
package SQL::Admin::Driver::DB2::Producer;
use base qw( SQL::Admin::Driver::Base::Producer );

use strict;
use warnings;

our $VERSION = v0.5.0;

######################################################################

use SQL::Admin::Driver::DB2::Keywords qw( :all );

our %DATA_TYPE_MAP = (
    'int2' => 'smallint',
    'int4' => 'integer',
    'int8' => 'bigint',

    'text' => 'varchar',
);

######################################################################

our $ESCAPE_ALL_IDENTIFIERS      = 0;
our $ESCAPE_NONRESERVED_KEYWORDS = 0;
our $ESCAPE_SQL_KEYWORDS         = 0;

######################################################################

our %REFERENTIAL_ACTION = (
    cascade   => [ 'CASCADE' ],
    no_action => [ 'NO', 'ACTION' ],
    restrict  => [ 'RESTRICT' ],
    set_null  => [ 'SET', 'NULL' ],
);

######################################################################
######################################################################

sub _escape_identifier {                 # ;
    my ($self, $identifier) = @_;
    my $lc = lc $identifier;

    $identifier = '"' . $identifier . '"'
      if $ESCAPE_ALL_IDENTIFIERS
        || $RESERVED_KEYWORDS{ $lc }
        || ($ESCAPE_NONRESERVED_KEYWORDS && $NONRESERVED_KEYWORDS{ $lc })
        || ($ESCAPE_SQL_KEYWORDS         && $SQL_KEYWORDS{ $lc })
      ;

    ##################################################################

    $identifier;
}


######################################################################
######################################################################
sub data_type {                          # ;
    my ($self, $data, $parent) = @_;

    $data = $DATA_TYPE_MAP{lc $data}
      if exists $DATA_TYPE_MAP{lc $data};

    $self->SUPER::data_type ($data, $parent);
}


######################################################################
######################################################################
sub current_timestamp {                  # ;
    'CURRENT', 'TIMESTAMP';
}


######################################################################
######################################################################
sub current_time {                       # ;
    'CURRENT', 'TIME';
}


######################################################################
######################################################################
sub current_date {                       # ;
    'CURRENT', 'DATE';
}


######################################################################
######################################################################
sub default_clause {                     # ;
    my ($self, $def) = @_;

    'WITH', 'DEFAULT', $self->__apply (%$def);
}


######################################################################
######################################################################

sub db2_in_tablespace {                  # ;
    'IN', '"' . $_[1] . '"';
}


######################################################################
######################################################################
sub db2_data_capture {                   # ;
    'DATA', 'CAPTURE', $_[1];
}


######################################################################
######################################################################
sub db2_pctfree {                        # ;
    'PCTFREE', $_[1];
}


######################################################################
######################################################################
sub db2_locksize {                       # ;
}


######################################################################
######################################################################
sub db2_append {                         # ;
}


######################################################################
######################################################################
sub db2_volatile {                       # ;
}


######################################################################
######################################################################
sub db2_log_index {                      # ;
}


######################################################################
######################################################################
sub db2_enforced {                       # ;
    # $_ ? () : ('NOT'), 'ENFORCED';
}


######################################################################
######################################################################
sub db2_optimize {                       # ;
    # $_ ? 'ENABLED' : 'DISABLED', 'QUERY', 'OPTIMIZATION';
}


######################################################################
######################################################################
sub autoincrement {                      # ;
    my ($self, $def) = @_;
    my $options = do { local $, = ', '; $self->__apply (sequence_options => $def) };

    $options = '(' . $options . ')'
      if $options;

    'GENERATED', 'AS', 'IDENTITY', $options;
}


######################################################################
######################################################################

sub sequence_type {                      # ;
    my ($self, $def) = @_;

    'AS', $self->data_type ($def);
}


######################################################################
######################################################################
sub sequence_options {                   # ;
    my ($self, $def) = @_;

    map $self->__call ($_, $def), (
        'sequence_start_with',
        'sequence_increment_by',
        'sequence_minvalue',
        'sequence_maxvalue',
        'sequence_cache',
    );
}


######################################################################
######################################################################
sub sequence_start_with {                # ;
    map +('START', 'WITH', $_), grep $_ != 1, $_[1];
}


######################################################################
######################################################################
sub sequence_increment_by {              # ;
    map +('INCREMENT', 'BY', $_), grep $_ != 1, $_[1];
}


######################################################################
######################################################################
sub sequence_minvalue {                  # ;
    my ($self, $def) = @_;

    map +('MINVALUE', $_), grep defined, $def;
}


######################################################################
######################################################################
sub sequence_maxvalue {                  # ;
    my ($self, $def) = @_;

    map +('MAXVALUE', $_), grep defined, $def;
}


######################################################################
######################################################################
sub sequence_cache {                     # ;
    my ($self, $def) = @_;

    map +('CACHE', $_), grep $_ != 20, grep defined, $def;
}


######################################################################
######################################################################

sub alter_table_actions {                # ;
    my ($self, $def) = @_;

    map $self->__apply (alter_table_action => $_), @$def;
}


######################################################################
######################################################################
sub alter_table_action {                 # ;
    my ($self, $def) = @_;

    $self->__apply (%$def);
}


######################################################################
######################################################################
sub delete_rule {                        # ;
    my ($self, $def) = @_;

    map +('ON', 'DELETE', @{ $REFERENTIAL_ACTION{$_} }), grep defined, $def;
}


######################################################################
######################################################################
sub update_rule {                        # ;
    my ($self, $def) = @_;

    map +('ON', 'UPDATE', @{ $REFERENTIAL_ACTION{$_} }), grep defined, $def;
}


######################################################################
######################################################################
sub foreign_key_constraint {             # ;
    my ($self, $def) = @_;

    (
        $self->SUPER::foreign_key_constraint ($def),
        $self->__call (db2_enforced => $def),
        $self->__call (db2_optimize => $def),
    );
}


######################################################################
######################################################################

sub create_sequence {                    # ;
    my ($self, $def) = @_;

    (
        'CREATE',
        'SEQUENCE',
        $self->__call (sequence_name => $def),
        $self->__call (sequence_type => $def),
        $self->__call (sequence_options => $def),
    );
}


######################################################################
######################################################################
sub create_index {                       # ;
    my ($self, $def) = @_;
    join ' ', (
        'CREATE',
        $self->__call (index_unique => $def),
        'INDEX',
        $self->__call (index_name => $def),
        'ON',
        $self->__call (table_name => $def),
        $self->__call (index_column_list => $def),
        $self->__call (index_options => $def),
    );
}


######################################################################
######################################################################
sub create_table {                       # ;
    my ($self, $def) = @_;

    join ' ', (
        'CREATE',
        'TABLE',
        $self->__call (table_name    => $def),
        $self->__call (table_content => $def),
        $self->__call (table_options => $def),
    );
}


######################################################################
######################################################################
sub alter_table {                        # ;
    my ($self, $def) = @_;

    my @list = map $self->__apply (%$_), @{ $def->{alter_table_actions} };

    return unless @list;

    (
        'ALTER',
        'TABLE',
        $self->__call (table_name => $def),
        @list
    );
}


######################################################################
######################################################################

sub create_schema {                      # ;
    my ($self, $def) = @_;

    (
        'CREATE',
        'SCHEMA',
        $self->_escape_identifier ($def->{schema_identifier}),
    );
}


######################################################################
######################################################################

sub select_what {                        # ;
    my ($self, $def, $parent) = @_;
    my ($key, @res) = $self->SUPER::select_what ($def, $parent);

    ($key, @res) = ('VALUES', '(', @res, ')')
      unless exists $parent->{select_from};

    ($key, @res);
}


######################################################################
######################################################################

package SQL::Admin::Driver::DB2::Producer;

1;
