use strict;
use warnings;

use Test::More;
use Test::Number::Delta relative => 0.0000000001;
use PDL;
use PDL::DateTime;

##                datetime            epoch_us           epoch_dbl                  rd                  sd                  jd                 mjd
my @data = (qw/
   0001-01-01T00:00:00.000  -62135596800000000      -62135596800.0                 1.0               367.0           1721425.5           -678575.0
   0001-01-01T00:00:00.001  -62135596799999000 -62135596799.999001  1.0000000115740741  367.00000001157406  1721425.5000000116 -678574.99999998847
   0001-01-01T00:00:00.002  -62135596799998000 -62135596799.998001  1.0000000231481481  367.00000002314817  1721425.5000000231 -678574.99999997683
   0001-01-01T00:00:00.010  -62135596799990000 -62135596799.989998  1.0000001157407408  367.00000011574076  1721425.5000001157 -678574.99999988428
   0001-01-01T00:00:00.020  -62135596799980000 -62135596799.980003  1.0000002314814815  367.00000023148147  1721425.5000002314 -678574.99999976857
   0001-01-01T00:00:00.100  -62135596799900000 -62135596799.900002  1.0000011574074075  367.00000115740738  1721425.5000011574  -678574.9999988426
   0001-01-01T00:00:00.200  -62135596799800000 -62135596799.800003  1.0000023148148147  367.00000231481482  1721425.5000023148  -678574.9999976852
   0001-01-01T00:00:00.300  -62135596799700000 -62135596799.699997  1.0000034722222222   367.0000034722222  1721425.5000034722  -678574.9999965278
   0001-01-01T00:00:00.400  -62135596799600000 -62135596799.599998  1.0000046296296297  367.00000462962964  1721425.5000046296  -678574.9999953704
   0001-01-01T00:00:00.490  -62135596799510000 -62135596799.510002  1.0000056712962964  367.00000567129632  1721425.5000056713 -678574.99999432871
   0001-01-01T00:00:00.499  -62135596799501000 -62135596799.500999   1.000005775462963  367.00000577546297  1721425.5000057754 -678574.99999422452
   0001-01-01T00:00:00.500  -62135596799500000      -62135596799.5  1.0000057870370371  367.00000578703703   1721425.500005787 -678574.99999421299
   0001-01-01T00:00:00.501  -62135596799499000 -62135596799.499001  1.0000057986111111  367.00000579861108  1721425.5000057986 -678574.99999420135
   0001-01-01T00:00:00.510  -62135596799490000 -62135596799.489998  1.0000059027777777  367.00000590277779  1721425.5000059027 -678574.99999409728
   0001-01-01T00:00:00.600  -62135596799400000 -62135596799.400002  1.0000069444444444  367.00000694444446  1721425.5000069444 -678574.99999305559
   0001-01-01T00:00:00.700  -62135596799300000 -62135596799.300003  1.0000081018518518  367.00000810185185  1721425.5000081018 -678574.99999189819
   0001-01-01T00:00:00.800  -62135596799200000 -62135596799.199997  1.0000092592592593  367.00000925925929  1721425.5000092592 -678574.99999074079
   0001-01-01T00:00:00.900  -62135596799100000 -62135596799.099998  1.0000104166666666  367.00001041666667  1721425.5000104166 -678574.99998958339
   0001-01-01T00:00:00.980  -62135596799020000 -62135596799.019997  1.0000113425925925  367.00001134259259  1721425.5000113426 -678574.99998865742
   0001-01-01T00:00:00.990  -62135596799010000 -62135596799.010002  1.0000114583333333  367.00001145833335  1721425.5000114583 -678574.99998854171
   0001-01-01T00:00:00.998  -62135596799002000 -62135596799.001999   1.000011550925926  367.00001155092593   1721425.500011551 -678574.99998844904
   0001-01-01T00:00:00.999  -62135596799001000 -62135596799.000999  1.0000115624999999  367.00001156249999  1721425.5000115626 -678574.99998843751
   0001-01-01T00:00:01.000  -62135596799000000      -62135596799.0   1.000011574074074  367.00001157407405   1721425.500011574 -678574.99998842587
   0001-01-01T00:00:01.001  -62135596798999000 -62135596798.999001  1.0000115856481482  367.00001158564817  1721425.5000115857 -678574.99998841435
   0001-01-01T00:00:01.002  -62135596798998000 -62135596798.998001  1.0000115972222223  367.00001159722223  1721425.5000115973 -678574.99998840282
   0001-01-01T00:00:01.010  -62135596798990000 -62135596798.989998  1.0000116898148148  367.00001168981481  1721425.5000116897 -678574.99998831016
   0001-01-01T00:00:01.020  -62135596798980000 -62135596798.980003  1.0000118055555556  367.00001180555557  1721425.5000118054 -678574.99998819444
   0001-01-01T00:00:01.100  -62135596798900000 -62135596798.900002  1.0000127314814815  367.00001273148149  1721425.5000127314 -678574.99998726847
   0001-01-01T00:00:01.200  -62135596798800000 -62135596798.800003   1.000013888888889  367.00001388888887  1721425.5000138888 -678574.99998611107
   1969-12-31T23:59:58.000            -2000000                -2.0  719162.99997685186  719528.99997685186   2440587.499976852  40586.999976851854
   1969-12-31T23:59:58.500            -1500000                -1.5  719162.99998263887  719528.99998263887  2440587.4999826388  40586.999982638888
   1969-12-31T23:59:58.800            -1200000                -1.2  719162.99998611107  719528.99998611107  2440587.4999861112  40586.999986111114
   1969-12-31T23:59:58.900            -1100000 -1.1000000000000001  719162.99998726847  719528.99998726847  2440587.4999872684  40586.999987268522
   1969-12-31T23:59:58.980            -1020000               -1.02  719162.99998819444  719528.99998819444  2440587.4999881946  40586.999988194446
   1969-12-31T23:59:58.990            -1010000               -1.01  719162.99998831016  719528.99998831016    2440587.49998831  40586.999988310185
   1969-12-31T23:59:58.998            -1002000              -1.002  719162.99998840282  719528.99998840282  2440587.4999884027  40586.999988402778
   1969-12-31T23:59:58.999            -1001000 -1.0009999999999999  719162.99998841435  719528.99998841435  2440587.4999884143  40586.999988414354
   1969-12-31T23:59:59.000            -1000000                -1.0  719162.99998842587  719528.99998842587   2440587.499988426  40586.999988425923
   1969-12-31T23:59:59.001             -999000              -0.999  719162.99998843751  719528.99998843751  2440587.4999884376  40586.999988437499
   1969-12-31T23:59:59.002             -998000              -0.998  719162.99998844904  719528.99998844904  2440587.4999884493  40586.999988449075
   1969-12-31T23:59:59.010             -990000 -0.98999999999999999  719162.99998854171  719528.99998854171  2440587.4999885415  40586.999988541669
   1969-12-31T23:59:59.020             -980000 -0.97999999999999998  719162.99998865742  719528.99998865742  2440587.4999886574  40586.999988657408
   1969-12-31T23:59:59.100             -900000 -0.90000000000000002  719162.99998958339  719528.99998958339  2440587.4999895832  40586.999989583332
   1969-12-31T23:59:59.200             -800000 -0.80000000000000004  719162.99999074079  719528.99999074079  2440587.4999907408   40586.99999074074
   1969-12-31T23:59:59.300             -700000 -0.69999999999999996  719162.99999189819  719528.99999189819   2440587.499991898  40586.999991898148
   1969-12-31T23:59:59.400             -600000 -0.59999999999999998  719162.99999305559  719528.99999305559  2440587.4999930556  40586.999993055557
   1969-12-31T23:59:59.490             -510000 -0.51000000000000001  719162.99999409728  719528.99999409728  2440587.4999940973  40586.999994097219
   1969-12-31T23:59:59.499             -501000              -0.501  719162.99999420135  719528.99999420135  2440587.4999942016  40586.999994201389
   1969-12-31T23:59:59.500             -500000                -0.5  719162.99999421299  719528.99999421299  2440587.4999942128  40586.999994212965
   1969-12-31T23:59:59.501             -499000              -0.499  719162.99999422452  719528.99999422452  2440587.4999942244  40586.999994224534
   1969-12-31T23:59:59.510             -490000 -0.48999999999999999  719162.99999432871  719528.99999432871  2440587.4999943287  40586.999994328704
   1969-12-31T23:59:59.600             -400000 -0.40000000000000002   719162.9999953704   719528.9999953704  2440587.4999953704  40586.999995370374
   1969-12-31T23:59:59.700             -300000 -0.29999999999999999   719162.9999965278   719528.9999965278  2440587.4999965276  40586.999996527775
   1969-12-31T23:59:59.800             -200000 -0.20000000000000001   719162.9999976852   719528.9999976852  2440587.4999976852  40586.999997685183
   1969-12-31T23:59:59.900             -100000 -0.10000000000000001   719162.9999988426   719528.9999988426  2440587.4999988424  40586.999998842592
   1969-12-31T23:59:59.980              -20000               -0.02  719162.99999976857  719528.99999976857  2440587.4999997686  40586.999999768515
   1969-12-31T23:59:59.990              -10000               -0.01  719162.99999988428  719528.99999988428  2440587.4999998841  40586.999999884261
   1969-12-31T23:59:59.998               -2000              -0.002  719162.99999997683  719528.99999997683  2440587.4999999767  40586.999999976855
   1969-12-31T23:59:59.999               -1000              -0.001  719162.99999998847  719528.99999998847  2440587.4999999884  40586.999999988424
   1970-01-01T00:00:00.000                   0                 0.0            719163.0            719529.0           2440587.5             40587.0
   1970-01-01T00:00:00.001                1000               0.001  719163.00000001153  719529.00000001153  2440587.5000000116  40587.000000011576
   1970-01-01T00:00:00.002                2000               0.002  719163.00000002317  719529.00000002317  2440587.5000000233  40587.000000023145
   1970-01-01T00:00:00.010               10000                0.01  719163.00000011572  719529.00000011572  2440587.5000001159  40587.000000115739
   1970-01-01T00:00:00.020               20000                0.02  719163.00000023143  719529.00000023143  2440587.5000002314  40587.000000231485
   1970-01-01T00:00:00.100              100000 0.10000000000000001   719163.0000011574   719529.0000011574  2440587.5000011576  40587.000001157408
   1970-01-01T00:00:00.200              200000 0.20000000000000001   719163.0000023148   719529.0000023148  2440587.5000023148  40587.000002314817
   1970-01-01T00:00:00.300              300000 0.29999999999999999   719163.0000034722   719529.0000034722  2440587.5000034724  40587.000003472225
   1970-01-01T00:00:00.400              400000 0.40000000000000002   719163.0000046296   719529.0000046296  2440587.5000046296  40587.000004629626
   1970-01-01T00:00:00.490              490000 0.48999999999999999  719163.00000567129  719529.00000567129  2440587.5000056713  40587.000005671296
   1970-01-01T00:00:00.499              499000               0.499  719163.00000577548  719529.00000577548  2440587.5000057756  40587.000005775466
   1970-01-01T00:00:00.500              500000                 0.5  719163.00000578701  719529.00000578701  2440587.5000057872  40587.000005787035
   1970-01-01T00:00:00.501              501000               0.501  719163.00000579865  719529.00000579865  2440587.5000057984  40587.000005798611
   1970-01-01T00:00:00.510              510000 0.51000000000000001  719163.00000590272  719529.00000590272  2440587.5000059027  40587.000005902781
   1970-01-01T00:00:00.600              600000 0.59999999999999998  719163.00000694441  719529.00000694441  2440587.5000069444  40587.000006944443
   1970-01-01T00:00:00.700              700000 0.69999999999999996  719163.00000810181  719529.00000810181   2440587.500008102  40587.000008101852
   1970-01-01T00:00:00.800              800000 0.80000000000000004  719163.00000925921  719529.00000925921  2440587.5000092592   40587.00000925926
   1970-01-01T00:00:00.900              900000 0.90000000000000002  719163.00001041661  719529.00001041661  2440587.5000104168  40587.000010416668
   1970-01-01T00:00:00.980              980000 0.97999999999999998  719163.00001134258  719529.00001134258  2440587.5000113426  40587.000011342592
   1970-01-01T00:00:00.990              990000 0.98999999999999999  719163.00001145829  719529.00001145829  2440587.5000114585  40587.000011458331
   1970-01-01T00:00:00.998              998000               0.998  719163.00001155096  719529.00001155096  2440587.5000115507  40587.000011550925
   1970-01-01T00:00:00.999              999000               0.999  719163.00001156249  719529.00001156249  2440587.5000115624  40587.000011562501
   1970-01-01T00:00:01.000             1000000                 1.0  719163.00001157413  719529.00001157413   2440587.500011574  40587.000011574077
   1970-01-01T00:00:01.001             1001000  1.0009999999999999  719163.00001158565  719529.00001158565  2440587.5000115857  40587.000011585646
   1970-01-01T00:00:01.002             1002000               1.002  719163.00001159718  719529.00001159718  2440587.5000115973  40587.000011597222
   1970-01-01T00:00:01.010             1010000                1.01  719163.00001168984  719529.00001168984    2440587.50001169  40587.000011689815
   1970-01-01T00:00:01.020             1020000                1.02  719163.00001180556  719529.00001180556  2440587.5000118054  40587.000011805554
   1970-01-01T00:00:01.100             1100000  1.1000000000000001  719163.00001273153  719529.00001273153  2440587.5000127316  40587.000012731478
   1970-01-01T00:00:01.200             1200000                 1.2  719163.00001388893  719529.00001388893  2440587.5000138888  40587.000013888886
   9999-12-31T23:59:58.000  253402300798000000      253402300798.0   3652059.999976852   3652425.999976852   5373484.499976852   2973483.999976852
   9999-12-31T23:59:58.500  253402300798500000      253402300798.5  3652059.9999826388  3652425.9999826388  5373484.4999826392  2973483.9999826388
   9999-12-31T23:59:58.800  253402300798800000  253402300798.79999  3652059.9999861112  3652425.9999861112  5373484.4999861112  2973483.9999861112
   9999-12-31T23:59:58.900  253402300798900000  253402300798.89999  3652059.9999872684  3652425.9999872684  5373484.4999872688  2973483.9999872684
   9999-12-31T23:59:58.980  253402300798980000  253402300798.98001  3652059.9999881946  3652425.9999881946  5373484.4999881946  2973483.9999881946
   9999-12-31T23:59:58.990  253402300798990000  253402300798.99002    3652059.99998831    3652425.99998831    5373484.49998831    2973483.99998831
   9999-12-31T23:59:58.998  253402300798998000  253402300798.99802  3652059.9999884027  3652425.9999884027  5373484.4999884032  2973483.9999884027
   9999-12-31T23:59:58.999  253402300798999000  253402300798.99899  3652059.9999884143  3652425.9999884143  5373484.4999884143  2973483.9999884143
   9999-12-31T23:59:59.000  253402300799000000      253402300799.0   3652059.999988426   3652425.999988426  5373484.4999884255   2973483.999988426
   9999-12-31T23:59:59.001  253402300799001000  253402300799.00101  3652059.9999884376  3652425.9999884376  5373484.4999884376  2973483.9999884376
   9999-12-31T23:59:59.002  253402300799002000  253402300799.00198  3652059.9999884493  3652425.9999884493  5373484.4999884488  2973483.9999884493
   9999-12-31T23:59:59.010  253402300799010000  253402300799.00998  3652059.9999885415  3652425.9999885415  5373484.4999885419  2973483.9999885415
   9999-12-31T23:59:59.020  253402300799020000  253402300799.01999  3652059.9999886574  3652425.9999886574  5373484.4999886574  2973483.9999886574
   9999-12-31T23:59:59.100  253402300799100000  253402300799.10001  3652059.9999895832  3652425.9999895832  5373484.4999895832  2973483.9999895832
   9999-12-31T23:59:59.200  253402300799200000  253402300799.20001  3652059.9999907408  3652425.9999907408  5373484.4999907408  2973483.9999907408
   9999-12-31T23:59:59.300  253402300799300000  253402300799.29999   3652059.999991898   3652425.999991898  5373484.4999918984   2973483.999991898
   9999-12-31T23:59:59.400  253402300799400000  253402300799.39999  3652059.9999930556  3652425.9999930556  5373484.4999930551  2973483.9999930556
   9999-12-31T23:59:59.490  253402300799490000  253402300799.48999  3652059.9999940973  3652425.9999940973  5373484.4999940973  2973483.9999940973
   9999-12-31T23:59:59.499  253402300799499000  253402300799.49899  3652059.9999942016  3652425.9999942016  5373484.4999942016  2973483.9999942016
   9999-12-31T23:59:59.500  253402300799500000      253402300799.5  3652059.9999942128  3652425.9999942128  5373484.4999942128  2973483.9999942128
   9999-12-31T23:59:59.501  253402300799501000  253402300799.50101  3652059.9999942244  3652425.9999942244  5373484.4999942249  2973483.9999942244
   9999-12-31T23:59:59.510  253402300799510000  253402300799.51001  3652059.9999943287  3652425.9999943287  5373484.4999943282  2973483.9999943287
   9999-12-31T23:59:59.600  253402300799600000  253402300799.60001  3652059.9999953704  3652425.9999953704  5373484.4999953704  2973483.9999953704
   9999-12-31T23:59:59.700  253402300799700000  253402300799.70001  3652059.9999965276  3652425.9999965276   5373484.499996528  2973483.9999965276
   9999-12-31T23:59:59.800  253402300799800000  253402300799.79999  3652059.9999976852  3652425.9999976852  5373484.4999976847  2973483.9999976852
   9999-12-31T23:59:59.900  253402300799900000  253402300799.89999  3652059.9999988424  3652425.9999988424  5373484.4999988424  2973483.9999988424
   9999-12-31T23:59:59.980  253402300799980000  253402300799.98001  3652059.9999997686  3652425.9999997686  5373484.4999997681  2973483.9999997686
   9999-12-31T23:59:59.990  253402300799990000  253402300799.99002  3652059.9999998841  3652425.9999998841  5373484.4999998845  2973483.9999998841
   9999-12-31T23:59:59.998  253402300799998000  253402300799.99802  3652059.9999999767  3652425.9999999767  5373484.4999999767  2973483.9999999767
   9999-12-31T23:59:59.999  253402300799999000  253402300799.99899  3652059.9999999884  3652425.9999999884  5373484.4999999888  2973483.9999999884
   0400-02-28T00:00:00.001  -49539427199999000 -49539427199.999001  145790.00000001158  146156.00000001158  1867214.5000000116 -532785.99999998847
   0400-02-29T00:00:00.000  -49539340800000000      -49539340800.0            145791.0            146157.0           1867215.5           -532785.0
   0400-03-01T00:12:00.001  -49539253679999000 -49539253679.999001  145792.00833334491  146158.00833334491  1867216.5083333449 -532783.99166665506
   0400-12-31T00:12:00.000  -49512901680000000      -49512901680.0  146097.00833333333  146463.00833333333  1867521.5083333333  -532478.9916666667
   0500-02-28T00:00:00.001  -46383667199999000 -46383667199.999001  182315.00000001158  182681.00000001158  1903739.5000000116 -496260.99999998842
   0500-03-01T00:00:00.000  -46383580800000000      -46383580800.0            182316.0            182682.0           1903740.5           -496260.0
   0500-12-31T00:00:00.001  -46357228799999000 -46357228799.999001  182621.00000001158  182987.00000001158  1904045.5000000116 -495954.99999998842
   1000-02-28T00:00:00.000  -30605212800000000      -30605212800.0            364936.0            365302.0           2086360.5           -313640.0
   1000-03-01T00:00:00.001  -30605126399999000 -30605126399.999001  364937.00000001158  365303.00000001158  2086361.5000000116 -313638.99999998842
   1000-12-31T00:00:00.000  -30578774400000000      -30578774400.0            365242.0            365608.0           2086666.5           -313334.0
   1500-02-28T00:00:00.001  -14826758399999000 -14826758399.999001  547557.00000001153  547923.00000001153  2268981.5000000116 -131018.99999998843
   1500-03-01T00:00:00.000  -14826672000000000      -14826672000.0            547558.0            547924.0           2268982.5           -131018.0
   1500-12-31T00:00:00.001  -14800319999999000 -14800319999.999001  547863.00000001153  548229.00000001153  2269287.5000000116 -130712.99999998843
   1600-02-28T00:00:00.000  -11671084800000000      -11671084800.0            584081.0            584447.0           2305505.5            -94495.0
   1600-02-29T00:00:00.001  -11670998399999000 -11670998399.999001  584082.00000001153  584448.00000001153  2305506.5000000116 -94493.999999988431
   1600-03-01T00:12:00.000  -11670911280000000      -11670911280.0   584083.0083333333   584449.0083333333  2305507.5083333333 -94492.991666666669
   1600-12-31T00:12:00.001  -11644559279999000 -11644559279.999001  584388.00833334494  584754.00833334494  2305812.5083333449   -94187.9916666551
   2000-02-28T00:00:00.000     951696000000000         951696000.0            730178.0            730544.0           2451602.5             51602.0
   2000-02-29T00:00:00.001     951782400001000  951782400.00100005  730179.00000001153  730545.00000001153  2451603.5000000116  51603.000000011576
   2000-03-01T00:00:00.000     951868800000000         951868800.0            730180.0            730546.0           2451604.5             51604.0
   2000-12-31T00:00:00.001     978220800001000  978220800.00100005  730485.00000001153  730851.00000001153  2451909.5000000116  51909.000000011576
   3000-02-28T00:00:00.000   32508691200000000       32508691200.0           1095421.0           1095787.0           2816845.5            416845.0
   3000-03-01T00:00:00.001   32508777600001000  32508777600.000999  1095422.0000000116  1095788.0000000116  2816846.5000000116  416846.00000001158
   3000-12-31T00:00:00.000   32535129600000000       32535129600.0           1095727.0           1096093.0           2817151.5            417151.0
   4000-02-28T00:00:00.001   64065600000001000  64065600000.000999  1460663.0000000116  1461029.0000000116  3182087.5000000116  782087.00000001153
   4000-02-29T00:00:00.000   64065686400000000       64065686400.0           1460664.0           1461030.0           3182088.5            782088.0
   4000-03-01T00:00:00.001   64065772800001000  64065772800.000999  1460665.0000000116  1461031.0000000116  3182089.5000000116  782089.00000001153
   4000-12-31T00:00:00.000   64092124800000000       64092124800.0           1460970.0           1461336.0           3182394.5            782394.0
   9600-02-28T00:00:00.001  240784531200001000  240784531200.00101  3506021.0000000116  3506387.0000000116  5227445.5000000112  2827445.0000000116
   9600-02-29T00:00:00.000  240784617600000000      240784617600.0           3506022.0           3506388.0           5227446.5           2827446.0
   9600-03-01T00:12:00.001  240784704720001000  240784704720.00101  3506023.0083333449  3506389.0083333449  5227447.5083333449  2827447.0083333449
   9600-12-31T00:12:00.000  240811056720000000      240811056720.0  3506328.0083333333  3506694.0083333333  5227752.5083333338  2827752.0083333333
/);

my @datetime;
my @datetime_microsec;
my @epoch_us;
my @ep;
my @rd;
my @sd;
my @jd;
my @mjd;

while (@data) {
  my $d = shift(@data);
  push @datetime, $d;
  push @datetime_microsec, $d . "999";
  push @epoch_us, int(shift(@data));
  push @ep      , 1.0 * shift(@data);
  push @rd      , 1.0 * shift(@data);
  push @sd      , 1.0 * shift(@data);
  push @jd      , 1.0 * shift(@data);
  push @mjd     , 1.0 * shift(@data);
}

my $pdl_epoch_us = longlong(\@epoch_us);
my $pdl_ep       =   double(\@ep);
my $pdl_rd       =   double(\@rd);
my $pdl_sd       =   double(\@sd);
my $pdl_jd       =   double(\@jd);
my $pdl_mjd      =   double(\@mjd);

my $dt0 = PDL::DateTime->new($pdl_epoch_us);
my $dt1 = PDL::DateTime->new_from_datetime(\@datetime);
my $dt2 = PDL::DateTime->new_from_datetime(\@datetime_microsec);
my $dt3 = PDL::DateTime->new_from_epoch     (\@ep);
my $dt4 = PDL::DateTime->new_from_ratadie   (\@rd);
my $dt5 = PDL::DateTime->new_from_serialdate(\@sd);
my $dt6 = PDL::DateTime->new_from_juliandate(\@jd);
my $dt7 = PDL::DateTime->new_from_epoch     ($dt1->double_epoch);
my $dt8 = PDL::DateTime->new_from_ratadie   ($dt1->double_ratadie);
my $dt9 = PDL::DateTime->new_from_serialdate($dt1->double_serialdate);
my $dtA = PDL::DateTime->new_from_juliandate($dt1->double_juliandate);
# microseconds should be dropped before exporting via double_*
# like: 9600-12-31T00:12:00.000999 > 9600-12-31T00:12:00.000
my $dtB = PDL::DateTime->new_from_epoch     ($dt2->double_epoch);
my $dtC = PDL::DateTime->new_from_ratadie   ($dt2->double_ratadie);
my $dtD = PDL::DateTime->new_from_serialdate($dt2->double_serialdate);
my $dtE = PDL::DateTime->new_from_juliandate($dt2->double_juliandate);

is(ref $dt1->double_epoch, 'PDL');
is(ref $dt1->longlong_epoch, 'PDL');
is(ref $dt1->double_ratadie, 'PDL');
is(ref $dt1->double_serialdate, 'PDL');
is(ref $dt1->double_juliandate, 'PDL');

is($dt1->double_epoch->type, 'double');
is($dt1->double_ratadie->type, 'double');
is($dt1->double_serialdate->type, 'double');
is($dt1->double_juliandate->type, 'double');

ok(all($dt1 == $dt0),          'compare dt/pdl');
ok(all($dt1 == $dt3),          'compare dt/ep1');
ok(all($dt1 == $dt4),          'compare dt/rd1');
ok(all($dt1 == $dt5),          'compare dt/sd1');
ok(all($dt1 == $dt6),          'compare dt/jd1');
ok(all($dt1 == $dt7),          'compare dt/ep2');
ok(all($dt1 == $dt8),          'compare dt/rd2');
ok(all($dt1 == $dt9),          'compare dt/sd2');
ok(all($dt1 == $dtA),          'compare dt/jd2');
ok(all($dt1 == $dtB),          'compare dt/ep3');
ok(all($dt1 == $dtC),          'compare dt/rd3');
ok(all($dt1 == $dtD),          'compare dt/sd3');
ok(all($dt1 == $dtE),          'compare dt/jd3');
ok(all($dt1 == $pdl_epoch_us), 'compare epoch_us');

is_deeply($dt1->dt_unpdl("%Y-%m-%dT%H:%M:%S.%3N"), \@datetime, "dt_unpdl('..')");
is_deeply($dt1->unpdl, \@epoch_us, "dt_unpdl");
delta_ok($dt1->dt_unpdl('epoch'), \@ep, "dt_unpdl('epoch')");

delta_ok($dt1->double_epoch->unpdl,      \@ep, '1-double_epoch');
delta_ok($dt1->double_ratadie->unpdl,    \@rd, '1-double_ratadie');
delta_ok($dt1->double_serialdate->unpdl, \@sd, '1-double_serialdate');
delta_ok($dt1->double_juliandate->unpdl, \@jd, '1-double_juliandate');
delta_ok($dt2->double_epoch->unpdl,      \@ep, '2-double_epoch');
delta_ok($dt2->double_ratadie->unpdl,    \@rd, '2-double_ratadie');
delta_ok($dt2->double_serialdate->unpdl, \@sd, '2-double_serialdate');
delta_ok($dt2->double_juliandate->unpdl, \@jd, '2-double_juliandate');


done_testing();

__END__

##### generated by the following script #####

use strict;
use warnings;

use Time::Moment;
use DateTime;

my $test1 = [qw/
0001-01-01T00:00:00.000Z
0001-01-01T00:00:00.001Z
0001-01-01T00:00:00.002Z
0001-01-01T00:00:00.01Z
0001-01-01T00:00:00.02Z
0001-01-01T00:00:00.1Z
0001-01-01T00:00:00.2Z
0001-01-01T00:00:00.3Z
0001-01-01T00:00:00.4Z
0001-01-01T00:00:00.49Z
0001-01-01T00:00:00.499Z
0001-01-01T00:00:00.5Z
0001-01-01T00:00:00.501Z
0001-01-01T00:00:00.51Z
0001-01-01T00:00:00.6Z
0001-01-01T00:00:00.7Z
0001-01-01T00:00:00.8Z
0001-01-01T00:00:00.9Z
0001-01-01T00:00:00.98Z
0001-01-01T00:00:00.99Z
0001-01-01T00:00:00.998Z
0001-01-01T00:00:00.999Z
0001-01-01T00:00:01Z
0001-01-01T00:00:01.001Z
0001-01-01T00:00:01.002Z
0001-01-01T00:00:01.01Z
0001-01-01T00:00:01.02Z
0001-01-01T00:00:01.1Z
0001-01-01T00:00:01.2Z
1969-12-31T23:59:58Z
1969-12-31T23:59:58.5Z
1969-12-31T23:59:58.8Z
1969-12-31T23:59:58.9Z
1969-12-31T23:59:58.98Z
1969-12-31T23:59:58.99Z
1969-12-31T23:59:58.998Z
1969-12-31T23:59:58.999Z
1969-12-31T23:59:59Z
1969-12-31T23:59:59.001Z
1969-12-31T23:59:59.002Z
1969-12-31T23:59:59.01Z
1969-12-31T23:59:59.02Z
1969-12-31T23:59:59.1Z
1969-12-31T23:59:59.2Z
1969-12-31T23:59:59.3Z
1969-12-31T23:59:59.4Z
1969-12-31T23:59:59.49Z
1969-12-31T23:59:59.499Z
1969-12-31T23:59:59.5Z
1969-12-31T23:59:59.501Z
1969-12-31T23:59:59.51Z
1969-12-31T23:59:59.6Z
1969-12-31T23:59:59.7Z
1969-12-31T23:59:59.8Z
1969-12-31T23:59:59.9Z
1969-12-31T23:59:59.98Z
1969-12-31T23:59:59.99Z
1969-12-31T23:59:59.998Z
1969-12-31T23:59:59.999Z
1970-01-01T00:00:00Z
1970-01-01T00:00:00.001Z
1970-01-01T00:00:00.002Z
1970-01-01T00:00:00.01Z
1970-01-01T00:00:00.02Z
1970-01-01T00:00:00.1Z
1970-01-01T00:00:00.2Z
1970-01-01T00:00:00.3Z
1970-01-01T00:00:00.4Z
1970-01-01T00:00:00.49Z
1970-01-01T00:00:00.499Z
1970-01-01T00:00:00.5Z
1970-01-01T00:00:00.501Z
1970-01-01T00:00:00.51Z
1970-01-01T00:00:00.6Z
1970-01-01T00:00:00.7Z
1970-01-01T00:00:00.8Z
1970-01-01T00:00:00.9Z
1970-01-01T00:00:00.98Z
1970-01-01T00:00:00.99Z
1970-01-01T00:00:00.998Z
1970-01-01T00:00:00.999Z
1970-01-01T00:00:01Z
1970-01-01T00:00:01.001Z
1970-01-01T00:00:01.002Z
1970-01-01T00:00:01.01Z
1970-01-01T00:00:01.02Z
1970-01-01T00:00:01.1Z
1970-01-01T00:00:01.2Z
9999-12-31T23:59:58Z
9999-12-31T23:59:58.5Z
9999-12-31T23:59:58.8Z
9999-12-31T23:59:58.9Z
9999-12-31T23:59:58.98Z
9999-12-31T23:59:58.99Z
9999-12-31T23:59:58.998Z
9999-12-31T23:59:58.999Z
9999-12-31T23:59:59Z
9999-12-31T23:59:59.001Z
9999-12-31T23:59:59.002Z
9999-12-31T23:59:59.01Z
9999-12-31T23:59:59.02Z
9999-12-31T23:59:59.1Z
9999-12-31T23:59:59.2Z
9999-12-31T23:59:59.3Z
9999-12-31T23:59:59.4Z
9999-12-31T23:59:59.49Z
9999-12-31T23:59:59.499Z
9999-12-31T23:59:59.5Z
9999-12-31T23:59:59.501Z
9999-12-31T23:59:59.51Z
9999-12-31T23:59:59.6Z
9999-12-31T23:59:59.7Z
9999-12-31T23:59:59.8Z
9999-12-31T23:59:59.9Z
9999-12-31T23:59:59.98Z
9999-12-31T23:59:59.99Z
9999-12-31T23:59:59.998Z
9999-12-31T23:59:59.999Z
0400-02-28T00:00:00.001Z
0400-02-29T00:00:00.000Z
0400-03-01T00:12:00.001Z
0400-12-31T00:12:00.000Z
0500-02-28T00:00:00.001Z
0500-03-01T00:00:00.000Z
0500-12-31T00:00:00.001Z
1000-02-28T00:00:00.000Z
1000-03-01T00:00:00.001Z
1000-12-31T00:00:00.000Z
1500-02-28T00:00:00.001Z
1500-03-01T00:00:00.000Z
1500-12-31T00:00:00.001Z
1600-02-28T00:00:00.000Z
1600-02-29T00:00:00.001Z
1600-03-01T00:12:00.000Z
1600-12-31T00:12:00.001Z
2000-02-28T00:00:00.000Z
2000-02-29T00:00:00.001Z
2000-03-01T00:00:00.000Z
2000-12-31T00:00:00.001Z
3000-02-28T00:00:00.000Z
3000-03-01T00:00:00.001Z
3000-12-31T00:00:00.000Z
4000-02-28T00:00:00.001Z
4000-02-29T00:00:00.000Z
4000-03-01T00:00:00.001Z
4000-12-31T00:00:00.000Z
9600-02-28T00:00:00.001Z
9600-02-29T00:00:00.000Z
9600-03-01T00:12:00.001Z
9600-12-31T00:12:00.000Z
/];

printf("##% 24s % 19s % 19s % 19s % 19s % 19s % 19s\n",
       'datetime' ,
       'epoch_us' ,
       'epoch_dbl',
       'rd'       ,
       'sd'       ,
       'jd'       ,
       'mjd'      ,
);
for my $iso (@$test1) {
  my $tm = Time::Moment->from_string($iso);
  my $microsecond   = $tm->microsecond   ;
  my $epoch         = $tm->epoch         ;
  my $ep            = sprintf("%.22f", ($epoch*1000000+$microsecond)/1000000 ) =~ s/([^\.])0+$/$1/r;
  my $jd            = sprintf("%.22f", $tm->jd ) =~ s/([^\.])0+$/$1/r;
  my $mjd           = sprintf("%.22f", $tm->mjd) =~ s/([^\.])0+$/$1/r;
  my $rd            = sprintf("%.22f", $tm->rd ) =~ s/([^\.])0+$/$1/r;
  my $sd            = sprintf("%.22f", $rd + 366.0) =~ s/([^\.])0+$/$1/r;
  my $dt = DateTime->from_object(object=>$tm);
  die "jumboepoch: $iso" if $dt->epoch*1000000 + $dt->microsecond != $epoch*1000000+$microsecond;
  my ($rdd, $rds, $rdn) = $dt->utc_rd_values();
  printf("  % 24s % 19s % 19s % 19s % 19s % 19s % 19s\n",
         $tm->strftime("%Y-%m-%dT%H:%M:%S.%3N"),
         $epoch*1000000+$microsecond,
         $ep,
         $rd,
         $sd,
         $jd,
         $mjd,
  );
}
