#!/usr/bin/env perl

use Bio::GeneDesign;
use Getopt::Long;
use Pod::Usage;

use strict;
use warnings;

my $VERSION = '5.55';
my $GDV = "GD_Graph_RSCU_Values_$VERSION";
my $GDS = "_GRV";

local $| = 1;

##Get Arguments
my %p = ();
GetOptions (
      'input=s'     => \$p{INPUT},
      'rscu=s'      => \$p{FILES},
      'organism=s'  => \$p{ORGS},
      'window=i'    => \$p{WINDOW},
      'output=s'    => \$p{OUTPUT},
      'align!'      => \$p{ALIGNS},
      'stats!'      => \$p{STATISTICS},
      'help'        => \$p{HELP}
);

################################################################################
################################ SANITY  CHECK #################################
################################################################################
pod2usage(-verbose=>99, -sections=>"NAME|VERSION|DESCRIPTION|ARGUMENTS|USAGE")
  if ($p{HELP});

my $GD = Bio::GeneDesign->new();

#The input file must exist and be a format we care to read.
die "\n GDERROR: You must supply an input file.\n"
  if (! $p{INPUT});
my ($iterator, $filename, $suffix) = $GD->import_seqs($p{INPUT});

#The output path must exist, and we'll need it to end with a slash
$p{OUTPUT} = $p{OUTPUT} || ".";
$p{OUTPUT} .= "/" if (substr($p{OUTPUT}, -1, 1) !~ /[\/]/);
die "\n GDERROR: $p{OUTPUT} does not exist.\n"
  if ($p{OUTPUT} && ! -e $p{OUTPUT});

#We must get a list of organisms or a set of rscu files
die "\n GDERROR: Neither an organism nor an RSCU table were supplied.\n"
  if (! $p{ORGS} && ! $p{FILES});

$p{ORGS}   = $p{ORGS}   || q{};
$p{FILES}  = $p{FILES}  || q{};
$p{WINDOW} = $p{WINDOW} || 10;

################################################################################
################################# CONFIGURING ##################################
################################################################################
my @fileswritten;

my %works = ();
foreach my $org (split (",", $p{ORGS}))
{
  $works{$org} = {on => $org, path => undef};
}
foreach my $file ( split ( ",", $p{FILES} ) )
{
  $works{$file} = {on => basename($file), path => $file};
}

################################################################################
################################## GRAPHING  ###################################
################################################################################
my @allseqs;
my ($maxlen, $minlen) = (0, 0);
while ( my $obj = $iterator->next_seq() )
{
  my $name = $obj->id;
  if (length($obj->seq) % 3)
  {
    warn "\nGDWARNING: Skipping $name (wrong length to be a gene).\n";
    next;
  }
  push @allseqs, $obj;
}

foreach my $work (keys %works)
{
  $GD->set_organism(
      -organism_name => $works{$work}->{on},
      -rscu_path     => $works{$work}->{path});
  my ($graph, $format) = $GD->make_graph(-sequences => \@allseqs,
                                         -window    => $p{WINDOW});
  my $path = $p{OUTPUT} . $filename . $GDS . "_" . $works{$work}->{on} ;
  $path .= ".$format";
  open   (my $IMG, '>', $path) or die "can't create graph file, $!";
  binmode $IMG;
  print   $IMG $graph;
  close   $IMG;
  push @fileswritten, $path;
}

print "\n";
print "Wrote $_\n" foreach @fileswritten;
print "\n";
print $GD->attitude() . " brought to you by $GDV\n\n";

exit;

__END__

=head1 NAME

  GD_Graph_RSCU_Values.pl

=head1 VERSION

  Version 5.55

=head1 DESCRIPTION

-r OR -org must be provided. If both are given the table will be treated as
    another organism, named after the table's filename.

=head1 ARGUMENTS

Required arguments:

  -i,   --input : a file containing nucleotide sequences.
  -org, --organism : an organism whose RSCU table can be found in the config
    directory, or several separated by commas
  AND/OR
  -r,   --rscu : path to an RSCU table generated by GD_Generate_RSCU_Table.pl,
    or several separated by commas

Optional arguments:

  -out, --output : path to an output directory
  -w,   --window : the width of the averaging window
  -h,   --help : display this message

=head1 COPYRIGHT AND LICENSE

Copyright (c) 2015, GeneDesign developers
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

* Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright notice, this
list of conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.

* The names of Johns Hopkins, the Joint Genome Institute, the Lawrence Berkeley
National Laboratory, the Department of Energy, and the GeneDesign developers may
not be used to endorse or promote products derived from this software without
specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE DEVELOPERS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=cut
