// Copyright 2018 Ulf Adams
//
// The contents of this file may be used under the terms of the Apache License,
// Version 2.0.
//
//    (See accompanying file LICENSE-Apache or copy at
//     http://www.apache.org/licenses/LICENSE-2.0)
//
// Alternatively, the contents of this file may be used under the terms of
// the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE-Boost or copy at
//     https://www.boost.org/LICENSE_1_0.txt)
//
// Unless required by applicable law or agreed to in writing, this software
// is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.

#include <math.h>
#include <stdint.h>

#include "ryu/ryu.h"
#include "third_party/gtest/gtest.h"

struct test_case {
  double value;
  uint32_t fixed_precision;
  const char* fixed_string;
  uint32_t exp_precision;
  const char* exp_string;
};

// These values test every power of ten that's within the range of doubles.
const test_case all_powers_of_ten[] = {
  {
    1e-323,
    1073,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000098813129168249308835313758573644274"
    "473011960522864952885117136500135101454041750373059967272327198475959312939089143546185331"
    "342071187959279754959202156375625260142638062280905569163433569796420737743727211399746144"
    "610001277481830712996877462494679454633923028006343077079614825247713118234205331711337353"
    "637407912062124986389054318298491065861091308880225496025941999908386397881816083312664904"
    "951429573802945356031871047722310026960705298694403875805362142149834066644536895066714416"
    "648638721847657869167361202120230123396195061566845546366584958099650494615527518544957493"
    "121695564074689393990672940359453554351702513211023982630097822029020757254763345019116747"
    "7946719798732961988232841140527418055848553508913045817507736501283943653106689453125",
    749,
    "9.8813129168249308835313758573644274473011960522864952885117136500135101454041750373059967"
    "272327198475959312939089143546185331342071187959279754959202156375625260142638062280905569"
    "163433569796420737743727211399746144610001277481830712996877462494679454633923028006343077"
    "079614825247713118234205331711337353637407912062124986389054318298491065861091308880225496"
    "025941999908386397881816083312664904951429573802945356031871047722310026960705298694403875"
    "805362142149834066644536895066714416648638721847657869167361202120230123396195061566845546"
    "366584958099650494615527518544957493121695564074689393990672940359453554351702513211023982"
    "630097822029020757254763345019116747794671979873296198823284114052741805584855350891304581"
    "7507736501283943653106689453125e-324"
  },
  {
    1e-322,
    1072,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000988131291682493088353137585736442744"
    "730119605228649528851171365001351014540417503730599672723271984759593129390891435461853313"
    "420711879592797549592021563756252601426380622809055691634335697964207377437272113997461446"
    "100012774818307129968774624946794546339230280063430770796148252477131182342053317113373536"
    "374079120621249863890543182984910658610913088802254960259419999083863978818160833126649049"
    "514295738029453560318710477223100269607052986944038758053621421498340666445368950667144166"
    "486387218476578691673612021202301233961950615668455463665849580996504946155275185449574931"
    "216955640746893939906729403594535543517025132110239826300978220290207572547633450191167477"
    "946719798732961988232841140527418055848553508913045817507736501283943653106689453125",
    749,
    "9.8813129168249308835313758573644274473011960522864952885117136500135101454041750373059967"
    "272327198475959312939089143546185331342071187959279754959202156375625260142638062280905569"
    "163433569796420737743727211399746144610001277481830712996877462494679454633923028006343077"
    "079614825247713118234205331711337353637407912062124986389054318298491065861091308880225496"
    "025941999908386397881816083312664904951429573802945356031871047722310026960705298694403875"
    "805362142149834066644536895066714416648638721847657869167361202120230123396195061566845546"
    "366584958099650494615527518544957493121695564074689393990672940359453554351702513211023982"
    "630097822029020757254763345019116747794671979873296198823284114052741805584855350891304581"
    "7507736501283943653106689453125e-323"
  },
  {
    1e-321,
    1073,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000009980126045993180192366689615938071721"
    "774208012809360241396830786513645246858216787679056694505047046071890606848003498164718465"
    "549189983887255250879417793938151274406444290371462485506790549438494512116448351374360605"
    "610129025664902012684623711962624918026225828640650785041097350019024941654738502845072717"
    "378199118274623625294486148147597651970222196902775098620141990747026186063424414579155400"
    "094386954097480959218975819953312723031235168134791456341576357133240731098226401738156081"
    "512510906613444785903481414143242463015701218251400183025080768064699956168279373040706805"
    "291251971543628793057966976304808989521953834313422245639880024931096482731097846930791527"
    "2618699672029160811516955193269223640703904400217627568281386629678308963775634765625",
    751,
    "9.9801260459931801923666896159380717217742080128093602413968307865136452468582167876790566"
    "945050470460718906068480034981647184655491899838872552508794177939381512744064442903714624"
    "855067905494384945121164483513743606056101290256649020126846237119626249180262258286406507"
    "850410973500190249416547385028450727173781991182746236252944861481475976519702221969027750"
    "986201419907470261860634244145791554000943869540974809592189758199533127230312351681347914"
    "563415763571332407310982264017381560815125109066134447859034814141432424630157012182514001"
    "830250807680646999561682793730407068052912519715436287930579669763048089895219538343134222"
    "456398800249310964827310978469307915272618699672029160811516955193269223640703904400217627"
    "568281386629678308963775634765625e-322"
  },
  {
    1e-320,
    1071,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000099998886718268300541337523676528005766"
    "688104049139332319738542138136722671490251377536686879595124857670824694358213268739555318"
    "176042214791112018712582252132763264349719028276435993394772633977786596651937936543098345"
    "321292811612681552839992044615608089530104342419194004570203150685675653015795691873401881"
    "056807006870486225722970118072958651424404586788201978253303907287034656397876312416883810"
    "846728688580700304253500294977747284233762278736722315026487855632075444271337807514989648"
    "422386509829763597369536545672884876949402305647692922983977596846300550913848767496983039"
    "155910843585666718561015643766997003922943369556270421658995893369006341820505159346148768"
    "20804363177575320916352342137470725187361510200023673178293392993509769439697265625",
    750,
    "9.9998886718268300541337523676528005766688104049139332319738542138136722671490251377536686"
    "879595124857670824694358213268739555318176042214791112018712582252132763264349719028276435"
    "993394772633977786596651937936543098345321292811612681552839992044615608089530104342419194"
    "004570203150685675653015795691873401881056807006870486225722970118072958651424404586788201"
    "978253303907287034656397876312416883810846728688580700304253500294977747284233762278736722"
    "315026487855632075444271337807514989648422386509829763597369536545672884876949402305647692"
    "922983977596846300550913848767496983039155910843585666718561015643766997003922943369556270"
    "421658995893369006341820505159346148768208043631775753209163523421374707251873615102000236"
    "73178293392993509769439697265625e-321"
  },
  {
    1e-319,
    1070,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000999988867182683005413375236765280057666"
    "881040491393323197385421381367226714902513775366868795951248576708246943582132687395553181"
    "760422147911120187125822521327632643497190282764359933947726339777865966519379365430983453"
    "212928116126815528399920446156080895301043424191940045702031506856756530157956918734018810"
    "568070068704862257229701180729586514244045867882019782533039072870346563978763124168838108"
    "467286885807003042535002949777472842337622787367223150264878556320754442713378075149896484"
    "223865098297635973695365456728848769494023056476929229839775968463005509138487674969830391"
    "559108435856667185610156437669970039229433695562704216589958933690063418205051593461487682"
    "0804363177575320916352342137470725187361510200023673178293392993509769439697265625",
    750,
    "9.9998886718268300541337523676528005766688104049139332319738542138136722671490251377536686"
    "879595124857670824694358213268739555318176042214791112018712582252132763264349719028276435"
    "993394772633977786596651937936543098345321292811612681552839992044615608089530104342419194"
    "004570203150685675653015795691873401881056807006870486225722970118072958651424404586788201"
    "978253303907287034656397876312416883810846728688580700304253500294977747284233762278736722"
    "315026487855632075444271337807514989648422386509829763597369536545672884876949402305647692"
    "922983977596846300550913848767496983039155910843585666718561015643766997003922943369556270"
    "421658995893369006341820505159346148768208043631775753209163523421374707251873615102000236"
    "73178293392993509769439697265625e-320"
  },
  {
    1e-318,
    1073,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000009999987484955998303442587681411374220943"
    "283416874456096926739330950172402250479179504041747926784812965558428748760416017501717148"
    "946292667070481626217427369651951695114540889924504908640696967575080402937520865709580676"
    "739282438749985996996081924055488407644357269925743534099929893815278419813774519051525459"
    "318108599110747558686066125594356208301549987700423321356332728611852037669447325001045989"
    "624298431872975781381900545497038450336933172366635378454147705357378493778317646565679258"
    "887289704824017606121015769408717818336425626336137844764344642729705586000404268243261408"
    "712779922641361250092237317059153946646039468838066148529687158929654939305279279633993568"
    "5990351574486171151756262515234669929463655509149777600441666436381638050079345703125",
    754,
    "9.9999874849559983034425876814113742209432834168744560969267393309501724022504791795040417"
    "479267848129655584287487604160175017171489462926670704816262174273696519516951145408899245"
    "049086406969675750804029375208657095806767392824387499859969960819240554884076443572699257"
    "435340999298938152784198137745190515254593181085991107475586860661255943562083015499877004"
    "233213563327286118520376694473250010459896242984318729757813819005454970384503369331723666"
    "353784541477053573784937783176465656792588872897048240176061210157694087178183364256263361"
    "378447643446427297055860004042682432614087127799226413612500922373170591539466460394688380"
    "661485296871589296549393052792796339935685990351574486171151756262515234669929463655509149"
    "777600441666436381638050079345703125e-319"
  },
  {
    1e-317,
    1074,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000100000023069253735408389129784751602675844"
    "543686685345266696720985206474225156972857665977069218756620453298226457012793890336449486"
    "476033452643735894613076931082954841359365992666407440152120030445435135990799474195425984"
    "307826303722606039456135434296903258394457241266949956618721176024353875489053188082260623"
    "637197859200663066444242733391298681807136840324571457602240285981099973517197374979457253"
    "670128679434175847866812026553849543810389671707959598249520266798536037749981808256864213"
    "845855131011662864961199497267523368458548855711646767193323864446531601927333960250050326"
    "810342572525646591908382581130719797987948458197197459242018322340080528934937813868610807"
    "68235954429611544999118868631378263784093853548673447306782691157422959804534912109375",
    757,
    "1.0000002306925373540838912978475160267584454368668534526669672098520647422515697285766597"
    "706921875662045329822645701279389033644948647603345264373589461307693108295484135936599266"
    "640744015212003044543513599079947419542598430782630372260603945613543429690325839445724126"
    "694995661872117602435387548905318808226062363719785920066306644424273339129868180713684032"
    "457145760224028598109997351719737497945725367012867943417584786681202655384954381038967170"
    "795959824952026679853603774998180825686421384585513101166286496119949726752336845854885571"
    "164676719332386444653160192733396025005032681034257252564659190838258113071979798794845819"
    "719745924201832234008052893493781386861080768235954429611544999118868631378263784093853548"
    "673447306782691157422959804534912109375e-317"
  },
  {
    1e-316,
    1074,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000999999983659714433460619209563119592647759"
    "254336952145504584997059223491913816093472283838042269385386536792366287780216044499031536"
    "405156556539159558732763919890485263237064770961810478612616379963299515548676713454894481"
    "553259843521483612069160686732333947359764842653641873488174697124255959305018551544262852"
    "278458811851318198469791538166759153418640131045150835957547860040033740467433541512910274"
    "322712859834395088588442646232720370702133470343586292981797312610775210888475844901856096"
    "836954505497483976693591967374658376095000903199353806016776249216189782734520806138109535"
    "299186815069742434107143460408564094000228298944414635844938668328253396212469776138316207"
    "33691549327791400285367657800597186444724697763908327630133499042131006717681884765625",
    757,
    "9.9999998365971443346061920956311959264775925433695214550458499705922349191381609347228383"
    "804226938538653679236628778021604449903153640515655653915955873276391989048526323706477096"
    "181047861261637996329951554867671345489448155325984352148361206916068673233394735976484265"
    "364187348817469712425595930501855154426285227845881185131819846979153816675915341864013104"
    "515083595754786004003374046743354151291027432271285983439508858844264623272037070213347034"
    "358629298179731261077521088847584490185609683695450549748397669359196737465837609500090319"
    "935380601677624921618978273452080613810953529918681506974243410714346040856409400022829894"
    "441463584493866832825339621246977613831620733691549327791400285367657800597186444724697763"
    "908327630133499042131006717681884765625e-317"
  },
  {
    1e-315,
    1074,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000009999999984816838086980155348601833786944004"
    "252887462239343279298267939669340813115785463940012644762356165637601847210794160309593361"
    "064672347330515219766442433468290522584604803039463139871314154327626262102357951648564032"
    "447600351437582190186923061065358655548532968545933350501169209114129270401493513009634553"
    "240699986606369464281496859115328132978038273771846603614391600262917001497059540098100100"
    "654272959048368919932233033910668747462392651477468743526016339332503208851563791618632593"
    "342503137746326570686961476926928946044301624343806379717639929311373569268499339198531592"
    "674411496809458432057444014624821271529836759260682332394533416326065098006842778911837195"
    "06116290258908432677169195113883135284975280272778963563951037940569221973419189453125",
    758,
    "9.9999999848168380869801553486018337869440042528874622393432792982679396693408131157854639"
    "400126447623561656376018472107941603095933610646723473305152197664424334682905225846048030"
    "394631398713141543276262621023579516485640324476003514375821901869230610653586555485329685"
    "459333505011692091141292704014935130096345532406999866063694642814968591153281329780382737"
    "718466036143916002629170014970595400981001006542729590483689199322330339106687474623926514"
    "774687435260163393325032088515637916186325933425031377463265706869614769269289460443016243"
    "438063797176399293113735692684993391985315926744114968094584320574440146248212715298367592"
    "606823323945334163260650980068427789118371950611629025890843267716919511388313528497528027"
    "2778963563951037940569221973419189453125e-316"
  },
  {
    1e-314,
    1074,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000099999999996388074622175516738988975729906454"
    "238392563177730222310355101443610783338917264959716398532052454089957441516575318415211607"
    "659830255244071830103227569246343116060005123815989752458291897970893727639170333585259541"
    "391005430598567971364546804395605737436214227468848120631114329012862381366243127663351562"
    "863111734156882122398550068601017928572019701038804280182829002491749611793319525949998363"
    "969873951188107233370136910690152515064984462816293248968206606549783188482443258786397558"
    "397989460234752510620656572449634645537308835788332116726276730263211434608284669802752166"
    "426658314206618411560449556787393046825921362423359297890480896304182115950572806647047072"
    "30363699570078756595184568246742623687481105361485322902126426924951374530792236328125",
    759,
    "9.9999999996388074622175516738988975729906454238392563177730222310355101443610783338917264"
    "959716398532052454089957441516575318415211607659830255244071830103227569246343116060005123"
    "815989752458291897970893727639170333585259541391005430598567971364546804395605737436214227"
    "468848120631114329012862381366243127663351562863111734156882122398550068601017928572019701"
    "038804280182829002491749611793319525949998363969873951188107233370136910690152515064984462"
    "816293248968206606549783188482443258786397558397989460234752510620656572449634645537308835"
    "788332116726276730263211434608284669802752166426658314206618411560449556787393046825921362"
    "423359297890480896304182115950572806647047072303636995700787565951845682467426236874811053"
    "61485322902126426924951374530792236328125e-315"
  },
  {
    1e-313,
    1074,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000001000000000013287310805879821807546636585886679"
    "620431612038734699546109582686175384116193524783693968956688140137554071635297755925208742"
    "269338146420358178511876770651243790671370269300350309164635764607147645263566941552468486"
    "215054944726595070143906775203397101679103788691652744850950702752480372779533942489184305"
    "449821297599883717180027845159424818650742664828155549841261024871689316874103301156316092"
    "174454298782545011717304630762680164130197277510134427584747136572748918146701035397332792"
    "304213963271354040790246325556461517071185888666743940446059781681939593390610457300000410"
    "827430924103528812599832038053657245435064880839104970219857874056331538133109738980029096"
    "93370594694452375893009888170063327154053821159418458108802951755933463573455810546875",
    761,
    "1.0000000000132873108058798218075466365858866796204316120387346995461095826861753841161935"
    "247836939689566881401375540716352977559252087422693381464203581785118767706512437906713702"
    "693003503091646357646071476452635669415524684862150549447265950701439067752033971016791037"
    "886916527448509507027524803727795339424891843054498212975998837171800278451594248186507426"
    "648281555498412610248716893168741033011563160921744542987825450117173046307626801641301972"
    "775101344275847471365727489181467010353973327923042139632713540407902463255564615170711858"
    "886667439404460597816819395933906104573000004108274309241035288125998320380536572454350648"
    "808391049702198578740563315381331097389800290969337059469445237589300988817006332715405382"
    "1159418458108802951755933463573455810546875e-313"
  },
  {
    1e-312,
    1074,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000009999999999984653414306424254822495727998400384"
    "494798179603049566133420122111551188980872622277349738658390603661601746944343843932809425"
    "680274682264662152679964471949000016499745599582144737901207291376845346020078598425065645"
    "235547531043204631943751558291951834840153344907012832890084789653234050444031427324837024"
    "042101107905649692216696974146511587715789684961217254373697248854313571918308886594163564"
    "317398627121032083125239736043336600860914827059738462139428152502738081500205011373256298"
    "069181549942053604151421452384269985617566294317171084910720379669920191982813295182567868"
    "591765894923254035012310969997392122823095038574513728253432007519784245448952372271615847"
    "64505149963529329106606264592722000702809908960488893825413470040075480937957763671875",
    761,
    "9.9999999999846534143064242548224957279984003844947981796030495661334201221115511889808726"
    "222773497386583906036616017469443438439328094256802746822646621526799644719490000164997455"
    "995821447379012072913768453460200785984250656452355475310432046319437515582919518348401533"
    "449070128328900847896532340504440314273248370240421011079056496922166969741465115877157896"
    "849612172543736972488543135719183088865941635643173986271210320831252397360433366008609148"
    "270597384621394281525027380815002050113732562980691815499420536041514214523842699856175662"
    "943171710849107203796699201919828132951825678685917658949232540350123109699973921228230950"
    "385745137282534320075197842454489523722716158476450514996352932910660626459272200070280990"
    "8960488893825413470040075480937957763671875e-313"
  },
  {
    1e-311,
    1074,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000099999999999994753836816616511477927917844470256"
    "657499736814793090661876925865714541989788848333087337492396834329956438852072154647372253"
    "815853604585541159238447954053438055211413089242805732757223268463084566816376801350275669"
    "270477226654792388932831776661537530352417991079642944520270134403910181791622270815376270"
    "877122947149684401750551218912852475949533812932510787775885488405715316005812990910633000"
    "601130631914738865300203932016831049050206218638990435102324738252131914968855456332634605"
    "664773582192022845265256327023045041269955535521979168657081233848677661751424229643114925"
    "600202295344574441109119110513101559758504155561673256479466808241373590371668740244833598"
    "14252296616287720889555263035131277865827392238584625516750747920013964176177978515625",
    762,
    "9.9999999999994753836816616511477927917844470256657499736814793090661876925865714541989788"
    "848333087337492396834329956438852072154647372253815853604585541159238447954053438055211413"
    "089242805732757223268463084566816376801350275669270477226654792388932831776661537530352417"
    "991079642944520270134403910181791622270815376270877122947149684401750551218912852475949533"
    "812932510787775885488405715316005812990910633000601130631914738865300203932016831049050206"
    "218638990435102324738252131914968855456332634605664773582192022845265256327023045041269955"
    "535521979168657081233848677661751424229643114925600202295344574441109119110513101559758504"
    "155561673256479466808241373590371668740244833598142522966162877208895552630351312778658273"
    "92238584625516750747920013964176177978515625e-312"
  },
  {
    1e-310,
    1074,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000999999999999996944932750289769196936057731524703"
    "811503348409363383061327826907212970624909358517403358560131942537544044990266118246815203"
    "829571639835051469864080618722193182185449923568510112153949469529056037035631619202629765"
    "009772905288839245826756497862715030841141424799600983742508756662958377033325374009431385"
    "589933427527906510700039348277770292425883783765220625771825884011346358999037950762662458"
    "487021106048861331017974844029465503982414835737106288925928453596236183010823010859703264"
    "972055182844057387236243871290565474397652886003214459754104817536601864822006055703628002"
    "562870735483089108086527575310742374760892812222244556109716993428246282344069074957894355"
    "32256565156542018836716836053949868937516689930411484255046161706559360027313232421875",
    763,
    "9.9999999999999694493275028976919693605773152470381150334840936338306132782690721297062490"
    "935851740335856013194253754404499026611824681520382957163983505146986408061872219318218544"
    "992356851011215394946952905603703563161920262976500977290528883924582675649786271503084114"
    "142479960098374250875666295837703332537400943138558993342752790651070003934827777029242588"
    "378376522062577182588401134635899903795076266245848702110604886133101797484402946550398241"
    "483573710628892592845359623618301082301085970326497205518284405738723624387129056547439765"
    "288600321445975410481753660186482200605570362800256287073548308910808652757531074237476089"
    "281222224455610971699342824628234406907495789435532256565156542018836716836053949868937516"
    "689930411484255046161706559360027313232421875e-311"
  },
  {
    1e-309,
    1074,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000010000000000000018855892087022346387017456602069175"
    "351539464355066307055836837322197256976114460360563569237483024613420106372205754241244703"
    "966751992330154576120407265409744451925818266825553906121211480188770739228179797726170722"
    "402729691629307814766003709874490035728375761999181375964894979253440329450356405949982532"
    "718038231310127600194920641926948457189383492092319005731229840067656788931287549282957037"
    "345925847390085988195683964155810053304501006718264827161965607037278863430498556130389858"
    "044871189364402806946119313965769805674626390815567370724340654415843895527824316308758772"
    "189555136868235777860612223287150524784779378827957552412218845296973202068072422088501927"
    "12299250559084998308332566242135779654409666848680071638000299572013318538665771484375",
    765,
    "1.0000000000000018855892087022346387017456602069175351539464355066307055836837322197256976"
    "114460360563569237483024613420106372205754241244703966751992330154576120407265409744451925"
    "818266825553906121211480188770739228179797726170722402729691629307814766003709874490035728"
    "375761999181375964894979253440329450356405949982532718038231310127600194920641926948457189"
    "383492092319005731229840067656788931287549282957037345925847390085988195683964155810053304"
    "501006718264827161965607037278863430498556130389858044871189364402806946119313965769805674"
    "626390815567370724340654415843895527824316308758772189555136868235777860612223287150524784"
    "779378827957552412218845296973202068072422088501927122992505590849983083325662421357796544"
    "09666848680071638000299572013318538665771484375e-309"
  },
  {
    1e-308,
    1073,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000099999999999999990932662533724846199547048873403204"
    "569370722504933164788134100221702366853061102859515757830175849182282437843879255320076376"
    "983377547382986251285668341346193998972906543693727922885247662294865916794343554462214934"
    "807294361329416721666282173755541448015911563979127605489720142038977058035153396077150619"
    "905566488977026029171097782672502440171652303162739065260414400859795093549243326204240563"
    "556399326294969169893097546113480479123599469793840520008931786073120501015911771170469747"
    "151434449948712331126470735417237809953873785021982614510236627959137966047188125997672735"
    "652160240532978990624776352152598139144388761857527558861992808911690506171197530846785775"
    "6405810961619074331866883961080943542712559830853980002984826569445431232452392578125",
    764,
    "9.9999999999999990932662533724846199547048873403204569370722504933164788134100221702366853"
    "061102859515757830175849182282437843879255320076376983377547382986251285668341346193998972"
    "906543693727922885247662294865916794343554462214934807294361329416721666282173755541448015"
    "911563979127605489720142038977058035153396077150619905566488977026029171097782672502440171"
    "652303162739065260414400859795093549243326204240563556399326294969169893097546113480479123"
    "599469793840520008931786073120501015911771170469747151434449948712331126470735417237809953"
    "873785021982614510236627959137966047188125997672735652160240532978990624776352152598139144"
    "388761857527558861992808911690506171197530846785775640581096161907433186688396108094354271"
    "2559830853980002984826569445431232452392578125e-309"
  },
  {
    1e-307,
    1072,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000999999999999999909326625337248461995470488734032045"
    "693707225049331647881341002217023668530611028595157578301758491822824378438792553200763769"
    "833775473829862512856683413461939989729065436937279228852476622948659167943435544622149348"
    "072943613294167216662821737555414480159115639791276054897201420389770580351533960771506199"
    "055664889770260291710977826725024401716523031627390652604144008597950935492433262042405635"
    "563993262949691698930975461134804791235994697938405200089317860731205010159117711704697471"
    "514344499487123311264707354172378099538737850219826145102366279591379660471881259976727356"
    "521602405329789906247763521525981391443887618575275588619928089116905061711975308467857756"
    "405810961619074331866883961080943542712559830853980002984826569445431232452392578125",
    764,
    "9.9999999999999990932662533724846199547048873403204569370722504933164788134100221702366853"
    "061102859515757830175849182282437843879255320076376983377547382986251285668341346193998972"
    "906543693727922885247662294865916794343554462214934807294361329416721666282173755541448015"
    "911563979127605489720142038977058035153396077150619905566488977026029171097782672502440171"
    "652303162739065260414400859795093549243326204240563556399326294969169893097546113480479123"
    "599469793840520008931786073120501015911771170469747151434449948712331126470735417237809953"
    "873785021982614510236627959137966047188125997672735652160240532978990624776352152598139144"
    "388761857527558861992808911690506171197530846785775640581096161907433186688396108094354271"
    "2559830853980002984826569445431232452392578125e-308"
  },
  {
    1e-306,
    1069,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000010000000000000000279023803391476325978469990224051750"
    "613215776767695913434815660171857902754611290428295390285511299939755539656995254561861674"
    "442608993809982188077260011126903019002311116743659118485969067043640532359081983018447216"
    "049451462723640722590746925490298257198232733988877473927392106870263222325803588251110234"
    "205543842448102753778430086832136807498326022836612478352744084880146129506125620176035215"
    "057087515132261261692207184015768235888410563716898510557524313110058901325619857847547714"
    "927109657043127542655407967166542476141719241000408007422682293109602540105142822306763482"
    "676370822194171790365710499573256566659306344285043677760454755517299704176913224907978537"
    "594173374670297704548248979442337094143862519235455010857549495995044708251953125",
    763,
    "1.0000000000000000279023803391476325978469990224051750613215776767695913434815660171857902"
    "754611290428295390285511299939755539656995254561861674442608993809982188077260011126903019"
    "002311116743659118485969067043640532359081983018447216049451462723640722590746925490298257"
    "198232733988877473927392106870263222325803588251110234205543842448102753778430086832136807"
    "498326022836612478352744084880146129506125620176035215057087515132261261692207184015768235"
    "888410563716898510557524313110058901325619857847547714927109657043127542655407967166542476"
    "141719241000408007422682293109602540105142822306763482676370822194171790365710499573256566"
    "659306344285043677760454755517299704176913224907978537594173374670297704548248979442337094"
    "143862519235455010857549495995044708251953125e-306"
  },
  {
    1e-305,
    1066,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000099999999999999999628217900530785377054659627883900722"
    "995775030945280642024408233714255781016776892345034950140642648166857382519099952140686141"
    "479811923402869722078131107249021865458693174447612971629816436950941715791549065392595532"
    "974473747817824410007390455073240023696790443685796272726246660775812439763465267748307025"
    "658385238493027973334562682769653967428338344198908910697296851733096562843141535755075192"
    "125128789628361223902198313043761496141536078948061079803654582305898880631017936340615816"
    "514657471306223613198521203818060812738950439865022590496104271907209573354547629628995046"
    "869450171551657295955570904230051927338583020097669333441417250244332809616705208468049446"
    "512230390757498683137398144735633162723253963832803492550738155841827392578125",
    760,
    "9.9999999999999999628217900530785377054659627883900722995775030945280642024408233714255781"
    "016776892345034950140642648166857382519099952140686141479811923402869722078131107249021865"
    "458693174447612971629816436950941715791549065392595532974473747817824410007390455073240023"
    "696790443685796272726246660775812439763465267748307025658385238493027973334562682769653967"
    "428338344198908910697296851733096562843141535755075192125128789628361223902198313043761496"
    "141536078948061079803654582305898880631017936340615816514657471306223613198521203818060812"
    "738950439865022590496104271907209573354547629628995046869450171551657295955570904230051927"
    "338583020097669333441417250244332809616705208468049446512230390757498683137398144735633162"
    "723253963832803492550738155841827392578125e-306"
  },
  {
    1e-304,
    1062,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000999999999999999970986017938236030708706274083986072964"
    "866688415599378481654095393107971837933080827946997879690663671822989712784178773583416591"
    "227895116453079951225559040330152058071587800564304010459170501625708295521324535590942309"
    "584410442829581572873274152094459850683609267234134994881071344343566129678087759380707725"
    "407426897026283217747721343282826813844008368651362124331816540784047770686498029508536253"
    "685317002729601094862994913526248059474804323713218591863837432701467745308734226327267503"
    "095061920061821625140744295401696537963568637537766035115293445561569111970431548780832229"
    "540437312396088097943436310280409359349986818954553781109130063014014238947635759786607029"
    "68627706001115993261324824985442939750956981015406199730932712554931640625",
    757,
    "9.9999999999999997098601793823603070870627408398607296486668841559937848165409539310797183"
    "793308082794699787969066367182298971278417877358341659122789511645307995122555904033015205"
    "807158780056430401045917050162570829552132453559094230958441044282958157287327415209445985"
    "068360926723413499488107134434356612967808775938070772540742689702628321774772134328282681"
    "384400836865136212433181654078404777068649802950853625368531700272960109486299491352624805"
    "947480432371321859186383743270146774530873422632726750309506192006182162514074429540169653"
    "796356863753776603511529344556156911197043154878083222954043731239608809794343631028040935"
    "934998681895455378110913006301401423894763575978660702968627706001115993261324824985442939"
    "750956981015406199730932712554931640625e-305"
  },
  {
    1e-303,
    1059,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000009999999999999999305121602309211138097617585722213781407"
    "209893854338937799101162826526342823575798751416352849454431760700551329332655770659048735"
    "155365283320923199363557888740455036470374903053828811167803130117741156906587462549214773"
    "278871862717215293522655142737552326287369958360106230708389228802729009475838904169276755"
    "251461163798887927910725682208862371410082513109989521059733783089791982946303046409911855"
    "797635730431832642086137664680610163699139784853910619875040081294340477064220070010424438"
    "126414512611584141895959069554379948820714197578302433620946079447265174503599527662430468"
    "939342674033123193637999390482334968926374077191304958210621599271120673965696399563871329"
    "88634103909036894596075133851385829952818085075705312192440032958984375",
    755,
    "9.9999999999999993051216023092111380976175857222137814072098938543389377991011628265263428"
    "235757987514163528494544317607005513293326557706590487351553652833209231993635578887404550"
    "364703749030538288111678031301177411569065874625492147732788718627172152935226551427375523"
    "262873699583601062307083892288027290094758389041692767552514611637988879279107256822088623"
    "714100825131099895210597337830897919829463030464099118557976357304318326420861376646806101"
    "636991397848539106198750400812943404770642200700104244381264145126115841418959590695543799"
    "488207141975783024336209460794472651745035995276624304689393426740331231936379993904823349"
    "689263740771913049582106215992711206739656963995638713298863410390903689459607513385138582"
    "9952818085075705312192440032958984375e-304"
  },
  {
    1e-302,
    1054,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000099999999999999996289124639677304732891737098163313400003"
    "754860956628154130529957101690432681798063738592536074161957267240279681399613427991424768"
    "542339882888242496771839003893074718667773851251978459069246390292145955519137772373814313"
    "310579151800956416907242453031892707263481295451012051902486005090748393198698558795171543"
    "097074089700433275639158827043869850340834518328948988664790828903405620812448453502724006"
    "420631679231752873211868411461065085382625466765308588857074778706100578827178246202249123"
    "857782630168898295051461771244482934726919398177887676465367803820059306641722957791439301"
    "113670339753294222750903603397418685851693670746912405151648239663380463742253582056305034"
    "674846879073532500981362665382068399821946513839066028594970703125",
    751,
    "9.9999999999999996289124639677304732891737098163313400003754860956628154130529957101690432"
    "681798063738592536074161957267240279681399613427991424768542339882888242496771839003893074"
    "718667773851251978459069246390292145955519137772373814313310579151800956416907242453031892"
    "707263481295451012051902486005090748393198698558795171543097074089700433275639158827043869"
    "850340834518328948988664790828903405620812448453502724006420631679231752873211868411461065"
    "085382625466765308588857074778706100578827178246202249123857782630168898295051461771244482"
    "934726919398177887676465367803820059306641722957791439301113670339753294222750903603397418"
    "685851693670746912405151648239663380463742253582056305034674846879073532500981362665382068"
    "399821946513839066028594970703125e-303"
  },
  {
    1e-301,
    1052,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000001000000000000000066504322127499234590215330691750752749850"
    "538126789922377769886093782568469091263076567653603289384041799915321232333917364744245029"
    "061384418610761068078713766563526513526532775357875707211346754592959921695798423951473709"
    "805328306131275582854537351322749293107827733708512353219858996938149482076890135228643129"
    "609539351774060645412452429006574863088645574619210784806404225209601531305860195942614414"
    "423096789547175207334420583569481202345538450891562371984314691467271650191063937158643001"
    "574226431386802985454492134866699635902071498414503652842702337317635037800515375262700586"
    "184498579038935391378146388344394749331429470152734388970314299103363808151802585925985892"
    "7144364121703023337768036216122169435038813389837741851806640625",
    751,
    "1.0000000000000000665043221274992345902153306917507527498505381267899223777698860937825684"
    "690912630765676536032893840417999153212323339173647442450290613844186107610680787137665635"
    "265135265327753578757072113467545929599216957984239514737098053283061312755828545373513227"
    "492931078277337085123532198589969381494820768901352286431296095393517740606454124524290065"
    "748630886455746192107848064042252096015313058601959426144144230967895471752073344205835694"
    "812023455384508915623719843146914672716501910639371586430015742264313868029854544921348666"
    "996359020714984145036528427023373176350378005153752627005861844985790389353913781463883443"
    "947493314294701527343889703142991033638081518025859259858927144364121703023337768036216122"
    "169435038813389837741851806640625e-301"
  },
  {
    1e-300,
    1049,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000010000000000000000250590918352087596856961468077037052499253"
    "423199004660431840514846763028121819501008949623062702782541489103114649988041308122460916"
    "061901827194266279345842755104147827870150702226392606037936139243597750940301438661414791"
    "255135908825910173416922229212204049186218220291556195418594185258832620409283163178720501"
    "540199698661694898041067655794243192165254180873224255430058507393834020333099315764646743"
    "363847906553166172481259959859490629378249375961717786188879297047653054233513471041822963"
    "756663795076749714785423658979515204489204917602528975670926176708182492472010563233775561"
    "653805064365381258305022465963115930056323650792902539887815381155401398600958797808116743"
    "2804936359631140419153283449560376539011485874652862548828125",
    749,
    "1.0000000000000000250590918352087596856961468077037052499253423199004660431840514846763028"
    "121819501008949623062702782541489103114649988041308122460916061901827194266279345842755104"
    "147827870150702226392606037936139243597750940301438661414791255135908825910173416922229212"
    "204049186218220291556195418594185258832620409283163178720501540199698661694898041067655794"
    "243192165254180873224255430058507393834020333099315764646743363847906553166172481259959859"
    "490629378249375961717786188879297047653054233513471041822963756663795076749714785423658979"
    "515204489204917602528975670926176708182492472010563233775561653805064365381258305022465963"
    "115930056323650792902539887815381155401398600958797808116743280493635963114041915328344956"
    "0376539011485874652862548828125e-300"
  },
  {
    1e-299,
    1046,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000099999999999999999190290760137637976208079970046606724998518"
    "567438890097551538379739129028665449972035680926865499362402810630365113071354366664694164"
    "203479400635907581928068266792539819540090611445010331775110138947965781261551979787569458"
    "166181868364336493141612019999729436725709268567023259945975579607028601215886118925518658"
    "960446433985656531743023483770388411882929286181173813228715116320889861526972008354488226"
    "701519154182974517909032591912335141165412695985930392654652029476022960918127506061373221"
    "681833800437256029778255072295302808639968643685229334660484195336481840454960117191913215"
    "008604835462031339238693319784506794499468102053494600355532932528120522673051486467229961"
    "8939724737118660523316204802313222899101674556732177734375",
    746,
    "9.9999999999999999190290760137637976208079970046606724998518567438890097551538379739129028"
    "665449972035680926865499362402810630365113071354366664694164203479400635907581928068266792"
    "539819540090611445010331775110138947965781261551979787569458166181868364336493141612019999"
    "729436725709268567023259945975579607028601215886118925518658960446433985656531743023483770"
    "388411882929286181173813228715116320889861526972008354488226701519154182974517909032591912"
    "335141165412695985930392654652029476022960918127506061373221681833800437256029778255072295"
    "302808639968643685229334660484195336481840454960117191913215008604835462031339238693319784"
    "506794499468102053494600355532932528120522673051486467229961893972473711866052331620480231"
    "3222899101674556732177734375e-300"
  },
  {
    1e-298,
    1042,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000999999999999999912328065440178667945403966643095736050128809"
    "725161144813110581347907260225388618807065241978378310511738176684897847296134517208981728"
    "061861094996950742552059845950875175526912254796125831249071305767376337220422034037811676"
    "417565406168999146753473669061829043981742261305303937700565244519143543112168880574714035"
    "007251076705546549406561057574839884358592320586088346562272180390090411973212500537381300"
    "528153669461252213404717758741643748844181432709364664927137710748047655173102156049178235"
    "583038396446773463958994302956646416349753660690843217434180231476584382186108808418914513"
    "379348958017563540910181041525407799464239279522206839012428228659762106653639065937800317"
    "076591475056075719596293538415920920670032501220703125",
    743,
    "9.9999999999999991232806544017866794540396664309573605012880972516114481311058134790726022"
    "538861880706524197837831051173817668489784729613451720898172806186109499695074255205984595"
    "087517552691225479612583124907130576737633722042203403781167641756540616899914675347366906"
    "182904398174226130530393770056524451914354311216888057471403500725107670554654940656105757"
    "483988435859232058608834656227218039009041197321250053738130052815366946125221340471775874"
    "164374884418143270936466492713771074804765517310215604917823558303839644677346395899430295"
    "664641634975366069084321743418023147658438218610880841891451337934895801756354091018104152"
    "540779946423927952220683901242822865976210665363906593780031707659147505607571959629353841"
    "5920920670032501220703125e-299"
  },
  {
    1e-297,
    1039,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000010000000000000000396478128980950068520868995348882659698990112"
    "439255546729582652670817083234140282683317496428210034914020640749031007639891563097175904"
    "185537531763508653178563611101120073253024302424898096523194397070266978525784561784243248"
    "083706501279844022137081185585735612223029402891897965152701270009714935868765744634701223"
    "627922977471765782444391057813106595117131865471280037220785529001835372476246333493828469"
    "074142652508409585016908153523760093400942761492674835181498451675387815861788033524646055"
    "595177691280323980766845749508570884296461025491634241072389864977588179676965900192627321"
    "100679925819633032729844916368640323129460651425895022810699832540710987766403439129992000"
    "576046943562114055481515606516040861606597900390625",
    742,
    "1.0000000000000000396478128980950068520868995348882659698990112439255546729582652670817083"
    "234140282683317496428210034914020640749031007639891563097175904185537531763508653178563611"
    "101120073253024302424898096523194397070266978525784561784243248083706501279844022137081185"
    "585735612223029402891897965152701270009714935868765744634701223627922977471765782444391057"
    "813106595117131865471280037220785529001835372476246333493828469074142652508409585016908153"
    "523760093400942761492674835181498451675387815861788033524646055595177691280323980766845749"
    "508570884296461025491634241072389864977588179676965900192627321100679925819633032729844916"
    "368640323129460651425895022810699832540710987766403439129992000576046943562114055481515606"
    "516040861606597900390625e-297"
  },
  {
    1e-296,
    1035,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000100000000000000000569588024265064981030478409707692465796029083"
    "892171204366554955530185549727391907866068093230295203015836077090169983922791921618802712"
    "343571099517749924697729040098218551239838345679274874478660398978660168400341027426095190"
    "415591840558833409097893202610835662482009256015356749801549167248404012695452275980183240"
    "131463880274190388767162625958511947088095562418409514613018689749203717444806459951576787"
    "294477304028395980916466692284740654118418456529339856054660932234114787602503073825490689"
    "912326974636334897863383575240090920834145138694470232775665182544563509260651492583654044"
    "854291669812336664290357160714237021995758897715412540943218202892203421050666978627949792"
    "93345025428418892943227547220885753631591796875",
    739,
    "1.0000000000000000056958802426506498103047840970769246579602908389217120436655495553018554"
    "972739190786606809323029520301583607709016998392279192161880271234357109951774992469772904"
    "009821855123983834567927487447866039897866016840034102742609519041559184055883340909789320"
    "261083566248200925601535674980154916724840401269545227598018324013146388027419038876716262"
    "595851194708809556241840951461301868974920371744480645995157678729447730402839598091646669"
    "228474065411841845652933985605466093223411478760250307382549068991232697463633489786338357"
    "524009092083414513869447023277566518254456350926065149258365404485429166981233666429035716"
    "071423702199575889771541254094321820289220342105066697862794979293345025428418892943227547"
    "220885753631591796875e-296"
  },
  {
    1e-295,
    1032,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000001000000000000000060018972491361621077156168797575070757062243486"
    "927860250533894694149620019098093782134390869131834368148286057303941318845898565835328395"
    "624578485054884960383803535589900413044858313908046196839141137370755553723483720922348550"
    "899489161422043087345630478052683980792648926611533925622908198063965662829805485671096339"
    "678893113837382858499593494345983536212525100894348867647572501798437291530574599303094328"
    "095960577175157717206504410093171019440331099651934492711786674657361812271066920990424755"
    "754468757033827535515018469930795962428893246494657174928387301146727692750635075318447106"
    "983038112267265251033043654697029568739150841850728404052663989160537516320548389031021334"
    "56680944423311530044884420931339263916015625",
    737,
    "1.0000000000000000600189724913616210771561687975750707570622434869278602505338946941496200"
    "190980937821343908691318343681482860573039413188458985658353283956245784850548849603838035"
    "355899004130448583139080461968391411373707555537234837209223485508994891614220430873456304"
    "780526839807926489266115339256229081980639656628298054856710963396788931138373828584995934"
    "943459835362125251008943488676475725017984372915305745993030943280959605771751577172065044"
    "100931710194403310996519344927117866746573618122710669209904247557544687570338275355150184"
    "699307959624288932464946571749283873011467276927506350753184471069830381122672652510330436"
    "546970295687391508418507284040526639891605375163205483890310213345668094442331153004488442"
    "0931339263916015625e-295"
  },
  {
    1e-294,
    1029,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000010000000000000000165604986923928440636750610371765538777806813685"
    "229416850392185830714084016387540193554229196687284977563458281821481351515150861174873778"
    "734844931529763896585930279037284925276784282158082351971114193034324579474249635932312335"
    "046325567550758902522717164972220960146038334451607835369749776000252341295793049756851889"
    "874896649609996818372197065372922839472695195261458904336640183533171978645665994732331639"
    "750105476621993907730344202965594368354138721651057469796447928043906632742379748020104704"
    "495095484974446900100722959068865591589397588546932971909989205858536126353389557329217802"
    "309409809521463645294660166533020897139013500934460083562784209697348716694455068298026103"
    "80963923120134495547972619533538818359375",
    735,
    "1.0000000000000000165604986923928440636750610371765538777806813685229416850392185830714084"
    "016387540193554229196687284977563458281821481351515150861174873778734844931529763896585930"
    "279037284925276784282158082351971114193034324579474249635932312335046325567550758902522717"
    "164972220960146038334451607835369749776000252341295793049756851889874896649609996818372197"
    "065372922839472695195261458904336640183533171978645665994732331639750105476621993907730344"
    "202965594368354138721651057469796447928043906632742379748020104704495095484974446900100722"
    "959068865591589397588546932971909989205858536126353389557329217802309409809521463645294660"
    "166533020897139013500934460083562784209697348716694455068298026103809639231201344955479726"
    "19533538818359375e-294"
  },
  {
    1e-293,
    1022,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000100000000000000005132727773156786567445994724549536738120593106324"
    "687653743495947193397769560622582957859727923921319406989801147958268210702186989176019207"
    "435968667450324623876143405266602894142233676959860451073519375729093456827196945652508742"
    "051784048864964792695872574159160383703990797825929720572155397117757708976024953201410954"
    "061242406210622316711873678424528575947398462070827220479080510941327279737299933712209527"
    "177057127256605191981041213384870291934765415456874356535829828676758247170113175274189869"
    "347691532655096641402923512601408177490254896666439938090962503455287672757585140134204163"
    "261868600424147373232812708828407293410094349927192491338687552237698739032781259077758972"
    "9640340010519139468669891357421875",
    729,
    "1.0000000000000000513272777315678656744599472454953673812059310632468765374349594719339776"
    "956062258295785972792392131940698980114795826821070218698917601920743596866745032462387614"
    "340526660289414223367695986045107351937572909345682719694565250874205178404886496479269587"
    "257415916038370399079782592972057215539711775770897602495320141095406124240621062231671187"
    "367842452857594739846207082722047908051094132727973729993371220952717705712725660519198104"
    "121338487029193476541545687435653582982867675824717011317527418986934769153265509664140292"
    "351260140817749025489666643993809096250345528767275758514013420416326186860042414737323281"
    "270882840729341009434992719249133868755223769873903278125907775897296403400105191394686698"
    "91357421875e-293"
  },
  {
    1e-292,
    1021,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000001000000000000000051327277731567865674459947245495367381205931063246"
    "876537434959471933977695606225829578597279239213194069898011479582682107021869891760192074"
    "359686674503246238761434052666028941422336769598604510735193757290934568271969456525087420"
    "517840488649647926958725741591603837039907978259297205721553971177577089760249532014109540"
    "612424062106223167118736784245285759473984620708272204790805109413272797372999337122095271"
    "770571272566051919810412133848702919347654154568743565358298286767582471701131752741898693"
    "476915326550966414029235126014081774902548966664399380909625034552876727575851401342041632"
    "618686004241473732328127088284072934100943499271924913386875522376987390327812590777589729"
    "640340010519139468669891357421875",
    729,
    "1.0000000000000000513272777315678656744599472454953673812059310632468765374349594719339776"
    "956062258295785972792392131940698980114795826821070218698917601920743596866745032462387614"
    "340526660289414223367695986045107351937572909345682719694565250874205178404886496479269587"
    "257415916038370399079782592972057215539711775770897602495320141095406124240621062231671187"
    "367842452857594739846207082722047908051094132727973729993371220952717705712725660519198104"
    "121338487029193476541545687435653582982867675824717011317527418986934769153265509664140292"
    "351260140817749025489666643993809096250345528767275758514013420416326186860042414737323281"
    "270882840729341009434992719249133868755223769873903278125907775897296403400105191394686698"
    "91357421875e-292"
  },
  {
    1e-291,
    1019,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000009999999999999999623243233912798103508506385521992048124372918447536"
    "033153018627964458003030494979954072709187387723715072044222381502419009245034296217877201"
    "191912593944933935303143113859357222379308718952590678583311554132344189036344464928213958"
    "515141307008282797599820760056638116035571735271022137303184610275791116970314678120729246"
    "181607632734773625772193520456011202305539786285748707062310138073209693886156855664311520"
    "649108300273993840638730303881817444771722615434723059317242518826693261954499588694423889"
    "204562440388988198994707250476238780378062800183777747382216458827606514493984901861724443"
    "237610708779941730011243747301958903899843803575785271892318676131711448691098426816425970"
    "2871277113445103168487548828125",
    727,
    "9.9999999999999996232432339127981035085063855219920481243729184475360331530186279644580030"
    "304949799540727091873877237150720442223815024190092450342962178772011919125939449339353031"
    "431138593572223793087189525906785833115541323441890363444649282139585151413070082827975998"
    "207600566381160355717352710221373031846102757911169703146781207292461816076327347736257721"
    "935204560112023055397862857487070623101380732096938861568556643115206491083002739938406387"
    "303038818174447717226154347230593172425188266932619544995886944238892045624403889881989947"
    "072504762387803780628001837777473822164588276065144939849018617244432376107087799417300112"
    "437473019589038998438035757852718923186761317114486910984268164259702871277113445103168487"
    "548828125e-292"
  },
  {
    1e-290,
    1015,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000100000000000000006912786859962547673918180898415459989495965890694553"
    "118186157880703161317411757139641286255133930135858243672932786917014824134318418787294520"
    "778575752499680780765800092204758525921794913927359931056627766647459814563645853154062545"
    "110576023941185639847447470879184212717813920573620411980107320757668537289314485451686381"
    "127672187277232802704027068522268732267074912421167160771992853446316296987606743322809571"
    "170336107378242695971995454080715432175053317379781724361309374456510080226811151638995438"
    "820714305337993285518800620737335427549750399360370214390571228689994280114198357321547027"
    "767099091416964419352763099484834284313532305479419062640425332975063941955314039677915616"
    "266545839607715606689453125",
    725,
    "1.0000000000000000691278685996254767391818089841545998949596589069455311818615788070316131"
    "741175713964128625513393013585824367293278691701482413431841878729452077857575249968078076"
    "580009220475852592179491392735993105662776664745981456364585315406254511057602394118563984"
    "744747087918421271781392057362041198010732075766853728931448545168638112767218727723280270"
    "402706852226873226707491242116716077199285344631629698760674332280957117033610737824269597"
    "199545408071543217505331737978172436130937445651008022681115163899543882071430533799328551"
    "880062073733542754975039936037021439057122868999428011419835732154702776709909141696441935"
    "276309948483428431353230547941906264042533297506394195531403967791561626654583960771560668"
    "9453125e-290"
  },
  {
    1e-289,
    1012,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000001000000000000000012165977821841121332071851420445055850947729807109836"
    "319696396934719179642881265582543213680619019232142312832213352408416339028648419294158493"
    "868691855394986859741366502787924981198174609132515869374212464746502549902052110890369664"
    "656891152167282191278528733790225847913624177131409245410346711577979412433583765213429574"
    "948210619815013120469114077424518206875138193205377793592507346653993059870530437603059100"
    "080677849044804081934928326073602404642121637624211210605711418220687678631763438017919472"
    "073330245656672612138789588840300282062184540149874194207543538025654080212120433459189768"
    "919033561542726224245894320367034868121486949612503459912314280908242325983381615372991291"
    "224025189876556396484375",
    723,
    "1.0000000000000000121659778218411213320718514204450558509477298071098363196963969347191796"
    "428812655825432136806190192321423128322133524084163390286484192941584938686918553949868597"
    "413665027879249811981746091325158693742124647465025499020521108903696646568911521672821912"
    "785287337902258479136241771314092454103467115779794124335837652134295749482106198150131204"
    "691140774245182068751381932053777935925073466539930598705304376030591000806778490448040819"
    "349283260736024046421216376242112106057114182206876786317634380179194720733302456566726121"
    "387895888403002820621845401498741942075435380256540802121204334591897689190335615427262242"
    "458943203670348681214869496125034599123142809082423259833816153729912912240251898765563964"
    "84375e-289"
  },
  {
    1e-288,
    1009,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000010000000000000000577354904440686056577598174714126910861572730869783922"
    "094285424325691264678703102336389327771952449332944119499049658178018608802770341571878650"
    "023443910764436180746740381956532036139942332453826223278646261289790264895772474105742938"
    "159864219629415570352855137915188713252362000152451449229279083769441808012326366561769640"
    "110196221808650457260393636630534995116269380104128448944442969013289878749600341030883893"
    "788244288349023841629492978604439383288508665630960370116172792962181775408419007155474049"
    "803804918352808065781628836667434768104401029129365539660785371250850569560109452642141759"
    "205994436442605996712836599520812481325558337578531931058655199821600008391886404979231883"
    "771717548370361328125",
    721,
    "1.0000000000000000577354904440686056577598174714126910861572730869783922094285424325691264"
    "678703102336389327771952449332944119499049658178018608802770341571878650023443910764436180"
    "746740381956532036139942332453826223278646261289790264895772474105742938159864219629415570"
    "352855137915188713252362000152451449229279083769441808012326366561769640110196221808650457"
    "260393636630534995116269380104128448944442969013289878749600341030883893788244288349023841"
    "629492978604439383288508665630960370116172792962181775408419007155474049803804918352808065"
    "781628836667434768104401029129365539660785371250850569560109452642141759205994436442605996"
    "712836599520812481325558337578531931058655199821600008391886404979231883771717548370361328"
    "125e-288"
  },
  {
    1e-287,
    1006,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000100000000000000002127988034628661819720944463063858289798963846308354749"
    "764282603428916900787907451276235749993426437237273265575167509029344339897414226676436809"
    "542236253127821140802800986947062568133853395508921996494289702299784521955713819441059048"
    "871020612641406442988008979048445259594658170817642531286295093777236610711353950197905276"
    "077242028818350552049913467222526540243594216638480385289473670346024547141635690306495794"
    "030716500282374238053252043097071137946748341198817588689259043579377841357913055744505865"
    "474029489239425102666424780558892101183565270248666615925053784554027556089853582019465031"
    "934673796303309933097218828404414412370072644157340655102452872302586095454302039797767065"
    "465450286865234375",
    719,
    "1.0000000000000000212798803462866181972094446306385828979896384630835474976428260342891690"
    "078790745127623574999342643723727326557516750902934433989741422667643680954223625312782114"
    "080280098694706256813385339550892199649428970229978452195571381944105904887102061264140644"
    "298800897904844525959465817081764253128629509377723661071135395019790527607724202881835055"
    "204991346722252654024359421663848038528947367034602454714163569030649579403071650028237423"
    "805325204309707113794674834119881758868925904357937784135791305574450586547402948923942510"
    "266642478055889210118356527024866661592505378455402755608985358201946503193467379630330993"
    "309721882840441441237007264415734065510245287230258609545430203979776706546545028686523437"
    "5e-287"
  },
  {
    1e-286,
    1003,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000001000000000000000050444368424512208165649742903257869448523746162199423267"
    "071399152913134975872063089463617721743048821110076091074307672300177384016455779103165620"
    "959985367410536741344832530416688027463093387323941855280280307782790235573225567341553150"
    "531178795636058514204428991311987579378276353831401000914916889109817862408817225337381760"
    "970181802328737684931317864887852689788738841607236686134384861755239394251298663083703091"
    "120976068486655806465942374549292938974189932874464786672341524133297715389346683926935715"
    "252452446703495467863156494512565650719212870846576404712937269176100676988463375410270800"
    "348902508015099603221365618473827330784812294597235794897321730333172862259516477934084832"
    "668304443359375",
    717,
    "1.0000000000000000504443684245122081656497429032578694485237461621994232670713991529131349"
    "758720630894636177217430488211100760910743076723001773840164557791031656209599853674105367"
    "413448325304166880274630933873239418552802803077827902355732255673415531505311787956360585"
    "142044289913119875793782763538314010009149168891098178624088172253373817609701818023287376"
    "849313178648878526897887388416072366861343848617552393942512986630837030911209760684866558"
    "064659423745492929389741899328744647866723415241332977153893466839269357152524524467034954"
    "678631564945125656507192128708465764047129372691761006769884633754102708003489025080150996"
    "03221365618473827330784812294597235794897321730333172862259516477934084832668304443359375e"
    "-286"
  },
  {
    1e-285,
    999,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000010000000000000000737759588870926801404019815213532986889510323214921238826"
    "142576478123077502664539508246258991900763800999508393324137379055645720503065889742036413"
    "900836363163970079982906591735379043627409331117193675501869356107462483860954656863232799"
    "879569310136537816639003519740155661236320703553815513564896501797792666450394040240449611"
    "283910136449234164770644190179225196709761817851829527261033883912345325192520710986992117"
    "720249210169865472126799294121581865795551495834959064961423948049131568375195851124373636"
    "621784901508910208222834456514813618260610055345046010828568080847607698604054195827671851"
    "506341440006998210207074860175738964520809770162991899955561361790223884327133418992161750"
    "79345703125",
    714,
    "1.0000000000000000737759588870926801404019815213532986889510323214921238826142576478123077"
    "502664539508246258991900763800999508393324137379055645720503065889742036413900836363163970"
    "079982906591735379043627409331117193675501869356107462483860954656863232799879569310136537"
    "816639003519740155661236320703553815513564896501797792666450394040240449611283910136449234"
    "164770644190179225196709761817851829527261033883912345325192520710986992117720249210169865"
    "472126799294121581865795551495834959064961423948049131568375195851124373636621784901508910"
    "208222834456514813618260610055345046010828568080847607698604054195827671851506341440006998"
    "21020707486017573896452080977016299189995556136179022388432713341899216175079345703125e-28"
    "5"
  },
  {
    1e-284,
    996,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000100000000000000003644541414696392498079839973240061190426737446662380289774"
    "568405597363131123542857264701281527483228571615124211944403293694507119614529318054280870"
    "192640606702058135275765316257810132330485985127534791833633108601662788550362833469107285"
    "711191440950135372874617491477078733106292391701267064997323246784101986708391812538384087"
    "525627553902624600386993240981079185939643750046892617935374577364231129052661827470541873"
    "034675696845736201789984163157379041097080284904611477806100173032845052044294321563472620"
    "661682063505813608768032382921622405510399003381948689098554583090462126529814890677296946"
    "786352642373947254176049794757939138445108514579775783838108682566314655559835955500602722"
    "16796875",
    712,
    "1.0000000000000000364454141469639249807983997324006119042673744666238028977456840559736313"
    "112354285726470128152748322857161512421194440329369450711961452931805428087019264060670205"
    "813527576531625781013233048598512753479183363310860166278855036283346910728571119144095013"
    "537287461749147707873310629239170126706499732324678410198670839181253838408752562755390262"
    "460038699324098107918593964375004689261793537457736423112905266182747054187303467569684573"
    "620178998416315737904109708028490461147780610017303284505204429432156347262066168206350581"
    "360876803238292162240551039900338194868909855458309046212652981489067729694678635264237394"
    "725417604979475793913844510851457977578383810868256631465555983595550060272216796875e-284"
  },
  {
    1e-283,
    993,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000999999999999999946852106770654912597749803438914163621026595614939832534061"
    "107435560807857560967665020741413878246459195032208808316741012258269146158183275756810250"
    "349053468517157403478438736274574028658284026209700801894880226665538684083218690773775743"
    "083874559535526684376149972583318228896972464927356954333829959189227599990751968597152267"
    "732904084873036868203164550342645111605051217155262467154603491420980341585531497120315430"
    "319163251987317550427630958171239606368370686366614654665658351325155359459002663308396313"
    "268813797059212724632831455779893404807152832175212830494516421649864637040699284386851829"
    "214044239034636192287726579592579222139344656594320661160968377600966050522401928901672363"
    "28125",
    709,
    "9.9999999999999994685210677065491259774980343891416362102659561493983253406110743556080785"
    "756096766502074141387824645919503220880831674101225826914615818327575681025034905346851715"
    "740347843873627457402865828402620970080189488022666553868408321869077377574308387455953552"
    "668437614997258331822889697246492735695433382995918922759999075196859715226773290408487303"
    "686820316455034264511160505121715526246715460349142098034158553149712031543031916325198731"
    "755042763095817123960636837068636661465466565835132515535945900266330839631326881379705921"
    "272463283145577989340480715283217521283049451642164986463704069928438685182921404423903463"
    "619228772657959257922213934465659432066116096837760096605052240192890167236328125e-284"
  },
  {
    1e-282,
    986,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000010000000000000000185267526717021225041886804737033222476192186962870088250087"
    "687318910666205005363911217585349955151204119274354572185745520077107861478711995856090116"
    "109355473198965629018102773173958643755446862622184950480409141464100452195464059076134343"
    "063064395081883198721699263332935106297336265956079108453519661106614136652848940265031537"
    "516012481956041767365788379171625098381602438061934369139173171980451007384009191883980703"
    "412382251633531244053994968932802500503164165102147533819330545277914882461551051694602279"
    "472192674583514150708253545289579250446225934906320788873399490536699396466589822957459401"
    "3362998679850527186594367398202895198873704795707040293706313605071045458316802978515625",
    704,
    "1.0000000000000000185267526717021225041886804737033222476192186962870088250087687318910666"
    "205005363911217585349955151204119274354572185745520077107861478711995856090116109355473198"
    "965629018102773173958643755446862622184950480409141464100452195464059076134343063064395081"
    "883198721699263332935106297336265956079108453519661106614136652848940265031537516012481956"
    "041767365788379171625098381602438061934369139173171980451007384009191883980703412382251633"
    "531244053994968932802500503164165102147533819330545277914882461551051694602279472192674583"
    "514150708253545289579250446225934906320788873399490536699396466589822957459401336299867985"
    "0527186594367398202895198873704795707040293706313605071045458316802978515625e-282"
  },
  {
    1e-281,
    985,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000100000000000000001852675267170212250418868047370332224761921869628700882500876"
    "873189106662050053639112175853499551512041192743545721857455200771078614787119958560901161"
    "093554731989656290181027731739586437554468626221849504804091414641004521954640590761343430"
    "630643950818831987216992633329351062973362659560791084535196611066141366528489402650315375"
    "160124819560417673657883791716250983816024380619343691391731719804510073840091918839807034"
    "123822516335312440539949689328025005031641651021475338193305452779148824615510516946022794"
    "721926745835141507082535452895792504462259349063207888733994905366993964665898229574594013"
    "362998679850527186594367398202895198873704795707040293706313605071045458316802978515625",
    704,
    "1.0000000000000000185267526717021225041886804737033222476192186962870088250087687318910666"
    "205005363911217585349955151204119274354572185745520077107861478711995856090116109355473198"
    "965629018102773173958643755446862622184950480409141464100452195464059076134343063064395081"
    "883198721699263332935106297336265956079108453519661106614136652848940265031537516012481956"
    "041767365788379171625098381602438061934369139173171980451007384009191883980703412382251633"
    "531244053994968932802500503164165102147533819330545277914882461551051694602279472192674583"
    "514150708253545289579250446225934906320788873399490536699396466589822957459401336299867985"
    "0527186594367398202895198873704795707040293706313605071045458316802978515625e-281"
  },
  {
    1e-280,
    981,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000999999999999999957364388169475170050694172070683240219593513666870751723400097"
    "759022579142792104411515557258308779196173510175383489009931421520586690004171251700735334"
    "129506741559146860533228960854564563482256350736730224010460829399817049880088993405271129"
    "831235264850390915566232466647936884444101972033761288853186871037892663016797660123731015"
    "646335493680073454731979186907663345907207730732342719302852534949839594619012356967550855"
    "734248053136101277703677183805567441722726787676002477378641128208658347785021448019020754"
    "979932717753065897070560929756201108648397168140986925463872335756081422906706746822965482"
    "25014870667032395919842010303184918392220732752395662146227550692856311798095703125",
    700,
    "9.9999999999999995736438816947517005069417207068324021959351366687075172340009775902257914"
    "279210441151555725830877919617351017538348900993142152058669000417125170073533412950674155"
    "914686053322896085456456348225635073673022401046082939981704988008899340527112983123526485"
    "039091556623246664793688444410197203376128885318687103789266301679766012373101564633549368"
    "007345473197918690766334590720773073234271930285253494983959461901235696755085573424805313"
    "610127770367718380556744172272678767600247737864112820865834778502144801902075497993271775"
    "306589707056092975620110864839716814098692546387233575608142290670674682296548225014870667"
    "032395919842010303184918392220732752395662146227550692856311798095703125e-281"
  },
  {
    1e-279,
    979,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000010000000000000000552241713730382939762853855155153714644346417139367630859739713"
    "156121591071255955788854793010075566749549777915014563133243594249058225914165859539773770"
    "191716668990125265765063313206442627821442091075539424591861366161821213461960561383322121"
    "915620541910772461321426532808548769073413697524005792512336544355262666457518463308073931"
    "741958167586387056869531553154177335120654514700934306859959959022574246700632872563820325"
    "406114294833382820169887189650596154726303437380039246657025675411861851771554023249522625"
    "628683027104245750782306884789594062071112841267340644655950844182545809103476250997249244"
    "578896536062406099908263094272136716259523347982707264236523769795894622802734375",
    700,
    "1.0000000000000000552241713730382939762853855155153714644346417139367630859739713156121591"
    "071255955788854793010075566749549777915014563133243594249058225914165859539773770191716668"
    "990125265765063313206442627821442091075539424591861366161821213461960561383322121915620541"
    "910772461321426532808548769073413697524005792512336544355262666457518463308073931741958167"
    "586387056869531553154177335120654514700934306859959959022574246700632872563820325406114294"
    "833382820169887189650596154726303437380039246657025675411861851771554023249522625628683027"
    "104245750782306884789594062071112841267340644655950844182545809103476250997249244578896536"
    "062406099908263094272136716259523347982707264236523769795894622802734375e-279"
  },
  {
    1e-278,
    976,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000099999999999999993779243152876254526557592938171681397062528805745754945088532304"
    "770466314992540617804157284976902370041721665215989554925283393972286348672218485008692555"
    "157375649117372732457348676134862362227877906256548032071576795654403558686758085865891335"
    "916990698225364945305042932135328595145409249003343077357751435836594229100682288231574014"
    "076342906436040454098439322611246838623618658479257702622384275935602860880217090978462037"
    "297537786665387684101487677366900697274607646360218792122884034215278030659465715783445346"
    "334560076159416146902697801164944913665434494383749766352778602364679130597857116761359380"
    "860051728314062903993886175277628638145832606909380047000013291835784912109375",
    697,
    "9.9999999999999993779243152876254526557592938171681397062528805745754945088532304770466314"
    "992540617804157284976902370041721665215989554925283393972286348672218485008692555157375649"
    "117372732457348676134862362227877906256548032071576795654403558686758085865891335916990698"
    "225364945305042932135328595145409249003343077357751435836594229100682288231574014076342906"
    "436040454098439322611246838623618658479257702622384275935602860880217090978462037297537786"
    "665387684101487677366900697274607646360218792122884034215278030659465715783445346334560076"
    "159416146902697801164944913665434494383749766352778602364679130597857116761359380860051728"
    "314062903993886175277628638145832606909380047000013291835784912109375e-279"
  },
  {
    1e-277,
    973,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000999999999999999969107562153902744921765117684063095968974449032518673086908962585"
    "813328738512123351599947903432632493627286289317645086338574069104985914640691811124379276"
    "266532599930740458422245310494127398242893741229070224307866265780858456021840933238459714"
    "474479571273275234141689043887043539690699759998003700952485045608695452272162468580180949"
    "678732449501284846576063116593872419790657220872804668829750264129734225138468602210596951"
    "011658297769718221274568024706502572715214403442651053088500928561688272077522535732535889"
    "884987947948938431481300802932104355442862059276582144079065595542201867143652216176615315"
    "077620302633957293508847799292922446656728396874314057640731334686279296875",
    695,
    "9.9999999999999996910756215390274492176511768406309596897444903251867308690896258581332873"
    "851212335159994790343263249362728628931764508633857406910498591464069181112437927626653259"
    "993074045842224531049412739824289374122907022430786626578085845602184093323845971447447957"
    "127327523414168904388704353969069975999800370095248504560869545227216246858018094967873244"
    "950128484657606311659387241979065722087280466882975026412973422513846860221059695101165829"
    "776971822127456802470650257271521440344265105308850092856168827207752253573253588988498794"
    "794893843148130080293210435544286205927658214407906559554220186714365221617661531507762030"
    "2633957293508847799292922446656728396874314057640731334686279296875e-278"
  },
  {
    1e-276,
    957,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000010000000000000000943680846544635435465218708934482239623710929327631676310035207382"
    "479910928589920458334481180870676664675648379486432346815345866334756263147196552741941750"
    "376370349587929938172795070761425020993524558834298386762595027281499326388812315566451356"
    "927699273517783585067279340220738926371288398562954104523677945797080973335208136379441853"
    "399459900648060689427426785194885540085397651937152392533802832245566904836593719145032631"
    "567800222330837423133330288564849725917661628045035805271432741973201340089840473248655960"
    "425366933680462812985919680627252305969305210329200662841838831238441118039764104287013409"
    "86032380607270307788791985359466707450337707996368408203125",
    681,
    "1.0000000000000000943680846544635435465218708934482239623710929327631676310035207382479910"
    "928589920458334481180870676664675648379486432346815345866334756263147196552741941750376370"
    "349587929938172795070761425020993524558834298386762595027281499326388812315566451356927699"
    "273517783585067279340220738926371288398562954104523677945797080973335208136379441853399459"
    "900648060689427426785194885540085397651937152392533802832245566904836593719145032631567800"
    "222330837423133330288564849725917661628045035805271432741973201340089840473248655960425366"
    "933680462812985919680627252305969305210329200662841838831238441118039764104287013409860323"
    "80607270307788791985359466707450337707996368408203125e-276"
  },
  {
    1e-275,
    966,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000099999999999999993403461585374572130683322678543526013082338874045021461456248630313"
    "162327929500011721456784332939064523200829570096560480254512419700879537196401476243110461"
    "062335812288574851163573545116316916308530112584953228471615943561684256906964970936779653"
    "335827157129435931947815464923504086569961763768202229251787589681191165498213196400724369"
    "359265814349890431339283925469990220965010846294970911113385878318393484181518669350318361"
    "102421491997587538371382354450750074977991082133234540568107178371135073671331248668357216"
    "087429923158823353245927669553051039972288998480752586163247501934203863876144178603122782"
    "32649208014296495104634271942900536345177897601388394832611083984375",
    690,
    "9.9999999999999993403461585374572130683322678543526013082338874045021461456248630313162327"
    "929500011721456784332939064523200829570096560480254512419700879537196401476243110461062335"
    "812288574851163573545116316916308530112584953228471615943561684256906964970936779653335827"
    "157129435931947815464923504086569961763768202229251787589681191165498213196400724369359265"
    "814349890431339283925469990220965010846294970911113385878318393484181518669350318361102421"
    "491997587538371382354450750074977991082133234540568107178371135073671331248668357216087429"
    "923158823353245927669553051039972288998480752586163247501934203863876144178603122782326492"
    "08014296495104634271942900536345177897601388394832611083984375e-276"
  },
  {
    1e-274,
    963,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000999999999999999966101309613889285754770955607037852897132929578912805217850693190154"
    "896842007798502938343898280926049479119604150501130778343016684302161560515142864783718696"
    "026093490067197572764489776159035750338732077365593563024828094123460583031966078823264365"
    "240602727391159156928110523802811219985462081404699924487859633391148790689868298794632022"
    "863324527760337239262807107657632569428039809102815139583143671458485970184023737703199520"
    "175376382597448769637664606902995118177161217966592429973512266433107391167459454319976937"
    "206778058879843085685814968969454439164416194430033946142756640242453271804435514565250455"
    "81841276259778116612913882732538439768177340738475322723388671875",
    688,
    "9.9999999999999996610130961388928575477095560703785289713292957891280521785069319015489684"
    "200779850293834389828092604947911960415050113077834301668430216156051514286478371869602609"
    "349006719757276448977615903575033873207736559356302482809412346058303196607882326436524060"
    "272739115915692811052380281121998546208140469992448785963339114879068986829879463202286332"
    "452776033723926280710765763256942803980910281513958314367145848597018402373770319952017537"
    "638259744876963766460690299511817716121796659242997351226643310739116745945431997693720677"
    "805887984308568581496896945443916441619443003394614275664024245327180443551456525045581841"
    "276259778116612913882732538439768177340738475322723388671875e-275"
  },
  {
    1e-273,
    954,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000010000000000000000943680846544635435465218708934482239623710929327631676310035207382479"
    "910928589920458334481180870676664675648379486432346815345866334756263147196552741941750376"
    "370349587929938172795070761425020993524558834298386762595027281499326388812315566451356927"
    "699273517783585067279340220738926371288398562954104523677945797080973335208136379441853399"
    "459900648060689427426785194885540085397651937152392533802832245566904836593719145032631567"
    "800222330837423133330288564849725917661628045035805271432741973201340089840473248655960425"
    "366933680462812985919680627252305969305210329200662841838831238441118039764104287013409860"
    "32380607270307788791985359466707450337707996368408203125",
    681,
    "1.0000000000000000943680846544635435465218708934482239623710929327631676310035207382479910"
    "928589920458334481180870676664675648379486432346815345866334756263147196552741941750376370"
    "349587929938172795070761425020993524558834298386762595027281499326388812315566451356927699"
    "273517783585067279340220738926371288398562954104523677945797080973335208136379441853399459"
    "900648060689427426785194885540085397651937152392533802832245566904836593719145032631567800"
    "222330837423133330288564849725917661628045035805271432741973201340089840473248655960425366"
    "933680462812985919680627252305969305210329200662841838831238441118039764104287013409860323"
    "80607270307788791985359466707450337707996368408203125e-273"
  },
  {
    1e-272,
    956,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000099999999999999993018661260252849357308069932684294899886624383983470374216790147668883"
    "045176946431092771471673520639672235493868702134168544937709853359142933787939014879092037"
    "502987882397462430028493216366517261488941166760493131911919659604840739417174503314039353"
    "239183256274333898415994428690842318531630443530097668147784842240319869720360383275709408"
    "017817738753236228844311234497456647675670141133638771994459659098870641092624819918170192"
    "607554446128657740296261702004142557224077373623576276597892578473993817881485056720358771"
    "440177231324038607209210271783711498990683965282489149124122483398888279628253860714510735"
    "8501766089473711783541798325813942938111722469329833984375",
    683,
    "9.9999999999999993018661260252849357308069932684294899886624383983470374216790147668883045"
    "176946431092771471673520639672235493868702134168544937709853359142933787939014879092037502"
    "987882397462430028493216366517261488941166760493131911919659604840739417174503314039353239"
    "183256274333898415994428690842318531630443530097668147784842240319869720360383275709408017"
    "817738753236228844311234497456647675670141133638771994459659098870641092624819918170192607"
    "554446128657740296261702004142557224077373623576276597892578473993817881485056720358771440"
    "177231324038607209210271783711498990683965282489149124122483398888279628253860714510735850"
    "1766089473711783541798325813942938111722469329833984375e-273"
  },
  {
    1e-271,
    953,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000999999999999999963022907012915503567768933640164003991567213658420396519935025329000662"
    "579987369857908861397005578650671396918539345720284666419005521998406414234566957867743827"
    "430894817778462896129360959432557962402706020051680307195902906825253691583707355539453379"
    "898936405866372532914759844305265974021014807322871818741194679542025661925610655042743253"
    "340554871239678379290193773690454889358399872116960852012322184129061860615381459997992896"
    "864882185777724588975864913027658811025179889704715641437980091818752339861345426882078678"
    "860091459848568576067294719315811378029678306273170029769604636013467032308116625984283093"
    "2775343256405271852305105539926444180309772491455078125",
    681,
    "9.9999999999999996302290701291550356776893364016400399156721365842039651993502532900066257"
    "998736985790886139700557865067139691853934572028466641900552199840641423456695786774382743"
    "089481777846289612936095943255796240270602005168030719590290682525369158370735553945337989"
    "893640586637253291475984430526597402101480732287181874119467954202566192561065504274325334"
    "055487123967837929019377369045488935839987211696085201232218412906186061538145999799289686"
    "488218577772458897586491302765881102517988970471564143798009181875233986134542688207867886"
    "009145984856857606729471931581137802967830627317002976960463601346703230811662598428309327"
    "75343256405271852305105539926444180309772491455078125e-272"
  },
  {
    1e-270,
    948,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "010000000000000000418300135978443275550206959921345359740495412230260591865761225745490596"
    "877103431706636134296544720601490976701849242289227873195822941751513974869912996521201131"
    "933332029076755261559900692742827964346124659238778785799980526896848053724169292971970139"
    "159856293616530499263171820576886669123197001754201481732256966752103772584270285283012689"
    "302608321372369973231892026085870796024761779903363689748636076659149398693012859570912267"
    "592927245564778354076598561946185841077546580302025425397104288079063263729330901104569935"
    "600574117082065856077555228646827095244910745490385222377161608724691987695038711983048567"
    "39378092440884149283419901621527969837188720703125",
    678,
    "1.0000000000000000418300135978443275550206959921345359740495412230260591865761225745490596"
    "877103431706636134296544720601490976701849242289227873195822941751513974869912996521201131"
    "933332029076755261559900692742827964346124659238778785799980526896848053724169292971970139"
    "159856293616530499263171820576886669123197001754201481732256966752103772584270285283012689"
    "302608321372369973231892026085870796024761779903363689748636076659149398693012859570912267"
    "592927245564778354076598561946185841077546580302025425397104288079063263729330901104569935"
    "600574117082065856077555228646827095244910745490385222377161608724691987695038711983048567"
    "39378092440884149283419901621527969837188720703125e-270"
  },
  {
    1e-269,
    944,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "099999999999999995776909990725358196861881615003263519273505848744668567549228551263076943"
    "947250497039187792816231909003955020176297381970879169230040385329008201773866841545207504"
    "673225876984872079425235210977630680057892366020046910362989710095828399779338395560380429"
    "779979096668716511398935512177112782826114779936859677905927839981334757008956347703938563"
    "457447384650780475466074509591274334212811839206915088148608922660498866512013826738634153"
    "858814985914103921374525014986149281967490514968318136453140268614207409390624933352012454"
    "676039639125937543126399907921995593002412172144546360506386768953355454402414297001497571"
    "3411407853836720960316597484052181243896484375",
    674,
    "9.9999999999999995776909990725358196861881615003263519273505848744668567549228551263076943"
    "947250497039187792816231909003955020176297381970879169230040385329008201773866841545207504"
    "673225876984872079425235210977630680057892366020046910362989710095828399779338395560380429"
    "779979096668716511398935512177112782826114779936859677905927839981334757008956347703938563"
    "457447384650780475466074509591274334212811839206915088148608922660498866512013826738634153"
    "858814985914103921374525014986149281967490514968318136453140268614207409390624933352012454"
    "676039639125937543126399907921995593002412172144546360506386768953355454402414297001497571"
    "3411407853836720960316597484052181243896484375e-270"
  },
  {
    1e-268,
    943,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "999999999999999957769099907253581968618816150032635192735058487446685675492285512630769439"
    "472504970391877928162319090039550201762973819708791692300403853290082017738668415452075046"
    "732258769848720794252352109776306800578923660200469103629897100958283997793383955603804297"
    "799790966687165113989355121771127828261147799368596779059278399813347570089563477039385634"
    "574473846507804754660745095912743342128118392069150881486089226604988665120138267386341538"
    "588149859141039213745250149861492819674905149683181364531402686142074093906249333520124546"
    "760396391259375431263999079219955930024121721445463605063867689533554544024142970014975713"
    "411407853836720960316597484052181243896484375",
    674,
    "9.9999999999999995776909990725358196861881615003263519273505848744668567549228551263076943"
    "947250497039187792816231909003955020176297381970879169230040385329008201773866841545207504"
    "673225876984872079425235210977630680057892366020046910362989710095828399779338395560380429"
    "779979096668716511398935512177112782826114779936859677905927839981334757008956347703938563"
    "457447384650780475466074509591274334212811839206915088148608922660498866512013826738634153"
    "858814985914103921374525014986149281967490514968318136453140268614207409390624933352012454"
    "676039639125937543126399907921995593002412172144546360506386768953355454402414297001497571"
    "3411407853836720960316597484052181243896484375e-269"
  },
  {
    1e-267,
    935,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009"
    "999999999999999846685922882426205562674176995052434427556929628320851990391133724446223189"
    "086131944788332886398080404746053916579979506572702930306087562857029678995104111858472536"
    "445608939532985100084216024183834834696571845772401360677068893507708376729184649136313756"
    "192592530762482539342597412647403351598845597050932251925322479403970703575522934431882891"
    "941084995411463765898514999685309454394974635146606713668951271841730504581055280919048092"
    "136137622888139957979240841837620318604260714493769405886910451066348231948383821399226310"
    "154412926824826877412866905680370802535466262752423675126015080729606961776560030677376417"
    "2074692169189802370965480804443359375",
    667,
    "9.9999999999999998466859228824262055626741769950524344275569296283208519903911337244462231"
    "890861319447883328863980804047460539165799795065727029303060875628570296789951041118584725"
    "364456089395329851000842160241838348346965718457724013606770688935077083767291846491363137"
    "561925925307624825393425974126474033515988455970509322519253224794039707035755229344318828"
    "919410849954114637658985149996853094543949746351466067136689512718417305045810552809190480"
    "921361376228881399579792408418376203186042607144937694058869104510663482319483838213992263"
    "101544129268248268774128669056803708025354662627524236751260150807296069617765600306773764"
    "172074692169189802370965480804443359375e-268"
  },
  {
    1e-266,
    934,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000099"
    "999999999999998466859228824262055626741769950524344275569296283208519903911337244462231890"
    "861319447883328863980804047460539165799795065727029303060875628570296789951041118584725364"
    "456089395329851000842160241838348346965718457724013606770688935077083767291846491363137561"
    "925925307624825393425974126474033515988455970509322519253224794039707035755229344318828919"
    "410849954114637658985149996853094543949746351466067136689512718417305045810552809190480921"
    "361376228881399579792408418376203186042607144937694058869104510663482319483838213992263101"
    "544129268248268774128669056803708025354662627524236751260150807296069617765600306773764172"
    "074692169189802370965480804443359375",
    667,
    "9.9999999999999998466859228824262055626741769950524344275569296283208519903911337244462231"
    "890861319447883328863980804047460539165799795065727029303060875628570296789951041118584725"
    "364456089395329851000842160241838348346965718457724013606770688935077083767291846491363137"
    "561925925307624825393425974126474033515988455970509322519253224794039707035755229344318828"
    "919410849954114637658985149996853094543949746351466067136689512718417305045810552809190480"
    "921361376228881399579792408418376203186042607144937694058869104510663482319483838213992263"
    "101544129268248268774128669056803708025354662627524236751260150807296069617765600306773764"
    "172074692169189802370965480804443359375e-267"
  },
  {
    1e-265,
    933,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000999"
    "999999999999984668592288242620556267417699505243442755692962832085199039113372444622318908"
    "613194478833288639808040474605391657997950657270293030608756285702967899510411185847253644"
    "560893953298510008421602418383483469657184577240136067706889350770837672918464913631375619"
    "259253076248253934259741264740335159884559705093225192532247940397070357552293443188289194"
    "108499541146376589851499968530945439497463514660671366895127184173050458105528091904809213"
    "613762288813995797924084183762031860426071449376940588691045106634823194838382139922631015"
    "441292682482687741286690568037080253546626275242367512601508072960696177656003067737641720"
    "74692169189802370965480804443359375",
    667,
    "9.9999999999999998466859228824262055626741769950524344275569296283208519903911337244462231"
    "890861319447883328863980804047460539165799795065727029303060875628570296789951041118584725"
    "364456089395329851000842160241838348346965718457724013606770688935077083767291846491363137"
    "561925925307624825393425974126474033515988455970509322519253224794039707035755229344318828"
    "919410849954114637658985149996853094543949746351466067136689512718417305045810552809190480"
    "921361376228881399579792408418376203186042607144937694058869104510663482319483838213992263"
    "101544129268248268774128669056803708025354662627524236751260150807296069617765600306773764"
    "172074692169189802370965480804443359375e-266"
  },
  {
    1e-264,
    925,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000010000"
    "000000000000122136724863753960700195856861651942907768226656267343111510651008940076674511"
    "880159438755777687567257201019061105026607485123801783385769532188208642126148172299935227"
    "582690363860909426367628838700067497683135390536732840241126646773617095618024468943033063"
    "947783386693892378420716261995422241910022896655860329841884224957586319728414406822075246"
    "143842472881974452564577216574512303496326748626962048403693772578610441840030544015983340"
    "887991121353726198621929297657051383994953379612104713543246863450099863535679265958692926"
    "072717397445183740292045884721780884777288209358202601049382573125959828533489137658563095"
    "101271755993366241455078125",
    661,
    "1.0000000000000000122136724863753960700195856861651942907768226656267343111510651008940076"
    "674511880159438755777687567257201019061105026607485123801783385769532188208642126148172299"
    "935227582690363860909426367628838700067497683135390536732840241126646773617095618024468943"
    "033063947783386693892378420716261995422241910022896655860329841884224957586319728414406822"
    "075246143842472881974452564577216574512303496326748626962048403693772578610441840030544015"
    "983340887991121353726198621929297657051383994953379612104713543246863450099863535679265958"
    "692926072717397445183740292045884721780884777288209358202601049382573125959828533489137658"
    "563095101271755993366241455078125e-264"
  },
  {
    1e-263,
    924,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000"
    "000000000001221367248637539607001958568616519429077682266562673431115106510089400766745118"
    "801594387557776875672572010190611050266074851238017833857695321882086421261481722999352275"
    "826903638609094263676288387000674976831353905367328402411266467736170956180244689430330639"
    "477833866938923784207162619954222419100228966558603298418842249575863197284144068220752461"
    "438424728819744525645772165745123034963267486269620484036937725786104418400305440159833408"
    "879911213537261986219292976570513839949533796121047135432468634500998635356792659586929260"
    "727173974451837402920458847217808847772882093582026010493825731259598285334891376585630951"
    "01271755993366241455078125",
    661,
    "1.0000000000000000122136724863753960700195856861651942907768226656267343111510651008940076"
    "674511880159438755777687567257201019061105026607485123801783385769532188208642126148172299"
    "935227582690363860909426367628838700067497683135390536732840241126646773617095618024468943"
    "033063947783386693892378420716261995422241910022896655860329841884224957586319728414406822"
    "075246143842472881974452564577216574512303496326748626962048403693772578610441840030544015"
    "983340887991121353726198621929297657051383994953379612104713543246863450099863535679265958"
    "692926072717397445183740292045884721780884777288209358202601049382573125959828533489137658"
    "563095101271755993366241455078125e-263"
  },
  {
    1e-262,
    923,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000"
    "000000000012213672486375396070019585686165194290776822665626734311151065100894007667451188"
    "015943875577768756725720101906110502660748512380178338576953218820864212614817229993522758"
    "269036386090942636762883870006749768313539053673284024112664677361709561802446894303306394"
    "778338669389237842071626199542224191002289665586032984188422495758631972841440682207524614"
    "384247288197445256457721657451230349632674862696204840369377257861044184003054401598334088"
    "799112135372619862192929765705138399495337961210471354324686345009986353567926595869292607"
    "271739744518374029204588472178088477728820935820260104938257312595982853348913765856309510"
    "1271755993366241455078125",
    661,
    "1.0000000000000000122136724863753960700195856861651942907768226656267343111510651008940076"
    "674511880159438755777687567257201019061105026607485123801783385769532188208642126148172299"
    "935227582690363860909426367628838700067497683135390536732840241126646773617095618024468943"
    "033063947783386693892378420716261995422241910022896655860329841884224957586319728414406822"
    "075246143842472881974452564577216574512303496326748626962048403693772578610441840030544015"
    "983340887991121353726198621929297657051383994953379612104713543246863450099863535679265958"
    "692926072717397445183740292045884721780884777288209358202601049382573125959828533489137658"
    "563095101271755993366241455078125e-262"
  },
  {
    1e-261,
    920,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000009999999"
    "999999999840075103634874339439373656678254046224031858499650136203484265309618370705435913"
    "987636722737007132720287134753111378376150804829390632405896825874283575582986940678874841"
    "569513044080660004385672118069109345174821566111745152759912370810960390248128330653547680"
    "004699501466869737653360653141903412050830634885323614013688267018389662002912484351541763"
    "583081989728893217809824559093005170114548158227707844413715390150040396446928048862747223"
    "614050542827541975695738579437017547092760509181115071263967217787118420176810424732967893"
    "091851084038061008702315466379055082798141457246466614257485362505823312667674329645706166"
    "0476028919219970703125",
    658,
    "9.9999999999999998400751036348743394393736566782540462240318584996501362034842653096183707"
    "054359139876367227370071327202871347531113783761508048293906324058968258742835755829869406"
    "788748415695130440806600043856721180691093451748215661117451527599123708109603902481283306"
    "535476800046995014668697376533606531419034120508306348853236140136882670183896620029124843"
    "515417635830819897288932178098245590930051701145481582277078444137153901500403964469280488"
    "627472236140505428275419756957385794370175470927605091811150712639672177871184201768104247"
    "329678930918510840380610087023154663790550827981414572464666142574853625058233126676743296"
    "457061660476028919219970703125e-262"
  },
  {
    1e-260,
    916,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000099999999"
    "999999996144258066517706424307158965315357288770427639743563706770631567501610059301751410"
    "501950963044627850907560273067164597910833496514764297149885360067967351308386532737926486"
    "728323906176469137911388524703986748063663896148690820665248485660522080270558407499346657"
    "817497475264627912030395793176326136731778181053186317172589145640456088737170141725782593"
    "755692759324457403959109467575665091447877832402353969896696393817192415800352751662722921"
    "123938941086166587088913173455243888042652268422362936776502319369223277748459975650013493"
    "914139951215175820274600620575913272060871364815590661574151517426191360105036665117950178"
    "682804107666015625",
    655,
    "9.9999999999999996144258066517706424307158965315357288770427639743563706770631567501610059"
    "301751410501950963044627850907560273067164597910833496514764297149885360067967351308386532"
    "737926486728323906176469137911388524703986748063663896148690820665248485660522080270558407"
    "499346657817497475264627912030395793176326136731778181053186317172589145640456088737170141"
    "725782593755692759324457403959109467575665091447877832402353969896696393817192415800352751"
    "662722921123938941086166587088913173455243888042652268422362936776502319369223277748459975"
    "650013493914139951215175820274600620575913272060871364815590661574151517426191360105036665"
    "117950178682804107666015625e-261"
  },
  {
    1e-259,
    909,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000"
    "000000069754243217066838807227314523578365214259041769576644520388447783555635685142685114"
    "991490318067565371250534304941206899940713450546460263134832737073356930115043281818717457"
    "689952724010974864489852734420989257495123679987422139478495534161148268820379228727713405"
    "190856644041613416458073367413244588591133864934254674011980634489706389385527103160307957"
    "163030215539363198269628596767208179963758318010314462508924306966078494112058890935196332"
    "034580795945818024575817538469154858904258206885442609197176401786357130427524797124075915"
    "351202192092603006676600280061735404794787615311529703775216340599918396492284955456852912"
    "90283203125",
    650,
    "1.0000000000000000697542432170668388072273145235783652142590417695766445203884477835556356"
    "851426851149914903180675653712505343049412068999407134505464602631348327370733569301150432"
    "818187174576899527240109748644898527344209892574951236799874221394784955341611482688203792"
    "287277134051908566440416134164580733674132445885911338649342546740119806344897063893855271"
    "031603079571630302155393631982696285967672081799637583180103144625089243069660784941120588"
    "909351963320345807959458180245758175384691548589042582068854426091971764017863571304275247"
    "971240759153512021920926030066766002800617354047947876153115297037752163405999183964922849"
    "5545685291290283203125e-259"
  },
  {
    1e-258,
    910,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000009999999999"
    "999999542218031617177459387945413284585867326006253726262365708608402011134649202091693710"
    "213775846048593849306072923870085843861763994543884853897883249200946186151201304166346945"
    "894581509482724800888207478811260288460733135868739444640841447681589716312643980778501230"
    "405826265532568338936835693865958192328916735717037382401521778655511872374463715309938029"
    "165207517582547623458590810226137634464463244244213813974999135856472022629587583400314031"
    "863766518560557273100193476246578151946736493795084850028796464859578206217380871252055407"
    "274126668223685491506332674722925416629753836788650765392684298393799480209054308943450450"
    "897216796875",
    651,
    "9.9999999999999995422180316171774593879454132845858673260062537262623657086084020111346492"
    "020916937102137758460485938493060729238700858438617639945438848538978832492009461861512013"
    "041663469458945815094827248008882074788112602884607331358687394446408414476815897163126439"
    "807785012304058262655325683389368356938659581923289167357170373824015217786555118723744637"
    "153099380291652075175825476234585908102261376344644632442442138139749991358564720226295875"
    "834003140318637665185605572731001934762465781519467364937950848500287964648595782062173808"
    "712520554072741266682236854915063326747229254166297538367886507653926842983937994802090543"
    "08943450450897216796875e-259"
  },
  {
    1e-257,
    904,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000099999999999"
    "999997732829117278756451248109596748254242893230865201631816076636171760189907319587251981"
    "540013129740058219459269489784824749708380967280284093879720735074708091510476069705124720"
    "955706556081295696902714518909867457588338686698358346696642264675683106908736420782277947"
    "063743005092815040656152899192557310454011184421392539451786919038222766706251785685663376"
    "582264451447644953061298417153264674990872314159999761978479226173346063277878485906438895"
    "601748067400818676317898579355722393659056088069530984173899754603768258289542912497961565"
    "217057187641544065582666999018111428933783000539800198645801199148763771518133580684661865"
    "234375",
    646,
    "9.9999999999999997732829117278756451248109596748254242893230865201631816076636171760189907"
    "319587251981540013129740058219459269489784824749708380967280284093879720735074708091510476"
    "069705124720955706556081295696902714518909867457588338686698358346696642264675683106908736"
    "420782277947063743005092815040656152899192557310454011184421392539451786919038222766706251"
    "785685663376582264451447644953061298417153264674990872314159999761978479226173346063277878"
    "485906438895601748067400818676317898579355722393659056088069530984173899754603768258289542"
    "912497961565217057187641544065582666999018111428933783000539800198645801199148763771518133"
    "580684661865234375e-258"
  },
  {
    1e-256,
    903,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000999999999999"
    "999977328291172787564512481095967482542428932308652016318160766361717601899073195872519815"
    "400131297400582194592694897848247497083809672802840938797207350747080915104760697051247209"
    "557065560812956969027145189098674575883386866983583466966422646756831069087364207822779470"
    "637430050928150406561528991925573104540111844213925394517869190382227667062517856856633765"
    "822644514476449530612984171532646749908723141599997619784792261733460632778784859064388956"
    "017480674008186763178985793557223936590560880695309841738997546037682582895429124979615652"
    "170571876415440655826669990181114289337830005398001986458011991487637715181335806846618652"
    "34375",
    646,
    "9.9999999999999997732829117278756451248109596748254242893230865201631816076636171760189907"
    "319587251981540013129740058219459269489784824749708380967280284093879720735074708091510476"
    "069705124720955706556081295696902714518909867457588338686698358346696642264675683106908736"
    "420782277947063743005092815040656152899192557310454011184421392539451786919038222766706251"
    "785685663376582264451447644953061298417153264674990872314159999761978479226173346063277878"
    "485906438895601748067400818676317898579355722393659056088069530984173899754603768258289542"
    "912497961565217057187641544065582666999018111428933783000539800198645801199148763771518133"
    "580684661865234375e-257"
  },
  {
    1e-255,
    898,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000010000000000000"
    "000069045958269569322867998859054332057202368632496356225958454292587070947890188525502717"
    "489910638533146924940101117230162790452947523732160415285768619822326590850874559844345632"
    "836762648647673756913337433036611100402806655239213906557383313620911495007608541877797011"
    "075785279474355430453172867476580602501128330269649521059540861659594169711851539610572529"
    "290672424402091270979802021488173783405934995886263843094369671238713461484188034266107411"
    "577415609873348632233226497484671262442076022144456354789669029399058931768268846904315558"
    "606903455954617824742252130784872510817613033601465588606771461854805238544940948486328125",
    643,
    "1.0000000000000000069045958269569322867998859054332057202368632496356225958454292587070947"
    "890188525502717489910638533146924940101117230162790452947523732160415285768619822326590850"
    "874559844345632836762648647673756913337433036611100402806655239213906557383313620911495007"
    "608541877797011075785279474355430453172867476580602501128330269649521059540861659594169711"
    "851539610572529290672424402091270979802021488173783405934995886263843094369671238713461484"
    "188034266107411577415609873348632233226497484671262442076022144456354789669029399058931768"
    "268846904315558606903455954617824742252130784872510817613033601465588606771461854805238544"
    "940948486328125e-255"
  },
  {
    1e-254,
    894,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000099999999999999"
    "991226042093361495540897975810399108318806228853725384840359241312717046849838531645281143"
    "263981120457069920980142732375617676854249774801571278819442602974707834804182739823503135"
    "852201189897407436593036984770420073822051019484003484992814062525889217789158581977896360"
    "310340148572310629719474331698620197810966882523836782408241965801781726344980322690209418"
    "851451295617841834599290417707136735860835402501433783057390987455706336559018146750102870"
    "890672265406094308144470993648891935253809335321109551106496085279130027635405361582066405"
    "23112442193941772020484998068937735011811498812839271721486511523835361003875732421875",
    639,
    "9.9999999999999991226042093361495540897975810399108318806228853725384840359241312717046849"
    "838531645281143263981120457069920980142732375617676854249774801571278819442602974707834804"
    "182739823503135852201189897407436593036984770420073822051019484003484992814062525889217789"
    "158581977896360310340148572310629719474331698620197810966882523836782408241965801781726344"
    "980322690209418851451295617841834599290417707136735860835402501433783057390987455706336559"
    "018146750102870890672265406094308144470993648891935253809335321109551106496085279130027635"
    "405361582066405231124421939417720204849980689377350118114988128392717214865115238353610038"
    "75732421875e-255"
  },
  {
    1e-253,
    890,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000001000000000000000"
    "063691100762962118413491962586298479239541608077064687111197238937629070563398974208747938"
    "801815442561088464535322362572340411346105148336238772806323553724007527314833136154282438"
    "768816644243356486575767377235255621516758721370204139224445804190503894482415208580143610"
    "263603824463005094190812806061175213314731548000902675077086070724376795826364401550348173"
    "398880130627552349172580931864984929784586926793612196754801473459915818115976602076568627"
    "042463987299215308449413635654050379209307531186359459708068192174648917110583335198090945"
    "7178064211023456375312410414433376301093954474641378737942432053387165069580078125",
    637,
    "1.0000000000000000636911007629621184134919625862984792395416080770646871111972389376290705"
    "633989742087479388018154425610884645353223625723404113461051483362387728063235537240075273"
    "148331361542824387688166442433564865757673772352556215167587213702041392244458041905038944"
    "824152085801436102636038244630050941908128060611752133147315480009026750770860707243767958"
    "263644015503481733988801306275523491725809318649849297845869267936121967548014734599158181"
    "159766020765686270424639872992153084494136356540503792093075311863594597080681921746489171"
    "105833351980909457178064211023456375312410414433376301093954474641378737942432053387165069"
    "580078125e-253"
  },
  {
    1e-252,
    890,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000009999999999999999"
    "425465568994843880098821990004525623983581524452160161451133782892621889113880480039987338"
    "722120521687770607482063315194094971032192280798179851168055345424641838964285458188815745"
    "713728480279307900594493536104117148797599001460687077874016610452145212097516975391996045"
    "354419534710860565939572148011966251506813697908747942813529405591291699251154618317449854"
    "913863910682451466288395280300908728436006053701927038100881943376338560953404944161366925"
    "338709207085975268456506763219455578723361888061483007935823206679700045053595596961494309"
    "9255665973581088914504805380368632696679899451996931247776956297457218170166015625",
    637,
    "9.9999999999999994254655689948438800988219900045256239835815244521601614511337828926218891"
    "138804800399873387221205216877706074820633151940949710321922807981798511680553454246418389"
    "642854581888157457137284802793079005944935361041171487975990014606870778740166104521452120"
    "975169753919960453544195347108605659395721480119662515068136979087479428135294055912916992"
    "511546183174498549138639106824514662883952803009087284360060537019270381008819433763385609"
    "534049441613669253387092070859752684565067632194555787233618880614830079358232066797000450"
    "535955969614943099255665973581088914504805380368632696679899451996931247776956297457218170"
    "166015625e-253"
  },
  {
    1e-251,
    886,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000100000000000000001"
    "523328321757102625204805715196011250306822582432521872476369467828231790259460372684825682"
    "997408640416390302047595015116804564895078023367045773051634605139018994747130002012209308"
    "983912575718620796924016778531805886195919288054996664962814693238814517334980416376600797"
    "233907606623747915207056955718377804911147671689152275879281865827774546586482566290689823"
    "588263480382946815508437033102730700819239822424439957691616181100303330772215901239585323"
    "902676066296819580790845192120845067451899423427499614227384357197735206849382499731433982"
    "770651655573173817676384638747710885235686628647044926765374839305877685546875",
    635,
    "1.0000000000000000152332832175710262520480571519601125030682258243252187247636946782823179"
    "025946037268482568299740864041639030204759501511680456489507802336704577305163460513901899"
    "474713000201220930898391257571862079692401677853180588619591928805499666496281469323881451"
    "733498041637660079723390760662374791520705695571837780491114767168915227587928186582777454"
    "658648256629068982358826348038294681550843703310273070081923982242443995769161618110030333"
    "077221590123958532390267606629681958079084519212084506745189942342749961422738435719773520"
    "684938249973143398277065165557317381767638463874771088523568662864704492676537483930587768"
    "5546875e-251"
  },
  {
    1e-250,
    881,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000001000000000000000053"
    "999537253883899981203181499430805892246931626516793433910530085759720031238100112368002407"
    "447171329703552232353080088105938206674274715725109791162112189484059841360768927450369633"
    "021140546122430854461935345268108985798815672273304709482272738880744620602127696868089805"
    "350874783651571183064358760376926261607533744100444613427420311156985754264486372859918366"
    "280631462807772969081619558193405229308021079738637319224411130133261154325713463734072281"
    "776541971965885921112598907481993502349823795942566994909322454114604102165433157935624539"
    "7864401930228576603456024321655258579877312286043888889253139495849609375",
    631,
    "1.0000000000000000539995372538838999812031814994308058922469316265167934339105300857597200"
    "312381001123680024074471713297035522323530800881059382066742747157251097911621121894840598"
    "413607689274503696330211405461224308544619353452681089857988156722733047094822727388807446"
    "206021276968680898053508747836515711830643587603769262616075337441004446134274203111569857"
    "542644863728599183662806314628077729690816195581934052293080210797386373192244111301332611"
    "543257134637340722817765419719658859211125989074819935023498237959425669949093224541146041"
    "021654331579356245397864401930228576603456024321655258579877312286043888889253139495849609"
    "375e-250"
  },
  {
    1e-249,
    880,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000010000000000000000539"
    "995372538838999812031814994308058922469316265167934339105300857597200312381001123680024074"
    "471713297035522323530800881059382066742747157251097911621121894840598413607689274503696330"
    "211405461224308544619353452681089857988156722733047094822727388807446206021276968680898053"
    "508747836515711830643587603769262616075337441004446134274203111569857542644863728599183662"
    "806314628077729690816195581934052293080210797386373192244111301332611543257134637340722817"
    "765419719658859211125989074819935023498237959425669949093224541146041021654331579356245397"
    "864401930228576603456024321655258579877312286043888889253139495849609375",
    631,
    "1.0000000000000000539995372538838999812031814994308058922469316265167934339105300857597200"
    "312381001123680024074471713297035522323530800881059382066742747157251097911621121894840598"
    "413607689274503696330211405461224308544619353452681089857988156722733047094822727388807446"
    "206021276968680898053508747836515711830643587603769262616075337441004446134274203111569857"
    "542644863728599183662806314628077729690816195581934052293080210797386373192244111301332611"
    "543257134637340722817765419719658859211125989074819935023498237959425669949093224541146041"
    "021654331579356245397864401930228576603456024321655258579877312286043888889253139495849609"
    "375e-249"
  },
  {
    1e-248,
    876,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000099999999999999997956"
    "832950416318242122534275228707458502381648630896999234860610340310794424258705217009089869"
    "884827266742574554899060918518449584516531018017783472224120434382964509298862538007867011"
    "167215136488291483614163016401274802673991216449563456235119041495368187766651331209268596"
    "822124621651448355628349024608169361510425185931465252898513762884440053713780975011971591"
    "647787756942772620690104203449664476602519718970085399257243740322368884688891716469171969"
    "696185869032090376063669383679127291463103754083095784920300041108019751594548954275789259"
    "29868094239082518686308263637652071764705397072248160839080810546875",
    627,
    "9.9999999999999997956832950416318242122534275228707458502381648630896999234860610340310794"
    "424258705217009089869884827266742574554899060918518449584516531018017783472224120434382964"
    "509298862538007867011167215136488291483614163016401274802673991216449563456235119041495368"
    "187766651331209268596822124621651448355628349024608169361510425185931465252898513762884440"
    "053713780975011971591647787756942772620690104203449664476602519718970085399257243740322368"
    "884688891716469171969696185869032090376063669383679127291463103754083095784920300041108019"
    "75159454895427578925929868094239082518686308263637652071764705397072248160839080810546875e"
    "-249"
  },
  {
    1e-247,
    872,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000001000000000000000019264"
    "973637347565119880190084097064615542811227753142494514965560659967723973527350942310370031"
    "287236420026538511171664609586474954023659804141544823505729751952416435804786484233850330"
    "055295235575149303231611552864074838513650889193807852976016263375515864045811208624482972"
    "303132848544723293923634315865463211066647721250631674817230177186455858390376742012329444"
    "026456363211855740084250652581223188423001215800302116219740192577003768928675335028019472"
    "737919103955579681683207780899128613400508688423510947933375719626279995672246018953437762"
    "8286140100146030563490161247042209424762404523789882659912109375",
    625,
    "1.0000000000000000192649736373475651198801900840970646155428112277531424945149655606599677"
    "239735273509423103700312872364200265385111716646095864749540236598041415448235057297519524"
    "164358047864842338503300552952355751493032316115528640748385136508891938078529760162633755"
    "158640458112086244829723031328485447232939236343158654632110666477212506316748172301771864"
    "558583903767420123294440264563632118557400842506525812231884230012158003021162197401925770"
    "037689286753350280194727379191039555796816832077808991286134005086884235109479333757196262"
    "799956722460189534377628286140100146030563490161247042209424762404523789882659912109375e-2"
    "47"
  },
  {
    1e-246,
    870,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000009999999999999999557503"
    "430242525528020324343532010805667124196414424664910487904290489920764040228729067592158993"
    "848944158652697716819759305433083798503004057996086614824890989559822872989287175855619806"
    "422650424675741558876413307019519396756689296767305879762949059005814858389345741736077657"
    "721142372963397136994038042114318575268143421530650414858820998391673443862695549841477999"
    "487302931572484869911168636458977125865147740411851183840557296116998936650622624899398314"
    "962224421400982080087854703265594953693548522754259899647752259525480281094356570405654910"
    "81726672244441155999912507201798916867119260132312774658203125",
    623,
    "9.9999999999999995575034302425255280203243435320108056671241964144246649104879042904899207"
    "640402287290675921589938489441586526977168197593054330837985030040579960866148248909895598"
    "228729892871758556198064226504246757415588764133070195193967566892967673058797629490590058"
    "148583893457417360776577211423729633971369940380421143185752681434215306504148588209983916"
    "734438626955498414779994873029315724848699111686364589771258651477404118511838405572961169"
    "989366506226248993983149622244214009820800878547032655949536935485227542598996477522595254"
    "8028109435657040565491081726672244441155999912507201798916867119260132312774658203125e-247"
  },
  {
    1e-245,
    863,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000099999999999999993034449"
    "077901454787489333206084268694718026300691819608966232037640460181737622108169253875424662"
    "395761420076227588610045892604175018095664646283419667319283775740862789658561092624664087"
    "705296522454409695005324183710278014047614586989968197640636291060773455618392039325768315"
    "970679279698628160971159954981931611088099051403838815334286890025193878462668017287514231"
    "763986513540558575386334807176752225192019733753831924978194442557834355961703347470797499"
    "954377741390561853901654609753184815689331781619200677733502848305524108431151227541658238"
    "2971737136376395860271060200830106623470783233642578125",
    617,
    "9.9999999999999993034449077901454787489333206084268694718026300691819608966232037640460181"
    "737622108169253875424662395761420076227588610045892604175018095664646283419667319283775740"
    "862789658561092624664087705296522454409695005324183710278014047614586989968197640636291060"
    "773455618392039325768315970679279698628160971159954981931611088099051403838815334286890025"
    "193878462668017287514231763986513540558575386334807176752225192019733753831924978194442557"
    "834355961703347470797499954377741390561853901654609753184815689331781619200677733502848305"
    "5241084311512275416582382971737136376395860271060200830106623470783233642578125e-246"
  },
  {
    1e-244,
    862,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000999999999999999930344490"
    "779014547874893332060842686947180263006918196089662320376404601817376221081692538754246623"
    "957614200762275886100458926041750180956646462834196673192837757408627896585610926246640877"
    "052965224544096950053241837102780140476145869899681976406362910607734556183920393257683159"
    "706792796986281609711599549819316110880990514038388153342868900251938784626680172875142317"
    "639865135405585753863348071767522251920197337538319249781944425578343559617033474707974999"
    "543777413905618539016546097531848156893317816192006777335028483055241084311512275416582382"
    "971737136376395860271060200830106623470783233642578125",
    617,
    "9.9999999999999993034449077901454787489333206084268694718026300691819608966232037640460181"
    "737622108169253875424662395761420076227588610045892604175018095664646283419667319283775740"
    "862789658561092624664087705296522454409695005324183710278014047614586989968197640636291060"
    "773455618392039325768315970679279698628160971159954981931611088099051403838815334286890025"
    "193878462668017287514231763986513540558575386334807176752225192019733753831924978194442557"
    "834355961703347470797499954377741390561853901654609753184815689331781619200677733502848305"
    "5241084311512275416582382971737136376395860271060200830106623470783233642578125e-245"
  },
  {
    1e-243,
    860,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000009999999999999999953834725"
    "268238404883694339292801746131825839913003283172116837111742408804873936672009431360776919"
    "558264619014651235416662662443221344766703649768265849912664257571959665839639740939106759"
    "958829667010478302787493311166285505696724153868013361210329649405378400255940709538946474"
    "698507153310677593236434835474221356703707099466206846433001038753771248324396897331458532"
    "313608713234129212323479415408095084823136988741250335228345020495118295568197537015276310"
    "411591129586475816249921238426250207948460318310037371819340049567758686293228741977886517"
    "8437101302218181775316452331026084721088409423828125",
    616,
    "9.9999999999999999538347252682384048836943392928017461318258399130032831721168371117424088"
    "048739366720094313607769195582646190146512354166626624432213447667036497682658499126642575"
    "719596658396397409391067599588296670104783027874933111662855056967241538680133612103296494"
    "053784002559407095389464746985071533106775932364348354742213567037070994662068464330010387"
    "537712483243968973314585323136087132341292123234794154080950848231369887412503352283450204"
    "951182955681975370152763104115911295864758162499212384262502079484603183100373718193400495"
    "677586862932287419778865178437101302218181775316452331026084721088409423828125e-244"
  },
  {
    1e-242,
    850,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000099999999999999996936787982"
    "770012344297899318190517954678165559754747542619193837726638525524292463299758138334526475"
    "654155744578942856518333016329335306866080411977462027189495841776873858462275495500275641"
    "871586983826747818854633351108918653226179719195359223516494320741652648892459987541005236"
    "462754799315329947882591005617972575461863158332767212312762242600178875013588298994443899"
    "476257695628205428474799363149460585746715433980272002647847146104452158090524210410657844"
    "220643333743596458161371331831427523423474557540495324317179619616195490219863468530614425"
    "931746236386743476032279431819915771484375",
    607,
    "9.9999999999999996936787982770012344297899318190517954678165559754747542619193837726638525"
    "524292463299758138334526475654155744578942856518333016329335306866080411977462027189495841"
    "776873858462275495500275641871586983826747818854633351108918653226179719195359223516494320"
    "741652648892459987541005236462754799315329947882591005617972575461863158332767212312762242"
    "600178875013588298994443899476257695628205428474799363149460585746715433980272002647847146"
    "104452158090524210410657844220643333743596458161371331831427523423474557540495324317179619"
    "616195490219863468530614425931746236386743476032279431819915771484375e-243"
  },
  {
    1e-241,
    849,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000999999999999999969367879827"
    "700123442978993181905179546781655597547475426191938377266385255242924632997581383345264756"
    "541557445789428565183330163293353068660804119774620271894958417768738584622754955002756418"
    "715869838267478188546333511089186532261797191953592235164943207416526488924599875410052364"
    "627547993153299478825910056179725754618631583327672123127622426001788750135882989944438994"
    "762576956282054284747993631494605857467154339802720026478471461044521580905242104106578442"
    "206433337435964581613713318314275234234745575404953243171796196161954902198634685306144259"
    "31746236386743476032279431819915771484375",
    607,
    "9.9999999999999996936787982770012344297899318190517954678165559754747542619193837726638525"
    "524292463299758138334526475654155744578942856518333016329335306866080411977462027189495841"
    "776873858462275495500275641871586983826747818854633351108918653226179719195359223516494320"
    "741652648892459987541005236462754799315329947882591005617972575461863158332767212312762242"
    "600178875013588298994443899476257695628205428474799363149460585746715433980272002647847146"
    "104452158090524210410657844220643333743596458161371331831427523423474557540495324317179619"
    "616195490219863468530614425931746236386743476032279431819915771484375e-242"
  },
  {
    1e-240,
    848,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000009999999999999999693678798277"
    "001234429789931819051795467816555975474754261919383772663852552429246329975813833452647565"
    "415574457894285651833301632933530686608041197746202718949584177687385846227549550027564187"
    "158698382674781885463335110891865322617971919535922351649432074165264889245998754100523646"
    "275479931532994788259100561797257546186315833276721231276224260017887501358829899444389947"
    "625769562820542847479936314946058574671543398027200264784714610445215809052421041065784422"
    "064333374359645816137133183142752342347455754049532431717961961619549021986346853061442593"
    "1746236386743476032279431819915771484375",
    607,
    "9.9999999999999996936787982770012344297899318190517954678165559754747542619193837726638525"
    "524292463299758138334526475654155744578942856518333016329335306866080411977462027189495841"
    "776873858462275495500275641871586983826747818854633351108918653226179719195359223516494320"
    "741652648892459987541005236462754799315329947882591005617972575461863158332767212312762242"
    "600178875013588298994443899476257695628205428474799363149460585746715433980272002647847146"
    "104452158090524210410657844220643333743596458161371331831427523423474557540495324317179619"
    "616195490219863468530614425931746236386743476032279431819915771484375e-241"
  },
  {
    1e-239,
    845,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000100000000000000007592774752331"
    "086846089823848315315933875985829835916086780881526495296189624426979709455112253728656481"
    "252609623707518885743635118724171586796539025946776244048864006266446992438854796959500679"
    "229858821580035001781170337842162949568931804995119168036022628142673512275341288295391562"
    "164140925092700319869107630863676953914455937585140575410644264316534325227541009743170786"
    "919068405008530211738026804844700883860075238691610755277275140661505025108160714320988751"
    "660906591874799129168282589108905049857407833757225634180327963654552849951972843449508193"
    "6805860323147499002516269683837890625",
    606,
    "1.0000000000000000759277475233108684608982384831531593387598582983591608678088152649529618"
    "962442697970945511225372865648125260962370751888574363511872417158679653902594677624404886"
    "400626644699243885479695950067922985882158003500178117033784216294956893180499511916803602"
    "262814267351227534128829539156216414092509270031986910763086367695391445593758514057541064"
    "426431653432522754100974317078691906840500853021173802680484470088386007523869161075527727"
    "514066150502510816071432098875166090659187479912916828258910890504985740783375722563418032"
    "79636545528499519728434495081936805860323147499002516269683837890625e-239"
  },
  {
    1e-238,
    840,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000999999999999999990679853366822"
    "272446562842242154775505177296137709812514515313754803700583443193665816975331183669118195"
    "751175878957889918151400872130798102236373871589770004064462227523761683081673596124136331"
    "155588257142620840629149547033551708575617172864026468026611189506538164230582904632674826"
    "366676372824983700466260205507957602734178537307979652919229330064068759161473975037537383"
    "899701835660488221870958805374087741443622319559242693331719116940274774410004713904731268"
    "468483132521263549307219829637997487511275991477045805797612856873027458811693931814423841"
    "33106315857730805873870849609375",
    601,
    "9.9999999999999999067985336682227244656284224215477550517729613770981251451531375480370058"
    "344319366581697533118366911819575117587895788991815140087213079810223637387158977000406446"
    "222752376168308167359612413633115558825714262084062914954703355170857561717286402646802661"
    "118950653816423058290463267482636667637282498370046626020550795760273417853730797965291922"
    "933006406875916147397503753738389970183566048822187095880537408774144362231955924269333171"
    "911694027477441000471390473126846848313252126354930721982963799748751127599147704580579761"
    "285687302745881169393181442384133106315857730805873870849609375e-239"
  },
  {
    1e-237,
    839,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000009999999999999999906798533668222"
    "724465628422421547755051772961377098125145153137548037005834431936658169753311836691181957"
    "511758789578899181514008721307981022363738715897700040644622275237616830816735961241363311"
    "555882571426208406291495470335517085756171728640264680266111895065381642305829046326748263"
    "666763728249837004662602055079576027341785373079796529192293300640687591614739750375373838"
    "997018356604882218709588053740877414436223195592426933317191169402747744100047139047312684"
    "684831325212635493072198296379974875112759914770458057976128568730274588116939318144238413"
    "3106315857730805873870849609375",
    601,
    "9.9999999999999999067985336682227244656284224215477550517729613770981251451531375480370058"
    "344319366581697533118366911819575117587895788991815140087213079810223637387158977000406446"
    "222752376168308167359612413633115558825714262084062914954703355170857561717286402646802661"
    "118950653816423058290463267482636667637282498370046626020550795760273417853730797965291922"
    "933006406875916147397503753738389970183566048822187095880537408774144362231955924269333171"
    "911694027477441000471390473126846848313252126354930721982963799748751127599147704580579761"
    "285687302745881169393181442384133106315857730805873870849609375e-238"
  },
  {
    1e-236,
    835,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000100000000000000004523850562697497"
    "389573749583639374115867013592052539546062315472129922782363588238983462383764998428403048"
    "712490815296123929376907380178547230294435983168516337593604201381495751807319514549342628"
    "710823068356751402598399912192149232838573419981220392012484833546421768519409075826893534"
    "250541481027617933014251151039724203682227397577235767904585044888443475439309336980649448"
    "593045289236911499691672094075724362418556266763620337397978233213107947983026866003126727"
    "845611570636930442760770896667141459146949297798054884123959586342812486483601353004502243"
    "493334390223026275634765625",
    599,
    "1.0000000000000000452385056269749738957374958363937411586701359205253954606231547212992278"
    "236358823898346238376499842840304871249081529612392937690738017854723029443598316851633759"
    "360420138149575180731951454934262871082306835675140259839991219214923283857341998122039201"
    "248483354642176851940907582689353425054148102761793301425115103972420368222739757723576790"
    "458504488844347543930933698064944859304528923691149969167209407572436241855626676362033739"
    "797823321310794798302686600312672784561157063693044276077089666714145914694929779805488412"
    "3959586342812486483601353004502243493334390223026275634765625e-236"
  },
  {
    1e-235,
    829,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000999999999999999957944662010730651"
    "577058050085611396113081592268020462746850609174906384239327580431406386227303880018694909"
    "606461440847125465979951128205680196431578644620908477577938829729718419833836711322074076"
    "676273018052836591048875780529838323956036062555026490502994209182532157816192957318360981"
    "178947633808213147930821906493819152592295307304030063339417833179353805722504038175917547"
    "964665321359685995384056034086040135284373694206587306362717705160991368109381051551269182"
    "499342610200096234987102040793131263159890576484959971436813478787059179808682785051132668"
    "741047382354736328125",
    593,
    "9.9999999999999995794466201073065157705805008561139611308159226802046274685060917490638423"
    "932758043140638622730388001869490960646144084712546597995112820568019643157864462090847757"
    "793882972971841983383671132207407667627301805283659104887578052983832395603606255502649050"
    "299420918253215781619295731836098117894763380821314793082190649381915259229530730403006333"
    "941783317935380572250403817591754796466532135968599538405603408604013528437369420658730636"
    "271770516099136810938105155126918249934261020009623498710204079313126315989057648495997143"
    "6813478787059179808682785051132668741047382354736328125e-236"
  },
  {
    1e-234,
    828,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000009999999999999999579446620107306515"
    "770580500856113961130815922680204627468506091749063842393275804314063862273038800186949096"
    "064614408471254659799511282056801964315786446209084775779388297297184198338367113220740766"
    "762730180528365910488757805298383239560360625550264905029942091825321578161929573183609811"
    "789476338082131479308219064938191525922953073040300633394178331793538057225040381759175479"
    "646653213596859953840560340860401352843736942065873063627177051609913681093810515512691824"
    "993426102000962349871020407931312631598905764849599714368134787870591798086827850511326687"
    "41047382354736328125",
    593,
    "9.9999999999999995794466201073065157705805008561139611308159226802046274685060917490638423"
    "932758043140638622730388001869490960646144084712546597995112820568019643157864462090847757"
    "793882972971841983383671132207407667627301805283659104887578052983832395603606255502649050"
    "299420918253215781619295731836098117894763380821314793082190649381915259229530730403006333"
    "941783317935380572250403817591754796466532135968599538405603408604013528437369420658730636"
    "271770516099136810938105155126918249934261020009623498710204079313126315989057648495997143"
    "6813478787059179808682785051132668741047382354736328125e-235"
  },
  {
    1e-233,
    827,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000099999999999999995794466201073065157"
    "705805008561139611308159226802046274685060917490638423932758043140638622730388001869490960"
    "646144084712546597995112820568019643157864462090847757793882972971841983383671132207407667"
    "627301805283659104887578052983832395603606255502649050299420918253215781619295731836098117"
    "894763380821314793082190649381915259229530730403006333941783317935380572250403817591754796"
    "466532135968599538405603408604013528437369420658730636271770516099136810938105155126918249"
    "934261020009623498710204079313126315989057648495997143681347878705917980868278505113266874"
    "1047382354736328125",
    593,
    "9.9999999999999995794466201073065157705805008561139611308159226802046274685060917490638423"
    "932758043140638622730388001869490960646144084712546597995112820568019643157864462090847757"
    "793882972971841983383671132207407667627301805283659104887578052983832395603606255502649050"
    "299420918253215781619295731836098117894763380821314793082190649381915259229530730403006333"
    "941783317935380572250403817591754796466532135968599538405603408604013528437369420658730636"
    "271770516099136810938105155126918249934261020009623498710204079313126315989057648495997143"
    "6813478787059179808682785051132668741047382354736328125e-234"
  },
  {
    1e-232,
    823,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000001000000000000000024986333908006291117"
    "803864422212237108093593793144251071027924154536088112076356335479272712049688094472633140"
    "628515750764885721997341514960534233394596286256239516962075107182047281663988765672574288"
    "016505168108861079050506718628599358044231968538756452578178166866642842418468448402090677"
    "674425335611175869399522291647927680918924687705672201958082040855974301516644868200636322"
    "390175894927468561142682409524414760486825800532446292623338674019037911022734863907720194"
    "144348058048126919728159867652859301665220833572223445350350191397625909672797206440009176"
    "731109619140625",
    591,
    "1.0000000000000000249863339080062911178038644222122371080935937931442510710279241545360881"
    "120763563354792727120496880944726331406285157507648857219973415149605342333945962862562395"
    "169620751071820472816639887656725742880165051681088610790505067186285993580442319685387564"
    "525781781668666428424184684484020906776744253356111758693995222916479276809189246877056722"
    "019580820408559743015166448682006363223901758949274685611426824095244147604868258005324462"
    "926233386740190379110227348639077201941443480580481269197281598676528593016652208335722234"
    "45350350191397625909672797206440009176731109619140625e-232"
  },
  {
    1e-231,
    819,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000009999999999999999892307755627926166960"
    "727634426917885774205263130782306314666894987335793799436758533070665868519237873911805890"
    "727424688238618595726944166776874057594220647374331494830109058681126428227741290867088950"
    "866453732969837296239860846071191436398749479988812794667280418882508284315291790468322783"
    "534698543322276354914605328059143003196620830786808086032826480548177260432546323163225316"
    "035868094653495333631366369994657065598222837617049966736674670709473501471704305005000902"
    "500878671404512223275630279080080260520771809790223741306854424171213224781240569427609443"
    "66455078125",
    587,
    "9.9999999999999998923077556279261669607276344269178857742052631307823063146668949873357937"
    "994367585330706658685192378739118058907274246882386185957269441667768740575942206473743314"
    "948301090586811264282277412908670889508664537329698372962398608460711914363987494799888127"
    "946672804188825082843152917904683227835346985433222763549146053280591430031966208307868080"
    "860328264805481772604325463231632253160358680946534953336313663699946570655982228376170499"
    "667366746707094735014717043050050009025008786714045122232756302790800802605207718097902237"
    "4130685442417121322478124056942760944366455078125e-232"
  },
  {
    1e-230,
    817,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000100000000000000004643966891513449577084"
    "252500992450622649743428118386333476466494801759335135596462478259638716834667872150467156"
    "197971992950003945212977393024232997570081916752333744951362797040806496871754762409352377"
    "442104499598748870641909904148748668684618086268155644504824185367805081063201001524571726"
    "699862437854480974435934695313570927848225048186703797096167653666824680067900973122459686"
    "212934620473997781040987303938419284302791059489107019314139818789224754807352395205225648"
    "090417259273233816851797149977006703310702512814408808214904078592866198960109613835811614"
    "990234375",
    587,
    "1.0000000000000000464396689151344957708425250099245062264974342811838633347646649480175933"
    "513559646247825963871683466787215046715619797199295000394521297739302423299757008191675233"
    "374495136279704080649687175476240935237744210449959874887064190990414874866868461808626815"
    "564450482418536780508106320100152457172669986243785448097443593469531357092784822504818670"
    "379709616765366682468006790097312245968621293462047399778104098730393841928430279105948910"
    "701931413981878922475480735239520522564809041725927323381685179714997700670331070251281440"
    "8808214904078592866198960109613835811614990234375e-230"
  },
  {
    1e-229,
    813,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000001000000000000000069323226256071247400750"
    "429636817593286128197468426116416083855127731198939920880133372808307294915835253634304557"
    "674620371755311403903916831264299662212320939559412636114716811326233827094915039047375249"
    "531313675588992339392304815234823905634674008201667236376321839848939762273142402611092832"
    "410132397071679445518872612024272862010317443141529718033287929408455103648760697185422965"
    "546360900496155589319167455351587356310161328165499600930970634670206508434761332673122973"
    "230694773644784504899948936474883425518962787792773662734480133451597794191911816596984863"
    "28125",
    584,
    "1.0000000000000000693232262560712474007504296368175932861281974684261164160838551277311989"
    "399208801333728083072949158352536343045576746203717553114039039168312642996622123209395594"
    "126361147168113262338270949150390473752495313136755889923393923048152348239056346740082016"
    "672363763218398489397622731424026110928324101323970716794455188726120242728620103174431415"
    "297180332879294084551036487606971854229655463609004961555893191674553515873563101613281654"
    "996009309706346702065084347613326731229732306947736447845048999489364748834255189627877927"
    "7366273448013345159779419191181659698486328125e-229"
  },
  {
    1e-228,
    810,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000010000000000000000327095345105724447928977"
    "822337886539907189763688385114859731508401894299982170153196284692350924051848022268917645"
    "627796641468762810652881896291481637939181043016923375529746658571636536911271751212128893"
    "548837882265865266351755772390843555730849753694899702513938619755174396473305828264919277"
    "517195674286879236636315578025711283654103051023429227187097010241218188971591516481012000"
    "791373872862711430642963898037561350585601549264125484676547198254721718567815236797365855"
    "082592841848703666887850377471771976598625323548767337977771774149005068466067314147949218"
    "75",
    582,
    "1.0000000000000000327095345105724447928977822337886539907189763688385114859731508401894299"
    "982170153196284692350924051848022268917645627796641468762810652881896291481637939181043016"
    "923375529746658571636536911271751212128893548837882265865266351755772390843555730849753694"
    "899702513938619755174396473305828264919277517195674286879236636315578025711283654103051023"
    "429227187097010241218188971591516481012000791373872862711430642963898037561350585601549264"
    "125484676547198254721718567815236797365855082592841848703666887850377471771976598625323548"
    "76733797777177414900506846606731414794921875e-228"
  },
  {
    1e-227,
    807,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000099999999999999994483667432137531853405142"
    "846651919968173684572982825965370746055008918453812773976664205546180637962371884910106109"
    "436196588663198625257944970478456758975129968316362100479351673139523752203630169842322493"
    "145205855681257601806540604930943542527129657226453155156671507930386534538221534344975657"
    "152877628550827121105302767048696761763317380829461396372195290172193549331544235852896295"
    "780095558254847205260583248896120405471733915260362255569652419810976406962998209560925497"
    "44141094810764349819916808006822507980219193039241043496900829268270172178745269775390625",
    579,
    "9.9999999999999994483667432137531853405142846651919968173684572982825965370746055008918453"
    "812773976664205546180637962371884910106109436196588663198625257944970478456758975129968316"
    "362100479351673139523752203630169842322493145205855681257601806540604930943542527129657226"
    "453155156671507930386534538221534344975657152877628550827121105302767048696761763317380829"
    "461396372195290172193549331544235852896295780095558254847205260583248896120405471733915260"
    "362255569652419810976406962998209560925497441410948107643498199168080068225079802191930392"
    "41043496900829268270172178745269775390625e-228"
  },
  {
    1e-226,
    803,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000999999999999999921403911604256084865025734"
    "128580678532674944226092192498436609806062452415437266285845678455596772807429948356873502"
    "783913017233507635857119058287608601973485118222629925278543631190326543612068785679314418"
    "536930644872855853502693732036123385854315559671081231612809240313578864862650681305177590"
    "144565313993697223698752968597858084892605463215064962391886735748633252290453214643033058"
    "777907128222426449488350538349222453692588279587908979174338697479788659722904339841966832"
    "0553962267313865268467856149502649681977558236700698302001910633407533168792724609375",
    576,
    "9.9999999999999992140391160425608486502573412858067853267494422609219249843660980606245241"
    "543726628584567845559677280742994835687350278391301723350763585711905828760860197348511822"
    "262992527854363119032654361206878567931441853693064487285585350269373203612338585431555967"
    "108123161280924031357886486265068130517759014456531399369722369875296859785808489260546321"
    "506496239188673574863325229045321464303305877790712822242644948835053834922245369258827958"
    "790897917433869747978865972290433984196683205539622673138652684678561495026496819775582367"
    "00698302001910633407533168792724609375e-227"
  },
  {
    1e-225,
    799,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000009999999999999999588963319516468587354668450"
    "692823123711739866320698999468699709965052238117420238551198816655321437134921895475736493"
    "087976082710734226128480926827429824179884221282156525025005915181841090908414460695712392"
    "011353039764081168030334396734226489214851798206017435390585826980372336939541407365039603"
    "593028684170156034655924916204333372775148153423433645199926013059168379304358448605208972"
    "147846551440994144763216593283930153321896764130507016098354984877493155742287490696278598"
    "293374336834640550786179121214422959164173920765250613840180449187755584716796875",
    573,
    "9.9999999999999995889633195164685873546684506928231237117398663206989994686997099650522381"
    "174202385511988166553214371349218954757364930879760827107342261284809268274298241798842212"
    "821565250250059151818410909084144606957123920113530397640811680303343967342264892148517982"
    "060174353905858269803723369395414073650396035930286841701560346559249162043333727751481534"
    "234336451999260130591683793043584486052089721478465514409941447632165932839301533218967641"
    "305070160983549848774931557422874906962785982933743368346405507861791212144229591641739207"
    "65250613840180449187755584716796875e-226"
  },
  {
    1e-224,
    796,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000100000000000000001888420450747209692817262257"
    "440492651277245448163423186436334890121365804582963596595860680142873716319177545269388374"
    "861295393117868142201454771495799112919370837715281606083172804275621385687770269398215226"
    "386275854209173808357697189310146982895657205983456262105753051317062382403967582662615270"
    "288295549432501109253572845655374109336977874598880792496198619757057495440805320850143871"
    "378869821877615845707545289506591395555191133327745750663038010048636493634780383388550426"
    "764336480678810024954958759532602026627590152685343127814121544361114501953125",
    572,
    "1.0000000000000000188842045074720969281726225744049265127724544816342318643633489012136580"
    "458296359659586068014287371631917754526938837486129539311786814220145477149579911291937083"
    "771528160608317280427562138568777026939821522638627585420917380835769718931014698289565720"
    "598345626210575305131706238240396758266261527028829554943250110925357284565537410933697787"
    "459888079249619861975705749544080532085014387137886982187761584570754528950659139555519113"
    "332774575066303801004863649363478038338855042676433648067881002495495875953260202662759015"
    "2685343127814121544361114501953125e-224"
  },
  {
    1e-223,
    792,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000999999999999999970893906462811906374008000570"
    "306835199493680201982766330368646577446910658559546277287626692711462403432106728597696196"
    "760677403094474374681383689185984160229479378003085214166818823098530044048697394453421813"
    "680794889544841059142146117358413102979458268448307355458372261063911719971247754528398828"
    "018885832477484990981138987657418040685808023072453200986478284247585335230286530117005514"
    "585463759034763272472418041727595056862123397096052789194474810296725446652560022479388716"
    "99861990812886411280424721621904078638909396659073536284267902374267578125",
    568,
    "9.9999999999999997089390646281190637400800057030683519949368020198276633036864657744691065"
    "855954627728762669271146240343210672859769619676067740309447437468138368918598416022947937"
    "800308521416681882309853004404869739445342181368079488954484105914214611735841310297945826"
    "844830735545837226106391171997124775452839882801888583247748499098113898765741804068580802"
    "307245320098647828424758533523028653011700551458546375903476327247241804172759505686212339"
    "709605278919447481029672544665256002247938871699861990812886411280424721621904078638909396"
    "659073536284267902374267578125e-224"
  },
  {
    1e-222,
    790,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000010000000000000000476783833342682112606713957768"
    "637813007397190494251111847601702954737064781916897791611948666591020190475766871515962797"
    "243198480292056504144461304211953105722457766426545688306735745508241445751058736993905339"
    "719367336198762982378673585473038645428403346663157804170254644346510864807326698848050278"
    "013972914335267534684821378915349249801611797386207593472909455643687259147132155320986333"
    "106388946209955678372738070689052947657840949863003370919232746001486301649501207291735980"
    "302117459836419315967918227902079542079860630110488273203372955322265625",
    568,
    "1.0000000000000000476783833342682112606713957768637813007397190494251111847601702954737064"
    "781916897791611948666591020190475766871515962797243198480292056504144461304211953105722457"
    "766426545688306735745508241445751058736993905339719367336198762982378673585473038645428403"
    "346663157804170254644346510864807326698848050278013972914335267534684821378915349249801611"
    "797386207593472909455643687259147132155320986333106388946209955678372738070689052947657840"
    "949863003370919232746001486301649501207291735980302117459836419315967918227902079542079860"
    "630110488273203372955322265625e-222"
  },
  {
    1e-221,
    787,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000100000000000000001696459258568568930600603769424"
    "100286024130351044817324300356080826298815033883237841176759708004617280138870373003624653"
    "886287005531314012122115392711085043513921718682682696513167396990650436454248200100304585"
    "547999598986220259957886207174755991748750817911241043356418308635533987693870374224254788"
    "839270785111004847354487779788817126241991707215373600296588103765536964094254136606138582"
    "056884038650264969133150093238119960431981583020131793294388887877935675999408142925964561"
    "757501084173080407977398016174108708042922444292344152927398681640625",
    566,
    "1.0000000000000000169645925856856893060060376942410028602413035104481732430035608082629881"
    "503388323784117675970800461728013887037300362465388628700553131401212211539271108504351392"
    "171868268269651316739699065043645424820010030458554799959898622025995788620717475599174875"
    "081791124104335641830863553398769387037422425478883927078511100484735448777978881712624199"
    "170721537360029658810376553696409425413660613858205688403865026496913315009323811996043198"
    "158302013179329438888787793567599940814292596456175750108417308040797739801617410870804292"
    "2444292344152927398681640625e-221"
  },
  {
    1e-220,
    783,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000999999999999999992393559986819671742273751228142"
    "780107842571079266622889598273218494413488056546457812225781416801495804438316992788219990"
    "497287676199131886641172731843282325453969622164633472698153505172392196091768642293055362"
    "314605885850926088948064891302516217205246989349714446795158007718742593903530828192563957"
    "989040985176684477595069722970768288226906938980117327505829416284684621926002033231587828"
    "512796998908315174577656023161923475148392505322102605760380301683938036029249989328483687"
    "46562272820190206615970605896759337838375358842313289642333984375",
    562,
    "9.9999999999999999239355998681967174227375122814278010784257107926662288959827321849441348"
    "805654645781222578141680149580443831699278821999049728767619913188664117273184328232545396"
    "962216463347269815350517239219609176864229305536231460588585092608894806489130251621720524"
    "698934971444679515800771874259390353082819256395798904098517668447759506972297076828822690"
    "693898011732750582941628468462192600203323158782851279699890831517457765602316192347514839"
    "250532210260576038030168393803602924998932848368746562272820190206615970605896759337838375"
    "358842313289642333984375e-221"
  },
  {
    1e-219,
    777,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000010000000000000000317072121450052998442454095738999"
    "365116805429691571034550467333621241329477082039307714926864779929789995589357723850624678"
    "822194827815450619691426442713913009503657256241430605917862487469716656129100162290401513"
    "792300522689685059573403800145861376568648929700280256255981335372982467598074906725382466"
    "349079706700668711147626428386130469357231520579072082419120104777806523524649657592646158"
    "024664190592504013838078779127652818226698251288471292539940250366079943729802932183427756"
    "40643709848145287942544623405183301656506955623626708984375",
    558,
    "1.0000000000000000317072121450052998442454095738999365116805429691571034550467333621241329"
    "477082039307714926864779929789995589357723850624678822194827815450619691426442713913009503"
    "657256241430605917862487469716656129100162290401513792300522689685059573403800145861376568"
    "648929700280256255981335372982467598074906725382466349079706700668711147626428386130469357"
    "231520579072082419120104777806523524649657592646158024664190592504013838078779127652818226"
    "698251288471292539940250366079943729802932183427756406437098481452879425446234051833016565"
    "06955623626708984375e-219"
  },
  {
    1e-218,
    776,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000100000000000000003170721214500529984424540957389993"
    "651168054296915710345504673336212413294770820393077149268647799297899955893577238506246788"
    "221948278154506196914264427139130095036572562414306059178624874697166561291001622904015137"
    "923005226896850595734038001458613765686489297002802562559813353729824675980749067253824663"
    "490797067006687111476264283861304693572315205790720824191201047778065235246496575926461580"
    "246641905925040138380787791276528182266982512884712925399402503660799437298029321834277564"
    "0643709848145287942544623405183301656506955623626708984375",
    558,
    "1.0000000000000000317072121450052998442454095738999365116805429691571034550467333621241329"
    "477082039307714926864779929789995589357723850624678822194827815450619691426442713913009503"
    "657256241430605917862487469716656129100162290401513792300522689685059573403800145861376568"
    "648929700280256255981335372982467598074906725382466349079706700668711147626428386130469357"
    "231520579072082419120104777806523524649657592646158024664190592504013838078779127652818226"
    "698251288471292539940250366079943729802932183427756406437098481452879425446234051833016565"
    "06955623626708984375e-218"
  },
  {
    1e-217,
    773,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000001000000000000000082028686907482903814769132256469096"
    "708593146988216918578820762345970173856062325496159354324956318077489313327810269020838934"
    "932195207033926388944132179370789585752738052315333095636160522433386599970974867100681381"
    "948985284062799729213005566035635834935809604029406528561494585049482415841618646905336101"
    "617712101596334819969580269454338083007904797478255584097731044943571298337519394690903533"
    "243276852447491695682251993842794365744794481480119325819590928025541052955082197362408504"
    "6705663553366118245779858852984034456312656402587890625",
    556,
    "1.0000000000000000820286869074829038147691322564690967085931469882169185788207623459701738"
    "560623254961593543249563180774893133278102690208389349321952070339263889441321793707895857"
    "527380523153330956361605224333865999709748671006813819489852840627997292130055660356358349"
    "358096040294065285614945850494824158416186469053361016177121015963348199695802694543380830"
    "079047974782555840977310449435712983375193946909035332432768524474916956822519938427943657"
    "447944814801193258195909280255410529550821973624085046705663553366118245779858852984034456"
    "312656402587890625e-217"
  },
  {
    1e-216,
    768,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000010000000000000000417715070975008206383501541104137685"
    "510630637729690664798015391588933411293790282438490650141736579986975098141799618541420927"
    "620252666428348531029418529871986774431281097775150925562311020640098103222079566522573797"
    "738388719873647117149051248760372924790762968283018061908057468484938910143162674116645282"
    "499189563727638558040303247813051651801026058214177103491545912132361416394764863498733486"
    "217906178898194461827527289807843312848189993737272683591382148915037089752510141467022134"
    "49081149583552718951295901206322014331817626953125",
    552,
    "1.0000000000000000417715070975008206383501541104137685510630637729690664798015391588933411"
    "293790282438490650141736579986975098141799618541420927620252666428348531029418529871986774"
    "431281097775150925562311020640098103222079566522573797738388719873647117149051248760372924"
    "790762968283018061908057468484938910143162674116645282499189563727638558040303247813051651"
    "801026058214177103491545912132361416394764863498733486217906178898194461827527289807843312"
    "848189993737272683591382148915037089752510141467022134490811495835527189512959012063220143"
    "31817626953125e-216"
  },
  {
    1e-215,
    767,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000100000000000000004177150709750082063835015411041376855"
    "106306377296906647980153915889334112937902824384906501417365799869750981417996185414209276"
    "202526664283485310294185298719867744312810977751509255623110206400981032220795665225737977"
    "383887198736471171490512487603729247907629682830180619080574684849389101431626741166452824"
    "991895637276385580403032478130516518010260582141771034915459121323614163947648634987334862"
    "179061788981944618275272898078433128481899937372726835913821489150370897525101414670221344"
    "9081149583552718951295901206322014331817626953125",
    552,
    "1.0000000000000000417715070975008206383501541104137685510630637729690664798015391588933411"
    "293790282438490650141736579986975098141799618541420927620252666428348531029418529871986774"
    "431281097775150925562311020640098103222079566522573797738388719873647117149051248760372924"
    "790762968283018061908057468484938910143162674116645282499189563727638558040303247813051651"
    "801026058214177103491545912132361416394764863498733486217906178898194461827527289807843312"
    "848189993737272683591382148915037089752510141467022134490811495835527189512959012063220143"
    "31817626953125e-215"
  },
  {
    1e-214,
    763,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000999999999999999912948531705558154473809424043036718446"
    "966797484175939762940024960247476403992477036456139219669145746563738570562978920712197817"
    "481457391341938411132808559707770852376293656497482700456956882004083446153843217300572813"
    "370353345972655720983713165321956617529713784766694604601464605330611566948653031915493472"
    "980891657336770474270501827599828131135592519536514353709939276163640205739179658576757833"
    "034667305268247784355081422352221012897456633272684485689532862584208239791227856442081540"
    "328491173763580945887952111661434173583984375",
    548,
    "9.9999999999999991294853170555815447380942404303671844696679748417593976294002496024747640"
    "399247703645613921966914574656373857056297892071219781748145739134193841113280855970777085"
    "237629365649748270045695688200408344615384321730057281337035334597265572098371316532195661"
    "752971378476669460460146460533061156694865303191549347298089165733677047427050182759982813"
    "113559251953651435370993927616364020573917965857675783303466730526824778435508142235222101"
    "289745663327268448568953286258420823979122785644208154032849117376358094588795211166143417"
    "3583984375e-215"
  },
  {
    1e-213,
    758,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000009999999999999999541718838309798076464624576645973744802"
    "776026965897403123357095038141531161161734228218754739105896672465453685204134594097641997"
    "354763518196711125117027765048609614168748152231178543046833402459160464911595164867510407"
    "202793112181136390385649127508640932246203586979303121868349231428609901062896334351846016"
    "010723662734377797936451727855359868050367761384970722522278960268406645351177997916668854"
    "365714923246312718423286410504962999123503102181513251931111118384484751383594693253237511"
    "2934186489610510761849582195281982421875",
    544,
    "9.9999999999999995417188383097980764646245766459737448027760269658974031233570950381415311"
    "611617342282187547391058966724654536852041345940976419973547635181967111251170277650486096"
    "141687481522311785430468334024591604649115951648675104072027931121811363903856491275086409"
    "322462035869793031218683492314286099010628963343518460160107236627343777979364517278553598"
    "680503677613849707225222789602684066453511779979166688543657149232463127184232864105049629"
    "991235031021815132519311111183844847513835946932532375112934186489610510761849582195281982"
    "421875e-214"
  },
  {
    1e-212,
    757,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000099999999999999995417188383097980764646245766459737448027"
    "760269658974031233570950381415311611617342282187547391058966724654536852041345940976419973"
    "547635181967111251170277650486096141687481522311785430468334024591604649115951648675104072"
    "027931121811363903856491275086409322462035869793031218683492314286099010628963343518460160"
    "107236627343777979364517278553598680503677613849707225222789602684066453511779979166688543"
    "657149232463127184232864105049629991235031021815132519311111183844847513835946932532375112"
    "934186489610510761849582195281982421875",
    544,
    "9.9999999999999995417188383097980764646245766459737448027760269658974031233570950381415311"
    "611617342282187547391058966724654536852041345940976419973547635181967111251170277650486096"
    "141687481522311785430468334024591604649115951648675104072027931121811363903856491275086409"
    "322462035869793031218683492314286099010628963343518460160107236627343777979364517278553598"
    "680503677613849707225222789602684066453511779979166688543657149232463127184232864105049629"
    "991235031021815132519311111183844847513835946932532375112934186489610510761849582195281982"
    "421875e-213"
  },
  {
    1e-211,
    753,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000001000000000000000086086610632329097798952165253591473786872"
    "179376313902070401900043227518594912001859192231487483210213431527121984203983241976622948"
    "337025348415756924164270255549310346734523145150346617408006619780367570571673882521368240"
    "042400003578976814090504523368015448321395277884576460019940142059144210726758298196213185"
    "650634870773157467703877379801124947258397264841771587551479589082132682119851679375853122"
    "664890905058431801519740884977218360010076443645211604561509452017228249180630551698825692"
    "06407652018242515623569488525390625",
    542,
    "1.0000000000000000860866106323290977989521652535914737868721793763139020704019000432275185"
    "949120018591922314874832102134315271219842039832419766229483370253484157569241642702555493"
    "103467345231451503466174080066197803675705716738825213682400424000035789768140905045233680"
    "154483213952778845764600199401420591442107267582981962131856506348707731574677038773798011"
    "249472583972648417715875514795890821326821198516793758531226648909050584318015197408849772"
    "183600100764436452116045615094520172282491806305516988256920640765201824251562356948852539"
    "0625e-211"
  },
  {
    1e-210,
    747,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000010000000000000000438738980558973249501554588251133620087619"
    "148388021703078207190706152416416973367595537175631399716386523329608757910156156686475202"
    "216098192174707121765922553290386891794166100999490773361133801437848251597835158748634337"
    "182115922300687259223151561667603367370635722992118926007347023157348973068783420324974785"
    "855889196258366120050919096362807417474785044114678002479328491648628750791950753089834631"
    "150033593217406145785889379433244567589512514871679528973822156752272537178589592588018319"
    "92968800477683544158935546875",
    537,
    "1.0000000000000000438738980558973249501554588251133620087619148388021703078207190706152416"
    "416973367595537175631399716386523329608757910156156686475202216098192174707121765922553290"
    "386891794166100999490773361133801437848251597835158748634337182115922300687259223151561667"
    "603367370635722992118926007347023157348973068783420324974785855889196258366120050919096362"
    "807417474785044114678002479328491648628750791950753089834631150033593217406145785889379433"
    "24456758951251487167952897382215675227253717858959258801831992968800477683544158935546875e"
    "-210"
  },
  {
    1e-209,
    746,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000100000000000000004387389805589732495015545882511336200876191"
    "483880217030782071907061524164169733675955371756313997163865233296087579101561566864752022"
    "160981921747071217659225532903868917941661009994907733611338014378482515978351587486343371"
    "821159223006872592231515616676033673706357229921189260073470231573489730687834203249747858"
    "558891962583661200509190963628074174747850441146780024793284916486287507919507530898346311"
    "500335932174061457858893794332445675895125148716795289738221567522725371785895925880183199"
    "2968800477683544158935546875",
    537,
    "1.0000000000000000438738980558973249501554588251133620087619148388021703078207190706152416"
    "416973367595537175631399716386523329608757910156156686475202216098192174707121765922553290"
    "386891794166100999490773361133801437848251597835158748634337182115922300687259223151561667"
    "603367370635722992118926007347023157348973068783420324974785855889196258366120050919096362"
    "807417474785044114678002479328491648628750791950753089834631150033593217406145785889379433"
    "24456758951251487167952897382215675227253717858959258801831992968800477683544158935546875e"
    "-209"
  },
  {
    1e-208,
    740,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000001000000000000000097906170153729994196615243053565345084743053"
    "446817186963924630715558956141812108087091015386299317014369701487094559614177342856068209"
    "341696591277063520820095610986410849952974964457928628136726878610739287003185182389585813"
    "172758756671078777597546184366879565008155448478538897317665187298818484324685922053583628"
    "847737094407307299537311447281324801454517762256647996472676258968228091235528514576627338"
    "859417864705333863263430146708652920391497449463827027465078192988527910206597582032372883"
    "9866816997528076171875",
    532,
    "1.0000000000000000979061701537299941966152430535653450847430534468171869639246307155589561"
    "418121080870910153862993170143697014870945596141773428560682093416965912770635208200956109"
    "864108499529749644579286281367268786107392870031851823895858131727587566710787775975461843"
    "668795650081554484785388973176651872988184843246859220535836288477370944073072995373114472"
    "813248014545177622566479964726762589682280912355285145766273388594178647053338632634301467"
    "086529203914974494638270274650781929885279102065975820323728839866816997528076171875e-208"
  },
  {
    1e-207,
    740,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000009999999999999999250028994406654526079439335225189992416034099"
    "011691336643921134517390697414448398389716623521894118120741222031945000987799853887146485"
    "996889950967392192910067087537015042366073980296044936620173271678140799002433983058991092"
    "970258715435496406938981280259425155854893708252707482521839982942707164963854754740474904"
    "195211949810823573120256520794590287312750397323352011452295578310984527060782566785018225"
    "200305272182321523050550958792252037827103701170298111999181361524504946941549476946420327"
    "2946178913116455078125",
    532,
    "9.9999999999999992500289944066545260794393352251899924160340990116913366439211345173906974"
    "144483983897166235218941181207412220319450009877998538871464859968899509673921929100670875"
    "370150423660739802960449366201732716781407990024339830589910929702587154354964069389812802"
    "594251558548937082527074825218399829427071649638547547404749041952119498108235731202565207"
    "945902873127503973233520114522955783109845270607825667850182252003052721823215230505509587"
    "922520378271037011702981119991813615245049469415494769464203272946178913116455078125e-208"
  },
  {
    1e-206,
    737,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000100000000000000002874486186850417756114671924114680674748719602"
    "855796564411162381003100158166520078784327417265535493345146977353453580801839986912678504"
    "489355280493380020846005009332711166642793788659897434684305803356920416200846875611113162"
    "246560262006712283608696183050474523908901741723163769147271169699937719336574342176917347"
    "645073463681732264719712920057849236522067324692287834169757851337623582374841141737713232"
    "366292971049317888008012637688183374798261772510814095901417025409694400162052829728054348"
    "3793735504150390625",
    531,
    "1.0000000000000000287448618685041775611467192411468067474871960285579656441116238100310015"
    "816652007878432741726553549334514697735345358080183998691267850448935528049338002084600500"
    "933271116664279378865989743468430580335692041620084687561111316224656026200671228360869618"
    "305047452390890174172316376914727116969993771933657434217691734764507346368173226471971292"
    "005784923652206732469228783416975785133762358237484114173771323236629297104931788800801263"
    "76881833747982617725108140959014170254096944001620528297280543483793735504150390625e-206"
  },
  {
    1e-205,
    731,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000001000000000000000001080338554413850906959309716179391412584853061"
    "254277116186421047819819757606437868144177687197770101084177088110526285554822674350215326"
    "172337416081911963805825736093616351809127258067112830889529802701171025537783302721259002"
    "348340999662460931503272815954817331462444992708733840995721456271734340837671969043391327"
    "936190728621331891151401968279968729501837643032831089306106331435493659036370153677049709"
    "360955712556905126740118244173399090576685029620586356588561160324557519629706803243607282"
    "6385498046875",
    526,
    "1.0000000000000000010803385544138509069593097161793914125848530612542771161864210478198197"
    "576064378681441776871977701010841770881105262855548226743502153261723374160819119638058257"
    "360936163518091272580671128308895298027011710255377833027212590023483409996624609315032728"
    "159548173314624449927087338409957214562717343408376719690433913279361907286213318911514019"
    "682799687295018376430328310893061063314354936590363701536770497093609557125569051267401182"
    "441733990905766850296205863565885611603245575196297068032436072826385498046875e-205"
  },
  {
    1e-204,
    730,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000010000000000000000010803385544138509069593097161793914125848530612"
    "542771161864210478198197576064378681441776871977701010841770881105262855548226743502153261"
    "723374160819119638058257360936163518091272580671128308895298027011710255377833027212590023"
    "483409996624609315032728159548173314624449927087338409957214562717343408376719690433913279"
    "361907286213318911514019682799687295018376430328310893061063314354936590363701536770497093"
    "609557125569051267401182441733990905766850296205863565885611603245575196297068032436072826"
    "385498046875",
    526,
    "1.0000000000000000010803385544138509069593097161793914125848530612542771161864210478198197"
    "576064378681441776871977701010841770881105262855548226743502153261723374160819119638058257"
    "360936163518091272580671128308895298027011710255377833027212590023483409996624609315032728"
    "159548173314624449927087338409957214562717343408376719690433913279361907286213318911514019"
    "682799687295018376430328310893061063314354936590363701536770497093609557125569051267401182"
    "441733990905766850296205863565885611603245575196297068032436072826385498046875e-204"
  },
  {
    1e-203,
    723,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000100000000000000003649092839644946902431919390813768304125985205940"
    "299843193068058345013249240165440535902118858347868651431172545325847430820148366422456613"
    "549311381232891696323291335249035452120486258789557131004593821225344022026068306029595609"
    "843587378042816937039475457872505322445769609805076960626896440311719207360342853239247803"
    "480693111220005888993282562207898322194721601209157236719072431964362986778297121315545566"
    "748242991533553101532865404019545205627887984465624769334080035321221657795831561088562011"
    "71875",
    520,
    "1.0000000000000000364909283964494690243191939081376830412598520594029984319306805834501324"
    "924016544053590211885834786865143117254532584743082014836642245661354931138123289169632329"
    "133524903545212048625878955713100459382122534402202606830602959560984358737804281693703947"
    "545787250532244576960980507696062689644031171920736034285323924780348069311122000588899328"
    "256220789832219472160120915723671907243196436298677829712131554556674824299153355310153286"
    "540401954520562788798446562476933408003532122165779583156108856201171875e-203"
  },
  {
    1e-202,
    722,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000001000000000000000036490928396449469024319193908137683041259852059402"
    "998431930680583450132492401654405359021188583478686514311725453258474308201483664224566135"
    "493113812328916963232913352490354521204862587895571310045938212253440220260683060295956098"
    "435873780428169370394754578725053224457696098050769606268964403117192073603428532392478034"
    "806931112200058889932825622078983221947216012091572367190724319643629867782971213155455667"
    "482429915335531015328654040195452056278879844656247693340800353212216577958315610885620117"
    "1875",
    520,
    "1.0000000000000000364909283964494690243191939081376830412598520594029984319306805834501324"
    "924016544053590211885834786865143117254532584743082014836642245661354931138123289169632329"
    "133524903545212048625878955713100459382122534402202606830602959560984358737804281693703947"
    "545787250532244576960980507696062689644031171920736034285323924780348069311122000588899328"
    "256220789832219472160120915723671907243196436298677829712131554556674824299153355310153286"
    "540401954520562788798446562476933408003532122165779583156108856201171875e-202"
  },
  {
    1e-201,
    720,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000009999999999999999458398184008382866438778903767244564718518546241422"
    "718636253761722365318913259000700890218250360647078131670538558640710995517318203609118298"
    "145276224615168802705395697729075782861950146917558335246313038824586331185893923613544981"
    "929960384320404305625917015212855137051754213994323632673435867770929096188922405495337823"
    "494527355775494792938308262767336984667091851847357308146785362197045393661583207247451227"
    "740334777536960707900047811967666685186232710373264651049218798561923904344439506530761718"
    "75",
    518,
    "9.9999999999999994583981840083828664387789037672445647185185462414227186362537617223653189"
    "132590007008902182503606470781316705385586407109955173182036091182981452762246151688027053"
    "956977290757828619501469175583352463130388245863311858939236135449819299603843204043056259"
    "170152128551370517542139943236326734358677709290961889224054953378234945273557754947929383"
    "082627673369846670918518473573081467853621970453936615832072474512277403347775369607079000"
    "4781196766668518623271037326465104921879856192390434443950653076171875e-202"
  },
  {
    1e-200,
    715,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000099999999999999998210026239908275959605441178928974709961505359824656"
    "249094749793672197213175620180419702157045503029929362492249482183238301163255790637355208"
    "596209840847691345548908285988635545366204397328202413315406723085126797542685953519513829"
    "014713523049200649545685240279259800618369205996726046799191331313257521270675728671148333"
    "244408622655324354942874459763350785891191594747038536509684958927467073288347769702934065"
    "739205278643004860546448479624082362272590048489495639927326919860206544399261474609375",
    514,
    "9.9999999999999998210026239908275959605441178928974709961505359824656249094749793672197213"
    "175620180419702157045503029929362492249482183238301163255790637355208596209840847691345548"
    "908285988635545366204397328202413315406723085126797542685953519513829014713523049200649545"
    "685240279259800618369205996726046799191331313257521270675728671148333244408622655324354942"
    "874459763350785891191594747038536509684958927467073288347769702934065739205278643004860546"
    "448479624082362272590048489495639927326919860206544399261474609375e-201"
  },
  {
    1e-199,
    714,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000999999999999999982100262399082759596054411789289747099615053598246562"
    "490947497936721972131756201804197021570455030299293624922494821832383011632557906373552085"
    "962098408476913455489082859886355453662043973282024133154067230851267975426859535195138290"
    "147135230492006495456852402792598006183692059967260467991913313132575212706757286711483332"
    "444086226553243549428744597633507858911915947470385365096849589274670732883477697029340657"
    "39205278643004860546448479624082362272590048489495639927326919860206544399261474609375",
    514,
    "9.9999999999999998210026239908275959605441178928974709961505359824656249094749793672197213"
    "175620180419702157045503029929362492249482183238301163255790637355208596209840847691345548"
    "908285988635545366204397328202413315406723085126797542685953519513829014713523049200649545"
    "685240279259800618369205996726046799191331313257521270675728671148333244408622655324354942"
    "874459763350785891191594747038536509684958927467073288347769702934065739205278643004860546"
    "448479624082362272590048489495639927326919860206544399261474609375e-200"
  },
  {
    1e-198,
    709,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000009999999999999999124802099224533715278754906771643890943097115679663244"
    "864890241489099268701300224747096620592506163636511458147080229307187686231418190870453248"
    "079045903136497403860177328871032921253477527517381647903616019374090502989225614211093036"
    "170293774649807043557627102989961482478123917402578427471263639364172725828851513302974451"
    "006929804660161786807414215058738258826728830198486282936879197000185087711763102436423213"
    "435887235808111997818538852504458228488519455634531141186016611754894256591796875",
    510,
    "9.9999999999999991248020992245337152787549067716438909430971156796632448648902414890992687"
    "013002247470966205925061636365114581470802293071876862314181908704532480790459031364974038"
    "601773288710329212534775275173816479036160193740905029892256142110930361702937746498070435"
    "576271029899614824781239174025784274712636393641727258288515133029744510069298046601617868"
    "074142150587382588267288301984862829368791970001850877117631024364232134358872358081119978"
    "18538852504458228488519455634531141186016611754894256591796875e-199"
  },
  {
    1e-197,
    706,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000099999999999999998674159923085805213393300653009810429996874306693191169"
    "124472952257610848253128042616284553786865789500312352968060842582729449985231219265253670"
    "571132968779770316262053501963893109782372131737653104498093944552523710205533344674022258"
    "247562069380821486359171562550479671275070451572730967489910974565240871501542907022905826"
    "697910962572537414527814270868346111386548510042114755039370057964754782429778948172054646"
    "195039061999776584332685697351547605103705418372328495024703443050384521484375",
    508,
    "9.9999999999999998674159923085805213393300653009810429996874306693191169124472952257610848"
    "253128042616284553786865789500312352968060842582729449985231219265253670571132968779770316"
    "262053501963893109782372131737653104498093944552523710205533344674022258247562069380821486"
    "359171562550479671275070451572730967489910974565240871501542907022905826697910962572537414"
    "527814270868346111386548510042114755039370057964754782429778948172054646195039061999776584"
    "332685697351547605103705418372328495024703443050384521484375e-198"
  },
  {
    1e-196,
    703,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000001000000000000000046150710677581796618779019212445076464495968266104381455"
    "049293821509053772452286787325392320763091120084705701658676821914115201220706677138306223"
    "956721187116073383902776725667442275804496169887224048676409452018186544561551067244957754"
    "832615276870223269854919886711715484701354736102883217117306393040517620719651262174348800"
    "008012953492730516907519670931169298819566764879162955758325283350779066794972872183126556"
    "639724251347018692505234351971198612785141079939421615563333034515380859375",
    507,
    "1.0000000000000000461507106775817966187790192124450764644959682661043814550492938215090537"
    "724522867873253923207630911200847057016586768219141152012207066771383062239567211871160733"
    "839027767256674422758044961698872240486764094520181865445615510672449577548326152768702232"
    "698549198867117154847013547361028832171173063930405176207196512621743488000080129534927305"
    "169075196709311692988195667648791629557583252833507790667949728721831265566397242513470186"
    "92505234351971198612785141079939421615563333034515380859375e-196"
  },
  {
    1e-195,
    699,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000010000000000000000936779998349607922066558293583226541961177484254423572660"
    "929452606554100043890918762554297470786377001499714392411315387835717623154222647269218385"
    "530343865707695609285700904902512181891160518957784516327854572125460985665251636487458927"
    "182109433198299948654832956772505022618749124033420508918637109510047452830290157305812264"
    "311356157066156142110212407293358467828320964455752800500250463133640607927195845531906323"
    "91191156426420971847936254735776662183610596912330947816371917724609375",
    504,
    "1.0000000000000000936779998349607922066558293583226541961177484254423572660929452606554100"
    "043890918762554297470786377001499714392411315387835717623154222647269218385530343865707695"
    "609285700904902512181891160518957784516327854572125460985665251636487458927182109433198299"
    "948654832956772505022618749124033420508918637109510047452830290157305812264311356157066156"
    "142110212407293358467828320964455752800500250463133640607927195845531906323911911564264209"
    "71847936254735776662183610596912330947816371917724609375e-195"
  },
  {
    1e-194,
    697,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000100000000000000001763433718315439926605293312491852982552290017050159596842"
    "310295802124003329020373396736986497376317204554625910920399179244126456387732458513685519"
    "893326744325567768730070677375691037372424068209140690258384890157081215856660940268487210"
    "125787700045923484858184133239447416504263032260791685257200229422534598162461004060934415"
    "413935616439945852541872905226937004160756593931556118330542557322807039632484476108811118"
    "884410829937732489961321031245178314605936975567601621150970458984375",
    503,
    "1.0000000000000000176343371831543992660529331249185298255229001705015959684231029580212400"
    "332902037339673698649737631720455462591092039917924412645638773245851368551989332674432556"
    "776873007067737569103737242406820914069025838489015708121585666094026848721012578770004592"
    "348485818413323944741650426303226079168525720022942253459816246100406093441541393561643994"
    "585254187290522693700416075659393155611833054255732280703963248447610881111888441082993773"
    "2489961321031245178314605936975567601621150970458984375e-194"
  },
  {
    1e-193,
    693,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000001000000000000000048051802243876956442294091618280179573760839472477900487491"
    "039879074908021729758990882593817815712983287316331161975010588893463664495300641850848540"
    "573715094261230983808460260354633499880965167566224794664492225960926721750031101109280348"
    "039103528207538855342423070336885403775543154901570468288685756937105702186372316598097064"
    "937859981285920799659733723095960738097378141819448729993273869282466554882740677929119669"
    "78292755019478367894242808178173476107986061833798885345458984375",
    500,
    "1.0000000000000000480518022438769564422940916182801795737608394724779004874910398790749080"
    "217297589908825938178157129832873163311619750105888934636644953006418508485405737150942612"
    "309838084602603546334998809651675662247946644922259609267217500311011092803480391035282075"
    "388553424230703368854037755431549015704682886857569371057021863723165980970649378599812859"
    "207996597337230959607380973781418194487299932738692824665548827406779291196697829275501947"
    "8367894242808178173476107986061833798885345458984375e-193"
  },
  {
    1e-192,
    689,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000010000000000000000967197463410330479242799452076588191709415423556399877179997"
    "389527607768032330474019469521423628326812741484464464086406632169822254840623325932378871"
    "984313358701162582208658389109905017317243443259334219935215449851100228435058185883335428"
    "890659726048252661593538510447433857482036865714162534353792972759212550851919581801017222"
    "154660883042604384453411964185058524810776658256688046938311429695004085753741448747332392"
    "8503835150271772586917651270965734511264599859714508056640625",
    497,
    "1.0000000000000000967197463410330479242799452076588191709415423556399877179997389527607768"
    "032330474019469521423628326812741484464464086406632169822254840623325932378871984313358701"
    "162582208658389109905017317243443259334219935215449851100228435058185883335428890659726048"
    "252661593538510447433857482036865714162534353792972759212550851919581801017222154660883042"
    "604384453411964185058524810776658256688046938311429695004085753741448747332392850383515027"
    "1772586917651270965734511264599859714508056640625e-192"
  },
  {
    1e-191,
    684,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000100000000000000001885103578558330155310257946465299581545241774258064814918582"
    "043486338675282778594424397882308744116449521706199131483254429935252790204362740541493259"
    "888534929589981916101691322081929877050966151039961826707463454641674109394627062184843112"
    "912606156916700885226460191217061459194683589966299720066963273381637044708053164889427057"
    "129631707491701638836923910243366946715842741571668517293950507024624266716059776175152808"
    "16610694100232507863790232249812106601893901824951171875",
    493,
    "1.0000000000000000188510357855833015531025794646529958154524177425806481491858204348633867"
    "528277859442439788230874411644952170619913148325442993525279020436274054149325988853492958"
    "998191610169132208192987705096615103996182670746345464167410939462706218484311291260615691"
    "670088522646019121706145919468358996629972006696327338163704470805316488942705712963170749"
    "170163883692391024336694671584274157166851729395050702462426671605977617515280816610694100"
    "232507863790232249812106601893901824951171875e-191"
  },
  {
    1e-190,
    683,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000001000000000000000018851035785583301553102579464652995815452417742580648149185820"
    "434863386752827785944243978823087441164495217061991314832544299352527902043627405414932598"
    "885349295899819161016913220819298770509661510399618267074634546416741093946270621848431129"
    "126061569167008852264601912170614591946835899662997200669632733816370447080531648894270571"
    "296317074917016388369239102433669467158427415716685172939505070246242667160597761751528081"
    "6610694100232507863790232249812106601893901824951171875",
    493,
    "1.0000000000000000188510357855833015531025794646529958154524177425806481491858204348633867"
    "528277859442439788230874411644952170619913148325442993525279020436274054149325988853492958"
    "998191610169132208192987705096615103996182670746345464167410939462706218484311291260615691"
    "670088522646019121706145919468358996629972006696327338163704470805316488942705712963170749"
    "170163883692391024336694671584274157166851729395050702462426671605977617515280816610694100"
    "232507863790232249812106601893901824951171875e-190"
  },
  {
    1e-189,
    680,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000010000000000000000686870105410711392306560935401767227629654574949386254732267282"
    "863177163850871532771738817474236917352337331480425748697404066355343545355987256216235425"
    "947807033983401593202256625288686656870585123412526520006572271804414136643813203989026554"
    "876046319882935288017213570171881319512203295850811908838180407634966154718446288670396235"
    "649706616968065048312917847198665960667399980860416663101533257689088484172679140598232518"
    "2252994934771483936941649517393670976161956787109375",
    491,
    "1.0000000000000000686870105410711392306560935401767227629654574949386254732267282863177163"
    "850871532771738817474236917352337331480425748697404066355343545355987256216235425947807033"
    "983401593202256625288686656870585123412526520006572271804414136643813203989026554876046319"
    "882935288017213570171881319512203295850811908838180407634966154718446288670396235649706616"
    "968065048312917847198665960667399980860416663101533257689088484172679140598232518225299493"
    "4771483936941649517393670976161956787109375e-189"
  },
  {
    1e-188,
    676,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000099999999999999994908067112790032880452765975891977808893416208927947989552854944"
    "282732526766467167814211472901669036546129454151955078046974915631886855486755712556527769"
    "214532540188976339227580242590091726130570768133012817820279334756064634091564387777099221"
    "990128121721030511263468938541163594069769777207961436977330409039381133269347693239389812"
    "020205342531022532236534723299348668678980039958608222059751251451001340125954851991484343"
    "502465496900111219247264671139419078826904296875",
    487,
    "9.9999999999999994908067112790032880452765975891977808893416208927947989552854944282732526"
    "766467167814211472901669036546129454151955078046974915631886855486755712556527769214532540"
    "188976339227580242590091726130570768133012817820279334756064634091564387777099221990128121"
    "721030511263468938541163594069769777207961436977330409039381133269347693239389812020205342"
    "531022532236534723299348668678980039958608222059751251451001340125954851991484343502465496"
    "900111219247264671139419078826904296875e-189"
  },
  {
    1e-187,
    672,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000001000000000000000012870718814924761031796157775590148581750852972297690870300911492"
    "688867196956661864292390472167091096006595131665163628080766478567127744590846990129685640"
    "217526999996641220515727814150383088373870166622140883511824725097055580097338022374545962"
    "676401628454691080147578789025767146309768072347121843930496982715306873574091297538285024"
    "078644503441574393792780559325811689429905832362393735027279583522725409797343474532661241"
    "69414530431510002017603255808353424072265625",
    485,
    "1.0000000000000000128707188149247610317961577755901485817508529722976908703009114926888671"
    "969566618642923904721670910960065951316651636280807664785671277445908469901296856402175269"
    "999966412205157278141503830883738701666221408835118247250970555800973380223745459626764016"
    "284546910801475787890257671463097680723471218439304969827153068735740912975382850240786445"
    "034415743937927805593258116894299058323623937350272795835227254097973434745326612416941453"
    "0431510002017603255808353424072265625e-187"
  },
  {
    1e-186,
    669,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000009999999999999999108066425156856694681665609489175557932441475594685533106651322129"
    "104001100894775664519492831264499271341141874321830719031387629699130410335832068266329233"
    "020044430256366953318471929512377650648101901549205550173745210388007974066273910088599742"
    "362089704636735321269557432431572173304555919191098852789883550009140081651083133072659778"
    "760987784314158795088872371940916852057371399202953119396522731023861961368715471441527510"
    "22960767812719495850615203380584716796875",
    482,
    "9.9999999999999991080664251568566946816656094891755579324414755946855331066513221291040011"
    "008947756645194928312644992713411418743218307190313876296991304103358320682663292330200444"
    "302563669533184719295123776506481019015492055501737452103880079740662739100885997423620897"
    "046367353212695574324315721733045559191910988527898835500091400816510831330726597787609877"
    "843141587950888723719409168520573713992029531193965227310238619613687154714415275102296076"
    "7812719495850615203380584716796875e-187"
  },
  {
    1e-185,
    666,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000099999999999999999245790355507694271907023841025563002404951188973186335837375563673"
    "317377958322500472430223435896286223209894281856751684524093544768480387939423346907509683"
    "442248860244031547895168991055402371205817132869681781293468428540462355919589610140876498"
    "836309685648757054345417986924516051390557626151945220019525717242830049229470069208121483"
    "813535843954269093600189489946768858507209387397405040975412143865756706524908905495954355"
    "99083970146239153109490871429443359375",
    480,
    "9.9999999999999999245790355507694271907023841025563002404951188973186335837375563673317377"
    "958322500472430223435896286223209894281856751684524093544768480387939423346907509683442248"
    "860244031547895168991055402371205817132869681781293468428540462355919589610140876498836309"
    "685648757054345417986924516051390557626151945220019525717242830049229470069208121483813535"
    "843954269093600189489946768858507209387397405040975412143865756706524908905495954355990839"
    "70146239153109490871429443359375e-186"
  },
  {
    1e-184,
    663,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000001000000000000000057778912386589961319793180379326089408693803353942511396540654375791"
    "392715178222955342184595344973210310486747127675072798922673429902214156043054783028835660"
    "356925063883211596635287478007030629856556267717828049382814882687684481250700175447797590"
    "086397970738801276652929170115515060665563735447105737160778909639734354043810599933404407"
    "764622446044140077693621063768491288540057036917041185835600107674663807951122583604977589"
    "46650037614745087921619415283203125",
    479,
    "1.0000000000000000577789123865899613197931803793260894086938033539425113965406543757913927"
    "151782229553421845953449732103104867471276750727989226734299022141560430547830288356603569"
    "250638832115966352874780070306298565562677178280493828148826876844812507001754477975900863"
    "979707388012766529291701155150606655637354471057371607789096397343540438105999334044077646"
    "224460441400776936210637684912885400570369170411858356001076746638079511225836049775894665"
    "0037614745087921619415283203125e-184"
  },
  {
    1e-183,
    660,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000010000000000000000055221053213795464392148268040697219009783701825739929660071353845448"
    "175667022245948478787065561649318477765036803890280359772830441282859347239977318658445996"
    "093758947288947024884094240446250956178483165010198602243104048612357436068569162165715087"
    "077570793378166900939297294192314232575737563049829075883615198705872646445226736516527520"
    "612112408429807643402401323278491257656865065626485649704171724609864137894957606886303658"
    "20017686928622424602508544921875",
    477,
    "1.0000000000000000055221053213795464392148268040697219009783701825739929660071353845448175"
    "667022245948478787065561649318477765036803890280359772830441282859347239977318658445996093"
    "758947288947024884094240446250956178483165010198602243104048612357436068569162165715087077"
    "570793378166900939297294192314232575737563049829075883615198705872646445226736516527520612"
    "112408429807643402401323278491257656865065626485649704171724609864137894957606886303658200"
    "17686928622424602508544921875e-183"
  },
  {
    1e-182,
    655,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000100000000000000004732755097354787834367750966427481590715071671966880771043395057754207"
    "768548302328324332341758721155461794469843821786384633359535274742851177924337279623744820"
    "741523005234821780591186721454952300881467747446641155111398712239473372193152360155237381"
    "066979245860435934112928197625833318396573961868117124629543168590493616395301467705407662"
    "394020500390821502294487748036285598518293084616266166256352063192832911879721902170814473"
    "720383830368518829345703125",
    473,
    "1.0000000000000000473275509735478783436775096642748159071507167196688077104339505775420776"
    "854830232832433234175872115546179446984382178638463335953527474285117792433727962374482074"
    "152300523482178059118672145495230088146774744664115511139871223947337219315236015523738106"
    "697924586043593411292819762583331839657396186811712462954316859049361639530146770540766239"
    "402050039082150229448774803628559851829308461626616625635206319283291187972190217081447372"
    "0383830368518829345703125e-182"
  },
  {
    1e-181,
    654,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "001000000000000000047327550973547878343677509664274815907150716719668807710433950577542077"
    "685483023283243323417587211554617944698438217863846333595352747428511779243372796237448207"
    "415230052348217805911867214549523008814677474466411551113987122394733721931523601552373810"
    "669792458604359341129281976258333183965739618681171246295431685904936163953014677054076623"
    "940205003908215022944877480362855985182930846162661662563520631928329118797219021708144737"
    "20383830368518829345703125",
    473,
    "1.0000000000000000473275509735478783436775096642748159071507167196688077104339505775420776"
    "854830232832433234175872115546179446984382178638463335953527474285117792433727962374482074"
    "152300523482178059118672145495230088146774744664115511139871223947337219315236015523738106"
    "697924586043593411292819762583331839657396186811712462954316859049361639530146770540766239"
    "402050039082150229448774803628559851829308461626616625635206319283291187972190217081447372"
    "0383830368518829345703125e-181"
  },
  {
    1e-180,
    647,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "010000000000000000205720657561601459248213926337435557432004149359281262740007888540238312"
    "094633121226702388025273417160450370537932074089277055554752311772624638861626007860251046"
    "700554453379680027103035857978894785962064514606187019596944752529800482837748751646201448"
    "056560613002510229215683397611108310748702979142825052177281241016263915175964207972289037"
    "936679409146465860138405827540686447052193047136397795898578025255033080442856885383662302"
    "0470142364501953125",
    467,
    "1.0000000000000000205720657561601459248213926337435557432004149359281262740007888540238312"
    "094633121226702388025273417160450370537932074089277055554752311772624638861626007860251046"
    "700554453379680027103035857978894785962064514606187019596944752529800482837748751646201448"
    "056560613002510229215683397611108310748702979142825052177281241016263915175964207972289037"
    "936679409146465860138405827540686447052193047136397795898578025255033080442856885383662302"
    "0470142364501953125e-180"
  },
  {
    1e-179,
    646,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "100000000000000002057206575616014592482139263374355574320041493592812627400078885402383120"
    "946331212267023880252734171604503705379320740892770555547523117726246388616260078602510467"
    "005544533796800271030358579788947859620645146061870195969447525298004828377487516462014480"
    "565606130025102292156833976111083107487029791428250521772812410162639151759642079722890379"
    "366794091464658601384058275406864470521930471363977958985780252550330804428568853836623020"
    "470142364501953125",
    467,
    "1.0000000000000000205720657561601459248213926337435557432004149359281262740007888540238312"
    "094633121226702388025273417160450370537932074089277055554752311772624638861626007860251046"
    "700554453379680027103035857978894785962064514606187019596944752529800482837748751646201448"
    "056560613002510229215683397611108310748702979142825052177281241016263915175964207972289037"
    "936679409146465860138405827540686447052193047136397795898578025255033080442856885383662302"
    "0470142364501953125e-179"
  },
  {
    1e-178,
    641,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "999999999999999952078023599647550932549733035583529723487642369551981796731894841817120230"
    "852851551603142187974074929298393483501980644336017773388789574064216571704500430381961642"
    "408451391728506514300696193707641236920632565789008124705298570090643745538135611970760193"
    "466884201733728309821430328221607674244836751047328058807005885153374082925684779698740218"
    "533059651111387470386124875573053082277758604143759177280959254269232516776355623733252286"
    "9110107421875",
    462,
    "9.9999999999999995207802359964755093254973303558352972348764236955198179673189484181712023"
    "085285155160314218797407492929839348350198064433601777338878957406421657170450043038196164"
    "240845139172850651430069619370764123692063256578900812470529857009064374553813561197076019"
    "346688420173372830982143032822160767424483675104732805880700588515337408292568477969874021"
    "853305965111138747038612487557305308227775860414375917728095925426923251677635562373325228"
    "69110107421875e-179"
  },
  {
    1e-177,
    640,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009"
    "999999999999999520780235996475509325497330355835297234876423695519817967318948418171202308"
    "528515516031421879740749292983934835019806443360177733887895740642165717045004303819616424"
    "084513917285065143006961937076412369206325657890081247052985700906437455381356119707601934"
    "668842017337283098214303282216076742448367510473280588070058851533740829256847796987402185"
    "330596511113874703861248755730530822777586041437591772809592542692325167763556237332522869"
    "110107421875",
    462,
    "9.9999999999999995207802359964755093254973303558352972348764236955198179673189484181712023"
    "085285155160314218797407492929839348350198064433601777338878957406421657170450043038196164"
    "240845139172850651430069619370764123692063256578900812470529857009064374553813561197076019"
    "346688420173372830982143032822160767424483675104732805880700588515337408292568477969874021"
    "853305965111138747038612487557305308227775860414375917728095925426923251677635562373325228"
    "69110107421875e-178"
  },
  {
    1e-176,
    632,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000099"
    "999999999999999591421057981561172760359517840594637610381681203271426218398700962941525716"
    "354631708608402128816567281624536848836577367469795392411220011109485295768465799357318010"
    "252751732178407974254554038401714686355665848001217909837164713986265000964892566636634526"
    "795754478479686133945236527071065064513189551784144051652154369610524111495583091804490661"
    "938365977391453819697791781023172096034811422121224133013894785904085438232868909835815429"
    "6875",
    455,
    "9.9999999999999999591421057981561172760359517840594637610381681203271426218398700962941525"
    "716354631708608402128816567281624536848836577367469795392411220011109485295768465799357318"
    "010252751732178407974254554038401714686355665848001217909837164713986265000964892566636634"
    "526795754478479686133945236527071065064513189551784144051652154369610524111495583091804490"
    "661938365977391453819697791781023172096034811422121224133013894785904085438232868909835815"
    "4296875e-177"
  },
  {
    1e-175,
    631,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000999"
    "999999999999995914210579815611727603595178405946376103816812032714262183987009629415257163"
    "546317086084021288165672816245368488365773674697953924112200111094852957684657993573180102"
    "527517321784079742545540384017146863556658480012179098371647139862650009648925666366345267"
    "957544784796861339452365270710650645131895517841440516521543696105241114955830918044906619"
    "383659773914538196977917810231720960348114221212241330138947859040854382328689098358154296"
    "875",
    455,
    "9.9999999999999999591421057981561172760359517840594637610381681203271426218398700962941525"
    "716354631708608402128816567281624536848836577367469795392411220011109485295768465799357318"
    "010252751732178407974254554038401714686355665848001217909837164713986265000964892566636634"
    "526795754478479686133945236527071065064513189551784144051652154369610524111495583091804490"
    "661938365977391453819697791781023172096034811422121224133013894785904085438232868909835815"
    "4296875e-176"
  },
  {
    1e-174,
    630,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000009999"
    "999999999999959142105798156117276035951784059463761038168120327142621839870096294152571635"
    "463170860840212881656728162453684883657736746979539241122001110948529576846579935731801025"
    "275173217840797425455403840171468635566584800121790983716471398626500096489256663663452679"
    "575447847968613394523652707106506451318955178414405165215436961052411149558309180449066193"
    "836597739145381969779178102317209603481142212122413301389478590408543823286890983581542968"
    "75",
    455,
    "9.9999999999999999591421057981561172760359517840594637610381681203271426218398700962941525"
    "716354631708608402128816567281624536848836577367469795392411220011109485295768465799357318"
    "010252751732178407974254554038401714686355665848001217909837164713986265000964892566636634"
    "526795754478479686133945236527071065064513189551784144051652154369610524111495583091804490"
    "661938365977391453819697791781023172096034811422121224133013894785904085438232868909835815"
    "4296875e-175"
  },
  {
    1e-173,
    624,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000"
    "000000000004080246604750770598173875001265610102838277944113298430680692938946920536410569"
    "775694061645860179459417852569871442414611750645879228256918309821296094530706786339470126"
    "146992930030675499927138062607864511092939560033079687847803826280818847855889066704471225"
    "664806909105809390693120899209847903412345564714338706557804386194710076938736661290721977"
    "94446443422556352914330611026469713197725405241789164989540950045920908451080322265625",
    451,
    "1.0000000000000000408024660475077059817387500126561010283827794411329843068069293894692053"
    "641056977569406164586017945941785256987144241461175064587922825691830982129609453070678633"
    "947012614699293003067549992713806260786451109293956003307968784780382628081884785588906670"
    "447122566480690910580939069312089920984790341234556471433870655780438619471007693873666129"
    "072197794446443422556352914330611026469713197725405241789164989540950045920908451080322265"
    "625e-173"
  },
  {
    1e-172,
    623,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000001000000"
    "000000000040802466047507705981738750012656101028382779441132984306806929389469205364105697"
    "756940616458601794594178525698714424146117506458792282569183098212960945307067863394701261"
    "469929300306754999271380626078645110929395600330796878478038262808188478558890667044712256"
    "648069091058093906931208992098479034123455647143387065578043861947100769387366612907219779"
    "4446443422556352914330611026469713197725405241789164989540950045920908451080322265625",
    451,
    "1.0000000000000000408024660475077059817387500126561010283827794411329843068069293894692053"
    "641056977569406164586017945941785256987144241461175064587922825691830982129609453070678633"
    "947012614699293003067549992713806260786451109293956003307968784780382628081884785588906670"
    "447122566480690910580939069312089920984790341234556471433870655780438619471007693873666129"
    "072197794446443422556352914330611026469713197725405241789164989540950045920908451080322265"
    "625e-172"
  },
  {
    1e-171,
    618,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000009999999"
    "999999999833454990488618253364457518248159030734657072758846386496895631432742740272197439"
    "139268149388403495748348068760250694293907115725610244967709339121567716762527719200148820"
    "105916795361790584957049666459647214626236474966227897344883106057195766283635621494235537"
    "958651944862482050868079918452516392716952558437127692140787333634389280351124458664512728"
    "40010194777314250133539987861677276049891804898805958146112971007823944091796875",
    446,
    "9.9999999999999998334549904886182533644575182481590307346570727588463864968956314327427402"
    "721974391392681493884034957483480687602506942939071157256102449677093391215677167625277192"
    "001488201059167953617905849570496664596472146262364749662278973448831060571957662836356214"
    "942355379586519448624820508680799184525163927169525584371276921407873336343892803511244586"
    "6451272840010194777314250133539987861677276049891804898805958146112971007823944091796875e-"
    "172"
  },
  {
    1e-170,
    617,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000099999999"
    "999999998334549904886182533644575182481590307346570727588463864968956314327427402721974391"
    "392681493884034957483480687602506942939071157256102449677093391215677167625277192001488201"
    "059167953617905849570496664596472146262364749662278973448831060571957662836356214942355379"
    "586519448624820508680799184525163927169525584371276921407873336343892803511244586645127284"
    "0010194777314250133539987861677276049891804898805958146112971007823944091796875",
    446,
    "9.9999999999999998334549904886182533644575182481590307346570727588463864968956314327427402"
    "721974391392681493884034957483480687602506942939071157256102449677093391215677167625277192"
    "001488201059167953617905849570496664596472146262364749662278973448831060571957662836356214"
    "942355379586519448624820508680799184525163927169525584371276921407873336343892803511244586"
    "6451272840010194777314250133539987861677276049891804898805958146112971007823944091796875e-"
    "171"
  },
  {
    1e-169,
    614,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000001000000000"
    "000000020117957927995188949433270665033629764612633461643579870244677540839030082826754373"
    "455647911487674387214786922546256448095860299749029663114719064671442799974430463814164864"
    "567756829347660592137388682880170721357697310494206530360280015299673863900909282408323621"
    "275688292229454267224632007317171976821906275504318887016415656982506500555112772543117066"
    "9760491634397165652335013242654640323869852380767042632214725017547607421875",
    445,
    "1.0000000000000000201179579279951889494332706650336297646126334616435798702446775408390300"
    "828267543734556479114876743872147869225462564480958602997490296631147190646714427999744304"
    "638141648645677568293476605921373886828801707213576973104942065303602800152996738639009092"
    "824083236212756882922294542672246320073171719768219062755043188870164156569825065005551127"
    "725431170669760491634397165652335013242654640323869852380767042632214725017547607421875e-1"
    "69"
  },
  {
    1e-168,
    610,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000010000000000"
    "000000495359250313018798398232857372078111175301744102507328466887690588908349273123627410"
    "787142896055342371187709597632060630599792814994337961897471866831796989517572988535911477"
    "486186638825422692833263124125301283449371615913399670578555429637516523307869887961394816"
    "040833370144536115579441369695981409232266209375586253665614918173692729092462974165924485"
    "487326723400897105883120943360144183831295094933011569082736968994140625",
    442,
    "1.0000000000000000495359250313018798398232857372078111175301744102507328466887690588908349"
    "273123627410787142896055342371187709597632060630599792814994337961897471866831796989517572"
    "988535911477486186638825422692833263124125301283449371615913399670578555429637516523307869"
    "887961394816040833370144536115579441369695981409232266209375586253665614918173692729092462"
    "974165924485487326723400897105883120943360144183831295094933011569082736968994140625e-168"
  },
  {
    1e-167,
    607,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000"
    "000000246717766601117441519926162172912095286210889247928808437822263000794717613538935288"
    "180808461695847727239650021608667911738891069878718326970219146440066058803436279050909465"
    "923972862673158584982610516075507716535339983592646834173469870122719084298265857563410507"
    "865126535845466062464472952571627836111406824437504400632815608158883714263265761903183803"
    "243905809949267802061486222313380078929867522674612700939178466796875",
    440,
    "1.0000000000000000024671776660111744151992616217291209528621088924792880843782226300079471"
    "761353893528818080846169584772723965002160866791173889106987871832697021914644006605880343"
    "627905090946592397286267315858498261051607550771653533998359264683417346987012271908429826"
    "585756341050786512653584546606246447295257162783611140682443750440063281560815888371426326"
    "5761903183803243905809949267802061486222313380078929867522674612700939178466796875e-167"
  },
  {
    1e-166,
    604,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000001000000000000"
    "000040122175558243738754898480914112073084596561306696443894226659773114257377076968063439"
    "333048607819085149496067853782186271461207339304473605738187639423891279012711640974737130"
    "742876831380132596626270962175118109020409240257267314631374111246760033226122752038406298"
    "996922683253821371284255480821768410804110398921909094514824670213185755923569457080326445"
    "473949491970304074772647913438294864363342639990150928497314453125",
    438,
    "1.0000000000000000401221755582437387548984809141120730845965613066964438942266597731142573"
    "770769680634393330486078190851494960678537821862714612073393044736057381876394238912790127"
    "116409747371307428768313801325966262709621751181090204092402572673146313741112467600332261"
    "227520384062989969226832538213712842554808217684108041103989219090945148246702131857559235"
    "69457080326445473949491970304074772647913438294864363342639990150928497314453125e-166"
  },
  {
    1e-165,
    600,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000010000000000000"
    "000099981772444576872831391054802057113792089993753227192463479100586292092163237050949933"
    "130774151305988478164137436257805482033700268906413369093906994053067262300325606022231535"
    "403582676612951991861383210390853540868017167926281363140337832311046810313514109149653227"
    "203968234144927739726347167373763710520766752844170239654897993137068652908399866415357150"
    "46036377988203231446419361194699604311608709394931793212890625",
    435,
    "1.0000000000000000099981772444576872831391054802057113792089993753227192463479100586292092"
    "163237050949933130774151305988478164137436257805482033700268906413369093906994053067262300"
    "325606022231535403582676612951991861383210390853540868017167926281363140337832311046810313"
    "514109149653227203968234144927739726347167373763710520766752844170239654897993137068652908"
    "39986641535715046036377988203231446419361194699604311608709394931793212890625e-165"
  },
  {
    1e-164,
    597,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000099999999999999"
    "996179977994240000492832410478595553265058890028512475980974191051545313215911848434547968"
    "112350682902076512896716737553139099083032702850970678331559537557144177774603200620079001"
    "632856571115536328192609522143294619302967924920545100628925840605611751971726511745976067"
    "795544767156701827404149420234910744882271746442971108655400587454064027847283393947054636"
    "13753956168418821244536776049471882288344204425811767578125",
    432,
    "9.9999999999999996179977994240000492832410478595553265058890028512475980974191051545313215"
    "911848434547968112350682902076512896716737553139099083032702850970678331559537557144177774"
    "603200620079001632856571115536328192609522143294619302967924920545100628925840605611751971"
    "726511745976067795544767156701827404149420234910744882271746442971108655400587454064027847"
    "28339394705463613753956168418821244536776049471882288344204425811767578125e-165"
  },
  {
    1e-163,
    594,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000999999999999999"
    "923241062100753859044472104230555389667692821012966392260457110880912270513354307745868775"
    "560380187758298979009906375332065220798567138804402682455512151783214215916809129382899197"
    "104804151043494558556314567311019878012049214467302760093638546017266710409948479455311044"
    "002347077226413715166916174327296566219551208439860783405371123207660268579111777818411413"
    "64660971975280266688113073314525536261498928070068359375",
    430,
    "9.9999999999999992324106210075385904447210423055538966769282101296639226045711088091227051"
    "335430774586877556038018775829897900990637533206522079856713880440268245551215178321421591"
    "680912938289919710480415104349455855631456731101987801204921446730276009363854601726671040"
    "994847945531104400234707722641371516691617432729656621955120843986078340537112320766026857"
    "911177781841141364660971975280266688113073314525536261498928070068359375e-164"
  },
  {
    1e-162,
    591,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000009999999999999999"
    "540880363740707757515537046748755040540096844306930862998849505885449598299656490255575000"
    "108815007682718989757151754915258368239750505686459631435787308137962653801874308372118524"
    "838133991329895372521390906085609300261532422578213570501344340483473578558017898588707511"
    "648275526988973622665785967447452723020842132317410259242789242740442764940895071401193718"
    "29638437424066232939168230586801655590534210205078125",
    428,
    "9.9999999999999995408803637407077575155370467487550405400968443069308629988495058854495982"
    "996564902555750001088150076827189897571517549152583682397505056864596314357873081379626538"
    "018743083721185248381339913298953725213909060856093002615324225782135705013443404834735785"
    "580178985887075116482755269889736226657859674474527230208421323174102592427892427404427649"
    "4089507140119371829638437424066232939168230586801655590534210205078125e-163"
  },
  {
    1e-161,
    587,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000100000000000000002"
    "812077463003137584854954574124377858117015663323715199451176588686341418983286809681043869"
    "208465199220690689365629587423131528495403880282983679493852048719318409229535432756222539"
    "343559454777748612211794652265945486000290895506598974572456532294091172584973482741404835"
    "478069383285811530576841054662216690016342473225360796965764683336589549003605751221847146"
    "8907359835098791478458224446512758731842041015625",
    426,
    "1.0000000000000000281207746300313758485495457412437785811701566332371519945117658868634141"
    "898328680968104386920846519922069068936562958742313152849540388028298367949385204871931840"
    "922953543275622253934355945477774861221179465226594548600029089550659897457245653229409117"
    "258497348274140483547806938328581153057684105466221669001634247322536079696576468333658954"
    "90036057512218471468907359835098791478458224446512758731842041015625e-161"
  },
  {
    1e-160,
    582,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000999999999999999988"
    "636647560185722463485097172514032166684571458546983624044131061093571864570351258808871395"
    "442971339441569337421031670121726772431911744598437514213299328048160779171128466042026508"
    "303756993223913391462556701806908281949753383202185641409828643157682995157497510857623186"
    "805685228079047018200509852287823114521178598646897545455661468394365358864563980432284994"
    "63060121588142692417022772133350372314453125",
    421,
    "9.9999999999999998863664756018572246348509717251403216668457145854698362404413106109357186"
    "457035125880887139544297133944156933742103167012172677243191174459843751421329932804816077"
    "917112846604202650830375699322391339146255670180690828194975338320218564140982864315768299"
    "515749751085762318680568522807904701820050985228782311452117859864689754545566146839436535"
    "886456398043228499463060121588142692417022772133350372314453125e-161"
  },
  {
    1e-159,
    581,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000009999999999999999886"
    "366475601857224634850971725140321666845714585469836240441310610935718645703512588088713954"
    "429713394415693374210316701217267724319117445984375142132993280481607791711284660420265083"
    "037569932239133914625567018069082819497533832021856414098286431576829951574975108576231868"
    "056852280790470182005098522878231145211785986468975454556614683943653588645639804322849946"
    "3060121588142692417022772133350372314453125",
    421,
    "9.9999999999999998863664756018572246348509717251403216668457145854698362404413106109357186"
    "457035125880887139544297133944156933742103167012172677243191174459843751421329932804816077"
    "917112846604202650830375699322391339146255670180690828194975338320218564140982864315768299"
    "515749751085762318680568522807904701820050985228782311452117859864689754545566146839436535"
    "886456398043228499463060121588142692417022772133350372314453125e-160"
  },
  {
    1e-158,
    577,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000100000000000000006444"
    "617153428937696280883842447514528249689499395210689534198992657166912907438358777188060499"
    "499819275101744539273894201213671647439569640272413320572395360660554036964212016080836775"
    "688509796677303432090515784379771181181208118068952169412306834148215808659315864595950931"
    "770174925485813033087918540976318295429117517178155992347336913970321071383156146176302524"
    "197776477876686840318143367767333984375",
    419,
    "1.0000000000000000644461715342893769628088384244751452824968949939521068953419899265716691"
    "290743835877718806049949981927510174453927389420121367164743956964027241332057239536066055"
    "403696421201608083677568850979667730343209051578437977118118120811806895216941230683414821"
    "580865931586459595093177017492548581303308791854097631829542911751717815599234733691397032"
    "1071383156146176302524197776477876686840318143367767333984375e-158"
  },
  {
    1e-157,
    574,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000999999999999999943150"
    "933175723529763890852421337364297197177337303909661265415741806713505867931861431065869711"
    "755227455900472638007306987480806006421373515865542817844552713093922452020273570757392631"
    "880130378197605747547628184774624032518164415083313239251988047403497400040122184621393298"
    "475315373560350922288252414659073461311052731967136774974327947162647755003431814598176263"
    "775286543022957630455493927001953125",
    416,
    "9.9999999999999994315093317572352976389085242133736429719717733730390966126541574180671350"
    "586793186143106586971175522745590047263800730698748080600642137351586554281784455271309392"
    "245202027357075739263188013037819760574754762818477462403251816441508331323925198804740349"
    "740004012218462139329847531537356035092228825241465907346131105273196713677497432794716264"
    "7755003431814598176263775286543022957630455493927001953125e-158"
  },
  {
    1e-156,
    571,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000010000000000000000401871"
    "238625762075230252412238475890854369514626224674485266750896186780044330932425037176579383"
    "495996919940508417926150072055343808008318253524151281480734279032167861177508427981727318"
    "841044490579486062336519119930942559532978275682800031488522826664259492825513536918861138"
    "564624785985744491609988107423610556951506838186752937735609011950981220659355323300554463"
    "372691316180862486362457275390625",
    415,
    "1.0000000000000000401871238625762075230252412238475890854369514626224674485266750896186780"
    "044330932425037176579383495996919940508417926150072055343808008318253524151281480734279032"
    "167861177508427981727318841044490579486062336519119930942559532978275682800031488522826664"
    "259492825513536918861138564624785985744491609988107423610556951506838186752937735609011950"
    "981220659355323300554463372691316180862486362457275390625e-156"
  },
  {
    1e-155,
    567,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000100000000000000001431080"
    "634608216012058940420984486247523967836253751870525700593020215413814905020755101051441125"
    "776709570242998744986620194560681429964294282258251206713457062073829702509023692063137188"
    "304469682852384391737891806816886303726225090562219944302181992964500281790357527308802975"
    "482325058838150866159977178681769719272456332493168876043211345311135751593454093488766432"
    "07404413260519504547119140625",
    412,
    "1.0000000000000000143108063460821601205894042098448624752396783625375187052570059302021541"
    "381490502075510105144112577670957024299874498662019456068142996429428225825120671345706207"
    "382970250902369206313718830446968285238439173789180681688630372622509056221994430218199296"
    "450028179035752730880297548232505883815086615997717868176971927245633249316887604321134531"
    "113575159345409348876643207404413260519504547119140625e-155"
  },
  {
    1e-154,
    563,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000999999999999999972908698"
    "319691684276692064987440499898924041402401600716025535275135715952094581351626679084767910"
    "834941635836620501468113529722707897740730774850326337632398968772714476833267516565195881"
    "349093261444224211342127788288234371605328245369713513693079550795488474467129803011095192"
    "200485772072803862561309457948323588842770534941920739426053065932534235932954702619213094"
    "2945368587970733642578125",
    408,
    "9.9999999999999997290869831969168427669206498744049989892404140240160071602553527513571595"
    "209458135162667908476791083494163583662050146811352972270789774073077485032633763239896877"
    "271447683326751656519588134909326144422421134212778828823437160532824536971351369307955079"
    "548847446712980301109519220048577207280386256130945794832358884277053494192073942605306593"
    "25342359329547026192130942945368587970733642578125e-155"
  },
  {
    1e-153,
    561,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000010000000000000000391520711"
    "619164456269278077432874800210290605386190694987958883232420170497817315211056093721972659"
    "263881423860076189050549951372781407842700512218235048358736119176465540444185630710774840"
    "620589687716157410009922360972402366564045017736910006190641569547114239654425551341904923"
    "969094781667315410228491841393213550536389989255495730357496854186514839354926742487350566"
    "07984006404876708984375",
    408,
    "1.0000000000000000391520711619164456269278077432874800210290605386190694987958883232420170"
    "497817315211056093721972659263881423860076189050549951372781407842700512218235048358736119"
    "176465540444185630710774840620589687716157410009922360972402366564045017736910006190641569"
    "547114239654425551341904923969094781667315410228491841393213550536389989255495730357496854"
    "18651483935492674248735056607984006404876708984375e-153"
  },
  {
    1e-152,
    557,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000100000000000000006564942029"
    "880635016702210484562627206987106819310605701190402954248453748885659158889718148716900796"
    "296674500576246587983158130310623800168576177042237171726346917561938492887898167343012514"
    "724525170257235286453801522084258267683500433528199538945799941840060376476765598342861247"
    "547896060430261240746507461572046153798638451900110647962833321309838313650786290054384153"
    "3362865447998046875",
    405,
    "1.0000000000000000656494202988063501670221048456262720698710681931060570119040295424845374"
    "888565915888971814871690079629667450057624658798315813031062380016857617704223717172634691"
    "756193849288789816734301251472452517025723528645380152208425826768350043352819953894579994"
    "184006037647676559834286124754789606043026124074650746157204615379863845190011064796283332"
    "13098383136507862900543841533362865447998046875e-152"
  },
  {
    1e-151,
    553,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000999999999999999938462144441"
    "734808374569478754400070235429431451568516948984951690120439381297263263497635335304646187"
    "389452430939200400903967707131371358090351137147810686592154337349796683468972382137447938"
    "351093633980615919518275427551321778768592039645220491567555592692540728007171907085636098"
    "345444903961469761308800329004750413118933670433745949010823799753266971634957371861673891"
    "544342041015625",
    401,
    "9.9999999999999993846214444173480837456947875440007023542943145156851694898495169012043938"
    "129726326349763533530464618738945243093920040090396770713137135809035113714781068659215433"
    "734979668346897238213744793835109363398061591951827542755132177876859203964522049156755559"
    "269254072800717190708563609834544490396146976130880032900475041311893367043374594901082379"
    "9753266971634957371861673891544342041015625e-152"
  },
  {
    1e-150,
    551,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000010000000000000000062953582321"
    "729639972108793363873778804649710470552049825417932113812917053289050370440599496323058010"
    "306751375116086563320282916513002346745701415609099029501889177602437476876440041602091164"
    "279779372295422901954699839733275910706785973181671037757922997368410142794300811352234994"
    "833199441434125059254799485864630130482407896696715653401621535373289262338403204921633005"
    "1422119140625",
    401,
    "1.0000000000000000062953582321729639972108793363873778804649710470552049825417932113812917"
    "053289050370440599496323058010306751375116086563320282916513002346745701415609099029501889"
    "177602437476876440041602091164279779372295422901954699839733275910706785973181671037757922"
    "997368410142794300811352234994833199441434125059254799485864630130482407896696715653401621"
    "5353732892623384032049216330051422119140625e-150"
  },
  {
    1e-149,
    547,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000099999999999999997916207271599"
    "770174815431910359245482245075520886052976911905660287695077571624832762549010390124195557"
    "418605488264535416080405784332868404088253979567021640697508559606492200017535535110464519"
    "722421592997174192459216140452526614984397424898845889249761691912089977052681151538853902"
    "816992807063540807880810075377797307651794197530131880842681202370414439428714103996753692"
    "626953125",
    397,
    "9.9999999999999997916207271599770174815431910359245482245075520886052976911905660287695077"
    "571624832762549010390124195557418605488264535416080405784332868404088253979567021640697508"
    "559606492200017535535110464519722421592997174192459216140452526614984397424898845889249761"
    "691912089977052681151538853902816992807063540807880810075377797307651794197530131880842681"
    "202370414439428714103996753692626953125e-150"
  },
  {
    1e-148,
    544,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000999999999999999935748815890117"
    "282149663822731120577929628009867749049427642678029270005288335997592555778350731539802843"
    "803522676304070686845283750574203026982376971286717937832954133378800900225517256537491228"
    "018261850658864691187645294441546276508577338302627079226124410768716556281580123652602299"
    "929902354192051524133144221481909121761385664308924357650265601904493451002053916454315185"
    "546875",
    395,
    "9.9999999999999993574881589011728214966382273112057792962800986774904942764267802927000528"
    "833599759255577835073153980284380352267630407068684528375057420302698237697128671793783295"
    "413337880090022551725653749122801826185065886469118764529444154627650857733830262707922612"
    "441076871655628158012365260229992990235419205152413314422148190912176138566430892435765026"
    "5601904493451002053916454315185546875e-149"
  },
  {
    1e-147,
    540,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000009999999999999999704794213508216"
    "178284562198290980794438862061406382337008237808881555616782401981806115477532673015250281"
    "095484413770974660123030247777878381025072307935167131466593035276977801853877321912144033"
    "830251141091664779112581825085221751768948668512925298433184174504631276777652370413516825"
    "219229273467367678731094473187602855666307131028399182715027393442142056301236152648925781"
    "25",
    392,
    "9.9999999999999997047942135082161782845621982909807944388620614063823370082378088815556167"
    "824019818061154775326730152502810954844137709746601230302477778783810250723079351671314665"
    "930352769778018538773219121440338302511410916647791125818250852217517689486685129252984331"
    "841745046312767776523704135168252192292734673676787310944731876028556663071310283991827150"
    "2739344214205630123615264892578125e-148"
  },
  {
    1e-146,
    538,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000100000000000000002604839008794855"
    "491452405518586208186669932017726092853791354546237245190208691912150077879732452028052299"
    "918966549394031267953386350352353589471564600439475364858757576593278812118049323717148396"
    "664633562964933666903880341568361304620291252915725083082882814125764191166141846335069466"
    "915584439423315785705380865772214765502279117310481526548215924776741303503513336181640625",
    392,
    "1.0000000000000000260483900879485549145240551858620818666993201772609285379135454623724519"
    "020869191215007787973245202805229991896654939403126795338635035235358947156460043947536485"
    "875757659327881211804932371714839666463356296493366690388034156836130462029125291572508308"
    "288281412576419116614184633506946691558443942331578570538086577221476550227911731048152654"
    "8215924776741303503513336181640625e-146"
  },
  {
    1e-145,
    534,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000999999999999999914910452613694680"
    "742388384472334077021073092104015538863734016313938671454393477239722316709210082769533219"
    "907217260254619345072186052052140310298815582638672644731031289462772249594971145257322799"
    "403892588683619153477561601360737307586821173427808855808006759668613443869055619352670374"
    "69001029924037788916508597979842347823863503257502127752331944066099822521209716796875",
    388,
    "9.9999999999999991491045261369468074238838447233407702107309210401553886373401631393867145"
    "439347723972231670921008276953321990721726025461934507218605205214031029881558263867264473"
    "103128946277224959497114525732279940389258868361915347756160136073730758682117342780885580"
    "800675966861344386905561935267037469001029924037788916508597979842347823863503257502127752"
    "331944066099822521209716796875e-146"
  },
  {
    1e-144,
    527,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000009999999999999999504745926054559204"
    "774717991006630385716734850874540635594714656414374811976553786418914245774067027730499492"
    "776006950340412120999228365229868973122013176006185659651255219331773285023382146698543729"
    "214743617926487584571589819440575439439704072612302878146696017771025535626117294320381489"
    "1907720963806747888947723673401521480956499754455535367014817893505096435546875",
    382,
    "9.9999999999999995047459260545592047747179910066303857167348508745406355947146564143748119"
    "765537864189142457740670277304994927760069503404121209992283652298689731220131760061856596"
    "512552193317732850233821466985437292147436179264875845715898194405754394397040726123028781"
    "466960177710255356261172943203814891907720963806747888947723673401521480956499754455535367"
    "014817893505096435546875e-145"
  },
  {
    1e-143,
    526,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000099999999999999995047459260545592047"
    "747179910066303857167348508745406355947146564143748119765537864189142457740670277304994927"
    "760069503404121209992283652298689731220131760061856596512552193317732850233821466985437292"
    "147436179264875845715898194405754394397040726123028781466960177710255356261172943203814891"
    "907720963806747888947723673401521480956499754455535367014817893505096435546875",
    382,
    "9.9999999999999995047459260545592047747179910066303857167348508745406355947146564143748119"
    "765537864189142457740670277304994927760069503404121209992283652298689731220131760061856596"
    "512552193317732850233821466985437292147436179264875845715898194405754394397040726123028781"
    "466960177710255356261172943203814891907720963806747888947723673401521480956499754455535367"
    "014817893505096435546875e-144"
  },
  {
    1e-142,
    521,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000001000000000000000041518790984364694199"
    "285340549185180141210491125056686780559335919834434140405846231444340719990049982052776465"
    "782288069361191690929004768354160066468799103200124324406757057414330505197912365935201126"
    "483700951764547204928276237357349018272445874789153751726477574834674378115371235219650945"
    "4885002561528285839188544891300604311457078665625886060297489166259765625",
    379,
    "1.0000000000000000415187909843646941992853405491851801412104911250566867805593359198344341"
    "404058462314443407199900499820527764657822880693611916909290047683541600664687991032001243"
    "244067570574143305051979123659352011264837009517645472049282762373573490182724458747891537"
    "517264775748346743781153712352196509454885002561528285839188544891300604311457078665625886"
    "060297489166259765625e-142"
  },
  {
    1e-141,
    520,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000010000000000000000415187909843646941992"
    "853405491851801412104911250566867805593359198344341404058462314443407199900499820527764657"
    "822880693611916909290047683541600664687991032001243244067570574143305051979123659352011264"
    "837009517645472049282762373573490182724458747891537517264775748346743781153712352196509454"
    "885002561528285839188544891300604311457078665625886060297489166259765625",
    379,
    "1.0000000000000000415187909843646941992853405491851801412104911250566867805593359198344341"
    "404058462314443407199900499820527764657822880693611916909290047683541600664687991032001243"
    "244067570574143305051979123659352011264837009517645472049282762373573490182724458747891537"
    "517264775748346743781153712352196509454885002561528285839188544891300604311457078665625886"
    "060297489166259765625e-141"
  },
  {
    1e-140,
    517,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000099999999999999998325050402186307901732"
    "467402213100953670680726099100791906309894166038425704554697413047438873670776829096706534"
    "606852675640475268505709131911190373761094154792697446676657790264922336770584044347107527"
    "772388993044240635592788964547377071914116211147955201007706428611705619304048118348964858"
    "527426057820477947621912585655923333405326047795824706554412841796875",
    376,
    "9.9999999999999998325050402186307901732467402213100953670680726099100791906309894166038425"
    "704554697413047438873670776829096706534606852675640475268505709131911190373761094154792697"
    "446676657790264922336770584044347107527772388993044240635592788964547377071914116211147955"
    "201007706428611705619304048118348964858527426057820477947621912585655923333405326047795824"
    "706554412841796875e-141"
  },
  {
    1e-139,
    514,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000001000000000000000029865133591864371162893"
    "207243774346020309754352243551008260088524199624163733786379981567453739381539300414585695"
    "044160840234303280215232947150433922561470869684854418758961511994248831871060836855116242"
    "505539397726245213806567814973968761784932253618911783197472724962913730905084412418686107"
    "855057037903823030327416475360191583376945345662534236907958984375",
    375,
    "1.0000000000000000298651335918643711628932072437743460203097543522435510082600885241996241"
    "637337863799815674537393815393004145856950441608402343032802152329471504339225614708696848"
    "544187589615119942488318710608368551162425055393977262452138065678149739687617849322536189"
    "117831974727249629137309050844124186861078550570379038230303274164753601915833769453456625"
    "34236907958984375e-139"
  },
  {
    1e-138,
    511,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000010000000000000000671568372478654048793480"
    "338210890152071921120252455854796176801902310160890843779046624419057415205561079726019742"
    "246681072979437563417462495812580705218943270911583803528683994702692032032371515623490143"
    "308589715533163001095103505741271958999483673303996016937994760395997611967669955619161259"
    "196942056630578736140489704009581828358932398259639739990234375",
    373,
    "1.0000000000000000671568372478654048793480338210890152071921120252455854796176801902310160"
    "890843779046624419057415205561079726019742246681072979437563417462495812580705218943270911"
    "583803528683994702692032032371515623490143308589715533163001095103505741271958999483673303"
    "996016937994760395997611967669955619161259196942056630578736140489704009581828358932398259"
    "639739990234375e-138"
  },
  {
    1e-137,
    508,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000099999999999999997765674847346292395985645"
    "003553380915867445361004070274835946019175567546824295824542834322093638691576983336290419"
    "145066634520661363811432374728011541687802931602887252749186952782031200601399626499036195"
    "009199436834569298244826513374695402390969442282883730261527345555328849672879601816408256"
    "456500304089424972613098230311834413441829383373260498046875",
    370,
    "9.9999999999999997765674847346292395985645003553380915867445361004070274835946019175567546"
    "824295824542834322093638691576983336290419145066634520661363811432374728011541687802931602"
    "887252749186952782031200601399626499036195009199436834569298244826513374695402390969442282"
    "883730261527345555328849672879601816408256456500304089424972613098230311834413441829383373"
    "260498046875e-138"
  },
  {
    1e-136,
    504,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000001000000000000000001523438813303585538387539"
    "045015197438279162520762004810028318858015766300467336821224102870217755886526670493322866"
    "975317265936518359082837303007570111549042056063407947592277512473349658606837677619335918"
    "296521617671188216331487917848351857520007198181041140264394144632347883405649229831294125"
    "93279040680454942957577912920896778814494609832763671875",
    368,
    "1.0000000000000000015234388133035855383875390450151974382791625207620048100283188580157663"
    "004673368212241028702177558865266704933228669753172659365183590828373030075701115490420560"
    "634079475922775124733496586068376776193359182965216176711882163314879178483518575200071981"
    "810411402643941446323478834056492298312941259327904068045494295757791292089677881449460983"
    "2763671875e-136"
  },
  {
    1e-135,
    499,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000010000000000000000397101433570486440640372814"
    "601854186856466967779160881086984927240319116320263425424973183090679462397376099019927478"
    "147587391043659126324589921714976230226624401186646197529302879182099027553839378256942674"
    "237652165919805905446443724105883912965076387445672805029872471588429020886143684988326241"
    "212501922610289550874540509539656341075897216796875",
    364,
    "1.0000000000000000397101433570486440640372814601854186856466967779160881086984927240319116"
    "320263425424973183090679462397376099019927478147587391043659126324589921714976230226624401"
    "186646197529302879182099027553839378256942674237652165919805905446443724105883912965076387"
    "445672805029872471588429020886143684988326241212501922610289550874540509539656341075897216"
    "796875e-135"
  },
  {
    1e-134,
    498,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000100000000000000003971014335704864406403728146"
    "018541868564669677791608810869849272403191163202634254249731830906794623973760990199274781"
    "475873910436591263245899217149762302266244011866461975293028791820990275538393782569426742"
    "376521659198059054464437241058839129650763874456728050298724715884290208861436849883262412"
    "12501922610289550874540509539656341075897216796875",
    364,
    "1.0000000000000000397101433570486440640372814601854186856466967779160881086984927240319116"
    "320263425424973183090679462397376099019927478147587391043659126324589921714976230226624401"
    "186646197529302879182099027553839378256942674237652165919805905446443724105883912965076387"
    "445672805029872471588429020886143684988326241212501922610289550874540509539656341075897216"
    "796875e-134"
  },
  {
    1e-133,
    493,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000001000000000000000064149634265045481520453116605"
    "894360283961918702494701419847403998282244644224106204112176189932068065792611123541471552"
    "001281931788346904216873236411230365779485914028889935748064202920459010453544357763610865"
    "201119901287710041064503330419772913467920705224010255686832775799714045712057246057262961"
    "864454953175851414926000870764255523681640625",
    360,
    "1.0000000000000000641496342650454815204531166058943602839619187024947014198474039982822446"
    "442241062041121761899320680657926111235414715520012819317883469042168732364112303657794859"
    "140288899357480642029204590104535443577636108652011199012877100410645033304197729134679207"
    "052240102556868327757997140457120572460572629618644549531758514149260008707642555236816406"
    "25e-133"
  },
  {
    1e-132,
    487,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000009999999999999999859432633594556016599224441396"
    "257471693532085438431388241708879206811790051912624869446309711668782224166072145855555928"
    "251448840365572345916538286876868678049393688632253507311800918466789942308034551417118526"
    "062293115049276525200843869593517391950184311224750470481588015379157829994532549384186718"
    "988143383057831670157611370086669921875",
    354,
    "9.9999999999999998594326335945560165992244413962574716935320854384313882417088792068117900"
    "519126248694463097116687822241660721458555559282514488403655723459165382868768686780493936"
    "886322535073118009184667899423080345514171185260622931150492765252008438695935173919501843"
    "11224750470481588015379157829994532549384186718988143383057831670157611370086669921875e-13"
    "3"
  },
  {
    1e-131,
    486,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000099999999999999998594326335945560165992244413962"
    "574716935320854384313882417088792068117900519126248694463097116687822241660721458555559282"
    "514488403655723459165382868768686780493936886322535073118009184667899423080345514171185260"
    "622931150492765252008438695935173919501843112247504704815880153791578299945325493841867189"
    "88143383057831670157611370086669921875",
    354,
    "9.9999999999999998594326335945560165992244413962574716935320854384313882417088792068117900"
    "519126248694463097116687822241660721458555559282514488403655723459165382868768686780493936"
    "886322535073118009184667899423080345514171185260622931150492765252008438695935173919501843"
    "11224750470481588015379157829994532549384186718988143383057831670157611370086669921875e-13"
    "2"
  },
  {
    1e-130,
    482,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000001000000000000000086047418118610647881401704896449"
    "571956052357546917138946636828500010543023153302444919088851186321221937892218049128020570"
    "600305158848011711934670573822545212358946675276019552791754021117414995911810497742588727"
    "689266426889109856940634588690842264333341972440114105661488593017559271586363570539363054"
    "8343253394705243408679962158203125",
    352,
    "1.0000000000000000860474181186106478814017048964495719560523575469171389466368285000105430"
    "231533024449190888511863212219378922180491280205706003051588480117119346705738225452123589"
    "466752760195527917540211174149959118104977425887276892664268891098569406345886908422643333"
    "419724401141056614885930175592715863635705393630548343253394705243408679962158203125e-130"
  },
  {
    1e-129,
    477,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000009999999999999999258807705039625739270348876855314"
    "522973337191419987387506913235730835605944140385121599562431552124227038362125074121361778"
    "716313631827683194853235560054613604876221759949494382130945420159417717384419280934109333"
    "533385517507781179706383817482773534294846124960068136571893048547172361733897591462572052"
    "02346085570752620697021484375",
    347,
    "9.9999999999999992588077050396257392703488768553145229733371914199873875069132357308356059"
    "441403851215995624315521242270383621250741213617787163136318276831948532355600546136048762"
    "217599494943821309454201594177173844192809341093335333855175077811797063838174827735342948"
    "4612496006813657189304854717236173389759146257205202346085570752620697021484375e-130"
  },
  {
    1e-128,
    478,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000100000000000000005401408859568103309052834145426594"
    "802430862986593345890744772751462514653740544965836726232958009946209108101694078484369205"
    "457039971496303344480117025912844198468177541980552987602212529712035107713678381275316882"
    "208085186144350914663534730232928215257050045129264726062873538499086450376880826074188490"
    "792948869056999683380126953125",
    350,
    "1.0000000000000000540140885956810330905283414542659480243086298659334589074477275146251465"
    "374054496583672623295800994620910810169407848436920545703997149630334448011702591284419846"
    "817754198055298760221252971203510771367838127531688220808518614435091466353473023292821525"
    "7050045129264726062873538499086450376880826074188490792948869056999683380126953125e-128"
  },
  {
    1e-127,
    473,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000001000000000000000028387424977337341257829650700519048"
    "878913647721146514876096446726316829348807167429125801112295122054213632056054110302189217"
    "982592408524090652905647408395025685269855534834311543436608640884635209397812668884721728"
    "332391839310430911435954191518896407953322860235480539940849278936138837692998437844948966"
    "8128080666065216064453125",
    346,
    "1.0000000000000000283874249773373412578296507005190488789136477211465148760964467263168293"
    "488071674291258011122951220542136320560541103021892179825924085240906529056474083950256852"
    "698555348343115434366086408846352093978126688847217283323918393104309114359541915188964079"
    "533228602354805399408492789361388376929984378449489668128080666065216064453125e-127"
  },
  {
    1e-126,
    470,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000009999999999999999463821013986375273931938402885289716"
    "136497048578282939757723482037302143452926642955531252169831943490057953812167517693801409"
    "016090279194737188399742860480935271517119029264128791629553409303444326331050085056910283"
    "373197684845805587978962369256620251783545688525470337396137395610167062504070045747539552"
    "3943006992340087890625",
    343,
    "9.9999999999999994638210139863752739319384028852897161364970485782829397577234820373021434"
    "529266429555312521698319434900579538121675176938014090160902791947371883997428604809352715"
    "171190292641287916295534093034443263310500850569102833731976848458055879789623692566202517"
    "835456885254703373961373956101670625040700457475395523943006992340087890625e-127"
  },
  {
    1e-125,
    468,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000100000000000000001198636026159737848490248861812103342"
    "586085914848287069603162702179950634810426680241126593323273651317206472108663859562740256"
    "639573240316726609251278392563925364622680845273181058187798089377705404487113680891558833"
    "337742514526084090834260060024953139832920195889383870060217106111441140448015119090996449"
    "81324672698974609375",
    343,
    "1.0000000000000000119863602615973784849024886181210334258608591484828706960316270217995063"
    "481042668024112659332327365131720647210866385956274025663957324031672660925127839256392536"
    "462268084527318105818779808937770540448711368089155883333774251452608409083426006002495313"
    "983292019588938387006021710611144114044801511909099644981324672698974609375e-125"
  },
  {
    1e-124,
    464,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000999999999999999933261249626045557174852110622610559251"
    "207473999697378631720492440116355944730343794181497073733285916172541513242774404130688568"
    "651687022735009389466586472584381852808921821149092879170812937657908350751782845046116338"
    "108237152444502375806964190744523934359642231277672747416053260997165219592375251522753387"
    "6895904541015625",
    339,
    "9.9999999999999993326124962604555717485211062261055925120747399969737863172049244011635594"
    "473034379418149707373328591617254151324277440413068856865168702273500938946658647258438185"
    "280892182114909287917081293765790835075178284504611633810823715244450237580696419074452393"
    "43596422312776727474160532609971652195923752515227533876895904541015625e-125"
  },
  {
    1e-123,
    459,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000010000000000000000592214266429284712709327154154273179306"
    "528902377541659346183077708093965901286206073491272489324068713717786457929571105254309650"
    "421596314266201143405023974721767222775404316814412035022816674485414613427491872372715305"
    "389379409506440278639824459525358767109377954635382725138417411847591154128167545422911643"
    "98193359375",
    336,
    "1.0000000000000000592214266429284712709327154154273179306528902377541659346183077708093965"
    "901286206073491272489324068713717786457929571105254309650421596314266201143405023974721767"
    "222775404316814412035022816674485414613427491872372715305389379409506440278639824459525358"
    "76710937795463538272513841741184759115412816754542291164398193359375e-123"
  },
  {
    1e-122,
    458,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000100000000000000005922142664292847127093271541542731793065"
    "289023775416593461830777080939659012862060734912724893240687137177864579295711052543096504"
    "215963142662011434050239747217672227754043168144120350228166744854146134274918723727153053"
    "893794095064402786398244595253587671093779546353827251384174118475911541281675454229116439"
    "8193359375",
    336,
    "1.0000000000000000592214266429284712709327154154273179306528902377541659346183077708093965"
    "901286206073491272489324068713717786457929571105254309650421596314266201143405023974721767"
    "222775404316814412035022816674485414613427491872372715305389379409506440278639824459525358"
    "76710937795463538272513841741184759115412816754542291164398193359375e-122"
  },
  {
    1e-121,
    452,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000999999999999999978606913352123406249441128348024592375807"
    "823845397822060763705959165850577073723446921843936804969460044266880960840178432795831352"
    "257161863989250421196205543988005817624520940738275930141680382536270564530728234932207383"
    "289436306713370547490762619408233606108634384584336451257113864699022087734192609786987304"
    "6875",
    330,
    "9.9999999999999997860691335212340624944112834802459237580782384539782206076370595916585057"
    "707372344692184393680496946004426688096084017843279583135225716186398925042119620554398800"
    "581762452094073827593014168038253627056453072823493220738328943630671337054749076261940823"
    "36061086343845843364512571138646990220877341926097869873046875e-122"
  },
  {
    1e-120,
    451,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000009999999999999999786069133521234062494411283480245923758078"
    "238453978220607637059591658505770737234469218439368049694600442668809608401784327958313522"
    "571618639892504211962055439880058176245209407382759301416803825362705645307282349322073832"
    "894363067133705474907626194082336061086343845843364512571138646990220877341926097869873046"
    "875",
    330,
    "9.9999999999999997860691335212340624944112834802459237580782384539782206076370595916585057"
    "707372344692184393680496946004426688096084017843279583135225716186398925042119620554398800"
    "581762452094073827593014168038253627056453072823493220738328943630671337054749076261940823"
    "36061086343845843364512571138646990220877341926097869873046875e-121"
  },
  {
    1e-119,
    448,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000100000000000000001300243902286690065861087216344975527920838"
    "550613652878027500273213376354264381290203748481664600942221067190062254340279232015505994"
    "888221071175236010018076668185817385530952343819169425474153069848529657060407593031859636"
    "613162145711766919388084154269466416907644493801583795988818792466190643608570098876953125",
    329,
    "1.0000000000000000130024390228669006586108721634497552792083855061365287802750027321337635"
    "426438129020374848166460094222106719006225434027923201550599488822107117523601001807666818"
    "581738553095234381916942547415306984852965706040759303185963661316214571176691938808415426"
    "9466416907644493801583795988818792466190643608570098876953125e-119"
  },
  {
    1e-118,
    444,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000999999999999999985486018486272105131275077111109624956487936"
    "177545563404665965313759433170187741337944972112773177452477547884893180823304700696093795"
    "505933333750808977000588526776288870678657278259082964292612168713510938703403131829625904"
    "77536966211997183138626383512581772072279665507232859766872934415005147457122802734375",
    325,
    "9.9999999999999998548601848627210513127507711110962495648793617754556340466596531375943317"
    "018774133794497211277317745247754788489318082330470069609379550593333375080897700058852677"
    "628887067865727825908296429261216871351093870340313182962590477536966211997183138626383512"
    "581772072279665507232859766872934415005147457122802734375e-119"
  },
  {
    1e-117,
    441,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000010000000000000000295122913448237779750123491948538334728406551"
    "032911080056404251831583617661174558404929924389697086040505463100601609504848918304396409"
    "079771385532907740888735749073048460880431341512610290108818163483679497444796094119786429"
    "45372534116287611377588167235972038088633907781943577219863072969019412994384765625",
    324,
    "1.0000000000000000295122913448237779750123491948538334728406551032911080056404251831583617"
    "661174558404929924389697086040505463100601609504848918304396409079771385532907740888735749"
    "073048460880431341512610290108818163483679497444796094119786429453725341162876113775881672"
    "35972038088633907781943577219863072969019412994384765625e-117"
  },
  {
    1e-116,
    437,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000099999999999999999429127305798243970002253152785846665975847996"
    "269467232486085728763921888937368423845457617801248368279214756992657684874073892296296458"
    "634209471130533641824553640249206576053444943751857723626609824048234091161842734609645240"
    "9370236519234987384528701547848584195964105614251593578956089913845062255859375",
    320,
    "9.9999999999999999429127305798243970002253152785846665975847996269467232486085728763921888"
    "937368423845457617801248368279214756992657684874073892296296458634209471130533641824553640"
    "249206576053444943751857723626609824048234091161842734609645240937023651923498738452870154"
    "7848584195964105614251593578956089913845062255859375e-117"
  },
  {
    1e-115,
    431,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000001000000000000000050644902316928580940006239795051053560689960187"
    "648969414108165920469847492163718801716042195544043556805585554140311411531383574925646700"
    "958164858482036691250398010192514284548344979506500075651247213099315044196318651507957266"
    "9739126745191857734238466488461104242357890825587674044072628021240234375",
    316,
    "1.0000000000000000506449023169285809400062397950510535606899601876489694141081659204698474"
    "921637188017160421955440435568055855541403114115313835749256467009581648584820366912503980"
    "101925142845483449795065000756512472130993150441963186515079572669739126745191857734238466"
    "488461104242357890825587674044072628021240234375e-115"
  },
  {
    1e-114,
    430,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000010000000000000000506449023169285809400062397950510535606899601876"
    "489694141081659204698474921637188017160421955440435568055855541403114115313835749256467009"
    "581648584820366912503980101925142845483449795065000756512472130993150441963186515079572669"
    "739126745191857734238466488461104242357890825587674044072628021240234375",
    316,
    "1.0000000000000000506449023169285809400062397950510535606899601876489694141081659204698474"
    "921637188017160421955440435568055855541403114115313835749256467009581648584820366912503980"
    "101925142845483449795065000756512472130993150441963186515079572669739126745191857734238466"
    "488461104242357890825587674044072628021240234375e-114"
  },
  {
    1e-113,
    428,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000099999999999999997851225686547752015282709321304454232749766549970"
    "746913987161087044664288059247456074136569310364691806838493434673117115935842041341359424"
    "959507009586034180417515233594017381055868576195884123825652814958815449661778058123104924"
    "1207195755411835638060919569276852048034243125584907829761505126953125",
    314,
    "9.9999999999999997851225686547752015282709321304454232749766549970746913987161087044664288"
    "059247456074136569310364691806838493434673117115935842041341359424959507009586034180417515"
    "233594017381055868576195884123825652814958815449661778058123104924120719575541183563806091"
    "9569276852048034243125584907829761505126953125e-114"
  },
  {
    1e-112,
    424,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000999999999999999949659198684897095837955434580241937834220747624530"
    "869030176988850437361035963976864351495092127488262573504686429299075010548358608520351566"
    "167154741389802025686009193310529515444168264142347473060254169697398616737432210540562148"
    "125004249902260523746627858543423172932307352311909198760986328125",
    311,
    "9.9999999999999994965919868489709583795543458024193783422074762453086903017698885043736103"
    "596397686435149509212748826257350468642929907501054835860852035156616715474138980202568600"
    "919331052951544416826414234747306025416969739861673743221054056214812500424990226052374662"
    "7858543423172932307352311909198760986328125e-113"
  },
  {
    1e-111,
    421,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000010000000000000000881538779516831325493393960176944394019499534253785"
    "495567111745464819138901807658070228739768130498089489298764329731365248366552720079164466"
    "211484428483929624338962779328221319938522536615175460023692731730268401631043898549001949"
    "195234763482210724552280700638817734215990640223026275634765625",
    310,
    "1.0000000000000000881538779516831325493393960176944394019499534253785495567111745464819138"
    "901807658070228739768130498089489298764329731365248366552720079164466211484428483929624338"
    "962779328221319938522536615175460023692731730268401631043898549001949195234763482210724552"
    "280700638817734215990640223026275634765625e-111"
  },
  {
    1e-110,
    417,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000100000000000000005122196348054018942630367296770710565055549854515250"
    "141630205836087003312905628875564383960756356672991548315909866005345435977616174456581183"
    "341678912610204596779305536687743424726985645640552655113857891285931391625847537107671577"
    "43183492959649261329346803250928132911212742328643798828125",
    307,
    "1.0000000000000000512219634805401894263036729677071056505554985451525014163020583608700331"
    "290562887556438396075635667299154831590986600534543597761617445658118334167891261020459677"
    "930553668774342472698564564055265511385789128593139162584753710767157743183492959649261329"
    "346803250928132911212742328643798828125e-110"
  },
  {
    1e-109,
    414,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000999999999999999992130900326711480429446516087727371648324370736790824"
    "391647472463891023911257125473437384616764393803461968411363759120541596769585323204796173"
    "046143170436579622027899261365917852738020928226295429169468096591271921305012196959149141"
    "99014601235509201726525674303047708235681056976318359375",
    304,
    "9.9999999999999999213090032671148042944651608772737164832437073679082439164747246389102391"
    "125712547343738461676439380346196841136375912054159676958532320479617304614317043657962202"
    "789926136591785273802092822629542916946809659127192130501219695914914199014601235509201726"
    "525674303047708235681056976318359375e-110"
  },
  {
    1e-108,
    411,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000010000000000000000394037508497744476269322415917111588501092729834801660"
    "113711411814742312854964560992025486094037321446247802095516798668718071748464602936087013"
    "426599349689526986400241457751309683634893507696803267447567496057055172677827362532024478"
    "52708639242959309800795608680346049368381500244140625",
    303,
    "1.0000000000000000394037508497744476269322415917111588501092729834801660113711411814742312"
    "854964560992025486094037321446247802095516798668718071748464602936087013426599349689526986"
    "400241457751309683634893507696803267447567496057055172677827362532024478527086392429593098"
    "00795608680346049368381500244140625e-108"
  },
  {
    1e-107,
    407,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000100000000000000000158547043132407386894366118852412908868135118612869271"
    "559220620740766538610499159859041741529226147169453077100134326980763885063755062255867870"
    "544652334305423735032423824776047586311461273497240868452582719415864049756630481795980316"
    "2658537732665475772364516160450875759124755859375",
    300,
    "1.0000000000000000015854704313240738689436611885241290886813511861286927155922062074076653"
    "861049915985904174152922614716945307710013432698076388506375506225586787054465233430542373"
    "503242382477604758631146127349724086845258271941586404975663048179598031626585377326654757"
    "72364516160450875759124755859375e-107"
  },
  {
    1e-106,
    405,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000999999999999999941076221761803475856161932543424881470396676310366335442"
    "345910248901159947078648397611007504713908395006131669320804714504969531903295148878642485"
    "905064741616699286804386203967687862515031879439739788156351335683637665220014521571571658"
    "57837531619534132687476812861859798431396484375",
    298,
    "9.9999999999999994107622176180347585616193254342488147039667631036633544234591024890115994"
    "707864839761100750471390839500613166932080471450496953190329514887864248590506474161669928"
    "680438620396768786251503187943973978815635133568363766522001452157157165857837531619534132"
    "687476812861859798431396484375e-107"
  },
  {
    1e-105,
    401,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000009999999999999999652799212296117150612746240014645805177105462606712783516"
    "444286323037621226891856780027714689452496256814913099930201366260372593969973383506569737"
    "216481821917145122123270214848030627548642216528073467041416790736387981585306401268642602"
    "1044028278339510507066734135150909423828125",
    295,
    "9.9999999999999996527992122961171506127462400146458051771054626067127835164442863230376212"
    "268918567800277146894524962568149130999302013662603725939699733835065697372164818219171451"
    "221232702148480306275486422165280734670414167907363879815853064012686426021044028278339510"
    "507066734135150909423828125e-106"
  },
  {
    1e-104,
    398,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000099999999999999992655400208111853233309431766860106204200835434018336969676"
    "679921885959864171232602937594912617510365660091588491747546123232889540707383519543379321"
    "511467727169015155962171345741874237113247411189925302767712964963698545690485043839609759"
    "9136336242509059957228600978851318359375",
    293,
    "9.9999999999999992655400208111853233309431766860106204200835434018336969676679921885959864"
    "171232602937594912617510365660091588491747546123232889540707383519543379321511467727169015"
    "155962171345741874237113247411189925302767712964963698545690485043839609759913633624250905"
    "9957228600978851318359375e-105"
  },
  {
    1e-103,
    395,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000999999999999999957534737399913078515638562734891876822570107876573696620668"
    "902749614929426493813748277407000391220431865376224977911201547295586599012637719612337620"
    "341481207709640081785959879326198678117872144625727968848769188838435618205482189170627688"
    "1794934752178960479795932769775390625",
    291,
    "9.9999999999999995753473739991307851563856273489187682257010787657369662066890274961492942"
    "649381374827740700039122043186537622497791120154729558659901263771961233762034148120770964"
    "008178595987932619867811787214462572796884876918883843561820548218917062768817949347521789"
    "60479795932769775390625e-104"
  },
  {
    1e-102,
    391,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000009999999999999999327501491448774415696031666818592249981207050474614350815472"
    "199250106647986686235731562407010183270116538079529295626092953222336454615957002695020961"
    "600380588940492640545627418002336325295537184445480159114575574772754891649767885510036169"
    "449676890508271753787994384765625",
    288,
    "9.9999999999999993275014914487744156960316668185922499812070504746143508154721992501066479"
    "866862357315624070101832701165380795292956260929532223364546159570026950209616003805889404"
    "926405456274180023363252955371844454801591145755747727548916497678855100361694496768905082"
    "71753787994384765625e-103"
  },
  {
    1e-101,
    388,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000100000000000000005171617276904849891057306773641595375547783862720029046933129"
    "748311113501222953641373783893800821542866933565876163585210479432782250659739311511261223"
    "096517320888518916526900192486585135348216411421179001055338801084410855940271152519915887"
    "069146265275776386260986328125",
    287,
    "1.0000000000000000517161727690484989105730677364159537554778386272002904693312974831111350"
    "122295364137378389380082154286693356587616358521047943278225065973931151126122309651732088"
    "851891652690019248658513534821641142117900105533880108441085594027115251991588706914626527"
    "5776386260986328125e-101"
  },
  {
    1e-100,
    381,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000001000000000000000019991899802602883619647760788534159420182603005936595699255543"
    "467617676288613292989582746074810911850798528270539749654022268436041961263608356283141278"
    "717942724928942469080665891630593000434578602301450250794499868559143387555798732080347690"
    "49845635890960693359375",
    281,
    "1.0000000000000000199918998026028836196477607885341594201826030059365956992555434676176762"
    "886132929895827460748109118507985282705397496540222684360419612636083562831412787179427249"
    "289424690806658916305930004345786023014502507944998685591433875557987320803476904984563589"
    "0960693359375e-100"
  },
  {
    1e-99,
    380,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000010000000000000000199918998026028836196477607885341594201826030059365956992555434"
    "676176762886132929895827460748109118507985282705397496540222684360419612636083562831412787"
    "179427249289424690806658916305930004345786023014502507944998685591433875557987320803476904"
    "9845635890960693359375",
    281,
    "1.0000000000000000199918998026028836196477607885341594201826030059365956992555434676176762"
    "886132929895827460748109118507985282705397496540222684360419612636083562831412787179427249"
    "289424690806658916305930004345786023014502507944998685591433875557987320803476904984563589"
    "0960693359375e-99"
  },
  {
    1e-98,
    378,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000099999999999999993877776100850210847487897500195676592182679981550153708786161318"
    "795442195615570982374570834502581469144926135669172098693100215308376520911937367969564096"
    "503268600095092683852564654833161663275969181098046581174622430963254762770198169619106920"
    "43602466583251953125",
    279,
    "9.9999999999999993877776100850210847487897500195676592182679981550153708786161318795442195"
    "615570982374570834502581469144926135669172098693100215308376520911937367969564096503268600"
    "095092683852564654833161663275969181098046581174622430963254762770198169619106920436024665"
    "83251953125e-99"
  },
  {
    1e-97,
    375,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000001000000000000000036234727561423038648601517945849638119853763644023607421534329523"
    "550327155104809622750153620767931282668381653309355387440521692633600474506152803830406268"
    "524732714540777529093940647045277194942384399544207791050597409045555541418082157179014757"
    "27558135986328125",
    278,
    "1.0000000000000000362347275614230386486015179458496381198537636440236074215343295235503271"
    "551048096227501536207679312826683816533093553874405216926336004745061528038304062685247327"
    "145407775290939406470452771949423843995442077910505974090455555414180821571790147572755813"
    "5986328125e-97"
  },
  {
    1e-96,
    370,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000009999999999999999062921054908617984169714606873258085224844785393275136433040410760"
    "891202231726765574108932531117758277095545911525095200944956399004867873237806383173858638"
    "686704297543099416695485154270631120321276147925518186447666098282116564632815425284206867"
    "218017578125",
    273,
    "9.9999999999999990629210549086179841697146068732580852248447853932751364330404107608912022"
    "317267655741089325311177582770955459115250952009449563990048678732378063831738586386867042"
    "975430994166954851542706311203212761479255181864476660982821165646328154252842068672180175"
    "78125e-97"
  },
  {
    1e-95,
    368,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000099999999999999998945538361602099216521469733278105946480082100633301366137142568246"
    "429265960924171922801988841171531888320391093289087519595231364967954712049882424571892284"
    "855029201764919762115947966272012509469195703361164098449832132731174283435393590480089187"
    "6220703125",
    272,
    "9.9999999999999998945538361602099216521469733278105946480082100633301366137142568246429265"
    "960924171922801988841171531888320391093289087519595231364967954712049882424571892284855029"
    "201764919762115947966272012509469195703361164098449832132731174283435393590480089187622070"
    "3125e-96"
  },
  {
    1e-94,
    364,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000999999999999999956190072365957314665917402674598959087874284019530813654144471839914"
    "223685034615654501169234291739522413744183020738333155369644150002443201811549874385699256"
    "598347112313495240515093968457319869666220137187712048605636727671708285924978554248809814"
    "453125",
    269,
    "9.9999999999999995619007236595731466591740267459895908787428401953081365414447183991422368"
    "503461565450116923429173952241374418302073833315536964415000244320181154987438569925659834"
    "711231349524051509396845731986966622013718771204860563672767170828592497855424880981445312"
    "5e-95"
  },
  {
    1e-93,
    360,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000009999999999999999029655743658554306670417312215075984847918248406472936425813456918341"
    "133257152139509382081876997782480626086183612942658904373729505190769319119108802525415094"
    "752352637763714314840768576368315096250411029094257511773413682476530084386467933654785156"
    "25",
    266,
    "9.9999999999999990296557436585543066704173122150759848479182484064729364258134569183411332"
    "571521395093820818769977824806260861836129426589043737295051907693191191088025254150947523"
    "5263776371431484076857636831509625041102909425751177341368247653008438646793365478515625e-"
    "94"
  },
  {
    1e-92,
    358,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000099999999999999998812477116601844506524280554645377544972375952686092566108234752876228"
    "990062625667663894586224691628702442552181640477351432900686969246296375133327086559390487"
    "221422143576952593370423494961288569092755775468382706261394332614145241677761077880859375",
    265,
    "9.9999999999999998812477116601844506524280554645377544972375952686092566108234752876228990"
    "062625667663894586224691628702442552181640477351432900686969246296375133327086559390487221"
    "422143576952593370423494961288569092755775468382706261394332614145241677761077880859375e-9"
    "3"
  },
  {
    1e-91,
    355,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "001000000000000000022188449886083650824523235276432246235696533401346378468482748263533560"
    "530590673766919240932065771502609152283198448976563885660437361817376487102227110814863031"
    "005804499528763713555185874725436117282139692787057416722973357536830008029937744140625",
    264,
    "1.0000000000000000221884498860836508245232352764322462356965334013463784684827482635335605"
    "305906737669192409320657715026091522831984489765638856604373618173764871022271108148630310"
    "05804499528763713555185874725436117282139692787057416722973357536830008029937744140625e-91"
  },
  {
    1e-90,
    351,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "009999999999999999949375069100314862170988914924494696069183143017580162225624276757165440"
    "266191400946950048762106873301413708740928136141242403375832263338462984870621146380965039"
    "72538048521373489674425134635395776198474142304473133435749332420527935028076171875",
    260,
    "9.9999999999999999493750691003148621709889149244946960691831430175801622256242767571654402"
    "661914009469500487621068733014137087409281361412424033758322633384629848706211463809650397"
    "2538048521373489674425134635395776198474142304473133435749332420527935028076171875e-91"
  },
  {
    1e-89,
    348,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "100000000000000003853901567171494958897784154682191221296346486109939581603494061622377043"
    "297359397025378256557882200608982112866183019402767285414984310749460027132610852092294722"
    "57643701331978478836423187794603219323390230766079866953077726066112518310546875",
    259,
    "1.0000000000000000385390156717149495889778415468219122129634648610993958160349406162237704"
    "329735939702537825655788220060898211286618301940276728541498431074946002713261085209229472"
    "257643701331978478836423187794603219323390230766079866953077726066112518310546875e-89"
  },
  {
    1e-88,
    345,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "999999999999999933895394643674637496468361416328049958455103518680084791700909559006427057"
    "722904668912716111095298783813540517696190402259434814389962850738675989092523202139483418"
    "02119826481938818152107683370541217106330922348433887236751616001129150390625",
    256,
    "9.9999999999999993389539464367463749646836141632804995845510351868008479170090955900642705"
    "772290466891271611109529878381354051769619040225943481438996285073867598909252320213948341"
    "802119826481938818152107683370541217106330922348433887236751616001129150390625e-89"
  },
  {
    1e-87,
    342,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
    "000000000000000176102914661068871704759455207231397620617925926155336111681344047803017579"
    "234561099855692746821173616345650064687022356740252461978670561434154148793914571662544642"
    "15735759522155943218070390309339980083880305983257130719721317291259765625",
    255,
    "1.0000000000000000176102914661068871704759455207231397620617925926155336111681344047803017"
    "579234561099855692746821173616345650064687022356740252461978670561434154148793914571662544"
    "64215735759522155943218070390309339980083880305983257130719721317291259765625e-87"
  },
  {
    1e-86,
    338,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000100"
    "000000000000008458220892405268690968201280423921160494714385176389266674191428139940151808"
    "389726284385180555157222389138459748671170240569759164419042046720695550888606118767130117"
    "1365755284370152575665235592482227300337171982391737401485443115234375",
    252,
    "1.0000000000000000845822089240526869096820128042392116049471438517638926667419142813994015"
    "180838972628438518055515722238913845974867117024056975916441904204672069555088860611876713"
    "01171365755284370152575665235592482227300337171982391737401485443115234375e-86"
  },
  {
    1e-85,
    334,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000999"
    "999999999999977427140991339407326952305150613496656330581837126518177823866478808841901827"
    "191418270599756160444444280473251857896555635021838930073037549140490501694694753404362042"
    "357762064827417603513483139454631754006186383776366710662841796875",
    248,
    "9.9999999999999997742714099133940732695230515061349665633058183712651817782386647880884190"
    "182719141827059975616044444428047325185789655563502183893007303754914049050169469475340436"
    "2042357762064827417603513483139454631754006186383776366710662841796875e-86"
  },
  {
    1e-84,
    328,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000010000"
    "000000000000345765105554531564377414825658805446289260815782664512385801586401904736971641"
    "012020430008491690459267396259695265979672460489070442689751054426051721967568516800629111"
    "620251152502095886610844477360160453827120363712310791015625",
    244,
    "1.0000000000000000345765105554531564377414825658805446289260815782664512385801586401904736"
    "971641012020430008491690459267396259695265979672460489070442689751054426051721967568516800"
    "629111620251152502095886610844477360160453827120363712310791015625e-84"
  },
  {
    1e-83,
    327,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000100000"
    "000000000003457651055545315643774148256588054462892608157826645123858015864019047369716410"
    "120204300084916904592673962596952659796724604890704426897510544260517219675685168006291116"
    "20251152502095886610844477360160453827120363712310791015625",
    244,
    "1.0000000000000000345765105554531564377414825658805446289260815782664512385801586401904736"
    "971641012020430008491690459267396259695265979672460489070442689751054426051721967568516800"
    "629111620251152502095886610844477360160453827120363712310791015625e-83"
  },
  {
    1e-82,
    320,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000999999"
    "999999999961425317513387557575931335474338723224003841909607336920812104673621984999132856"
    "678814327450118036029191910490910660160383934259858098175033375898393954117365887165799092"
    "5684107090358476721547731358441524207592010498046875",
    237,
    "9.9999999999999996142531751338755757593133547433872322400384190960733692081210467362198499"
    "913285667881432745011803602919191049091066016038393425985809817503337589839395411736588716"
    "57990925684107090358476721547731358441524207592010498046875e-83"
  },
  {
    1e-81,
    319,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000009999999"
    "999999999614253175133875575759313354743387232240038419096073369208121046736219849991328566"
    "788143274501180360291919104909106601603839342598580981750333758983939541173658871657990925"
    "684107090358476721547731358441524207592010498046875",
    237,
    "9.9999999999999996142531751338755757593133547433872322400384190960733692081210467362198499"
    "913285667881432745011803602919191049091066016038393425985809817503337589839395411736588716"
    "57990925684107090358476721547731358441524207592010498046875e-82"
  },
  {
    1e-80,
    318,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000099999999"
    "999999996142531751338755757593133547433872322400384190960733692081210467362198499913285667"
    "881432745011803602919191049091066016038393425985809817503337589839395411736588716579909256"
    "84107090358476721547731358441524207592010498046875",
    237,
    "9.9999999999999996142531751338755757593133547433872322400384190960733692081210467362198499"
    "913285667881432745011803602919191049091066016038393425985809817503337589839395411736588716"
    "57990925684107090358476721547731358441524207592010498046875e-81"
  },
  {
    1e-79,
    313,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000999999999"
    "999999998878728350925144193178130785208135783324028619960803451509348304505051212524853874"
    "707408230432153096736340815962020317497336959217417624670274052264414348782613129120472130"
    "243434116803058486766531132161617279052734375",
    233,
    "9.9999999999999999887872835092514419317813078520813578332402861996080345150934830450505121"
    "252485387470740823043215309673634081596202031749733695921741762467027405226441434878261312"
    "9120472130243434116803058486766531132161617279052734375e-80"
  },
  {
    1e-78,
    312,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000009999999999"
    "999999988787283509251441931781307852081357833240286199608034515093483045050512125248538747"
    "074082304321530967363408159620203174973369592174176246702740522644143487826131291204721302"
    "43434116803058486766531132161617279052734375",
    233,
    "9.9999999999999999887872835092514419317813078520813578332402861996080345150934830450505121"
    "252485387470740823043215309673634081596202031749733695921741762467027405226441434878261312"
    "9120472130243434116803058486766531132161617279052734375e-79"
  },
  {
    1e-77,
    306,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000099999999999"
    "999992696817954285297788806428378833886366942927013608214771257064053320956408281221925859"
    "269313222904832705103459186340881583960377644752428136742959683313070446249927954342337152"
    "46019613687167293392121791839599609375",
    228,
    "9.9999999999999992696817954285297788806428378833886366942927013608214771257064053320956408"
    "281221925859269313222904832705103459186340881583960377644752428136742959683313070446249927"
    "95434233715246019613687167293392121791839599609375e-78"
  },
  {
    1e-76,
    305,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000999999999999"
    "999926968179542852977888064283788338863669429270136082147712570640533209564082812219258592"
    "693132229048327051034591863408815839603776447524281367429596833130704462499279543423371524"
    "6019613687167293392121791839599609375",
    228,
    "9.9999999999999992696817954285297788806428378833886366942927013608214771257064053320956408"
    "281221925859269313222904832705103459186340881583960377644752428136742959683313070446249927"
    "95434233715246019613687167293392121791839599609375e-77"
  },
  {
    1e-75,
    299,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000009999999999999"
    "999576500137009637688449128585070030864380243670892037074945178225156289719248229433614683"
    "049074623730287834319141454830565469032677626787745099765644838117260390811886962975085779"
    "7034768736921250820159912109375",
    223,
    "9.9999999999999995765001370096376884491285850700308643802436708920370749451782251562897192"
    "482294336146830490746237302878343191414548305654690326776267877450997656448381172603908118"
    "869629750857797034768736921250820159912109375e-76"
  },
  {
    1e-74,
    298,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000099999999999999"
    "995765001370096376884491285850700308643802436708920370749451782251562897192482294336146830"
    "490746237302878343191414548305654690326776267877450997656448381172603908118869629750857797"
    "034768736921250820159912109375",
    223,
    "9.9999999999999995765001370096376884491285850700308643802436708920370749451782251562897192"
    "482294336146830490746237302878343191414548305654690326776267877450997656448381172603908118"
    "869629750857797034768736921250820159912109375e-75"
  },
  {
    1e-73,
    295,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000999999999999999"
    "996922761423345581269679034146893291581826091189199304015410215453125813962596670213149087"
    "979761028647000900486666538084652246616646076525732436683076683433657106032411976404006281"
    "882175244390964508056640625",
    221,
    "9.9999999999999999692276142334558126967903414689329158182609118919930401541021545312581396"
    "259667021314908797976102864700090048666653808465224661664607652573243668307668343365710603"
    "2411976404006281882175244390964508056640625e-74"
  },
  {
    1e-72,
    291,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000009999999999999999"
    "655045632454401313298660936349811274667847119092028267986963011031283403323776887318044615"
    "219221041524269256286496940621679719375393583247544685882023860675626861574394332876636326"
    "54584944248199462890625",
    218,
    "9.9999999999999996550456324544013132986609363498112746678471190920282679869630110312834033"
    "237768873180446152192210415242692562864969406216797193753935832475446858820238606756268615"
    "7439433287663632654584944248199462890625e-73"
  },
  {
    1e-71,
    287,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000099999999999999991"
    "523544616079141142616538881592166488271850506120846325195403814313238252402731836165305918"
    "937982496110856585582274362619313245096860920318971963640351028181161435748336430151539389"
    "0440464019775390625",
    215,
    "9.9999999999999991523544616079141142616538881592166488271850506120846325195403814313238252"
    "402731836165305918937982496110856585582274362619313245096860920318971963640351028181161435"
    "7483364301515393890440464019775390625e-72"
  },
  {
    1e-70,
    285,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000999999999999999995"
    "666033496229363272086516526416805017224436017999444926741658879125915017387910953895302921"
    "447471667217941492345864323752875629481807797693317959281711539013329237413074679352575913"
    "07163238525390625",
    214,
    "9.9999999999999999566603349622936327208651652641680501722443601799944492674165887912591501"
    "738791095389530292144747166721794149234586432375287562948180779769331795928171153901332923"
    "741307467935257591307163238525390625e-71"
  },
  {
    1e-69,
    280,
    "0.0000000000000000000000000000000000000000000000000000000000000000000009999999999999999634"
    "937985620541825337180654422187489634220636352830522568266105847285020200436739169984054286"
    "204129847741912377366160447289783580765283598918786301304310361326432854411905282177031040"
    "191650390625",
    210,
    "9.9999999999999996349379856205418253371806544221874896342206363528305225682661058472850202"
    "004367391699840542862041298477419123773661604472897835807652835989187863013043103613264328"
    "54411905282177031040191650390625e-70"
  },
  {
    1e-68,
    278,
    "0.0000000000000000000000000000000000000000000000000000000000000000000100000000000000006644"
    "495035141476089649710891165252833558965525997550880055476512680022361154523243506847740566"
    "700076859419205248621053760544962657342256085648448341452864535083833175121981184929609298"
    "7060546875",
    210,
    "1.0000000000000000664449503514147608964971089116525283355896552599755088005547651268002236"
    "115452324350684774056670007685941920524862105376054496265734225608564844834145286453508383"
    "31751219811849296092987060546875e-68"
  },
  {
    1e-67,
    275,
    "0.0000000000000000000000000000000000000000000000000000000000000000000999999999999999942903"
    "568204182066861162256748331993088988545310344560948080979676314157701743362213384391033211"
    "095428010191074786697146153684104377149519698957459473611514289004276179184671491384506225"
    "5859375",
    207,
    "9.9999999999999994290356820418206686116225674833199308898854531034456094808097967631415770"
    "174336221338439103321109542801019107478669714615368410437714951969895745947361151428900427"
    "61791846714913845062255859375e-68"
  },
  {
    1e-66,
    271,
    "0.0000000000000000000000000000000000000000000000000000000000000000009999999999999999758479"
    "367767774519372515506585508024880821746302461470420739891297771086110238609391668140658660"
    "035188325913355065673838741549102961556640076313325245227492388266909983940422534942626953"
    "125",
    204,
    "9.9999999999999997584793677677745193725155065855080248808217463024614704207398912977710861"
    "102386093916681406586600351883259133550656738387415491029615566400763133252452274923882669"
    "09983940422534942626953125e-67"
  },
  {
    1e-65,
    268,
    "0.0000000000000000000000000000000000000000000000000000000000000000099999999999999992313694"
    "706062483581550868040220070744953236771840360929168517400423638715617506297791493721361815"
    "057351675091835477500352140162082574583311375313564306477331911082728765904903411865234375",
    202,
    "9.9999999999999992313694706062483581550868040220070744953236771840360929168517400423638715"
    "617506297791493721361815057351675091835477500352140162082574583311375313564306477331911082"
    "728765904903411865234375e-66"
  },
  {
    1e-64,
    265,
    "0.0000000000000000000000000000000000000000000000000000000000000000999999999999999965305738"
    "833546928712902976607280783480372213247877639491996226104668964320054101346916438695416432"
    "929769423252076208907803604252402073697828855693148231154054883518256247043609619140625",
    200,
    "9.9999999999999996530573883354692871290297660728078348037221324787763949199622610466896432"
    "005410134691643869541643292976942325207620890780360425240207369782885569314823115405488351"
    "8256247043609619140625e-65"
  },
  {
    1e-63,
    262,
    "0.0000000000000000000000000000000000000000000000000000000000000010000000000000000665108390"
    "885599516666492874994729659543878425186153119727427511457071495133637934325200422517323105"
    "847758368530076502780808905681852605731451018311606304678207379765808582305908203125",
    199,
    "1.0000000000000000665108390885599516666492874994729659543878425186153119727427511457071495"
    "133637934325200422517323105847758368530076502780808905681852605731451018311606304678207379"
    "765808582305908203125e-63"
  },
  {
    1e-62,
    255,
    "0.0000000000000000000000000000000000000000000000000000000000000100000000000000003952281235"
    "388981221231693792822171729465034137975193264454367780143030012848120887635908130338140987"
    "67741265594259325793402808839764107397274361943573239841498434543609619140625",
    193,
    "1.0000000000000000395228123538898122123169379282217172946503413797519326445436778014303001"
    "284812088763590813033814098767741265594259325793402808839764107397274361943573239841498434"
    "543609619140625e-62"
  },
  {
    1e-61,
    254,
    "0.0000000000000000000000000000000000000000000000000000000000001000000000000000039522812353"
    "889812212316937928221717294650341379751932644543677801430300128481208876359081303381409876"
    "7741265594259325793402808839764107397274361943573239841498434543609619140625",
    193,
    "1.0000000000000000395228123538898122123169379282217172946503413797519326445436778014303001"
    "284812088763590813033814098767741265594259325793402808839764107397274361943573239841498434"
    "543609619140625e-61"
  },
  {
    1e-60,
    251,
    "0.0000000000000000000000000000000000000000000000000000000000009999999999999999704334639131"
    "342552092261230258185207257223384642616815643540500400815657031817924125870212756031040642"
    "8974820785673527056432009240175516617821216414085938595235347747802734375",
    190,
    "9.9999999999999997043346391313425520922612302581852072572233846426168156435405004008156570"
    "318179241258702127560310406428974820785673527056432009240175516617821216414085938595235347"
    "747802734375e-61"
  },
  {
    1e-59,
    245,
    "0.0000000000000000000000000000000000000000000000000000000000100000000000000002570494266573"
    "870081169877494774107798086474079665388242850575224916055324342132558360466929782574871427"
    "7250889112093117585088725661479625017591388314031064510345458984375",
    186,
    "1.0000000000000000257049426657387008116987749477410779808647407966538824285057522491605532"
    "434213255836046692978257487142772508891120931175850887256614796250175913883140310645103454"
    "58984375e-59"
  },
  {
    1e-58,
    244,
    "0.0000000000000000000000000000000000000000000000000000000001000000000000000025704942665738"
    "700811698774947741077980864740796653882428505752249160553243421325583604669297825748714277"
    "250889112093117585088725661479625017591388314031064510345458984375",
    186,
    "1.0000000000000000257049426657387008116987749477410779808647407966538824285057522491605532"
    "434213255836046692978257487142772508891120931175850887256614796250175913883140310645103454"
    "58984375e-58"
  },
  {
    1e-57,
    242,
    "0.0000000000000000000000000000000000000000000000000000000009999999999999999549574498624050"
    "104405337804876802046942824658111918653223915734215394491919147231247020798293807635622932"
    "4745710523507339850487508903231770318598137237131595611572265625",
    184,
    "9.9999999999999995495744986240501044053378048768020469428246581119186532239157342153944919"
    "191472312470207982938076356229324745710523507339850487508903231770318598137237131595611572"
    "265625e-58"
  },
  {
    1e-56,
    238,
    "0.0000000000000000000000000000000000000000000000000000000100000000000000003985444122640543"
    "888593177383975325263818119579374628584972858801468477405372264607538518719151474574467405"
    "157551346472642240549577596908648047246970236301422119140625",
    182,
    "1.0000000000000000398544412264054388859317738397532526381811957937462858497285880146847740"
    "537226460753851871915147457446740515755134647264224054957759690864804724697023630142211914"
    "0625e-56"
  },
  {
    1e-55,
    235,
    "0.0000000000000000000000000000000000000000000000000000000999999999999999994576045832271877"
    "048386177385314293734768539803050594901815513565007267460758420501687529931709955247404289"
    "379029075578142991831409602809799253009259700775146484375",
    179,
    "9.9999999999999999457604583227187704838617738531429373476853980305059490181551356500726746"
    "075842050168752993170995524740428937902907557814299183140960280979925300925970077514648437"
    "5e-56"
  },
  {
    1e-54,
    230,
    "0.0000000000000000000000000000000000000000000000000000010000000000000000307987621475787265"
    "184226545488654608574986645956071476601459731247492727351298009606456557395537876452200991"
    "3621658689676652276290269583114422857761383056640625",
    176,
    "1.0000000000000000307987621475787265184226545488654608574986645956071476601459731247492727"
    "3512980096064565573955378764522009913621658689676652276290269583114422857761383056640625e-"
    "54"
  },
  {
    1e-53,
    229,
    "0.0000000000000000000000000000000000000000000000000000100000000000000003079876214757872651"
    "842265454886546085749866459560714766014597312474927273512980096064565573955378764522009913"
    "621658689676652276290269583114422857761383056640625",
    176,
    "1.0000000000000000307987621475787265184226545488654608574986645956071476601459731247492727"
    "3512980096064565573955378764522009913621658689676652276290269583114422857761383056640625e-"
    "53"
  },
  {
    1e-52,
    221,
    "0.0000000000000000000000000000000000000000000000000001000000000000000007616223705782342857"
    "599309164192713898951384728370953894814479006514389359532117466912455222533734910617980891"
    "6165796528474629667471162974834442138671875",
    169,
    "1.0000000000000000076162237057823428575993091641927138989513847283709538948144790065143893"
    "595321174669124552225337349106179808916165796528474629667471162974834442138671875e-52"
  },
  {
    1e-51,
    220,
    "0.0000000000000000000000000000000000000000000000000010000000000000000076162237057823428575"
    "993091641927138989513847283709538948144790065143893595321174669124552225337349106179808916"
    "165796528474629667471162974834442138671875",
    169,
    "1.0000000000000000076162237057823428575993091641927138989513847283709538948144790065143893"
    "595321174669124552225337349106179808916165796528474629667471162974834442138671875e-51"
  },
  {
    1e-50,
    219,
    "0.0000000000000000000000000000000000000000000000000100000000000000000761622370578234285759"
    "930916419271389895138472837095389481447900651438935953211746691245522253373491061798089161"
    "65796528474629667471162974834442138671875",
    169,
    "1.0000000000000000076162237057823428575993091641927138989513847283709538948144790065143893"
    "595321174669124552225337349106179808916165796528474629667471162974834442138671875e-50"
  },
  {
    1e-49,
    215,
    "0.0000000000000000000000000000000000000000000000000999999999999999936399465612583852251549"
    "992142478035242294140976221366647716129075296827629696033774164063234248132909920273644205"
    "3573995281112729571759700775146484375",
    165,
    "9.9999999999999993639946561258385225154999214247803524229414097622136664771612907529682762"
    "96960337741640632342481329099202736442053573995281112729571759700775146484375e-50"
  },
  {
    1e-48,
    209,
    "0.0000000000000000000000000000000000000000000000009999999999999999743817365956230472414429"
    "612207258638591780043107011465128352490386128605522752784102965389613337873102923841761580"
    "0926796509884297847747802734375",
    160,
    "9.9999999999999997438173659562304724144296122072586385917800431070114651283524903861286055"
    "227527841029653896133378731029238417615800926796509884297847747802734375e-49"
  },
  {
    1e-47,
    208,
    "0.0000000000000000000000000000000000000000000000099999999999999997438173659562304724144296"
    "122072586385917800431070114651283524903861286055227527841029653896133378731029238417615800"
    "926796509884297847747802734375",
    160,
    "9.9999999999999997438173659562304724144296122072586385917800431070114651283524903861286055"
    "227527841029653896133378731029238417615800926796509884297847747802734375e-48"
  },
  {
    1e-46,
    205,
    "0.0000000000000000000000000000000000000000000001000000000000000022999043453913216828505961"
    "640883084488789349378835264740187722591657382693176711544546107892003424942768685657057403"
    "659564442932605743408203125",
    159,
    "1.0000000000000000229990434539132168285059616408830844887893493788352647401877225916573826"
    "93176711544546107892003424942768685657057403659564442932605743408203125e-46"
  },
  {
    1e-45,
    202,
    "0.0000000000000000000000000000000000000000000009999999999999999841051979672810811588555613"
    "047573079851002733243279701583057437492217649804555650371464527474677148367876444723378881"
    "462849676609039306640625",
    156,
    "9.9999999999999998410519796728108115885556130475730798510027332432797015830574374922176498"
    "04555650371464527474677148367876444723378881462849676609039306640625e-46"
  },
  {
    1e-44,
    199,
    "0.0000000000000000000000000000000000000000000099999999999999995299012157797537262313524103"
    "585668678214901248072213449280016067527327081027864783122672863183914675200281152456227573"
    "566138744354248046875",
    154,
    "9.9999999999999995299012157797537262313524103585668678214901248072213449280016067527327081"
    "027864783122672863183914675200281152456227573566138744354248046875e-45"
  },
  {
    1e-43,
    195,
    "0.0000000000000000000000000000000000000000001000000000000000077450427135198206766016522111"
    "459171593954055855145477154822492971067247490986316654905625094353419091142143315664725378"
    "15570831298828125",
    152,
    "1.0000000000000000774504271351982067660165221114591715939540558551454771548224929710672474"
    "9098631665490562509435341909114214331566472537815570831298828125e-43"
  },
  {
    1e-42,
    192,
    "0.0000000000000000000000000000000000000000010000000000000000376231293568868998402945121672"
    "663764541764419753300075029753466364131749531598626313283782263488519426175571425119414925"
    "57525634765625",
    150,
    "1.0000000000000000376231293568868998402945121672663764541764419753300075029753466364131749"
    "53159862631328378226348851942617557142511941492557525634765625e-42"
  },
  {
    1e-41,
    189,
    "0.0000000000000000000000000000000000000000100000000000000000576129113423785429971690421191"
    "214034235435087147763178149762956868991692289869941246658073194519822379788820398971438407"
    "89794921875",
    148,
    "1.0000000000000000057612911342378542997169042119121403423543508714776317814976295686899169"
    "228986994124665807319451982237978882039897143840789794921875e-41"
  },
  {
    1e-40,
    183,
    "0.0000000000000000000000000000000000000000999999999999999929292879399880145002330645119061"
    "973673981332222231930049951108606154097650271907687198266745376429298630682751536369323730"
    "46875",
    142,
    "9.9999999999999992929287939988014500233064511906197367398133222223193004995110860615409765"
    "027190768719826674537642929863068275153636932373046875e-41"
  },
  {
    1e-39,
    182,
    "0.0000000000000000000000000000000000000009999999999999999292928793998801450023306451190619"
    "736739813322222319300499511086061540976502719076871982667453764292986306827515363693237304"
    "6875",
    142,
    "9.9999999999999992929287939988014500233064511906197367398133222223193004995110860615409765"
    "027190768719826674537642929863068275153636932373046875e-40"
  },
  {
    1e-38,
    177,
    "0.0000000000000000000000000000000000000099999999999999996191940173987276763588211566534471"
    "14524871535125767627887442908835027138732593388233127473796457707067020237445831298828125",
    138,
    "9.9999999999999996191940173987276763588211566534471145248715351257676278874429088350271387"
    "32593388233127473796457707067020237445831298828125e-39"
  },
  {
    1e-37,
    175,
    "0.0000000000000000000000000000000000001000000000000000066324273227849160063246821413449472"
    "343705781641680227552882474171018285786819118458879085409307663212530314922332763671875",
    138,
    "1.0000000000000000663242732278491600632468214134494723437057816416802275528824741710182857"
    "86819118458879085409307663212530314922332763671875e-37"
  },
  {
    1e-36,
    171,
    "0.0000000000000000000000000000000000009999999999999999410384274422774891504091745157237592"
    "74243427886756069835916654225999599490547382896199479773713392205536365509033203125",
    134,
    "9.9999999999999994103842744227748915040917451572375927424342788675606983591665422599959949"
    "0547382896199479773713392205536365509033203125e-37"
  },
  {
    1e-35,
    169,
    "0.0000000000000000000000000000000000100000000000000000785754519458238030392258619451080624"
    "462334988938228728496509153000956551522564186296193611269700340926647186279296875",
    134,
    "1.0000000000000000078575451945823803039225861945108062446233498893822872849650915300095655"
    "1522564186296193611269700340926647186279296875e-35"
  },
  {
    1e-34,
    165,
    "0.0000000000000000000000000000000000999999999999999927674603891816510919706492179966349880"
    "16744348623082634610696676519760628561173110284698850591666996479034423828125",
    130,
    "9.9999999999999992767460389181651091970649217996634988016744348623082634610696676519760628"
    "561173110284698850591666996479034423828125e-35"
  },
  {
    1e-33,
    160,
    "0.0000000000000000000000000000000010000000000000000559673099762419019344522426032374800632"
    "968937312731638482799663888967410529939883190309046767652034759521484375",
    127,
    "1.0000000000000000559673099762419019344522426032374800632968937312731638482799663888967410"
    "529939883190309046767652034759521484375e-33"
  },
  {
    1e-32,
    159,
    "0.0000000000000000000000000000000100000000000000005596730997624190193445224260323748006329"
    "68937312731638482799663888967410529939883190309046767652034759521484375",
    127,
    "1.0000000000000000559673099762419019344522426032374800632968937312731638482799663888967410"
    "529939883190309046767652034759521484375e-32"
  },
  {
    1e-31,
    148,
    "0.0000000000000000000000000000001000000000000000083336420607585985350931336026868654502364"
    "509783548862515410206308619223136702203191816806793212890625",
    117,
    "1.0000000000000000833364206075859853509313360268686545023645097835488625154102063086192231"
    "36702203191816806793212890625e-31"
  },
  {
    1e-30,
    147,
    "0.0000000000000000000000000000010000000000000000833364206075859853509313360268686545023645"
    "09783548862515410206308619223136702203191816806793212890625",
    117,
    "1.0000000000000000833364206075859853509313360268686545023645097835488625154102063086192231"
    "36702203191816806793212890625e-30"
  },
  {
    1e-29,
    149,
    "0.0000000000000000000000000000099999999999999994320657417510427825855837769787704137433831"
    "5595897285339703377919640114868116143043152987957000732421875",
    119,
    "9.9999999999999994320657417510427825855837769787704137433831559589728533970337791964011486"
    "8116143043152987957000732421875e-30"
  },
  {
    1e-28,
    146,
    "0.0000000000000000000000000000999999999999999971232543461600619677032969363675363999943554"
    "4334276007748447435974359365218333550728857517242431640625",
    117,
    "9.9999999999999997123254346160061967703296936367536399994355443342760077484474359743593652"
    "18333550728857517242431640625e-29"
  },
  {
    1e-27,
    138,
    "0.0000000000000000000000000010000000000000000384948697491918390813719893615913383013961276"
    "43500357819184021224145908490754663944244384765625",
    111,
    "1.0000000000000000384948697491918390813719893615913383013961276435003578191840212241459084"
    "90754663944244384765625e-27"
  },
  {
    1e-26,
    137,
    "0.0000000000000000000000000100000000000000003849486974919183908137198936159133830139612764"
    "3500357819184021224145908490754663944244384765625",
    111,
    "1.0000000000000000384948697491918390813719893615913383013961276435003578191840212241459084"
    "90754663944244384765625e-26"
  },
  {
    1e-25,
    136,
    "0.0000000000000000000000001000000000000000038494869749191839081371989361591338301396127643"
    "500357819184021224145908490754663944244384765625",
    111,
    "1.0000000000000000384948697491918390813719893615913383013961276435003578191840212241459084"
    "90754663944244384765625e-25"
  },
  {
    1e-24,
    132,
    "0.0000000000000000000000009999999999999999237004995517028246313000618984814088269170693649"
    "76185796844987407894222997128963470458984375",
    107,
    "9.9999999999999992370049955170282463130006189848140882691706936497618579684498740789422299"
    "7128963470458984375e-25"
  },
  {
    1e-23,
    129,
    "0.0000000000000000000000099999999999999996043469801489930925532307868667658625875036801410"
    "39208439934782290947623550891876220703125",
    105,
    "9.9999999999999996043469801489930925532307868667658625875036801410392084399347822909476235"
    "50891876220703125e-24"
  },
  {
    1e-22,
    125,
    "0.0000000000000000000001000000000000000048596774326570872352978318978345012095150284772010"
    "4849571498561999760568141937255859375",
    103,
    "1.0000000000000000485967743265708723529783189783450120951502847720104849571498561999760568"
    "141937255859375e-22"
  },
  {
    1e-21,
    122,
    "0.0000000000000000000009999999999999999075374522278963713967299345116755307569104179593599"
    "8237609965144656598567962646484375",
    100,
    "9.9999999999999990753745222789637139672993451167553075691041795935998237609965144656598567"
    "962646484375e-22"
  },
  {
    1e-20,
    119,
    "0.0000000000000000000099999999999999994515327145420957165172950370278739244710771577606678"
    "3064379706047475337982177734375",
    98,
    "9.9999999999999994515327145420957165172950370278739244710771577606678306437970604747533798"
    "2177734375e-21"
  },
  {
    1e-19,
    114,
    "0.0000000000000000000999999999999999975245926835260131855729159055676881799265554029432223"
    "61500374972820281982421875",
    94,
    "9.9999999999999997524592683526013185572915905567688179926555402943222361500374972820281982"
    "421875e-20"
  },
  {
    1e-18,
    110,
    "0.0000000000000000010000000000000000715424240546219245085280561849232477261706364402016333"
    "7700068950653076171875",
    92,
    "1.0000000000000000715424240546219245085280561849232477261706364402016333770006895065307617"
    "1875e-18"
  },
  {
    1e-17,
    109,
    "0.0000000000000000100000000000000007154242405462192450852805618492324772617063644020163337"
    "700068950653076171875",
    92,
    "1.0000000000000000715424240546219245085280561849232477261706364402016333770006895065307617"
    "1875e-17"
  },
  {
    1e-16,
    104,
    "0.0000000000000000999999999999999979097786724034603561841114940846736436341757325863000005"
    "4836273193359375",
    87,
    "9.999999999999999790977867240346035618411149408467364363417573258630000054836273193359375e"
    "-17"
  },
  {
    1e-15,
    101,
    "0.0000000000000010000000000000000777053998766610792383071856011950151454925617144908756017"
    "6849365234375",
    86,
    "1.00000000000000007770539987666107923830718560119501514549256171449087560176849365234375e-"
    "15"
  },
  {
    1e-14,
    99,
    "0.0000000000000099999999999999999881930935455989869713432907291639217817191820358857512474"
    "06005859375",
    84,
    "9.999999999999999988193093545598986971343290729163921781719182035885751247406005859375e-15"
  },
  {
    1e-13,
    95,
    "0.0000000000001000000000000000030373745563400370913603471684227841365100175607949495315551"
    "7578125",
    82,
    "1.0000000000000000303737455634003709136034716842278413651001756079494953155517578125e-13"
  },
  {
    1e-12,
    92,
    "0.0000000000009999999999999999798866476292556153672528435061295226660149637609720230102539"
    "0625",
    79,
    "9.9999999999999997988664762925561536725284350612952266601496376097202301025390625e-13"
  },
  {
    1e-11,
    89,
    "0.0000000000099999999999999993949696928193981093017234096365086770674679428339004516601562"
    "5",
    77,
    "9.99999999999999939496969281939810930172340963650867706746794283390045166015625e-12"
  },
  {
    1e-10,
    86,
    "0.00000000010000000000000000364321973154977415791655470655996396089904010295867919921875",
    76,
    "1.0000000000000000364321973154977415791655470655996396089904010295867919921875e-10"
  },
  {
    1e-9,
    82,
    "0.0000000010000000000000000622815914577798564188970686927859787829220294952392578125",
    73,
    "1.0000000000000000622815914577798564188970686927859787829220294952392578125e-09"
  },
  {
    1e-8,
    78,
    "0.000000010000000000000000209225608301284726753266340892878361046314239501953125",
    70,
    "1.0000000000000000209225608301284726753266340892878361046314239501953125e-08"
  },
  {
    1e-7,
    73,
    "0.0000000999999999999999954748111825886258685613938723690807819366455078125",
    65,
    "9.99999999999999954748111825886258685613938723690807819366455078125e-08"
  },
  {
    1e-6,
    72,
    "0.000000999999999999999954748111825886258685613938723690807819366455078125",
    65,
    "9.99999999999999954748111825886258685613938723690807819366455078125e-07"
  },
  {
    1e-5,
    69,
    "0.000010000000000000000818030539140313095458623138256371021270751953125",
    64,
    "1.0000000000000000818030539140313095458623138256371021270751953125e-05"
  },
  {
    1e-4,
    66,
    "0.000100000000000000004792173602385929598312941379845142364501953125",
    62,
    "1.00000000000000004792173602385929598312941379845142364501953125e-04"
  },
  {
    1e-3,
    60,
    "0.001000000000000000020816681711721685132943093776702880859375",
    57,
    "1.000000000000000020816681711721685132943093776702880859375e-03"
  },
  {
    1e-2,
    59,
    "0.01000000000000000020816681711721685132943093776702880859375",
    57,
    "1.000000000000000020816681711721685132943093776702880859375e-02"
  },
  {
    1e-1,
    55,
    "0.1000000000000000055511151231257827021181583404541015625",
    54,
    "1.000000000000000055511151231257827021181583404541015625e-01"
  },
  {
    1e0,
    0,
    "1",
    0,
    "1e+00"
  },
  {
    1e1,
    0,
    "10",
    0,
    "1e+01"
  },
  {
    1e2,
    0,
    "100",
    0,
    "1e+02"
  },
  {
    1e3,
    0,
    "1000",
    0,
    "1e+03"
  },
  {
    1e4,
    0,
    "10000",
    0,
    "1e+04"
  },
  {
    1e5,
    0,
    "100000",
    0,
    "1e+05"
  },
  {
    1e6,
    0,
    "1000000",
    0,
    "1e+06"
  },
  {
    1e7,
    0,
    "10000000",
    0,
    "1e+07"
  },
  {
    1e8,
    0,
    "100000000",
    0,
    "1e+08"
  },
  {
    1e9,
    0,
    "1000000000",
    0,
    "1e+09"
  },
  {
    1e10,
    0,
    "10000000000",
    0,
    "1e+10"
  },
  {
    1e11,
    0,
    "100000000000",
    0,
    "1e+11"
  },
  {
    1e12,
    0,
    "1000000000000",
    0,
    "1e+12"
  },
  {
    1e13,
    0,
    "10000000000000",
    0,
    "1e+13"
  },
  {
    1e14,
    0,
    "100000000000000",
    0,
    "1e+14"
  },
  {
    1e15,
    0,
    "1000000000000000",
    0,
    "1e+15"
  },
  {
    1e16,
    0,
    "10000000000000000",
    0,
    "1e+16"
  },
  {
    1e17,
    0,
    "100000000000000000",
    0,
    "1e+17"
  },
  {
    1e18,
    0,
    "1000000000000000000",
    0,
    "1e+18"
  },
  {
    1e19,
    0,
    "10000000000000000000",
    0,
    "1e+19"
  },
  {
    1e20,
    0,
    "100000000000000000000",
    0,
    "1e+20"
  },
  {
    1e21,
    0,
    "1000000000000000000000",
    0,
    "1e+21"
  },
  {
    1e22,
    0,
    "10000000000000000000000",
    0,
    "1e+22"
  },
  {
    1e23,
    0,
    "99999999999999991611392",
    22,
    "9.9999999999999991611392e+22"
  },
  {
    1e24,
    0,
    "999999999999999983222784",
    23,
    "9.99999999999999983222784e+23"
  },
  {
    1e25,
    0,
    "10000000000000000905969664",
    25,
    "1.0000000000000000905969664e+25"
  },
  {
    1e26,
    0,
    "100000000000000004764729344",
    26,
    "1.00000000000000004764729344e+26"
  },
  {
    1e27,
    0,
    "1000000000000000013287555072",
    27,
    "1.000000000000000013287555072e+27"
  },
  {
    1e28,
    0,
    "9999999999999999583119736832",
    27,
    "9.999999999999999583119736832e+27"
  },
  {
    1e29,
    0,
    "99999999999999991433150857216",
    28,
    "9.9999999999999991433150857216e+28"
  },
  {
    1e30,
    0,
    "1000000000000000019884624838656",
    30,
    "1.000000000000000019884624838656e+30"
  },
  {
    1e31,
    0,
    "9999999999999999635896294965248",
    30,
    "9.999999999999999635896294965248e+30"
  },
  {
    1e32,
    0,
    "100000000000000005366162204393472",
    32,
    "1.00000000000000005366162204393472e+32"
  },
  {
    1e33,
    0,
    "999999999999999945575230987042816",
    32,
    "9.99999999999999945575230987042816e+32"
  },
  {
    1e34,
    0,
    "9999999999999999455752309870428160",
    32,
    "9.99999999999999945575230987042816e+33"
  },
  {
    1e35,
    0,
    "99999999999999996863366107917975552",
    34,
    "9.9999999999999996863366107917975552e+34"
  },
  {
    1e36,
    0,
    "1000000000000000042420637374017961984",
    36,
    "1.000000000000000042420637374017961984e+36"
  },
  {
    1e37,
    0,
    "9999999999999999538762658202121142272",
    36,
    "9.999999999999999538762658202121142272e+36"
  },
  {
    1e38,
    0,
    "99999999999999997748809823456034029568",
    37,
    "9.9999999999999997748809823456034029568e+37"
  },
  {
    1e39,
    0,
    "999999999999999939709166371603178586112",
    38,
    "9.99999999999999939709166371603178586112e+38"
  },
  {
    1e40,
    0,
    "10000000000000000303786028427003666890752",
    40,
    "1.0000000000000000303786028427003666890752e+40"
  },
  {
    1e41,
    0,
    "100000000000000000620008645040778319495168",
    41,
    "1.00000000000000000620008645040778319495168e+41"
  },
  {
    1e42,
    0,
    "1000000000000000044885712678075916785549312",
    42,
    "1.000000000000000044885712678075916785549312e+42"
  },
  {
    1e43,
    0,
    "10000000000000000139372116959414099130712064",
    43,
    "1.0000000000000000139372116959414099130712064e+43"
  },
  {
    1e44,
    0,
    "100000000000000008821361405306422640701865984",
    44,
    "1.00000000000000008821361405306422640701865984e+44"
  },
  {
    1e45,
    0,
    "999999999999999929757289024535551219930759168",
    44,
    "9.99999999999999929757289024535551219930759168e+44"
  },
  {
    1e46,
    0,
    "9999999999999999931398190359470212947659194368",
    45,
    "9.999999999999999931398190359470212947659194368e+45"
  },
  {
    1e47,
    0,
    "100000000000000004384584304507619735463404765184",
    47,
    "1.00000000000000004384584304507619735463404765184e+47"
  },
  {
    1e48,
    0,
    "1000000000000000043845843045076197354634047651840",
    47,
    "1.00000000000000004384584304507619735463404765184e+48"
  },
  {
    1e49,
    0,
    "9999999999999999464902769475481793196872414789632",
    48,
    "9.999999999999999464902769475481793196872414789632e+48"
  },
  {
    1e50,
    0,
    "100000000000000007629769841091887003294964970946560",
    49,
    "1.0000000000000000762976984109188700329496497094656e+50"
  },
  {
    1e51,
    0,
    "999999999999999993220948674361627976461708441944064",
    50,
    "9.99999999999999993220948674361627976461708441944064e+50"
  },
  {
    1e52,
    0,
    "9999999999999999932209486743616279764617084419440640",
    50,
    "9.99999999999999993220948674361627976461708441944064e+51"
  },
  {
    1e53,
    0,
    "99999999999999999322094867436162797646170844194406400",
    50,
    "9.99999999999999993220948674361627976461708441944064e+52"
  },
  {
    1e54,
    0,
    "1000000000000000078291540404596243842305360299886116864",
    54,
    "1.000000000000000078291540404596243842305360299886116864e+54"
  },
  {
    1e55,
    0,
    "10000000000000000102350670204085511496304388135324745728",
    55,
    "1.0000000000000000102350670204085511496304388135324745728e+55"
  },
  {
    1e56,
    0,
    "100000000000000009190283508143378238084034459715684532224",
    56,
    "1.00000000000000009190283508143378238084034459715684532224e+56"
  },
  {
    1e57,
    0,
    "1000000000000000048346692115553659057528394845890514255872",
    57,
    "1.000000000000000048346692115553659057528394845890514255872e+57"
  },
  {
    1e58,
    0,
    "9999999999999999438119489974413630815797154428513196965888",
    57,
    "9.999999999999999438119489974413630815797154428513196965888e+57"
  },
  {
    1e59,
    0,
    "99999999999999997168788049560464200849936328366177157906432",
    58,
    "9.9999999999999997168788049560464200849936328366177157906432e+58"
  },
  {
    1e60,
    0,
    "999999999999999949387135297074018866963645011013410073083904",
    59,
    "9.99999999999999949387135297074018866963645011013410073083904e+59"
  },
  {
    1e61,
    0,
    "9999999999999999493871352970740188669636450110134100730839040",
    59,
    "9.99999999999999949387135297074018866963645011013410073083904e+60"
  },
  {
    1e62,
    0,
    "100000000000000003502199685943161173046080317798311825604870144",
    62,
    "1.00000000000000003502199685943161173046080317798311825604870144e+62"
  },
  {
    1e63,
    0,
    "1000000000000000057857959942726969827393378689175040438172647424",
    63,
    "1.000000000000000057857959942726969827393378689175040438172647424e+63"
  },
  {
    1e64,
    0,
    "10000000000000000213204190094543968723012578712679649467743338496",
    64,
    "1.0000000000000000213204190094543968723012578712679649467743338496e+64"
  },
  {
    1e65,
    0,
    "99999999999999999209038626283633850822756121694230455365568299008",
    64,
    "9.9999999999999999209038626283633850822756121694230455365568299008e+64"
  },
  {
    1e66,
    0,
    "999999999999999945322333868247445125709646570021247924665841614848",
    65,
    "9.99999999999999945322333868247445125709646570021247924665841614848e+65"
  },
  {
    1e67,
    0,
    "9999999999999999827367757839185598317239782875580932278577147150336",
    66,
    "9.999999999999999827367757839185598317239782875580932278577147150336e+66"
  },
  {
    1e68,
    0,
    "99999999999999995280522225138166806691251291352861698530421623488512",
    67,
    "9.9999999999999995280522225138166806691251291352861698530421623488512e+67"
  },
  {
    1e69,
    0,
    "1000000000000000072531436381529235126158374409646521955518210155479040",
    68,
    "1.00000000000000007253143638152923512615837440964652195551821015547904e+69"
  },
  {
    1e70,
    0,
    "10000000000000000725314363815292351261583744096465219555182101554790400",
    68,
    "1.00000000000000007253143638152923512615837440964652195551821015547904e+70"
  },
  {
    1e71,
    0,
    "100000000000000004188152556421145795899143386664033828314342771180699648",
    71,
    "1.00000000000000004188152556421145795899143386664033828314342771180699648e+71"
  },
  {
    1e72,
    0,
    "999999999999999943801810948794571024057224129020550531544123892056457216",
    71,
    "9.99999999999999943801810948794571024057224129020550531544123892056457216e+71"
  },
  {
    1e73,
    0,
    "9999999999999999830336967949613257980309080240684656321838454199566729216",
    72,
    "9.999999999999999830336967949613257980309080240684656321838454199566729216e+72"
  },
  {
    1e74,
    0,
    "99999999999999995164818811802792197885196090803013355167206819763650035712",
    73,
    "9.9999999999999995164818811802792197885196090803013355167206819763650035712e+73"
  },
  {
    1e75,
    0,
    "999999999999999926539781176481198923508803215199467887262646419780362305536",
    74,
    "9.99999999999999926539781176481198923508803215199467887262646419780362305536e+74"
  },
  {
    1e76,
    0,
    "10000000000000000470601344959054695891559601407866630764278709534898249531392",
    76,
    "1.0000000000000000470601344959054695891559601407866630764278709534898249531392e+76"
  },
  {
    1e77,
    0,
    "99999999999999998278261272554585856747747644714015897553975120217811154108416",
    76,
    "9.9999999999999998278261272554585856747747644714015897553975120217811154108416e+76"
  },
  {
    1e78,
    0,
    "1000000000000000008493621433689702976148869924598760615894999102702796905906176",
    78,
    "1.000000000000000008493621433689702976148869924598760615894999102702796905906176e+78"
  },
  {
    1e79,
    0,
    "9999999999999999673560075006595519222746403606649979913266024618633003221909504",
    78,
    "9.999999999999999673560075006595519222746403606649979913266024618633003221909504e+78"
  },
  {
    1e80,
    0,
    "100000000000000000026609864708367276537402401181200809098131977453489758916313088",
    80,
    "1.00000000000000000026609864708367276537402401181200809098131977453489758916313088e+80"
  },
  {
    1e81,
    0,
    "999999999999999921281879895665782741935503249059183851809998224123064148429897728",
    80,
    "9.99999999999999921281879895665782741935503249059183851809998224123064148429897728e+80"
  },
  {
    1e82,
    0,
    "9999999999999999634067965630886574211027143225273567793680363843427086501542887424",
    81,
    "9.999999999999999634067965630886574211027143225273567793680363843427086501542887424e+81"
  },
  {
    1e83,
    0,
    "100000000000000003080666323096525690777025204007643346346089744069413985291331436544",
    83,
    "1.00000000000000003080666323096525690777025204007643346346089744069413985291331436544e+83"
  },
  {
    1e84,
    0,
    "1000000000000000057766609898115896702437267127096064137098041863234712334016924614656",
    84,
    "1.000000000000000057766609898115896702437267127096064137098041863234712334016924614656e+84"
  },
  {
    1e85,
    0,
    "10000000000000000146306952306748730309700429878646550592786107871697963642511482159104",
    85,
    "1.0000000000000000146306952306748730309700429878646550592786107871697963642511482159104e+8"
    "5"
  },
  {
    1e86,
    0,
    "100000000000000001463069523067487303097004298786465505927861078716979636425114821591040",
    85,
    "1.0000000000000000146306952306748730309700429878646550592786107871697963642511482159104e+8"
    "6"
  },
  {
    1e87,
    0,
    "999999999999999959416724456350362731491996089648451439669739009806703922950954425516032",
    86,
    "9.99999999999999959416724456350362731491996089648451439669739009806703922950954425516032e+"
    "86"
  },
  {
    1e88,
    0,
    "9999999999999999594167244563503627314919960896484514396697390098067039229509544255160320",
    86,
    "9.99999999999999959416724456350362731491996089648451439669739009806703922950954425516032e+"
    "87"
  },
  {
    1e89,
    0,
    "99999999999999999475366575191804932315794610450682175621941694731908308538307845136842752",
    88,
    "9.9999999999999999475366575191804932315794610450682175621941694731908308538307845136842752"
    "e+88"
  },
  {
    1e90,
    0,
    "999999999999999966484112715463900049825186092620125502979674597309179755437379230686511104",
    89,
    "9.9999999999999996648411271546390004982518609262012550297967459730917975543737923068651110"
    "4e+89"
  },
  {
    1e91,
    0,
    "100000000000000007956232486128049714315622614016691051593864399734879307522017611341417676"
    "80",
    90,
    "1.0000000000000000795623248612804971431562261401669105159386439973487930752201761134141767"
    "68e+91"
  },
  {
    1e92,
    0,
    "100000000000000004337729697461918607329029332495193931179177378933611681288968111094132375"
    "552",
    92,
    "1.0000000000000000433772969746191860732902933249519393117917737893361168128896811109413237"
    "5552e+92"
  },
  {
    1e93,
    0,
    "100000000000000004337729697461918607329029332495193931179177378933611681288968111094132375"
    "5520",
    92,
    "1.0000000000000000433772969746191860732902933249519393117917737893361168128896811109413237"
    "5552e+93"
  },
  {
    1e94,
    0,
    "100000000000000002021887912715594698857609632321435774113777685620800400499816430935869782"
    "75328",
    94,
    "1.0000000000000000202188791271559469885760963232143577411377768562080040049981643093586978"
    "275328e+94"
  },
  {
    1e95,
    0,
    "100000000000000002021887912715594698857609632321435774113777685620800400499816430935869782"
    "753280",
    94,
    "1.0000000000000000202188791271559469885760963232143577411377768562080040049981643093586978"
    "275328e+95"
  },
  {
    1e96,
    0,
    "100000000000000004986165397190889301701026848543846215157489293061198839909930581538445901"
    "5356416",
    96,
    "1.0000000000000000498616539719088930170102684854384621515748929306119883990993058153844590"
    "15356416e+96"
  },
  {
    1e97,
    0,
    "100000000000000007357587384771124983975760621521774567992458579013517591438021902020506796"
    "56153088",
    97,
    "1.0000000000000000735758738477112498397576062152177456799245857901351759143802190202050679"
    "656153088e+97"
  },
  {
    1e98,
    0,
    "999999999999999997690370245143708006966125479924038389205568639660975865481296764779119324"
    "78685184",
    97,
    "9.9999999999999999769037024514370800696612547992403838920556863966097586548129676477911932"
    "478685184e+97"
  },
  {
    1e99,
    0,
    "999999999999999967336168804116691273849533185806555472917961779471295845921727862608739868"
    "455469056",
    98,
    "9.9999999999999996733616880411669127384953318580655547291796177947129584592172786260873986"
    "8455469056e+98"
  },
  {
    1e100,
    0,
    "100000000000000001590289110975991804683608085639452813897813275577478387721703810608134699"
    "85856815104",
    100,
    "1.0000000000000000159028911097599180468360808563945281389781327557747838772170381060813469"
    "985856815104e+100"
  },
  {
    1e101,
    0,
    "999999999999999977049513265245336628446842719924150006129995974731993452180789911303261294"
    "48151154688",
    100,
    "9.9999999999999997704951326524533662844684271992415000612999597473199345218078991130326129"
    "448151154688e+100"
  },
  {
    1e102,
    0,
    "999999999999999977049513265245336628446842719924150006129995974731993452180789911303261294"
    "481511546880",
    100,
    "9.9999999999999997704951326524533662844684271992415000612999597473199345218078991130326129"
    "448151154688e+101"
  },
  {
    1e103,
    0,
    "100000000000000000191567508573466873621595512726519201115280351459937932420398875596123614"
    "51081803235328",
    103,
    "1.0000000000000000019156750857346687362159551272651920111528035145993793242039887559612361"
    "451081803235328e+103"
  },
  {
    1e104,
    0,
    "100000000000000000191567508573466873621595512726519201115280351459937932420398875596123614"
    "510818032353280",
    103,
    "1.0000000000000000019156750857346687362159551272651920111528035145993793242039887559612361"
    "451081803235328e+104"
  },
  {
    1e105,
    0,
    "999999999999999938258300825281978540327027364472124478294416212538871491824599713636820527"
    "503908255301632",
    104,
    "9.9999999999999993825830082528197854032702736447212447829441621253887149182459971363682052"
    "7503908255301632e+104"
  },
  {
    1e106,
    0,
    "100000000000000009103599905036843501046045399517548655715454573748409028953513341521541800"
    "97541612190564352",
    106,
    "1.0000000000000000910359990503684350104604539951754865571545457374840902895351334152154180"
    "097541612190564352e+106"
  },
  {
    1e107,
    0,
    "999999999999999968813840470299269834353712690612796894066442117527915251366706453952540023"
    "95395884805259264",
    106,
    "9.9999999999999996881384047029926983435371269061279689406644211752791525136670645395254002"
    "395395884805259264e+106"
  },
  {
    1e108,
    0,
    "100000000000000003399899171300282459494397471971289804771343071483787527172320083329274161"
    "6380733445921308672",
    108,
    "1.0000000000000000339989917130028245949439747197128980477134307148378752717232008332927416"
    "16380733445921308672e+108"
  },
  {
    1e109,
    0,
    "999999999999999981850870718839980786471765096432817124795839836989907255438005329820580342"
    "4393137676263358464",
    108,
    "9.9999999999999998185087071883998078647176509643281712479583983698990725543800532982058034"
    "24393137676263358464e+108"
  },
  {
    1e110,
    0,
    "100000000000000002356936751417025583324953279505688186312991253926828166846616173259830936"
    "159244951026231410688",
    110,
    "1.0000000000000000235693675141702558332495327950568818631299125392682816684661617325983093"
    "6159244951026231410688e+110"
  },
  {
    1e111,
    0,
    "999999999999999956819772641641815758405104477258378281795396215622882607621111488153942930"
    "947432322044748890112",
    110,
    "9.9999999999999995681977264164181575840510447725837828179539621562288260762111148815394293"
    "0947432322044748890112e+110"
  },
  {
    1e112,
    0,
    "999999999999999930119934692630439728467333150138976849261589686164722983283091390376196358"
    "6894254467577228034048",
    111,
    "9.9999999999999993011993469263043972846733315013897684926158968616472298328309139037619635"
    "86894254467577228034048e+111"
  },
  {
    1e113,
    0,
    "100000000000000001555941612946684302426820139692106143336977058043083378116475570326498538"
    "991504744767620628086784",
    113,
    "1.0000000000000000155594161294668430242682013969210614333697705804308337811647557032649853"
    "8991504744767620628086784e+113"
  },
  {
    1e114,
    0,
    "100000000000000001555941612946684302426820139692106143336977058043083378116475570326498538"
    "9915047447676206280867840",
    113,
    "1.0000000000000000155594161294668430242682013969210614333697705804308337811647557032649853"
    "8991504744767620628086784e+114"
  },
  {
    1e115,
    0,
    "100000000000000001555941612946684302426820139692106143336977058043083378116475570326498538"
    "99150474476762062808678400",
    113,
    "1.0000000000000000155594161294668430242682013969210614333697705804308337811647557032649853"
    "8991504744767620628086784e+115"
  },
  {
    1e116,
    0,
    "100000000000000001555941612946684302426820139692106143336977058043083378116475570326498538"
    "991504744767620628086784000",
    113,
    "1.0000000000000000155594161294668430242682013969210614333697705804308337811647557032649853"
    "8991504744767620628086784e+116"
  },
  {
    1e117,
    0,
    "100000000000000005055542772599503381422823703080300327902048147472223276397708540582423337"
    "7105062219252417113236701184",
    117,
    "1.0000000000000000505554277259950338142282370308030032790204814747222327639770854058242333"
    "77105062219252417113236701184e+117"
  },
  {
    1e118,
    0,
    "999999999999999966564999894327375918324151509486342849458775328422875205227494119682038207"
    "8490267674695111155514343424",
    117,
    "9.9999999999999996656499989432737591832415150948634284945877532842287520522749411968203820"
    "78490267674695111155514343424e+117"
  },
  {
    1e119,
    0,
    "999999999999999944167552472549333812749728703801900068242320356076379856227603110044119496"
    "04741731366073618283536318464",
    118,
    "9.9999999999999994416755247254933381274972870380190006824232035607637985622760311004411949"
    "604741731366073618283536318464e+118"
  },
  {
    1e120,
    0,
    "999999999999999980003468347394201181668805192897008518188648311830772414627428725464789434"
    "929992439754776075181077037056",
    119,
    "9.9999999999999998000346834739420118166880519289700851818864831183077241462742872546478943"
    "4929992439754776075181077037056e+119"
  },
  {
    1e121,
    0,
    "100000000000000003734093374714598897193932757544918203810277304103780050806714971013786133"
    "71421126415052399029342192009216",
    121,
    "1.0000000000000000373409337471459889719393275754491820381027730410378005080671497101378613"
    "371421126415052399029342192009216e+121"
  },
  {
    1e122,
    0,
    "100000000000000001440594758724527385583111862242831263013712314935498927069126131626863257"
    "625726456080505437183296233537536",
    122,
    "1.0000000000000000144059475872452738558311186224283126301371231493549892706912613162686325"
    "7625726456080505437183296233537536e+122"
  },
  {
    1e123,
    0,
    "999999999999999977709969731404129670057984297594921577392083322662491290889839886077866558"
    "841507631684757522070951350501376",
    122,
    "9.9999999999999997770996973140412967005798429759492157739208332266249129088983988607786655"
    "8841507631684757522070951350501376e+122"
  },
  {
    1e124,
    0,
    "999999999999999948353187446731214321439476837728208735196051461308492907048702741925253744"
    "9089020883885200422613425626021888",
    123,
    "9.9999999999999994835318744673121432143947683772820873519605146130849290704870274192525374"
    "49089020883885200422613425626021888e+123"
  },
  {
    1e125,
    0,
    "999999999999999924867761618992882042544670869834838461439225972225294199975793026603163493"
    "76281765375153005841365553228283904",
    124,
    "9.9999999999999992486776161899288204254467086983483846143922597222529419997579302660316349"
    "376281765375153005841365553228283904e+124"
  },
  {
    1e126,
    0,
    "999999999999999924867761618992882042544670869834838461439225972225294199975793026603163493"
    "762817653751530058413655532282839040",
    124,
    "9.9999999999999992486776161899288204254467086983483846143922597222529419997579302660316349"
    "376281765375153005841365553228283904e+125"
  },
  {
    1e127,
    0,
    "999999999999999954929106678497947359530022508738352411847962598251788545029117462215439015"
    "2298057300868772377386949310916067328",
    126,
    "9.9999999999999995492910667849794735953002250873835241184796259825178854502911746221543901"
    "52298057300868772377386949310916067328e+126"
  },
  {
    1e128,
    0,
    "100000000000000007517448691651820862747142906435240821348290910235776592524241520466454110"
    "109775803542826595503885252632667750400",
    126,
    "1.0000000000000000751744869165182086274714290643524082134829091023577659252424152046645411"
    "01097758035428265955038852526326677504e+128"
  },
  {
    1e129,
    0,
    "999999999999999998217443564185241415988928868759412500436543339729940401905904649497115766"
    "142268560009777175966751665376232210432",
    128,
    "9.9999999999999999821744356418524141598892886875941250043654333972994040190590464949711576"
    "6142268560009777175966751665376232210432e+128"
  },
  {
    1e130,
    0,
    "100000000000000005978307824605161518517492902523380907087363594983220082057511309363105603"
    "41066601403445681992244323541365884452864",
    130,
    "1.0000000000000000597830782460516151851749290252338090708736359498322008205751130936310560"
    "341066601403445681992244323541365884452864e+130"
  },
  {
    1e131,
    0,
    "999999999999999912025555009572318139128528649695257301824613685586775815769012827709599390"
    "99212034754106974340599870111173348163584",
    130,
    "9.9999999999999991202555500957231813912852864969525730182461368558677581576901282770959939"
    "099212034754106974340599870111173348163584e+130"
  },
  {
    1e132,
    0,
    "999999999999999990829567402361276563686608849982484911984092226517669151665599636201042933"
    "986541570369602253175829982724989462249472",
    131,
    "9.9999999999999999082956740236127656368660884998248491198409222651766915166559963620104293"
    "3986541570369602253175829982724989462249472e+131"
  },
  {
    1e133,
    0,
    "100000000000000002235117235947685993350984093009737595604788364289002648602423435959762035"
    "11843100595010152570837624953702918544949248",
    133,
    "1.0000000000000000223511723594768599335098409300973759560478836428900264860242343595976203"
    "511843100595010152570837624953702918544949248e+133"
  },
  {
    1e134,
    0,
    "999999999999999921482036496706993150075498273729724615043751110498483016076603244728572616"
    "15145089428049364457837845490532419930947584",
    133,
    "9.9999999999999992148203649670699315007549827372972461504375111049848301607660324472857261"
    "615145089428049364457837845490532419930947584e+133"
  },
  {
    1e135,
    0,
    "999999999999999961829690841814939863449235336276785151445404123455100404055655690676191710"
    "164594560368702289580532071091311261383655424",
    134,
    "9.9999999999999996182969084181493986344923533627678515144540412345510040405565569067619171"
    "0164594560368702289580532071091311261383655424e+134"
  },
  {
    1e136,
    0,
    "100000000000000005866406127007401197554620428638973043880937135455098213520538156095047753"
    "57961393589804030375857007499376802103616864256",
    136,
    "1.0000000000000000586640612700740119755462042863897304388093713545509821352053815609504775"
    "357961393589804030375857007499376802103616864256e+136"
  },
  {
    1e137,
    0,
    "100000000000000003284156248920492607898701256635961169551231342625874700689878799554400131"
    "562772741268394950478432243557864849063421149184",
    137,
    "1.0000000000000000328415624892049260789870125663596116955123134262587470068987879955440013"
    "1562772741268394950478432243557864849063421149184e+137"
  },
  {
    1e138,
    0,
    "100000000000000003284156248920492607898701256635961169551231342625874700689878799554400131"
    "5627727412683949504784322435578648490634211491840",
    137,
    "1.0000000000000000328415624892049260789870125663596116955123134262587470068987879955440013"
    "1562772741268394950478432243557864849063421149184e+138"
  },
  {
    1e139,
    0,
    "100000000000000003284156248920492607898701256635961169551231342625874700689878799554400131"
    "56277274126839495047843224355786484906342114918400",
    137,
    "1.0000000000000000328415624892049260789870125663596116955123134262587470068987879955440013"
    "1562772741268394950478432243557864849063421149184e+139"
  },
  {
    1e140,
    0,
    "100000000000000005928380124081487003706362488767045328864850074482999577828473980652023296"
    "508018124569151792237293382948229697163514582401024",
    140,
    "1.0000000000000000592838012408148700370636248876704532886485007448299957782847398065202329"
    "6508018124569151792237293382948229697163514582401024e+140"
  },
  {
    1e141,
    0,
    "100000000000000001697621923823895970414104517357310673963060103511599774406721690895826232"
    "5956255112879408454231155599236459402033650892537856",
    141,
    "1.0000000000000000169762192382389597041410451735731067396306010351159977440672169089582623"
    "25956255112879408454231155599236459402033650892537856e+141"
  },
  {
    1e142,
    0,
    "100000000000000005082228484029968797047910894485098397884492080288719617144123522700783883"
    "72553960191290960287445781834331294577148468377157632",
    142,
    "1.0000000000000000508222848402996879704791089448509839788449208028871961714412352270078388"
    "372553960191290960287445781834331294577148468377157632e+142"
  },
  {
    1e143,
    0,
    "100000000000000002374543235865110535740865792782868218747346498867023742954202057256817762"
    "821608329412934596913384011607579341316989008157343744",
    143,
    "1.0000000000000000237454323586511053574086579278286821874734649886702374295420205725681776"
    "2821608329412934596913384011607579341316989008157343744e+143"
  },
  {
    1e144,
    0,
    "100000000000000002374543235865110535740865792782868218747346498867023742954202057256817762"
    "8216083294129345969133840116075793413169890081573437440",
    143,
    "1.0000000000000000237454323586511053574086579278286821874734649886702374295420205725681776"
    "2821608329412934596913384011607579341316989008157343744e+144"
  },
  {
    1e145,
    0,
    "999999999999999989087061182140919612678480626040135894518001546472530239911025814885411280"
    "6457630061296658928320953898584032761523454337112604672",
    144,
    "9.9999999999999998908706118214091961267848062604013589451800154647253023991102581488541128"
    "06457630061296658928320953898584032761523454337112604672e+144"
  },
  {
    1e146,
    0,
    "999999999999999933633667299724622421110196943178461825789260038956198736501434202592985124"
    "53325054533017777074930382791057905692427399713177731072",
    145,
    "9.9999999999999993363366729972462242111019694317846182578926003895619873650143420259298512"
    "453325054533017777074930382791057905692427399713177731072e+145"
  },
  {
    1e147,
    0,
    "999999999999999977996382405657660174364823889467801080772253244969263939229107492426926049"
    "423260513969768268415537077468838432306731146395363835904",
    146,
    "9.9999999999999997799638240565766017436482388946780108077225324496926393922910749242692604"
    "9423260513969768268415537077468838432306731146395363835904e+146"
  },
  {
    1e148,
    0,
    "100000000000000004897672657515052057957222700353074388874504237459016826359338475616123152"
    "92472764637931130646815102767620534329186625852171022761984",
    148,
    "1.0000000000000000489767265751505205795722270035307438887450423745901682635933847561612315"
    "292472764637931130646815102767620534329186625852171022761984e+148"
  },
  {
    1e149,
    0,
    "100000000000000004897672657515052057957222700353074388874504237459016826359338475616123152"
    "924727646379311306468151027676205343291866258521710227619840",
    148,
    "1.0000000000000000489767265751505205795722270035307438887450423745901682635933847561612315"
    "292472764637931130646815102767620534329186625852171022761984e+149"
  },
  {
    1e150,
    0,
    "999999999999999980835596172437374590573120014030318793091164810154100112203678582976298268"
    "616221151962702060266176005440567032331208403948233373515776",
    149,
    "9.9999999999999998083559617243737459057312001403031879309116481015410011220367858297629826"
    "8616221151962702060266176005440567032331208403948233373515776e+149"
  },
  {
    1e151,
    0,
    "100000000000000001717753238721771911803931040843054551077323284452000312627818854200826267"
    "42861173182722545959543542834786931126445173006249634549465088",
    151,
    "1.0000000000000000171775323872177191180393104084305455107732328445200031262781885420082626"
    "742861173182722545959543542834786931126445173006249634549465088e+151"
  },
  {
    1e152,
    0,
    "100000000000000004625108135904199474001226272395072688491888727201272553753779650923383419"
    "882203425131989662450489690590919397689516441796634752009109504",
    152,
    "1.0000000000000000462510813590419947400122627239507268849188872720127255375377965092338341"
    "9882203425131989662450489690590919397689516441796634752009109504e+152"
  },
  {
    1e153,
    0,
    "999999999999999999733403004123153744855539019118436686285840188024369679522423761672919759"
    "564567158443669378824028710020392594094129030220133015859757056",
    152,
    "9.9999999999999999973340300412315374485553901911843668628584018802436967952242376167291975"
    "9564567158443669378824028710020392594094129030220133015859757056e+152"
  },
  {
    1e154,
    0,
    "100000000000000003694754568805822654098091798298426884519227785521505436593472195972165131"
    "09705408327446511753687232667314337003349573404171046192448274432",
    154,
    "1.0000000000000000369475456880582265409809179829842688451922778552150543659347219597216513"
    "109705408327446511753687232667314337003349573404171046192448274432e+154"
  },
  {
    1e155,
    0,
    "100000000000000000717623154091016830408061481189160311806712772146250661680488340128266606"
    "984576189330386573813296762136260081534229469225952733653677113344",
    155,
    "1.0000000000000000071762315409101683040806148118916031180671277214625066168048834012826660"
    "6984576189330386573813296762136260081534229469225952733653677113344e+155"
  },
  {
    1e156,
    0,
    "999999999999999983359180223191721714560372275017470536367007614460468417501012554531477876"
    "945938741751237388344363105067534507348164573733465510370326085632",
    155,
    "9.9999999999999998335918022319172171456037227501747053636700761446046841750101255453147787"
    "6945938741751237388344363105067534507348164573733465510370326085632e+155"
  },
  {
    1e157,
    0,
    "999999999999999983359180223191721714560372275017470536367007614460468417501012554531477876"
    "9459387417512373883443631050675345073481645737334655103703260856320",
    155,
    "9.9999999999999998335918022319172171456037227501747053636700761446046841750101255453147787"
    "6945938741751237388344363105067534507348164573733465510370326085632e+156"
  },
  {
    1e158,
    0,
    "999999999999999952873354536512110079974461827818580831790853877497859522392057870689956990"
    "03416510776387310061494932420984963311567802202010637287727642443776",
    157,
    "9.9999999999999995287335453651211007997446182781858083179085387749785952239205787068995699"
    "003416510776387310061494932420984963311567802202010637287727642443776e+157"
  },
  {
    1e159,
    0,
    "999999999999999928484693987168420772305733470059469068129930887927772406304894123616740280"
    "504746200573981670431418299523701733729688780649419062882836695482368",
    158,
    "9.9999999999999992848469398716842077230573347005946906812993088792777240630489412361674028"
    "0504746200573981670431418299523701733729688780649419062882836695482368e+158"
  },
  {
    1e160,
    0,
    "100000000000000000652840774506822655684566421488862671184488445455205117778381811425103375"
    "09988867035816342470187175785193750117648543530356184548650438281396224",
    160,
    "1.0000000000000000065284077450682265568456642148886267118448844545520511777838181142510337"
    "509988867035816342470187175785193750117648543530356184548650438281396224e+160"
  },
  {
    1e161,
    0,
    "100000000000000003774589324822814887066163651282028976933086588120176268637538771050475113"
    "919654290478469527765363729011764432297892058199009821165792668120252416",
    161,
    "1.0000000000000000377458932482281488706616365128202897693308658812017626863753877105047511"
    "3919654290478469527765363729011764432297892058199009821165792668120252416e+161"
  },
  {
    1e162,
    0,
    "999999999999999937849939638116397466450525159438967985375725315922685858882365002492855496"
    "964043060934899979621894213003182527093908649335762989920701551401238528",
    161,
    "9.9999999999999993784993963811639746645052515943896798537572531592268585888236500249285549"
    "6964043060934899979621894213003182527093908649335762989920701551401238528e+161"
  },
  {
    1e163,
    0,
    "999999999999999937849939638116397466450525159438967985375725315922685858882365002492855496"
    "9640430609348999796218942130031825270939086493357629899207015514012385280",
    161,
    "9.9999999999999993784993963811639746645052515943896798537572531592268585888236500249285549"
    "6964043060934899979621894213003182527093908649335762989920701551401238528e+162"
  },
  {
    1e164,
    0,
    "100000000000000000178334994858791836514563642560301392710701527770129502847789953562046870"
    "799284296099876897036220978235643807646031628623453753183252563447406133248",
    164,
    "1.0000000000000000017833499485879183651456364256030139271070152777012950284778995356204687"
    "0799284296099876897036220978235643807646031628623453753183252563447406133248e+164"
  },
  {
    1e165,
    0,
    "999999999999999899489893451833484927233458399740540420336951338855520357125044282616287570"
    "346763120896578585177704871391229197474064067196498264773607101557544845312",
    164,
    "9.9999999999999989948989345183348492723345839974054042033695133885552035712504428261628757"
    "0346763120896578585177704871391229197474064067196498264773607101557544845312e+164"
  },
  {
    1e166,
    0,
    "999999999999999940407276050535258302398329610085529823044976914393830225666186383817960025"
    "4051950569374547392515068357773127490685649548117139715971745147241514401792",
    165,
    "9.9999999999999994040727605053525830239832961008552982304497691439383022566618638381796002"
    "54051950569374547392515068357773127490685649548117139715971745147241514401792e+165"
  },
  {
    1e167,
    0,
    "100000000000000003860899428741951440279402051491350438954423829568577391016492742670197391"
    "754543170343555750902863155030391327289536708508823166797373630632400726786048",
    167,
    "1.0000000000000000386089942874195144027940205149135043895442382956857739101649274267019739"
    "1754543170343555750902863155030391327289536708508823166797373630632400726786048e+167"
  },
  {
    1e168,
    0,
    "999999999999999933860494834742974562371950216430331518611692822307700646699603647625692432"
    "595845947170914554599698521475539380813444812793279458505403728617494385000448",
    167,
    "9.9999999999999993386049483474297456237195021643033151861169282230770064669960364762569243"
    "2595845947170914554599698521475539380813444812793279458505403728617494385000448e+167"
  },
  {
    1e169,
    0,
    "999999999999999933860494834742974562371950216430331518611692822307700646699603647625692432"
    "5958459471709145545996985214755393808134448127932794585054037286174943850004480",
    167,
    "9.9999999999999993386049483474297456237195021643033151861169282230770064669960364762569243"
    "2595845947170914554599698521475539380813444812793279458505403728617494385000448e+168"
  },
  {
    1e170,
    0,
    "100000000000000003441905430931245280917713770297417747470693647675065097962631447553892265"
    "814744827318497179085147422915077831721209019419643357959500300321574675254607872",
    170,
    "1.0000000000000000344190543093124528091771377029741774747069364767506509796263144755389226"
    "5814744827318497179085147422915077831721209019419643357959500300321574675254607872e+170"
  },
  {
    1e171,
    0,
    "999999999999999953972206729656870211732987713739100709830741553196290713284945813208338477"
    "706166412373726001850053663010587168093173889073910282723323583537144858509574144",
    170,
    "9.9999999999999995397220672965687021173298771373910070983074155319629071328494581320833847"
    "7706166412373726001850053663010587168093173889073910282723323583537144858509574144e+170"
  },
  {
    1e172,
    0,
    "100000000000000008268716285710580236764362769651522353363265343088326713943113567293727316"
    "64122173896717192642523265688348930066834399772699475577180106550229078889679814656",
    172,
    "1.0000000000000000826871628571058023676436276965152235336326534308832671394311356729372731"
    "664122173896717192642523265688348930066834399772699475577180106550229078889679814656e+172"
  },
  {
    1e173,
    0,
    "100000000000000001403918625579970521782461970570129136093830042945021304548650108108184133"
    "243565686844612285763778101906192989276863139689872767772084421689716760605683089408",
    173,
    "1.0000000000000000140391862557997052178246197057012913609383004294502130454865010810818413"
    "3243565686844612285763778101906192989276863139689872767772084421689716760605683089408e+173"
  },
  {
    1e174,
    0,
    "100000000000000006895756753684458293767982609835243709909378283059665632064220875456618679"
    "9616905285426599982929417458880300383900478261195703581718577367397759832385751351296",
    174,
    "1.0000000000000000689575675368445829376798260983524370990937828305966563206422087545661867"
    "99616905285426599982929417458880300383900478261195703581718577367397759832385751351296e+17"
    "4"
  },
  {
    1e175,
    0,
    "999999999999999937153452462336876410027330755989687327520625067845192460268510338203757678"
    "3819090846734548822294900033162112051840457868829614121240178061963384891963422539776",
    174,
    "9.9999999999999993715345246233687641002733075598968732752062506784519246026851033820375767"
    "83819090846734548822294900033162112051840457868829614121240178061963384891963422539776e+17"
    "4"
  },
  {
    1e176,
    0,
    "100000000000000000744898050207431989144199493858315387235964254131263985246781616026371987"
    "637390705840846560260278464628372543383280977318309056924111623883709653889736043921408",
    176,
    "1.0000000000000000074489805020743198914419949385831538723596425413126398524678161602637198"
    "7637390705840846560260278464628372543383280977318309056924111623883709653889736043921408e+"
    "176"
  },
  {
    1e177,
    0,
    "100000000000000000744898050207431989144199493858315387235964254131263985246781616026371987"
    "6373907058408465602602784646283725433832809773183090569241116238837096538897360439214080",
    176,
    "1.0000000000000000074489805020743198914419949385831538723596425413126398524678161602637198"
    "7637390705840846560260278464628372543383280977318309056924111623883709653889736043921408e+"
    "177"
  },
  {
    1e178,
    0,
    "100000000000000005243811844750628371954738001544297246105661372433180618347537188638209568"
    "30887857615988724636416932177829345401680187244151732297960592357271816907060120777654272",
    178,
    "1.0000000000000000524381184475062837195473800154429724610566137243318061834753718863820956"
    "830887857615988724636416932177829345401680187244151732297960592357271816907060120777654272"
    "e+178"
  },
  {
    1e179,
    0,
    "999999999999999980455497734815141594578763892467262719141459831501140053863282724592694392"
    "34497983649422148597943950338419997003168440244384097290815044070304544781216945608327168",
    178,
    "9.9999999999999998045549773481514159457876389246726271914145983150114005386328272459269439"
    "234497983649422148597943950338419997003168440244384097290815044070304544781216945608327168"
    "e+178"
  },
  {
    1e180,
    0,
    "100000000000000000924854601989159844456621034165754661590752138863340650570811838930845490"
    "864250220653608187704434098914369379808621813123237387566331395871269994496970650475613388"
    "8",
    180,
    "1.0000000000000000092485460198915984445662103416575466159075213886334065057081183893084549"
    "086425022065360818770443409891436937980862181312323738756633139587126999449697065047561338"
    "88e+180"
  },
  {
    1e181,
    0,
    "999999999999999917110791507646936524606381704248638146256124405810153859804644262218021256"
    "490430622402128625636656234713313548311710199109068586846790701081805554065587949002974822"
    "4",
    180,
    "9.9999999999999991711079150764693652460638170424863814625612440581015385980464426221802125"
    "649043062240212862563665623471331354831171019910906858684679070108180555406558794900297482"
    "24e+180"
  },
  {
    1e182,
    0,
    "100000000000000006453119872723839559654210752410289169769835957832735809325020286556271509"
    "993374515701645382788895184180192194795092289050635704895322791329123657951217763820802932"
    "736",
    182,
    "1.0000000000000000645311987272383955965421075241028916976983595783273580932502028655627150"
    "999337451570164538278889518418019219479509228905063570489532279132912365795121776382080293"
    "2736e+182"
  },
  {
    1e183,
    0,
    "999999999999999946594872951565228338993526868219488856544571440313594706493755982886960025"
    "179093529324993666087115356131035228239552737388526279268078143523691759154905886843985723"
    "392",
    182,
    "9.9999999999999994659487295156522833899352686821948885654457144031359470649375598288696002"
    "517909352932499366608711535613103522823955273738852627926807814352369175915490588684398572"
    "3392e+182"
  },
  {
    1e184,
    0,
    "100000000000000001735666841696912869352267526174953056123684432312185273854762411249241307"
    "003188450593986976316821724753356726006637482925922474107916800538421865136926893766241188"
    "57728",
    184,
    "1.0000000000000000173566684169691286935226752617495305612368443231218527385476241124924130"
    "700318845059398697631682172475335672600663748292592247410791680053842186513692689376624118"
    "857728e+184"
  },
  {
    1e185,
    0,
    "999999999999999979617044168753715171107129451866841652067632118957448454785561110036171446"
    "110395985078602511391629572118883509758736380261518894779920079058604308854941977225917932"
    "50304",
    184,
    "9.9999999999999997961704416875371517110712945186684165206763211895744845478556111003617144"
    "611039598507860251139162957211888350975873638026151889477992007905860430885494197722591793"
    "250304e+184"
  },
  {
    1e186,
    0,
    "999999999999999979617044168753715171107129451866841652067632118957448454785561110036171446"
    "110395985078602511391629572118883509758736380261518894779920079058604308854941977225917932"
    "503040",
    184,
    "9.9999999999999997961704416875371517110712945186684165206763211895744845478556111003617144"
    "611039598507860251139162957211888350975873638026151889477992007905860430885494197722591793"
    "250304e+185"
  },
  {
    1e187,
    0,
    "999999999999999907156965612180121208069281496892078946462744686961792229962400145320187528"
    "181138025024969387980581235322690709168070558185923669885364060513424771227434213187849542"
    "2251008",
    186,
    "9.9999999999999990715696561218012120806928149689207894646274468696179222996240014532018752"
    "818113802502496938798058123532269070916807055818592366988536406051342477122743421318784954"
    "22251008e+186"
  },
  {
    1e188,
    0,
    "100000000000000002309309130269787154892983822485169927543056457815484218967945768886576179"
    "686795076111078238543825857419659919011313587350687602971665369018571203143144663564875896"
    "666980352",
    188,
    "1.0000000000000000230930913026978715489298382248516992754305645781548421896794576888657617"
    "968679507611107823854382585741965991901131358735068760297166536901857120314314466356487589"
    "6666980352e+188"
  },
  {
    1e189,
    0,
    "100000000000000002309309130269787154892983822485169927543056457815484218967945768886576179"
    "686795076111078238543825857419659919011313587350687602971665369018571203143144663564875896"
    "6669803520",
    188,
    "1.0000000000000000230930913026978715489298382248516992754305645781548421896794576888657617"
    "968679507611107823854382585741965991901131358735068760297166536901857120314314466356487589"
    "6666980352e+189"
  },
  {
    1e190,
    0,
    "100000000000000007255917159731877836103034242878113728245683439839721017249206890744520681"
    "817432419517406259768686757211613347531636374137714903657800393217922126245182526923208032"
    "10995433472",
    190,
    "1.0000000000000000725591715973187783610303424287811372824568343983972101724920689074452068"
    "181743241951740625976868675721161334753163637413771490365780039321792212624518252692320803"
    "210995433472e+190"
  },
  {
    1e191,
    0,
    "100000000000000007255917159731877836103034242878113728245683439839721017249206890744520681"
    "817432419517406259768686757211613347531636374137714903657800393217922126245182526923208032"
    "109954334720",
    190,
    "1.0000000000000000725591715973187783610303424287811372824568343983972101724920689074452068"
    "181743241951740625976868675721161334753163637413771490365780039321792212624518252692320803"
    "210995433472e+191"
  },
  {
    1e192,
    0,
    "100000000000000004090088020876139800128601973826629695796002171344209466349199772755436200"
    "453824519737356326184775781344763153278629790594017431218673977730337535459878294373875465"
    "4264509857792",
    192,
    "1.0000000000000000409008802087613980012860197382662969579600217134420946634919977275543620"
    "045382451973735632618477578134476315327862979059401743121867397773033753545987829437387546"
    "54264509857792e+192"
  },
  {
    1e193,
    0,
    "100000000000000006622751331960730228908147789067816921755747186140618707069205467146703785"
    "544710839561396273051904562038243308681035057428975409169975110120405208088121680413341518"
    "77325366493184",
    193,
    "1.0000000000000000662275133196073022890814778906781692175574718614061870706920546714670378"
    "554471083956139627305190456203824330868103505742897540916997511012040520808812168041334151"
    "877325366493184e+193"
  },
  {
    1e194,
    0,
    "999999999999999944659674387546961707663278759101182371489711151178543516131781340686193771"
    "084565044060045280896864147095385627494897766211771150037296746480803794725534274239044627"
    "08600804999168",
    193,
    "9.9999999999999994465967438754696170766327875910118237148971115117854351613178134068619377"
    "108456504406004528089686414709538562749489776621177115003729674648080379472553427423904462"
    "708600804999168e+193"
  },
  {
    1e195,
    0,
    "999999999999999977077764769429719196041465194188378863774447340572581797347854228894418860"
    "247909937807756600796112539971931616645685181699233267813951241073670004367049615544210109"
    "925082343145472",
    194,
    "9.9999999999999997707776476942971919604146519418837886377444734057258179734785422889441886"
    "024790993780775660079611253997193161664568518169923326781395124107367000436704961554421010"
    "9925082343145472e+194"
  },
  {
    1e196,
    0,
    "999999999999999951143292463923513205338916046118621669946658389057351172374995918327838788"
    "917234022809587544876713825670694825325055249309263573592627645399377036653837342500077723"
    "6538229086224384",
    195,
    "9.9999999999999995114329246392351320533891604611862166994665838905735117237499591832783878"
    "891723402280958754487671382567069482532505524930926357359262764539937703665383734250007772"
    "36538229086224384e+195"
  },
  {
    1e197,
    0,
    "999999999999999951143292463923513205338916046118621669946658389057351172374995918327838788"
    "917234022809587544876713825670694825325055249309263573592627645399377036653837342500077723"
    "65382290862243840",
    195,
    "9.9999999999999995114329246392351320533891604611862166994665838905735117237499591832783878"
    "891723402280958754487671382567069482532505524930926357359262764539937703665383734250007772"
    "36538229086224384e+196"
  },
  {
    1e198,
    0,
    "100000000000000001753554156601940054153744186517720008614579810493634157230551319337828377"
    "152376436520490032803037453428186101110586787622758599079921605032556703399966076149305663"
    "2508247061001404416",
    198,
    "1.0000000000000000175355415660194005415374418651772000861457981049363415723055131933782837"
    "715237643652049003280303745342818610111058678762275859907992160503255670339996607614930566"
    "32508247061001404416e+198"
  },
  {
    1e199,
    0,
    "100000000000000009720624048853446534497567284804749418558476576399113005222213392343881775"
    "065160077607927566781476738461526043404284302852957289144712213623699503081464886428463132"
    "31335560438561636352",
    199,
    "1.0000000000000000972062404885344653449756728480474941855847657639911300522221339234388177"
    "506516007760792756678147673846152604340428430285295728914471221362369950308146488642846313"
    "231335560438561636352e+199"
  },
  {
    1e200,
    0,
    "999999999999999969733122212510361659474503275455023626482417509503468484355540755341963384"
    "047062518680275124159738824081821357343682784846393850410472398778710235910667899818111818"
    "13306167128854888448",
    199,
    "9.9999999999999996973312221251036165947450327545502362648241750950346848435554075534196338"
    "404706251868027512415973882408182135734368278484639385041047239877871023591066789981811181"
    "813306167128854888448e+199"
  },
  {
    1e201,
    0,
    "100000000000000003771878529305655029174179371417100792467033657856355465388439044499361904"
    "623614958929307541410908738969965553158323491481075600563001892542312879319279108086692222"
    "0799992003324610084864",
    201,
    "1.0000000000000000377187852930565502917417937141710079246703365785635546538843904449936190"
    "462361495892930754141090873896996555315832349148107560056300189254231287931927910808669222"
    "20799992003324610084864e+201"
  },
  {
    1e202,
    0,
    "999999999999999901747459131964173027207212836739039328294498440443382314826691065690307721"
    "857975448067474834210390258463987183104130654882031695190925872134291678628544718769301415"
    "4661313392524876840960",
    200,
    "9.9999999999999990174745913196417302720721283673903932829449844044338231482669106569030772"
    "185797544806747483421039025846398718310413065488203169519092587213429167862854471876930141"
    "5466131339252487684096e+201"
  },
  {
    1e203,
    0,
    "999999999999999988769107875063294476509344598295499229975034848840292611823618668444426969"
    "460006898451859205345556422454814926130757381236415253871945426239147431949662390511778730"
    "87980216425864602058752",
    202,
    "9.9999999999999998876910787506329447650934459829549922997503484884029261182361866844442696"
    "946000689845185920534555642245481492613075738123641525387194542623914743194966239051177873"
    "087980216425864602058752e+202"
  },
  {
    1e204,
    0,
    "999999999999999988769107875063294476509344598295499229975034848840292611823618668444426969"
    "460006898451859205345556422454814926130757381236415253871945426239147431949662390511778730"
    "879802164258646020587520",
    202,
    "9.9999999999999998876910787506329447650934459829549922997503484884029261182361866844442696"
    "946000689845185920534555642245481492613075738123641525387194542623914743194966239051177873"
    "087980216425864602058752e+203"
  },
  {
    1e205,
    0,
    "100000000000000001661603547285501334028602676199356639851280649952730390686263550132574512"
    "869265696257486220410880959493187980389927793366981799264987168355270127301242004546937147"
    "18121768282606166882648064",
    205,
    "1.0000000000000000166160354728550133402860267619935663985128064995273039068626355013257451"
    "286926569625748622041088095949318798038992779336698179926498716835527012730124200454693714"
    "718121768282606166882648064e+205"
  },
  {
    1e206,
    0,
    "100000000000000003889357755108838843130737249295202013334302382007691294289384896763079965"
    "607877701387326460311941213291353170611409437561654018367221268940354434586262616943544566"
    "455807655946219322240663552",
    206,
    "1.0000000000000000388935775510883884313073724929520201333430238200769129428938489676307996"
    "560787770138732646031194121329135317061140943756165401836722126894035443458626261694354456"
    "6455807655946219322240663552e+206"
  },
  {
    1e207,
    0,
    "100000000000000003889357755108838843130737249295202013334302382007691294289384896763079965"
    "607877701387326460311941213291353170611409437561654018367221268940354434586262616943544566"
    "4558076559462193222406635520",
    206,
    "1.0000000000000000388935775510883884313073724929520201333430238200769129428938489676307996"
    "560787770138732646031194121329135317061140943756165401836722126894035443458626261694354456"
    "6455807655946219322240663552e+207"
  },
  {
    1e208,
    0,
    "999999999999999981863069830810948198292727421698378572177667479469913810653942493889860065"
    "970309682549354461652269635680502836444164284232931374655019714425386079366098492082295731"
    "1285732475861572950035529728",
    207,
    "9.9999999999999998186306983081094819829272742169837857217766747946991381065394249388986006"
    "597030968254935446165226963568050283644416428423293137465501971442538607936609849208229573"
    "11285732475861572950035529728e+207"
  },
  {
    1e209,
    0,
    "100000000000000007311188218325485257111615953570420507004223762444111242223779285187536341"
    "014385741266761068799969763125334902791605243044670546908252847439043930576054277584733562"
    "461577854658781477884848504832",
    209,
    "1.0000000000000000731118821832548525711161595357042050700422376244411124222377928518753634"
    "101438574126676106879996976312533490279160524304467054690825284743904393057605427758473356"
    "2461577854658781477884848504832e+209"
  },
  {
    1e210,
    0,
    "999999999999999927113782419344605574598668153294882673458925392487194643703632279098558059"
    "466181044478400725843812838336795121561031396504666917998514458446354143529431921823271795"
    "036250068185162804696593727488",
    209,
    "9.9999999999999992711378241934460557459866815329488267345892539248719464370363227909855805"
    "946618104447840072584381283833679512156103139650466691799851445844635414352943192182327179"
    "5036250068185162804696593727488e+209"
  },
  {
    1e211,
    0,
    "999999999999999956313402372126654973902166429776747152775587838877978199410464393653919129"
    "601716318116242718274989796920105902832035603293074628215317261635171175975654092628084560"
    "9521557638656931995269719916544",
    210,
    "9.9999999999999995631340237212665497390216642977674715277558783887797819941046439365391912"
    "960171631811624271827498979692010590283203560329307462821531726163517117597565409262808456"
    "09521557638656931995269719916544e+210"
  },
  {
    1e212,
    0,
    "999999999999999909594010447675375935016569187405763985868927924652724510279533010365341417"
    "384859880295695530385106663186808652798428872431622291868432776533063924061698619340384135"
    "48670665077684456779836676898816",
    211,
    "9.9999999999999990959401044767537593501656918740576398586892792465272451027953301036534141"
    "738485988029569553038510666318680865279842887243162229186843277653306392406169861934038413"
    "548670665077684456779836676898816e+211"
  },
  {
    1e213,
    0,
    "999999999999999984345037526797422397233524775199337052919583787413130412889023223627065756"
    "931830180808571031008919677160084252852199641809946030023447952696435527124027376600704816"
    "231425231719002378564135125254144",
    212,
    "9.9999999999999998434503752679742239723352477519933705291958378741313041288902322362706575"
    "693183018080857103100891967716008425285219964180994603002344795269643552712402737660070481"
    "6231425231719002378564135125254144e+212"
  },
  {
    1e214,
    0,
    "999999999999999954444626695148603812346742540081907826099321442308968051845227138322376021"
    "113042060603420830759394471570774012830691334058616534761441882231086885899095873696576543"
    "9335377993421392542578277827477504",
    213,
    "9.9999999999999995444462669514860381234674254008190782609932144230896805184522713832237602"
    "111304206060342083075939447157077401283069133405861653476144188223108688589909587369657654"
    "39335377993421392542578277827477504e+213"
  },
  {
    1e215,
    0,
    "999999999999999906603969364510494076527890963894021063186901690142308274175153401834872443"
    "802981068275180510360154142627877628796278041656489342342232169486529059939205469049971308"
    "25691790753915825536773603473752064",
    214,
    "9.9999999999999990660396936451049407652789096389402106318690169014230827417515340183487244"
    "380298106827518051036015414262787762879627804165648934234223216948652905993920546904997130"
    "825691790753915825536773603473752064e+214"
  },
  {
    1e216,
    0,
    "100000000000000002142154695804195744249313474674494929417670909534229174058333036940488102"
    "934712744986295727931833093209082895047886994342159460414833548007346784224294244020182387"
    "3880805647866312652703956229962072064",
    216,
    "1.0000000000000000214215469580419574424931347467449492941767090953422917405833303694048810"
    "293471274498629572793183309320908289504788699434215946041483354800734678422429424402018238"
    "73880805647866312652703956229962072064e+216"
  },
  {
    1e217,
    0,
    "999999999999999960185505574825176980645004729224454237648811812568967225165635986700876450"
    "390249379682809669207303311043921578914820929146871797851747047760433825014282722254169172"
    "2147321863584969741246387925089779712",
    216,
    "9.9999999999999996018550557482517698064500472922445423764881181256896722516563598670087645"
    "039024937968280966920730331104392157891482092914687179785174704776043382501428272225416917"
    "22147321863584969741246387925089779712e+216"
  },
  {
    1e218,
    0,
    "100000000000000008265758834125873790434126476426544435070460637811561625600102475210888560"
    "830400552004310488942935855313773632204291895769631741044492391238650185947160215814947857"
    "554687910937412833128327366741516615680",
    217,
    "1.0000000000000000826575883412587379043412647642654443507046063781156162560010247521088856"
    "083040055200431048894293585531377363220429189576963174104449239123865018594716021581494785"
    "755468791093741283312832736674151661568e+218"
  },
  {
    1e219,
    0,
    "999999999999999965084388885482519417592855130626093842171043595190833186399051537317196816"
    "706799625297221478016185520727674168639944850288849622355474122345476546392575499689981548"
    "348018063279122228410984187505225498624",
    218,
    "9.9999999999999996508438888548251941759285513062609384217104359519083318639905153731719681"
    "670679962529722147801618552072767416863994485028884962235547412234547654639257549968998154"
    "8348018063279122228410984187505225498624e+218"
  },
  {
    1e220,
    0,
    "999999999999999996437242073689511014059097699596587311113327003970775338292911061261647161"
    "132721197229457054393031662703690742880737945597507699179327399689749963213649275279180755"
    "6010476755711238558435947154812096741376",
    219,
    "9.9999999999999999643724207368951101405909769959658731111332700397077533829291106126164716"
    "113272119722945705439303166270369074288073794559750769917932739968974996321364927527918075"
    "56010476755711238558435947154812096741376e+219"
  },
  {
    1e221,
    0,
    "100000000000000004660180717482069756840508580994937686142098045801868278132308629957276771"
    "221419571232103397659598548986531726166600689809136062209749264344058743012736731622189948"
    "720589505523832645973577156024278435495936",
    221,
    "1.0000000000000000466018071748206975684050858099493768614209804580186827813230862995727677"
    "122141957123210339765959854898653172616660068980913606220974926434405874301273673162218994"
    "8720589505523832645973577156024278435495936e+221"
  },
  {
    1e222,
    0,
    "100000000000000004660180717482069756840508580994937686142098045801868278132308629957276771"
    "221419571232103397659598548986531726166600689809136062209749264344058743012736731622189948"
    "7205895055238326459735771560242784354959360",
    221,
    "1.0000000000000000466018071748206975684050858099493768614209804580186827813230862995727677"
    "122141957123210339765959854898653172616660068980913606220974926434405874301273673162218994"
    "8720589505523832645973577156024278435495936e+222"
  },
  {
    1e223,
    0,
    "100000000000000004660180717482069756840508580994937686142098045801868278132308629957276771"
    "221419571232103397659598548986531726166600689809136062209749264344058743012736731622189948"
    "72058950552383264597357715602427843549593600",
    221,
    "1.0000000000000000466018071748206975684050858099493768614209804580186827813230862995727677"
    "122141957123210339765959854898653172616660068980913606220974926434405874301273673162218994"
    "8720589505523832645973577156024278435495936e+223"
  },
  {
    1e224,
    0,
    "999999999999999969549035179483195020929648072447492112148424752601096948828737133526886545"
    "753050857140371824092248411345058928811833787060802532495190829039301080947896405333883515"
    "46084948006950326015738792668900564521713664",
    223,
    "9.9999999999999996954903517948319502092964807244749211214842475260109694882873713352688654"
    "575305085714037182409224841134505892881183378706080253249519082903930108094789640533388351"
    "546084948006950326015738792668900564521713664e+223"
  },
  {
    1e225,
    0,
    "999999999999999928454223448636526995609414612446486912536395043045051171498417578302416590"
    "307106934377352009423588636134254484622941461177838218040629861358615028052178586193608330"
    "530158506646130887048916655460323666687950848",
    224,
    "9.9999999999999992845422344863652699560941461244648691253639504304505117149841757830241659"
    "030710693437735200942358863613425448462294146117783821804062986135861502805217858619360833"
    "0530158506646130887048916655460323666687950848e+224"
  },
  {
    1e226,
    0,
    "999999999999999961330072833313861415865601380447291072226018810689887793362673222481992554"
    "663862072587767861158516456302898039974055321884209669604278635503163870368752841505828478"
    "4747112853848287855356936724432692495112994816",
    225,
    "9.9999999999999996133007283331386141586560138044729107222601881068988779336267322248199255"
    "466386207258776786115851645630289803997405532188420966960427863550316387036875284150582847"
    "84747112853848287855356936724432692495112994816e+225"
  },
  {
    1e227,
    0,
    "100000000000000009283347037202319909689034845245050771098451388126923428081969579920029641"
    "209088262542943126809822773697747226137851076470969547585887373208135923963504986275470907"
    "025292240033962037948280174037505158080469401600",
    225,
    "1.0000000000000000928334703720231990968903484524505077109845138812692342808196957992002964"
    "120908826254294312680982277369774722613785107647096954758588737320813592396350498627547090"
    "70252922400339620379482801740375051580804694016e+227"
  },
  {
    1e228,
    0,
    "999999999999999924509121522475246865178672200286390413373640190927670776874706901000867474"
    "584296317792102107215397297714017257980807797893073643852992008461269166974189675556141912"
    "776812173197487139230503413422370196749149011968",
    227,
    "9.9999999999999992450912152247524686517867220028639041337364019092767077687470690100086747"
    "458429631779210210721539729771401725798080779789307364385299200846126916697418967555614191"
    "2776812173197487139230503413422370196749149011968e+227"
  },
  {
    1e229,
    0,
    "999999999999999991838861062294427757863342701152037332417989667064296178452702460280639049"
    "586930840847033771568529473419399259339888984619722376655344697909305196038533750435568775"
    "7672562640543404353314227442034427503713670135808",
    228,
    "9.9999999999999999183886106229442775786334270115203733241798966706429617845270246028063904"
    "958693084084703377156852947341939925933988898461972237665534469790930519603853375043556877"
    "57672562640543404353314227442034427503713670135808e+228"
  },
  {
    1e230,
    0,
    "100000000000000009956644432600511718615881550253707240288894882888289682097749535512827356"
    "959114607773492443453354095454801046151441888338236034913910900102616284254148427024265175"
    "655196680942530570909289367345315883616691581616128",
    230,
    "1.0000000000000000995664443260051171861588155025370724028889488288828968209774953551282735"
    "695911460777349244345335409545480104615144188833823603491391090010261628425414842702426517"
    "5655196680942530570909289367345315883616691581616128e+230"
  },
  {
    1e231,
    0,
    "100000000000000005647541102052084141484062638198305837470056516415545656396757819718921976"
    "158945998297976816934753636209656598064460692387730516014560327977941978394030406231981856"
    "4238082591276919599588305301753272401848696295129088",
    231,
    "1.0000000000000000564754110205208414148406263819830583747005651641554565639675781971892197"
    "615894599829797681693475363620965659806446069238773051601456032797794197839403040623198185"
    "64238082591276919599588305301753272401848696295129088e+231"
  },
  {
    1e232,
    0,
    "100000000000000005647541102052084141484062638198305837470056516415545656396757819718921976"
    "158945998297976816934753636209656598064460692387730516014560327977941978394030406231981856"
    "42380825912769195995883053017532724018486962951290880",
    231,
    "1.0000000000000000564754110205208414148406263819830583747005651641554565639675781971892197"
    "615894599829797681693475363620965659806446069238773051601456032797794197839403040623198185"
    "64238082591276919599588305301753272401848696295129088e+232"
  },
  {
    1e233,
    0,
    "999999999999999973740627073991031933909703270519351440578868527878771270508537253946236450"
    "226222681049868140190407544589792577374567961627599197278072294985673111426038063107978834"
    "99542489243201826933949562808949044795771481474727936",
    232,
    "9.9999999999999997374062707399103193390970327051935144057886852787877127050853725394623645"
    "022622268104986814019040754458979257737456796162759919727807229498567311142603806310797883"
    "499542489243201826933949562808949044795771481474727936e+232"
  },
  {
    1e234,
    0,
    "100000000000000001786584517880693032373952892996666180544377340055967009368669242367582754"
    "961994924207914815574087624726007172578525540816077571080742215354233800343364659602096002"
    "3924842331815965645472194120710174156699571604284243968",
    234,
    "1.0000000000000000178658451788069303237395289299666618054437734005596700936866924236758275"
    "496199492420791481557408762472600717257852554081607757108074221535423380034336465960209600"
    "23924842331815965645472194120710174156699571604284243968e+234"
  },
  {
    1e235,
    0,
    "100000000000000005316601966265964903560338945752451009733569729870438915222921655945950042"
    "913493049090257216818125120939629504451380536538731692163090204038766991703973342235134497"
    "50683762833231235463783529148067211236930570359138156544",
    235,
    "1.0000000000000000531660196626596490356033894575245100973356972987043891522292165594595004"
    "291349304909025721681812512093962950445138053653873169216309020403876699170397334223513449"
    "750683762833231235463783529148067211236930570359138156544e+235"
  },
  {
    1e236,
    0,
    "100000000000000005316601966265964903560338945752451009733569729870438915222921655945950042"
    "913493049090257216818125120939629504451380536538731692163090204038766991703973342235134497"
    "506837628332312354637835291480672112369305703591381565440",
    235,
    "1.0000000000000000531660196626596490356033894575245100973356972987043891522292165594595004"
    "291349304909025721681812512093962950445138053653873169216309020403876699170397334223513449"
    "750683762833231235463783529148067211236930570359138156544e+236"
  },
  {
    1e237,
    0,
    "999999999999999940205461314330949157639035769339395563281540824641288164893139324951747214"
    "686990494667615328372051330560380424582445502262385046995766402482607793500255578094113131"
    "409067638500218263478644773697770829313903654699186257920",
    235,
    "9.9999999999999994020546131433094915763903576933939556328154082464128816489313932495174721"
    "468699049466761532837205133056038042458244550226238504699576640248260779350025557809411313"
    "140906763850021826347864477369777082931390365469918625792e+236"
  },
  {
    1e238,
    0,
    "100000000000000004864759732872650104048481530999710551597353103974186511273577347007919030"
    "055701289105317389458888321424285845971655097086231964664549661487146743209815430858105570"
    "13220039375302073350623645891623631119178909006652304785408",
    238,
    "1.0000000000000000486475973287265010404848153099971055159735310397418651127357734700791903"
    "005570128910531738945888832142428584597165509708623196466454966148714674320981543085810557"
    "013220039375302073350623645891623631119178909006652304785408e+238"
  },
  {
    1e239,
    0,
    "999999999999999990811791454382206702967066221646326874537802925021557407219701926011220654"
    "759667612980875992606572876278870174311694720942354526832307168264075624845941652321352997"
    "36843791138087983021771402091458056119576436948334022754304",
    238,
    "9.9999999999999999081179145438220670296706622164632687453780292502155740721970192601122065"
    "475966761298087599260657287627887017431169472094235452683230716826407562484594165232135299"
    "736843791138087983021771402091458056119576436948334022754304e+238"
  },
  {
    1e240,
    0,
    "100000000000000001394611380411992443797416585698663833111209417090968048942613054363840851"
    "307860572420979515339949701146446548847363722091034057475758294690703234774682671482523407"
    "8949864321840610832155574248213693581484614981956096327942144",
    240,
    "1.0000000000000000139461138041199244379741658569866383311120941709096804894261305436384085"
    "130786057242097951533994970114644654884736372209103405747575829469070323477468267148252340"
    "78949864321840610832155574248213693581484614981956096327942144e+240"
  },
  {
    1e241,
    0,
    "100000000000000005096102956370027281398552527353113666163096016433067742095641633184190908"
    "638890670217606581066817562776141799113274522085911825143802419273576310438824281483144380"
    "94801465785761804352561506118922744139467759619125060885807104",
    241,
    "1.0000000000000000509610295637002728139855252735311366616309601643306774209564163318419090"
    "863889067021760658106681756277614179911327452208591182514380241927357631043882428148314438"
    "094801465785761804352561506118922744139467759619125060885807104e+241"
  },
  {
    1e242,
    0,
    "100000000000000005096102956370027281398552527353113666163096016433067742095641633184190908"
    "638890670217606581066817562776141799113274522085911825143802419273576310438824281483144380"
    "948014657857618043525615061189227441394677596191250608858071040",
    241,
    "1.0000000000000000509610295637002728139855252735311366616309601643306774209564163318419090"
    "863889067021760658106681756277614179911327452208591182514380241927357631043882428148314438"
    "094801465785761804352561506118922744139467759619125060885807104e+242"
  },
  {
    1e243,
    0,
    "100000000000000007465057564983169577463279530011961559316303440012011545713579923629214945"
    "330749932807447903132012994219146759283457434082633596451350659006615078863874911883541803"
    "7019527222886944981240519484646566146722558989084608335389392896",
    243,
    "1.0000000000000000746505756498316957746327953001196155931630344001201154571357992362921494"
    "533074993280744790313201299421914675928345743408263359645135065900661507886387491188354180"
    "37019527222886944981240519484646566146722558989084608335389392896e+243"
  },
  {
    1e244,
    0,
    "100000000000000007465057564983169577463279530011961559316303440012011545713579923629214945"
    "330749932807447903132012994219146759283457434082633596451350659006615078863874911883541803"
    "70195272228869449812405194846465661467225589890846083353893928960",
    243,
    "1.0000000000000000746505756498316957746327953001196155931630344001201154571357992362921494"
    "533074993280744790313201299421914675928345743408263359645135065900661507886387491188354180"
    "37019527222886944981240519484646566146722558989084608335389392896e+244"
  },
  {
    1e245,
    0,
    "100000000000000004432795665958347438500428966608636256080197937830963477082618911859584178"
    "365170076692451010888562841972100410265623306726829729177688912148325455279810104971033102"
    "576911999816916636238052732752107272876955671430431745947427930112",
    245,
    "1.0000000000000000443279566595834743850042896660863625608019793783096347708261891185958417"
    "836517007669245101088856284197210041026562330672682972917768891214832545527981010497103310"
    "2576911999816916636238052732752107272876955671430431745947427930112e+245"
  },
  {
    1e246,
    0,
    "100000000000000006858605185178205149670709417331296498669082339575801931987387721275288791"
    "937633961584448524683322963769737489479890608611472822996618309634957154147061950501040063"
    "4769445777943389257468521053221467463131958534128550160206370177024",
    246,
    "1.0000000000000000685860518517820514967070941733129649866908233957580193198738772127528879"
    "193763396158444852468332296376973748947989060861147282299661830963495715414706195050104006"
    "34769445777943389257468521053221467463131958534128550160206370177024e+246"
  },
  {
    1e247,
    0,
    "999999999999999952147194929228881360533632538625273342424372112005773484444974360799066467"
    "898073141028604584684743791410795092514075595651859726657572016991249995842530919570066511"
    "5678820350271193610461511698595727381924297989722331966923339726848",
    246,
    "9.9999999999999995214719492922888136053363253862527334242437211200577348444497436079906646"
    "789807314102860458468474379141079509251407559565185972665757201699124999584253091957006651"
    "15678820350271193610461511698595727381924297989722331966923339726848e+246"
  },
  {
    1e248,
    0,
    "100000000000000004529828046727141746947240184637542665783753313900757015278809664236212362"
    "908068632088130911440353246844005893434193998802215452930446088047790723234500178792233381"
    "012913302936013527818404707654908851814405278709728676750356293615616",
    248,
    "1.0000000000000000452982804672714174694724018463754266578375331390075701527880966423621236"
    "290806863208813091144035324684400589343419399880221545293044608804779072323450017879223338"
    "1012913302936013527818404707654908851814405278709728676750356293615616e+248"
  },
  {
    1e249,
    0,
    "999999999999999921096833083214702657554042769375222237286651769671841261663933600278047414"
    "170535414411036408111814232401040478571454131528428125775275729162364250341707296785977412"
    "047465036916114055333519200963067478208555469597215339755257651527680",
    247,
    "9.9999999999999992109683308321470265755404276937522223728665176967184126166393360027804741"
    "417053541441103640811181423240104047857145413152842812577527572916236425034170729678597741"
    "204746503691611405533351920096306747820855546959721533975525765152768e+248"
  },
  {
    1e250,
    0,
    "999999999999999921096833083214702657554042769375222237286651769671841261663933600278047414"
    "170535414411036408111814232401040478571454131528428125775275729162364250341707296785977412"
    "0474650369161140553335192009630674782085554695972153397552576515276800",
    247,
    "9.9999999999999992109683308321470265755404276937522223728665176967184126166393360027804741"
    "417053541441103640811181423240104047857145413152842812577527572916236425034170729678597741"
    "204746503691611405533351920096306747820855546959721533975525765152768e+249"
  },
  {
    1e251,
    0,
    "100000000000000004827911520448877862495844246422343156393075429187162764617507655537214145"
    "823852994263659565935453370610499537728043164857800396298916132410948026391308085570960636"
    "368309306117879178753245974556315302310250472271728848176952226298724352",
    251,
    "1.0000000000000000482791152044887786249584424642234315639307542918716276461750765553721414"
    "582385299426365956593545337061049953772804316485780039629891613241094802639130808557096063"
    "6368309306117879178753245974556315302310250472271728848176952226298724352e+251"
  },
  {
    1e252,
    0,
    "100000000000000009915202805299840901192020234216271529458839530075154219997953373740977907"
    "586572775392681935985162149558657733676402265539783429787471556208832666934163027927905794"
    "4337344270883862880412035963403187241060084423965317738575228107571068928",
    252,
    "1.0000000000000000991520280529984090119202023421627152945883953007515421999795337374097790"
    "758657277539268193598516214955865773367640226553978342978747155620883266693416302792790579"
    "44337344270883862880412035963403187241060084423965317738575228107571068928e+252"
  },
  {
    1e253,
    0,
    "999999999999999936358706937767591773642570732757007356483944072335815627805270754889338699"
    "458694757798103518260940569245515066416531433574377226240942000556018171970272123856812886"
    "2437403998276353831973920663150777435958293799716241167969694049028276224",
    252,
    "9.9999999999999993635870693776759177364257073275700735648394407233581562780527075488933869"
    "945869475779810351826094056924551506641653143357437722624094200055601817197027212385681288"
    "62437403998276353831973920663150777435958293799716241167969694049028276224e+252"
  },
  {
    1e254,
    0,
    "999999999999999936358706937767591773642570732757007356483944072335815627805270754889338699"
    "458694757798103518260940569245515066416531433574377226240942000556018171970272123856812886"
    "24374039982763538319739206631507774359582937997162411679696940490282762240",
    252,
    "9.9999999999999993635870693776759177364257073275700735648394407233581562780527075488933869"
    "945869475779810351826094056924551506641653143357437722624094200055601817197027212385681288"
    "62437403998276353831973920663150777435958293799716241167969694049028276224e+253"
  },
  {
    1e255,
    0,
    "999999999999999988452569694641453289891412847766833896677368465428848130901034909295879619"
    "908945316559292587569958465674654992927728624557883489163749540246356891129106733591931304"
    "833693638565628182306078113383272782784390994049606075766012189756664840192",
    254,
    "9.9999999999999998845256969464145328989141284776683389667736846542884813090103490929587961"
    "990894531655929258756995846567465499292772862455788348916374954024635689112910673359193130"
    "4833693638565628182306078113383272782784390994049606075766012189756664840192e+254"
  },
  {
    1e256,
    0,
    "100000000000000003012765990014054250289048653977469512883210797990327413337764623282111235"
    "626914576356824384301717278281796693413668637734468849950199557199862786645617442138002603"
    "97056562295560224215930269510378288141352402853119916429412464176397346144256",
    256,
    "1.0000000000000000301276599001405425028904865397746951288321079799032741333776462328211123"
    "562691457635682438430171727828179669341366863773446884995019955719986278664561744213800260"
    "397056562295560224215930269510378288141352402853119916429412464176397346144256e+256"
  },
  {
    1e257,
    0,
    "100000000000000003012765990014054250289048653977469512883210797990327413337764623282111235"
    "626914576356824384301717278281796693413668637734468849950199557199862786645617442138002603"
    "970565622955602242159302695103782881413524028531199164294124641763973461442560",
    256,
    "1.0000000000000000301276599001405425028904865397746951288321079799032741333776462328211123"
    "562691457635682438430171727828179669341366863773446884995019955719986278664561744213800260"
    "397056562295560224215930269510378288141352402853119916429412464176397346144256e+257"
  },
  {
    1e258,
    0,
    "100000000000000005679971763165995959920989370265972631741114126916690677496267747987726130"
    "753967404965397264650338994578968657651041933912824370611847303232008129066549774156440667"
    "0023712287789874734736674207136744674199783831719918405933396323484899269935104",
    258,
    "1.0000000000000000567997176316599595992098937026597263174111412691669067749626774798772613"
    "075396740496539726465033899457896865765104193391282437061184730323200812906654977415644066"
    "70023712287789874734736674207136744674199783831719918405933396323484899269935104e+258"
  },
  {
    1e259,
    0,
    "999999999999999928773840520366757536876739320811576612231781480701470095354527494007746341"
    "441138276442474389769547563525432293116501122567178714359381222777104854460745804679379644"
    "4970432082673836316471673778619485458899748089618699435710767754281089234894848",
    258,
    "9.9999999999999992877384052036675753687673932081157661223178148070147009535452749400774634"
    "144113827644247438976954756352543229311650112256717871435938122277710485446074580467937964"
    "44970432082673836316471673778619485458899748089618699435710767754281089234894848e+258"
  },
  {
    1e260,
    0,
    "100000000000000006533477610574617307003210399478293629775643192173126922026988747893522897"
    "194624310120140586361897943794063686207001388689898137223574581962294638641248120402340847"
    "172549022642470747494264132908839774942043776657045497009088429335535195969814528",
    260,
    "1.0000000000000000653347761057461730700321039947829362977564319217312692202698874789352289"
    "719462431012014058636189794379406368620700138868989813722357458196229463864124812040234084"
    "7172549022642470747494264132908839774942043776657045497009088429335535195969814528e+260"
  },
  {
    1e261,
    0,
    "999999999999999928773840520366757536876739320811576612231781480701470095354527494007746341"
    "441138276442474389769547563525432293116501122567178714359381222777104854460745804679379644"
    "497043208267383631647167377861948545889974808961869943571076775428108923489484800",
    258,
    "9.9999999999999992877384052036675753687673932081157661223178148070147009535452749400774634"
    "144113827644247438976954756352543229311650112256717871435938122277710485446074580467937964"
    "44970432082673836316471673778619485458899748089618699435710767754281089234894848e+260"
  },
  {
    1e262,
    0,
    "100000000000000001617283929500958347809617271215324681096755776296054153530035788436133522"
    "496440536428819053303318396315116321724674929173953241540025456475844343490985646025955809"
    "39232492998880708913562707066468760361494711018313643605437535869015444666630275072",
    262,
    "1.0000000000000000161728392950095834780961727121532468109675577629605415353003578843613352"
    "249644053642881905330331839631511632172467492917395324154002545647584434349098564602595580"
    "939232492998880708913562707066468760361494711018313643605437535869015444666630275072e+262"
  },
  {
    1e263,
    0,
    "100000000000000001617283929500958347809617271215324681096755776296054153530035788436133522"
    "496440536428819053303318396315116321724674929173953241540025456475844343490985646025955809"
    "392324929988807089135627070664687603614947110183136436054375358690154446666302750720",
    262,
    "1.0000000000000000161728392950095834780961727121532468109675577629605415353003578843613352"
    "249644053642881905330331839631511632172467492917395324154002545647584434349098564602595580"
    "939232492998880708913562707066468760361494711018313643605437535869015444666630275072e+263"
  },
  {
    1e264,
    0,
    "100000000000000004414051890289528777928639139738258127456300617328344439608302360927448366"
    "769185083239881969887754761103139711296842870587468559973333403419247178065357187004521519"
    "7739635249206690814463183771858052833032509915549602573975010166573043840478561173504",
    264,
    "1.0000000000000000441405189028952877792863913973825812745630061732834443960830236092744836"
    "676918508323988196988775476110313971129684287058746855997333340341924717806535718700452151"
    "97739635249206690814463183771858052833032509915549602573975010166573043840478561173504e+26"
    "4"
  },
  {
    1e265,
    0,
    "100000000000000006651466258920385122023856634556604884543936490154176668470915618920500242"
    "187380720688732303155303852933558422954577223718280814719979760973969445724854419787374088"
    "07927440086615867529487142240269942705389409665241931447200154303102433395309881065472",
    265,
    "1.0000000000000000665146625892038512202385663455660488454393649015417666847091561892050024"
    "218738072068873230315530385293355842295457722371828081471997976097396944572485441978737408"
    "807927440086615867529487142240269942705389409665241931447200154303102433395309881065472e+2"
    "65"
  },
  {
    1e266,
    0,
    "100000000000000003071603269111014971471508642847250073203719093632845102290734406131617241"
    "518267700770571769927225306004888484302202258708981207125345588886413817469658847334809978"
    "790776999353375325137186550055668797052865128496484823152800700833072414104710501367808",
    266,
    "1.0000000000000000307160326911101497147150864284725007320371909363284510229073440613161724"
    "151826770077057176992722530600488848430220225870898120712534558888641381746965884733480997"
    "8790776999353375325137186550055668797052865128496484823152800700833072414104710501367808e+"
    "266"
  },
  {
    1e267,
    0,
    "999999999999999973438224854160227305877518561122823750593712591987145964024444656694044404"
    "476868689015149167622996309190165824584023146941018349739309135463248122613459314107074039"
    "291811569329219648848907543004197890512187794469896370420793533163493423472892065087488",
    266,
    "9.9999999999999997343822485416022730587751856112282375059371259198714596402444465669404440"
    "447686868901514916762299630919016582458402314694101834973930913546324812261345931410707403"
    "9291811569329219648848907543004197890512187794469896370420793533163493423472892065087488e+"
    "266"
  },
  {
    1e268,
    0,
    "999999999999999973438224854160227305877518561122823750593712591987145964024444656694044404"
    "476868689015149167622996309190165824584023146941018349739309135463248122613459314107074039"
    "2918115693292196488489075430041978905121877944698963704207935331634934234728920650874880",
    266,
    "9.9999999999999997343822485416022730587751856112282375059371259198714596402444465669404440"
    "447686868901514916762299630919016582458402314694101834973930913546324812261345931410707403"
    "9291811569329219648848907543004197890512187794469896370420793533163493423472892065087488e+"
    "267"
  },
  {
    1e269,
    0,
    "100000000000000004675381888545612798918960543133041028684136487274401643939455589461036825"
    "818030333693907688813404495028932616818466243033147431327741697981638738927986463793558699"
    "752023835231102266007829372867138519293326106230343475263802678137754874196788463928344576",
    269,
    "1.0000000000000000467538188854561279891896054313304102868413648727440164393945558946103682"
    "581803033369390768881340449502893261681846624303314743132774169798163873892798646379355869"
    "975202383523110226600782937286713851929332610623034347526380267813775487419678846392834457"
    "6e+269"
  },
  {
    1e270,
    0,
    "100000000000000004675381888545612798918960543133041028684136487274401643939455589461036825"
    "818030333693907688813404495028932616818466243033147431327741697981638738927986463793558699"
    "752023835231102266007829372867138519293326106230343475263802678137754874196788463928344576"
    "0",
    269,
    "1.0000000000000000467538188854561279891896054313304102868413648727440164393945558946103682"
    "581803033369390768881340449502893261681846624303314743132774169798163873892798646379355869"
    "975202383523110226600782937286713851929332610623034347526380267813775487419678846392834457"
    "6e+270"
  },
  {
    1e271,
    0,
    "999999999999999952909858525397375114550134237464699520444369953375222230920813510077473725"
    "439906987596449405879902689682400928375844147591690679948638939044369127946865823435090410"
    "987852070094314805704679411017385445834287279476505623399968223663557934294294144312619827"
    "2",
    270,
    "9.9999999999999995290985852539737511455013423746469952044436995337522223092081351007747372"
    "543990698759644940587990268968240092837584414759169067994863893904436912794686582343509041"
    "098785207009431480570467941101738544583428727947650562339996822366355793429429414431261982"
    "72e+270"
  },
  {
    1e272,
    0,
    "100000000000000006552261095746787856411749967010355244012076385661777528108930437151694716"
    "472838260680760238458487340241071121614642608687943103994317258797079104154646440083568631"
    "482671560875436423095301659220218514235305581886882057848563849292034690350260273827761094"
    "656",
    272,
    "1.0000000000000000655226109574678785641174996701035524401207638566177752810893043715169471"
    "647283826068076023845848734024107112161464260868794310399431725879707910415464644008356863"
    "148267156087543642309530165922021851423530558188688205784856384929203469035026027382776109"
    "4656e+272"
  },
  {
    1e273,
    0,
    "999999999999999945402341696592674884578976541955442659132610359825718694242914119314842162"
    "820675279649039207299571308833846909191138684972507989282336695782607667040225918275050684"
    "065261167516978177354790265605065466066369376850351293060923539046438669680406904714953752"
    "576",
    272,
    "9.9999999999999994540234169659267488457897654195544265913261035982571869424291411931484216"
    "282067527964903920729957130883384690919113868497250798928233669578260766704022591827505068"
    "406526116751697817735479026560506546606636937685035129306092353904643866968040690471495375"
    "2576e+272"
  },
  {
    1e274,
    0,
    "999999999999999921378287844417634148671271916325820702934979660467307376873636068874421162"
    "439133814217326571842510890118474047800081204591123379150169517344970992138978221762923557"
    "912970279269500966635145000285641530809032088446657435975980548271657022915967738002422313"
    "7792",
    273,
    "9.9999999999999992137828784441763414867127191632582070293497966046730737687363606887442116"
    "243913381421732657184251089011847404780008120459112337915016951734497099213897822176292355"
    "791297027926950096663514500028564153080903208844665743597598054827165702291596773800242231"
    "37792e+273"
  },
  {
    1e275,
    0,
    "999999999999999959816774007897699326123599317333215832851188779440765484664480949579094763"
    "049600158908066788573807560063070626025773173201338755361637002845189671980974536182326959"
    "756635700465464503786577424796719827220771749892567607311889333511307657739070404742472615"
    "85408",
    274,
    "9.9999999999999995981677400789769932612359931733321583285118877944076548466448094957909476"
    "304960015890806678857380756006307062602577317320133875536163700284518967198097453618232695"
    "975663570046546450378657742479671982722077174989256760731188933351130765773907040474247261"
    "585408e+274"
  },
  {
    1e276,
    0,
    "100000000000000005206914080024985575200918507975096414465009066497706494336250866327031140"
    "451471938616584330872891956793010241376743389786585565826915896804571450360176569078889512"
    "418143271133577699295001524362330773860894693736275201851807041808646918131451680491859334"
    "0833792",
    276,
    "1.0000000000000000520691408002498557520091850797509641446500906649770649433625086632703114"
    "045147193861658433087289195679301024137674338978658556582691589680457145036017656907888951"
    "241814327113357769929500152436233077386089469373627520185180704180864691813145168049185933"
    "40833792e+276"
  },
  {
    1e277,
    0,
    "100000000000000000286787851099537232487020600646149837835734299269103856539022721596832919"
    "573332246496169583131285983040101879363854817804477997671848058660543459340401040833205876"
    "982154097220494366539618174024912751920192017071198699920810717297971636874094539149132895"
    "41779456",
    277,
    "1.0000000000000000028678785109953723248702060064614983783573429926910385653902272159683291"
    "957333224649616958313128598304010187936385481780447799767184805866054345934040104083320587"
    "698215409722049436653961817402491275192019201707119869992081071729797163687409453914913289"
    "541779456e+277"
  },
  {
    1e278,
    0,
    "999999999999999963506868679591785583159022747829925765323144854862217463012402058126743428"
    "708204927998377849380012040377751897535439602187919431477937881453210665245806182366589686"
    "333627580900277003353114937549783343676298757391374983760136576894314118682088260749517444"
    "85326848",
    277,
    "9.9999999999999996350686867959178558315902274782992576532314485486221746301240205812674342"
    "870820492799837784938001204037775189753543960218791943147793788145321066524580618236658968"
    "633362758090027700335311493754978334367629875739137498376013657689431411868208826074951744"
    "485326848e+277"
  },
  {
    1e279,
    0,
    "100000000000000005797329227496039376326586256854570003660522038565138810871918243694654926"
    "956848701671034100601884673643359244818290018424438474005524037381854809282549632468371548"
    "670461972003147699225647526402820936493779014936084382083526600749927951882334537452986506"
    "7232493568",
    279,
    "1.0000000000000000579732922749603937632658625685457000366052203856513881087191824369465492"
    "695684870167103410060188467364335924481829001842443847400552403738185480928254963246837154"
    "867046197200314769922564752640282093649377901493608438208352660074992795188233453745298650"
    "67232493568e+279"
  },
  {
    1e280,
    0,
    "100000000000000003278224598286209824857070528302149356426333357744094260319737433592793437"
    "867241179305381749758182415081870163467691069569599399110129304252112477880424562006581527"
    "327235514959649032854891251030062909260139244483565213094856482600462207878567681085510570"
    "12647002112",
    280,
    "1.0000000000000000327822459828620982485707052830214935642633335774409426031973743359279343"
    "786724117930538174975818241508187016346769106956959939911012930425211247788042456200658152"
    "732723551495964903285489125103006290926013924448356521309485648260046220787856768108551057"
    "012647002112e+280"
  },
  {
    1e281,
    0,
    "100000000000000003278224598286209824857070528302149356426333357744094260319737433592793437"
    "867241179305381749758182415081870163467691069569599399110129304252112477880424562006581527"
    "327235514959649032854891251030062909260139244483565213094856482600462207878567681085510570"
    "126470021120",
    280,
    "1.0000000000000000327822459828620982485707052830214935642633335774409426031973743359279343"
    "786724117930538174975818241508187016346769106956959939911012930425211247788042456200658152"
    "732723551495964903285489125103006290926013924448356521309485648260046220787856768108551057"
    "012647002112e+281"
  },
  {
    1e282,
    0,
    "100000000000000003278224598286209824857070528302149356426333357744094260319737433592793437"
    "867241179305381749758182415081870163467691069569599399110129304252112477880424562006581527"
    "327235514959649032854891251030062909260139244483565213094856482600462207878567681085510570"
    "1264700211200",
    280,
    "1.0000000000000000327822459828620982485707052830214935642633335774409426031973743359279343"
    "786724117930538174975818241508187016346769106956959939911012930425211247788042456200658152"
    "732723551495964903285489125103006290926013924448356521309485648260046220787856768108551057"
    "012647002112e+282"
  },
  {
    1e283,
    0,
    "999999999999999955395351773536134427427182101891131281229057302618454010234379849598749433"
    "839668705980977279663290767809757055586510986875337610314766840775440358130963455479625817"
    "608438389220211297639279730849502495983978696534263259616618796453034422989958983246244929"
    "0116390191104",
    282,
    "9.9999999999999995539535177353613442742718210189113128122905730261845401023437984959874943"
    "383966870598097727966329076780975705558651098687533761031476684077544035813096345547962581"
    "760843838922021129763927973084950249598397869653426325961661879645303442298995898324624492"
    "90116390191104e+282"
  },
  {
    1e284,
    0,
    "100000000000000007921438250845767654125681919169971093408389934233443575897517102772544534"
    "557205764529752162833294418062406838213115052098838781957320876356853543120821491881752894"
    "667070520582225774709469217797130505057184069381648545374773244373557467226310750742042216"
    "461653692645376",
    284,
    "1.0000000000000000792143825084576765412568191916997109340838993423344357589751710277254453"
    "455720576452975216283329441806240683821311505209883878195732087635685354312082149188175289"
    "466707052058222577470946921779713050505718406938164854537477324437355746722631075074204221"
    "6461653692645376e+284"
  },
  {
    1e285,
    0,
    "999999999999999980159157920520442850193109519852847211800025710561650359982538085224088616"
    "186146493844286149397221450372619320895438893697947652166455225334059372746413748147206443"
    "420891752540620587530362220273863006901551095990707698442841525909542472844588688081080376"
    "132618600579072",
    284,
    "9.9999999999999998015915792052044285019310951985284721180002571056165035998253808522408861"
    "618614649384428614939722145037261932089543889369794765216645522533405937274641374814720644"
    "342089175254062058753036222027386300690155109599070769844284152590954247284458868808108037"
    "6132618600579072e+284"
  },
  {
    1e286,
    0,
    "100000000000000003298861103408696748542708801150450786368475831417380257277860898789147887"
    "185863244128601173816294023984005882022115176158618240811672377905911327059270770583804511"
    "182079226095749373929800486437916543019237221483112250127211668208342631253446539172872932"
    "99907083743789056",
    286,
    "1.0000000000000000329886110340869674854270880115045078636847583141738025727786089878914788"
    "718586324412860117381629402398400588202211517615861824081167237790591132705927077058380451"
    "118207922609574937392980048643791654301923722148311225012721166820834263125344653917287293"
    "299907083743789056e+286"
  },
  {
    1e287,
    0,
    "100000000000000007525217352494018719361427080482583638519254439706352434301546571002539107"
    "639662119923939220917551527141401041968172205589677021287693862203915638886974287199071604"
    "654071266769099226071211897966340736882502910990345434353553680702253338428636675464684849"
    "307718019341877248",
    287,
    "1.0000000000000000752521735249401871936142708048258363851925443970635243430154657100253910"
    "763966211992393922091755152714140104196817220558967702128769386220391563888697428719907160"
    "465407126676909922607121189796634073688250291099034543435355368070225333842863667546468484"
    "9307718019341877248e+287"
  },
  {
    1e288,
    0,
    "100000000000000000763047353957503566051477833551171075078008666443996951063649495461113154"
    "913583918651398345555539522089568786054480958499982972526059487327108739962648660614644255"
    "098884001691739462644953639520862026701277807778772339591406460711996206948332457397785783"
    "2138825282954985472",
    288,
    "1.0000000000000000076304735395750356605147783355117107507800866644399695106364949546111315"
    "491358391865139834555553952208956878605448095849998297252605948732710873996264866061464425"
    "509888400169173946264495363952086202670127780777877233959140646071199620694833245739778578"
    "32138825282954985472e+288"
  },
  {
    1e289,
    0,
    "100000000000000006172783352786715688699437231096301125831005285053881337653967155894253917"
    "094446479669431045845149126131034590785433956171738211535366987228554259102109161882186134"
    "743033813753627273385960246277244994846257890348030815401124236704201912132575831851305036"
    "08895092113260150784",
    289,
    "1.0000000000000000617278335278671568869943723109630112583100528505388133765396715589425391"
    "709444647966943104584514912613103459078543395617173821153536698722855425910210916188218613"
    "474303381375362727338596024627724499484625789034803081540112423670420191213257583185130503"
    "608895092113260150784e+289"
  },
  {
    1e290,
    0,
    "100000000000000006172783352786715688699437231096301125831005285053881337653967155894253917"
    "094446479669431045845149126131034590785433956171738211535366987228554259102109161882186134"
    "743033813753627273385960246277244994846257890348030815401124236704201912132575831851305036"
    "088950921132601507840",
    289,
    "1.0000000000000000617278335278671568869943723109630112583100528505388133765396715589425391"
    "709444647966943104584514912613103459078543395617173821153536698722855425910210916188218613"
    "474303381375362727338596024627724499484625789034803081540112423670420191213257583185130503"
    "608895092113260150784e+290"
  },
  {
    1e291,
    0,
    "999999999999999957860902350346284132153551878096514283852517773229033154005572478626236537"
    "071903625148082612890986863714202457020042006419681526374965874177788623543449994485057258"
    "262661745948026767632275613049896960078961318150545418464661067991669581788285529005480705"
    "688196068853638234112",
    290,
    "9.9999999999999995786090235034628413215355187809651428385251777322903315400557247862623653"
    "707190362514808261289098686371420245702004200641968152637496587417778862354344999448505725"
    "826266174594802676763227561304989696007896131815054541846466106799166958178828552900548070"
    "5688196068853638234112e+290"
  },
  {
    1e292,
    0,
    "100000000000000001325659897835741626806865610895864600356320314779424927269042532146159794"
    "180393624997273746385658920909881229746500070257845517383027467316859073953152552746468610"
    "581875582146175794962018326623525855388355736365975221075617109415185600287493768340951785"
    "51288964115055725510656",
    292,
    "1.0000000000000000132565989783574162680686561089586460035632031477942492726904253214615979"
    "418039362499727374638565892090988122974650007025784551738302746731685907395315255274646861"
    "058187558214617579496201832662352585538835573636597522107561710941518560028749376834095178"
    "551288964115055725510656e+292"
  },
  {
    1e293,
    0,
    "999999999999999924623484373539604850604489339579235252026106548489903482794660772925019694"
    "232684050253289702311625456483436552753066788724417337901780594783307353950604674697279949"
    "729005300639788058439531021138680003796203690845021343089755052295557729136294236363058416"
    "02377586326247764393984",
    292,
    "9.9999999999999992462348437353960485060448933957923525202610654848990348279466077292501969"
    "423268405025328970231162545648343655275306678872441733790178059478330735395060467469727994"
    "972900530063978805843953102113868000379620369084502134308975505229555772913629423636305841"
    "602377586326247764393984e+292"
  },
  {
    1e294,
    0,
    "100000000000000006643646774124810311854715617058629245448546110737685674662788405058354489"
    "034668756980440612078356746066803774429216105089087787538737112019976077088007803912512979"
    "947260613395493988432857461329320568393596956734859073135602071926563496711812375163739351"
    "8591968740451429495341056",
    294,
    "1.0000000000000000664364677412481031185471561705862924544854611073768567466278840505835448"
    "903466875698044061207835674606680377442921610508908778753873711201997607708800780391251297"
    "994726061339549398843285746132932056839359695673485907313560207192656349671181237516373935"
    "18591968740451429495341056e+294"
  },
  {
    1e295,
    0,
    "999999999999999981348677720623004157781556071982058133009848372044684788327950083988429772"
    "678285458073736269700402258157277029368704493591001552896016804949888720722394020468419889"
    "626445633965848788795148458000490275852110041446449098396261319083588624329026042472792457"
    "0510530141380583845003264",
    294,
    "9.9999999999999998134867772062300415778155607198205813300984837204468478832795008398842977"
    "267828545807373626970040225815727702936870449359100155289601680494988872072239402046841988"
    "962644563396584878879514845800049027585211004144644909839626131908358862432902604247279245"
    "70510530141380583845003264e+294"
  },
  {
    1e296,
    0,
    "999999999999999981348677720623004157781556071982058133009848372044684788327950083988429772"
    "678285458073736269700402258157277029368704493591001552896016804949888720722394020468419889"
    "626445633965848788795148458000490275852110041446449098396261319083588624329026042472792457"
    "05105301413805838450032640",
    294,
    "9.9999999999999998134867772062300415778155607198205813300984837204468478832795008398842977"
    "267828545807373626970040225815727702936870449359100155289601680494988872072239402046841988"
    "962644563396584878879514845800049027585211004144644909839626131908358862432902604247279245"
    "70510530141380583845003264e+295"
  },
  {
    1e297,
    0,
    "100000000000000001765280146275637971437487878071986477683944313911974482386925524306901222"
    "288347035907882207282921941122853493440271262470561545049232797945650079545633920176194945"
    "116080744729452765622274361759204884996789010583136286179242532982792839725237439838302224"
    "3308510390698430058459037696",
    297,
    "1.0000000000000000176528014627563797143748787807198647768394431391197448238692552430690122"
    "228834703590788220728292194112285349344027126247056154504923279794565007954563392017619494"
    "511608074472945276562227436175920488499678901058313628617924253298279283972523743983830222"
    "43308510390698430058459037696e+297"
  },
  {
    1e298,
    0,
    "999999999999999959566203475342978823825562446739374146712091511799648767003166988540080302"
    "555174517470684787823111966314522286348299614922233214338230100245921475882026911692302152"
    "705828545968641468338591362245555131382642002815500840358562912636984760575017028926654585"
    "2965785882018353801250996224",
    297,
    "9.9999999999999995956620347534297882382556244673937414671209151179964876700316698854008030"
    "255517451747068478782311196631452228634829961492223321433823010024592147588202691169230215"
    "270582854596864146833859136224555513138264200281550084035856291263698476057501702892665458"
    "52965785882018353801250996224e+297"
  },
  {
    1e299,
    0,
    "100000000000000005250476025520442024870446858110815915491585411551180245798890819578637137"
    "508044786404370444383288387817694252323536043057564479218478670698284838720092657580373783"
    "023379478809005936895323497079994508111903896764088007465274278014249457925878882005684283"
    "811566947219638686545940054016",
    299,
    "1.0000000000000000525047602552044202487044685811081591549158541155118024579889081957863713"
    "750804478640437044438328838781769425232353604305756447921847867069828483872009265758037378"
    "302337947880900593689532349707999450811190389676408800746527427801424945792587888200568428"
    "3811566947219638686545940054016e+299"
  },
  {
    1e300,
    0,
    "100000000000000005250476025520442024870446858110815915491585411551180245798890819578637137"
    "508044786404370444383288387817694252323536043057564479218478670698284838720092657580373783"
    "023379478809005936895323497079994508111903896764088007465274278014249457925878882005684283"
    "8115669472196386865459400540160",
    299,
    "1.0000000000000000525047602552044202487044685811081591549158541155118024579889081957863713"
    "750804478640437044438328838781769425232353604305756447921847867069828483872009265758037378"
    "302337947880900593689532349707999450811190389676408800746527427801424945792587888200568428"
    "3811566947219638686545940054016e+300"
  },
  {
    1e301,
    0,
    "100000000000000005250476025520442024870446858110815915491585411551180245798890819578637137"
    "508044786404370444383288387817694252323536043057564479218478670698284838720092657580373783"
    "023379478809005936895323497079994508111903896764088007465274278014249457925878882005684283"
    "81156694721963868654594005401600",
    299,
    "1.0000000000000000525047602552044202487044685811081591549158541155118024579889081957863713"
    "750804478640437044438328838781769425232353604305756447921847867069828483872009265758037378"
    "302337947880900593689532349707999450811190389676408800746527427801424945792587888200568428"
    "3811566947219638686545940054016e+301"
  },
  {
    1e302,
    0,
    "100000000000000007629703079084894925347346855150656811701601734206211380288125794484142188"
    "964691784076639747577138548761372210387844799938291815611350519830750167649856488981626536"
    "368095414607314235151058373458986890825155659063617715863205282622390509284183439858617103"
    "083735673849899204570498157510656",
    302,
    "1.0000000000000000762970307908489492534734685515065681170160173420621138028812579448414218"
    "896469178407663974757713854876137221038784479993829181561135051983075016764985648898162653"
    "636809541460731423515105837345898689082515565906361771586320528262239050928418343985861710"
    "3083735673849899204570498157510656e+302"
  },
  {
    1e303,
    0,
    "100000000000000000016176507678645643821266864623165943829549501710111749922573874786526024"
    "303421391525377977356818033741602744582056777919964339154160602606861115074612228497617725"
    "665004420052727680732706769046211266142750019705122648989826067876339144937608854729232081"
    "4127957486330655468919122263277568",
    303,
    "1.0000000000000000001617650767864564382126686462316594382954950171011174992257387478652602"
    "430342139152537797735681803374160274458205677791996433915416060260686111507461222849761772"
    "566500442005272768073270676904621126614275001970512264898982606787633914493760885472923208"
    "14127957486330655468919122263277568e+303"
  },
  {
    1e304,
    0,
    "999999999999999939253552505536462186004028722011732495319077157132320456301323390284330925"
    "744050774843685611805616217257871719374263603053023579884086688277498730144168201104106771"
    "025316244090584371980254855159907663968255082183265954911226960794980534603491866257240640"
    "7604380845959862074904348138143744",
    303,
    "9.9999999999999993925355250553646218600402872201173249531907715713232045630132339028433092"
    "574405077484368561180561621725787171937426360305302357988408668827749873014416820110410677"
    "102531624409058437198025485515990766396825508218326595491122696079498053460349186625724064"
    "07604380845959862074904348138143744e+303"
  },
  {
    1e305,
    0,
    "999999999999999939253552505536462186004028722011732495319077157132320456301323390284330925"
    "744050774843685611805616217257871719374263603053023579884086688277498730144168201104106771"
    "025316244090584371980254855159907663968255082183265954911226960794980534603491866257240640"
    "76043808459598620749043481381437440",
    303,
    "9.9999999999999993925355250553646218600402872201173249531907715713232045630132339028433092"
    "574405077484368561180561621725787171937426360305302357988408668827749873014416820110410677"
    "102531624409058437198025485515990766396825508218326595491122696079498053460349186625724064"
    "07604380845959862074904348138143744e+304"
  },
  {
    1e306,
    0,
    "100000000000000001721606459673645482883108782501323898232889201789238067124457504798792045"
    "187545959456860613886169829106031104922553294852069693880571144065012262851466942846035699"
    "262496802832955068922417528434673006071608882921425543969463011979454650551241561798214326"
    "2670862918816362862119154749127262208",
    306,
    "1.0000000000000000172160645967364548288310878250132389823288920178923806712445750479879204"
    "518754595945686061388616982910603110492255329485206969388057114406501226285146694284603569"
    "926249680283295506892241752843467300607160888292142554396946301197945465055124156179821432"
    "62670862918816362862119154749127262208e+306"
  },
  {
    1e307,
    0,
    "999999999999999986031059760256457771700264183812636387524966073588356585267274384906484641"
    "422896066678637928039265461539335317285025210333627595237061539701073069166468937517856903"
    "985107314633964162326607112672001102016955330401859645781268856194720117148846117292182213"
    "9066929851282122002676667750021070848",
    306,
    "9.9999999999999998603105976025645777170026418381263638752496607358835658526727438490648464"
    "142289606667863792803926546153933531728502521033362759523706153970107306916646893751785690"
    "398510731463396416232660711267200110201695533040185964578126885619472011714884611729218221"
    "39066929851282122002676667750021070848e+306"
  },
  {
    1e308,
    0,
    "100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328"
    "978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697"
    "489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105"
    "291802893207873272974885715430223118336",
    308,
    "1.0000000000000000109790636294404554174049230967731184633681068290315758540491149153716332"
    "897849468889906124966972117251561159028374314008832830700919814604603127166450293302718569"
    "748969958855904333838446616500117842689762621294517762809119578670745812278397017178441510"
    "5291802893207873272974885715430223118336e+308"
  },
};

// These values test every binary exponent (which would be more obvious with hexfloats).
// The mantissas were randomly generated.
const test_case all_binary_exponents[] = {
  {
    8.667315560151837e-309,
    1074,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000086673155601518373204745615934992065170006880827787"
    "289636461039456117181014435225731420739682086431846196084055971776197678140145261761209000"
    "307554293026077781251866965924172972401068174664168662349984846821559496820210791534029454"
    "719229889718739090124407402205278009112442448229109027730844488969500029631608590626296609"
    "882885374947604962348974323036144310657614326754120115556509383797380330234783520222245524"
    "583876671362471849408043185861726089996302842330553125394555226033512393900059324065691066"
    "929863218174741264971497401985883562857614025818809269477830515183406516097939800531603134"
    "250276197118962555458534787484939232126232153092624291889535798750202922731324119623206905"
    "40877906325907159510518339181289381166448950420677643791123045957647264003753662109375",
    765,
    "8.6673155601518373204745615934992065170006880827787289636461039456117181014435225731420739"
    "682086431846196084055971776197678140145261761209000307554293026077781251866965924172972401"
    "068174664168662349984846821559496820210791534029454719229889718739090124407402205278009112"
    "442448229109027730844488969500029631608590626296609882885374947604962348974323036144310657"
    "614326754120115556509383797380330234783520222245524583876671362471849408043185861726089996"
    "302842330553125394555226033512393900059324065691066929863218174741264971497401985883562857"
    "614025818809269477830515183406516097939800531603134250276197118962555458534787484939232126"
    "232153092624291889535798750202922731324119623206905408779063259071595105183391812893811664"
    "48950420677643791123045957647264003753662109375e-309"
  },
  {
    3.402496288854889e-308,
    1072,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000340249628885488918707803802237219634159360572672157"
    "684890228809310363407096641170467659879771805264411927667980233377849338161398396361988635"
    "343561859686478002041658296279167269633549887180398681467085048366045915667637095150966312"
    "151585104419778664778568818969883177514928542970375740015926895932459510900145974010899870"
    "643490118898178199420014167358923990328782986564433565017977882465941632972037475759286148"
    "645435617069627150190560661004463760153276800654567748203691908193307868018274961903878508"
    "519765463087823925880361092267988073184612955956100615232697812832213574317600267780811339"
    "722336120022236052218268188972350774933242797322404573240208707312500506083305050530743859"
    "909422095574730572935613200283098897446182155845928463122618268243968486785888671875",
    764,
    "3.4024962888548891870780380223721963415936057267215768489022880931036340709664117046765987"
    "977180526441192766798023337784933816139839636198863534356185968647800204165829627916726963"
    "354988718039868146708504836604591566763709515096631215158510441977866477856881896988317751"
    "492854297037574001592689593245951090014597401089987064349011889817819942001416735892399032"
    "878298656443356501797788246594163297203747575928614864543561706962715019056066100446376015"
    "327680065456774820369190819330786801827496190387850851976546308782392588036109226798807318"
    "461295595610061523269781283221357431760026778081133972233612002223605221826818897235077493"
    "324279732240457324020870731250050608330505053074385990942209557473057293561320028309889744"
    "6182155845928463122618268243968486785888671875e-308"
  },
  {
    5.674095874064163e-308,
    1068,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000567409587406416315113843845367765611463402272605773"
    "579309805044238876264896175716378374414880313837549557854086594144651757106826827082053224"
    "819137614951757164822186954713633389739294124192094600295957101411513892229222775656755797"
    "001569791833154964609151380870932502682826741267747086941264281692372695316993299328269619"
    "707002817860044551711084995833922394577051992453091701226019440181486036346841321732890759"
    "392314160217956773831481083699810908312929539145137356475139290129992626736771448814573026"
    "906644585538148884867233811394239534391479312597313753909725290390851223382408696877615721"
    "229933837546642765730256661271873445479155973842968298711175582031872758195259747569288096"
    "49794138442664770396059164903717939140888326665645990942721255123615264892578125",
    760,
    "5.6740958740641631511384384536776561146340227260577357930980504423887626489617571637837441"
    "488031383754955785408659414465175710682682708205322481913761495175716482218695471363338973"
    "929412419209460029595710141151389222922277565675579700156979183315496460915138087093250268"
    "282674126774708694126428169237269531699329932826961970700281786004455171108499583392239457"
    "705199245309170122601944018148603634684132173289075939231416021795677383148108369981090831"
    "292953914513735647513929012999262673677144881457302690664458553814888486723381139423953439"
    "147931259731375390972529039085122338240869687761572122993383754664276573025666127187344547"
    "915597384296829871117558203187275819525974756928809649794138442664770396059164903717939140"
    "888326665645990942721255123615264892578125e-308"
  },
  {
    1.5540526173622352e-307,
    1072,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000001554052617362235237599344524566268329346499410781820"
    "240377373211399724292118817533439328423661067387776003211400451369308809328471980946673335"
    "155886367588496521935894886476286230147591992600623876485166587589022024414313323949909142"
    "159529537456605070198347779192598243095403947592025125119457352938379814319820278343589774"
    "730296120303664088965600921426179216076245193704452662458429835783277758594438863220080552"
    "115251534233085611706230850346463513607451896716425893805860802544849432794739623937039499"
    "266977883786031463582126957812370158648390166491038532530966495890940486561351377873613480"
    "311323741789526145720019072115667475187757369075258822710974186182997578768175923453881771"
    "250830539545867001930118621195145977916950326414013971998429042287170886993408203125",
    765,
    "1.5540526173622352375993445245662683293464994107818202403773732113997242921188175334393284"
    "236610673877760032114004513693088093284719809466733351558863675884965219358948864762862301"
    "475919926006238764851665875890220244143133239499091421595295374566050701983477791925982430"
    "954039475920251251194573529383798143198202783435897747302961203036640889656009214261792160"
    "762451937044526624584298357832777585944388632200805521152515342330856117062308503464635136"
    "074518967164258938058608025448494327947396239370394992669778837860314635821269578123701586"
    "483901664910385325309664958909404865613513778736134803113237417895261457200190721156674751"
    "877573690752588227109741861829975787681759234538817712508305395458670019301186211951459779"
    "16950326414013971998429042287170886993408203125e-307"
  },
  {
    1.990320005135143e-307,
    1071,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000001990320005135142880561598924319312704477072269270245"
    "936137839955953465230545195415127755767083901447533947828678376962884082735327248251273418"
    "654936650677742426069650358358476393365497312325560741512426890538065472696077887804553644"
    "884150043878406627507745699995507099147377175105877153481526330603572499358626260756240816"
    "174503154885783858040748854022822093131458054271092923292182143582852711297479202477529680"
    "286154546026438265660119066696435931082590283442526670222949478974719486467415453971990534"
    "454470599633245287624684280561552964343633309851882307590384491177630806444328543647075668"
    "536514853052975155234251635207407297067719395388405831420311524001253621714021938613841338"
    "24689620350520245119381221877393126954932690841904019407593295909464359283447265625",
    764,
    "1.9903200051351428805615989243193127044770722692702459361378399559534652305451954151277557"
    "670839014475339478286783769628840827353272482512734186549366506777424260696503583584763933"
    "654973123255607415124268905380654726960778878045536448841500438784066275077456999955070991"
    "473771751058771534815263306035724993586262607562408161745031548857838580407488540228220931"
    "314580542710929232921821435828527112974792024775296802861545460264382656601190666964359310"
    "825902834425266702229494789747194864674154539719905344544705996332452876246842805615529643"
    "436333098518823075903844911776308064443285436470756685365148530529751552342516352074072970"
    "677193953884058314203115240012536217140219386138413382468962035052024511938122187739312695"
    "4932690841904019407593295909464359283447265625e-307"
  },
  {
    5.107418539046847e-307,
    1070,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000005107418539046846927047373682830132718075006256443666"
    "199351873286782041441258279676361749462207484160512079462800015135124764152600937802744585"
    "502617458841717705897103693609800077882088804203101790655437325897778001026322540349851343"
    "686142888941809812697130982668148593939074359425039990092398040350897342987033684062247889"
    "191004578703872406975692839527795367059563095996889887433505152749668146313379417615714580"
    "759187210921661443825061507972674365098970496562064290123980918945039727952980165470646623"
    "479202855278722883908955176044191810941642460040564082188712438936906291867000270523960550"
    "332664002210898302673308442754015967789648657719964005852935700696823519264465123996032196"
    "8186374102417320609944271040560945042782563241257065556055749766528606414794921875",
    763,
    "5.1074185390468469270473736828301327180750062564436661993518732867820414412582796763617494"
    "622074841605120794628000151351247641526009378027445855026174588417177058971036936098000778"
    "820888042031017906554373258977780010263225403498513436861428889418098126971309826681485939"
    "390743594250399900923980403508973429870336840622478891910045787038724069756928395277953670"
    "595630959968898874335051527496681463133794176157145807591872109216614438250615079726743650"
    "989704965620642901239809189450397279529801654706466234792028552787228839089551760441918109"
    "416424600405640821887124389369062918670002705239605503326640022108983026733084427540159677"
    "896486577199640058529357006968235192644651239960321968186374102417320609944271040560945042"
    "782563241257065556055749766528606414794921875e-307"
  },
  {
    9.580341507134765e-307,
    1068,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000009580341507134764786508271836926643404057427002697780"
    "971266720840479203611958030287584788324659120693742922007657201247570143253344392092094340"
    "465398937193159387800074315496349939162152176053622628582694074268423342012580732514751221"
    "293610595763003495795061946838666946253209134456556726989193259291618902375718355472162404"
    "243905684503604058791100485208980042190187445440799433710816373805294721006156102426371505"
    "496796959141102703635606828289861024299682102873191783632690913901906663418356736427846861"
    "198091606413597986305146589529621994975436450348729144771801748911075356977674248196373371"
    "459254492612305520113008095746852968491757589431223267360016445450314712551946974418885474"
    "26893677131265836011722089207730797071581319013233724035671912133693695068359375",
    761,
    "9.5803415071347647865082718369266434040574270026977809712667208404792036119580302875847883"
    "246591206937429220076572012475701432533443920920943404653989371931593878000743154963499391"
    "621521760536226285826940742684233420125807325147512212936105957630034957950619468386669462"
    "532091344565567269891932592916189023757183554721624042439056845036040587911004852089800421"
    "901874454407994337108163738052947210061561024263715054967969591411027036356068282898610242"
    "996821028731917836326909139019066634183567364278468611980916064135979863051465895296219949"
    "754364503487291447718017489110753569776742481963733714592544926123055201130080957468529684"
    "917575894312232673600164454503147125519469744188854742689367713126583601172208920773079707"
    "1581319013233724035671912133693695068359375e-307"
  },
  {
    2.7686246461533843e-306,
    1067,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000027686246461533842708986290401534510677245449052480438"
    "452772935563477911887903937433332521175778793903095384041223955723192328355566029243799092"
    "760177657489398060155725073732975431091566182547204021375695183029148269715561698427005215"
    "406766097411258573897066330232423648553484787712302958508292914387235309239329235907024151"
    "611338319751130631733719608246390666274473865694983392385750305737090058291815147877218349"
    "104719543513706308913668377494760282493669343297776248265117141371394177596003869591975610"
    "272425141960038017763507149362458614439882589567047333185916661995863312395185290317290485"
    "203772131643631923277169601170745162275359509369132897727641328747183370449707056957130139"
    "3853010023568263232603252582470613227076661200243279381538741290569305419921875",
    761,
    "2.7686246461533842708986290401534510677245449052480438452772935563477911887903937433332521"
    "175778793903095384041223955723192328355566029243799092760177657489398060155725073732975431"
    "091566182547204021375695183029148269715561698427005215406766097411258573897066330232423648"
    "553484787712302958508292914387235309239329235907024151611338319751130631733719608246390666"
    "274473865694983392385750305737090058291815147877218349104719543513706308913668377494760282"
    "493669343297776248265117141371394177596003869591975610272425141960038017763507149362458614"
    "439882589567047333185916661995863312395185290317290485203772131643631923277169601170745162"
    "275359509369132897727641328747183370449707056957130139385301002356826323260325258247061322"
    "7076661200243279381538741290569305419921875e-306"
  },
  {
    4.123539674247709e-306,
    1066,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000041235396742477088973208679969440262899254505285796142"
    "223814175822702981218018251380021070120502969772416262035601116862165953295290600674843653"
    "142629610874437148521841848923905230805636075377711443081016998939526959461400771643206600"
    "320959274437690468693402265462651475007058394578286178914389155810293540857408925099086126"
    "202065003470601897807688053820010379771148507245760602767529755342564783534772481563133530"
    "608872155816317027321781075294059535644107131784837791685211177196255873378934891728850369"
    "864191164109645279967863311832414284700383379389103398768464735139083439964508520572018273"
    "546018301229487468604876815644353595698302012244034576878689343471222581935495027065359114"
    "155162611459186679259780987690986908851542569465209453483112156391143798828125",
    760,
    "4.1235396742477088973208679969440262899254505285796142223814175822702981218018251380021070"
    "120502969772416262035601116862165953295290600674843653142629610874437148521841848923905230"
    "805636075377711443081016998939526959461400771643206600320959274437690468693402265462651475"
    "007058394578286178914389155810293540857408925099086126202065003470601897807688053820010379"
    "771148507245760602767529755342564783534772481563133530608872155816317027321781075294059535"
    "644107131784837791685211177196255873378934891728850369864191164109645279967863311832414284"
    "700383379389103398768464735139083439964508520572018273546018301229487468604876815644353595"
    "698302012244034576878689343471222581935495027065359114155162611459186679259780987690986908"
    "851542569465209453483112156391143798828125e-306"
  },
  {
    1.0292523314007669e-305,
    1065,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000102925233140076692326385741229598904670825352808317734"
    "627817052485596428748937263427579368777377894657327312479354581078416852016211748037289062"
    "294153412444887077225201260286873016177402040776581672627422105268655741919632420847158211"
    "926010156094848635703736551013937954000440830813259373879599843777175295670976367503830147"
    "413165532296085557798963633773657299527935426265350193083637673227681353297917537837040452"
    "104722871743258998969100598727128250967239986818240346849429624604806528929509814942265446"
    "828862812886192478721784982976392274686964984945034554476060657432739979953382458426601623"
    "596352504949545393950116767250021882228645032139324151043434437466649570437521385841177305"
    "81730514644229269575090721314998575865173879861913519562222063541412353515625",
    760,
    "1.0292523314007669232638574122959890467082535280831773462781705248559642874893726342757936"
    "877737789465732731247935458107841685201621174803728906229415341244488707722520126028687301"
    "617740204077658167262742210526865574191963242084715821192601015609484863570373655101393795"
    "400044083081325937387959984377717529567097636750383014741316553229608555779896363377365729"
    "952793542626535019308363767322768135329791753783704045210472287174325899896910059872712825"
    "096723998681824034684942962460480652892950981494226544682886281288619247872178498297639227"
    "468696498494503455447606065743273997995338245842660162359635250494954539395011676725002188"
    "222864503213932415104343443746664957043752138584117730581730514644229269575090721314998575"
    "865173879861913519562222063541412353515625e-305"
  },
  {
    1.869562966771459e-305,
    1065,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000186956296677145895916750114590693491523617790849939055"
    "674886628194086158815830540057384771933110599023450761887740398618582452621943937695612634"
    "176016332254849337060923845633060080657730525723232050132901795302428774762723203745726636"
    "865115551617454503357810581663654899620416060014611741224579629350785694495714670630402541"
    "133627186270376597920952277811842692283769740789234856500541032298070456110972851620621529"
    "465978994343606420788243552352751894800076450567507229035568346316425077269088928498850264"
    "418366552561657569635100768695240717484275349057666918708640594618603373147397342615169441"
    "685070750007633706104159956396068792531136855537815677721698849612362059270919596759679334"
    "03932478188107975984517174399812395983733193105535974609665572643280029296875",
    760,
    "1.8695629667714589591675011459069349152361779084993905567488662819408615881583054005738477"
    "193311059902345076188774039861858245262194393769561263417601633225484933706092384563306008"
    "065773052572323205013290179530242877476272320374572663686511555161745450335781058166365489"
    "962041606001461174122457962935078569449571467063040254113362718627037659792095227781184269"
    "228376974078923485650054103229807045611097285162062152946597899434360642078824355235275189"
    "480007645056750722903556834631642507726908892849885026441836655256165756963510076869524071"
    "748427534905766691870864059461860337314739734261516944168507075000763370610415995639606879"
    "253113685553781567772169884961236205927091959675967933403932478188107975984517174399812395"
    "983733193105535974609665572643280029296875e-305"
  },
  {
    2.83258275754601e-305,
    1063,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000283258275754601010515391461536595545429499538498991017"
    "040932231183342824512646699604684458146742737342772784980027834494082685766178127629247208"
    "184380986441039452161190285373582999883474215846885769779038605962665767818924814452656629"
    "427958160627630628802053684614164417649788178993281566147411667544202074381710741967860334"
    "032389811562883073602025172957382372382340446938272102534472564616360448591956313016751702"
    "183344954371143800460586565184398113067165957481092210378398497900972951098730107411912334"
    "069394868562044554181578968968188138457198638986666974656524280375457293019436257007627084"
    "926464281428764805946511594819986443861653023407022266493420721660512347699421921315325612"
    "490897898554760724211631142191989730927392798776054405607283115386962890625",
    758,
    "2.8325827575460101051539146153659554542949953849899101704093223118334282451264669960468445"
    "814674273734277278498002783449408268576617812762924720818438098644103945216119028537358299"
    "988347421584688576977903860596266576781892481445265662942795816062763062880205368461416441"
    "764978817899328156614741166754420207438171074196786033403238981156288307360202517295738237"
    "238234044693827210253447256461636044859195631301675170218334495437114380046058656518439811"
    "306716595748109221037839849790097295109873010741191233406939486856204455418157896896818813"
    "845719863898666697465652428037545729301943625700762708492646428142876480594651159481998644"
    "386165302340702226649342072166051234769942192131532561249089789855476072421163114219198973"
    "0927392798776054405607283115386962890625e-305"
  },
  {
    6.143575467580988e-305,
    1061,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000614357546758098837646073896440391226534816436897514942"
    "795483804072267347155569125119595580491351990480913034570147895374919927596970561282274649"
    "091727304992573536378826177765491606914330790712251813627403807741060322895257455018949778"
    "790865949313723431353352031229523443870605177028088216561214763278933728712879023635836842"
    "283965042552837102397922174874664511453596393052110185322246667762253606317075736272542989"
    "625758436326997585642760239301816630363203045431540296803267094601248551668127076407345103"
    "299971987320334913146074020655703982659448361025988551869998742523823644028696175451495318"
    "256465296055176473524028167772526378981266133601228562153297584372631820351071291137909988"
    "9297786747317827073427028343011006839002874357902328483760356903076171875",
    756,
    "6.1435754675809883764607389644039122653481643689751494279548380407226734715556912511959558"
    "049135199048091303457014789537491992759697056128227464909172730499257353637882617776549160"
    "691433079071225181362740380774106032289525745501894977879086594931372343135335203122952344"
    "387060517702808821656121476327893372871287902363583684228396504255283710239792217487466451"
    "145359639305211018532224666776225360631707573627254298962575843632699758564276023930181663"
    "036320304543154029680326709460124855166812707640734510329997198732033491314607402065570398"
    "265944836102598855186999874252382364402869617545149531825646529605517647352402816777252637"
    "898126613360122856215329758437263182035107129113790998892977867473178270734270283430110068"
    "39002874357902328483760356903076171875e-305"
  },
  {
    1.4317575812076783e-304,
    1062,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000001431757581207678327909696250760734478106796867078939935"
    "710607595708573102607339856178147302943011543601422950427382699571971944822634973906669124"
    "996214522364687200673305673317795296527695107187920756052082552297804453598386016350631514"
    "998361352765835015287841808273033244090798155636616618100736973069441228252750043602020368"
    "724304231730496903611600686763522497807086760639060265790948955651670281170776489599001405"
    "555481916829703134521996204956173351597027462353230835934530650721449729975296707605662039"
    "145192784354233319802901383227718965593270032595543344712007425459523297691036565265412105"
    "198702522141620817697556902837198527180570950660691034388286525048452890343118312657240371"
    "18307569089764413738898069327720556642713489736706833355128765106201171875",
    758,
    "1.4317575812076783279096962507607344781067968670789399357106075957085731026073398561781473"
    "029430115436014229504273826995719719448226349739066691249962145223646872006733056733177952"
    "965276951071879207560520825522978044535983860163506315149983613527658350152878418082730332"
    "440907981556366166181007369730694412282527500436020203687243042317304969036116006867635224"
    "978070867606390602657909489556516702811707764895990014055554819168297031345219962049561733"
    "515970274623532308359345306507214497299752967076056620391451927843542333198029013832277189"
    "655932700325955433447120074254595232976910365652654121051987025221416208176975569028371985"
    "271805709506606910343882865250484528903431183126572403711830756908976441373889806932772055"
    "6642713489736706833355128765106201171875e-304"
  },
  {
    1.8736954404407843e-304,
    1061,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000001873695440440784291437168256570360327506576425988595810"
    "379182000643821407577934103660676935011263322031392440132868733732781942957134732893539802"
    "696482540088235117407956259171072576386135085206493071283939781926200705468314546127800658"
    "230827802488328422886555037223958144179314615586216250084158766674880816820412018646350655"
    "714263340405276148348836991757774602316265058289810704660352544391098522007138900938734431"
    "720321349687880063890794970931337153937613121698048349149678306083161410614237531716215527"
    "522398672689857313452931836000009671729651430504379359358929467240989725060006847711032116"
    "813580555184077897939000848084702416809622562481289228530618262874396186671763441010328385"
    "5733573331697661905063280118823205999678549460440990515053272247314453125",
    757,
    "1.8736954404407842914371682565703603275065764259885958103791820006438214075779341036606769"
    "350112633220313924401328687337327819429571347328935398026964825400882351174079562591710725"
    "763861350852064930712839397819262007054683145461278006582308278024883284228865550372239581"
    "441793146155862162500841587666748808168204120186463506557142633404052761483488369917577746"
    "023162650582898107046603525443910985220071389009387344317203213496878800638907949709313371"
    "539376131216980483491496783060831614106142375317162155275223986726898573134529318360000096"
    "717296514305043793593589294672409897250600068477110321168135805551840778979390008480847024"
    "168096225624812892285306182628743961866717634410103283855733573331697661905063280118823205"
    "999678549460440990515053272247314453125e-304"
  },
  {
    3.8678085703767116e-304,
    1060,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000003867808570376711567970796017679738247589622247284651871"
    "120320458105704644504432357204561914952702147103351310377425294926652258552911586065040963"
    "047639543693593550751347496826577633402771938975855549088764275281223404237262544370036021"
    "050362028274509177214367131685009066980529198965345188509320693984189852122869096724030776"
    "266880448498682233610964812352992899821397333236093549925309032199246559964137142197652767"
    "564372214748185530284478963436221847787757895824702105267097245995765157219337219107573831"
    "427387260013587206163815924229271802232530916741889867942756272048490315195062592530929515"
    "848282749067855559474139733793821353946484844518861577118236048680062180232720103374715840"
    "572747272930896699870573302785896341315918789405259303748607635498046875",
    756,
    "3.8678085703767115679707960176797382475896222472846518711203204581057046445044323572045619"
    "149527021471033513103774252949266522585529115860650409630476395436935935507513474968265776"
    "334027719389758555490887642752812234042372625443700360210503620282745091772143671316850090"
    "669805291989653451885093206939841898521228690967240307762668804484986822336109648123529928"
    "998213973332360935499253090321992465599641371421976527675643722147481855302844789634362218"
    "477877578958247021052670972459957651572193372191075738314273872600135872061638159242292718"
    "022325309167418898679427562720484903151950625925309295158482827490678555594741397337938213"
    "539464848445188615771182360486800621802327201033747158405727472729308966998705733027858963"
    "41315918789405259303748607635498046875e-304"
  },
  {
    7.635297514304952e-304,
    1058,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000007635297514304952261443821158790115458810239952267930887"
    "072096693486627813429923330320913248325488240687702430323149784113266837858828284498895767"
    "310851990363570938799161191914775489620184508104363684371235378791715398901706373669321385"
    "471295753589758753964662654457905715120673904659565201490447825557005135025380834617028352"
    "782431197582947539338933804403231554930541964237271712749234912326942139379518950109342558"
    "327795161374824729966907064566186584779903792990625554511619140066469170627538130724031488"
    "543581914672473801419603174369718591349917430634535930397154004116848462943127209818005654"
    "101392783570316361193216938245623206104415409210489113384830637380982062653562435586682508"
    "2456994424052303706479922855952791049816141821793280541896820068359375",
    754,
    "7.6352975143049522614438211587901154588102399522679308870720966934866278134299233303209132"
    "483254882406877024303231497841132668378588282844988957673108519903635709387991611919147754"
    "896201845081043636843712353787917153989017063736693213854712957535897587539646626544579057"
    "151206739046595652014904478255570051350253808346170283527824311975829475393389338044032315"
    "549305419642372717127492349123269421393795189501093425583277951613748247299669070645661865"
    "847799037929906255545116191400664691706275381307240314885435819146724738014196031743697185"
    "913499174306345359303971540041168484629431272098180056541013927835703163611932169382456232"
    "061044154092104891133848306373809820626535624355866825082456994424052303706479922855952791"
    "049816141821793280541896820068359375e-304"
  },
  {
    1.9931833457401843e-303,
    1056,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000019931833457401843380206268213719062063350814301599018679"
    "933261580130337290316898735876549592164313534803701036771696187184346023873676317354073889"
    "463149604434151301408676960767531338973466054442984713722344923736818547328281889907631436"
    "769135497980501110004490481603566627111774459204577291011436150930984708896739289978721590"
    "778489898585846173250830089770352087321756188087231770443888379781496362819885950130996391"
    "895508049596794526555364932541343904335282144008519337590160439055816970092210788775942229"
    "804547504844829843518212715292498986710254817625716748036998815898643899774750851440391076"
    "114615083678303082058198490658570727053842544909780275364598179935132913048405234257751121"
    "41496214071257787596000063921886091833357568248175084590911865234375",
    753,
    "1.9931833457401843380206268213719062063350814301599018679933261580130337290316898735876549"
    "592164313534803701036771696187184346023873676317354073889463149604434151301408676960767531"
    "338973466054442984713722344923736818547328281889907631436769135497980501110004490481603566"
    "627111774459204577291011436150930984708896739289978721590778489898585846173250830089770352"
    "087321756188087231770443888379781496362819885950130996391895508049596794526555364932541343"
    "904335282144008519337590160439055816970092210788775942229804547504844829843518212715292498"
    "986710254817625716748036998815898643899774750851440391076114615083678303082058198490658570"
    "727053842544909780275364598179935132913048405234257751121414962140712577875960000639218860"
    "91833357568248175084590911865234375e-303"
  },
  {
    5.7104439440441904e-303,
    1057,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000057104439440441903852943648577406503088003161021002627368"
    "323935629069966209695533414405887372311247087423593978194334966196995272439997264356514564"
    "356559668034115419136524556384062112763062297964116687569901112750485684621049253917448239"
    "836386573363932764688094380605542392071070703906743190711382422419361043000641426414253161"
    "862928768191727964756739054251002351820210559683067678277426478312879053806872980671807735"
    "494036931597606968803747088298646260019156930298862902090377660482588271076855164676075898"
    "354655432385191518035743314649666577661985721153255879260996383923341453293337109740226159"
    "592310831356510276969070105365016105975780684355323209040588603358187555694328693197982659"
    "012778489710933784939697285300275186870067045674659311771392822265625",
    754,
    "5.7104439440441903852943648577406503088003161021002627368323935629069966209695533414405887"
    "372311247087423593978194334966196995272439997264356514564356559668034115419136524556384062"
    "112763062297964116687569901112750485684621049253917448239836386573363932764688094380605542"
    "392071070703906743190711382422419361043000641426414253161862928768191727964756739054251002"
    "351820210559683067678277426478312879053806872980671807735494036931597606968803747088298646"
    "260019156930298862902090377660482588271076855164676075898354655432385191518035743314649666"
    "577661985721153255879260996383923341453293337109740226159592310831356510276969070105365016"
    "105975780684355323209040588603358187555694328693197982659012778489710933784939697285300275"
    "186870067045674659311771392822265625e-303"
  },
  {
    5.845773229837182e-303,
    1056,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000058457732298371823574823131482226670449533326157358746716"
    "791953135062569212292668611641801097780408195948596588837344740206406272732220721797510164"
    "066073898934074165728050918928927661521204749793435540643180664127884788340231230981632935"
    "135316224370499662844245788803915476702717548531421965873211848640221635060306200642833463"
    "853585974609101781931276219629563822287439933439020777086511705876102130660051067922171320"
    "065746506246599474937571308257737619536205685975992457989856816189970750416649874540490206"
    "936606726096421881504633395241137765062919851836419722129628383272371360025970533430231669"
    "060142034064679181658499517524396626757148515684537136969992272393011690298548643680023668"
    "42252656116249943591260695464895125184057178557850420475006103515625",
    753,
    "5.8457732298371823574823131482226670449533326157358746716791953135062569212292668611641801"
    "097780408195948596588837344740206406272732220721797510164066073898934074165728050918928927"
    "661521204749793435540643180664127884788340231230981632935135316224370499662844245788803915"
    "476702717548531421965873211848640221635060306200642833463853585974609101781931276219629563"
    "822287439933439020777086511705876102130660051067922171320065746506246599474937571308257737"
    "619536205685975992457989856816189970750416649874540490206936606726096421881504633395241137"
    "765062919851836419722129628383272371360025970533430231669060142034064679181658499517524396"
    "626757148515684537136969992272393011690298548643680023668422526561162499435912606954648951"
    "25184057178557850420475006103515625e-303"
  },
  {
    1.6446596413123796e-302,
    1054,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000164465964131237955496522777053303591275019495978878844846"
    "436113300880266183458890074114483112001652990041532139330705876914793120460121891845777978"
    "753428196824338319708537694880490448415548357571084171954782371013982793608098323673105692"
    "542988889792850824706000259122737766555960728014769438575916147201919943102204875392527648"
    "062645020023888520863214641824353895255575638074536400353005626823857188489512901031327512"
    "343674332673547866665016504944648849864967346476782260339605421755639569644367437952386484"
    "883839418341180039062684146052818964935808566020049555301613761823243644333562714391756208"
    "995511664892316163547208497824657837748660221129693644603102397975669649131600761665002532"
    "724955406576363194307423103157628219150865334086120128631591796875",
    752,
    "1.6446596413123795549652277705330359127501949597887884484643611330088026618345889007411448"
    "311200165299004153213933070587691479312046012189184577797875342819682433831970853769488049"
    "044841554835757108417195478237101398279360809832367310569254298888979285082470600025912273"
    "776655596072801476943857591614720191994310220487539252764806264502002388852086321464182435"
    "389525557563807453640035300562682385718848951290103132751234367433267354786666501650494464"
    "884986496734647678226033960542175563956964436743795238648488383941834118003906268414605281"
    "896493580856602004955530161376182324364433356271439175620899551166489231616354720849782465"
    "783774866022112969364460310239797566964913160076166500253272495540657636319430742310315762"
    "8219150865334086120128631591796875e-302"
  },
  {
    3.571334515579776e-302,
    1054,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000357133451557977617610792878229505570226756358469896911919"
    "377105068137826586828649404727968260620169232867708166475001047504811177502998554808270085"
    "595314612727547047624036293060053880752125848194352774957680860559029018459961625418467498"
    "057908461053265054227705934347738190108734915700270888252381358986019181991946386455724737"
    "425855460042359579974778458859418696543460213592095168174624769832598626020025085824933983"
    "826300131126260391727473707770159740877054653682600781506186186387821934514680593847124459"
    "886451259056737591228212753379128164286102263071648036852588478724203711246744640823260641"
    "230432712249084182956955311914162575870246911724423990175156552477624283439156405949424531"
    "825414560528190620328410111172789953570827492512762546539306640625",
    752,
    "3.5713345155797761761079287822950557022675635846989691191937710506813782658682864940472796"
    "826062016923286770816647500104750481117750299855480827008559531461272754704762403629306005"
    "388075212584819435277495768086055902901845996162541846749805790846105326505422770593434773"
    "819010873491570027088825238135898601918199194638645572473742585546004235957997477845885941"
    "869654346021359209516817462476983259862602002508582493398382630013112626039172747370777015"
    "974087705465368260078150618618638782193451468059384712445988645125905673759122821275337912"
    "816428610226307164803685258847872420371124674464082326064123043271224908418295695531191416"
    "257587024691172442399017515655247762428343915640594942453182541456052819062032841011117278"
    "9953570827492512762546539306640625e-302"
  },
  {
    5.226450761688444e-302,
    1053,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000522645076168844375698058090407392073363666940952778105547"
    "306570912930855963880587815215988005244769961306374980627154903920257708024997792739202136"
    "400792180949605152479609662491266228594913712258572249852752399855128509501475677628955545"
    "887994275596738657711180366820971473683202760016403753438000065807028111002033265199878410"
    "071679692989050499741043577435715581470077666354778579928550640390691438050605710431110522"
    "225336076906939348214560580910775118826868200509986914895110839876152537097330539855968230"
    "909381107897375007290398837838513819151467101011152518705481159715971039220099109912777591"
    "446151811407376335370810682702451339907560494143445416915111573503047799447714154135643683"
    "59181745069897590897094359629893034480119240470230579376220703125",
    751,
    "5.2264507616884437569805809040739207336366694095277810554730657091293085596388058781521598"
    "800524476996130637498062715490392025770802499779273920213640079218094960515247960966249126"
    "622859491371225857224985275239985512850950147567762895554588799427559673865771118036682097"
    "147368320276001640375343800006580702811100203326519987841007167969298905049974104357743571"
    "558147007766635477857992855064039069143805060571043111052222533607690693934821456058091077"
    "511882686820050998691489511083987615253709733053985596823090938110789737500729039883783851"
    "381915146710101115251870548115971597103922009910991277759144615181140737633537081068270245"
    "133990756049414344541691511157350304779944771415413564368359181745069897590897094359629893"
    "034480119240470230579376220703125e-302"
  },
  {
    1.0747997253788843e-301,
    1051,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000001074799725378884349002591708831714014935426753194183154300"
    "531753894711710626048627725959770241731634484316278231555944004086406896017916323123187937"
    "924522777002178813750085517779550084791043067516845837986400111177331237195678228062266622"
    "507530803180421764933136874457770345473896308774311676909174769953465658733136531173064674"
    "382607369955012283179212129357477946380958365891562954818745714107417513297964491278143411"
    "211987964497322424618499276748938027265740133673157093792377088193412887758414403805099514"
    "431584478602627110212024512841842169839736663696096780505831114559890703110871001542936248"
    "745358663880699070477580187175555004706042629340722247723902377618834842156673903433698480"
    "020775608092608474045288879938908621625159867107868194580078125",
    750,
    "1.0747997253788843490025917088317140149354267531941831543005317538947117106260486277259597"
    "702417316344843162782315559440040864068960179163231231879379245227770021788137500855177795"
    "500847910430675168458379864001111773312371956782280622666225075308031804217649331368744577"
    "703454738963087743116769091747699534656587331365311730646743826073699550122831792121293574"
    "779463809583658915629548187457141074175132979644912781434112119879644973224246184992767489"
    "380272657401336731570937923770881934128877584144038050995144315844786026271102120245128418"
    "421698397366636960967805058311145598907031108710015429362487453586638806990704775801871755"
    "550047060426293407222477239023776188348421566739034336984800207756080926084740452888799389"
    "08621625159867107868194580078125e-301"
  },
  {
    1.90892926200005e-301,
    1051,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000001908929262000050146246109184641947605537081104324320091424"
    "275239881358691794611711727670645040801753875872256603903100157907411727213651838643291366"
    "454916544765083809468705448549739616445772463810826572502865543786724684318613551976033658"
    "170744074959929074525489851922078469090207322060182315897761166701384940345796648239069268"
    "255063916895324452269427403961247476556898979817349667497675420979079930454125061487858631"
    "618590818324348506617419383780879472992819215872496857301165261550671277026380409356090075"
    "981969035112260538333946368989549925329930627669708876781138383078123518967695199641472080"
    "864590952079516774516291036652778177765975414972155413953814454153462956899623245175859438"
    "180611960265891792845084629082208493855432607233524322509765625",
    750,
    "1.9089292620000501462461091846419476055370811043243200914242752398813586917946117117276706"
    "450408017538758722566039031001579074117272136518386432913664549165447650838094687054485497"
    "396164457724638108265725028655437867246843186135519760336581707440749599290745254898519220"
    "784690902073220601823158977611667013849403457966482390692682550639168953244522694274039612"
    "474765568989798173496674976754209790799304541250614878586316185908183243485066174193837808"
    "794729928192158724968573011652615506712770263804093560900759819690351122605383339463689895"
    "499253299306276697088767811383830781235189676951996414720808645909520795167745162910366527"
    "781777659754149721554139538144541534629568996232451758594381806119602658917928450846290822"
    "08493855432607233524322509765625e-301"
  },
  {
    6.786272682763782e-301,
    1047,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000006786272682763782230838381983740574462538627006938330369010"
    "760005283210472937468617690271829803643955687653670078357664384731585035733412334209913567"
    "418276410462791835406312031133362466615467364233496864997820370930641229855313759587426623"
    "336318947131414886263684536725151767642125347109002448147389830994122526274370789761701835"
    "855397480023329148528522435511879625527696702064745017607821933043748687600890171795457338"
    "357909910477142886261852651034820072508122374304467400027175334823146270544254593822739470"
    "223273544960101498579166998179831269447169218862728847291808670566291805820903410461158893"
    "419653332870539994687457679135320191821821021237042126628866238914410876818740171241379001"
    "52328702188653090400516365576777388923801481723785400390625",
    746,
    "6.7862726827637822308383819837405744625386270069383303690107600052832104729374686176902718"
    "298036439556876536700783576643847315850357334123342099135674182764104627918354063120311333"
    "624666154673642334968649978203709306412298553137595874266233363189471314148862636845367251"
    "517676421253471090024481473898309941225262743707897617018358553974800233291485285224355118"
    "796255276967020647450176078219330437486876008901717954573383579099104771428862618526510348"
    "200725081223743044674000271753348231462705442545938227394702232735449601014985791669981798"
    "312694471692188627288472918086705662918058209034104611588934196533328705399946874576791353"
    "201918218210212370421266288662389144108768187401712413790015232870218865309040051636557677"
    "7388923801481723785400390625e-301"
  },
  {
    1.2172372613102281e-300,
    1044,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000012172372613102280993179252931186926982692345919033968631102"
    "111765103315033304380176933385417160002408427950029547023800373790313896422917002964114755"
    "006655529461744412575668035201443743111285696889724935095981675922360465248279596722223732"
    "319388253286085175337115483791998375928132921436272911472395242334119323746974950076147620"
    "325900633348562563414895932402962713954954623414922689851738589686706046155327824502166704"
    "335474048380283048951935513621194283558946156241969867855011200207355333726764618109660919"
    "284199142502850576367653106436586300779749343132601043725371192211934157616065169378053005"
    "362625213894304321239311659199881966417445153541195275166274485974787587695265918859779451"
    "90412930384876548373096483146582613699138164520263671875",
    744,
    "1.2172372613102280993179252931186926982692345919033968631102111765103315033304380176933385"
    "417160002408427950029547023800373790313896422917002964114755006655529461744412575668035201"
    "443743111285696889724935095981675922360465248279596722223732319388253286085175337115483791"
    "998375928132921436272911472395242334119323746974950076147620325900633348562563414895932402"
    "962713954954623414922689851738589686706046155327824502166704335474048380283048951935513621"
    "194283558946156241969867855011200207355333726764618109660919284199142502850576367653106436"
    "586300779749343132601043725371192211934157616065169378053005362625213894304321239311659199"
    "881966417445153541195275166274485974787587695265918859779451904129303848765483730964831465"
    "82613699138164520263671875e-300"
  },
  {
    1.7966676643662128e-300,
    1042,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000017966676643662128335665392247768118683231502978059871166650"
    "153469309831612890500966882039821861868400342764656165871202066209954558783721103637054681"
    "294735049443223800468376366456756897344067365028385484351159634636984427195354961155387264"
    "339862668638629505370619690828822916514604841335766672558694378764961002206903271627540775"
    "195602113421651211440307831129924696845524348946549788527717960851496934136922932274256932"
    "207250250473001752070024993926310544782418200142226965748136861450091739855897290050893649"
    "253632501203390546203427335170679143242133197314758507297389888532047996981382454299196612"
    "660175644807829650921817679145247728070114038343552765434718432890065190765206137218078347"
    "442517180856079439532635433351970277726650238037109375",
    742,
    "1.7966676643662128335665392247768118683231502978059871166650153469309831612890500966882039"
    "821861868400342764656165871202066209954558783721103637054681294735049443223800468376366456"
    "756897344067365028385484351159634636984427195354961155387264339862668638629505370619690828"
    "822916514604841335766672558694378764961002206903271627540775195602113421651211440307831129"
    "924696845524348946549788527717960851496934136922932274256932207250250473001752070024993926"
    "310544782418200142226965748136861450091739855897290050893649253632501203390546203427335170"
    "679143242133197314758507297389888532047996981382454299196612660175644807829650921817679145"
    "247728070114038343552765434718432890065190765206137218078347442517180856079439532635433351"
    "970277726650238037109375e-300"
  },
  {
    5.462819694528703e-300,
    1047,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000054628196945287030878864501228515552805464723717056197760606"
    "756889643903510249553048439017711133773671440354639386335467946363525894874070569406966734"
    "125739688803637146671524346615605337829373355492821602887244992662702819402000347666640659"
    "247971465273660948829887189895772285486610572716404702769335978868511669673099220490286146"
    "185662487877704337685653857174654805590027514341847429667755625338708736633025559983835650"
    "906911708214459739844502268662973726280484820160402787520901996323488834872773275355713754"
    "101050144654204486060330761248897894980459029454932175335312351551030494535048881789983902"
    "715942651427865592007898400364624653290699411376938730047962082755849242672666320969811738"
    "37936942963677930569510454716919412021525204181671142578125",
    747,
    "5.4628196945287030878864501228515552805464723717056197760606756889643903510249553048439017"
    "711133773671440354639386335467946363525894874070569406966734125739688803637146671524346615"
    "605337829373355492821602887244992662702819402000347666640659247971465273660948829887189895"
    "772285486610572716404702769335978868511669673099220490286146185662487877704337685653857174"
    "654805590027514341847429667755625338708736633025559983835650906911708214459739844502268662"
    "973726280484820160402787520901996323488834872773275355713754101050144654204486060330761248"
    "897894980459029454932175335312351551030494535048881789983902715942651427865592007898400364"
    "624653290699411376938730047962082755849242672666320969811738379369429636779305695104547169"
    "19412021525204181671142578125e-300"
  },
  {
    6.961706248722051e-300,
    1043,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000069617062487220514621989094015930584333539423148144476551620"
    "642624680348776495275585358068724521997148409410638391145465795792701718124545287291547462"
    "211109638222107168169752091421481830386570720575104449835102504445368841983656051665210319"
    "551112196424192202989543478892145345310948377900918236569840573531393000071258003293035204"
    "706020167948760755422183994450699458391794656901738793039684960506013257067218773020004462"
    "578765143900984223269242821113257087327961856883065410269528691705370808275396621026048371"
    "816525487907526542036986587976346660964440251449023590091913852768941014859350462308144180"
    "303940443624892373450800025135071962212292725330139318787534438587025433685744672341079724"
    "9302829991612732996042467448205570690333843231201171875",
    743,
    "6.9617062487220514621989094015930584333539423148144476551620642624680348776495275585358068"
    "724521997148409410638391145465795792701718124545287291547462211109638222107168169752091421"
    "481830386570720575104449835102504445368841983656051665210319551112196424192202989543478892"
    "145345310948377900918236569840573531393000071258003293035204706020167948760755422183994450"
    "699458391794656901738793039684960506013257067218773020004462578765143900984223269242821113"
    "257087327961856883065410269528691705370808275396621026048371816525487907526542036986587976"
    "346660964440251449023590091913852768941014859350462308144180303940443624892373450800025135"
    "071962212292725330139318787534438587025433685744672341079724930282999161273299604246744820"
    "5570690333843231201171875e-300"
  },
  {
    1.5112507155402445e-299,
    1045,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000151125071554024447020135306022393565196762089755575456624338"
    "495151308209685256453809677900586317175819989462383489016678007812825705103181297543850063"
    "798852533579279995518512249438119478194724227415435927609478553047898933437022912325781333"
    "212370953323947954132838321509883392993414469072134683187383175819910501151266701414832623"
    "120087426760602897763558894964250015209221575190675349103449428597587562568657357560934571"
    "444329997703898734942358076870125601377536544377173388533764111459177194183621681873341290"
    "551265610317343464385367201962064725112486504236979032316608499911148570785237902780477588"
    "448058280897721387267446752139959763555849512939487567275285870905514013401789321488674482"
    "336764102707025482497105173251839005388319492340087890625",
    746,
    "1.5112507155402444702013530602239356519676208975557545662433849515130820968525645380967790"
    "058631717581998946238348901667800781282570510318129754385006379885253357927999551851224943"
    "811947819472422741543592760947855304789893343702291232578133321237095332394795413283832150"
    "988339299341446907213468318738317581991050115126670141483262312008742676060289776355889496"
    "425001520922157519067534910344942859758756256865735756093457144432999770389873494235807687"
    "012560137753654437717338853376411145917719418362168187334129055126561031734346438536720196"
    "206472511248650423697903231660849991114857078523790278047758844805828089772138726744675213"
    "995976355584951293948756727528587090551401340178932148867448233676410270702548249710517325"
    "1839005388319492340087890625e-299"
  },
  {
    2.673980873749012e-299,
    1044,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000267398087374901225364009190269428022346895766923714793667844"
    "056434839900624330085514973193324262004035356895030064921783009805791888470580796419933084"
    "847761281293880427379962271237851947800328657537013648988481294951696202424421176096074833"
    "609809099484601564965538617055249331116206925771990037990303050700858852603141997675889409"
    "904994988078689242985218461431861416650687300826323297128877359071851056100330924999478855"
    "370872777021779956743000754547833513581633478384242279856707134228104860925927074204711269"
    "107333714275775531468377613626493739012218735422714121430942971549508289032953889347174441"
    "040327095431644121171298500061017726989443110670420595265393011954189484319419575166250838"
    "48704359299592279997082044928902178071439266204833984375",
    745,
    "2.6739808737490122536400919026942802234689576692371479366784405643483990062433008551497319"
    "332426200403535689503006492178300980579188847058079641993308484776128129388042737996227123"
    "785194780032865753701364898848129495169620242442117609607483360980909948460156496553861705"
    "524933111620692577199003799030305070085885260314199767588940990499498807868924298521846143"
    "186141665068730082632329712887735907185105610033092499947885537087277702177995674300075454"
    "783351358163347838424227985670713422810486092592707420471126910733371427577553146837761362"
    "649373901221873542271412143094297154950828903295388934717444104032709543164412117129850006"
    "101772698944311067042059526539301195418948431941957516625083848704359299592279997082044928"
    "902178071439266204833984375e-299"
  },
  {
    8.503369157355244e-299,
    1043,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000850336915735524367430323595018676754258445568824857213848746"
    "280704072236823268383060174370779364946428938559999659947344275015058137995788308286051472"
    "366936034180700376051612641241581279358969439631011755807941264018000920525850166281682999"
    "885676817002390365053716091930171460789885945120581757248789493843788302263359833762908222"
    "914044189248632446027235425101963979115169194359369052429197321029095626722937610274060480"
    "758657835985167284620389463467848900582426969335519962264577071777190480699878841459712168"
    "172042966775236944743132671233141904503644142717629031982855171454949581396225686127777916"
    "417788318143148689664629360275192119733837689447193749842155872701062990042439317031447394"
    "0204826819442250618319434352088137529790401458740234375",
    744,
    "8.5033691573552436743032359501867675425844556882485721384874628070407223682326838306017437"
    "077936494642893855999965994734427501505813799578830828605147236693603418070037605161264124"
    "158127935896943963101175580794126401800092052585016628168299988567681700239036505371609193"
    "017146078988594512058175724878949384378830226335983376290822291404418924863244602723542510"
    "196397911516919435936905242919732102909562672293761027406048075865783598516728462038946346"
    "784890058242696933551996226457707177719048069987884145971216817204296677523694474313267123"
    "314190450364414271762903198285517145494958139622568612777791641778831814314868966462936027"
    "519211973383768944719374984215587270106299004243931703144739402048268194422506183194343520"
    "88137529790401458740234375e-299"
  },
  {
    1.3233106901096892e-298,
    1042,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000001323310690109689206537939780857239039544548016633334237413237"
    "030012119277536525667655183086798246156040464737778965504678479790147488445784672755983143"
    "351130755953690975933005374858558848382606556623636905890286546498407090340200266273232047"
    "853174074580694696818844330785856152531370577370227513485232394640688728493031113584757313"
    "581824158502490046319712249459640319924660673321900548594141665580600670298168713356715944"
    "526652647848979125983846783922008641892438293864255983743979381309089682372829493735657101"
    "058874392424964214213457126729435160471794471801982845652213879420707161040622898225638702"
    "937020553616740975943614118011210770298005618586189797547525922561378864683438391102840453"
    "976506918695559000820338724224711768329143524169921875",
    744,
    "1.3233106901096892065379397808572390395445480166333342374132370300121192775365256676551830"
    "867982461560404647377789655046784797901474884457846727559831433511307559536909759330053748"
    "585588483826065566236369058902865464984070903402002662732320478531740745806946968188443307"
    "858561525313705773702275134852323946406887284930311135847573135818241585024900463197122494"
    "596403199246606733219005485941416655806006702981687133567159445266526478489791259838467839"
    "220086418924382938642559837439793813090896823728294937356571010588743924249642142134571267"
    "294351604717944718019828456522138794207071610406228982256387029370205536167409759436141180"
    "112107702980056185861897975475259225613788646834383911028404539765069186955590008203387242"
    "24711768329143524169921875e-298"
  },
  {
    2.129178977672023e-298,
    1041,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000002129178977672023197474417184045234455038251105719793602087695"
    "352964073068726597457261574546506799204162006987775011216832563456063549134019607044230298"
    "420813677748315867515045245227766744237490627806921707486467127042943546561615399294326604"
    "589514550653819458352282966533750051174678662358361943789708146595937329877498914000348684"
    "048434867379969507431598347670943694548493918720371503317755078814637245395045814662285721"
    "966172731060086325202379832766248446232108304114349781946689185851295651283336594853825113"
    "363205107977583948497066534832654912056496337016989948283722604021746733182517415331234370"
    "695363197261511199956180205135316943056191497132454762416801541867974930419986231366709130"
    "61162976041319760955872197882854379713535308837890625",
    743,
    "2.1291789776720231974744171840452344550382511057197936020876953529640730687265974572615745"
    "465067992041620069877750112168325634560635491340196070442302984208136777483158675150452452"
    "277667442374906278069217074864671270429435465616153992943266045895145506538194583522829665"
    "337500511746786623583619437897081465959373298774989140003486840484348673799695074315983476"
    "709436945484939187203715033177550788146372453950458146622857219661727310600863252023798327"
    "662484462321083041143497819466891858512956512833365948538251133632051079775839484970665348"
    "326549120564963370169899482837226040217467331825174153312343706953631972615111999561802051"
    "353169430561914971324547624168015418679749304199862313667091306116297604131976095587219788"
    "2854379713535308837890625e-298"
  },
  {
    4.178648615235546e-298,
    1036,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000004178648615235546137477685826354126061495049537861130696785666"
    "437631785670860809373661431538965440533278098806894269459412201201534219040541068631174967"
    "609576401398921281304202277625907550397298041578342227031998628589539320999048052554586101"
    "710114818979133947966212532305905925289774542948861040799079637308077821570870307250237717"
    "296235792871419986889904702428266107609034227421206913924373960475229255293263340951844946"
    "909235155930998157636257462474157080728381442870731913798291348362591890965605707331439159"
    "255467480055655804018976745664144920158206659169946960403857699609960069561793275867483380"
    "048607554657047054862232324598082497358302008075588962346666145156130842715967918196794561"
    "008886211006529975975354318507015705108642578125",
    738,
    "4.1786486152355461374776858263541260614950495378611306967856664376317856708608093736614315"
    "389654405332780988068942694594122012015342190405410686311749676095764013989212813042022776"
    "259075503972980415783422270319986285895393209990480525545861017101148189791339479662125323"
    "059059252897745429488610407990796373080778215708703072502377172962357928714199868899047024"
    "282661076090342274212069139243739604752292552932633409518449469092351559309981576362574624"
    "741570807283814428707319137982913483625918909656057073314391592554674800556558040189767456"
    "641449201582066591699469604038576996099600695617932758674833800486075546570470548622323245"
    "980824973583020080755889623466661451561308427159679181967945610088862110065299759753543185"
    "07015705108642578125e-298"
  },
  {
    9.216189377840373e-298,
    1038,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000009216189377840373251386225234344523900898114056892264368766389"
    "694116364433408434582133098995372021049984062191604412148079156247005812992176943721470600"
    "997305826086517349632165288322679331343733551533593428833082741031301445760468599085716685"
    "646380991547104769050790451044679611799053816971826590912754216056544099398818252885384806"
    "815030050912205675383269371309430717708417954717530276617121140004348164540143690585881527"
    "779717788936032409975347303070407536869519995182748268401349997198899341885648856633807514"
    "762006717154913065225875510502534842446581576198070129986491767883536217287215961728669453"
    "640209658291317958839138963689034141887896327158871100869864921806575732263683426851889148"
    "74974230699578026104745731572620570659637451171875",
    740,
    "9.2161893778403732513862252343445239008981140568922643687663896941163644334084345821330989"
    "953720210499840621916044121480791562470058129921769437214706009973058260865173496321652883"
    "226793313437335515335934288330827410313014457604685990857166856463809915471047690507904510"
    "446796117990538169718265909127542160565440993988182528853848068150300509122056753832693713"
    "094307177084179547175302766171211400043481645401436905858815277797177889360324099753473030"
    "704075368695199951827482684013499971988993418856488566338075147620067171549130652258755105"
    "025348424465815761980701299864917678835362172872159617286694536402096582913179588391389636"
    "890341418878963271588711008698649218065757322636834268518891487497423069957802610474573157"
    "2620570659637451171875e-298"
  },
  {
    2.0166542228894894e-297,
    1036,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000020166542228894893663309521343065653027086245497896068869059180"
    "976583615938098889986991041485859172655850284923121278179049365606431887608215844284091438"
    "869551607151387128254162790496000752925692565575836623800465609350566271394565375282111983"
    "641947496028367649029875650259442211307086734438281633873595053753702900094452745724436276"
    "139585419606556712353250612586551744106137528572254128169189469681750106767803119244277794"
    "048375021047157915946259985742137994843962286893735657826695990959412074136251486985267738"
    "409725755584184052152367967583389723881746268838805220315370781221914602974072747830678356"
    "773916331273177786861178117851977338577887683038299362987414194915000779858380976854620285"
    "433129268926055743804681696929037570953369140625",
    739,
    "2.0166542228894893663309521343065653027086245497896068869059180976583615938098889986991041"
    "485859172655850284923121278179049365606431887608215844284091438869551607151387128254162790"
    "496000752925692565575836623800465609350566271394565375282111983641947496028367649029875650"
    "259442211307086734438281633873595053753702900094452745724436276139585419606556712353250612"
    "586551744106137528572254128169189469681750106767803119244277794048375021047157915946259985"
    "742137994843962286893735657826695990959412074136251486985267738409725755584184052152367967"
    "583389723881746268838805220315370781221914602974072747830678356773916331273177786861178117"
    "851977338577887683038299362987414194915000779858380976854620285433129268926055743804681696"
    "929037570953369140625e-297"
  },
  {
    3.5449747515908894e-297,
    1034,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000035449747515908893859520264700887688271107416558032715906653570"
    "211101157114219987495688096790234026558681710675314185695039044167338904689114033379126717"
    "553821191514288183644714890227289713124205442476032376273487541411901781650597957764510578"
    "764204872280947507808314952379965305580442332928843004930869937736534492024578818090715338"
    "972401729596065603472996108834056777416770233084403727965152368376850711830195137130645640"
    "590704660731909870705380652957093534288362044270149179527256787036157563590500335249951282"
    "415497432618748669890602656629029754708350031903441565340107701654986531704968754126524580"
    "974219213031353431952978338287844239630161275452657017530603328409563879230671334653052471"
    "6480707592136667472004774026572704315185546875",
    737,
    "3.5449747515908893859520264700887688271107416558032715906653570211101157114219987495688096"
    "790234026558681710675314185695039044167338904689114033379126717553821191514288183644714890"
    "227289713124205442476032376273487541411901781650597957764510578764204872280947507808314952"
    "379965305580442332928843004930869937736534492024578818090715338972401729596065603472996108"
    "834056777416770233084403727965152368376850711830195137130645640590704660731909870705380652"
    "957093534288362044270149179527256787036157563590500335249951282415497432618748669890602656"
    "629029754708350031903441565340107701654986531704968754126524580974219213031353431952978338"
    "287844239630161275452657017530603328409563879230671334653052471648070759213666747200477402"
    "6572704315185546875e-297"
  },
  {
    8.064939085743458e-297,
    1036,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000080649390857434579193115655390618626369629653008353154439327413"
    "620740314818119036634456720430925246670592130257684011506550587519070226234644358041357171"
    "274978820469569512301227708726310523920189182047928453396981851864551773190750301107980576"
    "020914739320849080150345675252860472734215665659222290616514062474620136017099678838136927"
    "214185461292825117230319532485842557112068939568153450865993280216546928857042166319106957"
    "693584960248973265389426026882676162107949206515499685204300681329542005627271444844193825"
    "095928434951827897281098737015287344758519589835254867679214792038215348813575508972323673"
    "031296110651949553961754642256090570653475980420477041519013852253561174979884533208594899"
    "267372506368456441805392387323081493377685546875",
    739,
    "8.0649390857434579193115655390618626369629653008353154439327413620740314818119036634456720"
    "430925246670592130257684011506550587519070226234644358041357171274978820469569512301227708"
    "726310523920189182047928453396981851864551773190750301107980576020914739320849080150345675"
    "252860472734215665659222290616514062474620136017099678838136927214185461292825117230319532"
    "485842557112068939568153450865993280216546928857042166319106957693584960248973265389426026"
    "882676162107949206515499685204300681329542005627271444844193825095928434951827897281098737"
    "015287344758519589835254867679214792038215348813575508972323673031296110651949553961754642"
    "256090570653475980420477041519013852253561174979884533208594899267372506368456441805392387"
    "323081493377685546875e-297"
  },
  {
    2.1182119864764602e-296,
    1035,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000211821198647646021306420947830872065837349416562163106754912736"
    "319440133647537047417519506625089140293248385386753482217342594034540534307701346332311222"
    "871574796733810364341680474829175453754906619121554670765885550307369656911442002153761634"
    "189363912594872690005956120312163191838878685861089147282669130448902104159282049214250431"
    "374142847079932175793656516248460353248085399130697050117059833885306031409904703709059143"
    "662429130360743269466709195978765360244654637182808686838641860759085152084357532663577478"
    "471344174751577529097224074854822396702272280914491477744407730301465429670751676432675885"
    "802073163203570625390902269155329902800456852585594756353628350150050201991314356647042088"
    "16162996076304381176669267006218433380126953125",
    739,
    "2.1182119864764602130642094783087206583734941656216310675491273631944013364753704741751950"
    "662508914029324838538675348221734259403454053430770134633231122287157479673381036434168047"
    "482917545375490661912155467076588555030736965691144200215376163418936391259487269000595612"
    "031216319183887868586108914728266913044890210415928204921425043137414284707993217579365651"
    "624846035324808539913069705011705983388530603140990470370905914366242913036074326946670919"
    "597876536024465463718280868683864186075908515208435753266357747847134417475157752909722407"
    "485482239670227228091449147774440773030146542967075167643267588580207316320357062539090226"
    "915532990280045685258559475635362835015005020199131435664704208816162996076304381176669267"
    "006218433380126953125e-296"
  },
  {
    3.820427653349436e-296,
    1034,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000382042765334943598395327786957950037434467474350925682292758702"
    "707539053966097537580947663100241695943403082531804530234533427497205555769804527722891005"
    "758348613020336577997700742420034577833321799026159177505203867462164483788332186044790557"
    "458920378993259626445306500420115807513364697741528648873115208090540520328862666794570556"
    "183215614239563950863023436570620060327664713533021804518289012696278260248731747094663669"
    "522603971081606399156333469955622929484291725938608707418721047578358097887363479034370662"
    "134536373850191404136724772853159069217212919724652162503614927500301311909590549418776185"
    "714671504837824213395361253602301932304828486128552452977230201644649186653479078364185839"
    "1019539842314856059601879678666591644287109375",
    738,
    "3.8204276533494359839532778695795003743446747435092568229275870270753905396609753758094766"
    "310024169594340308253180453023453342749720555576980452772289100575834861302033657799770074"
    "242003457783332179902615917750520386746216448378833218604479055745892037899325962644530650"
    "042011580751336469774152864887311520809054052032886266679457055618321561423956395086302343"
    "657062006032766471353302180451828901269627826024873174709466366952260397108160639915633346"
    "995562292948429172593860870741872104757835809788736347903437066213453637385019140413672477"
    "285315906921721291972465216250361492750030131190959054941877618571467150483782421339536125"
    "360230193230482848612855245297723020164464918665347907836418583910195398423148560596018796"
    "78666591644287109375e-296"
  },
  {
    9.166969936802109e-296,
    1033,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000916696993680210866828300309093813571285621232474662339098971852"
    "561732904368691968438376659413371715296794753554027780942902843142861485494708552404216026"
    "343714053469828571423620780701853124085458958213035274017523356243037902328013591119341990"
    "815640508353888461299416321909396083754094018415827176690224623856388154284062514578743601"
    "147119588910563403995608284494480455504769946667523216484187271326674880791332001238898632"
    "099770597084200640414102147020755519357329067231460009340640433675296982654429366131186868"
    "621479458580681292807701451756080357474567103969854609319502944972326497871151715832670224"
    "162682601508281235495203396273563440918968053523398319267275737964394466129772993407912666"
    "757701714015382776779006235301494598388671875",
    737,
    "9.1669699368021086682830030909381357128562123247466233909897185256173290436869196843837665"
    "941337171529679475355402778094290284314286148549470855240421602634371405346982857142362078"
    "070185312408545895821303527401752335624303790232801359111934199081564050835388846129941632"
    "190939608375409401841582717669022462385638815428406251457874360114711958891056340399560828"
    "449448045550476994666752321648418727132667488079133200123889863209977059708420064041410214"
    "702075551935732906723146000934064043367529698265442936613118686862147945858068129280770145"
    "175608035747456710396985460931950294497232649787115171583267022416268260150828123549520339"
    "627356344091896805352339831926727573796439446612977299340791266675770171401538277677900623"
    "5301494598388671875e-296"
  },
  {
    1.0587007370833377e-295,
    1031,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000001058700737083337693071645886124072693146296874798954178832516189"
    "855441157660004353211757630771024341596578033719821846802609255108564396255638193782798864"
    "423529412273792202206404073625371548623741287108101146745677514478491771437336532301602490"
    "071369439129589842422269461377654471657420744038482489238158431343585984336411227352371449"
    "776491123135684782630561684391683828579425996992353709635774270637096026351859440339529808"
    "571821550153320544369606029850373971274571312490881640236871417465348136536681156868276078"
    "287691586326216348211661922776018594564117076133002365617217752004182146230232982446412088"
    "873912163365353558350257031136436641825883900966469893445370409827399399146934862679071893"
    "1843515715485182226984761655330657958984375",
    736,
    "1.0587007370833376930716458861240726931462968747989541788325161898554411576600043532117576"
    "307710243415965780337198218468026092551085643962556381937827988644235294122737922022064040"
    "736253715486237412871081011467456775144784917714373365323016024900713694391295898424222694"
    "613776544716574207440384824892381584313435859843364112273523714497764911231356847826305616"
    "843916838285794259969923537096357742706370960263518594403395298085718215501533205443696060"
    "298503739712745713124908816402368714174653481365366811568682760782876915863262163482116619"
    "227760185945641170761330023656172177520041821462302329824464120888739121633653535583502570"
    "311364366418258839009664698934453704098273993991469348626790718931843515715485182226984761"
    "655330657958984375e-295"
  },
  {
    2.5617734189585968e-295,
    1030,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000002561773418958596768681829976825737256842310654327258445744819313"
    "471812859917700055329134698004320247395612121803671260214123922447411930937455720319306039"
    "833598839043844471067605414500422343705554909234106365193995271125408960594257515760719605"
    "036779487548177007694984011771817662329599641936649037337848258202676646918032486840997585"
    "611913527817469545238997407045234095644380673653509287505277908336999504527719309410219875"
    "178825293965486069247083227741334581779697376731123802020068320774184500229739579005871615"
    "352548376739622189939870489332022672990844324757213970295295844625265990020341139334420341"
    "388520264993199887535744618476933977079195980768174754058802045425915342354542418540247245"
    "180296875588510374655015766620635986328125",
    735,
    "2.5617734189585967686818299768257372568423106543272584457448193134718128599177000553291346"
    "980043202473956121218036712602141239224474119309374557203193060398335988390438444710676054"
    "145004223437055549092341063651939952711254089605942575157607196050367794875481770076949840"
    "117718176623295996419366490373378482582026766469180324868409975856119135278174695452389974"
    "070452340956443806736535092875052779083369995045277193094102198751788252939654860692470832"
    "277413345817796973767311238020200683207741845002297395790058716153525483767396221899398704"
    "893320226729908443247572139702952958446252659900203411393344203413885202649931998875357446"
    "184769339770791959807681747540588020454259153423545424185402472451802968755885103746550157"
    "66620635986328125e-295"
  },
  {
    6.117667163341597e-295,
    1030,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000006117667163341597375960900171171739569686686959974202247177759940"
    "425621023390290215523752040825861534988212323450527213368448456470177724705860105376186286"
    "618828790462167147855728450720401421620722177241528155204277434678258158119268340086515696"
    "034960671064545207292832947525798713291829364263480267044200323637785949438810254882961227"
    "449807676379388096070674965149803890959401033610344996445005362652164402969795474430171420"
    "624878611838431720806079540967050219910492570494935946901416032623576393016004846717787329"
    "973919115251418314656160745439460383763946147928236577727544572145499722938153200720552176"
    "047316890086301362717234566541647151923209709182593285429056144712640551237160829125782371"
    "580577478908935518120415508747100830078125",
    735,
    "6.1176671633415973759609001711717395696866869599742022471777599404256210233902902155237520"
    "408258615349882123234505272133684484564701777247058601053761862866188287904621671478557284"
    "507204014216207221772415281552042774346782581581192683400865156960349606710645452072928329"
    "475257987132918293642634802670442003236377859494388102548829612274498076763793880960706749"
    "651498038909594010336103449964450053626521644029697954744301714206248786118384317208060795"
    "409670502199104925704949359469014160326235763930160048467177873299739191152514183146561607"
    "454394603837639461479282365777275445721454997229381532007205521760473168900863013627172345"
    "665416471519232097091825932854290561447126405512371608291257823715805774789089355181204155"
    "08747100830078125e-295"
  },
  {
    1.2442918509206014e-294,
    1028,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000012442918509206014012028953502157484899302248375526922745314229166"
    "692585546772990892217496229641056244180835561315195689137757789588502002249677241041398831"
    "705904524537761121338841823910667676546146884838374628066409297381547429916722251684413546"
    "501381958225583188649380446140070327456087784429836326560673077172766655358175666838652658"
    "690245143514862764213610885902701104197953459657922797754442670964483945940232205602535173"
    "374756171647220221181740970825836072592287546555077976093708600776605467700992906496776057"
    "604651411381222236360843414899349118824035810960881746734551421883024974474404488990219518"
    "000424623267392931588862254591600849528519615415787550882158766787784759894027766182909768"
    "7051781662148641771636903285980224609375",
    734,
    "1.2442918509206014012028953502157484899302248375526922745314229166692585546772990892217496"
    "229641056244180835561315195689137757789588502002249677241041398831705904524537761121338841"
    "823910667676546146884838374628066409297381547429916722251684413546501381958225583188649380"
    "446140070327456087784429836326560673077172766655358175666838652658690245143514862764213610"
    "885902701104197953459657922797754442670964483945940232205602535173374756171647220221181740"
    "970825836072592287546555077976093708600776605467700992906496776057604651411381222236360843"
    "414899349118824035810960881746734551421883024974474404488990219518000424623267392931588862"
    "254591600849528519615415787550882158766787784759894027766182909768705178166214864177163690"
    "3285980224609375e-294"
  },
  {
    3.100051235126117e-294,
    1027,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000031000512351261170557348686264751638282347697441603952716948646046"
    "693942464502542196818852403552824031750041494745884790683734926184617588302516616480649859"
    "990878395191582611359155219189289993802671771294630735339482556380025342809655953530929191"
    "046934934274971727445056851627390907747951068768205696935466992391847079285679683543224431"
    "856784444083274091500700420703555836987399712520363953870514465047148032607184325622329241"
    "423209045337661652529605739247872738438576073597412440399102893350949320021448840462295019"
    "834812872987934725143073300044324617637500126768038853874523788584278674681881882414761448"
    "603612119759437604335951296607240419261927649158173573533413935670423949606511027942182777"
    "417112760076634003780782222747802734375",
    733,
    "3.1000512351261170557348686264751638282347697441603952716948646046693942464502542196818852"
    "403552824031750041494745884790683734926184617588302516616480649859990878395191582611359155"
    "219189289993802671771294630735339482556380025342809655953530929191046934934274971727445056"
    "851627390907747951068768205696935466992391847079285679683543224431856784444083274091500700"
    "420703555836987399712520363953870514465047148032607184325622329241423209045337661652529605"
    "739247872738438576073597412440399102893350949320021448840462295019834812872987934725143073"
    "300044324617637500126768038853874523788584278674681881882414761448603612119759437604335951"
    "296607240419261927649158173573533413935670423949606511027942182777417112760076634003780782"
    "222747802734375e-294"
  },
  {
    5.514599643126014e-294,
    1026,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000055145996431260139100993467191842543303968665047751359175490329248"
    "140080201690655843472208689617281422163505946515376031144432861141969013465023152411931094"
    "094468689740147397230391483091741506595344330145468538812284152241341540438198885491520708"
    "164396533783108870322254998368214367155967708666623505471085546790829367464674575770464713"
    "312264697289274394842495745574537422516867037314585991641886069493735438794004631014086781"
    "490233046315499495508653223255778244007921674070986270919136424783243756152774132037466449"
    "463259190112730265872484256651189076460562833874448737314181117507577452776597522475895186"
    "411139468734071358074828945392781426760447467958524915859532843339277430462825648345122700"
    "46662507638757233507931232452392578125",
    732,
    "5.5145996431260139100993467191842543303968665047751359175490329248140080201690655843472208"
    "689617281422163505946515376031144432861141969013465023152411931094094468689740147397230391"
    "483091741506595344330145468538812284152241341540438198885491520708164396533783108870322254"
    "998368214367155967708666623505471085546790829367464674575770464713312264697289274394842495"
    "745574537422516867037314585991641886069493735438794004631014086781490233046315499495508653"
    "223255778244007921674070986270919136424783243756152774132037466449463259190112730265872484"
    "256651189076460562833874448737314181117507577452776597522475895186411139468734071358074828"
    "945392781426760447467958524915859532843339277430462825648345122700466625076387572335079312"
    "32452392578125e-294"
  },
  {
    8.019518080011996e-294,
    1025,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000080195180800119955218580043524324296264728715452014332332170246674"
    "538871845805219648492909707144892198471904040082534603052959979879360004683448264744443748"
    "768376775122009757291979124279989252469842282297844622116572379825674959048399002791467863"
    "983608186561865709542458437166988944208543524566204381852429181095626666053625235057734407"
    "156147486304856183344925087301696163775028804963431540984237380150218682479654930440154361"
    "420853333545363389746232673796364968764826502068547422850220259764855484242040344069618313"
    "480682182145847678546584539645537394235440766309812696813128435634752133024754997149657335"
    "823917906222732053570409088684078732220291515017341550578558846136936725864647215018202880"
    "4099587659948156215250492095947265625",
    731,
    "8.0195180800119955218580043524324296264728715452014332332170246674538871845805219648492909"
    "707144892198471904040082534603052959979879360004683448264744443748768376775122009757291979"
    "124279989252469842282297844622116572379825674959048399002791467863983608186561865709542458"
    "437166988944208543524566204381852429181095626666053625235057734407156147486304856183344925"
    "087301696163775028804963431540984237380150218682479654930440154361420853333545363389746232"
    "673796364968764826502068547422850220259764855484242040344069618313480682182145847678546584"
    "539645537394235440766309812696813128435634752133024754997149657335823917906222732053570409"
    "088684078732220291515017341550578558846136936725864647215018202880409958765994815621525049"
    "2095947265625e-294"
  },
  {
    2.2148549407571022e-293,
    1022,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000221485494075710218030587580892378748180044045162290960266128758237"
    "081853875197114341184482311756965524739725514269341594904066527742443470740785459170633244"
    "811385112780556696588157177143608961573643126150562761915684928175093318190655666848029988"
    "762200952919039153240117642410861794838726888533081918214628310376972606506185797714332742"
    "644691314774842762584499878681779361342808953291656008895208453361683573332178323164956420"
    "441319644683365839431195202310742805276534309836915086270614032968345195613629397653273340"
    "702900233535205348360972985481672988378027259918186784684652101441052017008490362120457218"
    "093597760723253451093267846381682202986771828182657442023610074758531854529463439845010652"
    "7178604665095917880535125732421875",
    729,
    "2.2148549407571021803058758089237874818004404516229096026612875823708185387519711434118448"
    "231175696552473972551426934159490406652774244347074078545917063324481138511278055669658815"
    "717714360896157364312615056276191568492817509331819065566684802998876220095291903915324011"
    "764241086179483872688853308191821462831037697260650618579771433274264469131477484276258449"
    "987868177936134280895329165600889520845336168357333217832316495642044131964468336583943119"
    "520231074280527653430983691508627061403296834519561362939765327334070290023353520534836097"
    "298548167298837802725991818678468465210144105201700849036212045721809359776072325345109326"
    "784638168220298677182818265744202361007475853185452946343984501065271786046650959178805351"
    "25732421875e-293"
  },
  {
    4.20911452464387e-293,
    1024,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000420911452464386982964641976788584620223206018410308358432061499907"
    "914886710591233068837766126994241087546476736165489660708158188897653252826134814761213834"
    "692117553854271187898625371210169220451412074088885812485571054146189807584099035414602348"
    "779516882092482242540342614631484930791776599206297780531376028608003691256716592817476525"
    "620377364696154878934954777146874097164792640219120647730346822277636460817321960525720307"
    "042509474279237735500174408103971978728587499232657269969095263398622200501943268247798636"
    "403094152328953203114763477020376923929730346797771145340153925341989297019083584973670152"
    "505364011998054250156547601443853999776062961816363960978393038157465241961259638119909054"
    "498890554896206595003604888916015625",
    731,
    "4.2091145246438698296464197678858462022320601841030835843206149990791488671059123306883776"
    "612699424108754647673616548966070815818889765325282613481476121383469211755385427118789862"
    "537121016922045141207408888581248557105414618980758409903541460234877951688209248224254034"
    "261463148493079177659920629778053137602860800369125671659281747652562037736469615487893495"
    "477714687409716479264021912064773034682227763646081732196052572030704250947427923773550017"
    "440810397197872858749923265726996909526339862220050194326824779863640309415232895320311476"
    "347702037692392973034679777114534015392534198929701908358497367015250536401199805425015654"
    "760144385399977606296181636396097839303815746524196125963811990905449889055489620659500360"
    "4888916015625e-293"
  },
  {
    6.673970944986261e-293,
    1022,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000667397094498626075656756877829201194838756325023076671857801591621"
    "689938639801840509711072309067042884640156496699439994300838785152866487131226692050111421"
    "937130047388309431505994270060996058822819880128635044711488425409256110905357459453600332"
    "108237323622834385376369052090681423022586076490602959231413081621310102421941979163256973"
    "626937425960768196935871679150780952465291867950902919984043604391377263345732481961082670"
    "822286777585677288460873248318553492874197977095601357714718256395148609413826610224397173"
    "111647233642602892621576139275784829747348261571176126724916879947403586696937124033326308"
    "885551681539766822653019749388452598312993795479193611550664943279677707777413654937487264"
    "0072453577886335551738739013671875",
    729,
    "6.6739709449862607565675687782920119483875632502307667185780159162168993863980184050971107"
    "230906704288464015649669943999430083878515286648713122669205011142193713004738830943150599"
    "427006099605882281988012863504471148842540925611090535745945360033210823732362283438537636"
    "905209068142302258607649060295923141308162131010242194197916325697362693742596076819693587"
    "167915078095246529186795090291998404360439137726334573248196108267082228677758567728846087"
    "324831855349287419797709560135771471825639514860941382661022439717311164723364260289262157"
    "613927578482974734826157117612672491687994740358669693712403332630888555168153976682265301"
    "974938845259831299379547919361155066494327967770777741365493748726400724535778863355517387"
    "39013671875e-293"
  },
  {
    1.746546364023377e-292,
    1021,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000001746546364023376910063819736077110430205085110398918660610208193165"
    "002648830459932732755875328272920006833298046786276680503784255684600677870427122738543551"
    "273069990575464714222204379153421619003148917627092920573721527715623421389755093972134547"
    "171078825977573421206869147881140899358764839567176481371382813534485115983567870442751891"
    "903601589626615453304416936900776008651698533435544636347626690765031072431532774639002545"
    "780750939311942703695306465163350453096344539050958448889513198989804953099819252130727537"
    "927730108381018242065386226332023454226886482050639863587133447070386576679785063884990939"
    "398460469896517681871811495270565050105345772489759853919318571093384770703357713843295595"
    "523841257090680301189422607421875",
    729,
    "1.7465463640233769100638197360771104302050851103989186606102081931650026488304599327327558"
    "753282729200068332980467862766805037842556846006778704271227385435512730699905754647142222"
    "043791534216190031489176270929205737215277156234213897550939721345471710788259775734212068"
    "691478811408993587648395671764813713828135344851159835678704427518919036015896266154533044"
    "169369007760086516985334355446363476266907650310724315327746390025457807509393119427036953"
    "064651633504530963445390509584488895131989898049530998192521307275379277301083810182420653"
    "862263320234542268864820506398635871334470703865766797850638849909393984604698965176818718"
    "114952705650501053457724897598539193185710933847707033577138432955955238412570906803011894"
    "22607421875e-292"
  },
  {
    2.5532945426002905e-292,
    1021,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000002553294542600290478050715753818166924852760785307324274455464703276"
    "896685649750398556670281450079476171301511416012248448222007463781644071630086456019890128"
    "740564440929852319292026496727875740396972429247798368290103900949007393832538886143168315"
    "948223660409479333004620541843867294642578845204520026497108979679013560929523117558123789"
    "003921199749409846260814033055309256968619621536147314096602667572767547368480537021868582"
    "121265835469099565600764222773843384879631837745132522662819011867985241984197020689816378"
    "628029697694502793001789351434068617111543379937626164684311069822291752668796387656882250"
    "217177699698369291172140147851454964940605010325751218296624302065265450254613545551241049"
    "480495348689146339893341064453125",
    729,
    "2.5532945426002904780507157538181669248527607853073242744554647032768966856497503985566702"
    "814500794761713015114160122484482220074637816440716300864560198901287405644409298523192920"
    "264967278757403969724292477983682901039009490073938325388861431683159482236604094793330046"
    "205418438672946425788452045200264971089796790135609295231175581237890039211997494098462608"
    "140330553092569686196215361473140966026675727675473684805370218685821212658354690995656007"
    "642227738433848796318377451325226628190118679852419841970206898163786280296976945027930017"
    "893514340686171115433799376261646843110698222917526687963876568822502171776996983692911721"
    "401478514549649406050103257512182966243020652654502546135455512410494804953486891463398933"
    "41064453125e-292"
  },
  {
    7.167199051004838e-292,
    1020,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000007167199051004837761401912654283331028999961614491290387680525327758"
    "435160264770312345835553652819216736444016623532872322399497645329801848435825622595072704"
    "153115885130532082832887438015637784060687495723078001946125568827170156775233327361851224"
    "388982041403917785806215783044313688219646403826782962055511360668264737689708896100223615"
    "047109409764110696035061612142836930506582657157327364164772577568894370334356710122547899"
    "533333122870576400085144538798657786553932591607401743677737356750489495324217151349494977"
    "027083612852377322542673884527343694192326023200301290131404052076274549091612163177499275"
    "646656982709711584001572445734847331676355472196136755105828981107801935019171101978741056"
    "37083630426786839962005615234375",
    728,
    "7.1671990510048377614019126542833310289999616144912903876805253277584351602647703123458355"
    "536528192167364440166235328723223994976453298018484358256225950727041531158851305320828328"
    "874380156377840606874957230780019461255688271701567752333273618512243889820414039177858062"
    "157830443136882196464038267829620555113606682647376897088961002236150471094097641106960350"
    "616121428369305065826571573273641647725775688943703343567101225478995333331228705764000851"
    "445387986577865539325916074017436777373567504894953242171513494949770270836128523773225426"
    "738845273436941923260232003012901314040520762745490916121631774992756466569827097115840015"
    "724457348473316763554721961367551058289811078019350191711019787410563708363042678683996200"
    "5615234375e-292"
  },
  {
    1.4331413352279302e-291,
    1018,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000014331413352279302128149191072605813876697528611345885441115000029609"
    "556842920386965482577284207290323742533219950959430783015194668198486665582222833914383540"
    "319427025322391306167843986091706504184233238705137007795009442881486987886279579164967250"
    "385961545640275014746461793974420301827541350542923014361442430351791371270610282715417684"
    "609169626900527234807117188403332704905395975649634750649953559577652559528484291031633467"
    "618609466234606109062833705290871317724766461968322473754304645412774754269679895564496612"
    "117160486257131756955895785755258835428073160833017461267243813064775784942918437098744770"
    "077870682783671486446180107131875144882846097603267302176234293033266604144621188254848931"
    "364904274232685565948486328125",
    727,
    "1.4331413352279302128149191072605813876697528611345885441115000029609556842920386965482577"
    "284207290323742533219950959430783015194668198486665582222833914383540319427025322391306167"
    "843986091706504184233238705137007795009442881486987886279579164967250385961545640275014746"
    "461793974420301827541350542923014361442430351791371270610282715417684609169626900527234807"
    "117188403332704905395975649634750649953559577652559528484291031633467618609466234606109062"
    "833705290871317724766461968322473754304645412774754269679895564496612117160486257131756955"
    "895785755258835428073160833017461267243813064775784942918437098744770077870682783671486446"
    "180107131875144882846097603267302176234293033266604144621188254848931364904274232685565948"
    "486328125e-291"
  },
  {
    1.6537479214140957e-291,
    1018,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000016537479214140957496644432073018089012144452120603766472454112579144"
    "784777132047524978251678680652305622500049925363586631576797699752300460218057335597840075"
    "718257506403605395840821019033944723087937905141708969026006406748767209958799224140895104"
    "171783821717312533219301444313149132987103107728754026411305784146551911643305304636464367"
    "182427851593250624197965599005919337369891600964947788560805510699332326217569801868088227"
    "035341727693943399255277868009887421134563906845805007487145749111361724158631173517732532"
    "866699194559493616073891745826415206333041128369417623853620031942980159827745311600728794"
    "900450186108482723520894492971559993956728196016916768467473936479509527842902093242027916"
    "630831896327435970306396484375",
    727,
    "1.6537479214140957496644432073018089012144452120603766472454112579144784777132047524978251"
    "678680652305622500049925363586631576797699752300460218057335597840075718257506403605395840"
    "821019033944723087937905141708969026006406748767209958799224140895104171783821717312533219"
    "301444313149132987103107728754026411305784146551911643305304636464367182427851593250624197"
    "965599005919337369891600964947788560805510699332326217569801868088227035341727693943399255"
    "277868009887421134563906845805007487145749111361724158631173517732532866699194559493616073"
    "891745826415206333041128369417623853620031942980159827745311600728794900450186108482723520"
    "894492971559993956728196016916768467473936479509527842902093242027916630831896327435970306"
    "396484375e-291"
  },
  {
    4.290843909020547e-291,
    1017,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000042908439090205467290104179871798651941226153898698783477673232932487"
    "248769134773470874791558624108005131697353608862902655753654574915526919429334606006207109"
    "483961973599274460268272087202048998576465160522712368344493153385893958587111544463848228"
    "814456540883014806068336437696695281211569137591800855366984873526161749640088098036503504"
    "388607252321301774363944935742492341884553608844792004824932319261176330222096044026500175"
    "605116486722498994442457577038571735777909120861981871166561089515170564279345262414214018"
    "962657061977903858087530248130776497757976257129093373104376726882365594396845743694614128"
    "973681910045280458678843904647399071795286872800267566138071165770151527092462692691832160"
    "10787873528897762298583984375",
    726,
    "4.2908439090205467290104179871798651941226153898698783477673232932487248769134773470874791"
    "558624108005131697353608862902655753654574915526919429334606006207109483961973599274460268"
    "272087202048998576465160522712368344493153385893958587111544463848228814456540883014806068"
    "336437696695281211569137591800855366984873526161749640088098036503504388607252321301774363"
    "944935742492341884553608844792004824932319261176330222096044026500175605116486722498994442"
    "457577038571735777909120861981871166561089515170564279345262414214018962657061977903858087"
    "530248130776497757976257129093373104376726882365594396845743694614128973681910045280458678"
    "843904647399071795286872800267566138071165770151527092462692691832160107878735288977622985"
    "83984375e-291"
  },
  {
    9.634563122144926e-291,
    1014,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000096345631221449255842528627098837665212135824138580247641095033482922"
    "043986659116736534146977997184184550644460199475556134749771118880464823522080576561857560"
    "131686314305476183221143225933427668290811069426797326793633645257479047211320140997921897"
    "193252937112350877184381764987726442694148275944171196784772719961039624464068850461484304"
    "306996962247952090789771393359568041684499232396514214051736756405489381778675872598742815"
    "737011666738940745479418888631692294493983939129999574504327346013314634388492904199491990"
    "467623230884876216765889872914995150950084872213663276043304693018490908555227406043470100"
    "780821263144234975457396321979176485835934727447870660921832860224943445191445046127398654"
    "78943102061748504638671875",
    723,
    "9.6345631221449255842528627098837665212135824138580247641095033482922043986659116736534146"
    "977997184184550644460199475556134749771118880464823522080576561857560131686314305476183221"
    "143225933427668290811069426797326793633645257479047211320140997921897193252937112350877184"
    "381764987726442694148275944171196784772719961039624464068850461484304306996962247952090789"
    "771393359568041684499232396514214051736756405489381778675872598742815737011666738940745479"
    "418888631692294493983939129999574504327346013314634388492904199491990467623230884876216765"
    "889872914995150950084872213663276043304693018490908555227406043470100780821263144234975457"
    "396321979176485835934727447870660921832860224943445191445046127398654789431020617485046386"
    "71875e-291"
  },
  {
    2.1086057222075156e-290,
    1010,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000210860572220751561406999234414564500843088972527279057134370861085961"
    "669874684090783975879714875336360665125559425961800525159988473481938995387145032563193842"
    "335341149252710258276907339069644310335615204766793420504624545655865003873573080087197810"
    "388799456453427347327491782658321032413252539952417504938151519582936573695643702439651686"
    "452362202454062268891390078907577990174502268833095111096637846568643857395612744263557657"
    "149281665440973055056669372106589677993963623493823619251634129655150817988363340633174195"
    "752548807091782327159310514974938480008066183768903720394052389144246035868562963466692565"
    "989516150888961289555167178857386348057311567111069383877686712814214422401226656234030087"
    "9441201686859130859375",
    720,
    "2.1086057222075156140699923441456450084308897252727905713437086108596166987468409078397587"
    "971487533636066512555942596180052515998847348193899538714503256319384233534114925271025827"
    "690733906964431033561520476679342050462454565586500387357308008719781038879945645342734732"
    "749178265832103241325253995241750493815151958293657369564370243965168645236220245406226889"
    "139007890757799017450226883309511109663784656864385739561274426355765714928166544097305505"
    "666937210658967799396362349382361925163412965515081798836334063317419575254880709178232715"
    "931051497493848000806618376890372039405238914424603586856296346669256598951615088896128955"
    "516717885738634805731156711106938387768671281421442240122665623403008794412016868591308593"
    "75e-290"
  },
  {
    3.4206800637910767e-290,
    1013,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000342068006379107669759863482487568092455857226172452538849137654670829"
    "026558635397447930919129012919655902035289765511658279266066746655823602750349947919758040"
    "236162078686374492458909606117047221706343896128942681534774709663134694577537000845025759"
    "746268314909933266944228604633857834010155349164209247894596457376326553039760576060487737"
    "178789286028980473242284769127953331015775341858001266767203925650569921328324381543299424"
    "787438555730075039358292293164678724048090901309369949159601150254447351579618707251348992"
    "673994468585960715058863125448305733139414075897329752062164516636621228240923863329090281"
    "273746303889955924595344943621294856344793778243749110618353186426858156005820221778890299"
    "9018318951129913330078125",
    723,
    "3.4206800637910766975986348248756809245585722617245253884913765467082902655863539744793091"
    "912901291965590203528976551165827926606674665582360275034994791975804023616207868637449245"
    "890960611704722170634389612894268153477470966313469457753700084502575974626831490993326694"
    "422860463385783401015534916420924789459645737632655303976057606048773717878928602898047324"
    "228476912795333101577534185800126676720392565056992132832438154329942478743855573007503935"
    "829229316467872404809090130936994915960115025444735157961870725134899267399446858596071505"
    "886312544830573313941407589732975206216451663662122824092386332909028127374630388995592459"
    "534494362129485634479377824374911061835318642685815600582022177889029990183189511299133300"
    "78125e-290"
  },
  {
    6.629016758967737e-290,
    1013,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000662901675896773679513756174229292716806068642209468475059928574348081"
    "435492672562858635936768075345718116507990734126024664323017252510560123157213222771972932"
    "719343366101285834110047644303294398466421171333829299433821937324738327358468415836107634"
    "278887292642282611558725963507700368491900368482366904539555779940351079192322691980254988"
    "685866272949097261799666956920307209623381975737540495544653710372564881978390565989219877"
    "292992603247756361908752192553418763996358838205450960619495274938457068746433456118860927"
    "509511328976925764743028537705307563412694477807516786328337968625833339283430334801991758"
    "626792648077951794847891892706360760921326723768717127744443022976484974299188346336109134"
    "4268061220645904541015625",
    723,
    "6.6290167589677367951375617422929271680606864220946847505992857434808143549267256285863593"
    "676807534571811650799073412602466432301725251056012315721322277197293271934336610128583411"
    "004764430329439846642117133382929943382193732473832735846841583610763427888729264228261155"
    "872596350770036849190036848236690453955577994035107919232269198025498868586627294909726179"
    "966695692030720962338197573754049554465371037256488197839056598921987729299260324775636190"
    "875219255341876399635883820545096061949527493845706874643345611886092750951132897692576474"
    "302853770530756341269447780751678632833796862583333928343033480199175862679264807795179484"
    "789189270636076092132672376871712774444302297648497429918834633610913442680612206459045410"
    "15625e-290"
  },
  {
    1.352094685757577e-289,
    1011,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000001352094685757577046692433600904978812788038658784057375085512637014821"
    "192293633284354988083332910409714056212910610846709279699966459815043752343271443778019240"
    "677407660680483711914428816683238889899138321961096952897929141749716680995649364750852884"
    "940442785486690547343872512681947134948592736866712512591490524779325509073167336423676664"
    "987077644126779433703191479472341766481221651746023800800090309455088100077933460409533060"
    "257217016013325784579980763394136373279286222394859212718266463408191189737964960538213358"
    "597170666187905138168357733394815415483037124393427107254818942235087012334513591656298823"
    "628107449960996084615787537902816882118416869199211414207183456647512172568714738929429586"
    "27827465534210205078125",
    722,
    "1.3520946857575770466924336009049788127880386587840573750855126370148211922936332843549880"
    "833329104097140562129106108467092796999664598150437523432714437780192406774076606804837119"
    "144288166832388898991383219610969528979291417497166809956493647508528849404427854866905473"
    "438725126819471349485927368667125125914905247793255090731673364236766649870776441267794337"
    "031914794723417664812216517460238008000903094550881000779334604095330602572170160133257845"
    "799807633941363732792862223948592127182664634081911897379649605382133585971706661879051381"
    "683577333948154154830371243934271072548189422350870123345135916562988236281074499609960846"
    "157875379028168821184168691992114142071834566475121725687147389294295862782746553421020507"
    "8125e-289"
  },
  {
    3.065153212531983e-289,
    1011,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000003065153212531982804361140388649718163177630153098638242256322057332264"
    "316825884745387709988626966778193900377967818621973079307255143430435512542117022799111672"
    "966962084842623959299291197967912215720692209412461126684609711196673612053116397582155620"
    "461783165845586140232413390195514862846452362791383900061129942543777968385631793034647417"
    "070723446154644578522256496639523318507897582380933808362832961397125905373505903881692268"
    "239311221184441430130140035397035115057622604905647100707015420593956409893712632669418292"
    "832471584470128679373107055110632193260404785063947141156633187195789592983001451297372755"
    "688720100822729534737038975000464217296181756085459582003497648781210676779962476246055302"
    "90864408016204833984375",
    722,
    "3.0651532125319828043611403886497181631776301530986382422563220573322643168258847453877099"
    "886269667781939003779678186219730793072551434304355125421170227991116729669620848426239592"
    "992911979679122157206922094124611266846097111966736120531163975821556204617831658455861402"
    "324133901955148628464523627913839000611299425437779683856317930346474170707234461546445785"
    "222564966395233185078975823809338083628329613971259053735059038816922682393112211844414301"
    "301400353970351150576226049056471007070154205939564098937126326694182928324715844701286793"
    "731070551106321932604047850639471411566331871957895929830014512973727556887201008227295347"
    "370389750004642172961817560854595820034976487812106767799624762460553029086440801620483398"
    "4375e-289"
  },
  {
    5.074022975467963e-289,
    1006,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000005074022975467963001559749763541125170478284238884770689255999822246103"
    "901985835478854589953636416469537561590049706630652513764526444447223678686798336319614568"
    "798213633255002967621393591400598311356640376360911037851826795084045937024297722699696856"
    "108091789878307056427588017412016141242065429389459914673058410334447502904563069562881886"
    "485204081056812066585987639722063880607676286374127002491780885114417306881402349312394657"
    "949491179323334737191953828072140031195288300223306007714107976176291860917574307762494879"
    "344728134817295607280179935636124185858339459651464992994383411267201247553011716792219570"
    "153380458235643643535470204673305630435954878625940750273716264569061340505840007608640007"
    "674694061279296875",
    717,
    "5.0740229754679630015597497635411251704782842388847706892559998222461039019858354788545899"
    "536364164695375615900497066306525137645264444472236786867983363196145687982136332550029676"
    "213935914005983113566403763609110378518267950840459370242977226996968561080917898783070564"
    "275880174120161412420654293894599146730584103344475029045630695628818864852040810568120665"
    "859876397220638806076762863741270024917808851144173068814023493123946579494911793233347371"
    "919538280721400311952883002233060077141079761762918609175743077624948793447281348172956072"
    "801799356361241858583394596514649929943834112672012475530117167922195701533804582356436435"
    "35470204673305630435954878625940750273716264569061340505840007608640007674694061279296875e"
    "-289"
  },
  {
    1.5066401531008766e-288,
    1009,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000015066401531008765660209715620847113430721240845311467067488393060399424"
    "689768179013689935270456617217772626244929423631519228493856034858979575127394053875725554"
    "906220641969564204691181701460559413479173509541754734966610625691182595384833564715001503"
    "127651700215702343139831898865637821458732404403238288912908071942138146945071589392359015"
    "982750809131377045618332285170839599383602029439803065564549875367002305011744027345479199"
    "876728897725328043145101457251586625197382756110368594459048791089589822516229907409639132"
    "745173540057020090898673548797117118192982546034189803865994207837905829666676125695490011"
    "950379447954129812962290407978653640090229326186696355379052802457638497724001780397884431"
    "295096874237060546875",
    721,
    "1.5066401531008765660209715620847113430721240845311467067488393060399424689768179013689935"
    "270456617217772626244929423631519228493856034858979575127394053875725554906220641969564204"
    "691181701460559413479173509541754734966610625691182595384833564715001503127651700215702343"
    "139831898865637821458732404403238288912908071942138146945071589392359015982750809131377045"
    "618332285170839599383602029439803065564549875367002305011744027345479199876728897725328043"
    "145101457251586625197382756110368594459048791089589822516229907409639132745173540057020090"
    "898673548797117118192982546034189803865994207837905829666676125695490011950379447954129812"
    "962290407978653640090229326186696355379052802457638497724001780397884431295096874237060546"
    "875e-288"
  },
  {
    2.8015937941494583e-288,
    1008,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000028015937941494582893752219699378451541676163091913910886367587801515223"
    "826775770979017901213581846339239008597274356448340618213879408241570820677798537246273752"
    "960433178316105236985225964064168271940724334740127274569320116276953185468471169819979865"
    "384740374064133028144008765753349163678336931070726360114289479247710431832859153979633857"
    "644425954502850284440493944167381091614718033977997952112454266036283567034492773027029721"
    "407239251594118127204450343993494664215703368452650479479324179410443022095032721729177309"
    "167120831352005701355101360756688184442333876875901605840993973675678463540511605697472364"
    "864896178452003484902882020513499857730237568149478961646701990361894504810713613096595508"
    "97777080535888671875",
    720,
    "2.8015937941494582893752219699378451541676163091913910886367587801515223826775770979017901"
    "213581846339239008597274356448340618213879408241570820677798537246273752960433178316105236"
    "985225964064168271940724334740127274569320116276953185468471169819979865384740374064133028"
    "144008765753349163678336931070726360114289479247710431832859153979633857644425954502850284"
    "440493944167381091614718033977997952112454266036283567034492773027029721407239251594118127"
    "204450343993494664215703368452650479479324179410443022095032721729177309167120831352005701"
    "355101360756688184442333876875901605840993973675678463540511605697472364864896178452003484"
    "902882020513499857730237568149478961646701990361894504810713613096595508977770805358886718"
    "75e-288"
  },
  {
    5.38303645298669e-288,
    1007,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000053830364529866897070892653133613309687768553145998845020770002508286663"
    "820178028904357041106164871662930240429232628188396390915409645247289394775553893424233091"
    "454382695903114622156824669251256935907514980419516224552868613254602516808760199014692331"
    "121708195844041802689709960389889426829945558691778772225727267044096706370606363455491707"
    "358638817637507737802603692179707323013726519499078873069293802377310441059050095455840943"
    "125657666622548189687664269085772631674942093221544460565434112292880654790600255536877008"
    "967815920183219564371809676229791203099000592768545655827251767634688185250351421699914433"
    "173200215226306149229367128413055299890736661514938449689868803541784730848362983124388847"
    "5000858306884765625",
    719,
    "5.3830364529866897070892653133613309687768553145998845020770002508286663820178028904357041"
    "106164871662930240429232628188396390915409645247289394775553893424233091454382695903114622"
    "156824669251256935907514980419516224552868613254602516808760199014692331121708195844041802"
    "689709960389889426829945558691778772225727267044096706370606363455491707358638817637507737"
    "802603692179707323013726519499078873069293802377310441059050095455840943125657666622548189"
    "687664269085772631674942093221544460565434112292880654790600255536877008967815920183219564"
    "371809676229791203099000592768545655827251767634688185250351421699914433173200215226306149"
    "229367128413055299890736661514938449689868803541784730848362983124388847500085830688476562"
    "5e-288"
  },
  {
    8.441234238050258e-288,
    1006,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000084412342380502576546081430450113714547500629438411347771053360322063370"
    "388173904974678292331037009777086711209941481547644874475926368827721249742015399233877781"
    "800693488890921357295751195663473803676336514458617753228583892708430291151666078954288316"
    "578566036261766999842737938804261922861942664436862436088366000013544530110132454205033116"
    "021887062016086384595971121585437748216881228804338113019161359341335927611034275239403794"
    "071774633231535502365664703148223664552800461726097420248810912726615257924499835178142225"
    "373823617542437015507730018442924520553705165386495075945109984765230847584543031880965883"
    "034097696847994272836703489527961893337125050753747867908906204526854610836750225644209422"
    "171115875244140625",
    718,
    "8.4412342380502576546081430450113714547500629438411347771053360322063370388173904974678292"
    "331037009777086711209941481547644874475926368827721249742015399233877781800693488890921357"
    "295751195663473803676336514458617753228583892708430291151666078954288316578566036261766999"
    "842737938804261922861942664436862436088366000013544530110132454205033116021887062016086384"
    "595971121585437748216881228804338113019161359341335927611034275239403794071774633231535502"
    "365664703148223664552800461726097420248810912726615257924499835178142225373823617542437015"
    "507730018442924520553705165386495075945109984765230847584543031880965883034097696847994272"
    "836703489527961893337125050753747867908906204526854610836750225644209422171115875244140625"
    "e-288"
  },
  {
    1.7229963037415354e-287,
    1004,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000172299630374153538600369392745126026077034739345703168137555632023714502"
    "853526675561807275891540870413620561283459465386257149814404426731073509224486647540553797"
    "898000159803219506010895542832878489629573788723953174999998180200141739610815637795828297"
    "676462623559959560259504121419437835956086250386086680790917264600299756592001149217860301"
    "770599147634700558856107451291371285917065599081112280820640917253408975640335333977713665"
    "527965071968852553983251712474158073574273322543402496753700449881082906684354028745885662"
    "376033485132298748829297504679992953922134958726162710070275535075485078389037082746948188"
    "422864599080818474824767419308316855767579388678417811522174932139378489281966722046490758"
    "6574554443359375",
    717,
    "1.7229963037415353860036939274512602607703473934570316813755563202371450285352667556180727"
    "589154087041362056128345946538625714981440442673107350922448664754055379789800015980321950"
    "601089554283287848962957378872395317499999818020014173961081563779582829767646262355995956"
    "025950412141943783595608625038608668079091726460029975659200114921786030177059914763470055"
    "885610745129137128591706559908111228082064091725340897564033533397771366552796507196885255"
    "398325171247415807357427332254340249675370044988108290668435402874588566237603348513229874"
    "882929750467999295392213495872616271007027553507548507838903708274694818842286459908081847"
    "48247674193083168557675793886784178115221749321393784892819667220464907586574554443359375e"
    "-287"
  },
  {
    4.998022788892795e-287,
    1004,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000499802278889279473615442768760489455270526286941463611538933406026259074"
    "910585749895445770331696354343239202363631494369169360431693425214763451868479262473007994"
    "553742587698599717845223124301708762653254310627635861592921388581071235595388256740574555"
    "405967438747656240522437476590984427613225064118246657061085520840842189180590379919744790"
    "061588693056169849068456226653932205312729989102037696430382138353314730376082250724904487"
    "349755042619016279363453628277094581074974439583045948813522912747444842573673747533613488"
    "115993276538259526079768136866769281524479504986239782116242972307245998903143064844887648"
    "585462105052312954180436121820844169910821850286572016679104482877321613543131206824909895"
    "6584930419921875",
    717,
    "4.9980227888927947361544276876048945527052628694146361153893340602625907491058574989544577"
    "033169635434323920236363149436916936043169342521476345186847926247300799455374258769859971"
    "784522312430170876265325431062763586159292138858107123559538825674057455540596743874765624"
    "052243747659098442761322506411824665706108552084084218918059037991974479006158869305616984"
    "906845622665393220531272998910203769643038213835331473037608225072490448734975504261901627"
    "936345362827709458107497443958304594881352291274744484257367374753361348811599327653825952"
    "607976813686676928152447950498623978211624297230724599890314306484488764858546210505231295"
    "41804361218208441699108218502865720166791044828773216135431312068249098956584930419921875e"
    "-287"
  },
  {
    8.728697912491584e-287,
    1003,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000872869791249158400318111712497687467444368897106753274009448112897434374"
    "990551504083361434472076990675548257442958324490534088914416947203034596292953382062295275"
    "457233755524482929174694081985804881831251240710578812695110499946091913817865773144736149"
    "599230716456945111511342511929392766564124544226552357746710397528073591643337960451898139"
    "342063879562295690992026703839973859592630317510343598115293358217353215269579574482300932"
    "318121087160706915252100028942112875612578710323463601402080899019147011869735277375358119"
    "396699255537108445244031567738587083167210365733362211947829707918613171686490030796833827"
    "666928068725356332339361214932314316194561659184333914511633536505164077112794984714128077"
    "030181884765625",
    716,
    "8.7286979124915840031811171249768746744436889710675327400944811289743437499055150408336143"
    "447207699067554825744295832449053408891441694720303459629295338206229527545723375552448292"
    "917469408198580488183125124071057881269511049994609191381786577314473614959923071645694511"
    "151134251192939276656412454422655235774671039752807359164333796045189813934206387956229569"
    "099202670383997385959263031751034359811529335821735321526957957448230093231812108716070691"
    "525210002894211287561257871032346360140208089901914701186973527737535811939669925553710844"
    "524403156773858708316721036573336221194782970791861317168649003079683382766692806872535633"
    "2339361214932314316194561659184333914511633536505164077112794984714128077030181884765625e-"
    "287"
  },
  {
    1.9042271010497776e-286,
    1002,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000001904227101049777633200170880133061562636160392816356868148687074254559912"
    "708970128427928098420716969202321713200302826873037284450151621445592420898736600143493974"
    "313640366068432903490079568251503921823960776996107081360034058760638306226987187123573951"
    "079730136436300311799651097396464377723858909261474710812858063665151628506070959937347627"
    "275851899282446151613688832986623643166576916545974660203523412854521672957349134716034187"
    "586798702358180168471635443626177543811614397048963093383644738803425453347827659425941599"
    "260242926776721855571408975646106864867688734105951682537834904812715247258315613008475070"
    "905740733513696526664356906800610754349802581407714837019774399215488180914235272211953997"
    "61199951171875",
    716,
    "1.9042271010497776332001708801330615626361603928163568681486870742545599127089701284279280"
    "984207169692023217132003028268730372844501516214455924208987366001434939743136403660684329"
    "034900795682515039218239607769961070813600340587606383062269871871235739510797301364363003"
    "117996510973964643777238589092614747108128580636651516285060709599373476272758518992824461"
    "516136888329866236431665769165459746602035234128545216729573491347160341875867987023581801"
    "684716354436261775438116143970489630933836447388034254533478276594259415992602429267767218"
    "555714089756461068648676887341059516825378349048127152472583156130084750709057407335136965"
    "2666435690680061075434980258140771483701977439921548818091423527221195399761199951171875e-"
    "286"
  },
  {
    2.9003728382951163e-286,
    1000,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000002900372838295116300133638652361706238128613374409059474358941441136834065"
    "355513724427153759145165568068731597325059613096888192036043781625220722568191932257798370"
    "805328126288578604126089293435096994847057184959678498590349546873362607386039910958079726"
    "915480506740564727937744213886081662414338883958167769615634834449520820533245514148363043"
    "893443796967648502707271130763132701946276020956850825940639071222081359496315692610547887"
    "294288506372101623606589888698780133492591674630318264384897830299365082980617116182683763"
    "019320636233705491974210920522838275477083252004563254028764269886961406388361787734609729"
    "556483215835413375983163021711842878443424060979670008305676166815434413592811324633657932"
    "281494140625",
    714,
    "2.9003728382951163001336386523617062381286133744090594743589414411368340653555137244271537"
    "591451655680687315973250596130968881920360437816252207225681919322577983708053281262885786"
    "041260892934350969948470571849596784985903495468733626073860399109580797269154805067405647"
    "279377442138860816624143388839581677696156348344495208205332455141483630438934437969676485"
    "027072711307631327019462760209568508259406390712220813594963156926105478872942885063721016"
    "236065898886987801334925916746303182643848978302993650829806171161826837630193206362337054"
    "919742109205228382754770832520045632540287642698869614063883617877346097295564832158354133"
    "75983163021711842878443424060979670008305676166815434413592811324633657932281494140625e-28"
    "6"
  },
  {
    6.030230011480532e-286,
    999,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000006030230011480532336086714061366629154844710765750815164166009510289390408"
    "928352130382822822659766062818235069111539868695121950322320668137817151073876683000486770"
    "999433168325930329507270225333839962048307808736870336786474106255162862393383020101537774"
    "908499396139970912718835720263245510686995538097893670858302693490789150845596176334231938"
    "129908970145201797688554174500309060673869117866459629299845875451222894126638524480070648"
    "366890280832688105349742657556076984650423537189166528748077671084144704218518914901055445"
    "765459659578607346221279870994992550342662655560173662144012241644703804158632807596075223"
    "431027561864483265724367888001009670016871045645619533597274128378096236247074557468295097"
    "35107421875",
    713,
    "6.0302300114805323360867140613666291548447107657508151641660095102893904089283521303828228"
    "226597660628182350691115398686951219503223206681378171510738766830004867709994331683259303"
    "295072702253338399620483078087368703367864741062551628623933830201015377749084993961399709"
    "127188357202632455106869955380978936708583026934907891508455961763342319381299089701452017"
    "976885541745003090606738691178664596292998458754512228941266385244800706483668902808326881"
    "053497426575560769846504235371891665287480776710841447042185189149010554457654596595786073"
    "462212798709949925503426626555601736621440122416447038041586328075960752234310275618644832"
    "6572436788800100967001687104564561953359727412837809623624707455746829509735107421875e-286"
  },
  {
    1.2707290656410452e-285,
    997,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000012707290656410451972148714179141781588967973845926807547334471277675980850"
    "408554878248054803892782582018822253418256902829942626257005944569498834870031877782525851"
    "878453560585742513029967130042607138362997785889553873374682190314949102440662567707200467"
    "372036967286727601923707111389242837687724759407856937248266989860466008993894741453396502"
    "146202005519277127551424134439683925893021558019255827189881118691971239229665920665706074"
    "589881220195814966137824984006613325593667427422619760789724850783643890523872738445881101"
    "341725133790825008856838582077932509180210169815958254867618135914649572471737779074176755"
    "172573523955032354919765037037450328159638720514623892016668749160857032620697282254695892"
    "333984375",
    712,
    "1.2707290656410451972148714179141781588967973845926807547334471277675980850408554878248054"
    "803892782582018822253418256902829942626257005944569498834870031877782525851878453560585742"
    "513029967130042607138362997785889553873374682190314949102440662567707200467372036967286727"
    "601923707111389242837687724759407856937248266989860466008993894741453396502146202005519277"
    "127551424134439683925893021558019255827189881118691971239229665920665706074589881220195814"
    "966137824984006613325593667427422619760789724850783643890523872738445881101341725133790825"
    "008856838582077932509180210169815958254867618135914649572471737779074176755172573523955032"
    "354919765037037450328159638720514623892016668749160857032620697282254695892333984375e-285"
  },
  {
    3.3470583063356332e-285,
    998,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000033470583063356332108498766757166506385353764031783485196998561866179457340"
    "599432256217213352642082359348776006561937984312164867983255432833872617315767109984107363"
    "501898855634912640532492943310363892141101000040699709755413970624032998366918838727201015"
    "912757725678364994741447092697451640335531433818659809481148134509563323398425378499328625"
    "730378201514987590090742627365532394510810911008494010157265556059636318908537716658826575"
    "242739741476951917424048116264826783100622927498570470307345841986086200183468126440895261"
    "027969581202720428111002643330429924059661140864805033477404803079770373937677023271704288"
    "608613344385210559976398751938405173696118664965805355180402530512751013702654745429754257"
    "2021484375",
    713,
    "3.3470583063356332108498766757166506385353764031783485196998561866179457340599432256217213"
    "352642082359348776006561937984312164867983255432833872617315767109984107363501898855634912"
    "640532492943310363892141101000040699709755413970624032998366918838727201015912757725678364"
    "994741447092697451640335531433818659809481148134509563323398425378499328625730378201514987"
    "590090742627365532394510810911008494010157265556059636318908537716658826575242739741476951"
    "917424048116264826783100622927498570470307345841986086200183468126440895261027969581202720"
    "428111002643330429924059661140864805033477404803079770373937677023271704288608613344385210"
    "5599763987519384051736961186649658053551804025305127510137026547454297542572021484375e-285"
  },
  {
    6.5966578537292035e-285,
    997,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000065966578537292035157422433162339949684328657517781425278937040263737397772"
    "577187341772038360860221866728139740168744657011708467799022498428904836700880711205803008"
    "034428677795267551261459571487880448925447707365536317882093730294389117373968872265253444"
    "767942288787748068170159406223098141031473011581809098490153784813123115801307697707920891"
    "837072640679333216386798989999483042211053745064747961945094371289546480468588624655464858"
    "151737047145695739493473550732927723664462733063194768431159909034458907314518955367901770"
    "778500947430433396811313170358968075237471306363006317588661654766564093933011047378369831"
    "611942979426305500815782254835380854001782230228668880262464513375419983276515267789363861"
    "083984375",
    712,
    "6.5966578537292035157422433162339949684328657517781425278937040263737397772577187341772038"
    "360860221866728139740168744657011708467799022498428904836700880711205803008034428677795267"
    "551261459571487880448925447707365536317882093730294389117373968872265253444767942288787748"
    "068170159406223098141031473011581809098490153784813123115801307697707920891837072640679333"
    "216386798989999483042211053745064747961945094371289546480468588624655464858151737047145695"
    "739493473550732927723664462733063194768431159909034458907314518955367901770778500947430433"
    "396811313170358968075237471306363006317588661654766564093933011047378369831611942979426305"
    "500815782254835380854001782230228668880262464513375419983276515267789363861083984375e-285"
  },
  {
    7.427398668320939e-285,
    995,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000074273986683209386002029183146497792358537735412418076229677375034053108781"
    "538417130596628794716377703048792043944787439164083049732926376919766937265457021089676010"
    "081722203913305906746813299677405365107619022696735894410742592300656279570382931855202352"
    "919060319481153813978290928253191543124720263401838189446826111111156145441474169706609373"
    "281586299060755244822009110650534665364827404192153653867280756103054142218360222093458106"
    "173489557405135404872456559194372267970465578962160683119840868995665182041483933327146440"
    "118387784509668794940763056593474207263889043578452279585749951569369127133062372556281897"
    "782699606993494880824110638641864303065100892872446960209925626017479771689977496862411499"
    "0234375",
    710,
    "7.4273986683209386002029183146497792358537735412418076229677375034053108781538417130596628"
    "794716377703048792043944787439164083049732926376919766937265457021089676010081722203913305"
    "906746813299677405365107619022696735894410742592300656279570382931855202352919060319481153"
    "813978290928253191543124720263401838189446826111111156145441474169706609373281586299060755"
    "244822009110650534665364827404192153653867280756103054142218360222093458106173489557405135"
    "404872456559194372267970465578962160683119840868995665182041483933327146440118387784509668"
    "794940763056593474207263889043578452279585749951569369127133062372556281897782699606993494"
    "8808241106386418643030651008928724469602099256260174797716899774968624114990234375e-285"
  },
  {
    2.5312725644714646e-284,
    994,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000253127256447146459629479277438058098046188843976178214375723184585507431180"
    "848627052892059439660381190032909589290532205874362493462899587538183794311484367769029075"
    "747596682977338151350446246593552098587417080875785146924874009337685002273425338234764049"
    "746755196726829036283855389169626596626336915717363443186177773028468002427637656813657782"
    "095383164365461668044042733324278614600752670646079645747018634289655624083620110521953699"
    "389815696244858788510527465075072050708829210926602831437546879251441709037514462450585484"
    "637061375098642942286941572809016802017675262755381602721660037317089179663865389175421046"
    "586315246560167775846890651442682066021379729291296127070831634142678012722171843051910400"
    "390625",
    710,
    "2.5312725644714645962947927743805809804618884397617821437572318458550743118084862705289205"
    "943966038119003290958929053220587436249346289958753818379431148436776902907574759668297733"
    "815135044624659355209858741708087578514692487400933768500227342533823476404974675519672682"
    "903628385538916962659662633691571736344318617777302846800242763765681365778209538316436546"
    "166804404273332427861460075267064607964574701863428965562408362011052195369938981569624485"
    "878851052746507507205070882921092660283143754687925144170903751446245058548463706137509864"
    "294228694157280901680201767526275538160272166003731708917966386538917542104658631524656016"
    "7775846890651442682066021379729291296127070831634142678012722171843051910400390625e-284"
  },
  {
    4.7441324100905746e-284,
    994,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000474413241009057455545742873020846912803820795971935816315591113366331582329"
    "509971297928949808058941930664947024126350169270196995072745313996409498284229154731375549"
    "048678636733572900612861491971019072075638834606056231951943868820381043495128969548495024"
    "328077902850589424563896572146049347495861624467613219291137449097446855966560461497784520"
    "971287868115519659684239948033351708868051182201085858324140277023786909650546821132637590"
    "793226042392427342459479251395153348234719604672659565852875863171207615153939055490446947"
    "865662706945150283306277973264345335504598562037303734193017156401514386707515682692206984"
    "106501442412970154277186018963546023944039247179557390425574947556697225081734359264373779"
    "296875",
    710,
    "4.7441324100905745554574287302084691280382079597193581631559111336633158232950997129792894"
    "980805894193066494702412635016927019699507274531399640949828422915473137554904867863673357"
    "290061286149197101907207563883460605623195194386882038104349512896954849502432807790285058"
    "942456389657214604934749586162446761321929113744909744685596656046149778452097128786811551"
    "965968423994803335170886805118220108585832414027702378690965054682113263759079322604239242"
    "734245947925139515334823471960467265956585287586317120761515393905549044694786566270694515"
    "028330627797326434533550459856203730373419301715640151438670751568269220698410650144241297"
    "0154277186018963546023944039247179557390425574947556697225081734359264373779296875e-284"
  },
  {
    7.352811059453878e-284,
    992,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000735281105945387824823657738971686079656276703283988537435722942874894629410"
    "400049236090868524904221483780423221265352794066364321977056316193229087193345799980407047"
    "318859371585571006327608060193056679922161895138902345085771102988084674734794978126694767"
    "146579114049889801890230001840919242737520380012954161991335366879133367705657730561047169"
    "112644611788011278170299217145724065948735254510907592500616061752976888536427415063374341"
    "194168250819599742158320993808193105179075468474640923879534600345563234330664743865595922"
    "943046902701324836502573710829659557876474079365954015320683991715767424110859991121944220"
    "032704692638637963279470617377073838620213650458048732644344980258210853207856416702270507"
    "8125",
    708,
    "7.3528110594538782482365773897168607965627670328398853743572294287489462941040004923609086"
    "852490422148378042322126535279406636432197705631619322908719334579998040704731885937158557"
    "100632760806019305667992216189513890234508577110298808467473479497812669476714657911404988"
    "980189023000184091924273752038001295416199133536687913336770565773056104716911264461178801"
    "127817029921714572406594873525451090759250061606175297688853642741506337434119416825081959"
    "974215832099380819310517907546847464092387953460034556323433066474386559592294304690270132"
    "483650257371082965955787647407936595401532068399171576742411085999112194422003270469263863"
    "79632794706173770738386202136504580487326443449802582108532078564167022705078125e-284"
  },
  {
    1.2773195914029405e-283,
    990,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000001277319591402940483999712136501984667801236065235048806444537810208068087817"
    "829275181212790228225665073254762457032074167125945963486726250164598954251991932062236692"
    "745745907816152797925582528203979735159648420370516008716769106316730250486845936703583635"
    "694598574992126846759388865096988008602912529868185336100299108055371952565779056254804462"
    "551389480809263139877181540239827869316534678386403409720339026305076449997313311765579574"
    "003533288958339344621641262152896488412328146841515549505266094253645456458030149584498583"
    "883878330615551788311376615569801461643158576202752253276793725354647037131210454236435093"
    "465752334159968626474049856438694212827733612090766804336716688794695073738694190979003906"
    "25",
    707,
    "1.2773195914029404839997121365019846678012360652350488064445378102080680878178292751812127"
    "902282256650732547624570320741671259459634867262501645989542519919320622366927457459078161"
    "527979255825282039797351596484203705160087167691063167302504868459367035836356945985749921"
    "268467593888650969880086029125298681853361002991080553719525657790562548044625513894808092"
    "631398771815402398278693165346783864034097203390263050764499973133117655795740035332889583"
    "393446216412621528964884123281468415155495052660942536454564580301495844985838838783306155"
    "517883113766155698014616431585762027522532767937253546470371312104542364350934657523341599"
    "6862647404985643869421282773361209076680433671668879469507373869419097900390625e-283"
  },
  {
    3.1336514683809336e-283,
    991,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000003133651468380933582364201405690516581105746242498610056637907764625652226456"
    "555415141340665125388584830097774288102121532719018587676854488355808095725570494685970999"
    "130667700552896985066640196158390384755347105355795137683512491796292640286340784407826504"
    "261360116668998806692920326410697970081470553311956742163565291948891903034583447144949763"
    "401577373388034436986851036531041736832716725000154529591297240790424338568500905683261889"
    "088144262991753471099460781687848132567528325818282317505927988919104977176535125982092562"
    "126152163332546148270577877121959807562829701730894716563197686457689793871379392247928709"
    "120246303744498721385387058298586846638108429548307855283528766676681698299944400787353515"
    "625",
    708,
    "3.1336514683809335823642014056905165811057462424986100566379077646256522264565554151413406"
    "651253885848300977742881021215327190185876768544883558080957255704946859709991306677005528"
    "969850666401961583903847553471053557951376835124917962926402863407844078265042613601166689"
    "988066929203264106979700814705533119567421635652919488919030345834471449497634015773733880"
    "344369868510365310417368327167250001545295912972407904243385685009056832618890881442629917"
    "534710994607816878481325675283258182823175059279889191049771765351259820925621261521633325"
    "461482705778771219598075628297017308947165631976864576897938713793922479287091202463037444"
    "98721385387058298586846638108429548307855283528766676681698299944400787353515625e-283"
  },
  {
    4.624339948076099e-283,
    990,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000004624339948076098665175804885959189867025753248155197523423615068565730024182"
    "113965787802278458523387479124049634922415502894962122620968309353505083636846805273805862"
    "071611545196722839517377749398114834627280834441777428783029826723607606656979273452995057"
    "139008359627554000935187030159957139984855268085241021567271742343195487623036364350704462"
    "319528994542283472195164243165427286927827946326324892850537364653645448662928503889332720"
    "476917768510116269567142557142973316677614446620377585232395445207964980626580356717625140"
    "533962546352624989961931110902588588758282909374019543277991567132082147774864908128579658"
    "562342391043382460857663733513396639783638668964394283222851811387954512611031532287597656"
    "25",
    707,
    "4.6243399480760986651758048859591898670257532481551975234236150685657300241821139657878022"
    "784585233874791240496349224155028949621226209683093535050836368468052738058620716115451967"
    "228395173777493981148346272808344417774287830298267236076066569792734529950571390083596275"
    "540009351870301599571399848552680852410215672717423431954876230363643507044623195289945422"
    "834721951642431654272869278279463263248928505373646536454486629285038893327204769177685101"
    "162695671425571429733166776144466203775852323954452079649806265803567176251405339625463526"
    "249899619311109025885887582829093740195432779915671320821477748649081285796585623423910433"
    "8246085766373351339663978363866896439428322285181138795451261103153228759765625e-283"
  },
  {
    8.894087772740341e-283,
    989,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000008894087772740341347805442246955158536136060213214647927655374856057226385535"
    "490010817438841387381081057086944476526633185508976830686400273575138150735473070565202327"
    "083734754289519859380869209653817053628562931552372597405794368429177559397287482636064879"
    "735202907026717819924195853075278831851189896959260779613681724950714153558562125071160140"
    "041412588933795776394239933049382324414558667115560425786369347695780938584808156024325242"
    "043096442949805033001474195239588995180541026614271779826149866978171627988590381787070670"
    "292743666765992474172041115451984467344953683138400045170040908790923436569293770430710009"
    "650254127241328424225196370041609537158768310444265786651740768320451024919748306274414062"
    "5",
    706,
    "8.8940877727403413478054422469551585361360602132146479276553748560572263855354900108174388"
    "413873810810570869444765266331855089768306864002735751381507354730705652023270837347542895"
    "198593808692096538170536285629315523725974057943684291775593972874826360648797352029070267"
    "178199241958530752788318511898969592607796136817249507141535585621250711601400414125889337"
    "957763942399330493823244145586671155604257863693476957809385848081560243252420430964429498"
    "050330014741952395889951805410266142717798261498669781716279885903817870706702927436667659"
    "924741720411154519844673449536831384000451700409087909234365692937704307100096502541272413"
    "284242251963700416095371587683104442657866517407683204510249197483062744140625e-283"
  },
  {
    2.3953652083727512e-282,
    988,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000023953652083727512388926071411736651094684359629139545046331701056101374156295"
    "569337074287726281108064839092019722035425712564337078891872125343041100049783302647091843"
    "806474890630383680366777926439903799649329468277990705395909920541788815631955088422371660"
    "022548387789414322879137643336776999777150826689368702734206841637889564941327849417969580"
    "798865985391354437385154150217050073066367310703953519083722746947699236233991091448525170"
    "238716635753128040232958011984837624799412519247652005406902117824401817579020483417060068"
    "752200612802921994964281796368082442449256349318801912013921428220611600517618709876720043"
    "379713076738404219858403977956304139210111692789677284576299598484183661639690399169921875",
    706,
    "2.3953652083727512388926071411736651094684359629139545046331701056101374156295569337074287"
    "726281108064839092019722035425712564337078891872125343041100049783302647091843806474890630"
    "383680366777926439903799649329468277990705395909920541788815631955088422371660022548387789"
    "414322879137643336776999777150826689368702734206841637889564941327849417969580798865985391"
    "354437385154150217050073066367310703953519083722746947699236233991091448525170238716635753"
    "128040232958011984837624799412519247652005406902117824401817579020483417060068752200612802"
    "921994964281796368082442449256349318801912013921428220611600517618709876720043379713076738"
    "404219858403977956304139210111692789677284576299598484183661639690399169921875e-282"
  },
  {
    3.761567466326268e-282,
    985,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000037615674663262681818855898106340652384824488135624420876733253077105060946198"
    "861732627493193519901047247753085151645061052614831077461787550378952294019131437758380555"
    "817601695966731928617166028263195969591766026279481616008935858411029622674719705669607199"
    "726168155240523302904487611579487237192838928096033934801529226963821818326771744159065697"
    "180266351038889725789977121282586495071803503310337622258942765229731736751373980228545367"
    "332367240256732267829774012143595967393133021823905661846519903267598683144976498528075164"
    "459010510716339007682295712921936766580692236547094510725167610756795304960818801634786246"
    "087789365986160581038975154763000147609318065712591394600394778535701334476470947265625",
    703,
    "3.7615674663262681818855898106340652384824488135624420876733253077105060946198861732627493"
    "193519901047247753085151645061052614831077461787550378952294019131437758380555817601695966"
    "731928617166028263195969591766026279481616008935858411029622674719705669607199726168155240"
    "523302904487611579487237192838928096033934801529226963821818326771744159065697180266351038"
    "889725789977121282586495071803503310337622258942765229731736751373980228545367332367240256"
    "732267829774012143595967393133021823905661846519903267598683144976498528075164459010510716"
    "339007682295712921936766580692236547094510725167610756795304960818801634786246087789365986"
    "160581038975154763000147609318065712591394600394778535701334476470947265625e-282"
  },
  {
    1.2928212167160486e-281,
    985,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000129282121671604856472890507259207739874861836069414361195057298546848250710618"
    "797414084049769873325428052026475209722120894427905829017312531660824109185257625493974124"
    "746918871520377551204868603233138855433546350713748211658517353171708928420997443975709481"
    "547631303263398724665746621742408228790498004979928517309236043210339586064195965826264730"
    "306788592310470685037034319253555665752608782580961488429909489306613884615945556331126875"
    "486164601415077624257882874367569017493013432167566077044569124299536340949294173450937422"
    "124241084868415551323836836579618717002473556037170692671052731601410246362458410305421679"
    "975388848942695423067389883463190992127388374323404551802241257973946630954742431640625",
    704,
    "1.2928212167160485647289050725920773987486183606941436119505729854684825071061879741408404"
    "976987332542805202647520972212089442790582901731253166082410918525762549397412474691887152"
    "037755120486860323313885543354635071374821165851735317170892842099744397570948154763130326"
    "339872466574662174240822879049800497992851730923604321033958606419596582626473030678859231"
    "047068503703431925355566575260878258096148842990948930661388461594555633112687548616460141"
    "507762425788287436756901749301343216756607704456912429953634094929417345093742212424108486"
    "841555132383683657961871700247355603717069267105273160141024636245841030542167997538884894"
    "2695423067389883463190992127388374323404551802241257973946630954742431640625e-281"
  },
  {
    2.2480352182123753e-281,
    982,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000224803521821237528081365449403889786634144528684028071721643530480871593182406"
    "991088523084109355277990423722607479167570828666405584023758943395449623547786265060474719"
    "000797492782936683920058858257998717582822367577990545492185684071261692556135231155476178"
    "082232105070475071483579364503128180913173446500505425910316207702954217797026532255821206"
    "910785854698054868524203836021409630916609905079528396656843124885091925936276146801902962"
    "301794667900084167998719481932390233466300406339130835304539931159145009172042189525782968"
    "134988938499671626068560607947883035386797154297720090004196560429993659113890327918951122"
    "551671928810933831476652227485316062853136671872977814246041816659271717071533203125",
    701,
    "2.2480352182123752808136544940388978663414452868402807172164353048087159318240699108852308"
    "410935527799042372260747916757082866640558402375894339544962354778626506047471900079749278"
    "293668392005885825799871758282236757799054549218568407126169255613523115547617808223210507"
    "047507148357936450312818091317344650050542591031620770295421779702653225582120691078585469"
    "805486852420383602140963091660990507952839665684312488509192593627614680190296230179466790"
    "008416799871948193239023346630040633913083530453993115914500917204218952578296813498893849"
    "967162606856060794788303538679715429772009000419656042999365911389032791895112255167192881"
    "0933831476652227485316062853136671872977814246041816659271717071533203125e-281"
  },
  {
    3.154850939942071e-281,
    983,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000315485093994207126392720592764003333168825472939840263152474766394932559633481"
    "401213456905267591366748121685127366607186078984663434814626878761946743591353198878185556"
    "202712852448632736187785808850221879652949417246693530261947587649437530428425424208486244"
    "001935716761281663696262512957905776664489604940787095546497473357872213460835120581938813"
    "017269185088133887540329540019387686130807401092927971033628483700107282777002447037363112"
    "380936616653230090833282091176580530259321841919447337344677998409580350026291730548714720"
    "245346490759597607641662215910884925487059406888174771174585898341218787818889729662339730"
    "6560923894227742617663131411366557437721031172117580609892684151418507099151611328125",
    702,
    "3.1548509399420712639272059276400333316882547293984026315247476639493255963348140121345690"
    "526759136674812168512736660718607898466343481462687876194674359135319887818555620271285244"
    "863273618778580885022187965294941724669353026194758764943753042842542420848624400193571676"
    "128166369626251295790577666448960494078709554649747335787221346083512058193881301726918508"
    "813388754032954001938768613080740109292797103362848370010728277700244703736311238093661665"
    "323009083328209117658053025932184191944733734467799840958035002629173054871472024534649075"
    "959760764166221591088492548705940688817477117458589834121878781888972966233973065609238942"
    "27742617663131411366557437721031172117580609892684151418507099151611328125e-281"
  },
  {
    6.071363984983044e-281,
    982,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000607136398498304414319506668851861770752265151539926629076264973010953983907324"
    "466041869432033823459140388847323282265478315251107842926851637284944473721522256210665632"
    "511691427823143735225088177817782851258604105880246542796056822782023655166993164757976638"
    "529752050918411004666702491746580352817671265382760485469067605397173946721621406254985045"
    "767233104233430770581482344127010001310021576891807049979818732256016778786725558225608403"
    "941078353606559612621001943710817919983755745476941534320115237355705280174943756986505561"
    "652164232362998125337108087881313272243305213974456813336047435231192311584703601500865988"
    "451180860293358890540549489846723710069942465950010301867223461158573627471923828125",
    701,
    "6.0713639849830441431950666885186177075226515153992662907626497301095398390732446604186943"
    "203382345914038884732328226547831525110784292685163728494447372152225621066563251169142782"
    "314373522508817781778285125860410588024654279605682278202365516699316475797663852975205091"
    "841100466670249174658035281767126538276048546906760539717394672162140625498504576723310423"
    "343077058148234412701000131002157689180704997981873225601677878672555822560840394107835360"
    "655961262100194371081791998375574547694153432011523735570528017494375698650556165216423236"
    "299812533710808788131327224330521397445681333604743523119231158470360150086598845118086029"
    "3358890540549489846723710069942465950010301867223461158573627471923828125e-281"
  },
  {
    2.1910202165183368e-280,
    981,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000002191020216518336796103114473432901746322697938270186945371351301396824908535678"
    "593228801202439354195354555298866455116158899738801666640075186148021138255908028754924203"
    "055744169368398927068607311923093337658489709908439585083718902707887189552668448593889249"
    "735204217996658815646772179955571823374833497551591956309240154132915496583793843897155454"
    "758741822546081513346145952348420009435809153123845362687245912248059666995383487273354828"
    "366695696929216032838347225549591313620931641745286926091481043349304738854389188835525615"
    "948363008632224664238662463784686332242084676173707932159699210038441526890720252029751413"
    "67168786824130511190159598977575695967511849870934526052224100567400455474853515625",
    701,
    "2.1910202165183367961031144734329017463226979382701869453713513013968249085356785932288012"
    "024393541953545552988664551161588997388016666400751861480211382559080287549242030557441693"
    "683989270686073119230933376584897099084395850837189027078871895526684485938892497352042179"
    "966588156467721799555718233748334975515919563092401541329154965837938438971554547587418225"
    "460815133461459523484200094358091531238453626872459122480596669953834872733548283666956969"
    "292160328383472255495913136209316417452869260914810433493047388543891888355256159483630086"
    "322246642386624637846863322420846761737079321596992100384415268907202520297514136716878682"
    "4130511190159598977575695967511849870934526052224100567400455474853515625e-280"
  },
  {
    4.3370603492214e-280,
    980,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000004337060349221399924660510986356257892269790420502347908835422511841116263760317"
    "988870371022660947524084318977783088056734831661091131387005107336708003217862286442962963"
    "104808766013182276600273231744109599629523231073474743346300893802816394194037660492099236"
    "546107192724017030062164767337377700574128707882663321060086396869899652135274086981645091"
    "967175659794850779480003111895553824321087854495212577457398595402036090648629548430689644"
    "857400023575624192252226181173523607099393731967176783142832496258252948756690113046205630"
    "355933690645526817739372794594388572468529067558242499477463165957541382915860867194951359"
    "6104297734334686807656652381430992566229733198301943275509984232485294342041015625",
    700,
    "4.3370603492213999246605109863562578922697904205023479088354225118411162637603179888703710"
    "226609475240843189777830880567348316610911313870051073367080032178622864429629631048087660"
    "131822766002732317441095996295232310734747433463008938028163941940376604920992365461071927"
    "240170300621647673373777005741287078826633210600863968698996521352740869816450919671756597"
    "948507794800031118955538243210878544952125774573985954020360906486295484306896448574000235"
    "756241922522261811735236070993937319671767831428324962582529487566901130462056303559336906"
    "455268177393727945943885724685290675582424994774631659575413829158608671949513596104297734"
    "334686807656652381430992566229733198301943275509984232485294342041015625e-280"
  },
  {
    6.968471572869319e-280,
    980,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000006968471572869318512895878527464490889745995933947468338899191644265737622764483"
    "380773896231028946301101354121009652116589677772323678127928668665427120766382896068444585"
    "878929476759055769794509853336509082254542351258556803112976410432769807463403605097818593"
    "159599336715606881192650863075807426528516907258685085390493758905299511165343506074808910"
    "456780543266574359886133871760507672067140680759980765754218311821351038664767580912988223"
    "632453672861098702524281622665167193050254697249693893013988680158710797262499451731496172"
    "625833737082375357463190498230523012813555991495326869696530083748073407599377627499057902"
    "9055978506925292033909987407906166508790406222129121260877582244575023651123046875",
    700,
    "6.9684715728693185128958785274644908897459959339474683388991916442657376227644833807738962"
    "310289463011013541210096521165896777723236781279286686654271207663828960684445858789294767"
    "590557697945098533365090822545423512585568031129764104327698074634036050978185931595993367"
    "156068811926508630758074265285169072586850853904937589052995111653435060748089104567805432"
    "665743598861338717605076720671406807599807657542183118213510386647675809129882236324536728"
    "610987025242816226651671930502546972496938930139886801587107972624994517314961726258337370"
    "823753574631904982305230128135559914953268696965300837480734075993776274990579029055978506"
    "925292033909987407906166508790406222129121260877582244575023651123046875e-280"
  },
  {
    1.473331115243779e-279,
    979,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000014733311152437789540227053092678323460332933852257475188822441222074344436589917"
    "689823247953665290710696624690991611809635867231547228216053927389433947999419316933052062"
    "593639751923591813024934073088290349330762108956128527803667315384993505450499669750869202"
    "270344601516626630114787421766985497027790072719001808892865490797177322439918350145063046"
    "375849294907349943601734582058091677850466600549179567598052592528981026904494706085683657"
    "881489962319474536509309542947708760242719655622242489529407657845641312452150759617245693"
    "882513311297300622084838131937035382857449276854843252431117615501846144503730639485737457"
    "925483068648203400069389217127511839951041483498528350537526421248912811279296875",
    700,
    "1.4733311152437789540227053092678323460332933852257475188822441222074344436589917689823247"
    "953665290710696624690991611809635867231547228216053927389433947999419316933052062593639751"
    "923591813024934073088290349330762108956128527803667315384993505450499669750869202270344601"
    "516626630114787421766985497027790072719001808892865490797177322439918350145063046375849294"
    "907349943601734582058091677850466600549179567598052592528981026904494706085683657881489962"
    "319474536509309542947708760242719655622242489529407657845641312452150759617245693882513311"
    "297300622084838131937035382857449276854843252431117615501846144503730639485737457925483068"
    "648203400069389217127511839951041483498528350537526421248912811279296875e-279"
  },
  {
    2.66044060949483e-279,
    971,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000026604406094948299284840023312889151895045655703225518408579970423545100011582978"
    "514470567087920738321442927530728543237114459190980071924595778114172961293166843965348287"
    "532775482096395427042896234914919972683569811845927352458649219997270927226435832857204041"
    "623139257269006142406123561344190322624285588439618233913287064205337047968262378349013995"
    "361823268143063235754023395627339613127590917817943514492812352700718204492686590850424955"
    "549214973042837434213790076783891838501409937789616117232173259445897443207834377147143620"
    "375601946213583247296053366325035705814006820872606451086884513265855542938243808692453580"
    "2237305967208190452841033281990330838784331035640207119286060333251953125",
    692,
    "2.6604406094948299284840023312889151895045655703225518408579970423545100011582978514470567"
    "087920738321442927530728543237114459190980071924595778114172961293166843965348287532775482"
    "096395427042896234914919972683569811845927352458649219997270927226435832857204041623139257"
    "269006142406123561344190322624285588439618233913287064205337047968262378349013995361823268"
    "143063235754023395627339613127590917817943514492812352700718204492686590850424955549214973"
    "042837434213790076783891838501409937789616117232173259445897443207834377147143620375601946"
    "213583247296053366325035705814006820872606451086884513265855542938243808692453580223730596"
    "7208190452841033281990330838784331035640207119286060333251953125e-279"
  },
  {
    5.683973436284207e-279,
    976,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000056839734362842070796608423358963762521942847688739811715193332262874402507007578"
    "228137833409228851937575142988133388681897276244108465150313548284187815791299849102347119"
    "715319356793833082027918260374748768515780444991433596564874088121580880010149173444397592"
    "879832160925139100533234002565933700504350023969679949986347831174253791387681379454495092"
    "330365305712925878132005668316354328176029389998813412358306819750991589464158840440187540"
    "370845417980702155638277291777631628526639882289293616569552317153915953591994318932060391"
    "110575065116088021731477788062053688563492814750780987212117930534020118387378410418859756"
    "797404092331605782924669070403089223518511285959675660706125199794769287109375",
    697,
    "5.6839734362842070796608423358963762521942847688739811715193332262874402507007578228137833"
    "409228851937575142988133388681897276244108465150313548284187815791299849102347119715319356"
    "793833082027918260374748768515780444991433596564874088121580880010149173444397592879832160"
    "925139100533234002565933700504350023969679949986347831174253791387681379454495092330365305"
    "712925878132005668316354328176029389998813412358306819750991589464158840440187540370845417"
    "980702155638277291777631628526639882289293616569552317153915953591994318932060391110575065"
    "116088021731477788062053688563492814750780987212117930534020118387378410418859756797404092"
    "331605782924669070403089223518511285959675660706125199794769287109375e-279"
  },
  {
    1.4089452631685327e-278,
    976,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000140894526316853269684332430741095179680058969564052435377386295567255430891923404"
    "569767456183925094563844213295122043611679935471391755079980979501871944052554986009587710"
    "715393810226977108801992230611058211129108948552569189907327071476906753177066757260195759"
    "536819288900310550921239838805599316023954250349483506722832453795638182481885774321780697"
    "855175221447135604212500292255605581080737160756336541704916605654622573208077336936537681"
    "601495287937831361476225032018916859215383170855286140431926630333405822225809755086425841"
    "340353130483280917725185195942242116357590456907071250675585191116081834231777596928361527"
    "719075306480008642654648867933132806891383115299731798586435616016387939453125",
    698,
    "1.4089452631685326968433243074109517968005896956405243537738629556725543089192340456976745"
    "618392509456384421329512204361167993547139175507998097950187194405255498600958771071539381"
    "022697710880199223061105821112910894855256918990732707147690675317706675726019575953681928"
    "890031055092123983880559931602395425034948350672283245379563818248188577432178069785517522"
    "144713560421250029225560558108073716075633654170491660565462257320807733693653768160149528"
    "793783136147622503201891685921538317085528614043192663033340582222580975508642584134035313"
    "048328091772518519594224211635759045690707125067558519111608183423177759692836152771907530"
    "6480008642654648867933132806891383115299731798586435616016387939453125e-278"
  },
  {
    2.2755564362864927e-278,
    973,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000227555643628649269402817734805183107904798614676370506960981914851328415362342551"
    "959688942020547681275943405052471207178431622061755181896773453067772352444724262922219671"
    "446487226142118896359570378668235229095798116631696871576847687837771732367655827591311062"
    "549641717133217902876406824174212928756933047798953864072230789811106381721434276949076044"
    "338371290870631124853148741448273362510869730859167323400369922515612597339675515620741030"
    "786110317330091445693562528485582028687881785896904192513842055970675225542253236829083202"
    "359917015583355637858258854105483583225892878121852955409331433196899194011282853032812443"
    "010830497437627730945405107483288202618876283622739720158278942108154296875",
    695,
    "2.2755564362864926940281773480518310790479861467637050696098191485132841536234255195968894"
    "202054768127594340505247120717843162206175518189677345306777235244472426292221967144648722"
    "614211889635957037866823522909579811663169687157684768783777173236765582759131106254964171"
    "713321790287640682417421292875693304779895386407223078981110638172143427694907604433837129"
    "087063112485314874144827336251086973085916732340036992251561259733967551562074103078611031"
    "733009144569356252848558202868788178589690419251384205597067522554225323682908320235991701"
    "558335563785825885410548358322589287812185295540933143319689919401128285303281244301083049"
    "7437627730945405107483288202618876283622739720158278942108154296875e-278"
  },
  {
    3.821906858286435e-278,
    973,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000382190685828643527239677479313436165621657209781173761641840447788828198561093401"
    "360596513533007600193800457963875686854151814858111036445511242704770934444137666535548776"
    "414562807926044830228719368181619572195896346535421406710261927177639840972811133467472976"
    "053742964274022009430049671629851932890300242977032830060419466177625626933355434946418113"
    "486593200998002484261631559385150974748579850886791518129421085948180303284306883294509926"
    "394689017938918992450263082868445216830695420503163246032988791848623232836381428857315417"
    "646392893556327776291022365585138473611071423840023808221102670757369528894829189660777104"
    "364548668994317476597365746052831735057875306438290863297879695892333984375",
    695,
    "3.8219068582864352723967747931343616562165720978117376164184044778882819856109340136059651"
    "353300760019380045796387568685415181485811103644551124270477093444413766653554877641456280"
    "792604483022871936818161957219589634653542140671026192717763984097281113346747297605374296"
    "427402200943004967162985193289030024297703283006041946617762562693335543494641811348659320"
    "099800248426163155938515097474857985088679151812942108594818030328430688329450992639468901"
    "793891899245026308286844521683069542050316324603298879184862323283638142885731541764639289"
    "355632777629102236558513847361107142384002380822110267075736952889482918966077710436454866"
    "8994317476597365746052831735057875306438290863297879695892333984375e-278"
  },
  {
    8.103117627864485e-278,
    973,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000810311762786448460701916265178721805318331135326132318795500180952292783697914667"
    "643762630124008413474467884537069420798897444792340459678011732950473018974004142598230372"
    "463984780283182332380363238844680055174104488268540267140959039040182960785950101183230945"
    "330299216050026619036803998081725752847418224769149833023540219642792861909876996758822377"
    "118135075109536171233688494942452416618125742680387879907346830121637034403253494540299453"
    "377386417370013446614658571822640671013053242841183193770612060018133599299419386567835242"
    "230298744858673499671828294369071353516818115351395829197920942068227227696883240538718869"
    "600739480881585534337397162967550449941267487474760855548083782196044921875",
    695,
    "8.1031176278644846070191626517872180531833113532613231879550018095229278369791466764376263"
    "012400841347446788453706942079889744479234045967801173295047301897400414259823037246398478"
    "028318233238036323884468005517410448826854026714095903904018296078595010118323094533029921"
    "605002661903680399808172575284741822476914983302354021964279286190987699675882237711813507"
    "510953617123368849494245241661812574268038787990734683012163703440325349454029945337738641"
    "737001344661465857182264067101305324284118319377061206001813359929941938656783524223029874"
    "485867349967182829436907135351681811535139582919792094206822722769688324053871886960073948"
    "0881585534337397162967550449941267487474760855548083782196044921875e-278"
  },
  {
    1.832473425427903e-277,
    970,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000001832473425427902886938778437285144804786082325832375061113482101690797286266586294"
    "867945954372905818531402941092342910125905372135803452206403364874424956200760739434749051"
    "288205708442549672897714910848208299823169103036957205913876376666459893503223052173089296"
    "631419583528827717374000015474402335079386664374426327190155534896975435589342426819275753"
    "768845584435883886178808128569655382229457502908034012259240062189848521009805863770517872"
    "740795257317589786558096192524505382458240446829343654441424414437135054656069423642614933"
    "323336396499777095021957372015947166181084102535422709985885556677780495536599067966610216"
    "695720290798005524230155507763549704858263567075482569634914398193359375",
    693,
    "1.8324734254279028869387784372851448047860823258323750611134821016907972862665862948679459"
    "543729058185314029410923429101259053721358034522064033648744249562007607394347490512882057"
    "084425496728977149108482082998231691030369572059138763766664598935032230521730892966314195"
    "835288277173740000154744023350793866643744263271901555348969754355893424268192757537688455"
    "844358838861788081285696553822294575029080340122592400621898485210098058637705178727407952"
    "573175897865580961925245053824582404468293436544414244144371350546560694236426149333233363"
    "964997770950219573720159471661810841025354227099858855566777804955365990679666102166957202"
    "90798005524230155507763549704858263567075482569634914398193359375e-277"
  },
  {
    2.727222054531752e-277,
    970,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000002727222054531751786888915796533769602527450842088929858871118159521370404337824879"
    "190994854038061643561159339103362125367673189880817810703976647945600559439685847375781053"
    "451117128981590541661315279874114722174017866347800567633763552707314540341731219481235621"
    "611130507775509864789785266383510150307545015561531035709618518535142230162959301887003584"
    "356228912327582537870139843038204744666191730945069746273591847403191866223408504411485305"
    "382847908319671602904529683414737087204591391215000382637027252736901813808270570320601259"
    "532974827496579814102373524204368860193693113227607483201472667063978336888662018926203111"
    "235696960255284736973595876500901791228415049772593192756175994873046875",
    693,
    "2.7272220545317517868889157965337696025274508420889298588711181595213704043378248791909948"
    "540380616435611593391033621253676731898808178107039766479456005594396858473757810534511171"
    "289815905416613152798741147221740178663478005676337635527073145403417312194812356216111305"
    "077755098647897852663835101503075450155615310357096185185351422301629593018870035843562289"
    "123275825378701398430382047446661917309450697462735918474031918662234085044114853053828479"
    "083196716029045296834147370872045913912150003826370272527369018138082705703206012595329748"
    "274965798141023735242043688601936931132276074832014726670639783368886620189262031112356969"
    "60255284736973595876500901791228415049772593192756175994873046875e-277"
  },
  {
    7.882910515028218e-277,
    964,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000007882910515028217908439923837091164211498369045702899188062518103364864188409315336"
    "136915242577447392215731577741539262476980101322752505084581018100122515177424096906241307"
    "958185574840128623212622353572694115995685913932502489931128923562170444579956137349168575"
    "918204365789567048958596408790512900515843150356938508578817174199076915484678106188840973"
    "883118189313794396946312462030987453218823755018816819599892055694230286747924272367378510"
    "090522287033623889106823256500201136355675691255491301645782530488062210583120588582336679"
    "214533967590618353077697183881161890577696002739400403994708829285246662769338823361787863"
    "906471416520890065817296459005358855165468412451446056365966796875",
    687,
    "7.8829105150282179084399238370911642114983690457028991880625181033648641884093153361369152"
    "425774473922157315777415392624769801013227525050845810181001225151774240969062413079581855"
    "748401286232126223535726941159956859139325024899311289235621704445799561373491685759182043"
    "657895670489585964087905129005158431503569385085788171741990769154846781061888409738831181"
    "893137943969463124620309874532188237550188168195998920556942302867479242723673785100905222"
    "870336238891068232565002011363556756912554913016457825304880622105831205885823366792145339"
    "675906183530776971838811618905776960027394004039947088292852466627693388233617878639064714"
    "16520890065817296459005358855165468412451446056365966796875e-277"
  },
  {
    1.479332526434952e-276,
    966,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000014793325264349519609224189087172778660628550715159699314589946835960775139104114196"
    "387124361744349388346423526249997529063816286588004877609496510635406949740021097961541031"
    "838846597715150779814417131533030432343598859756309545824257288522974794220778148135398753"
    "826867598954413321397706023813578090773636202866475879056725196621533667569514255202620311"
    "480396908342237186203988319154990018200960799763004452095958354279543752826246109554840754"
    "022410900257769037244635988497968052021678176154757506096344675620304382033400422909910848"
    "021464805357944988357217013009920224794942969316921145421278874324059335947390098990213512"
    "04586356951492886187572435401083037476155368494801223278045654296875",
    690,
    "1.4793325264349519609224189087172778660628550715159699314589946835960775139104114196387124"
    "361744349388346423526249997529063816286588004877609496510635406949740021097961541031838846"
    "597715150779814417131533030432343598859756309545824257288522974794220778148135398753826867"
    "598954413321397706023813578090773636202866475879056725196621533667569514255202620311480396"
    "908342237186203988319154990018200960799763004452095958354279543752826246109554840754022410"
    "900257769037244635988497968052021678176154757506096344675620304382033400422909910848021464"
    "805357944988357217013009920224794942969316921145421278874324059335947390098990213512045863"
    "56951492886187572435401083037476155368494801223278045654296875e-276"
  },
  {
    2.2180410620430226e-276,
    968,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000022180410620430225517898657079973304666136981289307679736753455928805959725081462934"
    "224670303724159485938268882871438022974238697141104193414482484553095139683028128964862127"
    "811987704870946024473429743997903245712442109101017222429201543366300060869912356695082101"
    "858263488196182299378666763642961264891058651396058283602048459899956394727237587803958256"
    "596040689177850038369317144434041219912680004534927123313725716843926392914184693322765428"
    "244423296707499699242123935215592872574467881567912099808514600224816487485667120098756254"
    "128750336587423385132645636143667475868457811347741242328297088498995213516845729206540275"
    "1394463997041174444432233190359349261910892892046831548213958740234375",
    692,
    "2.2180410620430225517898657079973304666136981289307679736753455928805959725081462934224670"
    "303724159485938268882871438022974238697141104193414482484553095139683028128964862127811987"
    "704870946024473429743997903245712442109101017222429201543366300060869912356695082101858263"
    "488196182299378666763642961264891058651396058283602048459899956394727237587803958256596040"
    "689177850038369317144434041219912680004534927123313725716843926392914184693322765428244423"
    "296707499699242123935215592872574467881567912099808514600224816487485667120098756254128750"
    "336587423385132645636143667475868457811347741242328297088498995213516845729206540275139446"
    "3997041174444432233190359349261910892892046831548213958740234375e-276"
  },
  {
    5.690649743658698e-276,
    966,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000056906497436586979168324009384548588036589095543759130627636379562863411660649404239"
    "561145787562444857179638324291687916940100137337435741474876016235005649571805858842353409"
    "648548706755316590971565738140297140444224680947567069010740829878166746254719191628174707"
    "445624038508617426383266868388313295431669065308685440594898149718409995904694095670341472"
    "420644316999405402196876112361507834250398148271011353848871090482305740770992731270378503"
    "731024908696903489732478795538444073009335548789702016273085718942681812282917892426850234"
    "788281317837349312491862062886002571561525296660278854846520211943833204206612478024408951"
    "24858403164728133943685880850871239289290315355174243450164794921875",
    690,
    "5.6906497436586979168324009384548588036589095543759130627636379562863411660649404239561145"
    "787562444857179638324291687916940100137337435741474876016235005649571805858842353409648548"
    "706755316590971565738140297140444224680947567069010740829878166746254719191628174707445624"
    "038508617426383266868388313295431669065308685440594898149718409995904694095670341472420644"
    "316999405402196876112361507834250398148271011353848871090482305740770992731270378503731024"
    "908696903489732478795538444073009335548789702016273085718942681812282917892426850234788281"
    "317837349312491862062886002571561525296660278854846520211943833204206612478024408951248584"
    "03164728133943685880850871239289290315355174243450164794921875e-276"
  },
  {
    1.4121156100358157e-275,
    966,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000141211561003581572966648844034396616463022763650784856871320010353040705077207668546"
    "950355581578583044505766940217400099355023661276743227694506565541149890385524583996128825"
    "300819463743839851012791085031812663780907811398349872282098767141228431386675167164986720"
    "254381508123799657134532912400274735245070827402120103392493826710535824080534156754898869"
    "176837682567321817374385812537835559417946028789607212568576973910770764877147702942096611"
    "971251748274003472604976609538520625018538574328879978037098590719449636953103849833441874"
    "982393256143008247135479928276969604075487324380734820010970183375991778926381861307427550"
    "69952437588982806943523648328386865813399708713404834270477294921875",
    691,
    "1.4121156100358157296664884403439661646302276365078485687132001035304070507720766854695035"
    "558157858304450576694021740009935502366127674322769450656554114989038552458399612882530081"
    "946374383985101279108503181266378090781139834987228209876714122843138667516716498672025438"
    "150812379965713453291240027473524507082740212010339249382671053582408053415675489886917683"
    "768256732181737438581253783555941794602878960721256857697391077076487714770294209661197125"
    "174827400347260497660953852062501853857432887997803709859071944963695310384983344187498239"
    "325614300824713547992827696960407548732438073482001097018337599177892638186130742755069952"
    "437588982806943523648328386865813399708713404834270477294921875e-275"
  },
  {
    2.6685811370633676e-275,
    965,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000266858113706336764691514561244629338089446086679562287777776698876674681021413432252"
    "249735738226278585206628706394471551392518714620362982432694174419506683788723487695530093"
    "086709272946848081120027413065121574478655777215008729624317664749625244047624479962497629"
    "023902079409646657019072114403155241006282098943412166788299697774572999840965867011569785"
    "318005012698958062884482095994760870515369048323272748656907984060140419242923626411226561"
    "350556759451418729720362364708522318857675731726321139280873439306988245518166084854544461"
    "716750017222352059049601995839940619334313072944140131845188217756339608299374433232455048"
    "7662094880548934381955914103834448525276457075960934162139892578125",
    690,
    "2.6685811370633676469151456124462933808944608667956228777777669887667468102141343225224973"
    "573822627858520662870639447155139251871462036298243269417441950668378872348769553009308670"
    "927294684808112002741306512157447865577721500872962431766474962524404762447996249762902390"
    "207940964665701907211440315524100628209894341216678829969777457299984096586701156978531800"
    "501269895806288448209599476087051536904832327274865690798406014041924292362641122656135055"
    "675945141872972036236470852231885767573172632113928087343930698824551816608485454446171675"
    "001722235205904960199583994061933431307294414013184518821775633960829937443323245504876620"
    "94880548934381955914103834448525276457075960934162139892578125e-275"
  },
  {
    4.6692815012344967e-275,
    963,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000466928150123449666988828448754373667890059011889759344171317627022843286418622584881"
    "256584554850146178922587849067391433971807460553436148881053030070580286108356418420118416"
    "969143514528711425882108264857440652948294995179746931897601003101521159351968523561435049"
    "920800565524605112055758790462197138895366175519849995707572817304695817966816083401433075"
    "234723709158340884914908030521084546247013987864207024521100899305768953323642622737252465"
    "685844839888300778476275250469032520933241067415921740697019684544854238709595905534560300"
    "448531889011772684649785578117332036865720081190298097835924725016740857998905937273501925"
    "41377257601235072778330693898940495500937686301767826080322265625",
    688,
    "4.6692815012344966698882844875437366789005901188975934417131762702284328641862258488125658"
    "455485014617892258784906739143397180746055343614888105303007058028610835641842011841696914"
    "351452871142588210826485744065294829499517974693189760100310152115935196852356143504992080"
    "056552460511205575879046219713889536617551984999570757281730469581796681608340143307523472"
    "370915834088491490803052108454624701398786420702452110089930576895332364262273725246568584"
    "483988830077847627525046903252093324106741592174069701968454485423870959590553456030044853"
    "188901177268464978557811733203686572008119029809783592472501674085799890593727350192541377"
    "257601235072778330693898940495500937686301767826080322265625e-275"
  },
  {
    9.834548098669551e-275,
    958,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000983454809866955140896188759264612084833254520187790203799630446960504329333498437153"
    "616827417615849080912491003925803323420285422299200257544927820640319697426292214256590284"
    "651687321803691614450944195461737083147005394945013001995512331413015381585120956882968225"
    "427866716275630234476191960199520698720821775622452030124901681866799001372244395911617635"
    "616315293464515485572364258841282031378610280969479425754178210176017601378479729168995921"
    "054485337026942189492031327824997457509579952046482421043196522791788544424088239606919583"
    "423224940988305307278449983173081604885897405160185992663959592128805251547178743063123170"
    "001280186158806417666411459066466704825870692729949951171875",
    683,
    "9.8345480986695514089618875926461208483325452018779020379963044696050432933349843715361682"
    "741761584908091249100392580332342028542229920025754492782064031969742629221425659028465168"
    "732180369161445094419546173708314700539494501300199551233141301538158512095688296822542786"
    "671627563023447619196019952069872082177562245203012490168186679900137224439591161763561631"
    "529346451548557236425884128203137861028096947942575417821017601760137847972916899592105448"
    "533702694218949203132782499745750957995204648242104319652279178854442408823960691958342322"
    "494098830530727844998317308160488589740516018599266395959212880525154717874306312317000128"
    "0186158806417666411459066466704825870692729949951171875e-275"
  },
  {
    1.8600057734718372e-274,
    962,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000001860005773471837171482527351863497835435843065226252667540223733159853584429545480842"
    "165284796862395516622946505738661111515527860596046535739252667433961994327915595476704261"
    "945824525762053904102146775348361843974555536608449663541605202919437531227398833959534590"
    "546980892951601368099412773750211386643708216564687986861470559574764829423944855177411757"
    "744897431022543324456960570281971353204529907162058888376007918463781982884240607983616391"
    "110433956964751638884396648717288372304417549115776831956237245914282005255165195223843673"
    "716690874231955397032122819347024452498018750025400425613047170373078212498439755828093049"
    "5901982551036706501029587002005172280405531637370586395263671875",
    688,
    "1.8600057734718371714825273518634978354358430652262526675402237331598535844295454808421652"
    "847968623955166229465057386611115155278605960465357392526674339619943279155954767042619458"
    "245257620539041021467753483618439745555366084496635416052029194375312273988339595345905469"
    "808929516013680994127737502113866437082165646879868614705595747648294239448551774117577448"
    "974310225433244569605702819713532045299071620588883760079184637819828842406079836163911104"
    "339569647516388843966487172883723044175491157768319562372459142820052551651952238436737166"
    "908742319553970321228193470244524980187500254004256130471703730782124984397558280930495901"
    "982551036706501029587002005172280405531637370586395263671875e-274"
  },
  {
    2.811775675630843e-274,
    961,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000002811775675630843181004570741845754971971061000084980110558541339231221112326541290736"
    "699617979395977326231025284075045635632891833639494133197161881204622450439803808980479756"
    "669619405485279364942521657922822272709751021750995630547875850829787122147998996258733220"
    "571087825557332332005588075878604400572352057444304256611566580487469293921017145091996165"
    "557798542355615838210352261328359536538279552301352677428931245555964648511872460375830460"
    "315773282037723059187498467233103614182839694522559328292760649843538562452984192640813659"
    "122168387059488839582188262378663355081166412581348801204378576235540860124864292440125940"
    "486330437256312075595780786141375529041397385299205780029296875",
    687,
    "2.8117756756308431810045707418457549719710610000849801105585413392312211123265412907366996"
    "179793959773262310252840750456356328918336394941331971618812046224504398038089804797566696"
    "194054852793649425216579228222727097510217509956305478758508297871221479989962587332205710"
    "878255573323320055880758786044005723520574443042566115665804874692939210171450919961655577"
    "985423556158382103522613283595365382795523013526774289312455559646485118724603758304603157"
    "732820377230591874984672331036141828396945225593282927606498435385624529841926408136591221"
    "683870594888395821882623786633550811664125813488012043785762355408601248642924401259404863"
    "30437256312075595780786141375529041397385299205780029296875e-274"
  },
  {
    6.27142702954425e-274,
    959,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000006271427029544250460343525111013240914753466797573592625786027826229961270121367161220"
    "735064190560294766189444944164164163537272678618355342497912064256570488921297897723656238"
    "744717518171920071527371003534580160225562053419694214377441724668769025436402835091526788"
    "281492455721704077602519026610127870369118313142031480989753345061094924931145668236421252"
    "221734067473178795384718700740529819486819550341451155693941748487731802872607782124208196"
    "306724517738148599429059277140081298100966429737409600754952964540583049594691499489910019"
    "786761795794622366208715634510022620666391042308290790968346187058669856385832192052476042"
    "5968748800699115977970443669420319565688259899616241455078125",
    685,
    "6.2714270295442504603435251110132409147534667975735926257860278262299612701213671612207350"
    "641905602947661894449441641641635372726786183553424979120642565704889212978977236562387447"
    "175181719200715273710035345801602255620534196942143774417246687690254364028350915267882814"
    "924557217040776025190266101278703691183131420314809897533450610949249311456682364212522217"
    "340674731787953847187007405298194868195503414511556939417484877318028726077821242081963067"
    "245177381485994290592771400812981009664297374096007549529645405830495946914994899100197867"
    "617957946223662087156345100226206663910423082907909683461870586698563858321920524760425968"
    "748800699115977970443669420319565688259899616241455078125e-274"
  },
  {
    1.0250975871333601e-273,
    959,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000010250975871333601410285213100183232774538473685218055938897189096735340237340018400725"
    "666682494033072536608452334448790215856221307378430596060981382469462541903710949602109595"
    "152690417802943353760982516151826097946117833795514011639741468644583294671854431626452799"
    "129204600257369066120851024982982459912280838189402775148363138806380170433459153981158132"
    "727283701179492276737893371060445823347383802426142678436476115955450138058270748161861808"
    "671057012019317372254774761292737409045092429887831908577354822444176298074044434113188615"
    "703069177793172139697811909954855649628510951777351226849677509048789269483000628875368584"
    "8963361461824513759001631374445651090354658663272857666015625",
    686,
    "1.0250975871333601410285213100183232774538473685218055938897189096735340237340018400725666"
    "682494033072536608452334448790215856221307378430596060981382469462541903710949602109595152"
    "690417802943353760982516151826097946117833795514011639741468644583294671854431626452799129"
    "204600257369066120851024982982459912280838189402775148363138806380170433459153981158132727"
    "283701179492276737893371060445823347383802426142678436476115955450138058270748161861808671"
    "057012019317372254774761292737409045092429887831908577354822444176298074044434113188615703"
    "069177793172139697811909954855649628510951777351226849677509048789269483000628875368584896"
    "3361461824513759001631374445651090354658663272857666015625e-273"
  },
  {
    2.5253325080332935e-273,
    958,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000025253325080332935122289495566127846548934475987700357144674961351479357467092094630453"
    "535541284919659245903800349407085632973490108353504676078653305638868047686444694829231245"
    "812521767196094379515678278711538665184899956122603320973051083740781815076031746325340815"
    "480734713963425049837104122401942575852346334914282204317410475710039015821596102435733830"
    "265474911659400368786801131458888124352482480324532223906317300645196650798384111356497633"
    "385763926314710642986472226366485103040564918136701115032953166852184372020053702812309748"
    "306020305670681970673625585530694816712467933603012948860835146458029043511285542534192722"
    "288067603144347717143702869435628599603660404682159423828125",
    685,
    "2.5253325080332935122289495566127846548934475987700357144674961351479357467092094630453535"
    "541284919659245903800349407085632973490108353504676078653305638868047686444694829231245812"
    "521767196094379515678278711538665184899956122603320973051083740781815076031746325340815480"
    "734713963425049837104122401942575852346334914282204317410475710039015821596102435733830265"
    "474911659400368786801131458888124352482480324532223906317300645196650798384111356497633385"
    "763926314710642986472226366485103040564918136701115032953166852184372020053702812309748306"
    "020305670681970673625585530694816712467933603012948860835146458029043511285542534192722288"
    "067603144347717143702869435628599603660404682159423828125e-273"
  },
  {
    6.913296125147658e-273,
    957,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000069132961251476578929222904530052273628363552443315557555004747661057069612999210997894"
    "255772698684551025199124351522536488288269888962458462802724946749266694110307871159166970"
    "945639300297781454867471403144249057918732350248384232683453828202175745255871641949482204"
    "895994855547128746509985097710587405234913016896864323179317665178723422213216184384022812"
    "397384381918414301497856335169286849906525371490446315447355139287718496849222455179777434"
    "039582883600813461913264068300273626962586715002997274461341233852580066298338975595440749"
    "606615692544348444168496966436111362304807468045695546066275705657497354443900990463419932"
    "34153112576404723484124748011936389957554638385772705078125",
    684,
    "6.9132961251476578929222904530052273628363552443315557555004747661057069612999210997894255"
    "772698684551025199124351522536488288269888962458462802724946749266694110307871159166970945"
    "639300297781454867471403144249057918732350248384232683453828202175745255871641949482204895"
    "994855547128746509985097710587405234913016896864323179317665178723422213216184384022812397"
    "384381918414301497856335169286849906525371490446315447355139287718496849222455179777434039"
    "582883600813461913264068300273626962586715002997274461341233852580066298338975595440749606"
    "615692544348444168496966436111362304807468045695546066275705657497354443900990463419932341"
    "53112576404723484124748011936389957554638385772705078125e-273"
  },
  {
    9.747372287629775e-273,
    956,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000097473722876297753859094243167038918335732475648235279596858447770771804886894929506661"
    "512363400799235132329961109412782244052135199979147620902244163877962659735170124597001357"
    "267514566904245611685092350584690212210195439103504468405558964086408218439614002201877363"
    "936379516464663015757564910152203684316656461318510509727736135505497758609491007485313307"
    "150194231963853639010109636728185073120919295228223264260018917015441561102712444279374287"
    "475396552967106691035119145349584539628136809362468969569946583405784058237616365401337479"
    "270578926052771153043617919331512765273483588087026932592725793809455763126996835657504433"
    "7499124795891878615672343499909402453340590000152587890625",
    683,
    "9.7473722876297753859094243167038918335732475648235279596858447770771804886894929506661512"
    "363400799235132329961109412782244052135199979147620902244163877962659735170124597001357267"
    "514566904245611685092350584690212210195439103504468405558964086408218439614002201877363936"
    "379516464663015757564910152203684316656461318510509727736135505497758609491007485313307150"
    "194231963853639010109636728185073120919295228223264260018917015441561102712444279374287475"
    "396552967106691035119145349584539628136809362468969569946583405784058237616365401337479270"
    "578926052771153043617919331512765273483588087026932592725793809455763126996835657504433749"
    "9124795891878615672343499909402453340590000152587890625e-273"
  },
  {
    1.9721670909138485e-272,
    953,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000197216709091384854067082098752459887548554995843593801435382496634863181333354250488013"
    "601769333154658301020938111316690961968819801684276456747470648078446092573136974003930909"
    "131364649434410677191755964374461168944998514974184752287051265514823164239054882545965461"
    "813599846305931493984585486386237883843017791960545685776619450711678109955600103313337459"
    "763032012643595551046517388510084614608233716254602267850305675356162650233030360899161133"
    "192808264578667329162878470873772247704774334161447668174066561587041726468755775880275549"
    "546347394511667795970444005134973927905389687889706272452193245281982712102143805928570275"
    "0231882048720898070737206353442161343991756439208984375",
    681,
    "1.9721670909138485406708209875245988754855499584359380143538249663486318133335425048801360"
    "176933315465830102093811131669096196881980168427645674747064807844609257313697400393090913"
    "136464943441067719175596437446116894499851497418475228705126551482316423905488254596546181"
    "359984630593149398458548638623788384301779196054568577661945071167810995560010331333745976"
    "303201264359555104651738851008461460823371625460226785030567535616265023303036089916113319"
    "280826457866732916287847087377224770477433416144766817406656158704172646875577588027554954"
    "634739451166779597044400513497392790538968788970627245219324528198271210214380592857027502"
    "31882048720898070737206353442161343991756439208984375e-272"
  },
  {
    4.469048372679605e-272,
    953,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000446904837267960489127366487311229552573541881960242319865883123307422572380214566349088"
    "266541272348399162694465100758770795918495191617148046111636565021984454301308771295767628"
    "366526457752766226774136157158677204977723241472823784778380018581930015069549486826539998"
    "998430341818691393638710377829584670755858769889897445377431599053469908067403008989949485"
    "236875367828209803100829565629765163934287231863692549726979176395237517451055069267609733"
    "839616408831778853430757895403549124610101368994821095535082159636545165567339743831147356"
    "314912660732629308735805522284710389546444222168239476217622026150424150094836999566430914"
    "8768991554272721200558038390227011404931545257568359375",
    681,
    "4.4690483726796048912736648731122955257354188196024231986588312330742257238021456634908826"
    "654127234839916269446510075877079591849519161714804611163656502198445430130877129576762836"
    "652645775276622677413615715867720497772324147282378477838001858193001506954948682653999899"
    "843034181869139363871037782958467075585876988989744537743159905346990806740300898994948523"
    "687536782820980310082956562976516393428723186369254972697917639523751745105506926760973383"
    "961640883177885343075789540354912461010136899482109553508215963654516556733974383114735631"
    "491266073262930873580552228471038954644422216823947621762202615042415009483699956643091487"
    "68991554272721200558038390227011404931545257568359375e-272"
  },
  {
    1.0205638445192075e-271,
    952,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "001020563844519207454068603914961949216603622130365562288058257422369971885119195604032070"
    "972015707144596472334283003344969574819842726371410401166526430641092990020007149125267223"
    "500095048465228875009377422994597770942012244050224737963234632599223698388649382113915392"
    "050770236622145013987906289250667977440016565465777278612474582456571687609089135806919631"
    "306079776803215097870129800335260946331259196166059113373568611705869136348407003151491682"
    "756842641906922895435050439101390973708990716810066400092462166906705150685302857597044837"
    "404540088869708210335900301553691604441179068784764324781639073532912521042956144573308772"
    "953464440883496189282908517270698212087154388427734375",
    681,
    "1.0205638445192074540686039149619492166036221303655622880582574223699718851191956040320709"
    "720157071445964723342830033449695748198427263714104011665264306410929900200071491252672235"
    "000950484652288750093774229945977709420122440502247379632346325992236983886493821139153920"
    "507702366221450139879062892506679774400165654657772786124745824565716876090891358069196313"
    "060797768032150978701298003352609463312591961660591133735686117058691363484070031514916827"
    "568426419069228954350504391013909737089907168100664000924621669067051506853028575970448374"
    "045400888697082103359003015536916044411790687847643247816390735329125210429561445733087729"
    "53464440883496189282908517270698212087154388427734375e-271"
  },
  {
    1.5594946591458144e-271,
    952,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "001559494659145814441681581920546808616715984366442620364478829976811035089889563564047368"
    "317717978748493547107618900970773911702439074983599285664500289449009987812362032309365505"
    "683561381853263422607449477979590448089980215073756826092589717035140761699187009768022799"
    "578563932926539959419689583937077771224903982883050660375451535251837042673071950369041861"
    "598994502394042224648739892050532127005297477317011060627275918383352054171181466701597646"
    "454049096032781767016048297431985804844124924866637988340032817743663903473681657595362339"
    "231688365149446598673054797852034665234088747269980370542955955889926977401007116018807400"
    "721172485495540083189069946456584148108959197998046875",
    681,
    "1.5594946591458144416815819205468086167159843664426203644788299768110350898895635640473683"
    "177179787484935471076189009707739117024390749835992856645002894490099878123620323093655056"
    "835613818532634226074494779795904480899802150737568260925897170351407616991870097680227995"
    "785639329265399594196895839370777712249039828830506603754515352518370426730719503690418615"
    "989945023940422246487398920505321270052974773170110606272759183833520541711814667015976464"
    "540490960327817670160482974319858048441249248666379883400328177436639034736816575953623392"
    "316883651494465986730547978520346652340887472699803705429559558899269774010071160188074007"
    "21172485495540083189069946456584148108959197998046875e-271"
  },
  {
    2.3721122629194564e-271,
    949,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "002372112262919456368899269596102259197270096576996496202397961351883127358688939107282331"
    "299624534155208390877862968305992335609703366546415633283533678270611937503390558491520767"
    "144739936842779782727826567796786519248093278523380946182387181270224161693471305960716801"
    "796086350401820941530293610124339885099370122455633921348077316650373845372983672347797623"
    "392143268249901056594934855686217622990248877923391269195571001807761624217336055192512549"
    "943990505216069472298347039010013433733539919557001959878811967117782576113053728770778315"
    "767911073839509938476337268634020082539441903467893818834311806047336796227578822931941039"
    "658316800094184362279747801949270069599151611328125",
    678,
    "2.3721122629194563688992695961022591972700965769964962023979613518831273586889391072823312"
    "996245341552083908778629683059923356097033665464156332835336782706119375033905584915207671"
    "447399368427797827278265677967865192480932785233809461823871812702241616934713059607168017"
    "960863504018209415302936101243398850993701224556339213480773166503738453729836723477976233"
    "921432682499010565949348556862176229902488779233912691955710018077616242173360551925125499"
    "439905052160694722983470390100134337335399195570019598788119671177825761130537287707783157"
    "679110738395099384763372686340200825394419034678938188343118060473367962275788229319410396"
    "58316800094184362279747801949270069599151611328125e-271"
  },
  {
    5.551135919284155e-271,
    944,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "005551135919284154688627170825331662268083845008514417554841019502743301366429368819546119"
    "006284763004333001348277225050221860684150032510350617852415448793291743558291284923804209"
    "319488728339663950293202832391065128400274603099891349456086079489507202429780609710291427"
    "660660349740546127638820776459982111416694789552131981129329805742459868881489125410526449"
    "219152345004917839581742185059741768986228887671664675393120074838168629509013042697933604"
    "905361403885064909007008182829068069618562258374252943072040208118699107159634152413919749"
    "269646896273238207930381447337881848530251103441328808489089045268485281854014309335709649"
    "5846052242040258306587929837405681610107421875",
    673,
    "5.5511359192841546886271708253316622680838450085144175548410195027433013664293688195461190"
    "062847630043330013482772250502218606841500325103506178524154487932917435582912849238042093"
    "194887283396639502932028323910651284002746030998913494560860794895072024297806097102914276"
    "606603497405461276388207764599821114166947895521319811293298057424598688814891254105264492"
    "191523450049178395817421850597417689862288876716646753931200748381686295090130426979336049"
    "053614038850649090070081828290680696185622583742529430720402081186991071596341524139197492"
    "696468962732382079303814473378818485302511034413288084890890452684852818540143093357096495"
    "846052242040258306587929837405681610107421875e-271"
  },
  {
    1.0734739674446298e-270,
    947,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "010734739674446298024207248229790974212690206929738748612035688298132431930940767808978781"
    "611808395357190463149503662940609545550797429425138405559672522900633988700654397921947407"
    "990507847670142363385136500330409655335816372646993301610537006049401730121980298934824484"
    "645281004444729389311880410282095677589494819051830490015311349228608723684009581342357121"
    "335471812569531901439452988004676219145306228626404952817818807907754264843610519108607195"
    "980800602836302109764033446426430688987258461030549877555191854576387103486467093916977848"
    "983572370055145295457364582701096635871225741875069893746756102447305593731669685472484568"
    "2348285079432681499156387872062623500823974609375",
    677,
    "1.0734739674446298024207248229790974212690206929738748612035688298132431930940767808978781"
    "611808395357190463149503662940609545550797429425138405559672522900633988700654397921947407"
    "990507847670142363385136500330409655335816372646993301610537006049401730121980298934824484"
    "645281004444729389311880410282095677589494819051830490015311349228608723684009581342357121"
    "335471812569531901439452988004676219145306228626404952817818807907754264843610519108607195"
    "980800602836302109764033446426430688987258461030549877555191854576387103486467093916977848"
    "983572370055145295457364582701096635871225741875069893746756102447305593731669685472484568"
    "2348285079432681499156387872062623500823974609375e-270"
  },
  {
    2.4398554262702667e-270,
    948,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "024398554262702667444398148018038098952470544215124227046286789042553768286136848758427981"
    "512172714445409930115643233877845863399498871974798169753440335024240198843109802361840133"
    "842149912774785011897996278137062515822184149067261369312941603854124691974404403809095705"
    "720893892088782330632634700155655823455700481093598639683646815093873774935940375258540408"
    "008971091122482936133784909715421457345554809011396322005411855059916233822920918416811292"
    "732601739460526050660702480916614575619329299225550434439953381872738847202063720298418370"
    "992597319324438515921184124113074610598625092228081771573367525233816132412752749548779228"
    "57555982462385291142936694086529314517974853515625",
    678,
    "2.4398554262702667444398148018038098952470544215124227046286789042553768286136848758427981"
    "512172714445409930115643233877845863399498871974798169753440335024240198843109802361840133"
    "842149912774785011897996278137062515822184149067261369312941603854124691974404403809095705"
    "720893892088782330632634700155655823455700481093598639683646815093873774935940375258540408"
    "008971091122482936133784909715421457345554809011396322005411855059916233822920918416811292"
    "732601739460526050660702480916614575619329299225550434439953381872738847202063720298418370"
    "992597319324438515921184124113074610598625092228081771573367525233816132412752749548779228"
    "57555982462385291142936694086529314517974853515625e-270"
  },
  {
    6.8642158764630495e-270,
    942,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "068642158764630495478107887035397014093806406478739147400934802190758122966145509273130663"
    "656083449116193388822368564057256191430774573151546980833069733638423224241396929306353656"
    "907772341676239689270753025870742445454674389256122142758816013351276338542915435493127984"
    "766811052649793408810278491886555260153716643670663196129370831861216770934862924684216985"
    "743727871460443148241127262602309896093569590960002026708618897030775910938683812388348909"
    "731662562903690820461225303088678149847111147570170513001326360333196264837122448428318182"
    "457682024477158763481970961132923122335059870501903517632427470761892269884581646023315241"
    "96780765390002443382400088012218475341796875",
    672,
    "6.8642158764630495478107887035397014093806406478739147400934802190758122966145509273130663"
    "656083449116193388822368564057256191430774573151546980833069733638423224241396929306353656"
    "907772341676239689270753025870742445454674389256122142758816013351276338542915435493127984"
    "766811052649793408810278491886555260153716643670663196129370831861216770934862924684216985"
    "743727871460443148241127262602309896093569590960002026708618897030775910938683812388348909"
    "731662562903690820461225303088678149847111147570170513001326360333196264837122448428318182"
    "457682024477158763481970961132923122335059870501903517632427470761892269884581646023315241"
    "96780765390002443382400088012218475341796875e-270"
  },
  {
    1.2055339469240307e-269,
    941,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "120553394692403073774034025943543586744076424658214403161853873665328348183768707944967517"
    "035919903205770271070113114076479488114135898545635834909792131037998106545850994330544371"
    "266945740534776833488374278778857421007601195635674727657767439409444359581853444749924911"
    "534199867608439448630787618396950190532644723179216748177578061386432901621795953399500436"
    "704223434169274034194895415420453624725368318043357630277210489316956915292444618610846509"
    "129090912794604455025923507204700630210668066737375536920033779883548191818661184896304872"
    "235473291423567414142218851169522350509165130461225880879621586985693455105301566944376223"
    "5800432502941248458228074014186859130859375",
    672,
    "1.2055339469240307377403402594354358674407642465821440316185387366532834818376870794496751"
    "703591990320577027107011311407647948811413589854563583490979213103799810654585099433054437"
    "126694574053477683348837427877885742100760119563567472765776743940944435958185344474992491"
    "153419986760843944863078761839695019053264472317921674817757806138643290162179595339950043"
    "670422343416927403419489541542045362472536831804335763027721048931695691529244461861084650"
    "912909091279460445502592350720470063021066806673737553692003377988354819181866118489630487"
    "223547329142356741414221885116952235050916513046122588087962158698569345510530156694437622"
    "35800432502941248458228074014186859130859375e-269"
  },
  {
    1.5235738192618518e-269,
    945,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "152357381926185181899366185943531983734900389336415111790903062676051820241330533383850487"
    "198054690809001545957303087100560223792006622980536966737234211755461011587426519659509365"
    "351506623505775740873732101274705930224448068431482455371738523277055151472825558244365940"
    "377898401366734923803609217954694493129932033108617575053210778718770408543207716233661942"
    "141010409767355749589386637921051247047454063093657663296180147824771690982829988866222038"
    "776007404704345799976200349382370719198260584387074295575308489654601314392380880624765048"
    "696519163589838148827237065921523213070630583889965535394674617067198773261453892494336037"
    "59474424588216123765960219316184520721435546875",
    676,
    "1.5235738192618518189936618594353198373490038933641511179090306267605182024133053338385048"
    "719805469080900154595730308710056022379200662298053696673723421175546101158742651965950936"
    "535150662350577574087373210127470593022444806843148245537173852327705515147282555824436594"
    "037789840136673492380360921795469449312993203310861757505321077871877040854320771623366194"
    "214101040976735574958938663792105124704745406309365766329618014782477169098282998886622203"
    "877600740470434579997620034938237071919826058438707429557530848965460131439238088062476504"
    "869651916358983814882723706592152321307063058388996553539467461706719877326145389249433603"
    "759474424588216123765960219316184520721435546875e-269"
  },
  {
    4.043876497442965e-269,
    944,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "404387649744296471810836932787936669995896517883712316059693838777295245607266108475249394"
    "878935376323626898846861862368755886367916636298143998039008428394610535517912002245989309"
    "306202382454578439356783495073513291830890396871175295526018781733233852564049716575023193"
    "809645819080820642844591136688492350108089352395412060930293857687138582394910001512259594"
    "635315835889637361803405730106505082375740159759460103926440144688165168171854738715370375"
    "794081116073467023962896828734809612148557370012497858203113645628993222560967395033666435"
    "514345191977194511511319116625250918055314935106348254811998312185844938649143881639477818"
    "8001261909345629419476608745753765106201171875",
    675,
    "4.0438764974429647181083693278793666999589651788371231605969383877729524560726610847524939"
    "487893537632362689884686186236875588636791663629814399803900842839461053551791200224598930"
    "930620238245457843935678349507351329183089039687117529552601878173323385256404971657502319"
    "380964581908082064284459113668849235010808935239541206093029385768713858239491000151225959"
    "463531583588963736180340573010650508237574015975946010392644014468816516817185473871537037"
    "579408111607346702396289682873480961214855737001249785820311364562899322256096739503366643"
    "551434519197719451151131911662525091805531493510634825481199831218584493864914388163947781"
    "88001261909345629419476608745753765106201171875e-269"
  },
  {
    8.743556251142875e-269,
    941,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "874355625114287464079178581212143729028790517766789817379763220476136106242826852095074666"
    "987781422882240275455082159625173427591657526511616774859950448070519505134802050234509915"
    "695099201889128074783085073031931481918818302926382994469764881550786694181892787805824256"
    "654363232794053063133755509954035540578163117869928089247703873466663428261756878987457088"
    "692773098795098157293253829450151542285341405243742641265914881758531162780816922738713821"
    "526003173659290088391480469742480863935095626092090116984973808179531069736545259366585454"
    "730577039559422654220273607660927072429693858230954951579111153307874908127934588949050980"
    "7252804908017651541740633547306060791015625",
    672,
    "8.7435562511428746407917858121214372902879051776678981737976322047613610624282685209507466"
    "698778142288224027545508215962517342759165752651161677485995044807051950513480205023450991"
    "569509920188912807478308507303193148191881830292638299446976488155078669418189278780582425"
    "665436323279405306313375550995403554057816311786992808924770387346666342826175687898745708"
    "869277309879509815729325382945015154228534140524374264126591488175853116278081692273871382"
    "152600317365929008839148046974248086393509562609209011698497380817953106973654525936658545"
    "473057703955942265422027360766092707242969385823095495157911115330787490812793458894905098"
    "07252804908017651541740633547306060791015625e-269"
  },
  {
    1.4649312749107113e-268,
    941,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
    "464931274910711269924907861225346625513919061086604893020723479984156700388998530701157329"
    "530892948055206250467791745632187137591881800959448706575506783553330529398084568218901876"
    "519235886802696545938601215144349181485500931165058227711343981064209269072796191838297693"
    "555979340057353274795768482966261660740325721905169907017991808666016701814237069186041942"
    "684378647424132138357106197054181850987434271954643223838347525698626002419348990652336080"
    "238406606469810367043062463283050782737870209062873732838285505327727773761545042382603061"
    "313562240235190280769184374891864894990808094527898696863693474382944773192204352614134786"
    "1806163432873972851666621863842010498046875",
    673,
    "1.4649312749107112699249078612253466255139190610866048930207234799841567003889985307011573"
    "295308929480552062504677917456321871375918818009594487065755067835533305293980845682189018"
    "765192358868026965459386012151443491814855009311650582277113439810642092690727961918382976"
    "935559793400573532747957684829662616607403257219051699070179918086660167018142370691860419"
    "426843786474241321383571061970541818509874342719546432238383475256986260024193489906523360"
    "802384066064698103670430624632830507827378702090628737328382855053277277737615450423826030"
    "613135622402351902807691843748918648949908080945278986968636934743829447731922043526141347"
    "861806163432873972851666621863842010498046875e-268"
  },
  {
    3.98040680032538e-268,
    941,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003"
    "980406800325380192488237571941781253611862147212238841684360793753033502180071560424510118"
    "766755474701399549993774340528518781782521015393593943194136695829287851610474123432316708"
    "833531119052176167643254590466677828829988706129136467711963276762541096135592752188769472"
    "589597065685993694758879579630673912948899179398462873439668863966430183380257085490192418"
    "953077896482673340235492057075596872283134858371284398948959714256144129887323486148624003"
    "771536424647219733725766804094500279087617702031764610696181549390555658797691437413131078"
    "539175704147903187697872905257705223315772497825560770376041116692908571522268713665397078"
    "0240940907646063351421616971492767333984375",
    673,
    "3.9804068003253801924882375719417812536118621472122388416843607937530335021800715604245101"
    "187667554747013995499937743405285187817825210153935939431941366958292878516104741234323167"
    "088335311190521761676432545904666778288299887061291364677119632767625410961355927521887694"
    "725895970656859936947588795796306739129488991793984628734396688639664301833802570854901924"
    "189530778964826733402354920570755968722831348583712843989489597142561441298873234861486240"
    "037715364246472197337257668040945002790876177020317646106961815493905556587976914374131310"
    "785391757041479031876978729052577052233157724978255607703760411166929085715222687136653970"
    "780240940907646063351421616971492767333984375e-268"
  },
  {
    8.586286660615516e-268,
    938,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008"
    "586286660615515786205822133634582057484588037887861653476646133471193396760860140523709430"
    "528521295122472188112244145499998342419246971681850985483625015372900193140148991767079401"
    "875309567807263252352799973261009452705579499169901168981268202020899909243284637492060661"
    "433758304632176006814240421686409284667595109084881478052642492376705841628178069182902580"
    "136059445292592764666886134366208157303804289513110088392566492734345663801695248299650254"
    "552073655270093165014095374523848126486834586665705309940163984722579358218701475792921574"
    "606565186297310529472246116485610860468577453516637959521359804094235499407265480713324131"
    "5619809341797008528374135494232177734375",
    670,
    "8.5862866606155157862058221336345820574845880378878616534766461334711933967608601405237094"
    "305285212951224721881122441454999983424192469716818509854836250153729001931401489917670794"
    "018753095678072632523527999732610094527055794991699011689812682020208999092432846374920606"
    "614337583046321760068142404216864092846675951090848814780526424923767058416281780691829025"
    "801360594452925927646668861343662081573038042895131100883925664927343456638016952482996502"
    "545520736552700931650140953745238481264868345866657053099401639847225793582187014757929215"
    "746065651862973105294722461164856108604685774535166379595213598040942354994072654807133241"
    "315619809341797008528374135494232177734375e-268"
  },
  {
    1.657418018730156e-267,
    939,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000016"
    "574180187301560277042993359422084230465392733787051007866897996899438544490745152444688834"
    "530645625295640123271066862185300846827639112830109790029239800333627418242022537447931069"
    "471421920449829106085977935166798767407915171285448859248403302521616064815753537409498761"
    "876477493712341293605688302238016410600642223723867362631145241282908960084578297501714717"
    "916105706062353536412253151235023622382582131429436080455694095554769027101991142854931736"
    "085157727810031776394869393752948429061437092252824723387139651155755397089393249747931236"
    "021807212049169581849793930624443797538885162639441859409425487723575214692806037462753957"
    "97560303634554657037369906902313232421875",
    672,
    "1.6574180187301560277042993359422084230465392733787051007866897996899438544490745152444688"
    "834530645625295640123271066862185300846827639112830109790029239800333627418242022537447931"
    "069471421920449829106085977935166798767407915171285448859248403302521616064815753537409498"
    "761876477493712341293605688302238016410600642223723867362631145241282908960084578297501714"
    "717916105706062353536412253151235023622382582131429436080455694095554769027101991142854931"
    "736085157727810031776394869393752948429061437092252824723387139651155755397089393249747931"
    "236021807212049169581849793930624443797538885162639441859409425487723575214692806037462753"
    "95797560303634554657037369906902313232421875e-267"
  },
  {
    3.613106834542997e-267,
    937,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000036"
    "131068345429969451130373310115607099294864130669813892800930118616243487082666684796081077"
    "226753480935212068872259503118509581066207485830029333326409179897675170932942171931932924"
    "947144911321258510421618036490209030524556424721142161449304687328816892134309737914248109"
    "407440703653634171991075281655928489915633883714239891254785518595031411885155607054339603"
    "011357809855510923231449073199376203490368131774971870870529128925452259061106855564430654"
    "962589375807588265121141153314482687818031206887535601284141804226785072747406311526247953"
    "867656568422352120973656418427195192919635507954677513543575460907844889158905506818431248"
    "305945047150089521892368793487548828125",
    670,
    "3.6131068345429969451130373310115607099294864130669813892800930118616243487082666684796081"
    "077226753480935212068872259503118509581066207485830029333326409179897675170932942171931932"
    "924947144911321258510421618036490209030524556424721142161449304687328816892134309737914248"
    "109407440703653634171991075281655928489915633883714239891254785518595031411885155607054339"
    "603011357809855510923231449073199376203490368131774971870870529128925452259061106855564430"
    "654962589375807588265121141153314482687818031206887535601284141804226785072747406311526247"
    "953867656568422352120973656418427195192919635507954677513543575460907844889158905506818431"
    "248305945047150089521892368793487548828125e-267"
  },
  {
    5.9183241751764955e-267,
    936,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000059"
    "183241751764955109389764976714155872620609643410243773660851543979583503245329654584813732"
    "786527982073834052163642085272541799726161544741798141501688273444897338956418960616682248"
    "146734909590274760063649781587056559493140177302285701706465537194317455996997555599630976"
    "046292790637546422298642121147072796753317178168440072594592218022370998643157234465370967"
    "008505124175029486015944278448019481201861981653551367410858983035952773046534667845276266"
    "870926697195576559022854063157687024035059232711863381343216333489380995691010581869340586"
    "546165069007260684235784797752703442201127162548351380099067929652319412523908895030480201"
    "14928107204832485876977443695068359375",
    669,
    "5.9183241751764955109389764976714155872620609643410243773660851543979583503245329654584813"
    "732786527982073834052163642085272541799726161544741798141501688273444897338956418960616682"
    "248146734909590274760063649781587056559493140177302285701706465537194317455996997555599630"
    "976046292790637546422298642121147072796753317178168440072594592218022370998643157234465370"
    "967008505124175029486015944278448019481201861981653551367410858983035952773046534667845276"
    "266870926697195576559022854063157687024035059232711863381343216333489380995691010581869340"
    "586546165069007260684235784797752703442201127162548351380099067929652319412523908895030480"
    "20114928107204832485876977443695068359375e-267"
  },
  {
    1.1798475245572565e-266,
    935,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000117"
    "984752455725652588364632089286836354160191553757116368488357751653846078375633343585731394"
    "837217385287091120028390252414848147343275955457018828059343282834073062154638169876178683"
    "144127461935561949669627002417368555744345925384275224184718385202716127015783952729102014"
    "850606111389694902294315413360375916657066256903455238672169922008443398352980324725285604"
    "802315720532675758147358190312305588395835973484510540261572242255926498453096842484137847"
    "010445108819535016652319334499549429994953654304666027843239794666289393464823312954846864"
    "007303061943225206493148761483349501485878134634611784514926947757918302750382791982668856"
    "3897931771862204186618328094482421875",
    669,
    "1.1798475245572565258836463208928683635416019155375711636848835775165384607837563334358573"
    "139483721738528709112002839025241484814734327595545701882805934328283407306215463816987617"
    "868314412746193556194966962700241736855574434592538427522418471838520271612701578395272910"
    "201485060611138969490229431541336037591665706625690345523867216992200844339835298032472528"
    "560480231572053267575814735819031230558839583597348451054026157224225592649845309684248413"
    "784701044510881953501665231933449954942999495365430466602784323979466628939346482331295484"
    "686400730306194322520649314876148334950148587813463461178451492694775791830275038279198266"
    "88563897931771862204186618328094482421875e-266"
  },
  {
    3.084511928870661e-266,
    934,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000308"
    "451192887066092417761273105509457702606990972946360902325553169062823977589756568024449658"
    "175724356488309154179460646805809256909673683564407525994808351915660324637706817216948128"
    "696931536528727801890757276085782199857118411215409399609811355261055441011497274965443507"
    "875714439025081725135936635724083460958833375244726025227358583038570468844027336764155121"
    "379327919906700196084011500970718879296894750190780528244326958169030004334188899958031100"
    "736000503800060878357350714914804570705622357760565433666940637984589125351111757820970086"
    "953232936644936090553222688991897010997992044067666407349229132534396534265265865464543393"
    "326124487430206499993801116943359375",
    668,
    "3.0845119288706609241776127310550945770260699097294636090232555316906282397758975656802444"
    "965817572435648830915417946064680580925690967368356440752599480835191566032463770681721694"
    "812869693153652872780189075727608578219985711841121540939960981135526105544101149727496544"
    "350787571443902508172513593663572408346095883337524472602522735858303857046884402733676415"
    "512137932791990670019608401150097071887929689475019078052824432695816903000433418889995803"
    "110073600050380006087835735071491480457070562235776056543366694063798458912535111175782097"
    "008695323293664493609055322268899189701099799204406766640734922913253439653426526586546454"
    "3393326124487430206499993801116943359375e-266"
  },
  {
    4.1367697432326765e-266,
    930,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000413"
    "676974323267653523408170351827433582573548126839675514706183070775290299430288530880767625"
    "546394325091548046642335686562972830560848535832043202751676119898478727926728300673178723"
    "638882161179194197617980196622390883823248545471276786695488010798658633161409367103816859"
    "479020234929452747630391237612010703256493228671989267940923791939790011605202831187408904"
    "570506065615283688529715886701601418737963348910052650002158795052277642750755249858943871"
    "277223658376131404786971097472549039306650994514211262733568954010543397543745196493711733"
    "971931671734205726078356664400682450680392885775405230731524652911620065749649698474965653"
    "36086860042996704578399658203125",
    664,
    "4.1367697432326765352340817035182743358257354812683967551470618307077529029943028853088076"
    "762554639432509154804664233568656297283056084853583204320275167611989847872792672830067317"
    "872363888216117919419761798019662239088382324854547127678669548801079865863316140936710381"
    "685947902023492945274763039123761201070325649322867198926794092379193979001160520283118740"
    "890457050606561528368852971588670160141873796334891005265000215879505227764275075524985894"
    "387127722365837613140478697109747254903930665099451421126273356895401054339754374519649371"
    "173397193167173420572607835666440068245068039288577540523073152465291162006574964969847496"
    "565336086860042996704578399658203125e-266"
  },
  {
    1.1356367562498619e-265,
    933,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000001135"
    "636756249861914531024438569847352542728356480390247148093097331978873723151459297710811496"
    "506381505534558309773790241440886806822023647805957728246263626555076263986569766308170038"
    "666025207186054552343785147353087788741804559931776255025847515183240478076966312904995778"
    "458881231949209035981411121604045278278843061697726854718396016385001732159199309344782124"
    "866924011933370531388442338358094661908139596342648995217761099977664396650272093960020341"
    "884435861220916521937924700057559397741056523935875704687296084673219908834218208302180369"
    "362225717108121963705936493691882457994726492485413609210469134484467589008870790296193420"
    "41002028054208494722843170166015625",
    668,
    "1.1356367562498619145310244385698473525427283564803902471480930973319788737231514592977108"
    "114965063815055345583097737902414408868068220236478059577282462636265550762639865697663081"
    "700386660252071860545523437851473530877887418045599317762550258475151832404780769663129049"
    "957784588812319492090359814111216040452782788430616977268547183960163850017321591993093447"
    "821248669240119333705313884423383580946619081395963426489952177610999776643966502720939600"
    "203418844358612209165219379247000575593977410565239358757046872960846732199088342182083021"
    "803693622257171081219637059364936918824579947264924854136092104691344844675890088707902961"
    "9342041002028054208494722843170166015625e-265"
  },
  {
    1.5494624584252724e-265,
    928,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000001549"
    "462458425272397590624958337535230325665434318107517130127590869403722238026359250431068983"
    "678010402175817581233673382519031060069659513380216030715235291169465216919075736584984253"
    "493928692951277180004339298850163756715521637210713685129495086748985049799710299739775854"
    "204400714728515170031210558291615920547395211099278260930260258856832464997016742744530897"
    "530563097291977203548901120184100112729702315119501770992283784215283680031165586637831064"
    "539612758832139342286457927335845030154873887997206834618872714160608665204019257084947262"
    "289315617219879609164279766856157954598104804131015062387788537884648628367603544198072285"
    "098533029668033123016357421875",
    663,
    "1.5494624584252723975906249583375352303256654343181075171301275908694037222380263592504310"
    "689836780104021758175812336733825190310600696595133802160307152352911694652169190757365849"
    "842534939286929512771800043392988501637567155216372107136851294950867489850497997102997397"
    "758542044007147285151700312105582916159205473952110992782609302602588568324649970167427445"
    "308975305630972919772035489011201841001127297023151195017709922837842152836800311655866378"
    "310645396127588321393422864579273358450301548738879972068346188727141606086652040192570849"
    "472622893156172198796091642797668561579545981048041310150623877885378846486283676035441980"
    "72285098533029668033123016357421875e-265"
  },
  {
    3.028575537095104e-265,
    931,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000003028"
    "575537095103868160868369690562103377227833072855355220692967731775893727875231391193715089"
    "712768013783170035525245663579807030327581339575641841515558865022387505800532674156354288"
    "525775481285433117832241724948681614868585096210717704389551477048999401562959159801158673"
    "751335958285076385209833933722745096602678100314306172093410279496783605205489310551622455"
    "888155041168642269270378241902705974557075324087799607548440887496138095190515484126289206"
    "547636858207838426948184712878923707785167016810786001253026125137784148959428881521605742"
    "355054257217013573044606787074642577830411006902168840320052258179872988498243578485424887"
    "247745573404245078563690185546875",
    666,
    "3.0285755370951038681608683696905621033772278330728553552206929677317758937278752313911937"
    "150897127680137831700355252456635798070303275813395756418415155588650223875058005326741563"
    "542885257754812854331178322417249486816148685850962107177043895514770489994015629591598011"
    "586737513359582850763852098339337227450966026781003143061720934102794967836052054893105516"
    "224558881550411686422692703782419027059745570753240877996075484408874961380951905154841262"
    "892065476368582078384269481847128789237077851670168107860012530261251377841489594288815216"
    "057423550542572170135730446067870746425778304110069021688403200522581798729884982435784854"
    "24887247745573404245078563690185546875e-265"
  },
  {
    9.42740333348115e-265,
    930,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000009427"
    "403333481150374852840475728938472680677003267432740281813334184073733241175012142621582185"
    "336547439135572303877902850839331914579032892703455962702636721963543941824273155260008806"
    "499771416561057873271352997399155480063881212524740495031457329664415784460015536413542144"
    "918144920982279053472736448128968434148254398788827723100221099352075038650140212247562074"
    "777411134643948266030983658278327971341011673659661946388286738125049216155470349506228879"
    "832318739988654166524681444784216263188371836785780017874527628767521143515008392129613740"
    "783506615926436788653883816643418293983827325355790595619401175112158521708318652601588105"
    "71056164917536079883575439453125",
    665,
    "9.4274033334811503748528404757289384726806770032674327402818133341840737332411750121426215"
    "821853365474391355723038779028508393319145790328927034559627026367219635439418242731552600"
    "088064997714165610578732713529973991554800638812125247404950314573296644157844600155364135"
    "421449181449209822790534727364481289684341482543987888277231002210993520750386501402122475"
    "620747774111346439482660309836582783279713410116736596619463882867381250492161554703495062"
    "288798323187399886541665246814447842162631883718367857800178745276287675211435150083921296"
    "137407835066159264367886538838166434182939838273253557905956194011751121585217083186526015"
    "8810571056164917536079883575439453125e-265"
  },
  {
    1.729174836691517e-264,
    929,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000017291"
    "748366915169805535173361311690336060863250937536754419109791788940622928061477911081864929"
    "176484648733039274002682060801271760199087832663704763871824868199399153100645610343074809"
    "480658801773804131716859477870218786924969042662004277956316388108603543168796578661210760"
    "299937517417296459170632249395100355118985709999359928300674405002289342033156169213644776"
    "362614794679384856915080325537702909530580016470514923381118473118916750055309888956981579"
    "069546748968265626703041678002524692963359611838622545244218073070283969366388186699448626"
    "513452331388797913337924984827533241380476016103568773774841888147830421940782334028208211"
    "4072181866504251956939697265625",
    665,
    "1.7291748366915169805535173361311690336060863250937536754419109791788940622928061477911081"
    "864929176484648733039274002682060801271760199087832663704763871824868199399153100645610343"
    "074809480658801773804131716859477870218786924969042662004277956316388108603543168796578661"
    "210760299937517417296459170632249395100355118985709999359928300674405002289342033156169213"
    "644776362614794679384856915080325537702909530580016470514923381118473118916750055309888956"
    "981579069546748968265626703041678002524692963359611838622545244218073070283969366388186699"
    "448626513452331388797913337924984827533241380476016103568773774841888147830421940782334028"
    "2082114072181866504251956939697265625e-264"
  },
  {
    2.1531241335994103e-264,
    928,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000021531"
    "241335994102581545421543905718553261205634596225205065032112054906757350946064401973884364"
    "945201367158919349888958734251838952734069161919669821826840595044492491299418032955578572"
    "499044671525846348582137071002979147208900977791692058838094574979400811697266763299744631"
    "812474416498884413904210064969720257993874449869399622491655605240174565953844905584693276"
    "335440945576505682668112743945032654000227377822639620983203242462816986727896006612405764"
    "230193507933353399837465664150855113198426026263463928569770441386979156660470583445551686"
    "157682934718600188777199278014219625707936771490825261708861973418770466995124900078106833"
    "234414807520806789398193359375",
    664,
    "2.1531241335994102581545421543905718553261205634596225205065032112054906757350946064401973"
    "884364945201367158919349888958734251838952734069161919669821826840595044492491299418032955"
    "578572499044671525846348582137071002979147208900977791692058838094574979400811697266763299"
    "744631812474416498884413904210064969720257993874449869399622491655605240174565953844905584"
    "693276335440945576505682668112743945032654000227377822639620983203242462816986727896006612"
    "405764230193507933353399837465664150855113198426026263463928569770441386979156660470583445"
    "551686157682934718600188777199278014219625707936771490825261708861973418770466995124900078"
    "106833234414807520806789398193359375e-264"
  },
  {
    7.094381157920946e-264,
    927,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000070943"
    "811579209456674393546810681034475092272127137349769143575260247208850042176344557556673219"
    "122391550803213462791453166083465372413326621666485198457943128035461525487634326545044008"
    "463689494846305885375027924541805941906149994290111567688775222681488025519832201394364297"
    "664644611649841656631410298544880487054675623483604810872383651838979512701583025652350883"
    "872278828061474056265146942266842117494952581371563236372454474548015943400930092469885047"
    "371085489250666827759464535698435489894328280374088995472760504795222945131995366238007333"
    "908788553441113595577240303829348570057347486203100038390329418341112596675191240280788917"
    "97122196294367313385009765625",
    663,
    "7.0943811579209456674393546810681034475092272127137349769143575260247208850042176344557556"
    "673219122391550803213462791453166083465372413326621666485198457943128035461525487634326545"
    "044008463689494846305885375027924541805941906149994290111567688775222681488025519832201394"
    "364297664644611649841656631410298544880487054675623483604810872383651838979512701583025652"
    "350883872278828061474056265146942266842117494952581371563236372454474548015943400930092469"
    "885047371085489250666827759464535698435489894328280374088995472760504795222945131995366238"
    "007333908788553441113595577240303829348570057347486203100038390329418341112596675191240280"
    "78891797122196294367313385009765625e-264"
  },
  {
    1.037244238361319e-263,
    922,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000103724"
    "423836131899697140862584597222526602109048921353546664564824864365134133293000960441974952"
    "192749638978020863794678225957572854635073507252775535164961781534910344480358232648898270"
    "595208601349292946268441297565981132921340471932372253107009896365339079786399349223759876"
    "084238301219869927432405499615954732945986405809249696734057569539201247561455043693431979"
    "851892080485777689297288928968886924569014126842036461159664146976253775304747117052161568"
    "491733834101439034106718671249125299580172355437057762980472448608774453369969366167791474"
    "556463187639204268002177968393288896566289594361861665631675773451618959540565523269606273"
    "970566689968109130859375",
    659,
    "1.0372442383613189969714086258459722252660210904892135354666456482486436513413329300096044"
    "197495219274963897802086379467822595757285463507350725277553516496178153491034448035823264"
    "889827059520860134929294626844129756598113292134047193237225310700989636533907978639934922"
    "375987608423830121986992743240549961595473294598640580924969673405756953920124756145504369"
    "343197985189208048577768929728892896888692456901412684203646115966414697625377530474711705"
    "216156849173383410143903410671867124912529958017235543705776298047244860877445336996936616"
    "779147455646318763920426800217796839328889656628959436186166563167577345161895954056552326"
    "9606273970566689968109130859375e-263"
  },
  {
    1.6682266784627316e-263,
    925,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000166822"
    "667846273160541267866349470998488926530117711608469902247927774169651256336593638231661507"
    "087835116568115883612889554220377387265788086029467082903772667492283129753284952803154692"
    "864161969436641653582454220202573116690753284416570045692728515391617646959570588674431452"
    "489502213935230455934325594185788806886599903199142652772824242601660519132167498352835764"
    "317466115385064714278273070236040126348067956104322034201963154877669659580960959116015504"
    "309990919735860857543724951224894412485861253372083552618782145415839816201669515226115269"
    "594664342648507328527043804187676658034302406565002561725539804953326260251143808854834560"
    "406743548810482025146484375",
    662,
    "1.6682266784627316054126786634947099848892653011771160846990224792777416965125633659363823"
    "166150708783511656811588361288955422037738726578808602946708290377266749228312975328495280"
    "315469286416196943664165358245422020257311669075328441657004569272851539161764695957058867"
    "443145248950221393523045593432559418578880688659990319914265277282424260166051913216749835"
    "283576431746611538506471427827307023604012634806795610432203420196315487766965958096095911"
    "601550430999091973586085754372495122489441248586125337208355261878214541583981620166951522"
    "611526959466434264850732852704380418767665803430240656500256172553980495332626025114380885"
    "4834560406743548810482025146484375e-263"
  },
  {
    5.516139500818587e-263,
    924,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000551613"
    "950081858674334153572992664732028557024426522550154174733465165230596318238791373343416884"
    "120642522016359703316218905950439100436732933226969082782786169311545430229194074907477275"
    "505071792114832480787270717891563710707060775822387041028381848087100938280170291721801830"
    "329053039152820516093748361374472209437575566477972180950716455329797096025381305113905665"
    "218350796699468382475616011900085039348409957586580232901106296333354593470671297966973754"
    "686923720279528690270848804834944868781772303189955069294165504271947347376758303349873670"
    "188126416909359452860562506156155770638857504834718541747179856392234209615590967423770507"
    "59387202560901641845703125",
    661,
    "5.5161395008185867433415357299266473202855702442652255015417473346516523059631823879137334"
    "341688412064252201635970331621890595043910043673293322696908278278616931154543022919407490"
    "747727550507179211483248078727071789156371070706077582238704102838184808710093828017029172"
    "180183032905303915282051609374836137447220943757556647797218095071645532979709602538130511"
    "390566521835079669946838247561601190008503934840995758658023290110629633335459347067129796"
    "697375468692372027952869027084880483494486878177230318995506929416550427194734737675830334"
    "987367018812641690935945286056250615615577063885750483471854174717985639223420961559096742"
    "377050759387202560901641845703125e-263"
  },
  {
    1.1456020464232508e-262,
    919,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000001145602"
    "046423250767505712227754368511921646055716825274815762124952088415596615808373168469160246"
    "822484321000751593912242355663941083913700201743951110967290075316318238089280652430141301"
    "846061544973865241960172181110143712808599775710021657762313485166911340457125405068169443"
    "597769672899512513065315032891834699262732190803538664503898660143905166724356797913967243"
    "092159996770248028065053108625679906847658543128458538824047203145322198383475617623515440"
    "685661963579197311084394989515298325853204831853278799683622888677696336423887444215207972"
    "301820659320054036430922363633794449770968981950183436840696718769035296336955553897496429"
    "271996021270751953125",
    657,
    "1.1456020464232507675057122277543685119216460557168252748157621249520884155966158083731684"
    "691602468224843210007515939122423556639410839137002017439511109672900753163182380892806524"
    "301413018460615449738652419601721811101437128085997757100216577623134851669113404571254050"
    "681694435977696728995125130653150328918346992627321908035386645038986601439051667243567979"
    "139672430921599967702480280650531086256799068476585431284585388240472031453221983834756176"
    "235154406856619635791973110843949895152983258532048318532787996836228886776963364238874442"
    "152079723018206593200540364309223636337944497709689819501834368406967187690352963369555538"
    "97496429271996021270751953125e-262"
  },
  {
    2.066981873698563e-262,
    919,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000002066981"
    "873698563122921143360917330077152183825978741816512849212959334684319910588999155563054688"
    "552042487345002100121410104592008447301112604018951149526548034818889350469271995230153107"
    "121905488673510151431288317341497991035832678659314223700819815496411587815428866045015831"
    "262423046888744843665130751642206640470596191378580566938709560071348382472890960052707601"
    "002928528908244858324430310039391980073912226287873779846512676915660226793076428687633580"
    "029559705975990491787084759280280522694043190075215391510094658283357751675552194266495084"
    "558603653401864419204157889257764915433524177885184490559892343474975885675437581312507973"
    "052561283111572265625",
    657,
    "2.0669818736985631229211433609173300771521838259787418165128492129593346843199105889991555"
    "630546885520424873450021001214101045920084473011126040189511495265480348188893504692719952"
    "301531071219054886735101514312883173414979910358326786593142237008198154964115878154288660"
    "450158312624230468887448436651307516422066404705961913785805669387095600713483824728909600"
    "527076010029285289082448583244303100393919800739122262878737798465126769156602267930764286"
    "876335800295597059759904917870847592802805226940431900752153915100946582833577516755521942"
    "664950845586036534018644192041578892577649154335241778851844905598923434749758856754375813"
    "12507973052561283111572265625e-262"
  },
  {
    2.7219184652406806e-262,
    921,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000002721918"
    "465240680599311320006972457103311639579426002251096560090007625639505670844147437379205645"
    "010682127734111790249585302708172431629584243318004556707822556023275699223159569591370783"
    "239612711241429196781408366427948378212113066025708694772668990832057641669827087574149746"
    "812408127966499041381146615522441977831117503820749504056135520839464464897001756137549386"
    "645421003994961660982636991816058707634629532208736580752557522215085254672270475091825412"
    "352898008934744774445658102994845690034954430706160250879228653865045231218586303252491596"
    "734499946262272599981000921593324190166840578960162765846077581071744836208048390346903033"
    "78738462924957275390625",
    659,
    "2.7219184652406805993113200069724571033116395794260022510965600900076256395056708441474373"
    "792056450106821277341117902495853027081724316295842433180045567078225560232756992231595695"
    "913707832396127112414291967814083664279483782121130660257086947726689908320576416698270875"
    "741497468124081279664990413811466155224419778311175038207495040561355208394644648970017561"
    "375493866454210039949616609826369918160587076346295322087365807525575222150852546722704750"
    "918254123528980089347447744456581029948456900349544307061602508792286538650452312185863032"
    "524915967344999462622725999810009215933241901668405789601627658460775810717448362080483903"
    "4690303378738462924957275390625e-262"
  },
  {
    5.081283542674735e-262,
    918,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000005081283"
    "542674734507555361192347761739898782036964893955130680321274137157152649946485593063004909"
    "706459221011223798923824797171065383696947332357164386562648631284907319152056535540918700"
    "633625172752827170086544593535922167621228929536166429804941621134212881365185181432564583"
    "359823929240075429731442028191437183310347432220683509750220766806479109777703074729320843"
    "100873842723453005642196320580008440470383450260257320640044222252784994928836137114749728"
    "430244847021545183325639612301159891871794896234026549933805970090869002934536386965363555"
    "076444252747751067112437620044942160741851891286975974928623936015302867130749575608206214"
    "38324451446533203125",
    656,
    "5.0812835426747345075553611923477617398987820369648939551306803212741371571526499464855930"
    "630049097064592210112237989238247971710653836969473323571643865626486312849073191520565355"
    "409187006336251727528271700865445935359221676212289295361664298049416211342128813651851814"
    "325645833598239292400754297314420281914371833103474322206835097502207668064791097777030747"
    "293208431008738427234530056421963205800084404703834502602573206400442222527849949288361371"
    "147497284302448470215451833256396123011598918717948962340265499338059700908690029345363869"
    "653635550764442527477510671124376200449421607418518912869759749286239360153028671307495756"
    "0820621438324451446533203125e-262"
  },
  {
    1.6876650530578208e-261,
    917,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000016876650"
    "530578207621570229952797785644682110844600730382828697983205295811627431458729119705328829"
    "243777096719665579994254117849067464302618370312175425353893782406244120818820673988034798"
    "407822409393517355645199684285124185451841481271060464145664678393904220903523430075496975"
    "928589508099769233106360418159498800345136248794879877104664837625430195393564841311745786"
    "113921932455203585099235154424807349383493024231486017916745364147754223484345302292969504"
    "582896292277375000907594687526700757402369552884056881367623282875449130686468138056422456"
    "062612577365229328729474127552578325580741923701134651242893653288936611778225937996467109"
    "7695827484130859375",
    656,
    "1.6876650530578207621570229952797785644682110844600730382828697983205295811627431458729119"
    "705328829243777096719665579994254117849067464302618370312175425353893782406244120818820673"
    "988034798407822409393517355645199684285124185451841481271060464145664678393904220903523430"
    "075496975928589508099769233106360418159498800345136248794879877104664837625430195393564841"
    "311745786113921932455203585099235154424807349383493024231486017916745364147754223484345302"
    "292969504582896292277375000907594687526700757402369552884056881367623282875449130686468138"
    "056422456062612577365229328729474127552578325580741923701134651242893653288936611778225937"
    "9964671097695827484130859375e-261"
  },
  {
    2.276883127404377e-261,
    916,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000022768831"
    "274043772176382445707214941137050004819857296684931317704292669869186330235385392974936135"
    "583425559939139633685462503798116631815904395994711723284574730699615502393988875684314432"
    "124512756030968025540881982395362955033752704466769495216363444832802971422047971056504139"
    "431435243914932068348782120967302820976716001570512136786783004641864845978867193614401580"
    "700868334274804209917513296587055016135300878023975459549254745964452344219743352297906726"
    "945322169747046918521525407257698517891017369605878013657851075208247736015104064738182848"
    "988116399189438269004085404020737336043775752403727901886111359007654197927905670439940877"
    "258777618408203125",
    655,
    "2.2768831274043772176382445707214941137050004819857296684931317704292669869186330235385392"
    "974936135583425559939139633685462503798116631815904395994711723284574730699615502393988875"
    "684314432124512756030968025540881982395362955033752704466769495216363444832802971422047971"
    "056504139431435243914932068348782120967302820976716001570512136786783004641864845978867193"
    "614401580700868334274804209917513296587055016135300878023975459549254745964452344219743352"
    "297906726945322169747046918521525407257698517891017369605878013657851075208247736015104064"
    "738182848988116399189438269004085404020737336043775752403727901886111359007654197927905670"
    "439940877258777618408203125e-261"
  },
  {
    6.29191834725416e-261,
    917,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000062919183"
    "472541600602721754928426186640791713919207735791429339647422629100785659522780692862200166"
    "897491701255556801910407106573720448533498719438330123476869662624070493420911558233828611"
    "769601530923806891926087928583503064203162299093150391838929264231274969654087918033117141"
    "971771938897830608189267730961476989883421151170779345326896958734602721257311763761737474"
    "562447460745166179129141239961621209497591323866150735669664874909372816156647940669783864"
    "971383067108306325702411659261123396468735369415898747994942738027015984558389681209132390"
    "569976243538178107157484089420553314117632985609329840997837206327623210677657539235951844"
    "6028232574462890625",
    656,
    "6.2919183472541600602721754928426186640791713919207735791429339647422629100785659522780692"
    "862200166897491701255556801910407106573720448533498719438330123476869662624070493420911558"
    "233828611769601530923806891926087928583503064203162299093150391838929264231274969654087918"
    "033117141971771938897830608189267730961476989883421151170779345326896958734602721257311763"
    "761737474562447460745166179129141239961621209497591323866150735669664874909372816156647940"
    "669783864971383067108306325702411659261123396468735369415898747994942738027015984558389681"
    "209132390569976243538178107157484089420553314117632985609329840997837206327623210677657539"
    "2359518446028232574462890625e-261"
  },
  {
    9.114070309202437e-261,
    915,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000091140703"
    "092024367155067044395221164452736675833151013263211220841163780409432338993660009501481842"
    "296218708349541201452089361954411446146803975046420958720254804201695009067469100778501634"
    "093845484930634578062741492093406330473821405287256234524085595589584395481645609669462191"
    "164735496063043957677650482578113270421879425006157651070495838384845023596106870545840216"
    "347820462109279705594517393524899645116315147968384649516659855648207409482502910088356334"
    "794730960275068225436448290138284866407640307260385855138228837609625546739209590904446038"
    "894150322252727288772880171330343400982118253807645982089771097148581113778220696985954418"
    "77841949462890625",
    654,
    "9.1140703092024367155067044395221164452736675833151013263211220841163780409432338993660009"
    "501481842296218708349541201452089361954411446146803975046420958720254804201695009067469100"
    "778501634093845484930634578062741492093406330473821405287256234524085595589584395481645609"
    "669462191164735496063043957677650482578113270421879425006157651070495838384845023596106870"
    "545840216347820462109279705594517393524899645116315147968384649516659855648207409482502910"
    "088356334794730960275068225436448290138284866407640307260385855138228837609625546739209590"
    "904446038894150322252727288772880171330343400982118253807645982089771097148581113778220696"
    "98595441877841949462890625e-261"
  },
  {
    2.5433112499173348e-260,
    909,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000254331124"
    "991733477931941256217101636450793693314232028988762740870132656644179963370891130188870966"
    "286895286909980735596949777809152245542352924571623599896408538766930694228602626651083775"
    "966167899682411094921866452845328367725707232322960573641809348591616813120749535926959438"
    "011678275350175333529052571637682063781316513136496003491077890873874611509896910053785379"
    "496743989865981922331313110583463218632304343712779751159601698778173244506374886663889558"
    "909304985006734120160002143650400648095280239164877277385388078424667808899882139383977796"
    "058528578685504934650156703479301676139297467896571367640791774822339732509135501459240913"
    "39111328125",
    649,
    "2.5433112499173347793194125621710163645079369331423202898876274087013265664417996337089113"
    "018887096628689528690998073559694977780915224554235292457162359989640853876693069422860262"
    "665108377596616789968241109492186645284532836772570723232296057364180934859161681312074953"
    "592695943801167827535017533352905257163768206378131651313649600349107789087387461150989691"
    "005378537949674398986598192233131311058346321863230434371277975115960169877817324450637488"
    "666388955890930498500673412016000214365040064809528023916487727738538807842466780889988213"
    "938397779605852857868550493465015670347930167613929746789657136764079177482233973250913550"
    "145924091339111328125e-260"
  },
  {
    4.8383034142174994e-260,
    914,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000483830341"
    "421749941042720698381277829592162948904446248604881177404794594426120880857998471497917185"
    "926395692514342169545748844407930478209370082097698966466303161266693588296916477694843719"
    "922217257850078663588627429918121588283573141295106697492102121175682912180817585981879144"
    "933649348743440084259962655671200187615096677180556085028974520284948959468340530672611063"
    "094169124950837753490510108692200824269667675272188181979310566617032157898763990519492387"
    "274070392056149682430213652686973136279413174816037833439958789749907392526753047381170083"
    "533242864428241299085122350500381603684503670453820188503635745750491858885311557969544082"
    "8800201416015625",
    654,
    "4.8383034142174994104272069838127782959216294890444624860488117740479459442612088085799847"
    "149791718592639569251434216954574884440793047820937008209769896646630316126669358829691647"
    "769484371992221725785007866358862742991812158828357314129510669749210212117568291218081758"
    "598187914493364934874344008425996265567120018761509667718055608502897452028494895946834053"
    "067261106309416912495083775349051010869220082426966767527218818197931056661703215789876399"
    "051949238727407039205614968243021365268697313627941317481603783343995878974990739252675304"
    "738117008353324286442824129908512235050038160368450367045382018850363574575049185888531155"
    "79695440828800201416015625e-260"
  },
  {
    9.557162041008455e-260,
    913,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000955716204"
    "100845536122313278764271137902613442730146513121346531376923699321195592174941483374402823"
    "022241130263997567506024287148387776798461354301437476844010018790776658181862075147468952"
    "135479241966863723499456267193061648534010724768410763732197245185338921411846718211238899"
    "707637053935778806345289337615172974840829152707894689434168397447083294697957498770890977"
    "466411575537371988696981399871854364602534911494453108101743200834799840591710632180550687"
    "823669819744527982460287775503302538194712879042143801864025232229533473067828671438671860"
    "982520349384527056410394878097917201397138322894657648983246446723616163154702007886953651"
    "905059814453125",
    653,
    "9.5571620410084553612231327876427113790261344273014651312134653137692369932119559217494148"
    "337440282302224113026399756750602428714838777679846135430143747684401001879077665818186207"
    "514746895213547924196686372349945626719306164853401072476841076373219724518533892141184671"
    "821123889970763705393577880634528933761517297484082915270789468943416839744708329469795749"
    "877089097746641157553737198869698139987185436460253491149445310810174320083479984059171063"
    "218055068782366981974452798246028777550330253819471287904214380186402523222953347306782867"
    "143867186098252034938452705641039487809791720139713832289465764898324644672361616315470200"
    "7886953651905059814453125e-260"
  },
  {
    2.3218163991190385e-259,
    912,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000002321816399"
    "119038536450898425044770684828672070243333428291577919705134892889011156980930242366813130"
    "676885496716060042971897736006971123622013129470717465474989351231296863119163127182130194"
    "734615854702366253421790700948554276270180610568750669698346074396782598841969164523910845"
    "529402228142249335459416846310411634554333872883229576923965865788474373561291533235429216"
    "484906094517687214723033131816735076669297720633441014191257900017539357278220796859945952"
    "519805757175348077621113645858326414929343829861412960782560144906919068890930223248032001"
    "378178220026349482108995119241996449400126609003650787411201237078750558850970264757052063"
    "94195556640625",
    653,
    "2.3218163991190385364508984250447706848286720702433334282915779197051348928890111569809302"
    "423668131306768854967160600429718977360069711236220131294707174654749893512312968631191631"
    "271821301947346158547023662534217907009485542762701806105687506696983460743967825988419691"
    "645239108455294022281422493354594168463104116345543338728832295769239658657884743735612915"
    "332354292164849060945176872147230331318167350766692977206334410141912579000175393572782207"
    "968599459525198057571753480776211136458583264149293438298614129607825601449069190688909302"
    "232480320013781782200263494821089951192419964494001266090036507874112012370787505588509702"
    "6475705206394195556640625e-259"
  },
  {
    4.4322942901196377e-259,
    911,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000004432294290"
    "119637686336031968093369465430899792116492210274548405275696535832050413393762041703842835"
    "100613439369769339676083420741214153242100785872435174265771853500939243101562641192258386"
    "136963986594969899104208353518232315947890478652282094992701578651352178291486176574508412"
    "703120463808237396444949047768512976373125132751101023361048082931070567956561305906497339"
    "743765419970162656685419340316738036723361873504929779450561348524467585813726066495148491"
    "623149065684298295442331640639910549887152295344393856875857793189322574286727246285838594"
    "808424556563360450020368516054773532563608037624243787652624903400377931461662228684872388"
    "8397216796875",
    652,
    "4.4322942901196376863360319680933694654308997921164922102745484052756965358320504133937620"
    "417038428351006134393697693396760834207412141532421007858724351742657718535009392431015626"
    "411922583861369639865949698991042083535182323159478904786522820949927015786513521782914861"
    "765745084127031204638082373964449490477685129763731251327511010233610480829310705679565613"
    "059064973397437654199701626566854193403167380367233618735049297794505613485244675858137260"
    "664951484916231490656842982954423316406399105498871522953443938568758577931893225742867272"
    "462858385948084245565633604500203685160547735325636080376242437876526249034003779314616622"
    "286848723888397216796875e-259"
  },
  {
    7.248624545665186e-259,
    904,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000007248624545"
    "665186202347257437087681926041137071824981981874333594377620530764610176592097380225029628"
    "645932726108934612739313174583226276319058355309446786731754362240349683816673035395724140"
    "251754746538460606263717734775468528471925761988836228340907693385947639524028564463791929"
    "467246414768399226257697924451237021644396948072980080190210135300113856092594880192491767"
    "239908048905028678287251757469304085099656023708399752447431206991973014885061270696168035"
    "586546178057383592410412468013840504378503033669907061366859852254705696051810053431151300"
    "980240986285936866882956104322697194439226699982917385446656854952607318409718573093414306"
    "640625",
    645,
    "7.2486245456651862023472574370876819260411370718249819818743335943776205307646101765920973"
    "802250296286459327261089346127393131745832262763190583553094467867317543622403496838166730"
    "353957241402517547465384606062637177347754685284719257619888362283409076933859476395240285"
    "644637919294672464147683992262576979244512370216443969480729800801902101353001138560925948"
    "801924917672399080489050286782872517574693040850996560237083997524474312069919730148850612"
    "706961680355865461780573835924104124680138405043785030336699070613668598522547056960518100"
    "534311513009802409862859368668829561043226971944392266999829173854466568549526073184097185"
    "73093414306640625e-259"
  },
  {
    1.4667081257360905e-258,
    909,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000014667081257"
    "360905122219545067067544733163746805490966430488449206414100150968687775032402756052292826"
    "192310903305035134133781411481712341507954193064787326215057170476767913122486458282807168"
    "688861118879940910902450113853945115500835908444037691717404158138311187678374885000008722"
    "719000441283697017571498316595670519134569889281021321327141354177750096990863112151715710"
    "379567247962185037697773480199531512073678158854750890584017056925661719743417354466583312"
    "700152556120020313103113546357174122883684230464811770930065682639372866674335927762613779"
    "704327156873657196475914112946384824437746406875000046416248136726956374786823289468884468"
    "07861328125",
    651,
    "1.4667081257360905122219545067067544733163746805490966430488449206414100150968687775032402"
    "756052292826192310903305035134133781411481712341507954193064787326215057170476767913122486"
    "458282807168688861118879940910902450113853945115500835908444037691717404158138311187678374"
    "885000008722719000441283697017571498316595670519134569889281021321327141354177750096990863"
    "112151715710379567247962185037697773480199531512073678158854750890584017056925661719743417"
    "354466583312700152556120020313103113546357174122883684230464811770930065682639372866674335"
    "927762613779704327156873657196475914112946384824437746406875000046416248136726956374786823"
    "28946888446807861328125e-258"
  },
  {
    3.832202635359104e-258,
    908,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000038322026353"
    "591041945549619532350115070542582231900572219630335353346952308164980213456390925558452250"
    "257609523249459132862945644945228012525507914866849886180070675458520881782697612666888814"
    "512039619026161160917276211912379314587278960377880520380258796896112144791614399610027999"
    "724187449071693914235140233274635231409141504165184730552533850275002320357839853008554356"
    "923544887943781694190440637960551088058658109244864183491834764218822863830891904140211787"
    "521427609138820308878984427755726564461594229503735299174555080532497679385174476335641680"
    "250480253832251745231631731715116558106582871154291903995157270168547825051064137369394302"
    "3681640625",
    650,
    "3.8322026353591041945549619532350115070542582231900572219630335353346952308164980213456390"
    "925558452250257609523249459132862945644945228012525507914866849886180070675458520881782697"
    "612666888814512039619026161160917276211912379314587278960377880520380258796896112144791614"
    "399610027999724187449071693914235140233274635231409141504165184730552533850275002320357839"
    "853008554356923544887943781694190440637960551088058658109244864183491834764218822863830891"
    "904140211787521427609138820308878984427755726564461594229503735299174555080532497679385174"
    "476335641680250480253832251745231631731715116558106582871154291903995157270168547825051064"
    "1373693943023681640625e-258"
  },
  {
    6.943148025853397e-258,
    905,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000069431480258"
    "533971970776226476780070833439447534419340955589877460326824250672274858385074795336068109"
    "179111518174172764432638216994241523277261186746220234484198234243683185988178010416964894"
    "015165298895010820982753705460755534031325385861623735456539486105828313019407233007056859"
    "766035924495330774146275818492932334977877716272934729294493458229693156484855489071519117"
    "809921208470019603290425328615499359681908998029766237263426183580374474662876558240267887"
    "653911901083546355304644391124352657346049170317211404039027681728984589057874362030696687"
    "237509291158628880287376106739222007686699951653712772122600815905002491490449756383895874"
    "0234375",
    647,
    "6.9431480258533971970776226476780070833439447534419340955589877460326824250672274858385074"
    "795336068109179111518174172764432638216994241523277261186746220234484198234243683185988178"
    "010416964894015165298895010820982753705460755534031325385861623735456539486105828313019407"
    "233007056859766035924495330774146275818492932334977877716272934729294493458229693156484855"
    "489071519117809921208470019603290425328615499359681908998029766237263426183580374474662876"
    "558240267887653911901083546355304644391124352657346049170317211404039027681728984589057874"
    "362030696687237509291158628880287376106739222007686699951653712772122600815905002491490449"
    "7563838958740234375e-258"
  },
  {
    9.18729939902564e-258,
    906,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000091872993990"
    "256398393531416600457280450059068320796514631056090469764224373506303894785318601668954036"
    "095739737917845209608647602558431586717467239245039611550399241162357633390875607028636644"
    "965611983869125852342343033041414416362095382760444374342874460306324504150263432744815692"
    "363238442130953624758949364538354946315916381176504061003068786853124720741143508000759975"
    "764912058080348409246107484554164396410838347897285039581676105234110261940749964363668337"
    "453955227885141656326569257081685102455251171799298121850947044540408510585924341296807202"
    "561779059541387540100382232276547130221839669816463156163874326232132716540945693850517272"
    "94921875",
    648,
    "9.1872993990256398393531416600457280450059068320796514631056090469764224373506303894785318"
    "601668954036095739737917845209608647602558431586717467239245039611550399241162357633390875"
    "607028636644965611983869125852342343033041414416362095382760444374342874460306324504150263"
    "432744815692363238442130953624758949364538354946315916381176504061003068786853124720741143"
    "508000759975764912058080348409246107484554164396410838347897285039581676105234110261940749"
    "964363668337453955227885141656326569257081685102455251171799298121850947044540408510585924"
    "341296807202561779059541387540100382232276547130221839669816463156163874326232132716540945"
    "69385051727294921875e-258"
  },
  {
    3.0584908958671776e-257,
    905,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000305849089586"
    "717763378339977282125527431971696049669791108458130402764519100128543072177565353659388779"
    "586721422062480805494048637982077473308559243951566395413290867225547157080991564751569942"
    "298190170759723875656705341842611794125335326790923669652646044512020828048743867346249036"
    "151195606903793715067066354657074373462967696577422694299341639036998899439039226956072612"
    "080444035045033212003702250424000664409060170693287784261706490929305618215679483092331173"
    "245544667709092235091659004587744352478247058204575908244766418147686277313391994062912785"
    "384861137995527175891946466904938951882934222675166659449058625952488910115789622068405151"
    "3671875",
    648,
    "3.0584908958671776337833997728212552743197169604966979110845813040276451910012854307217756"
    "535365938877958672142206248080549404863798207747330855924395156639541329086722554715708099"
    "156475156994229819017075972387565670534184261179412533532679092366965264604451202082804874"
    "386734624903615119560690379371506706635465707437346296769657742269429934163903699889943903"
    "922695607261208044403504503321200370225042400066440906017069328778426170649092930561821567"
    "948309233117324554466770909223509165900458774435247824705820457590824476641814768627731339"
    "199406291278538486113799552717589194646690493895188293422267516665944905862595248891011578"
    "96220684051513671875e-257"
  },
  {
    3.332730679024704e-257,
    901,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000333273067902"
    "470373369076750343925684542725953694117389008676174880079134735517335342013230631997238070"
    "193934483504697879606878598819097351469420895437630770473232697024322335525318888227840552"
    "509675600524255646902861973961176501903653653403505030125752870521573682831018912910951341"
    "784667074265786122307968298108639579889275677667923731753622660647224638526422602163660051"
    "587329345627129548295038999732084305256140469500499943939385349824496739105901495187144509"
    "403790073622444198594469799057201540623756926169050435684444872560782485693221692904431184"
    "025729121928690077617301262945222865568847219920199827225548716569392126984894275665283203"
    "125",
    644,
    "3.3327306790247037336907675034392568454272595369411738900867617488007913473551733534201323"
    "063199723807019393448350469787960687859881909735146942089543763077047323269702432233552531"
    "888822784055250967560052425564690286197396117650190365365340350503012575287052157368283101"
    "891291095134178466707426578612230796829810863957988927567766792373175362266064722463852642"
    "260216366005158732934562712954829503899973208430525614046950049994393938534982449673910590"
    "149518714450940379007362244419859446979905720154062375692616905043568444487256078248569322"
    "169290443118402572912192869007761730126294522286556884721992019982722554871656939212698489"
    "4275665283203125e-257"
  },
  {
    7.495951055510854e-257,
    902,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000749595105551"
    "085387993357548133567017547923550656468733893779401769112399719866594903832403760079779042"
    "083118660847570367526920460011559675059654679049732952640292043242998460981893007748067031"
    "656426282809466359894469361467018538353244246621869926740522382813897864544876313000958927"
    "410841974990867044072596703748130020628516337485234925482107813006708874340589640891230115"
    "442226884232370871021353363060667674967440542821407338711019067174933523839551077258961555"
    "519024174473109573715905891230366304042366550548239550909206406025216068495902106667958268"
    "045483901331181996094249458058254613569349715151079408675949267149007937405258417129516601"
    "5625",
    645,
    "7.4959510555108538799335754813356701754792355065646873389377940176911239971986659490383240"
    "376007977904208311866084757036752692046001155967505965467904973295264029204324299846098189"
    "300774806703165642628280946635989446936146701853835324424662186992674052238281389786454487"
    "631300095892741084197499086704407259670374813002062851633748523492548210781300670887434058"
    "964089123011544222688423237087102135336306066767496744054282140733871101906717493352383955"
    "107725896155551902417447310957371590589123036630404236655054823955090920640602521606849590"
    "210666795826804548390133118199609424945805825461356934971515107940867594926714900793740525"
    "84171295166015625e-257"
  },
  {
    1.6420266726944826e-256,
    900,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000001642026672694"
    "482563045480808805433079068803918442976624844924474480251023441651216759373536510779461101"
    "387097822996439468312579313736912924505271471334743943687783285894643203892794441529295961"
    "297750910791697510144519842247604265947164098444353337876295194407250912855850267351168187"
    "138849274287240408555196983493342848993346734258797359871022071513859831539102646945647750"
    "121861491898836431598708185841590353804696289196689248002933137399349822865047328348427089"
    "299098885077622250958963685912045889391996834305140188793741993208422307913615130573653829"
    "855907178088163064254606588826707581097183952044795199332050117391190724447369575500488281"
    "25",
    644,
    "1.6420266726944825630454808088054330790688039184429766248449244744802510234416512167593735"
    "365107794611013870978229964394683125793137369129245052714713347439436877832858946432038927"
    "944415292959612977509107916975101445198422476042659471640984443533378762951944072509128558"
    "502673511681871388492742872404085551969834933428489933467342587973598710220715138598315391"
    "026469456477501218614918988364315987081858415903538046962891966892480029331373993498228650"
    "473283484270892990988850776222509589636859120458893919968343051401887937419932084223079136"
    "151305736538298559071780881630642546065888267075810971839520447951993320501173911907244473"
    "6957550048828125e-256"
  },
  {
    3.9297951542678785e-256,
    900,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000003929795154267"
    "878456008504733028918171536724362499944041153562633585534912427028975058137822897912262984"
    "183657160412761749470512260593211414155724417537757763613607501931069650336151626581015217"
    "450051090530338351759584991103790007219099337019871347846863373747416772168678914168817718"
    "265590672635122732019115088840685528430843925553042590159268313066447848416962657456742414"
    "794883703454284745859365496546440420879817851523567449094063593575883610144277670625619799"
    "170002059668286123375913221735313424498080681026351680633436427361899279318539070692012686"
    "415588482858054953378973526677058505748782057145603377468212613621290074661374092102050781"
    "25",
    644,
    "3.9297951542678784560085047330289181715367243624999440411535626335855349124270289750581378"
    "228979122629841836571604127617494705122605932114141557244175377577636136075019310696503361"
    "516265810152174500510905303383517595849911037900072190993370198713478468633737474167721686"
    "789141688177182655906726351227320191150888406855284308439255530425901592683130664478484169"
    "626574567424147948837034542847458593654965464404208798178515235674490940635935758836101442"
    "776706256197991700020596682861233759132217353134244980806810263516806334364273618992793185"
    "390706920126864155884828580549533789735266770585057487820571456033774682126136212900746613"
    "7409210205078125e-256"
  },
  {
    8.581799124622545e-256,
    896,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000008581799124622"
    "545274109204209186824395047743643806198358871437940846768983603461477579537133361928488981"
    "466248817297542853053040287162959373058321187685047415890640017752408772117556502060458027"
    "930982317492005560331959933582336136059033009322539085521752257781672307684288330420314630"
    "742326494970432677741777569981162003949009552105845802595438484895481477647889805237520204"
    "087352263260159463187607331748523728837618007403262411653389035074149781685161401176489280"
    "819623071406746696594722690994184749489112933249743339804295660478638146733257119542873384"
    "7935092625307951380612214241116466115044845931920935999670518867787905037403106689453125",
    640,
    "8.5817991246225452741092042091868243950477436438061983588714379408467689836034614775795371"
    "333619284889814662488172975428530530402871629593730583211876850474158906400177524087721175"
    "565020604580279309823174920055603319599335823361360590330093225390855217522577816723076842"
    "883304203146307423264949704326777417775699811620039490095521058458025954384848954814776478"
    "898052375202040873522632601594631876073317485237288376180074032624116533890350741497816851"
    "614011764892808196230714067466965947226909941847494891129332497433398042956604786381467332"
    "571195428733847935092625307951380612214241116466115044845931920935999670518867787905037403"
    "106689453125e-256"
  },
  {
    1.3408011506316038e-255,
    899,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000013408011506316"
    "038440876133546193212552863655819964643115232648414608851460824412779274915182857333649885"
    "355057613535589004444814459107485028897497795071367868656328786520143020463781583605327723"
    "161143191908569107421765042632813837046883789348426353501078266264918417967508329964480915"
    "840659977385708189979315550219954067467015447979030230032418560932618957121652453967921612"
    "902472152407298980304430344846944714451589948955752543741374305143435234244962714228070337"
    "425103386522711053691324762952533567334027342714762568043635971696114388117135246368031971"
    "464249607959690187724230739800061456545774157963438893981233945851272437721490859985351562"
    "5",
    644,
    "1.3408011506316038440876133546193212552863655819964643115232648414608851460824412779274915"
    "182857333649885355057613535589004444814459107485028897497795071367868656328786520143020463"
    "781583605327723161143191908569107421765042632813837046883789348426353501078266264918417967"
    "508329964480915840659977385708189979315550219954067467015447979030230032418560932618957121"
    "652453967921612902472152407298980304430344846944714451589948955752543741374305143435234244"
    "962714228070337425103386522711053691324762952533567334027342714762568043635971696114388117"
    "135246368031971464249607959690187724230739800061456545774157963438893981233945851272437721"
    "4908599853515625e-255"
  },
  {
    3.074598537137766e-255,
    898,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000030745985371377"
    "658309028784463801492290964511331496359833560993895247133471183641249699681449552243237472"
    "526943183175593100589037207439488188256624549597810763535809834706047710687710457290485581"
    "924290225213346294452679474573378393850945499015474401134459423551359389643794124668315519"
    "672669186779315388763994430005381086549704757852670657762062506022796246089046295743650517"
    "804858349388973761791892305019059446288013874091013238298511526040244838541574337722601610"
    "235500443826320534848230461794121362728275321908101523040331777496075732358246932713115094"
    "147010664020747197187540795725493563067684264279201544756148223314085043966770172119140625",
    643,
    "3.0745985371377658309028784463801492290964511331496359833560993895247133471183641249699681"
    "449552243237472526943183175593100589037207439488188256624549597810763535809834706047710687"
    "710457290485581924290225213346294452679474573378393850945499015474401134459423551359389643"
    "794124668315519672669186779315388763994430005381086549704757852670657762062506022796246089"
    "046295743650517804858349388973761791892305019059446288013874091013238298511526040244838541"
    "574337722601610235500443826320534848230461794121362728275321908101523040331777496075732358"
    "246932713115094147010664020747197187540795725493563067684264279201544756148223314085043966"
    "770172119140625e-255"
  },
  {
    8.304947313449564e-255,
    893,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000083049473134495"
    "637562783372335308040826947988947911227489305609217047398914763715260955078525106138710236"
    "649863613071683150200481577000742878690170022000140942294975068168620988708304294276938218"
    "452197083289885669815644294270309079599562503903545750178103969934357011546883992223120023"
    "520675646862992886411946496804125192055831811634890274031010872847387842015145697773860849"
    "597339982312872612038388001018205969465761016324883622189071362242178588791610258925279276"
    "057872913529097906000892216093855115948305886496986647151398105787713255416001049323291103"
    "1509454101705527925632329519805437285589837757837639031777143827639520168304443359375",
    638,
    "8.3049473134495637562783372335308040826947988947911227489305609217047398914763715260955078"
    "525106138710236649863613071683150200481577000742878690170022000140942294975068168620988708"
    "304294276938218452197083289885669815644294270309079599562503903545750178103969934357011546"
    "883992223120023520675646862992886411946496804125192055831811634890274031010872847387842015"
    "145697773860849597339982312872612038388001018205969465761016324883622189071362242178588791"
    "610258925279276057872913529097906000892216093855115948305886496986647151398105787713255416"
    "001049323291103150945410170552792563232951980543728558983775783763903177714382763952016830"
    "4443359375e-255"
  },
  {
    1.4474411243713317e-254,
    893,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000144744112437133"
    "168672089773523152089127483001868331390354539841675354753639148030452143451158399149830703"
    "468107963464265324484638353285518964551567088265107554403873137980901895793359340168401743"
    "333116776228534381841104488553374627443452462050621912599530572335752731477616643207466908"
    "074162720773069693095320347726002453085865689342835245234405968927024173724912466484808925"
    "444162531720822045192149514230329917332339732221913704414744084396980182542537483986194053"
    "369028249533947601013624027010399377778605496838258484958724080102850988532089529783148304"
    "0255143194596899156238644941664600889849492774963390928633089060895144939422607421875",
    639,
    "1.4474411243713316867208977352315208912748300186833139035453984167535475363914803045214345"
    "115839914983070346810796346426532448463835328551896455156708826510755440387313798090189579"
    "335934016840174333311677622853438184110448855337462744345246205062191259953057233575273147"
    "761664320746690807416272077306969309532034772600245308586568934283524523440596892702417372"
    "491246648480892544416253172082204519214951423032991733233973222191370441474408439698018254"
    "253748398619405336902824953394760101362402701039937777860549683825848495872408010285098853"
    "208952978314830402551431945968991562386449416646008898494927749633909286330890608951449394"
    "22607421875e-254"
  },
  {
    1.9586457971591685e-254,
    895,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000195864579715916"
    "846281232176987731089358335622641445846231437198642769046938714077954561728750159436703931"
    "768778961167502825335230168197000394874158844610101174297566165868577953600731530124390559"
    "329841336668647646246830897438240884544600449291108586027013595024498883173088962375136713"
    "500786346081145555140083705002874641669261418256760684453011673855148836981775461769273081"
    "094502232268677049521764173824448926122463433864969518502089292929143453729778070058866366"
    "076873565593983608492151377810843966575548269321853362612694434735933876190649002975901808"
    "094524753036686073646775963942949006348507646514678359750405434169806540012359619140625",
    641,
    "1.9586457971591684628123217698773108935833562264144584623143719864276904693871407795456172"
    "875015943670393176877896116750282533523016819700039487415884461010117429756616586857795360"
    "073153012439055932984133666864764624683089743824088454460044929110858602701359502449888317"
    "308896237513671350078634608114555514008370500287464166926141825676068445301167385514883698"
    "177546176927308109450223226867704952176417382444892612246343386496951850208929292914345372"
    "977807005886636607687356559398360849215137781084396657554826932185336261269443473593387619"
    "064900297590180809452475303668607364677596394294900634850764651467835975040543416980654001"
    "2359619140625e-254"
  },
  {
    4.686715367492042e-254,
    894,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000468671536749204"
    "167908733372448499710764842207767871104099458795015130431473931089518435134487283977625540"
    "227879888252572412233135500143521775531145980390850992942943675033022307280642493210611912"
    "876680597019732548242153165649865911515051284692517601874935517887864604171008282403333235"
    "182426869285183040103510515405771113750924931599214258728181447617501366603311762665787718"
    "585372488276856930054676752303283379947593630831921961798272782320798169726042997383300927"
    "368593567279231201682441735999054882957709060176324055801659088172672443295479954938335855"
    "82346727901495312758623628952562840067335857459351888820719977957196533679962158203125",
    640,
    "4.6867153674920416790873337244849971076484220776787110409945879501513043147393108951843513"
    "448728397762554022787988825257241223313550014352177553114598039085099294294367503302230728"
    "064249321061191287668059701973254824215316564986591151505128469251760187493551788786460417"
    "100828240333323518242686928518304010351051540577111375092493159921425872818144761750136660"
    "331176266578771858537248827685693005467675230328337994759363083192196179827278232079816972"
    "604299738330092736859356727923120168244173599905488295770906017632405580165908817267244329"
    "547995493833585582346727901495312758623628952562840067335857459351888820719977957196533679"
    "962158203125e-254"
  },
  {
    1.3394037849881335e-253,
    893,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000001339403784988133"
    "476521505999288467858205925231145677811397409862151397128824178273542061091712001697600489"
    "741012371831182295184552571827172752298276597827206935919744912385758389839526972019112258"
    "391424075181085718387875308026359155400107105741151539207807846271470685787797198661393857"
    "298908016252366323393251433958973746042608632819414012825780849282620222035381866840828228"
    "463067214889679313826047600506279372884213489373140373034023253715300471680741138472033401"
    "549031643231565079911419324701771396734220770968637372847417278863808972654184165400453746"
    "9523907692015816885749998277404037048816097012216008721452453755773603916168212890625",
    640,
    "1.3394037849881334765215059992884678582059252311456778113974098621513971288241782735420610"
    "917120016976004897410123718311822951845525718271727522982765978272069359197449123857583898"
    "395269720191122583914240751810857183878753080263591554001071057411515392078078462714706857"
    "877971986613938572989080162523663233932514339589737460426086328194140128257808492826202220"
    "353818668408282284630672148896793138260476005062793728842134893731403730340232537153004716"
    "807411384720334015490316432315650799114193247017713967342207709686373728474172788638089726"
    "541841654004537469523907692015816885749998277404037048816097012216008721452453755773603916"
    "168212890625e-253"
  },
  {
    2.3636119615931954e-253,
    892,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000002363611961593195"
    "446025016786923891101115432649147943090987270779390484348136119955071672482176734551019886"
    "979338416843661814920768214468158394547246496896565295091270383161124144797268623314664619"
    "388240949767977379143220992333278223132260837716239494499620700131088794260669791925469170"
    "854867933113640126889557448864829949832543487809635599055594111467444144186491447911019685"
    "947006619810233218963054329841873754169720351780057245913342890720914510934494465099588466"
    "301358667734487117385222582358444773436903965534204976160588602232307735599284641966598526"
    "646351415873074695035701931182197892727567777038100160780231817625463008880615234375",
    639,
    "2.3636119615931954460250167869238911011154326491479430909872707793904843481361199550716724"
    "821767345510198869793384168436618149207682144681583945472464968965652950912703831611241447"
    "972686233146646193882409497679773791432209923332782231322608377162394944996207001310887942"
    "606697919254691708548679331136401268895574488648299498325434878096355990555941114674441441"
    "864914479110196859470066198102332189630543298418737541697203517800572459133428907209145109"
    "344944650995884663013586677344871173852225823584447734369039655342049761605886022323077355"
    "992846419665985266463514158730746950357019311821978927275677770381001607802318176254630088"
    "80615234375e-253"
  },
  {
    2.941610445695594e-253,
    891,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000002941610445695593"
    "722154377639815670668588766122303313614997814419541448706988004808096826758429733606650625"
    "431153569494377252120909156008352007049354054676255691194440055586490578539069593613203273"
    "212857982258930235229317177289893607816570293832641314174027421888172164237619051576765081"
    "507797361682755327770880513040405757582635820404630355922568672916090185698671904807786729"
    "693180484357443084910488894820841696016046353183524828315637170466422689323272068186618987"
    "108680163778027664811007531304324861921612998953412634463343014399966471714416366962788461"
    "68977248472239754341550217931078610631942006625738628144972608424723148345947265625",
    638,
    "2.9416104456955937221543776398156706685887661223033136149978144195414487069880048080968267"
    "584297336066506254311535694943772521209091560083520070493540546762556911944400555864905785"
    "390695936132032732128579822589302352293171772898936078165702938326413141740274218881721642"
    "376190515767650815077973616827553277708805130404057575826358204046303559225686729160901856"
    "986719048077867296931804843574430849104888948208416960160463531835248283156371704664226893"
    "232720681866189871086801637780276648110075313043248619216129989534126344633430143999664717"
    "144163669627884616897724847223975434155021793107861063194200662573862814497260842472314834"
    "5947265625e-253"
  },
  {
    5.910628843543644e-253,
    890,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000005910628843543644"
    "183292532143743927306650247731203331124696823772527675163921479725896891521721045849840651"
    "915999551426802023426459381988765566611779796927133403617908751963516378284148518150925815"
    "308915225628892551785526041987019154864916301964965994784344521781015097700885197871228199"
    "743117772247657733964373336298903334302413332395800228970615626622172238942304148983022767"
    "574352760471327229523240260429752604361048901195445218743348681868283996003972638298232644"
    "555585672903656506864598109553545629515220084227822470783061790824274297798382230073449857"
    "0619115622471228714886455545532536860558157909562027043648413382470607757568359375",
    637,
    "5.9106288435436441832925321437439273066502477312033311246968237725276751639214797258968915"
    "217210458498406519159995514268020234264593819887655666117797969271334036179087519635163782"
    "841485181509258153089152256288925517855260419870191548649163019649659947843445217810150977"
    "008851978712281997431177722476577339643733362989033343024133323958002289706156266221722389"
    "423041489830227675743527604713272295232402604297526043610489011954452187433486818682839960"
    "039726382982326445555856729036565068645981095535456295152200842278224707830617908242742977"
    "983822300734498570619115622471228714886455545532536860558157909562027043648413382470607757"
    "568359375e-253"
  },
  {
    1.6810396848558274e-252,
    889,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000016810396848558273"
    "510235799637525049545294830249948222214411090835531380563125319563203943253866909216175104"
    "325704880448761140455703556626180185908048579022126094709857134231392761386602121769040513"
    "881861098500622660566656475507360117047354461533394875438034303678412951001675794861033325"
    "878295628561693365087411561166993821573800278394690338640699834125530610486716022332187450"
    "360770206194564289972002012617700416236636708103854388850007470083332807751966286065831315"
    "220704183198185446579202353813809894749489068183936217686839899494855221091836732260120480"
    "167878388960328175864398165605800093799944427974679683757130987942218780517578125",
    637,
    "1.6810396848558273510235799637525049545294830249948222214411090835531380563125319563203943"
    "253866909216175104325704880448761140455703556626180185908048579022126094709857134231392761"
    "386602121769040513881861098500622660566656475507360117047354461533394875438034303678412951"
    "001675794861033325878295628561693365087411561166993821573800278394690338640699834125530610"
    "486716022332187450360770206194564289972002012617700416236636708103854388850007470083332807"
    "751966286065831315220704183198185446579202353813809894749489068183936217686839899494855221"
    "091836732260120480167878388960328175864398165605800093799944427974679683757130987942218780"
    "517578125e-252"
  },
  {
    3.105977744955943e-252,
    887,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000031059777449559429"
    "265900676399183362537453270079840668482441014488765656125936602920378930602673659646522508"
    "075048508831193127035440881908926520811716105080120700006784839696588034619889411366519970"
    "659057734269692442626166725983900779675226751564905756203390337365440185753314262361871401"
    "259321742934139049311496866768064298338564655493854637072357239725464925528075313439590693"
    "619626864606148281041428613600174497025784538263345957961162159163723866639150220568159037"
    "280157348845983734567189129321409288748503731499488443530660279868727628854598982775836943"
    "5248803257946775797031683187521866108295187647314605783321894705295562744140625",
    635,
    "3.1059777449559429265900676399183362537453270079840668482441014488765656125936602920378930"
    "602673659646522508075048508831193127035440881908926520811716105080120700006784839696588034"
    "619889411366519970659057734269692442626166725983900779675226751564905756203390337365440185"
    "753314262361871401259321742934139049311496866768064298338564655493854637072357239725464925"
    "528075313439590693619626864606148281041428613600174497025784538263345957961162159163723866"
    "639150220568159037280157348845983734567189129321409288748503731499488443530660279868727628"
    "854598982775836943524880325794677579703168318752186610829518764731460578332189470529556274"
    "4140625e-252"
  },
  {
    5.296557380144569e-252,
    886,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000052965573801445690"
    "722158614076883511123532332685924214717774752426044438667279139986749437517515188177140269"
    "548537520221293766437965393848671855127178764030257644725829744125373451251905877184795307"
    "199084173034379573279304521433472921187557715830098799832308554648947403113284151115720925"
    "616345099709078914248467108669426864597398228408661760562863344203484553975704767864965698"
    "077298580010056954322574128346908080869265075775573926971696184685845668852822480375903592"
    "455411221140540262039171474711461420140476650439880829255552973718941456131767751776122974"
    "578049807479540550113655955376254567783245175149886563303880393505096435546875",
    634,
    "5.2965573801445690722158614076883511123532332685924214717774752426044438667279139986749437"
    "517515188177140269548537520221293766437965393848671855127178764030257644725829744125373451"
    "251905877184795307199084173034379573279304521433472921187557715830098799832308554648947403"
    "113284151115720925616345099709078914248467108669426864597398228408661760562863344203484553"
    "975704767864965698077298580010056954322574128346908080869265075775573926971696184685845668"
    "852822480375903592455411221140540262039171474711461420140476650439880829255552973718941456"
    "131767751776122974578049807479540550113655955376254567783245175149886563303880393505096435"
    "546875e-252"
  },
  {
    1.4534181673457758e-251,
    886,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000145341816734577584"
    "009490552366952210486647130952500537974265197678532370418359567919765802256363710982793213"
    "214068181046129672798657879984495167728061963953124586161238507439156789985644488887055291"
    "390992087782055270133690654450277439818859160478101811450411073560794624280264880695474926"
    "329152097412240493821112790893435756312850326668997982936470519578964687774479451070591815"
    "469564082406643364225160041453045884503886325221070737739693194936546919810702692652603043"
    "565212622312212995327472532618638556231986680206742586143722403550783309924285289349297072"
    "101420663057753817823364411493644089180821143969524200656451284885406494140625",
    635,
    "1.4534181673457758400949055236695221048664713095250053797426519767853237041835956791976580"
    "225636371098279321321406818104612967279865787998449516772806196395312458616123850743915678"
    "998564448888705529139099208778205527013369065445027743981885916047810181145041107356079462"
    "428026488069547492632915209741224049382111279089343575631285032666899798293647051957896468"
    "777447945107059181546956408240664336422516004145304588450388632522107073773969319493654691"
    "981070269265260304356521262231221299532747253261863855623198668020674258614372240355078330"
    "992428528934929707210142066305775381782336441149364408918082114396952420065645128488540649"
    "4140625e-251"
  },
  {
    2.3938479247234097e-251,
    885,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000239384792472340970"
    "185504585115448173677393615362009509028148250898916697102141063617998075450938596876943234"
    "520222814663334325501959922339511022417169200258138866716240572516473903243780418299855591"
    "784687744221561467373049527462784044598049545505369194626527596159094494289488416114618612"
    "591881209540514149037606579125729900073752772185000313791009131273648743151758347743969426"
    "824721451716012475547094198982983592160806543815234719036377453626738864303434918589623115"
    "748337487232265520789623861131272396012356699775618072348868787763728788948474012071889835"
    "56061557308620341306536104552493931900414292357481826911680400371551513671875",
    634,
    "2.3938479247234097018550458511544817367739361536200950902814825089891669710214106361799807"
    "545093859687694323452022281466333432550195992233951102241716920025813886671624057251647390"
    "324378041829985559178468774422156146737304952746278404459804954550536919462652759615909449"
    "428948841611461861259188120954051414903760657912572990007375277218500031379100913127364874"
    "315175834774396942682472145171601247554709419898298359216080654381523471903637745362673886"
    "430343491858962311574833748723226552078962386113127239601235669977561807234886878776372878"
    "894847401207188983556061557308620341306536104552493931900414292357481826911680400371551513"
    "671875e-251"
  },
  {
    5.13191114201037e-251,
    884,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000513191114201037004"
    "992362062638780342121062029509309775518579982202893087476301157658778928854723620392753220"
    "523269264416102554328703892784714509871799045575377396811614393505255988764201594860924728"
    "188446262020599581622437360640217692654088383155871759121945521808738422425573794503626410"
    "345483286454198260168164150495417189876565458563960716202617750457144060022015399770404420"
    "122250191866657902531789248571633199703390242215773002375760167632603757343376191928042229"
    "339921704578692579360050525881528852289207475083143721754146523124340918095432167038719733"
    "3315664529506371145190073701706340823791574479173505096696317195892333984375",
    633,
    "5.1319111420103700499236206263878034212106202950930977551857998220289308747630115765877892"
    "885472362039275322052326926441610255432870389278471450987179904557537739681161439350525598"
    "876420159486092472818844626202059958162243736064021769265408838315587175912194552180873842"
    "242557379450362641034548328645419826016816415049541718987656545856396071620261775045714406"
    "002201539977040442012225019186665790253178924857163319970339024221577300237576016763260375"
    "734337619192804222933992170457869257936005052588152885228920747508314372175414652312434091"
    "809543216703871973333156645295063711451900737017063408237915744791735050966963171958923339"
    "84375e-251"
  },
  {
    8.74142907505523e-251,
    883,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000874142907505522955"
    "463100673044175380899987646178394042990683537048153973431496755091359582283788293865850167"
    "188491498788205872086312288388767426146999394659547867198614986995481660971205928823308477"
    "880542196613894427562357498716843156874530436462166924145700803375823071463433379915920838"
    "348016899016107847998673137727450201118541698490307800110552346677522177410624411455346538"
    "905850111542451572988081998398568760735245436792821070637754721727631828680882026195550243"
    "758494821815186708842256768881464944978789741055024675233939100433403212219702473933537591"
    "600476985467616468889370663931655085210348232749311137013137340545654296875",
    632,
    "8.7414290750552295546310067304417538089998764617839404299068353704815397343149675509135958"
    "228378829386585016718849149878820587208631228838876742614699939465954786719861498699548166"
    "097120592882330847788054219661389442756235749871684315687453043646216692414570080337582307"
    "146343337991592083834801689901610784799867313772745020111854169849030780011055234667752217"
    "741062441145534653890585011154245157298808199839856876073524543679282107063775472172763182"
    "868088202619555024375849482181518670884225676888146494497878974105502467523393910043340321"
    "221970247393353759160047698546761646888937066393165508521034823274931113701313734054565429"
    "6875e-251"
  },
  {
    2.0810847152471556e-250,
    881,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000002081084715247155625"
    "440131123203218872844726711736017571789163675562034816765950484069543464753999344218966790"
    "668885556434954834640963848940274729780129572762106340724642693902978618255323860021609284"
    "800723548659943879732460682150385691303824800896612452662280810299151162782505065316201313"
    "287687887744161894093458421161051328341356651611615952602293086761737680125529654918947772"
    "720360259548131643864589751988643147070399821513779604354178902911542059198107738793838794"
    "825326292221503233637576194543561945855186719195814109635961633124924541600378757481248046"
    "5286687861085268674943439588960990282995311417835182510316371917724609375",
    631,
    "2.0810847152471556254401311232032188728447267117360175717891636755620348167659504840695434"
    "647539993442189667906688855564349548346409638489402747297801295727621063407246426939029786"
    "182553238600216092848007235486599438797324606821503856913038248008966124526622808102991511"
    "627825050653162013132876878877441618940934584211610513283413566516116159526022930867617376"
    "801255296549189477727203602595481316438645897519886431470703998215137796043541789029115420"
    "591981077387938387948253262922215032336375761945435619458551867191958141096359616331249245"
    "416003787574812480465286687861085268674943439588960990282995311417835182510316371917724609"
    "375e-250"
  },
  {
    3.470210840000035e-250,
    879,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000003470210840000034776"
    "617506383084352377035377861826138134977804691284033800891212333097988673493814013195364736"
    "504043808795941139087383925791711802331911838123827244326417953056298727749836386220636723"
    "508760945712336499953240251855646705291887761133361595958660743063765329731272882479029390"
    "612416402919177315324330478981790631410002131020758100979035042132866099441313028954497942"
    "180207200515744500844214749161058296204205458513208863139333891611154059887707939231034083"
    "780526190117163341860564382148578700780055367272010676619957857904664512865881373748236253"
    "48339837864856184966498995800830706937034619841142557561397552490234375",
    629,
    "3.4702108400000347766175063830843523770353778618261381349778046912840338008912123330979886"
    "734938140131953647365040438087959411390873839257917118023319118381238272443264179530562987"
    "277498363862206367235087609457123364999532402518556467052918877611333615959586607430637653"
    "297312728824790293906124164029191773153243304789817906314100021310207581009790350421328660"
    "994413130289544979421802072005157445008442147491610582962042054585132088631393338916111540"
    "598877079392310340837805261901171633418605643821485787007800553672720106766199578579046645"
    "128658813737482362534833983786485618496649899580083070693703461984114255756139755249023437"
    "5e-250"
  },
  {
    7.63186683615216e-250,
    879,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000007631866836152159702"
    "492103294116391219584304736080757263385584406312313745307865574620244170078956471117970699"
    "169233373895272809046711033266481008823220114800551436505352505370172613800372893811656190"
    "836025152631706262769131163996536699740657297258151427870476072005518538156986453060386081"
    "899065708959573969700226946302971992875773637706668478397145737128433755602138394162137298"
    "950946352337338739362742305387139962103296257360220339421554482680435146681018012505959237"
    "765754657839926762185576094185762219623704191708392077042381946054396918836354683063036494"
    "84833033050509910062272350022606615116416151067824102938175201416015625",
    629,
    "7.6318668361521597024921032941163912195843047360807572633855844063123137453078655746202441"
    "700789564711179706991692333738952728090467110332664810088232201148005514365053525053701726"
    "138003728938116561908360251526317062627691311639965366997406572972581514278704760720055185"
    "381569864530603860818990657089595739697002269463029719928757736377066684783971457371284337"
    "556021383941621372989509463523373387393627423053871399621032962573602203394215544826804351"
    "466810180125059592377657546578399267621855760941857622196237041917083920770423819460543969"
    "188363546830630364948483303305050991006227235002260661511641615106782410293817520141601562"
    "5e-250"
  },
  {
    1.5834019337617896e-249,
    872,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000015834019337617895673"
    "529802567022957766337379198918511085508103045110528907733717602402102884849998115672191187"
    "006109820501000606499411634194787087402593470868198920066841575310557741200108285451578606"
    "682174248568014560625483576893416195879013848840237648540669357740358470908301870658810585"
    "475242568679239518498422119432276079112931267920833853415931213541677648134703690053765744"
    "168086411452423513659991329474857843477918188270714321800366826427248381640323472424199547"
    "164205151630092903849126794586850072188668859638179105507019966741223228692609209496056165"
    "7785392136164166225682260746854268518291064538061618804931640625",
    623,
    "1.5834019337617895673529802567022957766337379198918511085508103045110528907733717602402102"
    "884849998115672191187006109820501000606499411634194787087402593470868198920066841575310557"
    "741200108285451578606682174248568014560625483576893416195879013848840237648540669357740358"
    "470908301870658810585475242568679239518498422119432276079112931267920833853415931213541677"
    "648134703690053765744168086411452423513659991329474857843477918188270714321800366826427248"
    "381640323472424199547164205151630092903849126794586850072188668859638179105507019966741223"
    "2286926092094960561657785392136164166225682260746854268518291064538061618804931640625e-249"
  },
  {
    2.760497402953147e-249,
    876,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000027604974029531470755"
    "379958955203393724195949942057591244473815246912413391595013067101010555177517397100426985"
    "139428091311014234980202789586468665938033192658135488171299928713448075884778010385159464"
    "580198234297906418071487197676982517965040927251347607393335338895689232107223514901456151"
    "263751669808618192311950045569505026974147140173697808659679428521673427481132186985576932"
    "729538148337201682004999096184931755553173054112814485123238132174498935401641579922957354"
    "412202193532663584112586965020021304973020405545448709984133475441937262407388611982403639"
    "22061035190857842312351787435187322472529558581300079822540283203125",
    627,
    "2.7604974029531470755379958955203393724195949942057591244473815246912413391595013067101010"
    "555177517397100426985139428091311014234980202789586468665938033192658135488171299928713448"
    "075884778010385159464580198234297906418071487197676982517965040927251347607393335338895689"
    "232107223514901456151263751669808618192311950045569505026974147140173697808659679428521673"
    "427481132186985576932729538148337201682004999096184931755553173054112814485123238132174498"
    "935401641579922957354412202193532663584112586965020021304973020405545448709984133475441937"
    "26240738861198240363922061035190857842312351787435187322472529558581300079822540283203125e"
    "-249"
  },
  {
    4.645119455324602e-249,
    877,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000046451194553246022272"
    "173425910248807842189717398564905420998902159587754760551012912411892518528475138617823127"
    "607714941375176711195036461654593756557922664803433514884132688607890111007414815002525439"
    "250352630552903702165581057093062553194679218849243341393884915957796909710627176082816754"
    "842114154611178188766399561619813289346817198949598305014082475316385753923603720505835516"
    "199626324546859661234530477365388955308858864327508806609889314792645398167383551277259549"
    "976575383096677175116552815735162561369485758686904947085020888284307723768352498369060642"
    "533108685449620935516154367726782137282270923606120049953460693359375",
    628,
    "4.6451194553246022272173425910248807842189717398564905420998902159587754760551012912411892"
    "518528475138617823127607714941375176711195036461654593756557922664803433514884132688607890"
    "111007414815002525439250352630552903702165581057093062553194679218849243341393884915957796"
    "909710627176082816754842114154611178188766399561619813289346817198949598305014082475316385"
    "753923603720505835516199626324546859661234530477365388955308858864327508806609889314792645"
    "398167383551277259549976575383096677175116552815735162561369485758686904947085020888284307"
    "723768352498369060642533108685449620935516154367726782137282270923606120049953460693359375"
    "e-249"
  },
  {
    9.91990200069646e-249,
    874,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000099199020006964591277"
    "293509665148114175738068218299722216171684761860715405043509729734134051835419456145780406"
    "172178277413161498636992137303800662181445156443401801518547889441823791990054965892553944"
    "073623865723326893735869331809786597219148070113001514189219716428613451801337230880142783"
    "809598221914169988718354134816229989291886742495498869896064739850070478724667412357597823"
    "910441887617574862626088827907557313143938169737105103111386182522107665148933770898445415"
    "333007078802205277104248481222090156777529683923001278925352576743515243575159677709123598"
    "388542132205569638845363104884267357874705339781939983367919921875",
    625,
    "9.9199020006964591277293509665148114175738068218299722216171684761860715405043509729734134"
    "051835419456145780406172178277413161498636992137303800662181445156443401801518547889441823"
    "791990054965892553944073623865723326893735869331809786597219148070113001514189219716428613"
    "451801337230880142783809598221914169988718354134816229989291886742495498869896064739850070"
    "478724667412357597823910441887617574862626088827907557313143938169737105103111386182522107"
    "665148933770898445415333007078802205277104248481222090156777529683923001278925352576743515"
    "243575159677709123598388542132205569638845363104884267357874705339781939983367919921875e-2"
    "49"
  },
  {
    2.5293584256227207e-248,
    874,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000252935842562272067344"
    "366904488604388635604233221972876730712892277079692204821068839754613660527696056655673032"
    "934775744927789366034483299943037941202054718946505243272833507365629419200820002804983158"
    "179977271757167172107499183953448138573770848806627257225257124535066798220272652101781155"
    "228576723908703627594244414733748435575816138235978871187024048835402668122957444444007747"
    "178455282919057421350105555224402231830887206967806285670112170221507665071279926429611065"
    "289317694700042420246011589644121085837055088423442887441142392232124354630488844367280025"
    "213630868625148079983856914954021277708307025022804737091064453125",
    626,
    "2.5293584256227206734436690448860438863560423322197287673071289227707969220482106883975461"
    "366052769605665567303293477574492778936603448329994303794120205471894650524327283350736562"
    "941920082000280498315817997727175716717210749918395344813857377084880662725722525712453506"
    "679822027265210178115522857672390870362759424441473374843557581613823597887118702404883540"
    "266812295744444400774717845528291905742135010555522440223183088720696780628567011217022150"
    "766507127992642961106528931769470004242024601158964412108583705508842344288744114239223212"
    "4354630488844367280025213630868625148079983856914954021277708307025022804737091064453125e-"
    "248"
  },
  {
    6.890528046616735e-248,
    874,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000689052804661673526327"
    "766200722678956074254195627208870727866361131430199870943435827924716622928660764729397730"
    "215669170795196785528888685159578923864553514694529384313751243344600123412643351227172604"
    "529578659637209614120604229768515984554638080689615221252011195162819761944782242323568906"
    "379347205811122606796367345501745911420372302547838391326416268023703544913664214321284062"
    "433477199095122086565542807986691910549101079454291551252805162525706181057897779052882188"
    "595582910816002410243781113060844501237388400738713446722875308537861185231999546786141281"
    "236847796378186571867436281679797360766315250657498836517333984375",
    626,
    "6.8905280466167352632776620072267895607425419562720887072786636113143019987094343582792471"
    "662292866076472939773021566917079519678552888868515957892386455351469452938431375124334460"
    "012341264335122717260452957865963720961412060422976851598455463808068961522125201119516281"
    "976194478224232356890637934720581112260679636734550174591142037230254783839132641626802370"
    "354491366421432128406243347719909512208656554280798669191054910107945429155125280516252570"
    "618105789777905288218859558291081600241024378111306084450123738840073871344672287530853786"
    "1185231999546786141281236847796378186571867436281679797360766315250657498836517333984375e-"
    "248"
  },
  {
    9.9700406511611e-248,
    870,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000997004065116110063804"
    "500584101227039020675203594199457958995118256301185979588665807563367554229093351299347926"
    "878163027036020163695376425344287053227483975642933460875671431629596491665913660803544069"
    "466490750869997096100357298585967016002827929499382370060231163916146463812768120436189614"
    "584850362451100469688146558380363769218710763222339968107214709966218547012769183801629746"
    "774301015743591908967152567417961150919091220856185361682832504647890500198973106127845308"
    "319244621499182967427340316386207631543160167022751161619375392013461083373840233598023551"
    "41440629266624923426713678065169688125024549663066864013671875",
    622,
    "9.9700406511611006380450058410122703902067520359419945795899511825630118597958866580756336"
    "755422909335129934792687816302703602016369537642534428705322748397564293346087567143162959"
    "649166591366080354406946649075086999709610035729858596701600282792949938237006023116391614"
    "646381276812043618961458485036245110046968814655838036376921871076322233996810721470996621"
    "854701276918380162974677430101574359190896715256741796115091909122085618536168283250464789"
    "050019897310612784530831924462149918296742734031638620763154316016702275116161937539201346"
    "108337384023359802355141440629266624923426713678065169688125024549663066864013671875e-248"
  },
  {
    2.0741563200652303e-247,
    872,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000002074156320065230251967"
    "851641691749063731953710366825298727522336814562733196900633406750711502066586090349737651"
    "414619121929946067878998595023137607518164790410230208118160529480080499815620607474676602"
    "822200093647710183989389227941857128261486792713783713285962553612344064327011914894418682"
    "422167270443852098206799122784778262467935959595002436605631518180277713808418326936404454"
    "561887278313580962716953304739999657551660090606873802274979700164372578814557790179804100"
    "883390247924759915758121835403426071643823602740995877008693348283789669168579218682702252"
    "2786412573603601400469044480079361392199643887579441070556640625",
    625,
    "2.0741563200652302519678516416917490637319537103668252987275223368145627331969006334067507"
    "115020665860903497376514146191219299460678789985950231376075181647904102302081181605294800"
    "804998156206074746766028222000936477101839893892279418571282614867927137837132859625536123"
    "440643270119148944186824221672704438520982067991227847782624679359595950024366056315181802"
    "777138084183269364044545618872783135809627169533047399996575516600906068738022749797001643"
    "725788145577901798041008833902479247599157581218354034260716438236027409958770086933482837"
    "896691685792186827022522786412573603601400469044480079361392199643887579441070556640625e-2"
    "47"
  },
  {
    4.6848861238967826e-247,
    869,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000004684886123896782552391"
    "943552353513866956887559172169395249747944696096889275440032086587785713772492575863764663"
    "121769661013949840449013072081217002421796823094420634858798663255295979116339739115048007"
    "859096080046421048235855444747600807137572005573286478427366497261165548010683612698376700"
    "103460067306048636372592132405966064272140020327580686777794712375577576860577298197494436"
    "116715794435286740615955354035910308168210866360531162725448806722244300846882708100059764"
    "808580923393744064725832585116533702112456551715355301713486847537555217153236325879679194"
    "9094141543712314728307999356360369347385130822658538818359375",
    622,
    "4.6848861238967825523919435523535138669568875591721693952497479446960968892754400320865877"
    "857137724925758637646631217696610139498404490130720812170024217968230944206348587986632552"
    "959791163397391150480078590960800464210482358554447476008071375720055732864784273664972611"
    "655480106836126983767001034600673060486363725921324059660642721400203275806867777947123755"
    "775768605772981974944361167157944352867406159553540359103081682108663605311627254488067222"
    "443008468827081000597648085809233937440647258325851165337021124565517153553017134868475375"
    "552171532363258796791949094141543712314728307999356360369347385130822658538818359375e-247"
  },
  {
    7.812879056117969e-247,
    870,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000007812879056117968926160"
    "547011986219301798506983896702553026108758793305517195870978404841713188396988128923287974"
    "065721352393075122418551066596657853413338080275298916399348535473471641911040849613085324"
    "792857849224142225226040785724236443793549159284110455362352309285581855331258281918297212"
    "200744899407805498796041724022360603035463337699744766841240868742014227868880393447341639"
    "066667237163433456130097453860186207032859848044188224565027108655355814594743568647730686"
    "615345405439108736657707455029296866147264289601403943229593597521723889485660706762928170"
    "51125806874533856884006144394305692912894301116466522216796875",
    623,
    "7.8128790561179689261605470119862193017985069838967025530261087587933055171958709784048417"
    "131883969881289232879740657213523930751224185510665966578534133380802752989163993485354734"
    "716419110408496130853247928578492241422252260407857242364437935491592841104553623523092855"
    "818553312582819182972122007448994078054987960417240223606030354633376997447668412408687420"
    "142278688803934473416390666672371634334561300974538601862070328598480441882245650271086553"
    "558145947435686477306866153454054391087366577074550292968661472642896014039432295935975217"
    "2388948566070676292817051125806874533856884006144394305692912894301116466522216796875e-247"
  },
  {
    1.16023447415079e-246,
    868,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000011602344741507901062609"
    "933663400332688034477906059336896625075268185503321277861155577877309099200703570062871759"
    "815522881113374815935124759215572993917116224795133831483509297677113816145396480072519049"
    "504190350301159988613736859277354679987749939428592328886512170606035388921350956155878372"
    "140350612102114235858507254986658169040416093975183029957643060233052384979024889678988680"
    "203420436124343358353115842501466575805640352097289296560418368734678913228931200939437087"
    "788159961581465910047408274093123274016665575087389782711415272427588336281066986388648508"
    "170505840000460355369849008155824776622466742992401123046875",
    622,
    "1.1602344741507901062609933663400332688034477906059336896625075268185503321277861155577877"
    "309099200703570062871759815522881113374815935124759215572993917116224795133831483509297677"
    "113816145396480072519049504190350301159988613736859277354679987749939428592328886512170606"
    "035388921350956155878372140350612102114235858507254986658169040416093975183029957643060233"
    "052384979024889678988680203420436124343358353115842501466575805640352097289296560418368734"
    "678913228931200939437087788159961581465910047408274093123274016665575087389782711415272427"
    "588336281066986388648508170505840000460355369849008155824776622466742992401123046875e-246"
  },
  {
    2.6285314099653654e-246,
    867,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000026285314099653654176280"
    "875743233612539556461168217321507191666512583849543277500473619693088454262930856098098303"
    "948802604312284693797178542387164814895571426387662670835577249199490374816560695683896184"
    "092123750070933504577295817555209891764246524928552438724622589910299910967160163719671228"
    "539360923776921131148693216974910909635803632416076140521430405408494331098431557109315555"
    "609723028171644630679617395772994661804039589896223778792770293045849647466844098260212735"
    "564789473032006673525509815362732759220556089373890101809605078825348777384719057604803200"
    "75691097888868409150620042424861821928061544895172119140625",
    621,
    "2.6285314099653654176280875743233612539556461168217321507191666512583849543277500473619693"
    "088454262930856098098303948802604312284693797178542387164814895571426387662670835577249199"
    "490374816560695683896184092123750070933504577295817555209891764246524928552438724622589910"
    "299910967160163719671228539360923776921131148693216974910909635803632416076140521430405408"
    "494331098431557109315555609723028171644630679617395772994661804039589896223778792770293045"
    "849647466844098260212735564789473032006673525509815362732759220556089373890101809605078825"
    "34877738471905760480320075691097888868409150620042424861821928061544895172119140625e-246"
  },
  {
    7.24619526711458e-246,
    866,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000072461952671145799308682"
    "942273613242790942104994852816207973587954637784400200604552323859641436019959259805541627"
    "524381368324353088153031282671758700037764963692127107204873182521274349285736712540229538"
    "974855777690099886074995168948554199676178910782151410536448387623324085520914592982182357"
    "388364013606173365687123972838007379314935535311449332150300365272142072421910507865927246"
    "673247351952415023600273667576940226363301653096492037705698968488662811198909785131615586"
    "198786918782826768903042966247621547375693560086863739901394552436861153882487228074815937"
    "8906498225093109517847000944357205298729240894317626953125",
    620,
    "7.2461952671145799308682942273613242790942104994852816207973587954637784400200604552323859"
    "641436019959259805541627524381368324353088153031282671758700037764963692127107204873182521"
    "274349285736712540229538974855777690099886074995168948554199676178910782151410536448387623"
    "324085520914592982182357388364013606173365687123972838007379314935535311449332150300365272"
    "142072421910507865927246673247351952415023600273667576940226363301653096492037705698968488"
    "662811198909785131615586198786918782826768903042966247621547375693560086863739901394552436"
    "8611538824872280748159378906498225093109517847000944357205298729240894317626953125e-246"
  },
  {
    1.513557579372713e-245,
    864,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000151355757937271292434155"
    "488817066399727646021384774898326133016611255256532612911904572526460038796615464064557235"
    "535184255428202266594734780713103215069500727504595579802054769661280199593841908822623154"
    "438852886516721904821163361435303795960577174561817499899974031496977273359331148325279487"
    "993459134939192402413911960848312506968068474497203654267210596943820018376325451286018463"
    "106301504920946287872817633761992078128830310281174844229789408905630908727536976405794432"
    "973052507474674698762818002597976646771449616515170649132357047121718410320718543846680133"
    "79767883698598547603797470628705923445522785186767578125",
    619,
    "1.5135575793727129243415548881706639972764602138477489832613301661125525653261291190457252"
    "646003879661546406455723553518425542820226659473478071310321506950072750459557980205476966"
    "128019959384190882262315443885288651672190482116336143530379596057717456181749989997403149"
    "697727335933114832527948799345913493919240241391196084831250696806847449720365426721059694"
    "382001837632545128601846310630150492094628787281763376199207812883031028117484422978940890"
    "563090872753697640579443297305250747467469876281800259797664677144961651517064913235704712"
    "171841032071854384668013379767883698598547603797470628705923445522785186767578125e-245"
  },
  {
    2.2022598987138438e-245,
    864,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000220225989871384377549550"
    "881409508224140794258910315178257547061031803808940293348910079770489697857931874563628273"
    "401003849229027947643460709548320874194553229940197985486063866961467079881830131956202351"
    "541355456753780910128309683401338833970019687723299566551735852677241638071401747016365464"
    "120366734729673913484098454622686279912973913718209179981656395546975501841471468433090568"
    "357915781915959205034872994262578079066845040859907941076591375398817610400020832219000691"
    "738654346851177693702055925078500930211965695729283618046819289817657206713828774896261972"
    "46411346537353463659325569778957287780940532684326171875",
    619,
    "2.2022598987138437754955088140950822414079425891031517825754706103180380894029334891007977"
    "048969785793187456362827340100384922902794764346070954832087419455322994019798548606386696"
    "146707988183013195620235154135545675378091012830968340133883397001968772329956655173585267"
    "724163807140174701636546412036673472967391348409845462268627991297391371820917998165639554"
    "697550184147146843309056835791578191595920503487299426257807906684504085990794107659137539"
    "881761040002083221900069173865434685117769370205592507850093021196569572928361804681928981"
    "765720671382877489626197246411346537353463659325569778957287780940532684326171875e-245"
  },
  {
    4.1410759136055036e-245,
    864,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000414107591360550356975543"
    "556125232733498057795754435160244844333255885048261735011062749061211793783227645942484379"
    "691318093841368847098623863226362751057949455437358357281291709007819852095728818477728142"
    "143877797061861882849928412094624260406390654427012375485147264953579827210698221103730583"
    "044395153862193840527289541637022653688133092670011734741150030964716207520130641260704021"
    "824268518292725366826052714592611208714273261395627080752743556294388041183383938489151395"
    "241871318175793857142069803602514255759195099358106829389259414039612286925866469910881124"
    "98341686942122073789018799061523168347775936126708984375",
    619,
    "4.1410759136055035697554355612523273349805779575443516024484433325588504826173501106274906"
    "121179378322764594248437969131809384136884709862386322636275105794945543735835728129170900"
    "781985209572881847772814214387779706186188284992841209462426040639065442701237548514726495"
    "357982721069822110373058304439515386219384052728954163702265368813309267001173474115003096"
    "471620752013064126070402182426851829272536682605271459261120871427326139562708075274355629"
    "438804118338393848915139524187131817579385714206980360251425575919509935810682938925941403"
    "961228692586646991088112498341686942122073789018799061523168347775936126708984375e-245"
  },
  {
    1.0252325908734784e-244,
    863,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000001025232590873478418789840"
    "784631216966293396445161720613561905787036150617718363325006003067828838099295381539103507"
    "207744202525643700611524672299844933001035613433505064997893247944671857324939620948627992"
    "142713220434893279799034506854982649170149103444477505174541550420185719057992592872659016"
    "813377466213470076718167158792818482892885532012914901509468511115943526881436543514623318"
    "028051207569954102646295942779533234962168490646675608744310157317163502352340937966413998"
    "580071972072654694838719624118122768904909732503571094374091831312455529822849937930466625"
    "3435895481214111535361865890081389807164669036865234375",
    619,
    "1.0252325908734784187898407846312169662933964451617206135619057870361506177183633250060030"
    "678288380992953815391035072077442025256437006115246722998449330010356134335050649978932479"
    "446718573249396209486279921427132204348932797990345068549826491701491034444775051745415504"
    "201857190579925928726590168133774662134700767181671587928184828928855320129149015094685111"
    "159435268814365435146233180280512075699541026462959427795332349621684906466756087443101573"
    "171635023523409379664139985800719720726546948387196241181227689049097325035710943740918313"
    "124555298228499379304666253435895481214111535361865890081389807164669036865234375e-244"
  },
  {
    2.0526028348315092e-244,
    861,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000002052602834831509226932649"
    "149141667161471727484417155219977134028714962975171049035381782547160052232774227698485667"
    "471867245925706032627831596226796183758396050199760459851947025984940932987904537296293869"
    "846480690036876519192087476613750641520994696959915507812672382558594635321159278065876914"
    "295340826056875441592600141361458893737882740856230002509639149203945405128352892860259015"
    "339215355673194397338372708285535055693381899508696665501113683684770786168217064751366954"
    "802701707958201691744906250314898776237606183664545829671163361569714429189618563946869924"
    "46379464696305794380037923474446870386600494384765625",
    617,
    "2.0526028348315092269326491491416671614717274844171552199771340287149629751710490353817825"
    "471600522327742276984856674718672459257060326278315962267961837583960501997604598519470259"
    "849409329879045372962938698464806900368765191920874766137506415209946969599155078126723825"
    "585946353211592780658769142953408260568754415926001413614588937378827408562300025096391492"
    "039454051283528928602590153392153556731943973383727082855350556933818995086966655011136836"
    "847707861682170647513669548027017079582016917449062503148987762376061836645458296711633615"
    "6971442918961856394686992446379464696305794380037923474446870386600494384765625e-244"
  },
  {
    3.9232668413253934e-244,
    861,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000003923266841325393410957631"
    "552337541082482763500601453247856637679543576166071328111203385529055820908854409551186443"
    "084919373917067982109478797832735088299408640774255550297436654337964480265428817925237078"
    "455293928888040646795226366213825411580640005811347292313508735062094848631128449978823754"
    "429190527768902694777788490021724731119659671404644229656835657996647977868270739479902408"
    "549191018791857386839430814658014354114909166773527884807665163068492284062280504681162738"
    "206750305934487612711077981280396055004386628939434565467838907125584137242034985328974841"
    "91055958980230700117175501873134635388851165771484375",
    617,
    "3.9232668413253934109576315523375410824827635006014532478566376795435761660713281112033855"
    "290558209088544095511864430849193739170679821094787978327350882994086407742555502974366543"
    "379644802654288179252370784552939288880406467952263662138254115806400058113472923135087350"
    "620948486311284499788237544291905277689026947777884900217247311196596714046442296568356579"
    "966479778682707394799024085491910187918573868394308146580143541149091667735278848076651630"
    "684922840622805046811627382067503059344876127110779812803960550043866289394345654678389071"
    "2558413724203498532897484191055958980230700117175501873134635388851165771484375e-244"
  },
  {
    7.579615073559597e-244,
    859,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000007579615073559596851823898"
    "310748383907000314295598697431158191668022579145348683370076416288747863020785670478229418"
    "594802605843659633218761991003725554393124746818759551343919858321903795678002384262202672"
    "348423611239623046545996988440924432526210697817982616974975257322206843303081668533190742"
    "330428145605682185623573267026212237936980552458729472632565335040983151789434800446792823"
    "265074553050656043056416296959360170882045939731413452970282373048701017908669290277725775"
    "612223691751332374494642214309992083112160331397139425617718942914363891383632845680383464"
    "106053752022684744105163190397433936595916748046875",
    615,
    "7.5796150735595968518238983107483839070003142955986974311581916680225791453486833700764162"
    "887478630207856704782294185948026058436596332187619910037255543931247468187595513439198583"
    "219037956780023842622026723484236112396230465459969884409244325262106978179826169749752573"
    "222068433030816685331907423304281456056821856235732670262122379369805524587294726325653350"
    "409831517894348004467928232650745530506560430564162969593601708820459397314134529702823730"
    "487010179086692902777257756122236917513323744946422143099920831121603313971394256177189429"
    "14363891383632845680383464106053752022684744105163190397433936595916748046875e-244"
  },
  {
    2.042372149921196e-243,
    859,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000020423721499211961109568525"
    "534671443647740451225589577776776615350824924185902447511045534069936905640957606537200670"
    "668241598816941076161810442082354034465365585049545316189811674090908528332216379781669042"
    "671392805883865914697756117935589096077795688376869135178989276965123735720160481020902714"
    "651968376310215669498093458778713417135141374746058126296881347472422680267110011339066236"
    "275056110971927834750821976896185521894260300273463567943024440052209000198482863479912953"
    "156904928607571710364397252501191609041151171163802675789249238420106837535506252866076169"
    "083468642463652376051186365657486021518707275390625",
    616,
    "2.0423721499211961109568525534671443647740451225589577776776615350824924185902447511045534"
    "069936905640957606537200670668241598816941076161810442082354034465365585049545316189811674"
    "090908528332216379781669042671392805883865914697756117935589096077795688376869135178989276"
    "965123735720160481020902714651968376310215669498093458778713417135141374746058126296881347"
    "472422680267110011339066236275056110971927834750821976896185521894260300273463567943024440"
    "052209000198482863479912953156904928607571710364397252501191609041151171163802675789249238"
    "420106837535506252866076169083468642463652376051186365657486021518707275390625e-243"
  },
  {
    2.9353293314833334e-243,
    858,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000029353293314833333854985102"
    "733201521796250036157410578174458181974941681427901748305530776598182206753889861863637101"
    "626780047066935764018072155601016400421464552029813635340871920828654050237207770779587686"
    "438912345309870394701848727504274788835609186466843505051307228378630149997278755552156853"
    "040864016042768011731203139790194066844214273369426057653128075479513670720217838142259228"
    "046519573053627935149750677307039941356482814713297285599712220196545670395402241262550735"
    "056162451065557414323405285188470820020307461454661961993618302085325455509316070608901872"
    "86750154134484791512704759952612221240997314453125",
    615,
    "2.9353293314833333854985102733201521796250036157410578174458181974941681427901748305530776"
    "598182206753889861863637101626780047066935764018072155601016400421464552029813635340871920"
    "828654050237207770779587686438912345309870394701848727504274788835609186466843505051307228"
    "378630149997278755552156853040864016042768011731203139790194066844214273369426057653128075"
    "479513670720217838142259228046519573053627935149750677307039941356482814713297285599712220"
    "196545670395402241262550735056162451065557414323405285188470820020307461454661961993618302"
    "08532545550931607060890187286750154134484791512704759952612221240997314453125e-243"
  },
  {
    6.372363858570045e-243,
    855,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000063723638585700451642795559"
    "570510679148094993613910324952290173092838588831267228266631303318254630028045739040718717"
    "806151057630003575437223382249922228296120345672094131497663649713882505386851200191831956"
    "237611785031221668198294255663107081123276328396877658665421755136440776486320603951713109"
    "073638238366471068582232777412680176689313447386866112250866307040221174507703114483598510"
    "540103447645068390784004440801839078883719477888542923897825206398740323615820485633543592"
    "686389160458875278305783242808662350782677502424919580739557162653494711579018111213675370"
    "37413035505561964555454323999583721160888671875",
    612,
    "6.3723638585700451642795559570510679148094993613910324952290173092838588831267228266631303"
    "318254630028045739040718717806151057630003575437223382249922228296120345672094131497663649"
    "713882505386851200191831956237611785031221668198294255663107081123276328396877658665421755"
    "136440776486320603951713109073638238366471068582232777412680176689313447386866112250866307"
    "040221174507703114483598510540103447645068390784004440801839078883719477888542923897825206"
    "398740323615820485633543592686389160458875278305783242808662350782677502424919580739557162"
    "65349471157901811121367537037413035505561964555454323999583721160888671875e-243"
  },
  {
    1.575234414235322e-242,
    854,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000157523441423532194506941897"
    "024519912969104334084307890190041251318636292965114311542306595038907176200315901038316721"
    "233788818826571623317203349289987417121376686501945841032567911901250552698960793265396245"
    "960534737316705565182045191204654661888815341264524249008725135008757664918769938301063898"
    "226941808254088456283277502440824661639800744384833042204150079229430393907506981793864701"
    "769660337112702767082883519285744806346620427578497508169144643312340209721649688496292864"
    "219377630628268116030388752066570843025641533813750666150515363044098947446032461106652930"
    "6552773392890998138682334683835506439208984375",
    612,
    "1.5752344142353219450694189702451991296910433408430789019004125131863629296511431154230659"
    "503890717620031590103831672123378881882657162331720334928998741712137668650194584103256791"
    "190125055269896079326539624596053473731670556518204519120465466188881534126452424900872513"
    "500875766491876993830106389822694180825408845628327750244082466163980074438483304220415007"
    "922943039390750698179386470176966033711270276708288351928574480634662042757849750816914464"
    "331234020972164968849629286421937763062826811603038875206657084302564153381375066615051536"
    "30440989474460324611066529306552773392890998138682334683835506439208984375e-242"
  },
  {
    3.3434436478293493e-242,
    855,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000334344364782934932805935712"
    "711780083796320795003859755482839391634978522148790448445176707594970914929266046301395881"
    "911376324184025424812370739165394698174516004668202850196235934904450092005834916657352885"
    "854495679712284975688945885665603008963754001731201807545843037503035671240236961951717629"
    "396479659702810769273368776557703225763240573620548788476008314685255562870023588785334442"
    "000080417182360569694778279347726566779815145097506338920779765572188303868670858298304546"
    "602128519970722727663497018972024558384033232006396023882173613333680306588354432532810297"
    "21128182752298840796356671489775180816650390625",
    613,
    "3.3434436478293493280593571271178008379632079500385975548283939163497852214879044844517670"
    "759497091492926604630139588191137632418402542481237073916539469817451600466820285019623593"
    "490445009200583491665735288585449567971228497568894588566560300896375400173120180754584303"
    "750303567124023696195171762939647965970281076927336877655770322576324057362054878847600831"
    "468525556287002358878533444200008041718236056969477827934772656677981514509750633892077976"
    "557218830386867085829830454660212851997072272766349701897202455838403323200639602388217361"
    "333368030658835443253281029721128182752298840796356671489775180816650390625e-242"
  },
  {
    5.199566493700135e-242,
    853,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000519956649370013508636642582"
    "095069161256120458874210354955399682199215520014904045126665970571265114347528813007435581"
    "863058379511824026420279450528771594711743194984119825629737962368975331651816862223263764"
    "907329895984505915243616827483678788641024554143279148411712282323650863417608588959251340"
    "984986920056515476688722390502792545337079153649654816344849430258535645372560190074871913"
    "197870664373018730201713610922019398386184775914931861779759491034394263205676462386582564"
    "671334894431929147821752438599108003402914657920461899226466173041809225803734159361765368"
    "727488489941634952629101462662220001220703125",
    611,
    "5.1995664937001350863664258209506916125612045887421035495539968219921552001490404512666597"
    "057126511434752881300743558186305837951182402642027945052877159471174319498411982562973796"
    "236897533165181686222326376490732989598450591524361682748367878864102455414327914841171228"
    "232365086341760858895925134098498692005651547668872239050279254533707915364965481634484943"
    "025853564537256019007487191319787066437301873020171361092201939838618477591493186177975949"
    "103439426320567646238658256467133489443192914782175243859910800340291465792046189922646617"
    "3041809225803734159361765368727488489941634952629101462662220001220703125e-242"
  },
  {
    1.151222665835575e-241,
    853,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000001151222665835574948384798160"
    "790374918696845627147066842290551496804868286255058796458842284516181576719673471613817362"
    "866823301977849457479564381865329613399427500201855688873879142642733519107599455632142550"
    "666846367743093899226009412278214746750427881209859743419090772304032977787125925338802665"
    "415242857226084498164308310215184350361304036906423619218512946699949724125654106786260102"
    "557363229481670642614150662832046947746967360569276734749642475323944536173037037021708328"
    "999399961193013811652519172663544961165454577744539450895523040343609292008370421060948365"
    "454650044181761359141091816127300262451171875",
    612,
    "1.1512226658355749483847981607903749186968456271470668422905514968048682862550587964588422"
    "845161815767196734716138173628668233019778494574795643818653296133994275002018556888738791"
    "426427335191075994556321425506668463677430938992260094122782147467504278812098597434190907"
    "723040329777871259253388026654152428572260844981643083102151843503613040369064236192185129"
    "466999497241256541067862601025573632294816706426141506628320469477469673605692767347496424"
    "753239445361730370370217083289993999611930138116525191726635449611654545777445394508955230"
    "40343609292008370421060948365454650044181761359141091816127300262451171875e-241"
  },
  {
    1.5543636774561637e-241,
    850,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000001554363677456163697498436121"
    "811101181578749611598812525176151274579920323911457023153031158762892959294137625850420107"
    "270440168478704645026759363769681184230190786395976567038018780134837227102915702621913476"
    "848336805205669043926853377700998806089359780103711297258044030751523125011328665317845898"
    "485403235762350467927613415678794454917406859654077089489887692454210664663765567103840566"
    "472303455745447620744679132024829071450081674006838972988367188884108104737049840838885337"
    "987296921979337482709916364728062993584182504817741094771758489864324415337151128538238880"
    "998079149975410473416559398174285888671875",
    609,
    "1.5543636774561636974984361218111011815787496115988125251761512745799203239114570231530311"
    "587628929592941376258504201072704401684787046450267593637696811842301907863959765670380187"
    "801348372271029157026219134768483368052056690439268533777009988060893597801037112972580440"
    "307515231250113286653178458984854032357623504679276134156787944549174068596540770894898876"
    "924542106646637655671038405664723034557454476207446791320248290714500816740068389729883671"
    "888841081047370498408388853379872969219793374827099163647280629935841825048177410947717584"
    "89864324415337151128538238880998079149975410473416559398174285888671875e-241"
  },
  {
    4.079493099457495e-241,
    850,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000004079493099457494944252531607"
    "549343636245547309268443899793498613543765446694133273976895637787747339760713440649581560"
    "845325794100875391063273453728186643522471972773614713700301177847896189832079249276514506"
    "877114770356574608290123358451973486669321692492332398543131893173963499548478234080992976"
    "620729264092514266793841953819578726079145271155664076730226953212874304860215791047963356"
    "608824459478793671211948274702724136559074011196819859825079215570247071425975730487942082"
    "183311410950781004256077318512834026216912470392928342115674805610333210644028408620173696"
    "379207495255059257033281028270721435546875",
    609,
    "4.0794930994574949442525316075493436362455473092684438997934986135437654466941332739768956"
    "377877473397607134406495815608453257941008753910632734537281866435224719727736147137003011"
    "778478961898320792492765145068771147703565746082901233584519734866693216924923323985431318"
    "931739634995484782340809929766207292640925142667938419538195787260791452711556640767302269"
    "532128743048602157910479633566088244594787936712119482747027241365590740111968198598250792"
    "155702470714259757304879420821833114109507810042560773185128340262169124703929283421156748"
    "05610333210644028408620173696379207495255059257033281028270721435546875e-241"
  },
  {
    8.503783454502844e-241,
    850,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000008503783454502843972651117397"
    "038706537649545180122135874554146602211105527944781436847646569250153389780805284146206065"
    "382019591751710733514433667516784023869511295085720936858126749712265756822014702395749725"
    "069109128943544677360375146178941605556087049517009816921069940296076289616976017445345589"
    "202204019215772204351631100875755262589943645429889209734631936720431395561041243551295323"
    "767386709839198875183729692636944707657278930317739103339487124899124826072841078946429617"
    "996885861361489855103899126561059706739026389257434729720838400566841392990613565908529915"
    "098046370047768505173735320568084716796875",
    609,
    "8.5037834545028439726511173970387065376495451801221358745541466022111055279447814368476465"
    "692501533897808052841462060653820195917517107335144336675167840238695112950857209368581267"
    "497122657568220147023957497250691091289435446773603751461789416055560870495170098169210699"
    "402960762896169760174453455892022040192157722043516311008757552625899436454298892097346319"
    "367204313955610412435512953237673867098391988751837296926369447076572789303177391033394871"
    "248991248260728410789464296179968858613614898551038991265610597067390263892574347297208384"
    "00566841392990613565908529915098046370047768505173735320568084716796875e-241"
  },
  {
    2.0072820595479573e-240,
    845,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000020072820595479573216169126231"
    "394408838551922619989874786871658199647732697034762028792666015861804714982292219159115495"
    "529137800501558783955553148061206323583534701281077232750185616658820946457957244860450889"
    "990617721872236848551624424241757114637727052791809821264163180310743723028655601288256323"
    "233178896944978433607754085808572590924978633945247230403112787302032626015620605415136044"
    "530348825415809501368263327044824152414594545569962369609544152106912922650934411810223780"
    "925102998679548866177843754000205462166075319744190834334340034460126199155776480361704910"
    "9707437537508667446672916412353515625",
    605,
    "2.0072820595479573216169126231394408838551922619989874786871658199647732697034762028792666"
    "015861804714982292219159115495529137800501558783955553148061206323583534701281077232750185"
    "616658820946457957244860450889990617721872236848551624424241757114637727052791809821264163"
    "180310743723028655601288256323233178896944978433607754085808572590924978633945247230403112"
    "787302032626015620605415136044530348825415809501368263327044824152414594545569962369609544"
    "152106912922650934411810223780925102998679548866177843754000205462166075319744190834334340"
    "0344601261991557764803617049109707437537508667446672916412353515625e-240"
  },
  {
    3.536438208901198e-240,
    845,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000035364382089011979980897109687"
    "165254886696841520981254446556125956556978294321082019870841710940758511838532489417943905"
    "736435746115951713790347474465462971745337544820877439873033371186200763396273825803241774"
    "138006853211153930782641403622243258240706664373441913699394432174922219936686057162089906"
    "945262639497251751272531533913011070118815751448623244063021481460622877756524201582813510"
    "487724558551743402584791531847833324394767751212559672612147109528806384219695184783904856"
    "874579756560539783035465943746368278612717453039965305218598014221274914886311047134467150"
    "3768353204577579163014888763427734375",
    605,
    "3.5364382089011979980897109687165254886696841520981254446556125956556978294321082019870841"
    "710940758511838532489417943905736435746115951713790347474465462971745337544820877439873033"
    "371186200763396273825803241774138006853211153930782641403622243258240706664373441913699394"
    "432174922219936686057162089906945262639497251751272531533913011070118815751448623244063021"
    "481460622877756524201582813510487724558551743402584791531847833324394767751212559672612147"
    "109528806384219695184783904856874579756560539783035465943746368278612717453039965305218598"
    "0142212749148863110471344671503768353204577579163014888763427734375e-240"
  },
  {
    8.722809014595794e-240,
    845,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000087228090145957935347248318547"
    "742302587087102313285501442514035526226306674555848729098848343284853736028606487383125492"
    "860996732221717787417179695557893446531314740287886503942082756517687856792743547708418920"
    "059578068476465991094429066781466951295814066435042468232392713814855820467535314718622583"
    "843522225733754988426169068002044002105223868118456214469943463514242867324395558742911540"
    "780602189115030142752959592996857877227304052880736262738027290210660571731655711256049169"
    "304129861923207797896858618680332588281978330716855688129759881700104170470699446423342576"
    "7840657272245152853429317474365234375",
    605,
    "8.7228090145957935347248318547742302587087102313285501442514035526226306674555848729098848"
    "343284853736028606487383125492860996732221717787417179695557893446531314740287886503942082"
    "756517687856792743547708418920059578068476465991094429066781466951295814066435042468232392"
    "713814855820467535314718622583843522225733754988426169068002044002105223868118456214469943"
    "463514242867324395558742911540780602189115030142752959592996857877227304052880736262738027"
    "290210660571731655711256049169304129861923207797896858618680332588281978330716855688129759"
    "8817001041704706994464233425767840657272245152853429317474365234375e-240"
  },
  {
    1.3749379101340066e-239,
    846,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000137493791013400660838573373481"
    "760128793029989420406438919594134474628099232137107923859209133670605866208605258647049621"
    "760492047179451844836572119347356991890024090747092362463756927276284737937087463531842640"
    "685811082271690106032106931918033063009535007119845401401760338121534727954319107576984927"
    "194147223289450923041797310685308345211110611634762026407877267878991523018875324844981756"
    "700122806053953717458149970358111047972034718728424120220308910285188823533296077514778262"
    "102253723563125528563041143324629454753231680542831915726299330596087249617289222650617572"
    "24113847541957511566579341888427734375",
    607,
    "1.3749379101340066083857337348176012879302998942040643891959413447462809923213710792385920"
    "913367060586620860525864704962176049204717945184483657211934735699189002409074709236246375"
    "692727628473793708746353184264068581108227169010603210693191803306300953500711984540140176"
    "033812153472795431910757698492719414722328945092304179731068530834521111061163476202640787"
    "726787899152301887532484498175670012280605395371745814997035811104797203471872842412022030"
    "891028518882353329607751477826210225372356312552856304114332462945475323168054283191572629"
    "933059608724961728922265061757224113847541957511566579341888427734375e-239"
  },
  {
    3.05119881849728e-239,
    845,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000305119881849727993830712119246"
    "344871754601809353260541161021859815864083210394212669807093524175183158874045050170880613"
    "243376562389933452426086185695180699312804245142060569472810956533457237033496816457196253"
    "387812514946594440827064963710383559844214050994602894288883408781805491312795928309791112"
    "301557669219108560685682834968523172307496695822817771595479770237903897871933313393475369"
    "503781814634341296303969712079941349950377993759212633121226292844763708145719324648259205"
    "908825462198868419797842376770017948515553052857064769335480791578431489872109360326785792"
    "9939310679401387460529804229736328125",
    606,
    "3.0511988184972799383071211924634487175460180935326054116102185981586408321039421266980709"
    "352417518315887404505017088061324337656238993345242608618569518069931280424514206056947281"
    "095653345723703349681645719625338781251494659444082706496371038355984421405099460289428888"
    "340878180549131279592830979111230155766921910856068568283496852317230749669582281777159547"
    "977023790389787193331339347536950378181463434129630396971207994134995037799375921263312122"
    "629284476370814571932464825920590882546219886841979784237677001794851555305285706476933548"
    "07915784314898721093603267857929939310679401387460529804229736328125e-239"
  },
  {
    6.2081425746751336e-239,
    844,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000620814257467513356101466802954"
    "351018594616556819890365863277752172387490613035017353323713829002805755157194020730148118"
    "435599036452088715313932099113134973129083783128920389682715075150575144731203122222001783"
    "294942440690555379559136961157326843128151822805761824940197594589024349297466336899455289"
    "602378238178255391653370447191285677213947446712621494044714867485373926983317672765804299"
    "630860902353138272962413113555038517755830887058852014800933522702452536975068231984788501"
    "210512796491195164930952739801311157528383120640408842401705946246458190149534645571431645"
    "816318223296548239886760711669921875",
    605,
    "6.2081425746751335610146680295435101859461655681989036586327775217238749061303501735332371"
    "382900280575515719402073014811843559903645208871531393209911313497312908378312892038968271"
    "507515057514473120312222200178329494244069055537955913696115732684312815182280576182494019"
    "759458902434929746633689945528960237823817825539165337044719128567721394744671262149404471"
    "486748537392698331767276580429963086090235313827296241311355503851775583088705885201480093"
    "352270245253697506823198478850121051279649119516493095273980131115752838312064040884240170"
    "5946246458190149534645571431645816318223296548239886760711669921875e-239"
  },
  {
    1.5212878641555803e-238,
    841,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000001521287864155580272495084608651"
    "614503230032699869119090029230173561903140575526338086781944551107256257294332875937033885"
    "350270044302018784621809744448807229376800166043171230470979247077718611566183061663289888"
    "074173424610772542026483506595446813945016920935740824692420808377728446284988055043397756"
    "717427418778466960476498441651068958217130044681113095462381198186573982658960792484830731"
    "627960784313379336537148325565729100673397232328082703744278459276106826127533120342240662"
    "852596857525177657403125501719700324721350369868638181213742553728087480359007130879811509"
    "061028118594549596309661865234375",
    603,
    "1.5212878641555802724950846086516145032300326998691190900292301735619031405755263380867819"
    "445511072562572943328759370338853502700443020187846218097444488072293768001660431712304709"
    "792470777186115661830616632898880741734246107725420264835065954468139450169209357408246924"
    "208083777284462849880550433977567174274187784669604764984416510689582171300446811130954623"
    "811981865739826589607924848307316279607843133793365371483255657291006733972323280827037442"
    "784592761068261275331203422406628525968575251776574031255017197003247213503698686381812137"
    "42553728087480359007130879811509061028118594549596309661865234375e-238"
  },
  {
    2.521616635184641e-238,
    842,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000002521616635184641090459880412650"
    "391997596020634142155668067312213943083961889306877895352865071445289971934552461081251587"
    "394827416804758179858983102808612721778326780698686015680956862173798760468031620107081126"
    "308023604716065459630277834590133356599359591175082540998292572569927744367695512232618054"
    "512372669109059329227873428969756808668222882011242817553250634028377706332201193105591187"
    "575198187287184221450436040247100286697114970765495289023647586135740814347953444228162789"
    "526700205662670272193162254169393650069587137994906638875265550194637753235738224667078499"
    "7633063539979048073291778564453125",
    604,
    "2.5216166351846410904598804126503919975960206341421556680673122139430839618893068778953528"
    "650714452899719345524610812515873948274168047581798589831028086127217783267806986860156809"
    "568621737987604680316201070811263080236047160654596302778345901333565993595911750825409982"
    "925725699277443676955122326180545123726691090593292278734289697568086682228820112428175532"
    "506340283777063322011931055911875751981872871842214504360402471002866971149707654952890236"
    "475861357408143479534442281627895267002056626702721931622541693936500695871379949066388752"
    "655501946377532357382246670784997633063539979048073291778564453125e-238"
  },
  {
    4.498289788351931e-238,
    841,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000004498289788351931234970025785207"
    "428524283933840589162361607524486702394246801779245582632820621839309957978885381833266783"
    "882815436988386959011380163998921014626575641453243971748047954427020762647175726857865966"
    "998309206594117460390492150507731013549139591322588458630372846181251298644962778269180438"
    "519777882585752366206164783368230851366025853127421809094303484111099302047709084993491900"
    "227885557798282272888791052050805713091213003012308739061423106369221258417858013955079753"
    "813197763552984657907220444177753024394698564843952193959423860928797599218080852621903797"
    "938784919097088277339935302734375",
    603,
    "4.4982897883519312349700257852074285242839338405891623616075244867023942468017792455826328"
    "206218393099579788853818332667838828154369883869590113801639989210146265756414532439717480"
    "479544270207626471757268578659669983092065941174603904921505077310135491395913225884586303"
    "728461812512986449627782691804385197778825857523662061647833682308513660258531274218090943"
    "034841110993020477090849934919002278855577982822728887910520508057130912130030123087390614"
    "231063692212584178580139550797538131977635529846579072204441777530243946985648439521939594"
    "23860928797599218080852621903797938784919097088277339935302734375e-238"
  },
  {
    7.28128014416453e-238,
    834,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000007281280144164530396641873252286"
    "060523444678158170116295187470244542162501469787325772110024713328093333754746328221475697"
    "303727345561926510751935093640686385450454336346624796049720140323046251187122647847538623"
    "304240701146998515213155895112699926325225487803168318982122886793550787654412114711135260"
    "065548158754163814626779217530007597288032053181584232489033439281492659500912807121143377"
    "612195059168237794781205290847875237820453998789574995457953460635879854777311114798495215"
    "561348833515894505972328373361159647799754037753597939411348758369777102060146911632898536"
    "48166172206401824951171875",
    596,
    "7.2812801441645303966418732522860605234446781581701162951874702445421625014697873257721100"
    "247133280933337547463282214756973037273455619265107519350936406863854504543363466247960497"
    "201403230462511871226478475386233042407011469985152131558951126999263252254878031683189821"
    "228867935507876544121147111352600655481587541638146267792175300075972880320531815842324890"
    "334392814926595009128071211433776121950591682377947812052908478752378204539987895749954579"
    "534606358798547773111147984952155613488335158945059723283733611596477997540377535979394113"
    "4875836977710206014691163289853648166172206401824951171875e-238"
  },
  {
    2.0208837171448896e-237,
    839,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000020208837171448895837318180903259"
    "214725922207568227959012108677599688049530743023941864936723736235817948008261140342779894"
    "654766198813974216962059622039149375372207140382159277201024253418658949728696651807649436"
    "206784207363219077106619115568943818501083465062656303197679139193990301914319651109378310"
    "305112195045328740586466146512917773158338656319118663906595686588537648936843031647093427"
    "657363683641219540298646042122056831742657703852094032730167092498886612777005716954892482"
    "974151250256729385145281839942068187782374817984939873744033326205027976993735313454569890"
    "7231781049631536006927490234375",
    602,
    "2.0208837171448895837318180903259214725922207568227959012108677599688049530743023941864936"
    "723736235817948008261140342779894654766198813974216962059622039149375372207140382159277201"
    "024253418658949728696651807649436206784207363219077106619115568943818501083465062656303197"
    "679139193990301914319651109378310305112195045328740586466146512917773158338656319118663906"
    "595686588537648936843031647093427657363683641219540298646042122056831742657703852094032730"
    "167092498886612777005716954892482974151250256729385145281839942068187782374817984939873744"
    "0333262050279769937353134545698907231781049631536006927490234375e-237"
  },
  {
    4.2234650668214423e-237,
    837,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000042234650668214423350279081718143"
    "018052918828845233278512008791451987305685594591145976330970127359480860248854891726431558"
    "443351245555468482610003496561352434651925296169879290047618996071855769729895077818141001"
    "396402280260955848718102896835082397944689967869877193663152427771849982557993069202722817"
    "172139278832385788203065386664239152516789871769170789609239941892434305575451880664013613"
    "320226313501911783489649376683084871481472461594614907247440858246857575468466973071241319"
    "175342027247791966720048497263178860953612995223921886377337938565547890301275432700656153"
    "88375590555369853973388671875",
    600,
    "4.2234650668214423350279081718143018052918828845233278512008791451987305685594591145976330"
    "970127359480860248854891726431558443351245555468482610003496561352434651925296169879290047"
    "618996071855769729895077818141001396402280260955848718102896835082397944689967869877193663"
    "152427771849982557993069202722817172139278832385788203065386664239152516789871769170789609"
    "239941892434305575451880664013613320226313501911783489649376683084871481472461594614907247"
    "440858246857575468466973071241319175342027247791966720048497263178860953612995223921886377"
    "33793856554789030127543270065615388375590555369853973388671875e-237"
  },
  {
    8.084125743595665e-237,
    836,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000080841257435956645678547405463747"
    "941035254532491290665291476902755342035620999121915876359426286744536442548879145579602755"
    "273295208707182241809884602210363950548931232190091262986853262347513509500936326592103187"
    "518710790216114113327410337752704088937019771852461546408738525360130912696153171899353016"
    "883165954976283632052474049117604877290387290489379052271964810487511690636116959829601543"
    "260594988729338797802211604439634988154451173709514406892911690109024301612727090904032911"
    "743576303129133173491689700883272188659710649614449984150537949653226767871266184029011014"
    "8998792283236980438232421875",
    599,
    "8.0841257435956645678547405463747941035254532491290665291476902755342035620999121915876359"
    "426286744536442548879145579602755273295208707182241809884602210363950548931232190091262986"
    "853262347513509500936326592103187518710790216114113327410337752704088937019771852461546408"
    "738525360130912696153171899353016883165954976283632052474049117604877290387290489379052271"
    "964810487511690636116959829601543260594988729338797802211604439634988154451173709514406892"
    "911690109024301612727090904032911743576303129133173491689700883272188659710649614449984150"
    "5379496532267678712661840290110148998792283236980438232421875e-237"
  },
  {
    1.6733590187416607e-236,
    835,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000167335901874166069420287882958227"
    "341470430879462336079686510169450060430637434099569829226051651502624936952359551337435141"
    "043924366256262954714703084104196403532472922571050719624290744697858239688419404203520754"
    "755791685071755349284205778174468869373760892895245843164670931855028839648439115777627948"
    "400437021229776220857412792707372131989621735723259580134638531177608776650138680397146190"
    "891510483936959754580000980633446858750098421368476749851793586866701372231665192694279106"
    "769119933450898417201549643514656077765935283060136103656512062492185832101622422474207496"
    "634335257112979888916015625",
    599,
    "1.6733590187416606942028788295822734147043087946233607968651016945006043063743409956982922"
    "605165150262493695235955133743514104392436625626295471470308410419640353247292257105071962"
    "429074469785823968841940420352075475579168507175534928420577817446886937376089289524584316"
    "467093185502883964843911577762794840043702122977622085741279270737213198962173572325958013"
    "463853117760877665013868039714619089151048393695975458000098063344685875009842136847674985"
    "179358686670137223166519269427910676911993345089841720154964351465607776593528306013610365"
    "6512062492185832101622422474207496634335257112979888916015625e-236"
  },
  {
    2.744824473945537e-236,
    835,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000274482447394553695998529494925529"
    "235079483338182081970084681581290838101596954832563602691754150154664120314225695400387005"
    "807802038517353586335988207343902959393348397621769684884913575361723052144412545323525101"
    "284320736262347790089324721363668078139228045580363211528278559333278052034025187825562865"
    "563283869051179985863500856441073408264495205553706152853901140239390583813083453138929782"
    "946517121080110383876637670106173438419253507596733224476414075014265345891995296496120856"
    "593048414341827141680276415523884377632032961134065066676791255166649173319499965817414022"
    "239972837269306182861328125",
    599,
    "2.7448244739455369599852949492552923507948333818208197008468158129083810159695483256360269"
    "175415015466412031422569540038700580780203851735358633598820734390295939334839762176968488"
    "491357536172305214441254532352510128432073626234779008932472136366807813922804558036321152"
    "827855933327805203402518782556286556328386905117998586350085644107340826449520555370615285"
    "390114023939058381308345313892978294651712108011038387663767010617343841925350759673322447"
    "641407501426534589199529649612085659304841434182714168027641552388437763203296113406506667"
    "6791255166649173319499965817414022239972837269306182861328125e-236"
  },
  {
    6.9330212812845956e-236,
    832,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000693302128128459556019730517054098"
    "598201899479144069915162346218677004387806355846786422452402293772645156697338726349371873"
    "983367424350097381613788724747210201244748437625890393184241908098354893458187915104119341"
    "768446546069323194322225287096847520558919427628701557805617700409219136650896656765033481"
    "448988584849094261228708706732282266890675010748937677467175997919721398957588961379994861"
    "200564333509007908515664863917534010543666617786469916707237836024118874742072934775462928"
    "779342329475149180326686989774048800335725396492694612182396772755548224227706288758099617"
    "552943527698516845703125",
    596,
    "6.9330212812845955601973051705409859820189947914406991516234621867700438780635584678642245"
    "240229377264515669733872634937187398336742435009738161378872474721020124474843762589039318"
    "424190809835489345818791510411934176844654606932319432222528709684752055891942762870155780"
    "561770040921913665089665676503348144898858484909426122870870673228226689067501074893767746"
    "717599791972139895758896137999486120056433350900790851566486391753401054366661778646991670"
    "723783602411887474207293477546292877934232947514918032668698977404880033572539649269461218"
    "2396772755548224227706288758099617552943527698516845703125e-236"
  },
  {
    9.887608848404935e-236,
    831,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000988760884840493529227740212029802"
    "706280150875620124272927780386820912731682208114320336162897601090023104023704996008069111"
    "083741196140661388861826358577996302406407082910484532702903501127304650131518239172470834"
    "283437373135351085880700942884901749797220162385036649402713026839454655625233726392208888"
    "970284441975958990633872635738715737326127582607828697580094942636293280089814722615886903"
    "963713220175723695858462099026339510542868187895386861625212297072254733429651534552220213"
    "138019454443987615077904499633483996281342741497111270645929917489286081408450534979692747"
    "58361279964447021484375",
    595,
    "9.8876088484049352922774021202980270628015087562012427292778038682091273168220811432033616"
    "289760109002310402370499600806911108374119614066138886182635857799630240640708291048453270"
    "290350112730465013151823917247083428343737313535108588070094288490174979722016238503664940"
    "271302683945465562523372639220888897028444197595899063387263573871573732612758260782869758"
    "009494263629328008981472261588690396371322017572369585846209902633951054286818789538686162"
    "521229707225473342965153455222021313801945444398761507790449963348399628134274149711127064"
    "592991748928608140845053497969274758361279964447021484375e-236"
  },
  {
    1.8140520866375598e-235,
    832,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000001814052086637559759281714017319832"
    "845615347108651544389194730141906742701634981466325264328182775378221400043060725800937244"
    "923408999789410546799241994708361732797013064162048498558045662932317080483637542466656897"
    "132773851376950137481730406496137930367940298307822959184929691157149248171962689707484240"
    "913709878645248765151617092817569160946057840256881823779799796725908957115177386006632445"
    "051153556989518300004374198462856061146613925239488684555318551006250185081535696799617385"
    "213871147992274592745652135539672861870128479085175141572936063038323285280528640583952437"
    "737025320529937744140625",
    597,
    "1.8140520866375597592817140173198328456153471086515443891947301419067427016349814663252643"
    "281827753782214000430607258009372449234089997894105467992419947083617327970130641620484985"
    "580456629323170804836375424666568971327738513769501374817304064961379303679402983078229591"
    "849296911571492481719626897074842409137098786452487651516170928175691609460578402568818237"
    "797997967259089571151773860066324450511535569895183000043741984628560611466139252394886845"
    "553185510062501850815356967996173852138711479922745927456521355396728618701284790851751415"
    "72936063038323285280528640583952437737025320529937744140625e-235"
  },
  {
    3.4634694166405886e-235,
    831,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000003463469416640588569972266599615681"
    "919083530515793937908223043111319858481440614215948950670924303810991380480870918391562773"
    "396968168321180155700058682912089873920922417568865509996831765897577861846007082332959929"
    "104748617248162737955257920342448513168770535182891455893675197708497078872366516469116334"
    "460618374815662621599597147245714196487683149990226170180629337325683581680262139018535551"
    "156640609221578327596539256658584427377168271165435763538417856513893401175314602355933911"
    "012032764195989159737458242017980813345760508704309560215451591550216712059317991645457368"
    "57689917087554931640625",
    596,
    "3.4634694166405885699722665996156819190835305157939379082230431113198584814406142159489506"
    "709243038109913804808709183915627733969681683211801557000586829120898739209224175688655099"
    "968317658975778618460070823329599291047486172481627379552579203424485131687705351828914558"
    "936751977084970788723665164691163344606183748156626215995971472457141964876831499902261701"
    "806293373256835816802621390185355511566406092215783275965392566585844273771682711654357635"
    "384178565138934011753146023559339110120327641959891597374582420179808133457605087043095602"
    "1545159155021671205931799164545736857689917087554931640625e-235"
  },
  {
    1.0308725436052673e-234,
    830,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000010308725436052673004078265535849362"
    "842941462603767068009055817328335165518994401933172454910493217867696432184596271668631487"
    "447065193359138589446927491863106418507505948623568420731561025482104068058179310778139467"
    "469606257489743456072635739148206436273894425724466559082563146527373333362075667563902933"
    "222563487012479314565117858212385111597451728156265292208695902500402336262572667655661806"
    "388175890193990786445710715532644427065621287497880140069704688156339008987222694650154503"
    "308016083770941603769316724062995277155244166736146518217642835268000201659477355065064330"
    "1196396350860595703125",
    596,
    "1.0308725436052673004078265535849362842941462603767068009055817328335165518994401933172454"
    "910493217867696432184596271668631487447065193359138589446927491863106418507505948623568420"
    "731561025482104068058179310778139467469606257489743456072635739148206436273894425724466559"
    "082563146527373333362075667563902933222563487012479314565117858212385111597451728156265292"
    "208695902500402336262572667655661806388175890193990786445710715532644427065621287497880140"
    "069704688156339008987222694650154503308016083770941603769316724062995277155244166736146518"
    "2176428352680002016594773550650643301196396350860595703125e-234"
  },
  {
    1.3123008280909585e-234,
    829,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000013123008280909585060094739769474378"
    "460538866107048392129824696126606516279205972547670492262550710764284363323435464168252607"
    "282504216121550058386417219638918387595741601006739821592204253433007371504412472409664067"
    "406303371832601371951685878765808244431347109769240070179015042453669205703336495525246088"
    "927422770958942579696379671576766979558709660405125643620837514272390698664170189566381311"
    "138897213106203693198480966230974668304648108481706583483004968448046454807759853953231524"
    "663460193768051528285130784630367073426811242164361861337955626193876000618043775602927780"
    "710160732269287109375",
    595,
    "1.3123008280909585060094739769474378460538866107048392129824696126606516279205972547670492"
    "262550710764284363323435464168252607282504216121550058386417219638918387595741601006739821"
    "592204253433007371504412472409664067406303371832601371951685878765808244431347109769240070"
    "179015042453669205703336495525246088927422770958942579696379671576766979558709660405125643"
    "620837514272390698664170189566381311138897213106203693198480966230974668304648108481706583"
    "483004968448046454807759853953231524663460193768051528285130784630367073426811242164361861"
    "337955626193876000618043775602927780710160732269287109375e-234"
  },
  {
    3.87322423781539e-234,
    825,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000038732242378153897318097485186350960"
    "300031186584656562863006373476206011523682262557811183503862971612833114691641746125502072"
    "880341115574094325771751109269324097178608013529608749513409124237583837416850526113038004"
    "925981667230897823396742262778016289726881528972185781894934891703312957475605352520561179"
    "848546186825239305131571924373146542287418567990354344016310437205184558983660529561615320"
    "898762527140916102609030211008210383276559453108001243028592320154717229013144171564493874"
    "024049142539487772727966163437511539617658723409899308317807842304628107577002538164379075"
    "16956329345703125",
    591,
    "3.8732242378153897318097485186350960300031186584656562863006373476206011523682262557811183"
    "503862971612833114691641746125502072880341115574094325771751109269324097178608013529608749"
    "513409124237583837416850526113038004925981667230897823396742262778016289726881528972185781"
    "894934891703312957475605352520561179848546186825239305131571924373146542287418567990354344"
    "016310437205184558983660529561615320898762527140916102609030211008210383276559453108001243"
    "028592320154717229013144171564493874024049142539487772727966163437511539617658723409899308"
    "31780784230462810757700253816437907516956329345703125e-234"
  },
  {
    9.2010739647813e-234,
    827,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000092010739647813000085411005568727105"
    "998799066899037448965302675135776884816654819628173659937259171048554565929979214712659389"
    "623130600333190907794838391075515187831400099110511118637642668704758301617460535276681485"
    "343318182462982458160681059970485369318360943437521600454275315514820798660680377329862207"
    "465454929638901238271162895703777874458709089484490860087292013612010824285906041126662575"
    "922290223290517197672726442811029075924386055866495721558512948039413487672903898945678035"
    "746290351193063272360450441041190170760119477152441024662891387077019028639668363211967516"
    "6904926300048828125",
    593,
    "9.2010739647813000085411005568727105998799066899037448965302675135776884816654819628173659"
    "937259171048554565929979214712659389623130600333190907794838391075515187831400099110511118"
    "637642668704758301617460535276681485343318182462982458160681059970485369318360943437521600"
    "454275315514820798660680377329862207465454929638901238271162895703777874458709089484490860"
    "087292013612010824285906041126662575922290223290517197672726442811029075924386055866495721"
    "558512948039413487672903898945678035746290351193063272360450441041190170760119477152441024"
    "6628913870770190286396683632119675166904926300048828125e-234"
  },
  {
    1.1992483584063614e-233,
    817,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000119924835840636139373963820028986190"
    "298195386861881910513764685874183980449238197882579818103124509702374293152236901426529753"
    "933885926217522939375787436168025095420854421089105328643012783084193445775178532627019300"
    "202128858764260740848896566266991633767322300226451099463423684184093373772462383201655323"
    "699202867840150792223802687054011435753250588695677560267140673869453512352957484048081742"
    "681160517935100483419861018435189240188373810482122213082644041469933620211270257379648322"
    "982766563842384538563467443736569589192277871478824920734897185869982649819576181471347808"
    "837890625",
    584,
    "1.1992483584063613937396382002898619029819538686188191051376468587418398044923819788257981"
    "810312450970237429315223690142652975393388592621752293937578743616802509542085442108910532"
    "864301278308419344577517853262701930020212885876426074084889656626699163376732230022645109"
    "946342368418409337377246238320165532369920286784015079222380268705401143575325058869567756"
    "026714067386945351235295748404808174268116051793510048341986101843518924018837381048212221"
    "308264404146993362021127025737964832298276656384238453856346744373656958919227787147882492"
    "0734897185869982649819576181471347808837890625e-233"
  },
  {
    2.3452909097606e-233,
    823,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000234529090976060021029983888573873983"
    "201465825159325691596229417485075020716028162190038577919142927662102871282920701702839101"
    "779318975472524788417303881315933266593759151136812057701602281728908118286422653068981297"
    "672141753077006927761131860435877673374316145811845349627845260180344150355363220174745189"
    "922773287927939393830707610654468322973526832192717737890383870841507295666699812163491329"
    "352763693592948219303919368730207897372104156558695370518639326240457297236641350456607151"
    "007096954463810541500796919853141331376838730027058339967416700395930195810478835483081638"
    "813018798828125",
    590,
    "2.3452909097606002102998388857387398320146582515932569159622941748507502071602816219003857"
    "791914292766210287128292070170283910177931897547252478841730388131593326659375915113681205"
    "770160228172890811828642265306898129767214175307700692776113186043587767337431614581184534"
    "962784526018034415035536322017474518992277328792793939383070761065446832297352683219271773"
    "789038387084150729566669981216349132935276369359294821930391936873020789737210415655869537"
    "051863932624045729723664135045660715100709695446381054150079691985314133137683873002705833"
    "9967416700395930195810478835483081638813018798828125e-233"
  },
  {
    6.55832259537843e-233,
    824,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000655832259537843040439548346141760881"
    "942954059403416011980731538759886012372617043790559933365976788191707725308732928182976190"
    "209935831121926799822295206356039751458151538698202210999800319499370069053119068921318649"
    "551370599754841947072256810504631662142490871995594322252196592214081064665147791627754299"
    "619721002440783544482018040091729190864251451963319322672293756387264440779599986292594881"
    "072366219565719189998103239748735717531246698300106404621320272068687120569178549223991517"
    "121489275313276613889947072468874020996202783357889095986976502657833654197361283877398818"
    "7313079833984375",
    591,
    "6.5583225953784304043954834614176088194295405940341601198073153875988601237261704379055993"
    "336597678819170772530873292818297619020993583112192679982229520635603975145815153869820221"
    "099980031949937006905311906892131864955137059975484194707225681050463166214249087199559432"
    "225219659221408106466514779162775429961972100244078354448201804009172919086425145196331932"
    "267229375638726444077959998629259488107236621956571918999810323974873571753124669830010640"
    "462132027206868712056917854922399151712148927531327661388994707246887402099620278335788909"
    "59869765026578336541973612838773988187313079833984375e-233"
  },
  {
    1.01023596160976e-232,
    822,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000001010235961609760000694404369626630431"
    "040256612388038133052586867836058641770028431371199769739320783499649064124775574220382629"
    "810839018012388827720150663315264203463880475089652049354996245677556813988458561715262242"
    "364880332481311367223365533330138935433511898794179657484124403659524277293102547115573188"
    "942514605904126273728889908537679517889697751910500073905342109787791803143772166329228442"
    "333870056541814956903081509819768976437949905372395341814698585477427930455840480364970460"
    "096700113363793784472737601635503939348175160230514832760854515916748574966277374187484383"
    "58306884765625",
    590,
    "1.0102359616097600006944043696266304310402566123880381330525868678360586417700284313711997"
    "697393207834996490641247755742203826298108390180123888277201506633152642034638804750896520"
    "493549962456775568139884585617152622423648803324813113672233655333301389354335118987941796"
    "574841244036595242772931025471155731889425146059041262737288899085376795178896977519105000"
    "739053421097877918031437721663292284423338700565418149569030815098197689764379499053723953"
    "418146985854774279304558404803649704600967001133637937844727376016355039393481751602305148"
    "3276085451591674857496627737418748438358306884765625e-232"
  },
  {
    2.5973213970621353e-232,
    822,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000002597321397062135304482454518820059218"
    "341904238563972729205356266624967554494269191251702550594714010912120186917916765895983048"
    "599631991780520962203620444536908024011856433780670158318226201921782197919397572309149709"
    "039667116952289847151837424216761341021681339723578337857881286638542378091891939886703814"
    "853841423404349829762547421686641518521487126745136915551420174622436559040848737208152598"
    "070647863885904017009536566406455700942818368886508589824748630609074368927306714824973309"
    "805714092994491615308634723898225453608078472782239545021688549922278532733344036387279629"
    "70733642578125",
    590,
    "2.5973213970621353044824545188200592183419042385639727292053562666249675544942691912517025"
    "505947140109121201869179167658959830485996319917805209622036204445369080240118564337806701"
    "583182262019217821979193975723091497090396671169522898471518374242167613410216813397235783"
    "378578812866385423780918919398867038148538414234043498297625474216866415185214871267451369"
    "155514201746224365590408487372081525980706478638859040170095365664064557009428183688865085"
    "898247486306090743689273067148249733098057140929944916153086347238982254536080784727822395"
    "4502168854992227853273334403638727962970733642578125e-232"
  },
  {
    4.955980006523515e-232,
    821,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000004955980006523515348979641019416139096"
    "648835561758566654270733913690055871121878555736385611230374810791309986802372782433508435"
    "375322008993365340619986890224801448685945509568718742271638818953199636583745220519046712"
    "966152320510867618543585720024531194940872820672919233356870518352801434898546241842045716"
    "492357747647795795854006844271213204574970248111617111988412283896397690238370324810245202"
    "312443034057065200979920481652492615232633927148499564024439563982325881645117144098140065"
    "448506502009259057265030332876997865741831119812975552591579209843475339880569663364440202"
    "7130126953125",
    589,
    "4.9559800065235153489796410194161390966488355617585666542707339136900558711218785557363856"
    "112303748107913099868023727824335084353753220089933653406199868902248014486859455095687187"
    "422716388189531996365837452205190467129661523205108676185435857200245311949408728206729192"
    "333568705183528014348985462418420457164923577476477957958540068442712132045749702481116171"
    "119884122838963976902383703248102452023124430340570652009799204816524926152326339271484995"
    "640244395639823258816451171440981400654485065020092590572650303328769978657418311198129755"
    "525915792098434753398805696633644402027130126953125e-232"
  },
  {
    1.1244377262803144e-231,
    820,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000011244377262803143878068016423688012596"
    "039718894748761522693123703181634130104650640135262083103846749672518801249249538170476428"
    "596608475348806502147009220854218318208403259597181016504446505711542445446519189204333592"
    "427511602808427136327807740545054320196126248781724647343012731676781634311202424056214158"
    "247761264608947838926595118197402289470545118797483097395936263215464251151336740856339180"
    "880724959346882809972866164269263377530400296198176892310064309601980174227287902248981317"
    "884777041785984059282988554452900238197385204296812693280688440211623202458213199861347675"
    "323486328125",
    589,
    "1.1244377262803143878068016423688012596039718894748761522693123703181634130104650640135262"
    "083103846749672518801249249538170476428596608475348806502147009220854218318208403259597181"
    "016504446505711542445446519189204333592427511602808427136327807740545054320196126248781724"
    "647343012731676781634311202424056214158247761264608947838926595118197402289470545118797483"
    "097395936263215464251151336740856339180880724959346882809972866164269263377530400296198176"
    "892310064309601980174227287902248981317884777041785984059282988554452900238197385204296812"
    "693280688440211623202458213199861347675323486328125e-231"
  },
  {
    2.4089050126555283e-231,
    819,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000024089050126555282541403121147937146640"
    "322688813110950873494165256577427141764900209847715481143046869400508714270672918718286848"
    "988872663121901055634647037723578276123125516128349187148157183017179831731319145661743097"
    "832289929873953098742149380851677801977037134500583883415558104439060594682754979707091804"
    "867067185871313640308216356155999604435917605136780020449205792873192769293405985822537229"
    "012759146679613240165097560521086933490729562560414901156584319689412299906187005948185278"
    "022005662480251971989026839432736507264895208278184952210973154840267795862018829211592674"
    "25537109375",
    588,
    "2.4089050126555282541403121147937146640322688813110950873494165256577427141764900209847715"
    "481143046869400508714270672918718286848988872663121901055634647037723578276123125516128349"
    "187148157183017179831731319145661743097832289929873953098742149380851677801977037134500583"
    "883415558104439060594682754979707091804867067185871313640308216356155999604435917605136780"
    "020449205792873192769293405985822537229012759146679613240165097560521086933490729562560414"
    "901156584319689412299906187005948185278022005662480251971989026839432736507264895208278184"
    "95221097315484026779586201882921159267425537109375e-231"
  },
  {
    3.1957993198054626e-231,
    815,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000031957993198054625725213645741173162786"
    "935684857677527438090979500370647810397958920480824463940923329563437363685177400336842377"
    "167211712128391896467408379088000492459813190319379094782060739161361771636335060579156222"
    "329606273263928601159509409470295790985613570978544141169919509750076880703428339239981167"
    "605168701215749176970374592880103567554395837929673462006697904438147786695003031463211995"
    "666528121763991142543575975589971847109339742740076626470903744949365521927945602356867632"
    "925861853636270287275936796300506314881253772642310270832993746026318149233702570199966430"
    "6640625",
    584,
    "3.1957993198054625725213645741173162786935684857677527438090979500370647810397958920480824"
    "463940923329563437363685177400336842377167211712128391896467408379088000492459813190319379"
    "094782060739161361771636335060579156222329606273263928601159509409470295790985613570978544"
    "141169919509750076880703428339239981167605168701215749176970374592880103567554395837929673"
    "462006697904438147786695003031463211995666528121763991142543575975589971847109339742740076"
    "626470903744949365521927945602356867632925861853636270287275936796300506314881253772642310"
    "2708329937460263181492337025701999664306640625e-231"
  },
  {
    6.380419758740376e-231,
    816,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000063804197587403760326652898054200801828"
    "323214780442215032022007620478359597818401693284013081678136665027405649019757328785359072"
    "410017997799082473115643094090029365795499978289712081064470112832907688547498226131520045"
    "345679802404848525663468490921519087993712872572553824454547352150009875039283616403045378"
    "848893428643145781279385770628277566694325711581556041680649755580831709124446178875091795"
    "627785989583994623827758905875393700067186061579595903977837711660628927349432190801081573"
    "067233350735781687425858348858427949253371551367650219054386551142954431270482018589973449"
    "70703125",
    585,
    "6.3804197587403760326652898054200801828323214780442215032022007620478359597818401693284013"
    "081678136665027405649019757328785359072410017997799082473115643094090029365795499978289712"
    "081064470112832907688547498226131520045345679802404848525663468490921519087993712872572553"
    "824454547352150009875039283616403045378848893428643145781279385770628277566694325711581556"
    "041680649755580831709124446178875091795627785989583994623827758905875393700067186061579595"
    "903977837711660628927349432190801081573067233350735781687425858348858427949253371551367650"
    "21905438655114295443127048201858997344970703125e-231"
  },
  {
    1.6095894110621704e-230,
    816,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000160958941106217041538322012940559715323"
    "903916159110124181021736915658438094084503336096695857168934173226321447397301047417148502"
    "624258582835574723816240415594393369665164751768647196218708742592134423663760259422572558"
    "244922466260598140262058533327578702173157350780950647812841845521229652314867912626642669"
    "965240116553476867889831360769565542738939395573853790203734536452220441066673265983380330"
    "480583155873811564176938159689442060769621685872330312916464823415664426762643483429352265"
    "211453025979662093409239736334483968002895346372280144893116005100353049783734604716300964"
    "35546875",
    586,
    "1.6095894110621704153832201294055971532390391615911012418102173691565843809408450333609669"
    "585716893417322632144739730104741714850262425858283557472381624041559439336966516475176864"
    "719621870874259213442366376025942257255824492246626059814026205853332757870217315735078095"
    "064781284184552122965231486791262664266996524011655347686788983136076956554273893939557385"
    "379020373453645222044106667326598338033048058315587381156417693815968944206076962168587233"
    "031291646482341566442676264348342935226521145302597966209340923973633448396800289534637228"
    "014489311600510035304978373460471630096435546875e-230"
  },
  {
    4.032612821947131e-230,
    813,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000403261282194713087763407746328019687304"
    "227981270857179095475826427227806099871552796153902810669339478058442834132325348297619070"
    "203917114895484113342497422436622422171014950377706618395303383927177010501146744877553895"
    "356422544544335952168545277006149704782564931265353207175263173410499299572631303771514474"
    "195966762133785775576328291545437786357784006648441476501265864809787107226992156017700321"
    "936913898614935271862924100599178352375439954606326097129388309765290899844628674147158970"
    "800023347955167381151032697590479982052777906725897711162631686931945296237245202064514160"
    "15625",
    583,
    "4.0326128219471308776340774632801968730422798127085717909547582642722780609987155279615390"
    "281066933947805844283413232534829761907020391711489548411334249742243662242217101495037770"
    "661839530338392717701050114674487755389535642254454433595216854527700614970478256493126535"
    "320717526317341049929957263130377151447419596676213378577557632829154543778635778400664844"
    "147650126586480978710722699215601770032193691389861493527186292410059917835237543995460632"
    "609712938830976529089984462867414715897080002334795516738115103269759047998205277790672589"
    "771116263168693194529623724520206451416015625e-230"
  },
  {
    6.0387918131443855e-230,
    811,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000603879181314438552939855044884559157244"
    "512926852670147678298490735004120607449545855310636730305293393951523888733562888957496230"
    "925242955930292779412423416985599139667873737076545274846108104562187768695151949854512801"
    "214193336497448752237649116269977376321099510707654908498919534668861605879616023392761122"
    "673420777016870697029781556749566357933326573876209439797879935135106867765540555947104291"
    "944949386627062503390682659995039549772250462149515137830868639868155808286646814645933763"
    "259664624485459438133730144593675207962419919759523671831669133780451375059783458709716796"
    "875",
    581,
    "6.0387918131443855293985504488455915724451292685267014767829849073500412060744954585531063"
    "673030529339395152388873356288895749623092524295593029277941242341698559913966787373707654"
    "527484610810456218776869515194985451280121419333649744875223764911626997737632109951070765"
    "490849891953466886160587961602339276112267342077701687069702978155674956635793332657387620"
    "943979787993513510686776554055594710429194494938662706250339068265999503954977225046214951"
    "513783086863986815580828664681464593376325966462448545943813373014459367520796241991975952"
    "3671831669133780451375059783458709716796875e-230"
  },
  {
    8.974323011245674e-230,
    813,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000897432301124567425428992797962682729344"
    "377879282934480862981070776476162765089889547916595804288044644798103641898249421376652724"
    "137290600459136448447242365922273624432734917290803189814744215370570273183759401554479459"
    "386097693911427911040229202448323679333061875705982148284852062747637365268829089956792615"
    "954904294463285439047758892458893062225649922349005898381028619643998845806942071486240555"
    "869742445831171652450554772427198667133558451445735864240418102343992211275762410613327206"
    "706574095487836196314795231910512044608674787857666979226656245316462445771321654319763183"
    "59375",
    583,
    "8.9743230112456742542899279796268272934437787928293448086298107077647616276508988954791659"
    "580428804464479810364189824942137665272413729060045913644844724236592227362443273491729080"
    "318981474421537057027318375940155447945938609769391142791104022920244832367933306187570598"
    "214828485206274763736526882908995679261595490429446328543904775889245889306222564992234900"
    "589838102861964399884580694207148624055586974244583117165245055477242719866713355845144573"
    "586424041810234399221127576241061332720670657409548783619631479523191051204460867478785766"
    "697922665624531646244577132165431976318359375e-230"
  },
  {
    1.8798157375327938e-229,
    812,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000001879815737532793783617736555231155172229"
    "856699268913749540840118006446081027566955736826248605047653570682229143154372846202159429"
    "019436045435644649504351411553584618810450861106936705352227437584256873127730164648403232"
    "716340427184106949371967480783761089412133835752241924595873877701261614154333943909110324"
    "386106585710639204443313355279898818069270560229701821174932349894397049419446397740811025"
    "213895891696714349771919521346467022907845451052601328720021485657518430282316906559267028"
    "608176010620252575967031451476037437836512635943567086499461638027241860982030630111694335"
    "9375",
    583,
    "1.8798157375327937836177365552311551722298566992689137495408401180064460810275669557368262"
    "486050476535706822291431543728462021594290194360454356446495043514115535846188104508611069"
    "367053522274375842568731277301646484032327163404271841069493719674807837610894121338357522"
    "419245958738777012616141543339439091103243861065857106392044433133552798988180692705602297"
    "018211749323498943970494194463977408110252138958916967143497719195213464670229078454510526"
    "013287200214856575184302823169065592670286081760106202525759670314514760374378365126359435"
    "670864994616380272418609820306301116943359375e-229"
  },
  {
    6.1019603119906395e-229,
    810,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000006101960311990639482370335989341904104348"
    "892866877712732160977960229595775584620677158743966749819160787012731884077873234225312484"
    "501399731681232773444628204378897443670340405165260057441307837979208879343268808910151822"
    "705828875469292565269036546147893915456863015857572909063802064779293512026308928862177518"
    "529874001744914269355008873261922774852235236855717527848126431411854646898935281848896381"
    "491380652507002409142303849126507863537916582989505083248647557799326948431744454808215738"
    "540980419242520356242812167321960954886041044316043538420224479068565415218472480773925781"
    "25",
    581,
    "6.1019603119906394823703359893419041043488928668777127321609779602295957755846206771587439"
    "667498191607870127318840778732342253124845013997316812327734446282043788974436703404051652"
    "600574413078379792088793432688089101518227058288754692925652690365461478939154568630158575"
    "729090638020647792935120263089288621775185298740017449142693550088732619227748522352368557"
    "175278481264314118546468989352818488963814913806525070024091423038491265078635379165829895"
    "050832486475577993269484317444548082157385409804192425203562428121673219609548860410443160"
    "4353842022447906856541521847248077392578125e-229"
  },
  {
    1.1326182622927228e-228,
    808,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000011326182622927227585026998401953453564387"
    "331526632995841434572566012270184131360172862035269820314552997461241377632945337361515620"
    "918822560229961170393130765553757248043014053314277604328004618600569540026322640540986956"
    "589927517636396806648073068142760962572911325417218855366383540612118299239233361846081797"
    "682496615655564150929904808543900488190715743181006200383902980812754628242850109644997858"
    "821492841688957682488157080189277388885933187971360453197973651452791340741998288731311405"
    "618485841516869165310136725186574173985326892508292140337999853727524168789386749267578125",
    580,
    "1.1326182622927227585026998401953453564387331526632995841434572566012270184131360172862035"
    "269820314552997461241377632945337361515620918822560229961170393130765553757248043014053314"
    "277604328004618600569540026322640540986956589927517636396806648073068142760962572911325417"
    "218855366383540612118299239233361846081797682496615655564150929904808543900488190715743181"
    "006200383902980812754628242850109644997858821492841688957682488157080189277388885933187971"
    "360453197973651452791340741998288731311405618485841516869165310136725186574173985326892508"
    "292140337999853727524168789386749267578125e-228"
  },
  {
    1.3463284210193606e-228,
    808,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000013463284210193605937745722937725857242949"
    "709999335060039840423204785367847390302267195263556416334323998196073782473607018630567269"
    "118364564600234366654623643949382152730430690017780065413052347695105366591075498020066423"
    "695822701524011562699271477756277346523276928305646394985430718139457738670216500334869260"
    "533748607064447801305583373185311623132192011604915251465977752968767098478902675252505947"
    "952937650369501082058516769660425361944455031347308706252788388788524194595110661637313910"
    "166749819315348258679689093276443200938289032121165933464368436034419573843479156494140625",
    580,
    "1.3463284210193605937745722937725857242949709999335060039840423204785367847390302267195263"
    "556416334323998196073782473607018630567269118364564600234366654623643949382152730430690017"
    "780065413052347695105366591075498020066423695822701524011562699271477756277346523276928305"
    "646394985430718139457738670216500334869260533748607064447801305583373185311623132192011604"
    "915251465977752968767098478902675252505947952937650369501082058516769660425361944455031347"
    "308706252788388788524194595110661637313910166749819315348258679689093276443200938289032121"
    "165933464368436034419573843479156494140625e-228"
  },
  {
    4.951126320251196e-228,
    806,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000049511263202511957641512399699829602641557"
    "623549736408253556822425590975395266427611114962041653194091053383333687977528459131961733"
    "246268075570183108576832296256725838984844196198969342687811673558788740825937435923292871"
    "694567290111231754249585805269961897940577926102625382575379827069744888403165357553159629"
    "221212082104534907468772395489064614074689875744296450508670594617851315848483996401206801"
    "717130288469826373770474107040550642701421757615256057500503308103419946710300086061346893"
    "7483063135363487535101835560640066559693432397641632558560331744956783950328826904296875",
    578,
    "4.9511263202511957641512399699829602641557623549736408253556822425590975395266427611114962"
    "041653194091053383333687977528459131961733246268075570183108576832296256725838984844196198"
    "969342687811673558788740825937435923292871694567290111231754249585805269961897940577926102"
    "625382575379827069744888403165357553159629221212082104534907468772395489064614074689875744"
    "296450508670594617851315848483996401206801717130288469826373770474107040550642701421757615"
    "256057500503308103419946710300086061346893748306313536348753510183556064006655969343239764"
    "1632558560331744956783950328826904296875e-228"
  },
  {
    8.472853590985601e-228,
    806,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000084728535909856012410200041906372414632296"
    "215272648004484427766845608477344705431493395232423944742460409199077822362080714239493454"
    "257568353046797628519690492599331320691011073453588367065068265407767505172055549071537857"
    "531129026112470221377199546949364922377684655844054764241349832683592295417675956421436580"
    "698658868311486350995177436653307794720582217765353864645824269735870431707954815959499644"
    "615997063996698543462258749378243096021674191619062240164618779799121232700677981587852874"
    "1174104469600086982166560311473505952658790405830624425931318910443224012851715087890625",
    578,
    "8.4728535909856012410200041906372414632296215272648004484427766845608477344705431493395232"
    "423944742460409199077822362080714239493454257568353046797628519690492599331320691011073453"
    "588367065068265407767505172055549071537857531129026112470221377199546949364922377684655844"
    "054764241349832683592295417675956421436580698658868311486350995177436653307794720582217765"
    "353864645824269735870431707954815959499644615997063996698543462258749378243096021674191619"
    "062240164618779799121232700677981587852874117410446960008698216656031147350595265879040583"
    "0624425931318910443224012851715087890625e-228"
  },
  {
    1.268665118648964e-227,
    802,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000126866511864896404186137290900006248836641"
    "257278266915285398248979462732789726736621235569136796631789208189983431026863877108668219"
    "641584809556168837216776958126517591068005717065263272298248242603898184394956661293554521"
    "864207181714532221752102556004126914623940394801367161197085055601344592896420774087053315"
    "725742354741926248513238428796400051451892563704678510696141675186808839002532227560976045"
    "649471966663022054729635272862815877076419614503481104276280881883230397522547283862747567"
    "583163122464808826042178125167560452464976701592025420950449188239872455596923828125",
    575,
    "1.2686651186489640418613729090000624883664125727826691528539824897946273278972673662123556"
    "913679663178920818998343102686387710866821964158480955616883721677695812651759106800571706"
    "526327229824824260389818439495666129355452186420718171453222175210255600412691462394039480"
    "136716119708505560134459289642077408705331572574235474192624851323842879640005145189256370"
    "467851069614167518680883900253222756097604564947196666302205472963527286281587707641961450"
    "348110427628088188323039752254728386274756758316312246480882604217812516756045246497670159"
    "2025420950449188239872455596923828125e-227"
  },
  {
    2.688551806792703e-227,
    805,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000268855180679270301356005408617076550568021"
    "599393737480835413901747223865340145085524986538766100510099067445588072524415445950949218"
    "647103736177038525626564873722274278821070632004195312288860329438863021795170526034848830"
    "564357071296682910382818492048390164518266365230117018909142870410476778790448300028738043"
    "622467217814381441088518652939570939011802965618361964376477901864639499728004298783375283"
    "138485961700105907608224914495063158132615052678462862682436495900588406487060191263024372"
    "546385805536617848116566363084955103460608268235720874628214005497284233570098876953125",
    578,
    "2.6885518067927030135600540861707655056802159939373748083541390174722386534014508552498653"
    "876610051009906744558807252441544595094921864710373617703852562656487372227427882107063200"
    "419531228886032943886302179517052603484883056435707129668291038281849204839016451826636523"
    "011701890914287041047677879044830002873804362246721781438144108851865293957093901180296561"
    "836196437647790186463949972800429878337528313848596170010590760822491449506315813261505267"
    "846286268243649590058840648706019126302437254638580553661784811656636308495510346060826823"
    "5720874628214005497284233570098876953125e-227"
  },
  {
    4.421464033750511e-227,
    803,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000442146403375051115549449229048232483436235"
    "645945164491769292916975053886089953905581257128754791912335240586693816810894915724128757"
    "425416511778571176251130211532941252220691325000829704363646059477032385920079354496268156"
    "739306001262573963473550597176323648730299698711974353214972921366758853059392782604085205"
    "992710613407198318101619344895552833595052575153604474771728012139453372565783915378276333"
    "979576459174713878464261200380015132547253153164479656660812754868324751497403180618654707"
    "0836287468682968970207399435141959840122704954168408875148088554851710796356201171875",
    576,
    "4.4214640337505111554944922904823248343623564594516449176929291697505388608995390558125712"
    "875479191233524058669381681089491572412875742541651177857117625113021153294125222069132500"
    "082970436364605947703238592007935449626815673930600126257396347355059717632364873029969871"
    "197435321497292136675885305939278260408520599271061340719831810161934489555283359505257515"
    "360447477172801213945337256578391537827633397957645917471387846426120038001513254725315316"
    "447965666081275486832475149740318061865470708362874686829689702073994351419598401227049541"
    "68408875148088554851710796356201171875e-227"
  },
  {
    1.6504299596692555e-226,
    803,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000001650429959669255466289843831556874268026308"
    "299105688301448050801820596161594006406167919149116567972342502528633809026611619473914203"
    "575226695110014755603134810874402588512571826899337246445297246553276626469882336762161281"
    "034853480590013149229622340795138373021923715805777359887470617461808606384004424205619224"
    "951313346682081318857464033916723241671119693909166036489531757375394715795013793737293261"
    "186978408512639099218568828122064104941327084405161941916779504534198182319569569273053021"
    "4763142734991744126675332954466959597313317938127585904339866829104721546173095703125",
    577,
    "1.6504299596692554662898438315568742680263082991056883014480508018205961615940064061679191"
    "491165679723425025286338090266116194739142035752266951100147556031348108744025885125718268"
    "993372464452972465532766264698823367621612810348534805900131492296223407951383730219237158"
    "057773598874706174618086063840044242056192249513133466820813188574640339167232416711196939"
    "091660364895317573753947157950137937372932611869784085126390992185688281220641049413270844"
    "051619419167795045341981823195695692730530214763142734991744126675332954466959597313317938"
    "127585904339866829104721546173095703125e-226"
  },
  {
    2.290009142698998e-226,
    799,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000002290009142698997935839253060546178501675948"
    "201500495043391589290713325015288644784094897823568726988932085980115553293515389936957901"
    "739981436559145450779378992877790037673882467134229994518733990751968572879016232254708606"
    "129229326286431030847922256382404862982847219322903060522246908430848008821265886585966924"
    "413435690755441409434776003663674043233887504353546349781354426946893811482717809150546571"
    "673421249634131103639734074609825126412085626048859210797049749415351752705234733937121147"
    "766631468394943062111239263831349277623049276453315314938663505017757415771484375",
    573,
    "2.2900091426989979358392530605461785016759482015004950433915892907133250152886447840948978"
    "235687269889320859801155532935153899369579017399814365591454507793789928777900376738824671"
    "342299945187339907519685728790162322547086061292293262864310308479222563824048629828472193"
    "229030605222469084308480088212658865859669244134356907554414094347760036636740432338875043"
    "535463497813544269468938114827178091505465716734212496341311036397340746098251264120856260"
    "488592107970497494153517527052347339371211477666314683949430621112392638313492776230492764"
    "53315314938663505017757415771484375e-226"
  },
  {
    4.095081668886285e-226,
    800,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000004095081668886285171574198171651055549113180"
    "692375307962564687320010101184183310556371391571292008962787733866727329374672025191194795"
    "224807234077953539142049054179596385647635507320267095499467503726553661467687518465985902"
    "038355614934756776928998890897769920782971767051608290557297810235286435333791164042348138"
    "624570904649638129484294437380588130732521005433668371194610258926496822790877767878174007"
    "929794976185694127951447015973190588292187212001502539099993036783572947568341165095404737"
    "7465328570858001204067822030524909524755276801766257221970590762794017791748046875",
    574,
    "4.0950816688862851715741981716510555491131806923753079625646873200101011841833105563713915"
    "712920089627877338667273293746720251911947952248072340779535391420490541795963856476355073"
    "202670954994675037265536614676875184659859020383556149347567769289988908977699207829717670"
    "516082905572978102352864353337911640423481386245709046496381294842944373805881307325210054"
    "336683711946102589264968227908777678781740079297949761856941279514470159731905882921872120"
    "015025390999930367835729475683411650954047377465328570858001204067822030524909524755276801"
    "766257221970590762794017791748046875e-226"
  },
  {
    1.15703960191472e-225,
    799,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000011570396019147199951614746811927139404489382"
    "923242367733464441788630888078074734225096038097286016984649018125775151464503555615646404"
    "060658691264580754160801113496274674184840568134100418868606634608488383801544877093923722"
    "784361068947147251684888914022566139588955681859345232808037850254879457610336826024268079"
    "617729904564830592280904276028380829012893805188297596888949701179576800381646868176872342"
    "258629059497492830636304470371179529986886830620720903236045776519393561035298250206099701"
    "234656483366474654869577789030389398873717443638753366030869074165821075439453125",
    574,
    "1.1570396019147199951614746811927139404489382923242367733464441788630888078074734225096038"
    "097286016984649018125775151464503555615646404060658691264580754160801113496274674184840568"
    "134100418868606634608488383801544877093923722784361068947147251684888914022566139588955681"
    "859345232808037850254879457610336826024268079617729904564830592280904276028380829012893805"
    "188297596888949701179576800381646868176872342258629059497492830636304470371179529986886830"
    "620720903236045776519393561035298250206099701234656483366474654869577789030389398873717443"
    "638753366030869074165821075439453125e-225"
  },
  {
    1.8221726162720878e-225,
    799,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000018221726162720878219391559104158395374959669"
    "147213245613503896745402028338069685891693215118247249574470577577363283406905925589340504"
    "067750696872317929067358648190651662372963944126659917530731609662268783558670926187288647"
    "139247271777198722046011940213644445587076392379048934061006896153708602934342271317121484"
    "057989380802416798615537137977507699992247403662018950839836603506336485592101596794523530"
    "482290152376909489707030059187727235006050422294906448778167618793769252410314007579540051"
    "336232970229014522482537912632759093969035679716483855372644029557704925537109375",
    574,
    "1.8221726162720878219391559104158395374959669147213245613503896745402028338069685891693215"
    "118247249574470577577363283406905925589340504067750696872317929067358648190651662372963944"
    "126659917530731609662268783558670926187288647139247271777198722046011940213644445587076392"
    "379048934061006896153708602934342271317121484057989380802416798615537137977507699992247403"
    "662018950839836603506336485592101596794523530482290152376909489707030059187727235006050422"
    "294906448778167618793769252410314007579540051336232970229014522482537912632759093969035679"
    "716483855372644029557704925537109375e-225"
  },
  {
    3.7536981380660996e-225,
    798,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000037536981380660995920317409206965536403197907"
    "919708438483243207276046259780782636795750670960498823863331556436197349751507464907406536"
    "791259502997841818725518956116584948522260009384734395632938955897555755943399454267975263"
    "596254224807511403010194004559027011419730714020701866858639463694156994769756096553422426"
    "077154175594829868582239458210962130867456517266283551635612083202636192488879561782024006"
    "010046525100412652755517015868480555727850528566770216506559421430507788004742729225958245"
    "14705341240110874954266700298560860654499549138307656903634779155254364013671875",
    573,
    "3.7536981380660995920317409206965536403197907919708438483243207276046259780782636795750670"
    "960498823863331556436197349751507464907406536791259502997841818725518956116584948522260009"
    "384734395632938955897555755943399454267975263596254224807511403010194004559027011419730714"
    "020701866858639463694156994769756096553422426077154175594829868582239458210962130867456517"
    "266283551635612083202636192488879561782024006010046525100412652755517015868480555727850528"
    "566770216506559421430507788004742729225958245147053412401108749542667002985608606544995491"
    "38307656903634779155254364013671875e-225"
  },
  {
    7.40644242989838e-225,
    797,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000074064424298983799758409921962326965702801554"
    "143551704944707467008311784842257503029702562542519306298330314198119432646936936219640005"
    "302425837793809866997855583039763458231724336941595820523128212963394766431215319763781916"
    "023622838051887986541964209509915484701324383469822898179296169276538100368828338614294568"
    "521970713465987069320473011773610399506362043394265338556489274108520861453579183080406614"
    "208415614734431946343515294217972587643229238349975638672734043378699267533306073702414230"
    "5637908247415113277343120253686591211890466457390402865712530910968780517578125",
    572,
    "7.4064424298983799758409921962326965702801554143551704944707467008311784842257503029702562"
    "542519306298330314198119432646936936219640005302425837793809866997855583039763458231724336"
    "941595820523128212963394766431215319763781916023622838051887986541964209509915484701324383"
    "469822898179296169276538100368828338614294568521970713465987069320473011773610399506362043"
    "394265338556489274108520861453579183080406614208415614734431946343515294217972587643229238"
    "349975638672734043378699267533306073702414230563790824741511327734312025368659121189046645"
    "7390402865712530910968780517578125e-225"
  },
  {
    1.5302655517853282e-224,
    796,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000153026555178532818655190880366416723494533640"
    "101523627609129374874427510679603081556815132040430603531140119169719954301232130009138770"
    "629329047623431367136583488292739563960649306317491883315147232877811185355163837018346607"
    "302335041417943065326781838394059886318734114461916993898805850494618899016611168903714117"
    "484190493108735630851325223633022032463961318185608942822039197627528288012138016609431683"
    "474435877212217601824120899083102402432899611516052729144447370268452974713939382192826509"
    "551728150820699522893409931803161412415537068199000714230351150035858154296875",
    572,
    "1.5302655517853281865519088036641672349453364010152362760912937487442751067960308155681513"
    "204043060353114011916971995430123213000913877062932904762343136713658348829273956396064930"
    "631749188331514723287781118535516383701834660730233504141794306532678183839405988631873411"
    "446191699389880585049461889901661116890371411748419049310873563085132522363302203246396131"
    "818560894282203919762752828801213801660943168347443587721221760182412089908310240243289961"
    "151605272914444737026845297471393938219282650955172815082069952289340993180316141241553706"
    "8199000714230351150035858154296875e-224"
  },
  {
    3.266244852085614e-224,
    794,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000326624485208561415823599691161976473147582190"
    "087362967971765252144015208673404054440196263960025977722528663186968836682916017000501882"
    "086507974134707585824404879354484989379169992763394149298116714053649735117003462047395167"
    "397663234733373692739982763510086321240330369083136374788335771909573664617846699061080110"
    "891451006885468613446077088830524248699862618018880463506378464151478648017516799147765506"
    "530148904271560421265143357707928115720602473152213992952494569908513824173282587811025445"
    "4482628152931101638475376877122978262757424516848914208821952342987060546875",
    570,
    "3.2662448520856141582359969116197647314758219008736296797176525214401520867340405444019626"
    "396002597772252866318696883668291601700050188208650797413470758582440487935448498937916999"
    "276339414929811671405364973511700346204739516739766323473337369273998276351008632124033036"
    "908313637478833577190957366461784669906108011089145100688546861344607708883052424869986261"
    "801888046350637846415147864801751679914776550653014890427156042126514335770792811572060247"
    "315221399295249456990851382417328258781102544544826281529311016384753768771229782627574245"
    "16848914208821952342987060546875e-224"
  },
  {
    6.839580807204396e-224,
    794,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000683958080720439604379468886159131440228186845"
    "974804091394166048549028060480158919478176429734948862245893917947413915725902749084228620"
    "358340824715778220783690279558522472606442707272495233483210036215958136787138474704512521"
    "924672147437938676670341733412221965453163657716020881022933527140567274495010312512752887"
    "240094353007082555344371913060998677817406095665300076462642939779912060608663258697480177"
    "184536071352071608186568842594609317420574039197696219126111789580806886704377222403255162"
    "6944497564790705804008078262816775448607042875437400653026998043060302734375",
    570,
    "6.8395808072043960437946888615913144022818684597480409139416604854902806048015891947817642"
    "973494886224589391794741391572590274908422862035834082471577822078369027955852247260644270"
    "727249523348321003621595813678713847470451252192467214743793867667034173341222196545316365"
    "771602088102293352714056727449501031251275288724009435300708255534437191306099867781740609"
    "566530007646264293977991206060866325869748017718453607135207160818656884259460931742057403"
    "919769621912611178958080688670437722240325516269444975647907058040080782628167754486070428"
    "75437400653026998043060302734375e-224"
  },
  {
    1.1596240640257126e-223,
    792,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000001159624064025712602258076375880833395654491457"
    "569681287715268605797942742380617012463104314373535120241196943636624576522441056300581422"
    "419793065918182403765774576619664789514847825369394357789994302581511747355595089966426087"
    "501178358421511195473450370805053307045519754477962046904488712634815387123346504771003022"
    "944272275341196010053479196568052744594315357037044626160527110141276029627190805466957974"
    "108660494425525185133740063498266042385379033759626084965822995882878273767335675757022446"
    "49391501104268570722067296520471808840557770281520788557827472686767578125",
    569,
    "1.1596240640257126022580763758808333956544914575696812877152686057979427423806170124631043"
    "143735351202411969436366245765224410563005814224197930659181824037657745766196647895148478"
    "253693943577899943025815117473555950899664260875011783584215111954734503708050533070455197"
    "544779620469044887126348153871233465047710030229442722753411960100534791965680527445943153"
    "570370446261605271101412760296271908054669579741086604944255251851337400634982660423853790"
    "337596260849658229958828782737673356757570224464939150110426857072206729652047180884055777"
    "0281520788557827472686767578125e-223"
  },
  {
    2.137739829010208e-223,
    792,
    "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
    "000000000000000000000000000000000000000000002137739829010208164254710599823755493032966693"
    "365861876059496163944557351065295515739128097243708385367203985409989418142339845464744509"
    "848763411466950826902047362041530912923617136105382832462853751569708914889364657721411115"
    "003246578851078377498053405000389321952672989234337447900215597922146055742073876247357396"
    "257935852169042456718340668938648811212769473931510486147915190498056902726387493418137772"
    "399190753868947235717472056123782214877819185392685523254459125164136720192630371751914248"
    "32565863233869284296762895038863071250911929155336110852658748626708984375",
    569,
    "2.1377398290102081642547105998237554930329666933658618760594961639445573510652955157391280"
    "9724370838536720398540998941814233984546474450984876341146695082690