package Module::Format::Module;
$Module::Format::Module::VERSION = '0.2.0';
use warnings;
use strict;


sub _new
{
    my $class = shift;
    my $self = bless {}, $class;
    $self->_init(@_);
    return $self;
}

sub _components
{
    my $self = shift;

    if (@_)
    {
        $self->{_components} = shift;
    }
    return $self->{_components};
}

sub _init
{
    my ( $self, $args ) = @_;

    $self->_components( [ @{ $args->{_components} } ] );

    return;
}


sub from_components
{
    my ( $class, $args ) = @_;

    return $class->_new( { _components => [ @{ $args->{components} } ] } );
}


my $dash_re      = qr{(?:\w+-)*\w+};
my $colon_re     = qr{(?:\w+::)*\w+};
my $METACPAN_REL = 'https://metacpan.org/release/';

sub _gen_dash_format
{
    my ($args) = @_;

    my $prefix = $args->{prefix};
    my $name   = $args->{name};

    return +{
        name  => $name,
        regex => qr{\A\Q$prefix\E$dash_re\z},
        input => sub {
            my ( $class, $value ) = @_;

            if ( $value !~ s{\A\Q$prefix\E}{} )
            {
                die "$name value does not start with the '$prefix' prefix.";
            }

            return $class->_calc_components_from_string(
                { format => 'dash', value => $value } );
        },
        format => sub {
            my ($self) = @_;

            return $prefix . $self->format_as('dash');
        },
    };
}

sub _gen_colon_format
{
    my ($args) = @_;

    my $prefix = $args->{prefix};
    my $suffix = $args->{suffix};
    my $name   = $args->{name};

    return +{
        name  => $name,
        regex => qr{\A\Q$prefix\E$colon_re\Q$suffix\E\z},
        input => sub {
            my ( $class, $value ) = @_;

            if ( $value !~ m{\A\Q$prefix\E((?:\w+::)*\w+)\Q$suffix\E\z} )
            {
                die "Improper value for $name";
            }

            return $class->_calc_components_from_string(
                { format => 'colon', value => $1 } );
        },
        format => sub {
            my ($self) = @_;

            return $prefix . $self->format_as('colon') . $suffix;
        },
    };
}

my @formats_by_priority = (
    _gen_dash_format(
        {
            name   => 'rpm_dash',
            prefix => 'perl-',
        }
    ),
    _gen_colon_format(
        {
            name   => 'rpm_colon',
            prefix => 'perl(',
            suffix => ')',
        }
    ),
    {
        name  => 'colon',
        regex => qr{\A$colon_re\z},
        input => sub {
            my ( $class, $value ) = @_;
            return [ split( /::/, $value, -1 ) ];
        },
        format => sub {
            my ($self) = @_;

            return join( '::', @{ $self->_components() } );
        },
    },
    {
        name  => 'dash',
        regex => qr{\A$dash_re\z},
        input => sub {
            my ( $class, $value ) = @_;
            return [ split( /-/, $value, -1 ) ];
        },
        format => sub {
            my ($self) = @_;

            return join( '-', @{ $self->_components() } );
        },
    },
    {
        name  => 'unix',
        regex => qr{\A(?:\w+/)*\w+\.pm\z},
        input => sub {
            my ( $class, $value ) = @_;

            if ( $value !~ s{\.pm\z}{} )
            {
                die "Cannot find a .pm suffix in the 'unix' format.";
            }

            return [ split( m{/}, $value, -1 ) ];
        },
        format => sub {
            my ($self) = @_;

            return join( '/', @{ $self->_components() } ) . '.pm';
        },
    },
    _gen_dash_format(
        {
            name   => 'metacpan_rel',
            prefix => $METACPAN_REL,
        }
    ),
    _gen_colon_format(
        {
            name   => 'metacpan_pod',
            prefix => 'https://metacpan.org/pod/',
            suffix => '',
        }
    ),
    {
        name   => 'debian',
        format => sub {
            my ($self) = @_;

            return 'lib' . lc( $self->format_as('dash') ) . '-perl';
        },
    },
);

my %formats = ( map { $_->{name} => $_ } @formats_by_priority );

sub _calc_components_from_string
{
    my ( $class, $args ) = @_;

    my $format = $args->{format};
    my $value  = $args->{value};

    if ( exists( $formats{$format} ) )
    {
        if ( my $handler = $formats{$format}->{'input'} )
        {
            return $class->$handler($value);
        }
        else
        {
            die "Format $format is output-only!";
        }
    }
    else
    {
        die "Unknown format '$format'!";
    }
}

sub from
{
    my ( $class, $args ) = @_;

    my $format = $args->{format};
    my $value  = $args->{value};

    return $class->_new(
        {
            _components => $class->_calc_components_from_string($args)
        }
    );
}


sub get_components_list
{
    my $self = shift;

    return [ @{ $self->_components() } ];
}


sub format_as
{
    my ( $self, $format ) = @_;

    if ( exists( $formats{$format} ) )
    {
        my $handler = $formats{$format}->{'format'};
        return $self->$handler();
    }
    else
    {
        die "Unknown format '$format';";
    }

    return;
}


sub clone
{
    my $self = shift;

    return
        ref($self)
        ->from_components( { components => $self->get_components_list() } );
}


sub _all
{
    my ( $cb, $l ) = @_;

    foreach (@$l)
    {
        if ( not $cb->($_) )
        {
            return;
        }
    }

    return 1;
}

sub is_sane
{
    my $self = shift;

    return _all( sub { m{\A\w+\z}; }, $self->_components() );
}


sub from_guess
{
    my ( $class, $args ) = @_;

    my $dummy_format_string;

    my $string = $args->{value};
    my $out_format_ref = ( $args->{format_ref} || ( \$dummy_format_string ) );

    # TODO : After the previous line the indentation in vim is reset to the
    # first column.

    foreach my $format_record (@formats_by_priority)
    {
        if ( my $regex = $format_record->{regex} )
        {
            if ( $string =~ $regex )
            {
                my $format_id = $format_record->{name};

                ${$out_format_ref} = $format_id;

                return $class->from(
                    { value => $string, format => $format_id, } );
            }
        }
    }

    die "Could not guess the format of the value '$string'!";
}


1;    # End of Module::Format::Module

__END__

=pod

=encoding UTF-8

=head1 NAME

Module::Format::Module - encapsulates a single Perl module.

=head1 VERSION

version 0.2.0

=head1 SYNOPSIS

    use Module::Format::Module;

    my $module = Module::Format::Module->from(
        {
            format => 'colon',
            value => 'XML::RSS',
        }
        );

    my $module = Module::Format::Module->from(
        {
            format => 'dash',
            value => 'XML-RSS',
        }
        );

    my $module = Module::Format::Module->from(
        {
            format => 'unix',
            value => 'XML/RSS.pm',
        }
        );

    my $module = Module::Format::Module->from(
        {
            format => 'rpm_dash',
            value => 'perl-XML-RSS',
        }
        );

    my $module = Module::Format::Module->from(
        {
            format => 'rpm_colon',
            value => 'perl(XML::RSS)',
        }
        );

    my $module = Module::Format::Module->from_guess(
        {
            value => # All of the above...
        }
    );

    # Prints "XML-RSS"
    print $module->format_as('dash'), "\n";

    # Prints "perl(XML::RSS)"
    print $module->format_as('rpm_colon'), "\n";

    # Prints "libxml-rss-perl"
    print $module->format_as('debian'), "\n";

=head1 VERSION

version 0.2.0

=head1 FUNCTIONS

=head2 my $module = Module::Format::Module->from_components({components => [@components]})

Constructs a module from @components which are strings that indicate the
individual components of the module. For example:

    my $module = Module::Format::Module->from_components(
        {
            components => [qw(XML Grammar Fiction)],
        }
    );

=head2 my $module = Module::Format::Module->from({format => $format, value => $string});

Construct a Module::Format::Module object from the given format $format
and the string value $string .

Valid formats are:

=over 4

=item * 'colon'

Separated by a double colon, e.g: C<XML::RSS>, C<Data::Dumper>,
C<Catalyst::Plugin::Model::DBIx::Class>, etc.

=item * 'dash'

Separated by a double colon, e.g: C<XML-RSS>, C<Data-Dumper>,
C<Catalyst-Plugin-Model-DBIx-Class>, etc.

=item * 'unix'

UNIX path, e.g: C<XML/RSS.pm>, C<Catalyst/Plugin/Model/DBIx/Class.pm> .

This is commonly displayed in Perl error messages.

=item * 'rpm_dash'

Like dash, only with C<'perl-'> prefixed: C<perl-XML-RSS>, C<perl-Data-Dumper>.

=item * 'rpm_colon'

Like colon only wrapped inside C<perl(...)> - useful for rpm provides for
individual modules. E.g: C<perl(XML::RSS)>,
C<perl(Catalyst::Plugin::Authentication)> .

=item * 'metacpan_rel'

A MetaCPAN release page URL, e.g:
C<https://metacpan.org/release/Text-Sprintf-Named> , C<https://metacpan.org/release/Class-XSAccessor> .

=item * 'metacpan_pod'

A MetaCPAN POD page URL, e.g:
C<https://metacpan.org/pod/Module::Format> , C<https://metacpan.org/pod/Data::ParseBinary> .

=item * 'debian'

Debian package format, such as C<libxml-rss-perl>,
C<libcatalyst-plugin-authentication-perl> . Output only.

=back

=head2 my $array_ref = $module->get_components_list()

Returns the components list of the module as an array reference. For
example for the module C<One::Two::Three> it would be
C<["One", "Two", "Three"]>.

=head2 $module->format_as($format)

Format the module in the given format. See from() for a list.

=head2 my $clone = $module->clone()

Duplicates the object.

=head2 my $bool = $module->is_sane();

Returns a boolean depending on if the component of the module do not contain
any funny character (only alphanumeric characters and underscore.).

=head2 my $module = Module::Format::Module->from_guess({ value => $string});

Initialises a module object from a string while trying to guess its format.
It accepts a hash-ref with the following keys: C<'value'> that points to the
string to serve as an input, and an optional C<'format_ref'> that will give
the format that was decided upon.

=head1 AUTHOR

Shlomi Fish, L<http://www.shlomifish.org/> .

=head1 BUGS

Please report any bugs or feature requests to C<bug-module-format at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Module-Format>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Module::Format::Module

You can also look for information at:

=over 4

=item * MetaCPAN

L<https://metacpan.org/release/Module-Format>

=item * RT: CPAN's request tracker

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Module-Format>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Module-Format>

=back

=head1 ACKNOWLEDGEMENTS

=head1 COPYRIGHT & LICENSE

Copyright 2010 Shlomi Fish.

This program is distributed under the MIT (X11) License:
L<http://www.opensource.org/licenses/mit-license.php>

Permission is hereby granted, free of charge, to any person
obtaining a copy of this software and associated documentation
files (the "Software"), to deal in the Software without
restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following
conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

=head1 AUTHOR

Shlomi Fish <shlomif@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2018 by Shlomi Fish.

This is free software, licensed under:

  The MIT (X11) License

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
L<https://github.com/shlomif/perl-Module-Format/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=for :stopwords cpan testmatrix url annocpan anno bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Perldoc

You can find documentation for this module with the perldoc command.

  perldoc Module::Format::Module

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<https://metacpan.org/release/Module-Format>

=item *

Search CPAN

The default CPAN search engine, useful to view POD in HTML format.

L<http://search.cpan.org/dist/Module-Format>

=item *

RT: CPAN's Bug Tracker

The RT ( Request Tracker ) website is the default bug/issue tracking system for CPAN.

L<https://rt.cpan.org/Public/Dist/Display.html?Name=Module-Format>

=item *

AnnoCPAN

The AnnoCPAN is a website that allows community annotations of Perl module documentation.

L<http://annocpan.org/dist/Module-Format>

=item *

CPAN Ratings

The CPAN Ratings is a website that allows community ratings and reviews of Perl modules.

L<http://cpanratings.perl.org/d/Module-Format>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/Module-Format>

=item *

CPAN Testers

The CPAN Testers is a network of smoke testers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/M/Module-Format>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=Module-Format>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=Module::Format>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests by email to C<bug-module-format at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/Public/Bug/Report.html?Queue=Module-Format>. You will be automatically notified of any
progress on the request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/shlomif/perl-Module-Format>

  git clone git://github.com/shlomif/perl-Module-Format.git

=cut
