% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generics.R
\name{predict.bsim}
\alias{predict.bsim}
\title{Prediction Method for \code{BayesSIM}}
\usage{
\method{predict}{bsim}(
  object,
  newdata = NULL,
  se.fit = FALSE,
  type = c("response", "latent", "index"),
  method = c("mean", "median"),
  interval = c("none", "credible"),
  level = 0.95,
  ...
)
}
\arguments{
\item{object}{A fitted object of \code{BayesSIM} or individual model.}

\item{newdata}{Optional data frame for which predictions should be made.
If \code{NULL}, predictions are returned for the training data.}

\item{se.fit}{A logical value indicating whether standard errors are required.
Default is \code{FALSE}.}

\item{type}{Character string specifying the type of prediction. \code{"response"}
is the default.
\describe{
\item{\code{"response"}}{
Posterior predictive summaries of the response \eqn{Y}.
This corresponds to draws from the posterior predictive distribution
\eqn{Y^{(m)} \sim N(f(X'\theta^{(m)}), \sigma^{2(m)})}
and therefore incorporates both the uncertainty in the link function
and the variability of the error term for each \eqn{m^{th}} MCMC sample.}
\item{\code{"latent"}}{
Posterior summaries of the latent mean structure
\eqn{E(Y \mid X) = f^{(m)}(t^{(m)})}, where \eqn{t^{(m)} = X'\theta^{(m)}}.
Unlike \code{"response"}, it excludes the noise term and calculated by
\eqn{f^{(m)}(X'\theta^{(m)})} for each \eqn{m^{th}} MCMC sample of
\eqn{\theta}.}
\item{\code{"index"}}{
Posterior summaries of the single index \eqn{t^{(m)} = X'\theta^{(m)}}.}
}}

\item{method}{Character string determining the posterior summary used for
point predictions. Options are \code{"mean"} or \code{"median"}. Default is \code{"mean"}.}

\item{interval}{Character string indicating whether a credible interval should be returned. Default is \code{"none"}.
\describe{
\item{\code{"none"}}{Return only point predictions.}
\item{\code{"credible"}}{Return a \eqn{100 \times \text{level}\%} credible interval.}
}}

\item{level}{Numeric value between 0 and 1 specifying the credible level.
\code{level = 0.95} yields a 95\% credible interval. Default is \code{0.95}.}

\item{...}{Additional arguments.}
}
\value{
A list containing at least the following components:
\describe{
\item{\code{fitted}}{Posterior mean or median predictions. If \code{se.fit = TRUE} or
\code{interval = "credible"}, standard error and lower, upper bounds of the credible interval is also included.}
\item{\code{truey}}{True response value of test data. When true response value is not available, \code{NULL} is saved.}
\item{\code{idxValue}}{Linear index value is saved where \eqn{\theta} is estimated by \code{method}.}
\item{\code{level}}{Credible level.}
}
}
\description{
Generate predictions from a fitted Bayesian single-index model.
}
\details{
This method extracts MCMC posterior samples stored in a BayesSIM object
and computes posterior summaries of:
\itemize{
\item the \strong{posterior predictive response} \eqn{Y \mid X} (type \code{"response"}),
\item the \strong{latent link function} evaluation \eqn{E(Y \mid X) = f(X'\theta)} (type \code{"latent"}), or
\item the \strong{single index} \eqn{X'\theta} (type \code{"index"}).
}

The key distinction is that \code{"response"} incorporates the posterior
variability of the error term \eqn{\epsilon}, whereas \code{"latent"} represents
the noiseless conditional expectation \eqn{E(Y \mid X)} computed directly
from the link function and the posterior draws of \eqn{\theta}.

When \code{interval = "credible"}, the returned object includes lower and upper
credible bounds computed via posterior quantiles for the chosen prediction
scale.

If \code{newdata} is supplied, predictions are evaluated at the new covariate
values by computing the corresponding posterior index \eqn{t = X'\theta}
and applying the link function.
}
\examples{
\donttest{
simdata2 <- data.frame(DATA1$X, y = DATA1$y)

# 1. One tool version
fit_one <- BayesSIM(y ~ ., data = simdata2,
                    niter = 5000, nburnin = 1000, nchain = 1)

# Check median index vector estimates with standard errors
coef(fit_one, method = "median", se = TRUE)

# Fitted index values of median prediction
fitted(fit_one, type = "linpred", method = "median")

# Residuals of median prediction
residuals(fit_one, method = "median")

# Summary of the model
summary(fit_one)

# Convergence diagnostics
nimTraceplot(fit_one)

# Goodness of fit
GOF(fit_one)

# Fitted plot
plot(fit_one)

# Prediction with 95\% credible interval at new data
newx <- data.frame(X1 = rnorm(3), X2 = rnorm(3), X3 = rnorm(3), X4 = rnorm(3))
pred <- predict(fit_one, newdata = newx, interval = "credible", level = 0.95)
plot(pred)


# 2. Split version
models <- BayesSIM_setup(y ~ ., data = simdata2)
Ccompile <- compileModelAndMCMC(models)
nimSampler <- get_sampler(Ccompile)
initList <- getInit(models)
mcmc.out <- runMCMC(nimSampler, niter = 5000, nburnin = 1000, thin = 1,
                    nchains = 1, setSeed = TRUE, inits = initList,
                    summary = TRUE, samplesAsCodaMCMC = TRUE)

# "fit_split" becomes exactly the same as the class of "fit_one" object and apply generic functions.
fit_split <- as_bsim(models, mcmc.out)

}
}
