\name{pdbart}
\title{Partial Dependence Plots for BART}
\alias{pdbart}
\alias{plot.pdbart}
\alias{pd2bart}
\alias{plot.pd2bart}
\description{
   Run \code{bart} at test observations constructed so that
   a  plot can be created
   displaying the effect of
   a single variable (\code{pdbart}) or pair of variables (\code{pd2bart}).
   Note the y is a binary with \eqn{P(Y=1 | x) =F(f(x))} with \eqn{F} the standard
   normal cdf, then the plots are all on the \eqn{f} scale.
}
\usage{
   pdbart(
      x.train, y.train,
      xind=1:ncol(x.train), levs=NULL, levquants=c(.05,(1:9)/10,.95),
      pl=TRUE,  plquants=c(.05,.95), \dots)
   \method{plot}{pdbart}(
      x,
      xind = 1:length(x$fd),
      plquants =c(.05,.95),cols=c('black','blue'), \dots)
   pd2bart(
      x.train, y.train,
      xind=1:2, levs=NULL, levquants=c(.05,(1:9)/10,.95),
      pl=TRUE, plquants=c(.05,.95), \dots)
   \method{plot}{pd2bart}(
      x,
      plquants =c(.05,.95), contour.color='white',
      justmedian=TRUE, \dots)
}
\arguments{
   \item{x.train}{
   Explanatory variables for training (in sample) data.\cr
   Must be a matrix (typeof double)
   with (as usual) rows corresponding to observations and columns to variables.\cr
   Note that for a categorical variable you need to use dummies and if there
   are more than two categories, you need to put all the dummies in
   (unlike linear regression).
   }

   \item{y.train}{
   Dependent variable for training (in sample) data.\cr
   Must be a vector (typeof double) with length equal to the number of observations
   (equal to the number of rows of x.train).
   }

   \item{xind}{
   Integer vector indicating which variables are to be plotted.\cr
   In \code{pdbart}, variables (columns of x.train) for which plot is to be constructed.\cr
   In \code{plot.pdbart}, indices in list returned by \code{pdbart} for which plot is to be constructed.\cr
   In \code{pd2bart}, integer vector of length 2,
   indicating the pair of variables (columns of x.train) to plot.
   }

   \item{levs}{
   Gives the values of a variable at which the plot is to be constructed.\cr
   List, where
   \eqn{i^{th}}{i\^th} component gives the values for \eqn{i^{th}}{i\^th} variable.\cr
   In \code{pdbart}, should have same length as xind.\cr
   In \code{pd2bart}, should have length 2.\cr
   See also argument levquants.
   }

   \item{levquants}{
   If levs in NULL, the values of each variable used in the plot is
   set to the quantiles (in x.train) indicated by levquants.\cr
   Double vector.
   }

   \item{pl}{
   For \code{pdbart} and \code{pd2bart}, if true, plot is made (by calling plot.*).
   }

   \item{plquants}{
   In the plots, beliefs about \eqn{f(x)} are indicated by plotting the
   posterior median and a lower and upper quantile.
   plquants is a double vector of length two giving the lower and upper quantiles.
   }

   \item{\dots}{
   Additional arguments.\cr
   In \code{pdbart},\code{pd2bart}, passed on to \code{\link{bart}}.\cr
   In \code{plot.pdbart}, passed on to \code{\link{plot}}.\cr
   In \code{plot.pd2bart}, passed on to \code{\link{image}}
   }

   \item{x}{
   For plot.*, object returned from pdbart or pd2bart.
   }

   \item{cols}{
   Vector of two colors.\cr
   First color is for median of \eqn{f}, second color is for the upper and lower quantiles.
   }

   \item{contour.color}{
   Color for contours plotted on top of the image.
   }

   \item{justmedian}{
   Boolean, if true just one plot is created for
   the median of \eqn{f(x)} draws.  If false, three plots are created
   one for the median and two additional ones for the lower and upper quantiles.
   In this case, mfrow is set to c(1,3).
   }
}
\details{
   We divide the predictor vector \eqn{x} into a subgroup of interest,
   \eqn{x_s} and the complement \eqn{x_c=x\setminus x_s}{x_c = x - x_s}.
   A prediction \eqn{f(x)} can
   then be written as \eqn{f(x_s,x_c)}. To estimate the effect of \eqn{x_s}
   on the prediction, Friedman suggests the partial dependence
   function
   \deqn{
   f_s(x_s) = \frac{1}{n}\sum_{i=1}^n f(x_s,x_{ic})
   }{
   f_s(x_s) = (1/n) \sum_{i=1}\^n f(x_s,x_{ic})
   }
   where \eqn{x_{ic}} is the \eqn{i^{th}}{i\^th} observation of \eqn{x_c} in the data. Note
   that \eqn{(x_s,x_{ic})} will generally not be one of the observed data
   points. Using BART it is straightforward to then estimate and even
   obtain uncertainty bounds for \eqn{f_s(x_s)}.  A draw of \eqn{f^*_s(x_s)}{f*_s(x_s)}
   from the induced BART posterior on \eqn{f_s(x_s)} is obtained by
   simply computing \eqn{f^*_s(x_s)}{f*_s(x_s)} as a byproduct of each MCMC draw
   \eqn{f^*}{f*}. The median (or average)
   of these MCMC draws \eqn{f^*_s(x_s)}{f*_s(x_s)} then yields an
   estimate of \eqn{f_s(x_s)}, and lower and upper quantiles can be used
   to obtain intervals for \eqn{f_s(x_s)}.

   In \code{pdbart} \eqn{x_s} consists of a single variable in \eqn{x} and in
   \code{pd2bart} it is a pair of variables.

   This is a computationally intensive procedure.
   For example, in \code{pdbart}, to compute the partial dependence plot
   for 5 \eqn{x_s} values, we need
   to compute \eqn{f(x_s,x_c)} for all possible \eqn{(x_s,x_{ic})} and there
   would be \eqn{5n} of these where \eqn{n} is the sample size.
   All of that computation would be done for each kept BART draw.
   For this reason running BART with keepevery larger than 1 (eg. 10)
   makes the procedure much faster.
}
\value{
   The plot methods produce the plots and don't return anything.

   \code{pdbart} and \code{pd2bart} return lists with components
   given below.  The list returned by \code{pdbart} is assigned class
   \sQuote{pdbart} and the list returned by \code{pd2bart} is assigned
   class \sQuote{pd2bart}.

   \item{fd}{
   A matrix whose \eqn{(i,j)} value is the \eqn{i^{th}}{i\^th}
   draw of \eqn{f_s(x_s)} for the \eqn{j^{th}}{j\^th} value of \eqn{x_s}.
   \dQuote{fd} is for \dQuote{function draws}.

   For \code{pdbart} fd is actually a list whose
   \eqn{k^{th}}{k\^th} component is the matrix described above
   corresponding to the \eqn{k^{th}}{k\^th} variable chosen by argument xind.\cr
   The number of columns in each matrix will equal the number of values
   given in the corresponding component of argument levs (or number of values in levquants).

   For \code{pd2bart}, fd is a single matrix.
   The columns correspond to all possible pairs of values for the pair
   of variables indicated by xind.
   That is, all possible \eqn{(x_i,x_j)} where \eqn{x_i} is a value in
   the levs component corresponding to the first \eqn{x} and
   \eqn{x_j} is a value in the levs components corresponding to the second one.\cr
   The first \eqn{x} changes first.
   }

   \item{levs}{
   The list of levels used, each component corresponding to a variable.\cr
   If argument levs was supplied it is unchanged.\cr
   Otherwise, the levels in levs are as constructed using argument levquants.
   }

   \item{xlbs}{
   vector of character strings which are the plotting labels used for the variables.
   }
   
   The remaining components returned in the list are the same as in the value of \code{\link{bart}}.
   They are simply passed on from the BART run used to create the partial dependence plot.
   The function \code{\link{plot.bart}} can be applied to the object returned by \code{pdbart} or
   \code{pd2bart} to examine the BART run.
}
\references{
Chipman, H., George, E., and McCulloch R. (2010)
   Bayesian Additive Regression Trees.
   \emph{The Annals of Applied Statistics}, \bold{4,1}, 266-298.
}
\author{
Hugh Chipman: \email{hugh.chipman@gmail.com}.\cr
Robert McCulloch: \email{robert.e.mcculloch@gmail.com}.
}
\examples{
##simulate data 
f = function(x) { return(.5*x[,1] + 2*x[,2]*x[,3]) }
sigma=.2 # y = f(x) + sigma*z
n=100 #number of observations
set.seed(27)
x = matrix(2*runif(n*3)-1,ncol=3) ; colnames(x) = c('rob','hugh','ed')
Ey = f(x)
y = Ey +  sigma*rnorm(n)
lmFit = lm(y~.,data.frame(x,y)) #compare lm fit to BART later
par(mfrow=c(1,3)) #first two for pdbart, third for pd2bart
##pdbart: one dimensional partial dependence plot
set.seed(99)
pdb1 = pdbart(x,y,xind=c(1,2),
   levs=list(seq(-1,1,.2),seq(-1,1,.2)),pl=FALSE,
   keepevery=10,ntree=100,nskip=100,ndpost=200) #should run longer!
plot(pdb1,ylim=c(-.6,.6))
##pd2bart: two dimensional partial dependence plot
set.seed(99)
pdb2 = pd2bart(x,y,xind=c(2,3),
   levquants=c(.05,.1,.25,.5,.75,.9,.95),pl=FALSE,
   ntree=100,keepevery=10,verbose=FALSE,nskip=100,ndpost=200) #should run longer!
plot(pdb2)
##compare BART fit to linear model and truth = Ey
fitmat = cbind(y,Ey,lmFit$fitted,pdb1$yhat.train.mean)
colnames(fitmat) = c('y','Ey','lm','bart')
print(cor(fitmat))
## plot.bart(pdb1) displays the BART run used to get the plot.
}
\keyword{nonparametric}
\keyword{tree}
\keyword{regression}
\keyword{nonlinear}
\keyword{dplot}
