% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimationCKT.classif.tree.R
\name{CKT.fit.tree}
\alias{CKT.fit.tree}
\alias{CKT.predict.tree}
\title{Estimation of conditional Kendall's taus using a classification tree}
\usage{
CKT.fit.tree(datasetPairs, mindev = 0.008, mincut = 0)

CKT.predict.tree(fit, newZ)
}
\arguments{
\item{datasetPairs}{the matrix of pairs and corresponding values of the kernel
as provided by \code{\link{datasetPairs}}.}

\item{mindev}{a factor giving the minimum deviation for a node to be splitted.
See \code{tree::\link[tree]{tree.control}()} for more details.}

\item{mincut}{the minimum number of observations (of pairs) in a node
See \code{tree::\link[tree]{tree.control}()} for more details.}

\item{fit}{result of a call to \code{CKT.fit.tree}}

\item{newZ}{new matrix of observations, with the same number of variables.
and same names as the \code{designMatrix} that was used to fit the tree.}
}
\value{
\code{CKT.fit.tree} returns the fitted tree.

\code{CKT.predict.tree} returns
a vector of (predicted) conditional Kendall's taus of the same size
as the number of rows of \code{newZ}.
}
\description{
Let \eqn{X_1} and \eqn{X_2} be two random variables.
The goal of this function is to estimate the conditional Kendall's tau
(a dependence measure) between \eqn{X_1} and \eqn{X_2} given \eqn{Z=z}
for a conditioning variable \eqn{Z}.
Conditional Kendall's tau between \eqn{X_1} and \eqn{X_2} given \eqn{Z=z}
is defined as:
\deqn{P( (X_{1,1} - X_{2,1})(X_{1,2} - X_{2,2}) > 0 | Z_1 = Z_2 = z)}
\deqn{- P( (X_{1,1} - X_{2,1})(X_{1,2} - X_{2,2}) < 0 | Z_1 = Z_2 = z),}
where \eqn{(X_{1,1}, X_{1,2}, Z_1)} and \eqn{(X_{2,1}, X_{2,2}, Z_2)}
are two independent and identically distributed copies of \eqn{(X_1, X_2, Z)}.
In other words, conditional Kendall's tau is the difference
between the probabilities of observing concordant and discordant pairs
from the conditional law of \deqn{(X_1, X_2) | Z=z.}
These functions estimate and predict conditional Kendall's tau using a
\strong{classification tree}. This is possible by the relationship between
estimation of conditional Kendall's tau and classification problems
(see Derumigny and Fermanian (2019)): estimation of conditional Kendall's tau
is equivalent to the prediction of concordance in the space of pairs
of observations.
}
\examples{
# We simulate from a conditional copula
set.seed(1)
N = 800
Z = rnorm(n = N, mean = 5, sd = 2)
conditionalTau = -0.9 + 1.8 * pnorm(Z, mean = 5, sd = 2)
simCopula = VineCopula::BiCopSim(N=N , family = 1,
    par = VineCopula::BiCopTau2Par(1 , conditionalTau ))
X1 = qnorm(simCopula[,1])
X2 = qnorm(simCopula[,2])

datasetP = datasetPairs(X1 = X1, X2 = X2, Z = Z, h = 0.07, cut = 0.9)
est_Tree = CKT.fit.tree(datasetPairs = datasetP, mindev = 0.008)
print(est_Tree)

newZ = seq(1,10,by = 0.1)
prediction = CKT.predict.tree(fit = est_Tree, newZ = data.frame(x=newZ))
# Comparison between true Kendall's tau (in red)
# and estimated Kendall's tau (in black)
plot(newZ, prediction, type = "l", ylim = c(-1,1))
lines(newZ, -0.9 + 1.8 * pnorm(newZ, mean = 5, sd = 2), col="red")

}
\references{
Derumigny, A., & Fermanian, J. D. (2019).
A classification point-of-view about conditional Kendall’s tau.
Computational Statistics & Data Analysis, 135, 70-94.
(Section 3.2)
\doi{10.1016/j.csda.2019.01.013}
}
\seealso{
See also other estimators of conditional Kendall's tau:
\code{\link{CKT.fit.nNets}}, \code{\link{CKT.fit.randomForest}},
\code{\link{CKT.fit.GLM}}, \code{\link{CKT.predict.kNN}},
\code{\link{CKT.kernel}}, \code{\link{CKT.kendallReg.fit}},
and the more general wrapper \code{\link{CKT.estimate}}.
}
