# LogisGompertz.R
#' Logistic-Gompertz Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic-Gompertz distribution.
#'
#' The Logistic-Gompertz distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic-Gompertz distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1-\frac{1}{1+\left[\exp \left\{\frac{\lambda}{\beta}
#' \left(e^{\beta x}-1\right)\right\}-1\right]^\alpha}  \, ; x \geq 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dlogis.gpz()} — Density function
#'   \item \code{plogis.gpz()} — Distribution function
#'   \item \code{qlogis.gpz()} — Quantile function
#'   \item \code{rlogis.gpz()} — Random generation
#'   \item \code{hlogis.gpz()} — Hazard function
#' }
#'
#' @usage
#' dlogis.gpz(x, alpha, beta, lambda, log = FALSE)
#' plogis.gpz(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.gpz(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.gpz(n, alpha, beta, lambda)
#' hlogis.gpz(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.gpz}: numeric vector of (log-)densities
#'   \item \code{plogis.gpz}: numeric vector of probabilities
#'   \item \code{qlogis.gpz}: numeric vector of quantiles
#'   \item \code{rlogis.gpz}: numeric vector of random variates
#'   \item \code{hlogis.gpz}: numeric vector of hazard values
#' }
#'
#' @references 
#' Joshi, R. K.,  & Kumar, V. (2020). 
#' The Logistic Gompertz Distribution with Properties and Applications. 
#' \emph{Bull. Math. & Stat. Res.},  \bold{8(4)}, 81--94.     
#'  
#' @examples
#' x <- seq(0.1, 2.0, 0.2)
#' dlogis.gpz(x, 2.0, 1.5, 0.1)
#' plogis.gpz(x, 2.0, 1.5, 0.1)
#' qlogis.gpz(0.5, 2.0, 1.5, 0.1)
#' rlogis.gpz(10, 2.0, 1.5, 0.1)
#' hlogis.gpz(x, 2.0, 1.5, 0.1)
#' 
#' # Data
#' x <- stress
#' # ML estimates
#' params = list(alpha=2.09377, beta=0.30392, lambda=0.17763)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.gpz, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.gpz, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlogis.gpz, pfun=plogis.gpz, plot=TRUE)
#' print.gofic(out)
#' @name LogisGompertz
#' @aliases dlogis.gpz plogis.gpz qlogis.gpz rlogis.gpz hlogis.gpz
#' @export
NULL

#' @rdname LogisGompertz
#' @usage NULL
#' @export   
dlogis.gpz <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	z <- exp(x[valid] *beta)
    u <- exp((lambda/beta) * (z-1))
    num <- alpha *lambda * z * u * ((u-1)^(alpha - 1.0))
    deno <- (1.0 + (u-1) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
	}
    if (log) 
       pdf<- log(pdf)
    return(pdf)   
}

#' @rdname LogisGompertz
#' @usage NULL
#' @export   
plogis.gpz <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {	
    u <- exp((lambda/beta) * (exp(q[valid] * beta) -1))
	tmp <- 1 + ((u - 1) ^ alpha)
    cdf[valid] <- 1 - (1/tmp)
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LogisGompertz
#' @usage NULL
#' @export 
qlogis.gpz <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	qtl[valid] <- (1/beta)*log((beta/lambda) *log(1 + ((p[valid]/(1- p[valid])) ^ (1/alpha))) + 1)
    }   
	
    return(qtl)   
}

#' @rdname LogisGompertz
#' @usage NULL
#' @export 
rlogis.gpz <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qlogis.gpz(u, alpha, beta, lambda)
    return(q)
}

#' @rdname LogisGompertz
#' @usage NULL
#' @export 
hlogis.gpz <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
     
    nume <- dlogis.gpz(x, alpha, beta, lambda)
	surv <- 1.0 - plogis.gpz(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 


