% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/upAndDownSearch.R
\name{upAndDownSearch}
\alias{upAndDownSearch}
\alias{stochBlockForUDS}
\title{Perform Up-and-Down Search Optimization}
\usage{
upAndDownSearch(
  data,
  initPart,
  initFit,
  optimFun,
  nRep = 100,
  minPmove = 0.2,
  pFitSumTreshMoveBest = 5,
  ...
)

stochBlockForUDS(data, initPart, ...)
}
\arguments{
\item{data}{A data object (e.g., data frame or matrix) on which the search is performed.}

\item{initPart}{Initial partition or configuration to start the search from. It can be a vector for simple datasets and a list of vectors for temporal or linked networks}

\item{initFit}{Initial fitness value associated with the initial partition.}

\item{optimFun}{A function used for optimization. A function must accept a data and a initial partition and return a list with an element \code{part} holding the final partition and an element \code{fit} fitness value.}

\item{nRep}{Integer. The number of repetitions for the search algorithm. Defaults to 100.}

\item{minPmove}{Numeric. The minimum probability of moving to a new partition when the fitness does not improve. Defaults to 0.2.}

\item{pFitSumTreshMoveBest}{Numeric. The threshold for the sum of 1 - probabilities of moving to a new partition before reverting to the best partition. Defaults to 5.}

\item{...}{Additional paramters to optimFun.}
}
\value{
A list with the following components:
\describe{
  \item{data}{The input data.}
  \item{finalPart}{The final partition obtained.}
  \item{finalFit}{The final fitness value after the search.}
  \item{searchHistory}{A list containing the history of partitions and fitness values during the search.}
  \item{callUsed}{The call used to invoke the function, capturing the parameters passed.}
  \item{initial.param}{A list of initial parameters used in the function call withiut the data.}
}
}
\description{
The \code{upAndDownSearch} function performs an up-and-down search (increasing and lowering the number of clusters) on a given dataset, starting from an initial partition and fitness value. The function assumes that the higher fitness value is better. Experimental! The \code{stochBlockForUDS} is just a wrapper to \code{\link{stochBlock}} function to be used within \code{upAndDownSearch} function.
}
\examples{
# Create a synthetic network matrix
set.seed(2022)
library(blockmodeling)
k<-2 # number of blocks to generate
blockSizes<-rep(20,k)
IM<-matrix(c(0.8,.4,0.2,0.8), nrow=2)
clu<-rep(1:k, times=blockSizes)
n<-length(clu)
M<-matrix(rbinom(n*n,1,IM[clu,clu]),ncol=n, nrow=n)
initClu<-rep(1, times=n)
initFit<-ICLStochBlock(M, initClu) # Initial fitness value
# Using up-and-down search to optimise the partition
res<-upAndDownSearch(data=M,initPart=initClu, initFit=initFit, optimFun=stochBlockForUDS, nRep=10) 
plotMat(res$data, clu=res$bestPart) # Have a look at the optimised parition
print(res$bestFit) # Print the final fitness value

# Create a synthetic linked-network matrix
set.seed(2022)
library(blockmodeling)
IM<-matrix(c(0.9,.5,0.1,0.8), nrow=2)
clu<-rep(1:2, each=20) # Partition to generate
n<-length(clu)
nClu<-length(unique(clu)) # Number of clusters to generate
M1<-matrix(rbinom(n^2,1,IM[clu,clu]),ncol=n, nrow=n) # First network
M2<-matrix(rbinom(n^2,1,IM[clu,clu]),ncol=n, nrow=n) # Second network
M12<-diag(n) # Linking network
nn<-c(n,n)
k<-c(2,2)
Ml<-matrix(0, nrow=sum(nn),ncol=sum(nn)) 
Ml[1:n,1:n]<-M1
Ml[n+1:n,n+1:n]<-M2
Ml[n+1:n, 1:n]<-M12 
plotMat(Ml) # Linked network
clu1<-rep(1, n)
clu2<-rep(2, n)
initClu<-list(clu1, clu2)
initFit<-ICLStochBlock(Ml, initClu) # Initial fitness value
# Using up-and-down search to optimise the partition
res<-upAndDownSearch(data=Ml,initPart=initClu, initFit=initFit, optimFun=stochBlockForUDS, nRep=10)
plotMat(res$data, clu=res$bestPart) # Have a look at the optimised parition
print(res$bestFit) # Print the final fitness value
}
