% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DupuyGalichon.R
\name{estimate.affinity.matrix}
\alias{estimate.affinity.matrix}
\title{Estimate Dupuy and Galichon's model}
\usage{
estimate.affinity.matrix(
  X,
  Y,
  w = rep(1, N),
  A0 = matrix(0, nrow = Kx, ncol = Ky),
  lb = matrix(-Inf, nrow = Kx, ncol = Ky),
  ub = matrix(Inf, nrow = Kx, ncol = Ky),
  pr = 0.05,
  max_iter = 10000,
  tol_level = 1e-06,
  scale = 1,
  nB = 2000,
  verbose = TRUE
)
}
\arguments{
\item{X}{The matrix of men's traits. Its rows must be ordered so that the
i-th man is matched with the i-th woman: this means that \code{nrow(X)}
must be equal to \code{nrow(Y)}. Its columns correspond to the different
matching variables: \code{ncol(X)} can be different from \code{ncol(Y)}.
For the sake of clarity of exposition when using descriptive tools such
as \code{\link{show.correlations}}, it is recommended assigning the same
matching variable to the k-th column of \code{X} and to the k-th column
of \code{Y}, whenever possible. If \code{X} has more matching variables
than \code{Y}, then those variables that appear in \code{X} but no in {Y}
should be found in the last columns of \code{X} (and vice versa). The
matrix is demeaned and rescaled before the start of the estimation
algorithm.}

\item{Y}{The matrix of women's traits. Its rows must be ordered so that the
i-th woman is matched with the i-th man: this means that \code{nrow(Y)}
must be equal to \code{nrow(X)}.  Its columns correspond to the different
matching variables: \code{ncol(Y)} can be different from \code{ncol(X)}.
The matrix is demeaned and rescaled before the start of the estimation
algorithm.}

\item{w}{A vector of sample weights with length \code{nrow(X)}. Defaults to
uniform weights.}

\item{A0}{A vector or matrix with \code{ncol(X)*ncol(Y)} elements
corresponding to the initial values of the affinity matrix to be fed to
the estimation algorithm. Optional. Defaults to matrix of zeros.}

\item{lb}{A vector or matrix with \code{ncol(X)*ncol(Y)} elements
corresponding to the lower bounds of the elements of the affinity matrix.
Defaults to \code{-Inf} for all parameters.}

\item{ub}{A vector or matrix with \code{ncol(X)*ncol(Y)} elements
corresponding to the upper bounds of the elements of the affinity matrix.
Defaults to \code{Inf} for all parameters.}

\item{pr}{A probability indicating the significance level used to compute
bootstrap two-sided confidence intervals for \code{U}, \code{V} and
\code{lambda}. Defaults to 0.05.}

\item{max_iter}{An integer indicating the maximum number of iterations in the
Maximum Likelihood Estimation. See \code{\link[stats]{optim}} for the
\code{"L-BFGS-B"} method. Defaults to 10000.}

\item{tol_level}{A positive real number indicating the tolerance level in the
Maximum Likelihood Estimation. See \code{\link[stats]{optim}} for the
\code{"L-BFGS-B"} method. Defaults to 1e-6.}

\item{scale}{A positive real number indicating the scale of the model.
Defaults to 1.}

\item{nB}{An integer indicating the number of bootstrap replications used to
compute the confidence intervals of \code{U}, \code{V} and \code{lambda}.
Defaults to 2000.}

\item{verbose}{If \code{TRUE}, the function displays messages to keep track
of its progress. Defaults to \code{TRUE}.}
}
\value{
The function returns a list with elements: \code{X}, the demeaned and
rescaled matrix of men's traits; \code{Y}, the demeaned and rescaled
matrix of men's traits; \code{fx}, the empirical marginal distribution of
men; \code{fy}, the empirical marginal distribution of women;
\code{Aopt}, the estimated affinity matrix; \code{sdA}, the standard
errors of \code{Aopt}; \code{tA}, the Z-test statistics of \code{Aopt};
\code{VarCovA}, the full variance-covariance matrix of \code{Aopt};
\code{rank.tests}, a list with all the summaries of the rank tests on
\code{Aopt}; \code{U}, whose columns are the left-singular vectors of
\code{Aopt}; \code{V}, whose columns are the right-singular vectors of
\code{Aopt}; \code{lambda}, whose elements are the singular values of
\code{Aopt}; \code{UCI}, whose columns are the lower and the upper bounds
of the confidence intervals of \code{U}; \code{VCI}, whose columns are
the lower and the upper bounds of the confidence intervals of \code{V};
\code{lambdaCI}, whose columns are the lower and the upper bounds of the
confidence intervals of \code{lambda}; \code{df.bootstrap}, a data frame
resulting from the \code{nB} bootstrap replications and used to infer the
empirical distribution of the estimated objects.
}
\description{
This function estimates the affinity matrix of the matching model of Dupuy
and Galichon (2014), performs the saliency analysis and the rank tests. The
user must supply a \emph{matched sample} that is treated as the equilibrium
matching of a bipartite one-to-one matching model without frictions and with
Transferable Utility. For the sake of clarity, in the documentation we take
the example of the marriage market and refer to "men" as the observations on
one side of the market and to "women" as the observations on the other side.
Other applications may include matching between CEOs and firms, firms and
workers, buyers and sellers, etc.
}
\examples{

# Parameters
Kx = 4; Ky = 4; # number of matching variables on both sides of the market
N = 200 # sample size
mu = rep(0, Kx+Ky) # means of the data generating process
Sigma = matrix(c(1, 0.326, 0.1446, -0.0668, 0.5712, 0.4277, 0.1847, -0.2883,
                 0.326, 1, -0.0372, 0.0215, 0.2795, 0.8471, 0.1211, -0.0902,
                 0.1446, -0.0372, 1, -0.0244, 0.2186, 0.0636, 0.1489,
                 -0.1301, -0.0668, 0.0215, -0.0244, 1, 0.0192, 0.0452,
                 -0.0553, 0.2717, 0.5712, 0.2795, 0.2186, 0.0192, 1, 0.3309,
                 0.1324, -0.1896, 0.4277, 0.8471, 0.0636, 0.0452, 0.3309, 1,
                 0.0915, -0.1299, 0.1847, 0.1211, 0.1489, -0.0553, 0.1324,
                 0.0915, 1, -0.1959, -0.2883, -0.0902, -0.1301, 0.2717,
                 -0.1896, -0.1299, -0.1959, 1),
               nrow=Kx+Ky) # (normalized) variance-covariance matrix of the
               # data generating process
labels_x = c("Educ.", "Age", "Height", "BMI") # labels for men's matching variables
labels_y = c("Educ.", "Age", "Height", "BMI") # labels for women's matching variables

# Sample
data = MASS::mvrnorm(N, mu, Sigma) # generating sample
X = data[,1:Kx]; Y = data[,Kx+1:Ky] # men's and women's sample data
w = sort(runif(N-1)); w = c(w,1) - c(0,w) # sample weights

# Main estimation
res = estimate.affinity.matrix(X, Y, w = w, nB = 500)

# Summarize results
show.affinity.matrix(res, labels_x = labels_x, labels_y = labels_y)
show.diagonal(res, labels = labels_x)
show.test(res)
show.saliency(res, labels_x = labels_x, labels_y = labels_y,
              ncol_x = 2, ncol_y = 2)
show.correlations(res, labels_x = labels_x, labels_y = labels_y,
                  label_x_axis = "Husband", label_y_axis = "Wife", ndims = 2)

}
\seealso{
\strong{Dupuy, Arnaud, and Alfred Galichon}. "Personality traits and
the marriage market." \emph{Journal of Political Economy} 122, no. 6
(2014): 1271-1319.
}
