% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bcf.R
\name{predict.bcf}
\alias{predict.bcf}
\title{Takes a fitted bcf object produced by bcf() along with serialized tree samples and produces predictions for a new set of covariate values}
\usage{
\method{predict}{bcf}(
  object,
  x_predict_control,
  x_predict_moderate,
  pi_pred,
  z_pred,
  save_tree_directory,
  log_file = file.path(".", sprintf("bcf_log_\%s.txt", format(Sys.time(),
    "\%Y\%m\%d_\%H\%M\%S"))),
  n_cores = 2,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{object}{output from a BCF predict run}

\item{x_predict_control}{matrix of covariates for the "prognostic" function mu(x) for predictions (optional)}

\item{x_predict_moderate}{matrix of covariates for the covariate-dependent treatment effects tau(x) for predictions (optional)}

\item{pi_pred}{propensity score for prediction}

\item{z_pred}{Treatment variable for predictions (optional except if x_pre is not empty)}

\item{save_tree_directory}{directory where the trees have been saved}

\item{log_file}{File to log progress}

\item{n_cores}{An optional integer of the number of cores to run your MCMC chains on}

\item{verbose}{Logical; set to FALSE to suppress extra output}

\item{...}{additional arguments affecting the predictions produced.}
}
\value{
A list with elements: tau (samples of treatment effects), mu (samples of predicted control outcomes), yhat (samples of predicted values), and coda_chains (coda objects for scalar summaries)
}
\description{
This function takes in an existing BCF model fit and uses it to predict estimates for new data.
It is important to note that this function requires that you indicate where the trees from the model fit are saved.
You can do so using the save_tree_directory argument in bcf(). Otherwise, they will be saved in the working directory.
}
\examples{
\dontrun{

# data generating process
p = 3 #two control variables and one moderator
n = 250

x = matrix(rnorm(n*p), nrow=n)

# create targeted selection
q = -1*(x[,1]>(x[,2])) + 1*(x[,1]<(x[,2]))

# generate treatment variable
pi = pnorm(q)
z = rbinom(n,1,pi)

# tau is the true (homogeneous) treatment effect
tau = (0.5*(x[,3] > -3/4) + 0.25*(x[,3] > 0) + 0.25*(x[,3]>3/4))

# generate the response using q, tau and z
mu = (q + tau*z)

# set the noise level relative to the expected mean function of Y
sigma = diff(range(q + tau*pi))/8

# draw the response variable with additive error
y = mu + sigma*rnorm(n)

# If you didn't know pi, you would estimate it here
pihat = pnorm(q)

n_burn = 5000
n_sim = 5000

bcf_fit = bcf(y               = y,
              z               = z,
              x_control       = x,
              x_moderate      = x,
              pihat           = pihat,
              nburn           = n_burn,
              nsim            = n_sim,
              n_chains        = 2,
              update_interval = 100,
              save_tree_directory = './trees')

# Predict using new data

x_pred = matrix(rnorm(n*p), nrow=n)

pred_out = predict(bcf_out=bcf_fit,
                   x_predict_control=x_pred,
                   x_predict_moderate=x_pred,
                   pi_pred=pihat,
                   z_pred=z,
                   save_tree_directory = './trees')

}
}
