% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bin2norm.R
\name{bin2norm}
\alias{bin2norm}
\title{bin2norm: A user-friendly interface to estimate normal distribution parameters
from dichotomized data}
\usage{
bin2norm(
  scenario = c("single_threshold", "multiple_thresholds"),
  method = NULL,
  n_i = NULL,
  c_i = NULL,
  p_i_obs = NULL,
  data_list = NULL,
  ...
)
}
\arguments{
\item{scenario}{character string, either \code{"single_threshold"} or \code{"multiple_thresholds"}.}

\item{method}{character string indicating which estimation method to use.
\itemize{
\item For \code{scenario = "single_threshold"}, valid \code{method} are \code{"MLE"} or \code{"probit"}.
\item For \code{scenario = "multiple_thresholds"}, valid \code{method} are
\code{"MLE_integration"}, \code{"GLMM"}, or \code{"Bayesian"}.
}}

\item{n_i, c_i, p_i_obs}{used \strong{only if \code{scenario="single_threshold"}}. Numeric vectors of
the same length.  \eqn{n_i} is study sample size, \eqn{c_i} is threshold, \eqn{p_{i,\mathrm{obs}}} is observed
proportion above threshold.}

\item{data_list}{used \strong{only if \code{scenario="multiple_thresholds"}}, a list with:
\itemize{
\item \code{n_i}: numeric vector (length I) of sample sizes
\item \code{c_ij}: list of length I, where \code{c_ij[[i]]} is a numeric vector of thresholds in study i
\item \code{p_ij_obs}: list of length I, where \code{p_ij_obs[[i]]} is a numeric vector of observed proportions above each threshold
}}

\item{...}{additional arguments passed to lower-level functions (e.g. \code{use_wols_init},
\code{gh_points}, \code{iter}, \code{chains}, etc.).}
}
\value{
A list of estimated parameters, depending on the data-collection setting
(\code{scenario}) and the chosen method. Typically includes:
\itemize{
\item \code{mu} or \code{mu0}
\item \code{sigma}
\item \code{tau} (only for multiple-threshold methods)
}
}
\description{
This function handles two data-collection settings for estimating normal parameters
from threshold-based (dichotomized) data:

\itemize{
\item \strong{Single-threshold per study:}
Each of \eqn{I} studies reports one threshold \eqn{c_i}, a sample size \eqn{n_i},
and the observed proportion \eqn{p_i^{obs}} of samples above that threshold.
We assume one normal distribution \eqn{\mathcal{N}(\mu,\sigma^2)} across all studies.
Methods include \code{"MLE"} and \code{"probit"}.

\item \strong{Multiple-thresholds per study:}
Each study \eqn{i} reports \eqn{K_i} thresholds \eqn{\{c_{ij}\}}, each with an
observed proportion \eqn{p_{ij}^{obs}}. We assume the study-specific mean
\eqn{\mu_i \sim \mathcal{N}(\mu_0,\tau^2)} and within-study variance \eqn{\sigma^2}.
Because each study has multiple cutpoints, one can estimate \eqn{\mu_0, \sigma, \tau}.
Methods include \code{"MLE_integration"}, \code{"GLMM"}, or \code{"Bayesian"} (MCMC).
}
}
\examples{
\donttest{
# Single-threshold example
n_i <- c(100, 120, 80)
c_i <- c(1.2, 1.0, 1.5)
p_i_obs <- c(0.30, 0.25, 0.40)
bin2norm(scenario="single_threshold", method="MLE", n_i=n_i, c_i=c_i, p_i_obs=p_i_obs)

# Multiple-thresholds example
data_list <- list(
  n_i = c(100, 120),
  c_ij = list(c(1.0,1.2), c(0.8,1.5,2.0)),
  p_ij_obs = list(c(0.20,0.30), c(0.15,0.40,0.55))
)

# MLE with numeric integration
bin2norm(scenario="multiple_thresholds", method="MLE_integration",
         data_list=data_list, gh_points=5)

# GLMM approximation
# library(lme4)
bin2norm(scenario="multiple_thresholds", method="GLMM",
         data_list=data_list, use_lme4=TRUE)

# Bayesian MCMC approach
# library(rstan)
bin2norm(scenario="multiple_thresholds", method="Bayesian",
         data_list=data_list, iter=1000, chains=2)
}
}
