% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PolyaUrnBivarDirichlet.R
\name{PolyaUrnBivarDirichlet}
\alias{PolyaUrnBivarDirichlet}
\title{Calibrate and Summarise Multiple Radiocarbon Samples via
a Bayesian Non-Parametric DPMM (with Polya Urn Updating)}
\usage{
PolyaUrnBivarDirichlet(
  rc_determinations,
  rc_sigmas,
  calibration_curve,
  F14C_inputs = FALSE,
  n_iter = 1e+05,
  n_thin = 10,
  use_F14C_space = TRUE,
  slice_width = NA,
  slice_multiplier = 10,
  n_clust = min(10, length(rc_determinations)),
  show_progress = TRUE,
  sensible_initialisation = TRUE,
  lambda = NA,
  nu1 = NA,
  nu2 = NA,
  A = NA,
  B = NA,
  alpha_shape = NA,
  alpha_rate = NA,
  mu_phi = NA,
  calendar_ages = NA
)
}
\arguments{
\item{rc_determinations}{A vector of observed radiocarbon determinations. Can be provided either as
\eqn{{}^{14}}C ages (in \eqn{{}^{14}}C yr BP) or as F\eqn{{}^{14}}C concentrations.}

\item{rc_sigmas}{A vector of the (1-sigma) measurement uncertainties for the
radiocarbon determinations. Must be the same length as \code{rc_determinations} and
given in the same units.}

\item{calibration_curve}{A dataframe which must contain one column \code{calendar_age_BP}, and also
columns \code{c14_age} and \code{c14_sig} or \code{f14c} and \code{f14c_sig} (or both sets).
This format matches the curves supplied with this package, e.g., \link{intcal20},
\link{intcal13}, which contain all 5 columns.}

\item{F14C_inputs}{\code{TRUE} if the provided \code{rc_determinations} are F\eqn{{}^{14}}C
concentrations and \code{FALSE} if they are radiocarbon ages. Defaults to \code{FALSE}.}

\item{n_iter}{The number of MCMC iterations (optional). Default is 100,000.}

\item{n_thin}{How much to thin the MCMC output (optional). Will store every
\code{n_thin}\eqn{{}^\textrm{th}} iteration. 1 is no thinning, while a larger number will result
in more thinning. Default is 10. Must choose an integer greater than 1. Overall
number of MCMC realisations stored will be \eqn{n_{\textrm{out}} =
\textrm{floor}( n_{\textrm{iter}}/n_{\textrm{thin}}) + 1} so do not choose
\code{n_thin} too large to ensure there are enough samples from the posterior to
use for later inference.}

\item{use_F14C_space}{If \code{TRUE} (default) the calculations within the function are carried
out in F\eqn{{}^{14}}C space. If \code{FALSE} they are carried out in \eqn{{}^{14}}C
age space. We recommend selecting \code{TRUE} as, for very old samples, calibrating in
F\eqn{{}^{14}}C space removes the potential affect of asymmetry in the radiocarbon age
uncertainty.
\emph{Note:} This flag can be set independently of the format/scale on which
\code{rc_determinations} were originally provided.}

\item{slice_width}{Parameter for slice sampling (optional). If not given a value
is chosen intelligently based on the spread of the initial calendar ages.
Must be given if \code{sensible_initialisation} is \code{FALSE}.}

\item{slice_multiplier}{Integer parameter for slice sampling (optional).
Default is 10. Limits the slice size to \code{slice_multiplier * slice_width}.}

\item{n_clust}{The number of clusters with which to initialise the sampler (optional). Must
be less than the length of \code{rc_determinations}. Default is 10 or the length
of \code{rc_determinations} if that is less than 10.}

\item{show_progress}{Whether to show a progress bar in the console during
execution. Default is \code{TRUE}.}

\item{sensible_initialisation}{Whether to use sensible values to initialise the sampler
and an automated (adaptive) prior on \eqn{\mu_{\phi}} and (A, B) that is informed
by the observed \code{rc_determinations}. If this is \code{TRUE} (the recommended default), then
all the remaining arguments below are ignored.}

\item{lambda, nu1, nu2}{Hyperparameters for the prior on the mean
\eqn{\phi_j} and precision \eqn{\tau_j} of each individual calendar age
cluster \eqn{j}:
\deqn{(\phi_j, \tau_j)|\mu_{\phi} \sim
\textrm{NormalGamma}(\mu_{\phi}, \lambda, \nu_1, \nu_2)} where
\eqn{\mu_{\phi}} is the overall cluster centering. Required if
\code{sensible_initialisation} is \code{FALSE}.}

\item{A, B}{Prior on \eqn{\mu_{\phi}} giving the mean and precision of the
overall centering \eqn{\mu_{\phi} \sim N(A, B^{-1})}. Required if \code{sensible_initialisation} is \code{FALSE}.}

\item{alpha_shape, alpha_rate}{Shape and rate hyperparameters that specify
the prior for the Dirichlet Process (DP) concentration, \eqn{\alpha}. This
concentration \eqn{\alpha} determines the number of clusters we
expect to observe among our \eqn{n} sampled objects. The model
places a prior on
\eqn{\alpha \sim \Gamma(\eta_1, \eta_2)}, where \eqn{\eta_1, \eta_2} are
the \code{alpha_shape} and \code{alpha_rate}. A small \eqn{\alpha} means the DPMM is
more concentrated (i.e. we expect fewer calendar age clusters) while a large alpha means it is less
less concentrated (i.e. many clusters). Required if \code{sensible_initialisation} is \code{FALSE}.}

\item{mu_phi}{Initial value of the overall cluster centering \eqn{\mu_{\phi}}.
Required if \code{sensible_initialisation} is \code{FALSE}.}

\item{calendar_ages}{The initial estimate for the underlying calendar ages
(optional). If supplied, it must be a vector with the same length as
\code{rc_determinations}.  Required if \code{sensible_initialisation} is \code{FALSE}.}
}
\value{
A list with 10 items. The first 8 items contain output of the model, each of
which has one dimension of size \eqn{n_{\textrm{out}} =
\textrm{floor}( n_{\textrm{iter}}/n_{\textrm{thin}}) + 1}. The rows in these items store
the state of the MCMC from every \eqn{n_{\textrm{thin}}}\eqn{{}^\textrm{th}} iteration:

\describe{
\item{\code{cluster_identifiers}}{A list of length \eqn{n_{\textrm{out}}} each entry
gives the cluster allocation (an integer between 1 and \code{n_clust})
for each observation on the relevant MCMC iteration. Information on the state of
these calendar age clusters (means and precisions) can be found in the other output items.}
\item{\code{alpha}}{A double vector of length \eqn{n_{\textrm{out}}} giving the Dirichlet Process
concentration parameter \eqn{\alpha}.}
\item{\code{n_clust}}{An integer vector of length \eqn{n_{\textrm{out}}} giving
the current number of clusters in the model.}
\item{\code{phi}}{A list of length \eqn{n_{\textrm{out}}} each entry giving
a vector of length \code{n_clust} of the means of the current calendar age clusters
\eqn{\phi_j}.}
\item{\code{tau}}{A list of length \eqn{n_{\textrm{out}}} each entry giving
a vector of length \code{n_clust} of the precisions of the current calenadar age cluster
\eqn{\tau_j}.}
\item{\code{observations_per_cluster}}{A list of length \eqn{n_{\textrm{out}}} each entry giving
a vector of length \code{n_clust} of the number of observations for that cluster.}
\item{\code{calendar_ages}}{An \eqn{n_{\textrm{out}}} by \eqn{n_{\textrm{obs}}}
integer matrix. Gives the current estimate for the calendar age of each individual
observation.}
\item{\code{mu_phi}}{A vector of length \eqn{n_{\textrm{out}}} giving the overall
centering \eqn{\mu_{\phi}} of the calendar age clusters.}
}
where \eqn{n_{\textrm{obs}}} is the number of radiocarbon observations i.e.
the length of \code{rc_determinations}.

The remaining items give information about the input data, input parameters (or
those calculated using \code{sensible_initialisation}) and the update_type

\describe{
\item{\code{update_type}}{A string that always has the value "Polya Urn".}
\item{\code{input_data}}{A list containing the \eqn{{}^{14}}C data used, and the name of
the calibration curve used.}
\item{\code{input_parameters}}{A list containing the values of the fixed
hyperparameters \code{lambda}, \code{nu1}, \code{nu2}, \code{A}, \code{B}, \code{alpha_shape},
\code{alpha_rate} and \code{mu_phi}, and the slice parameters \code{slice_width} and
\code{slice_multiplier}.}
}
}
\description{
This function calibrates sets of multiple radiocarbon (\eqn{{}^{14}}C)
determinations, and simultaneously summarises the resultant calendar age information.
This is achieved using Bayesian non-parametric density estimation,
providing a statistically-rigorous alternative to summed probability
distributions (SPDs).

It takes as an input a set of \eqn{{}^{14}}C determinations and associated \eqn{1\sigma}
uncertainties, as well as the radiocarbon age calibration curve to be used. The samples
are assumed to arise from an (unknown) shared calendar age distribution \eqn{f(\theta)} that
we would like to estimate, alongside performing calibration of each sample.

The function models the underlying distribution \eqn{f(\theta)} as a Dirichlet process
mixture model (DPMM), whereby the samples are considered to arise from an unknown number of
distinct clusters. Fitting is achieved via MCMC.

It returns estimates for the calendar age of each individual radiocarbon sample; and broader
output (including the means and variances of the underpinning calendar age clusters)
that can be used by other library functions to provide a predictive estimate of the
shared calendar age density \eqn{f(\theta)}.

For more information read the vignette: \cr
\code{vignette("Non-parametric-summed-density", package = "carbondate")}

\strong{Note:} The library provides two slightly-different update schemes for the MCMC. In this
particular function, updating of the DPMM is achieved by a Polya Urn approach (Neal 2000)
This is our recommended updating approach based on testing. The alternative, slice-sampled, approach
can be found at \link{WalkerBivarDirichlet}

\strong{References:} \cr
Heaton, TJ. 2022. “Non-parametric Calibration of Multiple Related Radiocarbon
Determinations and their Calendar Age Summarisation.” \emph{Journal of the Royal Statistical
Society Series C: Applied Statistics} \strong{71} (5):1918-56. https://doi.org/10.1111/rssc.12599. \cr
Neal, RM. 2000. “Markov Chain Sampling Methods for Dirichlet Process Mixture Models.”
\emph{Journal of Computational and Graphical Statistics} \strong{9} (2):249 https://doi.org/10.2307/1390653.
}
\examples{
# Note these examples are shown with a small n_iter to speed up execution.
# When you run ensure n_iter gives convergence (try function default).

# Basic usage making use of sensible initialisation to set most values and
# using a saved example data set and the IntCal20 curve.
polya_urn_output <- PolyaUrnBivarDirichlet(
    two_normals$c14_age,
    two_normals$c14_sig,
    intcal20,
    n_iter = 100,
    show_progress = FALSE)

# The radiocarbon determinations can be given as F14C concentrations
polya_urn_output <- PolyaUrnBivarDirichlet(
    two_normals$f14c,
    two_normals$f14c_sig,
    intcal20,
    F14C_inputs = TRUE,
    n_iter = 100,
    show_progress = FALSE)
}
\seealso{
\link{WalkerBivarDirichlet} for our less-preferred MCMC method to update the Bayesian DPMM
(otherwise an identical model); and \link{PlotCalendarAgeDensityIndividualSample},
\link{PlotPredictiveCalendarAgeDensity} and \link{PlotNumberOfClusters}
to access the model output and estimate the calendar age information. \cr \cr
See also \link{PPcalibrate} for an an alternative (similarly rigorous) approach to
calibration and summarisation of related radiocarbon determinations using a variable-rate Poisson process
}
