% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/func.cle.prop.R
\name{func.cle.prop}
\alias{func.cle.prop}
\title{Composite Likelihood Estimation for Spatial Proportional Data}
\usage{
func.cle.prop(vec.yobs, mat.X, mat.lattice, radius, n.sim = 100,
  parallel = TRUE, n.core = max(detectCores()/2, 1), output = TRUE)
}
\arguments{
\item{vec.yobs}{a vector of observed responses for all N sites.}

\item{mat.X}{regression (design) matrix, including intercepts.}

\item{mat.lattice}{a data matrix containing geographical information of sites. The i-th row constitutes a set of geographical coordinates.}

\item{radius}{weight radius.}

\item{n.sim}{number of simulations used for calculate the Godambe matrix (default: 100).}

\item{parallel}{logical flag indicating using parallel processing (default: \code{TRUE})}

\item{n.core}{number of physical cores used for parallel processing (when \code{parallel} is \code{TRUE}), default value is \code{max(detectCores()/2,1)}).}

\item{output}{logical flag indicates whether printing out result (default: \code{TRUE}).}
}
\value{
\code{func.cle.prop} returns a list containing:

\code{vec.par}: a vector of estimator for \eqn{\theta=(\alpha,\beta,\sigma^2,\rho)};

\code{vec.se}: a vector of standard error for the estimator;

\code{mat.asyvar}: estimated asymptotic covariance matrix \eqn{H^{-1}(\theta)J(\theta)H^{-1}(\theta)} for the estimator; and

\code{vec.comp}: a vector of computational time for parameter and standard error estimation.

\code{CLIC}: Composite likelihood information criterion proposed by Varin and Vidoni (2005), i.e. \eqn{-2*logCL(\theta) + 2*trace(H^{-1}(\theta)J(\theta))}
}
\description{
\code{func.cle.prop} performs composite likelihood estimation of parameters and their standard errors in a spatial Tobit model by maximizing its composite log-likelihood.
}
\details{
Given the design matrix, the vector of observed responses, spatial lattice data, weight radius, and the prespecified number of simulated vectors of responses used in estimating the Godambe information matrix, this function assumes initial values of \eqn{\beta} as the estimates from the standard Type I Tobit model with independent responses. The initial value of \eqn{\alpha} and the right limit of the Tobit model are equally set to 1. Since there is only one cutoff point to be estimated, reparameterization is unnecessary. The function first estimates parameters of interest by maximizing the composite log-likelihood using \code{optim(...,method = "L-BFGS-B")}, then computes the simulated based standard error and asymptotic covariance matrix.
}
\examples{
# True parameter
alpha <- 4; vec.beta <- c(1, 2, 1, 0, -1); sigmasq <- 0.8; rho <- 0.6; radius <- 5
vec.par <- c(alpha, vec.beta, sigmasq, rho)

# Coordinate matrix
n.lati <- 30; n.long <- 30
n.site <- n.lati * n.long
mat.lattice <- cbind(rep(1:n.lati, n.long), rep(1:n.long, each=n.lati))
mat.dist <- as.matrix(dist(mat.lattice, upper=TRUE, diag=TRUE))
mat.cov <- sigmasq * rho^mat.dist

set.seed(1228)

# Generate regression (design) matrix with intercept
mat.X <- cbind(rep(1, n.site),scale(matrix(rnorm(n.site*(length(vec.beta)-1)),nrow=n.site)))
vec.Z <- t(chol(mat.cov)) \%*\% rnorm(n.site) + mat.X \%*\% vec.beta
vec.epsilon <- diag(sqrt(1-sigmasq), n.site) \%*\% rnorm(n.site)
vec.ylat <- as.numeric(vec.Z + vec.epsilon)

# Convert to the vector of observation
vec.yobs <- func.obs.prop(vec.ylat, alpha=alpha)

# With parallel computing

\dontrun{
prop.example <- func.cle.prop(vec.yobs, mat.X, mat.lattice, radius,
n.sim=100, parallel = TRUE, n.core = 2)

round(prop.example$vec.par,4)
# alpha   beta0   beta1   beta2   beta3   beta4 sigma^2     rho
# 3.8259  0.9921  1.9679  0.9455  0.0148 -0.9871  0.8386  0.5761

round(prop.example$vec.se ,4)
# alpha   beta0   beta1   beta2   beta3   beta4 sigma^2     rho
# 0.1902  0.1406  0.1103  0.0744  0.0385  0.0652  0.1527  0.1151
}

# Without parallel computing

\dontrun{
prop.example2 <- func.cle.prop(vec.yobs, mat.X, mat.lattice, radius, n.sim=100, parallel = FALSE)
}

}
\references{
Feng, Xiaoping, Zhu, Jun, Lin, Pei-Sheng, and Steen-Adams, Michelle M. (2014) Composite likelihood Estimation for Models of Spatial Ordinal Data and Spatial Proportional Data with Zero/One values. \emph{Environmetrics} 25(8): 571--583.
}
