# checkbox table frequency ------------------------------------------------

#' Frequency for Checkbox Fields.
#'
#' Each category (or option) is a variable in the dataframe, and we count the number of `sucess` in each column. The column `percent` is the ratio between this count by the observation number.
#'
#' Missing values are ignored.
#'
#' @param data dataframe object
#' @param columns character vector. Columns to count.
#' @param sucess character scalar. Category indicating the sucess. Defaults to "checked".
#' @param labels character vector.Label of each category. Defaults to NULL.
#' @param variable_name character scalar. Name of field in the form. Defaults to NULL.
#'
#' @returns return a dataframe with `n` and `percent`
#' @export
#' @import dplyr rlang
#'
#' @examples
#' data <- data.frame(
#'   x1 = c("checked", "checked", "unchecked"),
#'   x2 = c("checked", "unchecked", "checked")
#' )
#' tab_freq_checkbox(data, c("x1", "x2"))
tab_freq_checkbox <- function (data, columns, sucess = "checked",
                               labels = NULL, variable_name = NULL) {
  if (!is.null(labels) && (length(columns) != length(labels))) {
    stop("When 'labels' is provided, it must satisfy 'length(columns) == length(labels)'.")
  }

  if (!is.null(variable_name) && is.null(labels)) {
    stop("You have to provide 'labels' in order to use 'variable_name'.")
  }

  sample_size <- nrow(data)

  res <- columns |>
    lapply(\(column) {
      c(
        "n" = sum(dplyr::pull(data, {{ column }}) == sucess, na.rm = T),
        "percent" = sum(dplyr::pull(data, {{ column }}) == sucess, na.rm = T) / sample_size
      )
    }) |>
    bind_rows() |>
    mutate(coluna = columns) |>
    relocate(coluna, .before = n)

  if (!is.null(labels)) {
    res <- res |>
      mutate(categoria = labels) |>
      relocate(categoria, .before = n)
  }

  if (!is.null(variable_name)) {
    res <- res |>
      rename({{ variable_name }} := categoria)
  }

  res
}
