
#' Density-Convoluted Support Vector Machines
#'
#' This package implements the density-convoluted SVM for high-dimensional classification.
#'
#' @name dcsvm-package
#' @aliases dcsvm-package
#' @author Boxiang Wang, Le Zhou, Yuwen Gu, and Hui Zou\cr Maintainer:
#' Boxiang Wang <boxiang-wang@uiowa.edu>
#' @references
#' Wang, B., Zhou, L., Gu, Y., and Zou, H. (2023)
#' \emph{Density-Convoluted Support Vector Machines for High-Dimensional Classification, IEEE Transactions on Information Theory, Vol. 69(4), 2523-2536},

#' @description
#' This package provides tools to perform density-convoluted support vector machine (DCSVM) modeling for high-dimensional data classification.
#'
#' @details
#' \tabular{ll}{
#' Package: \tab dcsvm\cr
#' Type: \tab Package\cr
#' Version: \tab 0.0.1\cr
#' Date: \tab 2025-01-08\cr
#' License: \tab GPL-2\cr
#' }
#' The package \code{dcsvm} contains five main functions:\cr
#' \itemize{
#'   \item \code{dcsvm}
#'   \item \code{cv.dcsvm}
#'   \item \code{coef.dcsvm}
#'   \item \code{plot.dcsvm}
#'   \item \code{plot.cv.dcsvm}
#' }
#'
#' @keywords SVM
#' @keywords high-dimensional
#' @keywords classification
#'
#' @import methods
#' @import Matrix
#' @importFrom stats approx coef predict rnorm pnorm sd
#' @importFrom graphics abline axis matplot points segments text
#' @importFrom grDevices gray.colors
"_PACKAGE"


#' Internal DCSVM Functions
#'
#' Internal helper functions for the DCSVM package.
#'
#' @name dcsvm-internal
#' @title Internal DCSVM Functions
#' @description These internal functions are not intended for direct use by users. 
#' For example, \code{err} obtains error messages from Fortran code, 
#' \code{error.bars} helps to plot cross-validation error curves, 
#' and \code{getmin} identifies the best \code{lambda} through cross-validation.
#' Other functions assist with kernel loss computations, coefficient management, and interpolation.
#'
#' @aliases cv.gaussian cv.uniform cv.epanechnikov cvcompute dcsvm.gauss dcsvm.unif
#' dcsvm.epane err error.bars getmin getoutput lambda.interp lamfix loss.epanechnikov
#' loss.gaussian loss.uniform nonzero zeromat
#'
#' @usage
#' cv.gaussian(outlist, lambda, x, y, foldid, pred.loss, hval)
#' cv.uniform(outlist, lambda, x, y, foldid, pred.loss, hval)
#' cv.epanechnikov(outlist, lambda, x, y, foldid, pred.loss, hval)
#' cvcompute(mat, foldid, nlams)
#' dcsvm.gauss(x, y, alpha, lam2, hval, nlam, flmin, ulam, isd, eps, dfmax, pmax, jd, 
#'   pfncol, pf, pf2, maxit, istrong, nobs, nvars, vnames)
#' dcsvm.unif(x, y, alpha, lam2, hval, nlam, flmin, ulam, isd, eps, dfmax, pmax, jd, 
#'   pfncol, pf, pf2, maxit, istrong, nobs, nvars, vnames)
#' dcsvm.epane(x, y, alpha, lam2, hval, nlam, flmin, ulam, isd, eps, dfmax, pmax, jd, 
#'   pfncol, pf, pf2, maxit, istrong, nobs, nvars, vnames)
#' err(n, maxit, pmax)
#' error.bars(x, upper, lower, width = 0.02, ...)
#' getmin(lambda, cvm, cvsd)
#' getoutput(fit, maxit, pmax, nvars, vnames)
#' lambda.interp(lambda, s)
#' lamfix(lam)
#' loss.epanechnikov(u, hval)
#' loss.gaussian(u, hval)
#' loss.uniform(u, hval)
#' nonzero(beta, bystep = FALSE)
#' zeromat(nvars, nalam, vnames, stepnames)
#'
#' @details Most of these functions are adapted or directly copied from the \code{gcdnet} and \code{glmnet} packages.
#' 
#' @keywords internal
NULL
#' 
#' Simplified Gene Expression Data from Alon et al. (1999)
#'
#' Gene expression data (2000 genes for 62 samples) from a DNA microarray experiment of colon tissue samples (Alon et al., 1999).
#'
#' @name colon
#' @aliases colon
#' @title Simplified Gene Expression Data from Alon et al. (1999)
#'
#' @usage
#' data(colon)
#'
#' @description
#' This dataset contains 62 colon tissue samples with 2000 gene expression levels. Among these samples, 40 are tumor tissues (coded as 1) and 22 are normal tissues (coded as -1).
#'
#' @return
#' A list with the following elements:
#'   \item{x}{A matrix of 62 rows and 2000 columns representing the gene expression levels of 62 colon tissue samples. Each row corresponds to a sample, and each column corresponds to a gene.}
#'   \item{y}{A numeric vector of length 62 representing the tissue type (1 for tumor; -1 for normal).}
#'
#' @source
#' The data were introduced in Alon et al. (1999).
#'
#' @references
#' Alon, U., Barkai, N., Notterman, D.A., Gish, K., Ybarra, S., Mack, D., and Levine, A.J. (1999). 
#' ``Broad patterns of gene expression revealed by clustering analysis of tumor and normal colon tissues probed by oligonucleotide arrays,''
#' \emph{Proceedings of the National Academy of Sciences}, \bold{96}(12), 6745--6750.
#'
#' @examples
#' # Load the dcsvm library
#' library(dcsvm)
#'
#' # Load the dataset
#' data(colon)
#'
#' # Check the dimensions of the data
#' dim(colon$x)
#'
#' # Count the number of samples in each class
#' sum(colon$y == -1)
#' sum(colon$y == 1)
#'
#' @keywords data set
NULL
