% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_deepSTRAPP_over_time.R
\name{run_deepSTRAPP_over_time}
\alias{run_deepSTRAPP_over_time}
\title{Run deepSTRAPP to test for a relationship between diversification rates and trait data over multiple time steps}
\usage{
run_deepSTRAPP_over_time(
  contMap = NULL,
  densityMaps = NULL,
  ace = NULL,
  tip_data = NULL,
  trait_data_type,
  BAMM_object,
  time_steps = NULL,
  time_range = NULL,
  nb_time_steps = NULL,
  time_step_duration = NULL,
  keep_tip_labels = TRUE,
  rate_type = "net_diversification",
  seed = NULL,
  nb_permutations = NULL,
  replace_samples = FALSE,
  alpha = 0.05,
  two_tailed = TRUE,
  one_tailed_hypothesis = NULL,
  posthoc_pairwise_tests = FALSE,
  p.adjust_method = "none",
  return_perm_data = FALSE,
  nthreads = 1,
  print_hypothesis = TRUE,
  extract_trait_data_melted_df = FALSE,
  extract_diversification_data_melted_df = FALSE,
  return_STRAPP_results = FALSE,
  return_updated_trait_data_with_Map = FALSE,
  return_updated_BAMM_object = FALSE,
  verbose = TRUE,
  verbose_extended = FALSE
)
}
\arguments{
\item{contMap}{For continuous trait data. Object of class \code{"contMap"},
typically generated with \code{\link[=prepare_trait_data]{prepare_trait_data()}} or \code{\link[phytools:contMap]{phytools::contMap()}},
that contains a phylogenetic tree and associated continuous trait mapping.
The phylogenetic tree must be rooted and fully resolved/dichotomous,
but it does not need to be ultrametric (it can includes fossils).}

\item{densityMaps}{For categorical trait or biogeographic data. List of objects of class \code{"densityMap"},
typically generated with \code{\link[=prepare_trait_data]{prepare_trait_data()}},
that contains a phylogenetic tree and associated posterior probability of being in a given state/range along branches.
Each object (i.e., \code{densityMap}) corresponds to a state/range. The phylogenetic tree must be rooted and fully resolved/dichotomous,
but it does not need to be ultrametric (it can includes fossils).}

\item{ace}{(Optional) Ancestral Character Estimates (ACE) at the internal nodes.
Obtained with \code{\link[=prepare_trait_data]{prepare_trait_data()}} as output in the \verb{$ace} slot.
\itemize{
\item For continuous trait data: Named numerical vector typically generated with \code{\link[phytools:fastAnc]{phytools::fastAnc()}}, \code{\link[phytools:anc.ML]{phytools::anc.ML()}}, or \code{\link[ape:ace]{ape::ace()}}.
Names are nodes_ID of the internal nodes. Values are ACE of the trait.
\item For categorical trait or biogeographic data: Matrix that record the posterior probabilities of ancestral states/ranges.
Rows are internal nodes_ID. Columns are states/ranges. Values are posterior probabilities of each state per node.
Needed in all cases to provide accurate estimates of trait values.
}}

\item{tip_data}{(Optional) Named vector of tip values of the trait.
\itemize{
\item For continuous trait data: Named numerical vector of trait values.
\item For categorical trait or biogeographic data: Character string vector of states/ranges
Names are nodes_ID of the internal nodes. Needed to provide accurate tip values.
\item For biogeographic data, ranges should follow the coding scheme of BioGeoBEARS with a unique CAPITAL letter per unique areas
(ex: A, B), combined to form multi-area ranges (Ex: AB). Alternatively, you can provide tip_data as a matrix or data.frame of
binary presence/absence in each area (coded as unique CAPITAL letter). In this case, columns are unique areas, rows are taxa,
and values are integer (0/1) signaling absence or presence of the taxa in the area.
}}

\item{trait_data_type}{Character string. Specify the type of trait data. Must be one of "continuous", "categorical", "biogeographic".}

\item{BAMM_object}{Object of class \code{"bammdata"}, typically generated with \code{\link[=prepare_diversification_data]{prepare_diversification_data()}},
that contains a phylogenetic tree and associated diversification rate mapping across selected posterior samples.
The phylogenetic tree must the same as the one associated with the \code{contMap}, \code{ace} and \code{tip_data}.}

\item{time_steps}{Numerical vector. Time steps at which the STRAPP tests should be carried out. If \code{NULL} (the default),
\code{time_steps} will be generated from a combination of two arguments among \code{time_range}, \code{nb_time_steps}, and/or \code{time_step_duration}.}

\item{time_range}{Vector of two numerical values. Time boundaries within with the \code{time_steps} must be defined if not provided.
If \code{NULL} (the default), and \code{time_range} is needed to generate the \code{time_steps}, the depth of the tree is used by default: \code{c(0, root_age)}.
However, no time step will be generated for the 'root_age'.}

\item{nb_time_steps, time_step_duration}{Numerical. Number of time steps and duration of each time step used to generate \code{time_steps} if not provided.
You must provide at least one of those two arguments to be able to generate \code{time_steps}.}

\item{keep_tip_labels}{Logical. Specify whether terminal branches with a single descendant tip
must retained their initial \code{tip.label} on the updated phylogeny. Default is \code{TRUE}.}

\item{rate_type}{A character string specifying the type of diversification rates to use. Must be one of 'speciation', 'extinction' or 'net_diversification' (default).}

\item{seed}{Integer. Set the seed to ensure reproducibility. Default is \code{NULL} (a random seed is used).}

\item{nb_permutations}{Integer. To select the number of random permutations to perform during the tests.
If NULL (default), all posterior samples will be used once.}

\item{replace_samples}{Logical. To specify whether to allow 'replacement' (i.e., multiple use) of a posterior sample
when drawing samples used to carry out the STRAPP test. Default is \code{FALSE}.}

\item{alpha}{Numerical. Significance level to use to compute the \code{estimate} corresponding to the values of the test statistic used to assess significance of the test.
This does NOT affect p-values. Default is \code{0.05}.}

\item{two_tailed}{Logical. To define the type of tests. If \code{TRUE} (default), tests for correlations/differences in rates will be carried out with a null hypothesis
that rates are not correlated with trait values (continuous data) or equals between trait states (categorical and biogeographic data).
If \code{FALSE}, one-tailed tests are carried out.
\itemize{
\item For continuous data, it involves defining a \code{one_tailed_hypothesis} testing for either a "positive" or "negative" correlation under the alternative hypothesis.
\item For binary data (two states), it involves defining a \code{one_tailed_hypothesis} indicating which states have higher rates under the alternative hypothesis.
\item For multinominal data (more than two states), it defines the type of post hoc pairwise tests to carry out between pairs of states.
If \code{posthoc_pairwise_tests = TRUE}, all two-tailed (if \code{two_tailed = TRUE}) or one-tailed (if \code{two_tailed = FALSE}) tests are automatically carried out.
}}

\item{one_tailed_hypothesis}{A character string specifying the alternative hypothesis in the one-tailed test.
For continuous data, it is either "negative" or "positive" correlation.
For binary data, it lists the trait states with states ordered in increasing rates under the alternative hypothesis, separated by a greater-than such as c('A > B').}

\item{posthoc_pairwise_tests}{Logical. Only for multinominal data (with more than two states). If \code{TRUE}, all possible post hoc pairwise (Dunn) tests
will be computed across all pairs of states. This is a way to detect which pairs of states have significant differences in rates
if the overall test (Kruskal-Wallis) is significant. Default is \code{FALSE}.}

\item{p.adjust_method}{A character string. Only for multinominal data (with more than two states). It specifies the type of correction to apply to the p-values
in the post hoc pairwise tests to account for multiple comparisons. See \code{\link[stats:p.adjust]{stats::p.adjust()}} for the available methods. Default is \code{none}.}

\item{return_perm_data}{Logical. Whether to return the stats data computed from the posterior samples for observed and permuted data in the output.
This is needed to plot the histograms of the null distribution used to assess significance of the tests with \code{\link[=plot_histogram_STRAPP_test_for_focal_time]{plot_histogram_STRAPP_test_for_focal_time()}}.
(for a single \code{focal_time}) and \code{\link[=plot_histograms_STRAPP_tests_over_time]{plot_histograms_STRAPP_tests_over_time()}} (for multiple \code{time_steps}). Default is \code{FALSE}.}

\item{nthreads}{Integer. Number of threads to use for paralleled computing of the STRAPP tests across the permutations.
The R package \code{parallel} must be loaded for \code{nthreads > 1}. Default is \code{1}.}

\item{print_hypothesis}{Logical. Whether to print information on what test is carried out, detailing the null and alternative hypotheses,
and what significant level is used to rejected or not the null hypothesis. Default is \code{TRUE}.}

\item{extract_trait_data_melted_df}{Logical. Specify whether trait data must be extracted from the \code{updated_contMap}/\code{updated_densityMaps} objects at each time step
and returned in a melted data.frame. Default is \code{FALSE}.}

\item{extract_diversification_data_melted_df}{Logical. Specify whether diversification data (regimes ID and tip rates) must be extracted from the \code{updated_BAMM_object}
at each time step and returned in a melted data.frame. Default is \code{FALSE}.}

\item{return_STRAPP_results}{Logical. Specify whether the \code{STRAPP_results} objects summarizing the results of the STRAPP tests carried out at each time step
should be returned among the outputs in addition to the \verb{$pvalues_summary_df} already providing test stat estimates and p-values obtained across all \code{time_steps}.}

\item{return_updated_trait_data_with_Map}{Logical. Specify whether the \code{trait_data} extracted
for the given \code{focal_time} and the updated version of mapped phylogeny (\code{contMap}/\code{densityMaps}) provided as input
should be returned among the outputs. The updated \code{contMap}/\code{densityMaps} consists in cutting off branches and mapping
that are younger than the \code{focal_time}. Default is \code{FALSE}.}

\item{return_updated_BAMM_object}{Logical. Specify whether the \code{updated_BAMM_object} with phylogeny and
mapped diversification rates cut-off at the \code{focal_time} should be returned among the outputs.}

\item{verbose}{Logical. Should progression per \code{time_steps} be displayed? Default is \code{TRUE}.}

\item{verbose_extended}{Should progression per \code{time_steps} AND within each deepSTRAPP workflow de displayed?
In addition to printing progress along \code{time_steps}, a message will be printed at each step of the deepSTRAPP workflow,
and for every batch of 100 BAMM posterior samples whose rates are regimes are updated. If \code{extract_diversification_data_melted_df = TRUE},
a message for will also be printed when rates are extracted. Default is \code{FALSE}.}
}
\value{
The function returns a list with at least five elements.
\itemize{
\item \verb{$pvalues_summary_df} Data.frame with three columns providing test stat \verb{$estimate} and \verb{$p_value} obtained for each time step (i.e., \verb{$focal_time}),
that can be passed down to \code{\link[=plot_STRAPP_pvalues_over_time]{plot_STRAPP_pvalues_over_time()}} to generate a plot showing the evolution of the test results across time.
\item \verb{$time_steps} Numerical vector. Time steps at which the STRAPP tests were carried out in the same order as the objects returned in the output lists.
\item \verb{$trait_data_type} Character string. Specify the type of trait data. Possible values are: "continuous", "categorical", "biogeographic".
\item \verb{$trait_data_type_for_stats} Character string. The type of trait data used to select statistical method. One of 'continuous', 'binary', or 'multinominal'.
\item \verb{$rate_type} Character string. The type of diversification rates used in the tests: 'speciation', 'extinction' or 'net_diversification'.
}

Optional summary df for multinominal data, if \code{posthoc_pairwise_tests = TRUE}:
\itemize{
\item \verb{$pvalues_summary_df_for_posthoc_pairwise_tests} Data.frame with four or five columns providing test stat \verb{$estimate}, \verb{$p_value}, and \verb{$p_value_adjusted}
(if \code{p.adjust_method} used is not "none") for each \verb{$pair} of states involved in post hoc Dunn's tests obtained for each time step (i.e., \verb{$focal_time}).
This data.frame can be passed down to \code{\link[=plot_STRAPP_pvalues_over_time]{plot_STRAPP_pvalues_over_time()}} to generate a plot showing the evolution of the post hoc test results across time.
}

Optional melted data.frames:
\itemize{
\item \verb{$trait_data_df_over_time} Data.frame with three columns providing \verb{$trait_value} associated with each \verb{$tip_ID} found along each time step (i.e., \verb{$focal_time}).
Set \code{extract_trait_data_melted_df = TRUE} to include it in the output.
\item \verb{$diversification_data_df_over_time} Data.frame with six columns providing diversification regimes (\verb{$regime_ID}) and \verb{$rates} sorted by \verb{$rate_type} along tips (\verb{$tip_ID})
found across all posterior samples (\verb{$BAMM_sample_ID}) over each time step (i.e., \verb{$focal_time}).
Set \code{extract_diversification_data_melted_df = TRUE} to include it in the output.
\item Those data.frames can be passed down to \code{\link[=plot_rates_through_time]{plot_rates_through_time()}} to generate a plot showing
the evolution diversification rates across trait values over time.
}

Optional objects generated for each time step (i.e., \code{focal_time}) and ordered as in \verb{$time_steps}:
\itemize{
\item \verb{$STRAPP_results_over_time} List of objects summarizing the results of the STRAPP tests
See \code{\link[=compute_STRAPP_test_for_focal_time]{compute_STRAPP_test_for_focal_time()}} for a detailed description of the elements in each object.
Set \code{return_STRAPP_results = TRUE} to include it in the output.
Combined with \code{return_perm_data = TRUE}, it allows to plot the histograms of the null distributions
used to assess significance of the tests with \code{\link[=plot_histogram_STRAPP_test_for_focal_time]{plot_histogram_STRAPP_test_for_focal_time()}}.
(for a single \code{focal_time}) and \code{\link[=plot_histograms_STRAPP_tests_over_time]{plot_histograms_STRAPP_tests_over_time()}} (for multiple \code{time_steps}).
\item \verb{$updated_trait_data_with_Map_over_time} List of objects containing trait data and updated \code{contMap}/\code{densityMaps}.
Updated \code{contMap}/\code{densityMaps} can be respectively plotted with \code{\link[=plot_contMap]{plot_contMap()}} or \code{\link[=plot_densityMaps_overlay]{plot_densityMaps_overlay()}},
to display a phylogeny mapped with trait values with branches cut at each \code{focal_time}.
\item \verb{$updated_BAMM_objects_over_time} List of objects containing rates and regimes ID mapped on phylogeny.
Updated \code{BAMM_object} can be plotted with \code{\link[=plot_BAMM_rates]{plot_BAMM_rates()}} to display a phylogeny mapped with
diversification rates with branches cut at each \code{focal_time}.
}
}
\description{
Wrapper function to run deepSTRAPP workflows over multiple time steps in the past.
It starts from traits mapped on a phylogeny (trait data) and BAMM output (diversification data)
and carries out the appropriate statistical method to test for a relationship between diversification rates and trait data.
The workflow is repeated over multiple points in time (i.e. the \code{time_steps}) and results are summarized in a data.frame.
The function can also provide summaries of trait values and diversification rates
extracted along branches over the different \code{time_steps}.

Statistical tests are based on block-permutations: rates data are randomized across tips following blocks
defined by the diversification regimes identified on each tip (typically from a BAMM).
Such tests are called STructured RAte Permutations on Phylogenies (STRAPP) as described in
Rabosky, D. L., & Huang, H. (2016). A robust semi-parametric test for detecting trait-dependent diversification.
Systematic biology, 65(2), 181-193. \doi{10.1093/sysbio/syv066}.

See the original \code{\link[BAMMtools:traitDependentBAMM]{BAMMtools::traitDependentBAMM()}} function used to
carry out STRAPP test on extant time-calibrated phylogenies.

Tests can be carried out on speciation, extinction and net diversification rates.
}
\details{
The function is a wrapper of \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}} that runs the
deepSTRAPP workflow over multiple \code{time_steps}.

The deepSTRAPP workflow is described step by step in the \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}} documentation.

Its main output is the \verb{$pvalues_summary_df}: a data.frame providing test stat estimates and p-values obtained across all \code{time_steps},
that can be passed down to \code{\link[=plot_STRAPP_pvalues_over_time]{plot_STRAPP_pvalues_over_time()}} to generate a plot showing the evolution of the test results across time.
If using multinominal data (with more than two states) and \code{posthoc_pairwise_tests = TRUE}, the output will also contain
a data.frame providing test stat estimates and p-values for post hoc pairwise tests in \verb{$pvalues_summary_df_for_posthoc_pairwise_tests}.

The function offers options to generate summary data.frames of the data extracted across \code{time_steps}:
\itemize{
\item If \code{extract_trait_data_melted_df = TRUE}, a data.frame of trait values found along branches at each time step
is provided in \verb{$trait_data_df_over_time}.
\item If \code{extract_diversification_data_melted_df = TRUE}, a data.frame of diversification data (regimes ID and tip rates)
found along branches at each time step is provided in \verb{$diversification_data_df_over_time}.
\item Those data.frames can be passed down to \code{\link[=plot_rates_through_time]{plot_rates_through_time()}} to generate a plot showing
the evolution diversification rates across trait values over time.
}

The function also allows to keep records of the intermediate objects generated during the STRAPP workflow:
\itemize{
\item If \code{return_STRAPP_results = TRUE}, a list of STRAPP test outputs is provided in \verb{$STRAPP_results_over_time}.
Combined with \code{return_perm_data = TRUE}, it allows to plot the histograms of the null distributions
used to assess significance of the tests with \code{\link[=plot_histogram_STRAPP_test_for_focal_time]{plot_histogram_STRAPP_test_for_focal_time()}}.
(for a single \code{focal_time}) and \code{\link[=plot_histograms_STRAPP_tests_over_time]{plot_histograms_STRAPP_tests_over_time()}} (for multiple \code{time_steps}).
\item If \code{return_updated_trait_data_with_Map = TRUE}, a list of objects containing trait data and updated \code{contMap} or \code{densityMaps}
is provided in \verb{$updated_trait_data_with_Map_over_time}. Updated \code{contMap}/\code{densityMaps} can be respectively plotted with \code{\link[=plot_contMap]{plot_contMap()}}
or \code{\link[=plot_densityMaps_overlay]{plot_densityMaps_overlay()}}, to display a phylogeny mapped with trait values with branches cut at each \code{focal_time}.
\item If \code{return_updated_BAMM_object = TRUE}, a list of updated \code{BAMM_object} of class \code{"bammdata"} that contains rates and regimes ID
found at each \code{focal_time}. Updated \code{BAMM_object} can be plotted with \code{\link[=plot_BAMM_rates]{plot_BAMM_rates()}} to display
a phylogeny mapped with diversification rates with branches cut at each \code{focal_time}.
}
}
\examples{
if (deepSTRAPP::is_dev_version())
{
 # ----- Example 1: Continuous trait ----- #
 ## Load data

 # Load trait df
 data(Ponerinae_trait_tip_data, package = "deepSTRAPP")
 # Load phylogeny with old calibration
 data(Ponerinae_tree_old_calib, package = "deepSTRAPP")

 # Load the BAMM_object summarizing 1000 posterior samples of BAMM
 data(Ponerinae_BAMM_object_old_calib, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Prepare trait data

 # Extract continuous trait data as a named vector
 Ponerinae_cont_tip_data <- setNames(object = Ponerinae_trait_tip_data$fake_cont_tip_data,
                                     nm = Ponerinae_trait_tip_data$Taxa)

 # Select a color scheme from lowest to highest values
 color_scale = c("darkgreen", "limegreen", "orange", "red")

 # Get Ancestral Character Estimates based on a Brownian Motion model
 # To obtain values at internal nodes
 Ponerinae_ACE <- phytools::fastAnc(tree = Ponerinae_tree_old_calib, x = Ponerinae_cont_tip_data)

 \donttest{ # (May take several minutes to run)
 # Run a Stochastic Mapping based on a Brownian Motion model
 # to interpolate values along branches and obtain a "contMap" object
 Ponerinae_contMap <- phytools::contMap(Ponerinae_tree, x = Ponerinae_cont_tip_data,
                                        res = 100, # Number of time steps
                                        plot = FALSE)
 # Plot contMap = stochastic mapping of continuous trait
 plot_contMap(contMap = Ponerinae_contMap,
              color_scale = color_scale)

 ## Set for time steps of 5 My. Will generate deepSTRAPP workflows for 0 to 40 Mya.
 # nb_time_steps <- 5
 time_step_duration <- 5
 time_range <- c(0, 40)

 ## Run deepSTRAPP on net diversification rates
 Ponerinae_deepSTRAPP_cont_old_calib_0_40 <- run_deepSTRAPP_over_time(
    contMap = Ponerinae_contMap,
    ace = Ponerinae_ACE,
    tip_data = Ponerinae_cont_tip_data,
    trait_data_type = "continuous",
    BAMM_object = Ponerinae_BAMM_object_old_calib,
    # nb_time_steps = nb_time_steps,
    time_range = time_range,
    time_step_duration = time_step_duration,
    return_perm_data = TRUE,
    extract_trait_data_melted_df = TRUE,
    extract_diversification_data_melted_df = TRUE,
    return_STRAPP_results = TRUE,
    return_updated_trait_data_with_Map = TRUE,
    return_updated_BAMM_object = TRUE,
    verbose = TRUE,
    verbose_extended = TRUE) }

 ## Load directly trait data output
 data(Ponerinae_deepSTRAPP_cont_old_calib_0_40, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Explore output
 str(Ponerinae_deepSTRAPP_cont_old_calib_0_40, max.level = 1)

 # Display test summary
 # Can be passed down to [deepSTRAPP::plot_STRAPP_pvalues_over_time()] to generate a plot
 # showing the evolution of the test results across time.
 Ponerinae_deepSTRAPP_cont_old_calib_0_40$pvalues_summary_df

 # Access trait data in a melted data.frame
 head(Ponerinae_deepSTRAPP_cont_old_calib_0_40$trait_data_df_over_time)

 # Access the diversification data in a melted data.frame
 head(Ponerinae_deepSTRAPP_cont_old_calib_0_40$diversification_data_df_over_time)

 # Access deepSTRAPP results
 str(Ponerinae_deepSTRAPP_cont_old_calib_0_40$STRAPP_results, max.level = 2)

 ## Visualize updated phylogenies

 \donttest{ # (May take time to plot)
 # Plot updated contMap for time step n°2
 deepSTRAPP_outputs <- Ponerinae_deepSTRAPP_cont_old_calib_0_40
 contMap_step2 <- deepSTRAPP_outputs$updated_trait_data_with_Map_over_time[[2]]
 plot_contMap(contMap_step2$contMap, ftype = "off")

 # Plot diversification rates on updated phylogeny for time step n°2
 BAMM_object_step2 <- deepSTRAPP_outputs$updated_BAMM_objects_over_time[[2]]
 plot_BAMM_rates(BAMM_object = BAMM_object_step2,
    legend = TRUE, labels = FALSE,
    colorbreaks = BAMM_object_step2$initial_colorbreaks$net_diversification) }

 ## Visualize test results

 \donttest{ # (May take time to plot)
 # Plot p-values of Spearman tests across all time-steps
 plot_STRAPP_pvalues_over_time(
    deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cont_old_calib_0_40,
    alpha = 0.10)

 # Plot evolution of mean rates through time
 plot_rates_through_time(deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cont_old_calib_0_40)

 # Plot rates vs. trait values across branches for time step = 10 My
 plot_rates_vs_trait_data_for_focal_time(
    deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cont_old_calib_0_40,
    focal_time = 10)

 # Plot histogram of Spearman test stats for time step = 10 My
 plot_histogram_STRAPP_test_for_focal_time(
    deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cont_old_calib_0_40,
    focal_time = 10)

 # Plot histograms of Spearman test results (One plot per time-step)
 plot_histograms_STRAPP_tests_over_time(
    deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cont_old_calib_0_40,
    display_plots = TRUE)  }

 # ----- Example 2: Categorical trait ----- #

 ## Load data

 # Load trait df
 data(Ponerinae_trait_tip_data, package = "deepSTRAPP")
 # Load phylogeny
 data(Ponerinae_tree_old_calib, package = "deepSTRAPP")

 # Load the BAMM_object summarizing 1000 posterior samples of BAMM
 data(Ponerinae_BAMM_object_old_calib, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Prepare trait data

 # Extract categorical data with 3-levels
 Ponerinae_cat_3lvl_tip_data <- setNames(object = Ponerinae_trait_tip_data$fake_cat_3lvl_tip_data,
                                         nm = Ponerinae_trait_tip_data$Taxa)
 table(Ponerinae_cat_3lvl_tip_data)

 # Select color scheme for states
 colors_per_states <- c("forestgreen", "sienna", "goldenrod")
 names(colors_per_states) <- c("arboreal", "subterranean", "terricolous")

 \donttest{ # (May take several minutes to run)
 ## Produce densityMaps using stochastic character mapping based on an ARD Mk model
 Ponerinae_cat_3lvl_data_old_calib <- prepare_trait_data(
    tip_data = Ponerinae_cat_3lvl_tip_data,
    phylo = Ponerinae_tree_old_calib,
    trait_data_type = "categorical",
    colors_per_levels = colors_per_states,
    evolutionary_models = "ARD",
    nb_simulations = 100,
    return_best_model_fit = TRUE,
    return_model_selection_df = TRUE,
    plot_map = FALSE) }

  # Load directly trait data output
 data(Ponerinae_cat_3lvl_data_old_calib, package = "deepSTRAPP")

 ## Set for time steps of 5 My. Will generate deepSTRAPP workflows for 0 to 40 Mya.
 # nb_time_steps <- 5
 time_step_duration <- 5
 time_range <- c(0, 40)

 \donttest{ # (May take several minutes to run)
 ## Run deepSTRAPP on net diversification rates across time-steps.
 Ponerinae_deepSTRAPP_cat_3lvl_old_calib_0_40 <- run_deepSTRAPP_over_time(
    densityMaps = Ponerinae_cat_3lvl_data_old_calib$densityMaps,
    ace = Ponerinae_cat_3lvl_data_old_calib$ace,
    tip_data = Ponerinae_cat_3lvl_tip_data,
    trait_data_type = "categorical",
    BAMM_object = Ponerinae_BAMM_object_old_calib,
    # nb_time_steps = nb_time_steps,
    time_range = time_range,
    time_step_duration = time_step_duration,
    rate_type = "net_diversification",
    seed = 1234,
    alpha = 0.10, # Select a generous level of significance for the sake of the example
    posthoc_pairwise_tests = TRUE,
    return_perm_data = TRUE,
    extract_trait_data_melted_df = TRUE,
    extract_diversification_data_melted_df = TRUE,
    return_STRAPP_results = TRUE,
    return_updated_trait_data_with_Map = TRUE,
    return_updated_BAMM_object = TRUE,
    verbose = TRUE,
    verbose_extended = TRUE) }

 ## Load directly deepSTRAPP output
 data(Ponerinae_deepSTRAPP_cat_3lvl_old_calib_0_40, package = "deepSTRAPP")
 deepSTRAPP_outputs <- Ponerinae_deepSTRAPP_cat_3lvl_old_calib_0_40
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Explore output
 str(deepSTRAPP_outputs, max.level = 1)

 # Display test summaries
 # Can be passed down to [deepSTRAPP::plot_STRAPP_pvalues_over_time()] to generate a plot
 # showing the evolution of the test results across time.
 deepSTRAPP_outputs$pvalues_summary_df
 # Results for posthoc pairwise Dunn's tests over time-steps
 deepSTRAPP_outputs$pvalues_summary_df_for_posthoc_pairwise_tests

 # Access trait data in a melted data.frame
 head(deepSTRAPP_outputs$trait_data_df_over_time)

 # Access the diversification data in a melted data.frame
 head(deepSTRAPP_outputs$diversification_data_df_over_time)

 # Access details of deepSTRAPP results
 str(deepSTRAPP_outputs$STRAPP_results, max.level = 2)

 ## Visualize updated phylogenies

 \donttest{ # (May take time to plot)
 # Plot updated densityMaps for time step n°2 = 10My
 densityMaps_10My <- deepSTRAPP_outputs$updated_trait_data_with_Map_over_time[[2]]
 plot_densityMaps_overlay(densityMaps_10My$densityMaps)

 # Plot diversification rates on updated phylogeny for time step n°2
 BAMM_object_10My <- deepSTRAPP_outputs$updated_BAMM_objects_over_time[[2]]
 plot_BAMM_rates(BAMM_object = BAMM_object_10My,
    legend = TRUE, labels = FALSE,
    colorbreaks = BAMM_object_10My$initial_colorbreaks$net_diversification) }

 ## Visualize test results

 \donttest{ # (May take time to plot)
 # Plot p-values of overall Kruskal-Wallis test across all time-steps
 plot_STRAPP_pvalues_over_time(
    deepSTRAPP_outputs = deepSTRAPP_outputs,
    alpha = 0.10)

 # Plot p-values of post hoc pairwise Dunn's tests between pairs of tests across all time-steps
 plot_STRAPP_pvalues_over_time(
    deepSTRAPP_outputs = deepSTRAPP_outputs,
    alpha = 0.10,
    plot_posthoc_tests = TRUE)

 # Plot evolution of mean rates through time
 plot_rates_through_time(deepSTRAPP_outputs = deepSTRAPP_outputs,
    colors_per_levels = colors_per_states)

 # Plot rates vs. trait values across branches for time step n°2 = 10 My
 plot_rates_vs_trait_data_for_focal_time(
    deepSTRAPP_outputs = deepSTRAPP_outputs,
    focal_time = 10,
    colors_per_levels = colors_per_states)

 # Plot histogram of overall Kruskal-Wallis test for time step n°2 = 10 My
 plot_histogram_STRAPP_test_for_focal_time(
    deepSTRAPP_outputs = deepSTRAPP_outputs,
    focal_time = 10)

 # Plot histograms of overall Kruskal-Wallis test results across all time-steps
 # (One plot per time-step)
 plot_histograms_STRAPP_tests_over_time(
    deepSTRAPP_outputs = deepSTRAPP_outputs,
    plot_posthoc_tests = FALSE)

 # Plot histograms of post hoc pairwise Dunn's test results across all time-steps
 # (One multifaceted plot per time-step)
 plot_histograms_STRAPP_tests_over_time(
    deepSTRAPP_outputs = deepSTRAPP_outputs,
    plot_posthoc_tests = TRUE) }

 # ----- Example 3: Biogeographic ranges ----- #

 ## Load data

 # Load phylogeny
 data(Ponerinae_tree_old_calib, package = "deepSTRAPP")
 # Load trait df
 data(Ponerinae_binary_range_table, package = "deepSTRAPP")

 # Load the BAMM_object summarizing 1000 posterior samples of BAMM
 data(Ponerinae_BAMM_object_old_calib, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Prepare range data for Old World vs. New World

 # No overlap in ranges
 table(Ponerinae_binary_range_table$Old_World, Ponerinae_binary_range_table$New_World)

 Ponerinae_NO_data <- stats::setNames(object = Ponerinae_binary_range_table$Old_World,
                                      nm = Ponerinae_binary_range_table$Taxa)
 Ponerinae_NO_data <- as.character(Ponerinae_NO_data)
 Ponerinae_NO_data[Ponerinae_NO_data == "TRUE"] <- "O" # O = Old World
 Ponerinae_NO_data[Ponerinae_NO_data == "FALSE"] <- "N" # N = New World
 names(Ponerinae_NO_data) <- Ponerinae_binary_range_table$Taxa
 table(Ponerinae_NO_data)

 colors_per_ranges <- c("mediumpurple2", "peachpuff2")
 names(colors_per_ranges) <- c("N", "O")

 \donttest{ # (May take several minutes to run)
 ## Run evolutionary models
 Ponerinae_biogeo_data <- prepare_trait_data(
    tip_data = Ponerinae_NO_data,
    trait_data_type = "biogeographic",
    phylo = Ponerinae_tree_old_calib,
    evolutionary_models = "DEC+J", # Default = "DEC" for biogeographic
    BioGeoBEARS_directory_path = tempdir(), # Ex: "./BioGeoBEARS_directory/"
    keep_BioGeoBEARS_files = FALSE,
    prefix_for_files = "Ponerinae_old_calib",
    max_range_size = 2,
    split_multi_area_ranges = TRUE, # Set to TRUE to use only unique areas "A" and "B"
    nb_simulations = 100, # Reduce to save time (Default = '1000')
    colors_per_levels = colors_per_ranges,
    return_model_selection_df = TRUE,
    verbose = TRUE) }

 # Load directly output
 data(Ponerinae_biogeo_data_old_calib, package = "deepSTRAPP")

 ## Set for time steps of 5 My. Will generate deepSTRAPP workflows from 0 to 40 Mya.
 time_range <- c(0, 40)
 time_step_duration <- 10

 \donttest{ # (May take several minutes to run)
 ## Run deepSTRAPP on net diversification rates for time-steps = 0 to 40 Mya.
 Ponerinae_deepSTRAPP_biogeo_old_calib_0_40 <- run_deepSTRAPP_over_time(
    densityMaps = Ponerinae_biogeo_data_old_calib$densityMaps,
    ace = Ponerinae_biogeo_data_old_calib$ace,
    tip_data = Ponerinae_ON_tip_data,
    trait_data_type = "biogeographic",
    BAMM_object = Ponerinae_BAMM_object_old_calib,
    time_range = time_range,
    time_step_duration = time_step_duration,
    seed = 1234, # Set seed for reproducibility
    alpha = 0.10, # Select a generous level of significance for the sake of the example
    rate_type = "net_diversification",
    return_perm_data = TRUE,
    extract_trait_data_melted_df = TRUE,
    extract_diversification_data_melted_df = TRUE,
    return_STRAPP_results = TRUE,
    return_updated_trait_data_with_Map = TRUE,
    return_updated_BAMM_object = TRUE,
    verbose = TRUE,
    verbose_extended = TRUE) }

 ## Load directly output
 data(Ponerinae_deepSTRAPP_biogeo_old_calib_0_40, package = "deepSTRAPP")
 deepSTRAPP_outputs <- Ponerinae_deepSTRAPP_biogeo_old_calib_0_40
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Explore output
 str(deepSTRAPP_outputs, max.level = 1)

 # Display test summaries
 # Can be passed down to [deepSTRAPP::plot_STRAPP_pvalues_over_time()] to generate a plot
 # showing the evolution of the test results across time.
 deepSTRAPP_outputs$pvalues_summary_df

 # Access bioregeographic range data in a melted data.frame
 head(deepSTRAPP_outputs$trait_data_df_over_time)

 # Access the diversification data in a melted data.frame
 head(deepSTRAPP_outputs$diversification_data_df_over_time)

 # Access details of deepSTRAPP results
 str(deepSTRAPP_outputs$STRAPP_results, max.level = 2)

 ## Visualize updated phylogenies

 \donttest{ # (May take time to plot)
 # Plot updated densityMaps for time step n°2 = 10My
 densityMaps_10My <- deepSTRAPP_outputs$updated_trait_data_with_Map_over_time[[2]]
 plot_densityMaps_overlay(densityMaps_10My$densityMaps)

 # Plot diversification rates on updated phylogeny for time step n°2
 BAMM_object_10My <- deepSTRAPP_outputs$updated_BAMM_objects_over_time[[2]]
 plot_BAMM_rates(BAMM_object = BAMM_object_10My,
   legend = TRUE, labels = FALSE,
   colorbreaks = BAMM_object_10My$initial_colorbreaks$net_diversification) }

 ## Visualize test results

 \donttest{ # (May take time to plot)
 # Plot p-values of Mann-Whitney-Wilcoxon tests across all time-steps
 plot_STRAPP_pvalues_over_time(
    deepSTRAPP_outputs = deepSTRAPP_outputs,
    alpha = 0.05)

 # Plot evolution of mean rates through time
 plot_rates_through_time(deepSTRAPP_outputs = deepSTRAPP_outputs,
    colors_per_levels = colors_per_ranges)

 # Plot rates vs. trait values across branches for time step n°2 = 10 My
 plot_rates_vs_trait_data_for_focal_time(
    deepSTRAPP_outputs = deepSTRAPP_outputs,
    focal_time = 10,
    colors_per_levels = colors_per_ranges)

 # Plot histogram of Mann-Whitney-Wilcoxon test stats for time step n°2 = 10My
 plot_histogram_STRAPP_test_for_focal_time(
    deepSTRAPP_outputs = deepSTRAPP_outputs,
    focal_time = 10)

 # Plot histograms of Mann-Whitney-Wilcoxon test stats for all time-steps (One plot per time-step)
 plot_histograms_STRAPP_tests_over_time(
    deepSTRAPP_outputs = deepSTRAPP_outputs,
    display_plots = TRUE,
    plot_posthoc_tests = FALSE) }
}

}
\seealso{
To run the deepSTRAPP workflow for a single \code{focal_time}: \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}}
\code{\link[=extract_most_likely_trait_values_for_focal_time]{extract_most_likely_trait_values_for_focal_time()}} \code{\link[=update_rates_and_regimes_for_focal_time]{update_rates_and_regimes_for_focal_time()}}
\code{\link[=extract_diversification_data_melted_df_for_focal_time]{extract_diversification_data_melted_df_for_focal_time()}} \code{\link[=compute_STRAPP_test_for_focal_time]{compute_STRAPP_test_for_focal_time()}}

For a guided tutorial on complete deepSTRAPP workflow, see the associated vignettes:
\itemize{
\item For continuous trait data: \code{vignette("deepSTRAPP_continuous_data", package = "deepSTRAPP")}
\item For categorical trait data: \code{vignette("deepSTRAPP_categorical_3lvl_data", package = "deepSTRAPP")}
\item For biogeographic range data: \code{vignette("deepSTRAPP_biogeographic_data", package = "deepSTRAPP")}
}
}
\author{
Maël Doré
}
