#' Recycle objects to a given size
#'
#' The names of the `...` expressions, which should be variables within
#' the `.env` envrionment, are attempted to be recycled to the size specified
#' in the expression:
#' e.g., `name_of_object_to_recycle = size_to_recycle_to`. Expressions
#' are evaluated in the environment specified and objects are assigned back
#' into that same environment. The object recycling is from the
#' [vctrs](https://vctrs.r-lib.org/) package and thus stick to the
#' [vctrs recycling rules](https://vctrs.r-lib.org/reference/theory-faq-recycling.html).
#'
#' @param ... any number of named R expressions.
#' @param .env the environment to use for the evaluation of the recycling
#' expressions and the assignment of the recycled objects.
#' @param .error_call the call environment to use for error messages
#' (passed to [abort][rlang::abort]).
#' @return NULL, but objects named in `...` will be changed in the
#' `.env` environment specified.
#' @details See [favr::abort_if_not] for general validation,
#' [favr::recycle_if_not] for recycling, and [favr::enforce] and [favr::schema]
#' for non data-masked and data-masked validations, recycling and casting.
#' @examples
#' # NB: Some of these examples are expected to produce an error. To
#' #     prevent them from terminating a run with example() they are
#' #     piped into a call to try().
#'
#' x <- 1
#' recycle_if_not(x = 5)
#' length(x) # 5
#'
#' # recycle_if_not() follows `vctrs` recycling rules:
#' x <- c(1, 1)
#' recycle_if_not(x = 6) |> try()
#'
#' # Beware when using other objects as the size argument, e.g.:
#' x <- 1L
#' y <- c(1, 1, 1)
#' recycle_if_not(x = y) |> try()
#'
#' # When using other objects, call vctrs::vec_size() on them first:
#' recycle_if_not(x = vctrs::vec_size(y))
#' length(x) # 3
#'
#' # Changed objects are available immediately:
#' x <- y <- 1
#' recycle_if_not(x = 3, y = vctrs::vec_size(x))
#' cat(length(x), length(y), sep = ", ") # 3, 3
#'
#' myfunc <- function(x) {
#'   recycle_if_not(x = 3)
#'   length(x)
#' }
#' x <- 1L
#' myfunc(x) # x is recycled to length 3 within the function
#' length(x) # x is still scalar outside the function
#'
#' # The `.env` argument determines the expression and assignment
#' # environment:
#' x <- 1
#' e <- new.env()
#' e$x <- 1
#' recycle_if_not(x = 3, .env = e)
#' cat(
#'   "environment 'e'", length(e$x), "local environment", length(x),
#'   sep = ", "
#' ) # environment 'e', 3, local environment, 1
#'
#' # Named objects (lhs) are checked to be in the `.env` environment,
#' # throwing an error if not found:
#' x <- 1
#' e <- new.env()
#' recycle_if_not(x = 3, .env = e) |> try()
#'
#' # For expressions (rhs), the `.env` argument is preferentially
#' # chosen, but if not found then the normal R scoping rules
#' # apply:
#' x <- 3
#' e <- new.env()
#' e$z <- 1
#' recycle_if_not(z = x, .env = e)
#' length(e$z) # 3
#'
#' # The `.error_call` argument can be used to specify where the
#' # error occurs, by default this is the caller environment:
#' myfunc <- function(x) recycle_if_not(x = -5)
#' myfunc(1) |> try()
#'
#' #' # Injection can be used:
#' y <- 1L
#' x <- "y"
#' recycle_if_not(!!x := 5) |> try()
#' length(y) # 5
#'
#' y <- 1L
#' x <- list(y = 5)
#' recycle_if_not(!!!x)
#' length(y) # 5
#'
#' # Objects are reverted to their original values if an error
#' # occur:
#' x <- y <- 1L
#' recycle_if_not(x = 5, y = -5) |> try()
#' length(x) # 1
#' @export
recycle_if_not <- function(
    ...,
    .env = caller_env(),
    .error_call = caller_env()) {
  args <- enquos(...)
  caller_fn <- "recycle_if_not"

  check_env(.error_call, call = caller_env(), caller_fn = caller_fn)
  check_env(.env, call = .error_call, caller_fn = caller_fn)

  vars <- names2(args)
  check_args_named_and_vars_exist(
    vars,
    .env,
    action = "recycle",
    call = caller_env(),
    caller_fn = caller_fn
  )

  withCallingHandlers(
    for (i in seq_along(args)) {
      .favr_size_to_type <- eval_tidy(args[[i]], env = .env)

      collect_old_value(vars[i], .env)

      check_positive_scalar_integerish(.favr_size_to_type)

      # used to test first using vctrs::vec_size, but it is faster
      # to just use vec_recycle directly even if already that size
      vctrs_recycle_call <- call2(
        vec_recycle,
        sym(vars[i]),
        .favr_size_to_type,
        x_arg = vars[i]
      )

      assign(
        x = vars[i],
        value = eval_tidy(vctrs_recycle_call, env = .env),
        pos = .env
      )
    },
    error = function(cnd) {
      restore_old_values(.env)

      favr_error_handler(
        cnd = cnd,
        caller_fn = caller_fn,
        expr = paste(vars[i], "=", as_label(args[[i]])),
        call = .error_call
      )
    }
  )

  clean_favr_env_on_exit()
}

#--
