#' Plot Block‐Bootstrap Mann–Kendall Test Results
#'
#' Generates a histogram of bootstrapped Mann–Kendall S‐statistics with vertical
#' lines indicating the observed S‐statistic and confidence bounds.
#'
#' @param results List of BB‐MK test results generated by [eda_bbmk_test()].
#'
#' @param ... Optional named arguments: 'title', 'xlabel', and 'ylabel'.
#'
#' @return \code{ggplot}; A plot containing:
#'   - A gray histogram of the distribution of bootstrapped S‐statistics.
#'   - A red vertical line at the lower and upper confidence bounds.
#'   - A black vertical line at the observed S‐statistic.
#'
#' @examples
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#' results <- eda_bbmk_test(data, samples = 1000L)
#' plot_bbmk_test(results)
#'
#' @import ggplot2
#' @export

plot_bbmk_test <- function(results, ...) {

	# Capture optional arguments
	args <- list(...)

    # Override defaults if args is provided
    title <- args$title %||% "Block-Bootstrap Mann-Kendall Test" 
    xlabel <- args$xlabel %||% "S-Statistic"
    ylabel <- args$ylabel %||% "Frequency"

	# Generate the plot
	p1 <- ggplot() +
		geom_histogram(
			aes(x = results$bootstrap, color = "gray"), 
			fill = "lightgray",
			bins = 30
		)  +
		geom_vline(
			aes(xintercept = results$bounds, color = "red"),
			linewidth = 1.2
		) + 
		geom_vline(
			aes(xintercept = results$statistic, color = "black"),
			linewidth = 1.2
		) + 
		labs(
			title = title,
			x = xlabel,
			y = ylabel,
			color = "Legend"
		) + 
		scale_color_manual(
			values = c("gray" = "gray", "black" = "black", "red" = "red"),
			breaks = c("gray", "black", "red"),
			labels = c("Bootstrapped Statistics", "S-Statistic", "Confidence Bounds"),
		)

	# Return the plot with added theme
	add_theme(p1)

}
