\name{gensphere}
\alias{gensphere}
\alias{dgensphere}
\alias{rgensphere}
\title{Generalized spherical distribution definition, density, simulation
}
\description{Define a generalized spherical distribution by 
specifying a contour function, a radial density function, 
a radial simulation function, and a value of the density at the origin.
Once it is defined, compute density and simulate that distribution.
}
\usage{
gensphere(cfunc, dradial, rradial, g0)
dgensphere(x, gs.dist)
rgensphere(n, gs.dist)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{cfunc}{contour function object defined by \code{cfunc.new}, \code{cfunc.add.term} and \code{cfunc.finish}
}
  \item{dradial}{a function to evaluate the density for the radial component of distribution
}
  \item{rradial}{a function to simulate values of the radial distribution
}
  \item{g0}{g(0) = value of the multivariate density at the origin
}
  \item{x}{(d x n) matrix of point where the density is to be evaluated.  Columns x[,i] are vectors in d-space
}
  \item{gs.dist}{a generalized spherical distribution, an object returned by function \code{gensphere}
}
  \item{n}{number of values to generate
}
}
\details{A generalized spherical distribution is specified by calling function \code{gensphere} with
the contour function (defined via function \code{cfunc.new}, \code{cfunc.add.term} and \code{cfunc.finish}),
a function to compute the density of the radial term R, a runction to simulate from the radial term R,
and g(0)=the value of the density at the origin.  See the general representation of generalized spherical
laws in \link{gensphere-package}.  

If the distribution is d dimensional and the radial term is
a gamma distribution with shape=shape and scale=1,g(0)=0 if  d < shape, g(0)=cfunc$norm.const if d=shape,
\eqn{g(0)=\infty}{g(0)=Inf} if d > shape.  In general, 
\eqn{ g(0)=\lim_{r \to 0^+} r^{1-d} dradial(r)}{g(0)=lim_\{r -> 0+\} r^(1-d)*dradial(r)}. 

}
\value{\code{gensphere} returns an S3 object of class "gensphere.distribution" with components:
  \item{cfunc}{a contour function defined with \code{cfunc.new}, etc.}
  \item{dradial}{a function that evaluates the desnity of the radial component}
  \item{rradial}{a function that simulates values of the radial component}
  \item{g0}{g(0), the value of the multivariate density g(x) at the origin}
  
  \code{dgensphere} returns a numeric vector y that contains the value of the
  density of \bold{X}: y[i]=g(x[,i]), i=1,...,n.  Note that g(x) is the density of
  the vector \bold{X}, whereas \code{dradial} is the denis of the univariate radial term R.
  
  \code{rgensphere} returns a (d x n) matrix of simulated values of \bold{X}. Note that
  these values are an approximation to the distribution of X because the contour
  is approximated to a limited accuracy in \code{cfund.finish}.

Here are plots of the density surface and simulated points generated by the examples below.


\if{html}{
\figure{densitysurface.png}
\figure{rgensphere.png}
}
\if{latex}{
\figure{densitysurface.png}{options: width=2.5in}
\figure{rgensphere.png}{options: width=2.5in}
}

}
\seealso{
\code{\link{gensphere-package}}, \code{\link{cfunc.new}} 
}
\examples{

# define a diamond shaped contour
cfunc1 <- cfunc.new(d=2)
cfunc1 <- cfunc.add.term( cfunc1,"gen.lp.norm",k=c(1,1,2,0,0,1))
cfunc1 <- cfunc.finish( cfunc1 )
cfunc1

\donttest{
# define a generalized spherical distribution
rradial <- function( n ) { rgamma( n, shape=2 ) }
dradial <- function( x ) { dgamma( x, shape=2 ) }
dist1 <- gensphere( cfunc1, dradial, rradial, g0=cfunc1$norm.const ) 
dist1

# calculate density at a few points
dgensphere( x=matrix( c(0,0, 0,1, 0,2), nrow=2, ncol=3), dist1 )

# simulate values from the distribution
x <- rgensphere( 10000, dist1 )

# calculate and plot density surface on a grid
xy.grid <- seq(-3,3,.1)
if( interactive() ) {
  z <- gs.pdf2d.plot( dist1, xy.grid )
  title3d("density surface")
  plot(t(x),xlab="x",ylab="y",main="simulated points")
} 

}
}
