% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gs_update_ahr.R
\name{gs_update_ahr}
\alias{gs_update_ahr}
\title{Group sequential design using average hazard ratio under non-proportional hazards}
\usage{
gs_update_ahr(
  x = NULL,
  alpha = NULL,
  ustime = NULL,
  lstime = NULL,
  event_tbl = NULL
)
}
\arguments{
\item{x}{A design created by either \code{\link[=gs_design_ahr]{gs_design_ahr()}} or \code{\link[=gs_power_ahr]{gs_power_ahr()}}.}

\item{alpha}{Type I error for the updated design.}

\item{ustime}{Default is NULL in which case upper bound spending time is determined by timing.
Otherwise, this should be a vector of length k (total number of analyses)
with the spending time at each analysis.}

\item{lstime}{Default is NULL in which case lower bound spending time is determined by timing.
Otherwise, this should be a vector of length k (total number of analyses)
with the spending time at each analysis.}

\item{event_tbl}{A data frame with two columns: (1) analysis and (2) event,
which represents the events observed at each analysis per piecewise interval.
This can be defined via the \code{pw_observed_event()} function or manually entered.
For example, consider a scenario with two intervals in the piecewise model:
the first interval lasts 6 months with a hazard ratio (HR) of 1,
and the second interval follows with an HR of 0.6.
The data frame \code{event_tbl = data.frame(analysis = c(1, 1, 2, 2), event = c(30, 100, 30, 200))}
indicates that 30 events were observed during the delayed effect period,
130 events were observed at the IA, and 230 events were observed at the FA.}
}
\value{
A list with input parameters, enrollment rate, failure rate, analysis, and bound.
}
\description{
Group sequential design using average hazard ratio under non-proportional hazards
}
\examples{
library(gsDesign)
library(gsDesign2)

alpha <- 0.025
beta <- 0.1
ratio <- 1

# Enrollment
enroll_rate <- define_enroll_rate(
  duration = c(2, 2, 10),
  rate = (1:3) / 3)

# Failure and dropout
fail_rate <- define_fail_rate(
  duration = c(3, Inf), fail_rate = log(2) / 9,
  hr = c(1, 0.6), dropout_rate = .0001)

# IA and FA analysis time
analysis_time <- c(20, 36)

# Randomization ratio
ratio <- 1

# ------------------------------------------------- #
# Two-sided asymmetric design,
# beta-spending with non-binding lower bound
# ------------------------------------------------- #
# Original design
x <- gs_design_ahr(
  enroll_rate = enroll_rate, fail_rate = fail_rate,
  alpha = alpha, beta = beta, ratio = ratio,
  info_scale = "h0_info",
  info_frac = NULL, analysis_time = c(20, 36),
  upper = gs_spending_bound,
  upar = list(sf = sfLDOF, total_spend = alpha),
  test_upper = TRUE,
  lower = gs_spending_bound,
  lpar = list(sf = sfLDOF, total_spend = beta),
  test_lower = c(TRUE, FALSE),
  binding = FALSE) |> to_integer()

planned_event_ia <- x$analysis$event[1]
planned_event_fa <- x$analysis$event[2]


# Updated design with 190 events observed at IA,
# where 50 events observed during the delayed effect.
# IA spending = observed events / final planned events, the remaining alpha will be allocated to FA.
gs_update_ahr(
  x = x,
  ustime = c(190 / planned_event_fa, 1),
  lstime = c(190 / planned_event_fa, 1),
  event_tbl = data.frame(analysis = c(1, 1),
                         event = c(50, 140)))

# Updated design with 190 events observed at IA, and 300 events observed at FA,
# where 50 events observed during the delayed effect.
# IA spending = observed events / final planned events, the remaining alpha will be allocated to FA.
gs_update_ahr(
  x = x,
  ustime = c(190 / planned_event_fa, 1),
  lstime = c(190 / planned_event_fa, 1),
  event_tbl = data.frame(analysis = c(1, 1, 2, 2),
                         event = c(50, 140, 50, 250)))

# Updated design with 190 events observed at IA, and 300 events observed at FA,
# where 50 events observed during the delayed effect.
# IA spending = minimal of planned and actual information fraction spending
gs_update_ahr(
  x = x,
  ustime = c(min(190, planned_event_ia) / planned_event_fa, 1),
  lstime = c(min(190, planned_event_ia) / planned_event_fa, 1),
  event_tbl = data.frame(analysis = c(1, 1, 2, 2),
                         event = c(50, 140, 50, 250)))

# Alpha is updated to 0.05
gs_update_ahr(x = x, alpha = 0.05)
}
