% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_chem}
\alias{fmt_chem}
\title{Format chemical formulas}
\usage{
fmt_chem(data, columns = everything(), rows = everything())
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{columns}{\emph{Columns to target}

\code{\link[=rows-columns]{<column-targeting expression>}} // \emph{default:} \code{everything()}

Can either be a series of column names provided in \code{c()}, a vector of
column indices, or a select helper function (e.g. \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}} and \code{\link[=everything]{everything()}}).}

\item{rows}{\emph{Rows to target}

\code{\link[=rows-columns]{<row-targeting expression>}} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
undergo formatting. The default \code{\link[=everything]{everything()}} results in all rows in
\code{columns} being formatted. Alternatively, we can supply a vector of row
captions within \code{c()}, a vector of row indices, or a select helper
function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
\code{fmt_chem()} lets you format chemical formulas or even chemical
reactions in the table body. Often the input text will be in a common form
representing single compounds (like \code{"C2H4O"}, for acetaldehyde) but chemical
reactions can be used (e.g., \verb{2CH3OH -> CH3OCH3 + H2O"}). So long as the
text within the targeted cells conforms to \strong{gt}'s specialized chemistry
notation, the appropriate conversions will occur. Details pertaining to
chemistry notation can be found in the section entitled
\emph{How to use \strong{gt}'s chemistry notation}.
}
\section{How to use \strong{gt}'s chemistry notation}{


The chemistry notation involves a shorthand of writing chemical formulas and
chemical reactions, if needed. It should feel familiar in its basic usage and
the more advanced typesetting tries to limit the amount of syntax needed.
It's always best to show examples on usage:
\itemize{
\item \code{"CH3O2"} and \code{"(NH4)2S"} will render with subscripted numerals
\item Charges can be expressed with terminating \code{"+"} or \code{"-"}, as in \code{"H+"} and
\code{"[AgCl2]-"}; if any charges involve the use of a number, the following
incantations could be used: \code{"CrO4^2-"}, \code{"Fe^n+"}, \code{"Y^99+"}, \code{"Y^{99+}"}
(the final two forms produce equivalent output)
\item Stoichiometric values can be included with whole values prepending formulas
(e.g.,  \code{"2H2O2"}) or by setting them off with a space, like this:
\code{"2 H2O2"}, \code{"0.5 H2O"}, \code{"1/2 H2O"}, \code{"(1/2) H2O"}
\item Certain standalone, lowercase letters or combinations thereof will be
automatically stylized to fit conventions; \code{"NO_x"} and \code{"x Na(NH4)HPO4"}
will have italicized 'x' characters and you can always italicize letters
by surrounding with \code{"*"} (as in \code{"*n* H2O"} or \code{"*n*-C5H12"})
\item Chemical isotopes can be rendered using either of these two constructions
preceding an element: \code{"^{227}_{90}Th"} or \code{"^227_90Th"}; nuclides can
be represented in a similar manner, here are two examples:
\code{"^{0}_{-1}n^{-}"}, \code{"^0_-1n-"}
\item Chemical reactions can use \code{"+"} signs and a variety of reaction arrows:
(1) \code{"A -> B"}, (2) \code{"A <- B"}, (3) \code{"A <-> B"}, (4) \code{"A <--> B"}, (5)
\code{"A <=> B"}, (6) \code{"A <=>> B"}, or (7) \code{"A <<=> B"}
\item Center dots (useful in addition compounds) can be added by using a single
\code{"."} or \code{"*"} character, surrounded by spaces; here are two equivalent
examples \code{"KCr(SO4)2 . 12 H2O"} and \code{"KCr(SO4)2 * 12 H2O"}
\item Single and double bonds can be shown by inserting a \code{"-"} or \code{"="} between
adjacent characters (i.e., these shouldn't be at the beginning or end of
the markup); two examples: \code{"C6H5-CHO"}, \code{"CH3CH=CH2"}
\item as with units notation, Greek letters can be inserted by surrounding the
letter name with \code{":"}; here's an example that describes the delta value
of carbon-13: \code{":delta: ^13C"}
}
}

\section{Examples}{


Let's use the \code{\link{reactions}} dataset and create a new \strong{gt} table. The table
will be filtered down to only a few rows and columns. The column
\code{cmpd_formula} contains chemical formulas and the formatting of those will be
performed by \code{fmt_chem()}. Certain column labels with chemical names
(\code{o3_k298} and \code{no3_k298}) can be handled within \code{\link[=cols_label]{cols_label()}} by using
surrounding the text with \code{"{{\%"}/\code{"\%}}"}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{reactions |>
  dplyr::filter(cmpd_type == "terminal monoalkene") |>
  dplyr::filter(grepl("^1-", cmpd_name)) |>
  dplyr::select(cmpd_name, cmpd_formula, ends_with("k298")) |>
  gt() |>
  tab_header(title = "Gas-phase reactions of selected terminal alkenes") |>
  tab_spanner(
    label = "Reaction Rate Constant at 298 K",
    columns = ends_with("k298")
  ) |>
  fmt_chem(columns = cmpd_formula) |>
  fmt_scientific() |>
  sub_missing() |>
  cols_label(
    cmpd_name = "Alkene",
    cmpd_formula = "Formula",
    OH_k298 = "OH",
    O3_k298 = "\{\{\%O3\%\}\}",
    NO3_k298 = "\{\{\%NO3\%\}\}",
    Cl_k298 = "Cl"
  ) |>
  opt_align_table_header(align = "left")
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_chem_1.png" alt="This image of a table was generated from the first code example in the `fmt_chem()` help file." style="width:100\%;">
}}

Taking just a few rows from the \code{\link{photolysis}} dataset, let's create a new
\strong{gt} table. The \code{cmpd_formula} and \code{products} columns both contain text in
chemistry notation (the first has compounds, and the second column has the
products of photolysis reactions). These columns will be formatted by
\code{fmt_chem()}. The compound formulas will be merged with the compound
names with \code{\link[=cols_merge]{cols_merge()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{photolysis |>
  dplyr::filter(cmpd_name \%in\% c(
    "hydrogen peroxide", "nitrous acid",
    "nitric acid", "acetaldehyde",
    "methyl peroxide", "methyl nitrate",
    "ethyl nitrate", "isopropyl nitrate"
  )) |>
  dplyr::select(-c(l, m, n, quantum_yield, type)) |>
  gt() |>
  tab_header(title = "Photolysis pathways of selected VOCs") |>
  fmt_chem(columns = c(cmpd_formula, products)) |>
  cols_nanoplot(
    columns = sigma_298_cm2,
    columns_x_vals = wavelength_nm,
    expand_x = c(200, 400),
    new_col_name = "cross_section",
    new_col_label = "Absorption Cross Section",
    options = nanoplot_options(
      show_data_points = FALSE,
      data_line_stroke_width = 4,
      data_line_stroke_color = "black",
      show_data_area = FALSE
    )
  ) |>
  cols_merge(
    columns = c(cmpd_name, cmpd_formula),
    pattern = "\{1\}, \{2\}"
  ) |>
  cols_label(
    cmpd_name = "Compound",
    products = "Products"
  ) |>
  opt_align_table_header(align = "left")
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_chem_2.png" alt="This image of a table was generated from the second code example in the `fmt_chem()` help file." style="width:100\%;">
}}

\code{fmt_chem()} can handle the typesetting of nuclide notation.
Let's take a subset of columns and rows from the \code{\link{nuclides}} dataset and
make a new \strong{gt} table. The contents of the \code{nuclide} column contains
isotopes of hydrogen and carbon and this is placed in the table stub. Using
\code{fmt_chem()} makes it so that the subscripted and superscripted values are
properly formatted to the convention of formatting nuclides.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{nuclides |>
  dplyr::filter(element \%in\% c("H", "C")) |>
  dplyr::mutate(nuclide = gsub("[0-9]+$", "", nuclide)) |>
  dplyr::select(nuclide, atomic_mass, half_life, decay_1, is_stable) |>
  gt(rowname_col = "nuclide") |>
  tab_header(title = "Isotopes of Hydrogen and Carbon") |>
  tab_stubhead(label = "Isotope") |>
  fmt_chem(columns = nuclide) |>
  fmt_scientific(columns = half_life) |>
  fmt_number(
    columns = atomic_mass,
    decimals = 4,
    scale_by = 1 / 1e6
  ) |>
  sub_missing(
    columns = half_life,
    rows = is_stable,
    missing_text = md("**STABLE**")
  ) |>
  sub_missing(columns = half_life, rows = !is_stable) |>
  sub_missing(columns = decay_1) |>
  data_color(
    columns = decay_1,
    target_columns = c(atomic_mass, half_life, decay_1),
    palette = "LaCroixColoR::PassionFruit",
    na_color = "white"
  ) |>
  cols_label_with(fn = function(x) tools::toTitleCase(gsub("_", " ", x))) |>
  cols_label(decay_1 = "Decay Mode") |>
  cols_width(
    stub() ~ px(70),
    c(atomic_mass, half_life, decay_1) ~ px(120)
  ) |>
  cols_hide(columns = c(is_stable)) |>
  cols_align(align = "center", columns = decay_1) |>
  opt_align_table_header(align = "left") |>
  opt_vertical_padding(scale = 0.5)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_chem_3.png" alt="This image of a table was generated from the third code example in the `fmt_chem()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-20
}

\section{Function Introduced}{

\code{v0.11.0} (July 9, 2024)
}

\seealso{
Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_country}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_date}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_email}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_icon}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_tf}()},
\code{\link{fmt_time}()},
\code{\link{fmt_units}()},
\code{\link{fmt_url}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
