/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5VLmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5SLprivate.h" 
#include "H5VLpkg.h"     

typedef struct H5VL_dyn_op_t {
    char *op_name; 
    int   op_val;  
} H5VL_dyn_op_t;

static void H5VL__release_dyn_op(H5VL_dyn_op_t *dyn_op);

static int H5VL_opt_vals_g[H5VL_SUBCLS_TOKEN + 1] = {
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL, 
    H5VL_RESERVED_NATIVE_OPTIONAL  
};

static H5SL_t *H5VL_opt_ops_g[H5VL_SUBCLS_TOKEN + 1] = {
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
};

H5FL_DEFINE_STATIC(H5VL_dyn_op_t);

static void
H5VL__release_dyn_op(H5VL_dyn_op_t *dyn_op)
{
    FUNC_ENTER_PACKAGE_NOERR

    H5MM_xfree(dyn_op->op_name);
    H5FL_FREE(H5VL_dyn_op_t, dyn_op);

    FUNC_LEAVE_NOAPI_VOID
} 

static herr_t
H5VL__term_opt_operation_cb(void *_item, void H5_ATTR_UNUSED *key, void H5_ATTR_UNUSED *op_data)
{
    H5VL_dyn_op_t *item = (H5VL_dyn_op_t *)_item; 

    FUNC_ENTER_PACKAGE_NOERR

    
    H5VL__release_dyn_op(item);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5VL__term_opt_operation(void)
{
    size_t subcls; 

    FUNC_ENTER_PACKAGE_NOERR

    
    for (subcls = 0; subcls < NELMTS(H5VL_opt_vals_g); subcls++) {
        if (H5VL_opt_ops_g[subcls]) {
            H5SL_destroy(H5VL_opt_ops_g[subcls], H5VL__term_opt_operation_cb, NULL);
            H5VL_opt_ops_g[subcls] = NULL;
        }
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5VL__register_opt_operation(H5VL_subclass_t subcls, const char *op_name, int *op_val)
{
    H5VL_dyn_op_t *new_op;              
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(op_val);
    assert(op_name && *op_name);

    
    if (H5VL_opt_ops_g[subcls]) {
        if (NULL != H5SL_search(H5VL_opt_ops_g[subcls], op_name))
            HGOTO_ERROR(H5E_VOL, H5E_EXISTS, FAIL, "operation name already exists");
    } 
    else {
        
        if (NULL == (H5VL_opt_ops_g[subcls] = H5SL_create(H5SL_TYPE_STR, NULL)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTCREATE, FAIL, "can't create skip list for operations");
    } 

    
    if (NULL == (new_op = H5FL_CALLOC(H5VL_dyn_op_t)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTALLOC, FAIL, "can't allocate memory for dynamic operation info");
    if (NULL == (new_op->op_name = H5MM_strdup(op_name)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTALLOC, FAIL, "can't allocate name for dynamic operation info");
    new_op->op_val = H5VL_opt_vals_g[subcls]++;

    
    if (H5SL_insert(H5VL_opt_ops_g[subcls], new_op, new_op->op_name) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINSERT, FAIL, "can't insert operation info into skip list");

    
    *op_val = new_op->op_val;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5VL__num_opt_operation(void)
{
    size_t subcls;        
    size_t ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    for (subcls = 0; subcls < NELMTS(H5VL_opt_vals_g); subcls++)
        if (H5VL_opt_ops_g[subcls])
            ret_value += H5SL_count(H5VL_opt_ops_g[subcls]);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__find_opt_operation(H5VL_subclass_t subcls, const char *op_name, int *op_val)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(op_val);
    assert(op_name && *op_name);

    
    if (H5VL_opt_ops_g[subcls]) {
        H5VL_dyn_op_t *dyn_op; 

        
        if (NULL == (dyn_op = H5SL_search(H5VL_opt_ops_g[subcls], op_name)))
            HGOTO_ERROR(H5E_VOL, H5E_NOTFOUND, FAIL, "operation name isn't registered");

        
        *op_val = dyn_op->op_val;
    } 
    else
        HGOTO_ERROR(H5E_VOL, H5E_NOTFOUND, FAIL, "operation name isn't registered");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__unregister_opt_operation(H5VL_subclass_t subcls, const char *op_name)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(op_name && *op_name);

    
    if (H5VL_opt_ops_g[subcls]) {
        H5VL_dyn_op_t *dyn_op; 

        
        if (NULL == (dyn_op = H5SL_remove(H5VL_opt_ops_g[subcls], op_name)))
            HGOTO_ERROR(H5E_VOL, H5E_NOTFOUND, FAIL, "operation name isn't registered");

        
        H5VL__release_dyn_op(dyn_op);

        
        if (0 == H5SL_count(H5VL_opt_ops_g[subcls])) {
            if (H5SL_close(H5VL_opt_ops_g[subcls]) < 0)
                HGOTO_ERROR(H5E_VOL, H5E_CANTCLOSEOBJ, FAIL, "can't close dyn op skip list");
            H5VL_opt_ops_g[subcls] = NULL;
        } 
    }     
    else
        HGOTO_ERROR(H5E_VOL, H5E_NOTFOUND, FAIL, "operation name isn't registered");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
