% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hydra.R
\name{hydra}
\alias{hydra}
\title{Calculate hyperbolic embedding of distance data}
\usage{
hydra(D, dim = 2, curvature = 1, alpha = 1.1, equi.adj = 0.5,
  control = list())
}
\arguments{
\item{D}{a square symmetric matrix of distances (or dissimiliarities) to be embdedded, can also be a \code{\link[stats]{dist}} object}

\item{dim}{embedding dimension}

\item{curvature}{embedding curvature; if this argument is NULL, hydra tries to find the optimal curvature}

\item{alpha}{real number greater one; adjusts the hyperbolic curvature. Values larger than one yield a more distorted embedding where points are pushed
to the outer boundary (i.e. the ideal points) of hyperblic space. The interaction between \code{curvature} and \code{alpha} is non-linear.}

\item{equi.adj}{equi-angular adjustment; must be a real number between zero and one; only used if \code{dim} is 2. Value 0 means no ajustment, 1 adjusts
embedded data points such that their angular coordinates in the Poincare disc are uniformly distributed. Other values interpolate between the two extremes. Setting the parameter to non-zero values can make the embedding result look more harmoniuous in plots.}

\item{control}{a list which may contain the following boolean flags:
\itemize{
\item polar - return polar coordinates in dimension 2 (default: TRUE if \code{dim} is 2. This flag is ignored in higher dimension)
\item isotropic.adj - perform isotropic adjustment, ignoring Eigenvalues (default: TRUE if \code{dim} is 2, FALSE else)
\item return.lorentz - return raw Lorentz coordinates (before projection to hyperbolic space) (default: FALSE)
\item return.stress - return embedding stress (default: TRUE)
\item return.dist - return hyperbolic distance matrix of embedded points (default: FALSE)
\item use.eigs  - use \code{\link[RSpectra]{eigs}} function from \pkg{RSpectra} and \code{\link[Matrix]{norm}} function from \pkg{Matrix} to speed up computation (default: FALSE)
}}
}
\value{
A `hydra' object, which is a list with all or some of the following components:
          \describe{
          \item{r}{a vector containing the radial coordinates of the embedded points}
          \item{directional}{a matrix with \code{dim} columns containing as rows the directional coordinates of the embedded points}
          \item{theta}{a vector containing the angular coordinates of the embedded points (only returned if \code{dim} is 2 and \code{polar} flag  is TRUE)}
          \item{curvature}{the curvature used for the returned embedding}
          \item{dim}{the dimension used for the returned embedding}
          \item{stress}{the stress (i.e. the mean-square difference) between distances supplied in \code{D} and the hyperbolic distance matrix of the returned embedding}
          \item{dist}{the hyperbolic distance matrix of the returned embedding (only returned if flag \code{return.dist} is true. Computation may be time- and memory-intensive.)}
          \item{x0}{a vector containing the 'time-like' coordinate of the raw Lorentz embedding (only returned if flag \code{return.lorentz} is true)}
          \item{X}{a matrix with \code{dim} columns containing as rows the 'space-like' coordinate of the raw Lorentz embedding (only returned if flag \code{return.lorentz} is true)}
          }
}
\description{
Implements the HYDRA (hyperbolic distance recovery and approximation) method for embedding high-dimensional data points (represented by their distance matrix \code{D}) into low-dimensional hyperbolic space.
}
\details{
See \url{https://arxiv.org/abs/1903.08977} for more details.
}
\examples{
data(karate)
embedding <- hydra(karate$distance)
plot(embedding,labels=karate$label,lab.col=karate$group,graph.adj=karate$adjacency)

## Compare with Multidimensional scaling (MDS):
mds <- cmdscale(karate$distance) # Compute Euclidean embedding with MDS
mds.stress <- sqrt(sum((as.matrix(dist(mds)) - karate$distance)^2)) # Calculate embedding stress
c(embedding$stress,mds.stress) # Compare hyperbolic with Euclidean stress

}
\author{
Martin Keller-Ressel <martin.keller-ressel@tu-dresden.de>
}
