% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GradienBased.R
\name{IntegratedGradient}
\alias{IntegratedGradient}
\title{Integrated Gradients}
\description{
The \code{IntegratedGradient} class implements the method Integrated Gradients
(Sundararajan et al., 2017), which incorporates a reference value \eqn{x'}
(also known as baseline value) analogous to the \code{\link{DeepLift}} method.
Integrated Gradients helps to uncover the relative importance of input
features in the predictions \eqn{y = f(x)} made by a model compared to the
prediction of the reference value \eqn{y' = f(x')}. This is achieved through
the following formula:
\deqn{
(x - x') \times \int_{\alpha=0}^{1} \frac{\partial f(x' + \alpha (x - x'))}{\partial x} d\alpha
}
In simpler terms, it calculates how much each feature contributes to a
model's output by tracing a path from a baseline input \eqn{x'} to the actual
input \eqn{x} and measuring the average gradients along that path.

Similar to the other gradient-based methods, by default the integrated
gradient is multiplied by the input to get an approximate decomposition
of \eqn{y - y'}. However, with the parameter \code{times_input} only the gradient
describing the output sensitivity can be returned.

The R6 class can also be initialized using the \code{\link{run_intgrad}} function
as a helper function so that no prior knowledge of R6 classes is required.
}
\examples{
\dontshow{if (torch::torch_is_installed()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
#----------------------- Example 1: Torch ----------------------------------
library(torch)

# Create nn_sequential model and data
model <- nn_sequential(
  nn_linear(5, 12),
  nn_relu(),
  nn_linear(12, 2),
  nn_softmax(dim = 2)
)
data <- torch_randn(25, 5)
ref <- torch_randn(1, 5)

# Create Converter
converter <- convert(model, input_dim = c(5))

# Apply method IntegratedGradient
int_grad <- IntegratedGradient$new(converter, data, x_ref = ref)

# You can also use the helper function `run_intgrad` for initializing
# an R6 IntegratedGradient object
int_grad <- run_intgrad(converter, data, x_ref = ref)

# Print the result as a torch tensor for first two data points
get_result(int_grad, "torch.tensor")[1:2]

# Plot the result for both classes
plot(int_grad, output_idx = 1:2)

# Plot the boxplot of all datapoints and for both classes
boxplot(int_grad, output_idx = 1:2)

# ------------------------- Example 2: Neuralnet ---------------------------
if (require("neuralnet")) {
  library(neuralnet)
  data(iris)

  # Train a neural network
  nn <- neuralnet((Species == "setosa") ~ Petal.Length + Petal.Width,
    iris,
    linear.output = FALSE,
    hidden = c(3, 2), act.fct = "tanh", rep = 1
  )

  # Convert the model
  converter <- convert(nn)

  # Apply IntegratedGradient with a reference input of the feature means
  x_ref <- matrix(colMeans(iris[, c(3, 4)]), nrow = 1)
  int_grad <- run_intgrad(converter, iris[, c(3, 4)], x_ref = x_ref)

  # Get the result as a dataframe and show first 5 rows
  get_result(int_grad, type = "data.frame")[1:5, ]

  # Plot the result for the first datapoint in the data
  plot(int_grad, data_idx = 1)

  # Plot the result as boxplots
  boxplot(int_grad)
}
\dontshow{\}) # examplesIf}
\dontshow{if (torch::torch_is_installed() & Sys.getenv("INNSIGHT_EXAMPLE_KERAS", unset = 0) == 1) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# ------------------------- Example 3: Keras -------------------------------
if (require("keras") & keras::is_keras_available()) {
  library(keras)

  # Make sure keras is installed properly
  is_keras_available()

  data <- array(rnorm(10 * 32 * 32 * 3), dim = c(10, 32, 32, 3))

  model <- keras_model_sequential()
  model \%>\%
    layer_conv_2d(
      input_shape = c(32, 32, 3), kernel_size = 8, filters = 8,
      activation = "softplus", padding = "valid") \%>\%
    layer_conv_2d(
      kernel_size = 8, filters = 4, activation = "tanh",
      padding = "same") \%>\%
    layer_conv_2d(
      kernel_size = 4, filters = 2, activation = "relu",
      padding = "valid") \%>\%
    layer_flatten() \%>\%
    layer_dense(units = 64, activation = "relu") \%>\%
    layer_dense(units = 2, activation = "softmax")

  # Convert the model
  converter <- convert(model)

  # Apply the IntegratedGradient method with a zero baseline and n = 20
  # iteration steps
  int_grad <- run_intgrad(converter, data,
    channels_first = FALSE,
    n = 20
  )

  # Plot the result for the first image and both classes
  plot(int_grad, output_idx = 1:2)

  # Plot the pixel-wise median of the results
  plot_global(int_grad, output_idx = 1)
}
\dontshow{\}) # examplesIf}
\dontshow{if (torch::torch_is_installed() & Sys.getenv("RENDER_PLOTLY", unset = 0) == 1) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
#------------------------- Plotly plots ------------------------------------
if (require("plotly")) {
  # You can also create an interactive plot with plotly.
  # This is a suggested package, so make sure that it is installed
  library(plotly)
  boxplot(int_grad, as_plotly = TRUE)
}
\dontshow{\}) # examplesIf}
}
\references{
M. Sundararajan et al. (2017) \emph{Axiomatic attribution for deep networks.} ICML
2017, PMLR 70, pp. 3319-3328.
}
\seealso{
Other methods: 
\code{\link{ConnectionWeights}},
\code{\link{DeepLift}},
\code{\link{DeepSHAP}},
\code{\link{ExpectedGradient}},
\code{\link{Gradient}},
\code{\link{LIME}},
\code{\link{LRP}},
\code{\link{SHAP}},
\code{\link{SmoothGrad}}
}
\concept{methods}
\section{Super classes}{
\code{\link[innsight:InterpretingMethod]{innsight::InterpretingMethod}} -> \code{\link[innsight:GradientBased]{innsight::GradientBased}} -> \code{IntegratedGradient}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{n}}{(\code{integer(1)})\cr
Number of steps for the approximation of the integration path along
\eqn{\alpha} (default: \eqn{50}).\cr}

\item{\code{x_ref}}{(\code{list})\cr
The reference input for the IntegratedGradient method. This value is
stored as a list of \code{torch_tensor}s of shape \emph{(1, dim_in)} for each
input layer.\cr}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-IntegratedGradient-new}{\code{IntegratedGradient$new()}}
\item \href{#method-IntegratedGradient-clone}{\code{IntegratedGradient$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="innsight" data-topic="InterpretingMethod" data-id="get_result"><a href='../../innsight/html/InterpretingMethod.html#method-InterpretingMethod-get_result'><code>innsight::InterpretingMethod$get_result()</code></a></span></li>
<li><span class="pkg-link" data-pkg="innsight" data-topic="InterpretingMethod" data-id="plot"><a href='../../innsight/html/InterpretingMethod.html#method-InterpretingMethod-plot'><code>innsight::InterpretingMethod$plot()</code></a></span></li>
<li><span class="pkg-link" data-pkg="innsight" data-topic="InterpretingMethod" data-id="plot_global"><a href='../../innsight/html/InterpretingMethod.html#method-InterpretingMethod-plot_global'><code>innsight::InterpretingMethod$plot_global()</code></a></span></li>
<li><span class="pkg-link" data-pkg="innsight" data-topic="InterpretingMethod" data-id="print"><a href='../../innsight/html/InterpretingMethod.html#method-InterpretingMethod-print'><code>innsight::InterpretingMethod$print()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-IntegratedGradient-new"></a>}}
\if{latex}{\out{\hypertarget{method-IntegratedGradient-new}{}}}
\subsection{Method \code{new()}}{
Create a new instance of the \code{IntegratedGradient} R6 class. When
initialized, the method \emph{Integrated Gradient} is applied to the given
data and baseline value and the results are stored in the field \code{result}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{IntegratedGradient$new(
  converter,
  data,
  x_ref = NULL,
  n = 50,
  times_input = TRUE,
  channels_first = TRUE,
  output_idx = NULL,
  output_label = NULL,
  ignore_last_act = TRUE,
  verbose = interactive(),
  dtype = "float"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{converter}}{(\code{\link{Converter}})\cr
An instance of the \code{Converter} class that includes the
torch-converted model and some other model-specific attributes. See
\code{\link{Converter}} for details.\cr}

\item{\code{data}}{(\code{\link{array}}, \code{\link{data.frame}}, \code{\link[torch]{torch_tensor}} or \code{list})\cr
The data to which the method is to be applied. These must
have the same format as the input data of the passed model to the
converter object. This means either
\itemize{
\item an \code{array}, \code{data.frame}, \code{torch_tensor} or array-like format of
size \emph{(batch_size, dim_in)}, if e.g., the model has only one input layer, or
\item a \code{list} with the corresponding input data (according to the
upper point) for each of the input layers.\cr
}}

\item{\code{x_ref}}{(\code{\link{array}}, \code{\link{data.frame}}, \code{\link[torch]{torch_tensor}} or \code{list})\cr
The reference input for the IntegratedGradient method. This value
must have the same format as the input data of the passed model to the
converter object. This means either
\itemize{
\item an \code{array}, \code{data.frame}, \code{torch_tensor} or array-like format of
size \emph{(1, dim_in)}, if e.g., the model has only one input layer, or
\item a \code{list} with the corresponding input data (according to the upper point)
for each of the input layers.
\item It is also possible to use the default value \code{NULL} to take only
zeros as reference input.\cr
}}

\item{\code{n}}{(\code{integer(1)})\cr
Number of steps for the approximation of the integration path along
\eqn{\alpha} (default: \eqn{50}).\cr}

\item{\code{times_input}}{(\verb{logical(1})\cr
Multiplies the integrated gradients with the difference of the input
features and the baseline values. By default, the original definition of
IntegratedGradient is applied. However, by setting \code{times_input = FALSE}
only an approximation of the integral is calculated, which describes the
sensitivity of the features to the output.\cr}

\item{\code{channels_first}}{(\code{logical(1)})\cr
The channel position of the given data (argument
\code{data}). If \code{TRUE}, the channel axis is placed at the second position
between the batch size and the rest of the input axes, e.g.,
\code{c(10,3,32,32)} for a batch of ten images with three channels and a
height and width of 32 pixels. Otherwise (\code{FALSE}), the channel axis
is at the last position, i.e., \code{c(10,32,32,3)}. If the data
has no channel axis, use the default value \code{TRUE}.\cr}

\item{\code{output_idx}}{(\code{integer}, \code{list} or \code{NULL})\cr
These indices specify the output nodes for which
the method is to be applied. In order to allow models with multiple
output layers, there are the following possibilities to select
the indices of the output nodes in the individual output layers:
\itemize{
\item An \code{integer} vector of indices: If the model has only one output
layer, the values correspond to the indices of the output nodes, e.g.,
\code{c(1,3,4)} for the first, third and fourth output node. If there are
multiple output layers, the indices of the output nodes from the first
output layer are considered.
\item A \code{list} of \code{integer} vectors of indices: If the method is to be
applied to output nodes from different layers, a list can be passed
that specifies the desired indices of the output nodes for each
output layer. Unwanted output layers have the entry \code{NULL} instead of
a vector of indices, e.g., \code{list(NULL, c(1,3))} for the first and
third output node in the second output layer.
\item \code{NULL} (default): The method is applied to all output nodes in
the first output layer but is limited to the first ten as the
calculations become more computationally expensive for more output
nodes.\cr
}}

\item{\code{output_label}}{(\code{character}, \code{factor}, \code{list} or \code{NULL})\cr
These values specify the output nodes for which
the method is to be applied. Only values that were previously passed with
the argument \code{output_names} in the \code{converter} can be used. In order to
allow models with multiple
output layers, there are the following possibilities to select
the names of the output nodes in the individual output layers:
\itemize{
\item A \code{character} vector or \code{factor} of labels: If the model has only one output
layer, the values correspond to the labels of the output nodes named in the
passed \code{Converter} object, e.g.,
\code{c("a", "c", "d")} for the first, third and fourth output node if the
output names are \code{c("a", "b", "c", "d")}. If there are
multiple output layers, the names of the output nodes from the first
output layer are considered.
\item A \code{list} of \code{charactor}/\code{factor} vectors of labels: If the method is to be
applied to output nodes from different layers, a list can be passed
that specifies the desired labels of the output nodes for each
output layer. Unwanted output layers have the entry \code{NULL} instead of
a vector of labels, e.g., \code{list(NULL, c("a", "c"))} for the first and
third output node in the second output layer.
\item \code{NULL} (default): The method is applied to all output nodes in
the first output layer but is limited to the first ten as the
calculations become more computationally expensive for more output
nodes.\cr
}}

\item{\code{ignore_last_act}}{(\code{logical(1)})\cr
Set this logical value to include the last
activation functions for each output layer, or not (default: \code{TRUE}).
In practice, the last activation (especially for softmax activation) is
often omitted.\cr}

\item{\code{verbose}}{(\code{logical(1)})\cr
This logical argument determines whether a progress bar is
displayed for the calculation of the method or not. The default value is
the output of the primitive R function \code{\link[=interactive]{interactive()}}.\cr}

\item{\code{dtype}}{(\code{character(1)})\cr
The data type for the calculations. Use
either \code{'float'} for \code{\link[torch]{torch_float}} or \code{'double'} for
\code{\link[torch]{torch_double}}.\cr}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-IntegratedGradient-clone"></a>}}
\if{latex}{\out{\hypertarget{method-IntegratedGradient-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{IntegratedGradient$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
