% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sx2_fit.R
\name{sx2_fit}
\alias{sx2_fit}
\alias{sx2_fit.default}
\alias{sx2_fit.est_item}
\alias{sx2_fit.est_irt}
\title{S-X2 Fit Statistic}
\usage{
sx2_fit(x, ...)

\method{sx2_fit}{default}(
  x,
  data,
  D = 1,
  alpha = 0.05,
  min.collapse = 1,
  norm.prior = c(0, 1),
  nquad = 30,
  weights,
  pcm.loc = NULL,
  ...
)

\method{sx2_fit}{est_item}(
  x,
  alpha = 0.05,
  min.collapse = 1,
  norm.prior = c(0, 1),
  nquad = 30,
  weights,
  pcm.loc = NULL,
  ...
)

\method{sx2_fit}{est_irt}(
  x,
  alpha = 0.05,
  min.collapse = 1,
  norm.prior = c(0, 1),
  nquad = 30,
  weights,
  pcm.loc = NULL,
  ...
)
}
\arguments{
\item{x}{A data frame containing item metadata (e.g., item parameters, number
of categories, IRT model types, etc.); or an object of class \code{est_irt}
obtained from \code{\link[=est_irt]{est_irt()}}, or \code{est_item} from \code{\link[=est_item]{est_item()}}.

See \code{\link[=est_irt]{est_irt()}} or \code{\link[=simdat]{simdat()}} for more details about the item
metadata. This data frame can be easily created using the
\code{\link[=shape_df]{shape_df()}} function.}

\item{...}{Additional arguments passed to or from other methods.}

\item{data}{A matrix of examinees' item responses corresponding to the items
specified in the \code{x} argument. Rows represent examinees and columns
represent items.}

\item{D}{A scaling constant used in IRT models to make the logistic function
closely approximate the normal ogive function. A value of 1.7 is commonly
used for this purpose. Default is 1.}

\item{alpha}{A numeric value specifying the significance level (\eqn{\alpha})
for the hypothesis test associated with the \eqn{S\text{-}X^2} statistic.
Default is 0.05.}

\item{min.collapse}{An integer specifying the minimum expected frequency
required per cell before adjacent cells are collapsed. Default is 1. See
\strong{Details}.}

\item{norm.prior}{A numeric vector of length two specifying the mean and
standard deviation of the normal prior distribution. These values are used
to generate the Gaussian quadrature points and weights. Ignored if \code{method}
is \code{"ML"}, \code{"MLF"}, \code{"WL"}, or \code{"INV.TCC"}. Default is \code{c(0, 1)}.}

\item{nquad}{An integer specifying the number of Gaussian quadrature points
used to approximate the normal prior distribution. Default is 30.}

\item{weights}{A two-column matrix or data frame containing the quadrature
points (first column) and their corresponding weights (second column) for
the latent ability distribution. If omitted, default values are generated
using \code{\link[=gen.weight]{gen.weight()}} according to the \code{norm.prior} and \code{nquad} arguments.}

\item{pcm.loc}{An optional integer vector indicating the row indices of items
that follow the partial credit model (PCM), where slope parameters are
fixed. Default is \code{NULL}.}
}
\value{
A list containing the following components:
\item{fit_stat}{A data frame summarizing the \eqn{S\text{-}X^2} fit statistics
for all items, including the chi-square value, degrees of freedom, critical
value, and p-value.}
\item{item_df}{A data frame containing the item metadata as specified in the
input argument \code{x}.}
\item{exp_freq}{A list of collapsed expected frequency tables for all items.}
\item{obs_freq}{A list of collapsed observed frequency tables for all items.}
\item{exp_prob}{A list of collapsed expected probability tables for all items.}
\item{obs_prop}{A list of collapsed observed proportion tables for all items.}
}
\description{
Computes the \eqn{S\text{-}X^2} item fit statistic proposed by
Orlando and Thissen (2000, 2003). This statistic evaluates the fit of IRT
models by comparing observed and expected item response frequencies across
summed score groups.
}
\details{
The accuracy of the \eqn{\chi^{2}} approximation in item fit statistics can be
compromised when expected cell frequencies in contingency tables are too small
(Orlando & Thissen, 2000). To address this issue, Orlando and Thissen (2000)
proposed collapsing adjacent summed score groups to ensure a minimum expected
frequency of at least 1.

However, applying this collapsing approach directly to polytomous item data
can result in excessive information loss (Kang & Chen, 2008). To mitigate this,
Kang and Chen (2008) instead collapsed adjacent response categories \emph{within}
each summed score group, maintaining a minimum expected frequency of 1 per
category. The same collapsing strategies are implemented in \code{\link[=sx2_fit]{sx2_fit()}}.
If a different minimum expected frequency is desired, it can be specified via
the \code{min.collapse} argument.

When an item is labeled as "DRM" in the item metadata, it is treated as a 3PLM
item when computing the degrees of freedom for the \eqn{S\text{-}X^2} statistic.

Additionally, any missing responses in the \code{data} are automatically replaced
with incorrect responses (i.e., 0s).
}
\section{Methods (by class)}{
\itemize{
\item \code{sx2_fit(default)}: Default method for computing \eqn{S\text{-}X^{2}} fit
statistics from a data frame \code{x} containing item metadata.

\item \code{sx2_fit(est_item)}: An object created by the function \code{\link[=est_item]{est_item()}}.

\item \code{sx2_fit(est_irt)}: An object created by the function \code{\link[=est_irt]{est_irt()}}.

}}
\examples{
## Example 1: All five polytomous IRT items follow the GRM
## Import the "-prm.txt" output file from flexMIRT
flex_sam <- system.file("extdata", "flexmirt_sample-prm.txt", package = "irtQ")

# Select the item metadata
x <- bring.flexmirt(file = flex_sam, "par")$Group1$full_df

# Generate examinees' abilities from N(0, 1)
set.seed(23)
score <- rnorm(500, mean = 0, sd = 1)

# Simulate response data
data <- simdat(x = x, theta = score, D = 1)

\donttest{
# Compute fit statistics
fit1 <- sx2_fit(x = x, data = data, nquad = 30)

# Display fit statistics
fit1$fit_stat
}

## Example 2: Items 39 and 40 follow the GRM, and items 53, 54, and 55
##            follow the PCM (with slope parameters fixed to 1)
# Replace the model names with "GPCM" and
# set the slope parameters of items 53–55 to 1
x[53:55, 3] <- "GPCM"
x[53:55, 4] <- 1

# Generate examinees' abilities from N(0, 1)
set.seed(25)
score <- rnorm(1000, mean = 0, sd = 1)

# Simulate response data
data <- simdat(x = x, theta = score, D = 1)

\donttest{
# Compute fit statistics
fit2 <- sx2_fit(x = x, data = data, nquad = 30, pcm.loc = 53:55)

# Display fit statistics
fit2$fit_stat
}

}
\references{
Kang, T., & Chen, T. T. (2008). Performance of the generalized
S-X2 item fit index for polytomous IRT models.
\emph{Journal of Educational Measurement, 45}(4), 391-406.

Orlando, M., & Thissen, D. (2000). Likelihood-based item-fit indices for
dichotomous item response theory models.
\emph{Applied Psychological Measurement, 24}(1), 50-64.

Orlando, M., & Thissen, D. (2003). Further investigation of the performance
of S-X2: An item fit index for use with dichotomous item response theory
models. \emph{Applied Psychological Measurement, 27}(4), 289-298.
}
\seealso{
\code{\link[=irtfit]{irtfit()}}, \code{\link[=simdat]{simdat()}}, \code{\link[=shape_df]{shape_df()}},
\code{\link[=est_irt]{est_irt()}}, \code{\link[=est_item]{est_item()}}
}
\author{
Hwanggyu Lim \email{hglim83@gmail.com}
}
