% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/api_jt.R
\name{jt}
\alias{jt}
\alias{jt.charge}
\title{Junction Tree}
\usage{
jt(x, evidence = NULL, flow = "sum", propagate = "full")

\method{jt}{charge}(x, evidence = NULL, flow = "sum", propagate = "full")
}
\arguments{
\item{x}{An object return from \code{compile}}

\item{evidence}{A named vector. The names are the variabes and the elements
are the evidence}

\item{flow}{Either "sum" or "max"}

\item{propagate}{Either "no", "collect" or "full".}
}
\value{
A \code{jt} object
}
\description{
Construction of a junction tree and message passing
}
\details{
Evidence can be entered either at compile stage
or after compilation. Hence, one can also combine
evidence from before compilation with evidence
after compilation. Before refers to entering
evidence in the 'compile' function and after
refers to entering evidence in the 'jt' function.
}
\examples{

# Setting up the network
# ----------------------

library(igraph)
el <- matrix(c(
"A", "T",
"T", "E",
"S", "L",
"S", "B",
"L", "E",
"E", "X",
"E", "D",
"B", "D"),
 nc = 2,
 byrow = TRUE
)

g <- igraph::graph_from_edgelist(el)
plot(g)
# -----------------------

# Data
# ----
# We use the asia data; see the man page (?asia)

# Compilation
# -----------
cl <- cpt_list(asia, g) # Checking and conversion
cp <- compile(cl)

# After the network has been compiled, the graph has been triangulated and
# moralized. Furthermore, all conditional probability tables (CPTs) has been
# designated one of the cliques (in the triangulated and moralized graph).

# Example 1: sum-flow without evidence
# ------------------------------------
jt1 <- jt(cp)
plot(jt1)
print(jt1)
query_belief(jt1, c("E", "L", "T"))
query_belief(jt1, c("B", "D", "E"), type = "joint")


# Notice, that jt1 is equivalent to:
# jt1 <- jt(cp, propagate = "no")
# jt1 <- propagate(jt1, prop = "full")

# That is; it is possible to postpone the actual propagation
# In this setup, the junction tree is saved in the jt1 object,
# and one can repeadetly enter evidence for new observations
# using the set_evidence function on jt1 and then query
# several probabilites without repeadetly calculating the
# the junction tree over and over again. One just needs
# to use the propagate function on jt1.

# Example 2: sum-flow with evidence
# ---------------------------------

e2  <- c(A = "y", X = "n")
jt2 <- jt(cp, e2) 
query_belief(jt2, c("B", "D", "E"), type = "joint")

# Notice that, the configuration (D,E,B) = (y,y,n) has changed
# dramatically as a consequence of the evidence

# We can get the probability of the evidence:
query_evidence(jt2)

# Example 3: max-flow without evidence
# ------------------------------------
jt3 <- jt(cp, flow = "max")
mpe(jt3)


# Example 4: max-flow with evidence
# ---------------------------------
e4  <- c(T = "y", X = "y", D = "y")
jt4 <- jt(cp, e4, flow = "max")
mpe(jt4)

# Notice, that T, E, S, B, X and D has changed from "n" to "y"
# as a consequence of the new evidence e4


# Example 5: specifying a root node and only collect to save run time
# -------------------------------------------------------------------------

\donttest{
  cp5 <- compile(cpt_list(asia, g), root_node = "X")
  jt5 <- jt(cp5, propagate = "collect")
  query_belief(jt5, get_clique_root(jt5), "joint")
}

# We can only query from the variables in the root clique now
# but we have ensured that the node of interest, "X", does indeed live in
# this clique. The variables are found using 'get_clique_root'

# Example 6: Compiling from a list of conditional probabilities
# -------------------------------------------------------------------------

# * We need a list with CPTs which we extract from the asia2 object
#    - the list must be named with child nodes
#    - The elements need to be array-like objects

cl  <- cpt_list(asia2)
cp6 <- compile(cl)

# Inspection; see if the graph correspond to the cpts
# g <- get_graph(cp6)
# plot(g) 

# This time we specify that no propagation should be performed
jt6 <- jt(cp6, propagate = "no")

# We can now inspect the collecting junction tree and see which cliques
# are leaves and parents
plot(jt6)
get_cliques(jt6)
get_clique_root(jt6)

jt_leaves(jt6)
unlist(jt_parents(jt6))

# That is;
# - clique 2 is parent of clique 1
# - clique 3 is parent of clique 4 etc.

# Next, we send the messages from the leaves to the parents
jt6 <- send_messages(jt6)

# Inspect again
plot(jt6)

# Send the last message to the root and inspect
jt6 <- send_messages(jt6)
plot(jt6)

# The arrows are now reversed and the outwards (distribute) phase begins
jt_leaves(jt6)
jt_parents(jt6)

# Clique 2 (the root) is now a leave and it has 1, 3 and 6 as parents.

# Finishing the message passing
jt6 <- send_messages(jt6)
jt6 <- send_messages(jt6)

# Queries can now be performed as normal
query_belief(jt6, c("either", "tub"), "joint")

}
\seealso{
\code{\link{query_belief}}, \code{\link{mpe}},
\code{\link{get_cliques}}, \code{\link{get_clique_root}},
\code{\link{propagate}}
}
