## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----load packages------------------------------------------------------------
# Load required packages
library(tourr)
library(lionfish)
if (requireNamespace("flexclust")) {library(flexclust)}

## ----initialize python backend,  eval = FALSE---------------------------------
#  # Initialize python backend
#  if (check_venv()){
#  init_env(env_name = "r-lionfish", virtual_env = "virtual_env")
#  } else if (check_conda_env()){
#  init_env(env_name = "r-lionfish", virtual_env = "anaconda")
#  }

## ----Load and prepare flea data-----------------------------------------------
# Load dataset
data("flea")

# Prepare objects for later us
clusters_flea <- as.numeric(flea$species)
flea_subspecies <- unique(flea$species)

# Standardize data and calculate half_range
flea <- apply(flea[,1:6], 2, function(x) (x-mean(x))/sd(x))
feature_names_flea <- colnames(flea)
half_range_flea <- max(sqrt(rowSums(flea^2)))

## ----Load and prepare flea data winter activities data------------------------
if (requireNamespace("flexclust")) {
# Load dataset and set seed
data("winterActiv")
set.seed(42)

# Perform kmeans clustering 
clusters_winter <- stepcclust(winterActiv, k=6, nrep=20)
clusters_winter <- clusters_winter@cluster

# Get the names of our features
feature_names_winter <- colnames(winterActiv)
}

## ----1d tour------------------------------------------------------------------
guided_tour_flea_1d <- save_history(flea,
                                    tour_path=guided_tour(holes(),1))

obj_flea_1d_tour <- list(type="1d_tour",          # type of display
                         obj=guided_tour_flea_1d) # 1d tour history 

## ----2d tour------------------------------------------------------------------
grand_tour_flea_2d <- save_history(flea,
                                   tour_path = grand_tour(d=2))

obj_flea_2d_tour <- list(type="2d_tour",         # type of display
                         obj=grand_tour_flea_2d) # 2d tour history

## ----scatterplot--------------------------------------------------------------
obj_flea_scatter <- list(type="scatter",          # type of display
                         obj=c("tars1", "tars2")) # x and y axis of plot

## ----histogram----------------------------------------------------------------
obj_flea_histogram <- list(type="hist", # type of display
                           obj="head")  # x axis of histogram

## ----mosaic-------------------------------------------------------------------
obj_winter_mosaic <- list(type="mosaic",   # type of display
                  obj=c("subgroups_on_y")) # whether subgroups x or y axis

## ----heatmap------------------------------------------------------------------
obj_winter_heatmap <- list(type="heatmap",          # type of display
                           obj=c("Total fraction")) # initial metric

## ----categorical cluster interface--------------------------------------------
obj_winter_cat_clust <- list(type="cat_clust_interface", # type of display
                             obj=c("Total fraction"))    # initial metric

## ----interactive_tour flea----------------------------------------------------
# interactive_tour call of flea dataset. Insert plot objects of your liking.
if (interactive()){
interactive_tour(data=flea,
                 feature_names=feature_names_flea,
                 plot_objects=list(obj_flea_2d_tour),
                 half_range=half_range_flea,
                 n_plot_cols=2,
                 preselection=clusters_flea,
                 n_subsets=3,
                 preselection_names=flea_subspecies,
                 display_size=5)
}

## ----interactive_tour winterActiv---------------------------------------------
# interactive_tour call of winterActiv dataset. Insert plot objects of your liking.
if (interactive()){
interactive_tour(data=winterActiv,
                 feature_names=feature_names_winter,
                 plot_objects=list(obj_winter_cat_clust),
                 half_range=3,
                 n_plot_cols=2,
                 preselection=clusters_winter,
                 n_subsets=10,
                 display_size=5)
}

