#' Calculate Percentage Relative Efficiency (PRE)
#'
#' Computes the Percentage Relative Efficiency (PRE) of different MSE estimators
#' relative to the regression estimator MSE.
#'
#' @param result_all_mse A list containing MSE results from calculate_all_mse_neutrosophic()
#' @return A list containing PRE values for each estimator type:
#' \itemize{
#'   \item PRE_t0 - PRE for standard MSE estimator
#'   \item PRE_t1 - PRE for ratio-adjusted MSE estimator
#'   \item PRE_t2 - PRE for kurtosis-adjusted MSE estimator
#'   \item PRE_t3 - PRE for third estimator type (if available)
#'   \item PRE_exp - PRE for exponential MSE estimator
#'   \item PRE_r - Reference value (100) for regression estimator
#' }
#' @export
#' @examples
#' data(japan_neutro)
#' metrics <- compute_all_metrics(japan_neutro)
#' mse_results <- calculate_all_mse_neutrosophic(
#'   0.01, 0.02,
#'   metrics$mean_interval_Y[1], metrics$mean_interval_Y[2],
#'   metrics$mean_interval_X[1], metrics$mean_interval_X[2],
#'   metrics$cv_interval_X[1], metrics$cv_interval_X[2],
#'   metrics$cv_interval_Y[1], metrics$cv_interval_Y[2],
#'   metrics$correlation_results[1], metrics$correlation_results[2],
#'   metrics$kurtosis_interval_X[1], metrics$kurtosis_interval_X[2]
#' )
#' pre_results <- calculate_pre(mse_results)
#' print(pre_results)
calculate_pre <- function(result_all_mse) {
  if (!is.list(result_all_mse) ||
      !all(c("MSE", "MSE1", "MSE2", "MSE_exp", "MSE_r") %in% names(result_all_mse))) {
    stop("Input must be results from calculate_all_mse_neutrosophic()")
  }

  # Extract regression MSE values as reference
  mse_r_L <- result_all_mse$MSE_r["Lower"]
  mse_r_U <- result_all_mse$MSE_r["Upper"]

  # Calculate PRE for each estimator type
  pre_t0 <- c(
    Lower = (result_all_mse$MSE["Lower"] / mse_r_L) * 100,
    Upper = (result_all_mse$MSE["Upper"] / mse_r_U) * 100
  )

  pre_t1 <- c(
    Lower = (result_all_mse$MSE1["Lower"] / mse_r_L) * 100,
    Upper = (result_all_mse$MSE1["Upper"] / mse_r_U) * 100
  )

  pre_t2 <- c(
    Lower = (result_all_mse$MSE2["Lower"] / mse_r_L) * 100,
    Upper = (result_all_mse$MSE2["Upper"] / mse_r_U) * 100
  )
  pre_t3 <- c(
    Lower = (result_all_mse$MSE3["Lower"] / mse_r_L) * 100,
    Upper = (result_all_mse$MSE3["Upper"] / mse_r_U) * 100
  )
  pre_exp <- c(
    Lower = (result_all_mse$MSE_exp["Lower"] / mse_r_L) * 100,
    Upper = (result_all_mse$MSE_exp["Upper"] / mse_r_U) * 100
  )
  pre_r <- c(
    Lower = (result_all_mse$MSE_r["Lower"] / mse_r_L) * 100,
    Upper = (result_all_mse$MSE_r["Upper"] / mse_r_U) * 100
  )

  return(list(
    PRE_t0 = pre_t0,
    PRE_t1 = pre_t1,
    PRE_t2 = pre_t2,
    PRE_t3 = pre_t3,
    PRE_exp = pre_exp,
    PRE_r = pre_r
  ))
}
