#' Calculate All MSE Neutrosophic
#'
#' Computes various MSE estimates for neutrosophic interval data.
#' @param theta_L,theta_U Population parameters
#' @param Y_L,Y_U Study variable means
#' @param X_L,X_U Auxiliary variable means
#' @param Cx_L,Cx_U Auxiliary CVs
#' @param Cy_L,Cy_U Study CVs
#' @param rho_L,rho_U Correlations
#' @param B_L,B_U Kurtosis values
#' @return List of MSE estimates
#' @export
calculate_all_mse_neutrosophic <- function(
    theta_L, theta_U, Y_L, Y_U, X_L, X_U,
    Cx_L, Cx_U, Cy_L, Cy_U, rho_L, rho_U,
    B_L, B_U) {

  mse_L <- theta_L * (Y_L^2 * (Cy_L^2 + Cx_L^2 - 2 * Cx_L * Cy_L * rho_L))
  mse_U <- theta_U * (Y_U^2 * (Cy_U^2 + Cx_U^2 - 2 * Cx_U * Cy_U * rho_U))

  mse_L1 <- theta_L * (Y_L^2 * (Cy_L^2 + (X_L^2 / (X_L + Cx_L)^2) * Cx_L^2 - 2 * (X_L / (X_L + Cx_L)) * Cx_L * Cy_L * rho_L))
  mse_U1 <- theta_U * (Y_U^2 * (Cy_U^2 + (X_U^2 / (X_U + Cx_U)^2) * Cx_U^2 - 2 * (X_U / (X_U + Cx_U)) * Cx_U * Cy_U * rho_U))

  mse_L2 <- theta_L * (Y_L^2 * (Cy_L^2 + (X_L^2 / (X_L + B_L)^2) * Cx_L^2 - 2 * (X_L / (X_L + B_L)) * Cx_L * Cy_L * rho_L))
  mse_U2 <- theta_U * (Y_U^2 * (Cy_U^2 + (X_U^2 / (X_U + B_U)^2) * Cx_U^2 - 2 * (X_U / (X_U + B_U)) * Cx_U * Cy_U * rho_U))

  mse_L3 <- theta_L * (Y_L^2 * (Cy_L^2 + ((X_L*B_L) / (X_L*B_L+Cx_L))^2 * Cx_L^2 - 2 * ((X_L*B_L)/ (X_L*B_L+Cx_L)) * Cx_L * Cy_L * rho_L))
  mse_U3 <- theta_U * (Y_U^2 * (Cy_U^2 + ((X_U*B_U) / (X_U*B_U+Cx_U))^2 * Cx_U^2 - 2 * ((X_U*B_U)/ (X_U*B_U+Cx_U)) * Cx_U * Cy_U * rho_U))


  mse_exp_L <- theta_L * (Y_L^2 * (Cy_L^2 + (1 / 4) * (Cx_L^2) - (Cx_L * Cy_L * rho_L)))
  mse_exp_U <- theta_U * (Y_U^2 * (Cy_U^2 + (1 / 4) * (Cx_U^2) - (Cx_U * Cy_U * rho_U)))

mse_L1_r <- theta_L * (Y_L^2 * (Cy_L^2 * (1 - rho_L^2)))
mse_U1_r <- theta_U * (Y_U^2 * (Cy_U^2 * (1 - rho_U^2)))

list(
  MSE = c(Lower = mse_L, Upper = mse_U),
  MSE1 = c(Lower = mse_L1, Upper = mse_U1),
  MSE2 = c(Lower = mse_L2, Upper = mse_U2),
  MSE3 = c(Lower = mse_L3, Upper = mse_U3),
  MSE_exp = c(Lower = mse_exp_L, Upper = mse_exp_U),
  MSE_r = c(Lower = mse_L1_r, Upper = mse_U1_r)
)
}

#' Get User Inputs for Population Parameters
#'
#' Interactive function to get population parameters.
#' @return List with theta_L and theta_U
#' @export
get_user_inputs <- function() {
  N_L <- as.numeric(readline("Enter population size_L: "))
  N_U <- as.numeric(readline("Enter population size_U: "))
  n_L <- as.numeric(readline("Enter sample_size_L: "))
  n_U <- as.numeric(readline("Enter sample_size_U: "))

  .calculate_theta(N_L, N_U, n_L, n_U)
}
