% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions_ABC.R
\name{multipleABC}
\alias{multipleABC}
\title{Parameter estimation with Approximate Bayesian Computation for multiple
targets}
\usage{
multipleABC(
  targets,
  params,
  sumstats,
  limits,
  tol,
  method,
  parallel = FALSE,
  ncores = NA
)
}
\arguments{
\item{targets}{a matrix of observed summary statistics. Each row will be
considered a different target for parameter estimation. Each column should
be a different summary statistics and these statistics should correspond to
the statistics in the \code{sumstats} input.}

\item{params}{is a vector or matrix of simulated parameter values i.e.
numbers from the simulations. Each row or vector entry should be a
different simulation and each column of a matrix should be a different
parameter. This is the dependent variable for the regression, if a
regression step is performed.}

\item{sumstats}{is a vector or matrix of simulated summary statistics. Each
row or vector entry should be a different simulation and each column of a
matrix should be a different statistic. These act as the independent
variables if a regression step is performed.}

\item{limits}{is a matrix with two columns and as many rows as there are
parameters. Each row should contain the minimum value of the prior for a
given parameter in the first column and the maximum value in the second
column.}

\item{tol}{is the tolerance rate, indicating the required proportion of
points accepted nearest the target values.}

\item{method}{either "rejection" or "regression" indicating whether a
regression step should be performed during ABC parameter estimation.}

\item{parallel}{logical, indicating whether this function should be run using
parallel execution. The default setting is FALSE, meaning that this
function will utilize a single core.}

\item{ncores}{a non-negative integer that is required when \code{parallel} is
TRUE. It specifies the number of cores to use for parallel execution.}
}
\value{
the returned object is a list containing the following components:

\item{target}{parameter estimates obtained with the rejection sampling.}

\item{ss}{set of accepted summary statistics from the simulations.}

\item{unadjusted}{parameter estimates obtained with the rejection
sampling.}

\item{adjusted}{regression adjusted parameter values.}

\item{predmean}{estimates of the posterior mean for each parameter.}

\item{weights}{regression weights.}
}
\description{
Perform multivariate parameter estimation based on summary statistics using
an Approximate Bayesian Computation (ABC) algorithm. This function always
uses a rejection sampling algorithm while a local linear regression algorithm
might or might not be used.
}
\details{
To use this function, the usual steps of ABC parameter estimation have to be
performed. Briefly, data should have been simulated based on random draws
from the prior distributions of the parameters of interest and a set of
summary statistics should have been calculated from that data. The same set
of summary statistics should have been calculated from the observed data to
be used as the \code{targets} in this function. Parameter values are accepted if
the Euclidean distance between the set of summary statistics computed from
the simulated data and the set of summary statistics computed from the
observed data is sufficiently small. The percentage of accepted simulations
is determined by \code{tol}. This function performs a simple rejection by calling
the \code{\link[=rejABC]{rejABC()}} function.

When \code{method} is "regression", a local linear regression method is used to
correct for the imperfect match between the summary statistics computed from
the simulated data and the summary statistics computed from the observed
data. The output of the \code{\link[=rejABC]{rejABC()}} function is used as the input of the
\code{\link[=regABC]{regABC()}} function to apply this correction. The parameter values accepted
in the rejection step are weighted by a smooth function (kernel) of the
distance between the simulated and observed summary statistics and corrected
according to a linear transformation.

Please note that this functions performs parameter estimation for multiple
\code{targets}. The \code{targets} should contain multiple rows and each row will be
treated as an independent target for parameter estimation.
}
\examples{
# load the matrix with parameter values
data(params)
# load the matrix with simulated parameter values
data(sumstats)
# load the matrix with the prior limits
data(limits)

# select some random simulations to act as target just to test the function
targets <- sumstats[c(11:20) ,]
# we should remove those random simulation from the sumstats and params matrices
sumstats <- sumstats[-c(11:20), ]; params <- params[-c(11:20), ]

# parameter estimation for multiple targets
multipleABC(targets = targets, params = params, sumstats = sumstats, limits = limits,
tol = 0.01, method = "regression")

}
