% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/psymet_functions.R
\name{composite}
\alias{composite}
\title{Composite Reliability of a Score}
\usage{
composite(
  data,
  vrb.nm,
  level = 0.95,
  std = FALSE,
  ci.type = "delta",
  boot.ci.type = "bca.simple",
  R = 200L,
  fit.measures = c("chisq", "df", "tli", "cfi", "rmsea", "srmr"),
  se = "standard",
  test = "standard",
  missing = "fiml",
  ...
)
}
\arguments{
\item{data}{data.frame of data.}

\item{vrb.nm}{character vector of colnames in \code{data} specifying the set
of variables/items.}

\item{level}{double vector of length 1 with a value between 0 and 1
specifying what confidence level to use.}

\item{std}{logical element of length 1 specifying if the composite
reliability should be computed for the standardized version of the
variables \code{data[vrb.nm]}.}

\item{ci.type}{character vector of length 1 specifying which type of
confidence interval to compute. The "delta" option uses the delta method to
compute a standard error and a symmetrical confidence interval. The "boot"
option uses bootstrapping to compute an asymmetrical confidence interval as
well as a (pseudo) standard error.}

\item{boot.ci.type}{character vector of length 1 specifying which type of
bootstrapped confidence interval to compute. The options are: 1) "norm", 2)
"basic", 3) "perc", 4) "bca.simple". Only used if \code{ci.type} = "boot".
See \code{\link[lavaan]{parameterEstimates}} and its \code{boot.ci.type}
argument for details.}

\item{R}{integer vector of length 1 specifying how many bootstrapped
resamples to compute. Note, as the number of bootstrapped resamples
increases, the computation time will increase. Only used if \code{ci.type}
is "boot".}

\item{fit.measures}{character vector specifying which model fit indices to
include in the return object. The default option includes the chi-square
test statistic ("chisq"), degrees of freedom ("df"), tucker-lewis index
("tli"), comparative fit index ("cfi"), root mean square error of
approximation ("rmsea"), and standardized root mean residual ("srmr"). If
NULL, then no model fit indices are included in the return object. See
\code{\link[lavaan]{fitMeasures}} for details.}

\item{se}{character vector of length 1 specifying which type of standard
errors to compute. If ci.type = "boot", then the input value is ignored and
set to "bootstrap". See \code{\link[lavaan]{lavOptions}} and its \code{se}
argument for details.}

\item{test}{character vector of length 1 specifying which type of test
statistic to compute. If ci.type = "boot", then the input value is ignored
and set to "bootstrap". See \code{\link[lavaan]{lavOptions}} and its
\code{test} argument for details.}

\item{missing}{character vector of length 1 specifying how to handle missing
data. The default is "fiml" for full information maximum likelihood). See
\code{\link[lavaan]{lavOptions}} and its \code{missing} argument for
details.}

\item{...}{other arguments passed to \code{\link[lavaan]{cfa}}. Use at your
own peril as some argument values could cause the function to break.}
}
\value{
double vector where the first element is the composite reliability
  coefficient ("est") followed by its standard error ("se"), then its
  confidence interval ("lwr" and "upr"), the average standardized factor
  loading of the factor model ("average_l") and number of variables ("nvrb"),
  and finally any of the \code{fit.measures} requested.
}
\description{
\code{composite} computes the composite reliability coefficient (sometimes
referred to as omega) for a set of variables/items. The composite reliability
computed in \code{composite} assumes a undimensional factor model with no
error covariances. In addition to the coefficient itself, its standard error
and confidence interval are returned, the average standardized factor loading
from the factor model and number of variables/items, and (optional) model fit
indices of the factor model. Note, any reverse coded items need to be recoded
ahead of time so that all variables/items are keyed in the same direction.
}
\details{
The factor model is estimated using the R package \code{lavaan}. The
reliability coefficients are calculated based on the square of the sum of the
factor loadings divided by the sum of the square of the sum of the factors
loadings and the sum of the error variances (Raykov, 2001).

\code{composite} is only able to use the "ML" estimator at the moment and
cannot model items as categorical/ordinal. However, different versions of
standard errors and test statistics are possible. For example, the "MLM"
estimator can be specified by \code{se} = "robust.sem" and \code{test} =
"satorra.bentler"; the "MLR" estimator can be specified by \code{se} =
"robust.huber.white" and \code{test} = "yuan.bentler.mplus". See
\code{\link[lavaan]{lavOptions}} and scroll down to Estimation options.
}
\examples{

# data
dat <- psych::bfi[1:250, 2:5] # the first item is reverse coded

# delta method CI
composite(data = dat, vrb.nm = names(dat), ci.type = "delta")
composite(data = dat, vrb.nm = names(dat), ci.type = "delta", level = 0.99)
composite(data = dat, vrb.nm = names(dat), ci.type = "delta", std = TRUE)
composite(data = dat, vrb.nm = names(dat), ci.type = "delta", fit.measures = NULL)
composite(data = dat, vrb.nm = names(dat), ci.type = "delta",
   se = "robust.sem", test = "satorra.bentler", missing = "listwise") # MLM estimator
composite(data = dat, vrb.nm = names(dat), ci.type = "delta",
   se = "robust.huber.white", test = "yuan.bentler.mplus", missing = "fiml") # MLR estimator

\dontrun{
# bootstrapped CI
composite(data = dat, vrb.nm = names(dat), level = 0.95,
   ci.type = "boot") # slightly different estimate for some reason...
composite(data = dat, vrb.nm = names(dat), level = 0.95, ci.type = "boot",
   boot.ci.type = "perc", R = 250L) # probably want to use more resamples - this is just an example
}

# compare to semTools::reliability
psymet_obj <- composite(data = dat, vrb.nm = names(dat))
psymet_est <- unname(psymet_obj["est"])
lavaan_obj <- lavaan::cfa(model = make.latent(names(dat)), data = dat,
   std.lv = TRUE, missing = "fiml")
semTools_obj <- semTools::reliability(lavaan_obj)
semTools_est <- semTools_obj["omega", "latent"]
all.equal(psymet_est, semTools_est)

}
\references{
Raykov, T. (2001). Estimation of congeneric scale reliability using covariance
structure analysis with nonlinear constraints. British Journal of Mathematical
and Statistical Psychology, 54(2), 315–323.
}
\seealso{
\code{\link{composites}}
   \code{\link{cronbach}}
}
