#' Generate proc contents of data frame similar to that generated by SAS
#'
#' Pass in a data frame and the contents will be displayed.
#'
#' @param dsetin Incoming data frame to have proc contents generated.
#'
#' @return 'NULL' because the proc contents output will be displayed.
#'
#' @author Yongwei Wang, \email{yongwei.x.wang@viivhealthcare.com} \cr
#'         Chris Rook, \email{cr883296@gmail.com}
#'
#' @examples
#' library(repfun)
#' repfun::ru_contents(mtcars)
#'
#' @importFrom Hmisc label
#' @import haven
#' @export
#'
ru_contents <- function (dsetin) {
  if (typeof(dsetin) == "character") {
    if (base::file.exists(dsetin) && ! base::dir.exists(dsetin)) {
      df.1 <- haven::read_sas(dsetin)
      f_info <- base::file.info(dsetin)
      this_text <- unlist(strsplit(base::basename(dsetin), "\\."))[1]
      data_info <- list(
        DataSetName=this_text,
        MemberType="Data",
        Created=f_info$ctime,
        LastModified=f_info$mtime,
        Label=ifelse(is.null(base::attr(df.1, "label")), "", base::attr(df.1, "label")),
        Observations=nrow(df.1),
        Variables=length(names(df.1)),
        FileName=as.character(dsetin),
        FileSize=paste0(round(f_info$size/(10^6), digits=1), " MB"),
        FileSizeBytes=f_info$size
      )
    } else {
      return(-1)
    }
  } else {

    df.1 <- dsetin
    f.info <- NULL
    data_info <- list(
      DataSetName=base::deparse(base::substitute(dsetin)),
      MemberType="Data",
      Created=NA,
      LastModified=NA,
      Label=ifelse(is.null(base::attr(df.1, "label")), "", base::attr(df.1, "label")),
      Observations=base::nrow(df.1),
      Variables=length(names(df.1)),
      FileName=deparse(substitute(dsetin)),
      FileSize=paste0(round(utils::object.size(dsetin)/(10^6), digits=0), " MB"),
      FileSizeBytes=as.character(utils::object.size(dsetin))
    )
  }

  this_labels <-  list(
    DataSetName="Data Set Name",
    MemberType="Member Type",
    Created="Created",
    LastModified="Last Modified",
    Label="Label",
    Observations="Observations",
    Variables="Variables",
    FileName="Filename",
    FileSize="File Size",
    FileSizeBytes="File Size (bytes)"
  )

  data_info.1 <- as.data.frame(data_info)
  Hmisc::label(data_info.1) <- this_labels

  s_data_info <- sapply(data_info.1, function(x) {
    fmt_str <- base::sprintf("%-20s%-50s\n", Hmisc::label(x), x)
    return((fmt_str))
  })

  base::names(s_data_info) <- NULL

  n_nobs <- base::nrow(df.1)
  s_names <- base::names(df.1)
  fmt_str <- NULL
  fmt_str <- list()

  s_var_info <- sapply(seq_along(df.1), function(ind) {
    x <- df.1[[ind]]

    if (is.character(x)) {
      n_len <- base::max(nchar(x, type = "bytes"), na.rm = TRUE)  # Max string length
    } else {
      n_len <- base::floor(utils::object.size(x)/n_nobs)  # Memory usage for non-characters
    }
    s_class <- ""
    s_class.1 <- base::class(x)
    for (j in 1:length(s_class.1)) {
      if ((! s_class.1[j] %in% c("labeled", "numeric", "character")) & (! (base::typeof(x) == s_class.1))) s.class <- c(s.class, s_class.1)
    }
    s_fmt <- base::attr(x, "format")
    if (is.null(s_fmt)) s_fmt <-""
    fmt_str[[s_names[ind]]] <- base::sprintf("%3d %-10s%-11s%3d %-12s%-12s%-50s\n", ind, s_names[ind], typeof(x), n_len, s_class, s_fmt, Hmisc::label(x))
    return(fmt_str)
  })

  s_var_info_1 <-unlist(s_var_info[base::order(base::names(s_var_info), na.last = TRUE)])
  names(s_var_info_1) <- NULL

  fmt_str <- base::sprintf("\n\n%-4s %-10s%-11s%-3s %-12s%-12s%-50s\n", "  ", "Variable", "Type", "Len", "Class", "Format", "Label")

  return(list('s_data_info'=s_data_info,'fmt_str'=fmt_str,'s_var_info_1'=s_var_info_1))
}
