#' Get the current origin for each reuse
#'
#' @param as Whether to return the origin as `character` value (can
#' also be specified by passing `string` or `text`), as a `numeric`
#' value (can also be specified by passing `number`), or as `POSIX`
#' time (can also be specified by passing `time`).
#' @param suppressPrinting Whether to suppress printing the message
#' about how to store the origin in your R script.
#' @param format If returning `character`, the `format` to pass to
#' [base::format()] when formatting the time to a character value
#'
#' @returns The origin, in the format specified in `as`.
#' @export
#'
#' @examples squids::get_current_origin();
#'
#' squids::get_current_origin(
#'   as = "number"
#' );
#'
#' squids::get_current_origin(
#'   as = "time"
#' );
get_current_origin <- function(as = "time",
                               suppressPrinting = FALSE,
                               format = "%Y-%m-%d %H:%M:%S %Z") {

  origin <- Sys.time();

  if (!suppressPrinting) {

    cat0(
      "\U1F991 To human-readably store an origin so that\n",
      "\U1F991 you can reproduce the same sequence of SQUIDs,\n",
      "\U1F991 you can use the following line in your R script:\n\n",
      "SQUID_origin <- \"", format(origin, "%Y-%m-%d %H:%M:%S %Z"),
      "\";\n\n\U1F991 You can then pass `origin = SQUID_origin` when\n",
      "\U1F991 calling `squids::squids()`.\n\n"
    );

  }

  if (tolower(as) %in% c("character", "string", "text")) {
    return(
      format(origin, format)
    );
  } else if (tolower(as) %in% c("numeric", "number")) {
    return(
      as.numeric(origin)
    );
  } else if (tolower(as) %in% c("time", "posix")) {
    return(
      as.POSIXct(origin)
    );
  }
}
