#' @export

#' @title Bootstrap for stochcorr
#' @name stoch.bootstrap
#' @description `stoch.bootstrap` returns the Bootstrap confidence interval for estimated rho from `stochcorr`.
#' @usage stoch.bootstrap(est, S_1, S_2, boot_iter=500, p=0.05, seed = NULL)
#' @details This function returns a p% confidence interval for estimated rho from `stochcorr`
#' using parametric bootstrap. See section 4 of Majumdar and Laha (2024) <doi:10.48550/arXiv.2412.06343>.
#' @param est object of class `cbm` or `vmp`
#' @param S_1 historical price of the first asset
#' @param S_2 historical price of the second asset
#' @param boot_iter number of bootstrap iteration (Default is 500)
#' @param p 1-p% confidence interval (Default is 0.05)
#' @param seed (optional) seed value
#' @return Returns a matrix for the bootstrap (1-p)% confidence interval for rho. The first row of the matrix is the lower bound and the second
#' row is the upper bound.
#' @examples
#' \donttest{
#' library(stochcorr)
#'
#' data("nse2020")
#'
#' ## using von Mises process as the correlation process
#'
#' a <- stochcorr(nse2020$`USD/INR`, nse2020$Nifty, 1 / 250, corr_process = "vmp")
#' b <- stoch.bootstrap(a, nse2020$`USD/INR`, nse2020$Nifty, seed = 100)
#'
#' rho_data <- as.data.frame(cbind(a$rho, nse2020$Date))
#' rho_data[, 2] <- as.Date(rho_data[, 2], origin = "1970-01-01")
#' colnames(rho_data) <- c("Correlation", "Time")
#'
#' if(requireNamespace("ggplot2")){
#' library(ggplot2)
#' ggplot2::ggplot(rho_data, aes(x = Time, y = Correlation)) +
#'   theme_test() +
#'   theme(
#'     text = element_text(size = 15),
#'     axis.text.x = element_text(angle = 90, hjust = 1)
#'   ) +
#'   geom_line() +
#'   geom_ribbon(aes(ymin = b[1, ], ymax = b[2, ]), fill = "blue", alpha = 0.15) +
#'   scale_y_continuous(breaks = round(seq(-1, 1, by = 0.05), 1)) +
#'   scale_x_date(breaks = "1 month", date_labels = "%B %Y")
#'   }
#'
#' ## using Circular Brownian Motions as the correlation process
#'
#' a <- stochcorr(nse2020$`USD/INR`, nse2020$Nifty, 1 / 250, corr_process = "cbm")
#' b <- stoch.bootstrap(a, nse2020$`USD/INR`, nse2020$Nifty, seed = 100)
#'
#' rho_data <- as.data.frame(cbind(a$rho, nse2020$Date))
#' rho_data[, 2] <- as.Date(rho_data[, 2], origin = "1970-01-01")
#' colnames(rho_data) <- c("Correlation", "Time")
#'
#' if(requireNamespace("ggplot2")){
#' library(ggplot2)
#' ggplot2::ggplot(rho_data, aes(x = Time, y = Correlation)) +
#'   theme_test() +
#'   theme(
#'     text = element_text(size = 15),
#'     axis.text.x = element_text(angle = 90, hjust = 1)
#'   ) +
#'   geom_line() +
#'   geom_ribbon(aes(ymin = b[1, ], ymax = b[2, ]), fill = "blue", alpha = 0.15) +
#'   scale_y_continuous(breaks = round(seq(-1, 1, by = 0.05), 1)) +
#'   scale_x_date(breaks = "1 month", date_labels = "%B %Y")}
#'   }
#' @seealso [stochcorr()]

stoch.bootstrap <- function(est, S_1, S_2, boot_iter = 500, p = 0.05, seed = NULL) {
  if (inherits(est, "vmp")) {
    a <- stoch.bootstrap_vmp(est, S_1, S_2, boot_iter = boot_iter, p = p, seed = seed)
    return(a)
  } else if (inherits(est, "cbm")) {
    b <- stoch.bootstrap_cbm(a, S_1, S_2, boot_iter = boot_iter, p = p, seed = seed)
    return(b)
  } else {
    stop("incorrect est format")
  }
}

#' @export

#' @title Bootstrap for circdiff
#' @name circ.bootstrap
#' @description `stoch.bootstrap` returns the Bootstrap confidence interval for estimated parameters from `circdiff`.
#' @usage circ.bootstrap(est, theta, boot_iter=500, p=0.05, seed = NULL)
#' @details This function returns a (1-p)% confidence interval for estimated parameters from `circdiff`
#' using parametric bootstrap. See section 4 of Majumdar and Laha (2024) <doi:10.48550/arXiv.2412.06343>.
#' @param est object of class `cbm` or `vmp`
#' @param theta data of the discretely observed diffusion
#' @param boot_iter number of bootstrap iteration (Default is 500)
#' @param p 1-p% confidence interval (Default is 0.05)
#' @param seed (optional) seed value
#' @return Returns a matrix of the bootstrap (1-p)% confidence interval for the parameters. The first row is the lower bound and the second
#' row is the upper bound.
#' @examples
#' \donttest{
#' library(stochcorr)
#'
#'
#' data(wind)
#'
#' if(requireNamespace("ggplot2")){
#' library(ggplot2)
#' ggplot2::ggplot(wind, aes(x = Date, y = Dir)) +
#'   geom_line() +
#'   labs(title = "Sotavento Wind Farm",
#'        x = "Date",
#'        y = "Wind Direction") +
#'   scale_x_datetime(date_labels = "%d-%b", date_breaks = "2 day") +
#'   theme_test() +
#'   theme(
#'     text = element_text(size = 15),
#'     axis.text.x = element_text(angle = 90, hjust = 1)
#'   )
#' }
#'
#' a<-circdiff(wind$Dir,10/1440,"vmp")
#' a
#'
#' b<-circdiff(wind$Dir,10/1440,"cbm")
#' b
#'
#'
#'
#' estimates_vmp<-circ.bootstrap(a,wind$Dir, seed = 100)
#' estimates_vmp
#'
#' estimates_cbm<-circ.bootstrap(b,wind$Dir, seed = 100)
#' estimates_cbm}
#' @seealso [circdiff()]

circ.bootstrap <- function(est, theta, boot_iter = 500, p = 0.05, seed = NULL) {
  if (inherits(est, "vmp")) {
    a <- circ.bootstrap_vmp(est, theta, boot_iter = boot_iter, p = p, seed = seed)
    return(a)
  } else if (inherits(est,"cbm")) {
    b <- circ.bootstrap_cbm(est, theta, boot_iter = boot_iter, p = p, seed = seed)
    return(b)
  } else {
    stop("incorrect est format")
  }
}
