#' Plot Model-Free Tipping Point Results
#'
#' Visualizes the hazard ratios and confidence intervals across tipping point parameters
#' from model-free analyses, for both random sampling (% best event times) and deterministic
#' sampling (number of patients event-free at DCO). Highlights the tipping point where
#' the upper confidence interval crosses 1.
#'
#' @param tipse An S3 object of class `"tipse"` returned from \code{tipping_point_model_free()} or \code{tipping_point_model_based()}.
#'
#' @return A ggplot2 object showing HR and CI across tipping point parameters.
#' @keywords internal
#'
plot_tp <- function(tipse) {
  #----------------------------#
  # Input validation
  #----------------------------#
  summary_results <- tipse$imputation_results
  dat <- tipse$original_data
  arm <- tipse$arm_to_impute
  method <- tipse$method_to_impute

  control <- levels(dat[["TRT01P"]])[1]
  trt <- levels(dat[["TRT01P"]])[2]

  # Determine default x-axis label
  x_label <- switch(method,
    "random sampling" = ifelse(arm == control, paste("% best event times sampled from"),
      paste("% worst event times sampled from")
    ),
    "deterministic sampling" = ifelse(arm == control, paste("Number of ", arm, "subjects event-free at DCO"),
      paste("Number of ", arm, "subjects as events at censoring time")
    ),
    "hazard inflation" = paste("Hazard inflation factor (%) in", arm, "arm"),
    "hazard deflation" = paste("Hazard deflation factor (%) in", arm, "arm")
  )

  if (method == "hazard deflation") {
    summary_results$parameter <- (1 - summary_results$parameter) * 100
  }

  if (method == "hazard inflation") {
    summary_results$parameter <- (summary_results$parameter - 1) * 100
  }

  #----------------------------#
  # Identify tipping point
  #----------------------------#
  tipping_point <- summary_results$parameter[summary_results$tipping_point]

  if (method %in% c("random sampling", "hazard deflation")) {
    scale_x <- scale_x_reverse(
      name = x_label,
      breaks = seq(min(summary_results$parameter), max(summary_results$parameter), by = 10),
      labels = function(x) round(x)
    )
  } else {
    scale_x <- scale_x_continuous(
      name = x_label,
      breaks = seq(min(summary_results$parameter), max(summary_results$parameter), by = 10),
      labels = function(x) round(x)
    )
  }

  #----------------------------#
  # Build ggplot
  #----------------------------#
  p <- ggplot(summary_results, aes(x = parameter, y = HR)) +
    geom_point(size = 2) +
    geom_errorbar(
      aes(ymin = HR_lowerCI, ymax = HR_upperCI),
      width = 0.1 # controls horizontal width of the CI bar ends
    ) +
    geom_hline(yintercept = 1, linetype = "dashed", color = "black") +
    geom_vline(xintercept = tipping_point, linetype = "dashed", color = "red") +
    labs(
      x = x_label,
      y = "Hazard Ratio",
      title = "Tipping Point Analysis",
      subtitle = paste("Method:", method, "| Tipping point (HR_upperCL \u2265 1):", tipping_point)
    ) +
    scale_x +
    theme_minimal() +
    theme(
      plot.title = element_text(face = "bold"),
      plot.subtitle = element_text(color = "gray30"),
      plot.caption = element_text(hjust = 0),
      legend.position = "bottom",
      legend.title = element_text(hjust = 0.5)
    )


  return(p)
}
