use core::cmp::Ordering;
use core::num::FpCategory;
use core::ops::{Add, Div, Neg};

use core::f32;
use core::f64;

use crate::{Num, NumCast, ToPrimitive};




pub trait FloatCore: Num + NumCast + Neg<Output = Self> + PartialOrd + Copy {















    fn infinity() -> Self;
















    fn neg_infinity() -> Self;
















    fn nan() -> Self;


















    fn neg_zero() -> Self;
















    fn min_value() -> Self;
















    fn min_positive_value() -> Self;
















    fn epsilon() -> Self;
















    fn max_value() -> Self;


















    #[inline]
    #[allow(clippy::eq_op)]
    fn is_nan(self) -> bool {
        self != self
    }




















    #[inline]
    fn is_infinite(self) -> bool {
        self == Self::infinity() || self == Self::neg_infinity()
    }



















    #[inline]
    fn is_finite(self) -> bool {
        !(self.is_nan() || self.is_infinite())
    }



















    #[inline]
    fn is_normal(self) -> bool {
        self.classify() == FpCategory::Normal
    }






















    #[inline]
    fn is_subnormal(self) -> bool {
        self.classify() == FpCategory::Subnormal
    }























    fn classify(self) -> FpCategory;























    #[inline]
    fn floor(self) -> Self {
        let f = self.fract();
        if f.is_nan() || f.is_zero() {
            self
        } else if self < Self::zero() {
            self - f - Self::one()
        } else {
            self - f
        }
    }























    #[inline]
    fn ceil(self) -> Self {
        let f = self.fract();
        if f.is_nan() || f.is_zero() {
            self
        } else if self > Self::zero() {
            self - f + Self::one()
        } else {
            self - f
        }
    }






















    #[inline]
    fn round(self) -> Self {
        let one = Self::one();
        let h = Self::from(0.5).expect("Unable to cast from 0.5");
        let f = self.fract();
        if f.is_nan() || f.is_zero() {
            self
        } else if self > Self::zero() {
            if f < h {
                self - f
            } else {
                self - f + one
            }
        } else if -f < h {
            self - f
        } else {
            self - f - one
        }
    }























    #[inline]
    fn trunc(self) -> Self {
        let f = self.fract();
        if f.is_nan() {
            self
        } else {
            self - f
        }
    }























    #[inline]
    fn fract(self) -> Self {
        if self.is_zero() {
            Self::zero()
        } else {
            self % Self::one()
        }
    }





















    #[inline]
    fn abs(self) -> Self {
        if self.is_sign_positive() {
            return self;
        }
        if self.is_sign_negative() {
            return -self;
        }
        Self::nan()
    }
























    #[inline]
    fn signum(self) -> Self {
        if self.is_nan() {
            Self::nan()
        } else if self.is_sign_negative() {
            -Self::one()
        } else {
            Self::one()
        }
    }























    #[inline]
    fn is_sign_positive(self) -> bool {
        !self.is_sign_negative()
    }























    #[inline]
    fn is_sign_negative(self) -> bool {
        let (_, _, sign) = self.integer_decode();
        sign < 0
    }




















    #[inline]
    fn min(self, other: Self) -> Self {
        if self.is_nan() {
            return other;
        }
        if other.is_nan() {
            return self;
        }
        if self < other {
            self
        } else {
            other
        }
    }




















    #[inline]
    fn max(self, other: Self) -> Self {
        if self.is_nan() {
            return other;
        }
        if other.is_nan() {
            return self;
        }
        if self > other {
            self
        } else {
            other
        }
    }



























    fn clamp(self, min: Self, max: Self) -> Self {
        crate::clamp(self, min, max)
    }



















    #[inline]
    fn recip(self) -> Self {
        Self::one() / self
    }




















    #[inline]
    fn powi(mut self, mut exp: i32) -> Self {
        if exp < 0 {
            exp = exp.wrapping_neg();
            self = self.recip();
        }



        super::pow(self, (exp as u32).to_usize().unwrap())
    }


















    fn to_degrees(self) -> Self;


















    fn to_radians(self) -> Self;






















    fn integer_decode(self) -> (u64, i16, i8);
}

impl FloatCore for f32 {
    constant! {
        infinity() -> f32::INFINITY;
        neg_infinity() -> f32::NEG_INFINITY;
        nan() -> f32::NAN;
        neg_zero() -> -0.0;
        min_value() -> f32::MIN;
        min_positive_value() -> f32::MIN_POSITIVE;
        epsilon() -> f32::EPSILON;
        max_value() -> f32::MAX;
    }

    #[inline]
    fn integer_decode(self) -> (u64, i16, i8) {
        integer_decode_f32(self)
    }

    forward! {
        Self::is_nan(self) -> bool;
        Self::is_infinite(self) -> bool;
        Self::is_finite(self) -> bool;
        Self::is_normal(self) -> bool;
        Self::is_subnormal(self) -> bool;
        Self::clamp(self, min: Self, max: Self) -> Self;
        Self::classify(self) -> FpCategory;
        Self::is_sign_positive(self) -> bool;
        Self::is_sign_negative(self) -> bool;
        Self::min(self, other: Self) -> Self;
        Self::max(self, other: Self) -> Self;
        Self::recip(self) -> Self;
        Self::to_degrees(self) -> Self;
        Self::to_radians(self) -> Self;
    }

    #[cfg(feature = "std")]
    forward! {
        Self::floor(self) -> Self;
        Self::ceil(self) -> Self;
        Self::round(self) -> Self;
        Self::trunc(self) -> Self;
        Self::fract(self) -> Self;
        Self::abs(self) -> Self;
        Self::signum(self) -> Self;
        Self::powi(self, n: i32) -> Self;
    }

    #[cfg(all(not(feature = "std"), feature = "libm"))]
    forward! {
        libm::floorf as floor(self) -> Self;
        libm::ceilf as ceil(self) -> Self;
        libm::roundf as round(self) -> Self;
        libm::truncf as trunc(self) -> Self;
        libm::fabsf as abs(self) -> Self;
    }

    #[cfg(all(not(feature = "std"), feature = "libm"))]
    #[inline]
    fn fract(self) -> Self {
        self - libm::truncf(self)
    }
}

impl FloatCore for f64 {
    constant! {
        infinity() -> f64::INFINITY;
        neg_infinity() -> f64::NEG_INFINITY;
        nan() -> f64::NAN;
        neg_zero() -> -0.0;
        min_value() -> f64::MIN;
        min_positive_value() -> f64::MIN_POSITIVE;
        epsilon() -> f64::EPSILON;
        max_value() -> f64::MAX;
    }

    #[inline]
    fn integer_decode(self) -> (u64, i16, i8) {
        integer_decode_f64(self)
    }

    forward! {
        Self::is_nan(self) -> bool;
        Self::is_infinite(self) -> bool;
        Self::is_finite(self) -> bool;
        Self::is_normal(self) -> bool;
        Self::is_subnormal(self) -> bool;
        Self::clamp(self, min: Self, max: Self) -> Self;
        Self::classify(self) -> FpCategory;
        Self::is_sign_positive(self) -> bool;
        Self::is_sign_negative(self) -> bool;
        Self::min(self, other: Self) -> Self;
        Self::max(self, other: Self) -> Self;
        Self::recip(self) -> Self;
        Self::to_degrees(self) -> Self;
        Self::to_radians(self) -> Self;
    }

    #[cfg(feature = "std")]
    forward! {
        Self::floor(self) -> Self;
        Self::ceil(self) -> Self;
        Self::round(self) -> Self;
        Self::trunc(self) -> Self;
        Self::fract(self) -> Self;
        Self::abs(self) -> Self;
        Self::signum(self) -> Self;
        Self::powi(self, n: i32) -> Self;
    }

    #[cfg(all(not(feature = "std"), feature = "libm"))]
    forward! {
        libm::floor as floor(self) -> Self;
        libm::ceil as ceil(self) -> Self;
        libm::round as round(self) -> Self;
        libm::trunc as trunc(self) -> Self;
        libm::fabs as abs(self) -> Self;
    }

    #[cfg(all(not(feature = "std"), feature = "libm"))]
    #[inline]
    fn fract(self) -> Self {
        self - libm::trunc(self)
    }
}







#[cfg(any(feature = "std", feature = "libm"))]
pub trait Float: Num + Copy + NumCast + PartialOrd + Neg<Output = Self> {









    fn nan() -> Self;












    fn infinity() -> Self;












    fn neg_infinity() -> Self;













    fn neg_zero() -> Self;











    fn min_value() -> Self;











    fn min_positive_value() -> Self;
















    fn epsilon() -> Self {
        Self::from(f32::EPSILON).expect("Unable to cast from f32::EPSILON")
    }










    fn max_value() -> Self;













    fn is_nan(self) -> bool;



















    fn is_infinite(self) -> bool;


















    fn is_finite(self) -> bool;























    fn is_normal(self) -> bool;






















    #[inline]
    fn is_subnormal(self) -> bool {
        self.classify() == FpCategory::Subnormal
    }
















    fn classify(self) -> FpCategory;












    fn floor(self) -> Self;












    fn ceil(self) -> Self;













    fn round(self) -> Self;












    fn trunc(self) -> Self;














    fn fract(self) -> Self;



















    fn abs(self) -> Self;


















    fn signum(self) -> Self;



















    fn is_sign_positive(self) -> bool;



















    fn is_sign_negative(self) -> bool;



















    fn mul_add(self, a: Self, b: Self) -> Self;










    fn recip(self) -> Self;













    fn powi(self, n: i32) -> Self;











    fn powf(self, n: Self) -> Self;
















    fn sqrt(self) -> Self;















    fn exp(self) -> Self;













    fn exp2(self) -> Self;















    fn ln(self) -> Self;


















    fn log(self, base: Self) -> Self;













    fn log2(self) -> Self;













    fn log10(self) -> Self;












    #[inline]
    fn to_degrees(self) -> Self {
        let halfpi = Self::zero().acos();
        let ninety = Self::from(90u8).unwrap();
        self * ninety / halfpi
    }












    #[inline]
    fn to_radians(self) -> Self {
        let halfpi = Self::zero().acos();
        let ninety = Self::from(90u8).unwrap();
        self * halfpi / ninety
    }











    fn max(self, other: Self) -> Self;











    fn min(self, other: Self) -> Self;














    fn clamp(self, min: Self, max: Self) -> Self {
        crate::clamp(self, min, max)
    }


















    fn abs_sub(self, other: Self) -> Self;













    fn cbrt(self) -> Self;















    fn hypot(self, other: Self) -> Self;













    fn sin(self) -> Self;













    fn cos(self) -> Self;












    fn tan(self) -> Self;
















    fn asin(self) -> Self;
















    fn acos(self) -> Self;














    fn atan(self) -> Self;




























    fn atan2(self, other: Self) -> Self;

















    fn sin_cos(self) -> (Self, Self);














    fn exp_m1(self) -> Self;















    fn ln_1p(self) -> Self;

















    fn sinh(self) -> Self;

















    fn cosh(self) -> Self;

















    fn tanh(self) -> Self;













    fn asinh(self) -> Self;













    fn acosh(self) -> Self;














    fn atanh(self) -> Self;




















    fn integer_decode(self) -> (u64, i16, i8);






















    fn copysign(self, sign: Self) -> Self {
        if self.is_sign_negative() == sign.is_sign_negative() {
            self
        } else {
            self.neg()
        }
    }
}

#[cfg(feature = "std")]
macro_rules! float_impl_std {
    ($T:ident $decode:ident) => {
        impl Float for $T {
            constant! {
                nan() -> $T::NAN;
                infinity() -> $T::INFINITY;
                neg_infinity() -> $T::NEG_INFINITY;
                neg_zero() -> -0.0;
                min_value() -> $T::MIN;
                min_positive_value() -> $T::MIN_POSITIVE;
                epsilon() -> $T::EPSILON;
                max_value() -> $T::MAX;
            }

            #[inline]
            #[allow(deprecated)]
            fn abs_sub(self, other: Self) -> Self {
                <$T>::abs_sub(self, other)
            }

            #[inline]
            fn integer_decode(self) -> (u64, i16, i8) {
                $decode(self)
            }

            forward! {
                Self::is_nan(self) -> bool;
                Self::is_infinite(self) -> bool;
                Self::is_finite(self) -> bool;
                Self::is_normal(self) -> bool;
                Self::is_subnormal(self) -> bool;
                Self::classify(self) -> FpCategory;
                Self::clamp(self, min: Self, max: Self) -> Self;
                Self::floor(self) -> Self;
                Self::ceil(self) -> Self;
                Self::round(self) -> Self;
                Self::trunc(self) -> Self;
                Self::fract(self) -> Self;
                Self::abs(self) -> Self;
                Self::signum(self) -> Self;
                Self::is_sign_positive(self) -> bool;
                Self::is_sign_negative(self) -> bool;
                Self::mul_add(self, a: Self, b: Self) -> Self;
                Self::recip(self) -> Self;
                Self::powi(self, n: i32) -> Self;
                Self::powf(self, n: Self) -> Self;
                Self::sqrt(self) -> Self;
                Self::exp(self) -> Self;
                Self::exp2(self) -> Self;
                Self::ln(self) -> Self;
                Self::log(self, base: Self) -> Self;
                Self::log2(self) -> Self;
                Self::log10(self) -> Self;
                Self::to_degrees(self) -> Self;
                Self::to_radians(self) -> Self;
                Self::max(self, other: Self) -> Self;
                Self::min(self, other: Self) -> Self;
                Self::cbrt(self) -> Self;
                Self::hypot(self, other: Self) -> Self;
                Self::sin(self) -> Self;
                Self::cos(self) -> Self;
                Self::tan(self) -> Self;
                Self::asin(self) -> Self;
                Self::acos(self) -> Self;
                Self::atan(self) -> Self;
                Self::atan2(self, other: Self) -> Self;
                Self::sin_cos(self) -> (Self, Self);
                Self::exp_m1(self) -> Self;
                Self::ln_1p(self) -> Self;
                Self::sinh(self) -> Self;
                Self::cosh(self) -> Self;
                Self::tanh(self) -> Self;
                Self::asinh(self) -> Self;
                Self::acosh(self) -> Self;
                Self::atanh(self) -> Self;
                Self::copysign(self, sign: Self) -> Self;
            }
        }
    };
}

#[cfg(all(not(feature = "std"), feature = "libm"))]
macro_rules! float_impl_libm {
    ($T:ident $decode:ident) => {
        constant! {
            nan() -> $T::NAN;
            infinity() -> $T::INFINITY;
            neg_infinity() -> $T::NEG_INFINITY;
            neg_zero() -> -0.0;
            min_value() -> $T::MIN;
            min_positive_value() -> $T::MIN_POSITIVE;
            epsilon() -> $T::EPSILON;
            max_value() -> $T::MAX;
        }

        #[inline]
        fn integer_decode(self) -> (u64, i16, i8) {
            $decode(self)
        }

        #[inline]
        fn fract(self) -> Self {
            self - Float::trunc(self)
        }

        #[inline]
        fn log(self, base: Self) -> Self {
            self.ln() / base.ln()
        }

        forward! {
            Self::is_nan(self) -> bool;
            Self::is_infinite(self) -> bool;
            Self::is_finite(self) -> bool;
            Self::is_normal(self) -> bool;
            Self::is_subnormal(self) -> bool;
            Self::clamp(self, min: Self, max: Self) -> Self;
            Self::classify(self) -> FpCategory;
            Self::is_sign_positive(self) -> bool;
            Self::is_sign_negative(self) -> bool;
            Self::min(self, other: Self) -> Self;
            Self::max(self, other: Self) -> Self;
            Self::recip(self) -> Self;
            Self::to_degrees(self) -> Self;
            Self::to_radians(self) -> Self;
        }

        forward! {
            FloatCore::signum(self) -> Self;
            FloatCore::powi(self, n: i32) -> Self;
        }
    };
}

fn integer_decode_f32(f: f32) -> (u64, i16, i8) {
    let bits: u32 = f.to_bits();
    let sign: i8 = if bits >> 31 == 0 { 1 } else { -1 };
    let mut exponent: i16 = ((bits >> 23) & 0xff) as i16;
    let mantissa = if exponent == 0 {
        (bits & 0x7fffff) << 1
    } else {
        (bits & 0x7fffff) | 0x800000
    };

    exponent -= 127 + 23;
    (mantissa as u64, exponent, sign)
}

fn integer_decode_f64(f: f64) -> (u64, i16, i8) {
    let bits: u64 = f.to_bits();
    let sign: i8 = if bits >> 63 == 0 { 1 } else { -1 };
    let mut exponent: i16 = ((bits >> 52) & 0x7ff) as i16;
    let mantissa = if exponent == 0 {
        (bits & 0xfffffffffffff) << 1
    } else {
        (bits & 0xfffffffffffff) | 0x10000000000000
    };

    exponent -= 1023 + 52;
    (mantissa, exponent, sign)
}

#[cfg(feature = "std")]
float_impl_std!(f32 integer_decode_f32);
#[cfg(feature = "std")]
float_impl_std!(f64 integer_decode_f64);

#[cfg(all(not(feature = "std"), feature = "libm"))]
impl Float for f32 {
    float_impl_libm!(f32 integer_decode_f32);

    #[inline]
    #[allow(deprecated)]
    fn abs_sub(self, other: Self) -> Self {
        libm::fdimf(self, other)
    }

    forward! {
        libm::floorf as floor(self) -> Self;
        libm::ceilf as ceil(self) -> Self;
        libm::roundf as round(self) -> Self;
        libm::truncf as trunc(self) -> Self;
        libm::fabsf as abs(self) -> Self;
        libm::fmaf as mul_add(self, a: Self, b: Self) -> Self;
        libm::powf as powf(self, n: Self) -> Self;
        libm::sqrtf as sqrt(self) -> Self;
        libm::expf as exp(self) -> Self;
        libm::exp2f as exp2(self) -> Self;
        libm::logf as ln(self) -> Self;
        libm::log2f as log2(self) -> Self;
        libm::log10f as log10(self) -> Self;
        libm::cbrtf as cbrt(self) -> Self;
        libm::hypotf as hypot(self, other: Self) -> Self;
        libm::sinf as sin(self) -> Self;
        libm::cosf as cos(self) -> Self;
        libm::tanf as tan(self) -> Self;
        libm::asinf as asin(self) -> Self;
        libm::acosf as acos(self) -> Self;
        libm::atanf as atan(self) -> Self;
        libm::atan2f as atan2(self, other: Self) -> Self;
        libm::sincosf as sin_cos(self) -> (Self, Self);
        libm::expm1f as exp_m1(self) -> Self;
        libm::log1pf as ln_1p(self) -> Self;
        libm::sinhf as sinh(self) -> Self;
        libm::coshf as cosh(self) -> Self;
        libm::tanhf as tanh(self) -> Self;
        libm::asinhf as asinh(self) -> Self;
        libm::acoshf as acosh(self) -> Self;
        libm::atanhf as atanh(self) -> Self;
        libm::copysignf as copysign(self, other: Self) -> Self;
    }
}

#[cfg(all(not(feature = "std"), feature = "libm"))]
impl Float for f64 {
    float_impl_libm!(f64 integer_decode_f64);

    #[inline]
    #[allow(deprecated)]
    fn abs_sub(self, other: Self) -> Self {
        libm::fdim(self, other)
    }

    forward! {
        libm::floor as floor(self) -> Self;
        libm::ceil as ceil(self) -> Self;
        libm::round as round(self) -> Self;
        libm::trunc as trunc(self) -> Self;
        libm::fabs as abs(self) -> Self;
        libm::fma as mul_add(self, a: Self, b: Self) -> Self;
        libm::pow as powf(self, n: Self) -> Self;
        libm::sqrt as sqrt(self) -> Self;
        libm::exp as exp(self) -> Self;
        libm::exp2 as exp2(self) -> Self;
        libm::log as ln(self) -> Self;
        libm::log2 as log2(self) -> Self;
        libm::log10 as log10(self) -> Self;
        libm::cbrt as cbrt(self) -> Self;
        libm::hypot as hypot(self, other: Self) -> Self;
        libm::sin as sin(self) -> Self;
        libm::cos as cos(self) -> Self;
        libm::tan as tan(self) -> Self;
        libm::asin as asin(self) -> Self;
        libm::acos as acos(self) -> Self;
        libm::atan as atan(self) -> Self;
        libm::atan2 as atan2(self, other: Self) -> Self;
        libm::sincos as sin_cos(self) -> (Self, Self);
        libm::expm1 as exp_m1(self) -> Self;
        libm::log1p as ln_1p(self) -> Self;
        libm::sinh as sinh(self) -> Self;
        libm::cosh as cosh(self) -> Self;
        libm::tanh as tanh(self) -> Self;
        libm::asinh as asinh(self) -> Self;
        libm::acosh as acosh(self) -> Self;
        libm::atanh as atanh(self) -> Self;
        libm::copysign as copysign(self, sign: Self) -> Self;
    }
}

macro_rules! float_const_impl {
    ($(#[$doc:meta] $constant:ident,)+) => (
        #[allow(non_snake_case)]
        pub trait FloatConst {
            $(#[$doc] fn $constant() -> Self;)+
            #[doc = "Return the full circle constant `τ`."]
            #[inline]
            fn TAU() -> Self where Self: Sized + Add<Self, Output = Self> {
                Self::PI() + Self::PI()
            }
            #[doc = "Return `log10(2.0)`."]
            #[inline]
            fn LOG10_2() -> Self where Self: Sized + Div<Self, Output = Self> {
                Self::LN_2() / Self::LN_10()
            }
            #[doc = "Return `log2(10.0)`."]
            #[inline]
            fn LOG2_10() -> Self where Self: Sized + Div<Self, Output = Self> {
                Self::LN_10() / Self::LN_2()
            }
        }
        float_const_impl! { @float f32, $($constant,)+ }
        float_const_impl! { @float f64, $($constant,)+ }
    );
    (@float $T:ident, $($constant:ident,)+) => (
        impl FloatConst for $T {
            constant! {
                $( $constant() -> $T::consts::$constant; )+
                TAU() -> 6.28318530717958647692528676655900577;
                LOG10_2() -> 0.301029995663981195213738894724493027;
                LOG2_10() -> 3.32192809488736234787031942948939018;
            }
        }
    );
}

float_const_impl! {
    #[doc = "Return Euler’s number."]
    E,
    #[doc = "Return `1.0 / π`."]
    FRAC_1_PI,
    #[doc = "Return `1.0 / sqrt(2.0)`."]
    FRAC_1_SQRT_2,
    #[doc = "Return `2.0 / π`."]
    FRAC_2_PI,
    #[doc = "Return `2.0 / sqrt(π)`."]
    FRAC_2_SQRT_PI,
    #[doc = "Return `π / 2.0`."]
    FRAC_PI_2,
    #[doc = "Return `π / 3.0`."]
    FRAC_PI_3,
    #[doc = "Return `π / 4.0`."]
    FRAC_PI_4,
    #[doc = "Return `π / 6.0`."]
    FRAC_PI_6,
    #[doc = "Return `π / 8.0`."]
    FRAC_PI_8,
    #[doc = "Return `ln(10.0)`."]
    LN_10,
    #[doc = "Return `ln(2.0)`."]
    LN_2,
    #[doc = "Return `log10(e)`."]
    LOG10_E,
    #[doc = "Return `log2(e)`."]
    LOG2_E,
    #[doc = "Return Archimedes’ constant `π`."]
    PI,
    #[doc = "Return `sqrt(2.0)`."]
    SQRT_2,
}




pub trait TotalOrder {


















































    fn total_cmp(&self, other: &Self) -> Ordering;
}
macro_rules! totalorder_impl {
    ($T:ident, $I:ident, $U:ident, $bits:expr) => {
        impl TotalOrder for $T {
            #[inline]
            #[cfg(has_total_cmp)]
            fn total_cmp(&self, other: &Self) -> Ordering {

                Self::total_cmp(&self, other)
            }
            #[inline]
            #[cfg(not(has_total_cmp))]
            fn total_cmp(&self, other: &Self) -> Ordering {

                let mut left = self.to_bits() as $I;
                let mut right = other.to_bits() as $I;

                left ^= (((left >> ($bits - 1)) as $U) >> 1) as $I;
                right ^= (((right >> ($bits - 1)) as $U) >> 1) as $I;

                left.cmp(&right)
            }
        }
    };
}
totalorder_impl!(f64, i64, u64, 64);
totalorder_impl!(f32, i32, u32, 32);

#[cfg(test)]
mod tests {
    use core::f64::consts;

    const DEG_RAD_PAIRS: [(f64, f64); 7] = [
        (0.0, 0.),
        (22.5, consts::FRAC_PI_8),
        (30.0, consts::FRAC_PI_6),
        (45.0, consts::FRAC_PI_4),
        (60.0, consts::FRAC_PI_3),
        (90.0, consts::FRAC_PI_2),
        (180.0, consts::PI),
    ];

    #[test]
    fn convert_deg_rad() {
        use crate::float::FloatCore;

        for &(deg, rad) in &DEG_RAD_PAIRS {
            assert!((FloatCore::to_degrees(rad) - deg).abs() < 1e-6);
            assert!((FloatCore::to_radians(deg) - rad).abs() < 1e-6);

            let (deg, rad) = (deg as f32, rad as f32);
            assert!((FloatCore::to_degrees(rad) - deg).abs() < 1e-5);
            assert!((FloatCore::to_radians(deg) - rad).abs() < 1e-5);
        }
    }

    #[cfg(any(feature = "std", feature = "libm"))]
    #[test]
    fn convert_deg_rad_std() {
        for &(deg, rad) in &DEG_RAD_PAIRS {
            use crate::Float;

            assert!((Float::to_degrees(rad) - deg).abs() < 1e-6);
            assert!((Float::to_radians(deg) - rad).abs() < 1e-6);

            let (deg, rad) = (deg as f32, rad as f32);
            assert!((Float::to_degrees(rad) - deg).abs() < 1e-5);
            assert!((Float::to_radians(deg) - rad).abs() < 1e-5);
        }
    }

    #[test]
    fn to_degrees_rounding() {
        use crate::float::FloatCore;

        assert_eq!(
            FloatCore::to_degrees(1_f32),
            57.2957795130823208767981548141051703
        );
    }

    #[test]
    #[cfg(any(feature = "std", feature = "libm"))]
    fn extra_logs() {
        use crate::float::{Float, FloatConst};

        fn check<F: Float + FloatConst>(diff: F) {
            let _2 = F::from(2.0).unwrap();
            assert!((F::LOG10_2() - F::log10(_2)).abs() < diff);
            assert!((F::LOG10_2() - F::LN_2() / F::LN_10()).abs() < diff);

            let _10 = F::from(10.0).unwrap();
            assert!((F::LOG2_10() - F::log2(_10)).abs() < diff);
            assert!((F::LOG2_10() - F::LN_10() / F::LN_2()).abs() < diff);
        }

        check::<f32>(1e-6);
        check::<f64>(1e-12);
    }

    #[test]
    #[cfg(any(feature = "std", feature = "libm"))]
    fn copysign() {
        use crate::float::Float;
        test_copysign_generic(2.0_f32, -2.0_f32, f32::nan());
        test_copysign_generic(2.0_f64, -2.0_f64, f64::nan());
        test_copysignf(2.0_f32, -2.0_f32, f32::nan());
    }

    #[cfg(any(feature = "std", feature = "libm"))]
    fn test_copysignf(p: f32, n: f32, nan: f32) {
        use crate::float::Float;
        use core::ops::Neg;

        assert!(p.is_sign_positive());
        assert!(n.is_sign_negative());
        assert!(nan.is_nan());

        assert_eq!(p, Float::copysign(p, p));
        assert_eq!(p.neg(), Float::copysign(p, n));

        assert_eq!(n, Float::copysign(n, n));
        assert_eq!(n.neg(), Float::copysign(n, p));

        assert!(Float::copysign(nan, p).is_sign_positive());
        assert!(Float::copysign(nan, n).is_sign_negative());
    }

    #[cfg(any(feature = "std", feature = "libm"))]
    fn test_copysign_generic<F: crate::float::Float + ::core::fmt::Debug>(p: F, n: F, nan: F) {
        assert!(p.is_sign_positive());
        assert!(n.is_sign_negative());
        assert!(nan.is_nan());
        assert!(!nan.is_subnormal());

        assert_eq!(p, p.copysign(p));
        assert_eq!(p.neg(), p.copysign(n));

        assert_eq!(n, n.copysign(n));
        assert_eq!(n.neg(), n.copysign(p));

        assert!(nan.copysign(p).is_sign_positive());
        assert!(nan.copysign(n).is_sign_negative());
    }

    #[cfg(any(feature = "std", feature = "libm"))]
    fn test_subnormal<F: crate::float::Float + ::core::fmt::Debug>() {
        let min_positive = F::min_positive_value();
        let lower_than_min = min_positive / F::from(2.0f32).unwrap();
        assert!(!min_positive.is_subnormal());
        assert!(lower_than_min.is_subnormal());
    }

    #[test]
    #[cfg(any(feature = "std", feature = "libm"))]
    fn subnormal() {
        test_subnormal::<f64>();
        test_subnormal::<f32>();
    }

    #[test]
    fn total_cmp() {
        use crate::float::TotalOrder;
        use core::cmp::Ordering;
        use core::{f32, f64};

        fn check_eq<T: TotalOrder>(x: T, y: T) {
            assert_eq!(x.total_cmp(&y), Ordering::Equal);
        }
        fn check_lt<T: TotalOrder>(x: T, y: T) {
            assert_eq!(x.total_cmp(&y), Ordering::Less);
        }
        fn check_gt<T: TotalOrder>(x: T, y: T) {
            assert_eq!(x.total_cmp(&y), Ordering::Greater);
        }

        check_eq(f64::NAN, f64::NAN);
        check_eq(f32::NAN, f32::NAN);

        check_lt(-0.0_f64, 0.0_f64);
        check_lt(-0.0_f32, 0.0_f32);



        #[cfg(not(target_arch = "x86"))]
        {
            let s_nan = f64::from_bits(0x7ff4000000000000);
            let q_nan = f64::from_bits(0x7ff8000000000000);
            check_lt(s_nan, q_nan);

            let neg_s_nan = f64::from_bits(0xfff4000000000000);
            let neg_q_nan = f64::from_bits(0xfff8000000000000);
            check_lt(neg_q_nan, neg_s_nan);

            let s_nan = f32::from_bits(0x7fa00000);
            let q_nan = f32::from_bits(0x7fc00000);
            check_lt(s_nan, q_nan);

            let neg_s_nan = f32::from_bits(0xffa00000);
            let neg_q_nan = f32::from_bits(0xffc00000);
            check_lt(neg_q_nan, neg_s_nan);
        }

        check_lt(-f64::NAN, f64::NEG_INFINITY);
        check_gt(1.0_f64, -f64::NAN);
        check_lt(f64::INFINITY, f64::NAN);
        check_gt(f64::NAN, 1.0_f64);

        check_lt(-f32::NAN, f32::NEG_INFINITY);
        check_gt(1.0_f32, -f32::NAN);
        check_lt(f32::INFINITY, f32::NAN);
        check_gt(f32::NAN, 1.0_f32);
    }
}
