/*-
 * Copyright (c) 1996-1998,2000 Shunsuke Akiyama <akiyama@FreeBSD.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: camio.c,v 1.4 2000/09/27 14:41:36 akiyama Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/file.h>
#include <cam/scsi/scsi_all.h>
#include <cam/scsi/scsi_message.h>
#include <camlib.h>

#include "odcontrol.h"

char dname[64];
int unit;
struct cam_device *path;

/*
 *  scsi_format_unit() - set format unit command and parameters.
 */

void
scsi_format_unit(struct ccb_scsiio *csio, u_int32_t retries,
		 void (*cbfcnp)(struct cam_periph *, union ccb *),
		 u_int8_t tag_action, u_int8_t format_data, u_int8_t complist,
		 u_int8_t dlist_fmt, u_int16_t interleave,
		 u_int8_t *param_buf, u_int32_t param_len,
		 u_int8_t sense_len, u_int32_t timeout)
{
	u_int8_t cdb_len;
	struct scsi_format_unit *scsi_cmd;

	scsi_cmd = (struct scsi_format_unit *)&csio->cdb_io.cdb_bytes;
	bzero(scsi_cmd, sizeof(*scsi_cmd));
	scsi_cmd->opcode = SCSI_FORMAT_UNIT;
	scsi_cmd->byte2 |= format_data;
	scsi_cmd->byte2 |= complist;
	scsi_cmd->byte2 |= dlist_fmt;
	scsi_ulto2b(interleave, scsi_cmd->interleave);
	cdb_len = sizeof(*scsi_cmd);

	cam_fill_csio(csio,
		      retries,
		      cbfcnp,
		      /* flags */ CAM_DIR_OUT,
		      tag_action,
		      /* data_ptr */ param_buf,
		      /* dxfer_len */ param_len,
		      sense_len,
		      cdb_len,
		      timeout);
}

/*
 * scsi_verify_blocks() - set verify blocks command and parameters
 */

void
scsi_verify_blocks(struct ccb_scsiio *csio, u_int32_t retries,
		   void (*cbfcnp)(struct cam_periph *, union ccb *),
		   u_int8_t tag_action, u_int8_t dpo, u_int8_t bytchk,
		   u_int8_t reladr, u_int32_t lb_addr, u_int16_t lb_count,
		   u_int8_t sense_len, u_int32_t timeout)
{
	u_int8_t cdb_len;
	struct scsi_verify_blocks *scsi_cmd;

	scsi_cmd = (struct scsi_verify_blocks *)&csio->cdb_io.cdb_bytes;
	bzero(scsi_cmd, sizeof(*scsi_cmd));
	scsi_cmd->opcode = SCSI_VERIFY;
	scsi_cmd->byte2 |= dpo;
	scsi_cmd->byte2 |= bytchk;
	scsi_cmd->byte2 |= reladr;
	scsi_ulto4b(lb_addr, scsi_cmd->lb_addr);
	scsi_ulto2b(lb_count, scsi_cmd->lb_count);
	cdb_len = sizeof(*scsi_cmd);

	cam_fill_csio(csio,
		      retries,
		      cbfcnp,
		      /* flags */ CAM_DIR_NONE,
		      tag_action,
		      /* data_ptr */ NULL,
		      /* dxfer_len */ 0,
		      sense_len,
		      cdb_len,
		      timeout);
}

/*
 * scsi_reassign_blocks() - set reassign blocks command and parameters
 */

void
scsi_reassign_blocks(struct ccb_scsiio *csio, u_int32_t retries,
		     void (*cbfcnp)(struct cam_periph *, union ccb *),
		     u_int8_t tag_action,
		     u_int8_t *param_buf, u_int32_t param_len,
		     u_int8_t sense_len, u_int32_t timeout)
{
	u_int8_t cdb_len;
	struct scsi_reassign_blocks *scsi_cmd;

	scsi_cmd = (struct scsi_reassign_blocks *)&csio->cdb_io.cdb_bytes;
	bzero(scsi_cmd, sizeof(*scsi_cmd));
	scsi_cmd->opcode = SCSI_REASSIGN_BLOCKS;
	cdb_len = sizeof(*scsi_cmd);

	cam_fill_csio(csio,
		      retries,
		      cbfcnp,
		      /* flags */ CAM_DIR_OUT,
		      tag_action,
		      /* data_ptr */ param_buf,
		      /* dxfer_len */ param_len,
		      sense_len,
		      cdb_len,
		      timeout);
}

/*
 *  scsi_error() - print SCSI error.
 */

int
scsi_error(csio)
	struct ccb_scsiio *csio;
{
	int skey, asc, ascq;
	int sts;

	/* message termination */
	switch (have_message) {
	case 2:
		xfprintf(stderr, ", ");
		/* FALLTHROUGH */
	case 1:
		xfprintf(stderr, "aborted\n");
		break;
	default:
		break;
	}

	/* get error codes */
	skey = csio->sense_data.flags & SSD_KEY;
	if (csio->sense_data.extra_len >= 5) {
		asc = csio->sense_data.add_sense_code;
	} else {
		asc = 0;
	}
	if (csio->sense_data.extra_len >= 6) {
		ascq = csio->sense_data.add_sense_code_qual;
	} else {
		ascq = 0;
	}

	if (skey == SSD_KEY_UNIT_ATTENTION && asc == 0x28 && ascq == 0) {
		/*
		 * Media had been changed!
		 * This is not a really error.
		 */
		sts = ERR_NONE;
	} else if (csio->cdb_io.cdb_bytes[0] == SCSI_VERIFY &&
		   skey == SSD_KEY_MEDIUM_ERROR) {
		/*
		 * Error on verify!
		 */
		sts = ERR_BADBLOCK;
	} else {
		if (verbose >= 0) {
			if (have_message == 3) {
				xfprintf(stderr, ", aborted\n");
			}
			scsi_sense_print(path, csio, stderr);
		}
		sts = ERR_SCSI;
	}

	return sts;
}

/*
 * test_unit_ready() - send test unit ready to target.
 */

int
test_unit_ready()
{
	int sts;
	union ccb *ccb;

	ccb = cam_getccb(path);

	scsi_test_unit_ready(&ccb->csio,
			     /* retries */ 1,
			     /* cbfcnp */ NULL,
			     /* tag_action */ MSG_SIMPLE_Q_TAG,
			     /* sense_len */ SSD_FULL_SIZE,
			     /* timeout */ get_timeout(DEFAULT_TIMEOUT));

	/* Disable freezing the device queue */
	ccb->ccb_h.flags |= CAM_DEV_QFRZDIS;
	ccb->ccb_h.flags |= CAM_PASS_ERR_RECOVER;

	sts = cam_send_ccb(path, ccb);
	if (sts < 0) {
		xfprintf(stderr, "%s: error sending test unit ready\n",
			 program);
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			scsi_sense_print(path, &ccb->csio, stderr);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
		}
		cam_freeccb(ccb);
		return ERR_SCSI;
	}
	if ((ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			sts = scsi_error(&ccb->csio);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
			sts = ERR_SCSI;
		}
		cam_freeccb(ccb);
		return sts;
	}

	cam_freeccb(ccb);

	return sts;
}	

/*
 * get_cache_setting() - get current cache control setting
 */

int
get_cache_setting(curval, len)
	u_char *curval;
	int len;
{
	int sts;
	union ccb *ccb;

	ccb = cam_getccb(path);

	bzero(curval, len);
	scsi_mode_sense(&ccb->csio,
			/* retries */ 1,
			/* cbfcnp */ NULL,
			/* tag_action */ MSG_SIMPLE_Q_TAG,
			/* dbd */ 0,
			/* page_code */ SMS_PAGE_CTRL_CURRENT,
			/* page */ 8,
			/* param_buf */ curval,
			/* param_len */ len,
			/* sense_len */ SSD_FULL_SIZE,
			/* timeout */ get_timeout(DEFAULT_TIMEOUT));

	/* Disable freezing the device queue */
	ccb->ccb_h.flags |= CAM_DEV_QFRZDIS;

	sts = cam_send_ccb(path, ccb);
	if (sts < 0) {
		xfprintf(stderr, "%s: error sending mode sense\n",
			 program);
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			scsi_sense_print(path, &ccb->csio, stderr);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
		}
		cam_freeccb(ccb);
		return ERR_SCSI;
	}
	if ((ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			sts = scsi_error(&ccb->csio);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
			sts = ERR_SCSI;
		}
		cam_freeccb(ccb);
		return sts;
	}

	cam_freeccb(ccb);

	return sts;
}

/*
 * get_cache_mask() - get changable cache control parameter mask
 */

int
get_cache_mask(maskval, len)
	u_char *maskval;
	int len;
{
	int sts;
	union ccb *ccb;

	ccb = cam_getccb(path);

	bzero(maskval, len);
	scsi_mode_sense(&ccb->csio,
			/* retries */ 1,
			/* cbfcnp */ NULL,
			/* tag_action */ MSG_SIMPLE_Q_TAG,
			/* dbd */ 0,
			/* page_code */ SMS_PAGE_CTRL_CHANGEABLE,
			/* page */ 8,
			/* param_buf */ maskval,
			/* param_len */ len,
			/* sense_len */ SSD_FULL_SIZE,
			/* timeout */ get_timeout(DEFAULT_TIMEOUT));

	/* Disable freezing the device queue */
	ccb->ccb_h.flags |= CAM_DEV_QFRZDIS;

	sts = cam_send_ccb(path, ccb);
	if (sts < 0) {
		xfprintf(stderr, "%s: error sending mode sense\n",
			 program);
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			scsi_sense_print(path, &ccb->csio, stderr);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
		}
		cam_freeccb(ccb);
		return ERR_SCSI;
	}
	if ((ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			sts = scsi_error(&ccb->csio);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
			sts = ERR_SCSI;
		}
		cam_freeccb(ccb);
		return sts;
	}

	cam_freeccb(ccb);

	return sts;
}

/*
 * set_cache_setting() - set cache control parameter
 */

int
set_cache_setting(newval, len)
	u_char *newval;
	int len;
{
	int sts;
	union ccb *ccb;

	ccb = cam_getccb(path);

	scsi_mode_select(&ccb->csio,
			 /* retries */ 1,
			 /* cbfcnp */ NULL,
			 /* tag_action */ MSG_SIMPLE_Q_TAG,
			 /* scsi_page_fmt */ 1,
			 /* save_pages */ 0,
			 /* param_buf */ newval,
			 /* param_len */ len,
			 /* sense_len */ SSD_FULL_SIZE,
			 /* timeout */ get_timeout(DEFAULT_TIMEOUT));

	/* Disable freezing the device queue */
	ccb->ccb_h.flags |= CAM_DEV_QFRZDIS;

	sts = cam_send_ccb(path, ccb);
	if (sts < 0) {
		xfprintf(stderr, "%s: error sending mode select\n",
			 program);
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			scsi_sense_print(path, &ccb->csio, stderr);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
		}
		cam_freeccb(ccb);
		return ERR_SCSI;
	}
	if ((ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			sts = scsi_error(&ccb->csio);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
			sts = ERR_SCSI;
		}
		cam_freeccb(ccb);
		return sts;
	}

	cam_freeccb(ccb);

	return sts;
}

/*
 * format_unit() - initiate format unit command
 */

int
format_unit()
{
	int sts;
	union ccb *ccb;

	ccb = cam_getccb(path);

	scsi_format_unit(&ccb->csio,
			 /* retries */ 1,
			 /* cbfcnp */ NULL,
			 /* tag_action */ MSG_SIMPLE_Q_TAG,
			 /* format_data */ 0,
			 /* complist */ 0,
			 /* dlist_fmt */ 0,
			 /* interleave */ 0,
			 /* param_buf */ (u_int8_t *)&sts,	/* KLUDGE */
			 /* param_len */ 0,
			 /* sense_len */ SSD_FULL_SIZE,
			 /* timeout */ get_timeout(DEFAULT_TIMEOUT_LONG));

	/* Disable freezing the device queue */
	ccb->ccb_h.flags |= CAM_DEV_QFRZDIS;

	sts = cam_send_ccb(path, ccb);
	if (sts < 0) {
		xfprintf(stderr, "%s: error sending format unit\n",
			 program);
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			scsi_sense_print(path, &ccb->csio, stderr);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
		}
		cam_freeccb(ccb);
		return ERR_SCSI;
	}
	if ((ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			sts = scsi_error(&ccb->csio);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
			sts = ERR_SCSI;
		}
		cam_freeccb(ccb);
		return sts;
	}

	cam_freeccb(ccb);

	return sts;
}

/*
 * read_capacity() - get media capacity
 */

int
read_capacity(max)
	u_int *max;
{
	int sts;
	union ccb *ccb;
	struct scsi_read_capacity_data rdcap_data;

	ccb = cam_getccb(path);

	scsi_read_capacity(&ccb->csio,
			   /* retries */ 1,
			   /* cbfcnp */ NULL,
			   /* tag_action */ MSG_SIMPLE_Q_TAG,
			   /* rcap_buf */ &rdcap_data,
			   /* sense_len */ SSD_FULL_SIZE,
			   /* timeout */ get_timeout(DEFAULT_TIMEOUT));

	/* Disable freezing the device queue */
	ccb->ccb_h.flags |= CAM_DEV_QFRZDIS;

	sts = cam_send_ccb(path, ccb);
	if (sts < 0) {
		xfprintf(stderr, "%s: error sending read capacity\n",
			 program);
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			scsi_sense_print(path, &ccb->csio, stderr);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
		}
		cam_freeccb(ccb);
		return ERR_SCSI;
	}
	if ((ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			sts = scsi_error(&ccb->csio);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
			sts = ERR_SCSI;
		}
		cam_freeccb(ccb);
		return sts;
	}

	cam_freeccb(ccb);

	*max = scsi_4btoul(rdcap_data.addr) + 1;

	return ERR_NONE;
}

/*
 * verify_media() - verify media blocks
 */

int
verify_media(addr, len, bad)
	u_int addr;
	u_int len;
	u_int *bad;
{
	int sts;
	union ccb *ccb;

	ccb = cam_getccb(path);

	scsi_verify_blocks(&ccb->csio,
			   /* retries */ 1,
			   /* cbfcnp */ NULL,
			   /* tag_action */ MSG_SIMPLE_Q_TAG,
			   /* dpo */ 0,
			   /* bytchk */ 0,
			   /* reladr */ 0,
			   /* lb_addr */ addr,
			   /* lb_count */ len,
			   /* sense_len */ SSD_FULL_SIZE,
			   /* timeout */ get_timeout(DEFAULT_TIMEOUT));

	/* Disable freezing the device queue */
	ccb->ccb_h.flags |= CAM_DEV_QFRZDIS;

	sts = cam_send_ccb(path, ccb);
	if (sts < 0) {
		xfprintf(stderr, "%s: error sending verify\n",
			 program);
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			scsi_sense_print(path, &ccb->csio, stderr);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
		}
		cam_freeccb(ccb);
		return ERR_SCSI;
	}
	if ((ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			sts = scsi_error(&ccb->csio);
			if (sts == ERR_BADBLOCK) {
				*bad = scsi_4btoul(ccb->csio.sense_data.info);
			}
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
			sts = ERR_SCSI;
		}
		cam_freeccb(ccb);
		return sts;
	}

	cam_freeccb(ccb);

	return sts;
}

/*
 * reassign_block() - reassign bad block
 */

int
reassign_block(bad)
	u_int bad;
{
	int sts;
	union ccb *ccb;
	struct scsi_defect_list defect;

	ccb = cam_getccb(path);

	bzero(&defect, sizeof (defect));
	scsi_ulto2b(4, defect.list_len);
	scsi_ulto4b(bad, defect.lb_addr);

	scsi_reassign_blocks(&ccb->csio,
			     /* retries */ 1,
			     /* cbfcnp */ NULL,
			     /* tag_action */ MSG_SIMPLE_Q_TAG,
			     /* param_buf */ (u_int8_t *)&defect,
			     /* param_len */ sizeof (defect),
			     /* sense_len */ SSD_FULL_SIZE,
			     /* timeout */ get_timeout(DEFAULT_TIMEOUT));

	/* Disable freezing the device queue */
	ccb->ccb_h.flags |= CAM_DEV_QFRZDIS;

	sts = cam_send_ccb(path, ccb);
	if (sts < 0) {
		xfprintf(stderr, "%s: error sending reassign blocks\n",
			 program);
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			scsi_sense_print(path, &ccb->csio, stderr);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
		}
		cam_freeccb(ccb);
		return ERR_SCSI;
	}
	if ((ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			sts = scsi_error(&ccb->csio);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
			sts = ERR_SCSI;
		}
		cam_freeccb(ccb);
		return sts;
	}

	cam_freeccb(ccb);

	return sts;
}

/*
 * eject_media() - eject loaded media
 */

int
eject_media()
{
	int sts;
	union ccb *ccb;

	sts = open_device();
	if (sts != ERR_NONE) {
		return sts;
	}

	ccb = cam_getccb(path);

	scsi_start_stop(&ccb->csio,
			/* retries */ 1,
			/* cbfcnp */ NULL,
			/* tag_action */ MSG_ORDERED_Q_TAG,
			/* start/stop */ 0,
			/* load_eject */ 1,
			/* immediate */ 0,
			/* sense_len */ SSD_FULL_SIZE,
			/* timeout */ get_timeout(DEFAULT_TIMEOUT2));

	/* Disable freezing the device queue */
	ccb->ccb_h.flags |= CAM_DEV_QFRZDIS;

	sts = cam_send_ccb(path, ccb);
	if (sts < 0) {
		xfprintf(stderr, "%s: error sending start-stop unit\n",
			 program);
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			scsi_sense_print(path, &ccb->csio, stderr);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
		}
		cam_freeccb(ccb);
		close_device();
		return ERR_SCSI;
	}
	if ((ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		if ((ccb->ccb_h.status & CAM_STATUS_MASK) ==
		    CAM_SCSI_STATUS_ERROR) {
			sts = scsi_error(&ccb->csio);
		} else {
			xfprintf(stderr, "CAM status is %#x\n",
				 ccb->ccb_h.status);
			sts = ERR_SCSI;
		}
		cam_freeccb(ccb);
		close_device();
		return sts;
	}

	cam_freeccb(ccb);

	close_device();

	return sts;
}

/*
 * get_device() - get device name and check it
 */

int
get_device(dev, cmd)
	char *dev;
	struct cmd_tab *cmd;
{
	int sts;

	sts = cam_get_device(dev, dname, sizeof (dname), &unit);

	return sts;
}

/*
 * close_device() - close the device
 */

void
close_device()
{
	cam_close_spec_device(path);
}

/*
 * open_device() - open specified device
 */

int
open_device()
{
	int sts;

	path = cam_open_spec_device(dname, unit, O_RDWR, NULL);
	if (path == NULL) {
		perror(program);
		return ERR_COMMAND;
	}

	bus_no = path->bus_id;
	target_no = path->target_id;
	lun_no = path->target_lun;

	/* clear "Unit Attention" */
	sts = test_unit_ready();
	if (sts != ERR_NONE) {
		close_device();
		return sts;
	}

	return ERR_NONE;
}

