--
-- This file is part of TALER
-- Copyright (C) 2024 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>

BEGIN;

SELECT _v.register_patch('libeufin-bank-0009', NULL, NULL);
SET search_path TO libeufin_bank;

-- Add missing unique constraints
ALTER TABLE taler_exchange_outgoing ADD UNIQUE (exchange_outgoing_id);
ALTER TABLE taler_exchange_incoming ADD UNIQUE (exchange_incoming_id);
ALTER TABLE taler_withdrawal_operations ADD UNIQUE (withdrawal_id);

CREATE TYPE transfer_status AS ENUM
  ('permanent_failure'
  ,'success'
  );

CREATE TABLE transfer_operations
  ( transfer_operation_id INT8 GENERATED BY DEFAULT AS IDENTITY UNIQUE
   ,request_uid BYTEA UNIQUE NOT NULL CHECK (LENGTH(request_uid)=64)
   ,wtid BYTEA UNIQUE NOT NULL CHECK (LENGTH(wtid)=32)
   ,amount taler_amount NOT NULL
   ,exchange_base_url TEXT NOT NULL
   ,transfer_date INT8 NOT NULL
   ,exchange_outgoing_id INT8 UNIQUE REFERENCES taler_exchange_outgoing(exchange_outgoing_id) ON DELETE CASCADE
   ,creditor_payto TEXT NOT NULL
   ,status transfer_status NOT NULL
   ,status_msg TEXT
   ,exchange_id INT8 NOT NULL REFERENCES bank_accounts(bank_account_id) ON DELETE CASCADE
   ,CONSTRAINT transfer_operations_polymorphism CHECK(
    CASE status
      WHEN 'success' THEN exchange_outgoing_id IS NOT NULL
      ELSE exchange_outgoing_id IS NULL
    END
   )
  );
COMMENT ON TABLE transfer_operations
  IS 'Operation table for idempotent wire gateway transfers with status.';

-- Migrate data from taler_exchange_outgoing to transfer_operations
INSERT INTO transfer_operations(transfer_operation_id, request_uid, amount, wtid, exchange_base_url, transfer_date, exchange_outgoing_id, creditor_payto, status, status_msg, exchange_id)
  SELECT bank_transaction_id, request_uid, amount, wtid, exchange_base_url, transaction_date, exchange_outgoing_id, creditor_payto, 'success'::transfer_status, NULL, bank_account_id
  FROM taler_exchange_outgoing JOIN bank_account_transactions ON bank_transaction = bank_transaction_id;

CREATE INDEX transfer_operations_status_index ON transfer_operations (status);
COMMENT ON INDEX transfer_operations_status_index IS 'for listing taler transfers by status';

CREATE INDEX transfer_operations_account_index ON transfer_operations (exchange_id);
COMMENT ON INDEX transfer_operations_account_index IS 'for listing taler transfers by account';

-- Remove unused columns
ALTER TABLE taler_exchange_outgoing 
  DROP COLUMN request_uid,
  DROP COLUMN creditor_account_id,
  DROP COLUMN wtid,
  DROP COLUMN exchange_base_url;

COMMIT;
