{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A @GRegex@ is a compiled form of a regular expression.
-- 
-- After instantiating a @GRegex@, you can use its methods to find matches
-- in a string, replace matches within a string, or split the string at matches.
-- 
-- @GRegex@ implements regular expression pattern matching using syntax and
-- semantics (such as character classes, quantifiers, and capture groups)
-- similar to Perl regular expression. See the
-- <http://developer.gnome.org/glib/stable/man:pcre2pattern(3 PCRE documentation>) for details.
-- 
-- A typical scenario for regex pattern matching is to check if a string
-- matches a pattern. The following statements implement this scenario.
-- 
-- 
-- === /{ .c } code/
-- >const char *regex_pattern = ".*GLib.*";
-- >const char *string_to_search = "You will love the GLib implementation of regex";
-- >g_autoptr(GMatchInfo) match_info = NULL;
-- >g_autoptr(GRegex) regex = NULL;
-- >
-- >regex = g_regex_new (regex_pattern, G_REGEX_DEFAULT, G_REGEX_MATCH_DEFAULT, NULL);
-- >g_assert (regex != NULL);
-- >
-- >if (g_regex_match (regex, string_to_search, G_REGEX_MATCH_DEFAULT, &match_info))
-- >  {
-- >    int start_pos, end_pos;
-- >    g_match_info_fetch_pos (match_info, 0, &start_pos, &end_pos);
-- >    g_print ("Match successful! Overall pattern matches bytes %d to %d\n", start_pos, end_pos);
-- >  }
-- >else
-- >  {
-- >    g_print ("No match!\n");
-- >  }
-- 
-- 
-- The constructor for @GRegex@ includes two sets of bitmapped flags:
-- 
-- * [flags/@gLib@/.RegexCompileFlags]—These flags
-- control how GLib compiles the regex. There are options for case
-- sensitivity, multiline, ignoring whitespace, etc.
-- * [flags/@gLib@/.RegexMatchFlags]—These flags control
-- @GRegex@’s matching behavior, such as anchoring and customizing definitions
-- for newline characters.
-- 
-- Some regex patterns include backslash assertions, such as @\\d@ (digit) or
-- @\\D@ (non-digit). The regex pattern must escape those backslashes. For
-- example, the pattern @\"\\\\d\\\\D\"@ matches a digit followed by a non-digit.
-- 
-- GLib’s implementation of pattern matching includes a @start_position@
-- argument for some of the match, replace, and split methods. Specifying
-- a start position provides flexibility when you want to ignore the first
-- _n_ characters of a string, but want to incorporate backslash assertions
-- at character _n_ - 1. For example, a database field contains inconsistent
-- spelling for a job title: @healthcare provider@ and @health-care provider@.
-- The database manager wants to make the spelling consistent by adding a
-- hyphen when it is missing. The following regex pattern tests for the string
-- @care@ preceded by a non-word boundary character (instead of a hyphen)
-- and followed by a space.
-- 
-- 
-- === /{ .c } code/
-- >const char *regex_pattern = "\\Bcare\\s";
-- 
-- 
-- An efficient way to match with this pattern is to start examining at
-- @start_position@ 6 in the string @healthcare@ or @health-care@.
-- 
-- 
-- === /{ .c } code/
-- >const char *regex_pattern = "\\Bcare\\s";
-- >const char *string_to_search = "healthcare provider";
-- >g_autoptr(GMatchInfo) match_info = NULL;
-- >g_autoptr(GRegex) regex = NULL;
-- >
-- >regex = g_regex_new (
-- >  regex_pattern,
-- >  G_REGEX_DEFAULT,
-- >  G_REGEX_MATCH_DEFAULT,
-- >  NULL);
-- >g_assert (regex != NULL);
-- >
-- >g_regex_match_full (
-- >  regex,
-- >  string_to_search,
-- >  -1,
-- >  6, // position of 'c' in the test string.
-- >  G_REGEX_MATCH_DEFAULT,
-- >  &match_info,
-- >  NULL);
-- 
-- 
-- The method 'GI.GLib.Structs.Regex.regexMatchFull' (and other methods implementing
-- @start_pos@) allow for lookback before the start position to determine if
-- the previous character satisfies an assertion.
-- 
-- Unless you set the [flags/@gLib@/.RegexCompileFlags.RAW] as one of
-- the @GRegexCompileFlags@, all the strings passed to @GRegex@ methods must
-- be encoded in UTF-8. The lengths and the positions inside the strings are
-- in bytes and not in characters, so, for instance, @\\xc3\\xa0@ (i.e., @à@)
-- is two bytes long but it is treated as a single character. If you set
-- @G_REGEX_RAW@, the strings can be non-valid UTF-8 strings and a byte is
-- treated as a character, so @\\xc3\\xa0@ is two bytes and two characters long.
-- 
-- Regarding line endings, @\\n@ matches a @\\n@ character, and @\\r@ matches
-- a @\\r@ character. More generally, @\\R@ matches all typical line endings:
-- CR + LF (@\\r\\n@), LF (linefeed, U+000A, @\\n@), VT (vertical tab, U+000B,
-- @\\v@), FF (formfeed, U+000C, @\\f@), CR (carriage return, U+000D, @\\r@),
-- NEL (next line, U+0085), LS (line separator, U+2028), and PS (paragraph
-- separator, U+2029).
-- 
-- The behaviour of the dot, circumflex, and dollar metacharacters are
-- affected by newline characters. By default, @GRegex@ matches any newline
-- character matched by @\\R@. You can limit the matched newline characters by
-- specifying the [flags/@gLib@/.RegexMatchFlags.NEWLINE_CR],
-- [flags/@gLib@/.RegexMatchFlags.NEWLINE_LF], and
-- [flags/@gLib@/.RegexMatchFlags.NEWLINE_CRLF] compile options, and
-- with [flags/@gLib@/.RegexMatchFlags.NEWLINE_ANY],
-- [flags/@gLib@/.RegexMatchFlags.NEWLINE_CR],
-- [flags/@gLib@/.RegexMatchFlags.NEWLINE_LF] and
-- [flags/@gLib@/.RegexMatchFlags.NEWLINE_CRLF] match options.
-- These settings are also relevant when compiling a pattern if
-- [flags/@gLib@/.RegexCompileFlags.EXTENDED] is set and an unescaped
-- @#@ outside a character class is encountered. This indicates a comment
-- that lasts until after the next newline.
-- 
-- Because @GRegex@ does not modify its internal state between creation and
-- destruction, you can create and modify the same @GRegex@ instance from
-- different threads. In contrast, t'GI.GLib.Structs.MatchInfo.MatchInfo' is not thread safe.
-- 
-- The regular expression low-level functionalities are obtained through
-- the excellent <http://www.pcre.org/ PCRE> library written by Philip Hazel.
-- 
-- /Since: 2.14/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.Regex
    ( 

-- * Exported types
    Regex(..)                               ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [match]("GI.GLib.Structs.Regex#g:method:match"), [matchAll]("GI.GLib.Structs.Regex#g:method:matchAll"), [matchAllFull]("GI.GLib.Structs.Regex#g:method:matchAllFull"), [matchFull]("GI.GLib.Structs.Regex#g:method:matchFull"), [ref]("GI.GLib.Structs.Regex#g:method:ref"), [replace]("GI.GLib.Structs.Regex#g:method:replace"), [replaceEval]("GI.GLib.Structs.Regex#g:method:replaceEval"), [replaceLiteral]("GI.GLib.Structs.Regex#g:method:replaceLiteral"), [split]("GI.GLib.Structs.Regex#g:method:split"), [splitFull]("GI.GLib.Structs.Regex#g:method:splitFull"), [unref]("GI.GLib.Structs.Regex#g:method:unref").
-- 
-- ==== Getters
-- [getCaptureCount]("GI.GLib.Structs.Regex#g:method:getCaptureCount"), [getCompileFlags]("GI.GLib.Structs.Regex#g:method:getCompileFlags"), [getHasCrOrLf]("GI.GLib.Structs.Regex#g:method:getHasCrOrLf"), [getMatchFlags]("GI.GLib.Structs.Regex#g:method:getMatchFlags"), [getMaxBackref]("GI.GLib.Structs.Regex#g:method:getMaxBackref"), [getMaxLookbehind]("GI.GLib.Structs.Regex#g:method:getMaxLookbehind"), [getPattern]("GI.GLib.Structs.Regex#g:method:getPattern"), [getStringNumber]("GI.GLib.Structs.Regex#g:method:getStringNumber").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveRegexMethod                      ,
#endif

-- ** checkReplacement #method:checkReplacement#

    regexCheckReplacement                   ,


-- ** errorQuark #method:errorQuark#

    regexErrorQuark                         ,


-- ** escapeNul #method:escapeNul#

    regexEscapeNul                          ,


-- ** escapeString #method:escapeString#

    regexEscapeString                       ,


-- ** getCaptureCount #method:getCaptureCount#

#if defined(ENABLE_OVERLOADING)
    RegexGetCaptureCountMethodInfo          ,
#endif
    regexGetCaptureCount                    ,


-- ** getCompileFlags #method:getCompileFlags#

#if defined(ENABLE_OVERLOADING)
    RegexGetCompileFlagsMethodInfo          ,
#endif
    regexGetCompileFlags                    ,


-- ** getHasCrOrLf #method:getHasCrOrLf#

#if defined(ENABLE_OVERLOADING)
    RegexGetHasCrOrLfMethodInfo             ,
#endif
    regexGetHasCrOrLf                       ,


-- ** getMatchFlags #method:getMatchFlags#

#if defined(ENABLE_OVERLOADING)
    RegexGetMatchFlagsMethodInfo            ,
#endif
    regexGetMatchFlags                      ,


-- ** getMaxBackref #method:getMaxBackref#

#if defined(ENABLE_OVERLOADING)
    RegexGetMaxBackrefMethodInfo            ,
#endif
    regexGetMaxBackref                      ,


-- ** getMaxLookbehind #method:getMaxLookbehind#

#if defined(ENABLE_OVERLOADING)
    RegexGetMaxLookbehindMethodInfo         ,
#endif
    regexGetMaxLookbehind                   ,


-- ** getPattern #method:getPattern#

#if defined(ENABLE_OVERLOADING)
    RegexGetPatternMethodInfo               ,
#endif
    regexGetPattern                         ,


-- ** getStringNumber #method:getStringNumber#

#if defined(ENABLE_OVERLOADING)
    RegexGetStringNumberMethodInfo          ,
#endif
    regexGetStringNumber                    ,


-- ** match #method:match#

#if defined(ENABLE_OVERLOADING)
    RegexMatchMethodInfo                    ,
#endif
    regexMatch                              ,


-- ** matchAll #method:matchAll#

#if defined(ENABLE_OVERLOADING)
    RegexMatchAllMethodInfo                 ,
#endif
    regexMatchAll                           ,


-- ** matchAllFull #method:matchAllFull#

#if defined(ENABLE_OVERLOADING)
    RegexMatchAllFullMethodInfo             ,
#endif
    regexMatchAllFull                       ,


-- ** matchFull #method:matchFull#

#if defined(ENABLE_OVERLOADING)
    RegexMatchFullMethodInfo                ,
#endif
    regexMatchFull                          ,


-- ** matchSimple #method:matchSimple#

    regexMatchSimple                        ,


-- ** new #method:new#

    regexNew                                ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    RegexRefMethodInfo                      ,
#endif
    regexRef                                ,


-- ** replace #method:replace#

#if defined(ENABLE_OVERLOADING)
    RegexReplaceMethodInfo                  ,
#endif
    regexReplace                            ,


-- ** replaceEval #method:replaceEval#

#if defined(ENABLE_OVERLOADING)
    RegexReplaceEvalMethodInfo              ,
#endif
    regexReplaceEval                        ,


-- ** replaceLiteral #method:replaceLiteral#

#if defined(ENABLE_OVERLOADING)
    RegexReplaceLiteralMethodInfo           ,
#endif
    regexReplaceLiteral                     ,


-- ** split #method:split#

#if defined(ENABLE_OVERLOADING)
    RegexSplitMethodInfo                    ,
#endif
    regexSplit                              ,


-- ** splitFull #method:splitFull#

#if defined(ENABLE_OVERLOADING)
    RegexSplitFullMethodInfo                ,
#endif
    regexSplitFull                          ,


-- ** splitSimple #method:splitSimple#

    regexSplitSimple                        ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    RegexUnrefMethodInfo                    ,
#endif
    regexUnref                              ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import {-# SOURCE #-} qualified GI.GLib.Flags as GLib.Flags
import {-# SOURCE #-} qualified GI.GLib.Structs.MatchInfo as GLib.MatchInfo

#else
import qualified GI.GLib.Callbacks as GLib.Callbacks
import {-# SOURCE #-} qualified GI.GLib.Flags as GLib.Flags
import {-# SOURCE #-} qualified GI.GLib.Structs.MatchInfo as GLib.MatchInfo

#endif

-- | Memory-managed wrapper type.
newtype Regex = Regex (SP.ManagedPtr Regex)
    deriving (Eq)

instance SP.ManagedPtrNewtype Regex where
    toManagedPtr (Regex p) = p

foreign import ccall "g_regex_get_type" c_g_regex_get_type :: 
    IO GType

type instance O.ParentTypes Regex = '[]
instance O.HasParentTypes Regex

instance B.Types.TypedObject Regex where
    glibType = c_g_regex_get_type

instance B.Types.GBoxed Regex

-- | Convert t'Regex' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Regex) where
    gvalueGType_ = c_g_regex_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr Regex)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr Regex)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed Regex ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Regex
type instance O.AttributeList Regex = RegexAttributeList
type RegexAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method Regex::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "pattern"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the regular expression"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "compile_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexCompileFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "compile options for the regular expression, or 0"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "match options for the regular expression, or 0"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Regex" })
-- throws : True
-- Skip return : False

foreign import ccall "g_regex_new" g_regex_new :: 
    CString ->                              -- pattern : TBasicType TUTF8
    CUInt ->                                -- compile_options : TInterface (Name {namespace = "GLib", name = "RegexCompileFlags"})
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Regex)

-- | Compiles the regular expression to an internal form, and does
-- the initial setup of the t'GI.GLib.Structs.Regex.Regex' structure.
-- 
-- /Since: 2.14/
regexNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@pattern@/: the regular expression
    -> [GLib.Flags.RegexCompileFlags]
    -- ^ /@compileOptions@/: compile options for the regular expression, or 0
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: match options for the regular expression, or 0
    -> m (Maybe Regex)
    -- ^ __Returns:__ a t'GI.GLib.Structs.Regex.Regex' structure or 'P.Nothing' if an error occurred. Call
    --   'GI.GLib.Structs.Regex.regexUnref' when you are done with it /(Can throw 'Data.GI.Base.GError.GError')/
regexNew pattern compileOptions matchOptions = liftIO $ do
    pattern' <- textToCString pattern
    let compileOptions' = gflagsToWord compileOptions
    let matchOptions' = gflagsToWord matchOptions
    onException (do
        result <- propagateGError $ g_regex_new pattern' compileOptions' matchOptions'
        maybeResult <- convertIfNonNull result $ \result' -> do
            result'' <- (wrapBoxed Regex) result'
            return result''
        freeMem pattern'
        return maybeResult
     ) (do
        freeMem pattern'
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Regex::get_capture_count
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_get_capture_count" g_regex_get_capture_count :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    IO Int32

-- | Returns the number of capturing subpatterns in the pattern.
-- 
-- /Since: 2.14/
regexGetCaptureCount ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex'
    -> m Int32
    -- ^ __Returns:__ the number of capturing subpatterns
regexGetCaptureCount regex = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    result <- g_regex_get_capture_count regex'
    touchManagedPtr regex
    return result

#if defined(ENABLE_OVERLOADING)
data RegexGetCaptureCountMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod RegexGetCaptureCountMethodInfo Regex signature where
    overloadedMethod = regexGetCaptureCount

instance O.OverloadedMethodInfo RegexGetCaptureCountMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexGetCaptureCount",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexGetCaptureCount"
        })


#endif

-- method Regex::get_compile_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "GLib" , name = "RegexCompileFlags" })
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_get_compile_flags" g_regex_get_compile_flags :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    IO CUInt

-- | Returns the compile options that /@regex@/ was created with.
-- 
-- Depending on the version of PCRE that is used, this may or may not
-- include flags set by option expressions such as @(?i)@ found at the
-- top-level within the compiled pattern.
-- 
-- /Since: 2.26/
regexGetCompileFlags ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex'
    -> m [GLib.Flags.RegexCompileFlags]
    -- ^ __Returns:__ flags from t'GI.GLib.Flags.RegexCompileFlags'
regexGetCompileFlags regex = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    result <- g_regex_get_compile_flags regex'
    let result' = wordToGFlags result
    touchManagedPtr regex
    return result'

#if defined(ENABLE_OVERLOADING)
data RegexGetCompileFlagsMethodInfo
instance (signature ~ (m [GLib.Flags.RegexCompileFlags]), MonadIO m) => O.OverloadedMethod RegexGetCompileFlagsMethodInfo Regex signature where
    overloadedMethod = regexGetCompileFlags

instance O.OverloadedMethodInfo RegexGetCompileFlagsMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexGetCompileFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexGetCompileFlags"
        })


#endif

-- method Regex::get_has_cr_or_lf
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_get_has_cr_or_lf" g_regex_get_has_cr_or_lf :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    IO CInt

-- | Checks whether the pattern contains explicit CR or LF references.
-- 
-- /Since: 2.34/
regexGetHasCrOrLf ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the pattern contains explicit CR or LF references
regexGetHasCrOrLf regex = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    result <- g_regex_get_has_cr_or_lf regex'
    let result' = (/= 0) result
    touchManagedPtr regex
    return result'

#if defined(ENABLE_OVERLOADING)
data RegexGetHasCrOrLfMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod RegexGetHasCrOrLfMethodInfo Regex signature where
    overloadedMethod = regexGetHasCrOrLf

instance O.OverloadedMethodInfo RegexGetHasCrOrLfMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexGetHasCrOrLf",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexGetHasCrOrLf"
        })


#endif

-- method Regex::get_match_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" })
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_get_match_flags" g_regex_get_match_flags :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    IO CUInt

-- | Returns the match options that /@regex@/ was created with.
-- 
-- /Since: 2.26/
regexGetMatchFlags ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex'
    -> m [GLib.Flags.RegexMatchFlags]
    -- ^ __Returns:__ flags from t'GI.GLib.Flags.RegexMatchFlags'
regexGetMatchFlags regex = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    result <- g_regex_get_match_flags regex'
    let result' = wordToGFlags result
    touchManagedPtr regex
    return result'

#if defined(ENABLE_OVERLOADING)
data RegexGetMatchFlagsMethodInfo
instance (signature ~ (m [GLib.Flags.RegexMatchFlags]), MonadIO m) => O.OverloadedMethod RegexGetMatchFlagsMethodInfo Regex signature where
    overloadedMethod = regexGetMatchFlags

instance O.OverloadedMethodInfo RegexGetMatchFlagsMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexGetMatchFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexGetMatchFlags"
        })


#endif

-- method Regex::get_max_backref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_get_max_backref" g_regex_get_max_backref :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    IO Int32

-- | Returns the number of the highest back reference
-- in the pattern, or 0 if the pattern does not contain
-- back references.
-- 
-- /Since: 2.14/
regexGetMaxBackref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex'
    -> m Int32
    -- ^ __Returns:__ the number of the highest back reference
regexGetMaxBackref regex = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    result <- g_regex_get_max_backref regex'
    touchManagedPtr regex
    return result

#if defined(ENABLE_OVERLOADING)
data RegexGetMaxBackrefMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod RegexGetMaxBackrefMethodInfo Regex signature where
    overloadedMethod = regexGetMaxBackref

instance O.OverloadedMethodInfo RegexGetMaxBackrefMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexGetMaxBackref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexGetMaxBackref"
        })


#endif

-- method Regex::get_max_lookbehind
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_get_max_lookbehind" g_regex_get_max_lookbehind :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    IO Int32

-- | Gets the number of characters in the longest lookbehind assertion in the
-- pattern. This information is useful when doing multi-segment matching using
-- the partial matching facilities.
-- 
-- /Since: 2.38/
regexGetMaxLookbehind ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure
    -> m Int32
    -- ^ __Returns:__ the number of characters in the longest lookbehind assertion.
regexGetMaxLookbehind regex = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    result <- g_regex_get_max_lookbehind regex'
    touchManagedPtr regex
    return result

#if defined(ENABLE_OVERLOADING)
data RegexGetMaxLookbehindMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod RegexGetMaxLookbehindMethodInfo Regex signature where
    overloadedMethod = regexGetMaxLookbehind

instance O.OverloadedMethodInfo RegexGetMaxLookbehindMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexGetMaxLookbehind",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexGetMaxLookbehind"
        })


#endif

-- method Regex::get_pattern
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_get_pattern" g_regex_get_pattern :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    IO CString

-- | Gets the pattern string associated with /@regex@/, i.e. a copy of
-- the string passed to 'GI.GLib.Structs.Regex.regexNew'.
-- 
-- /Since: 2.14/
regexGetPattern ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure
    -> m T.Text
    -- ^ __Returns:__ the pattern of /@regex@/
regexGetPattern regex = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    result <- g_regex_get_pattern regex'
    checkUnexpectedReturnNULL "regexGetPattern" result
    result' <- cstringToText result
    touchManagedPtr regex
    return result'

#if defined(ENABLE_OVERLOADING)
data RegexGetPatternMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod RegexGetPatternMethodInfo Regex signature where
    overloadedMethod = regexGetPattern

instance O.OverloadedMethodInfo RegexGetPatternMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexGetPattern",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexGetPattern"
        })


#endif

-- method Regex::get_string_number
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GRegex structure" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "name of the subexpression"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_get_string_number" g_regex_get_string_number :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    CString ->                              -- name : TBasicType TUTF8
    IO Int32

-- | Retrieves the number of the subexpression named /@name@/.
-- 
-- /Since: 2.14/
regexGetStringNumber ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: t'GI.GLib.Structs.Regex.Regex' structure
    -> T.Text
    -- ^ /@name@/: name of the subexpression
    -> m Int32
    -- ^ __Returns:__ The number of the subexpression or -1 if /@name@/
    --   does not exists
regexGetStringNumber regex name = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    name' <- textToCString name
    result <- g_regex_get_string_number regex' name'
    touchManagedPtr regex
    freeMem name'
    return result

#if defined(ENABLE_OVERLOADING)
data RegexGetStringNumberMethodInfo
instance (signature ~ (T.Text -> m Int32), MonadIO m) => O.OverloadedMethod RegexGetStringNumberMethodInfo Regex signature where
    overloadedMethod = regexGetStringNumber

instance O.OverloadedMethodInfo RegexGetStringNumberMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexGetStringNumber",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexGetStringNumber"
        })


#endif

-- method Regex::match
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure from g_regex_new()"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to scan for matches"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "match options" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_info"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "MatchInfo" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "pointer to location where to store\n    the #GMatchInfo, or %NULL if you do not need it"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_match" g_regex_match :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    CString ->                              -- string : TBasicType TUTF8
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    Ptr (Ptr GLib.MatchInfo.MatchInfo) ->   -- match_info : TInterface (Name {namespace = "GLib", name = "MatchInfo"})
    IO CInt

-- | Scans for a match in /@string@/ for the pattern in /@regex@/.
-- The /@matchOptions@/ are combined with the match options specified
-- when the /@regex@/ structure was created, letting you have more
-- flexibility in reusing t'GI.GLib.Structs.Regex.Regex' structures.
-- 
-- Unless 'GI.GLib.Flags.RegexCompileFlagsRaw' is specified in the options, /@string@/ must be valid UTF-8.
-- 
-- A t'GI.GLib.Structs.MatchInfo.MatchInfo' structure, used to get information on the match,
-- is stored in /@matchInfo@/ if not 'P.Nothing'. Note that if /@matchInfo@/
-- is not 'P.Nothing' then it is created even if the function returns 'P.False',
-- i.e. you must free it regardless if regular expression actually matched.
-- 
-- To retrieve all the non-overlapping matches of the pattern in
-- string you can use 'GI.GLib.Structs.MatchInfo.matchInfoNext'.
-- 
-- 
-- === /C code/
-- >
-- >static void
-- >print_uppercase_words (const gchar *string)
-- >{
-- >  // Print all uppercase-only words.
-- >  GRegex *regex;
-- >  GMatchInfo *match_info;
-- > 
-- >  regex = g_regex_new ("[A-Z]+", G_REGEX_DEFAULT, G_REGEX_MATCH_DEFAULT, NULL);
-- >  g_regex_match (regex, string, 0, &match_info);
-- >  while (g_match_info_matches (match_info))
-- >    {
-- >      gchar *word = g_match_info_fetch (match_info, 0);
-- >      g_print ("Found: %s\n", word);
-- >      g_free (word);
-- >      g_match_info_next (match_info, NULL);
-- >    }
-- >  g_match_info_free (match_info);
-- >  g_regex_unref (regex);
-- >}
-- 
-- 
-- /@string@/ is not copied and is used in t'GI.GLib.Structs.MatchInfo.MatchInfo' internally. If
-- you use any t'GI.GLib.Structs.MatchInfo.MatchInfo' method (except 'GI.GLib.Structs.MatchInfo.matchInfoFree') after
-- freeing or modifying /@string@/ then the behaviour is undefined.
-- 
-- /Since: 2.14/
regexMatch ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure from 'GI.GLib.Structs.Regex.regexNew'
    -> T.Text
    -- ^ /@string@/: the string to scan for matches
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: match options
    -> m ((Bool, GLib.MatchInfo.MatchInfo))
    -- ^ __Returns:__ 'P.True' is the string matched, 'P.False' otherwise
regexMatch regex string matchOptions = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    string' <- textToCString string
    let matchOptions' = gflagsToWord matchOptions
    matchInfo <- callocMem :: IO (Ptr (Ptr GLib.MatchInfo.MatchInfo))
    result <- g_regex_match regex' string' matchOptions' matchInfo
    let result' = (/= 0) result
    matchInfo' <- peek matchInfo
    matchInfo'' <- (wrapBoxed GLib.MatchInfo.MatchInfo) matchInfo'
    touchManagedPtr regex
    freeMem string'
    freeMem matchInfo
    return (result', matchInfo'')

#if defined(ENABLE_OVERLOADING)
data RegexMatchMethodInfo
instance (signature ~ (T.Text -> [GLib.Flags.RegexMatchFlags] -> m ((Bool, GLib.MatchInfo.MatchInfo))), MonadIO m) => O.OverloadedMethod RegexMatchMethodInfo Regex signature where
    overloadedMethod = regexMatch

instance O.OverloadedMethodInfo RegexMatchMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexMatch",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexMatch"
        })


#endif

-- method Regex::match_all
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure from g_regex_new()"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to scan for matches"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "match options" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_info"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "MatchInfo" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "pointer to location where to store\n    the #GMatchInfo, or %NULL if you do not need it"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_match_all" g_regex_match_all :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    CString ->                              -- string : TBasicType TUTF8
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    Ptr (Ptr GLib.MatchInfo.MatchInfo) ->   -- match_info : TInterface (Name {namespace = "GLib", name = "MatchInfo"})
    IO CInt

-- | Using the standard algorithm for regular expression matching only
-- the longest match in the string is retrieved. This function uses
-- a different algorithm so it can retrieve all the possible matches.
-- For more documentation see 'GI.GLib.Structs.Regex.regexMatchAllFull'.
-- 
-- A t'GI.GLib.Structs.MatchInfo.MatchInfo' structure, used to get information on the match, is
-- stored in /@matchInfo@/ if not 'P.Nothing'. Note that if /@matchInfo@/ is
-- not 'P.Nothing' then it is created even if the function returns 'P.False',
-- i.e. you must free it regardless if regular expression actually
-- matched.
-- 
-- /@string@/ is not copied and is used in t'GI.GLib.Structs.MatchInfo.MatchInfo' internally. If
-- you use any t'GI.GLib.Structs.MatchInfo.MatchInfo' method (except 'GI.GLib.Structs.MatchInfo.matchInfoFree') after
-- freeing or modifying /@string@/ then the behaviour is undefined.
-- 
-- /Since: 2.14/
regexMatchAll ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure from 'GI.GLib.Structs.Regex.regexNew'
    -> T.Text
    -- ^ /@string@/: the string to scan for matches
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: match options
    -> m ((Bool, GLib.MatchInfo.MatchInfo))
    -- ^ __Returns:__ 'P.True' is the string matched, 'P.False' otherwise
regexMatchAll regex string matchOptions = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    string' <- textToCString string
    let matchOptions' = gflagsToWord matchOptions
    matchInfo <- callocMem :: IO (Ptr (Ptr GLib.MatchInfo.MatchInfo))
    result <- g_regex_match_all regex' string' matchOptions' matchInfo
    let result' = (/= 0) result
    matchInfo' <- peek matchInfo
    matchInfo'' <- (wrapBoxed GLib.MatchInfo.MatchInfo) matchInfo'
    touchManagedPtr regex
    freeMem string'
    freeMem matchInfo
    return (result', matchInfo'')

#if defined(ENABLE_OVERLOADING)
data RegexMatchAllMethodInfo
instance (signature ~ (T.Text -> [GLib.Flags.RegexMatchFlags] -> m ((Bool, GLib.MatchInfo.MatchInfo))), MonadIO m) => O.OverloadedMethod RegexMatchAllMethodInfo Regex signature where
    overloadedMethod = regexMatchAll

instance O.OverloadedMethodInfo RegexMatchAllMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexMatchAll",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexMatchAll"
        })


#endif

-- method Regex::match_all_full
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure from g_regex_new()"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TCArray False (-1) 2 (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to scan for matches"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string_len"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_position"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "starting index of the string to match, in bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "match options" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_info"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "MatchInfo" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "pointer to location where to store\n    the #GMatchInfo, or %NULL if you do not need it"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "string_len"
--              , argType = TBasicType TSSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_regex_match_all_full" g_regex_match_all_full :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    Ptr CString ->                          -- string : TCArray False (-1) 2 (TBasicType TUTF8)
    DI.Int64 ->                             -- string_len : TBasicType TSSize
    Int32 ->                                -- start_position : TBasicType TInt
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    Ptr (Ptr GLib.MatchInfo.MatchInfo) ->   -- match_info : TInterface (Name {namespace = "GLib", name = "MatchInfo"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Using the standard algorithm for regular expression matching only
-- the longest match in the /@string@/ is retrieved, it is not possible
-- to obtain all the available matches. For instance matching
-- @\"\<a> \<b> \<c>\"@ against the pattern @\"\<.*>\"@
-- you get @\"\<a> \<b> \<c>\"@.
-- 
-- This function uses a different algorithm (called DFA, i.e. deterministic
-- finite automaton), so it can retrieve all the possible matches, all
-- starting at the same point in the string. For instance matching
-- @\"\<a> \<b> \<c>\"@ against the pattern @\"\<.*>\"@
-- you would obtain three matches: @\"\<a> \<b> \<c>\"@,
-- @\"\<a> \<b>\"@ and @\"\<a>\"@.
-- 
-- The number of matched strings is retrieved using
-- 'GI.GLib.Structs.MatchInfo.matchInfoGetMatchCount'. To obtain the matched strings and
-- their position you can use, respectively, 'GI.GLib.Structs.MatchInfo.matchInfoFetch' and
-- 'GI.GLib.Structs.MatchInfo.matchInfoFetchPos'. Note that the strings are returned in
-- reverse order of length; that is, the longest matching string is
-- given first.
-- 
-- Note that the DFA algorithm is slower than the standard one and it
-- is not able to capture substrings, so backreferences do not work.
-- 
-- Setting /@startPosition@/ differs from just passing over a shortened
-- string and setting 'GI.GLib.Flags.RegexMatchFlagsNotbol' in the case of a pattern
-- that begins with any kind of lookbehind assertion, such as \"\\b\".
-- 
-- Unless 'GI.GLib.Flags.RegexCompileFlagsRaw' is specified in the options, /@string@/ must be valid UTF-8.
-- 
-- A t'GI.GLib.Structs.MatchInfo.MatchInfo' structure, used to get information on the match, is
-- stored in /@matchInfo@/ if not 'P.Nothing'. Note that if /@matchInfo@/ is
-- not 'P.Nothing' then it is created even if the function returns 'P.False',
-- i.e. you must free it regardless if regular expression actually
-- matched.
-- 
-- /@string@/ is not copied and is used in t'GI.GLib.Structs.MatchInfo.MatchInfo' internally. If
-- you use any t'GI.GLib.Structs.MatchInfo.MatchInfo' method (except 'GI.GLib.Structs.MatchInfo.matchInfoFree') after
-- freeing or modifying /@string@/ then the behaviour is undefined.
-- 
-- /Since: 2.14/
regexMatchAllFull ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure from 'GI.GLib.Structs.Regex.regexNew'
    -> [T.Text]
    -- ^ /@string@/: the string to scan for matches
    -> Int32
    -- ^ /@startPosition@/: starting index of the string to match, in bytes
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: match options
    -> m (GLib.MatchInfo.MatchInfo)
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
regexMatchAllFull regex string startPosition matchOptions = liftIO $ do
    let stringLen = fromIntegral $ P.length string
    regex' <- unsafeManagedPtrGetPtr regex
    string' <- packUTF8CArray string
    let matchOptions' = gflagsToWord matchOptions
    matchInfo <- callocMem :: IO (Ptr (Ptr GLib.MatchInfo.MatchInfo))
    onException (do
        _ <- propagateGError $ g_regex_match_all_full regex' string' stringLen startPosition matchOptions' matchInfo
        matchInfo' <- peek matchInfo
        matchInfo'' <- (wrapBoxed GLib.MatchInfo.MatchInfo) matchInfo'
        touchManagedPtr regex
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
        freeMem matchInfo
        return matchInfo''
     ) (do
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
        freeMem matchInfo
     )

#if defined(ENABLE_OVERLOADING)
data RegexMatchAllFullMethodInfo
instance (signature ~ ([T.Text] -> Int32 -> [GLib.Flags.RegexMatchFlags] -> m (GLib.MatchInfo.MatchInfo)), MonadIO m) => O.OverloadedMethod RegexMatchAllFullMethodInfo Regex signature where
    overloadedMethod = regexMatchAllFull

instance O.OverloadedMethodInfo RegexMatchAllFullMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexMatchAllFull",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexMatchAllFull"
        })


#endif

-- method Regex::match_full
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure from g_regex_new()"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TCArray False (-1) 2 (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to scan for matches"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string_len"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_position"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "starting index of the string to match, in bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "match options" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_info"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "MatchInfo" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "pointer to location where to store\n    the #GMatchInfo, or %NULL if you do not need it"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "string_len"
--              , argType = TBasicType TSSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_regex_match_full" g_regex_match_full :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    Ptr CString ->                          -- string : TCArray False (-1) 2 (TBasicType TUTF8)
    DI.Int64 ->                             -- string_len : TBasicType TSSize
    Int32 ->                                -- start_position : TBasicType TInt
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    Ptr (Ptr GLib.MatchInfo.MatchInfo) ->   -- match_info : TInterface (Name {namespace = "GLib", name = "MatchInfo"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Scans for a match in /@string@/ for the pattern in /@regex@/.
-- The /@matchOptions@/ are combined with the match options specified
-- when the /@regex@/ structure was created, letting you have more
-- flexibility in reusing t'GI.GLib.Structs.Regex.Regex' structures.
-- 
-- Setting /@startPosition@/ differs from just passing over a shortened
-- string and setting 'GI.GLib.Flags.RegexMatchFlagsNotbol' in the case of a pattern
-- that begins with any kind of lookbehind assertion, such as \"\\b\".
-- 
-- Unless 'GI.GLib.Flags.RegexCompileFlagsRaw' is specified in the options, /@string@/ must be valid UTF-8.
-- 
-- A t'GI.GLib.Structs.MatchInfo.MatchInfo' structure, used to get information on the match, is
-- stored in /@matchInfo@/ if not 'P.Nothing'. Note that if /@matchInfo@/ is
-- not 'P.Nothing' then it is created even if the function returns 'P.False',
-- i.e. you must free it regardless if regular expression actually
-- matched.
-- 
-- /@string@/ is not copied and is used in t'GI.GLib.Structs.MatchInfo.MatchInfo' internally. If
-- you use any t'GI.GLib.Structs.MatchInfo.MatchInfo' method (except 'GI.GLib.Structs.MatchInfo.matchInfoFree') after
-- freeing or modifying /@string@/ then the behaviour is undefined.
-- 
-- To retrieve all the non-overlapping matches of the pattern in
-- string you can use 'GI.GLib.Structs.MatchInfo.matchInfoNext'.
-- 
-- 
-- === /C code/
-- >
-- >static void
-- >print_uppercase_words (const gchar *string)
-- >{
-- >  // Print all uppercase-only words.
-- >  GRegex *regex;
-- >  GMatchInfo *match_info;
-- >  GError *error = NULL;
-- >  
-- >  regex = g_regex_new ("[A-Z]+", G_REGEX_DEFAULT, G_REGEX_MATCH_DEFAULT, NULL);
-- >  g_regex_match_full (regex, string, -1, 0, 0, &match_info, &error);
-- >  while (g_match_info_matches (match_info))
-- >    {
-- >      gchar *word = g_match_info_fetch (match_info, 0);
-- >      g_print ("Found: %s\n", word);
-- >      g_free (word);
-- >      g_match_info_next (match_info, &error);
-- >    }
-- >  g_match_info_free (match_info);
-- >  g_regex_unref (regex);
-- >  if (error != NULL)
-- >    {
-- >      g_printerr ("Error while matching: %s\n", error->message);
-- >      g_error_free (error);
-- >    }
-- >}
-- 
-- 
-- /Since: 2.14/
regexMatchFull ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure from 'GI.GLib.Structs.Regex.regexNew'
    -> [T.Text]
    -- ^ /@string@/: the string to scan for matches
    -> Int32
    -- ^ /@startPosition@/: starting index of the string to match, in bytes
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: match options
    -> m (GLib.MatchInfo.MatchInfo)
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
regexMatchFull regex string startPosition matchOptions = liftIO $ do
    let stringLen = fromIntegral $ P.length string
    regex' <- unsafeManagedPtrGetPtr regex
    string' <- packUTF8CArray string
    let matchOptions' = gflagsToWord matchOptions
    matchInfo <- callocMem :: IO (Ptr (Ptr GLib.MatchInfo.MatchInfo))
    onException (do
        _ <- propagateGError $ g_regex_match_full regex' string' stringLen startPosition matchOptions' matchInfo
        matchInfo' <- peek matchInfo
        matchInfo'' <- (wrapBoxed GLib.MatchInfo.MatchInfo) matchInfo'
        touchManagedPtr regex
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
        freeMem matchInfo
        return matchInfo''
     ) (do
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
        freeMem matchInfo
     )

#if defined(ENABLE_OVERLOADING)
data RegexMatchFullMethodInfo
instance (signature ~ ([T.Text] -> Int32 -> [GLib.Flags.RegexMatchFlags] -> m (GLib.MatchInfo.MatchInfo)), MonadIO m) => O.OverloadedMethod RegexMatchFullMethodInfo Regex signature where
    overloadedMethod = regexMatchFull

instance O.OverloadedMethodInfo RegexMatchFullMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexMatchFull",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexMatchFull"
        })


#endif

-- method Regex::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Regex" })
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_ref" g_regex_ref :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    IO (Ptr Regex)

-- | Increases reference count of /@regex@/ by 1.
-- 
-- /Since: 2.14/
regexRef ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex'
    -> m Regex
    -- ^ __Returns:__ /@regex@/
regexRef regex = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    result <- g_regex_ref regex'
    checkUnexpectedReturnNULL "regexRef" result
    result' <- (wrapBoxed Regex) result
    touchManagedPtr regex
    return result'

#if defined(ENABLE_OVERLOADING)
data RegexRefMethodInfo
instance (signature ~ (m Regex), MonadIO m) => O.OverloadedMethod RegexRefMethodInfo Regex signature where
    overloadedMethod = regexRef

instance O.OverloadedMethodInfo RegexRefMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexRef"
        })


#endif

-- method Regex::replace
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TCArray False (-1) 2 (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to perform matches against"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string_len"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_position"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "starting index of the string to match, in bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "replacement"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text to replace each match with"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "options for the match"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "string_len"
--              , argType = TBasicType TSSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_regex_replace" g_regex_replace :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    Ptr CString ->                          -- string : TCArray False (-1) 2 (TBasicType TUTF8)
    DI.Int64 ->                             -- string_len : TBasicType TSSize
    Int32 ->                                -- start_position : TBasicType TInt
    CString ->                              -- replacement : TBasicType TUTF8
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Replaces all occurrences of the pattern in /@regex@/ with the
-- replacement text. Backreferences of the form @\\number@ or
-- @\\g\<number>@ in the replacement text are interpolated by the
-- number-th captured subexpression of the match, @\\g\<name>@ refers
-- to the captured subexpression with the given name. @\\0@ refers
-- to the complete match, but @\\0@ followed by a number is the octal
-- representation of a character. To include a literal @\\@ in the
-- replacement, write @\\\\\\\\@.
-- 
-- There are also escapes that changes the case of the following text:
-- 
-- * \\l: Convert to lower case the next character
-- * \\u: Convert to upper case the next character
-- * \\L: Convert to lower case till \\E
-- * \\U: Convert to upper case till \\E
-- * \\E: End case modification
-- 
-- 
-- If you do not need to use backreferences use 'GI.GLib.Structs.Regex.regexReplaceLiteral'.
-- 
-- The /@replacement@/ string must be UTF-8 encoded even if 'GI.GLib.Flags.RegexCompileFlagsRaw' was
-- passed to 'GI.GLib.Structs.Regex.regexNew'. If you want to use not UTF-8 encoded strings
-- you can use 'GI.GLib.Structs.Regex.regexReplaceLiteral'.
-- 
-- Setting /@startPosition@/ differs from just passing over a shortened
-- string and setting 'GI.GLib.Flags.RegexMatchFlagsNotbol' in the case of a pattern that
-- begins with any kind of lookbehind assertion, such as \"\\b\".
-- 
-- /Since: 2.14/
regexReplace ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure
    -> [T.Text]
    -- ^ /@string@/: the string to perform matches against
    -> Int32
    -- ^ /@startPosition@/: starting index of the string to match, in bytes
    -> T.Text
    -- ^ /@replacement@/: text to replace each match with
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: options for the match
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string containing the replacements /(Can throw 'Data.GI.Base.GError.GError')/
regexReplace regex string startPosition replacement matchOptions = liftIO $ do
    let stringLen = fromIntegral $ P.length string
    regex' <- unsafeManagedPtrGetPtr regex
    string' <- packUTF8CArray string
    replacement' <- textToCString replacement
    let matchOptions' = gflagsToWord matchOptions
    onException (do
        result <- propagateGError $ g_regex_replace regex' string' stringLen startPosition replacement' matchOptions'
        checkUnexpectedReturnNULL "regexReplace" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr regex
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
        freeMem replacement'
        return result'
     ) (do
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
        freeMem replacement'
     )

#if defined(ENABLE_OVERLOADING)
data RegexReplaceMethodInfo
instance (signature ~ ([T.Text] -> Int32 -> T.Text -> [GLib.Flags.RegexMatchFlags] -> m T.Text), MonadIO m) => O.OverloadedMethod RegexReplaceMethodInfo Regex signature where
    overloadedMethod = regexReplace

instance O.OverloadedMethodInfo RegexReplaceMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexReplace",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexReplace"
        })


#endif

-- method Regex::replace_eval
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure from g_regex_new()"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TCArray False (-1) 2 (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "string to perform matches against"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string_len"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_position"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "starting index of the string to match, in bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "options for the match"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "eval"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexEvalCallback" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a function to call for each match"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeCall
--           , argClosure = 6
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data to pass to the function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "string_len"
--              , argType = TBasicType TSSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_regex_replace_eval" g_regex_replace_eval :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    Ptr CString ->                          -- string : TCArray False (-1) 2 (TBasicType TUTF8)
    DI.Int64 ->                             -- string_len : TBasicType TSSize
    Int32 ->                                -- start_position : TBasicType TInt
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    FunPtr GLib.Callbacks.C_RegexEvalCallback -> -- eval : TInterface (Name {namespace = "GLib", name = "RegexEvalCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Replaces occurrences of the pattern in regex with the output of
-- /@eval@/ for that occurrence.
-- 
-- Setting /@startPosition@/ differs from just passing over a shortened
-- string and setting 'GI.GLib.Flags.RegexMatchFlagsNotbol' in the case of a pattern
-- that begins with any kind of lookbehind assertion, such as \"\\b\".
-- 
-- The following example uses 'GI.GLib.Structs.Regex.regexReplaceEval' to replace multiple
-- strings at once:
-- 
-- === /C code/
-- >
-- >static gboolean
-- >eval_cb (const GMatchInfo *info,
-- >         GString          *res,
-- >         gpointer          data)
-- >{
-- >  gchar *match;
-- >  gchar *r;
-- >
-- >   match = g_match_info_fetch (info, 0);
-- >   r = g_hash_table_lookup ((GHashTable *)data, match);
-- >   g_string_append (res, r);
-- >   g_free (match);
-- >
-- >   return FALSE;
-- >}
-- >
-- >...
-- >
-- >GRegex *reg;
-- >GHashTable *h;
-- >gchar *res;
-- >
-- >h = g_hash_table_new (g_str_hash, g_str_equal);
-- >
-- >g_hash_table_insert (h, "1", "ONE");
-- >g_hash_table_insert (h, "2", "TWO");
-- >g_hash_table_insert (h, "3", "THREE");
-- >g_hash_table_insert (h, "4", "FOUR");
-- >
-- >reg = g_regex_new ("1|2|3|4", G_REGEX_DEFAULT, G_REGEX_MATCH_DEFAULT, NULL);
-- >res = g_regex_replace_eval (reg, text, -1, 0, 0, eval_cb, h, NULL);
-- >g_hash_table_destroy (h);
-- >
-- >...
-- 
-- 
-- /Since: 2.14/
regexReplaceEval ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure from 'GI.GLib.Structs.Regex.regexNew'
    -> [T.Text]
    -- ^ /@string@/: string to perform matches against
    -> Int32
    -- ^ /@startPosition@/: starting index of the string to match, in bytes
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: options for the match
    -> GLib.Callbacks.RegexEvalCallback
    -- ^ /@eval@/: a function to call for each match
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string containing the replacements /(Can throw 'Data.GI.Base.GError.GError')/
regexReplaceEval regex string startPosition matchOptions eval = liftIO $ do
    let stringLen = fromIntegral $ P.length string
    regex' <- unsafeManagedPtrGetPtr regex
    string' <- packUTF8CArray string
    let matchOptions' = gflagsToWord matchOptions
    eval' <- GLib.Callbacks.mk_RegexEvalCallback (GLib.Callbacks.wrap_RegexEvalCallback Nothing (GLib.Callbacks.drop_closures_RegexEvalCallback eval))
    let userData = nullPtr
    onException (do
        result <- propagateGError $ g_regex_replace_eval regex' string' stringLen startPosition matchOptions' eval' userData
        checkUnexpectedReturnNULL "regexReplaceEval" result
        result' <- cstringToText result
        freeMem result
        safeFreeFunPtr $ castFunPtrToPtr eval'
        touchManagedPtr regex
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
        return result'
     ) (do
        safeFreeFunPtr $ castFunPtrToPtr eval'
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
     )

#if defined(ENABLE_OVERLOADING)
data RegexReplaceEvalMethodInfo
instance (signature ~ ([T.Text] -> Int32 -> [GLib.Flags.RegexMatchFlags] -> GLib.Callbacks.RegexEvalCallback -> m T.Text), MonadIO m) => O.OverloadedMethod RegexReplaceEvalMethodInfo Regex signature where
    overloadedMethod = regexReplaceEval

instance O.OverloadedMethodInfo RegexReplaceEvalMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexReplaceEval",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexReplaceEval"
        })


#endif

-- method Regex::replace_literal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TCArray False (-1) 2 (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to perform matches against"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string_len"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_position"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "starting index of the string to match, in bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "replacement"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text to replace each match with"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "options for the match"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "string_len"
--              , argType = TBasicType TSSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_regex_replace_literal" g_regex_replace_literal :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    Ptr CString ->                          -- string : TCArray False (-1) 2 (TBasicType TUTF8)
    DI.Int64 ->                             -- string_len : TBasicType TSSize
    Int32 ->                                -- start_position : TBasicType TInt
    CString ->                              -- replacement : TBasicType TUTF8
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Replaces all occurrences of the pattern in /@regex@/ with the
-- replacement text. /@replacement@/ is replaced literally, to
-- include backreferences use 'GI.GLib.Structs.Regex.regexReplace'.
-- 
-- Setting /@startPosition@/ differs from just passing over a
-- shortened string and setting 'GI.GLib.Flags.RegexMatchFlagsNotbol' in the
-- case of a pattern that begins with any kind of lookbehind
-- assertion, such as \"\\b\".
-- 
-- /Since: 2.14/
regexReplaceLiteral ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure
    -> [T.Text]
    -- ^ /@string@/: the string to perform matches against
    -> Int32
    -- ^ /@startPosition@/: starting index of the string to match, in bytes
    -> T.Text
    -- ^ /@replacement@/: text to replace each match with
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: options for the match
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string containing the replacements /(Can throw 'Data.GI.Base.GError.GError')/
regexReplaceLiteral regex string startPosition replacement matchOptions = liftIO $ do
    let stringLen = fromIntegral $ P.length string
    regex' <- unsafeManagedPtrGetPtr regex
    string' <- packUTF8CArray string
    replacement' <- textToCString replacement
    let matchOptions' = gflagsToWord matchOptions
    onException (do
        result <- propagateGError $ g_regex_replace_literal regex' string' stringLen startPosition replacement' matchOptions'
        checkUnexpectedReturnNULL "regexReplaceLiteral" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr regex
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
        freeMem replacement'
        return result'
     ) (do
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
        freeMem replacement'
     )

#if defined(ENABLE_OVERLOADING)
data RegexReplaceLiteralMethodInfo
instance (signature ~ ([T.Text] -> Int32 -> T.Text -> [GLib.Flags.RegexMatchFlags] -> m T.Text), MonadIO m) => O.OverloadedMethod RegexReplaceLiteralMethodInfo Regex signature where
    overloadedMethod = regexReplaceLiteral

instance O.OverloadedMethodInfo RegexReplaceLiteralMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexReplaceLiteral",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexReplaceLiteral"
        })


#endif

-- method Regex::split
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to split with the pattern"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "match time option flags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_split" g_regex_split :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    CString ->                              -- string : TBasicType TUTF8
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    IO (Ptr CString)

-- | Breaks the string on the pattern, and returns an array of the tokens.
-- If the pattern contains capturing parentheses, then the text for each
-- of the substrings will also be returned. If the pattern does not match
-- anywhere in the string, then the whole string is returned as the first
-- token.
-- 
-- As a special case, the result of splitting the empty string \"\" is an
-- empty vector, not a vector containing a single string. The reason for
-- this special case is that being able to represent an empty vector is
-- typically more useful than consistent handling of empty elements. If
-- you do need to represent empty elements, you\'ll need to check for the
-- empty string before calling this function.
-- 
-- A pattern that can match empty strings splits /@string@/ into separate
-- characters wherever it matches the empty string between characters.
-- For example splitting \"ab c\" using as a separator \"\\s*\", you will get
-- \"a\", \"b\" and \"c\".
-- 
-- /Since: 2.14/
regexSplit ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure
    -> T.Text
    -- ^ /@string@/: the string to split with the pattern
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: match time option flags
    -> m [T.Text]
    -- ^ __Returns:__ a 'P.Nothing'-terminated gchar __ array. Free
    -- it using 'GI.GLib.Functions.strfreev'
regexSplit regex string matchOptions = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    string' <- textToCString string
    let matchOptions' = gflagsToWord matchOptions
    result <- g_regex_split regex' string' matchOptions'
    checkUnexpectedReturnNULL "regexSplit" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    touchManagedPtr regex
    freeMem string'
    return result'

#if defined(ENABLE_OVERLOADING)
data RegexSplitMethodInfo
instance (signature ~ (T.Text -> [GLib.Flags.RegexMatchFlags] -> m [T.Text]), MonadIO m) => O.OverloadedMethod RegexSplitMethodInfo Regex signature where
    overloadedMethod = regexSplit

instance O.OverloadedMethodInfo RegexSplitMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexSplit",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexSplit"
        })


#endif

-- method Regex::split_full
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex structure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TCArray False (-1) 2 (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to split with the pattern"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string_len"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_position"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "starting index of the string to match, in bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "match time option flags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "max_tokens"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the maximum number of tokens to split @string into.\n  If this is less than 1, the string is split completely"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "string_len"
--              , argType = TBasicType TSSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : True
-- Skip return : False

foreign import ccall "g_regex_split_full" g_regex_split_full :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    Ptr CString ->                          -- string : TCArray False (-1) 2 (TBasicType TUTF8)
    DI.Int64 ->                             -- string_len : TBasicType TSSize
    Int32 ->                                -- start_position : TBasicType TInt
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    Int32 ->                                -- max_tokens : TBasicType TInt
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr CString)

-- | Breaks the string on the pattern, and returns an array of the tokens.
-- If the pattern contains capturing parentheses, then the text for each
-- of the substrings will also be returned. If the pattern does not match
-- anywhere in the string, then the whole string is returned as the first
-- token.
-- 
-- As a special case, the result of splitting the empty string \"\" is an
-- empty vector, not a vector containing a single string. The reason for
-- this special case is that being able to represent an empty vector is
-- typically more useful than consistent handling of empty elements. If
-- you do need to represent empty elements, you\'ll need to check for the
-- empty string before calling this function.
-- 
-- A pattern that can match empty strings splits /@string@/ into separate
-- characters wherever it matches the empty string between characters.
-- For example splitting \"ab c\" using as a separator \"\\s*\", you will get
-- \"a\", \"b\" and \"c\".
-- 
-- Setting /@startPosition@/ differs from just passing over a shortened
-- string and setting 'GI.GLib.Flags.RegexMatchFlagsNotbol' in the case of a pattern
-- that begins with any kind of lookbehind assertion, such as \"\\b\".
-- 
-- /Since: 2.14/
regexSplitFull ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex' structure
    -> [T.Text]
    -- ^ /@string@/: the string to split with the pattern
    -> Int32
    -- ^ /@startPosition@/: starting index of the string to match, in bytes
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: match time option flags
    -> Int32
    -- ^ /@maxTokens@/: the maximum number of tokens to split /@string@/ into.
    --   If this is less than 1, the string is split completely
    -> m [T.Text]
    -- ^ __Returns:__ a 'P.Nothing'-terminated gchar __ array. Free
    -- it using 'GI.GLib.Functions.strfreev' /(Can throw 'Data.GI.Base.GError.GError')/
regexSplitFull regex string startPosition matchOptions maxTokens = liftIO $ do
    let stringLen = fromIntegral $ P.length string
    regex' <- unsafeManagedPtrGetPtr regex
    string' <- packUTF8CArray string
    let matchOptions' = gflagsToWord matchOptions
    onException (do
        result <- propagateGError $ g_regex_split_full regex' string' stringLen startPosition matchOptions' maxTokens
        checkUnexpectedReturnNULL "regexSplitFull" result
        result' <- unpackZeroTerminatedUTF8CArray result
        mapZeroTerminatedCArray freeMem result
        freeMem result
        touchManagedPtr regex
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
        return result'
     ) (do
        (mapCArrayWithLength stringLen) freeMem string'
        freeMem string'
     )

#if defined(ENABLE_OVERLOADING)
data RegexSplitFullMethodInfo
instance (signature ~ ([T.Text] -> Int32 -> [GLib.Flags.RegexMatchFlags] -> Int32 -> m [T.Text]), MonadIO m) => O.OverloadedMethod RegexSplitFullMethodInfo Regex signature where
    overloadedMethod = regexSplitFull

instance O.OverloadedMethodInfo RegexSplitFullMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexSplitFull",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexSplitFull"
        })


#endif

-- method Regex::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "regex"
--           , argType = TInterface Name { namespace = "GLib" , name = "Regex" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GRegex" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_unref" g_regex_unref :: 
    Ptr Regex ->                            -- regex : TInterface (Name {namespace = "GLib", name = "Regex"})
    IO ()

-- | Decreases reference count of /@regex@/ by 1. When reference count drops
-- to zero, it frees all the memory associated with the regex structure.
-- 
-- /Since: 2.14/
regexUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Regex
    -- ^ /@regex@/: a t'GI.GLib.Structs.Regex.Regex'
    -> m ()
regexUnref regex = liftIO $ do
    regex' <- unsafeManagedPtrGetPtr regex
    g_regex_unref regex'
    touchManagedPtr regex
    return ()

#if defined(ENABLE_OVERLOADING)
data RegexUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod RegexUnrefMethodInfo Regex signature where
    overloadedMethod = regexUnref

instance O.OverloadedMethodInfo RegexUnrefMethodInfo Regex where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Regex.regexUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Regex.html#v:regexUnref"
        })


#endif

-- method Regex::check_replacement
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "replacement"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the replacement string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "has_references"
--           , argType = TBasicType TBoolean
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location to store information about\n  references in @replacement or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_regex_check_replacement" g_regex_check_replacement :: 
    CString ->                              -- replacement : TBasicType TUTF8
    Ptr CInt ->                             -- has_references : TBasicType TBoolean
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Checks whether /@replacement@/ is a valid replacement string
-- (see 'GI.GLib.Structs.Regex.regexReplace'), i.e. that all escape sequences in
-- it are valid.
-- 
-- If /@hasReferences@/ is not 'P.Nothing' then /@replacement@/ is checked
-- for pattern references. For instance, replacement text \'foo\\n\'
-- does not contain references and may be evaluated without information
-- about actual match, but \'\\0\\1\' (whole match followed by first
-- subpattern) requires valid t'GI.GLib.Structs.MatchInfo.MatchInfo' object.
-- 
-- /Since: 2.14/
regexCheckReplacement ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@replacement@/: the replacement string
    -> m (Bool)
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
regexCheckReplacement replacement = liftIO $ do
    replacement' <- textToCString replacement
    hasReferences <- allocMem :: IO (Ptr CInt)
    onException (do
        _ <- propagateGError $ g_regex_check_replacement replacement' hasReferences
        hasReferences' <- peek hasReferences
        let hasReferences'' = (/= 0) hasReferences'
        freeMem replacement'
        freeMem hasReferences
        return hasReferences''
     ) (do
        freeMem replacement'
        freeMem hasReferences
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Regex::error_quark
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TBasicType TUInt32)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_error_quark" g_regex_error_quark :: 
    IO Word32

-- | /No description available in the introspection data./
regexErrorQuark ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Word32
regexErrorQuark  = liftIO $ do
    result <- g_regex_error_quark
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method Regex::escape_nul
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to escape"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of @string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_escape_nul" g_regex_escape_nul :: 
    CString ->                              -- string : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    IO CString

-- | Escapes the nul characters in /@string@/ to \"\\x00\".  It can be used
-- to compile a regex with embedded nul characters.
-- 
-- For completeness, /@length@/ can be -1 for a nul-terminated string.
-- In this case the output string will be of course equal to /@string@/.
-- 
-- /Since: 2.30/
regexEscapeNul ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@string@/: the string to escape
    -> Int32
    -- ^ /@length@/: the length of /@string@/
    -> m T.Text
    -- ^ __Returns:__ a newly-allocated escaped string
regexEscapeNul string length_ = liftIO $ do
    string' <- textToCString string
    result <- g_regex_escape_nul string' length_
    checkUnexpectedReturnNULL "regexEscapeNul" result
    result' <- cstringToText result
    freeMem result
    freeMem string'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Regex::escape_string
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to escape"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @string, in bytes, or -1 if @string is nul-terminated"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_escape_string" g_regex_escape_string :: 
    CString ->                              -- string : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    IO CString

-- | Escapes the special characters used for regular expressions
-- in /@string@/, for instance \"a.b*c\" becomes \"a\\.b\\*c\". This
-- function is useful to dynamically generate regular expressions.
-- 
-- /@string@/ can contain nul characters that are replaced with \"\\0\",
-- in this case remember to specify the correct length of /@string@/
-- in /@length@/.
-- 
-- /Since: 2.14/
regexEscapeString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@string@/: the string to escape
    -> Int32
    -- ^ /@length@/: the length of /@string@/, in bytes, or -1 if /@string@/ is nul-terminated
    -> m T.Text
    -- ^ __Returns:__ a newly-allocated escaped string
regexEscapeString string length_ = liftIO $ do
    string' <- textToCString string
    result <- g_regex_escape_string string' length_
    checkUnexpectedReturnNULL "regexEscapeString" result
    result' <- cstringToText result
    freeMem result
    freeMem string'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Regex::match_simple
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "pattern"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the regular expression"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to scan for matches"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "compile_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexCompileFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "compile options for the regular expression, or 0"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "match options, or 0"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_match_simple" g_regex_match_simple :: 
    CString ->                              -- pattern : TBasicType TUTF8
    CString ->                              -- string : TBasicType TUTF8
    CUInt ->                                -- compile_options : TInterface (Name {namespace = "GLib", name = "RegexCompileFlags"})
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    IO CInt

-- | Scans for a match in /@string@/ for /@pattern@/.
-- 
-- This function is equivalent to 'GI.GLib.Structs.Regex.regexMatch' but it does not
-- require to compile the pattern with 'GI.GLib.Structs.Regex.regexNew', avoiding some
-- lines of code when you need just to do a match without extracting
-- substrings, capture counts, and so on.
-- 
-- If this function is to be called on the same /@pattern@/ more than
-- once, it\'s more efficient to compile the pattern once with
-- 'GI.GLib.Structs.Regex.regexNew' and then use 'GI.GLib.Structs.Regex.regexMatch'.
-- 
-- /Since: 2.14/
regexMatchSimple ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@pattern@/: the regular expression
    -> T.Text
    -- ^ /@string@/: the string to scan for matches
    -> [GLib.Flags.RegexCompileFlags]
    -- ^ /@compileOptions@/: compile options for the regular expression, or 0
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: match options, or 0
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the string matched, 'P.False' otherwise
regexMatchSimple pattern string compileOptions matchOptions = liftIO $ do
    pattern' <- textToCString pattern
    string' <- textToCString string
    let compileOptions' = gflagsToWord compileOptions
    let matchOptions' = gflagsToWord matchOptions
    result <- g_regex_match_simple pattern' string' compileOptions' matchOptions'
    let result' = (/= 0) result
    freeMem pattern'
    freeMem string'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Regex::split_simple
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "pattern"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the regular expression"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to scan for matches"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "compile_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexCompileFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "compile options for the regular expression, or 0"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "match_options"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "RegexMatchFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "match options, or 0"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_regex_split_simple" g_regex_split_simple :: 
    CString ->                              -- pattern : TBasicType TUTF8
    CString ->                              -- string : TBasicType TUTF8
    CUInt ->                                -- compile_options : TInterface (Name {namespace = "GLib", name = "RegexCompileFlags"})
    CUInt ->                                -- match_options : TInterface (Name {namespace = "GLib", name = "RegexMatchFlags"})
    IO (Ptr CString)

-- | Breaks the string on the pattern, and returns an array of
-- the tokens. If the pattern contains capturing parentheses,
-- then the text for each of the substrings will also be returned.
-- If the pattern does not match anywhere in the string, then the
-- whole string is returned as the first token.
-- 
-- This function is equivalent to 'GI.GLib.Structs.Regex.regexSplit' but it does
-- not require to compile the pattern with 'GI.GLib.Structs.Regex.regexNew', avoiding
-- some lines of code when you need just to do a split without
-- extracting substrings, capture counts, and so on.
-- 
-- If this function is to be called on the same /@pattern@/ more than
-- once, it\'s more efficient to compile the pattern once with
-- 'GI.GLib.Structs.Regex.regexNew' and then use 'GI.GLib.Structs.Regex.regexSplit'.
-- 
-- As a special case, the result of splitting the empty string \"\"
-- is an empty vector, not a vector containing a single string.
-- The reason for this special case is that being able to represent
-- an empty vector is typically more useful than consistent handling
-- of empty elements. If you do need to represent empty elements,
-- you\'ll need to check for the empty string before calling this
-- function.
-- 
-- A pattern that can match empty strings splits /@string@/ into
-- separate characters wherever it matches the empty string between
-- characters. For example splitting \"ab c\" using as a separator
-- \"\\s*\", you will get \"a\", \"b\" and \"c\".
-- 
-- /Since: 2.14/
regexSplitSimple ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@pattern@/: the regular expression
    -> T.Text
    -- ^ /@string@/: the string to scan for matches
    -> [GLib.Flags.RegexCompileFlags]
    -- ^ /@compileOptions@/: compile options for the regular expression, or 0
    -> [GLib.Flags.RegexMatchFlags]
    -- ^ /@matchOptions@/: match options, or 0
    -> m [T.Text]
    -- ^ __Returns:__ a 'P.Nothing'-terminated array of strings. Free
    -- it using 'GI.GLib.Functions.strfreev'
regexSplitSimple pattern string compileOptions matchOptions = liftIO $ do
    pattern' <- textToCString pattern
    string' <- textToCString string
    let compileOptions' = gflagsToWord compileOptions
    let matchOptions' = gflagsToWord matchOptions
    result <- g_regex_split_simple pattern' string' compileOptions' matchOptions'
    checkUnexpectedReturnNULL "regexSplitSimple" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    freeMem pattern'
    freeMem string'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveRegexMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveRegexMethod "match" o = RegexMatchMethodInfo
    ResolveRegexMethod "matchAll" o = RegexMatchAllMethodInfo
    ResolveRegexMethod "matchAllFull" o = RegexMatchAllFullMethodInfo
    ResolveRegexMethod "matchFull" o = RegexMatchFullMethodInfo
    ResolveRegexMethod "ref" o = RegexRefMethodInfo
    ResolveRegexMethod "replace" o = RegexReplaceMethodInfo
    ResolveRegexMethod "replaceEval" o = RegexReplaceEvalMethodInfo
    ResolveRegexMethod "replaceLiteral" o = RegexReplaceLiteralMethodInfo
    ResolveRegexMethod "split" o = RegexSplitMethodInfo
    ResolveRegexMethod "splitFull" o = RegexSplitFullMethodInfo
    ResolveRegexMethod "unref" o = RegexUnrefMethodInfo
    ResolveRegexMethod "getCaptureCount" o = RegexGetCaptureCountMethodInfo
    ResolveRegexMethod "getCompileFlags" o = RegexGetCompileFlagsMethodInfo
    ResolveRegexMethod "getHasCrOrLf" o = RegexGetHasCrOrLfMethodInfo
    ResolveRegexMethod "getMatchFlags" o = RegexGetMatchFlagsMethodInfo
    ResolveRegexMethod "getMaxBackref" o = RegexGetMaxBackrefMethodInfo
    ResolveRegexMethod "getMaxLookbehind" o = RegexGetMaxLookbehindMethodInfo
    ResolveRegexMethod "getPattern" o = RegexGetPatternMethodInfo
    ResolveRegexMethod "getStringNumber" o = RegexGetStringNumberMethodInfo
    ResolveRegexMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveRegexMethod t Regex, O.OverloadedMethod info Regex p) => OL.IsLabel t (Regex -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveRegexMethod t Regex, O.OverloadedMethod info Regex p, R.HasField t Regex p) => R.HasField t Regex p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveRegexMethod t Regex, O.OverloadedMethodInfo info Regex) => OL.IsLabel t (O.MethodProxy info Regex) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


