{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A render node for a Cairo surface.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gsk.Objects.CairoNode
    ( 

-- * Exported types
    CairoNode(..)                           ,
    IsCairoNode                             ,
    toCairoNode                             ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [draw]("GI.Gsk.Objects.RenderNode#g:method:draw"), [ref]("GI.Gsk.Objects.RenderNode#g:method:ref"), [serialize]("GI.Gsk.Objects.RenderNode#g:method:serialize"), [unref]("GI.Gsk.Objects.RenderNode#g:method:unref"), [writeToFile]("GI.Gsk.Objects.RenderNode#g:method:writeToFile").
-- 
-- ==== Getters
-- [getBounds]("GI.Gsk.Objects.RenderNode#g:method:getBounds"), [getDrawContext]("GI.Gsk.Objects.CairoNode#g:method:getDrawContext"), [getNodeType]("GI.Gsk.Objects.RenderNode#g:method:getNodeType"), [getOpaqueRect]("GI.Gsk.Objects.RenderNode#g:method:getOpaqueRect"), [getSurface]("GI.Gsk.Objects.CairoNode#g:method:getSurface").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveCairoNodeMethod                  ,
#endif

-- ** getDrawContext #method:getDrawContext#

#if defined(ENABLE_OVERLOADING)
    CairoNodeGetDrawContextMethodInfo       ,
#endif
    cairoNodeGetDrawContext                 ,


-- ** getSurface #method:getSurface#

#if defined(ENABLE_OVERLOADING)
    CairoNodeGetSurfaceMethodInfo           ,
#endif
    cairoNodeGetSurface                     ,


-- ** new #method:new#

    cairoNodeNew                            ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.Surface as Cairo.Surface
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Gsk.Callbacks as Gsk.Callbacks
import {-# SOURCE #-} qualified GI.Gsk.Enums as Gsk.Enums
import {-# SOURCE #-} qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode

#else
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.Surface as Cairo.Surface
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import {-# SOURCE #-} qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode

#endif

-- | Memory-managed wrapper type.
newtype CairoNode = CairoNode (SP.ManagedPtr CairoNode)
    deriving (Eq)

instance SP.ManagedPtrNewtype CairoNode where
    toManagedPtr (CairoNode p) = p

foreign import ccall "gsk_cairo_node_get_type"
    c_gsk_cairo_node_get_type :: IO B.Types.GType

instance B.Types.TypedObject CairoNode where
    glibType = c_gsk_cairo_node_get_type

-- | Type class for types which can be safely cast to t'CairoNode', for instance with `toCairoNode`.
class (SP.BoxedPtr o, SP.TypedObject o, O.IsDescendantOf CairoNode o) => IsCairoNode o
instance (SP.BoxedPtr o, SP.TypedObject o, O.IsDescendantOf CairoNode o) => IsCairoNode o

instance O.HasParentTypes CairoNode
type instance O.ParentTypes CairoNode = '[Gsk.RenderNode.RenderNode]

-- | Cast to t'CairoNode', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toCairoNode :: (MIO.MonadIO m, IsCairoNode o) => o -> m CairoNode
toCairoNode = MIO.liftIO . B.ManagedPtr.unsafeCastTo CairoNode

--- XXX Missing getter and/or setter, so no GValue instance could be generated.
#if defined(ENABLE_OVERLOADING)
type family ResolveCairoNodeMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveCairoNodeMethod "draw" o = Gsk.RenderNode.RenderNodeDrawMethodInfo
    ResolveCairoNodeMethod "ref" o = Gsk.RenderNode.RenderNodeRefMethodInfo
    ResolveCairoNodeMethod "serialize" o = Gsk.RenderNode.RenderNodeSerializeMethodInfo
    ResolveCairoNodeMethod "unref" o = Gsk.RenderNode.RenderNodeUnrefMethodInfo
    ResolveCairoNodeMethod "writeToFile" o = Gsk.RenderNode.RenderNodeWriteToFileMethodInfo
    ResolveCairoNodeMethod "getBounds" o = Gsk.RenderNode.RenderNodeGetBoundsMethodInfo
    ResolveCairoNodeMethod "getDrawContext" o = CairoNodeGetDrawContextMethodInfo
    ResolveCairoNodeMethod "getNodeType" o = Gsk.RenderNode.RenderNodeGetNodeTypeMethodInfo
    ResolveCairoNodeMethod "getOpaqueRect" o = Gsk.RenderNode.RenderNodeGetOpaqueRectMethodInfo
    ResolveCairoNodeMethod "getSurface" o = CairoNodeGetSurfaceMethodInfo
    ResolveCairoNodeMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveCairoNodeMethod t CairoNode, O.OverloadedMethod info CairoNode p) => OL.IsLabel t (CairoNode -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveCairoNodeMethod t CairoNode, O.OverloadedMethod info CairoNode p, R.HasField t CairoNode p) => R.HasField t CairoNode p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveCairoNodeMethod t CairoNode, O.OverloadedMethodInfo info CairoNode) => OL.IsLabel t (O.MethodProxy info CairoNode) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- XXX Wrapping a foreign struct/union with no known destructor or size, leak?
instance BoxedPtr CairoNode where
    boxedPtrCopy = return
    boxedPtrFree = \_x -> return ()


-- method CairoNode::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "bounds"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Rect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rectangle to render to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "CairoNode" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_cairo_node_new" gsk_cairo_node_new :: 
    Ptr Graphene.Rect.Rect ->               -- bounds : TInterface (Name {namespace = "Graphene", name = "Rect"})
    IO (Ptr CairoNode)

-- | Creates a @GskRenderNode@ that will render a cairo surface
-- into the area given by /@bounds@/.
-- 
-- You can draw to the cairo surface using 'GI.Gsk.Objects.CairoNode.cairoNodeGetDrawContext'.
cairoNodeNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Graphene.Rect.Rect
    -- ^ /@bounds@/: the rectangle to render to
    -> m CairoNode
    -- ^ __Returns:__ A new @GskRenderNode@
cairoNodeNew bounds = liftIO $ do
    bounds' <- unsafeManagedPtrGetPtr bounds
    result <- gsk_cairo_node_new bounds'
    checkUnexpectedReturnNULL "cairoNodeNew" result
    result' <- (wrapPtr CairoNode) result
    touchManagedPtr bounds
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method CairoNode::get_draw_context
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "CairoNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskRenderNode` for a Cairo surface"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "cairo" , name = "Context" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_cairo_node_get_draw_context" gsk_cairo_node_get_draw_context :: 
    Ptr CairoNode ->                        -- node : TInterface (Name {namespace = "Gsk", name = "CairoNode"})
    IO (Ptr Cairo.Context.Context)

-- | Creates a Cairo context for drawing using the surface associated
-- to the render node.
-- 
-- If no surface exists yet, a surface will be created optimized for
-- rendering to /@renderer@/.
cairoNodeGetDrawContext ::
    (B.CallStack.HasCallStack, MonadIO m, IsCairoNode a) =>
    a
    -- ^ /@node@/: a @GskRenderNode@ for a Cairo surface
    -> m Cairo.Context.Context
    -- ^ __Returns:__ a Cairo context used for drawing; use
    --   @/cairo_destroy()/@ when done drawing
cairoNodeGetDrawContext node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    result <- gsk_cairo_node_get_draw_context node'
    checkUnexpectedReturnNULL "cairoNodeGetDrawContext" result
    result' <- (wrapBoxed Cairo.Context.Context) result
    touchManagedPtr node
    return result'

#if defined(ENABLE_OVERLOADING)
data CairoNodeGetDrawContextMethodInfo
instance (signature ~ (m Cairo.Context.Context), MonadIO m, IsCairoNode a) => O.OverloadedMethod CairoNodeGetDrawContextMethodInfo a signature where
    overloadedMethod = cairoNodeGetDrawContext

instance O.OverloadedMethodInfo CairoNodeGetDrawContextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.CairoNode.cairoNodeGetDrawContext",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-CairoNode.html#v:cairoNodeGetDrawContext"
        })


#endif

-- method CairoNode::get_surface
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "CairoNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskRenderNode` for a Cairo surface"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "cairo" , name = "Surface" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_cairo_node_get_surface" gsk_cairo_node_get_surface :: 
    Ptr CairoNode ->                        -- node : TInterface (Name {namespace = "Gsk", name = "CairoNode"})
    IO (Ptr Cairo.Surface.Surface)

-- | Retrieves the Cairo surface used by the render node.
cairoNodeGetSurface ::
    (B.CallStack.HasCallStack, MonadIO m, IsCairoNode a) =>
    a
    -- ^ /@node@/: a @GskRenderNode@ for a Cairo surface
    -> m Cairo.Surface.Surface
    -- ^ __Returns:__ a Cairo surface
cairoNodeGetSurface node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    result <- gsk_cairo_node_get_surface node'
    checkUnexpectedReturnNULL "cairoNodeGetSurface" result
    result' <- (newBoxed Cairo.Surface.Surface) result
    touchManagedPtr node
    return result'

#if defined(ENABLE_OVERLOADING)
data CairoNodeGetSurfaceMethodInfo
instance (signature ~ (m Cairo.Surface.Surface), MonadIO m, IsCairoNode a) => O.OverloadedMethod CairoNodeGetSurfaceMethodInfo a signature where
    overloadedMethod = cairoNodeGetSurface

instance O.OverloadedMethodInfo CairoNodeGetSurfaceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.CairoNode.cairoNodeGetSurface",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-CairoNode.html#v:cairoNodeGetSurface"
        })


#endif


