#pragma once
#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <initializer_list>
#include <iterator>
#include <memory>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>
#if __cplusplus >= 201703L
#include <string_view>
#endif
#if __cplusplus >= 202002L
#include <ranges>
#endif

#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdollar-in-identifier-extension"
#endif // __clang__

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

class Opaque;

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);
#ifdef __cpp_char8_t
  String(const char8_t *s);
  String(const char8_t *s, std::size_t len);
#endif

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) & noexcept;
  String &operator=(String &&) & noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_STR
#define CXXBRIDGE1_RUST_STR
class Str final {
public:
  Str() noexcept;
  Str(const String &) noexcept;
  Str(const std::string &);
  Str(const char *);
  Str(const char *, std::size_t);

  Str &operator=(const Str &) & noexcept = default;

  explicit operator std::string() const;
#if __cplusplus >= 201703L
  explicit operator std::string_view() const;
#endif

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  Str(const Str &) noexcept = default;
  ~Str() noexcept = default;

  using iterator = const char *;
  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const Str &) const noexcept;
  bool operator!=(const Str &) const noexcept;
  bool operator<(const Str &) const noexcept;
  bool operator<=(const Str &) const noexcept;
  bool operator>(const Str &) const noexcept;
  bool operator>=(const Str &) const noexcept;

  void swap(Str &) noexcept;

private:
  class uninit;
  Str(uninit) noexcept;
  friend impl<Str>;

  std::array<std::uintptr_t, 2> repr;
};
#endif // CXXBRIDGE1_RUST_STR

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) & noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) & noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C &c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) & noexcept = default;
  Slice &operator=(Slice<T> &&) & noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

#ifdef __cpp_deduction_guides
template <typename C>
explicit Slice(C &c)
    -> Slice<std::remove_reference_t<decltype(*std::declval<C>().data())>>;
#endif // __cpp_deduction_guides

template <typename T>
class Slice<T>::iterator final {
public:
#if __cplusplus >= 202002L
  using iterator_category = std::contiguous_iterator_tag;
#else
  using iterator_category = std::random_access_iterator_tag;
#endif
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  friend inline iterator operator+(difference_type lhs, iterator rhs) noexcept {
    return rhs + lhs;
  }
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

#if __cplusplus >= 202002L
static_assert(std::ranges::contiguous_range<rust::Slice<const uint8_t>>);
static_assert(std::contiguous_iterator<rust::Slice<const uint8_t>::iterator>);
#endif

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) & noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) & noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT
} // namespace cxxbridge1
} // namespace rust

#if __cplusplus >= 201402L
#define CXX_DEFAULT_VALUE(value) = value
#else
#define CXX_DEFAULT_VALUE(value)
#endif

namespace pdns {
  namespace rust {
    namespace settings {
      namespace rec {
        struct ForwardZone;
        struct AuthZone;
        struct TrustAnchor;
        struct NegativeTrustAnchor;
        struct ProtobufServer;
        struct DNSTapFrameStreamServer;
        struct DNSTapNODFrameStreamServer;
        struct TSIGTriplet;
        struct RPZ;
        struct ZoneToCache;
        struct SubnetOrder;
        struct SortList;
        struct AllowedAdditionalQType;
        struct ProxyMapping;
        struct ApiZones;
        struct XFR;
        struct FCZDefault;
        struct ForwardingCatalogZone;
        struct IncomingTLS;
        struct IncomingWSConfig;
        struct Value;
        struct OldStyle;
        struct Dnssec;
        struct Incoming;
        struct Recursor;
        struct Webservice;
        struct Carbon;
        struct Outgoing;
        struct Packetcache;
        struct Logging;
        struct Ecs;
        struct Recordcache;
        struct Nod;
        struct Snmp;
        struct Recursorsettings;
      }
    }
  }
}

namespace pdns {
namespace rust {
namespace settings {
namespace rec {
#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardZone
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardZone
struct ForwardZone final {
  ::rust::String zone;
  ::rust::Vec<::rust::String> forwarders;
  bool recurse CXX_DEFAULT_VALUE(false);
  bool notify_allowed CXX_DEFAULT_VALUE(false);

  void validate(::rust::Str field) const;
  bool operator==(ForwardZone const &) const noexcept;
  bool operator!=(ForwardZone const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardZone

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AuthZone
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AuthZone
struct AuthZone final {
  ::rust::String zone;
  ::rust::String file;

  void validate(::rust::Str field) const;
  bool operator==(AuthZone const &) const noexcept;
  bool operator!=(AuthZone const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AuthZone

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TrustAnchor
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TrustAnchor
struct TrustAnchor final {
  ::rust::String name;
  ::rust::Vec<::rust::String> dsrecords;

  void validate(::rust::Str field) const;
  bool operator==(TrustAnchor const &) const noexcept;
  bool operator!=(TrustAnchor const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TrustAnchor

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$NegativeTrustAnchor
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$NegativeTrustAnchor
struct NegativeTrustAnchor final {
  ::rust::String name;
  ::rust::String reason;

  void validate(::rust::Str field) const;
  bool operator==(NegativeTrustAnchor const &) const noexcept;
  bool operator!=(NegativeTrustAnchor const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$NegativeTrustAnchor

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProtobufServer
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProtobufServer
struct ProtobufServer final {
  ::rust::Vec<::rust::String> servers;
  ::std::uint64_t timeout CXX_DEFAULT_VALUE(0);
  ::std::uint64_t maxQueuedEntries CXX_DEFAULT_VALUE(0);
  ::std::uint64_t reconnectWaitTime CXX_DEFAULT_VALUE(0);
  bool taggedOnly CXX_DEFAULT_VALUE(false);
  bool asyncConnect CXX_DEFAULT_VALUE(false);
  bool logQueries CXX_DEFAULT_VALUE(false);
  bool logResponses CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::rust::String> exportTypes;
  bool logMappedFrom CXX_DEFAULT_VALUE(false);

  bool operator==(ProtobufServer const &) const noexcept;
  bool operator!=(ProtobufServer const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProtobufServer

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapFrameStreamServer
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapFrameStreamServer
struct DNSTapFrameStreamServer final {
  ::rust::Vec<::rust::String> servers;
  bool logQueries CXX_DEFAULT_VALUE(false);
  bool logResponses CXX_DEFAULT_VALUE(false);
  ::std::uint64_t bufferHint CXX_DEFAULT_VALUE(0);
  ::std::uint64_t flushTimeout CXX_DEFAULT_VALUE(0);
  ::std::uint64_t inputQueueSize CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outputQueueSize CXX_DEFAULT_VALUE(0);
  ::std::uint64_t queueNotifyThreshold CXX_DEFAULT_VALUE(0);
  ::std::uint64_t reopenInterval CXX_DEFAULT_VALUE(0);

  bool operator==(DNSTapFrameStreamServer const &) const noexcept;
  bool operator!=(DNSTapFrameStreamServer const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapFrameStreamServer

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapNODFrameStreamServer
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapNODFrameStreamServer
struct DNSTapNODFrameStreamServer final {
  ::rust::Vec<::rust::String> servers;
  bool logNODs CXX_DEFAULT_VALUE(false);
  bool logUDRs CXX_DEFAULT_VALUE(false);
  ::std::uint64_t bufferHint CXX_DEFAULT_VALUE(0);
  ::std::uint64_t flushTimeout CXX_DEFAULT_VALUE(0);
  ::std::uint64_t inputQueueSize CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outputQueueSize CXX_DEFAULT_VALUE(0);
  ::std::uint64_t queueNotifyThreshold CXX_DEFAULT_VALUE(0);
  ::std::uint64_t reopenInterval CXX_DEFAULT_VALUE(0);

  bool operator==(DNSTapNODFrameStreamServer const &) const noexcept;
  bool operator!=(DNSTapNODFrameStreamServer const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapNODFrameStreamServer

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TSIGTriplet
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TSIGTriplet
struct TSIGTriplet final {
  ::rust::String name;
  ::rust::String algo;
  ::rust::String secret;

  bool operator==(TSIGTriplet const &) const noexcept;
  bool operator!=(TSIGTriplet const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TSIGTriplet

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$RPZ
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$RPZ
struct RPZ final {
  ::rust::String name;
  ::rust::Vec<::rust::String> addresses;
  ::rust::String defcontent;
  ::rust::String defpol;
  bool defpolOverrideLocalData CXX_DEFAULT_VALUE(false);
  ::std::uint32_t defttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t extendedErrorCode CXX_DEFAULT_VALUE(0);
  ::rust::String extendedErrorExtra;
  bool includeSOA CXX_DEFAULT_VALUE(false);
  bool ignoreDuplicates CXX_DEFAULT_VALUE(false);
  ::std::uint32_t maxTTL CXX_DEFAULT_VALUE(0);
  ::rust::String policyName;
  ::rust::Vec<::rust::String> tags;
  bool overridesGettag CXX_DEFAULT_VALUE(false);
  ::std::uint32_t zoneSizeHint CXX_DEFAULT_VALUE(0);
  ::pdns::rust::settings::rec::TSIGTriplet tsig;
  ::std::uint32_t refresh CXX_DEFAULT_VALUE(0);
  ::std::uint32_t maxReceivedMBytes CXX_DEFAULT_VALUE(0);
  ::rust::String localAddress;
  ::std::uint32_t axfrTimeout CXX_DEFAULT_VALUE(0);
  ::rust::String dumpFile;
  ::rust::String seedFile;

  bool operator==(RPZ const &) const noexcept;
  bool operator!=(RPZ const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$RPZ

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ZoneToCache
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ZoneToCache
struct ZoneToCache final {
  ::rust::String zone;
  ::rust::String method;
  ::rust::Vec<::rust::String> sources;
  ::std::uint64_t timeout CXX_DEFAULT_VALUE(0);
  ::pdns::rust::settings::rec::TSIGTriplet tsig;
  ::std::uint64_t refreshPeriod CXX_DEFAULT_VALUE(0);
  ::std::uint64_t retryOnErrorPeriod CXX_DEFAULT_VALUE(0);
  ::std::uint64_t maxReceivedMBytes CXX_DEFAULT_VALUE(0);
  ::rust::String localAddress;
  ::rust::String zonemd;
  ::rust::String dnssec;

  bool operator==(ZoneToCache const &) const noexcept;
  bool operator!=(ZoneToCache const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ZoneToCache

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SubnetOrder
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SubnetOrder
struct SubnetOrder final {
  ::rust::String subnet;
  ::std::uint32_t order CXX_DEFAULT_VALUE(0);

  bool operator==(SubnetOrder const &) const noexcept;
  bool operator!=(SubnetOrder const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SubnetOrder

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SortList
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SortList
struct SortList final {
  ::rust::String key;
  ::rust::Vec<::pdns::rust::settings::rec::SubnetOrder> subnets;

  bool operator==(SortList const &) const noexcept;
  bool operator!=(SortList const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SortList

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AllowedAdditionalQType
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AllowedAdditionalQType
struct AllowedAdditionalQType final {
  ::rust::String qtype;
  ::rust::Vec<::rust::String> targets;
  ::rust::String mode;

  bool operator==(AllowedAdditionalQType const &) const noexcept;
  bool operator!=(AllowedAdditionalQType const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AllowedAdditionalQType

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProxyMapping
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProxyMapping
struct ProxyMapping final {
  ::rust::String subnet;
  ::rust::String address;
  ::rust::Vec<::rust::String> domains;

  bool operator==(ProxyMapping const &) const noexcept;
  bool operator!=(ProxyMapping const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProxyMapping

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ApiZones
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ApiZones
struct ApiZones final {
  ::rust::Vec<::pdns::rust::settings::rec::AuthZone> auth_zones;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> forward_zones;

  void validate(::rust::Str field) const;
  bool operator==(ApiZones const &) const noexcept;
  bool operator!=(ApiZones const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ApiZones

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$XFR
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$XFR
struct XFR final {
  ::rust::Vec<::rust::String> addresses;
  ::std::uint32_t zoneSizeHint CXX_DEFAULT_VALUE(0);
  ::pdns::rust::settings::rec::TSIGTriplet tsig;
  ::std::uint32_t refresh CXX_DEFAULT_VALUE(0);
  ::std::uint32_t maxReceivedMBytes CXX_DEFAULT_VALUE(0);
  ::rust::String localAddress;
  ::std::uint32_t axfrTimeout CXX_DEFAULT_VALUE(0);

  bool operator==(XFR const &) const noexcept;
  bool operator!=(XFR const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$XFR

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$FCZDefault
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$FCZDefault
struct FCZDefault final {
  ::rust::String name;
  ::rust::Vec<::rust::String> forwarders;
  bool recurse CXX_DEFAULT_VALUE(false);
  bool notify_allowed CXX_DEFAULT_VALUE(false);

  bool operator==(FCZDefault const &) const noexcept;
  bool operator!=(FCZDefault const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$FCZDefault

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardingCatalogZone
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardingCatalogZone
struct ForwardingCatalogZone final {
  ::rust::String zone;
  bool notify_allowed CXX_DEFAULT_VALUE(false);
  ::pdns::rust::settings::rec::XFR xfr;
  ::rust::Vec<::pdns::rust::settings::rec::FCZDefault> groups;

  bool operator==(ForwardingCatalogZone const &) const noexcept;
  bool operator!=(ForwardingCatalogZone const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardingCatalogZone

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingTLS
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingTLS
struct IncomingTLS final {
  ::rust::String certificate;
  ::rust::String key;

  bool operator==(IncomingTLS const &) const noexcept;
  bool operator!=(IncomingTLS const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingTLS

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingWSConfig
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingWSConfig
struct IncomingWSConfig final {
  ::rust::Vec<::rust::String> addresses;
  ::pdns::rust::settings::rec::IncomingTLS tls;

  bool operator==(IncomingWSConfig const &) const noexcept;
  bool operator!=(IncomingWSConfig const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingWSConfig

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Value
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Value
struct Value final {
  bool bool_val CXX_DEFAULT_VALUE(false);
  ::std::uint64_t u64_val CXX_DEFAULT_VALUE(0);
  double f64_val CXX_DEFAULT_VALUE(0);
  ::rust::String string_val;
  ::rust::Vec<::rust::String> vec_string_val;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> vec_forwardzone_val;
  ::rust::Vec<::pdns::rust::settings::rec::AuthZone> vec_authzone_val;
  ::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> vec_trustanchor_val;
  ::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> vec_negativetrustanchor_val;
  ::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> vec_protobufserver_val;
  ::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> vec_dnstap_framestream_server_val;
  ::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> vec_dnstap_nod_framestream_server_val;
  ::rust::Vec<::pdns::rust::settings::rec::RPZ> vec_rpz_val;
  ::rust::Vec<::pdns::rust::settings::rec::SortList> vec_sortlist_val;
  ::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> vec_zonetocache_val;
  ::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> vec_allowedadditionalqtype_val;
  ::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> vec_proxymapping_val;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> vec_forwardingcatalogzone_val;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Value

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$OldStyle
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$OldStyle
struct OldStyle final {
  ::rust::String section;
  ::rust::String name;
  ::rust::String old_name;
  ::rust::String type_name;
  ::pdns::rust::settings::rec::Value value;
  bool overriding CXX_DEFAULT_VALUE(false);

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$OldStyle

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Dnssec
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Dnssec
struct Dnssec final {
  ::std::uint64_t aggressive_nsec_cache_size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t aggressive_cache_min_nsec3_hit_ratio CXX_DEFAULT_VALUE(0);
  ::rust::String validation;
  ::rust::Vec<::rust::String> disabled_algorithms;
  bool log_bogus CXX_DEFAULT_VALUE(false);
  ::std::uint64_t nsec3_max_iterations CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_rrsigs_per_record CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_nsec3s_per_record CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_signature_validations_per_query CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_nsec3_hash_computations_per_query CXX_DEFAULT_VALUE(0);
  ::std::uint64_t aggressive_cache_max_nsec3_hash_cost CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_ds_per_zone CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_dnskeys CXX_DEFAULT_VALUE(0);
  ::std::uint64_t signature_inception_skew CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> x_dnssec_names;
  ::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> trustanchors;
  ::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> negative_trustanchors;
  ::rust::String trustanchorfile;
  ::std::uint64_t trustanchorfile_interval CXX_DEFAULT_VALUE(0);

  bool operator==(Dnssec const &) const noexcept;
  bool operator!=(Dnssec const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Dnssec

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Incoming
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Incoming
struct Incoming final {
  ::rust::Vec<::rust::String> allow_from;
  ::rust::String allow_from_file;
  ::rust::Vec<::rust::String> allow_notify_for;
  ::rust::String allow_notify_for_file;
  ::rust::Vec<::rust::String> allow_notify_from;
  ::rust::String allow_notify_from_file;
  bool allow_no_rd CXX_DEFAULT_VALUE(false);
  ::std::uint64_t tcp_timeout CXX_DEFAULT_VALUE(0);
  double distribution_load_factor CXX_DEFAULT_VALUE(0);
  ::std::uint64_t distribution_pipe_buffer_size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t distributor_threads CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> edns_padding_from;
  ::rust::String edns_padding_mode;
  ::std::uint64_t edns_padding_tag CXX_DEFAULT_VALUE(0);
  bool gettag_needs_edns_options CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::rust::String> listen;
  ::std::uint64_t port CXX_DEFAULT_VALUE(0);
  bool non_local_bind CXX_DEFAULT_VALUE(false);
  ::std::uint64_t max_concurrent_requests_per_tcp_connection CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_tcp_clients CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_tcp_per_client CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_tcp_queries_per_connection CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_udp_queries_per_round CXX_DEFAULT_VALUE(0);
  bool pdns_distributes_queries CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::rust::String> proxy_protocol_from;
  ::rust::Vec<::rust::String> proxy_protocol_exceptions;
  ::std::uint64_t proxy_protocol_maximum_size CXX_DEFAULT_VALUE(0);
  bool reuseport CXX_DEFAULT_VALUE(false);
  ::std::uint64_t tcp_fast_open CXX_DEFAULT_VALUE(0);
  ::std::uint64_t udp_truncation_threshold CXX_DEFAULT_VALUE(0);
  bool use_incoming_edns_subnet CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> proxymappings;

  bool operator==(Incoming const &) const noexcept;
  bool operator!=(Incoming const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Incoming

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursor
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursor
struct Recursor final {
  bool any_to_tcp CXX_DEFAULT_VALUE(false);
  bool allow_trust_anchor_query CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::pdns::rust::settings::rec::AuthZone> auth_zones;
  ::rust::String chroot;
  ::rust::String config_dir;
  ::rust::String config_name;
  ::rust::String cpu_map;
  bool daemon CXX_DEFAULT_VALUE(false);
  bool devonly_regression_test_mode CXX_DEFAULT_VALUE(false);
  ::rust::String dns64_prefix;
  ::rust::String etc_hosts_file;
  ::std::uint64_t event_trace_enabled CXX_DEFAULT_VALUE(0);
  bool export_etc_hosts CXX_DEFAULT_VALUE(false);
  ::rust::String export_etc_hosts_search_suffix;
  bool extended_resolution_errors CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> forward_zones;
  ::rust::String forward_zones_file;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> forward_zones_recurse;
  ::rust::String hint_file;
  ::rust::Vec<::rust::String> ignore_unknown_settings;
  ::rust::String include_dir;
  ::std::uint64_t latency_statistic_size CXX_DEFAULT_VALUE(0);
  ::rust::String lua_config_file;
  ::rust::String lua_global_include_dir;
  ::rust::String lua_dns_script;
  ::std::uint64_t lua_maintenance_interval CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_chain_length CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_include_depth CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_generate_steps CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_mthreads CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_cnames_followed CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_recursion_depth CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_total_msec CXX_DEFAULT_VALUE(0);
  ::std::uint64_t minimum_ttl_override CXX_DEFAULT_VALUE(0);
  ::rust::String nothing_below_nxdomain;
  ::rust::String public_suffix_list_file;
  bool qname_minimization CXX_DEFAULT_VALUE(false);
  ::std::uint64_t qname_max_minimize_count CXX_DEFAULT_VALUE(0);
  ::std::uint64_t qname_minimize_one_label CXX_DEFAULT_VALUE(0);
  bool root_nx_trust CXX_DEFAULT_VALUE(false);
  bool save_parent_ns_set CXX_DEFAULT_VALUE(false);
  ::rust::String security_poll_suffix;
  bool serve_rfc1918 CXX_DEFAULT_VALUE(false);
  bool serve_rfc6303 CXX_DEFAULT_VALUE(false);
  ::rust::String server_id;
  ::rust::String setgid;
  ::rust::String setuid;
  ::rust::String socket_dir;
  ::rust::String socket_group;
  ::rust::String socket_mode;
  ::rust::String socket_owner;
  ::std::uint64_t spoof_nearmiss_max CXX_DEFAULT_VALUE(0);
  ::std::uint64_t stack_cache_size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t stack_size CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> stats_api_disabled_list;
  ::rust::Vec<::rust::String> stats_carbon_disabled_list;
  ::rust::Vec<::rust::String> stats_rec_control_disabled_list;
  ::std::uint64_t stats_ringbuffer_entries CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> stats_snmp_disabled_list;
  ::std::uint64_t threads CXX_DEFAULT_VALUE(0);
  ::std::uint64_t tcp_threads CXX_DEFAULT_VALUE(0);
  ::rust::String version_string;
  bool write_pid CXX_DEFAULT_VALUE(false);
  ::std::uint64_t system_resolver_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint64_t system_resolver_interval CXX_DEFAULT_VALUE(0);
  bool system_resolver_self_resolve_check CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::pdns::rust::settings::rec::SortList> sortlists;
  ::rust::Vec<::pdns::rust::settings::rec::RPZ> rpzs;
  ::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> allowed_additional_qtypes;
  ::rust::String lua_start_stop_script;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> forwarding_catalog_zones;

  bool operator==(Recursor const &) const noexcept;
  bool operator!=(Recursor const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursor

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Webservice
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Webservice
struct Webservice final {
  ::rust::String api_dir;
  ::rust::String api_key;
  bool webserver CXX_DEFAULT_VALUE(false);
  ::rust::String address;
  ::rust::Vec<::pdns::rust::settings::rec::IncomingWSConfig> listen;
  ::rust::Vec<::rust::String> allow_from;
  bool hash_plaintext_credentials CXX_DEFAULT_VALUE(false);
  ::rust::String loglevel;
  ::rust::String password;
  ::std::uint64_t port CXX_DEFAULT_VALUE(0);

  bool operator==(Webservice const &) const noexcept;
  bool operator!=(Webservice const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Webservice

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Carbon
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Carbon
struct Carbon final {
  ::std::uint64_t interval CXX_DEFAULT_VALUE(0);
  ::rust::String ns;
  ::rust::String ourname;
  ::rust::String instance;
  ::rust::Vec<::rust::String> server;

  bool operator==(Carbon const &) const noexcept;
  bool operator!=(Carbon const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Carbon

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Outgoing
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Outgoing
struct Outgoing final {
  ::rust::Vec<::rust::String> dont_throttle_names;
  ::rust::Vec<::rust::String> dont_throttle_netmasks;
  ::rust::Vec<::rust::String> dot_to_auth_names;
  bool dot_to_port_853 CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::rust::String> dont_query;
  ::std::uint64_t edns_bufsize CXX_DEFAULT_VALUE(0);
  bool edns_padding CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::rust::String> edns_subnet_allow_list;
  bool edns_subnet_harden CXX_DEFAULT_VALUE(false);
  bool lowercase CXX_DEFAULT_VALUE(false);
  ::std::uint64_t max_busy_dot_probes CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_qperq CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_ns_address_qperq CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_ns_per_resolve CXX_DEFAULT_VALUE(0);
  ::std::uint64_t network_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint64_t non_resolving_ns_max_fails CXX_DEFAULT_VALUE(0);
  ::std::uint64_t non_resolving_ns_throttle_time CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> source_address;
  ::std::uint64_t server_down_max_fails CXX_DEFAULT_VALUE(0);
  ::std::uint64_t server_down_throttle_time CXX_DEFAULT_VALUE(0);
  ::std::uint64_t bypass_server_throttling_probability CXX_DEFAULT_VALUE(0);
  bool single_socket CXX_DEFAULT_VALUE(false);
  bool tcp_fast_open_connect CXX_DEFAULT_VALUE(false);
  ::std::uint64_t tcp_max_idle_ms CXX_DEFAULT_VALUE(0);
  ::std::uint64_t tcp_max_idle_per_auth CXX_DEFAULT_VALUE(0);
  ::std::uint64_t tcp_max_queries CXX_DEFAULT_VALUE(0);
  ::std::uint64_t tcp_max_idle_per_thread CXX_DEFAULT_VALUE(0);
  ::std::uint64_t udp_source_port_min CXX_DEFAULT_VALUE(0);
  ::std::uint64_t udp_source_port_max CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> udp_source_port_avoid;

  bool operator==(Outgoing const &) const noexcept;
  bool operator!=(Outgoing const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Outgoing

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Packetcache
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Packetcache
struct Packetcache final {
  bool disable CXX_DEFAULT_VALUE(false);
  ::std::uint64_t max_entries CXX_DEFAULT_VALUE(0);
  ::std::uint64_t ttl CXX_DEFAULT_VALUE(0);
  ::std::uint64_t negative_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint64_t servfail_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint64_t shards CXX_DEFAULT_VALUE(0);

  bool operator==(Packetcache const &) const noexcept;
  bool operator!=(Packetcache const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Packetcache

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Logging
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Logging
struct Logging final {
  bool disable_syslog CXX_DEFAULT_VALUE(false);
  bool timestamp CXX_DEFAULT_VALUE(false);
  ::std::uint64_t loglevel CXX_DEFAULT_VALUE(0);
  bool common_errors CXX_DEFAULT_VALUE(false);
  bool rpz_changes CXX_DEFAULT_VALUE(false);
  ::rust::String facility;
  bool protobuf_use_kernel_timestamp CXX_DEFAULT_VALUE(false);
  bool quiet CXX_DEFAULT_VALUE(false);
  ::std::uint64_t statistics_interval CXX_DEFAULT_VALUE(0);
  bool structured_logging CXX_DEFAULT_VALUE(false);
  ::rust::String structured_logging_backend;
  ::rust::String trace;
  ::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> protobuf_servers;
  ::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> outgoing_protobuf_servers;
  ::std::uint64_t protobuf_mask_v4 CXX_DEFAULT_VALUE(0);
  ::std::uint64_t protobuf_mask_v6 CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> dnstap_framestream_servers;
  ::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> dnstap_nod_framestream_servers;

  bool operator==(Logging const &) const noexcept;
  bool operator!=(Logging const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Logging

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Ecs
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Ecs
struct Ecs final {
  ::rust::Vec<::rust::String> add_for;
  ::std::uint64_t ipv4_bits CXX_DEFAULT_VALUE(0);
  ::std::uint64_t ipv4_cache_bits CXX_DEFAULT_VALUE(0);
  ::std::uint64_t ipv6_bits CXX_DEFAULT_VALUE(0);
  ::std::uint64_t ipv6_cache_bits CXX_DEFAULT_VALUE(0);
  bool ipv4_never_cache CXX_DEFAULT_VALUE(false);
  bool ipv6_never_cache CXX_DEFAULT_VALUE(false);
  ::std::uint64_t minimum_ttl_override CXX_DEFAULT_VALUE(0);
  ::std::uint64_t cache_limit_ttl CXX_DEFAULT_VALUE(0);
  ::rust::String scope_zero_address;

  bool operator==(Ecs const &) const noexcept;
  bool operator!=(Ecs const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Ecs

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recordcache
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recordcache
struct Recordcache final {
  ::std::uint64_t max_cache_bogus_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_entries CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_ttl CXX_DEFAULT_VALUE(0);
  bool limit_qtype_any CXX_DEFAULT_VALUE(false);
  ::std::uint64_t max_rrset_size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_negative_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint64_t locked_ttl_perc CXX_DEFAULT_VALUE(0);
  ::std::uint64_t shards CXX_DEFAULT_VALUE(0);
  ::std::uint64_t refresh_on_ttl_perc CXX_DEFAULT_VALUE(0);
  ::std::uint64_t serve_stale_extensions CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> zonetocaches;

  bool operator==(Recordcache const &) const noexcept;
  bool operator!=(Recordcache const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recordcache

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Nod
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Nod
struct Nod final {
  bool tracking CXX_DEFAULT_VALUE(false);
  bool log CXX_DEFAULT_VALUE(false);
  ::rust::String lookup;
  ::std::uint64_t db_size CXX_DEFAULT_VALUE(0);
  ::rust::String history_dir;
  ::std::uint64_t db_snapshot_interval CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> ignore_list;
  ::rust::String ignore_list_file;
  ::rust::String pb_tag;
  bool unique_response_tracking CXX_DEFAULT_VALUE(false);
  bool unique_response_log CXX_DEFAULT_VALUE(false);
  ::std::uint64_t unique_response_db_size CXX_DEFAULT_VALUE(0);
  ::rust::String unique_response_history_dir;
  ::rust::String unique_response_pb_tag;
  ::rust::Vec<::rust::String> unique_response_ignore_list;
  ::rust::String unique_response_ignore_list_file;

  bool operator==(Nod const &) const noexcept;
  bool operator!=(Nod const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Nod

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Snmp
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Snmp
struct Snmp final {
  bool agent CXX_DEFAULT_VALUE(false);
  ::rust::String daemon_socket;

  bool operator==(Snmp const &) const noexcept;
  bool operator!=(Snmp const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Snmp

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursorsettings
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursorsettings
struct Recursorsettings final {
  ::pdns::rust::settings::rec::Dnssec dnssec;
  ::pdns::rust::settings::rec::Incoming incoming;
  ::pdns::rust::settings::rec::Recursor recursor;
  ::pdns::rust::settings::rec::Webservice webservice;
  ::pdns::rust::settings::rec::Carbon carbon;
  ::pdns::rust::settings::rec::Outgoing outgoing;
  ::pdns::rust::settings::rec::Packetcache packetcache;
  ::pdns::rust::settings::rec::Logging logging;
  ::pdns::rust::settings::rec::Ecs ecs;
  ::pdns::rust::settings::rec::Recordcache recordcache;
  ::pdns::rust::settings::rec::Nod nod;
  ::pdns::rust::settings::rec::Snmp snmp;

  ::rust::String to_yaml_string() const;
  void validate() const;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursorsettings

::pdns::rust::settings::rec::Recursorsettings parse_yaml_string(::rust::Str str);

::rust::Vec<::rust::String> parse_yaml_string_to_allow_from(::rust::Str str);

::rust::Vec<::pdns::rust::settings::rec::ForwardZone> parse_yaml_string_to_forward_zones(::rust::Str str);

::rust::Vec<::rust::String> parse_yaml_string_to_allow_notify_for(::rust::Str str);

::pdns::rust::settings::rec::ApiZones parse_yaml_string_to_api_zones(::rust::Str str);

::rust::String map_to_yaml_string(::rust::Vec<::pdns::rust::settings::rec::OldStyle> const &map);

::rust::String forward_zones_to_yaml_string(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> const &vec);

::rust::String allow_from_to_yaml_string(::rust::Vec<::rust::String> const &vec);

::rust::String allow_from_to_yaml_string_incoming(::rust::String const &key, ::rust::String const &filekey, ::rust::Vec<::rust::String> const &vec);

::rust::String allow_for_to_yaml_string(::rust::Vec<::rust::String> const &vec);

void merge(::pdns::rust::settings::rec::Recursorsettings &lhs, ::rust::Str rhs);

void validate_auth_zones(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::AuthZone> const &vec);

void validate_forward_zones(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> const &vec);

void validate_allow_for(::rust::Str field, ::rust::Vec<::rust::String> const &vec);

void validate_allow_notify_for(::rust::Str field, ::rust::Vec<::rust::String> const &vec);

void validate_allow_from(::rust::Str field, ::rust::Vec<::rust::String> const &vec);

::std::unique_ptr<::pdns::rust::settings::rec::ApiZones> api_read_zones(::rust::Str path);

void api_add_auth_zone(::rust::Str file, ::pdns::rust::settings::rec::AuthZone authzone);

void api_add_forward_zone(::rust::Str file, ::pdns::rust::settings::rec::ForwardZone forwardzone);

void api_add_forward_zones(::rust::Str file, ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> &forwardzones);

void validate_trustanchors(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> const &vec);

void validate_negativetrustanchors(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> const &vec);

void api_delete_zone(::rust::Str file, ::rust::Str zone);

void api_delete_zones(::rust::Str file);
} // namespace rec
} // namespace settings
} // namespace rust
} // namespace pdns

#ifdef __clang__
#pragma clang diagnostic pop
#endif // __clang__
