// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A7__F209

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_32_0_F_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1152)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x05871798u, 0xf1846a49u, 0, 29 }, // 05871798f1846a49 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1152
    { 0x304b9cf6u, 0xca2561bbu, 0, 6 }, // 304b9cf6ca2561bb = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1152
    { 0xf36078afu, 0xb63cca23u, 0, 46 }, // f36078afb63cca23 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1152
    { 0xfb5abea2u, 0xd1cc839cu, 0, 63 }, // fb5abea2d1cc839c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1152
    { 0x0c7b4912u, 0xac6892bau, 23, 29 }, // 0c7b4912ac6892ba = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1152
    { 0x59fc1e06u, 0xa812ab91u, 23, 6 }, // 59fc1e06a812ab91 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1152
    { 0x4429093bu, 0x3b0e2231u, 23, 46 }, // 4429093b3b0e2231 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1152
    { 0xe16c40f8u, 0xeea0d14eu, 23, 63 }, // e16c40f8eea0d14e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1152
    { 0x8db035c3u, 0xd269e8dfu, 160, 126 }, // 8db035c3d269e8df = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__64_32__CO__wave2_warp4_stg1--Arch_gfx1152
    { 0x4d81d784u, 0x55296c59u, 160, 143 }, // 4d81d78455296c59 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1152
    { 0x4df76be6u, 0x4241fa9cu, 166, 109 }, // 4df76be64241fa9c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_F_1___gfx11xx__P__64_64__CO__wave1_warp4_stg1--Arch_gfx1152
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1152_mod0
{{ 6, 7, 4, 6, 4, 4, 4, 0, 0, 7},
 { 5, 5, 0, 4, 0, 7, 0, 1, 1, 1},
 { 7, 5, 5, 6, 7, 7, 0, 1, 2, 1},
 { 7, 5, 5, 5, 6, 6, 1, 3, 3, 1},
 { 4, 7, 6, 7, 5, 4, 2, 0, 1, 1},
 { 4, 6, 6, 6, 5, 4, 6,10,10,10},
 { 4, 4, 4, 5, 4, 7,10,10,10, 4},
 { 4, 5, 7, 4, 7, 0,10,10, 4, 4},
 { 7, 5, 4, 6, 5,10,10, 7, 7, 4},
 { 6, 9, 5, 9, 9,10, 5, 5, 8, 6}}
// End of GPU gfx1152_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A7__F209 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 32
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 32 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

