/*
  ==============================================================================

    This file contains the basic framework code for a JUCE plugin processor.

  ==============================================================================
*/

#pragma once

#include <JuceHeader.h>

//==============================================================================
/**
 */
class SG323AudioProcessor : public juce::AudioProcessor
#if JucePlugin_Enable_ARA
    ,
                            public juce::AudioProcessorARAExtension
#endif
{
public:
    //==============================================================================
    SG323AudioProcessor();
    ~SG323AudioProcessor() override;

    //==============================================================================
    void prepareToPlay(double sampleRate, int samplesPerBlock) override;
    void releaseResources() override;

#ifndef JucePlugin_PreferredChannelConfigurations
    bool isBusesLayoutSupported(const BusesLayout &layouts) const override;
#endif

    void processBlock(juce::AudioBuffer<float> &, juce::MidiBuffer &) override;

    //==============================================================================
    juce::AudioProcessorEditor *createEditor() override;
    bool hasEditor() const override;

    //==============================================================================
    const juce::String getName() const override;

    bool acceptsMidi() const override;
    bool producesMidi() const override;
    bool isMidiEffect() const override;
    double getTailLengthSeconds() const override;

    //==============================================================================
    int getNumPrograms() override;
    int getCurrentProgram() override;
    void setCurrentProgram(int index) override;
    const juce::String getProgramName(int index) override;
    void changeProgramName(int index, const juce::String &newName) override;

    //==============================================================================
    void getStateInformation(juce::MemoryBlock &destData) override;
    void setStateInformation(const void *data, int sizeInBytes) override;

    void updateFilter();
    juce::AudioProcessorValueTreeState apvts;

private:
    juce::AudioProcessorValueTreeState::ParameterLayout createParameters();
    juce::AudioBuffer<float> inputBuffer;
    juce::AudioBuffer<float> monoBuffer;
    juce::AudioBuffer<float> randomBuffer;
    juce::AudioBuffer<float> feedbackBuffer;
    juce::AudioBuffer<float> outputBuffer;
    juce::AudioBuffer<float> noiseBuffer;
    juce::AudioBuffer<float> bitBuffer;

    int fifoBufferSize = 8192;
    juce::AbstractFifo abstractFifo{fifoBufferSize};
    int start1, size1, start2, size2;
    juce::AudioBuffer<float> fifoBuffer;

    juce::Random random;

    juce::LinearSmoothedValue<float> inputGainSmooth{};
    juce::LinearSmoothedValue<float> highPassSmooth{};
    juce::LinearSmoothedValue<float> lowPassSmooth{};
    juce::LinearSmoothedValue<float> predelaySmooth{};
    juce::LinearSmoothedValue<float> decaySmooth{};
    juce::LinearSmoothedValue<float> mixSmooth{};

    juce::dsp::Gain<float> gainModule;
    juce::dsp::DelayLine<float, juce::dsp::DelayLineInterpolationTypes::Linear> fractionalDelay;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> inputHighPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> inputLowPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> randomHighPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> randomLowPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> preEmphasis;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> deEmphasis;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> feedBackDip;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> feedBackHighPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> feedBackLowPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> antiAliasFirstSection;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> antiAliasSecondSection;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> antiAliasThirdSection;

    float outputGainArray[8]{-0.996f, 0.996f, 0.622f, 0.378f, 0.378f, 0.622f, 0.966f, -0.966f};
    float outputDelayArray[16]{10.5f, 7.0f, 3.5f, 0.0f, 19.0f, 13.0f, 7.0f, 1.0f, 11.5f, 8.0f, 4.5f, 1.0f, 20.0f, 14.0f, 8.0f, 2.0f};
    unsigned long delayTaps[24];
    unsigned int gainCeiling[24];
    int initSampleRateCount{};
    float lastSampleRate{};
    float nextHighPassValue{};
    float nextLowPassValue{};

    int writeAddress{16383};
    unsigned int nROW{255};
    unsigned int nCOLUMN{255};
    unsigned int modRateCeiling{16};
    float modScale{1.0f};
    unsigned int MCCK{};
    unsigned int modClockOut{};
    unsigned int modCount{};
    unsigned int gainModContBaseAddr{};
    unsigned int gainModBaseAddr{};
    unsigned int delayModBaseAddr{};

    float s1a0{1.0f};
    float s1a1{1.0f};
    float s1a2{1.0f};
    float s1b0{1.0f};
    float s1b1{1.0f};
    float s1b2{1.0f};
    float s2a0{1.0f};
    float s2a1{1.0f};
    float s2a2{1.0f};
    float s2b0{1.0f};
    float s2b1{1.0f};
    float s2b2{1.0f};
    float s3a0{1.0f};
    float s3a1{1.0f};
    float s3a2{1.0f};
    float s3b0{1.0f};
    float s3b1{1.0f};
    float s3b2{1.0f};
    double ellipticCoefficients[108]{0.1590722219519817, 0.2925800084337282, 0.15907222195198167, 1.0, -0.1616482402221111, 0.19263764096311883, 1.0, 1.265871938992507, 0.9999999999999998, 1.0, 0.5376549648832389, 0.7201150300473758, 1.0, 1.0336348451847246, 1.0, 1.0, 0.8104458709784527, 0.9429830293245048, 0.12217156278343735, 0.21708074800925328, 0.12217156278343735, 1.0, -0.3622982289466266, 0.21069044780090654, 1.0, 1.036555560330608, 1.0000000000000002, 1.0, 0.2564381854840234, 0.7103381856084345, 1.0, 0.759333766447034, 0.9999999999999998, 1.0, 0.5147161937185064, 0.939618968627594, 0.03001113166367873, 0.02967221178596468, 0.030011131663678737, 1.0, -1.196963794847673, 0.44585426837637376, 1.0, -0.5800931166584409, 0.9999999999999999, 1.0, -1.0890277974018174, 0.7625980831299516, 1.0, -0.8811254991797578, 1.0000000000000002, 1.0, -1.0466322049865133, 0.9469912658394941, 0.026152742312366212, 0.021748966292658066, 0.02615274231236622, 1.0, -1.2666830304156562, 0.4787076012745491, 1.0, -0.7561492861543417, 1.0, 1.0, -1.1994214057403096, 0.7757059495161946, 1.0, -1.0338334903782682, 1.0, 1.0, -1.1783179130907315, 0.9498397819567047, 0.01345196428348894, -0.00578353415575715, 0.01345196428348894, 1.0, -1.609978347980477, 0.677670924022425, 1.0, -1.570407382335601, 0.9999999999999998, 1.0, -1.6792585172217143, 0.8621444371857153, 1.0, -1.6869584422150632, 0.9999999999999999, 1.0, -1.7263819265125393, 0.9694005278843073, 0.012734638654465117, -0.007569262219551063, 0.012734638654465117, 1.0, -1.6422421825069187, 0.6999451424751979, 1.0, -1.6334218277265786, 0.9999999999999997, 1.0, -1.7166739247723046, 0.8720771412216309, 1.0, -1.7341496621089312, 1.0000000000000002, 1.0, -1.7656027882101546, 0.9716653621519904};
    int modRateCountData[128]{12, 12, 12, 12, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 12, 11, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 11, 10, 9, 9, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 11, 9, 7, 7, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 10, 9, 7, 7, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 10, 8, 6, 6, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 9, 7, 5, 5, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 9, 6, 5, 5, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4};
    int delayData[512]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 4, 0, 5, 0, 7, 0, 8, 0, 9, 0, 11, 0, 13, 0, 14, 0, 15, 0, 16, 0, 18, 0, 20, 0, 21, 0, 22, 240, 3, 128, 3, 16, 3, 160, 2, 0, 5, 64, 4, 128, 3, 192, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 5, 0, 6, 0, 8, 0, 11, 0, 12, 0, 14, 0, 17, 0, 18, 0, 20, 0, 23, 0, 24, 0, 25, 0, 29, 0, 30, 64, 5, 96, 4, 128, 3, 160, 2, 96, 7, 224, 5, 96, 4, 224, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 7, 0, 9, 0, 11, 0, 14, 0, 16, 0, 20, 0, 22, 0, 23, 0, 25, 0, 28, 0, 31, 0, 33, 0, 35, 0, 37, 144, 6, 64, 5, 240, 3, 160, 2, 192, 9, 128, 7, 64, 5, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 10, 0, 11, 0, 16, 0, 17, 0, 21, 0, 22, 0, 25, 0, 28, 0, 31, 0, 34, 0, 36, 0, 39, 0, 42, 0, 44, 224, 7, 32, 6, 96, 4, 160, 2, 32, 12, 32, 9, 32, 6, 32, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 11, 0, 14, 0, 15, 0, 17, 0, 20, 0, 23, 0, 27, 0, 29, 0, 32, 0, 36, 0, 40, 0, 42, 0, 46, 0, 50, 0, 55, 48, 9, 0, 7, 208, 4, 160, 2, 128, 14, 192, 10, 0, 7, 64, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 15, 0, 16, 0, 17, 0, 20, 0, 21, 0, 25, 0, 28, 0, 32, 0, 36, 0, 40, 0, 44, 0, 47, 0, 52, 0, 59, 0, 63, 128, 10, 224, 7, 64, 5, 160, 2, 224, 16, 96, 12, 224, 7, 96, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 22, 0, 25, 0, 28, 0, 32, 0, 34, 0, 37, 0, 42, 0, 45, 0, 47, 0, 49, 0, 53, 0, 55, 0, 59, 0, 63, 208, 11, 192, 8, 176, 5, 160, 2, 64, 19, 0, 14, 192, 8, 128, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 25, 0, 29, 0, 31, 0, 33, 0, 37, 0, 41, 0, 43, 0, 48, 0, 50, 0, 52, 0, 55, 0, 56, 0, 58, 0, 61, 0, 63, 32, 13, 160, 9, 32, 6, 160, 2, 160, 21, 160, 15, 160, 9, 160, 3, 0, 0, 0};
    int delayModData[4096]{255, 255, 255, 255, 255, 255, 46, 237, 169, 92, 53, 239, 4, 172, 114, 166, 60, 173, 30, 237, 174, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 46, 237, 142, 92, 53, 239, 4, 172, 14, 166, 60, 173, 30, 237, 174, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 237, 142, 92, 53, 239, 4, 172, 14, 166, 60, 173, 30, 237, 174, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 237, 142, 92, 53, 239, 164, 172, 14, 166, 60, 36, 30, 237, 174, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 237, 142, 92, 53, 239, 164, 172, 14, 166, 60, 36, 30, 129, 43, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 237, 142, 92, 53, 53, 164, 172, 14, 232, 60, 36, 30, 129, 43, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 232, 142, 92, 53, 53, 164, 172, 14, 232, 60, 36, 30, 129, 43, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 232, 142, 92, 53, 53, 164, 100, 14, 232, 60, 36, 30, 129, 43, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 232, 142, 92, 53, 53, 164, 100, 14, 232, 69, 36, 59, 129, 43, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 232, 231, 21, 53, 53, 164, 100, 14, 232, 69, 36, 59, 129, 43, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 232, 231, 21, 135, 53, 164, 100, 14, 232, 69, 36, 59, 129, 43, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 232, 231, 21, 135, 53, 164, 100, 169, 232, 69, 36, 59, 129, 43, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 138, 232, 231, 21, 135, 53, 233, 100, 169, 232, 69, 36, 59, 129, 145, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 232, 231, 21, 135, 85, 233, 100, 169, 232, 69, 36, 59, 129, 145, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 255, 231, 21, 135, 85, 233, 100, 169, 232, 69, 59, 59, 129, 145, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 255, 231, 21, 135, 85, 233, 100, 169, 232, 69, 59, 59, 8, 145, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 255, 231, 21, 135, 85, 233, 100, 169, 185, 69, 59, 75, 8, 145, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 255, 231, 65, 135, 85, 233, 211, 169, 185, 69, 59, 75, 8, 145, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 255, 231, 65, 135, 85, 233, 211, 169, 185, 69, 59, 75, 8, 145, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 255, 231, 65, 135, 85, 233, 211, 169, 185, 239, 59, 75, 8, 145, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 255, 231, 65, 10, 85, 233, 211, 169, 185, 239, 59, 75, 8, 145, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 255, 43, 65, 10, 85, 74, 211, 169, 185, 239, 59, 75, 8, 145, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 255, 43, 65, 10, 85, 74, 211, 27, 185, 239, 59, 75, 8, 145, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 255, 43, 65, 10, 85, 74, 211, 27, 185, 239, 59, 75, 8, 161, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 255, 43, 65, 10, 229, 74, 211, 27, 185, 239, 15, 75, 8, 161, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 34, 43, 65, 10, 229, 74, 211, 27, 185, 239, 15, 75, 6, 161, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 34, 43, 65, 10, 229, 74, 211, 27, 214, 239, 15, 219, 6, 161, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 34, 43, 10, 10, 229, 74, 228, 27, 214, 239, 15, 219, 6, 161, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 34, 43, 10, 10, 229, 74, 228, 27, 214, 239, 15, 219, 6, 161, 68, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 34, 43, 10, 100, 229, 74, 228, 27, 214, 239, 15, 219, 6, 161, 68, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 34, 43, 10, 100, 229, 74, 228, 16, 214, 239, 15, 219, 6, 161, 68, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 34, 78, 10, 100, 229, 74, 228, 16, 214, 239, 15, 219, 6, 161, 68, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 34, 78, 10, 100, 229, 101, 228, 16, 214, 239, 97, 219, 6, 161, 68, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 34, 78, 10, 100, 229, 101, 228, 16, 214, 239, 97, 219, 6, 199, 68, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 34, 78, 10, 100, 54, 101, 228, 16, 214, 239, 97, 219, 75, 199, 68, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 112, 78, 10, 100, 54, 101, 104, 16, 214, 239, 97, 219, 75, 199, 68, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 112, 78, 10, 100, 54, 101, 104, 16, 205, 239, 97, 105, 75, 199, 68, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 112, 78, 139, 100, 54, 101, 104, 16, 205, 239, 97, 105, 75, 199, 68, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 112, 78, 139, 80, 54, 101, 104, 16, 205, 239, 97, 105, 75, 199, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 112, 78, 139, 80, 54, 101, 104, 16, 205, 108, 97, 105, 75, 199, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 112, 78, 139, 80, 54, 101, 104, 62, 205, 108, 97, 105, 75, 199, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 112, 196, 139, 80, 54, 101, 104, 62, 205, 108, 97, 105, 75, 199, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 112, 196, 139, 80, 54, 38, 104, 62, 205, 108, 97, 105, 75, 181, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 249, 112, 196, 139, 80, 54, 38, 104, 62, 205, 108, 97, 105, 75, 181, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 249, 112, 196, 139, 80, 241, 38, 104, 62, 205, 108, 60, 105, 75, 181, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 249, 177, 196, 139, 80, 241, 38, 104, 62, 205, 108, 60, 105, 41, 181, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 249, 177, 196, 139, 80, 241, 38, 104, 62, 161, 108, 60, 133, 41, 181, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 249, 177, 196, 209, 80, 241, 38, 79, 62, 161, 108, 60, 133, 41, 181, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 249, 177, 196, 209, 80, 241, 38, 79, 62, 161, 108, 60, 133, 41, 181, 40, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 249, 177, 196, 209, 222, 241, 38, 79, 62, 161, 97, 60, 133, 41, 181, 40, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 249, 177, 196, 209, 222, 241, 120, 79, 78, 161, 97, 60, 133, 41, 181, 40, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 249, 177, 24, 209, 222, 241, 120, 79, 78, 161, 97, 60, 133, 41, 181, 40, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 216, 177, 24, 209, 222, 241, 120, 79, 78, 161, 97, 60, 133, 41, 112, 40, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 216, 177, 24, 209, 222, 111, 120, 79, 78, 161, 97, 60, 133, 85, 112, 40, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 216, 181, 24, 209, 222, 111, 120, 79, 78, 161, 97, 14, 133, 85, 112, 40, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 216, 181, 24, 209, 222, 111, 120, 129, 78, 161, 97, 14, 133, 85, 112, 40, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 216, 181, 24, 209, 222, 111, 120, 129, 78, 53, 97, 14, 167, 85, 112, 40, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 216, 181, 24, 174, 222, 111, 120, 129, 78, 53, 97, 14, 167, 85, 112, 30, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 216, 181, 24, 174, 222, 111, 120, 129, 78, 53, 129, 14, 167, 85, 112, 30, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 216, 181, 24, 174, 128, 111, 120, 129, 78, 53, 129, 14, 167, 85, 112, 30, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 216, 181, 24, 174, 128, 111, 120, 129, 154, 53, 129, 14, 167, 85, 112, 30, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 216, 181, 128, 174, 128, 111, 120, 129, 154, 53, 129, 14, 167, 85, 112, 30, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 90, 181, 128, 174, 128, 111, 101, 129, 154, 53, 129, 14, 167, 85, 112, 30, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 90, 181, 128, 174, 128, 111, 101, 129, 154, 53, 129, 66, 167, 85, 25, 30, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 90, 181, 128, 174, 128, 4, 101, 129, 154, 53, 129, 66, 167, 49, 25, 30, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 90, 122, 128, 174, 128, 4, 101, 129, 154, 8, 129, 66, 167, 49, 25, 30, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 90, 122, 128, 174, 128, 4, 101, 197, 154, 8, 129, 66, 110, 49, 25, 30, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 90, 122, 128, 155, 128, 4, 101, 197, 154, 8, 129, 66, 110, 49, 25, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 90, 122, 128, 155, 128, 4, 101, 197, 154, 8, 213, 66, 110, 49, 25, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 90, 122, 128, 155, 15, 4, 101, 197, 154, 8, 213, 66, 110, 49, 25, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 90, 122, 128, 155, 15, 4, 101, 197, 216, 8, 213, 66, 110, 49, 25, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 90, 122, 45, 155, 15, 4, 131, 197, 216, 8, 213, 66, 110, 49, 25, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 246, 122, 45, 155, 15, 4, 131, 197, 216, 8, 213, 66, 110, 49, 25, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 246, 122, 45, 155, 15, 4, 131, 197, 216, 8, 213, 39, 110, 49, 47, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 246, 160, 45, 155, 15, 188, 131, 197, 216, 8, 213, 39, 110, 49, 47, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 246, 160, 45, 155, 15, 188, 131, 197, 216, 8, 213, 39, 250, 59, 47, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 246, 160, 45, 178, 15, 188, 131, 197, 216, 64, 213, 39, 250, 59, 47, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 246, 160, 45, 178, 15, 188, 131, 93, 216, 64, 213, 39, 250, 59, 47, 8, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 246, 160, 45, 178, 15, 188, 131, 93, 216, 64, 213, 39, 250, 59, 47, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 246, 160, 45, 178, 193, 188, 131, 93, 216, 64, 193, 39, 250, 59, 47, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 246, 160, 73, 178, 193, 188, 131, 93, 238, 64, 193, 39, 250, 59, 47, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 160, 73, 178, 193, 188, 131, 93, 238, 64, 193, 39, 250, 59, 47, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 160, 73, 178, 193, 188, 7, 93, 238, 64, 193, 39, 250, 59, 243, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 146, 73, 178, 193, 188, 7, 93, 238, 64, 193, 134, 250, 59, 243, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 146, 73, 178, 193, 88, 7, 93, 238, 64, 193, 134, 250, 59, 243, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 146, 73, 178, 193, 88, 7, 93, 238, 64, 193, 134, 250, 98, 243, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 146, 73, 178, 193, 88, 7, 93, 238, 99, 193, 134, 165, 98, 243, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 146, 73, 50, 193, 88, 7, 61, 238, 99, 193, 134, 165, 98, 243, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 146, 73, 50, 193, 88, 7, 61, 238, 99, 193, 134, 165, 98, 243, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 146, 73, 50, 193, 88, 7, 61, 238, 99, 180, 134, 165, 98, 243, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 146, 73, 50, 28, 88, 7, 61, 238, 99, 180, 134, 165, 98, 243, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 146, 82, 50, 28, 88, 7, 61, 60, 99, 180, 134, 165, 98, 243, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 69, 146, 82, 50, 28, 88, 64, 61, 60, 99, 180, 75, 165, 98, 243, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 146, 82, 50, 28, 88, 64, 61, 60, 99, 180, 75, 165, 98, 141, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 146, 82, 50, 28, 210, 64, 61, 60, 99, 180, 75, 165, 98, 141, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 192, 82, 50, 28, 210, 64, 61, 60, 99, 180, 75, 165, 25, 141, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 192, 82, 50, 28, 210, 64, 61, 60, 226, 180, 75, 142, 25, 141, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 192, 82, 155, 28, 210, 64, 66, 60, 226, 180, 75, 142, 25, 141, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 192, 82, 155, 28, 210, 64, 66, 60, 226, 180, 75, 142, 25, 141, 80, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 192, 13, 155, 28, 210, 64, 66, 60, 226, 13, 75, 142, 25, 141, 80, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 192, 13, 155, 168, 210, 64, 66, 60, 226, 13, 75, 142, 25, 141, 80, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 192, 13, 155, 168, 210, 64, 66, 164, 226, 13, 75, 142, 25, 141, 80, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 192, 13, 155, 168, 210, 176, 66, 164, 226, 13, 75, 142, 25, 141, 80, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 192, 13, 155, 168, 210, 176, 66, 164, 226, 13, 23, 142, 25, 147, 80, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 192, 13, 155, 168, 106, 176, 66, 164, 226, 13, 23, 142, 112, 147, 80, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 211, 13, 155, 168, 106, 176, 66, 164, 241, 13, 23, 142, 112, 147, 80, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 211, 13, 155, 168, 106, 176, 66, 164, 241, 13, 23, 44, 112, 147, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 211, 13, 248, 168, 106, 176, 17, 164, 241, 13, 23, 44, 112, 147, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 211, 13, 248, 168, 106, 176, 17, 164, 241, 130, 23, 44, 112, 147, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 211, 13, 248, 47, 106, 176, 17, 164, 241, 130, 23, 44, 112, 147, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 211, 96, 248, 47, 106, 176, 17, 164, 241, 130, 23, 44, 112, 147, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 211, 96, 248, 47, 106, 176, 17, 48, 241, 130, 23, 44, 112, 147, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 83, 211, 96, 248, 47, 106, 161, 17, 48, 241, 130, 23, 44, 112, 147, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 83, 211, 96, 248, 47, 124, 161, 17, 48, 241, 130, 23, 44, 112, 57, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 83, 15, 96, 248, 47, 124, 161, 17, 48, 241, 130, 21, 44, 112, 57, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 83, 15, 96, 248, 47, 124, 161, 17, 48, 241, 130, 21, 186, 42, 57, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 83, 15, 96, 62, 47, 124, 161, 17, 48, 131, 130, 21, 186, 42, 57, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 83, 15, 96, 62, 47, 124, 161, 15, 48, 131, 130, 21, 186, 42, 57, 109, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 83, 15, 96, 62, 47, 124, 161, 15, 48, 131, 130, 21, 186, 42, 57, 47, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 83, 15, 96, 62, 47, 124, 161, 15, 48, 131, 95, 21, 186, 42, 57, 47, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 83, 15, 96, 62, 146, 124, 4, 15, 48, 131, 95, 21, 186, 42, 57, 47, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 83, 15, 169, 62, 146, 124, 4, 15, 114, 131, 95, 21, 186, 42, 57, 47, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 46, 15, 169, 62, 146, 239, 4, 15, 114, 131, 95, 21, 186, 42, 57, 47, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 46, 15, 169, 62, 146, 239, 4, 15, 114, 131, 95, 173, 186, 42, 174, 47, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 46, 237, 169, 62, 146, 239, 4, 15, 114, 131, 95, 173, 186, 237, 174, 47, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 46, 237, 169, 62, 146, 239, 4, 15, 114, 166, 95, 173, 30, 237, 174, 47, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 46, 237, 169, 62, 146, 239, 4, 15, 114, 166, 95, 173, 30, 237, 174, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 46, 237, 169, 92, 146, 239, 4, 172, 114, 166, 95, 173, 30, 237, 174, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255};
    int gainData[2048]{255, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0, 128, 0, 128, 0, 0, 128, 0, 0, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 223, 50, 46, 43, 39, 36, 32, 29, 153, 22, 146, 15, 11, 136, 4, 1, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 239, 101, 93, 86, 79, 72, 65, 58, 179, 44, 165, 29, 22, 143, 8, 1, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 247, 127, 120, 112, 105, 98, 91, 83, 204, 69, 190, 54, 47, 168, 33, 26, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 251, 127, 121, 116, 110, 104, 99, 93, 216, 82, 204, 71, 65, 188, 54, 48, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 253, 127, 122, 117, 113, 108, 103, 99, 222, 89, 213, 80, 75, 198, 66, 61, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 254, 127, 123, 119, 115, 112, 108, 104, 228, 97, 221, 89, 85, 209, 78, 74, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 255, 127, 124, 120, 117, 114, 111, 107, 232, 101, 226, 94, 91, 216, 85, 81, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 191, 128, 128, 0, 128, 0, 0, 0, 0, 0, 128, 128, 128, 0, 128, 128, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 223, 178, 174, 43, 167, 36, 32, 29, 25, 22, 146, 143, 139, 8, 132, 129, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 239, 254, 245, 108, 227, 90, 81, 72, 64, 55, 174, 165, 156, 19, 138, 129, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 247, 255, 249, 114, 236, 102, 95, 89, 83, 76, 198, 192, 185, 51, 173, 166, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 251, 255, 250, 118, 241, 109, 104, 100, 95, 91, 214, 210, 205, 73, 196, 192, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 253, 255, 251, 120, 244, 112, 109, 105, 102, 98, 222, 219, 215, 84, 208, 204, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 254, 255, 252, 121, 247, 116, 113, 111, 108, 105, 231, 228, 225, 94, 220, 217, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 253, 123, 249, 119, 117, 115, 113, 111, 237, 235, 233, 103, 229, 227, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 191, 128, 128, 128, 0, 128, 128, 128, 0, 128, 0, 0, 0, 0, 128, 0, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 223, 183, 179, 175, 43, 167, 164, 160, 28, 152, 20, 16, 13, 9, 133, 1, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 239, 255, 247, 238, 102, 222, 213, 205, 69, 188, 52, 44, 35, 27, 147, 10, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 247, 255, 249, 242, 108, 230, 223, 217, 83, 204, 70, 64, 57, 51, 173, 38, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 251, 255, 250, 246, 113, 237, 232, 228, 95, 219, 86, 82, 77, 73, 196, 64, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 253, 255, 252, 248, 117, 242, 239, 235, 104, 229, 98, 94, 91, 88, 213, 81, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 254, 255, 253, 251, 120, 246, 244, 242, 112, 238, 107, 105, 103, 101, 227, 97, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 254, 253, 124, 251, 250, 250, 121, 248, 119, 118, 117, 116, 243, 114, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0, 128, 0, 0, 0, 128, 128, 128, 128, 128, 0, 128, 128, 0, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 223, 55, 51, 175, 43, 39, 36, 160, 156, 152, 148, 144, 13, 137, 133, 1, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 239, 126, 117, 236, 99, 90, 81, 200, 192, 183, 174, 165, 28, 147, 138, 1, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 247, 127, 121, 243, 109, 103, 97, 219, 213, 207, 201, 195, 61, 183, 177, 43, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 251, 127, 122, 245, 112, 107, 102, 226, 221, 216, 211, 206, 73, 196, 191, 59, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 253, 127, 123, 247, 115, 112, 108, 232, 228, 225, 221, 217, 85, 209, 206, 74, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 254, 127, 124, 249, 118, 115, 112, 236, 233, 230, 227, 224, 93, 218, 215, 84, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 255, 127, 125, 251, 121, 120, 118, 244, 242, 241, 239, 237, 107, 233, 232, 102, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0, 128, 128, 128, 0, 0, 0, 0, 0, 128, 128, 128, 128, 128, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 223, 68, 63, 186, 181, 177, 44, 39, 34, 30, 25, 148, 143, 139, 134, 129, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 239, 127, 119, 239, 231, 223, 87, 79, 71, 63, 55, 175, 167, 159, 151, 143, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 247, 127, 121, 242, 236, 230, 95, 89, 83, 76, 70, 192, 185, 179, 173, 166, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 251, 127, 122, 245, 240, 235, 102, 97, 91, 86, 81, 204, 199, 194, 189, 184, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 253, 127, 123, 247, 243, 240, 108, 104, 100, 97, 93, 217, 213, 209, 206, 202, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 254, 127, 124, 250, 247, 245, 114, 112, 109, 107, 104, 230, 227, 225, 222, 220, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 255, 127, 125, 251, 249, 248, 118, 116, 114, 113, 111, 237, 235, 233, 232, 230, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 191, 128, 0, 0, 0, 128, 128, 0, 128, 0, 0, 128, 128, 0, 128, 0, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 223, 211, 77, 71, 65, 187, 182, 48, 170, 36, 30, 152, 147, 13, 135, 1, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 239, 255, 119, 111, 103, 223, 215, 79, 199, 63, 55, 175, 167, 31, 151, 15, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 247, 255, 121, 115, 110, 232, 226, 92, 214, 81, 75, 197, 191, 57, 180, 46, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 251, 255, 123, 119, 114, 238, 234, 102, 226, 94, 89, 213, 209, 77, 201, 69, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 253, 255, 124, 121, 118, 243, 240, 110, 235, 104, 101, 226, 223, 92, 217, 86, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 254, 255, 125, 123, 121, 248, 246, 116, 242, 113, 111, 237, 235, 105, 232, 102, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 126, 125, 124, 251, 250, 120, 247, 118, 117, 244, 243, 114, 241, 112, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 191, 128, 128, 128, 0, 128, 128, 128, 128, 128, 128, 128, 0, 128, 0, 128, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 223, 208, 203, 197, 63, 186, 180, 174, 169, 163, 157, 152, 18, 140, 7, 129, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 239, 255, 247, 239, 102, 222, 214, 206, 198, 190, 182, 173, 37, 157, 21, 141, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 247, 255, 249, 242, 108, 230, 223, 217, 211, 204, 198, 192, 57, 179, 45, 166, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 251, 255, 250, 245, 113, 236, 231, 227, 222, 217, 213, 208, 75, 198, 66, 189, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 253, 255, 252, 248, 117, 242, 239, 235, 232, 229, 226, 222, 91, 216, 85, 209, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 254, 255, 253, 250, 120, 246, 243, 241, 238, 236, 234, 231, 101, 227, 96, 222, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 254, 252, 123, 250, 249, 247, 246, 245, 244, 242, 113, 240, 111, 237, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 191, 128, 128, 128, 0, 128, 0, 0, 0, 0, 128, 0, 0, 0, 128, 128, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 223, 203, 198, 193, 59, 182, 49, 43, 38, 33, 155, 22, 17, 12, 134, 129, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 239, 255, 247, 239, 103, 223, 87, 79, 71, 63, 183, 47, 39, 31, 151, 143, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 247, 255, 249, 244, 111, 233, 100, 94, 89, 83, 206, 73, 67, 62, 184, 179, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 251, 255, 251, 248, 117, 241, 110, 106, 103, 99, 224, 93, 89, 86, 210, 207, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 253, 255, 253, 250, 120, 246, 115, 113, 110, 108, 234, 103, 101, 99, 224, 222, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 254, 255, 254, 253, 124, 251, 122, 120, 119, 118, 245, 116, 115, 114, 241, 240, 127, 255, 207, 176, 176, 207, 255, 127, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 127, 255, 127, 127, 127, 127, 255, 127, 127, 127, 255, 255, 127, 255, 207, 176, 176, 207, 255, 127, 255};
    int gainModData[4096]{255, 192, 128, 64, 0, 63, 127, 191, 254, 193, 129, 65, 1, 62, 126, 190, 253, 194, 130, 66, 2, 61, 125, 189, 252, 195, 131, 67, 3, 60, 124, 188, 251, 196, 132, 68, 4, 59, 123, 187, 250, 197, 133, 69, 5, 58, 122, 186, 249, 198, 134, 70, 6, 57, 121, 185, 248, 199, 135, 71, 7, 56, 120, 184, 247, 200, 136, 72, 8, 55, 119, 183, 246, 201, 137, 73, 9, 54, 118, 182, 245, 202, 138, 74, 10, 53, 117, 181, 244, 203, 139, 75, 11, 52, 116, 180, 243, 204, 140, 76, 12, 51, 115, 179, 242, 205, 141, 77, 13, 50, 114, 178, 241, 206, 142, 78, 14, 49, 113, 177, 240, 207, 143, 79, 15, 48, 112, 176, 239, 208, 144, 80, 16, 47, 111, 175, 238, 209, 145, 81, 17, 46, 110, 174, 237, 210, 146, 82, 18, 45, 109, 173, 236, 211, 147, 83, 19, 44, 108, 172, 235, 212, 148, 84, 20, 43, 107, 171, 234, 213, 149, 85, 21, 42, 106, 170, 233, 214, 150, 86, 22, 41, 105, 169, 232, 215, 151, 87, 23, 40, 104, 168, 231, 216, 152, 88, 24, 39, 103, 167, 230, 217, 153, 89, 25, 38, 102, 166, 229, 218, 154, 90, 26, 37, 101, 165, 228, 219, 155, 91, 27, 36, 100, 164, 227, 220, 156, 92, 28, 35, 99, 163, 226, 221, 157, 93, 29, 34, 98, 162, 225, 222, 158, 94, 30, 33, 97, 161, 224, 223, 159, 95, 31, 32, 96, 160, 223, 224, 160, 96, 32, 31, 95, 159, 222, 225, 161, 97, 33, 30, 94, 158, 221, 226, 162, 98, 34, 29, 93, 157, 220, 227, 163, 99, 35, 28, 92, 156, 219, 228, 164, 100, 36, 27, 91, 155, 218, 229, 165, 101, 37, 26, 90, 154, 217, 230, 166, 102, 38, 25, 89, 153, 216, 231, 167, 103, 39, 24, 88, 152, 215, 232, 168, 104, 40, 23, 87, 151, 214, 233, 169, 105, 41, 22, 86, 150, 213, 234, 170, 106, 42, 21, 85, 149, 212, 235, 171, 107, 43, 20, 84, 148, 211, 236, 172, 108, 44, 19, 83, 147, 210, 237, 173, 109, 45, 18, 82, 146, 209, 238, 174, 110, 46, 17, 81, 145, 208, 239, 175, 111, 47, 16, 80, 144, 207, 240, 176, 112, 48, 15, 79, 143, 206, 241, 177, 113, 49, 14, 78, 142, 205, 242, 178, 114, 50, 13, 77, 141, 204, 243, 179, 115, 51, 12, 76, 140, 203, 244, 180, 116, 52, 11, 75, 139, 202, 245, 181, 117, 53, 10, 74, 138, 201, 246, 182, 118, 54, 9, 73, 137, 200, 247, 183, 119, 55, 8, 72, 136, 199, 248, 184, 120, 56, 7, 71, 135, 198, 249, 185, 121, 57, 6, 70, 134, 197, 250, 186, 122, 58, 5, 69, 133, 196, 251, 187, 123, 59, 4, 68, 132, 195, 252, 188, 124, 60, 3, 67, 131, 194, 253, 189, 125, 61, 2, 66, 130, 193, 254, 190, 126, 62, 1, 65, 129, 192, 255, 191, 127, 63, 0, 64, 128, 191, 255, 192, 128, 64, 0, 63, 127, 190, 254, 193, 129, 65, 1, 62, 126, 189, 253, 194, 130, 66, 2, 61, 125, 188, 252, 195, 131, 67, 3, 60, 124, 187, 251, 196, 132, 68, 4, 59, 123, 186, 250, 197, 133, 69, 5, 58, 122, 185, 249, 198, 134, 70, 6, 57, 121, 184, 248, 199, 135, 71, 7, 56, 120, 183, 247, 200, 136, 72, 8, 55, 119, 182, 246, 201, 137, 73, 9, 54, 118, 181, 245, 202, 138, 74, 10, 53, 117, 180, 244, 203, 139, 75, 11, 52, 116, 179, 243, 204, 140, 76, 12, 51, 115, 178, 242, 205, 141, 77, 13, 50, 114, 177, 241, 206, 142, 78, 14, 49, 113, 176, 240, 207, 143, 79, 15, 48, 112, 175, 239, 208, 144, 80, 16, 47, 111, 174, 238, 209, 145, 81, 17, 46, 110, 173, 237, 210, 146, 82, 18, 45, 109, 172, 236, 211, 147, 83, 19, 44, 108, 171, 235, 212, 148, 84, 20, 43, 107, 170, 234, 213, 149, 85, 21, 42, 106, 169, 233, 214, 150, 86, 22, 41, 105, 168, 232, 215, 151, 87, 23, 40, 104, 167, 231, 216, 152, 88, 24, 39, 103, 166, 230, 217, 153, 89, 25, 38, 102, 165, 229, 218, 154, 90, 26, 37, 101, 164, 228, 219, 155, 91, 27, 36, 100, 163, 227, 220, 156, 92, 28, 35, 99, 162, 226, 221, 157, 93, 29, 34, 98, 161, 225, 222, 158, 94, 30, 33, 97, 160, 224, 223, 159, 95, 31, 32, 96, 159, 223, 224, 160, 96, 32, 31, 95, 158, 222, 225, 161, 97, 33, 30, 94, 157, 221, 226, 162, 98, 34, 29, 93, 156, 220, 227, 163, 99, 35, 28, 92, 155, 219, 228, 164, 100, 36, 27, 91, 154, 218, 229, 165, 101, 37, 26, 90, 153, 217, 230, 166, 102, 38, 25, 89, 152, 216, 231, 167, 103, 39, 24, 88, 151, 215, 232, 168, 104, 40, 23, 87, 150, 214, 233, 169, 105, 41, 22, 86, 149, 213, 234, 170, 106, 42, 21, 85, 148, 212, 235, 171, 107, 43, 20, 84, 147, 211, 236, 172, 108, 44, 19, 83, 146, 210, 237, 173, 109, 45, 18, 82, 145, 209, 238, 174, 110, 46, 17, 81, 144, 208, 239, 175, 111, 47, 16, 80, 143, 207, 240, 176, 112, 48, 15, 79, 142, 206, 241, 177, 113, 49, 14, 78, 141, 205, 242, 178, 114, 50, 13, 77, 140, 204, 243, 179, 115, 51, 12, 76, 139, 203, 244, 180, 116, 52, 11, 75, 138, 202, 245, 181, 117, 53, 10, 74, 137, 201, 246, 182, 118, 54, 9, 73, 136, 200, 247, 183, 119, 55, 8, 72, 135, 199, 248, 184, 120, 56, 7, 71, 134, 198, 249, 185, 121, 57, 6, 70, 133, 197, 250, 186, 122, 58, 5, 69, 132, 196, 251, 187, 123, 59, 4, 68, 131, 195, 252, 188, 124, 60, 3, 67, 130, 194, 253, 189, 125, 61, 2, 66, 129, 193, 254, 190, 126, 62, 1, 65, 128, 192, 255, 191, 127, 63, 0, 64, 127, 191, 255, 192, 128, 64, 0, 63, 126, 190, 254, 193, 129, 65, 1, 62, 125, 189, 253, 194, 130, 66, 2, 61, 124, 188, 252, 195, 131, 67, 3, 60, 123, 187, 251, 196, 132, 68, 4, 59, 122, 186, 250, 197, 133, 69, 5, 58, 121, 185, 249, 198, 134, 70, 6, 57, 120, 184, 248, 199, 135, 71, 7, 56, 119, 183, 247, 200, 136, 72, 8, 55, 118, 182, 246, 201, 137, 73, 9, 54, 117, 181, 245, 202, 138, 74, 10, 53, 116, 180, 244, 203, 139, 75, 11, 52, 115, 179, 243, 204, 140, 76, 12, 51, 114, 178, 242, 205, 141, 77, 13, 50, 113, 177, 241, 206, 142, 78, 14, 49, 112, 176, 240, 207, 143, 79, 15, 48, 111, 175, 239, 208, 144, 80, 16, 47, 110, 174, 238, 209, 145, 81, 17, 46, 109, 173, 237, 210, 146, 82, 18, 45, 108, 172, 236, 211, 147, 83, 19, 44, 107, 171, 235, 212, 148, 84, 20, 43, 106, 170, 234, 213, 149, 85, 21, 42, 105, 169, 233, 214, 150, 86, 22, 41, 104, 168, 232, 215, 151, 87, 23, 40, 103, 167, 231, 216, 152, 88, 24, 39, 102, 166, 230, 217, 153, 89, 25, 38, 101, 165, 229, 218, 154, 90, 26, 37, 100, 164, 228, 219, 155, 91, 27, 36, 99, 163, 227, 220, 156, 92, 28, 35, 98, 162, 226, 221, 157, 93, 29, 34, 97, 161, 225, 222, 158, 94, 30, 33, 96, 160, 224, 223, 159, 95, 31, 32, 95, 159, 223, 224, 160, 96, 32, 31, 94, 158, 222, 225, 161, 97, 33, 30, 93, 157, 221, 226, 162, 98, 34, 29, 92, 156, 220, 227, 163, 99, 35, 28, 91, 155, 219, 228, 164, 100, 36, 27, 90, 154, 218, 229, 165, 101, 37, 26, 89, 153, 217, 230, 166, 102, 38, 25, 88, 152, 216, 231, 167, 103, 39, 24, 87, 151, 215, 232, 168, 104, 40, 23, 86, 150, 214, 233, 169, 105, 41, 22, 85, 149, 213, 234, 170, 106, 42, 21, 84, 148, 212, 235, 171, 107, 43, 20, 83, 147, 211, 236, 172, 108, 44, 19, 82, 146, 210, 237, 173, 109, 45, 18, 81, 145, 209, 238, 174, 110, 46, 17, 80, 144, 208, 239, 175, 111, 47, 16, 79, 143, 207, 240, 176, 112, 48, 15, 78, 142, 206, 241, 177, 113, 49, 14, 77, 141, 205, 242, 178, 114, 50, 13, 76, 140, 204, 243, 179, 115, 51, 12, 75, 139, 203, 244, 180, 116, 52, 11, 74, 138, 202, 245, 181, 117, 53, 10, 73, 137, 201, 246, 182, 118, 54, 9, 72, 136, 200, 247, 183, 119, 55, 8, 71, 135, 199, 248, 184, 120, 56, 7, 70, 134, 198, 249, 185, 121, 57, 6, 69, 133, 197, 250, 186, 122, 58, 5, 68, 132, 196, 251, 187, 123, 59, 4, 67, 131, 195, 252, 188, 124, 60, 3, 66, 130, 194, 253, 189, 125, 61, 2, 65, 129, 193, 254, 190, 126, 62, 1, 64, 128, 192, 255, 191, 127, 63, 0, 63, 127, 191, 255, 192, 128, 64, 0, 62, 126, 190, 254, 193, 129, 65, 1, 61, 125, 189, 253, 194, 130, 66, 2, 60, 124, 188, 252, 195, 131, 67, 3, 59, 123, 187, 251, 196, 132, 68, 4, 58, 122, 186, 250, 197, 133, 69, 5, 57, 121, 185, 249, 198, 134, 70, 6, 56, 120, 184, 248, 199, 135, 71, 7, 55, 119, 183, 247, 200, 136, 72, 8, 54, 118, 182, 246, 201, 137, 73, 9, 53, 117, 181, 245, 202, 138, 74, 10, 52, 116, 180, 244, 203, 139, 75, 11, 51, 115, 179, 243, 204, 140, 76, 12, 50, 114, 178, 242, 205, 141, 77, 13, 49, 113, 177, 241, 206, 142, 78, 14, 48, 112, 176, 240, 207, 143, 79, 15, 47, 111, 175, 239, 208, 144, 80, 16, 46, 110, 174, 238, 209, 145, 81, 17, 45, 109, 173, 237, 210, 146, 82, 18, 44, 108, 172, 236, 211, 147, 83, 19, 43, 107, 171, 235, 212, 148, 84, 20, 42, 106, 170, 234, 213, 149, 85, 21, 41, 105, 169, 233, 214, 150, 86, 22, 40, 104, 168, 232, 215, 151, 87, 23, 39, 103, 167, 231, 216, 152, 88, 24, 38, 102, 166, 230, 217, 153, 89, 25, 37, 101, 165, 229, 218, 154, 90, 26, 36, 100, 164, 228, 219, 155, 91, 27, 35, 99, 163, 227, 220, 156, 92, 28, 34, 98, 162, 226, 221, 157, 93, 29, 33, 97, 161, 225, 222, 158, 94, 30, 32, 96, 160, 224, 223, 159, 95, 31, 31, 95, 159, 223, 224, 160, 96, 32, 30, 94, 158, 222, 225, 161, 97, 33, 29, 93, 157, 221, 226, 162, 98, 34, 28, 92, 156, 220, 227, 163, 99, 35, 27, 91, 155, 219, 228, 164, 100, 36, 26, 90, 154, 218, 229, 165, 101, 37, 25, 89, 153, 217, 230, 166, 102, 38, 24, 88, 152, 216, 231, 167, 103, 39, 23, 87, 151, 215, 232, 168, 104, 40, 22, 86, 150, 214, 233, 169, 105, 41, 21, 85, 149, 213, 234, 170, 106, 42, 20, 84, 148, 212, 235, 171, 107, 43, 19, 83, 147, 211, 236, 172, 108, 44, 18, 82, 146, 210, 237, 173, 109, 45, 17, 81, 145, 209, 238, 174, 110, 46, 16, 80, 144, 208, 239, 175, 111, 47, 15, 79, 143, 207, 240, 176, 112, 48, 14, 78, 142, 206, 241, 177, 113, 49, 13, 77, 141, 205, 242, 178, 114, 50, 12, 76, 140, 204, 243, 179, 115, 51, 11, 75, 139, 203, 244, 180, 116, 52, 10, 74, 138, 202, 245, 181, 117, 53, 9, 73, 137, 201, 246, 182, 118, 54, 8, 72, 136, 200, 247, 183, 119, 55, 7, 71, 135, 199, 248, 184, 120, 56, 6, 70, 134, 198, 249, 185, 121, 57, 5, 69, 133, 197, 250, 186, 122, 58, 4, 68, 132, 196, 251, 187, 123, 59, 3, 67, 131, 195, 252, 188, 124, 60, 2, 66, 130, 194, 253, 189, 125, 61, 1, 65, 129, 193, 254, 190, 126, 62, 0, 64, 128, 192, 255, 191, 127, 63, 0, 63, 127, 191, 255, 192, 128, 64, 1, 62, 126, 190, 254, 193, 129, 65, 2, 61, 125, 189, 253, 194, 130, 66, 3, 60, 124, 188, 252, 195, 131, 67, 4, 59, 123, 187, 251, 196, 132, 68, 5, 58, 122, 186, 250, 197, 133, 69, 6, 57, 121, 185, 249, 198, 134, 70, 7, 56, 120, 184, 248, 199, 135, 71, 8, 55, 119, 183, 247, 200, 136, 72, 9, 54, 118, 182, 246, 201, 137, 73, 10, 53, 117, 181, 245, 202, 138, 74, 11, 52, 116, 180, 244, 203, 139, 75, 12, 51, 115, 179, 243, 204, 140, 76, 13, 50, 114, 178, 242, 205, 141, 77, 14, 49, 113, 177, 241, 206, 142, 78, 15, 48, 112, 176, 240, 207, 143, 79, 16, 47, 111, 175, 239, 208, 144, 80, 17, 46, 110, 174, 238, 209, 145, 81, 18, 45, 109, 173, 237, 210, 146, 82, 19, 44, 108, 172, 236, 211, 147, 83, 20, 43, 107, 171, 235, 212, 148, 84, 21, 42, 106, 170, 234, 213, 149, 85, 22, 41, 105, 169, 233, 214, 150, 86, 23, 40, 104, 168, 232, 215, 151, 87, 24, 39, 103, 167, 231, 216, 152, 88, 25, 38, 102, 166, 230, 217, 153, 89, 26, 37, 101, 165, 229, 218, 154, 90, 27, 36, 100, 164, 228, 219, 155, 91, 28, 35, 99, 163, 227, 220, 156, 92, 29, 34, 98, 162, 226, 221, 157, 93, 30, 33, 97, 161, 225, 222, 158, 94, 31, 32, 96, 160, 224, 223, 159, 95, 32, 31, 95, 159, 223, 224, 160, 96, 33, 30, 94, 158, 222, 225, 161, 97, 34, 29, 93, 157, 221, 226, 162, 98, 35, 28, 92, 156, 220, 227, 163, 99, 36, 27, 91, 155, 219, 228, 164, 100, 37, 26, 90, 154, 218, 229, 165, 101, 38, 25, 89, 153, 217, 230, 166, 102, 39, 24, 88, 152, 216, 231, 167, 103, 40, 23, 87, 151, 215, 232, 168, 104, 41, 22, 86, 150, 214, 233, 169, 105, 42, 21, 85, 149, 213, 234, 170, 106, 43, 20, 84, 148, 212, 235, 171, 107, 44, 19, 83, 147, 211, 236, 172, 108, 45, 18, 82, 146, 210, 237, 173, 109, 46, 17, 81, 145, 209, 238, 174, 110, 47, 16, 80, 144, 208, 239, 175, 111, 48, 15, 79, 143, 207, 240, 176, 112, 49, 14, 78, 142, 206, 241, 177, 113, 50, 13, 77, 141, 205, 242, 178, 114, 51, 12, 76, 140, 204, 243, 179, 115, 52, 11, 75, 139, 203, 244, 180, 116, 53, 10, 74, 138, 202, 245, 181, 117, 54, 9, 73, 137, 201, 246, 182, 118, 55, 8, 72, 136, 200, 247, 183, 119, 56, 7, 71, 135, 199, 248, 184, 120, 57, 6, 70, 134, 198, 249, 185, 121, 58, 5, 69, 133, 197, 250, 186, 122, 59, 4, 68, 132, 196, 251, 187, 123, 60, 3, 67, 131, 195, 252, 188, 124, 61, 2, 66, 130, 194, 253, 189, 125, 62, 1, 65, 129, 193, 254, 190, 126, 63, 0, 64, 128, 192, 255, 191, 127, 64, 0, 63, 127, 191, 255, 192, 128, 65, 1, 62, 126, 190, 254, 193, 129, 66, 2, 61, 125, 189, 253, 194, 130, 67, 3, 60, 124, 188, 252, 195, 131, 68, 4, 59, 123, 187, 251, 196, 132, 69, 5, 58, 122, 186, 250, 197, 133, 70, 6, 57, 121, 185, 249, 198, 134, 71, 7, 56, 120, 184, 248, 199, 135, 72, 8, 55, 119, 183, 247, 200, 136, 73, 9, 54, 118, 182, 246, 201, 137, 74, 10, 53, 117, 181, 245, 202, 138, 75, 11, 52, 116, 180, 244, 203, 139, 76, 12, 51, 115, 179, 243, 204, 140, 77, 13, 50, 114, 178, 242, 205, 141, 78, 14, 49, 113, 177, 241, 206, 142, 79, 15, 48, 112, 176, 240, 207, 143, 80, 16, 47, 111, 175, 239, 208, 144, 81, 17, 46, 110, 174, 238, 209, 145, 82, 18, 45, 109, 173, 237, 210, 146, 83, 19, 44, 108, 172, 236, 211, 147, 84, 20, 43, 107, 171, 235, 212, 148, 85, 21, 42, 106, 170, 234, 213, 149, 86, 22, 41, 105, 169, 233, 214, 150, 87, 23, 40, 104, 168, 232, 215, 151, 88, 24, 39, 103, 167, 231, 216, 152, 89, 25, 38, 102, 166, 230, 217, 153, 90, 26, 37, 101, 165, 229, 218, 154, 91, 27, 36, 100, 164, 228, 219, 155, 92, 28, 35, 99, 163, 227, 220, 156, 93, 29, 34, 98, 162, 226, 221, 157, 94, 30, 33, 97, 161, 225, 222, 158, 95, 31, 32, 96, 160, 224, 223, 159, 96, 32, 31, 95, 159, 223, 224, 160, 97, 33, 30, 94, 158, 222, 225, 161, 98, 34, 29, 93, 157, 221, 226, 162, 99, 35, 28, 92, 156, 220, 227, 163, 100, 36, 27, 91, 155, 219, 228, 164, 101, 37, 26, 90, 154, 218, 229, 165, 102, 38, 25, 89, 153, 217, 230, 166, 103, 39, 24, 88, 152, 216, 231, 167, 104, 40, 23, 87, 151, 215, 232, 168, 105, 41, 22, 86, 150, 214, 233, 169, 106, 42, 21, 85, 149, 213, 234, 170, 107, 43, 20, 84, 148, 212, 235, 171, 108, 44, 19, 83, 147, 211, 236, 172, 109, 45, 18, 82, 146, 210, 237, 173, 110, 46, 17, 81, 145, 209, 238, 174, 111, 47, 16, 80, 144, 208, 239, 175, 112, 48, 15, 79, 143, 207, 240, 176, 113, 49, 14, 78, 142, 206, 241, 177, 114, 50, 13, 77, 141, 205, 242, 178, 115, 51, 12, 76, 140, 204, 243, 179, 116, 52, 11, 75, 139, 203, 244, 180, 117, 53, 10, 74, 138, 202, 245, 181, 118, 54, 9, 73, 137, 201, 246, 182, 119, 55, 8, 72, 136, 200, 247, 183, 120, 56, 7, 71, 135, 199, 248, 184, 121, 57, 6, 70, 134, 198, 249, 185, 122, 58, 5, 69, 133, 197, 250, 186, 123, 59, 4, 68, 132, 196, 251, 187, 124, 60, 3, 67, 131, 195, 252, 188, 125, 61, 2, 66, 130, 194, 253, 189, 126, 62, 1, 65, 129, 193, 254, 190, 127, 63, 0, 64, 128, 192, 255, 191, 128, 64, 0, 63, 127, 191, 255, 192, 129, 65, 1, 62, 126, 190, 254, 193, 130, 66, 2, 61, 125, 189, 253, 194, 131, 67, 3, 60, 124, 188, 252, 195, 132, 68, 4, 59, 123, 187, 251, 196, 133, 69, 5, 58, 122, 186, 250, 197, 134, 70, 6, 57, 121, 185, 249, 198, 135, 71, 7, 56, 120, 184, 248, 199, 136, 72, 8, 55, 119, 183, 247, 200, 137, 73, 9, 54, 118, 182, 246, 201, 138, 74, 10, 53, 117, 181, 245, 202, 139, 75, 11, 52, 116, 180, 244, 203, 140, 76, 12, 51, 115, 179, 243, 204, 141, 77, 13, 50, 114, 178, 242, 205, 142, 78, 14, 49, 113, 177, 241, 206, 143, 79, 15, 48, 112, 176, 240, 207, 144, 80, 16, 47, 111, 175, 239, 208, 145, 81, 17, 46, 110, 174, 238, 209, 146, 82, 18, 45, 109, 173, 237, 210, 147, 83, 19, 44, 108, 172, 236, 211, 148, 84, 20, 43, 107, 171, 235, 212, 149, 85, 21, 42, 106, 170, 234, 213, 150, 86, 22, 41, 105, 169, 233, 214, 151, 87, 23, 40, 104, 168, 232, 215, 152, 88, 24, 39, 103, 167, 231, 216, 153, 89, 25, 38, 102, 166, 230, 217, 154, 90, 26, 37, 101, 165, 229, 218, 155, 91, 27, 36, 100, 164, 228, 219, 156, 92, 28, 35, 99, 163, 227, 220, 157, 93, 29, 34, 98, 162, 226, 221, 158, 94, 30, 33, 97, 161, 225, 222, 159, 95, 31, 32, 96, 160, 224, 223, 160, 96, 32, 31, 95, 159, 223, 224, 161, 97, 33, 30, 94, 158, 222, 225, 162, 98, 34, 29, 93, 157, 221, 226, 163, 99, 35, 28, 92, 156, 220, 227, 164, 100, 36, 27, 91, 155, 219, 228, 165, 101, 37, 26, 90, 154, 218, 229, 166, 102, 38, 25, 89, 153, 217, 230, 167, 103, 39, 24, 88, 152, 216, 231, 168, 104, 40, 23, 87, 151, 215, 232, 169, 105, 41, 22, 86, 150, 214, 233, 170, 106, 42, 21, 85, 149, 213, 234, 171, 107, 43, 20, 84, 148, 212, 235, 172, 108, 44, 19, 83, 147, 211, 236, 173, 109, 45, 18, 82, 146, 210, 237, 174, 110, 46, 17, 81, 145, 209, 238, 175, 111, 47, 16, 80, 144, 208, 239, 176, 112, 48, 15, 79, 143, 207, 240, 177, 113, 49, 14, 78, 142, 206, 241, 178, 114, 50, 13, 77, 141, 205, 242, 179, 115, 51, 12, 76, 140, 204, 243, 180, 116, 52, 11, 75, 139, 203, 244, 181, 117, 53, 10, 74, 138, 202, 245, 182, 118, 54, 9, 73, 137, 201, 246, 183, 119, 55, 8, 72, 136, 200, 247, 184, 120, 56, 7, 71, 135, 199, 248, 185, 121, 57, 6, 70, 134, 198, 249, 186, 122, 58, 5, 69, 133, 197, 250, 187, 123, 59, 4, 68, 132, 196, 251, 188, 124, 60, 3, 67, 131, 195, 252, 189, 125, 61, 2, 66, 130, 194, 253, 190, 126, 62, 1, 65, 129, 193, 254, 191, 127, 63, 0, 64, 128, 192, 255, 192, 128, 64, 0, 63, 127, 191, 255, 193, 129, 65, 1, 62, 126, 190, 254, 194, 130, 66, 2, 61, 125, 189, 253, 195, 131, 67, 3, 60, 124, 188, 252, 196, 132, 68, 4, 59, 123, 187, 251, 197, 133, 69, 5, 58, 122, 186, 250, 198, 134, 70, 6, 57, 121, 185, 249, 199, 135, 71, 7, 56, 120, 184, 248, 200, 136, 72, 8, 55, 119, 183, 247, 201, 137, 73, 9, 54, 118, 182, 246, 202, 138, 74, 10, 53, 117, 181, 245, 203, 139, 75, 11, 52, 116, 180, 244, 204, 140, 76, 12, 51, 115, 179, 243, 205, 141, 77, 13, 50, 114, 178, 242, 206, 142, 78, 14, 49, 113, 177, 241, 207, 143, 79, 15, 48, 112, 176, 240, 208, 144, 80, 16, 47, 111, 175, 239, 209, 145, 81, 17, 46, 110, 174, 238, 210, 146, 82, 18, 45, 109, 173, 237, 211, 147, 83, 19, 44, 108, 172, 236, 212, 148, 84, 20, 43, 107, 171, 235, 213, 149, 85, 21, 42, 106, 170, 234, 214, 150, 86, 22, 41, 105, 169, 233, 215, 151, 87, 23, 40, 104, 168, 232, 216, 152, 88, 24, 39, 103, 167, 231, 217, 153, 89, 25, 38, 102, 166, 230, 218, 154, 90, 26, 37, 101, 165, 229, 219, 155, 91, 27, 36, 100, 164, 228, 220, 156, 92, 28, 35, 99, 163, 227, 221, 157, 93, 29, 34, 98, 162, 226, 222, 158, 94, 30, 33, 97, 161, 225, 223, 159, 95, 31, 32, 96, 160, 224, 224, 160, 96, 32, 31, 95, 159, 223, 225, 161, 97, 33, 30, 94, 158, 222, 226, 162, 98, 34, 29, 93, 157, 221, 227, 163, 99, 35, 28, 92, 156, 220, 228, 164, 100, 36, 27, 91, 155, 219, 229, 165, 101, 37, 26, 90, 154, 218, 230, 166, 102, 38, 25, 89, 153, 217, 231, 167, 103, 39, 24, 88, 152, 216, 232, 168, 104, 40, 23, 87, 151, 215, 233, 169, 105, 41, 22, 86, 150, 214, 234, 170, 106, 42, 21, 85, 149, 213, 235, 171, 107, 43, 20, 84, 148, 212, 236, 172, 108, 44, 19, 83, 147, 211, 237, 173, 109, 45, 18, 82, 146, 210, 238, 174, 110, 46, 17, 81, 145, 209, 239, 175, 111, 47, 16, 80, 144, 208, 240, 176, 112, 48, 15, 79, 143, 207, 241, 177, 113, 49, 14, 78, 142, 206, 242, 178, 114, 50, 13, 77, 141, 205, 243, 179, 115, 51, 12, 76, 140, 204, 244, 180, 116, 52, 11, 75, 139, 203, 245, 181, 117, 53, 10, 74, 138, 202, 246, 182, 118, 54, 9, 73, 137, 201, 247, 183, 119, 55, 8, 72, 136, 200, 248, 184, 120, 56, 7, 71, 135, 199, 249, 185, 121, 57, 6, 70, 134, 198, 250, 186, 122, 58, 5, 69, 133, 197, 251, 187, 123, 59, 4, 68, 132, 196, 252, 188, 124, 60, 3, 67, 131, 195, 253, 189, 125, 61, 2, 66, 130, 194, 254, 190, 126, 62, 1, 65, 129, 193, 255, 191, 127, 63, 0, 64, 128, 192};
    int gainModControlData[4096]{15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 3, 4, 15, 7, 3, 5, 1, 4, 7, 1, 0, 0, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 3, 4, 1, 7, 3, 5, 1, 4, 7, 15, 0, 0, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 3, 4, 1, 7, 3, 5, 1, 4, 7, 15, 0, 0, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 15, 4, 1, 7, 3, 5, 1, 4, 7, 15, 0, 0, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 15, 15, 1, 7, 3, 5, 1, 4, 7, 4, 0, 0, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 5, 15, 1, 7, 3, 15, 1, 4, 7, 4, 0, 0, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 5, 6, 1, 7, 3, 15, 1, 4, 7, 4, 0, 0, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 5, 6, 1, 15, 3, 7, 1, 4, 15, 4, 0, 0, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 5, 6, 1, 0, 3, 7, 1, 4, 15, 4, 15, 0, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 5, 6, 1, 0, 3, 7, 15, 4, 15, 4, 15, 0, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 5, 6, 1, 0, 3, 7, 15, 4, 2, 4, 15, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 5, 6, 1, 0, 15, 7, 15, 4, 2, 4, 3, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 5, 5, 6, 1, 0, 15, 7, 4, 15, 2, 4, 3, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 15, 5, 6, 1, 0, 5, 7, 4, 15, 2, 4, 3, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 15, 5, 15, 1, 0, 5, 7, 4, 15, 2, 4, 3, 0, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 15, 5, 15, 1, 0, 5, 15, 4, 7, 2, 4, 3, 0, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 15, 5, 0, 1, 15, 5, 15, 4, 7, 2, 4, 3, 15, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 1, 5, 0, 15, 1, 5, 15, 4, 7, 2, 4, 3, 15, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 5, 0, 15, 1, 5, 2, 4, 7, 15, 4, 3, 15, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 5, 0, 15, 1, 5, 2, 4, 7, 15, 4, 15, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 5, 0, 4, 1, 5, 2, 15, 7, 4, 15, 15, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 1, 15, 0, 4, 1, 15, 2, 15, 7, 4, 15, 5, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 1, 15, 0, 4, 1, 15, 2, 6, 7, 4, 6, 5, 3, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 1, 7, 0, 4, 1, 7, 2, 6, 15, 4, 6, 5, 3, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 1, 7, 15, 4, 1, 7, 2, 6, 15, 4, 6, 5, 3, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 15, 7, 1, 4, 15, 7, 2, 6, 1, 4, 6, 5, 3, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 15, 7, 1, 4, 15, 7, 2, 6, 1, 4, 6, 5, 3, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 3, 7, 1, 4, 15, 7, 2, 6, 1, 4, 6, 5, 15, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 3, 7, 1, 15, 4, 7, 2, 6, 1, 4, 6, 5, 15, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 1, 15, 4, 7, 2, 6, 1, 4, 6, 15, 5, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 1, 6, 4, 7, 2, 15, 1, 4, 15, 6, 5, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 3, 15, 1, 6, 4, 7, 2, 15, 1, 4, 15, 6, 5, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 3, 15, 1, 6, 4, 7, 2, 0, 1, 4, 0, 6, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 3, 1, 15, 6, 4, 7, 2, 0, 15, 4, 0, 6, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 3, 1, 2, 6, 4, 7, 15, 0, 15, 4, 0, 6, 5, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 15, 1, 2, 6, 4, 7, 15, 0, 3, 4, 0, 6, 5, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 15, 1, 2, 6, 15, 7, 4, 0, 3, 15, 0, 6, 5, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 5, 1, 2, 6, 15, 7, 4, 0, 3, 15, 0, 6, 15, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 5, 1, 2, 15, 6, 7, 4, 0, 3, 15, 0, 15, 6, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 1, 2, 15, 6, 15, 4, 0, 3, 15, 0, 15, 6, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 1, 2, 0, 6, 15, 4, 15, 3, 0, 15, 15, 6, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 5, 15, 2, 0, 6, 15, 4, 15, 3, 0, 15, 1, 6, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 5, 15, 15, 0, 6, 15, 4, 2, 3, 0, 2, 1, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 5, 3, 15, 0, 6, 3, 4, 2, 15, 0, 2, 1, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 5, 3, 15, 0, 6, 3, 15, 2, 15, 0, 2, 1, 6, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 15, 3, 5, 0, 6, 3, 15, 2, 5, 0, 2, 1, 6, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 15, 3, 5, 0, 6, 3, 6, 2, 5, 0, 2, 1, 15, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 7, 3, 5, 0, 6, 3, 6, 2, 5, 0, 2, 1, 15, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 7, 3, 5, 15, 6, 3, 6, 2, 5, 15, 2, 1, 0, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 3, 5, 1, 6, 3, 6, 2, 5, 15, 2, 1, 0, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 7, 3, 5, 1, 6, 3, 6, 15, 5, 2, 15, 1, 0, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 7, 15, 5, 1, 6, 3, 6, 15, 5, 2, 15, 1, 0, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 7, 15, 5, 1, 6, 3, 6, 4, 5, 2, 4, 1, 0, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 7, 5, 15, 1, 6, 3, 6, 4, 15, 2, 4, 1, 0, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 7, 5, 15, 1, 15, 3, 15, 4, 6, 2, 4, 1, 0, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 15, 5, 7, 1, 15, 3, 15, 4, 6, 2, 4, 1, 0, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 15, 5, 7, 1, 15, 3, 0, 4, 6, 2, 4, 1, 15, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 1, 5, 7, 15, 15, 3, 0, 4, 6, 2, 4, 15, 15, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 5, 7, 15, 2, 3, 0, 4, 6, 15, 4, 15, 15, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 5, 7, 15, 2, 15, 0, 4, 6, 3, 4, 15, 3, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 1, 5, 7, 4, 2, 15, 0, 15, 6, 3, 15, 4, 3, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 1, 15, 7, 4, 2, 15, 0, 5, 6, 3, 15, 4, 3, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 1, 15, 7, 4, 2, 6, 0, 5, 15, 3, 6, 4, 3, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 1, 7, 15, 4, 2, 6, 0, 5, 15, 3, 6, 4, 3, 7, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 1, 7, 15, 4, 2, 6, 15, 5, 0, 3, 6, 4, 3, 7, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 15, 7, 1, 4, 2, 6, 15, 5, 0, 3, 6, 4, 3, 7, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 15, 7, 1, 4, 15, 6, 2, 5, 0, 3, 6, 4, 3, 7, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 3, 7, 1, 4, 3, 6, 2, 5, 0, 15, 6, 4, 15, 7, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 3, 7, 1, 15, 3, 6, 2, 5, 0, 15, 6, 15, 15, 7, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 1, 15, 3, 6, 2, 15, 0, 5, 6, 15, 5, 7, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 3, 7, 1, 6, 3, 15, 2, 15, 0, 5, 15, 15, 5, 7, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 3, 15, 1, 6, 3, 15, 2, 15, 0, 5, 7, 7, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 3, 0, 1, 6, 3, 15, 2, 0, 15, 5, 7, 7, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 3, 0, 15, 6, 3, 1, 2, 0, 15, 5, 7, 7, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 3, 0, 15, 6, 3, 1, 15, 0, 15, 5, 7, 7, 5, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 15, 0, 3, 6, 15, 1, 15, 0, 3, 5, 7, 7, 5, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 4, 0, 3, 6, 15, 1, 4, 0, 3, 5, 7, 7, 5, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 4, 0, 3, 6, 15, 1, 4, 0, 3, 15, 7, 7, 15, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 4, 0, 3, 15, 6, 1, 4, 0, 3, 15, 7, 7, 6, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 4, 0, 3, 15, 6, 1, 4, 0, 3, 7, 15, 15, 6, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 4, 15, 3, 0, 6, 1, 4, 15, 3, 7, 15, 15, 6, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 4, 15, 3, 0, 6, 15, 4, 15, 3, 7, 15, 1, 6, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 4, 15, 3, 0, 6, 15, 4, 2, 3, 7, 2, 1, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 4, 3, 15, 0, 6, 3, 4, 2, 15, 7, 2, 1, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 15, 3, 15, 0, 6, 3, 15, 2, 15, 7, 2, 1, 6, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 15, 3, 15, 0, 6, 3, 15, 2, 5, 7, 2, 1, 6, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 15, 3, 6, 0, 15, 3, 15, 2, 5, 7, 2, 1, 15, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 3, 6, 0, 15, 3, 7, 2, 5, 15, 2, 1, 15, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 3, 6, 15, 0, 3, 7, 2, 5, 0, 2, 1, 15, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 3, 6, 15, 0, 3, 7, 2, 5, 0, 2, 15, 1, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 3, 6, 2, 0, 3, 7, 15, 5, 0, 15, 15, 1, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 15, 6, 2, 0, 15, 7, 15, 5, 0, 15, 3, 1, 4, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 15, 6, 2, 0, 15, 7, 4, 5, 0, 4, 3, 1, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 5, 6, 2, 0, 5, 7, 4, 15, 0, 4, 3, 1, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 5, 15, 2, 0, 5, 7, 4, 15, 0, 4, 3, 1, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 5, 15, 2, 0, 5, 15, 4, 7, 0, 4, 3, 1, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 5, 0, 2, 15, 5, 15, 4, 7, 15, 4, 3, 1, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 5, 0, 2, 15, 5, 1, 4, 7, 15, 4, 3, 15, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 3, 7, 5, 0, 15, 2, 5, 1, 4, 7, 15, 4, 3, 15, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 5, 0, 15, 2, 5, 1, 4, 7, 3, 4, 15, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 7, 5, 0, 4, 2, 5, 1, 15, 7, 3, 15, 4, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 7, 15, 0, 4, 2, 15, 1, 5, 7, 3, 15, 4, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 7, 15, 0, 4, 2, 15, 1, 5, 7, 3, 6, 4, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 15, 7, 0, 4, 2, 7, 1, 5, 15, 3, 6, 4, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 15, 7, 15, 4, 2, 7, 1, 5, 0, 3, 6, 4, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 15, 7, 1, 4, 2, 7, 15, 5, 0, 3, 6, 4, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 2, 7, 1, 4, 15, 7, 15, 5, 0, 3, 6, 4, 3, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 2, 7, 1, 4, 15, 7, 3, 5, 0, 15, 6, 4, 15, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 2, 7, 1, 15, 4, 7, 3, 5, 0, 15, 6, 15, 15, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 2, 7, 1, 5, 4, 7, 3, 15, 0, 15, 6, 15, 5, 6, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 2, 7, 1, 5, 4, 7, 3, 15, 0, 6, 15, 15, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 2, 15, 1, 5, 4, 15, 3, 15, 0, 6, 7, 7, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 2, 0, 1, 5, 4, 15, 3, 0, 15, 6, 7, 7, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 2, 0, 15, 5, 4, 1, 3, 0, 15, 6, 7, 7, 5, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 15, 0, 15, 5, 4, 1, 3, 0, 15, 6, 7, 7, 5, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 15, 0, 15, 5, 4, 1, 15, 0, 3, 6, 7, 7, 5, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 15, 0, 4, 5, 4, 1, 15, 0, 3, 6, 7, 7, 5, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 6, 5, 0, 4, 15, 4, 1, 5, 0, 3, 6, 7, 7, 15, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 0, 4, 6, 4, 1, 5, 0, 3, 15, 7, 7, 15, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 5, 0, 4, 6, 4, 1, 5, 0, 3, 7, 15, 15, 7, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 0, 5, 15, 4, 6, 4, 1, 5, 15, 3, 7, 15, 0, 7, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 0, 5, 15, 4, 6, 4, 15, 5, 1, 3, 7, 1, 0, 7, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 0, 5, 15, 4, 6, 4, 15, 5, 1, 3, 7, 1, 0, 7, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 0, 5, 3, 4, 6, 4, 3, 5, 1, 15, 7, 1, 0, 7, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 0, 5, 3, 4, 6, 15, 3, 5, 1, 4, 7, 1, 0, 7, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 0, 5, 3, 4, 6, 15, 3, 5, 1, 4, 7, 1, 0, 7, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 0, 5, 3, 4, 15, 15, 3, 5, 1, 4, 7, 1, 0, 7, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 0, 5, 3, 4, 15, 7, 3, 5, 1, 4, 7, 1, 0, 15, 2, 15, 15, 15, 15, 15, 15, 15, 15, 15};
    int writeAddressArray[16384]{16383, 16319, 16382, 16318, 16381, 16317, 16380, 16316, 16379, 16315, 16378, 16314, 16377, 16313, 16376, 16312, 16375, 16311, 16374, 16310, 16373, 16309, 16372, 16308, 16371, 16307, 16370, 16306, 16369, 16305, 16368, 16304, 16367, 16303, 16366, 16302, 16365, 16301, 16364, 16300, 16363, 16299, 16362, 16298, 16361, 16297, 16360, 16296, 16359, 16295, 16358, 16294, 16357, 16293, 16356, 16292, 16355, 16291, 16354, 16290, 16353, 16289, 16352, 16288, 16351, 16287, 16350, 16286, 16349, 16285, 16348, 16284, 16347, 16283, 16346, 16282, 16345, 16281, 16344, 16280, 16343, 16279, 16342, 16278, 16341, 16277, 16340, 16276, 16339, 16275, 16338, 16274, 16337, 16273, 16336, 16272, 16335, 16271, 16334, 16270, 16333, 16269, 16332, 16268, 16331, 16267, 16330, 16266, 16329, 16265, 16328, 16264, 16327, 16263, 16326, 16262, 16325, 16261, 16324, 16260, 16323, 16259, 16322, 16258, 16321, 16257, 16320, 16256, 16255, 16191, 16254, 16190, 16253, 16189, 16252, 16188, 16251, 16187, 16250, 16186, 16249, 16185, 16248, 16184, 16247, 16183, 16246, 16182, 16245, 16181, 16244, 16180, 16243, 16179, 16242, 16178, 16241, 16177, 16240, 16176, 16239, 16175, 16238, 16174, 16237, 16173, 16236, 16172, 16235, 16171, 16234, 16170, 16233, 16169, 16232, 16168, 16231, 16167, 16230, 16166, 16229, 16165, 16228, 16164, 16227, 16163, 16226, 16162, 16225, 16161, 16224, 16160, 16223, 16159, 16222, 16158, 16221, 16157, 16220, 16156, 16219, 16155, 16218, 16154, 16217, 16153, 16216, 16152, 16215, 16151, 16214, 16150, 16213, 16149, 16212, 16148, 16211, 16147, 16210, 16146, 16209, 16145, 16208, 16144, 16207, 16143, 16206, 16142, 16205, 16141, 16204, 16140, 16203, 16139, 16202, 16138, 16201, 16137, 16200, 16136, 16199, 16135, 16198, 16134, 16197, 16133, 16196, 16132, 16195, 16131, 16194, 16130, 16193, 16129, 16192, 16128, 16127, 16063, 16126, 16062, 16125, 16061, 16124, 16060, 16123, 16059, 16122, 16058, 16121, 16057, 16120, 16056, 16119, 16055, 16118, 16054, 16117, 16053, 16116, 16052, 16115, 16051, 16114, 16050, 16113, 16049, 16112, 16048, 16111, 16047, 16110, 16046, 16109, 16045, 16108, 16044, 16107, 16043, 16106, 16042, 16105, 16041, 16104, 16040, 16103, 16039, 16102, 16038, 16101, 16037, 16100, 16036, 16099, 16035, 16098, 16034, 16097, 16033, 16096, 16032, 16095, 16031, 16094, 16030, 16093, 16029, 16092, 16028, 16091, 16027, 16090, 16026, 16089, 16025, 16088, 16024, 16087, 16023, 16086, 16022, 16085, 16021, 16084, 16020, 16083, 16019, 16082, 16018, 16081, 16017, 16080, 16016, 16079, 16015, 16078, 16014, 16077, 16013, 16076, 16012, 16075, 16011, 16074, 16010, 16073, 16009, 16072, 16008, 16071, 16007, 16070, 16006, 16069, 16005, 16068, 16004, 16067, 16003, 16066, 16002, 16065, 16001, 16064, 16000, 15999, 15935, 15998, 15934, 15997, 15933, 15996, 15932, 15995, 15931, 15994, 15930, 15993, 15929, 15992, 15928, 15991, 15927, 15990, 15926, 15989, 15925, 15988, 15924, 15987, 15923, 15986, 15922, 15985, 15921, 15984, 15920, 15983, 15919, 15982, 15918, 15981, 15917, 15980, 15916, 15979, 15915, 15978, 15914, 15977, 15913, 15976, 15912, 15975, 15911, 15974, 15910, 15973, 15909, 15972, 15908, 15971, 15907, 15970, 15906, 15969, 15905, 15968, 15904, 15967, 15903, 15966, 15902, 15965, 15901, 15964, 15900, 15963, 15899, 15962, 15898, 15961, 15897, 15960, 15896, 15959, 15895, 15958, 15894, 15957, 15893, 15956, 15892, 15955, 15891, 15954, 15890, 15953, 15889, 15952, 15888, 15951, 15887, 15950, 15886, 15949, 15885, 15948, 15884, 15947, 15883, 15946, 15882, 15945, 15881, 15944, 15880, 15943, 15879, 15942, 15878, 15941, 15877, 15940, 15876, 15939, 15875, 15938, 15874, 15937, 15873, 15936, 15872, 15871, 15807, 15870, 15806, 15869, 15805, 15868, 15804, 15867, 15803, 15866, 15802, 15865, 15801, 15864, 15800, 15863, 15799, 15862, 15798, 15861, 15797, 15860, 15796, 15859, 15795, 15858, 15794, 15857, 15793, 15856, 15792, 15855, 15791, 15854, 15790, 15853, 15789, 15852, 15788, 15851, 15787, 15850, 15786, 15849, 15785, 15848, 15784, 15847, 15783, 15846, 15782, 15845, 15781, 15844, 15780, 15843, 15779, 15842, 15778, 15841, 15777, 15840, 15776, 15839, 15775, 15838, 15774, 15837, 15773, 15836, 15772, 15835, 15771, 15834, 15770, 15833, 15769, 15832, 15768, 15831, 15767, 15830, 15766, 15829, 15765, 15828, 15764, 15827, 15763, 15826, 15762, 15825, 15761, 15824, 15760, 15823, 15759, 15822, 15758, 15821, 15757, 15820, 15756, 15819, 15755, 15818, 15754, 15817, 15753, 15816, 15752, 15815, 15751, 15814, 15750, 15813, 15749, 15812, 15748, 15811, 15747, 15810, 15746, 15809, 15745, 15808, 15744, 15743, 15679, 15742, 15678, 15741, 15677, 15740, 15676, 15739, 15675, 15738, 15674, 15737, 15673, 15736, 15672, 15735, 15671, 15734, 15670, 15733, 15669, 15732, 15668, 15731, 15667, 15730, 15666, 15729, 15665, 15728, 15664, 15727, 15663, 15726, 15662, 15725, 15661, 15724, 15660, 15723, 15659, 15722, 15658, 15721, 15657, 15720, 15656, 15719, 15655, 15718, 15654, 15717, 15653, 15716, 15652, 15715, 15651, 15714, 15650, 15713, 15649, 15712, 15648, 15711, 15647, 15710, 15646, 15709, 15645, 15708, 15644, 15707, 15643, 15706, 15642, 15705, 15641, 15704, 15640, 15703, 15639, 15702, 15638, 15701, 15637, 15700, 15636, 15699, 15635, 15698, 15634, 15697, 15633, 15696, 15632, 15695, 15631, 15694, 15630, 15693, 15629, 15692, 15628, 15691, 15627, 15690, 15626, 15689, 15625, 15688, 15624, 15687, 15623, 15686, 15622, 15685, 15621, 15684, 15620, 15683, 15619, 15682, 15618, 15681, 15617, 15680, 15616, 15615, 15551, 15614, 15550, 15613, 15549, 15612, 15548, 15611, 15547, 15610, 15546, 15609, 15545, 15608, 15544, 15607, 15543, 15606, 15542, 15605, 15541, 15604, 15540, 15603, 15539, 15602, 15538, 15601, 15537, 15600, 15536, 15599, 15535, 15598, 15534, 15597, 15533, 15596, 15532, 15595, 15531, 15594, 15530, 15593, 15529, 15592, 15528, 15591, 15527, 15590, 15526, 15589, 15525, 15588, 15524, 15587, 15523, 15586, 15522, 15585, 15521, 15584, 15520, 15583, 15519, 15582, 15518, 15581, 15517, 15580, 15516, 15579, 15515, 15578, 15514, 15577, 15513, 15576, 15512, 15575, 15511, 15574, 15510, 15573, 15509, 15572, 15508, 15571, 15507, 15570, 15506, 15569, 15505, 15568, 15504, 15567, 15503, 15566, 15502, 15565, 15501, 15564, 15500, 15563, 15499, 15562, 15498, 15561, 15497, 15560, 15496, 15559, 15495, 15558, 15494, 15557, 15493, 15556, 15492, 15555, 15491, 15554, 15490, 15553, 15489, 15552, 15488, 15487, 15423, 15486, 15422, 15485, 15421, 15484, 15420, 15483, 15419, 15482, 15418, 15481, 15417, 15480, 15416, 15479, 15415, 15478, 15414, 15477, 15413, 15476, 15412, 15475, 15411, 15474, 15410, 15473, 15409, 15472, 15408, 15471, 15407, 15470, 15406, 15469, 15405, 15468, 15404, 15467, 15403, 15466, 15402, 15465, 15401, 15464, 15400, 15463, 15399, 15462, 15398, 15461, 15397, 15460, 15396, 15459, 15395, 15458, 15394, 15457, 15393, 15456, 15392, 15455, 15391, 15454, 15390, 15453, 15389, 15452, 15388, 15451, 15387, 15450, 15386, 15449, 15385, 15448, 15384, 15447, 15383, 15446, 15382, 15445, 15381, 15444, 15380, 15443, 15379, 15442, 15378, 15441, 15377, 15440, 15376, 15439, 15375, 15438, 15374, 15437, 15373, 15436, 15372, 15435, 15371, 15434, 15370, 15433, 15369, 15432, 15368, 15431, 15367, 15430, 15366, 15429, 15365, 15428, 15364, 15427, 15363, 15426, 15362, 15425, 15361, 15424, 15360, 15359, 15295, 15358, 15294, 15357, 15293, 15356, 15292, 15355, 15291, 15354, 15290, 15353, 15289, 15352, 15288, 15351, 15287, 15350, 15286, 15349, 15285, 15348, 15284, 15347, 15283, 15346, 15282, 15345, 15281, 15344, 15280, 15343, 15279, 15342, 15278, 15341, 15277, 15340, 15276, 15339, 15275, 15338, 15274, 15337, 15273, 15336, 15272, 15335, 15271, 15334, 15270, 15333, 15269, 15332, 15268, 15331, 15267, 15330, 15266, 15329, 15265, 15328, 15264, 15327, 15263, 15326, 15262, 15325, 15261, 15324, 15260, 15323, 15259, 15322, 15258, 15321, 15257, 15320, 15256, 15319, 15255, 15318, 15254, 15317, 15253, 15316, 15252, 15315, 15251, 15314, 15250, 15313, 15249, 15312, 15248, 15311, 15247, 15310, 15246, 15309, 15245, 15308, 15244, 15307, 15243, 15306, 15242, 15305, 15241, 15304, 15240, 15303, 15239, 15302, 15238, 15301, 15237, 15300, 15236, 15299, 15235, 15298, 15234, 15297, 15233, 15296, 15232, 15231, 15167, 15230, 15166, 15229, 15165, 15228, 15164, 15227, 15163, 15226, 15162, 15225, 15161, 15224, 15160, 15223, 15159, 15222, 15158, 15221, 15157, 15220, 15156, 15219, 15155, 15218, 15154, 15217, 15153, 15216, 15152, 15215, 15151, 15214, 15150, 15213, 15149, 15212, 15148, 15211, 15147, 15210, 15146, 15209, 15145, 15208, 15144, 15207, 15143, 15206, 15142, 15205, 15141, 15204, 15140, 15203, 15139, 15202, 15138, 15201, 15137, 15200, 15136, 15199, 15135, 15198, 15134, 15197, 15133, 15196, 15132, 15195, 15131, 15194, 15130, 15193, 15129, 15192, 15128, 15191, 15127, 15190, 15126, 15189, 15125, 15188, 15124, 15187, 15123, 15186, 15122, 15185, 15121, 15184, 15120, 15183, 15119, 15182, 15118, 15181, 15117, 15180, 15116, 15179, 15115, 15178, 15114, 15177, 15113, 15176, 15112, 15175, 15111, 15174, 15110, 15173, 15109, 15172, 15108, 15171, 15107, 15170, 15106, 15169, 15105, 15168, 15104, 15103, 15039, 15102, 15038, 15101, 15037, 15100, 15036, 15099, 15035, 15098, 15034, 15097, 15033, 15096, 15032, 15095, 15031, 15094, 15030, 15093, 15029, 15092, 15028, 15091, 15027, 15090, 15026, 15089, 15025, 15088, 15024, 15087, 15023, 15086, 15022, 15085, 15021, 15084, 15020, 15083, 15019, 15082, 15018, 15081, 15017, 15080, 15016, 15079, 15015, 15078, 15014, 15077, 15013, 15076, 15012, 15075, 15011, 15074, 15010, 15073, 15009, 15072, 15008, 15071, 15007, 15070, 15006, 15069, 15005, 15068, 15004, 15067, 15003, 15066, 15002, 15065, 15001, 15064, 15000, 15063, 14999, 15062, 14998, 15061, 14997, 15060, 14996, 15059, 14995, 15058, 14994, 15057, 14993, 15056, 14992, 15055, 14991, 15054, 14990, 15053, 14989, 15052, 14988, 15051, 14987, 15050, 14986, 15049, 14985, 15048, 14984, 15047, 14983, 15046, 14982, 15045, 14981, 15044, 14980, 15043, 14979, 15042, 14978, 15041, 14977, 15040, 14976, 14975, 14911, 14974, 14910, 14973, 14909, 14972, 14908, 14971, 14907, 14970, 14906, 14969, 14905, 14968, 14904, 14967, 14903, 14966, 14902, 14965, 14901, 14964, 14900, 14963, 14899, 14962, 14898, 14961, 14897, 14960, 14896, 14959, 14895, 14958, 14894, 14957, 14893, 14956, 14892, 14955, 14891, 14954, 14890, 14953, 14889, 14952, 14888, 14951, 14887, 14950, 14886, 14949, 14885, 14948, 14884, 14947, 14883, 14946, 14882, 14945, 14881, 14944, 14880, 14943, 14879, 14942, 14878, 14941, 14877, 14940, 14876, 14939, 14875, 14938, 14874, 14937, 14873, 14936, 14872, 14935, 14871, 14934, 14870, 14933, 14869, 14932, 14868, 14931, 14867, 14930, 14866, 14929, 14865, 14928, 14864, 14927, 14863, 14926, 14862, 14925, 14861, 14924, 14860, 14923, 14859, 14922, 14858, 14921, 14857, 14920, 14856, 14919, 14855, 14918, 14854, 14917, 14853, 14916, 14852, 14915, 14851, 14914, 14850, 14913, 14849, 14912, 14848, 14847, 14783, 14846, 14782, 14845, 14781, 14844, 14780, 14843, 14779, 14842, 14778, 14841, 14777, 14840, 14776, 14839, 14775, 14838, 14774, 14837, 14773, 14836, 14772, 14835, 14771, 14834, 14770, 14833, 14769, 14832, 14768, 14831, 14767, 14830, 14766, 14829, 14765, 14828, 14764, 14827, 14763, 14826, 14762, 14825, 14761, 14824, 14760, 14823, 14759, 14822, 14758, 14821, 14757, 14820, 14756, 14819, 14755, 14818, 14754, 14817, 14753, 14816, 14752, 14815, 14751, 14814, 14750, 14813, 14749, 14812, 14748, 14811, 14747, 14810, 14746, 14809, 14745, 14808, 14744, 14807, 14743, 14806, 14742, 14805, 14741, 14804, 14740, 14803, 14739, 14802, 14738, 14801, 14737, 14800, 14736, 14799, 14735, 14798, 14734, 14797, 14733, 14796, 14732, 14795, 14731, 14794, 14730, 14793, 14729, 14792, 14728, 14791, 14727, 14790, 14726, 14789, 14725, 14788, 14724, 14787, 14723, 14786, 14722, 14785, 14721, 14784, 14720, 14719, 14655, 14718, 14654, 14717, 14653, 14716, 14652, 14715, 14651, 14714, 14650, 14713, 14649, 14712, 14648, 14711, 14647, 14710, 14646, 14709, 14645, 14708, 14644, 14707, 14643, 14706, 14642, 14705, 14641, 14704, 14640, 14703, 14639, 14702, 14638, 14701, 14637, 14700, 14636, 14699, 14635, 14698, 14634, 14697, 14633, 14696, 14632, 14695, 14631, 14694, 14630, 14693, 14629, 14692, 14628, 14691, 14627, 14690, 14626, 14689, 14625, 14688, 14624, 14687, 14623, 14686, 14622, 14685, 14621, 14684, 14620, 14683, 14619, 14682, 14618, 14681, 14617, 14680, 14616, 14679, 14615, 14678, 14614, 14677, 14613, 14676, 14612, 14675, 14611, 14674, 14610, 14673, 14609, 14672, 14608, 14671, 14607, 14670, 14606, 14669, 14605, 14668, 14604, 14667, 14603, 14666, 14602, 14665, 14601, 14664, 14600, 14663, 14599, 14662, 14598, 14661, 14597, 14660, 14596, 14659, 14595, 14658, 14594, 14657, 14593, 14656, 14592, 14591, 14527, 14590, 14526, 14589, 14525, 14588, 14524, 14587, 14523, 14586, 14522, 14585, 14521, 14584, 14520, 14583, 14519, 14582, 14518, 14581, 14517, 14580, 14516, 14579, 14515, 14578, 14514, 14577, 14513, 14576, 14512, 14575, 14511, 14574, 14510, 14573, 14509, 14572, 14508, 14571, 14507, 14570, 14506, 14569, 14505, 14568, 14504, 14567, 14503, 14566, 14502, 14565, 14501, 14564, 14500, 14563, 14499, 14562, 14498, 14561, 14497, 14560, 14496, 14559, 14495, 14558, 14494, 14557, 14493, 14556, 14492, 14555, 14491, 14554, 14490, 14553, 14489, 14552, 14488, 14551, 14487, 14550, 14486, 14549, 14485, 14548, 14484, 14547, 14483, 14546, 14482, 14545, 14481, 14544, 14480, 14543, 14479, 14542, 14478, 14541, 14477, 14540, 14476, 14539, 14475, 14538, 14474, 14537, 14473, 14536, 14472, 14535, 14471, 14534, 14470, 14533, 14469, 14532, 14468, 14531, 14467, 14530, 14466, 14529, 14465, 14528, 14464, 14463, 14399, 14462, 14398, 14461, 14397, 14460, 14396, 14459, 14395, 14458, 14394, 14457, 14393, 14456, 14392, 14455, 14391, 14454, 14390, 14453, 14389, 14452, 14388, 14451, 14387, 14450, 14386, 14449, 14385, 14448, 14384, 14447, 14383, 14446, 14382, 14445, 14381, 14444, 14380, 14443, 14379, 14442, 14378, 14441, 14377, 14440, 14376, 14439, 14375, 14438, 14374, 14437, 14373, 14436, 14372, 14435, 14371, 14434, 14370, 14433, 14369, 14432, 14368, 14431, 14367, 14430, 14366, 14429, 14365, 14428, 14364, 14427, 14363, 14426, 14362, 14425, 14361, 14424, 14360, 14423, 14359, 14422, 14358, 14421, 14357, 14420, 14356, 14419, 14355, 14418, 14354, 14417, 14353, 14416, 14352, 14415, 14351, 14414, 14350, 14413, 14349, 14412, 14348, 14411, 14347, 14410, 14346, 14409, 14345, 14408, 14344, 14407, 14343, 14406, 14342, 14405, 14341, 14404, 14340, 14403, 14339, 14402, 14338, 14401, 14337, 14400, 14336, 14335, 14271, 14334, 14270, 14333, 14269, 14332, 14268, 14331, 14267, 14330, 14266, 14329, 14265, 14328, 14264, 14327, 14263, 14326, 14262, 14325, 14261, 14324, 14260, 14323, 14259, 14322, 14258, 14321, 14257, 14320, 14256, 14319, 14255, 14318, 14254, 14317, 14253, 14316, 14252, 14315, 14251, 14314, 14250, 14313, 14249, 14312, 14248, 14311, 14247, 14310, 14246, 14309, 14245, 14308, 14244, 14307, 14243, 14306, 14242, 14305, 14241, 14304, 14240, 14303, 14239, 14302, 14238, 14301, 14237, 14300, 14236, 14299, 14235, 14298, 14234, 14297, 14233, 14296, 14232, 14295, 14231, 14294, 14230, 14293, 14229, 14292, 14228, 14291, 14227, 14290, 14226, 14289, 14225, 14288, 14224, 14287, 14223, 14286, 14222, 14285, 14221, 14284, 14220, 14283, 14219, 14282, 14218, 14281, 14217, 14280, 14216, 14279, 14215, 14278, 14214, 14277, 14213, 14276, 14212, 14275, 14211, 14274, 14210, 14273, 14209, 14272, 14208, 14207, 14143, 14206, 14142, 14205, 14141, 14204, 14140, 14203, 14139, 14202, 14138, 14201, 14137, 14200, 14136, 14199, 14135, 14198, 14134, 14197, 14133, 14196, 14132, 14195, 14131, 14194, 14130, 14193, 14129, 14192, 14128, 14191, 14127, 14190, 14126, 14189, 14125, 14188, 14124, 14187, 14123, 14186, 14122, 14185, 14121, 14184, 14120, 14183, 14119, 14182, 14118, 14181, 14117, 14180, 14116, 14179, 14115, 14178, 14114, 14177, 14113, 14176, 14112, 14175, 14111, 14174, 14110, 14173, 14109, 14172, 14108, 14171, 14107, 14170, 14106, 14169, 14105, 14168, 14104, 14167, 14103, 14166, 14102, 14165, 14101, 14164, 14100, 14163, 14099, 14162, 14098, 14161, 14097, 14160, 14096, 14159, 14095, 14158, 14094, 14157, 14093, 14156, 14092, 14155, 14091, 14154, 14090, 14153, 14089, 14152, 14088, 14151, 14087, 14150, 14086, 14149, 14085, 14148, 14084, 14147, 14083, 14146, 14082, 14145, 14081, 14144, 14080, 14079, 14015, 14078, 14014, 14077, 14013, 14076, 14012, 14075, 14011, 14074, 14010, 14073, 14009, 14072, 14008, 14071, 14007, 14070, 14006, 14069, 14005, 14068, 14004, 14067, 14003, 14066, 14002, 14065, 14001, 14064, 14000, 14063, 13999, 14062, 13998, 14061, 13997, 14060, 13996, 14059, 13995, 14058, 13994, 14057, 13993, 14056, 13992, 14055, 13991, 14054, 13990, 14053, 13989, 14052, 13988, 14051, 13987, 14050, 13986, 14049, 13985, 14048, 13984, 14047, 13983, 14046, 13982, 14045, 13981, 14044, 13980, 14043, 13979, 14042, 13978, 14041, 13977, 14040, 13976, 14039, 13975, 14038, 13974, 14037, 13973, 14036, 13972, 14035, 13971, 14034, 13970, 14033, 13969, 14032, 13968, 14031, 13967, 14030, 13966, 14029, 13965, 14028, 13964, 14027, 13963, 14026, 13962, 14025, 13961, 14024, 13960, 14023, 13959, 14022, 13958, 14021, 13957, 14020, 13956, 14019, 13955, 14018, 13954, 14017, 13953, 14016, 13952, 13951, 13887, 13950, 13886, 13949, 13885, 13948, 13884, 13947, 13883, 13946, 13882, 13945, 13881, 13944, 13880, 13943, 13879, 13942, 13878, 13941, 13877, 13940, 13876, 13939, 13875, 13938, 13874, 13937, 13873, 13936, 13872, 13935, 13871, 13934, 13870, 13933, 13869, 13932, 13868, 13931, 13867, 13930, 13866, 13929, 13865, 13928, 13864, 13927, 13863, 13926, 13862, 13925, 13861, 13924, 13860, 13923, 13859, 13922, 13858, 13921, 13857, 13920, 13856, 13919, 13855, 13918, 13854, 13917, 13853, 13916, 13852, 13915, 13851, 13914, 13850, 13913, 13849, 13912, 13848, 13911, 13847, 13910, 13846, 13909, 13845, 13908, 13844, 13907, 13843, 13906, 13842, 13905, 13841, 13904, 13840, 13903, 13839, 13902, 13838, 13901, 13837, 13900, 13836, 13899, 13835, 13898, 13834, 13897, 13833, 13896, 13832, 13895, 13831, 13894, 13830, 13893, 13829, 13892, 13828, 13891, 13827, 13890, 13826, 13889, 13825, 13888, 13824, 13823, 13759, 13822, 13758, 13821, 13757, 13820, 13756, 13819, 13755, 13818, 13754, 13817, 13753, 13816, 13752, 13815, 13751, 13814, 13750, 13813, 13749, 13812, 13748, 13811, 13747, 13810, 13746, 13809, 13745, 13808, 13744, 13807, 13743, 13806, 13742, 13805, 13741, 13804, 13740, 13803, 13739, 13802, 13738, 13801, 13737, 13800, 13736, 13799, 13735, 13798, 13734, 13797, 13733, 13796, 13732, 13795, 13731, 13794, 13730, 13793, 13729, 13792, 13728, 13791, 13727, 13790, 13726, 13789, 13725, 13788, 13724, 13787, 13723, 13786, 13722, 13785, 13721, 13784, 13720, 13783, 13719, 13782, 13718, 13781, 13717, 13780, 13716, 13779, 13715, 13778, 13714, 13777, 13713, 13776, 13712, 13775, 13711, 13774, 13710, 13773, 13709, 13772, 13708, 13771, 13707, 13770, 13706, 13769, 13705, 13768, 13704, 13767, 13703, 13766, 13702, 13765, 13701, 13764, 13700, 13763, 13699, 13762, 13698, 13761, 13697, 13760, 13696, 13695, 13631, 13694, 13630, 13693, 13629, 13692, 13628, 13691, 13627, 13690, 13626, 13689, 13625, 13688, 13624, 13687, 13623, 13686, 13622, 13685, 13621, 13684, 13620, 13683, 13619, 13682, 13618, 13681, 13617, 13680, 13616, 13679, 13615, 13678, 13614, 13677, 13613, 13676, 13612, 13675, 13611, 13674, 13610, 13673, 13609, 13672, 13608, 13671, 13607, 13670, 13606, 13669, 13605, 13668, 13604, 13667, 13603, 13666, 13602, 13665, 13601, 13664, 13600, 13663, 13599, 13662, 13598, 13661, 13597, 13660, 13596, 13659, 13595, 13658, 13594, 13657, 13593, 13656, 13592, 13655, 13591, 13654, 13590, 13653, 13589, 13652, 13588, 13651, 13587, 13650, 13586, 13649, 13585, 13648, 13584, 13647, 13583, 13646, 13582, 13645, 13581, 13644, 13580, 13643, 13579, 13642, 13578, 13641, 13577, 13640, 13576, 13639, 13575, 13638, 13574, 13637, 13573, 13636, 13572, 13635, 13571, 13634, 13570, 13633, 13569, 13632, 13568, 13567, 13503, 13566, 13502, 13565, 13501, 13564, 13500, 13563, 13499, 13562, 13498, 13561, 13497, 13560, 13496, 13559, 13495, 13558, 13494, 13557, 13493, 13556, 13492, 13555, 13491, 13554, 13490, 13553, 13489, 13552, 13488, 13551, 13487, 13550, 13486, 13549, 13485, 13548, 13484, 13547, 13483, 13546, 13482, 13545, 13481, 13544, 13480, 13543, 13479, 13542, 13478, 13541, 13477, 13540, 13476, 13539, 13475, 13538, 13474, 13537, 13473, 13536, 13472, 13535, 13471, 13534, 13470, 13533, 13469, 13532, 13468, 13531, 13467, 13530, 13466, 13529, 13465, 13528, 13464, 13527, 13463, 13526, 13462, 13525, 13461, 13524, 13460, 13523, 13459, 13522, 13458, 13521, 13457, 13520, 13456, 13519, 13455, 13518, 13454, 13517, 13453, 13516, 13452, 13515, 13451, 13514, 13450, 13513, 13449, 13512, 13448, 13511, 13447, 13510, 13446, 13509, 13445, 13508, 13444, 13507, 13443, 13506, 13442, 13505, 13441, 13504, 13440, 13439, 13375, 13438, 13374, 13437, 13373, 13436, 13372, 13435, 13371, 13434, 13370, 13433, 13369, 13432, 13368, 13431, 13367, 13430, 13366, 13429, 13365, 13428, 13364, 13427, 13363, 13426, 13362, 13425, 13361, 13424, 13360, 13423, 13359, 13422, 13358, 13421, 13357, 13420, 13356, 13419, 13355, 13418, 13354, 13417, 13353, 13416, 13352, 13415, 13351, 13414, 13350, 13413, 13349, 13412, 13348, 13411, 13347, 13410, 13346, 13409, 13345, 13408, 13344, 13407, 13343, 13406, 13342, 13405, 13341, 13404, 13340, 13403, 13339, 13402, 13338, 13401, 13337, 13400, 13336, 13399, 13335, 13398, 13334, 13397, 13333, 13396, 13332, 13395, 13331, 13394, 13330, 13393, 13329, 13392, 13328, 13391, 13327, 13390, 13326, 13389, 13325, 13388, 13324, 13387, 13323, 13386, 13322, 13385, 13321, 13384, 13320, 13383, 13319, 13382, 13318, 13381, 13317, 13380, 13316, 13379, 13315, 13378, 13314, 13377, 13313, 13376, 13312, 13311, 13247, 13310, 13246, 13309, 13245, 13308, 13244, 13307, 13243, 13306, 13242, 13305, 13241, 13304, 13240, 13303, 13239, 13302, 13238, 13301, 13237, 13300, 13236, 13299, 13235, 13298, 13234, 13297, 13233, 13296, 13232, 13295, 13231, 13294, 13230, 13293, 13229, 13292, 13228, 13291, 13227, 13290, 13226, 13289, 13225, 13288, 13224, 13287, 13223, 13286, 13222, 13285, 13221, 13284, 13220, 13283, 13219, 13282, 13218, 13281, 13217, 13280, 13216, 13279, 13215, 13278, 13214, 13277, 13213, 13276, 13212, 13275, 13211, 13274, 13210, 13273, 13209, 13272, 13208, 13271, 13207, 13270, 13206, 13269, 13205, 13268, 13204, 13267, 13203, 13266, 13202, 13265, 13201, 13264, 13200, 13263, 13199, 13262, 13198, 13261, 13197, 13260, 13196, 13259, 13195, 13258, 13194, 13257, 13193, 13256, 13192, 13255, 13191, 13254, 13190, 13253, 13189, 13252, 13188, 13251, 13187, 13250, 13186, 13249, 13185, 13248, 13184, 13183, 13119, 13182, 13118, 13181, 13117, 13180, 13116, 13179, 13115, 13178, 13114, 13177, 13113, 13176, 13112, 13175, 13111, 13174, 13110, 13173, 13109, 13172, 13108, 13171, 13107, 13170, 13106, 13169, 13105, 13168, 13104, 13167, 13103, 13166, 13102, 13165, 13101, 13164, 13100, 13163, 13099, 13162, 13098, 13161, 13097, 13160, 13096, 13159, 13095, 13158, 13094, 13157, 13093, 13156, 13092, 13155, 13091, 13154, 13090, 13153, 13089, 13152, 13088, 13151, 13087, 13150, 13086, 13149, 13085, 13148, 13084, 13147, 13083, 13146, 13082, 13145, 13081, 13144, 13080, 13143, 13079, 13142, 13078, 13141, 13077, 13140, 13076, 13139, 13075, 13138, 13074, 13137, 13073, 13136, 13072, 13135, 13071, 13134, 13070, 13133, 13069, 13132, 13068, 13131, 13067, 13130, 13066, 13129, 13065, 13128, 13064, 13127, 13063, 13126, 13062, 13125, 13061, 13124, 13060, 13123, 13059, 13122, 13058, 13121, 13057, 13120, 13056, 13055, 12991, 13054, 12990, 13053, 12989, 13052, 12988, 13051, 12987, 13050, 12986, 13049, 12985, 13048, 12984, 13047, 12983, 13046, 12982, 13045, 12981, 13044, 12980, 13043, 12979, 13042, 12978, 13041, 12977, 13040, 12976, 13039, 12975, 13038, 12974, 13037, 12973, 13036, 12972, 13035, 12971, 13034, 12970, 13033, 12969, 13032, 12968, 13031, 12967, 13030, 12966, 13029, 12965, 13028, 12964, 13027, 12963, 13026, 12962, 13025, 12961, 13024, 12960, 13023, 12959, 13022, 12958, 13021, 12957, 13020, 12956, 13019, 12955, 13018, 12954, 13017, 12953, 13016, 12952, 13015, 12951, 13014, 12950, 13013, 12949, 13012, 12948, 13011, 12947, 13010, 12946, 13009, 12945, 13008, 12944, 13007, 12943, 13006, 12942, 13005, 12941, 13004, 12940, 13003, 12939, 13002, 12938, 13001, 12937, 13000, 12936, 12999, 12935, 12998, 12934, 12997, 12933, 12996, 12932, 12995, 12931, 12994, 12930, 12993, 12929, 12992, 12928, 12927, 12863, 12926, 12862, 12925, 12861, 12924, 12860, 12923, 12859, 12922, 12858, 12921, 12857, 12920, 12856, 12919, 12855, 12918, 12854, 12917, 12853, 12916, 12852, 12915, 12851, 12914, 12850, 12913, 12849, 12912, 12848, 12911, 12847, 12910, 12846, 12909, 12845, 12908, 12844, 12907, 12843, 12906, 12842, 12905, 12841, 12904, 12840, 12903, 12839, 12902, 12838, 12901, 12837, 12900, 12836, 12899, 12835, 12898, 12834, 12897, 12833, 12896, 12832, 12895, 12831, 12894, 12830, 12893, 12829, 12892, 12828, 12891, 12827, 12890, 12826, 12889, 12825, 12888, 12824, 12887, 12823, 12886, 12822, 12885, 12821, 12884, 12820, 12883, 12819, 12882, 12818, 12881, 12817, 12880, 12816, 12879, 12815, 12878, 12814, 12877, 12813, 12876, 12812, 12875, 12811, 12874, 12810, 12873, 12809, 12872, 12808, 12871, 12807, 12870, 12806, 12869, 12805, 12868, 12804, 12867, 12803, 12866, 12802, 12865, 12801, 12864, 12800, 12799, 12735, 12798, 12734, 12797, 12733, 12796, 12732, 12795, 12731, 12794, 12730, 12793, 12729, 12792, 12728, 12791, 12727, 12790, 12726, 12789, 12725, 12788, 12724, 12787, 12723, 12786, 12722, 12785, 12721, 12784, 12720, 12783, 12719, 12782, 12718, 12781, 12717, 12780, 12716, 12779, 12715, 12778, 12714, 12777, 12713, 12776, 12712, 12775, 12711, 12774, 12710, 12773, 12709, 12772, 12708, 12771, 12707, 12770, 12706, 12769, 12705, 12768, 12704, 12767, 12703, 12766, 12702, 12765, 12701, 12764, 12700, 12763, 12699, 12762, 12698, 12761, 12697, 12760, 12696, 12759, 12695, 12758, 12694, 12757, 12693, 12756, 12692, 12755, 12691, 12754, 12690, 12753, 12689, 12752, 12688, 12751, 12687, 12750, 12686, 12749, 12685, 12748, 12684, 12747, 12683, 12746, 12682, 12745, 12681, 12744, 12680, 12743, 12679, 12742, 12678, 12741, 12677, 12740, 12676, 12739, 12675, 12738, 12674, 12737, 12673, 12736, 12672, 12671, 12607, 12670, 12606, 12669, 12605, 12668, 12604, 12667, 12603, 12666, 12602, 12665, 12601, 12664, 12600, 12663, 12599, 12662, 12598, 12661, 12597, 12660, 12596, 12659, 12595, 12658, 12594, 12657, 12593, 12656, 12592, 12655, 12591, 12654, 12590, 12653, 12589, 12652, 12588, 12651, 12587, 12650, 12586, 12649, 12585, 12648, 12584, 12647, 12583, 12646, 12582, 12645, 12581, 12644, 12580, 12643, 12579, 12642, 12578, 12641, 12577, 12640, 12576, 12639, 12575, 12638, 12574, 12637, 12573, 12636, 12572, 12635, 12571, 12634, 12570, 12633, 12569, 12632, 12568, 12631, 12567, 12630, 12566, 12629, 12565, 12628, 12564, 12627, 12563, 12626, 12562, 12625, 12561, 12624, 12560, 12623, 12559, 12622, 12558, 12621, 12557, 12620, 12556, 12619, 12555, 12618, 12554, 12617, 12553, 12616, 12552, 12615, 12551, 12614, 12550, 12613, 12549, 12612, 12548, 12611, 12547, 12610, 12546, 12609, 12545, 12608, 12544, 12543, 12479, 12542, 12478, 12541, 12477, 12540, 12476, 12539, 12475, 12538, 12474, 12537, 12473, 12536, 12472, 12535, 12471, 12534, 12470, 12533, 12469, 12532, 12468, 12531, 12467, 12530, 12466, 12529, 12465, 12528, 12464, 12527, 12463, 12526, 12462, 12525, 12461, 12524, 12460, 12523, 12459, 12522, 12458, 12521, 12457, 12520, 12456, 12519, 12455, 12518, 12454, 12517, 12453, 12516, 12452, 12515, 12451, 12514, 12450, 12513, 12449, 12512, 12448, 12511, 12447, 12510, 12446, 12509, 12445, 12508, 12444, 12507, 12443, 12506, 12442, 12505, 12441, 12504, 12440, 12503, 12439, 12502, 12438, 12501, 12437, 12500, 12436, 12499, 12435, 12498, 12434, 12497, 12433, 12496, 12432, 12495, 12431, 12494, 12430, 12493, 12429, 12492, 12428, 12491, 12427, 12490, 12426, 12489, 12425, 12488, 12424, 12487, 12423, 12486, 12422, 12485, 12421, 12484, 12420, 12483, 12419, 12482, 12418, 12481, 12417, 12480, 12416, 12415, 12351, 12414, 12350, 12413, 12349, 12412, 12348, 12411, 12347, 12410, 12346, 12409, 12345, 12408, 12344, 12407, 12343, 12406, 12342, 12405, 12341, 12404, 12340, 12403, 12339, 12402, 12338, 12401, 12337, 12400, 12336, 12399, 12335, 12398, 12334, 12397, 12333, 12396, 12332, 12395, 12331, 12394, 12330, 12393, 12329, 12392, 12328, 12391, 12327, 12390, 12326, 12389, 12325, 12388, 12324, 12387, 12323, 12386, 12322, 12385, 12321, 12384, 12320, 12383, 12319, 12382, 12318, 12381, 12317, 12380, 12316, 12379, 12315, 12378, 12314, 12377, 12313, 12376, 12312, 12375, 12311, 12374, 12310, 12373, 12309, 12372, 12308, 12371, 12307, 12370, 12306, 12369, 12305, 12368, 12304, 12367, 12303, 12366, 12302, 12365, 12301, 12364, 12300, 12363, 12299, 12362, 12298, 12361, 12297, 12360, 12296, 12359, 12295, 12358, 12294, 12357, 12293, 12356, 12292, 12355, 12291, 12354, 12290, 12353, 12289, 12352, 12288, 12287, 12223, 12286, 12222, 12285, 12221, 12284, 12220, 12283, 12219, 12282, 12218, 12281, 12217, 12280, 12216, 12279, 12215, 12278, 12214, 12277, 12213, 12276, 12212, 12275, 12211, 12274, 12210, 12273, 12209, 12272, 12208, 12271, 12207, 12270, 12206, 12269, 12205, 12268, 12204, 12267, 12203, 12266, 12202, 12265, 12201, 12264, 12200, 12263, 12199, 12262, 12198, 12261, 12197, 12260, 12196, 12259, 12195, 12258, 12194, 12257, 12193, 12256, 12192, 12255, 12191, 12254, 12190, 12253, 12189, 12252, 12188, 12251, 12187, 12250, 12186, 12249, 12185, 12248, 12184, 12247, 12183, 12246, 12182, 12245, 12181, 12244, 12180, 12243, 12179, 12242, 12178, 12241, 12177, 12240, 12176, 12239, 12175, 12238, 12174, 12237, 12173, 12236, 12172, 12235, 12171, 12234, 12170, 12233, 12169, 12232, 12168, 12231, 12167, 12230, 12166, 12229, 12165, 12228, 12164, 12227, 12163, 12226, 12162, 12225, 12161, 12224, 12160, 12159, 12095, 12158, 12094, 12157, 12093, 12156, 12092, 12155, 12091, 12154, 12090, 12153, 12089, 12152, 12088, 12151, 12087, 12150, 12086, 12149, 12085, 12148, 12084, 12147, 12083, 12146, 12082, 12145, 12081, 12144, 12080, 12143, 12079, 12142, 12078, 12141, 12077, 12140, 12076, 12139, 12075, 12138, 12074, 12137, 12073, 12136, 12072, 12135, 12071, 12134, 12070, 12133, 12069, 12132, 12068, 12131, 12067, 12130, 12066, 12129, 12065, 12128, 12064, 12127, 12063, 12126, 12062, 12125, 12061, 12124, 12060, 12123, 12059, 12122, 12058, 12121, 12057, 12120, 12056, 12119, 12055, 12118, 12054, 12117, 12053, 12116, 12052, 12115, 12051, 12114, 12050, 12113, 12049, 12112, 12048, 12111, 12047, 12110, 12046, 12109, 12045, 12108, 12044, 12107, 12043, 12106, 12042, 12105, 12041, 12104, 12040, 12103, 12039, 12102, 12038, 12101, 12037, 12100, 12036, 12099, 12035, 12098, 12034, 12097, 12033, 12096, 12032, 12031, 11967, 12030, 11966, 12029, 11965, 12028, 11964, 12027, 11963, 12026, 11962, 12025, 11961, 12024, 11960, 12023, 11959, 12022, 11958, 12021, 11957, 12020, 11956, 12019, 11955, 12018, 11954, 12017, 11953, 12016, 11952, 12015, 11951, 12014, 11950, 12013, 11949, 12012, 11948, 12011, 11947, 12010, 11946, 12009, 11945, 12008, 11944, 12007, 11943, 12006, 11942, 12005, 11941, 12004, 11940, 12003, 11939, 12002, 11938, 12001, 11937, 12000, 11936, 11999, 11935, 11998, 11934, 11997, 11933, 11996, 11932, 11995, 11931, 11994, 11930, 11993, 11929, 11992, 11928, 11991, 11927, 11990, 11926, 11989, 11925, 11988, 11924, 11987, 11923, 11986, 11922, 11985, 11921, 11984, 11920, 11983, 11919, 11982, 11918, 11981, 11917, 11980, 11916, 11979, 11915, 11978, 11914, 11977, 11913, 11976, 11912, 11975, 11911, 11974, 11910, 11973, 11909, 11972, 11908, 11971, 11907, 11970, 11906, 11969, 11905, 11968, 11904, 11903, 11839, 11902, 11838, 11901, 11837, 11900, 11836, 11899, 11835, 11898, 11834, 11897, 11833, 11896, 11832, 11895, 11831, 11894, 11830, 11893, 11829, 11892, 11828, 11891, 11827, 11890, 11826, 11889, 11825, 11888, 11824, 11887, 11823, 11886, 11822, 11885, 11821, 11884, 11820, 11883, 11819, 11882, 11818, 11881, 11817, 11880, 11816, 11879, 11815, 11878, 11814, 11877, 11813, 11876, 11812, 11875, 11811, 11874, 11810, 11873, 11809, 11872, 11808, 11871, 11807, 11870, 11806, 11869, 11805, 11868, 11804, 11867, 11803, 11866, 11802, 11865, 11801, 11864, 11800, 11863, 11799, 11862, 11798, 11861, 11797, 11860, 11796, 11859, 11795, 11858, 11794, 11857, 11793, 11856, 11792, 11855, 11791, 11854, 11790, 11853, 11789, 11852, 11788, 11851, 11787, 11850, 11786, 11849, 11785, 11848, 11784, 11847, 11783, 11846, 11782, 11845, 11781, 11844, 11780, 11843, 11779, 11842, 11778, 11841, 11777, 11840, 11776, 11775, 11711, 11774, 11710, 11773, 11709, 11772, 11708, 11771, 11707, 11770, 11706, 11769, 11705, 11768, 11704, 11767, 11703, 11766, 11702, 11765, 11701, 11764, 11700, 11763, 11699, 11762, 11698, 11761, 11697, 11760, 11696, 11759, 11695, 11758, 11694, 11757, 11693, 11756, 11692, 11755, 11691, 11754, 11690, 11753, 11689, 11752, 11688, 11751, 11687, 11750, 11686, 11749, 11685, 11748, 11684, 11747, 11683, 11746, 11682, 11745, 11681, 11744, 11680, 11743, 11679, 11742, 11678, 11741, 11677, 11740, 11676, 11739, 11675, 11738, 11674, 11737, 11673, 11736, 11672, 11735, 11671, 11734, 11670, 11733, 11669, 11732, 11668, 11731, 11667, 11730, 11666, 11729, 11665, 11728, 11664, 11727, 11663, 11726, 11662, 11725, 11661, 11724, 11660, 11723, 11659, 11722, 11658, 11721, 11657, 11720, 11656, 11719, 11655, 11718, 11654, 11717, 11653, 11716, 11652, 11715, 11651, 11714, 11650, 11713, 11649, 11712, 11648, 11647, 11583, 11646, 11582, 11645, 11581, 11644, 11580, 11643, 11579, 11642, 11578, 11641, 11577, 11640, 11576, 11639, 11575, 11638, 11574, 11637, 11573, 11636, 11572, 11635, 11571, 11634, 11570, 11633, 11569, 11632, 11568, 11631, 11567, 11630, 11566, 11629, 11565, 11628, 11564, 11627, 11563, 11626, 11562, 11625, 11561, 11624, 11560, 11623, 11559, 11622, 11558, 11621, 11557, 11620, 11556, 11619, 11555, 11618, 11554, 11617, 11553, 11616, 11552, 11615, 11551, 11614, 11550, 11613, 11549, 11612, 11548, 11611, 11547, 11610, 11546, 11609, 11545, 11608, 11544, 11607, 11543, 11606, 11542, 11605, 11541, 11604, 11540, 11603, 11539, 11602, 11538, 11601, 11537, 11600, 11536, 11599, 11535, 11598, 11534, 11597, 11533, 11596, 11532, 11595, 11531, 11594, 11530, 11593, 11529, 11592, 11528, 11591, 11527, 11590, 11526, 11589, 11525, 11588, 11524, 11587, 11523, 11586, 11522, 11585, 11521, 11584, 11520, 11519, 11455, 11518, 11454, 11517, 11453, 11516, 11452, 11515, 11451, 11514, 11450, 11513, 11449, 11512, 11448, 11511, 11447, 11510, 11446, 11509, 11445, 11508, 11444, 11507, 11443, 11506, 11442, 11505, 11441, 11504, 11440, 11503, 11439, 11502, 11438, 11501, 11437, 11500, 11436, 11499, 11435, 11498, 11434, 11497, 11433, 11496, 11432, 11495, 11431, 11494, 11430, 11493, 11429, 11492, 11428, 11491, 11427, 11490, 11426, 11489, 11425, 11488, 11424, 11487, 11423, 11486, 11422, 11485, 11421, 11484, 11420, 11483, 11419, 11482, 11418, 11481, 11417, 11480, 11416, 11479, 11415, 11478, 11414, 11477, 11413, 11476, 11412, 11475, 11411, 11474, 11410, 11473, 11409, 11472, 11408, 11471, 11407, 11470, 11406, 11469, 11405, 11468, 11404, 11467, 11403, 11466, 11402, 11465, 11401, 11464, 11400, 11463, 11399, 11462, 11398, 11461, 11397, 11460, 11396, 11459, 11395, 11458, 11394, 11457, 11393, 11456, 11392, 11391, 11327, 11390, 11326, 11389, 11325, 11388, 11324, 11387, 11323, 11386, 11322, 11385, 11321, 11384, 11320, 11383, 11319, 11382, 11318, 11381, 11317, 11380, 11316, 11379, 11315, 11378, 11314, 11377, 11313, 11376, 11312, 11375, 11311, 11374, 11310, 11373, 11309, 11372, 11308, 11371, 11307, 11370, 11306, 11369, 11305, 11368, 11304, 11367, 11303, 11366, 11302, 11365, 11301, 11364, 11300, 11363, 11299, 11362, 11298, 11361, 11297, 11360, 11296, 11359, 11295, 11358, 11294, 11357, 11293, 11356, 11292, 11355, 11291, 11354, 11290, 11353, 11289, 11352, 11288, 11351, 11287, 11350, 11286, 11349, 11285, 11348, 11284, 11347, 11283, 11346, 11282, 11345, 11281, 11344, 11280, 11343, 11279, 11342, 11278, 11341, 11277, 11340, 11276, 11339, 11275, 11338, 11274, 11337, 11273, 11336, 11272, 11335, 11271, 11334, 11270, 11333, 11269, 11332, 11268, 11331, 11267, 11330, 11266, 11329, 11265, 11328, 11264, 11263, 11199, 11262, 11198, 11261, 11197, 11260, 11196, 11259, 11195, 11258, 11194, 11257, 11193, 11256, 11192, 11255, 11191, 11254, 11190, 11253, 11189, 11252, 11188, 11251, 11187, 11250, 11186, 11249, 11185, 11248, 11184, 11247, 11183, 11246, 11182, 11245, 11181, 11244, 11180, 11243, 11179, 11242, 11178, 11241, 11177, 11240, 11176, 11239, 11175, 11238, 11174, 11237, 11173, 11236, 11172, 11235, 11171, 11234, 11170, 11233, 11169, 11232, 11168, 11231, 11167, 11230, 11166, 11229, 11165, 11228, 11164, 11227, 11163, 11226, 11162, 11225, 11161, 11224, 11160, 11223, 11159, 11222, 11158, 11221, 11157, 11220, 11156, 11219, 11155, 11218, 11154, 11217, 11153, 11216, 11152, 11215, 11151, 11214, 11150, 11213, 11149, 11212, 11148, 11211, 11147, 11210, 11146, 11209, 11145, 11208, 11144, 11207, 11143, 11206, 11142, 11205, 11141, 11204, 11140, 11203, 11139, 11202, 11138, 11201, 11137, 11200, 11136, 11135, 11071, 11134, 11070, 11133, 11069, 11132, 11068, 11131, 11067, 11130, 11066, 11129, 11065, 11128, 11064, 11127, 11063, 11126, 11062, 11125, 11061, 11124, 11060, 11123, 11059, 11122, 11058, 11121, 11057, 11120, 11056, 11119, 11055, 11118, 11054, 11117, 11053, 11116, 11052, 11115, 11051, 11114, 11050, 11113, 11049, 11112, 11048, 11111, 11047, 11110, 11046, 11109, 11045, 11108, 11044, 11107, 11043, 11106, 11042, 11105, 11041, 11104, 11040, 11103, 11039, 11102, 11038, 11101, 11037, 11100, 11036, 11099, 11035, 11098, 11034, 11097, 11033, 11096, 11032, 11095, 11031, 11094, 11030, 11093, 11029, 11092, 11028, 11091, 11027, 11090, 11026, 11089, 11025, 11088, 11024, 11087, 11023, 11086, 11022, 11085, 11021, 11084, 11020, 11083, 11019, 11082, 11018, 11081, 11017, 11080, 11016, 11079, 11015, 11078, 11014, 11077, 11013, 11076, 11012, 11075, 11011, 11074, 11010, 11073, 11009, 11072, 11008, 11007, 10943, 11006, 10942, 11005, 10941, 11004, 10940, 11003, 10939, 11002, 10938, 11001, 10937, 11000, 10936, 10999, 10935, 10998, 10934, 10997, 10933, 10996, 10932, 10995, 10931, 10994, 10930, 10993, 10929, 10992, 10928, 10991, 10927, 10990, 10926, 10989, 10925, 10988, 10924, 10987, 10923, 10986, 10922, 10985, 10921, 10984, 10920, 10983, 10919, 10982, 10918, 10981, 10917, 10980, 10916, 10979, 10915, 10978, 10914, 10977, 10913, 10976, 10912, 10975, 10911, 10974, 10910, 10973, 10909, 10972, 10908, 10971, 10907, 10970, 10906, 10969, 10905, 10968, 10904, 10967, 10903, 10966, 10902, 10965, 10901, 10964, 10900, 10963, 10899, 10962, 10898, 10961, 10897, 10960, 10896, 10959, 10895, 10958, 10894, 10957, 10893, 10956, 10892, 10955, 10891, 10954, 10890, 10953, 10889, 10952, 10888, 10951, 10887, 10950, 10886, 10949, 10885, 10948, 10884, 10947, 10883, 10946, 10882, 10945, 10881, 10944, 10880, 10879, 10815, 10878, 10814, 10877, 10813, 10876, 10812, 10875, 10811, 10874, 10810, 10873, 10809, 10872, 10808, 10871, 10807, 10870, 10806, 10869, 10805, 10868, 10804, 10867, 10803, 10866, 10802, 10865, 10801, 10864, 10800, 10863, 10799, 10862, 10798, 10861, 10797, 10860, 10796, 10859, 10795, 10858, 10794, 10857, 10793, 10856, 10792, 10855, 10791, 10854, 10790, 10853, 10789, 10852, 10788, 10851, 10787, 10850, 10786, 10849, 10785, 10848, 10784, 10847, 10783, 10846, 10782, 10845, 10781, 10844, 10780, 10843, 10779, 10842, 10778, 10841, 10777, 10840, 10776, 10839, 10775, 10838, 10774, 10837, 10773, 10836, 10772, 10835, 10771, 10834, 10770, 10833, 10769, 10832, 10768, 10831, 10767, 10830, 10766, 10829, 10765, 10828, 10764, 10827, 10763, 10826, 10762, 10825, 10761, 10824, 10760, 10823, 10759, 10822, 10758, 10821, 10757, 10820, 10756, 10819, 10755, 10818, 10754, 10817, 10753, 10816, 10752, 10751, 10687, 10750, 10686, 10749, 10685, 10748, 10684, 10747, 10683, 10746, 10682, 10745, 10681, 10744, 10680, 10743, 10679, 10742, 10678, 10741, 10677, 10740, 10676, 10739, 10675, 10738, 10674, 10737, 10673, 10736, 10672, 10735, 10671, 10734, 10670, 10733, 10669, 10732, 10668, 10731, 10667, 10730, 10666, 10729, 10665, 10728, 10664, 10727, 10663, 10726, 10662, 10725, 10661, 10724, 10660, 10723, 10659, 10722, 10658, 10721, 10657, 10720, 10656, 10719, 10655, 10718, 10654, 10717, 10653, 10716, 10652, 10715, 10651, 10714, 10650, 10713, 10649, 10712, 10648, 10711, 10647, 10710, 10646, 10709, 10645, 10708, 10644, 10707, 10643, 10706, 10642, 10705, 10641, 10704, 10640, 10703, 10639, 10702, 10638, 10701, 10637, 10700, 10636, 10699, 10635, 10698, 10634, 10697, 10633, 10696, 10632, 10695, 10631, 10694, 10630, 10693, 10629, 10692, 10628, 10691, 10627, 10690, 10626, 10689, 10625, 10688, 10624, 10623, 10559, 10622, 10558, 10621, 10557, 10620, 10556, 10619, 10555, 10618, 10554, 10617, 10553, 10616, 10552, 10615, 10551, 10614, 10550, 10613, 10549, 10612, 10548, 10611, 10547, 10610, 10546, 10609, 10545, 10608, 10544, 10607, 10543, 10606, 10542, 10605, 10541, 10604, 10540, 10603, 10539, 10602, 10538, 10601, 10537, 10600, 10536, 10599, 10535, 10598, 10534, 10597, 10533, 10596, 10532, 10595, 10531, 10594, 10530, 10593, 10529, 10592, 10528, 10591, 10527, 10590, 10526, 10589, 10525, 10588, 10524, 10587, 10523, 10586, 10522, 10585, 10521, 10584, 10520, 10583, 10519, 10582, 10518, 10581, 10517, 10580, 10516, 10579, 10515, 10578, 10514, 10577, 10513, 10576, 10512, 10575, 10511, 10574, 10510, 10573, 10509, 10572, 10508, 10571, 10507, 10570, 10506, 10569, 10505, 10568, 10504, 10567, 10503, 10566, 10502, 10565, 10501, 10564, 10500, 10563, 10499, 10562, 10498, 10561, 10497, 10560, 10496, 10495, 10431, 10494, 10430, 10493, 10429, 10492, 10428, 10491, 10427, 10490, 10426, 10489, 10425, 10488, 10424, 10487, 10423, 10486, 10422, 10485, 10421, 10484, 10420, 10483, 10419, 10482, 10418, 10481, 10417, 10480, 10416, 10479, 10415, 10478, 10414, 10477, 10413, 10476, 10412, 10475, 10411, 10474, 10410, 10473, 10409, 10472, 10408, 10471, 10407, 10470, 10406, 10469, 10405, 10468, 10404, 10467, 10403, 10466, 10402, 10465, 10401, 10464, 10400, 10463, 10399, 10462, 10398, 10461, 10397, 10460, 10396, 10459, 10395, 10458, 10394, 10457, 10393, 10456, 10392, 10455, 10391, 10454, 10390, 10453, 10389, 10452, 10388, 10451, 10387, 10450, 10386, 10449, 10385, 10448, 10384, 10447, 10383, 10446, 10382, 10445, 10381, 10444, 10380, 10443, 10379, 10442, 10378, 10441, 10377, 10440, 10376, 10439, 10375, 10438, 10374, 10437, 10373, 10436, 10372, 10435, 10371, 10434, 10370, 10433, 10369, 10432, 10368, 10367, 10303, 10366, 10302, 10365, 10301, 10364, 10300, 10363, 10299, 10362, 10298, 10361, 10297, 10360, 10296, 10359, 10295, 10358, 10294, 10357, 10293, 10356, 10292, 10355, 10291, 10354, 10290, 10353, 10289, 10352, 10288, 10351, 10287, 10350, 10286, 10349, 10285, 10348, 10284, 10347, 10283, 10346, 10282, 10345, 10281, 10344, 10280, 10343, 10279, 10342, 10278, 10341, 10277, 10340, 10276, 10339, 10275, 10338, 10274, 10337, 10273, 10336, 10272, 10335, 10271, 10334, 10270, 10333, 10269, 10332, 10268, 10331, 10267, 10330, 10266, 10329, 10265, 10328, 10264, 10327, 10263, 10326, 10262, 10325, 10261, 10324, 10260, 10323, 10259, 10322, 10258, 10321, 10257, 10320, 10256, 10319, 10255, 10318, 10254, 10317, 10253, 10316, 10252, 10315, 10251, 10314, 10250, 10313, 10249, 10312, 10248, 10311, 10247, 10310, 10246, 10309, 10245, 10308, 10244, 10307, 10243, 10306, 10242, 10305, 10241, 10304, 10240, 10239, 10175, 10238, 10174, 10237, 10173, 10236, 10172, 10235, 10171, 10234, 10170, 10233, 10169, 10232, 10168, 10231, 10167, 10230, 10166, 10229, 10165, 10228, 10164, 10227, 10163, 10226, 10162, 10225, 10161, 10224, 10160, 10223, 10159, 10222, 10158, 10221, 10157, 10220, 10156, 10219, 10155, 10218, 10154, 10217, 10153, 10216, 10152, 10215, 10151, 10214, 10150, 10213, 10149, 10212, 10148, 10211, 10147, 10210, 10146, 10209, 10145, 10208, 10144, 10207, 10143, 10206, 10142, 10205, 10141, 10204, 10140, 10203, 10139, 10202, 10138, 10201, 10137, 10200, 10136, 10199, 10135, 10198, 10134, 10197, 10133, 10196, 10132, 10195, 10131, 10194, 10130, 10193, 10129, 10192, 10128, 10191, 10127, 10190, 10126, 10189, 10125, 10188, 10124, 10187, 10123, 10186, 10122, 10185, 10121, 10184, 10120, 10183, 10119, 10182, 10118, 10181, 10117, 10180, 10116, 10179, 10115, 10178, 10114, 10177, 10113, 10176, 10112, 10111, 10047, 10110, 10046, 10109, 10045, 10108, 10044, 10107, 10043, 10106, 10042, 10105, 10041, 10104, 10040, 10103, 10039, 10102, 10038, 10101, 10037, 10100, 10036, 10099, 10035, 10098, 10034, 10097, 10033, 10096, 10032, 10095, 10031, 10094, 10030, 10093, 10029, 10092, 10028, 10091, 10027, 10090, 10026, 10089, 10025, 10088, 10024, 10087, 10023, 10086, 10022, 10085, 10021, 10084, 10020, 10083, 10019, 10082, 10018, 10081, 10017, 10080, 10016, 10079, 10015, 10078, 10014, 10077, 10013, 10076, 10012, 10075, 10011, 10074, 10010, 10073, 10009, 10072, 10008, 10071, 10007, 10070, 10006, 10069, 10005, 10068, 10004, 10067, 10003, 10066, 10002, 10065, 10001, 10064, 10000, 10063, 9999, 10062, 9998, 10061, 9997, 10060, 9996, 10059, 9995, 10058, 9994, 10057, 9993, 10056, 9992, 10055, 9991, 10054, 9990, 10053, 9989, 10052, 9988, 10051, 9987, 10050, 9986, 10049, 9985, 10048, 9984, 9983, 9919, 9982, 9918, 9981, 9917, 9980, 9916, 9979, 9915, 9978, 9914, 9977, 9913, 9976, 9912, 9975, 9911, 9974, 9910, 9973, 9909, 9972, 9908, 9971, 9907, 9970, 9906, 9969, 9905, 9968, 9904, 9967, 9903, 9966, 9902, 9965, 9901, 9964, 9900, 9963, 9899, 9962, 9898, 9961, 9897, 9960, 9896, 9959, 9895, 9958, 9894, 9957, 9893, 9956, 9892, 9955, 9891, 9954, 9890, 9953, 9889, 9952, 9888, 9951, 9887, 9950, 9886, 9949, 9885, 9948, 9884, 9947, 9883, 9946, 9882, 9945, 9881, 9944, 9880, 9943, 9879, 9942, 9878, 9941, 9877, 9940, 9876, 9939, 9875, 9938, 9874, 9937, 9873, 9936, 9872, 9935, 9871, 9934, 9870, 9933, 9869, 9932, 9868, 9931, 9867, 9930, 9866, 9929, 9865, 9928, 9864, 9927, 9863, 9926, 9862, 9925, 9861, 9924, 9860, 9923, 9859, 9922, 9858, 9921, 9857, 9920, 9856, 9855, 9791, 9854, 9790, 9853, 9789, 9852, 9788, 9851, 9787, 9850, 9786, 9849, 9785, 9848, 9784, 9847, 9783, 9846, 9782, 9845, 9781, 9844, 9780, 9843, 9779, 9842, 9778, 9841, 9777, 9840, 9776, 9839, 9775, 9838, 9774, 9837, 9773, 9836, 9772, 9835, 9771, 9834, 9770, 9833, 9769, 9832, 9768, 9831, 9767, 9830, 9766, 9829, 9765, 9828, 9764, 9827, 9763, 9826, 9762, 9825, 9761, 9824, 9760, 9823, 9759, 9822, 9758, 9821, 9757, 9820, 9756, 9819, 9755, 9818, 9754, 9817, 9753, 9816, 9752, 9815, 9751, 9814, 9750, 9813, 9749, 9812, 9748, 9811, 9747, 9810, 9746, 9809, 9745, 9808, 9744, 9807, 9743, 9806, 9742, 9805, 9741, 9804, 9740, 9803, 9739, 9802, 9738, 9801, 9737, 9800, 9736, 9799, 9735, 9798, 9734, 9797, 9733, 9796, 9732, 9795, 9731, 9794, 9730, 9793, 9729, 9792, 9728, 9727, 9663, 9726, 9662, 9725, 9661, 9724, 9660, 9723, 9659, 9722, 9658, 9721, 9657, 9720, 9656, 9719, 9655, 9718, 9654, 9717, 9653, 9716, 9652, 9715, 9651, 9714, 9650, 9713, 9649, 9712, 9648, 9711, 9647, 9710, 9646, 9709, 9645, 9708, 9644, 9707, 9643, 9706, 9642, 9705, 9641, 9704, 9640, 9703, 9639, 9702, 9638, 9701, 9637, 9700, 9636, 9699, 9635, 9698, 9634, 9697, 9633, 9696, 9632, 9695, 9631, 9694, 9630, 9693, 9629, 9692, 9628, 9691, 9627, 9690, 9626, 9689, 9625, 9688, 9624, 9687, 9623, 9686, 9622, 9685, 9621, 9684, 9620, 9683, 9619, 9682, 9618, 9681, 9617, 9680, 9616, 9679, 9615, 9678, 9614, 9677, 9613, 9676, 9612, 9675, 9611, 9674, 9610, 9673, 9609, 9672, 9608, 9671, 9607, 9670, 9606, 9669, 9605, 9668, 9604, 9667, 9603, 9666, 9602, 9665, 9601, 9664, 9600, 9599, 9535, 9598, 9534, 9597, 9533, 9596, 9532, 9595, 9531, 9594, 9530, 9593, 9529, 9592, 9528, 9591, 9527, 9590, 9526, 9589, 9525, 9588, 9524, 9587, 9523, 9586, 9522, 9585, 9521, 9584, 9520, 9583, 9519, 9582, 9518, 9581, 9517, 9580, 9516, 9579, 9515, 9578, 9514, 9577, 9513, 9576, 9512, 9575, 9511, 9574, 9510, 9573, 9509, 9572, 9508, 9571, 9507, 9570, 9506, 9569, 9505, 9568, 9504, 9567, 9503, 9566, 9502, 9565, 9501, 9564, 9500, 9563, 9499, 9562, 9498, 9561, 9497, 9560, 9496, 9559, 9495, 9558, 9494, 9557, 9493, 9556, 9492, 9555, 9491, 9554, 9490, 9553, 9489, 9552, 9488, 9551, 9487, 9550, 9486, 9549, 9485, 9548, 9484, 9547, 9483, 9546, 9482, 9545, 9481, 9544, 9480, 9543, 9479, 9542, 9478, 9541, 9477, 9540, 9476, 9539, 9475, 9538, 9474, 9537, 9473, 9536, 9472, 9471, 9407, 9470, 9406, 9469, 9405, 9468, 9404, 9467, 9403, 9466, 9402, 9465, 9401, 9464, 9400, 9463, 9399, 9462, 9398, 9461, 9397, 9460, 9396, 9459, 9395, 9458, 9394, 9457, 9393, 9456, 9392, 9455, 9391, 9454, 9390, 9453, 9389, 9452, 9388, 9451, 9387, 9450, 9386, 9449, 9385, 9448, 9384, 9447, 9383, 9446, 9382, 9445, 9381, 9444, 9380, 9443, 9379, 9442, 9378, 9441, 9377, 9440, 9376, 9439, 9375, 9438, 9374, 9437, 9373, 9436, 9372, 9435, 9371, 9434, 9370, 9433, 9369, 9432, 9368, 9431, 9367, 9430, 9366, 9429, 9365, 9428, 9364, 9427, 9363, 9426, 9362, 9425, 9361, 9424, 9360, 9423, 9359, 9422, 9358, 9421, 9357, 9420, 9356, 9419, 9355, 9418, 9354, 9417, 9353, 9416, 9352, 9415, 9351, 9414, 9350, 9413, 9349, 9412, 9348, 9411, 9347, 9410, 9346, 9409, 9345, 9408, 9344, 9343, 9279, 9342, 9278, 9341, 9277, 9340, 9276, 9339, 9275, 9338, 9274, 9337, 9273, 9336, 9272, 9335, 9271, 9334, 9270, 9333, 9269, 9332, 9268, 9331, 9267, 9330, 9266, 9329, 9265, 9328, 9264, 9327, 9263, 9326, 9262, 9325, 9261, 9324, 9260, 9323, 9259, 9322, 9258, 9321, 9257, 9320, 9256, 9319, 9255, 9318, 9254, 9317, 9253, 9316, 9252, 9315, 9251, 9314, 9250, 9313, 9249, 9312, 9248, 9311, 9247, 9310, 9246, 9309, 9245, 9308, 9244, 9307, 9243, 9306, 9242, 9305, 9241, 9304, 9240, 9303, 9239, 9302, 9238, 9301, 9237, 9300, 9236, 9299, 9235, 9298, 9234, 9297, 9233, 9296, 9232, 9295, 9231, 9294, 9230, 9293, 9229, 9292, 9228, 9291, 9227, 9290, 9226, 9289, 9225, 9288, 9224, 9287, 9223, 9286, 9222, 9285, 9221, 9284, 9220, 9283, 9219, 9282, 9218, 9281, 9217, 9280, 9216, 9215, 9151, 9214, 9150, 9213, 9149, 9212, 9148, 9211, 9147, 9210, 9146, 9209, 9145, 9208, 9144, 9207, 9143, 9206, 9142, 9205, 9141, 9204, 9140, 9203, 9139, 9202, 9138, 9201, 9137, 9200, 9136, 9199, 9135, 9198, 9134, 9197, 9133, 9196, 9132, 9195, 9131, 9194, 9130, 9193, 9129, 9192, 9128, 9191, 9127, 9190, 9126, 9189, 9125, 9188, 9124, 9187, 9123, 9186, 9122, 9185, 9121, 9184, 9120, 9183, 9119, 9182, 9118, 9181, 9117, 9180, 9116, 9179, 9115, 9178, 9114, 9177, 9113, 9176, 9112, 9175, 9111, 9174, 9110, 9173, 9109, 9172, 9108, 9171, 9107, 9170, 9106, 9169, 9105, 9168, 9104, 9167, 9103, 9166, 9102, 9165, 9101, 9164, 9100, 9163, 9099, 9162, 9098, 9161, 9097, 9160, 9096, 9159, 9095, 9158, 9094, 9157, 9093, 9156, 9092, 9155, 9091, 9154, 9090, 9153, 9089, 9152, 9088, 9087, 9023, 9086, 9022, 9085, 9021, 9084, 9020, 9083, 9019, 9082, 9018, 9081, 9017, 9080, 9016, 9079, 9015, 9078, 9014, 9077, 9013, 9076, 9012, 9075, 9011, 9074, 9010, 9073, 9009, 9072, 9008, 9071, 9007, 9070, 9006, 9069, 9005, 9068, 9004, 9067, 9003, 9066, 9002, 9065, 9001, 9064, 9000, 9063, 8999, 9062, 8998, 9061, 8997, 9060, 8996, 9059, 8995, 9058, 8994, 9057, 8993, 9056, 8992, 9055, 8991, 9054, 8990, 9053, 8989, 9052, 8988, 9051, 8987, 9050, 8986, 9049, 8985, 9048, 8984, 9047, 8983, 9046, 8982, 9045, 8981, 9044, 8980, 9043, 8979, 9042, 8978, 9041, 8977, 9040, 8976, 9039, 8975, 9038, 8974, 9037, 8973, 9036, 8972, 9035, 8971, 9034, 8970, 9033, 8969, 9032, 8968, 9031, 8967, 9030, 8966, 9029, 8965, 9028, 8964, 9027, 8963, 9026, 8962, 9025, 8961, 9024, 8960, 8959, 8895, 8958, 8894, 8957, 8893, 8956, 8892, 8955, 8891, 8954, 8890, 8953, 8889, 8952, 8888, 8951, 8887, 8950, 8886, 8949, 8885, 8948, 8884, 8947, 8883, 8946, 8882, 8945, 8881, 8944, 8880, 8943, 8879, 8942, 8878, 8941, 8877, 8940, 8876, 8939, 8875, 8938, 8874, 8937, 8873, 8936, 8872, 8935, 8871, 8934, 8870, 8933, 8869, 8932, 8868, 8931, 8867, 8930, 8866, 8929, 8865, 8928, 8864, 8927, 8863, 8926, 8862, 8925, 8861, 8924, 8860, 8923, 8859, 8922, 8858, 8921, 8857, 8920, 8856, 8919, 8855, 8918, 8854, 8917, 8853, 8916, 8852, 8915, 8851, 8914, 8850, 8913, 8849, 8912, 8848, 8911, 8847, 8910, 8846, 8909, 8845, 8908, 8844, 8907, 8843, 8906, 8842, 8905, 8841, 8904, 8840, 8903, 8839, 8902, 8838, 8901, 8837, 8900, 8836, 8899, 8835, 8898, 8834, 8897, 8833, 8896, 8832, 8831, 8767, 8830, 8766, 8829, 8765, 8828, 8764, 8827, 8763, 8826, 8762, 8825, 8761, 8824, 8760, 8823, 8759, 8822, 8758, 8821, 8757, 8820, 8756, 8819, 8755, 8818, 8754, 8817, 8753, 8816, 8752, 8815, 8751, 8814, 8750, 8813, 8749, 8812, 8748, 8811, 8747, 8810, 8746, 8809, 8745, 8808, 8744, 8807, 8743, 8806, 8742, 8805, 8741, 8804, 8740, 8803, 8739, 8802, 8738, 8801, 8737, 8800, 8736, 8799, 8735, 8798, 8734, 8797, 8733, 8796, 8732, 8795, 8731, 8794, 8730, 8793, 8729, 8792, 8728, 8791, 8727, 8790, 8726, 8789, 8725, 8788, 8724, 8787, 8723, 8786, 8722, 8785, 8721, 8784, 8720, 8783, 8719, 8782, 8718, 8781, 8717, 8780, 8716, 8779, 8715, 8778, 8714, 8777, 8713, 8776, 8712, 8775, 8711, 8774, 8710, 8773, 8709, 8772, 8708, 8771, 8707, 8770, 8706, 8769, 8705, 8768, 8704, 8703, 8639, 8702, 8638, 8701, 8637, 8700, 8636, 8699, 8635, 8698, 8634, 8697, 8633, 8696, 8632, 8695, 8631, 8694, 8630, 8693, 8629, 8692, 8628, 8691, 8627, 8690, 8626, 8689, 8625, 8688, 8624, 8687, 8623, 8686, 8622, 8685, 8621, 8684, 8620, 8683, 8619, 8682, 8618, 8681, 8617, 8680, 8616, 8679, 8615, 8678, 8614, 8677, 8613, 8676, 8612, 8675, 8611, 8674, 8610, 8673, 8609, 8672, 8608, 8671, 8607, 8670, 8606, 8669, 8605, 8668, 8604, 8667, 8603, 8666, 8602, 8665, 8601, 8664, 8600, 8663, 8599, 8662, 8598, 8661, 8597, 8660, 8596, 8659, 8595, 8658, 8594, 8657, 8593, 8656, 8592, 8655, 8591, 8654, 8590, 8653, 8589, 8652, 8588, 8651, 8587, 8650, 8586, 8649, 8585, 8648, 8584, 8647, 8583, 8646, 8582, 8645, 8581, 8644, 8580, 8643, 8579, 8642, 8578, 8641, 8577, 8640, 8576, 8575, 8511, 8574, 8510, 8573, 8509, 8572, 8508, 8571, 8507, 8570, 8506, 8569, 8505, 8568, 8504, 8567, 8503, 8566, 8502, 8565, 8501, 8564, 8500, 8563, 8499, 8562, 8498, 8561, 8497, 8560, 8496, 8559, 8495, 8558, 8494, 8557, 8493, 8556, 8492, 8555, 8491, 8554, 8490, 8553, 8489, 8552, 8488, 8551, 8487, 8550, 8486, 8549, 8485, 8548, 8484, 8547, 8483, 8546, 8482, 8545, 8481, 8544, 8480, 8543, 8479, 8542, 8478, 8541, 8477, 8540, 8476, 8539, 8475, 8538, 8474, 8537, 8473, 8536, 8472, 8535, 8471, 8534, 8470, 8533, 8469, 8532, 8468, 8531, 8467, 8530, 8466, 8529, 8465, 8528, 8464, 8527, 8463, 8526, 8462, 8525, 8461, 8524, 8460, 8523, 8459, 8522, 8458, 8521, 8457, 8520, 8456, 8519, 8455, 8518, 8454, 8517, 8453, 8516, 8452, 8515, 8451, 8514, 8450, 8513, 8449, 8512, 8448, 8447, 8383, 8446, 8382, 8445, 8381, 8444, 8380, 8443, 8379, 8442, 8378, 8441, 8377, 8440, 8376, 8439, 8375, 8438, 8374, 8437, 8373, 8436, 8372, 8435, 8371, 8434, 8370, 8433, 8369, 8432, 8368, 8431, 8367, 8430, 8366, 8429, 8365, 8428, 8364, 8427, 8363, 8426, 8362, 8425, 8361, 8424, 8360, 8423, 8359, 8422, 8358, 8421, 8357, 8420, 8356, 8419, 8355, 8418, 8354, 8417, 8353, 8416, 8352, 8415, 8351, 8414, 8350, 8413, 8349, 8412, 8348, 8411, 8347, 8410, 8346, 8409, 8345, 8408, 8344, 8407, 8343, 8406, 8342, 8405, 8341, 8404, 8340, 8403, 8339, 8402, 8338, 8401, 8337, 8400, 8336, 8399, 8335, 8398, 8334, 8397, 8333, 8396, 8332, 8395, 8331, 8394, 8330, 8393, 8329, 8392, 8328, 8391, 8327, 8390, 8326, 8389, 8325, 8388, 8324, 8387, 8323, 8386, 8322, 8385, 8321, 8384, 8320, 8319, 8255, 8318, 8254, 8317, 8253, 8316, 8252, 8315, 8251, 8314, 8250, 8313, 8249, 8312, 8248, 8311, 8247, 8310, 8246, 8309, 8245, 8308, 8244, 8307, 8243, 8306, 8242, 8305, 8241, 8304, 8240, 8303, 8239, 8302, 8238, 8301, 8237, 8300, 8236, 8299, 8235, 8298, 8234, 8297, 8233, 8296, 8232, 8295, 8231, 8294, 8230, 8293, 8229, 8292, 8228, 8291, 8227, 8290, 8226, 8289, 8225, 8288, 8224, 8287, 8223, 8286, 8222, 8285, 8221, 8284, 8220, 8283, 8219, 8282, 8218, 8281, 8217, 8280, 8216, 8279, 8215, 8278, 8214, 8277, 8213, 8276, 8212, 8275, 8211, 8274, 8210, 8273, 8209, 8272, 8208, 8271, 8207, 8270, 8206, 8269, 8205, 8268, 8204, 8267, 8203, 8266, 8202, 8265, 8201, 8264, 8200, 8263, 8199, 8262, 8198, 8261, 8197, 8260, 8196, 8259, 8195, 8258, 8194, 8257, 8193, 8256, 8192, 8191, 8127, 8190, 8126, 8189, 8125, 8188, 8124, 8187, 8123, 8186, 8122, 8185, 8121, 8184, 8120, 8183, 8119, 8182, 8118, 8181, 8117, 8180, 8116, 8179, 8115, 8178, 8114, 8177, 8113, 8176, 8112, 8175, 8111, 8174, 8110, 8173, 8109, 8172, 8108, 8171, 8107, 8170, 8106, 8169, 8105, 8168, 8104, 8167, 8103, 8166, 8102, 8165, 8101, 8164, 8100, 8163, 8099, 8162, 8098, 8161, 8097, 8160, 8096, 8159, 8095, 8158, 8094, 8157, 8093, 8156, 8092, 8155, 8091, 8154, 8090, 8153, 8089, 8152, 8088, 8151, 8087, 8150, 8086, 8149, 8085, 8148, 8084, 8147, 8083, 8146, 8082, 8145, 8081, 8144, 8080, 8143, 8079, 8142, 8078, 8141, 8077, 8140, 8076, 8139, 8075, 8138, 8074, 8137, 8073, 8136, 8072, 8135, 8071, 8134, 8070, 8133, 8069, 8132, 8068, 8131, 8067, 8130, 8066, 8129, 8065, 8128, 8064, 8063, 7999, 8062, 7998, 8061, 7997, 8060, 7996, 8059, 7995, 8058, 7994, 8057, 7993, 8056, 7992, 8055, 7991, 8054, 7990, 8053, 7989, 8052, 7988, 8051, 7987, 8050, 7986, 8049, 7985, 8048, 7984, 8047, 7983, 8046, 7982, 8045, 7981, 8044, 7980, 8043, 7979, 8042, 7978, 8041, 7977, 8040, 7976, 8039, 7975, 8038, 7974, 8037, 7973, 8036, 7972, 8035, 7971, 8034, 7970, 8033, 7969, 8032, 7968, 8031, 7967, 8030, 7966, 8029, 7965, 8028, 7964, 8027, 7963, 8026, 7962, 8025, 7961, 8024, 7960, 8023, 7959, 8022, 7958, 8021, 7957, 8020, 7956, 8019, 7955, 8018, 7954, 8017, 7953, 8016, 7952, 8015, 7951, 8014, 7950, 8013, 7949, 8012, 7948, 8011, 7947, 8010, 7946, 8009, 7945, 8008, 7944, 8007, 7943, 8006, 7942, 8005, 7941, 8004, 7940, 8003, 7939, 8002, 7938, 8001, 7937, 8000, 7936, 7935, 7871, 7934, 7870, 7933, 7869, 7932, 7868, 7931, 7867, 7930, 7866, 7929, 7865, 7928, 7864, 7927, 7863, 7926, 7862, 7925, 7861, 7924, 7860, 7923, 7859, 7922, 7858, 7921, 7857, 7920, 7856, 7919, 7855, 7918, 7854, 7917, 7853, 7916, 7852, 7915, 7851, 7914, 7850, 7913, 7849, 7912, 7848, 7911, 7847, 7910, 7846, 7909, 7845, 7908, 7844, 7907, 7843, 7906, 7842, 7905, 7841, 7904, 7840, 7903, 7839, 7902, 7838, 7901, 7837, 7900, 7836, 7899, 7835, 7898, 7834, 7897, 7833, 7896, 7832, 7895, 7831, 7894, 7830, 7893, 7829, 7892, 7828, 7891, 7827, 7890, 7826, 7889, 7825, 7888, 7824, 7887, 7823, 7886, 7822, 7885, 7821, 7884, 7820, 7883, 7819, 7882, 7818, 7881, 7817, 7880, 7816, 7879, 7815, 7878, 7814, 7877, 7813, 7876, 7812, 7875, 7811, 7874, 7810, 7873, 7809, 7872, 7808, 7807, 7743, 7806, 7742, 7805, 7741, 7804, 7740, 7803, 7739, 7802, 7738, 7801, 7737, 7800, 7736, 7799, 7735, 7798, 7734, 7797, 7733, 7796, 7732, 7795, 7731, 7794, 7730, 7793, 7729, 7792, 7728, 7791, 7727, 7790, 7726, 7789, 7725, 7788, 7724, 7787, 7723, 7786, 7722, 7785, 7721, 7784, 7720, 7783, 7719, 7782, 7718, 7781, 7717, 7780, 7716, 7779, 7715, 7778, 7714, 7777, 7713, 7776, 7712, 7775, 7711, 7774, 7710, 7773, 7709, 7772, 7708, 7771, 7707, 7770, 7706, 7769, 7705, 7768, 7704, 7767, 7703, 7766, 7702, 7765, 7701, 7764, 7700, 7763, 7699, 7762, 7698, 7761, 7697, 7760, 7696, 7759, 7695, 7758, 7694, 7757, 7693, 7756, 7692, 7755, 7691, 7754, 7690, 7753, 7689, 7752, 7688, 7751, 7687, 7750, 7686, 7749, 7685, 7748, 7684, 7747, 7683, 7746, 7682, 7745, 7681, 7744, 7680, 7679, 7615, 7678, 7614, 7677, 7613, 7676, 7612, 7675, 7611, 7674, 7610, 7673, 7609, 7672, 7608, 7671, 7607, 7670, 7606, 7669, 7605, 7668, 7604, 7667, 7603, 7666, 7602, 7665, 7601, 7664, 7600, 7663, 7599, 7662, 7598, 7661, 7597, 7660, 7596, 7659, 7595, 7658, 7594, 7657, 7593, 7656, 7592, 7655, 7591, 7654, 7590, 7653, 7589, 7652, 7588, 7651, 7587, 7650, 7586, 7649, 7585, 7648, 7584, 7647, 7583, 7646, 7582, 7645, 7581, 7644, 7580, 7643, 7579, 7642, 7578, 7641, 7577, 7640, 7576, 7639, 7575, 7638, 7574, 7637, 7573, 7636, 7572, 7635, 7571, 7634, 7570, 7633, 7569, 7632, 7568, 7631, 7567, 7630, 7566, 7629, 7565, 7628, 7564, 7627, 7563, 7626, 7562, 7625, 7561, 7624, 7560, 7623, 7559, 7622, 7558, 7621, 7557, 7620, 7556, 7619, 7555, 7618, 7554, 7617, 7553, 7616, 7552, 7551, 7487, 7550, 7486, 7549, 7485, 7548, 7484, 7547, 7483, 7546, 7482, 7545, 7481, 7544, 7480, 7543, 7479, 7542, 7478, 7541, 7477, 7540, 7476, 7539, 7475, 7538, 7474, 7537, 7473, 7536, 7472, 7535, 7471, 7534, 7470, 7533, 7469, 7532, 7468, 7531, 7467, 7530, 7466, 7529, 7465, 7528, 7464, 7527, 7463, 7526, 7462, 7525, 7461, 7524, 7460, 7523, 7459, 7522, 7458, 7521, 7457, 7520, 7456, 7519, 7455, 7518, 7454, 7517, 7453, 7516, 7452, 7515, 7451, 7514, 7450, 7513, 7449, 7512, 7448, 7511, 7447, 7510, 7446, 7509, 7445, 7508, 7444, 7507, 7443, 7506, 7442, 7505, 7441, 7504, 7440, 7503, 7439, 7502, 7438, 7501, 7437, 7500, 7436, 7499, 7435, 7498, 7434, 7497, 7433, 7496, 7432, 7495, 7431, 7494, 7430, 7493, 7429, 7492, 7428, 7491, 7427, 7490, 7426, 7489, 7425, 7488, 7424, 7423, 7359, 7422, 7358, 7421, 7357, 7420, 7356, 7419, 7355, 7418, 7354, 7417, 7353, 7416, 7352, 7415, 7351, 7414, 7350, 7413, 7349, 7412, 7348, 7411, 7347, 7410, 7346, 7409, 7345, 7408, 7344, 7407, 7343, 7406, 7342, 7405, 7341, 7404, 7340, 7403, 7339, 7402, 7338, 7401, 7337, 7400, 7336, 7399, 7335, 7398, 7334, 7397, 7333, 7396, 7332, 7395, 7331, 7394, 7330, 7393, 7329, 7392, 7328, 7391, 7327, 7390, 7326, 7389, 7325, 7388, 7324, 7387, 7323, 7386, 7322, 7385, 7321, 7384, 7320, 7383, 7319, 7382, 7318, 7381, 7317, 7380, 7316, 7379, 7315, 7378, 7314, 7377, 7313, 7376, 7312, 7375, 7311, 7374, 7310, 7373, 7309, 7372, 7308, 7371, 7307, 7370, 7306, 7369, 7305, 7368, 7304, 7367, 7303, 7366, 7302, 7365, 7301, 7364, 7300, 7363, 7299, 7362, 7298, 7361, 7297, 7360, 7296, 7295, 7231, 7294, 7230, 7293, 7229, 7292, 7228, 7291, 7227, 7290, 7226, 7289, 7225, 7288, 7224, 7287, 7223, 7286, 7222, 7285, 7221, 7284, 7220, 7283, 7219, 7282, 7218, 7281, 7217, 7280, 7216, 7279, 7215, 7278, 7214, 7277, 7213, 7276, 7212, 7275, 7211, 7274, 7210, 7273, 7209, 7272, 7208, 7271, 7207, 7270, 7206, 7269, 7205, 7268, 7204, 7267, 7203, 7266, 7202, 7265, 7201, 7264, 7200, 7263, 7199, 7262, 7198, 7261, 7197, 7260, 7196, 7259, 7195, 7258, 7194, 7257, 7193, 7256, 7192, 7255, 7191, 7254, 7190, 7253, 7189, 7252, 7188, 7251, 7187, 7250, 7186, 7249, 7185, 7248, 7184, 7247, 7183, 7246, 7182, 7245, 7181, 7244, 7180, 7243, 7179, 7242, 7178, 7241, 7177, 7240, 7176, 7239, 7175, 7238, 7174, 7237, 7173, 7236, 7172, 7235, 7171, 7234, 7170, 7233, 7169, 7232, 7168, 7167, 7103, 7166, 7102, 7165, 7101, 7164, 7100, 7163, 7099, 7162, 7098, 7161, 7097, 7160, 7096, 7159, 7095, 7158, 7094, 7157, 7093, 7156, 7092, 7155, 7091, 7154, 7090, 7153, 7089, 7152, 7088, 7151, 7087, 7150, 7086, 7149, 7085, 7148, 7084, 7147, 7083, 7146, 7082, 7145, 7081, 7144, 7080, 7143, 7079, 7142, 7078, 7141, 7077, 7140, 7076, 7139, 7075, 7138, 7074, 7137, 7073, 7136, 7072, 7135, 7071, 7134, 7070, 7133, 7069, 7132, 7068, 7131, 7067, 7130, 7066, 7129, 7065, 7128, 7064, 7127, 7063, 7126, 7062, 7125, 7061, 7124, 7060, 7123, 7059, 7122, 7058, 7121, 7057, 7120, 7056, 7119, 7055, 7118, 7054, 7117, 7053, 7116, 7052, 7115, 7051, 7114, 7050, 7113, 7049, 7112, 7048, 7111, 7047, 7110, 7046, 7109, 7045, 7108, 7044, 7107, 7043, 7106, 7042, 7105, 7041, 7104, 7040, 7039, 6975, 7038, 6974, 7037, 6973, 7036, 6972, 7035, 6971, 7034, 6970, 7033, 6969, 7032, 6968, 7031, 6967, 7030, 6966, 7029, 6965, 7028, 6964, 7027, 6963, 7026, 6962, 7025, 6961, 7024, 6960, 7023, 6959, 7022, 6958, 7021, 6957, 7020, 6956, 7019, 6955, 7018, 6954, 7017, 6953, 7016, 6952, 7015, 6951, 7014, 6950, 7013, 6949, 7012, 6948, 7011, 6947, 7010, 6946, 7009, 6945, 7008, 6944, 7007, 6943, 7006, 6942, 7005, 6941, 7004, 6940, 7003, 6939, 7002, 6938, 7001, 6937, 7000, 6936, 6999, 6935, 6998, 6934, 6997, 6933, 6996, 6932, 6995, 6931, 6994, 6930, 6993, 6929, 6992, 6928, 6991, 6927, 6990, 6926, 6989, 6925, 6988, 6924, 6987, 6923, 6986, 6922, 6985, 6921, 6984, 6920, 6983, 6919, 6982, 6918, 6981, 6917, 6980, 6916, 6979, 6915, 6978, 6914, 6977, 6913, 6976, 6912, 6911, 6847, 6910, 6846, 6909, 6845, 6908, 6844, 6907, 6843, 6906, 6842, 6905, 6841, 6904, 6840, 6903, 6839, 6902, 6838, 6901, 6837, 6900, 6836, 6899, 6835, 6898, 6834, 6897, 6833, 6896, 6832, 6895, 6831, 6894, 6830, 6893, 6829, 6892, 6828, 6891, 6827, 6890, 6826, 6889, 6825, 6888, 6824, 6887, 6823, 6886, 6822, 6885, 6821, 6884, 6820, 6883, 6819, 6882, 6818, 6881, 6817, 6880, 6816, 6879, 6815, 6878, 6814, 6877, 6813, 6876, 6812, 6875, 6811, 6874, 6810, 6873, 6809, 6872, 6808, 6871, 6807, 6870, 6806, 6869, 6805, 6868, 6804, 6867, 6803, 6866, 6802, 6865, 6801, 6864, 6800, 6863, 6799, 6862, 6798, 6861, 6797, 6860, 6796, 6859, 6795, 6858, 6794, 6857, 6793, 6856, 6792, 6855, 6791, 6854, 6790, 6853, 6789, 6852, 6788, 6851, 6787, 6850, 6786, 6849, 6785, 6848, 6784, 6783, 6719, 6782, 6718, 6781, 6717, 6780, 6716, 6779, 6715, 6778, 6714, 6777, 6713, 6776, 6712, 6775, 6711, 6774, 6710, 6773, 6709, 6772, 6708, 6771, 6707, 6770, 6706, 6769, 6705, 6768, 6704, 6767, 6703, 6766, 6702, 6765, 6701, 6764, 6700, 6763, 6699, 6762, 6698, 6761, 6697, 6760, 6696, 6759, 6695, 6758, 6694, 6757, 6693, 6756, 6692, 6755, 6691, 6754, 6690, 6753, 6689, 6752, 6688, 6751, 6687, 6750, 6686, 6749, 6685, 6748, 6684, 6747, 6683, 6746, 6682, 6745, 6681, 6744, 6680, 6743, 6679, 6742, 6678, 6741, 6677, 6740, 6676, 6739, 6675, 6738, 6674, 6737, 6673, 6736, 6672, 6735, 6671, 6734, 6670, 6733, 6669, 6732, 6668, 6731, 6667, 6730, 6666, 6729, 6665, 6728, 6664, 6727, 6663, 6726, 6662, 6725, 6661, 6724, 6660, 6723, 6659, 6722, 6658, 6721, 6657, 6720, 6656, 6655, 6591, 6654, 6590, 6653, 6589, 6652, 6588, 6651, 6587, 6650, 6586, 6649, 6585, 6648, 6584, 6647, 6583, 6646, 6582, 6645, 6581, 6644, 6580, 6643, 6579, 6642, 6578, 6641, 6577, 6640, 6576, 6639, 6575, 6638, 6574, 6637, 6573, 6636, 6572, 6635, 6571, 6634, 6570, 6633, 6569, 6632, 6568, 6631, 6567, 6630, 6566, 6629, 6565, 6628, 6564, 6627, 6563, 6626, 6562, 6625, 6561, 6624, 6560, 6623, 6559, 6622, 6558, 6621, 6557, 6620, 6556, 6619, 6555, 6618, 6554, 6617, 6553, 6616, 6552, 6615, 6551, 6614, 6550, 6613, 6549, 6612, 6548, 6611, 6547, 6610, 6546, 6609, 6545, 6608, 6544, 6607, 6543, 6606, 6542, 6605, 6541, 6604, 6540, 6603, 6539, 6602, 6538, 6601, 6537, 6600, 6536, 6599, 6535, 6598, 6534, 6597, 6533, 6596, 6532, 6595, 6531, 6594, 6530, 6593, 6529, 6592, 6528, 6527, 6463, 6526, 6462, 6525, 6461, 6524, 6460, 6523, 6459, 6522, 6458, 6521, 6457, 6520, 6456, 6519, 6455, 6518, 6454, 6517, 6453, 6516, 6452, 6515, 6451, 6514, 6450, 6513, 6449, 6512, 6448, 6511, 6447, 6510, 6446, 6509, 6445, 6508, 6444, 6507, 6443, 6506, 6442, 6505, 6441, 6504, 6440, 6503, 6439, 6502, 6438, 6501, 6437, 6500, 6436, 6499, 6435, 6498, 6434, 6497, 6433, 6496, 6432, 6495, 6431, 6494, 6430, 6493, 6429, 6492, 6428, 6491, 6427, 6490, 6426, 6489, 6425, 6488, 6424, 6487, 6423, 6486, 6422, 6485, 6421, 6484, 6420, 6483, 6419, 6482, 6418, 6481, 6417, 6480, 6416, 6479, 6415, 6478, 6414, 6477, 6413, 6476, 6412, 6475, 6411, 6474, 6410, 6473, 6409, 6472, 6408, 6471, 6407, 6470, 6406, 6469, 6405, 6468, 6404, 6467, 6403, 6466, 6402, 6465, 6401, 6464, 6400, 6399, 6335, 6398, 6334, 6397, 6333, 6396, 6332, 6395, 6331, 6394, 6330, 6393, 6329, 6392, 6328, 6391, 6327, 6390, 6326, 6389, 6325, 6388, 6324, 6387, 6323, 6386, 6322, 6385, 6321, 6384, 6320, 6383, 6319, 6382, 6318, 6381, 6317, 6380, 6316, 6379, 6315, 6378, 6314, 6377, 6313, 6376, 6312, 6375, 6311, 6374, 6310, 6373, 6309, 6372, 6308, 6371, 6307, 6370, 6306, 6369, 6305, 6368, 6304, 6367, 6303, 6366, 6302, 6365, 6301, 6364, 6300, 6363, 6299, 6362, 6298, 6361, 6297, 6360, 6296, 6359, 6295, 6358, 6294, 6357, 6293, 6356, 6292, 6355, 6291, 6354, 6290, 6353, 6289, 6352, 6288, 6351, 6287, 6350, 6286, 6349, 6285, 6348, 6284, 6347, 6283, 6346, 6282, 6345, 6281, 6344, 6280, 6343, 6279, 6342, 6278, 6341, 6277, 6340, 6276, 6339, 6275, 6338, 6274, 6337, 6273, 6336, 6272, 6271, 6207, 6270, 6206, 6269, 6205, 6268, 6204, 6267, 6203, 6266, 6202, 6265, 6201, 6264, 6200, 6263, 6199, 6262, 6198, 6261, 6197, 6260, 6196, 6259, 6195, 6258, 6194, 6257, 6193, 6256, 6192, 6255, 6191, 6254, 6190, 6253, 6189, 6252, 6188, 6251, 6187, 6250, 6186, 6249, 6185, 6248, 6184, 6247, 6183, 6246, 6182, 6245, 6181, 6244, 6180, 6243, 6179, 6242, 6178, 6241, 6177, 6240, 6176, 6239, 6175, 6238, 6174, 6237, 6173, 6236, 6172, 6235, 6171, 6234, 6170, 6233, 6169, 6232, 6168, 6231, 6167, 6230, 6166, 6229, 6165, 6228, 6164, 6227, 6163, 6226, 6162, 6225, 6161, 6224, 6160, 6223, 6159, 6222, 6158, 6221, 6157, 6220, 6156, 6219, 6155, 6218, 6154, 6217, 6153, 6216, 6152, 6215, 6151, 6214, 6150, 6213, 6149, 6212, 6148, 6211, 6147, 6210, 6146, 6209, 6145, 6208, 6144, 6143, 6079, 6142, 6078, 6141, 6077, 6140, 6076, 6139, 6075, 6138, 6074, 6137, 6073, 6136, 6072, 6135, 6071, 6134, 6070, 6133, 6069, 6132, 6068, 6131, 6067, 6130, 6066, 6129, 6065, 6128, 6064, 6127, 6063, 6126, 6062, 6125, 6061, 6124, 6060, 6123, 6059, 6122, 6058, 6121, 6057, 6120, 6056, 6119, 6055, 6118, 6054, 6117, 6053, 6116, 6052, 6115, 6051, 6114, 6050, 6113, 6049, 6112, 6048, 6111, 6047, 6110, 6046, 6109, 6045, 6108, 6044, 6107, 6043, 6106, 6042, 6105, 6041, 6104, 6040, 6103, 6039, 6102, 6038, 6101, 6037, 6100, 6036, 6099, 6035, 6098, 6034, 6097, 6033, 6096, 6032, 6095, 6031, 6094, 6030, 6093, 6029, 6092, 6028, 6091, 6027, 6090, 6026, 6089, 6025, 6088, 6024, 6087, 6023, 6086, 6022, 6085, 6021, 6084, 6020, 6083, 6019, 6082, 6018, 6081, 6017, 6080, 6016, 6015, 5951, 6014, 5950, 6013, 5949, 6012, 5948, 6011, 5947, 6010, 5946, 6009, 5945, 6008, 5944, 6007, 5943, 6006, 5942, 6005, 5941, 6004, 5940, 6003, 5939, 6002, 5938, 6001, 5937, 6000, 5936, 5999, 5935, 5998, 5934, 5997, 5933, 5996, 5932, 5995, 5931, 5994, 5930, 5993, 5929, 5992, 5928, 5991, 5927, 5990, 5926, 5989, 5925, 5988, 5924, 5987, 5923, 5986, 5922, 5985, 5921, 5984, 5920, 5983, 5919, 5982, 5918, 5981, 5917, 5980, 5916, 5979, 5915, 5978, 5914, 5977, 5913, 5976, 5912, 5975, 5911, 5974, 5910, 5973, 5909, 5972, 5908, 5971, 5907, 5970, 5906, 5969, 5905, 5968, 5904, 5967, 5903, 5966, 5902, 5965, 5901, 5964, 5900, 5963, 5899, 5962, 5898, 5961, 5897, 5960, 5896, 5959, 5895, 5958, 5894, 5957, 5893, 5956, 5892, 5955, 5891, 5954, 5890, 5953, 5889, 5952, 5888, 5887, 5823, 5886, 5822, 5885, 5821, 5884, 5820, 5883, 5819, 5882, 5818, 5881, 5817, 5880, 5816, 5879, 5815, 5878, 5814, 5877, 5813, 5876, 5812, 5875, 5811, 5874, 5810, 5873, 5809, 5872, 5808, 5871, 5807, 5870, 5806, 5869, 5805, 5868, 5804, 5867, 5803, 5866, 5802, 5865, 5801, 5864, 5800, 5863, 5799, 5862, 5798, 5861, 5797, 5860, 5796, 5859, 5795, 5858, 5794, 5857, 5793, 5856, 5792, 5855, 5791, 5854, 5790, 5853, 5789, 5852, 5788, 5851, 5787, 5850, 5786, 5849, 5785, 5848, 5784, 5847, 5783, 5846, 5782, 5845, 5781, 5844, 5780, 5843, 5779, 5842, 5778, 5841, 5777, 5840, 5776, 5839, 5775, 5838, 5774, 5837, 5773, 5836, 5772, 5835, 5771, 5834, 5770, 5833, 5769, 5832, 5768, 5831, 5767, 5830, 5766, 5829, 5765, 5828, 5764, 5827, 5763, 5826, 5762, 5825, 5761, 5824, 5760, 5759, 5695, 5758, 5694, 5757, 5693, 5756, 5692, 5755, 5691, 5754, 5690, 5753, 5689, 5752, 5688, 5751, 5687, 5750, 5686, 5749, 5685, 5748, 5684, 5747, 5683, 5746, 5682, 5745, 5681, 5744, 5680, 5743, 5679, 5742, 5678, 5741, 5677, 5740, 5676, 5739, 5675, 5738, 5674, 5737, 5673, 5736, 5672, 5735, 5671, 5734, 5670, 5733, 5669, 5732, 5668, 5731, 5667, 5730, 5666, 5729, 5665, 5728, 5664, 5727, 5663, 5726, 5662, 5725, 5661, 5724, 5660, 5723, 5659, 5722, 5658, 5721, 5657, 5720, 5656, 5719, 5655, 5718, 5654, 5717, 5653, 5716, 5652, 5715, 5651, 5714, 5650, 5713, 5649, 5712, 5648, 5711, 5647, 5710, 5646, 5709, 5645, 5708, 5644, 5707, 5643, 5706, 5642, 5705, 5641, 5704, 5640, 5703, 5639, 5702, 5638, 5701, 5637, 5700, 5636, 5699, 5635, 5698, 5634, 5697, 5633, 5696, 5632, 5631, 5567, 5630, 5566, 5629, 5565, 5628, 5564, 5627, 5563, 5626, 5562, 5625, 5561, 5624, 5560, 5623, 5559, 5622, 5558, 5621, 5557, 5620, 5556, 5619, 5555, 5618, 5554, 5617, 5553, 5616, 5552, 5615, 5551, 5614, 5550, 5613, 5549, 5612, 5548, 5611, 5547, 5610, 5546, 5609, 5545, 5608, 5544, 5607, 5543, 5606, 5542, 5605, 5541, 5604, 5540, 5603, 5539, 5602, 5538, 5601, 5537, 5600, 5536, 5599, 5535, 5598, 5534, 5597, 5533, 5596, 5532, 5595, 5531, 5594, 5530, 5593, 5529, 5592, 5528, 5591, 5527, 5590, 5526, 5589, 5525, 5588, 5524, 5587, 5523, 5586, 5522, 5585, 5521, 5584, 5520, 5583, 5519, 5582, 5518, 5581, 5517, 5580, 5516, 5579, 5515, 5578, 5514, 5577, 5513, 5576, 5512, 5575, 5511, 5574, 5510, 5573, 5509, 5572, 5508, 5571, 5507, 5570, 5506, 5569, 5505, 5568, 5504, 5503, 5439, 5502, 5438, 5501, 5437, 5500, 5436, 5499, 5435, 5498, 5434, 5497, 5433, 5496, 5432, 5495, 5431, 5494, 5430, 5493, 5429, 5492, 5428, 5491, 5427, 5490, 5426, 5489, 5425, 5488, 5424, 5487, 5423, 5486, 5422, 5485, 5421, 5484, 5420, 5483, 5419, 5482, 5418, 5481, 5417, 5480, 5416, 5479, 5415, 5478, 5414, 5477, 5413, 5476, 5412, 5475, 5411, 5474, 5410, 5473, 5409, 5472, 5408, 5471, 5407, 5470, 5406, 5469, 5405, 5468, 5404, 5467, 5403, 5466, 5402, 5465, 5401, 5464, 5400, 5463, 5399, 5462, 5398, 5461, 5397, 5460, 5396, 5459, 5395, 5458, 5394, 5457, 5393, 5456, 5392, 5455, 5391, 5454, 5390, 5453, 5389, 5452, 5388, 5451, 5387, 5450, 5386, 5449, 5385, 5448, 5384, 5447, 5383, 5446, 5382, 5445, 5381, 5444, 5380, 5443, 5379, 5442, 5378, 5441, 5377, 5440, 5376, 5375, 5311, 5374, 5310, 5373, 5309, 5372, 5308, 5371, 5307, 5370, 5306, 5369, 5305, 5368, 5304, 5367, 5303, 5366, 5302, 5365, 5301, 5364, 5300, 5363, 5299, 5362, 5298, 5361, 5297, 5360, 5296, 5359, 5295, 5358, 5294, 5357, 5293, 5356, 5292, 5355, 5291, 5354, 5290, 5353, 5289, 5352, 5288, 5351, 5287, 5350, 5286, 5349, 5285, 5348, 5284, 5347, 5283, 5346, 5282, 5345, 5281, 5344, 5280, 5343, 5279, 5342, 5278, 5341, 5277, 5340, 5276, 5339, 5275, 5338, 5274, 5337, 5273, 5336, 5272, 5335, 5271, 5334, 5270, 5333, 5269, 5332, 5268, 5331, 5267, 5330, 5266, 5329, 5265, 5328, 5264, 5327, 5263, 5326, 5262, 5325, 5261, 5324, 5260, 5323, 5259, 5322, 5258, 5321, 5257, 5320, 5256, 5319, 5255, 5318, 5254, 5317, 5253, 5316, 5252, 5315, 5251, 5314, 5250, 5313, 5249, 5312, 5248, 5247, 5183, 5246, 5182, 5245, 5181, 5244, 5180, 5243, 5179, 5242, 5178, 5241, 5177, 5240, 5176, 5239, 5175, 5238, 5174, 5237, 5173, 5236, 5172, 5235, 5171, 5234, 5170, 5233, 5169, 5232, 5168, 5231, 5167, 5230, 5166, 5229, 5165, 5228, 5164, 5227, 5163, 5226, 5162, 5225, 5161, 5224, 5160, 5223, 5159, 5222, 5158, 5221, 5157, 5220, 5156, 5219, 5155, 5218, 5154, 5217, 5153, 5216, 5152, 5215, 5151, 5214, 5150, 5213, 5149, 5212, 5148, 5211, 5147, 5210, 5146, 5209, 5145, 5208, 5144, 5207, 5143, 5206, 5142, 5205, 5141, 5204, 5140, 5203, 5139, 5202, 5138, 5201, 5137, 5200, 5136, 5199, 5135, 5198, 5134, 5197, 5133, 5196, 5132, 5195, 5131, 5194, 5130, 5193, 5129, 5192, 5128, 5191, 5127, 5190, 5126, 5189, 5125, 5188, 5124, 5187, 5123, 5186, 5122, 5185, 5121, 5184, 5120, 5119, 5055, 5118, 5054, 5117, 5053, 5116, 5052, 5115, 5051, 5114, 5050, 5113, 5049, 5112, 5048, 5111, 5047, 5110, 5046, 5109, 5045, 5108, 5044, 5107, 5043, 5106, 5042, 5105, 5041, 5104, 5040, 5103, 5039, 5102, 5038, 5101, 5037, 5100, 5036, 5099, 5035, 5098, 5034, 5097, 5033, 5096, 5032, 5095, 5031, 5094, 5030, 5093, 5029, 5092, 5028, 5091, 5027, 5090, 5026, 5089, 5025, 5088, 5024, 5087, 5023, 5086, 5022, 5085, 5021, 5084, 5020, 5083, 5019, 5082, 5018, 5081, 5017, 5080, 5016, 5079, 5015, 5078, 5014, 5077, 5013, 5076, 5012, 5075, 5011, 5074, 5010, 5073, 5009, 5072, 5008, 5071, 5007, 5070, 5006, 5069, 5005, 5068, 5004, 5067, 5003, 5066, 5002, 5065, 5001, 5064, 5000, 5063, 4999, 5062, 4998, 5061, 4997, 5060, 4996, 5059, 4995, 5058, 4994, 5057, 4993, 5056, 4992, 4991, 4927, 4990, 4926, 4989, 4925, 4988, 4924, 4987, 4923, 4986, 4922, 4985, 4921, 4984, 4920, 4983, 4919, 4982, 4918, 4981, 4917, 4980, 4916, 4979, 4915, 4978, 4914, 4977, 4913, 4976, 4912, 4975, 4911, 4974, 4910, 4973, 4909, 4972, 4908, 4971, 4907, 4970, 4906, 4969, 4905, 4968, 4904, 4967, 4903, 4966, 4902, 4965, 4901, 4964, 4900, 4963, 4899, 4962, 4898, 4961, 4897, 4960, 4896, 4959, 4895, 4958, 4894, 4957, 4893, 4956, 4892, 4955, 4891, 4954, 4890, 4953, 4889, 4952, 4888, 4951, 4887, 4950, 4886, 4949, 4885, 4948, 4884, 4947, 4883, 4946, 4882, 4945, 4881, 4944, 4880, 4943, 4879, 4942, 4878, 4941, 4877, 4940, 4876, 4939, 4875, 4938, 4874, 4937, 4873, 4936, 4872, 4935, 4871, 4934, 4870, 4933, 4869, 4932, 4868, 4931, 4867, 4930, 4866, 4929, 4865, 4928, 4864, 4863, 4799, 4862, 4798, 4861, 4797, 4860, 4796, 4859, 4795, 4858, 4794, 4857, 4793, 4856, 4792, 4855, 4791, 4854, 4790, 4853, 4789, 4852, 4788, 4851, 4787, 4850, 4786, 4849, 4785, 4848, 4784, 4847, 4783, 4846, 4782, 4845, 4781, 4844, 4780, 4843, 4779, 4842, 4778, 4841, 4777, 4840, 4776, 4839, 4775, 4838, 4774, 4837, 4773, 4836, 4772, 4835, 4771, 4834, 4770, 4833, 4769, 4832, 4768, 4831, 4767, 4830, 4766, 4829, 4765, 4828, 4764, 4827, 4763, 4826, 4762, 4825, 4761, 4824, 4760, 4823, 4759, 4822, 4758, 4821, 4757, 4820, 4756, 4819, 4755, 4818, 4754, 4817, 4753, 4816, 4752, 4815, 4751, 4814, 4750, 4813, 4749, 4812, 4748, 4811, 4747, 4810, 4746, 4809, 4745, 4808, 4744, 4807, 4743, 4806, 4742, 4805, 4741, 4804, 4740, 4803, 4739, 4802, 4738, 4801, 4737, 4800, 4736, 4735, 4671, 4734, 4670, 4733, 4669, 4732, 4668, 4731, 4667, 4730, 4666, 4729, 4665, 4728, 4664, 4727, 4663, 4726, 4662, 4725, 4661, 4724, 4660, 4723, 4659, 4722, 4658, 4721, 4657, 4720, 4656, 4719, 4655, 4718, 4654, 4717, 4653, 4716, 4652, 4715, 4651, 4714, 4650, 4713, 4649, 4712, 4648, 4711, 4647, 4710, 4646, 4709, 4645, 4708, 4644, 4707, 4643, 4706, 4642, 4705, 4641, 4704, 4640, 4703, 4639, 4702, 4638, 4701, 4637, 4700, 4636, 4699, 4635, 4698, 4634, 4697, 4633, 4696, 4632, 4695, 4631, 4694, 4630, 4693, 4629, 4692, 4628, 4691, 4627, 4690, 4626, 4689, 4625, 4688, 4624, 4687, 4623, 4686, 4622, 4685, 4621, 4684, 4620, 4683, 4619, 4682, 4618, 4681, 4617, 4680, 4616, 4679, 4615, 4678, 4614, 4677, 4613, 4676, 4612, 4675, 4611, 4674, 4610, 4673, 4609, 4672, 4608, 4607, 4543, 4606, 4542, 4605, 4541, 4604, 4540, 4603, 4539, 4602, 4538, 4601, 4537, 4600, 4536, 4599, 4535, 4598, 4534, 4597, 4533, 4596, 4532, 4595, 4531, 4594, 4530, 4593, 4529, 4592, 4528, 4591, 4527, 4590, 4526, 4589, 4525, 4588, 4524, 4587, 4523, 4586, 4522, 4585, 4521, 4584, 4520, 4583, 4519, 4582, 4518, 4581, 4517, 4580, 4516, 4579, 4515, 4578, 4514, 4577, 4513, 4576, 4512, 4575, 4511, 4574, 4510, 4573, 4509, 4572, 4508, 4571, 4507, 4570, 4506, 4569, 4505, 4568, 4504, 4567, 4503, 4566, 4502, 4565, 4501, 4564, 4500, 4563, 4499, 4562, 4498, 4561, 4497, 4560, 4496, 4559, 4495, 4558, 4494, 4557, 4493, 4556, 4492, 4555, 4491, 4554, 4490, 4553, 4489, 4552, 4488, 4551, 4487, 4550, 4486, 4549, 4485, 4548, 4484, 4547, 4483, 4546, 4482, 4545, 4481, 4544, 4480, 4479, 4415, 4478, 4414, 4477, 4413, 4476, 4412, 4475, 4411, 4474, 4410, 4473, 4409, 4472, 4408, 4471, 4407, 4470, 4406, 4469, 4405, 4468, 4404, 4467, 4403, 4466, 4402, 4465, 4401, 4464, 4400, 4463, 4399, 4462, 4398, 4461, 4397, 4460, 4396, 4459, 4395, 4458, 4394, 4457, 4393, 4456, 4392, 4455, 4391, 4454, 4390, 4453, 4389, 4452, 4388, 4451, 4387, 4450, 4386, 4449, 4385, 4448, 4384, 4447, 4383, 4446, 4382, 4445, 4381, 4444, 4380, 4443, 4379, 4442, 4378, 4441, 4377, 4440, 4376, 4439, 4375, 4438, 4374, 4437, 4373, 4436, 4372, 4435, 4371, 4434, 4370, 4433, 4369, 4432, 4368, 4431, 4367, 4430, 4366, 4429, 4365, 4428, 4364, 4427, 4363, 4426, 4362, 4425, 4361, 4424, 4360, 4423, 4359, 4422, 4358, 4421, 4357, 4420, 4356, 4419, 4355, 4418, 4354, 4417, 4353, 4416, 4352, 4351, 4287, 4350, 4286, 4349, 4285, 4348, 4284, 4347, 4283, 4346, 4282, 4345, 4281, 4344, 4280, 4343, 4279, 4342, 4278, 4341, 4277, 4340, 4276, 4339, 4275, 4338, 4274, 4337, 4273, 4336, 4272, 4335, 4271, 4334, 4270, 4333, 4269, 4332, 4268, 4331, 4267, 4330, 4266, 4329, 4265, 4328, 4264, 4327, 4263, 4326, 4262, 4325, 4261, 4324, 4260, 4323, 4259, 4322, 4258, 4321, 4257, 4320, 4256, 4319, 4255, 4318, 4254, 4317, 4253, 4316, 4252, 4315, 4251, 4314, 4250, 4313, 4249, 4312, 4248, 4311, 4247, 4310, 4246, 4309, 4245, 4308, 4244, 4307, 4243, 4306, 4242, 4305, 4241, 4304, 4240, 4303, 4239, 4302, 4238, 4301, 4237, 4300, 4236, 4299, 4235, 4298, 4234, 4297, 4233, 4296, 4232, 4295, 4231, 4294, 4230, 4293, 4229, 4292, 4228, 4291, 4227, 4290, 4226, 4289, 4225, 4288, 4224, 4223, 4159, 4222, 4158, 4221, 4157, 4220, 4156, 4219, 4155, 4218, 4154, 4217, 4153, 4216, 4152, 4215, 4151, 4214, 4150, 4213, 4149, 4212, 4148, 4211, 4147, 4210, 4146, 4209, 4145, 4208, 4144, 4207, 4143, 4206, 4142, 4205, 4141, 4204, 4140, 4203, 4139, 4202, 4138, 4201, 4137, 4200, 4136, 4199, 4135, 4198, 4134, 4197, 4133, 4196, 4132, 4195, 4131, 4194, 4130, 4193, 4129, 4192, 4128, 4191, 4127, 4190, 4126, 4189, 4125, 4188, 4124, 4187, 4123, 4186, 4122, 4185, 4121, 4184, 4120, 4183, 4119, 4182, 4118, 4181, 4117, 4180, 4116, 4179, 4115, 4178, 4114, 4177, 4113, 4176, 4112, 4175, 4111, 4174, 4110, 4173, 4109, 4172, 4108, 4171, 4107, 4170, 4106, 4169, 4105, 4168, 4104, 4167, 4103, 4166, 4102, 4165, 4101, 4164, 4100, 4163, 4099, 4162, 4098, 4161, 4097, 4160, 4096, 4095, 4031, 4094, 4030, 4093, 4029, 4092, 4028, 4091, 4027, 4090, 4026, 4089, 4025, 4088, 4024, 4087, 4023, 4086, 4022, 4085, 4021, 4084, 4020, 4083, 4019, 4082, 4018, 4081, 4017, 4080, 4016, 4079, 4015, 4078, 4014, 4077, 4013, 4076, 4012, 4075, 4011, 4074, 4010, 4073, 4009, 4072, 4008, 4071, 4007, 4070, 4006, 4069, 4005, 4068, 4004, 4067, 4003, 4066, 4002, 4065, 4001, 4064, 4000, 4063, 3999, 4062, 3998, 4061, 3997, 4060, 3996, 4059, 3995, 4058, 3994, 4057, 3993, 4056, 3992, 4055, 3991, 4054, 3990, 4053, 3989, 4052, 3988, 4051, 3987, 4050, 3986, 4049, 3985, 4048, 3984, 4047, 3983, 4046, 3982, 4045, 3981, 4044, 3980, 4043, 3979, 4042, 3978, 4041, 3977, 4040, 3976, 4039, 3975, 4038, 3974, 4037, 3973, 4036, 3972, 4035, 3971, 4034, 3970, 4033, 3969, 4032, 3968, 3967, 3903, 3966, 3902, 3965, 3901, 3964, 3900, 3963, 3899, 3962, 3898, 3961, 3897, 3960, 3896, 3959, 3895, 3958, 3894, 3957, 3893, 3956, 3892, 3955, 3891, 3954, 3890, 3953, 3889, 3952, 3888, 3951, 3887, 3950, 3886, 3949, 3885, 3948, 3884, 3947, 3883, 3946, 3882, 3945, 3881, 3944, 3880, 3943, 3879, 3942, 3878, 3941, 3877, 3940, 3876, 3939, 3875, 3938, 3874, 3937, 3873, 3936, 3872, 3935, 3871, 3934, 3870, 3933, 3869, 3932, 3868, 3931, 3867, 3930, 3866, 3929, 3865, 3928, 3864, 3927, 3863, 3926, 3862, 3925, 3861, 3924, 3860, 3923, 3859, 3922, 3858, 3921, 3857, 3920, 3856, 3919, 3855, 3918, 3854, 3917, 3853, 3916, 3852, 3915, 3851, 3914, 3850, 3913, 3849, 3912, 3848, 3911, 3847, 3910, 3846, 3909, 3845, 3908, 3844, 3907, 3843, 3906, 3842, 3905, 3841, 3904, 3840, 3839, 3775, 3838, 3774, 3837, 3773, 3836, 3772, 3835, 3771, 3834, 3770, 3833, 3769, 3832, 3768, 3831, 3767, 3830, 3766, 3829, 3765, 3828, 3764, 3827, 3763, 3826, 3762, 3825, 3761, 3824, 3760, 3823, 3759, 3822, 3758, 3821, 3757, 3820, 3756, 3819, 3755, 3818, 3754, 3817, 3753, 3816, 3752, 3815, 3751, 3814, 3750, 3813, 3749, 3812, 3748, 3811, 3747, 3810, 3746, 3809, 3745, 3808, 3744, 3807, 3743, 3806, 3742, 3805, 3741, 3804, 3740, 3803, 3739, 3802, 3738, 3801, 3737, 3800, 3736, 3799, 3735, 3798, 3734, 3797, 3733, 3796, 3732, 3795, 3731, 3794, 3730, 3793, 3729, 3792, 3728, 3791, 3727, 3790, 3726, 3789, 3725, 3788, 3724, 3787, 3723, 3786, 3722, 3785, 3721, 3784, 3720, 3783, 3719, 3782, 3718, 3781, 3717, 3780, 3716, 3779, 3715, 3778, 3714, 3777, 3713, 3776, 3712, 3711, 3647, 3710, 3646, 3709, 3645, 3708, 3644, 3707, 3643, 3706, 3642, 3705, 3641, 3704, 3640, 3703, 3639, 3702, 3638, 3701, 3637, 3700, 3636, 3699, 3635, 3698, 3634, 3697, 3633, 3696, 3632, 3695, 3631, 3694, 3630, 3693, 3629, 3692, 3628, 3691, 3627, 3690, 3626, 3689, 3625, 3688, 3624, 3687, 3623, 3686, 3622, 3685, 3621, 3684, 3620, 3683, 3619, 3682, 3618, 3681, 3617, 3680, 3616, 3679, 3615, 3678, 3614, 3677, 3613, 3676, 3612, 3675, 3611, 3674, 3610, 3673, 3609, 3672, 3608, 3671, 3607, 3670, 3606, 3669, 3605, 3668, 3604, 3667, 3603, 3666, 3602, 3665, 3601, 3664, 3600, 3663, 3599, 3662, 3598, 3661, 3597, 3660, 3596, 3659, 3595, 3658, 3594, 3657, 3593, 3656, 3592, 3655, 3591, 3654, 3590, 3653, 3589, 3652, 3588, 3651, 3587, 3650, 3586, 3649, 3585, 3648, 3584, 3583, 3519, 3582, 3518, 3581, 3517, 3580, 3516, 3579, 3515, 3578, 3514, 3577, 3513, 3576, 3512, 3575, 3511, 3574, 3510, 3573, 3509, 3572, 3508, 3571, 3507, 3570, 3506, 3569, 3505, 3568, 3504, 3567, 3503, 3566, 3502, 3565, 3501, 3564, 3500, 3563, 3499, 3562, 3498, 3561, 3497, 3560, 3496, 3559, 3495, 3558, 3494, 3557, 3493, 3556, 3492, 3555, 3491, 3554, 3490, 3553, 3489, 3552, 3488, 3551, 3487, 3550, 3486, 3549, 3485, 3548, 3484, 3547, 3483, 3546, 3482, 3545, 3481, 3544, 3480, 3543, 3479, 3542, 3478, 3541, 3477, 3540, 3476, 3539, 3475, 3538, 3474, 3537, 3473, 3536, 3472, 3535, 3471, 3534, 3470, 3533, 3469, 3532, 3468, 3531, 3467, 3530, 3466, 3529, 3465, 3528, 3464, 3527, 3463, 3526, 3462, 3525, 3461, 3524, 3460, 3523, 3459, 3522, 3458, 3521, 3457, 3520, 3456, 3455, 3391, 3454, 3390, 3453, 3389, 3452, 3388, 3451, 3387, 3450, 3386, 3449, 3385, 3448, 3384, 3447, 3383, 3446, 3382, 3445, 3381, 3444, 3380, 3443, 3379, 3442, 3378, 3441, 3377, 3440, 3376, 3439, 3375, 3438, 3374, 3437, 3373, 3436, 3372, 3435, 3371, 3434, 3370, 3433, 3369, 3432, 3368, 3431, 3367, 3430, 3366, 3429, 3365, 3428, 3364, 3427, 3363, 3426, 3362, 3425, 3361, 3424, 3360, 3423, 3359, 3422, 3358, 3421, 3357, 3420, 3356, 3419, 3355, 3418, 3354, 3417, 3353, 3416, 3352, 3415, 3351, 3414, 3350, 3413, 3349, 3412, 3348, 3411, 3347, 3410, 3346, 3409, 3345, 3408, 3344, 3407, 3343, 3406, 3342, 3405, 3341, 3404, 3340, 3403, 3339, 3402, 3338, 3401, 3337, 3400, 3336, 3399, 3335, 3398, 3334, 3397, 3333, 3396, 3332, 3395, 3331, 3394, 3330, 3393, 3329, 3392, 3328, 3327, 3263, 3326, 3262, 3325, 3261, 3324, 3260, 3323, 3259, 3322, 3258, 3321, 3257, 3320, 3256, 3319, 3255, 3318, 3254, 3317, 3253, 3316, 3252, 3315, 3251, 3314, 3250, 3313, 3249, 3312, 3248, 3311, 3247, 3310, 3246, 3309, 3245, 3308, 3244, 3307, 3243, 3306, 3242, 3305, 3241, 3304, 3240, 3303, 3239, 3302, 3238, 3301, 3237, 3300, 3236, 3299, 3235, 3298, 3234, 3297, 3233, 3296, 3232, 3295, 3231, 3294, 3230, 3293, 3229, 3292, 3228, 3291, 3227, 3290, 3226, 3289, 3225, 3288, 3224, 3287, 3223, 3286, 3222, 3285, 3221, 3284, 3220, 3283, 3219, 3282, 3218, 3281, 3217, 3280, 3216, 3279, 3215, 3278, 3214, 3277, 3213, 3276, 3212, 3275, 3211, 3274, 3210, 3273, 3209, 3272, 3208, 3271, 3207, 3270, 3206, 3269, 3205, 3268, 3204, 3267, 3203, 3266, 3202, 3265, 3201, 3264, 3200, 3199, 3135, 3198, 3134, 3197, 3133, 3196, 3132, 3195, 3131, 3194, 3130, 3193, 3129, 3192, 3128, 3191, 3127, 3190, 3126, 3189, 3125, 3188, 3124, 3187, 3123, 3186, 3122, 3185, 3121, 3184, 3120, 3183, 3119, 3182, 3118, 3181, 3117, 3180, 3116, 3179, 3115, 3178, 3114, 3177, 3113, 3176, 3112, 3175, 3111, 3174, 3110, 3173, 3109, 3172, 3108, 3171, 3107, 3170, 3106, 3169, 3105, 3168, 3104, 3167, 3103, 3166, 3102, 3165, 3101, 3164, 3100, 3163, 3099, 3162, 3098, 3161, 3097, 3160, 3096, 3159, 3095, 3158, 3094, 3157, 3093, 3156, 3092, 3155, 3091, 3154, 3090, 3153, 3089, 3152, 3088, 3151, 3087, 3150, 3086, 3149, 3085, 3148, 3084, 3147, 3083, 3146, 3082, 3145, 3081, 3144, 3080, 3143, 3079, 3142, 3078, 3141, 3077, 3140, 3076, 3139, 3075, 3138, 3074, 3137, 3073, 3136, 3072, 3071, 3007, 3070, 3006, 3069, 3005, 3068, 3004, 3067, 3003, 3066, 3002, 3065, 3001, 3064, 3000, 3063, 2999, 3062, 2998, 3061, 2997, 3060, 2996, 3059, 2995, 3058, 2994, 3057, 2993, 3056, 2992, 3055, 2991, 3054, 2990, 3053, 2989, 3052, 2988, 3051, 2987, 3050, 2986, 3049, 2985, 3048, 2984, 3047, 2983, 3046, 2982, 3045, 2981, 3044, 2980, 3043, 2979, 3042, 2978, 3041, 2977, 3040, 2976, 3039, 2975, 3038, 2974, 3037, 2973, 3036, 2972, 3035, 2971, 3034, 2970, 3033, 2969, 3032, 2968, 3031, 2967, 3030, 2966, 3029, 2965, 3028, 2964, 3027, 2963, 3026, 2962, 3025, 2961, 3024, 2960, 3023, 2959, 3022, 2958, 3021, 2957, 3020, 2956, 3019, 2955, 3018, 2954, 3017, 2953, 3016, 2952, 3015, 2951, 3014, 2950, 3013, 2949, 3012, 2948, 3011, 2947, 3010, 2946, 3009, 2945, 3008, 2944, 2943, 2879, 2942, 2878, 2941, 2877, 2940, 2876, 2939, 2875, 2938, 2874, 2937, 2873, 2936, 2872, 2935, 2871, 2934, 2870, 2933, 2869, 2932, 2868, 2931, 2867, 2930, 2866, 2929, 2865, 2928, 2864, 2927, 2863, 2926, 2862, 2925, 2861, 2924, 2860, 2923, 2859, 2922, 2858, 2921, 2857, 2920, 2856, 2919, 2855, 2918, 2854, 2917, 2853, 2916, 2852, 2915, 2851, 2914, 2850, 2913, 2849, 2912, 2848, 2911, 2847, 2910, 2846, 2909, 2845, 2908, 2844, 2907, 2843, 2906, 2842, 2905, 2841, 2904, 2840, 2903, 2839, 2902, 2838, 2901, 2837, 2900, 2836, 2899, 2835, 2898, 2834, 2897, 2833, 2896, 2832, 2895, 2831, 2894, 2830, 2893, 2829, 2892, 2828, 2891, 2827, 2890, 2826, 2889, 2825, 2888, 2824, 2887, 2823, 2886, 2822, 2885, 2821, 2884, 2820, 2883, 2819, 2882, 2818, 2881, 2817, 2880, 2816, 2815, 2751, 2814, 2750, 2813, 2749, 2812, 2748, 2811, 2747, 2810, 2746, 2809, 2745, 2808, 2744, 2807, 2743, 2806, 2742, 2805, 2741, 2804, 2740, 2803, 2739, 2802, 2738, 2801, 2737, 2800, 2736, 2799, 2735, 2798, 2734, 2797, 2733, 2796, 2732, 2795, 2731, 2794, 2730, 2793, 2729, 2792, 2728, 2791, 2727, 2790, 2726, 2789, 2725, 2788, 2724, 2787, 2723, 2786, 2722, 2785, 2721, 2784, 2720, 2783, 2719, 2782, 2718, 2781, 2717, 2780, 2716, 2779, 2715, 2778, 2714, 2777, 2713, 2776, 2712, 2775, 2711, 2774, 2710, 2773, 2709, 2772, 2708, 2771, 2707, 2770, 2706, 2769, 2705, 2768, 2704, 2767, 2703, 2766, 2702, 2765, 2701, 2764, 2700, 2763, 2699, 2762, 2698, 2761, 2697, 2760, 2696, 2759, 2695, 2758, 2694, 2757, 2693, 2756, 2692, 2755, 2691, 2754, 2690, 2753, 2689, 2752, 2688, 2687, 2623, 2686, 2622, 2685, 2621, 2684, 2620, 2683, 2619, 2682, 2618, 2681, 2617, 2680, 2616, 2679, 2615, 2678, 2614, 2677, 2613, 2676, 2612, 2675, 2611, 2674, 2610, 2673, 2609, 2672, 2608, 2671, 2607, 2670, 2606, 2669, 2605, 2668, 2604, 2667, 2603, 2666, 2602, 2665, 2601, 2664, 2600, 2663, 2599, 2662, 2598, 2661, 2597, 2660, 2596, 2659, 2595, 2658, 2594, 2657, 2593, 2656, 2592, 2655, 2591, 2654, 2590, 2653, 2589, 2652, 2588, 2651, 2587, 2650, 2586, 2649, 2585, 2648, 2584, 2647, 2583, 2646, 2582, 2645, 2581, 2644, 2580, 2643, 2579, 2642, 2578, 2641, 2577, 2640, 2576, 2639, 2575, 2638, 2574, 2637, 2573, 2636, 2572, 2635, 2571, 2634, 2570, 2633, 2569, 2632, 2568, 2631, 2567, 2630, 2566, 2629, 2565, 2628, 2564, 2627, 2563, 2626, 2562, 2625, 2561, 2624, 2560, 2559, 2495, 2558, 2494, 2557, 2493, 2556, 2492, 2555, 2491, 2554, 2490, 2553, 2489, 2552, 2488, 2551, 2487, 2550, 2486, 2549, 2485, 2548, 2484, 2547, 2483, 2546, 2482, 2545, 2481, 2544, 2480, 2543, 2479, 2542, 2478, 2541, 2477, 2540, 2476, 2539, 2475, 2538, 2474, 2537, 2473, 2536, 2472, 2535, 2471, 2534, 2470, 2533, 2469, 2532, 2468, 2531, 2467, 2530, 2466, 2529, 2465, 2528, 2464, 2527, 2463, 2526, 2462, 2525, 2461, 2524, 2460, 2523, 2459, 2522, 2458, 2521, 2457, 2520, 2456, 2519, 2455, 2518, 2454, 2517, 2453, 2516, 2452, 2515, 2451, 2514, 2450, 2513, 2449, 2512, 2448, 2511, 2447, 2510, 2446, 2509, 2445, 2508, 2444, 2507, 2443, 2506, 2442, 2505, 2441, 2504, 2440, 2503, 2439, 2502, 2438, 2501, 2437, 2500, 2436, 2499, 2435, 2498, 2434, 2497, 2433, 2496, 2432, 2431, 2367, 2430, 2366, 2429, 2365, 2428, 2364, 2427, 2363, 2426, 2362, 2425, 2361, 2424, 2360, 2423, 2359, 2422, 2358, 2421, 2357, 2420, 2356, 2419, 2355, 2418, 2354, 2417, 2353, 2416, 2352, 2415, 2351, 2414, 2350, 2413, 2349, 2412, 2348, 2411, 2347, 2410, 2346, 2409, 2345, 2408, 2344, 2407, 2343, 2406, 2342, 2405, 2341, 2404, 2340, 2403, 2339, 2402, 2338, 2401, 2337, 2400, 2336, 2399, 2335, 2398, 2334, 2397, 2333, 2396, 2332, 2395, 2331, 2394, 2330, 2393, 2329, 2392, 2328, 2391, 2327, 2390, 2326, 2389, 2325, 2388, 2324, 2387, 2323, 2386, 2322, 2385, 2321, 2384, 2320, 2383, 2319, 2382, 2318, 2381, 2317, 2380, 2316, 2379, 2315, 2378, 2314, 2377, 2313, 2376, 2312, 2375, 2311, 2374, 2310, 2373, 2309, 2372, 2308, 2371, 2307, 2370, 2306, 2369, 2305, 2368, 2304, 2303, 2239, 2302, 2238, 2301, 2237, 2300, 2236, 2299, 2235, 2298, 2234, 2297, 2233, 2296, 2232, 2295, 2231, 2294, 2230, 2293, 2229, 2292, 2228, 2291, 2227, 2290, 2226, 2289, 2225, 2288, 2224, 2287, 2223, 2286, 2222, 2285, 2221, 2284, 2220, 2283, 2219, 2282, 2218, 2281, 2217, 2280, 2216, 2279, 2215, 2278, 2214, 2277, 2213, 2276, 2212, 2275, 2211, 2274, 2210, 2273, 2209, 2272, 2208, 2271, 2207, 2270, 2206, 2269, 2205, 2268, 2204, 2267, 2203, 2266, 2202, 2265, 2201, 2264, 2200, 2263, 2199, 2262, 2198, 2261, 2197, 2260, 2196, 2259, 2195, 2258, 2194, 2257, 2193, 2256, 2192, 2255, 2191, 2254, 2190, 2253, 2189, 2252, 2188, 2251, 2187, 2250, 2186, 2249, 2185, 2248, 2184, 2247, 2183, 2246, 2182, 2245, 2181, 2244, 2180, 2243, 2179, 2242, 2178, 2241, 2177, 2240, 2176, 2175, 2111, 2174, 2110, 2173, 2109, 2172, 2108, 2171, 2107, 2170, 2106, 2169, 2105, 2168, 2104, 2167, 2103, 2166, 2102, 2165, 2101, 2164, 2100, 2163, 2099, 2162, 2098, 2161, 2097, 2160, 2096, 2159, 2095, 2158, 2094, 2157, 2093, 2156, 2092, 2155, 2091, 2154, 2090, 2153, 2089, 2152, 2088, 2151, 2087, 2150, 2086, 2149, 2085, 2148, 2084, 2147, 2083, 2146, 2082, 2145, 2081, 2144, 2080, 2143, 2079, 2142, 2078, 2141, 2077, 2140, 2076, 2139, 2075, 2138, 2074, 2137, 2073, 2136, 2072, 2135, 2071, 2134, 2070, 2133, 2069, 2132, 2068, 2131, 2067, 2130, 2066, 2129, 2065, 2128, 2064, 2127, 2063, 2126, 2062, 2125, 2061, 2124, 2060, 2123, 2059, 2122, 2058, 2121, 2057, 2120, 2056, 2119, 2055, 2118, 2054, 2117, 2053, 2116, 2052, 2115, 2051, 2114, 2050, 2113, 2049, 2112, 2048, 2047, 1983, 2046, 1982, 2045, 1981, 2044, 1980, 2043, 1979, 2042, 1978, 2041, 1977, 2040, 1976, 2039, 1975, 2038, 1974, 2037, 1973, 2036, 1972, 2035, 1971, 2034, 1970, 2033, 1969, 2032, 1968, 2031, 1967, 2030, 1966, 2029, 1965, 2028, 1964, 2027, 1963, 2026, 1962, 2025, 1961, 2024, 1960, 2023, 1959, 2022, 1958, 2021, 1957, 2020, 1956, 2019, 1955, 2018, 1954, 2017, 1953, 2016, 1952, 2015, 1951, 2014, 1950, 2013, 1949, 2012, 1948, 2011, 1947, 2010, 1946, 2009, 1945, 2008, 1944, 2007, 1943, 2006, 1942, 2005, 1941, 2004, 1940, 2003, 1939, 2002, 1938, 2001, 1937, 2000, 1936, 1999, 1935, 1998, 1934, 1997, 1933, 1996, 1932, 1995, 1931, 1994, 1930, 1993, 1929, 1992, 1928, 1991, 1927, 1990, 1926, 1989, 1925, 1988, 1924, 1987, 1923, 1986, 1922, 1985, 1921, 1984, 1920, 1919, 1855, 1918, 1854, 1917, 1853, 1916, 1852, 1915, 1851, 1914, 1850, 1913, 1849, 1912, 1848, 1911, 1847, 1910, 1846, 1909, 1845, 1908, 1844, 1907, 1843, 1906, 1842, 1905, 1841, 1904, 1840, 1903, 1839, 1902, 1838, 1901, 1837, 1900, 1836, 1899, 1835, 1898, 1834, 1897, 1833, 1896, 1832, 1895, 1831, 1894, 1830, 1893, 1829, 1892, 1828, 1891, 1827, 1890, 1826, 1889, 1825, 1888, 1824, 1887, 1823, 1886, 1822, 1885, 1821, 1884, 1820, 1883, 1819, 1882, 1818, 1881, 1817, 1880, 1816, 1879, 1815, 1878, 1814, 1877, 1813, 1876, 1812, 1875, 1811, 1874, 1810, 1873, 1809, 1872, 1808, 1871, 1807, 1870, 1806, 1869, 1805, 1868, 1804, 1867, 1803, 1866, 1802, 1865, 1801, 1864, 1800, 1863, 1799, 1862, 1798, 1861, 1797, 1860, 1796, 1859, 1795, 1858, 1794, 1857, 1793, 1856, 1792, 1791, 1727, 1790, 1726, 1789, 1725, 1788, 1724, 1787, 1723, 1786, 1722, 1785, 1721, 1784, 1720, 1783, 1719, 1782, 1718, 1781, 1717, 1780, 1716, 1779, 1715, 1778, 1714, 1777, 1713, 1776, 1712, 1775, 1711, 1774, 1710, 1773, 1709, 1772, 1708, 1771, 1707, 1770, 1706, 1769, 1705, 1768, 1704, 1767, 1703, 1766, 1702, 1765, 1701, 1764, 1700, 1763, 1699, 1762, 1698, 1761, 1697, 1760, 1696, 1759, 1695, 1758, 1694, 1757, 1693, 1756, 1692, 1755, 1691, 1754, 1690, 1753, 1689, 1752, 1688, 1751, 1687, 1750, 1686, 1749, 1685, 1748, 1684, 1747, 1683, 1746, 1682, 1745, 1681, 1744, 1680, 1743, 1679, 1742, 1678, 1741, 1677, 1740, 1676, 1739, 1675, 1738, 1674, 1737, 1673, 1736, 1672, 1735, 1671, 1734, 1670, 1733, 1669, 1732, 1668, 1731, 1667, 1730, 1666, 1729, 1665, 1728, 1664, 1663, 1599, 1662, 1598, 1661, 1597, 1660, 1596, 1659, 1595, 1658, 1594, 1657, 1593, 1656, 1592, 1655, 1591, 1654, 1590, 1653, 1589, 1652, 1588, 1651, 1587, 1650, 1586, 1649, 1585, 1648, 1584, 1647, 1583, 1646, 1582, 1645, 1581, 1644, 1580, 1643, 1579, 1642, 1578, 1641, 1577, 1640, 1576, 1639, 1575, 1638, 1574, 1637, 1573, 1636, 1572, 1635, 1571, 1634, 1570, 1633, 1569, 1632, 1568, 1631, 1567, 1630, 1566, 1629, 1565, 1628, 1564, 1627, 1563, 1626, 1562, 1625, 1561, 1624, 1560, 1623, 1559, 1622, 1558, 1621, 1557, 1620, 1556, 1619, 1555, 1618, 1554, 1617, 1553, 1616, 1552, 1615, 1551, 1614, 1550, 1613, 1549, 1612, 1548, 1611, 1547, 1610, 1546, 1609, 1545, 1608, 1544, 1607, 1543, 1606, 1542, 1605, 1541, 1604, 1540, 1603, 1539, 1602, 1538, 1601, 1537, 1600, 1536, 1535, 1471, 1534, 1470, 1533, 1469, 1532, 1468, 1531, 1467, 1530, 1466, 1529, 1465, 1528, 1464, 1527, 1463, 1526, 1462, 1525, 1461, 1524, 1460, 1523, 1459, 1522, 1458, 1521, 1457, 1520, 1456, 1519, 1455, 1518, 1454, 1517, 1453, 1516, 1452, 1515, 1451, 1514, 1450, 1513, 1449, 1512, 1448, 1511, 1447, 1510, 1446, 1509, 1445, 1508, 1444, 1507, 1443, 1506, 1442, 1505, 1441, 1504, 1440, 1503, 1439, 1502, 1438, 1501, 1437, 1500, 1436, 1499, 1435, 1498, 1434, 1497, 1433, 1496, 1432, 1495, 1431, 1494, 1430, 1493, 1429, 1492, 1428, 1491, 1427, 1490, 1426, 1489, 1425, 1488, 1424, 1487, 1423, 1486, 1422, 1485, 1421, 1484, 1420, 1483, 1419, 1482, 1418, 1481, 1417, 1480, 1416, 1479, 1415, 1478, 1414, 1477, 1413, 1476, 1412, 1475, 1411, 1474, 1410, 1473, 1409, 1472, 1408, 1407, 1343, 1406, 1342, 1405, 1341, 1404, 1340, 1403, 1339, 1402, 1338, 1401, 1337, 1400, 1336, 1399, 1335, 1398, 1334, 1397, 1333, 1396, 1332, 1395, 1331, 1394, 1330, 1393, 1329, 1392, 1328, 1391, 1327, 1390, 1326, 1389, 1325, 1388, 1324, 1387, 1323, 1386, 1322, 1385, 1321, 1384, 1320, 1383, 1319, 1382, 1318, 1381, 1317, 1380, 1316, 1379, 1315, 1378, 1314, 1377, 1313, 1376, 1312, 1375, 1311, 1374, 1310, 1373, 1309, 1372, 1308, 1371, 1307, 1370, 1306, 1369, 1305, 1368, 1304, 1367, 1303, 1366, 1302, 1365, 1301, 1364, 1300, 1363, 1299, 1362, 1298, 1361, 1297, 1360, 1296, 1359, 1295, 1358, 1294, 1357, 1293, 1356, 1292, 1355, 1291, 1354, 1290, 1353, 1289, 1352, 1288, 1351, 1287, 1350, 1286, 1349, 1285, 1348, 1284, 1347, 1283, 1346, 1282, 1345, 1281, 1344, 1280, 1279, 1215, 1278, 1214, 1277, 1213, 1276, 1212, 1275, 1211, 1274, 1210, 1273, 1209, 1272, 1208, 1271, 1207, 1270, 1206, 1269, 1205, 1268, 1204, 1267, 1203, 1266, 1202, 1265, 1201, 1264, 1200, 1263, 1199, 1262, 1198, 1261, 1197, 1260, 1196, 1259, 1195, 1258, 1194, 1257, 1193, 1256, 1192, 1255, 1191, 1254, 1190, 1253, 1189, 1252, 1188, 1251, 1187, 1250, 1186, 1249, 1185, 1248, 1184, 1247, 1183, 1246, 1182, 1245, 1181, 1244, 1180, 1243, 1179, 1242, 1178, 1241, 1177, 1240, 1176, 1239, 1175, 1238, 1174, 1237, 1173, 1236, 1172, 1235, 1171, 1234, 1170, 1233, 1169, 1232, 1168, 1231, 1167, 1230, 1166, 1229, 1165, 1228, 1164, 1227, 1163, 1226, 1162, 1225, 1161, 1224, 1160, 1223, 1159, 1222, 1158, 1221, 1157, 1220, 1156, 1219, 1155, 1218, 1154, 1217, 1153, 1216, 1152, 1151, 1087, 1150, 1086, 1149, 1085, 1148, 1084, 1147, 1083, 1146, 1082, 1145, 1081, 1144, 1080, 1143, 1079, 1142, 1078, 1141, 1077, 1140, 1076, 1139, 1075, 1138, 1074, 1137, 1073, 1136, 1072, 1135, 1071, 1134, 1070, 1133, 1069, 1132, 1068, 1131, 1067, 1130, 1066, 1129, 1065, 1128, 1064, 1127, 1063, 1126, 1062, 1125, 1061, 1124, 1060, 1123, 1059, 1122, 1058, 1121, 1057, 1120, 1056, 1119, 1055, 1118, 1054, 1117, 1053, 1116, 1052, 1115, 1051, 1114, 1050, 1113, 1049, 1112, 1048, 1111, 1047, 1110, 1046, 1109, 1045, 1108, 1044, 1107, 1043, 1106, 1042, 1105, 1041, 1104, 1040, 1103, 1039, 1102, 1038, 1101, 1037, 1100, 1036, 1099, 1035, 1098, 1034, 1097, 1033, 1096, 1032, 1095, 1031, 1094, 1030, 1093, 1029, 1092, 1028, 1091, 1027, 1090, 1026, 1089, 1025, 1088, 1024, 1023, 959, 1022, 958, 1021, 957, 1020, 956, 1019, 955, 1018, 954, 1017, 953, 1016, 952, 1015, 951, 1014, 950, 1013, 949, 1012, 948, 1011, 947, 1010, 946, 1009, 945, 1008, 944, 1007, 943, 1006, 942, 1005, 941, 1004, 940, 1003, 939, 1002, 938, 1001, 937, 1000, 936, 999, 935, 998, 934, 997, 933, 996, 932, 995, 931, 994, 930, 993, 929, 992, 928, 991, 927, 990, 926, 989, 925, 988, 924, 987, 923, 986, 922, 985, 921, 984, 920, 983, 919, 982, 918, 981, 917, 980, 916, 979, 915, 978, 914, 977, 913, 976, 912, 975, 911, 974, 910, 973, 909, 972, 908, 971, 907, 970, 906, 969, 905, 968, 904, 967, 903, 966, 902, 965, 901, 964, 900, 963, 899, 962, 898, 961, 897, 960, 896, 895, 831, 894, 830, 893, 829, 892, 828, 891, 827, 890, 826, 889, 825, 888, 824, 887, 823, 886, 822, 885, 821, 884, 820, 883, 819, 882, 818, 881, 817, 880, 816, 879, 815, 878, 814, 877, 813, 876, 812, 875, 811, 874, 810, 873, 809, 872, 808, 871, 807, 870, 806, 869, 805, 868, 804, 867, 803, 866, 802, 865, 801, 864, 800, 863, 799, 862, 798, 861, 797, 860, 796, 859, 795, 858, 794, 857, 793, 856, 792, 855, 791, 854, 790, 853, 789, 852, 788, 851, 787, 850, 786, 849, 785, 848, 784, 847, 783, 846, 782, 845, 781, 844, 780, 843, 779, 842, 778, 841, 777, 840, 776, 839, 775, 838, 774, 837, 773, 836, 772, 835, 771, 834, 770, 833, 769, 832, 768, 767, 703, 766, 702, 765, 701, 764, 700, 763, 699, 762, 698, 761, 697, 760, 696, 759, 695, 758, 694, 757, 693, 756, 692, 755, 691, 754, 690, 753, 689, 752, 688, 751, 687, 750, 686, 749, 685, 748, 684, 747, 683, 746, 682, 745, 681, 744, 680, 743, 679, 742, 678, 741, 677, 740, 676, 739, 675, 738, 674, 737, 673, 736, 672, 735, 671, 734, 670, 733, 669, 732, 668, 731, 667, 730, 666, 729, 665, 728, 664, 727, 663, 726, 662, 725, 661, 724, 660, 723, 659, 722, 658, 721, 657, 720, 656, 719, 655, 718, 654, 717, 653, 716, 652, 715, 651, 714, 650, 713, 649, 712, 648, 711, 647, 710, 646, 709, 645, 708, 644, 707, 643, 706, 642, 705, 641, 704, 640, 639, 575, 638, 574, 637, 573, 636, 572, 635, 571, 634, 570, 633, 569, 632, 568, 631, 567, 630, 566, 629, 565, 628, 564, 627, 563, 626, 562, 625, 561, 624, 560, 623, 559, 622, 558, 621, 557, 620, 556, 619, 555, 618, 554, 617, 553, 616, 552, 615, 551, 614, 550, 613, 549, 612, 548, 611, 547, 610, 546, 609, 545, 608, 544, 607, 543, 606, 542, 605, 541, 604, 540, 603, 539, 602, 538, 601, 537, 600, 536, 599, 535, 598, 534, 597, 533, 596, 532, 595, 531, 594, 530, 593, 529, 592, 528, 591, 527, 590, 526, 589, 525, 588, 524, 587, 523, 586, 522, 585, 521, 584, 520, 583, 519, 582, 518, 581, 517, 580, 516, 579, 515, 578, 514, 577, 513, 576, 512, 511, 447, 510, 446, 509, 445, 508, 444, 507, 443, 506, 442, 505, 441, 504, 440, 503, 439, 502, 438, 501, 437, 500, 436, 499, 435, 498, 434, 497, 433, 496, 432, 495, 431, 494, 430, 493, 429, 492, 428, 491, 427, 490, 426, 489, 425, 488, 424, 487, 423, 486, 422, 485, 421, 484, 420, 483, 419, 482, 418, 481, 417, 480, 416, 479, 415, 478, 414, 477, 413, 476, 412, 475, 411, 474, 410, 473, 409, 472, 408, 471, 407, 470, 406, 469, 405, 468, 404, 467, 403, 466, 402, 465, 401, 464, 400, 463, 399, 462, 398, 461, 397, 460, 396, 459, 395, 458, 394, 457, 393, 456, 392, 455, 391, 454, 390, 453, 389, 452, 388, 451, 387, 450, 386, 449, 385, 448, 384, 383, 319, 382, 318, 381, 317, 380, 316, 379, 315, 378, 314, 377, 313, 376, 312, 375, 311, 374, 310, 373, 309, 372, 308, 371, 307, 370, 306, 369, 305, 368, 304, 367, 303, 366, 302, 365, 301, 364, 300, 363, 299, 362, 298, 361, 297, 360, 296, 359, 295, 358, 294, 357, 293, 356, 292, 355, 291, 354, 290, 353, 289, 352, 288, 351, 287, 350, 286, 349, 285, 348, 284, 347, 283, 346, 282, 345, 281, 344, 280, 343, 279, 342, 278, 341, 277, 340, 276, 339, 275, 338, 274, 337, 273, 336, 272, 335, 271, 334, 270, 333, 269, 332, 268, 331, 267, 330, 266, 329, 265, 328, 264, 327, 263, 326, 262, 325, 261, 324, 260, 323, 259, 322, 258, 321, 257, 320, 256, 255, 191, 254, 190, 253, 189, 252, 188, 251, 187, 250, 186, 249, 185, 248, 184, 247, 183, 246, 182, 245, 181, 244, 180, 243, 179, 242, 178, 241, 177, 240, 176, 239, 175, 238, 174, 237, 173, 236, 172, 235, 171, 234, 170, 233, 169, 232, 168, 231, 167, 230, 166, 229, 165, 228, 164, 227, 163, 226, 162, 225, 161, 224, 160, 223, 159, 222, 158, 221, 157, 220, 156, 219, 155, 218, 154, 217, 153, 216, 152, 215, 151, 214, 150, 213, 149, 212, 148, 211, 147, 210, 146, 209, 145, 208, 144, 207, 143, 206, 142, 205, 141, 204, 140, 203, 139, 202, 138, 201, 137, 200, 136, 199, 135, 198, 134, 197, 133, 196, 132, 195, 131, 194, 130, 193, 129, 192, 128, 127, 63, 126, 62, 125, 61, 124, 60, 123, 59, 122, 58, 121, 57, 120, 56, 119, 55, 118, 54, 117, 53, 116, 52, 115, 51, 114, 50, 113, 49, 112, 48, 111, 47, 110, 46, 109, 45, 108, 44, 107, 43, 106, 42, 105, 41, 104, 40, 103, 39, 102, 38, 101, 37, 100, 36, 99, 35, 98, 34, 97, 33, 96, 32, 95, 31, 94, 30, 93, 29, 92, 28, 91, 27, 90, 26, 89, 25, 88, 24, 87, 23, 86, 22, 85, 21, 84, 20, 83, 19, 82, 18, 81, 17, 80, 16, 79, 15, 78, 14, 77, 13, 76, 12, 75, 11, 74, 10, 73, 9, 72, 8, 71, 7, 70, 6, 69, 5, 68, 4, 67, 3, 66, 2, 65, 1, 64, 0};
    //==============================================================================
    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR(SG323AudioProcessor)
};