/*
  libdal - a device abstraction library
  Copyright (C) 2001-2004 Yury Umanets <torque@ukrpost.net>.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.
                                                                                                 
  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.
                                                                                                 
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <dal/dal.h>

static int power_of_two(unsigned long value) {
	return (value & -value) == value;
}

void *libdal_malloc(size_t size) {
	void *mem;

	mem = (void *)malloc(size);
	if (!mem) {
		fprintf(stderr, "Out of memory.\n");
		return NULL;
	}

	return mem;
}

int libdal_realloc(void **old, size_t size) {
	void *mem;

	mem = (void *)realloc(*old, size);
	if (!mem) {
		fprintf(stderr, "Out of memory.\n");
		return 0;
	}
	*old = mem;
	return 1;
}

void libdal_free(void* ptr) {
	free (ptr);
}

dal_t *dal_create(struct dal_ops *ops, const void *dev, 
		  size_t blocksize, int flags, void *data) 
{
	dal_t *dal;
    
	if (!ops) return NULL;
    
	if (!power_of_two(blocksize)) {
		fprintf(stderr, "Block size isn't power of two.\n");
		return NULL;
	}	
    
	if (!(dal = (dal_t *)libdal_malloc(sizeof(*dal))))
		return NULL;

	memset(dal, 0, sizeof(*dal));
    
	dal->ops = ops;
	dal->dev = dev;
	dal->data = data;
	dal->flags = flags;
	dal->blocksize = blocksize;
	dal->len = 0;
	
	return dal;
}

void dal_free(dal_t *dal) {
	
	if (!dal) return;
    
	dal->ops = NULL;
	dal->dev = NULL;
	dal->data = NULL;
	libdal_free(dal);
}

int dal_set_block_size(dal_t *dal, size_t blocksize) {

	if (!dal) return 0;
    
	if (!power_of_two(blocksize))
		return 0;
    
	dal->blocksize = blocksize;
	dal_realize(dal);
	
	return 1;
}

size_t dal_block_size(dal_t *dal) {

	if (!dal) return 0;

	return dal->blocksize;
}

void dal_realize(dal_t *dal) {
    
	if (!dal) return;
	
	dal->len = 0;
}

blk_t dal_len(dal_t *dal) {
    
	if (!dal || !dal->ops->len) return 0;

	if (!dal->len)
		dal->len = dal->ops->len(dal);

	return dal->len;
}

int dal_read(dal_t *dal, void *buff, blk_t block, blk_t count) {

	if (!dal) return 0;

	if (dal->ops->read)
		return dal->ops->read(dal, buff, block, count);
	
	return 0;
}

int dal_write(dal_t *dal, void *buff, blk_t block, blk_t count) {

	if (!dal) return 0;
    
	if (dal->ops->write)
		return dal->ops->write(dal, buff, block, count);
    	
	return 0;
}
	
int dal_sync(dal_t *dal) {

	if (!dal) return 0;

	if (dal->ops->sync)
		return dal->ops->sync(dal);
	
	return 0;	
}

int dal_flags(dal_t *dal) {

	if (!dal) return 0;

	if (dal->ops->flags)
		return dal->ops->flags(dal);
	
	return 0;
}

int dal_equals(dal_t *dal1, dal_t *dal2) {
	
	if (!dal1 || !dal2) return 0;

	if (dal1->ops->equals)
		return dal1->ops->equals(dal1, dal2);
	
	return 0;	
}

int dal_stat(dal_t *dal, struct stat* st) {

	if (!dal) return 0;
    
	if (dal->ops->stat)
		return dal->ops->stat(dal, st);
	
	return 0;
}

dev_t dal_dev(dal_t *dal) {
    
	if (dal->ops->dev)
		return (dev_t)dal->ops->dev(dal);
	
	return 0;	
}

