/*
    The original data of this file is from unihan database by Unicode
    Licensed under https://www.unicode.org/license.txt

    License can be found under LICENSE file in this directory

    Data was modified for use with this project, modification:
        extracted kMandarin and kHanyuPinyin field
        converted to a map structure
        removed accents on letter
        mapped "ü" to both "u" and "v" (except 'lu' and 'nu' which exist and pronounce differently)

    See also: Unicode® Standard Annex #38 (https://www.unicode.org/reports/tr38/)
*/
#ifndef PINYIN_DATA_H
#define PINYIN_DATA_H

#include <map>
#include <string>

inline const std::map<std::u32string, std::u32string> pinyin_data = {
    {U"a", U"吖呵啊嗄腌錒锕阿𠼞𥥩𨉚"},
    {U"ai", U"㕌㗒㗨㘷㝵㝶㢊㤅㱯㶼㾢㾨㿄䀳䅬䑂䔇䔽䝽䠹䨠䬵䶣乂乃伌佁僾儗凒剴厓叆呃呆哀哎唉啀嗌嗳嘊噫噯埃堨塧壒奇娭娾嫒嬡嵦愛懓懝挨捱敱敳昹暧曖欬欸毐溰溾濭烠焥爱獃瑷璦癌皑皚皧瞹矮砹硋碍磑礙絠艾蔼薆藹諰謁譪譺賹躷醷銰鎄鑀锿閡阨阸隑隘霭靄靉餲馤騃鯦鱫鴱𠊎𠳳𡁍𡉓𡟓𡰽𡶃𢟪𢟰𢣏𢣕𢰇𣋞𣎴𣜬𣝅𣤃𣩄𣩱𤢵𤸖𤸳𤻢𥡽𥤦𥴨𦗍𦗐𦤦𦥂𦥈𦩴𧏹𧓁𧡋𧪚𧰿𧵨𨶂𩂒𩂢𩈋𩪂𩫇𩮖𪇈𪕭𫘤"},
    {U"an", U"㛺㜝㞄㟁㡋㫨㭺㱘㸩㽢䀂䅁䅖䎨䜙䢿䤃䤶䬓䮗䯥侒俺儑匼厂厈咹唵啽垵垾埯堓婩媕安屽岸峖干广庵按揞晻暗案桉氨洝犴玵痷盒盦盫碪罯胺腤荌菴萻葊蓭裺誝諳谙豻貋遃鉗銨錌铵闇陰隌雸鞌鞍韽頇頞馣鮟鴳鵪鶕鹌黯𠉬𠪚𠰑𠽪𠿑𡎑𡪁𡪙𡯏𡹼𡽜𢰍𣆛𣚖𣣚𣵱𣽥𤃷𤜁𤞿𤟉𥏮𥦍𥳬𦺽𧖮𧩸𧫥𧫧𧮍𨲊𩅝𩈴𩓤𩭢𩹎𩽾𪁟𪒠𪘒"},
    {U"ang", U"㭿㼜䀚䇦䒢䩕䭹䭺仰卬岇昂昻枊盎肮腌醃醠骯𠵫𠹃𡕉𡵙𢓋𣉗𣖮𤭒𦫫𨋕𩉰𩑝𩔘𩜟"},
    {U"ao", U"㑃㕭㘬㘭㜜㜩㟼㠂㠗㤇㥿㩠㿰䆟䉛䐿䚫䜒䥝䦋䫜䫨䮯䯠䴈䵅傲凹厫嗷嗸噢囂坳垇墺墽奡奥奧媪媼嫯岙岰嶅嶴廒慠懊扷抝拗摮擙敖柪梎棍泑滶澆澳熝熬燠爊獒獓璈眑磝磽礉翱翶翺聱芺蔜蝹螯袄襖謷謸軪遨郩鏊鏕鏖镺隞隩驁骜鰲鳌鴁鴢鷔鼇鿫𡊛𡏼𡒃𢁱𢕟𢧴𢳆𣊁𣷫𤇯𤏶𤺾𥂢𥑑𥜌𦪈𦽀𧅃𧨲𨱧𩈏𩑍𩑤𩕀𩘮𩟇𩣻𩥊𩮯𩱏𩼈𪁾𪃨𪉑"},
    {U"ba", U"㔜㞎㧊㭛㭭㶚㸭㺴㿬䃻䆉䇑䈈䎬䎱䟦䥯䩗䩻䮂䯲䰾䳊䶕丷仈伯八叐叭吧哱哵坝坺垻墢壩夿妭岜峇巴巼弝扒把抜抪拔捌捭朳杷柭欛湃灞炦爸犮玐疤癹皅矲笆粑紦罢罷羓耙胈芭茇茷菝萆蚆覇詙豝跁跋軷釛釟鈀钯霸靶颰魃魞鮁鮊鲃鲅鲌鼥𠐌𠛋𠵺𡚭𢃳𢇷𢠭𢺞𢻷𣬶𣬷𤜕𤜱𤣸𤤒𥎱𥐙𥝧𥥛𦓧𦫙𦳺𧎱𧲧𧺡𧺺𧿏𨊹𩃴𩊤𩖣𩖽𩙥𩚥𩠀𩡩𩨜𩹏𩽷"},
    {U"bai", U"㓦㔥㗑㠔㧳㿟䒔䙓䠋䢙䪹䳆佰唄啡庍扒拜拝挀捭排掰摆擘擺敗柏栢派猈瓸白百稗竡粨粺絔罷薜薭襬贁败鞁鞴韛𠟻𠫛𠮠𡏯𡭢𢈕𢛞𣧙𣺽𤁣𤙅𤽹𥬝𦣺𦣻𦩋𦳞𨃅𩋂𩎻𩏞𩑢𪡈"},
    {U"ban", U"㚘㢖㪵㻞䃑䈲䉽䛀䬳䰉並伴分办半卑坂坢埿姅岅彬怑扮扳拌搫搬攽斑斒昄朌板柈湴版班瓣瓪瘢癍秚籓粄絆绊舨般蝂螁螌褩覂豳賁跘辦辨辬辯鈑鉡钣闆阪靽頒颁魬鳻𠔯𠚼𠦒𠧫𠯘𠺚𡯘𢲔𢴬𣪂𤡰𤦦𤫫𤳖𥷁𥹓𦎊𦙹𦝤𧇥𧌿𧿨𨐦𨐱𨐾𨭉𩔮𩢔𩭧𩿉𪄕𪉒𪒋"},
    {U"bang", U"㙃㨍㭋㮄㿶䂜䎧䖫䧛䩷䰷並傍嗙垹埲塝嫎帮幇幚幫彭徬捠搒旁梆棒棓榜浜牓玤硥磅稖紡綁縍绑膀艕蒡蚄蚌蛖蜯螃謗谤邦邫鎊镑鞤騯髈𠨵𠬣𠲑𠳐𡽲𢁏𢄎𢜗𢪋𢮏𢶶𢸌𣘙𣮡𣮧𤚰𤱵𥀂𦰥𦾭𨢐𩋮𩍗𩦠𩮗"},
    {U"bao", U"㙅㙸㫧㲒㵡㻄㿺䈏䎂䤖䥤䨌䨔䪨䭋䯽䳈䳰䴐佨保儤刨剥勹勽包呆嘐嚗堡堢報媬嫑孢宝宲寚寳寶忁怉报抱暴曓枹瀑炮煲爆珤砲窇笣緥胞苞苴菢葆蕔薄藵虣蚫袌袍裒褒褓襃豹賲趵鉋鑤铇闁雹靌靤飹飽饱駂骲髱鮑鲍鳵鴇鸨齙龅𠅬𠣒𠣺𠤏𠹕𡂟𡉩𡧖𡶄𢼌𣋳𣭀𤔣𤝧𤞥𤿈𥄹𥭓𦡕𦢊𧘑𧝘𧭤𧵢𨇅𨚔𨠖𨰦𨰻𩊅𩍂𩛞𩬽𩭼𩾡𩿓𪏶"},
    {U"bei", U"㔗㔨㗗㛝㣁㤳㪍㫲㭭㰆㶔㷶㸢㸬㸽㻗㽡㾱䋳䎱䔒䚜䟛䟺䡶䣙䥯䩀䰽䱝俻俾倍偝偹備僃北卑呗哱唄垻埤备孛怫悖悲惫愂憊拔揹昁杮杯柸桮梖棑棓椑波焙牬犕狈狽珼琲盃碑碚禙箄糒背苝茀菩萆萯葡蓓藣蛽蜚被褙襬誖諀貝贝跋軰輩辈邶郥鄁鉳鋇錍鐴鐾钡陂鞁鞞鞴骳鵯鹎𠋭𠐡𠢥𡋭𡥞𡰯𢂏𢃍𢴾𢻵𣎵𣖾𣝁𣬍𣬪𤜲𤰈𤳦𤵛𤷁𤷒𤹲𤿒𤿾𥏓𥶓𦈧𦈶𦩖𦮷𦾙𧉥𧋲𧶙𧼠𨚍𩇩𩔹𩖠𩚾𬇙"},
    {U"ben", U"㡷㤓㨧㮥㮺䬱体倴喯坋坌夯夲奔奙捹撪本栟桳楍泍渀炃燌犇獖畚笨翉苯蟦賁贲軬輽逩錛鐼锛𡗢𡗲𣄏𣳰𣴞𥢊𦯀𨋒𩣺𩧼𩰶𪊜𪎝𪑖"},
    {U"beng", U"㑟㔙㧍㱶㷯㼞䋽䑫䙀䨜䨻䩬䭰䳞伻俸傍傰唪嗙嘣埄埲堋塴奟崩嵭平抨揼搒旁榜泵漨熢琣琫甏甭痭祊絣綳繃绷菶蚌跰蹦迸逬錋鏰镚閍鞛𠜳𠡮𡎾𡡈𡶤𡾛𢆸𢉁𢏳𢐒𣂤𣨥𤙾𤡭𤫬𥀂𥖗𥛱𥞩𥦜𦂌𦅈𦝷𦺑𧑑𧚭𧩱𧻓𨆊𨓁𨕧𨸂𨹹𨻱𩂦𩊌𩑚𩗴𪔑"},
    {U"bi", U"㓖㘠㘩㙄㚰㠲㡀㡙㢰㢶㢸㧙㪏㪤㮿㯅㯇㱸㳼㵒㵥㻫㻶㿫㿰䀣䁹䃾䄶䉾䊧䋔䌘䌟䎵䏟䏢䏶䕗䖩䘡䚜䟆䟤䠋䣥䧗䨆䩛䪐䫁䫾䬛䮠䮡䯗䵄䵗仳佊佖佛俾偪匕卑吡咇哔啚嗶坒埤堛壁夶奰妣妼娝婢媲嬖嬶屄崥币幅幣幤庇庳廦弊弻弼彃彼必怭怶悂愊愎拂捭敝斃旇服朼枇枈柀柲梐椑楅榌檗殍比毕毖毙毴沘泌波湢滗滭潷濞煏熚狴獘獙珌璧畀畁畐畢疕疪痹痺皀皕睤瞥碧祕禆秕秘稫笓笔筆筚箄箅箆篦篳粃粊紕紴綼縪繴罷罼翍聛肥肶肸胇脾腷臂舭芘苾茀荜荸萆萞蓖蓽蔽薜蘗虑蜌螕袐被裨襅襞襣觱詖诐豍貏貱費賁贔赑跛跸踾蹕躃躄辟逼避邲鄙鄨鄪鈚鉍錍鎞鏎鐴铋閇閈閉閟闭陂陛陴鞁鞞鞸韠飶饆馝馥駜驆髀髲魓魮鮅鮩鰏鲾鴓鵖鶝鷝鷩鸊鼊鼻𠈺𠋯𠐌𠓷𠛡𠡂𠦈𠧅𠨘𠩿𠬈𠮃𠽩𡚁𡛗𡠚𡳄𡻞𡽶𢁽𢅩𢍁𢐦𢖬𢘍𢞗𢟵𢡅𢩒𢲾𢳋𢴩𣁉𣁢𣋹𣔓𣕾𣘥𣚡𣝍𣢠𣥣𣦇𣦢𣩩𣭤𣮐𣯴𣴨𤂀𤅹𤐙𤗚𤙞𤜻𤝸𤠺𤡝𤢣𤵘𤷒𤹝𤹦𤻖𤽊𥆯𥈗𥏠𥛘𥟗𥢦𥳆𥴬𥷑𦀘𦂖𦑞𦔆𦠞𦤫𦭼𦯛𦰙𦱔𦳈𦸣𧏻𧒀𧓄𧓎𧝬𧤃𧥑𧫤𧲜𧳠𧼱𨅗𨋥𨋩𨐨𨚍𨚓𨟵𨠔𨲋𨳲𨵰𨸼𨻀𨻼𩉫𩊰𩏂𩑻𩣚𩧿𩪖𩪧𩭧𩯃𩲢𩾳𪋜𪌄𪍪𪏺𪐄"},
    {U"bian", U"㝸㣐㦚㭓㲢㳎㳒㴜㵷㺕㺹䁵䉸䐔䒪䛒䟍䡢䪻便匾卞变変封峅弁徧忭惼扁抃拚揙昪汳汴炞煸牑猵獱玣甂疺砭碥稨稹窆笾箯籩糄編緶缏编臱艑苄萹藊蝙褊覍覵變豍貶贬辡辧辨辩辫辮辯边辺遍邉邊邲釆鍽閞鞕鞭頨鯾鯿鳊鴘鶣𠐈𠑟𠓫𠪂𠭹𠯴𠷖𡈯𡊅𡬯𡬲𡬸𢩟𢬵𢭥𢴂𢻶𣈠𣝜𣩀𣪭𣸇𤀫𤀲𤄺𤺇𤻶𥍚𥣝𥣰𥿋𦇭𦉙𦟣𦽟𧩰𧸘𨐾𨖠𨖾𨚕𨥧𨧕𨩫𨳲𩩯𩰍𪉱𪏗𪓍𪖯"},
    {U"biang", U"𰻝𰻞"},
    {U"biao", U"㟽㠒㧼㯹㶾䁃䁭䅺䔸䙳䞄䮽俵僄儦剽墂婊嫖幖彪摽杓标標檦淲滮漂瀌灬熛爂猋瘭磦穮篻脿膔膘臕苞蔈藨表裱褾諘謤贆錶鏖鏢鑣镖镳颩颮颷飆飇飈飊飑飙飚驃驫骉骠髟鰾鳔麃𠔂𠚠𠬪𢅚𢒯𢿏𣄠𤂆𤆀𤐫𥘤𥲦𦔗𦔩𦠎𦾑𧝪𧥍𧳀𧴎𧴕𨭚𩙪𩪊𩴩𩽁"},
    {U"bie", U"㔡㢼㧙㭭㿜䇷䋢䌘䏟䘷䟤䠥䭱䳤別别咇彆徶憋扒拔捌撇柭柲瘪癟秘穪苾莂蔽虌蛂蟞襒蹩鱉鳖鼈龞𠍯𡐞𡘴𡙀𡙪𡷘𢆣𢐳𢛎𢠳𣇢𣊶𤉤𤷗𤺓𤾵𥞲𥡁𧆊𧌽𧝬𧧸𧿥𨂅𨒜𩓝𩠻𩡟𩦉𩵛𩸁𪂟𪐆"},
    {U"bin", U"㟗㯽㻞䐔䚔䧬䨈份傧儐宾彬摈擯攽斌梹椕槟檳殡殯氞汃浜滨濒濱濵瀕玢瑸璸砏繽缤膑臏虨訜豩豳賓賔贇邠鑌镔霦頻顮髌髕髩鬂鬓鬢𠴇𡦆𡦻𡧍𡧼𢲰𣉮𣢏𣰨𥃰𧷟𧸈𨐰𨽗𩆱𩴱𪇕"},
    {U"bing", U"㓈㨀䔊䗒䴵丙並仌仒併倂偋傡兵冫冰垪寎屏平并幷庰怲抦拼掤摒昞昺枋柄栟栤梹棅檳氷炳燹琕病痭癛眪禀秉稟窉竝絣綆苪蛃誁跰邴鈵鉼鋲陃靐鞆鞞鞸餅餠饼鮩𠊧𠒝𠛥𠱛𡇤𡖛𡚛𡲍𡹾𢆩𢊜𢎴𢔧𣅍𣦪𣰜𥖬𥲂𦡻𦼹𦿅𨆱𨋲𨹗𩊖𩋒𩏂𩬝𩮟𩶁𪑰"},
    {U"bo", U"㗘㝿㞈㟑㩧㩭㪍㬍㬥㬧㴾㵨㶿㹀㼎㼟㼣䂍䃗䄶䊿䌟䍸䑈䗚䙏䝛䞳䟔䟛䟮䢌䢪䥬䪇䪬䬪䭦䭯䮀䮡䯋䰊䳁䵗䶈亳仢伯佛侼僠僰剝剥勃募博卜哱啵噃嚗壆妭孛孹嶓帗帛彴怕愽懪拍拔拨挬搏撥播擗擘暴服柏柭桲榑檗檘欂殕泊波浡淿渤溊溥潑潘瀑煿爆牔犦犻狛猼玻瓝瓟番癶癷發白百皪盋砵碆磻礡礴秡穛箔箥簙簸簿糪紴缽肑胉脖膊般舶艊艴苩茀茷菠菩萡葧蒲蔔蔢蕃薄薜蘖蘗蚾袚袯袰袹襎襏襮詙譒豰趵跑跛踣蹳郣鈸鉑鉢鋍鎛鑮钵钹铂镈餑餺饽馎馛馞駁駮驋驳髆髉魄鮁鮊鱍鵓鹁𠧛𠮭𠱀𠴸𠷺𠸳𠺣𡀖𡅂𡋯𡯳𡯷𢂍𢐾𢠺𢣞𢩞𢫯𢺽𣋵𣛓𣧧𣭷𣽡𤃵𤒔𤕝𤗳𤗺𤚽𤜧𤶋𤾝𤿑𥜖𥭖𥮯𥴮𥸥𥸨𥹸𦃙𦈞𦋉𦢊𦤚𦤣𦯉𦰬𦲱𦼭𦽮𧇚𧘟𧙄𧟱𧲯𧿲𨈩𨍭𨏫𨨏𨭂𩃶𩄿𩈔𩌏𩍿𩏯𩓐𩗀𩗒𩗓𩙦𩜥𩟕𩣡𩧯𩬸𩯌𩱚𩷚𩽛𪌰𪍡𪓜𪙍𪚷𬭛"},
    {U"bu", U"㘵㙛㚴㨐㱛㳍㻉㾟䀯䊇䋠䍌䏽䑰䒀䝵䪁䪔䬏䳝䴺不佈僕勏卜卟吥咘哺喸埔埗埠堡婄尃峬布庯廍怖悑抪拊捕捗捬撲晡柨步歨歩溥瓿秿箁篰簿荹蔀薄补補誧踄輹轐逋部郶醭鈈鈽钚钸附陠鞴餔餢鯆鳪鵏鸔𠘁𠚉𠜙𡡐𢁻𢇴𢼹𣱶𤚵𤣰𤸵𥃨𥑢𥣌𥪀𥳖𥸨𥹴𥻞𧉩𧻷𨋞𨛒𨴪𩅇𩊬𩊶𩌏𩏮𩏵𩢕𩣝𩯏𩶉𩷖𩺼𩻗𪇰𫗦𬷕"},
    {U"ca", U"䃰䌨䟃䵽傪嚓囃拆擦攃磣礤礸蔡遪𤄖𥗭𥩝𨆾𨺭𪊗"},
    {U"cai", U"㒲㥒䌨䌽䐆䞗䟀䠕䣋䰂䴭倸偲啋埰婇寀彩才扐採揌材棌毝猜睬綵縩纔菜蔡裁財财跴踩采𡣮𢎂𤁱𤚀𤝭𤟖𤷕𦬁𧀊𧵤𨙴𨯓𩁞𩧇𪇭"},
    {U"can", U"㛑㜗㜞㣓㥇㦧㨻㱚㻮㽩㿊䅟䉔䍼䏼䗝䗞䘉䙁䛹䝳䟃䣟䱗䳻傪儏参參叄叅喰嘇噆囋嬠孱嵾惨惭慘慙慚憯戔掺摲摻朁残殘淺湌澯灿燦爘璨穇篸粲縿薒蚕蝅蠶蠺謲蹔鏒飡飱餐驂骖鰺黪黲𠠋𠡡𠫭𡆮𡛝𡡖𢦸𢧮𢾃𣦼𣶡𣻬𤅒𥂥𥠩𥢽𥮾𥹛𦪜𦪫𦺐𧅀𧓩𨅔𨞷𨲱𩀧𩈻𩈼𩛳𩝖𩞀𩟒𩯞𪆶"},
    {U"cang", U"㵴㶓䅮䢢仓仺伧倉傖凔匨嵢欌沧滄濸獊瑲篬臧舱艙苍蒼蔵藏螥賶鑶鶬鸧𠥐𡽴𡾻𡿄𤚬𦾝𨤃𩀞𩕹𩝞"},
    {U"cao", U"㜖㯥㿷䄚䎭䏆䐬䒃䒑傮嘈屮嶆愺慅慒慥懆撡操曹曺槽漕澡糙肏艚艸艹草蓸螬褿襙造鄵鏪鐰騲鼜𠀊𠹊𡮦𣈅𣉿𤒕𤡐𤵥𥕢𥲍𦋿𨎝𩞄𩠎𩫥"},
    {U"ce", U"㥽㨲㩍䇲䈟䊂䔴䜺侧側冊册厕厠墄嫧幘廁恻惻憡拺敇栅测測畟笧策筞筴箣簎粣荝萗萴蓛赦齰𡍫𢷾𢿸𣌧𥠉𥬰𥰡𥳯𦔎𦣧𦵪𧵡𨶨𩒄"},
    {U"cei", U"𤭢"},
    {U"cen", U"㞤㞥䅾䤁䨙䫈䲋參岑嵾梣汵涔硶穇笒篸𣡎𦊃𨁊𨥣𨱼𩅨𩅮𩻛"},
    {U"ceng", U"㣒㬝䁬䉕僧噌增层層嶒曽曾橧竲繒蹭鄫驓𠟂𡃆𡪠𡾓𢅋𤛢𦠇𧲅"},
    {U"cha", U"㛼㜘㡸㢉㢒㣾㤞㪯㫅㮑㲼㳗䁟䂳䅊䒲䓭䕓䛽䟕䠡䡨䤩䶪仛侘偛刹叉嗏嚓土垞奼姹察岎岔嵖差扠扱挿捈捷接插揷搽摖斜杈查梌楂槎檫汊猹疀碴秅紁肞臿艖芆苴茬茶荖荼衩褨訍詧詫诧蹅釵鉏銟鍤鎈鑔锸镲靫餷馇𠝞𠞊𠽹𡋨𡌚𡝐𡝙𡨀𡵌𢔣𢘹𢣼𢭅𣃍𣆗𣍏𣘤𣘻𣱱𤜫𤜯𤞠𤳅𤳵𤵾𤶠𥃀𥌀𥑥𥥸𥫢𥻗𦉆𦑈𦑣𦛝𦝥𦦘𦦜𦦱𦳘𧠈𧩫𧫗𧶵𨀸𨃓𨆇𨙳𨩨𨪺𨼑𩝟𩟔𩴳𪑂𪑨𪒼𪘾"},
    {U"chai", U"㑪㦅㳗㼮㾹䐤䓱䘍䜺䞗侪儕喍囆差扠拆搓查柴犲瘥祡芆茈茝蔕虿蠆袃訍豺釵钗齜𠞊𡟭𡺵𢹓𤝔𤞗𤠌𥐟𦏱𦐰𦑏𧀱𧒨𧓽𧔴𧕧𧪘𧸿𨌅𩑐"},
    {U"chan", U"㙴㙻㚲㢆㢟㤐㦃㨻㬄㯆㰫㶣㸥㹌㹽㺗㺥䀡䂁䊲䐮䑎䜛䠨䡲䣑䤘䤫䥀䦓䦲䧯䩥䩶䪜䫮䱿䴼䵐丳产亶佔僝僤儃儳兔冁刬剗剷劖厘啴單嘽嚵囅墠壥婵嬋嬗孱嵼嶄巉幝幨廛忏憚懴懺搀摌摲摻撣攙斺旵梴棎榐欃毚沾浐湹滻漸潹潺澶瀍瀺灛煘燀獑產産硟磛禅禪簅緂緾繟繵纏纒缠羼脠艬苫蒇蕆蝉螹蟬蟺蟾袩裧襜襝襢覘觇誗諂譂讇讒讖谄谗蹍躔辴辿鄽酁醦鉆鋋鋓鏟鑱铲镡镵閳闡阐韂顫颤饞馋骣𠁷𠋷𠐩𠑆𠑑𠑡𠟉𠣄𠹖𡍌𡎻𡖞𡖤𡝫𡮿𡶴𢁧𢌚𢥋𢱟𢷹𢺟𢽝𣃘𣔵𣔻𣤱𤎥𤗻𤚍𤪮𤮭𤯥𤴿𤸦𤼋𥊓𥧑𥭔𦆀𦈎𦝟𦢙𦸰𧈪𧐲𧓋𧕃𧠛𧥓𧨗𧬦𧴃𧾍𧾡𨄉𨇝𨇦𨊝𨔢𨩪𨪑𨬖𨮻𨲵𨳂𨵍𨷭𨼒𨽊𩍲𩖌𩝚𩟶𩥮𩬑𩮏𩽝𪏁𪏂𪏋𪏦𪓄𪖎𪗂𪙞𪚃𬊤𬳲"},
    {U"chang", U"㙊㦂㫤䅛䗅䗉䠀䠆䩨䮖䯴䱽仧仩伥倀倘倡偿僘償儻兏厂厰唱嘗嚐场場塲娼嫦尚尝常廠徜怅悵惝敞昌昶晿暢棖椙氅淌淐焻猖玚琩瑒瑺瓺甞畅畼肠脹腸膓苌菖萇蟐裮裳誯鋹鋿錩鏛锠長镸閶闛阊韔鬯鯧鱨鲳鲿鼚𠙁𠚊𡭿𢁝𢗺𢢌𤢄𤽣𤿼𥇔𥋤𥗊𥟚𥠴𥫅𦰱𦼳𧀄𨎖𨣛𨱮𨷇𩲹𩳤𪁺𪂇𪄹𪉨𬬮"},
    {U"chao", U"㥮㶤㷅䄻䎐䏚䜈䟁䫸䫿䮓䰫䲃䳂仦仯劋勦吵嘮嘲巐巢巣弨怊抄摷晁朝槱樔欩漅潮濤炒焣焯煼牊眧窲粆紹綽縐罺耖觘訬謅謿超趠趫轈鄛鈔钞麨鼂鼌𠰉𡏮𡡊𡯴𡻝𡼼𢁾𣰩𤙴𤰬𥕘𥧇𥲀𥿷𦙧𦨖𦸛𦾱𧧠𨄓𨌬𨗡𨢪𨨚𨴡𩈎𩖥𩱈𩱦𪌬𪍇𪍈𪍑𪎊"},
    {U"che", U"㒤㔭㤴㥉㨋㬚㱌㳧㵔㾝㿭䁤䊞䋲䒆䚢䛸䜠䞣䧪䰩伡俥偖勶呫唓喢坼奲宅尺屮彻徹扯拆掣揊摰撤撦斥池澈烢烲爡瞮砗硨硩聅莗蛼詀謵車车迠頙𡷖𢇛𢊏𢛁𣨊𤊿𤕛𤖷𤗙𤥭𤹞𥯥𥿊𦈈𦓍𦛖𧙝𧼳𨀠𨅊𨹡𩂻𩎚𩒷𩗙𩴟𪎺"},
    {U"chen", U"㕴㥲㧱㫳㲀㴴㽸䀼䆣䏹䐜䑣䒞䜟䞋䟢䠳䢅䢈䢻䣅䤟䫈䫖侲傖儭嗔嚫堪填塵墋夦宸尘帘忱愖抻捵揨敐晨曟枕桭棧棽榇樄櫬沈沉湛瀋煁琛疢疹瘎瘨眈瞋硶碜磣稱綝縝肜胂臣茞莀莐蔯薼螴衬襯訦諃諶謓謲讖谌谶賝贂趁趂趻跈踸軙辰迧郴醦鈂鍖闖陈陳霃鷐麎齓齔龀𠋆𡫁𢆺𢎕𣀍𣞟𤘣𤝚𤟸𤡳𤹛𥉜𥔪𥗒𥞁𥫹𦁄𦁟𧆂𧡬𧨡𧭼𧿒𨑌𨣔𨻖𨼌𨼐𨼤𩂉𩅌𩇖𩞀𪁏𪁧𪕁𬘭"},
    {U"cheng", U"㐼㓌㛵㞼㨃㲂㼩䁎䄇䆑䆵䇸䔲䕝䗀䚘䞓䟓䟫䧕䫆䮪丞乗乘侱倀偁傖僜净呈嗆噌城埕埩堘塍塖娍宬峸嵊庱徎徵悜惩憆憕懲成承挰掁搶摚撐撑撜敞晟朾枨柽棖棦棱椉槍樘橕橖橙檉檙氶泟洆浧浾淨湞溗澂澄瀓爯牚珵珹琤瑲畻盛盯睈睖瞠矃碀秤称程稱穪窚竀筬絾緽罉脀脭荿虰蛏蟶裎觕誠诚赪赬趟踜蹚逞郕郢酲醒鋮鎗鏳鏿鐣鐺铖阷靗頳饓騁騬骋鯎黨𠏧𠓷𠕠𠳽𡝚𡤿𡽊𢐞𢔤𢜻𢜼𢟊𢻓𢾊𢿦𢿧𣀏𣥺𣥻𤆁𤕀𤗓𤿣𥢲𥥱𦓬𦦢𧡈𧯒𧶔𧶸𧷒𧹓𨁎𨅝𨋬𨌤𨎋𨞐𨭃𨹚𩁷𩇜𩙆𩛦𩞦𩠏𩤙𩨆𩫹𩯎𪁋𬲜"},
    {U"chi", U"㑜㒆㓼㓾㔑㘜㙜㞴㞿㡿㢁㢋㢮㥡㩽㮛㰞㱀㶴㷰㺈㽚㾹䀸䄜䇪䊼䏧䑛䙙䜄䜉䜵䜻䞾䟷䠠䤲䧝䪧䮈䮻䰡䳵䶔䶵佁侈侙俿傺剟勅勑匙卙卶叱叺吃呎呬呹哆哧啻喜喫嗤嘯噄噭坻垑墀奓她妛媸尺岻弛彨彲彳徲恜恥慗慸憏懘扡抬抶拆拖拸持捇提搋摛摴攡敕斥杘柅柢樆欼歭歯汖池沱沶治泜湁滯漦灻炽烾熾狋瓻痓痴痸瘈瘛癡眙眵瞝硳祇离移穉竾笞筂箈箎篪粚紕絺翄翅翤翨耛耻肔胝胣胵脪腟芪茌茬荎莉菭蚇蚩蚳蛇蝭螭袲袳裭褫訵誀誃誺謘謻豉貾赤赿趍趐趩跅跢跮踅踟踶軧迟迡迣遅遟遫遲邌郗鉓鉹銐鍉離雴飭飾饎饬馳騺驪驰魑鳷鴟鵄鵣鶒鶗鷘鸱麶黐齒齝齣齿𠛔𠝨𠞄𠞩𠧚𠧵𠩺𠭋𠮟𠻟𡂙𡉪𡌞𡎍𡖳𡚨𡣀𡳭𡼁𢂝𢇕𢇛𢓎𢔊𢜳𢣕𢨒𢱟𢴨𣆚𣉄𣐃𣙰𣚩𣣷𣤌𣤩𣲋𣲑𣹡𤆍𤈔𤟆𤡏𤡢𤰠𤵬𤸪𥄇𥚚𥛚𥭘𥱻𦂋𦆤𦇲𦎚𦏸𦏿𦐁𦐉𦑡𦔫𦘪𦙆𦞲𦤸𦥊𦱰𦳚𦵟𧀤𧉀𧋗𧎨𧛧𧛺𧠫𧤍𧩚𧩴𧩼𧪡𧭟𧰲𧴁𧺏𧺠𧺧𧺿𧼪𨂰𨑠𨒬𨔤𨖎𨘾𨧳𨨬𨨲𨾛𩈕𩉬𩒐𩚉𩤖𩥲𩳲𩶅𩷧𩾕𩿪𪀦𪅍𪅙𪆵𪉄𪉅𪉗𪌫𪌹𪏐𪏸𫄨𫛶"},
    {U"chong", U"㓽㤝㧤㮔㳘㹐䂌䆔䆹䌬䖝䘪䝑䡴䳯偅傭僮充冲喠嘃埫宠寵崇崈徸忡憃憧揰摏樁橦沖浺涌漴潼烛爞珫痋盅祌种種緟罿翀舂艟茧茺虫蝩蟲衝褈蹖蹱酮重銃铳隀𠑙𠖥𠝤𠟍𡿂𢖄𢛒𢝈𢡹𢥞𣐯𣑁𥁵𥅻𥫯𥬱𥭥𦑝𦟛𧐍𧘂𧝎𧩃𧼙𧼩𨈮𨖼𨛱𨤩𨳁𨿿𩅃𩅞𩌨𩒘𩜖𩞉𩞋𩥫𩩳𩬤𩰀𪄻𪅈𪅖𪎽𪒒"},
    {U"chou", U"㐜㤽㦞㨨㮲㵞㾄㿒㿧䀺䊭䌧䌷䓓䔏䛣䪮䱔䲖丑丒仇侴俦偢儔吜嚋圳妯婤媿嬦帱幬怞惆愁懤扭抽掫揄搊擣杻杽栦椆檮殠溴燽牰犨犫畤畴疇瘳皗盩眣瞅矁稠筹篘簉籌紬絒綢绸臭臰菗薵裯詶謅譸讎讐跾踌躊遚酧酬醔醜醻鈕雔雠魗鮋鯈𠌪𠜋𠝽𠷎𠹝𠼡𠾉𡕐𡕪𢣊𢭆𣀓𣕾𣪐𣫐𤘶𤳝𤳠𤽯𤾊𤾦𥃧𥄨𥏈𥡀𥦅𥬠𥰞𥲅𥵬𥺣𥻤𦡴𦭸𧃝𧮻𨀔𨁀𨖬𨞪𨡑𨡲𨤷𩋄𩌄𩽀𩾂𪇘𬑡"},
    {U"chu", U"㔘㕏㕑㗙㙇㛀㡡㤕㶆㾥㾻䅳䅷䇍䊰䎌䎝䐍䐢䖏䙘䜴䝙䟞䟣䠂䠧䢺䦌䶥亍俶傗储儊儲処出刍初助厨嘔嘼埱处媰岀幮廚怵慉憷拀搐摢摴敊斶杵柠柷椘楚楮榋樗橱橻檚櫉櫖櫥欪歜涂淑滀滁濋炪犓珿琡璴畜矗础硫礎祝竌竐篨絀絮绌耝耡臅芻菆著蒢蒭蓫蕏藸處蜍蟵蠩褚觕触觸詘諔諸豖豠貙趎跦踀踰蹰躇躕鄐鉏鋤锄閦除雏雛鶵鸀黜齣齭齼𠁉𠇘𠧖𠰕𠿝𡐌𡝈𡳑𢅥𢊍𢒔𢕓𢣵𢣿𢨫𣢶𣥹𣦠𣦡𤏱𤙟𤝞𤻇𥁯𥒭𥹵𦷝𦺵𦿀𧃏𧎷𧢶𧯩𧰫𧲰𧺶𧽧𨁿𨃕𨅒𨕢𨞕𨴰𨼪𩂫𩈤𩙙𩨸𩹱𩿿𪀩𪁲𪆷𪇆𪓐𬺓"},
    {U"chua", U"㔍䊬䔴䫄䵵撮欻歘𠹐𠻦𣛕𣹶𤁫"},
    {U"chuai", U"㪓㪜䦟䦤䦷䴝啐啜嘬揣搋欼腄膗膪踹𠽶𢲽𢶀𣛹𣤌𣲂𧍒𨣅"},
    {U"chuan", U"㯌㱛㼷䁣串丳传傳僢剶喘圌團巛川惴掾暷椯椽歂氚汌猭玔瑏甎穿篅膞舛舡舩船荈賗踳踹輲遄釧钏鶨𠛖𠯀𠾮𣀔𣛹𣧒𤜼𤮍𤰌𤶱𥃹𥬫𥲏𦎇𦎜𦺛𧍒𧑝𨂦𨔡𨘼𨩴𩂍"},
    {U"chuang", U"㡖㼽䃥䄝䆫䇬䎫䚒䭚倉傸凔刅创刱剏剙創噇囱幢床怆愴戧摐摤朣橦漴漺牀牎牕疮瘡磢窓窗窻舂葱闖闯𠏨𠞮𠳹𡆪𡻯𣃘𥈄𥎒𥡟𥲡𦔛𧜧𧝎𧢆𧬧𨜾𨧖𩃕𩞆𩪘𪁱"},
    {U"chui", U"㝽㷃䅜䍋䕓䜅䞼倕吹圌垂埀惙捶搥棰椎槌炊箠腄菙郵錘鎚锤陲顀鬌魋龡𠄒𡍮𢏒𣇦𣟈𤙵𥙋𥞃𦉈𩌝𩗰𩭦"},
    {U"chun", U"㖺㝄㝇㵮㸪㿤䏛䐇䐏䓐䔚䞐䞺䡅䣨䣩䥎䦮䫃䮞䲠偆僢唇堾媋惷旾春暙朐杶椿楯槆橁櫄沌浱淳湻滣漘犉瑃睶箺純纯肫脣膞芚莼萅萶蒓蓴蝽蠢賰踳輇輴醇醕錞陙鯙鰆鶉鶞鹑𠵧𠸸𡉐𡗥𢾎𣌚𣌠𣘣𣚆𣮢𤘛𤪪𦎧𦚧𧇶𨉩𨋮𩨁𪂹𬭚"},
    {U"chuo", U"㚟㪕㪬㲋䄌䋘䍳䓎促吷啜嚽娕娖婥婼孎惙戳拺擉斫歠涰淖焯磭箹簇綴綽繛绰腏荃蔟趠趵跿踔踱躇輟辍辵辶逴酫醛鋜錣鏃鑡齪齱龊𠽶𡁇𢽸𢿭𣀔𣤌𣬋𤿫𥁯𥓑𦁶𧣺𨆬𨒢𨮸𨰆𩟫𩩟𪘛"},
    {U"ci", U"㐸㓨㘂㘹㞖㠿㢀㤵㩞㾊䂣䈘䓧䖪䗹䛐䧳䨏䭣䯸䰍䲿䳄䳐伺佌佽偨刺刾司呰呲啙垐堲姕嬨嵯差庛廁慈措朿枱柌柴栜栨次此泚滋澬濨玼珁瓷甆疵皉磁礠祠粢糍絘縒胔茈茦茨茲莿薋薺蚝蛓螅螆蠀詞词賜赐趀趑跐辝辞辤辭鈶雌飺餈骴髊鮆鴜鶿鷀鹚齹𠤫𠦐𠩆𠯂𡃸𡥎𡰾𢅜𢓗𢫴𢶴𣐑𣙼𣜁𣢕𤴼𥰭𥴺𥿆𥿴𦍧𦐨𦐾𦑺𦒁𦖝𦼡𧊒𧌐𧑖𧙈𧠎𧠥𧧒𧺼𨋰𨐕𨒤𨒮𨠐𨲁𨾅𩆂𩉋𩝐𩢑𩨨𩾔𪉈𪉪𪑟𪘲𫚖"},
    {U"cong", U"㗰㜡㞱㥖㼻䈡䉘䐋䐫䓗䕺䗓䡯䢨䰌䳷丛从偬匆叢囪囱婃孮従徖從忩怱悤悰慒憁暰枞棇楤樅樬樷欉淙漎漗潀潨灇焧熜爜琮瑽璁瞛碂窗篵総緫縱總繱聡聦聪聰苁茐菆葱蓯蔥藂蟌誴謥賨賩鍯鏓鏦騘驄骢𠂥𠕁𠙂𠢛𠤰𡅇𡟟𡦷𡵷𡹸𢃏𢊕𢐔𢔩𣃗𣊷𤄓𤧚𥍷𥎋𥡬𥣙𥮨𥵫𦇎𦇱𦗜𦝰𧐱𧓏𧝮𧩪𨂴𨍉𨑪𨑹𨒀𨡮𨦱𨱸𨲧𨽁𩬼𩯍𪻐𫓩"},
    {U"cou", U"㔌凑奏揍族楱湊簇腠蔟藪趣趨輳辏𢈾𣉅𣙘𣞜𤆑𦦅𦳿𦺀𧡣𧱪𨓭𨨯𩹀𪉮"},
    {U"cu", U"㗤䃚䓚䙯䛤䟟䠓䠞䢐䣯䥄䥘且促卒噈娕娖媨徂怚憱戚捽殂猝瘄瘯皻簇粗縐縬脨蔍蔖蔟觕誎趗趣趥趨踀踓踤踧蹙蹴蹵酢醋錯顣麁麄麆麤鼀𠑯𠛙𠱙𠴫𡄱𡘛𡝉𡞜𢄧𢈠𢪃𤗁𤛏𤠽𤿚𥅗𥪱𥷼𥻒𥾛𦈚𦟠𦠁𦯣𧆓𧺲𧼜𪓡𪓰𪕝𪚯"},
    {U"cuan", U"㠝㭫㸑䂎僔巑撺攛攢櫕欑殩汆灒熶爨穳窜窾竄篡簒菆襸蹲蹿躥鋑鑹镩𢖑𢸥𤐲𥍬𥎢𥎣𥎤𥤳𥨒𨣵𨼉"},
    {U"cui", U"㓹㜠㝮㯔㯜㱖㳃㵏㷪䃀䄟䆊䊫䔴䘹䙑䧽乼伜体倅催凗卒啐啛墔察崒崔嶉忰悴慛摧椊榱槯毳洒淬漼濢焠熣獕琗璀疩瘁皠磪竁粋粹紣綷縗繀缞翆翠脃脆脺膬膵臎萃衰襊趡踤鏙隹顇𠗚𠞿𠟓𠩪𠽶𢂕𢄸𢕘𢡈𢶓𣃍𣯧𣰚𣿒𣿓𤎋𤗯𤛍𥨒𥳈𥻮𥼂𥼛𥼺𦙈𦦣𧎃𧑎𧚥𧜱𧳚𧹺𧼬𧽠𨄍𨅎𨊉𨻵𨿐𩤏𩮴"},
    {U"cun", U"䍎䞭侟刌吋墫存寸忖拵村洊浚澊皴竴籿膥踆蹲邨𤿄𧚉𧽾𨀛𨙯𨙲𨚲"},
    {U"cuo", U"㟇㭫㽨㿷䂳䐤䑘䟶䠡䣜䰈䱜䴾剉剒厝夎嵯嵳差挫措搓摧撮斮昔最棤澨營玼瑳痤瘥睉矬磋縒脞莝莡蒫蓌蔖虘襊諎蹉躦逪遳酂酇醝銼錯锉错髊鹺鹾齹𠦏𢒐𢚂𢤎𢫬𢯽𣖵𣨎𣩈𤠝𥕉𥭭𥰭𧚏𧪘𨇃𨛏𩄝𩯉𪒙𪘓"},
    {U"da", U"㗳㙮㜓㟷㩉㯓㾑㿯㿴䃮䌋䐛䩢䪏䪚䳴䵣亣僤剳匒呾咑哒嗒噠垯塌塔墶大妲怛憚打搨搭撘毼汏沓溚炟燵畗畣疸瘩眔矺笚笪答繨羍耷胆荅荙薘蟽褡觰詚跶躂达迏迖迚迭逹達鎉鎝鐽阘靼鞑韃龖龘𠉤𠞈𠶫𠹥𡈐𡉑𡌩𡍲𡐿𡚻𢘇𢛁𢝉𢽇𣌭𣣴𣥾𣸉𤝰𤤊𤨑𥉌𥕇𦂀𦈘𦑻𦖿𦗧𦞂𦪭𦬹𨅓𨆰𨗾𨨹𨱏𩏒𩝣𩟐𩠅𩣯𩭣𩺗𫟼"},
    {U"dai", U"㐲㞭㯂㶡㻖㿃䈆䒫䚞䚟䠠䲦代侢傣叇呆呔嘚垈埭大媞岱帒带帯帶廗待怠懛戴曃柋棣歹殆毒瀻獃玳瑇甙箉簤紿緿绐艜蔕蚮蝳螮袋襶詒貸贷跢蹛軑軚軩載轪迨逮逯遞遰隶霴靆馱駘骀鮘鴏黛黱𠯈𠯪𠰺𠷂𡧹𢄔𢎌𣇨𣐮𣦶𣫹𤮼𤸊𥿝𦄂𦙯𦪍𧊇𧑔𨊺𨓞𨗼𨟲𨥶𨽿𩃠𩃷𪐝𪒡"},
    {U"dan", U"㐤㔊㕪㗖㠆㡺㯯㲷㴷䃫䄡䉞䊤䏙䐷䒟䗺䡲䦔䨢䨵䩥䪴䭛䳉丹丼亶伔但倓僤儃儋冉刐勯匰单単呾唌啖啗啿單嘽嘾噉噡嚪壇妉娊媅帎弹弾彈忱怛惔惮愖憚憺憾抌担掸撢撣擔旦柦檐欿殚殫氮沊泹淡湛潭澶澸澹燀狚玬瓭甔疍疸瘅癉癚皽眈石砃禫窞箪簞紞繵耼耽聃聸胆腅膻膽萏蓞蛋蜑蜒蟺衴褝襌襜覘觛訑詹誕譂诞贉贍赕蹛躭郸鄲酖醈霮頕餤饏馾駳髧鴠黕黮黵鿕𠆛𠆶𠇋𠈰𠓊𠹆𡖓𡦨𡵕𢅒𢉑𢋃𢎪𢑝𢘇𢻼𣅟𣇇𣋊𣛱𣱍𣲥𤁡𤢏𤲭𤺺𥄦𥐹𥨎𥱷𥲄𥲇𥳸𥳹𥾚𦅼𦋪𦬹𦻁𦽜𦽫𦾩𧀻𧂄𧙇𧡪𧭃𧴸𨢿𩄕𩅦𩅾𩈉𩈊𩏥𩕤𩩧𪀩𪆻𪒾𫢸𬘘"},
    {U"dang", U"㼕㽆䑗䣊䣣䦒偒儅党凼噹圵垱場壋婸宕崵嵣当愓挡擋攩档檔欓氹潒澢灙燙珰瑒璗璫瓽當瘍盪瞊砀碭礑筜簜簹艡荡菪蕩蘯蟷裆襠譡讜谠趤逿鐺铛闣雼黨𡇈𡇵𡗍𡢈𡰨𡾕𢠽𢡂𣂳𣃉𣗋𣺼𣻍𤔶𤗾𤢎𤣞𥢷𥤗𥯕𥸈𦗴𦳝𦼲𦿆𧅗𧑘𨎴𨝦𨷾𩟈𩼉𩽳𪇁𫽮𬍡"},
    {U"dao", U"㠀㧅㨶㿒䆃䊭䌦䧂佻倒儔刀刂到受叨啁嘄噵壔导導岛島嶋嶌嶹幬忉忑悼惆捣捯搗擣敦朷椡槝檤檮氘洮濤焘燾瓙盗盜祷禂禱稲稻箌絩纛翢翿舠艔菿薵虭衜衟裯蹈軇道醻釖陦陶隝隯魛鱽鳥𠐵𠴼𡄒𢭏𣀓𣁍𣫜𣱼𤓾𤘀𤷘𤹷𥓬𥗚𥺅𦒺𦦺𦦾𦩍𧼤𨗓𨱦𩈞𩕯𩬱𩭟𩵕"},
    {U"de", U"㝵㤫㥀㥁㯖䙷䙸嘚地底得徳德恴悳惪棏淂登的脦鍀锝陟𠮊𠵨𡋩𡭂𣌏𣮊𣮰𤷙𧊇𨁽"},
    {U"dei", U"得"},
    {U"den", U"㩐扥扽"},
    {U"deng", U"㔁㲪䒭䔲䙞䠬䮴䳾僜凳噔墱嬁嶝憕戥朩橙櫈澄灯燈璒登瞪磴竳等簦艠覴豋蹬邓鄧鐙镫隥𡦔𢯭𢿤𣩟𤮘𤺌𤼶𤾢𦩫𧄼𧾊𨄇𨎤𨮴𨶿𩍐𩞬𩯇𪌷𪑬𪒘𪔏"},
    {U"di", U"㓳㢩㣙㦅㪆㫝㭽㰅㹍㼵䀿䂡䃅䊮䍕䏄䏑䐎䐭䑭䑯䗖䞾䟡䢑䣌䧑䨀䨤䩘䩚䪘䯼䱱䴞䵠䶍仾低俤偙僀儥勺厎呧哋唙啇啲啻嘀嚁地坔坘坻埅埊埞堤墆墑墬奃娣媂嫡嵽嶳帝底廸弔弟弤彽怟慸扚抵拞掋提揥摕敌敵旳杓杕枤柢梊梑棣楴樀櫂氐浟涤渧滌滴焍牴狄玓珶甋疐的眱睇砥碮碲磾祶禘笛第篴籴糴約締缔羝翟聜肑胝腣芍苐苖荻莜菂菧蒂蓧蔋蔐蔕藋藡蚳蝃螮袛覿觌觝詆諟諦诋谛豴赿趆踧踶蹄蹢軧迪逐递逓逮遞適遰邸釱鉪鍉鏑镝阺隄隶靮鞮頔題馰骶髢鬄魡鯳鸐𠍪𠐑𠒿𠕳𠚭𠥖𠨿𠫜𠽰𡄷𡆣𡒱𡚙𡚷𡛜𡰖𡽢𢅊𢉆𢓧𢕚𣂉𣅥𣚌𣬴𣯵𣲢𤁰𤈥𤝬𤞈𤧛𤬵𤾠𥄿𥕐𥖾𥡦𥳠𥴦𥸚𥾬𥿄𦉹𦨢𦵦𧀶𧂨𧉛𧋍𧍝𧠫𧤲𧥮𧦚𧺽𧿠𨂇𨌮𨑩𨑼𨗼𨘬𨪾𨮹𩉱𩑾𩭲𩴺𩷎𪄱𬱖"},
    {U"dia", U"嗲"},
    {U"dian", U"㓠㚲㝪㞟㶘㸃㼭䀡䍄䓦䟍䩇佃佔傎典厧唸嚸坫垫埝墊壂奌奠婝婰嵮巅巓巔店惦扂拈掂攧敁敟椣槇槙橂橝殿沾涎淀滇澱点猠玷琔电甸痶瘨癜癫癲碘磹簟腍蒧蕇蜓蜔詀跕踮蹎鈿钿阽電靛頕顚顛颠驔點齻𠑘𠢣𠩷𠫉𠶧𡱇𡼓𢅝𢕯𢖩𢻅𣇖𣒂𣢥𣣈𣣣𣧛𣪀𣪪𤠶𤩱𤿶𥅑𥇞𥑼𥢏𥦟𥧑𥮏𥳢𥴫𥵏𦅆𦒻𦕒𦽄𧄺𧍿𧽍𨈀𩂵𩄠𩅀𩆔𩥄𩨋𩬑𪑩𪓼𪖚"},
    {U"diao", U"㒛㓮㚋㢯㣿㪕㹦䂏䂪䂽䄪䉆䔙䘟䯾䲽䳂伄佻倜凋刀刁刟叼吊啁奝嬥屌弔弴彫扚挑掉敦椆殦汈淍琱瘹瞗矵碉稠窎窵竨簓糶絩綢莜蓧藋虭蛁蜩訋誂調调貂趙跳踔軺釣鈟銚銱鋽錭鑃钓铞铫雕雿魡鮉鯛鲷鳥鳭鵃鵰鼦𠄏𠚥𠚻𠛪𠤼𠥑𠶰𡰯𢁕𢄦𢆴𣩰𤕷𤭈𤱩𥁮𥮐𥲟𥾯𦄋𦨣𦩍𦰏𦶌𦸔𧅈𧘨𧘩𧜣𨰑𨸓𩀜𩈮𩋙𩾗𫼛"},
    {U"die", U"㑙㗧㤴㥈㦅㦶㩸㩹㫼㬪㭯㲲㲳㷸㻡䏲䞇䠟䩞䪓䫕䳀䴑佚叠咥哆哋啑喋嗲垤堞峌崼嵽幉怢恎惵戜挃挕揲昳曡柣楪槢殜氎泆涉渫爹牃牒瓞畳疂疉疊眣眰碟窒絰绖耊耋胅至臷艓苵蜨蝶螲褋褶褺詄諜谍趃跌跕跮踢蹀蹛軼迭鐵镻鞢鰈鰨鲽𠅗𠆙𠗛𠗨𠠯𠲷𡅥𡇓𡖐𡱷𡹭𡺑𡼄𢎆𢜨𢬴𢲼𢶣𣈍𣛻𣡟𣧈𣨂𣶏𤖒𤗨𤚊𤴍𥈖𥉺𥑇𥥌𥶺𥷕𦁛𦁜𦄔𦈅𧍱𨄌𨈈𨐁𨓊𨭓𨳺𨴗𨸅𨻗𨾤𩋞𩻵𪀒𪑧𫶇"},
    {U"din", U"𨈖"},
    {U"ding", U"㝎㞟㣔㫀㴿䟓䦺䵺丁仃叮啶奠奵定嵿帄忊掟椗汀濎灯玎町疔盯矴碇碠磸耵聢腚艼萣葶薡虰蝊訂订酊釘鋌錠鐤钉铤锭靪頂顁顶飣饤鼎鼑𢑅𣆍𣢳𤐣𤛙𥇓𥯢𥳰𥸧𦨍𦩘𧇷𧌾𧳉𨡱𩜦𩠆𩠑𩡯𩸎𩾚𪔂"},
    {U"diu", U"丟丢銩铥颩𠲍𢒝"},
    {U"dong", U"㑈㓊㖦㗢㚵㢥㣚㣫㨂㼯䂢䆚䍶䞒䰤䳉䵔东侗倲働冬冻凍动動勭咚垌埬墥姛娻嬞岽峒崠崬徚恫懂戙挏揰昸東栋桐棟氡氭洞涷湩烔狪甬硐笗筒筩箽絧胨胴腖苳菄董蕫蝀衕詷諌迵酮霘駧騆鮗鯟鶇鶫鸫鼕𠄉𢔅𢛔𢳾𣿅𤤮𤦪𤲚𤷆𥫎𥳘𦡂𧄓𧆼𧓕𧡍𧯾𧲴𧳣𧼓𧼩𧽿𨩧𨿢𩂓𩐤𩐵𩜍𩣳𩧲𩭩𪐈𪔦𬟽"},
    {U"dou", U"㓸㛒㞳㢄㤱㨮㪷䄈䇺䕆䛠䬦乧侸兜兠剅吋吺唗唞投抖斗斣枓梪橷毭氀浢瀆痘瞗窦窬竇篼脰荳蔸蚪讀豆逗逾郖都酘鈄鋀閗闘阧陡餖饾鬥鬦鬪鬬鬭𠁁𠍄𠱑𠾇𡂛𡂝𡆏𡙬𡟳𢦍𢭃𣁵𣂮𣘛𣭗𤀨𤅋𤝈𤞟𤾒𥆖𥉝𥥷𥺉𦄓𦆘𧏆𧘞𧡸𧮡𧯞𧯠𧯤𧱓𨁋𨥪𨪐𨴜𨶜𨹜𩊪𩑯𩔡𩮷𩳈𪌉𪐺"},
    {U"du", U"㒔㓃㞘㡯㧻㱩㸿㾄䀾䈞䐁䐗䓯䙱䟻䢱䦠䩲䪅䫳䮷䰩䲧儥凟剢剫匵厾嘟噣土堵塗妒妬嬻宅帾度斁晵暏杜椟樚橐櫝殬殰毒涜渎渡瀆牍牘犊犢独獨琽瓄皾督睪睹碡秺竇竺笃篤纛罜肚芏荰蝳螙蠧蠹裻襡覩詫読讀讟读豄賭贕赌都醏錖鍍鍺鑟镀闍阇陼靯韇韣韥頓騳髑黩黷𠉩𠠔𠠠𠣰𡍨𡎉𡝜𡰪𢉜𢝂𢷺𢽚𢾀𢾅𣧃𣨲𣫔𣰬𤚚𤚡𤫻𤬂𤬪𤴱𤵊𤶮𥀁𥀲𥃾𥑯𥓇𥓍𥖠𥖿𥝟𥝾𥯖𥲗𥳉𥳲𦌷𦏕𦘴𦙋𦚈𦛯𦡄𦳔𦺇𦺥𧁿𧉓𧋌𧐰𧑠𧔬𧛔𧞹𧰵𧷿𧾥𨂭𨋈𨍛𨏔𨧀𨽍𩞾𩥋𩧈𩨳𩩮𩵚𪇆𪍹𪐞𬭊"},
    {U"duan", U"㟨㫁㱭䠪偳剬塅媏断斷椴段毈煅瑖短碫端篅簖籪緞缎耑腶葮褍踹躖鍛鍴锻𠡱𢭃𢯫𢷖𣠭𥠄𥵣𦾸𧤗𧶲𨱚𨺣𩏇𩤚𩤣"},
    {U"dui", U"㙂㟋㠚㢂㢈㨃㬣㳔䂙䇏䏨䔻䜃䨴䨺䬈䬽䭔䯟兊兌兑啍垖埻堆塠奪对対對嵟怼憝憞懟搥敦杸槌濧瀢瀩痽碓磓祋綐膭薱襨謉譈譵追鈗鋭錞鎚鐓鐜镦队陮隊頧鴭𠂤𠏮𠜑𠡒𠦗𠫨𡁨𡏩𡑈𡜥𡷋𡼻𢆻𢈹𢟋𣝉𣪌𤄛𤤷𤧫𤮩𤷎𤹵𥑵𥹲𦞱𦡷𦶏𧧆𨬍𨹅𨻵𩄮𩅆𩅥𩅲𩈁𩈜𩊭𩐌𩜀𩨽𪌤𪒛𪒡"},
    {U"dun", U"䃦䔻䤜䪃伅俊吨噸囤坉墩墪庉忳惇憞撉撴敦楯橔沌潡炖燉犜獤盹盾砘碷礅腞腯蜳豚趸踲蹲蹾躉逇遁遯鈍鐓鐜钝頓顿驐𠎻𡆰𡼖𢬼𣎴𣗁𣚪𣞇𤟢𤭞𥂦𥫬𥫱𥭒𦎧𦪔𦰭𦼿𧝗𧿗𨔡𩔂𩞤"},
    {U"duo", U"㖼㙍㙐㛆㛊㡯㣞㤞㥩㨊㪜㻔㻧䅜䐾䑨䒳䙃䙤䝐䠤䤪䤻䩔䫂䯬䲊亸仛兑凙刴剁剟剫咄哆哚喥嚉嚲垛垜埵堕墮墯多夛夺奪奲媠尮崜嶞度惰憜挅挆捶掇揣敓敚敠敪朵朶杂杕杝柁柂柮桗棰椯橢毲沰沱澤畓痥硾綞缍舵茤袳裰襗詑誃貀趓跢跥跺跿踱躱躲軃鄲酡鈬錞鍺鐸铎陀陊陏隋隓飿饳馱鬌鮵鵽點𠛫𡌭𡓉𡓷𡶲𡺇𢜬𢳽𢼠𢿎𣑧𣧷𣵺𣵻𤋨𤌃𤛛𤢕𤤸𤬾𥞛𥬲𥳔𥿰𦍦𦕰𦖋𧊱𧙤𧢵𧧇𧩧𧱫𨀟𨆅𨉡𨍏𨦃𨬍𨲉𨹃𩃒𩊜𩍜𩎫𩑒𩢎𩬻𩿩𪃒𪘉𪞝"},
    {U"e", U"㓵㔩㖾㗁㗉㛕㟧㠋㣂㦍㧖㧴㩵㩽㫊㮙㷈㼂䄉䆓䋪䑥䑪䔾䕏䖸䙳䚰䛖䛩䜙䝈䞩䣞䩹䫷䱮䳗䳘䳬亞佮侉俄偔偽僫匎匼卾厄叱吪呃呝咢咹哦啈啊啐啞噁噩囐囮垩埡堊堨堮妸妿姶娥娾娿婀媕屙屵岋峉峨峩崿庵廅恶悪惡愕戹扼搕搤搹擜曷枙椏櫮欸歞歹歺洝涐湂猗玀珴琧疴痷痾皒睋砈砐砨砵硆硪磀礘胺腭苊莪萼蒍蕚蘁蚅蛤蛾蝁覨訛詻誐誒諤譌讍讹谔豟軛軶輵轭迗遌遏遻邑鄂鈋鈪鋨鍔鑩锇锷閜閼阏阨阸阿隘頋頞頟額顎颚额餓餩饿騀鬲魤魥鰐鰪鱷鳄鴳鵈鵝鵞鶚鹅鹗齃齶齾𠥍𠥕𠥜𠯪𠰜𠱥𠱫𠷸𡀾𡅅𡅡𡪑𡪗𡴯𡹣𡾙𢃲𢨡𢼚𢼛𣄰𣎴𣘨𣢛𣤲𣦵𤂷𤎣𤡾𤪄𤭼𤸱𤻜𥋙𥑺𥑾𥓈𥔲𥯳𦊪𦛅𧊜𧊧𧌄𧍬𧒎𧔼𧙃𧚄𧠞𧢽𧨟𧭪𧼎𧽶𧿕𨂁𨃃𨌧𨤕𨱂𨵌𨶯𨸷𨺨𩇠𩉴𩊢𩋊𩋽𩐰𩑁𩒕𩒰𩕟𩕬𩖀𩚬𩣣𩤩𩨮𩪤𩸇𩸋𩸖𩽹𪀝𪅴𪘊𪘐𪙯𫫇"},
    {U"ee", U"欸誒"},
    {U"ei", U"欸誒诶𡅅"},
    {U"en", U"䅰䬶䭓䭡奀峎恩摁煾蒽饐𡟯𡵖𡷐𤇯𤫹"},
    {U"eng", U"鞥"},
    {U"er", U"㒃㖇㚷㛅㢽㧫䋙䋩䌺䎟䎠䎶䏪䣵䮘二佴侕儿児兒刵厼咡唲嬭尒尓尔峏弍弐杒栭栮樲毦洏洱濡爾珥粫而耏耳聏胹腝臑荋薾衈袻誀貮貳贰趰輀輭轜迩邇鉺铒陑陾隭餌饵駬髵髶鮞鲕鴯鸸𠚧𡦕𢀪𢄽𣧹𣩚𤽓𥅡𦓓𦓔𦖢𦗼𧌣𨎪𩚪𩤖𩰴𩱊𩱓𪐰𪕔𪕨"},
    {U"fa", U"㕹㘺㛲䂲䇅䣹乏伐佱傠发垡姂彂拔撥栰橃汎沷法泛浌灋珐琺疺発發瞂砝笩筏罚罰罸茷蕟藅貶醱鍅閥阀髪髮𠞵𠲎𤇰𤣹𤿓𥎰𥩱𦪑𧬋𨀳𨋺"},
    {U"fan", U"㕨㛯㠶㤆㮥㴀㶗㸋㺕㼝㽹䀀䀟䉊䉒䊩䋣䋦䌓䐪䒦䕰䛀䟪䡊䣲䪛䪤䪻䫶䭵䮳仮伋凡凢凣勫匥反噃墦奿婏嬎嬏帆幡忛憣払拚旙旛杋柉梵棥楓樊橎氾汎泛渢滼潘瀪瀿烦煩燔犯犿璠畈畨番盕矾礬笲笵範籓籵緐繁繙羳翻膰舤舧舩范蕃薠藩蘩蟠蠜袢襎訉販贩蹯軓軬轓返釩鐇鐢钒颿飜飯飰饭鱕鷭𠆩𠒾𡁈𡗹𡜀𡤎𡶉𢇪𢐲𢗰𢬵𢶃𣔶𣳜𤄑𤄫𤬨𤭍𤳖𥃵𥅒𥢌𥸨𥹇𥻫𥼞𥿋𦊻𦜒𦦽𦨲𦪖𧀭𧁉𧉤𧊾𧍙𧢜𧦟𧶶𨆌𨙮𨟄𨠒𨥧𩕏𩡫𩧅𩨏𩨩𪖇𫔍𬸪"},
    {U"fang", U"㑂㕫㤃㧍㯐䄱䢍䰃䲱仿倣匚坊埅堏妨彷房放方旊昉昘昞枋汸淓牥瓬眆眪祊紡纺肪舫芳蚄訪访趽邡鈁錺钫防雱髣魴鰟鲂鴋鶭𣄅𥫳𨾔𩇴𩗧𩬝𩮗𩲌𩷸𪕃"},
    {U"fei", U"㔗㥱㩌㫵㵒㹃䆏䈈䉬䍨䑔䒈䕁䕠䚨䛍䟛䠊䤵䨽䨾䩁䯋䰁俷剕匪厞吠啡墢奜妃婓婔屝废廃廢怫悱扉拂斐昲暃曊朏杮柹棐榧橃橨櫠沸淝渄濷犻狒猆疿痱癈砩祓笰篚紼緋绯翡肥肺胇胏胐腓芾茀茇菲萉蕜蕟蕡蜚蜰蟦裴裶襏誹诽費賁费鐨镄陫霏靅非靟飛飝飞餥馡騑騛髴鯡鲱鼣鼥𠏿𠮆𠲎𡌦𢑮𢒍𢳁𢾺𣍧𣙿𤝟𤷂𤺕𤼺𥄱𥇖𥝊𥝋𥟍𥠶𥭬𦃄𦈗𦱷𧌘𧍃𧑈𧓖𧕒𧕿𧚆𧝇𧿳𨚓𨵈𨻃𩄼𩆦𩇫𩇮𩇯𩇽𩙲𩡷𩦎𩯃𩰲𩰾𩱎𩵥𩹉𪁹𪂏𫂈𬴂"},
    {U"fen", U"㖹㤋㥹㬟㱵㷊㸮㿎䆏䩿䫞䭻䴅份偾僨兝兺分匪吩哛噴坆坋坟墳奋奔奮妢岎帉幩弅忿愍愤憤扮拚敃昐朆朌枌梤棻棼橨氛汾濆瀵炃焚燌燓燔獖玢盼砏秎竕粉粪糞紛纷羒羵翂肦膹芬葐蒶蕡蚠蚡衯訜豮豶賁躮轒酚鈖錀鐼隫雰頒餴饙馚馩魵鱝鲼鳻黂黺鼖鼢𠛸𠵮𠻫𡊄𡊅𡨖𢁤𢅯𢊱𢚅𢧝𢴢𢹔𣬩𣯻𣱦𣸣𤔟𤖘𤗸𤘝𤰪𥂙𥳡𥹻𥽒𦍏𦍪𦐈𦦑𦰛𦶚𧮱𧷐𧿚𨎾𨤘𨤚𨳚𨳣𨷒𩉵𩡷𩢈𩰟𩸂𩿈"},
    {U"feng", U"㐽㒥㕫㛔㜂㠦㡝㦀㵯䀱䏎䒠䙜䟪䩬䩼丰仹俸偑僼冯凤凨凬凮唪埄堸夆奉妦寷封峯峰崶捀捧摓方枫桻楓檒沣沨泛浲渢湗溄漨灃炐烽焨煈熢犎猦琒甮疯瘋盽砜碸篈綘縫缝肨舽艂莑葑蘴蚌蜂蠭覂諷讽豊豐賵赗逄逢鄷酆鋒鎽鏠锋闏霻靊風飌风馮鳯鳳鴌鵬麷𡉚𡕖𡨛𡵞𢇫𢓱𢪋𣿝𤖀𥊒𥍮𥛝𥷜𥽈𦜁𦧁𧆉𧍯𧥹𧾳𨝭𨩥𨲫𨺢𩉧𩊩𩋮𩐯𩘵𩙐𩙣𩪌𪐃"},
    {U"fiao", U"覅"},
    {U"fo", U"仏仸佛坲梻𧥚𧼴"},
    {U"fou", U"䬏不否垺妚殕炰紑缶缹缻芣衃裦雬鴀𡜊𤊻𤽦𧉈𧊦𨛔𩂆"},
    {U"fu", U"㓡㔗㕊㕮㙏㚆㚕㚘㜑㟊㠅㤔㤱㩤㪄㫙㬼㭪㲗㳇㷆㽬㾈䂤䃽䃿䄮䈏䋨䋹䌗䌿䍖䎔䑧䒄䒇䓏䓵䔰䕎䗄䘀䘠䝾䞜䞞䞤䞯䞸䟔䟮䠵䡍䦣䧞䨗䨱䩉䪔䪙䫍䫝䭮䭸䭻䮛䯱䱐䳕䴸䵾不乀乶仅付伏伕佛俌俘俛俯偩偪傅冨冹凫刜副包匐呋咈咐哹哺嘸坿垘垺報复夫妇妋姇娐婏婦媍嬎嬔孚孵宓富尃岪峊巿帗幅幞府弗弣彳彿復怀怤怫懯扶抚拂拊捊捬掊撨撫敷斧旉服枎枹柎柫柭栿桴棴椨椱榑氟汱沕沸泭洑浮涪溥滏澓炥烰焤父玞玸琈璷甫甶畉畐畗痡癁盙砆砩祓祔福禣秿稃稪竎符笰筟箁箙簠粰糐紨紱紼絥綍綒緮縛纀绂绋缚罘罦翇肤胕脯腐腑腹膚艀艴芙芣芾苻茀茯荂荴莆莩菔萉萯葍蓲蕧虙蚥蚨蚹蛗蜅蜉蝜蝠蝮衭袚袝袱複褔襆襥覄覆訃詂諨讣豧負費賦賻负赋赙赴趺跗踾軵輔輹輻辅辐还邚邞郍郙郛鄜酜酻釜釡鈇鉘鉜錇鍑鍢阜阝附陚鞴韍韛韨頫颫颰馥駙驸髴鬴鮄鮒鮲鰒鲋鳆鳧鳬鳺鴔鵩鶝麩麬麱麸黻黼𠋩𠌽𠓗𠟌𠡂𠣾𠪻𠬝𠲽𡏪𡐝𡞪𡠞𡦄𡫺𡬇𡵛𢀼𢁀𢂀𢂆𢌹𢏍𢒒𢗫𢗲𢘍𢞦𢠲𢯋𢰆𢻀𢻮𣀣𣀾𣄎𣆵𣑿𣘧𣞒𣥋𣭘𣹋𣻜𣿆𤆮𤉨𤕾𤙤𤙭𤝔𤝟𤠪𤭟𤰈𤱽𤶖𤸑𤸗𤿭𥄑𥄓𥒫𥒰𥘬𥦘𥧷𥨍𥪋𥪚𥰛𥱀𥲛𥳇𥷱𥼼𥾧𦂊𦇁𦊦𦊾𦎎𦎭𦐡𦑹𦔍𦖀𦨈𦨋𦨡𦩡𦮹𦰺𦱖𦲫𦳓𦸱𦺉𦽏𦿁𧀮𧀴𧄏𧉈𧉊𧌈𧌓𧒂𧒙𧕡𧖚𧥱𧳂𧴌𧻳𧼗𧼱𧿳𨁒𨋩𨌥𨑑𨚓𨦛𨵃𨵟𨺅𨾚𩂆𩂎𩂔𩂕𩅿𩉽𩋟𩋨𩌏𩍏𩍿𩎛𩐚𩑬𩒙𩒺𩓖𩖬𩖼𩜲𩠷𩢰𩢿𩣜𩣸𩬙𩭺𩳎𩳐𩵩𩵹𩽺𩽻𩾿𩿧𪀺𪂀𪂋𪂾𪃓𪆠𪊐𪍏𫓧𫖯𫚒"},
    {U"ga", U"㹢伽呷咖嘎嘠噶夾尕尜尬旮玍軋釓錷钆魀𠁥𡉅𡯰𡯽𡼛𣬫"},
    {U"gai", U"㕢㧉㮣㱾䀭䏗䐩䪱䬵䶣丐乢侅匃匄咳垓姟峐忋戤摡改晐杚核概槩槪汽溉漑瓂畡盖磑祴絠絯胲芥荄葢蓋該该豥賅賌赅郂鈣钙閡阣陔隑骸𠌰𡒖𡧣𢅤𢍓𢻉𦫻𧊏𧯺𨞨𨮂𨱕𨱣𨸛𩕭𬮿"},
    {U"gan", U"㓧㔶㤌㶥㺂㽏㿻䃭䇞䊻䤗䯎䲺䵟个乹乾亁仠佄倝凎凲咁坩奸尲尴尶尷干幹忓感扞捍擀攼敢旰杆柑桿榦橄檊汗汵泔浛淦漧澉灨玕玵甘疳皯盰矸秆稈竿笴筸篢簳粓紺绀肝芉苷虷衦詌諴豃贑贛赣赶趕迀酐釬錎飦骭魐鰔鱤鳡鳱𠇵𠖫𠥴𠿑𡯋𡶑𢁗𢧀𣁖𣆙𣔼𣗲𣘠𣦖𣰦𣵷𣵼𣹟𤌹𤭙𤮽𤯌𥕵𥘏𥸡𥾍𦪧𦼮𦾮𧆐𧹳𧾲𨝌𨣝𨳼𩉐𩖦𩚵𩠁𩢨𩬚𩴁𩴵𩹸𪉦𪉿𪊄𪊇𪔆𪚬"},
    {U"gang", U"㟠㟵㠮㧏㭎㼚㽘䚗䴚亢伉冈冮刚剛堈堽岗岡崗戅戆戇扛抗掆杠棡槓港溝焵焹牨犅犺疘矼碙筻綱纲缸罁罓罡肛肮釭鋼鎠钢阬頏鿍𠵹𡇬𢭈𢰌𣗵𣦐𤭛𦋳𦱌𨟼"},
    {U"gao", U"㚏㚖㤒㵆㾸䆁䓘勂吿告咎夰峼搞暠杲槀槁槔槹橰檺櫜浩滜獋皋皐睪睾祮祰禞稁稾稿筶篙糕縞缟羔羙膏臯菒蒿藁藳誥诰郜鋯鎬锆镐韟餻高髙鷎鷱鼛𠰛𡋟𡜲𡷥𡼗𢍎𢞟𣓌𣝏𣽎𤱟𥓖𥢐𥢑𦍱𦏦𦕺𦤎𦺆𧚡𧜉𧠼𧢌𨝲𩋺𩏤𩓢𩔇𩕍𩝝𩫓𪔘"},
    {U"ge", U"㓣㖵㗆㠷㤎㦴㪾㭘㵧㷴䅥䈓䐙䔅䗘䘁䛿䣬䧄䨣䩐䩡䪂䪺䫦䬣䰛个介仡佫佮個假割匌可各合吤呄咯哥哿嗝嗰嘅噶圪塥屹彁愅戈戓戨扢挌搁搿擖擱敋杚格槅櫊歌浩滆滒牫牱犵猲獦疙砝硌秴箇紇纥肐胳膈臈臵舸茖菏葛蓋虼蛒蛤袼裓觡詥諽謌輵轕鉀鉻鉿鎑鎘鎶铬镉閘閣閤阁隔革鞈鞷韐韚頜騔骼髂鬲魺鮥鮯鰪鴐鴚鴿鵅鸽鿔𠛊𠯫𠲱𠸲𠹓𠺝𡟍𢄍𢆜𢎄𢓜𢡍𢧧𢩓𢯹𢼛𣭝𤇞𤕒𤜊𤠇𤩲𤭻𤰙𥉅𥢸𥰮𥴩𥺊𦑜𦓱𦨜𧈌𧈑𧈖𧊧𧎺𧗶𧪞𧿩𨍮𨏚𨏴𨐥𨝆𨞛𨟶𨭛𨼣𨼥𨾓𩎎𩢅𩢛𩨀𩨘𩹺𩹿𩼙𩾷𪀁𪀉𪃿𪄎𪌣"},
    {U"gei", U"給给"},
    {U"gen", U"㫔㮓䫀亘亙哏揯搄根痕艮茛跟𠄣𠛵𥃩𨒼𩑡𩒝𩓓"},
    {U"geng", U"㪅㹴㹹㾘䋁䌄䎴䢚䱍䱎䱭䱴亘亙亢刯哽埂堩峺庚恆挭暅更梗椩浭焿畊硬絙絚綆緪縆绠羮羹耕耿莄菮賡赓邢郠頸骾鯁鲠鶊鹒𠡣𡍷𡩃𢙾𢞚𣆳𣈶𣎄𥅨𥉔𥔂𥾚𦚸𦛟𦜷𦞌𦣍𦵸𧀙𧋑𧙸𧰨𨁈𩂼𩜣𩱁𩱋𩱧𬒔"},
    {U"gong", U"㓋㓚㔶㕬㤨㧬㫒㭟㯯㺬㼦䂬䂵䇨䍔䐵䔈䡗䢼䰸䱋䲲䳍供公共功匑匔厷咣唝嗊塨宫宮工巩幊廾弓恭愩慐拱拲攻杛杠栱汞渱熕珙疘碽磺礦篢糼紅羾肱莻虹蚣蛩觥觵貢贛贡躬躳輁釭銾鑛鞏髸魟龏龔龚𠇒𠌕𠞖𡔕𡚑𡟫𢀜𢁠𢖷𢸁𤅐𤨶𤬳𤱨𥧂𥧡𥨐𥫋𥸲𦄜𦈩𦊫𦏺𦓳𦔸𦞗𦞨𦦧𦩼𦬘𦵡𧆷𧎡𧘏𨉫𨊧𨋑𨋝𨋷𨒱𨣂𨥑𨨭𨴛𩃙𩌌𩐣𩛘𪄌𪏠𪏢𫋐"},
    {U"gou", U"㗕㜌㝅㝤㡚㨌㺃㽛䃓䐟䑦䝭䪷䬲䵶佝傋冓勾區句呴坸垢够夠姤媾岣彀拘搆撀构枸構沟泃溝煹狗玽痀笱篝簼緱缑耇耈耉芶苟茩蚼袧褠覯觏訽詬诟豰豿購购軥遘鈎鉤钩雊鞲韝鮈鴝𠛎𡖑𡗁𢄇𣕉𣕌𣙱𣫌𤖮𤚼𤠼𤫱𥉇𥧒𥬉𥴴𥿺𦎯𦎼𦩷𦱣𦵷𦽋𧃛𧲿𧵈𨩦𩄢𪃺𪚭"},
    {U"gu", U"㒴㚉㧽㯏㱠㼋㽽㾶䀇䀜䀦䀰䅽䇢䉉䍛䐨䐻䓢䜼䡰䮩䵻䶜估傦僱凅古告呱呴咕哌唂唃啒嗗嘏固堌夃姑嫴孤家尳崓崮怘愲扢故枯柧梏棝榖榾橭櫎毂汩沽泒淈滑濲瀔焸牯牿瓠痼皋皷皼盬瞽磆祻稒穀笟箍箛篐糓縎罛罟羖股胍脵臌苦苽菇菰蓇薣蛄蛊蛌蠱角觚詁诂谷賈軱軲轂轱辜逧酤鈲鈷錮钴锢雇離顧顾餶馉骨骰鮕鯝鲴鴣鵠鶻鸪鹄鹘鼓鼔𠑹𠷞𠻧𠽿𡗷𡷓𡽂𢝳𢡇𢪏𣀐𣖫𣦩𣦭𣨍𣨺𣪸𣫀𣱫𣲖𤅱𤚱𥂰𥐬𥠳𥮝𥵠𥿍𦈔𦈲𦊬𦋆𦍩𦎰𦙶𦺠𦾫𧆻𧇡𧟣𧣡𧬕𧳸𧵎𨠋𨪷𨬕𨱃𨵐𨸯𩀠𩙏𩲱𩴡𪄮𪇗𪕷"},
    {U"gua", U"㒷㓡㧓㱙㶽䀨䈑䏦䒷䫚䯄䯏冎刮剐剮劀卦叧呱咶咼啩坬寡惴挂捖掛擖栝歄焻煱瓜絓緺罣罫聒胍舌苽袿褂詿諣诖趏踻銛銽颪颳騧鴰鸹𠆣𠈥𠊰𠙼𠛒𠜵𠟗𠮠𠯑𠵯𡜁𣅻𤆜𥄼𥈓𥝒𦊱𧤐𧿼𨵃𩢍𩻎𪇜"},
    {U"guai", U"㧔㷇㾩䂯䂷䃶䊽䓙乖叏噲夬怪恠拐掴摑枴柺箉罫𠛕𠦬𡇸𡌪𡖪𡧩𢶒𣲾𥑋𥑰𦫳𦮃𧊅𧱾𧴚𩶦"},
    {U"guan", U"㮡㴦䌯䎚䏓䗆䗰䘾䙛䙮䚪䝺䤽䦎䩪䪀䲘丱串倌关冠卝婠官幹悹悺惯慣懽掼摜斡果棺樌櫬權毌泴涫淉淪潅灌爟琯瓘痯瘝癏盥矔矜礶祼窤筦管綸罆罐舘莞菅蒄覌観觀观貫贯躀輨遦錧鏆鑵閞関闗關雚館馆鰥鱞鱹鳏鳤鵍鸛鹳𠬆𠴨𡅭𡠒𡭷𢇇𢉂𢺄𣥥𣩔𣬂𤮳𤼐𥈒𥉀𥊫𥍅𥎅𥜄𥷬𥿑𦛤𦺊𨝑𨱌𨳡𨴊𨵄𨷀𩖒𪈸"},
    {U"guang", U"㤮㫛侊俇僙光咣垙姯广広廣恍挄撗擴桄横櫎欟洸潢灮炗炚炛烡犷獷珖硄胱臦臩茪趪輄迋逛銧黆𠏤𢓯𤖖𤳭𤴀𥀱𦢎𧻺𨎩𨐈𨤡𨶰𩑈𩒚𩧉𪇵𪕓"},
    {U"gui", U"㔳㙺㧪㨳㩻㪈㰪㲹㸵䁛䃶䃽䅅䈐䌆䍯䐩䐴䝿䞈䞨䟸䠩䣀䤆䤥䮹䯣䲅䳏亀佹偽傀刽刿劊劌匦匭匱厬哇圭垝妫姽娃媯嫢嬀宄嶡巂帰庋庪廆归恑摫撅撌攰攱昋晷朹柜桂桅桧椝椢概槣槶槻槼檜櫃櫰櫷歸氿洼湀溎潙炅炔猤珪瑰璝瓌癐癸皈瞡瞶硅硊祈祪禬窐筀簂簋繪胿膭茥蓕蘬蛫螝蟡袿襘規规觖觤詭謉譌诡貴贵赽趹跪蹶軌轨邽郌閨闺陒隗鞼騩鬶鬹鬼鮭鱖鱥鲑鳜鳺鴂龜龟𠐽𠜾𠪑𠱓𠿥𡃩𡌲𡗤𡧭𡬂𡷺𡹙𢃯𢄊𢠿𢻂𢻗𣄜𣢪𣦦𣧎𣪕𣷾𤘧𤜂𤡱𤱺𤱾𤲉𤻿𤼮𤿡𥇳𥈸𥍁𥍨𥎛𥜏𥥠𥦣𦓯𦤇𦳛𧊄𧡫𧢦𧷱𧹑𧻜𧽸𨇙𨋡𨲿𨾚𨾴𩉝𩊛𩌃𩍨𩏐𩏡𩑬𩒙𩓠𩔆𩪁𩱻𩲡𩳝𩳧𪀗𪄯𪆳𪈥𪊧𪏤"},
    {U"gun", U"㙥㨰㫎㯻䃂䎾䜇䵪丨卷惃棍混渾滚滾琯璭睔睴磙緄緷绲蓘蔉衮袞裷謴輥辊錕鮌鯀鰥鲧𠃌𠞬𡈧𡘝𡻨𢃩𣮎𥕦𦓼𦠺𦫎𧬪𧸫𨍲𩨬𩩌"},
    {U"guo", U"㕵㖪㗥㗻㳀㳡㶁㿆䂸䆐䐸䙨䤋䬎䴹划呙咶咼唬啯嘓囗囯囶囻国圀國埚埻堝墎崞帼幗彉彍惈慖摑果椁楇槨櫎活淉渦漍濄猓瘑矌簂粿綶聒聝腂腘膕菓蔮虢蜮蜾蝈蝸蟈蠃裹褁輠过過郭鈛錁鍋鐹锅餜馃馘𠋜𠜴𠩥𠿤𡆸𡇄𡓣𢃦𢅗𢐚𢝸𢧰𢸗𢹖𢼛𣁯𣂄𣽅𣽰𤂁𤫵𤬁𤮋𥁁𥂣𥄍𥆘𥇜𥕖𥜭𦄰𦗒𦘌𦛢𦬗𦸈𧒖𧖻𧤯𧥵𧭕𧭣𧰒𧾛𨉹𨭗𨽏𩉕𩋗𩟂𩪐𩫏𩰬𩰭𩻧𪂠𪆹𪈃𪋊𬇹"},
    {U"ha", U"吓呵哈奤獬蛤蝦鉿铪𡄟𨉣"},
    {U"hai", U"㕢㜾㤥㧉㧡㨟㰧㰩㱼㺔㾂䇋䜕䝳䠽䪱䯐䱺亥侅咍咳咴嗐嗨嚡塰妎孩害拸氦海烸猲絯胲还還郂酼醢閡頦餀饚駭駴骇骸𠀅𠔑𠹛𡕗𡾨𢁓𢞐𢩸𢻜𣖻𣢇𣳠𤵽𤸎𥁐𥂧𥩤𥩲𦐤𦤦𦤬𦷷𧻲𧽊𧽖𨀖𨒨𨡬𨸜𩞞𩠚𩡔𩩰𩪃𩰶𩹄𬐚"},
    {U"han", U"㑵㒈㖤㘎㘕㘚㙔㙳㟏㟔㢨㤷㦑㨔㪋㮀㰹㲦㵄㶰㸁㺖㺝㼨䁔䈄䍐䍑䎏䎯䏷䐄䓍䓿䕿䖔䗙䗣䘶䛞䜗䣻䤴䥁䧲䨡䫲䮧䶃丆仠佄傼兯函凾厂厈含咁哻唅喊嚂圅垾娢嫨寒屽岾崡嵅嵌幹忓悍感憨憾扞捍撖撼攼旰旱晗晘暵桿梒椷榦欦歛汉汗汵泔浛浫涆涵淊淦漢澉澏澣瀚灘焊焓熯爳犴猂琀甘甝皔睅矸笒筨罕翰肣莟菡蔊蘫虷蚶蛿蜬蜭螒譀谽豃軒邗邯酣釬鈐銲鋎鋡閈闞闬阚雗靬韓韩頇頜頷顄顩顸颔馠馯駻鬫魽鳱鶾鼾𠗴𠢇𠤮𠤾𠥴𠦊𠲒𠵸𠹄𠽦𠿑𡁀𡇜𡣔𡬖𡷛𡻡𢀵𢃗𢄜𢇞𢎘𢔈𢔔𢧦𣐺𣒷𣘞𣛴𣝽𣢅𣢟𣢺𣵷𤀉𤌐𤞶𤬯𤭙𤳉𤸕𤿧𥀐𥆡𥇌𥉰𥤱𦋣𦒅𦒝𦛜𦜆𦞞𦥖𦺦𧂃𧃙𧑚𧭻𧮰𧮳𧯘𧰪𧵊𧹣𧾔𨁄𨛎𨢈𨸗𩄙𩈖𩈣𩎒𩕠𩖺𩗤𩞿𩠻𩦊𩭥𩹑𩹼𩼉𩾝𪁟𫒶𫘣"},
    {U"hang", U"㤚㬻㰠䀪䂫䘕䟘䢚䣈䦭䲳吭垳夯妔巷忼斻杭桁沆炕狠狼珩笐筕絎绗肮航苀蚢行貥迒邟酐頏颃魧𠡊𡕧𤰟𤵻𤼍𥮕𦐄𦨵𧘃𧦑𨁈𨾒𩉜𩔋𩠾𩲋𩶞𪐦𪕇𪗜"},
    {U"hao", U"㘪㙱㚪㝀㞻㠙㩝㬔㬶䒵䚽䝞䝥䧚䧫䪽䯫傐儫号呺哠唬嗥嘷噑嚆嚎壕好妞恏悎昊昦晧暠暤暭曍椃毜毫浩淏滈澔濠灏灝獆獋獔皋皓皜皞皡皥睾秏竓籇翯耗聕膠茠蒿薃薅薧藃號虠蚝蠔諕譹豪貉郝鄗鎒鎬顥颢鰝𠚃𠢕𡐒𡚌𡚽𡠖𡥆𡽝𢻇𣆧𣘫𣚧𣭖𣭹𣱓𣽎𤀃𤝐𤡇𤢨𤢭𤩩𤩭𤫧𥍣𥢑𦳁𧇼𧦝𧬁𧯌𨂜𨒑𨚙𨚮𨝲𨠬𨼍𩐮𩕍𩖸𩫕𩮘𩲊"},
    {U"he", U"㔠㕡㗿㚘㥺㧁㪃㪉㪋㬞㭘㭱㮝㮫㰤㰰㵑㷎㹇㾑㿣㿥䃒䅂䏜䐧䒩䓇䓼䕣䚂䞦䢔䫘䮤䯨䳚䳽䶅䶎何佫劾合呵呼咊和咼哈哧哬啝喛喝嗃嗑嗬噈嚇垎壑姀害寉峆惒愒抲挌揭敆曷柇核格楁欱毼河洽涸渮渴澕焃煂熆熇燺爀犵狢猲癋皬盇盉盍盒硅碋礉禾秴穒篕籺粭紇繳翮翯苛荷菏萂藃藿蚵蝎螛蠚袔褐覈訶訸詥謞诃貈貉賀贺赫輅轄郃鉌鑉閡闔阂阖隺霍靍靎靏鞨頜颌餄餲饸鬩魺鲄鵠鶡鶮鶴鸖鹖鹤麧齃齕龁龢𠀀𠁗𠗂𠘢𠚔𠛳𠡀𠧕𠰓𠳇𠳊𠵩𠶹𠶾𠻙𡀽𡇞𡇶𡫥𢄍𢅰𢥳𢬲𣆈𣏷𣣹𣰌𣲑𣲲𣿌𤈧𤌾𤖱𥉑𥋿𥘫𥝖𥝸𥞄𥞍𥞩𥟃𥻉𥽶𦃔𦇸𦒏𦘿𦛘𦛜𦞦𦤦𦳬𦺞𦼰𦼵𦽅𧀔𧇎𧇮𧊬𧝂𧝳𧨂𧪞𧬂𧬱𧭳𧮵𧯉𧲦𨋟𨍇𨜱𨜴𨨛𨴢𩄸𩅢𩌡𩐥𩑸𩩒𩩲𩵢𪈊𪖲𪘹𬌗"},
    {U"hei", U"㱄嗨嘿潶黑黒𢖛𢡀𥕙𨭆𩻤𬭶"},
    {U"hen", U"㯊䓳佷哏噷很恨拫掀狠痕艮詪鞎𦚣𬣳"},
    {U"heng", U"㔰㶇䬖䬝䯒亨佷哼啈堼姮恆恒悙桁横橫涥烆狟珩胻脝蘅行衡訇鑅鴴鵆鸻𠔲𠧿𡧦𣨉𤮏𥞧𦨾𦪗𦶙𧝒𩙯𪏓"},
    {U"hm", U"噷"},
    {U"hng", U"哼"},
    {U"hong", U"㖓㗢㢬㧦㬴㶹䀧䂫䃔䆖䆪䉺䎕䜫䞑䡌䡏䧆䨎䩑䪦䫹䫺䲨仜共厷叿吰吽呍哄哅嗊嚝垬妅娂宏宖屸巆弘彋愩揈撔晎汪汯泓洚洪浤浲港渱渹潂澋澒灴烘焢玒玜瓨硔硡竑竤篊粠紅紘紭綋红纮翃翝耾舼苰荭葒葓蕻薨虹訇訌謍讧谹谼谾軣輷轟轰鈜鉷銾鋐鍧閎閧闀闂闳霐霟鞃鬨魟鴻鸿黉黌𠐿𠪷𠲓𠳃𠹅𡇳𡵓𡺭𢂔𢗵𢘌𢝁𢝻𢦅𢬀𢼦𣽝𤂲𤃫𤄏𤟼𥈿𥏕𥓰𥔀𥕗𥥈𥥡𥰲𦁷𦏺𦐌𦐳𦑟𦑠𦒃𦕠𦕷𦶓𧈽𧊯𧋔𧐬𧮴𧾧𨋮𨌁𨌆𨎗𨢣𨥺𨹁𨾊𩐠𩒓𩒴𩒼𩓅𩕆𩕉𩖉𩗄𩗢𩘇𩘎𩙛𩰓𩶥𪈘𫚉𫟹𬭎"},
    {U"hou", U"㖃㗋㤧㫗㬋㮢㰯㰻㸸㺅䂉䗔䙈䞀䞧䪷䫛䳧侯候厚后吼吽呴喉垕堠帿後洉犼猴瘊睺矦篌糇翭翵腄葔詬豞逅郈鄇銗鍭餱骺鮜鯸鱟鲎鲘齁𠯜𠴣𠷋𡞥𡟑𡹵𢜴𣔹𣣠𣣡𤘽𤙽𥀃𥅠𥈑𥚦𦍵𦑚𦚥𦞈𦞕𧇹𧙺𧩨𧮶𧻿𧼵𨒙𩃺𩄬𩘋𩙡𪃶𪄗𪅺𪇂𪑻𪖙𬭤"},
    {U"hu", U"㕆㗅㦆㦌㦿㧮㧾㨭㪶㫚㯛㱿㳷㴶㷤㸦㺀㺉㽇㾰䁫䇘䈸䉉䉿䊀䊺䍓䎁䓤䕶䗂䚛䞱䠒䤕䧼䨚䨼䩐䩴䪝䬍䭅䭌䭍䰧䴣䴯乎乕乥乯互俿冱冴匢匫呼和唬唿喖嗀嗃嘑嘝嚛囫垀壶壷壺姱婟媩嫭嫮寣岵帍幠弖弧忽怘怙恗惚惡戯戲戶户戸戽扈抇护搰摢擭斛昈昒曶枑楛楜槲槴歑汩汻沍沪泘洿浒淈淲淴湖滬滸滹濩瀫烀焀煳熩狐猢琥瑚瓠瓡瓳礐祜穫笏箎箶簄粐糊絗綔縎縠羽胍胡膴芐芔芦芴苦苸萀葫蔛蔰虍虎虖虝蝴螜衚觳觷許謼護豰軤轷鄠醐鈷鋘錿鍙鍸隺雇雐雽韄頀頶餬鬍魱鯱鰗鱯鳠鳸鴩鵠鶘鶦鶮鶻鸌鹕鹱𠥰𠦪𠯳𠰛𠴱𡉴𡍐𡜂𡞠𡧥𡰅𡱽𡴱𡵘𡹹𡻮𡼘𢆰𢉢𢎵𢏯𢑢𢑹𢓲𢚪𢨥𢨦𢪏𢽨𣄟𣎚𣑂𣓗𣙶𣛫𣝗𣡾𣫈𣲑𣹬𤇠𤌍𤎲𤐀𤘔𤘵𤜷𤝘𤞲𤨖𤭱𤶘𤹣𤾅𥂤𥇰𥐿𥢍𥰪𥲉𥶜𥷆𥾨𦁕𦊂𦊘𦊧𦏗𦖼𦗣𦧘𦩕𦬚𦭈𦴉𦷳𦺟𧂔𧅰𧆢𧆮𧆯𧇛𧇰𧌧𧍵𧗌𧘢𧛞𧞒𧠩𧢰𧣼𧥮𧥯𧦚𧦝𧩓𧰴𧲇𧲥𧹲𧹾𧻰𧿓𧿠𨍲𨕚𨖃𨛵𨝘𨝞𨢋𨢤𨣗𨥛𨱀𨴬𩂂𩑶𩖨𩢪𩨔𩰯𩱍𩳨𩵬𩶈𩾇𩾻𪂒𪄮𪍂𪏳𪏻𪕉𪕮𪕱𪙈"},
    {U"hua", U"㓰㕦㕲㕷㚌㟆㠏㦊㩇㭉㮯㳸㼫䀨䅿䇈䋀䔢䛡䯏䱻䴳䶤侉划劃劐化华叱吪咶哇哗嘩埖夻姡婲婳嫿嬅學崋找搳摦撶敌杹桦椛槬樺檴滑澅澮猾獪画畫畵砉硴磆稞竵粿糀繣罫腂舙花芲華蒊蒍蕐蘤螖觟話誮諙諣譁譮话豁輠釪釫鋘錵鏵铧驊骅魤鮭鷨黊𠝐𠤎𠳂𠿜𡁑𡺛𢄶𢦚𢼤𣶩𤁪𤙕𤼒𥉄𥒶𥢮𥧰𦁊𦖍𦧠𦧵𦧹𦪠𦶎𦽊𧑍𧨋𧪮𧽌𨣄𨶬𨶱𩂤𩗐𩝨𩤉𩨔𩲏𩵏𩸄𪉊𪗽𫜸"},
    {U"huai", U"㜳㠢䃶䈭佪劃咶喟圳坏坯壊壞徊怀懐懷槐櫰淮瀤耲蘹蘾褢褱踝𣟉𣩹𣸎𤜄𦏨𦧬𧱳𩌃𩟮𪊉"},
    {U"huan", U"㕕㡲㣪㪱㬇㬊㵹㶎㹕㹖㼫㿪䀓䁵䄆䆠䈠䍺䑏䒛䝠䠉䥧䦡䭴䯘䴉䴋䴟䴷唤喚喛嚾圂圜垸奂奐嬛孉宦寏寰峘嵈巜幻患愌懁懽换換援擐攌桓梙槵欢欥歓歡汍洹浣涣渙漶澣澴灌烉焕煥犿狟獾环瑍瑗環瓛痪瘓皖眩睆睔瞏糫絙綄緩繯缓缳羦肒脘荁萈萑蒝藧螌蠸讙豢豩豲貆貛轘还逭還郇酄鉮鍰鐶锾镮闤阛雈雚驩鬟鯇鯶鰀鲩鴅鵍鸛鹮𠂄𠟼𠺐𠻍𡄤𡅱𡅻𡍦𡘍𡚊𡚜𡩂𡱌𡷗𢟿𣌓𣎅𣴟𤀣𤛚𤡟𤢁𤩽𤴯𤼐𤼢𤽅𤽕𥈉𥏇𥐓𥠅𥶍𥹚𥹳𦌦𦑛𦝝𦣴𦻃𦼉𧚁𧡩𧴊𨕹𨜌𨽧𩍡𩑖𩙽𩡧𩦘𩵄𩿊𪈩𪊥𪍺𬘫"},
    {U"huang", U"㞷㠵㡃㤺㨪㬻㼹㾮㿠䀮䁜䄓䅣䅿䊗䊣䌙䍿䐠䐵䑟䞹䪄䮲䳨䵃偟兤凰喤堭塃墴奛媓宺崲巟幌徨怳恍惶愰慌揘晃晄曂朚楻榥横櫎汻洸湟滉潢炾煌熀熿爌獚瑝璜癀皇皝皩磺穔篁篊簧縨肓艎芒茫荒葟蝗蟥衁詤諻謊谎趪遑鍠鎤鐄锽隍韹餭騜鰉鱑鳇鷬黃黄𠂸𡉚𡜋𡡄𡧽𡿰𢁹𢇟𣄙𣆖𣉪𣺬𤆴𤚝𤛥𤠛𤭉𤯷𤾑𥫼𥿪𦟮𦡽𦪗𦬺𦵽𧕸𧖬𧠬𧨆𨉁𨉤𨍧𨚳𨜔𨝴𨱑𨾡𩞩𩢯𪀞𪏍𪏒𪏙𫗮"},
    {U"hui", U"㑰㑹㒑㜇㞀㞧㣛㤬㥣㧑㨤㨹㩓㩨㫎㬩㰥㱱㷄㷇㷐㹆㺔㻅㾯䁤䂕䃣䅏䌇䏨䕇䖶䙡䛛䛼䜋䜐䝅䢈䤧䧥䩈䫭䵻䶐会佪僡儶匯卉叀咴哕喙嘒噅噕噦嚖囘回囬圚墮壞婎媈嬒孈寭屶屷幑廆廻廽彗彙彚徊徻徽恚恛恢恵悔惠慧憓懳戲拻挥揮撝晖晦暉暳會椲楎槥橞檅檓檜櫘殨毀毁毇汇沬泋洃洄浍涣湏滙潓澮濊瀈灰灳烜烠烣烩煇煒燬燴獩珲琿璤璯痐瘣皓眭睢睳瞺禈秽穢篲絵繢繪绘缋翙翚翬翽芔茴荟蒐蔧蕙薈薉藱蘬蘳虫虺蚘蛔蛕蜖螝蟪袆褘襘詯詼誨諱譓譭譮譿讳诙诲豗賄贿輝輠辉迴逥違銊鏸鐬鑴闠阓隓隳靧鞼韋韢頮顪颒餯鮰鰴麾鼿𠍗𠓊𠧩𠯠𠱥𠲛𠽡𠿔𡋙𡏁𡒾𡜦𡢕𡥋𡭛𡯥𡰋𡹎𡹯𢀡𢄣𢅫𢊄𢊇𢕺𢟾𢯷𢻔𣄓𣋘𣌭𣨶𣸀𤃽𤆳𤈦𤌋𤕚𤜋𤜡𤟤𤳳𤸁𤾈𥀠𥃌𥊔𥌍𥍋𥔯𥱵𥴯𥶵𥸃𦂆𦒎𦞙𦡖𦭹𦽐𧉇𧏧𧖢𧗏𧗼𧧾𧬨𧭾𧳐𨊢𨍹𨗥𨘇𨘲𨛤𨦗𨵘𩀶𩃾𩆁𩇻𩒏𩒳𩔁𩗝𩢱𩶥𩻟𪀟𪀬𪈑𪊂𪏇𪏏𪑀𪔊𪖕𪛂𫚔𬤝𬭬灰"},
    {U"hun", U"㑮㖧㗃㥵㨡㮯䅙䅱䊐䎜䐊䚠䛰䡣䧰䫟䮝䰟䴷䴹䵪俒倱圂堚婚婫忶惛惽慁捆掍揮昆昏昬梡梱棍棔殙浑涽混渾湣湷溷焄焝煇琿眃睧睯緄緡繉荤葷觨諢诨轋閽阍顐餛餫馄魂鼲𠉣𡇯𡨩𡨳𣇲𣝂𣣏𣣞𣨿𥄇𦃕𦞢𦟲𦡵𦵣𧠚𧣢𨂱𨋨𨏂𨡫𩅴𩇇𩏖𩔉𩧰𩽼𪌽𪑕"},
    {U"huo", U"㓉㖪㗲㘞㦎㦜㦯㨯㩇㯉㶁㶡㸌㺢䁨䂄䄀䄆䄑䉟䐸䣶䦝䨥䬉䰥䱛伙佸俰剨劐化吙呵和咟嚄嚯嚿壑夥奯姡惑或扮捇掝搉擭攉旤曤楇檴沎活湱漷濊濩瀖火灬焃獲瓠癨眓矆矐砉礊祸禍秮秳穫篧耠耯膕臛艧获萿蒦藿蠖諕謋豁豰貨货越趏過邩鈥鍃鑊钬锪镬閄隻霍靃騞𠙞𠯐𠵾𡄴𡓘𡪞𡯢𡾜𡿿𢃎𢋒𢛯𢝇𢞕𣄸𣉒𣤨𤁹𤆄𤊴𤏘𤐰𤬁𤻙𥇙𥊮𥒠𥙨𥝂𥽥𦑌𦑱𦒧𦞦𦨯𧆑𧤴𧯆𧯱𧵻𧹴𨌝𨐶𨘌𨙀𨷮𩆀𩞺𩟨𩟸𩦐𩪭𩭳𪒩𬴃"},
    {U"ji", U"㑧㒫㓹㔕㗊㗱㘍㙨㙫㚡㚻㛷㞃㞆㞋㞓㞛㞦㠍㠎㠱㡭㡮㤂㤅㥍㥛㦘㦸㧀㧗㨈㨳㫷㭲㮨㮷㰟㲅㲺㳵㴉㴕㸄㸅㹄㻑㻷㽺㾊㾵䀈䀘䁒䁶䂑䇧䇫䈕䋟䍤䐀䐕䐚䓽䕤䗁䗗䚐䛋䛴䜞䝸䞘䟇䟌䠏䢋䢳䣢䤒䦇䨖䩐䩯䮺䯂䰏䱥䲯䳭䳶䶓䶩丌丮乁乩亟亼亽伋伎佶倚偈偮僟其兾冀几击刉刏剂剞剤劑勣卙卟即卽厝及叝叽吇吉呰咭哜唧喞嗘嘰嚌圾坖垍基堲塈塉墼奇妀妓姞姫姬姼嫉季寂寄尐居屐屰岋岌峜嵆嵇嵴嶯己帺幾庴廭彐彑彶徛忌忣急悸惎愱憿懠懻戟戢技挤掎揖揤撃撠撽擊擠攲敧旡既旣暨暩曁朞机极枅梞棋棘楖楫極槉槣樭機橶檕檝檵櫅櫭殛毄汥汲泲洁洎济淁済湒漃漈潗激濈濟瀱焏犄犱狤猗玑璣璾畟畸畿疵疾痵瘠瘵癠癪皀皍睽瞉瞿矶磯祭禝禨积秸稘稩稷稽穄穊積穖穧笄笈筓箕箿簊簎籍粢系紀紒級結給継緝績繋繫繼级纪继绩缉罽羁羇羈耤耭肌胔脊脔脨膌臮艥艻芨芰苙茍茤荠莋萁葪蒩蒺蓟蔇蕀蕺薊薺藉蘄蘎蘮蘻虀虮蜡蝍螏蟣蟻蟿蠀裚襀襋覉覊覘覬覿觊觙觭計訐記誋諅諔譏譤计讥记诘谻谿賫賷赍趌趞跂跡跻跽踑踖踦蹐蹟躋躤躸輯轚辑迹郅郆鄿銈銡錤鍓鏶鐖鑇鑙际隔際隮集雞雦雧霁霵霽革鞊鞿韲颳飢饑饥騎驥骥髻鬾魕魝魢魥鮆鯚鯽鰶鰿鱀鱭鱾鲚鲫鳮鵋鶏鶺鷄鷑鸄鸡鹡麂齊齌齍齎齏齑𠀃𠀷𠋻𠍃𠑃𠓞𠔋𠗏𠚽𠟣𠢠𠦫𠨕𠨠𠫷𠮯𠯉𠱙𠱨𠲹𠴩𠴫𠶻𠷌𠹋𠼻𠿉𠿠𡁪𡁰𡃃𡅺𡇟𡋚𡜱𡥞𡦊𡦪𡪱𡫀𡬄𡳮𡹪𡽉𡿙𢁂𢃺𢆻𢉗𢍇𢏞𢓄𢔔𢗂𢗹𢚁𢜭𢡴𢢞𢨐𢩦𢭄𢰒𢱣𢺼𢼋𢼷𣄯𣄱𣇳𣏡𣔽𣖷𣛔𣜇𣣝𣪠𣬠𣯅𣰈𣱗𣲡𣳃𣹜𣽍𤊵𤋭𤌿𤎗𤓑𤛄𤜝𤜾𤠎𤤋𤫝𤳎𤵀𤷉𤺗𤺷𤿠𥈂𥊬𥋥𥒡𥕂𥖙𥘌𥝌𥠈𥠋𥡒𥡴𥣩𥨿𥪫𥪼𥫶𥭋𥭌𥭜𥮪𥰦𥳏𥷙𥺵𥾊𥾡𦁳𦂑𦆡𦇧𦋋𦌗𦌰𦎢𦜸𦝖𦞽𦠄𦠾𦩧𦪱𦮯𦮼𦳌𦵾𦶍𦺩𦺬𦺴𦺶𦼷𦾲𦿓𧃞𧇯𧈼𧉆𧉍𧎿𧐐𧓓𧗒𧙁𧟜𧡉𧡯𧢾𧤏𧥄𧧃𧧟𧧩𧩦𧪇𧪠𧫜𧫠𧮭𧽑𧾢𧾽𧾾𨀶𨂢𨄐𨅤𨊻𨋉𨍺𨐆𨒴𨖊𨛉𨛑𨛳𨜒𨢵𨣧𨤹𨦮𨪏𨮺𨲪𨳋𨳻𨸚𨹶𨻕𩀖𩉜𩉢𩐆𩓮𩚮𩜆𩞊𩠨𩢴𩥉𩦤𩧱𩨒𩩛𩯋𩯰𩴃𩼄𩼚𪂍𪂺𪄵𪄸𪅹𪊆𪌍𪔋𪘥𪟝𫌀𫓯𫓹𬯀𬶨𬶭"},
    {U"jia", U"㕅㚙㪴㮖㸦㹢㿓䀫䀹䂟䑝䕒䕛䘥䛟䩡䴥乫价伽佳假傢價加叚呷咖唊嘉嘏圿埉夏夹夾婽嫁宊家岬幏徦忦恝戛戞扴抸押拁拮挈挾揩揳擖斚斝暇架枷柙梜椵榎榢槚檟毠泇浃浹犌猰猳玾珈甲痂瘕稼笳筴糘絜耞胛腵茄荚莢葭蛱蛺蝦袈袷裌豭貑賈贾跏跲迦郏郟鉀鉫鉿鋏鎵钾铗镓頡頬頰颊餄駕駱驾骱鴶鵊麚𠷉𠺢𡊠𡩚𡭘𡶥𢉤𢗊𢜿𢪍𢫢𢱈𢱌𣦉𣪇𣮫𤖰𤗜𤟚𤠙𥇗𥋣𥑐𥑔𥝿𥞵𥡮𥹌𦎮𦎱𦙺𦣯𦧮𦨦𦸘𦺧𦽤𧉪𧦤𧿵𨒇𨔗𨔣𩉡𩊏𩌍𩚲𩛩𩠃𩨹𩲣𩶛𪆲𪇷𪈟𪐓𪔟𪙏𬂩"},
    {U"jian", U"㓺㔋㔓㡨㣤㦰㨴㨵㪠㭴㯺㰄㳨㵎㶕㹇䄯䅐䇟䉍䌑䌠䓸䔐䘋䚊䟅䟰䤔䥜䧖䫡䬻䭈䭕䭙䭠䮿䯡䵡䵤䶠䶢䶬件侟俭俴倹健傔僣僭儉兼冿减前剑剣剪剱劍劎劒劔劗咸喊囏囝坚堅堿塹墹奸姦姧孱寋尖帴幵建弿彅徤惤戋戔戩戬拣挸捡揀揃揵搛撿擶攕旔暕朁枧柬栫梘检検椷椾楗榗樫橏橺檢檻櫼歼殱殲毽沮洊涀涧淺渐減湔湕溅漸澗濫濺瀐瀳瀸瀽煎熞熸牋牮犍犴猏玪珔瑊瑐监監睑睷瞯瞷瞼硷碊碱磵礀礆礛稴笕笺筧简箋箭箴篯簡籈籛糋絸緘縑繝繭纖缄缣翦聻肩腱臶舰艦艰艱茛茧荐菅菺葌葥蒹蔪蕑蕳薦藆虃螹蠒袸裥襇襉襺見覵覸见詃諓諫謇謭譖譼譾谏谫豜豣賎賤贱趝趼跈践踐踺蹇軒轞醎醶釰釼鉴銒鋑鋻錢錽鍊鍳鍵鏩鐗鐧鐱鑑鑒鑬鑯鑳锏键閒間间險靬鞬鞯韀韉餞餰饯馢騫鬋鰎鰔鰜鰹鲣鳒鳽鵳鶼鹣鹸鹹鹻鹼麉黚黬齊𠊒𠍚𠏇𠐻𠫘𠷁𠹟𠼤𠽱𠿏𡄑𡄓𡅶𡑯𡬵𡭭𡸃𡾰𢃬𢆞𢆦𢍫𢐆𢦺𢨿𢩀𢳚𢵈𣘖𣘷𣚙𣜭𣝕𣠷𣥞𣮏𣳲𣴓𣽖𣽦𤀩𤄒𤍖𤧣𤪋𤷃𥀹𥊇𥌈𥍀𥍹𥜲𥡝𥢇𥯦𥳒𥳟𥳷𥴱𥻧𥽐𦁲𦂇𦋰𦏔𦞬𦢣𦣨𦩵𦺍𦺘𦽇𦾶𧀇𧀵𧂂𧂆𧂢𧅆𧗸𧙧𧢖𧤨𧤽𧥈𧧅𧬫𧮈𧲨𧸘𨍒𨎫𨏊𨔥𨢑𨣇𨣝𨤄𨪅𨰓𨳡𨳿𨴾𨵭𨷓𩅼𩆯𩆷𩇏𩉍𩉔𩋋𩌯𩍎𩎒𩛧𩟗𩤊𩱃𩻘𩽜𪃋𪃛𪆿𪉦𪋁𪏊𪐻𪒫𪒹𪙨𬣡"},
    {U"jiang", U"㢡㯍㹔䁰䉃䋌䒂䗵䜫䞪䥒傋僵勥匞匠塂壃夅奖奨奬姜将將嵹弜弶强彊摪摾杢桨槳橿櫤殭江洚浆滰漿犟獎畕畺疅疆礓糡糨紅絳繮绛缰翞耩膙茳葁蒋蔃蔣薑螀螿袶講謽讲豇酱醤醬降韁顜鱂鳉𠘌𠷐𠼢𡏞𡑶𡲣𡷍𢘸𢪇𣚦𣨣𣩴𣫳𤕭𤕯𤖛𤛜𥆅𥔣𥗪𥞜𥬮𥷃𦦗𧘍𨃇𨜰𨯞𩌾𩝽𩴒𩷄𩷭𪀘𫮬"},
    {U"jiao", U"㠐㡑㩰㬭㭂㰾㲬㳅㶀㹾㽱㽲䀊䁶䂃䉰䌭䍊䕧䘨䙼䚩䡈䢒䥞䰘䴔䶰交佼侥僑僥僬儌剿劋勦卻叫叽呌咬喬嘂嘄嘐嘦噍噭嚼妖姣娇嫶嬌嬓孂學峤峧嵺嶕嶠嶣徺徼恔悎憍憢憿挍挢捁搅摎摷撟撹攪敎教敥敫敽敿斠晈暞曒校椒樔橋櫵浇湫湬滘漖潐澆激灂灚烄焦煍燋燞爝狡獥珓璬皎皛皦皭矫矯礁稾穚窌窖笅筊簥糾絞繳纐绞缴胶脚腳膠膲臫艽芁茭茮菽萩蕉蕎藠虠蛟蟜蟭覺角訆譑譥賋趫趭跤踋蹻較轇轎轿较郊鄗酵醮釂釥鉸鐎铰隦餃饺驕骄骹鮫鱎鲛鵁鵤鷦鷮鹪𠕧𠘣𠜅𠝑𠞰𠩏𡏭𡓖𡙎𡟠𡥈𡬋𢀌𢄺𢅎𢒾𢕪𢥚𢧱𢯴𢶡𢺽𢻟𢼫𣁹𣏑𣝞𣧦𣩓𣺳𤃭𤉧𤕝𤫷𤶀𤶳𤺃𥂨𥃤𥃧𥃪𥄉𥅟𥇟𥉒𥉼𥏹𥘊𥡤𥦢𥳴𥹜𦅃𦌆𦗵𦫶𦮁𧂈𧎙𧎸𧣦𧺜𧾐𨇕𨎦𨎬𨓩𨖵𨝰𨡃𨨴𨱓𨲭𨶟𨶪𨶲𨸋𨺹𩊔𩎔𩏷𩥊𩯘𩱞𩴧𩵰𩾗𩾾𩿑𪁉𪖄𪚰"},
    {U"jie", U"㑘㓗㓩㔚㔾㘶㛃㝏㞏㞯㠎㠹㡇㦢㨗㨩㫸㮞㮮㸅㼪㾏㿍䀷䀹䁓䂝䂶䃈䅥䇒䌖䕙䕸䗻䛺䣠䥛䦈䫘䯰䰺䱄䲙䲸丯亥介价借倢假偈偕偼傑價刦刧刼劫劼卩卪吤唧唶啑喈喼嗟嚌圾堦堺契她妎姐婕媎媘媫嫅孑家尐屆届岊岕崨嵑嵥嶰嶻巀差幯庎徣悈戒截扢担拮拾捷接掲掶揭搩擑擮擳斺昅暨杰桀桔桝椄楐楬楶楷概榤檞櫭毑洁洯渴湝滐潔煯犗狤獬玠琾界畍疌疖疥痎癤皆睫砎砝碣礍祖秸稭竭節籍紇紒結絜结罝羯耤脥脻艐节芥苴莭菨蓵藉蚧蛣蛶蜐蝍蝔蠘蠞蠽街衱衸袓袷袺裓褯解觧訐詰誡誱諎謯讦诫趌跲踕迼鉣鍇鍻鎅阶階雃鞂鞊頡颉飷骱髻魝魪鮚鲒鶛𠂈𠄍𠅂𠎿𠐉𠓢𠙤𠬮𠯙𠲵𠷟𡉷𡔣𡗦𡗲𡙣𡣯𡨲𡩣𡴱𡵒𡵚𡸎𡽱𢈻𢎔𢎡𢢂𢨜𢪍𢫐𢬱𢮌𢱄𢴲𢷿𢻮𣆟𣓉𣙴𣚃𣬫𣮌𣮍𣰞𣲡𣳟𣶏𤁢𤘦𤙩𤭧𤮌𥁂𥅴𥇒𥓐𥝔𥝥𥠹𥢻𥵞𥷫𥾌𦀓𦀖𦁉𦈜𦈰𦎒𦝨𦵴𦺢𦿐𧉧𧍠𧍩𧏥𧜅𧞝𧞩𧞬𧣋𧪘𧫑𧼨𧽄𧽟𧾢𧾯𨃲𨐑𨓰𨕽𨥂𨵠𩔄𩘅𩟦𩟶𩡺𩢴𩧦𩧵𩩰𩯰𩾶𪀾𪁍𪃈𪅸𪇲𪉋𪉚𪌧𪑹𪖋𪙏"},
    {U"jin", U"㝻㦗㧆㨷㬐㬜㯲㯸㱈㴆㶦㶳㹏㻱䀆䃡䃸䆮䈥䈽䋮䌍䌝䐶䑤䒺䗯䘳䝲䤐䤺䥆䫴䭙䶖仅今伒侭僅僸儘兓凚劤劲勁卺厪吟唫噤嚍埐堇堻墐壗妗婜嫤嬐嬧寖尽嶜巹巾廑惍慬搢斤斳晉晋枃榗槿歏殣津浕浸湛溍漌濅濜烬煡燼珒琎琻瑨瑾璡璶盡矜矝砛祲禁竻笒筋紟紧紾緊縉缙肋臸荕荩菫菳蓳藎衿襟覲觐觔謹谨賮贐赆近进進金釒釿鋟錦钅锦靳饉馑馸鹶黅齽𠂟𠞱𠞾𠢱𠢵𠬶𠰇𠾤𠾬𡢳𡥦𡥨𡺽𢉅𢎭𢙿𢦊𢬬𢬶𢱷𢽖𣓏𣝌𤄼𤘡𤣶𤧫𤵞𥂵𥖜𥧲𥯑𦈟𦎷𦘔𦞬𦧈𦩏𦽔𧔷𧗁𧢹𨆃𨔥𨚡𨭺𩀿𩖗𩤿𩻛𪉢𪏴𪑙𪖼𬬱"},
    {U"jing", U"㘫㢣㣏㬌㵾㸒䔔䜘䝼䪫䴖䵞丼井京亰仱俓倞傹儆兢净凈刭剄勁坓坕坙境妌婙婛婧宑巠幜弪弳径徑惊憬憼擏敬旌旍晟景晶暻曔桱梷橸檠殑氏汫汬泾浄涇淨瀞烴燝猄獍獷璄璟璥痉痙睛秔稉穽竞竟竧竫競竸箐粇粳精経經经聙肼胫脛腈茎荆荊莖菁葏葝蜻蟼誩警踁迳逕醒鋞鏡镜阱陘青靓靖靘静靚靜頚頴頸颈驚鯨鲸鵛鶁鶄麖麠鼱𠑱𠗊𠗌𠦋𠭉𠭗𠲮𠳬𠷐𡁔𢀖𢈴𢹘𣋢𣬙𣻒𤜰𤰳𤷦𥅸𥠛𥯙𥶹𦀇𦂠𦜳𦥍𦳲𦽁𦾿𧑊𧓔𧚫𧤵𨁎𨙷𨥙𨵼𩃋𩇕𩓞𩓨𩩋𩰹𩳯𩹢𩻱𪂴𪇒"},
    {U"jiong", U"㓏㢠㤯㯋㷗㷡䅃䌹䢛侰僒冂冋冏囧坰垧埛扃扄昋泂浻澃瀅炅炯烱煚煛熒熲燛窘絅綗臦臩蘏蘔褧迥逈銄鎣顈颎駉駫𠕕𠖷𡆩𢂶𢄗𣅻𣔲𣕄𤌇𦀝𧍮𨴀𩓺𩚱𪔃𪕍𬳶"},
    {U"jiu", U"㙀㝌㠇㡱㤹㥢㧃㩆㲃㸨㺩㺵㽱䅢䆒䆶䊆䊘䓘䛮䡂䬨䰗䳎䳔丩久乆九乣倃僦剹勼匓匛匶厩咎啾噍奺就廄廏廐愁慦捄揂揪揫摎救旧朻杦柩柾桕樛欍殧氿汣湫灸牞玖疚稵穋究糺糾紤繆纠臼舅舊舏萛蝤赳酒镹阄韭韮鬏鬮鯦鳩鷲鸠鹫麔齨𠃖𠃺𠕴𠖬𠙔𠚨𠛩𠜃𠜉𠠳𠣿𠴄𠿈𡆥𡚮𢀙𢑇𢜥𢽭𣁭𣐤𣟼𣲄𤉥𤴥𤴦𤴪𤷑𥆷𥘦𥠃𥤳𦠢𦭺𦭻𦱠𦱱𦱲𦽬𧡑𧫾𧳫𧾻𨖏𨘂𨘮𨳊𨾉𨾞𩏶𩏷𩒦𩢹𩭓𩱼𩶧𪀏"},
    {U"ju", U"㖩㘌㘲㜘㞐㞫㠪㡹㥌㨿㩀㩴㪺㬬㮂㯯㵵㹼㽤䀠䃊䄔䅓䅕䈮䋰䎤䏱䕮䗇䛯䜯䝻䡞䢸䢹䣰䤎䪕䪶䰬䱟䱡䳔䴗䵕䶙䶥且举乬仇佝侷俥俱倨倶僪具冣凥剧劇勮匊句告咀啹坥埧埾壉姐姖娵娶婅婮寠局居屈屦屨岠岨崌巈巨巪弆忂怇怐怚惧愳懅懼抅拒拘拠拱挙挶捄据掬揈揟據擧昛枸柜桔梮椇椈椐榉榘橘檋櫸欅歫毩毱沮泃泦洰涺淗渠湨澽炬烥焗焣爠犋犑狊狙珇琚疽痀眗瞿矩砠租秬窭窶筥簍簴籧粔粷罝耟聚聥腒臄舉艍苣苴莒菊菹萭蒟蒩蓻蔞蘜蘧處虡蚷蛆蜛螶袓裾襷詎諊讵豦貗趄趉趜趡足跔跙距跼踘踞踽蹫蹻躆躹車軥輂遽邭郰郹鄒鄹醵鉅鉏鋤鋦鋸鐻钜锔锯閰陱雎雛鞠鞫颶飓駏駒駶驕驧驹鬻鮈鮍鮔鴡鵙鵴鶋鶪鼰鼳齟龃𠉧𠋬𠙆𠚵𠜹𠟰𠤄𠨭𠩉𠮑𠰾𡉎𡒍𡕖𡢒𡥶𡨅𡨢𡫬𡱾𡳘𡶋𡸘𡸨𡿾𢚆𢤫𢩁𢯺𣌬𣍇𣎛𣐊𣓌𣖣𣶝𣻐𤓯𤔋𤖵𤜔𤢓𤷢𤼳𥂃𥄷𥇛𥈋𥉁𥕑𥘮𥢧𥪏𥬙𥮗𥯔𥲜𥴧𥴪𥷚𦀣𦅽𦇙𦊐𦗻𦙮𦛓𦜛𦞇𦟳𦥑𦱅𦺖𦼈𧂜𧄛𧝲𧣒𧣻𧤑𧲋𧵞𧷾𧸧𧹕𧺤𧺹𧻗𧽻𧾣𧿻𨁺𨋔𨋧𨍯𨗝𨛮𨝮𨧙𨨠𨸰𩉸𩋜𩍔𩍸𩛺𩜃𩧒𩧺𩫴𩬜𩭊𩰤𩳵𩴘𩷐𩿝𪀏𪀣𪁥𪂓𪗖𬶋"},
    {U"juan", U"㢧㢾㪻㯞㷷䄅䅌䌸䖭䚈䡓䣺䳪倦劵勌勬卷呟圈埍埢奆姢娟婘巂巻帣弮悁惓慻捐捲擐朘桊梋棬泫涓淃焆狷獧瓹甄眩眷睃睊睠絭絹縳绢罥羂脧腃臇菤萒蔨蕊蜷蠲裐襈踡身鄄鋑鋗錈鎸鐫锩镌闂隽雋鞙韏飬餋鵍鵑鹃𠔉𠡶𠢚𡘰𡡀𡫂𡱑𢃩𢋄𢍏𢎠𢎥𢝓𣙢𣚓𣜨𣬋𣬏𤎱𤲨𤺻𥁠𥅬𥆞𥱽𦁆𦊌𦢿𦦽𦬾𦮻𦳽𦼱𧎖𧕲𧭦𧯦𨆈𨌫𨤑𨩸𨹵𩎳𩏗𩏹𩔱𩘒𩜇𩟥𩠉"},
    {U"jue", U"㔃㔢㟲㤜㩱㭈㭾㰐㲄㵐㷾㸕㹟㻕䀗䁷䇶䊽䍳䏐䏣䐘䖼䘿䙠䝌䞵䞷䟾䠇䡈䣤䤎䦆䦼䳏䶂乙亅倔傕决刔劂勪匷厥叕吷啳噘噱嚼埆壆夬妜孒孓屈屩屫崛崫嶡嶥弡彏憠憰戄抉挗捔掘撅撧攫斍柽桷梏構橛橜欔欮殌氒決泬潏灍焳熦燋爑爝爴爵狂獗玃玦玨珏瑴璚疦瘚矍矞矡砄穱穴絕絶繑繘绝脚臄芵蕝蕞蕨虳蚗蛙蟨蟩蠼袦覐覚覺觉角觖觳觼訣誳譎诀谲貜赽趉趹蹶蹷蹻躩較逫鈌鐍鐝钁镢闋闕鞽駃騤髉鱖鴂鴃鶌鷢龣𠀔𠄌𠄑𠊬𠎮𠜾𠢤𠨊𠫃𠶸𡈅𡚠𡲗𡳾𡾜𢁪𢎹𢏷𢔱𢖦𢨏𢩯𢯅𢱺𢴭𣅡𣖬𣨢𣬎𤛦𤞴𤟎𤹋𤼗𥄴𥆌𥈾𥏘𥕲𥛯𥤘𥪊𥾮𦁐𦏅𦓐𦛲𦠒𦪘𧍕𧗫𧝃𧣸𧤼𧥎𧮫𧮬𧱝𧺐𧼨𧽸𧾵𧿺𨊿𨏹𨟠𨬐𨰜𨼎𨼱𩊺𩌡𩍷𩏺𩓻𩧏𩧡𩪗𩰨𪁠𪈴𪖜𪚅𪨗𫏋𫔎𫘝𫛞"},
    {U"jun", U"㑺㒞㕙㖥㚬㝦㴫㻒㼱㽙䇹䐃䕑䜭䝍俊儁军匀卷君呁均埈姰寯峻懏捃攈攟旬晙桾棞汮浚濬焌焞燇狻珺畯皲皸皹睃碅竣筠箘箟莙菌葰蔨蚐蜠袀覠訇軍郡鈞銁銞鋆鍕钧陖隽雋餕馂駿骏鮶鲪鵔鵕鵘麇麏麕龜𠀹𠣕𠨢𢉦𢹲𢻸𤮪𥇘𥚂𥜮𥡣𦇘𦌺𦴌𦵼𧥺𧯖𧱝𧽔𨌘𨛐𨲄𨶊𪍁𪕞"},
    {U"ka", U"㮟䘔佧卡呿咔咖咯喀垰擖胩衉裃鉲"},
    {U"kai", U"㚊㪡㱾䁗䐩䒓䠽䡷䤤䫦凯凱剀剴劾勓喝喫嘅垲塏奒嵦幆开忾恺愒愷愾慨揩暟核楷欬欯渴溘濭炌炏烗蒈豈輆鍇鎎鎧鐦铠锎锴開閡闓闿雉颽𠢲𡁁𡙓𡳂𢋝𢔡𢢚𢾆𤉫𤐩𤡲𤸱𤻜𥃣𥎆𥏪𥻄𦂄𦈲𨴆𩋧𩫀"},
    {U"kan", U"㘛㙳㪁㸝䀍䁍䖔䘓䫲䳚侃偘冚凵刊勘喊坎埳堪堿塪墈崁嵁嵌惂戡扻栞槛檻欿歁監看瞰矙砍碪磡竷莰薟衎輡輱轁轗闞靬顑餡龕龛𠝲𠪚𡚜𡶪𡸞𡺗𢙮𢦟𣊟𣣒𣽌𤬪𥍓𥑫𥓒𥤱𥦔𦞖𧇦𧡵𧯰𧱄𨍜𨒞𩐬𩑟𩒃𩓟𩜱𪉯"},
    {U"kang", U"㝩㢜㱂㼹䆲䗧䡉䦎亢伉匟囥坑奋嫝嵻康忼慷扛抗摃杭槺沆漮炕犺砊穅粇糠羫荒躿邟鈧鏮钪閌闶阬骯鱇𠻞𠾨𡐓𡵻𡻚𢴦𣔛𤮊𥉽𥒳𥕎𥹺𨀫𨂟𨄗𨎍𨝎𨻷𩾌𪎵"},
    {U"kao", U"㸆㼥䎋䐧䯌䯪丂尻嵪拷搞撟攷栲槀槁洘烤焅熇犒稾考薧訄銬铐靠髛鮳鯌鲓𡭳𣐊𣧏𣨻𣩅𥬯𥹬𧋓𨃤𨘴𩝝𩩾"},
    {U"ke", U"㕉㕎㝓㞹㤩㥛㪙㪡㪼㵣㸯䂺䆟䈖䌀䐦䓇䙐䡷䫘䯊䶗克刻剋勀勊匼可呵咳喀嗑坷堁壳娔客尅岢峇嵑嵙嶱恪悈愘愙揢搕敤柯棵榼樖歁殼毼氪渇渴溘濭炣牁犐珂疴痾盍瞌砢硞碣碦磆磕礊礚科稞窠窼簻緙缂翗胢艐苛萪薖蚵蝌袔課课趷軻轲醘鈳鉿錁錒钶锞頦顆颏颗騍骒髁龕𠏀𠛳𠡜𠡤𠢹𠩧𠪒𠪟𠲙𠳭𠶲𠷄𡞢𡱼𡸡𡻘𢈈𢩏𢩐𢩘𢼐𢾩𣧤𣩄𣩱𣲊𣹇𤖇𤛗𤰙𤸎𤸱𥃕𥊉𥔽𥝹𥠁𥦨𥧇𥯚𦝣𧈗𧎗𧛾𧜡𧠋𧨵𧵛𧿫𨍰𨏿𨢸𩏭𩜭𩭽𩰻𩱘𪀉𪃭𪍎𪓮"},
    {U"kei", U"刻剋"},
    {U"ken", U"㸧啃垠垦墾恳懇掯狠珢硍肎肯肻裉褃豤貇錹頎齦龈𠳁𣍟𣥤𤀊𥖞𨼯𩎤"},
    {U"keng", U"㧶㰢䀴䃘䡩䡰劥吭坈坑奟妔忐挳揁摼殸牼硁硍硎硜硻脛誙踁鉺銵鍞鏗铿阬𠠷𠡮𡞚𡷨𣢐𣢴𣫒𥉸𥑅𥑒𥒁𥒱𥥳𧀘𨋔𨌳𨌶𨍑"},
    {U"kong", U"㚚㤟㲁㸜䅝倥埪孔崆恐悾控椌涳矼硿穹空箜腔躻錓鞚鵼𠀝𡤿𢃐𢪬𢷙𢽦𣏺𤗇𤤲𥔇𥥅𦁈𦱇𦶐𧌆𧚬𩲧𪔣"},
    {U"kou", U"㓂㜌㰯㸸䁱䍍䳹佝冦刳剾劶區口叩嫗宼寇彀彄怐扣抠挎摳敂毆溝滱眍眗瞉瞘窛竘筘簆芤茠蔲蔻釦鏂鷇𠛅𡠆𡯷𢂁𢄠𢚫𢟭𢼃𣂻𣻎𣿟𤘘𥊧𥲃𦬅𦴎𦶲𧥣𨙫𨥴𩀠𪄺𪇄𫃜𫸩"},
    {U"ku", U"㗄㠸㩿㪂㱠㲄㵠䂗䇢䉐䔯䧊䯇䵈俈刳古哭喾嚳圐圣堀崫库庫廤扝挎捁掘搰朏枯桍楛泏焅狜瘔矻硞秙窋窟絝绔苦袴裤褲趶跍跨郀酷骷鮬齁𠠶𠺟𡀙𡉄𡑚𡑣𡗵𡞯𡶏𡼿𢏆𢼁𣗺𥈷𥌄𥞴𥟾𥧋𥯶𦛏𦜇𦡆𧊘𧠂𧷎𧿉𧿋𨐡𨡱𩇵𩊓𩑔𩑡𩑶𩨳𩱙𪇄𪌓𪍠"},
    {U"kua", U"㐄㛻㡁䓙䠸䦚䦱䯞侉咵垮夸姱恗挎晇楇絓胯舿華袔誇跨銙錁顝骻髁𠇗𡇚𡕒𡗢𡗸𡱼𢄳𢓢𥏤𥑹𦮃𨃖𨕺𨵧𩊓"},
    {U"kuai", U"㔞㕟㙕㟴㧟㱮㹟䈛䓒䭝䯤侩傀儈凷哙噲圦块塊墤巜廥快擓旝會檜澮狤狯獪璯筷糩脍膾蒯蕢郐鄶駃鬠魁鱠鲙𠜐𠣲𡚅𡼾𢢞𢾒𣫉𥢶𦔦𦳋𨛖𩦱𫐆"},
    {U"kuan", U"㯘㱁䕀䥗䲌完宽寛寬梡棵欵款歀窽窾臗鑧顆髋髖𢕫𢴪𣎑𣢻𣽟𥟓𥦀"},
    {U"kuang", U"㑌㚚㾠䊯䒰䖱䯑䵃丱儣兄况劻匡匩卝呈哐圹壙夼岲廣忹恇懬懭抂旷昿曠枉框況洭湟爌狂狅眖眶矌矿砿硄磺礦穬筐筺絋絖纊纩誆誑诓诳貺贶軖軠軦軭迋逛邝邼鄺鉱鋛鑛鵟黋𡶢𡾇𢼑𢼳𣍦𣒸𣴥𤝿𤞇𥈏𦚞𦥰𧥌𧻔𧻺𧿈𨀕𨇁𨏆𨖢𨥑𨨭𨴑𩢼𩬹𩷗𪍿𪏪𫛭"},
    {U"kui", U"㒑㕟㙓㙺㚍㨒䃬䕚䕫䖯䙆䙌䙡䞚䟸䠏䠑䠿䤆䧶䫔䫥䯓䯣䰎䳫亏傀刲匮匱喟喹嘳夔奎媿嬇尯岿巋巙悝愦愧憒戣揆晆暌楏楑樻櫆欳歸殨溃潰煃犪盔睽瞆瞶磈窥窺篑簣籄缺聧聩聭聵胿腃膭臾葵蒉蒍蕢藈蘬蘷虁虧蝰觖謉跬踩蹞躨逵鄈鍨鍷鐀鑎闋闚隗頃頄頍頯顝餽饋馈馗騤骙魁鮭𠊾𠣠𠿥𡌤𡐠𡒾𡓰𡤞𡯵𢌳𢜽𢼀𣄲𣥮𣧼𤆂𤏜𤵮𥁇𥏙𥜶𥪊𦝢𧂠𧄑𧍜𧑋𧝷𧡦𧡫𧢦𧷛𧻜𧻲𨣈𨾎𨾗𩏣𩓗𩕜𩠮𩲅𩲷𩵉𩹍𪆴𪖢𫠆"},
    {U"kun", U"㡓㩲㫻㱎䐊䖵䠅䪲卵困坤堃堒壸壼婫尡崐崑悃捆昆晜梱涃混潉焜熴猑琨瑻睏硱祵稇稛綑罤菎蜫裈裍裩褌豤貇醌錕锟閫閸阃頑餛騉髠髡髨鯤鰥鲲鵾鶤鹍齦𠚯𡖉𢈛𣏔𣰘𣱂𥊽𥚛𦄐𦌸𦓾𧋕𧥊𨁉𨱙𩓽𩤋𩨫𩭋𩻋𩽞𪋆𪘩"},
    {U"kuo", U"㗥㾧䟯䦢䩹䯺噋噲廓懖扩拡括挄擴會栝桰漷濶燭秳筈萿葀蛞适鄺闊阔霩鞟鞹韕頢髺鬠𠚳𠠎𡎒𡻙𢠛𤫵𥕏𦧍𦧔𨓈𨨱𨶐𩋻𪗽"},
    {U"la", U"㕇㡴㩉㯿㱞㸊㻋㻝䂰䃳䏀䏠䓥䖃䗶䱨䱫䶛儠剌啦喇嚹垃拉揦揧搚摺擸攋旯柆楋溂爉瓎瘌癩砬磖翋腊臈臘菈落藞蜡蝋蝲蠟辢辣邋鑞镴鞡鬎鯻鱲𠾩𡅘𡉆𢃴𢈠𢉨𢤆𣤊𤀦𤊶𤛊𤰚𤷟𥀥𥀰𥈙𥖍𥗿𥘁𦅶𦆻𦇛𦎏𦒆𦒦𧗩𧙀𧞪𧩲𨭛𩃜𩋷𩑮𩘊𩤲𩧆𩨉𩯽𪇹𬶟"},
    {U"lai", U"㚓㥎㸊䂾䄤䅘䋱䓶䚅䠭䧒䲚來俫倈勑厲唻娕婡崃崍庲徕徠懶攋来梾棶櫴涞淶濑瀨瀬猍琜癘癞癩睐睞筙箂籁籟莱萊藾襰誺賚賴赉赖逨郲釐錸铼頼顂騋鯠鵣鶆麳黧𠎙𠘝𡂖𡃄𡓒𢅭𢑬𣖤𤢗𤢵𤦃𤲓𦆋𦓹𧯲𧳕𧳟𧵭𨂐𨇆𨦂𩳆𪈈𪑚"},
    {U"lan", U"㑣㔋㘓㘕㛦㜮㞩㦨㨫㩜㰖㱫㳕䃹䄤䆾䈒䌫䍀䑌䦨䪍䰐僋儖兰厱啉嚂囒囕坔壈壏婪嬾孄孏岚嵐幱廩惏懒懔懢懶拦揽擥攔攬斓斕暕栏榄欄欖欗浨湅滥漣漤澜濫瀾灆灠灡烂煉燗燣燷爁爛爤爦璼瓓礷篮籃籣糷繿纜缆罱葻蓝藍蘫蘭褴襕襤襴襽覧覽览諫譋讕谰躝連郴醂鑭钄镧闌阑韊顲𠓖𠓭𠼖𡒄𡓔𡮻𡽳𢅡𢉧𢊓𢒞𢛓𢹙𣋣𣛱𤂺𤃨𤑸𤣟𥌻𥗺𥗽𥜓𥦝𦧼𧮤𧸦𧼖𨅏𨅬𨊔𨎹𨣨𨣸𨬒𨷻𩈵𩉀𩔵𩟺𪇖𬒗"},
    {U"lang", U"㓪㙟㝗㟍㢃㫰㮾㱢㾿䀶䆡䍚䕞䡙䯖䱶俍勆哴唥啷埌塱嫏崀廊悢斏朖朗朤桹榔樃樠欴浪烺狼琅瑯硠稂筤羹脼艆莨蒗蓈蓢蜋螂誏踉躴郎郒郞鋃鎯锒閬阆駺𠺘𠻴𢳑𢽂𣊧𣻡𥇑𥍫𥧫𦵧𦺫𧚅𧻴𨞿𨱍𨶗𩛡𩲒𩳤𩷕𪁜"},
    {U"lao", U"㗦㞠㟉㟹㧯㨓㺐䃕䇭䕩䜎䜮䝁䝤䲏䳓䵏佬僗僚劳労勞咾哰唠嗠嘐嘮姥嫪嫽崂嶗恅憥憦捞撈撩朥栳橑橯浶涝潦澇烙牢狫獠珯痨癆硓磱窂簩粩絡老耂耢耮荖落蓼蛯蟧躼軂轑酪醪銠鐒铑铹顟髝鮱𠈭𡂕𡑍𡬘𢚄𢭂𣘪𣟽𣠼𤉍𤎤𤛮𤩂𤶁𥢒𦒴𦛨𦺜𧢋𧯍𨡤𨣃𨦭𨲮𪀧𪁔𫭼"},
    {U"le", U"㔹㖀㦡㿭乐了仂勒叻嘞忇扐楽樂氻泐牞玏砳竻簕肋艻阞韷餎饹鰳鳓𡃖𣂒𣲒𤟓𤨙𥖪𩐾"},
    {U"lei", U"㑍㒍㒦㔣㗊㙼㠥㲕㴃㵢㵽㶟㹎㼍㿔䃬䉂䉪䍣䐯䒹䛶䢮䣂䣦䨓䮑䴎傫儡儽勒厽嘞垒塁壘壨婁嫘擂攂樏檑櫐櫑欙泪洡涙淚漯灅瓃畾瘣癗盧矋磊磥礌礧礨祱禷类累絫縲纇纍纝缧罍羸耒肋腂蔂蕌蕾藟蘱蘲蘽虆蠝誄讄诔轠郲酹銇錑鐳鑘鑸镭雷靁頛頪類颣鱩鸓鼺𠱤𡈶𡔇𡚗𡰠𡻭𡻱𡼊𡾋𡾖𡿉𡿛𢴱𢹮𣀀𣀜𣚎𣠠𣡧𣡺𣨅𤃻𤜖𤡂𤢹𤮎𤮚𤮸𤳳𤳴𤼘𥅦𥍔𥑶𥗬𥗶𥣬𥤐𦇄𦓥𦢏𦣄𧒜𧒽𧞭𨀤𨄱𨊚𨏒𨞽𨶺𨻌𩑚𩔗𩛝𩴻𩵓𪑯"},
    {U"len", U"啉"},
    {U"leng", U"㘄㱥䉄䬋䮚倰冷堎塄崚愣棱楞睖碐稜薐踜輘𥈮𦼊𧼔𨈓𩩡"},
    {U"li", U"㑦㒧㒿㓯㔏㕸㗚㘑㛤㟳㠟㠣㡂㤡㤦㦒㧰㬏㮚㯤㰀㰚㱹㴝㷰㸚㹈㺡㻎㻺㼖㽁㽝㾐㾖㿛㿨䁻䃯䄜䅄䅻䇐䉫䊍䊪䋥䍠䍥䍦䍽䓞䔁䔆䔉䔣䔧䕻䖥䖽䖿䗍䘈䙰䚕䟏䟐䡃䣓䣫䤙䤚䥶䧉䬅䬆䮋䮥䰛䰜䱘䲞䴡䴻䵓䵩䶘丽仂位例俐俚俪傈儮儷兣凓刕列利剓剺劙力励勵历厉厘厤厯厲叓叕叻吏呖哩唎唳喱嚟嚦囄囇坜塛壢娌娳婯嫠孋孷屴岦峛峢峲巁廲悡悧悷慄戾扐扚捩搮擽攊攦攭斄暦曆曞朸李杝枥柂栃栎栗栛梨梩梸棃棙樆檪櫔櫟櫪欐欚歴歷氂沥沴泣浬涖淚溧漓澧濼濿瀝灑灕爄爏犁犂犛犡狸猁珕珞理琍瑮璃瓅瓈瓑瓥疠疬痢癘癧皪盠盭睝矖砅砬砺砾硌磿礪礫礰礼禮禲离秝穲立竰笠筣篥篱籬粒粝粴糎糲綟縭纅纚缡罹翮脷艃苈苙茘荔荲莅莉菞蒚蒞蓠蔾藜藶蘺蚸蛎蛠蜊蜧蝕蝷蟍蟸蠇蠡蠣蠫裏裡褵觻詈謧讈豊貍赲跞躒轢轣轹逦邌邐郦酈醨醴釃里釐鉝銐鋫鋰錅錑鎘鏫鑗鑠锂隶隷隸離雳霾靂靋颯驪骊鬁鬲鬴鯉鯏鯬鱧鱱鱳鱺鲡鲤鳢鳨鴗鵹鷅鸝鹂麗麜黎黐黧礼𠌯𠘞𠘟𠚄𠛘𠛦𠝄𠞉𠞙𠞤𠠏𠠝𠠵𠢠𠩵𠪄𠪺𠫌𠭰𠻗𠼝𠾆𡃷𡆯𡤌𡥽𡫯𡮰𡯄𡳸𡸉𡾒𡿋𡿎𢄡𢌈𢍼𢏃𢛮𢟢𢟤𢡑𢤂𢤆𢤩𢨨𢩑𢮃𢸀𢻠𣀂𣀥𣀷𣁟𣌅𣌜𣐬𣖊𣘬𣞴𣟌𣦯𣧿𣫥𣫧𣮉𣯤𣲒𣿞𤁼𤃀𤄽𤇃𤔨𤖢𤗫𤘃𤚓𤜜𤟑𤠫𤡿𤢴𤩮𤭜𤳓𤹇𤹈𤻤𤼚𥁟𥉆𥊈𥌛𥌤𥌮𥌿𥎓𥎔𥒢𥓃𥝢𥠲𥣥𥨻𥬭𥲧𥲪𥴡𥶗𥷅𥷗𥻑𥻿𥼅𥽈𥽗𦃇𦃊𦅺𦇔𦎐𦔓𦕸𦘊𦜏𦠓𦢱𦪶𦪾𦫈𦺙𧄚𧄻𧅮𧅯𧉲𧋎𧋠𧍀𧑇𧒈𧓽𧔝𧕮𧕯𧘫𧙉𧚩𧢝𧤜𧥖𧧋𧫬𧮛𧯏𧰡𧲡𧴠𧽲𨁸𨃙𨄛𨇎𨇗𨊛𨍫𨍮𨏬𨓦𨘸𨛋𨛫𨜼𨝏𨝖𨝟𨞺𨟑𨢌𨤫𨪹𨬑𨯽𨴻𨷦𨽹𨽻𩁟𩄞𩅩𩆝𩆲𩗅𩗭𩘟𩘡𩙖𩞨𩣫𩥬𩥴𩧃𩧋𩪸𩭇𩯺𩰲𩱇𩳓𩴣𩶘𩷋𩻌𩽏𩽵𪁐𪅆𪅼𪌱𪏵𪏼𪐅𪑚𪒔𪓀𪕴𪖂𪖍𪗁𪙺𪙽𫁡𫟷𫵷𬍛𭀖"},
    {U"lia", U"俩倆"},
    {U"lian", U"㜃㜕㜻㝺㟀㡘㢘㥕㦁㪘㪝㯬㰈㰸㱨㶌㶑㺦㼑㼓㾾䁠䃛䆂䌞䏈䙺䥥䨬䭑亷令僆劆匲匳嗹噒堜奁奩媡嫾嬚孌帘廉怜恋慩憐戀搛摙撿攣敛斂梿楝槏槤櫣欄歛殓殮浰涟湅溓漣潋澰濂濓瀲炼煉熑燫琏瑓璉瞵磏稴簾籢籨練縺纞练羷羸翴联聨聫聮聯脸膦臁臉苓莲萰蓮蔹薕薟蘝蘞螊蠊裢裣褳襝覝謰譧蹥輦连連鄻醶錬鍊鎌鏈鐮链镰零鬑鰊鰱鱄鲢𠋖𠒵𠔨𠗳𡆕𡟤𢅏𢅖𢉧𣀃𣝈𣞰𣟺𣿊𤑿𤒦𤗛𤣆𤬓𤹨𤼏𤾲𥖝𥲥𥽸𦆆𦈐𦔖𦖾𦣸𧍴𧐖𧡙𧡴𧸖𧸘𧽫𨎷𨏩𨏶𨬁𨽷𩄡𩞙𩟅𪍦𪍴𪐋𪐍𪖳𪚁𪛒𬶠"},
    {U"liang", U"㒳㔝㹁㾗䀶䁁䓣䝶䠃䣼䩫䭪両两亮俍倆倞兩凉哴唡啢喨墚悢惊掚晾梁椋樑涼湸煷簗粮粱糧綡緉脼良莨蜋蜽裲諒谅踉蹣輌輛輬辆辌量鍄閬靚駺魉魎𠓜𠯱𡑆𡮎𣄴𣓈𤙝𥈘𥛫𨄈𨎍𨎛𨱉𨵶𩗾𩘁𩞯𫟅𬜯"},
    {U"liao", U"㙩㝋㡻㵳㶫䄦䉼䎆䑠䒿䜍䜮䝀䝤䢧䨅䩍了佬僇僚勞叾嘹嫽寥寮尞尥尦屪嵺嶚嶛廖廫憀憭摎撂撩敹料暸曢樂樛漻潦炓燎爎爒獠璙疗療瞭窌窷竂簝繆繚缭聊膋膫蓼藔蟉蟟蟧豂賿蹘蹽轑辽遼鄝釕鏐鐐钌镣镽飂飉髎鷯鹩𠐋𠖂𠨥𠺕𡻪𢄷𢊻𢨺𢸘𢻢𢼙𢿞𣁰𣎸𣟆𣩢𤊽𤑗𤵠𥗀𥛰𥲊𦌒𦕵𦗖𦪕𦺹𧂏𧘈𧝜𧡜𧽽𨖚𨣀𩕐𩖝𩯊𩴤𪌀𪌵𪖷𪤗"},
    {U"lie", U"㤠㤡㧜㬯㭞㭩㯿㲱㸹㼲㽟䁽䃳䅀䉭䋑䓟䜲䝓䟩䟹䪉䬅䴕例倈儠冽列劣劦劽咧哷埒埓奊姴巁巤忚挒挘捩擖擸栗栵棙毟洌浖烈烮煭燤爄爉犣猎猟獦獵睙累綟聗脟膊臘茢蛚裂趔躐迾邋颲鬛鬣鮤鱲鴷𠛱𠠗𠷈𡁓𡂏𡂩𡊻𡏵𡒏𡓍𡭣𡿩𢣓𣁷𣁻𣋲𣖊𣝚𣰌𤁯𤐱𤓿𤖺𤜓𤞊𤡕𤢪𤱃𤱛𥪂𥬭𥲁𥶢𥷨𥸸𦅶𦓓𦓤𦖩𦾳𧀨𧓐𧞕𧞪𧭌𧭞𧰠𨆍𨕜𨤤𩆣𩙑𩢾𩧆𩧮𩨐𩭌𩼭𫚭"},
    {U"lin", U"㐭㔂㖁㝝㡘㨆㷠䉮䕲䗲䚏䚬䢯䫐䫰䮼临亃任伈僯冧凛凜厸吝啉壣崊嶙廩廪恡悋惏懍懔拎撛斴晽暽林橉檁檩淋溓滲潾澟瀶焛燐獜玪琳璘甐疄痳癛癝瞵碄磷稟箖粦粼繗翷膦臨菻蔺藺賃赁蹸躏躙躪轔轥辚遴邻鄰鏻閵隣霖顲驎魿鱗鳞麐麟𠐼𠓮𡃦𡬜𡰚𡳞𡶱𡹇𡻫𡿠𣇰𤂶𤌎𤎭𤗷𥓆𥳞𥶒𥷖𥻋𥼭𦺸𧃮𧖔𧲂𧵧𧶆𧹩𨏨𨸻𩞻𩣖𩱬𩴠𩻜𬭸𬴊"},
    {U"ling", U"㖫㡵㥄㦭㪮㬡㯪㱥㲆㸳㻏㾉䄥䈊䉁䉖䉹䌢䍅䔖䕘䖅䙥䚖䠲䡼䡿䧙䨩䮚䯍䰱䴇䴒䴫令伶倰冷凌刢另呤囹坽夌姈婈孁岭岺崚嶺彾怜拎掕昤朎柃棂棱櫺欞泠淩澪瀮灵炩燯爧狑玲琌瓴皊砱磷祾秢竛笭紷綾绫羚翎聆舲苓菱蓤蔆蕶蘦蛉衑袊裬詅跉軨輘酃醽釘鈴錂铃閝阾陵零霊霗霛霝靇靈領领駖魿鯪鲮鴒鸰鹷麢齡齢龄龗𠄖𠏡𠟨𠠢𠡭𠱠𠻠𠻱𠾥𡈍𡕮𡿡𢌔𢔁𢩗𢹝𢺰𣌟𣣋𣬹𤃩𤖦𤜙𤣘𤧘𤨻𤫩𤫲𤷖𤿅𥌼𥤜𥤞𥥋𥩔𥵝𥺙𥾂𦉢𦊓𦐽𦫃𦫊𧆺𧕅𧖜𧟙𧨈𧰻𧱢𧲙𧾇𧾮𨞎𨠎𨱋𨽲𩂙𩃞𩄊𩆒𩆚𩆮𩆻𩆼𩇄𩇎𩊂𩑊𩖊𩖵𩚹𩜁𩟃𩪥𩬔𩲩𩵀𪅋𪋳𪋾𪌏𪕌𪛈𫐉"},
    {U"liu", U"㐬㙀㚹㧕㨨㶯㽌㽞䄂䉧䗜䚧䝀䬟䰘䱖䱞䶉僂六刘劉嚠塯媹嬼嵧廇懰摎斿旈旒柳栁桞桺榴橊橮沠泖泵流浏游溜漻澑瀏熘熮珋琉瑠瑬璢畂畄留畱疁瘤癅硫碌磂磟窌綹绺罶羀翏聊蒥蓅蓼蔞藰蟉裗蹓遛鉚鋶鎏鎦鏐鐂锍镏镠陸雡霤飀飂飅飗餾馏駠駵騮驑骝鬸鰡鶹鷚鹠鹨麍𠗽𠛓𠪐𠺕𡻪𢏭𢔲𢞭𢣠𢤐𢷶𣞗𣟑𣠚𣱳𣲖𤥗𤮷𥀓𥆦𥌐𥛅𥠷𥥹𥧕𥨌𥰣𥶅𥹷𦀠𦃓𦉉𦊑𦊗𦊿𦌁𦑾𧏓𧮗𨋖𨍸𨢇𨦰𨪕𨪿𨶪𨻧𩆎𩖴𩗩𩙄𩢞𪃂𪆱𪇯𪎣"},
    {U"lo", U"咯囖"},
    {U"long", U"㑝㙙㚅㛞㝫㟖㡣㢅㦕㰍㳥㴳䃧䆍䏊䙪䡁䥢䪊䮾䰱儱咙哢嚨垄垅壟壠寵屸嶐巃巄弄徿拢攏昽曨朧栊梇槞櫳泷湰滝漋瀧爖珑瓏癃眬矓砻硦礱礲窿竉竜笼篢篭籠聋聾胧茏蕯蘢蝕蠪蠬衖襱谾豅贚躘鏧鑨陇隆隴霳靇驡鸗龍龐龒龓龙𠮽𠱚𠾐𡃡𡬍𡬕𡱯𢙱𢤱𢤲𢸭𣫣𤵸𤼃𤾭𥦌𥪢𥪻𥬆𥳌𥸉𦨩𦪽𧍰𧙥𧚂𨇘𨏠𨐇𨛓𨺚𩂽𩄺𩙘𩙠𩟭𩧪𪐖𪔕𪔳𪔷𪚑𪚓𪚘𪚝𪚠𬕂"},
    {U"lou", U"㔷㟺㡞㥪㪹㲎㺏䁖䄛䅹䝏䣚䫫䮫䱾偻僂剅喽嘍塿娄婁寠屚嵝嶁廔慺搂摟楼樓溇漊漏熡牢甊瘘瘺瘻瞜窶篓簍耧耬艛蒌蔞蝼螻謱軁遱鏤镂陋露鞻髅髏𠖛𠗩𠞭𠳴𡀔𡇭𡗆𡡎𡪅𡰌𢈢𣤋𣫻𤋏𤠋𤬏𥕍𦌁𦎹𦭯𦸢𧁾𧢃𧫞𧯨𧰃𧷡𨄋𨝢𨦖𨫒𨱐𨻻𩏝𩨇𪍣𪣻𫠥"},
    {U"lu", U"㓐㔪㖨㛬㜙㟤㠠㢚㢳㦇㪐㪖㪭㫽㭔㯝㯟㯭㱺㼾㿖䃙䌒䍡䎑䎼䐂䘵䚄䟿䡎䡜䥨䩮䮉䰕䱚䲐䴪侓僇六剹勎勠卢卤噜嚕嚧圥坴垆塶塷壚娽峍庐廘廬彔录戮掳摝撸擄擼攄攎曥枦栌椂樐樚橹櫓櫨氇氌泸淕淥渌滷漉潞澛瀂瀘炉熝爐獹玈琭璐璷瓐甪瘳盝盧睩矑硉硵碌磟磠祿禄稑穋箓簏簬簵簶籙籚粶緑繆纑罏翏胪膔膚臚舮舻艣艪艫芦菉蓼蓾蔍蕗蘆虂虏虜螰蠦角觮觻謢谷賁賂赂趢路踛蹗輅轆轤轳辂辘逯鄜酪醁鈩錄録錴鏀鏕鏴鐪鑥鑪镥陆陸露顱颅騄騼髗魯魲鯥鱳鱸鲁鲈鵦鵱鷺鸕鸬鹭鹵鹿麓黸𠀽𠿛𡀔𡉴𡳴𡴆𡷏𢂤𢊩𢋡𢟧𢫫𢯅𢲸𢻵𢾬𣆐𣞓𣥐𣩏𣱀𣼟𤝮𤟘𤢊𤣃𤨍𤬛𤮧𤺼𤻱𤽺𤿴𥀔𥀵𥈛𥉶𥒨𥚊𥛞𥛪𥣤𥫰𥲎𥶇𦋔𦌕𦌟𦗓𦪇𦸐𦼋𦽂𦽎𦾞𦾷𦿊𦿖𧀦𧆣𧇄𧌉𧌍𧐳𧨹𧫓𧽥𨁸𨇖𨌠𨏔𨽐𩄅𩅄𩌫𩍼𩓪𩛼𩣱𩯜𪉖𪉣𪍄𪑄𪒏𪖌𫘧𬬻"},
    {U"luan", U"㝈㡩㱍䏈䖂䜌乱乿亂卵圝圞奱娈孌孪孿峦巒挛攣曫栾欒滦灓灤癴癵羉脔脟臠臡薍虊覶釠銮鑾鵉鸞鸾𠦨𡄹𡡗𡭸𡰠𢌕𢺈𢿢𣨀𤔔𤲶𤼙𦣋𦣏𧖘𨄄𨇼𨈌𨈎𨊟𩪾𪢮"},
    {U"lue", U"㑼㔀㗉㨼䂮䌎䛚䤣剠圙寽掠擽率略畧稤藥詻鋝鋢锊𠢌𠼟𦊹𧎾𧐋𧐯𧑀𧕌𪅅"},
    {U"lun", U"㖮㷍䈁䑳仑伦侖倫囵圇埨婨崘崙惀抡掄棆沦淪溣睔碖磮稐綸纶耣腀菕蜦論论踚輪轮錀陯鯩𠔕𠼩𡃝𤲕𤷔𥿑𦓾𧣵𧱜𫭢𬬭"},
    {U"luo", U"㑩㒩㓢㔏㞅㦬㩡㪾㰁㱻㴖㸹㼈㽋㾧㿚䀩䃕䇔䈷䉓䊨䌱䌴䎊䗍䮑䯁䴹倮儸儽剆咯啰囉峈挼捋捰摞攎攞攭曪果格椤樂橐櫟欏欙泺洛洜漯濼烙爍犖猓猡玀珞瘰癳皪砢硌硦碌礫笿箩籮絡纙络罖罗羅脶腡臝茖荦萝落蓏蘿蛒蜾蝸螺蠃蠡袼裸覙覶覼詻路躒躶逻邏酪鉻鎯鏍鑼锣镙雒頱饠駱騾驘骆骡鮥鱳鴼鵅鸁𠉗𠏢𠜖𠶱𠻐𠻡𡁆𡆆𡤢𡿏𢅾𢺆𢺑𣁻𣂞𣎆𣛗𣜄𣜢𣧳𣨪𣵟𤄷𤔖𤔝𤗀𤨗𤬁𤽥𤽼𥡜𥯛𦆁𦣇𦣖𦣛𦿌𧄿𧈦𧟌𧭥𧷳𧹐𨇽𨏒𨟥𨬅𨰠𩂣𩉙𩊚𩌭𩍪𩎊𩮹𩵇𩼊𩽰𪇱𪈰𪌳𪎆𪑋𫌨"},
    {U"lv", U"㔧㛎㠥㡞㭚㲶㻲㾔䔞䕡䢖䥨䮫侣侶儢勴吕呂哷垏壘婁寠寽屡屢履嵂廬律慮慺挔捋捛旅梠榈樓櫖櫚櫨氀氯滤漊濾焒爈率瘻盧瞜祣稆穞穭箻簍累絽綠緑縷繂绿缕膂膐膟膢臚葎蔞藘虑褛褸謱軁郘鋁録鏤鑢铝閭闾馿驢驴魯鷜鹿𠜈𠣊𠷈𠻜𡀿𡡎𡾅𢅞𢈚𢙲𢟳𢣻𢯰𣀞𣥐𣭇𤁵𤗬𤝽𤾺𥖼𥡢𥭐𥰠𥶆𥶌𦆾𦊼𦛗𦝼𦭯𦳭𧃒𧈔𧍶𧓻𧜊𧭜𨂠𩄽𩥆𩲦𩳡𩴐𪈜"},
    {U"lve", U"㑼㔀㗉㨼䂮䌎䛚䤣剠圙寽掠擽率略畧稤藥詻鋝鋢锊𠢌𠼟𦊹𧎾𧐋𧐯𧑀𧕌𪅅"},
    {U"m", U"呣唔"},
    {U"ma", U"㐷㑻㕰㜫㦄㨸㾺䀛䀣䗫䣕䣖䧞䯦䳸么亇傌吗唛嗎嘛嘜嚜妈媽嫲嬤嬷孖抹摩杩榪溤犘犸獁玛瑪痲睰码碼礣祃禡罵蔴蚂螞蟆蟇貉貊遤鎷閁靡馬駡驀马骂鬕鰢鷌麻麽𡻤𢉿𢊀𢋚𢳀𣨜𤳂𥀏𥉊𥉵𥧓𧪨𨰾𨳿𩀪𩊃𩔶𩔷𩨲𩶞𪐎𪒜𪓹"},
    {U"mai", U"㜥㦟䁲䘑䚑䜕䨪䨫䮮买佅劢勱卖咪哩嘪埋売派脈脉荬蕒薶衇貍買賣迈邁霡霢霾鷶麥麦鿏𠿆𢠼𥇯𥌚𦏢𦙻𦚡𧱘𨤢𩈗𩊍𩍃𪄳𪍇𪒪"},
    {U"man", U"㒼㗈㙢㛧㡢㬅㵘㿸䅼䊡䐽䒥䕕䛲䜱䝡䝢䟂䡬䯶䰋僈埋墁姏嫚屘幔幕悗慢慲摱曼槾樠満满滿漫澫澷熳獌睌瞒瞞矕絻縵缦蔄蔓蘰蛮螨蟎蠻襔謾谩蹒蹣鄤鏋鏝镘鞔顢颟饅馒鬗鬘鰻鳗𡕢𡢚𡻩𢦈𢿜𣁜𤅎𤜘𥊑𥧭𥬈𥲈𥲑𦎌𦔔𧆏𧖵𧜞𧱼𨲛𨲾𩅍𩆓𩈦𩛎𩛟𩬞𩮉𪈿𪍩𪑪𬜬"},
    {U"mang", U"㝑㟌㟐㟿㡛㤶㬒㴇㻊䁳䅒䈍䏵䒎䓼䖟䵨厖吂哤壾娏尨庬忙恾朚朦杗杧氓汒浝漭牤牻狵甿痝盲盳瞢硥硭笀芒茫茻莽莾蘉蛖蟒蠎邙釯鋩铓駹鸏龍𠈵𠮵𡅖𡘪𡩩𡩽𡵀𣙷𣯬𣶈𤛘𤰡𥁃𥆙𥐞𥝕𥤩𥭚𥮎𦎨𦜭𨛌𩅁𩒿𩙸𩛲𩪎𩭒𩷶𪁪𪚢"},
    {U"mao", U"㒵㒻㚹㝟㡌㧇㧌㪞㫯㮘㲠㴘㺺㿞䀤䅦䋃䓮䡚䫉䭷乮侔兞冃冇冐冒勖務卯堥夘媢峁嵍帽愗懋戼描旄昴暓枆柕楙毛毣毷氂泖渵牟牦犛猫瑁皃眊瞀矛秏笷緢罞耄耗芼茂茅茆萺蓩蛑蝐蝥蟊袤覒貇貌貓貿贸軞鄚鄮酕鉚鉾錨铆锚霿髦髳鶜𠔼𠤝𡜢𡹰𢂹𢅉𢘅𢝌𢨯𢯾𢽢𣊃𣔺𣨇𣬵𣭮𣯀𣴟𣴼𣹪𤚜𤛖𤝄𤥰𤲰𥄸𥈆𥎟𥟪𦀸𦳁𦼪𧍟𧐟𧒚𧓿𧔨𧠊𨈥𨥨𨦜𨩩𨺸𩛨𩫁𩬞𩭾𩿂𪃑"},
    {U"me", U"么嘸嚒嚜没濹癦麼麽"},
    {U"mei", U"㙁㭑㺳䀛䆀䆊䉋䊈䍙䓺䜸䤂䰨䰪䱕䵢凂呅味嚜坆坶堳塺墨妹娒媄媒媚媺嬍寐嵄嵋徾抺挴攗旀昧枚某栂梅楣楳槑櫗毎每氼沒没沬浼渼湄湈溦煝煤燘猸玫珻瑂痗眉眊眛睂睸矀祙禖穈篃糜美羙脄脢腜膴苺莓葿蘪蝞袂跊躾郿酶鋂鎂鎇镁镅霉韎鬽魅鶥鹛黣黴𠊉𠍨𠪃𡲭𢮇𣟸𤈦𤚤𤽃𥞊𥧴𦼻𧭵𧳬𨉭𨜘𩈐𩋿𩎟𩫍𩲈𩲢𩴈𪂜𪃏𪉏𪎦𪎭"},
    {U"men", U"㡈㥃㦖㨺㱪㵍䊟䪸䫒亹们們怋悗悶惛懑懣扪捫暪椚殙汶滿焖燜玧璊瞞穈菛虋鍆钔門閅门闷鞔𣯩𤅣𧄸𧴺𨳔𨴺𩑥𩔉𫞩"},
    {U"meng", U"㙹㜴㝱㠓㩚䀄䁅䁫䇇䈍䉚䏵䑃䑅䒐䓝䗈䙦䙩䟥䠢䤓䥂䥰䰒䲛䴌䴿䵆儚冡勐夢夣嫇孟尨幪庬懜懞懵掹擝明曚朚朦梦橗檬氋氓溕濛猛獴瓾甍甿盟瞑瞢矇矒礞艋艨莔萌蒙蕄蘉虻蜢蝱蟊蟒蠓鄳鄸鋂錳锰雺霥霧霿靀顭饛髳鯍鯭鱦鸏鹲黽鼆𠐁𠐧𠖆𠵼𡒯𡚔𡬆𡬌𢄐𢕙𢤘𢿂𣊔𣓝𣞑𣰥𣽭𤯻𤱴𤼁𤾬𥂂𥄁𥉕𥋝𥌯𥌱𥣛𥭮𦆟𦊽𦢧𦫰𦱋𦳶𦴔𦷹𦿏𧀆𧀧𧁊𧂛𧂡𧈧𧓨𧞑𧭊𧲍𨞫𨢊𨢠𨣘𨨸𨮒𨼿𩄖𩆽𩕱𩟞𩦺𩴲𩶡𪅇𪇓𪈆𫑡"},
    {U"mi", U"㜆㜷㝥㟜㠧㣆㥝㨠㩢㫘㳴㳽㴵㵋㸏㸓䁇䈼䈿䉲䊳䋛䌏䌐䌕䍘䕳䕷䖑䖹䛑䛧䣾䤉䤍䥸䭧䮭䱊䴢侎冖冞冪劘咪嘧塓孊宓宻密峚幂幎幦幺弥弭彌戂摩摵擟攠敉榓樒檷櫁汨沕沵泌洣淧渳溟滵漞濔濗瀰灖熐爢爾猕獮獼瓕眫眯眽瞇瞴祕祢禰秘穈簚籋米粎糜糸縻羃羋脒芈苾葞蒾蓂蔝蔤藌蘼蜜袮覓覔覛觅詸謎謐谜谧辟迷醚醾醿釄銤鑖镾靡鸍麊麋麛鼏𠞧𠵸𡄣𡇒𡊭𡓭𡝠𡬍𡬐𡲼𡾱𢆯𢇲𢘺𢞞𢱮𣓔𣧲𤛬𤦀𥁑𥅘𥇆𥇎𥈕𥉴𥉿𥎖𥞪𥦀𥧧𥭫𥮜𥵨𥹄𥹫𥻒𥽰𥿫𦄐𦖬𦗕𦞟𦟂𦣥𦰴𦸡𧐎𧕵𧠟𧱻𧵬𧶡𧷦𧼊𧽨𨇻𨒲𨢎𨢥𨣯𨣾𨷬𩀶𩔢𩞁𩞇𩸹𪀿𪅮𪋗𪋢𪎔𪎗𪑸𪒄𪓬𪕈"},
    {U"mian", U"㒙㛯㝃㝰㡈㤁㨺㮌㰃㴐㻰䀎䃇䏃䛉䤄䩄䫵䯶䰓丏俛偭免冕冥勉勔喕娩婂媔嬵宀愐杣棉檰櫋汅沔泯渑湎湣澠牑眄眠瞑矈矊矏糆絻綿緜緡緬绵缅腼臱芇莬葂蝒蠠面靣靦鮸麪麫麵麺黽黾𡒳𡕢𡧍𡧒𡯫𢃮𢣔𣅍𣡠𣧾𥄝𥊿𥌂𥤵𥻩𥾝𦬛𦽃𧭇𧸨𨉥𨟺𨡞𨲛𨲾𩈹𩋠𩬞𩮉𩾃𪁼𪏊"},
    {U"miao", U"㑤㠺㦝䁧䅺䏚䖢仯吵喵妙媌嫹庙庿廟彯描杪淼渺猫玅眇瞄秒竗篎紗緢緲繆缈苗藐蜱訬邈鱙鶓鹋𠋝𡙎𡡺𢆷𢚋𢤧𢷕𤾛𥭝𦳥𩳸𪃐𪃦"},
    {U"mie", U"㒝㩢䁾䈼䌩䘊䩏乜吀咩咪哶孭幭懱搣櫗滅瀎灭烕眜篾羋蔑薎蠛衊覕鑖鱴鴓𠺗𡖺𡞙𡟬𢦼𢧞𢨖𤊾𤏿𥄲𥉓𥋚𥌨𥣫𥵒𥸴𥽘𥾝𦇪𧀅𧂝𨣱𩔠𩱷𪇴𪌺𪒍"},
    {U"min", U"㞴㞶㟩㟭㢯㥸㨉㬆㳷䁕䂥䃉䋋䝧䟨䡑䡻䪸䲄僶冺刡勄厸呡垊姄岷崏忞怋悯惽愍慜憫抿捪敃敏敯旻旼暋民汶泯渂湏湣潣玟珉琘琝瑉痻皿盷盿眠砇碈笢笽簢緍緡繩缗罠苠蠠鈱錉鍲閔閩闵闽鰵鳘鴖黽𠊟𢼖𢽹𣱈𣱉𣷠𣹒𤇜𤛎𤸅𤺖𤿕𥜐𦈏𦌡𦫮𦳜𧁋𧌙𧲃𧵣𨏵𩔉𩭷𪂆𪄴𪉎"},
    {U"ming", U"㝠㟰㫥䄙䆩䊅䏃䒌䫤䳟佲冥凕名命姳嫇慏掵明暝朙椧榠洺溟猽皿眀眳瞑茗萌蓂螟覭詺鄍酩銘铭鳴鸣𠋶𡥸𥌏𥥊𥹆𥿨𦡉𦫭𧟠𧱴𩣶𪂤𪗸"},
    {U"miu", U"嘐繆謬谬𨱯"},
    {U"mo", U"㱄㱳㵹㶬㷬㷵㹮䁼䁿䃺䊳䏞䒬䘃䜆䩋䬴䭩䮬䯢䱅䳮䴲万么伯佰冒劘劰勿唜嗼嘿嚤嚩嚰圽塻墨妺嫫嫼寞尛帓帕帞庅怽懡戂抹摩摸摹撫擵攠昧昩暯末枺模橅歾歿殁没沫湐漠瀎無爅狢獏瘼百皌眜眽眿瞐瞙砞磨礳秣粖糢絈絔縸纆耱膜艒茉莈莫蓦藐藦蘑蛨蟆蟔袜袹謨謩譕谟貃貈貉貊貌貘鄚銆鏌镆陌靺鞨饃饝馍驀髍魔魩魹麽麿默黙𠄮𠆮𠇱𠡞𠢓𠬛𠻚𡈗𡊉𡠜𡡉𡢜𡻟𡾉𢄏𢊗𢐖𢗿𢣗𢳀𣋟𣧒𣧣𣶊𣻕𤋂𤣻𤹴𤿖𥂓𥄕𥕓𥙎𥞪𥬎𥱹𥽘𥿉𦅔𦔭𦟟𦥦𦫕𦮅𧀅𧕤𧕥𧠓𧥟𧰱𧻙𧼟𧿴𨆽𨟖𨰞𨱱𩃁𩄻𩌧𩎟𩐻𩑦𩑷𩞁𩟠𩢖𩢷𩥔𩪮𩿣𪍇𪍤𪎠𪏟𪒂𪒇𬙊"},
    {U"mou", U"㭌䋷䍒䍙䏬䗋䜼䥐䱕件侔劺厶呣哞堥婺恈敄某桙毋洠牟畝眸瞴繆缪蛑蟱袤謀谋踎鉾鍪鞪鴾麰𠀱𠥨𢃱𢜯𣫬𥆆𥿵𦊋𦊎𦋡𦭷𦳑𦺒𧎄𧎻𨴍𩢫𩶢"},
    {U"mu", U"㙁㜈㟂㣎㧅㾇䀲䊾䑵䥈䱯亩仫凩募嘿坶墓墲姆姥娒婺峔幕幙慔慕拇暮木朰朷楘模樢母毣毪氁沐炑牟牡牧牳狇獏畆畒畝畞畮目睦砪穆縸繆胟艒苜茻莫莯萺蚞踇鉧鉬钼雮霂鞪鶩𠺖𡵬𢘃𢜯𢟨𣈊𣳠𤚅𤝂𤝕𤵝𥄈𥣸𥰻𦃤𦱒𧚀𧩒𧬏𧰷𧿹𨈶𨍎𨎸𨡭𨢢𩡨𩬍𩵦𩶖𩶩𪎫𬭁𬰃"},
    {U"n", U"㕶咹哏哽唵嗯𠮾𧗈"},
    {U"na", U"㗙㨥㪎㮏㵊䇱䈫䋈䋾䎎䏧䖓䖧䛔䟜䪏䫱乸内南吶呐呶哪嗱妠娜抐拏拿挐捺淰秅笚笝箬納絮纳肭蒘蒳衲袦訤詉誽豽貀蹃軜那郍鈉鎿钠镎雫靹魶𠕄𠱲𠴾𡤙𡰀𡷝𢇵𢓇𢜲𢡏𣅚𣡰𣸏𣹵𤓷𤔀𤝒𤡤𤬷𤭠𤱅𤱆𤷈𤸏𤸻𥍲𥑒𥹉𥿃𦙜𦛐𦣀𦬻𦰡𦳐𧋡𧘽𧤣𧦮𧰹𨙻𨚗𩏼𩚛𩟿𩮅𩮸𩹾𪌅𪐀𪗝"},
    {U"nai", U"㜨㮈㮏㲡㴎㾍䍲䘅䯮乃佴倷哪奈奶妳嬭孻廼掜搱摨柰氖渿熋疓耏耐能腉艿萘螚褦迺釢錼鼐𠧟𠧤𡞫𡨵𡮙𢉓𣉘𣮦𥄗𥉃𦓎𦔹𦠸𦳐𦶅𨎡𩹟𪌞"},
    {U"nan", U"㓓㫱㬮㽖䈒䊖䔜䛁䣸䩅䶲侽冉南喃囝囡妠娚婻嫨弇戁抩揇攤暔枏柟楠湳灘煵男畘罱腩莮萳蝻諵赧遖难難𡆤𡆱𡆲𢪈𢬷𤌔𤱣𤽲𤿏𦍀𦛚𦝧𦶈𧇙𧕴𧹞𨠹𨦳𨴌𨴘𨵴𩅠𩈑𩈶𩹞"},
    {U"nang", U"㚂㶞䁸乪儾噥嚢囊囔憹搑擃攮曩欜涳瀼灢蘘蠰譨饢馕鬞齉𠶬𡿝𢖧𣰶𦈃𦗳𦣘𧅺𧖒𧟘𨳆𩛡𩜒"},
    {U"nao", U"㑎㛴㞪㺀㺁㺒䃩䐉䛝䜀䜧䴃匘呶垴堖夒婥嫐孬峱嶩巎巙怓恼悩惱憹挠摎撓橈淖澆猱獶獿瑙硇碙碯脑脳腝腦膠臑蛲蝚蟯詉譊鐃铙閙闹鬧𠊦𠡷𡍍𡽧𡾂𡿺𢅈𢉵𢙐𢜸𢪼𣧽𣭺𣳦𤊲𤋫𤞍𤡤𤫕𤷻𥀮𥆲𥐻𥑪𥒢𦗮𧩣𧳀𧳦𧴓𧴙𨥸𨱵𩋈𩖯𩛋𩤘𩩀𩫔𩫺𩬷𩯆𫍢𬆛"},
    {U"ne", U"㕯䅞䎪䭆呐呢哪抐疒疔眲訥讷那𢗉𣧍𧤜"},
    {U"nei", U"㐻㨅㼏䲎內内娞婑氝浽脮腇那錗餒餧馁鮾鯘𠑚𠑛𡣢𢁩𢅼𢛉𣓃𥡭𨡌𩗔𩛞𩬀"},
    {U"nen", U"㜛㯎㶧媆嫩嫰恁枘腝臑𡞾𧮠𨈗"},
    {U"neng", U"㲌㴰䏻竜而耐能螚𠹌𢆂𨃳𨶙"},
    {U"ng", U"㕶哽唔唵嗯"},
    {U"ni", U"㘈㞾㠜㥾㦐㩘㪒㮏㲻㵫㹸䁥䕥䘌䘦䘽䛏䝚䦵䵑䵒伱伲你倪儗儞兒匿呢坭埿堄妮妳婗嫟嬭嬺孨孴尼屔屰嶷彌怩惄愵慝懝抐抳拟掜擬旎昵晲暱柅棿檷氼泥淣溺濔濘瀰灄爾狔猊痆眤睨禰秜籾縌聣聻胒腝腻膩臡苨薿蚭蛪蜺觬誽譺貎跜輗迡逆郳鈮鉨鑈铌隬霓馜鯢鲵麑齯鿭𠆵𠒰𠱘𠸺𠽬𡎳𡎿𡞭𡣁𡥦𡥨𡫸𡬗𢅟𢘝𢚮𢛜𢣚𢦱𣘗𣡋𣢞𣭙𣲷𤙌𤦤𥄽𥇄𥜦𥜬𥷄𥺜𦆦𦤽𦦃𦮾𦰫𧃩𧈞𧏾𧖷𧵼𧺰𨀀𨋗𨒫𨺙𨽦𩈢𩉹𩋪𩍦𩚮𩚯𩩢𩯨𩰞𩱄𩸦𩸧𩺝𩺱𩾆𪏵𪏸𪐌𪙛𫐐𫠜"},
    {U"nia", U"㖸"},
    {U"nian", U"㜤㞋㮟㲽䄭䄹䚓䧔䬯䴴卄哖唸埝姩年廿念拈捵捻撚撵攆榐涊淰溓焾痆碾秊秥簐粘艌蔫趁跈蹍蹨躎輦輾辇辗鮎鯰鲇鲶鵇黏𠕟𠗋𠣇𠫺𠬋𡝟𡰫𣎔𣐏𤁥𤽿𥮘𥺴𦁇𦭁𦷙𧹞𨀀𨇍𨋚𨏛𨚶𨢯𨴞𨸱𩉄𩊫𩽴𪐇𪑮"},
    {U"niang", U"䖆娘嬢孃酿醸釀𥽬𪓃"},
    {U"niao", U"㒟㜵㞙㠡㭤㳮䃵䙚䦊䮍嫋嬝嬲尥尿樢溺脲茑茮蔦袅裊褭鳥鸟𠒰𡘏𡝋𡝒𡠿𢶑𢸣𣟊𤲶𥤂𥾇𨳀𨽖𩖔𩭑𪅝𪈼"},
    {U"nie", U"㖏㖕㖖㘝㘨㘿㙞㚔㜸㡪㩶㮆㴪㸎䂼䄒䇣䌜䌰䜆䡾䭃䯀䯅䯵䳖乜倪哪啮喦嗫噛嚙囁囐囓囡圼埝孼孽峊嵒嵲嶭巕帇幸惗捏捻掜揑摰攝敜枿棿槷櫱泥涅湼痆篞籋糱糵聂聶臬臲苶菍蘖蠥褹諗讘踂踗踙蹑躡鉨鉩銸鋷錜鎳鑈鑷钀镊镍闑陧隉顳颞齧𠈊𠶿𡆣𡇄𡍤𡰆𡴎𡶫𡸣𡾦𡾲𡿖𡿗𢈸𢫻𢬴𣀳𣌍𣙗𣯭𣰼𤭂𤴘𤶚𤷈𤸏𤺐𥔄𥬞𥬬𥭙𥮤𦄌𦈙𦘒𦛠𦞆𦡃𦣀𦯖𦵐𧁈𧋖𧞍𧻼𨊞𨙓𨱺𨲀𨶠𨻄𩋏𩐭𩒕𩖁𩣘𪌊𪌿𪎃𪎅𪘎𫔶𬛸"},
    {U"nin", U"㤛䋻䚾䛘囜恁您拰脌𠽝"},
    {U"ning", U"㝕㣷㲰㿦䆨䔭䗿䭢佞侫倿儜冰凝咛嚀嬣宁寍寕寗寜寧年拧擰攘柠橣檸泞泥澝濘狞獰甯疑矃聍聹苧薴鑏鬡鬤鸋𡫃𡷪𣍆𤕦𤹧𤻝𥣗𥧤𥳥𦡲𦡼𧃱𧍖𧑗𧕝𧭈𩕳𫛢"},
    {U"niu", U"㖻㺲㽱䂇䋴䏔䒜妞忸怓扭拗杻汼沑炄牛牜狃紐纽莥蚴鈕钮靵𢔟𣅚𣧊𣲶𥀝𥍳𥝦𧘥𨋀𨙺𨳞𨷁𩈇𩙷𩚖𩜷𩲍𩵠𪏲"},
    {U"nong", U"㶶㺜䁸䢉䵜侬儂农咔哝噥弄憹挊挵檂欁浓濃燶癑禯秾穠繷脓膿莀蕽襛農辳醲齈𠌚𠘊𥂒𨑊𨠤𨲳𩅽𩇔𩟊𪆯𪒬𬪩"},
    {U"nou", U"㜌㝹㳶䅶䘫䨲䰭啂嬬搙擩槈檽獳羺耨譨譳鎒鐞𠲴𡝦𡨻𡭾𢉕𢉚𣻖𤟦𥀫𧂦𧃨𧅘𩆟𩒔𪋺"},
    {U"nu", U"㚢㣽仅伮傉努呶奴孥帑弩怒挐搙擩砮笯肭胬褥詉駑驽𠴂𢪦𢫓𢫭𥅄𥛑𥤨𥱂𧉭𧗈𧪅𧿔𪺹"},
    {U"nuan", U"㬉䙇奻暖渜湪濡煖煗臑餪𪋐"},
    {U"nue", U"䖈䖋䨋婩疟瘧硸虐𤷡𨵫"},
    {U"nun", U"媆黁"},
    {U"nuo", U"㐡㑚㔮㖠㛂㡅㰙䇔䎠䚥傩儺呐哪喏堧娜媠愞懦懧挪掉掿搙搦搻梛榒橠毭濡稬穤糑糥糯耎袲袳諾诺蹃逽那郍鍩锘難需𠸱𠹈𡖔𡖫𡬥𡿊𢜪𢰜𢾲𣃽𣆚𤘟𥑽𥻾𦀨𦂍𦓢𦡃𦩜𧣚𧣺𨁌𨎭𩈺𩴓𩷁"},
    {U"nv", U"㵖䏔䖡䘐䚼䶊女恧朒沑狃籹絮聏胬衂衄釹钕𥄋𥍞𦓕𦓖𧖷𩈢"},
    {U"nve", U"䖈䖋䨋婩疟瘧硸虐𤷡𨵫"},
    {U"o", U"哦喔噢"},
    {U"ou", U"㒖㭝㰶㸸㼴䉱䌂䌔䙔䥲䧢偶區吘吽呕嘔塸怄慪握摳敺樞櫙欧歐殴毆沤渥漚澫熰瓯甌筽紆耦腢膒蓲蕅藕藲謳讴遇醧鏂鴎鷗鸥齵𠙶𠢔𠥝𠴰𡂿𡈆𡩾𣂻𣉾𣓕𣢨𣽕𤁮𤛐𤵎𥈬𥐂𥧆𥱸𥻑𦂕𧏺𧖼𧪓𩀫𩔸𩥋𪊪𪙃𫭟𬉼"},
    {U"pa", U"㕷㞎䔤䯲叭吧啪妑帊帕怕扒把掱杷汃派潖爬琶皅筢耙舥芭苩葩袙趴跁鈀𣚒𣧜𣱺𤆵𤽉𥐙𥩙𦐆𧑡𧣃𧣣𨋐𩈆𪗔"},
    {U"pai", U"㭛㵒㵺䖰䱝俳哌啡廹徘拍排棑椑派渒湃牌犤猅箄簰簲脾蒎輫迫鎃𠂢𠸁𣏟𣖐𣝁𣲖𣴪𥯟𥱼𥴖𥿯𦔠𦩯𦫖𧵠𩛇"},
    {U"pan", U"㐴㢖㽃䃑䃲䆺䈲䏒䩔䰉䰔乑伴冸判半卞叛坢姍媻審幋弁彦扳拌拚搫攀柈槃沜泮洀湴溿潘瀊瀋炍爿片牉畔畨番皤盘盤盻盼眅眫磐磻籓縏繁聁胖膰般萠蒰螌蟠袢褩襻詊賁跘踫蹣鄱鋬鎜鑻闆鞶頖鵥𠽲𡞟𢬵𢰿𢲔𣁦𣔚𤄜𤄧𤖭𤠍𤺏𤻷𥈼𥉟𥌊𥕿𦙀𦪹𧺾𨂝𨃞𨃟𨒃𪄀𪒀𪒋"},
    {U"pang", U"㕩㥬㫄䅭䏺䒍䠙䨦乓仿傍厐厖嗙夆嫎尨庞彭彷徬房方旁榜汸沗滂炐牓磅篣耪肨胖胮膀膖舽蒡螃蠭覫趽逄逢鎊雱霶髈鰟鳑龎龐𠗵𠦲𡅃𢐊𣂆𤧭𤵸𥪴𦜍𦣂𧔧𧿆𨜷𩃎𩅅𩈈𩐨𪐿𪔔"},
    {U"pao", U"㘐㚿㯡㯱㲏䛌䩝䫽䶌刨包匏咆嚗垉奅庖抛抱拋摽泡炮炰爮犥狍瓟疱皰砲礟礮穮窌胞脟脬苞萢藨蚫袌袍褜謈趵跑軳鉋鞄颮鮑麃麅麭𠣳𡂘𡧙𡯈𡾌𢾳𢿏𣕅𣚇𣟏𣮃𣶐𤔉𥶔𦐸𦔩𦠖𧙌𨋛𨣙𩂞𩆘𩎘𩎾𩐜𩗥𪊳"},
    {U"pei", U"㚰㟝㤄㧩㯁㳈㾦䊃䏽䟺䣙䩛䪹䫊䯱伂佩俖倍呸啡坏垺培妃妚姵婄嶏帔怌抷掊攈斾旆昢柭柸棑棓毰沛浿淠犻珮琣肧肺胚艴茇茷蓜蜚衃裴裵賠赔轡辔配醅錇锫阫陪陫霈馷駍𢁖𢘀𢥐𣍺𣬆𣯱𤗏𤜲𤬃𥄔𥹂𦙂𦸪𧳏𧴥𨓿𨙶𨛬𩎜𩑢𩖭𩵣"},
    {U"pen", U"㖹吩呠喯喷噴歕汾湓濆瓫盆翸葐衯𠺔𠽾𡺜𩸂𪂽"},
    {U"peng", U"㛁㛔㠮㥊㧸㱶㼞䄘䍬䡫䣙䥋䦕䧛䰃䴶亨倗傍傰剻匉嘭堋塜塳庄弸彭怦恲憉抨挷捧掽搒摓旁朋梈棚椖椪槰樥泙洴淎淜滂漨漰澎烹熢痭皏砰硑硼碰磞稝竼篣篷絣纄胓膨芃苹荓莑蓬蘕蟚蟛踫軯輣輧逢逬錋鑝閛韸韼駍騯髼鬅鬔鵬鹏𡂫𡗗𡼜𢏳𢪋𢼩𢽩𣟀𣨞𤖳𤘾𤡭𥕱𥕽𦚝𦪪𦯰𧌇𧚋𧴂𨂃𨅘𨍩𨎧𨎳𨑎𨠟𨲰𨺀𩄦𩐛𩖛𩡕𩮗𩱀𩱋𩸀𪔍𪔔"},
    {U"pi", U"㓟㔥㨢㨽㮰㯅㱟㳪㵨㼰㽬㿙䇑䏘䑀䑄䚰䚹䠘䡟䤏䤨䤵䦼䪹䫌䫠䯱䰦䲹䴙䴽丕仳伓伾俾僻副劈匹卑否啤噼噽嚊嚭圮坏坯埤培壀奊妚媲嫓屁岯崥嶏帔庀庇庳怶悂憵扑批披抷拂揊擗旇朇枇枈椑比毗毘毞淠潎澼濞炋焷狉狓猈琵甓番疈疋疲痞痦癖皮睥砒磇礔礕秛秠稫笓篦篺粃紕纰罴罷羆翍耚肶脴脾腗膍芘苉苤萆蕃薜蚌蚍蚽蚾蜱螕螷蠯被裨諀譬豼豾貔辟邳郫鄱釽鈈鈚鈲鈹鉟銔銢錃錍鎞铍闢阰陂陴隦霹鞞頗駓髬魮魾鮍鲏鴄鵧鷿鸊鼙𠜱𠡄𠨸𠪮𠯔𠯭𠵬𠹦𡊝𡛘𡛡𡦟𡲮𡶌𡺮𢇳𢓖𢞗𢰘𢱧𢻹𢾇𢾱𣓋𣔬𣖰𣢋𣪉𣬉𣬮𣬼𣴪𣹚𣹮𤂃𤖿𤘢𤘤𤘹𤚪𤜻𤬭𤱍𤴣𤷒𤼜𤿇𤿎𤿐𤿙𥀘𥔁𥤻𥯡𥱼𦀘𦃋𦊁𦘩𦘲𦤢𦨭𦰽𦳈𦹽𧑜𧓎𧙁𧧺𧪫𧲺𧳼𧴉𧾑𨈚𨐴𨑜𨤽𨧦𨲐𨵡𨵩𨸆𨺤𨻀𩔙𩗫𩜰𩣚𩫫𪄆𪇊𪉔𪊕𪌈𪖞𪛎𬳵"},
    {U"pian", U"㓲㛹㸤㼐㾫䏒䮁便偏囨媥平徧扁楄楩片犏猵璸篇緶翩胼腁萹蝙褊覑諚諞谝貵賆跰蹁辨辯鍂駢騈騗騙骈骗骿魸鶣𠐈𠑟𠯯𠷊𡎚𢉞𢐃𢕨𦳄𧍲𧡤𧱩𨂯𨲜𨵸𨸇𪘀𪚏"},
    {U"piao", U"㩠㬓㵱㹾㼼䏇䕯䴩僄剽勡嘌嫖彯徱慓摽旚朴殍漂潎犥瓢皫瞟票篻縹缥翲膘莩蔈薸螵謤醥闝顠飃飄飘驃驫髟魒麃𠷻𡢱𡣋𣋳𣝐𣳭𦭼𧌠𧽤𨝓𨮬𩄷𩗏𩙒𩛞𩡦𩮳𪅃𪋖𪏫"},
    {U"pie", U"䥕丿嫳撆撇暼氕潎瞥苤蔽覕鐅𠟈𠢪𢳂𤮕𦒐𦗥𩓼𩠿𩮳𬭯"},
    {U"pin", U"㡦㰋㺍䎙匕品嚬姘娉娦嫔嬪拚拼榀汖泵牝玭琕矉砏礗穦聘薲蘋蠙貧贫頻顰频颦馪驞𠐺𠮰𡛞𢣐𢬵𢶳𣎳𥑓𥖶𦇖𧔪𧭹𧮝𨏞𨽗𩕵𩰗𬞟"},
    {U"ping", U"㵗㺸㻂䀻䈂䍈䓑䛣䶄乒俜倗凭凴呯坪堋塀娉屏屛岼帡帲幈平慿憑枰檘泙洴涄淜焩玶瓶甁甹砯砰硑秤竮箳簈缾聘聠胓艵苹荓萍蓱蘋蚲蛢評评軿輧郱鉼頩馮鮃鲆𠗥𠗦𡊞𢆟𢖊𤭔𤳊𥪁𥭢𥵪𦀔𦚓𦥚𦥤𦶊𧂋𧏑𨂲𩂾𩈚𩩍𪋋𪔾𪕒"},
    {U"po", U"㗶㛘㤕㧊㨇㩯䄸䋣䍨䎅䎊䞟䣪䣮䥽䨰䪖䪙䯙剖叵哱嘙坡奤婆尀尃屰岥岶巿搫敀昢朴桲櫇泊泼洦溌溥潑濼烞猼珀番皛皤破砶笸粕繁翍膊蒪蔢謈跛迫鄱酦醗醱釙鉕鏺钋钷陂霸頗颇馞駊髆魄𠰐𠰼𠵳𠷑𠾌𡊟𡶆𡼃𡽠𢂤𢱨𢶉𣍸𣎵𣬚𣲳𤀪𤖼𤝯𤱍𤽌𥄔𥗟𥬒𥵜𥹖𦃡𦍁𦐦𦑀𦑵𦒟𦥭𦥲𦫔𦾕𦿍𧂉𧘟𧙅𧴤𧻙𧿽𨂩𨅅𨆵𨍭𨑝𨠓𨡩𨫁𨸭𩊀𩑻𩑼𩒕𩔈𩕏𩢘𩸿"},
    {U"pou", U"㕻㧵㩠㰴䎧䬌䯽䳝剖吥咅哣垺培堷娝婄抔抙抱捊掊棓涪犃瓿箁裒襃踣部錇𠟌𢒷𦥑𦵿𦺎𧠾𩔻𩚭"},
    {U"pu", U"㒒㬥㯷㲫㹒㺪䈬䈻䑑䔕䗱䧤䮒䲕䴆仆僕剥匍卜噗圃圑圤埔堡墣巬巭扑扶抪捗撲擈攴攵普暜暴曝朴柨樸檏氆浦溥潽濮瀑炇烳獛璞甫痡瞨砲秿穙箁纀脯舖舗苻荹莆菐菩葡蒱蒲蜅襆諩譜谱豧贌蹼酺鋪鏷鐠铺镤镨陠鯆鵏𠽾𡜵𡰿𢈲𢼹𣋏𤆝𤗵𤰑𤾣𥐁𥐚𥛟𥣈𥼜𦬙𦮑𧙛𧦞𧭎𧱹𨁏𨛥𨽂𩂗𩑀𩪛𩯱𪋡𪒢𪔿𪖈𫚙"},
    {U"qi", U"㒅㖢㞓㞚㟓㟚㟢㠌㠱㣬㥓㩻㩽㩾㫓㬤㮑㯃㯦㰗㰟㱦㼮㾨䀙䁈䁉䄎䄢䄫䅤䅩䅲䇍䉻䋯䌌䎢䏅䏌䏠䏿䐡䑴䒗䒻䓅䓫䔇䔾䗁䗩䙄䚉䚍䛴䞚䟄䟚䟷䡋䡔䢀䣀䣛䥓䥛䥢䧵䩓䩯䫏䫑䫔䬣䭫䭬䭶䭼䯥䰇䰴䱈䲬䳢䳶䶒䶞七丌乞亓亝亟企伎俟倛偈傶僛其凄切刺剘勤吃启吱呇呮咠唘唭啓啔啟喰嘁噐器圻埼夡奇契妻娸婍宿屺岂岐岓崎嵜己帺幾弃忔忮忯忾恓恝悽愒愭愾慼慽憇憩懠戚扢扱扺技抵挈捿掑揭摖支攲敧斉斊旂旗晵暣朞期杞枝柒栔栖桤桼梩棄棊棋棨棲榿槭檱櫀欫欺歧气気氣汔汽沏泣洓淇淒湆湇溪滊漆漬濝濟炁焏猉玂玘琦琪璂甈甭畦畸疧盀盵矵砌碁碕碛碶磎磜磧磩礘示祁祇祈祺禥禨稘稽竒簯簱籏粸紪絜綥綦綨綮綺緀緕緝纃绮缼罊耆肐肵脐臍舙艩芑芞芪萁萋萕葺蕲薺藄蘄蚑蚔蚚蛣蛴蜝蜞螇螧蟣蟿蠐衹袳裿褀褄觭訖諆諬諿讫豈起趞趿跂踑踖踦蹊躩軙軝迄迉逗邔郪鄿釮錡鏚鐖锜闙隑霋頎颀饑騎騏騹骐骑鬐鬾鬿魌鮨鯕鰭鲯鳍鵸鶀鶈鸂麒麡鼜齊齐齮𠀁𠁭𠅚𠊔𠎰𠐾𠓪𠔶𠧒𠫸𠴩𠴹𡍪𡎫𡖾𡢖𡦍𡪵𡫁𡱔𡵒𡷞𡹉𡹓𡹘𡹩𡺓𡺸𡻧𡻰𡽼𢁒𢍁𢍆𢍑𢔆𢔠𢜱𢞒𢢖𢢞𢦽𢩡𢴰𢺵𢺷𢻋𢻚𢻪𢾦𢾪𣉓𣏶𣔘𣛺𣪠𣫱𣯆𣶠𣾤𤘌𤪌𤳃𤳤𤷍𤹸𤺗𤼅𤿡𥀻𥄜𥇚𥉐𥉙𥉷𥉻𥌁𥓾𥔩𥖫𥤥𥫟𥷇𥼘𥽳𦄊𦈦𦔌𦖊𦘸𦙊𦚊𦛰𦡹𦧉𦧯𦩣𦪊𦫡𦭲𦮼𦸆𦸓𦸗𧇜𧋉𧌞𧎪𧒕𧓑𧕉𧘗𧘧𧙞𧙾𧚨𧠪𧡘𧡺𧯯𧰙𧻕𧼕𧼘𧽓𨁐𨉸𨊰𨑤𨒅𨙬𨙸𨞢𨥦𨪌𨱜𨵆𨵎𨸒𨸔𨾛𩉬𩒛𩒨𩠦𩠨𩥂𩦋𩧌𩨘𩨝𩲪𩳣𩴪𩷾𩸞𩹵𩺲𪀩𪂛𪄖𪄭𪅾𪒆𪒑𪔪𪗅𪗆𪗍𪗏𪙧𬨂"},
    {U"qia", U"㓞㓣㓤㡊㤉㧎㮫䁍䂒䈓䛩䠍䨐䯊䶗䶝佉價冾卡咭圶客峠帢恰愘抲拤挈掐揢搳擖楬殎洽疴矻硈磍絜葜袷跒酠鞐髂鮚𠕣𠜤𠜼𠝘𠝛𠳌𡘧𡤫𢮌𢼣𣁴𣘟𣣟𣨄𤫶𤵹𥇌𥉰𥎸𥦞𥴭𦝣𦸉𧩶𩥌𩩱𩮁𩷻𪘺𫈰"},
    {U"qian", U"㐸㓺㗔㜞㟻㡨㢛㥶㦮㦿㧄㨜㩃㩮㩷㪁㪠㯠㸫㹂䀒䁮䅾䇂䇜䈤䈴䉦䊴䐶䑶䕭䖍䙴䞿䤘䥅䦲䪈䫡䭑䭤䵖䵛乾仟仱佥俔倩偂傔僉儙兛凵刋前千厱唊嗛圱圲堑塹墘壍奷婜媊嬱孅孯寨岍岒嵌嵰幵廞忏忴悓悭愆慊慳扦扲拑拪掔掮揃揵搴摼撁撍撖攐攑攓朁杄杴柑棈椠榩槏槧橬檶櫏欠欦欿歁歉歬汘汧浅涔淒淺湔漸潛潜濳灊灒煔熑燂燖牵牽犍玪瓩皘竏筋签箝箞篏篟簽籖籤粁綪縴繾缱羥羬肷脥腱膁臤艌芊芡茜茾葥葴蒨蔳蕁藖虔蚈蚙蜸褰諐謙譴谦谴谸赶軡輤迁遣遷釺鈆鈐鉆鉗鉛銭鋟錎錢鍼鎆鏲鐱鑓鑯钎钤钱钳铅開阡雃靬韆顅顩馯騚騝騫骞鬜鬝鰜鰬鳽鵮鶼鹐黔黚齦𠀼𠊭𠋵𠑲𠔺𠠃𠢍𠬾𠳋𠷁𡒌𢁮𢂺𢃘𢃥𢋔𢌍𢍱𢜩𢧥𢮄𣍰𣓅𣖳𣘝𣟋𣢖𣢬𣢲𣹥𤠿𤬯𤿷𥏥𥔮𥜴𥡝𥦃𥧬𥮒𥱺𥲢𥳐𥴤𥷪𦂒𦅋𦖎𦢙𦴑𦼓𧃑𧘜𧚫𧛓𧟑𧢞𧣑𧥛𧪯𧮮𧮽𧲀𧽐𨐋𨐩𨓲𨗦𨜻𨝍𨥞𨦄𨨘𨰂𨱫𨺩𨺫𨽨𩋆𩎖𩑟𩑳𩒣𩦊𩨃𩨊𩨓𩪢𩬚𪇇𪈇𪉻𪑙𪘦𬘬"},
    {U"qiang", U"㛨㩖㱿㳾㾤䤌䵁丬創勥呛哐唴啌嗆嗴墏墙墻嫱嬙將嶈廧強强彊慶戕戗戧抢控搶摪斨枪椌槍樯檣溬漒炝熗爿牄牆猐獇玱瑲矼箐篬繈繦羌羗羟羥羫羻腔艢蔃蔷薔蘠蜣襁謒跄跫蹌蹡錆鎗鏘鏹锖锵镪鶬𠼢𡠥𡬎𡸤𡺛𢈵𢏄𢐩𢧅𣫝𤕽𥇉𥓌𥴻𥶑𦯤𦳟𦷦𧇞𧖑𧭚𧱡𧽩𨄚𨶆𩚕𩣼𩩝𩼒𩿄𪁸𪎞𪙎"},
    {U"qiao", U"㚁㚽㝯㡑㢗㤍㪣㴥㺒䀉䂪䂭䃝䆻䇌䎗䚩䦒䩌䫞䯨䱁䲾䵲丂乔侨俏偢僑僺削劁喬喿嘺噭塙墝墧墽嫶峭嵪嶠巧帩幓幧悄愀愁憔招捎搞摮撬撽敫敲校桥槗樵橇橋橾殻殼毃毳潐焦燆燋犞癄睄瞧硗硚硝碻磝磽礄礉窍窯竅箾繑繰缲翘翹茭荍荞菬蕉蕎藮蟜誚譑譙诮谯趫趬跤跷踃踍蹺蹻躈郻鄗鄡鄥醮釥銚鍫鍬鏒鐈鐰锹陗雀鞒鞘鞩鞽韒頝顤顦驕骹髚髜𠏖𠿕𡌔𡩇𡰐𡰑𡺘𢄹𢐟𢘟𢩨𢮉𢶡𢻤𢿣𣂁𣂇𣂥𣒆𣖄𣜽𣟼𣦜𣯹𣹝𣺰𤴀𥀣𥁢𥉾𥟅𥪯𥹶𦅃𦢺𧄍𧣌𨃤𨅣𨜍𨜑𨝱𨞶𨸑𩖇𩨟𪑊"},
    {U"qie", U"㓶㗫㚗㛍㛗㛙㤲㥦㰤㰰㰼㹤㼤㾀㾜䈉䞣䟙䠍䤿䦧䫔且伽倢切匧厒唼喋嗛契妾婕帹怯悏惬愜慊挈捷朅椄沏洯淁渫漆猰疌癿砌稧穕窃竊笡箧篋籡緁聺脞苆茄蕺藒蛣蛪詧趄跙踥輵郄鍥鐑锲魥鯜鰈𠀃𠁠𠋧𠩂𠲵𡂠𡐤𡔣𡘍𡛠𡝍𡶐𢍆𢲶𢺅𣠺𤴼𤷾𥕑𥪵𥿚𦆍𦼰𦿋𧑨𧚨𧚪𧫕𧻘𧻧𨄊𨉪𨖰𨚧𩣴𪑗𪙌"},
    {U"qin", U"㓎㕂㕋㘦㝲㞬㢙㤈㩒㪁㮗㱽㾛㾣䃡䃢䈜䔷䜷䦦䫬䰼䵖亲侵儭勤吢吣唚嗪噙坅埁埐堇墐媇嫀寑寝寢寴嵚嶔嶜庈廑忴慬懃懄扲抋捦揿搇撳擒斳昑梫槿橬檎櫬欽沁浸溱滲澿瀙珡琴琹瘽矜禽秦笉綅耹肣臤芩芹菣菦菳蓁藽蘄蚙螓螼蠄衾衿親誛赺赾鈂鈊鈙鈫鋟钦锓雂靲頜顉顩駸骎鬵鮼鳹鵭𠓿𠔎𠖶𠘅𠜘𠦎𠻨𡫧𡵑𡹢𢫲𢱶𣆲𣖯𣜣𣢐𣨠𣪄𤙋𤚩𤥓𤴽𤵂𤿳𥍯𥎊𥎡𥘋𥱧𥵧𦧋𦯈𧡘𧯃𧼒𧾏𨙽𨛣𨾰𩂈𩎖𩐙𩓒𩔝𩔟𪒭𪒯𪙟"},
    {U"qing", U"㩩㯳㵾㷫䂩䋜䔛䝼䞍䡖䨝䯧䲔倩倾傾儬凊剠勍卿啨圊埥声夝寈庆庼廎情慶掅擎擏晴暒棾樈檠檾櫦殑殸氢氫氰涇淸清渹漀濪甠硘硜碃磬箐精綪綮罄胜苘莔葝蜻親請謦请軽輕轻郬鑋靑青靘頃顷鯖鯨鲭鶄黥𠑴𠗝𠨍𡄇𡄔𡲀𢹃𣩜𣫨𤭩𥃟𥱨𧕙𧖪𨆪𨓷𨻺𩇝𩇟𩑭𩒵𩔥𩗼𩷏𩽡𪄈𪏅"},
    {U"qiong", U"㑋㒌㝁㧭㮪㷀㼇䁚䃔䅃䆳䊄䓖䛪䠻儝卭嬛宆惸憌桏橩焪焭煢熍琁琼璚瓊瓗睘瞏穷穹窮竆笻筇舼芎茕藑藭蛩蛬赹跫邛銎鞠𠊫𠌖𠤊𡊼𡞦𡦃𡸕𡺺𢞏𢮍𢶇𣇬𣋶𣑦𣜧𣶆𤢶𤤑𤤶𥑎𥑱𥨪𥳎𦦧𦨰𦭭𦾵𨀯𨉫𨍶𩑓𩢽𩨯𩬛𩬰𪀛"},
    {U"qiu", U"㐀㐤㕤㚱㛏㞗㟈㤹㥢㧨㭝㳋㷕㺩㺫䆋䊆䊵䎿䐐䜪䞭䟬䟵䠓䠗䣇䤛䨂䱸䲡丘丠仇俅區厹叴唒囚团坵媝崷巯巰恘惆愀扏捄搝朹梂楸橚櫹殏毬氽氿求汓泅浗渞湫湭煪牫犰玌球璆皳盚秋秌穐篍糗紌絿緧肍艽莍萩蓲蘒虬虯蚯蛷蝤蝵蟗蠤裘觓觩訄訅賕赇趜趥踆逎逑遒邱邺酋醔釓釚釻銶鞦鞧馗鮂鯄鰌鰍鰽鱃鳅鳩鶖鹙鼽龜龝𠀉𠗈𠰋𡊣𡲚𢈝𢘄𢛃𢦎𣧝𣭳𤕾𤞰𥔻𥥽𥫷𥭑𦦄𦬖𦰪𦵩𧇸𧏋𧔭𧣕𧤕𧲰𧳫𧺤𧻁𧻱𧼭𨍊𨒊𨕦𨟽𨱇𨲒𨺧𩈸𩒮𩔕𩗕𩝠𩵍𩵰𩾁𪍗𪖛𪚺"},
    {U"qu", U"㖆㘗㜘㜹㠊㣄㧁㩴㫢㭕㯫㰦㲒㲘㸖㻃㽛㾀䁦䂂䆽䈌䋧䒧䒼䓚䓛䕮䖦䗇䝣䞤䟊䠐䢗䢹䧢䵶䶚伹佉佢刞劬匤区區厺去取句呿唟坥娶屈岖岨岴嶇巨弆忂怚憈戌戵抾敺斪曲朐枸欋欪毆氍浀淭渠灈焌璖璩癯瞿磲祛竘竬筁籧粬紶組絇翑翵耝胊胠脥臞苣菃葋蕖蘧蚼蛆蛐蜡蝺螶蟝蠷蠼衐衢袪覰覷覻觑詓詘誇誳诎趋趍趜趣趨跔跙跼躣躯軀軥迲遽鉤鐻鑺镼閴闃阒阹鞠鞫駆駈騶驅驱髷魼鮈鰸鱋鴝鶌鸜鸲麮麯麴麹黢鼁鼩齲龋𠇯𠍲𠏛𠣪𡟥𡡥𡱅𡱺𡲰𡳆𢂁𢌄𢌷𢎖𢦌𢴮𢼰𣖪𣭳𣮈𣯸𣰋𣰠𣰡𣰻𤖬𤙏𤢓𤨎𥃔𥗫𥧻𥬔𥶶𥺷𥽧𦄽𦐛𦔬𦕙𦗛𦛕𦛱𦣒𦸶𦼫𧄒𧉧𧊛𧌑𧐅𧕎𧗪𧝔𧠢𧲵𧻃𧾱𧾶𨃘𨄅𨊘𨎶𨐣𨓭𨞙𨧱𨱊𨸟𨼫𨼽𩇐𩉿𩍔𩖷𩢳𩣹𩧘𩪍𩴹𩵅𩽩𩿟𩿥𩿩𪀊𪁖𪄊𪆂𪉌𪋄𪌆𪌬𪍸𪛃𪨰"},
    {U"quan", U"㒰㒽㟫䀬䄐䅚䊎䌯䑏䟒䠰串佺全券劝勧勸卷啳圈圏圳埢奍姾婘孉峑巏弮恮悛惓拳拴捲搼权栓桊棬椦楾槫権權汱泉洤湶灥烇牶牷犈犬犭狋獾瑔甽畎痊矔硂筌純絟綣縓绻腃荃葲虇蜷蠸觠詮謜譔诠跧踡輇辁酄醛銓鐉铨闎韏顴颧駩騡鬈鰁鳈鸛齤𠛮𠤹𠥙𡇮𡈉𡙅𡙐𡰝𡴔𡺟𡿨𢍕𢎠𢑆𢔑𢫔𣍴𣸋𤜍𤥷𤬠𤰝𤷄𥁠𥁸𥤊𥹳𦋓𦍅𦏮𦓰𦨚𧈾𧍭𧸾𨛈𨜩𨟠𨨗𨩸𩓫𩘘𩜬𩧴𪈻𪐂"},
    {U"que", U"㕁㩁㰌㱋㱿㲉㴶㹱㾡䇎䍳䐨䟩䦬䧿䲵傕却卻埆塙墧屈崅悫愨慤搉攉敠榷汋決炔燩猎琷瘸皵硞确碏確碻礐礭缺舄芍蒛蚗觳趞踖闋闕阕阙隺雀鳥鵲鹊𠞗𡇱𡉉𢔱𢠬𣖫𣛵𣤇𣪹𤡯𤣅𤷽𤿋𤿩𤿵𥀎𥆸𥕹𥗙𥗮𥜵𥩢𧎯𧢩𧢭𨞩𨟠𨢜𨴊𨴒𨵗𩤈𩨭𩨷𩫠𪏈𪏨𪖀𬒈"},
    {U"qun", U"㟒㪊㿏䊎䭽囷夋宭峮帬歏箘羣群裙裠踆蹲輑逡遁麇𡈀𢛕𣀄𤛭𤸷𦃢𦽖𧼐𨞗𩎗𩤁"},
    {U"ran", U"㒄㚩㜣㦓㲯㸐㾆㿵䎃䑙䒣䔳䕼䖄䣸䤡䫇䳿冄冉呥嘫姌媣染柟橪然熯燃珃繎肰苒蒅蚦蚺衻袇袡蹨髥髯𠊌𠤀𠯍𠱞𡖝𡜉𡜫𢓒𣰦𤙼𤡮𤱋𤲗𥀫𥀭𥣺𥬕𥳚𦫉𨹌𩃵𩢡𩧬𩶎𪓘𪓚𪚮"},
    {U"rang", U"䉴䑋儴勷嚷壌壤孃忀懹攘欀瀼爙獽瓤禳穣穰纕蘘蠰譲讓让躟鑲鬤𢐿𣩽𣰶𤅑𤬥𤰂𥗝𧟄𨏛𨟚𩆶"},
    {U"rao", U"㑱㹛娆嬈扰撓擾桡橈犪穘繚繞绕荛蕘蟯襓遶隢饒饶𠒸𡈦𦪛𧳨𨇄"},
    {U"re", U"㳧偌喏惹捼渃热焫熱若蹃𢞇𤍠𤑄𧧏𩭿"},
    {U"ren", U"㠴㣼㲽㶵㸾䀔䀼䇮䋕䌾䏕䚾䛘䭃人亻仁仞仭任儿刃刄壬妊姙屻岃忈忍忎恁扨朲杒栠栣梕棯涊牣祍秂秹稔紉紝絍綛纫纴肕腍芢荏荵菍葚衽袵訒認认讱躵軔轫釰鈓銋靭靱韌韧飪餁饪魜鵀𠯄𠲏𡰥𢆉𢇦𣅉𦍌𦏀𦬄𧥷𧴬𨉃𩑉𩠈𩵕𪔺"},
    {U"reng", U"㭁㺱䄧䚮仍戎扔礽穰耳艿芿辸陾𠧟𠮨𠯷𠯹𢫨𣗐𥾋𧹈𨸐"},
    {U"ri", U"䒤囸日氜釰鈤馹驲𠟅𡆸𡈎𡉭𤝍𦨙"},
    {U"rong", U"㘇㝐㣑㭜㲓㲝㲨㺎㼸䄾䇀䇯䈶䘬䠜䡆䡥䢇䤊䩸傇傛冗坈媶嫆嬫宂容峵嵘嵤嶸巆戎搈搑曧栄榕榮榵毧氄溶瀜烿熔爃狨瑢穁穃絨縙縟绒羢肜茙茸荣蓉蝾融螎蠑褣軵鎔镕隔頌駥髶𠌚𠞕𠰽𠹍𡊫𡊸𡖢𡦼𡫦𡭋𢐿𢦿𢫨𣭲𣮪𣯍𣯏𣯐𣰇𣲽𤘺𤘻𥎂𥎜𥑳𥨳𥬪𥼬𦔋𦗋𦗨𦶇𧉡𧎣𨉴𨉷𨋠𨌣𨍅𨍷𨒆𨲟𩍉𩎂𩒷𩚗𩮠𩼅𪃾𪕁𪕎𪗴"},
    {U"rou", U"㖻㽥䏔䐓䧷䰆厹媃宍揉柔楺渘煣瑈瓇禸粈糅肉腬莥葇蝚蹂輮鍒鑐鞣韖騥髳鰇鶔𠠐𡗑𢔟𢘅𥠊𦍭𨛶𪑶𫐓"},
    {U"ru", U"㐵㦺㨎㳶㹘㼋㾒䄾䋈䞕䰰乳侞偄儒入吺咮嗕嚅女如媷嬬孺嶿帤扖挐擩曘月杁桇檽汝洳渪溽濡燸獳筎縟繻缛肉肗臑茹蒘蓐蕠薷蝡蠕袽褥襦辱邚鄏醹銣鑐铷需顬颥鱬鳰鴑鴽𠟺𠤦𡄲𡈴𡜃𡜚𡥶𡫽𡮚𢖵𢛚𣖹𣚐𣭠𣯋𣽈𣽉𤡤𥙦𥞚𦤊𦭰𦳾𦷸𧊟𨚴𨨜𩄋𩍥𩱨𩶫𩶯𩸐𪏮𪑾"},
    {U"rua", U"挼"},
    {U"ruan", U"㓴㧫㮕㼱㽭䆓䎡䓴䙇䞂䪭偄堧壖媆撋擩朊檽濡燸瑌瓀碝礝緛耎腝蝡軟輭软阮需𠟺𠤦𢘧𢡵𢱾𣃅𣡗𣽈𣽳𤧠𤲬𥈇𥊶𥎀𥎘𥩗𥯬𦶇𦺾𨋚𨒩𨨰𨪳𨬔𩏈"},
    {U"rui", U"㓹㢻㧫㨅㪫㮃㲊䂱䄲䅑䇤䌼䓲䜭䬐兑内叡壡婑惢抐撋枘桵棁橤汭瑞甤睿笍綏緌繠芮苼蕊蕋蕤蘂蘃蚋蜹踒鈉銳鋭鏸锐𡯒𢣳𣛚𣬘𥳝𦼆𧄜𨧨𨳙𪏩"},
    {U"run", U"㠈䏰䦞撋橍润潤瞤膶閏閠闰𠷀𥆧𨳝𨷎𩀋"},
    {U"ruo", U"䐞䚥偌叒婼嵶弱惹挼捼撋楉渃溺焫爇箬篛芮若蒻鄀鰙鰯鶸𢜪𤍽𤣼𦩸𧃪𨀝𨴚"},
    {U"sa", U"㒎㚫㪪㽂䊛䑥䙣䛽䬃仨卅挱挲摋撒攃檫櫒殺泧洒潵灑纚脎萨蔡薩虄訯趿躠鈒鎝鏾钑隡霅靸鞈颯飒馺𠎷𠦃𠬙𠮿𠱡𠿓𡄳𡐥𡒁𢓔𢕬𢫬𢶍𢻨𣀯𣘤𣜂𣬬𥋌𥵯𥸗𥻦𦠿𦻅𦼧𧀕𧭝𨃛𨆂𨐖𨷆𩆅𩎕𩐅𩗉𩗞𩣞𩨞"},
    {U"sai", U"㗷㘔㩙䈢䚡䰄僿嗮嘥噻塞思愢揌毢毸簺腮賽赛顋鰓鳃𡨄𡬉𦞫𪃄"},
    {U"san", U"㤾㧲㪔㪚䈀䉈䊉䫅䫩三仐伞俕傘傪厁叁參壭帴弎散橵毵毶毿潵犙糁糂糝糣糤繖蔘謲鏒鏾閐饊馓鬖𠮿𡙘𢁘𢕕𣀧𣀫𣬛𣮠𥊀𥒬𦙱𦙸𦡨𦷻𦺻𧗋𧱆𧽾𨸃𩀲𩀼𩜱𩞀𩨡𩯑"},
    {U"sang", U"䘮䡦䫙丧喪嗓搡桑桒槡磉纕褬鎟顙颡𡕏𡠏𣉕𣊝𣞙𤸯𥔫𦅇𦟄𧍨𨢆𩐷𩦌𩺞𪔬"},
    {U"sao", U"㛮㥰㲧㺐㺑㿋䐹䑹䕅哨埽嫂慅懆扫掃掻搔橾氉溞燥瘙矂縿繅繰缫臊螦鄵鐰颾騒騷骚髞鰠鰺鱢鳋𠋺𢔳𢜶𢠡𢤁𢮞𣉔𣔱𣰕𤠘𤢖𥰱𦏛𦕏𦞣𦟄𦺋𦾘𧂩𧑫𧖠𨃣𨧪𨪊𩙈𩙰𩫦𩮚𪍻"},
    {U"se", U"㒊㥶㮦㱇㴔㻭㾊䉢䊂䔼䨛啬嗇塞寨廧愬懎拺擌栜槭歮歰泣洓涩渋溹漬澀澁濇濏瀒琗瑟璱瘷穑穡穯粣繬色薔虩譅轖鉍銫鎍鎩鏼铯閪闟雭飋𠎸𠟦𠟩𠢳𠵭𠽼𠿗𡫟𡵶𢀋𢃢𢡉𢱢𣚟𣽤𤁧𤖗𤛷𤾿𥈽𥱁𥷹𥻨𦆄𦐅𧈈𧒓𧒗𧨷𨆙𩃑𩄜𩇣𩊯𩍙𩏫𩕡𩰙"},
    {U"sen", U"㜗傪摻森椮槮洒滲襂𣟹𧂅𩕌"},
    {U"seng", U"䒏僧鬙𡬙"},
    {U"sha", U"㛼㠺㬠㰱㰼㲚㵤㸺䈉䝊䤬䬊乷倽傻儍刹剎厦哈唦唼啑啥喢嗄噎帹廈挲接摋攝杀杉桬榝樧歃歰殺毮沙濈煞猀痧砂硰箑粆紗繌纱翜翣莎菨萐蔱裟賒鎩铩閯閷霅霎魦鯊鯋鲨𠄒𠍽𠚺𡺧𢅑𢇗𢔣𢩖𢶌𢼵𣉜𣓉𣛶𣡽𣣮𣣺𣲓𣲡𣶤𣻑𤍁𤑣𤟃𥈊𥯦𦀛𦔯𦔰𦕉𦩿𦭉𦱵𦾚𧋊𧏫𧚨𧜁𧫝𧲌𧳛𧻵𨖷𨘉𨪍𩊮𩮫𩵮𪄅𪌮"},
    {U"shai", U"㩄㬠㴓䵘摋攦晒曬殺筛篩簁簛繺色諰酾釃閷𠱡𢄌𣻑𧜁𨢦𩂃𩂝𩴇"},
    {U"shan", U"㚒㣌㣣㨛㨻㪎㪨㯆㰑㴸㶒㺑䀐䁴䄠䘰䚲䠾䡪䥇䦂䦅䩔䪌䱇䱉䴮傓僐僤儃儋删刪剡剼善單嘇圸埏墠墡壇姍姗嬗山嶦幓彡扇挻掞搧摻撣擅擔攙敾晱杉柵栅椫樿檀檆櫼歚汕潬潸澘澹灗炶烻煔煽熌狦猭珊疝痁睒磰禪穇笘笧縿繕纔缮羴羶脠膳膻舢芟苫葠蔪蟬蟮蟺衫襂襳覢訕謆譱讪贍赡赸跚軕邓邖鄯釤銏鐥钐閃閄閊闪陕陝顃顫饍騸骟髟鯅鱓鱔鱣鳝鿃𠚹𠫹𠿞𡟨𢒉𢒹𢕻𢩢𢫔𢿈𣆴𣓒𣖉𣧺𣩧𣪶𣬛𣲀𤀤𤇄𤊼𤢖𤮜𤺪𥄘𥈚𥊀𥔱𥰢𥸣𦍸𦎞𦏂𦘹𦳫𦶋𦺭𧀵𧎥𧛄𧛡𧧵𧨾𧭽𧲾𧴭𧷶𨁆𨏪𨝩𨝵𨹈𨹊𩁺𩆤𩆫𩌰𩟋𩦐𪍶𪑃𫮃"},
    {U"shang", U"䟫䬕䵰䵼丄上伤傷商垧埫場塲墒尙尚恦愓慯扄晌曏殇殤湯滳漡熵禓緔绱蔏螪蠰裳觞觴謪賞贘赏踼鑜鞝鬺𠼬𤎘𤔚𤳈𤵼𥏫𧡮𧶜𨢩𨶼𩞃𩞧𪄲𬀷"},
    {U"shao", U"㪢㲈㷹㸛䈰䈾䏴䒚䔠䙼䬰佋削劭勺卲召哨娋少弰招捎搜旓杓柖梢溲潲烧焼燒燿玿睄稍笤筲紹綃綤绍艄芍苕莦萷蕱蛸袑裢輎邵鞘韒韶颵髾鮹𠣫𠧙𠷃𡡏𢦽𢼼𢾐𤉎𤱠𥙬𥳓𥵦𦄏𦓴𦡱𦯐𦿃𧣪𧳹𨈘𨊢𨙹𨛍𨱭𨲆𩬏"},
    {U"she", U"㓭㢵㤴㴇㵃䀅䀹䄕䜓䞌䠶䤮䬦䬷佘厍厙奓奢射弽慑慴懾折抴拾挕捨揲摂摄摵攝檨欇歙涉涻渉滠灄猞畬畲睫碟磼社聶舌舍舎葉蔎虵蛇蛞蛥蠂設设賒賖赊赦輋邪鉈鍦闍鞨韘騇麝𠋞𠪣𠴯𠾏𡄢𢉃𢗭𢶅𣝒𣠞𣣭𣸚𤙱𤜣𤠭𤺔𥁹𥍉𥔡𥘦𥝀𥰰𥿞𦁗𦯬𦴍𦼢𧉮𧌖𧮿𧵳𨛭𨝫𨣍𩂨𩂴𩙝𩩗𩮐𪨶"},
    {U"shei", U"誰谁"},
    {U"shen", U"㑗㕥㚞㚨㜪㥲㮱㰂㰮㵕㶒㾕䅸䆦䫖䯂䰠䰼什伸侁侺信兟參吲呻哂嘇堔妽姺娠婶嫀嬸审宷審屾峷幓弞愼慎扟抌抻搷敒昚曋曑柛棯棽椹榊槮氠沈涁淰深渖渗湛滲瀋燊珅甚甡甧申瘆瘮眒眘瞋瞫矤矧砷神祳穼籶籸紳綝绅罙罧肾胂脤腎莘葚葠蓡蔘薓蜃蜄裑覾訠訷詵諗讅诜谂谉身邥鉮鋠震頣駪魫鯅鯓鯵鰰鰺鲹鵢黮𠂧𠃫𠗿𠘆𠻝𡖬𡼬𢈇𢈯𢊖𢊲𢏎𢏦𢘊𢸙𣃵𣇗𣔗𣘘𣘲𣿇𤏗𤕊𤶴𥆣𥏖𥥍𥥿𥬐𥳱𥸬𦌀𦐹𦕽𦜊𦜜𦸂𦸯𦺷𧀯𧢹𧵬𨊘𨐍𨐔𨐕𨝐𨞲𨴐𩅨𩉼𩶇𩺵𬬹𬳽"},
    {U"sheng", U"㗂㮐㱡㹌㼩㼳㾪䁞䎴䚇䞉䪿䱆䲼䴤丞乘偗冼剩剰勝升呏圣垩墭声姓娍媵嵊憴斘昇晟晠曻枡栍榺橳殅殸泩渻湦澠焺牲狌珄琞生甥甸盛省眚竔笙箵縄繩绳聖聲胜苼蕂譝貹賸鉎鍟阩陞陹鱦鵿鼪𠇷𠓸𠓽𠴢𡞞𡨽𢦑𣢡𣬺𤚣𤯡𥘥𥟎𦔄𦕡𦖞𦛙𦩱𦳗𧍖𧡶𧪝𧿘𨁠𨕻𨚱𨜜𨲉𨲓𨵥𩍋𪅻"},
    {U"shi", U"㒾㔺㕜㖷㢁㫑㱁㳏㵓㸷㹝㹬㹷䁺䂖䂠䄷䈕䊓䌤䌳䏉䏡䒨䖨䗐䙾䛈䜴䟗䤭䤱䦙䦹䩃䭄䰄䲽䴓䶡世丗乨乭亊事什仕似佦使侍兘冟势勢匙十卋厔叓史呞呩咶唑啇嗜嘘噬埘埶堤塒士失奭始姼媞嬕实実室宩宲寔實寺尸屍屎峕峙崼嵵市师師式弑弒彖徥忕忯恀恃惿戺拭拾挈提揓斯施时旹是昰時枾柹柿栻楴榁榯檡殖殺氏汁沶浉液湜湤湿溡溮溼澤澨濕灑炻烒煶狧狮狶獅瑡畤痑眂眎眡睗矢石示礻祏秲竍笶筮箷篒篩簭籂絁繹耆肢舍舐舓莳葹蒒蒔蓍虱蚀蝕蝨螫褆褷襫襹視视觢訑試詩誓諟諡謚識识试诗谥豉豕貰贳赫跩踶軾轼辻适逝遈遞適遰遾邿郝醳釃釈释釋釶鈰鉂鉃鉇鉈鉐鉽銴鍉鍦鎩铈食飠飭飾餙餝饣饰馶駛驶魳鮖鯴鰘鰣鰤鲥鲺鳲鳾鶳鸤鼫鼭齛𠀍𠁗𠇳𠓤𠘪𠡥𠥿𠩔𠫺𠯰𠰚𠰴𠷇𡀗𡂓𡅵𡉸𡚼𡟕𡠋𡣪𡫵𡰯𡱁𡶈𡷈𡺔𢀕𢁓𢂑𢃰𢏯𢝬𢟯𢧏𢨝𢺿𢻘𢻫𢼉𢼊𣁒𣆘𣏚𣚌𣤘𣧚𣬐𤆰𤉏𤑦𤖻𤜣𤢼𤯄𤯜𤸤𤹌𥅔𥅞𥇲𥍸𥐘𥑏𥛨𥜰𥥥𥫴𥫽𥰰𥼶𥿅𥿮𦌿𦒈𦔂𦔥𦚨𦰯𦳊𦿇𧄹𧊖𧋒𧍀𧕯𧜂𧝊𧞲𧠜𧠡𧥖𧧅𧩹𧳅𧵋𧻸𨒍𨒧𨙩𨟂𨱡𨴯𨸝𨽄𩋡𩒂𩗎𩙢𩛌𩛏𩥐𩬭𩭐𩰢𪀔𪊢𪓻𪓿𪗧𫚕𬤊"},
    {U"shou", U"㖟㝊㥅㧃䛵䭭兽収受售嘼垨壽夀守寿手扌授掱收敊涭濤熟狩獣獸痩瘦綬绶膄艏醻鏉首𠈅𠱔𠷃𠷎𡧕𡬳𡭮𣒻𤙘𤚔𤱜𥅪𥅷𥙰𥨝𥾹𦣻𧈙𧌅𧚯𧜃𧤙𧯼𧵃𨛶𨞪𨱒𩠦𩠶𩴍𪈀𫜷"},
    {U"shu", U"㑐㒔㓱㛸㜐㡏㣽㫹㯮㲓㵂㶖㷂㸡㻿㼡㽰㾁䃞䇬䉀䑕䘤䜹䝂䝪䞖䟉䠼䢞䢤䨹䩱䱙䴰书侸俆俶倏倐儵兪叔咰售嗽塾墅姝娶婌孎孰尌尗属屬庶庻忬怷恕悆戍抒捈捒掓揄摅攄数數暏暑曙書朮术朱束杸杼枢树梳樞樹橾殊殳毹毺氀沭涑淑漱潄潏潻澍濖瀭焂熟瑹璹疋疎疏癙秫稌竖竪籔糬紓紵絉綀纾署翛腧舒荗荼菽蒁蒣蔬薥薯藪藷虪蜀蠴蠾術裋襡襩謶豎豫贖赎跾踈軗輸输述透鄃野鉥錰鏣鐲陎除隃鮛鱪鱰鵨鶐鷸鸀黍鼠鼡𠊪𠐊𠘧𠙎𠦜𠲌𠾢𡂡𡊍𡑀𡒒𡔪𡣈𡤽𡦛𡧔𡱆𢋂𢞣𢠫𢧇𣀻𣉛𣏗𣤯𣰿𣻚𤍓𤕟𤗪𤘷𤞉𤱐𤴙𤻃𥌚𥍝𥞃𥣋𥳕𥿇𦈌𦈷𦍄𦐣𦒶𦠦𦤂𦶕𦺗𦺪𧁿𧄔𧇝𧊠𧑓𧒑𧗱𧞀𧞫𧠣𧼯𨁀𨅒𨐅𨔦𨛭𨶝𨷙𨽉𩛅𩢻𩪍𩳅𩷌𩾈𪅰𪌶𪐧𬬸"},
    {U"shua", U"㕞刷唆唰涮耍誜選𠛚𤔙𩈥𩉆𩤤"},
    {U"shuai", U"㲤䢦卛帅帥摔率甩綏縗蟀衰𠌭𠞩𢕅𢕑𣘚𣼧𤀤𤠠𤸬𤺀𧍓𧗿𧜠𨄮𩘱"},
    {U"shuan", U"䧠專拴栓槫汕涮腨踹閂闩𡭐𢩠𢮛𣔫𣟴𣠸𤅲𦁆𦝏𦺲𨄔𨏉"},
    {U"shuang", U"㕠㦼㼽䉶䌮䔪䗮䝄䨥䫪傱双塽孀孇慡樉欆淙漺瀧灀爽礵縔艭鏯雙霜騻驦骦鷞鸘鹴𠗾𡑽𥡠𥱶𥲚𦄍𦆌𧄐𧕟𧕺𧴅𨇯𩅪𩆿𩦌𩽧"},
    {U"shui", U"㥨㽷䬽䭨䳠娷帨捝水氵氺涗涚睡瞓祱稅税脽裞説誰閖𠻜𡯑𡱊𢇤𢏅𤆙𥌘𥫸𦙙𦣢𧀣𨓚𨿠𩟥𩩞"},
    {U"shun", U"㥧䀢䀵䐏䑞䴄俊吮巛巡恂楯橓盾眴瞚瞤瞬舜蕣輴順顺鬊𨝜𨺠"},
    {U"shuo", U"㮶䀥䁻䈾䌃䔠勺哾嗍嗽妁揱搠數朔杓槊欶汋洬溯濯烁燿爍獡療矟硕碩箾萷蒴藥說説说銏鎙鑠铄𠲾𠲿𣀝𣝇𣷥𣸛𣻘𤄶𤡯𤢴𥌞𦂗𦃗𦋞𧋊𨨺𩟧𪎒"},
    {U"si", U"㒋㕽㚶㟃㠼㣈㭒㮐㴲㸻㹑㺇㺨㽄䇁䇃䎣䏤䒨䔮䡳䦙䫢䲉丝亖以伺似佀佁価俟俬偲傂儩兕凘厮厶台司咝嗣嘶噝四姒娰媤孠寺巳廁廝徙思恖愢撕斯杫析枱柶梩楒榹死汜泀泗泤洍涘澌瀃燍牭磃祀祠禗禠禩私竢笥簛籭糸糹絲緦纟缌罳耜肂肄肆菥蕬蕼虒蛳蜤螄螔蟖蟴覗謕貄逘釲鈶鈻鉰銉銯鋖鍶鐁锶雉颸飔食飤飴飼饲駟騃騦驷鷉鷥鸶麗鼶𠀓𠋡𠖓𠭈𠳎𡡒𢈶𢊀𢍭𢛥𢠹𢦲𣂖𣙼𣚄𣣌𣣑𣩠𣱻𣽷𤆟𤣵𤱸𤺊𥄶𥐀𥒲𥕶𥙉𥝠𥠱𥯨𥹊𦇲𦇵𦔌𦭡𦮺𦸷𦽕𧀚𧀩𧝤𧣛𧬊𧱅𧳙𨮭𨽼𩅰𩆵𩵗𩸟𩺛𪆁𪆗𪊍𪕳𪖉"},
    {U"song", U"㕬㞞㣝㧐㨦㩳㬝㮸䈡䉥䛦䜬䢠䯳䯷䲲倯傱凇吅娀宋崧嵩嵷庺忪怂悚愯慫憁憽捒揔摗松枀枩柗梥棇楤檧淞漎濍硹竦耸聳菘蓯蘴蜙訟誦讼诵送鍶鎹頌颂餸駷鬆𠳼𡇝𡷽𡾼𢓣𢔋𢔩𢖗𢤄𢱤𣚜𣽫𤾥𥳺𦬘𦯕𦷴𧊕𧌻𨠤𨱛𨱿𨴏𩃍𩃭𩠌𩩺𩮀𪀚𪨊"},
    {U"sou", U"㖩㛐㟬䈭䈹䉤䏂䐹䑹䗏䤹䩳䬒䮟䱸傁凁叜叟嗖嗽嗾廀廋捒捜搜摉摗撨擞擻敕族棷櫢欶涑溲潚獀瘶瞍籔艘蒐蓃薮藪螋謏鄋醙鎪鏉锼颼颾飕餿馊騪𠋢𠌞𠌟𠘂𠝬𠪇𡠼𡣂𢲷𢴼𣔱𣮬𣯜𤕇𥈟𥖻𥯪𥰞𦢩𦺌𧔅𧩮𧳶𧼭𧽏𨡻𨤇𨺦𩗣𩘠𩙫𩨄𩮃𩮶𩮸"},
    {U"su", U"㑉㑛㓘㔄㕖㜚㝛㢝㨞㩋㪩㬘㯈㲞㴋㴑㴼䃤䅇䌚䎘䏋䑿䔎䘘䛾䥔䲆俗傃僁僳卹嗉嗖囌圱埣塐塑夙嫊宿愫愬憟捽搬摵梀棴榡樎樕橚櫯殐泝洬涑溯溸潚潥玊珟璛甦碿稡稣穌窣簌粛粟素縤縮肃肅膆苏莤蓿蔌藗蘇蘓觫訴謖诉谡趚蹜速遡遬酥鋉餗驌骕鯂鱐鷫鹔𠐍𠷃𠻜𡎮𡖯𢋈𢎎𢖏𢚑𢢒𢸫𣝝𣩷𣫎𣯼𣶘𣷶𣿈𤌂𤛝𤠚𤡃𤢂𤢘𤤐𤥔𤭴𤸮𤼀𦌉𦌊𦎄𧀌𧐁𧐒𧐴𧔖𧜦𧞺𧥆𧩝𧺷𧼭𧽷𨱈𩐫𩐼𩘰𩘹𩙨𩝥𩲵𩳒𪁽𪄑𪅄𪋝𪌔𪍛𪐮𪖶𫂙𫗧"},
    {U"suan", U"㔯䔉䝜匴撰狻痠祘笇筭算篹蒜選酸𠥘𤶤𥳪𥴵𦾹𨠡𩆑𩈲𪘑𪘝"},
    {U"sui", U"㒸㞸㥞㴚㵦㻟㻪㻽䅗䉌䍁䔹䜔䠔䡵䢫䥙䧌䪎䭉䯝䯿亗倠哸嗺埣夊娞嬘尿岁嵗彗挼撋旞檖歲歳毸浽滖澻濉瀡煫熣燧璲瓍眭睟睢砕碎祟禭穂穗穟篲粹綏縗繀繐繸绥脺膸芕荽荾莎葰蓑虽襚誶譢谇賥遀遂遺邃鏸鐆鐩陏隊隋随隧隨雖靃鞖韢髄髓𠌱𠕸𠨌𠭥𡑞𡝓𡶣𡷼𡹖𡻕𢅕𢇥𢈼𢒱𢟩𣄧𣩡𣮄𣯧𣯯𤡪𤬫𤭢𤯖𤻄𥊴𥕸𥡜𥢍𥤼𥱵𥴦𥶻𦃒𦄑𦅵𦇀𦉎𦵭𦸏𧃚𧆑𧈧𧌢𧞸𧡏𧨧𧲈𧸙𨆏𨒉𨣢𨷃𨾡𨾬𩃃𩌩𩍚𩎰𩏘𩏚𩏲𩗶𩙇𩝌𩞅𩮴𫟦𬭼"},
    {U"sun", U"㔼㦏㨚䁚䐣喰孙孫扻损損搎摌栒榫槂潠狲猻笋筍箰簨荪蓀蕵薞跣鎨隼飧飱餐鶽𠣬𣕍𦠆𧂍𧎤"},
    {U"suo", U"㛖㛗㪽㮦䂹䅴䈗䐝䓾䔋䖛䞆䞽䣔䯯䲃䵀乺些傞唆唢嗍嗦嗩娑嫅惢戲所抄挲摍暛桫梭歲沙溑溹犧獻琐琑瑣璅睃簑簔索縒縮缩羧莎莏葰蓑蜶衰褨趖逡逤鎈鎍鎖鎻鏁锁霍靃髿魦鮻𠈱𠋲𠗼𠘺𠝿𠞯𠩄𠱗𡩡𡱳𢘿𢚭𢱡𢱢𢷾𣒹𣯌𤀤𤸴𤺫𥁲𥆝𥇇𥔭𥕘𥰼𦅊𦟱𦵫𧎫𧎳𧛻𧨀𧴪𧴲𨒽𨻈𨻨𩊮𩋝𩌆𩌈𩌢𩘝𩙭𩡾𩪈𩮛𩹳𪌮𪍌𪍔𪍟𪍨"},
    {U"ta", U"㒓㗳㛥㣛㣵㧺㭼㯓㯚㳠㹺㺚㿹䂿䈋䈳䌈䍇䍝䎓䑜䑽䓠䜚䪚䳴䵬䶀䶁他侤傝呾咜哈嗒嚃嚺塌塔墖太她它崉拓挞搨搭撻榙榻橽毾沓涾溚溻漯澾濌濕牠狧獭獺祂禢荅褟誻譶趿踏蹋蹹躢达達遝遢鉈錔鎉鎑铊闒闟闥闧闼鞈鞜鞳韃鮙鰨鳎鿎𠉂𠴲𠷍𡂖𡌩𢃕𢞠𢺉𣗶𣝋𣥂𣥷𣯚𤄥𤒻𤛣𤠐𤠟𤿽𥗓𦈖𦍒𦐇𦑇𦑲𦑶𦑼𦧛𦧞𦧟𦧥𦧱𦨎𦪙𦭟𦱆𦶑𦾽𧌏𧔣𧖆𧪦𧮑𨃚𨆰𨌭𨓬𨔯𨙎𨰏𨵝𨶀𨸉𩋅𩌇𩌉𩌐𩌘𩎽𩥑𩨌𩫊𩷽𩺗𪂌𪔕𪘁"},
    {U"tai", U"㑷㒗㘆㙵㣍㥭㬃㷘㸀䈚䑓䔶䣭儓冭台呔咍囼坮大太夳奤嬯孡忕忲态態抬擡斄旲枱檯汏汰泰溙漦炱炲燤珆箈籉粏肽胎能臺舦苔菭薹詒跆邰酞釐鈦鈶钛颱駘騃鮐鲐𡇷𡒢𢖤𣣿𤗿𥙉𦒰𦻀𧉑𧉟𧭏𧮼𩬠𩿡𪐝𪐥𪒴"},
    {U"tan", U"㘱㛶㜤㤾㨏㫜㲜㲭㳩㴂㵅㶒㷋㽎㽑䀡䃪䆱䉡䊤䏙䐺䑙䕊䗊䜖䞡䦔但倓傝僋儃叹單嗿嘆嘽嘾坍坛坦埮墰墵壇壜婒彈忐怹惔憛憳憻探摊撢弹撣擹攤昙暺曇榃橝檀歎毯沈淡湛湠滩漢潬潭澹灘炎炭燂璮痑痰瘫癉癱碳磹禪緂繵罈罎胆舑舔舕菼蕁蕈藫袒裧襢覃談譚譠谈谭貚貪賧贪郯醈醓醰鉭錟钽锬顃餤鷤黮𠫶𠻪𡅄𡊨𢅀𢇧𢇰𣁗𣞔𣢌𣴽𣵢𣸙𣼚𣽯𤎥𤐔𤺺𥩒𥰨𥹠𥼟𥼮𦃖𦌪𦗡𦙇𦧏𦧴𦨸𦸁𦼎𧂇𧣁𧣹𧥞𧫿𧰘𧵊𧺟𧽼𨁴𨂞𨅍𨝸𨡍𨣕𩑰𩒢𩖖𩠽𩡄𩡝𩤞𩪺𪉧𪊇𪍵"},
    {U"tang", U"㑽㒉㓥㙶㜍㭻㲥㼒㼺㿩䅯䉎䌅䕋䞶䟖䠀䣘䧜伖倘偒傏傥儻劏唐啺嘡坣堂塘嵣帑惝愓戃搪摥擴攩曭棠榶樘橖欓汤淌湯溏漟漡烫煻燙爣瑭矘磄禟篖簜糃糖糛羰耥膅膛蓎蕩薚蝪螗螳赯趟踼蹚躺逿鄌醣鎕鎲鏜鐋鐺钂铴镋镗閶闛闣隚鞺餳餹饄饧鶶黨鼞𠗶𠢃𠹔𡿓𢠵𢴳𢻿𣎲𣙟𤚫𤠯𤾉𥋡𦪀𦳝𧱵𨆉𨉱𨌩𨍴𨎋𨎖𨲗𨶈𩘜𩥁𩹶𪕹"},
    {U"tao", U"㚐㣠㫦㲈㹗䀞䀺䄻䈱䑬䚯䛌䛬䤾䬞䵚匋叨咷啕夲夵套姚嫍幍弢慆抭挑掏搯桃梼槄檮洮涛涭淘滔濤燾瑫祹籌絛綢綯縚縧绦绹萄蜪裪討詜謟讨跳轁迯逃醄鋾錭陶鞀鞉鞱韜韬頫飸饀饕駣騊鼗𠇏𠌪𠓝𠗆𠚜𠞞𠬢𡍒𡺫𢓘𢔇𣨔𣰺𣺮𤘸𤙎𤚟𤨐𤴻𤵟𥰜𦃡𦍷𦺰𨌨𨡒𩋃𩎢𩏾𩘿𩙧𩛽𩤉𩥅𩹴𪇘𪌼𫘦"},
    {U"te", U"㥂㧹匿式忑忒慝特犆職脦螣蟘貣貸鋱铽𠈸𢘋𣘱𤙰𥊸𥌩𦗟"},
    {U"teng", U"䒅䕨䠮䮴䲍䲢僜儯幐滕漛熥疼痋籐籘縢腾膯藤虅螣誊謄邆霯駦騰驣鰧鼟𢚺𢟱𢥂𣽨𤃶𤳘𤹤𥉋𦡪𦩫𦪝𦫀𧄼𧈜𧭔𨃗𨄇𩩻𩴝𪒿𪔶"},
    {U"ti", U"㔸㖒㖷㗣㡗㣢㥴㬱㯩䄺䅠䈕䌡䎮䏲䔶䖙䙗䚣䛱䢰䣟䣠䣽䨑䩟䪆䬫䬾䯜䱱䴘䶏䶑体俶倜偍剃剔厗啑啼嗁嚏嚔堤奃姼媂媞屉屜屟崹弟徥徲悌悐惕惖惿戻折挮掦提揥擿是替朑桋梯棣楴歒殢洟涕渧漽狄珶瑅瓋睇碮磃禵稊穉笹籊綈緹绨缇罤肆苐荑蕛薙虒蝭蟬衹裼褅褆詆諦謕趧趯踢踶蹄蹏躍躰軆达逖逷遆適醍銻錫鍗鐟锑隄題题騠骵體髰鬀鬄鮧鮷鯷鳀鴺鵜鶗鶙鷈鷉鷤鹈𠞄𡥩𡰎𡲕𡲿𡸑𢓡𢝹𢞖𢧑𢱦𢳓𣄍𣈡𣉆𣖅𣖸𣜹𣤖𣧂𣸒𣹲𤗘𤗢𤚢𤟥𤟾𤭌𥉈𥉘𥡦𥫵𥳳𥶛𦌢𦻀𧀠𧀰𧋘𧔩𧙣𧛒𧝆𧝐𧡨𧤍𧨱𧼮𨁃𨉥𨔛𨠏𨪉𨲎𨲞𨴼𩋣𩓂𩛑𩛶𩝊𩤽𩬲𩮜𩳕𩿷𪁩𪍲𪕩𪖦𫘨𫛸"},
    {U"tian", U"㐁㖭㙉㜤㥏㧂㬲㮇㶺䀖䄼䄽䋬䏦䐌䑙䑚䚶䟧䠄䡒䡘䣯䣶䥖䧃䬯䵺佃倎兲典吞唺嗔塡填天奵娗婖寘屇忝恬悿捵掭搷撣晪栝殄沗沺沾淟添湉滇琠瑱璳甛甜田甸町畇畋畑畠痶盷睓睼瞋碵磌窴紾緂胋腆舔舚苫菾蚕蚺覥觍賟跈酟鈿銛錪鍩鎮闐阗靔靝靦顚餂鴫鷆鷏黇鿬𠂈𠗘𡒧𡙒𢇶𢓍𣊖𤘠𤤦𤩱𤫞𤲖𥧑𥪌𥪧𥳫𥵶𦊊𦔿𦗀𦗁𦧒𦧖𦧝𦧡𦬞𦳇𧉂𧌎𧨩𧨸𧰊𧹖𨆁𨉾𨌈𨡁𨡏𨸱𨹻𩈍𩉁𩚣𪅉𪌩𪎾"},
    {U"tiao", U"调㟘㨄㬸㸠䄻䎄䑬䒒䖺䟭䠷䩦䯾䱔䳂佻儵咷啁姚嬥宨岧岹庣恌挑斢旫晀朓朷条桃條樤眺祒祧稠窕窱笤粜糶絩聎脁脩艞芀苕萔蓚蓧蓨蜩螩覜誂調超趒趠跳踔迢銚鋚鎥鞗頫髫鯈鰷鲦齠龆𠛪𠤺𠧪𡠊𡯿𡳏𢈄𢓝𢖈𢳙𢺫𣂁𣂥𣒼𣟐𣬸𥎺𥶏𦩄𦴚𦺰𧌁𨋫𨾾𩲤𪌪"},
    {U"tie", U"䑜䥫䩞䴴䵿僣占呫帖怗惵聑萜蛈蝶詀貼贴跕鉄鉆銕鋨鐡鐵铁飻餮驖鴩𢶋𤝓𦝒𦧢𦧤𪎋"},
    {U"ting", U"㓅㹶㼗䁎䅍䇸䋼䗴䦐䯕䱓䵺亭侱侹停厅厛听圢奠娗婷嵉庁庍庭廰廳廷忊挺朾桯梃楟榳汀涏渟濎烃烴烶珵珽町甼筳綎耓聤聴聼聽脡艇艼莛葶蜓蝏誔諪邒鋌閮霆鞓頲颋鼮𠄚𠕊𠘋𡈼𡔛𢬫𢽄𣂴𣄿𣉡𤗞𤘖𤱹𥆑𥑈𥥶𥫙𥴑𦉬𦐿𦕢𦗟𦝞𧓴𧖨𧰩𧶺𨁗𨉬𨊡𨓍𨳑𨳝𨸁𩆆𩐴𩑙𩒞𩨑𩹇𪊶𬘩"},
    {U"tong", U"㛚㠉㠽㣚㣠㤏㪌㮔㸗㼧㼿䂈䆚䆹䮵䳋䴀䶱仝佟侗偅僮勭同哃嗵囲垌峂峒峝庝彤恫恸恿慟憅捅晍曈朣桐桶樋橦氃洞浵湩潼炵烔熥燑爞犝狪獞痌痛眮瞳砼硐硧秱穜童筒筩粡絧統綂统膧艟茼蓪蚒蜼蟲衕詷赨通酮重鉖鉵銅铜餇鮦鲖鼕𠖄𡠙𡦜𢄟𢈉𢏕𢓘𢳟𣌾𣑸𣪯𣻢𤱇𤺄𥦁𥩌𥫂𥲆𦏆𦒍𦨴𧇌𧊚𧋒𧋚𧌝𧙥𧳆𧳿𨀜𨈹𨚯𨜳𨝯𨠌𩍅𩩅𩻡𪀭𪌢𫍣"},
    {U"tou", U"㓱㖣㡏㢏㪗㰯㳆㼥䕱䚵䞬䟝䱏䵉亠偷偸埱头妵婾媮愉投敨斢紏綉緰蘣褕諭諳透逗鋀鍮钭頭飳骰黈𡇧𡷠𣂮𣛾𣪌𦈕𧺢𨔙𨪐𨯲𨱎𨷩𩜶𩿢𪁞𪉘𪌘𪎨"},
    {U"tu", U"㒔㟮㭸㻌㻠㻬㻯䅷䔑䖘䛢䞮䠈䣄䣝䤅䩣䳜余兎兔凃凸吐唋啚図图圕圖圗土圡堍堗塗墿宊屠峹嵞嶀庩廜徒怢悇捈捸揬摕斁杜梌檡汢涂涋湥潳瑹痜瘏禿秃稌突筡腞腯荼莵菟葖蒤趃跌跿迌途酴釷鈯鋀鋵鍎钍馟駼鵌鵚鵵鶟鷋鷵鼵𠊲𠞀𠟶𠫓𠫮𠳶𠸂𠻬𡇩𡸂𡺴𢉜𢝀𢬳𣅝𣈥𣒇𣔻𣘻𣥳𣲱𤙛𤟪𤷿𥂋𥥛𥧣𥨜𥯝𥱻𦔅𦝬𦩤𧛗𧧶𧳌𨑒𨙭𨝛𨢬𨨷𨱄𨴩𩣮𩥽𩸃𩾅𪉍𪑏𬳿"},
    {U"tuan", U"㩛䊜䜝䝎䵊䵎䵯剬剸团団團塼墥嫥專彖慱抟揣摶敦槫檲湍湪漙煓猯畽疃痪磚税篿糰蓴褍褖貒鏄鱄鶉鷒鷻𡁴𢣎𢮛𣶣𤱝𧐕𧓘𧰄𧳩𧽢𨪒𩃘𩘯𩜵𩠊𩠹𪈋𪏖"},
    {U"tui", U"㞂㞜㟎㢂㢈㢑㥆㱣㳷㷟㾯㾼㾽㿉㿗䀃䅪䫋侻俀僓啍墤娧尵弚弟忒怢推橔焞煺税穨聉脮脱腿蓷藬蘈蛻蜕褪謉讉蹆蹪追退隤頹頺頽颓饋駾騩骽魋𠺙𡯵𡷜𢈹𢉭𢊮𢓇𢟴𢠮𤍐𤗴𤸉𥢢𥲣𥶐𦖦𦜄𧆸𧝋𧮓𨆨𨌴𨗞𨘃𨽟𩓬𩘺𩙬𩳕𬓼𬯎"},
    {U"tun", U"㖔㧷㩔㬿㹠㼊䀫吞吨吴呑啍噋囤坉屯庉忳憞敦旽暾朜氽汭沌涒炖焞燉畽窀純肫膯臀臋芚蜳褪豘豚軘逐錪霕飩饨魨鲀黗𠭿𡉒𢞋𢥽𣋄𣵞𤶕𥴫𥸵𦍓𦜯𦜴𦟓𦟙𧑒𧰭𨁇𨔡𨙲𨧐𨳘𨹙𩂄𩖤𩷵𪌋𪎴𪎶𪏆𪑒"},
    {U"tuo", U"㟎㨊㸰㸱㼠㾃䅜䍫䓕䜏䡐䪑䭾䰿䲊䴱乇他仛佗侂侻咃唾嘽坨堶妥媠嫷它岮庹彵惰托扡拓拕拖挩捝撱杔杝柁柝棁椭楕槖橐橢毤毻汑池沰沱沲涶牠狏砣砤碢磚税箨籜紽綏脫脱舄莌萚蘀蛇蟺袉袘袥訑託詑説讬跅跎踻軃迆迤迱酡鉈鋖阤陀陁隋飥饦馱馲駄駝駞騨驒驝驮驼鬌魄魠鮀鰖鱓鴕鵎鸵鼉鼍鼧𠈁𠰹𠴻𡇈𡐏𡛵𡩆𡹬𢄿𢏜𢑠𢓰𢩷𢩻𢸨𢼊𣗸𣟁𣟄𣮆𣵺𣵻𣶦𣷿𤗙𤝛𤣯𤤩𤬾𤱡𤱧𤹢𥓿𥞒𥩀𦑑𦚈𦚐𦝦𧔳𧕦𧜲𧣖𧤓𧦧𧦭𧧉𧿧𧿶𨁡𨂫𨈷𨉋𨒙𨞌𨠑𨲉𨹔𨺖𩃰𩃱𩅡𩉺𩎼𩟰𩢊𩢵𩧐𩱾𩿽𪌂𪘕𪘗𬶍"},
    {U"wa", U"㒝㧚㼘䅅䍪䎳䖯䚴䠚䨟䯉䵷佤凹劸咓哇唲啘嗗嗢姽娃娲媧屲帓徍挖搲攨汙洼溛漥瓦瓩瓲畖砙穵窊窐窪聉腽膃蛙袜襪譁譌邷靺鞋韈韤鮭黳鼃𠲎𠴺𠹁𡁌𡚟𡧗𢈈𣐎𣢉𣢚𤞇𤬦𤬿𤮰𤿗𥤺𥥟𥿉𦘍𦘵𦚩𦞭𦤙𦫪𧧊𨀄𨩶𩎟𩨚𩨾𩩤𩿺"},
    {U"wai", U"㖞㗏䠿䴜䶐咼喎外夞夭崴歪瀤竵顡𠨃𠰻𢱉𤟷𤤫𤷹𦘍𨂿𨈕𨵞𩔀𩕕𩰬𪉭𪑷"},
    {U"wan", U"㘤㜶㝴㸘㹉㽜㿸䅋䑱䖤䗕䘎䘼䛃䛷䝹䥑䨲䩊䯈䯛䳃万丸倇免刓剜卍卐唍园埦塆壪夗夘妧娩婉婠完宛岏帵弯彎忨惋惌抏挽捖捥掔晚晥晩晼朊杤梚椀槾汍涴湾潫澫灣烷玩琓琬畹皖盌睕瞣碗笂箢紈絻綄綩綰纨绾翫脕脘腕芄莞莧莬菀萖萬蔓薍蚖蜿蟃豌貦貫贃贎踠輐輓鄤鋄鋔鋺錽鎫關頑顽骫魭𠒢𠝪𠠪𠣉𡆅𡇿𡤶𡩄𡸥𢀗𢓃𢓆𢛙𢨔𢯫𢯲𢺯𣆛𣡩𣥃𤗍𤥙𤧩𤻆𥆶𥝄𥟶𥤸𥿎𦂔𦙵𦜐𦣾𦲯𦽞𧚇𧠆𧯡𧲦𧹗𧿙𨂺𨈊𨉝𨌔𨞼𨥧𨩯𨩵𨩻𩅦𩈬𩊁𩢄𩣵𩾞𪂦𪂧𪋅𪎛𪐬𪑉𬇕"},
    {U"wang", U"㑌㓁㲿㳹㴏䋄䋞䒽䤑䰣亡亾仼兦匡妄尢尣尩尪尫彺往徃徍忘忹惘抂方旺暀朚望朢枉棢汪瀇王琞皇盳網网罒罔莣菵蚟蛧蝄誷輞辋迋迬魍𠕿𡔞𡝝𡯁𡷢𢁶𢓯𢛛𢼟𣢫𣥊𣶈𣷪𤷀𥆚𥆜𥝍𥲠𥾼𦓋𦖉𦣦𦣩𦬣𦯌𧈿𧎕𧧄𧧜𧫢𨕿𨳠𩖩𩵭𪁘"},
    {U"wei", U"㕒㖐㙎㙔㙗㛱㞇㞑㟪㠕㣦㣲㥜㥨㦣㧡㧪㨊㬙㭏㮃㱬㷉㹻䃬䇻䈧䉠䊊䋿䍴䍷䑊䔺䗽䘙䙟䙿䜅䜜䝐䞔䠑䡺䣀䤥䥩䦱䧦䪋䪘䫋䬑䬿䭳䮹䲁䲊䴧䵋䵳为于亹伟伪位倭偉偎偽僞儰卫危厃叞味唩唯喂喡喴噲囗围圍圩堤墛壝委威娓媁媙媦寪尉尾屗峗峞崣崴嵔嵬嶶巋巍帏帷幃廆徫微恑惟愄愇慰懀捤捼揋揻撝撱斖暐有未机桅梶椲椳楲欈沇沩洈洧浘涠渨渭湋溈溦潍潙潿濊濰濻瀢炜為烓煀煒煟煨熨熭燰爲犚犩猗猚猥猬玮琟瑋璏瓗畏痏痿癐癓眭睢瞶硊硙碨磈磑立維緭緯縅纬维罻胃腲膸艉芛芟苇苿茟荱荽菋萎葦葨葳蒍蓶蔚蔿薇薳藯蘶蜲蜹蜼蝛蝟螱衛衞褽覣覹觿詴諉謂讆讏诿谓趡踒踓躗躛軎轊违逶違遺鄬醀錗鍏鍡鏏闈闱阢隇隈隗隹霨霺韋韑韙韡韦韪頠颹餧餵饖骩骪骫魏鮇鮠鮪鰃鰄鰖鲔鳂鳚𠄿𠆟𠙕𠥎𠳿𠹤𡂗𡇦𡔱𡚈𡶎𡷕𡼱𢉝𢊯𢍚𢙓𢣘𢯷𢲴𢸦𢼸𢾁𣄧𣄺𣈎𣍟𣨙𣩪𣫪𣲗𣽴𤀷𤁿𤛲𤜂𤡪𤸆𤺉𤻅𤼒𥅵𥉖𥊪𥌰𥒮𥧙𥯜𥯤𥶽𥿫𦇅𦈓𦓽𦝛𦞽𦠻𦢿𦩏𦩝𦩬𦪒𦳢𦾛𧄸𧍥𧍫𧐌𧒭𧔥𧕞𧚷𧛚𧝕𧞸𧟼𧢒𧢧𧤖𧲄𧲗𧲝𧳞𧳪𧴖𧸽𨃄𨖿𨗨𨚘𨝀𨟗𨠥𨢉𨪈𨱖𨴓𨵋𨸔𨻒𨾂𨿭𩀣𩀶𩁌𩆁𩋾𩎵𩏉𩏏𩏐𩏿𩑵𩗘𩗜𩜧𩟟𩠯𩤸𩨅𩲂𩲄𩲈𩴞𩵠𩹂𩹥𩹷𩼂𩼌𩽎𪂄𪑐𪑭𫇭𬀩𬱟𬶏"},
    {U"wen", U"㒚㖧㗃㝧㡈㨉㬈㮧㳷㼔䎹䎽䐇䘇䦟䰚免刎匁吻呅呚呡問塭妏娩彣忞忟愠抆揾搵文昧昷桽榅榲歾殁殟汶渂温溫炆煴玟珳瑥璺瘒瘟眼稳穏穩笏紊紋絻緼纹聞肳脕脗芠莬蕰藴蚉蚊螡蟁褞豱輼轀辒鎾閺閿闅闦问闻阌限雯鞰韞顐饂馼駇魰鰛鰮鳁鳼鴍鴖鼤𠐢𠯳𡁋𢾿𣜺𣶌𤛁𤡕𤵒𥁕𥃮𥦊𥧚𥬼𦖞𦝮𦟕𦦯𦮶𧖁𧚇𨆲𨜵𨟸𨶭𨸩𩢌𩥈𩭋𪉃𪉸𫘜"},
    {U"weng", U"㘢㙂㜲㮬㹙㺋䈵䐥䩺䱵勜嗡塕壅奣嵡攚暡滃瓮甕瞈罋翁聬蓊蕹螉鎓鶲鹟齆𠰈𡍻𡩥𡻐𤌏𥕀𦞡𦧅𧚐𧛹𨜺𨞑𩄘𩔚𩡓𩮬𩰎𬭩"},
    {U"wo", U"㠛㦱㧴㱧㹻䀑䁊䂺䇶䠎䩊䮸䰀仴倭偓卧咼唩喔嗌噁嚄堝夭婐婑媉媪幄我挝捰捼捾握撾擭斡杌枂楃沃涡涴涹渥渦濄濣焥猧瓁瘟瞃矆硪窝窩肟腛臒臥艧莴萵蒦薶蜗蝸踒踠雘馧齷龌龏𠪧𠷏𠿟𡁮𡎔𡑟𡖲𢦴𢫷𣁳𣂽𣇫𣚝𤆏𤉦𤡓𤻌𥄗𥑣𥟿𥪍𦤨𦯏𦰖𦳹𦷵𧤒𧥋𧶕𨁟𨌝𩈱𩐦𩟓𩣵𩭏𩭝𩮑𩷯𪁕𪁾𪎤𪑱"},
    {U"wong", U"𥥈𥦷"},
    {U"wu", U"㐅㐳㑄㒇㘬㚢㡔㬳㮧㵲㷻㹳㻍㽾䀛䁷䃖䉑䍢䎸䑁䒉䓊䖚䛩䜑䟼䡧䦍䦜䨁䫓䮏䳇䳱䵦乄乌五亡仡仵伆伍侉侮俉倵儛兀剭务務勿午卼吳吴吾呉呒呜唔啎喔嗚嘸噁圬坞埡堥塢墲奦妩娒娪娬婺嫵寤屋屼岉峿嵍嵨巫幠庑廡弙忢忤怃悞悟悮惡憮戊扜扝扤捂揾摀敄於旄无旿晤杅杇杌柮梧橆歍武毋母汙汚污沕洖洿浯渞渥溩潕烏焐無熃熓物牾玝珷珸瑦璑甒痦盓瞀瞴矹碔祦禑窏窹笏筽箼粅膴舞芜芴茣莁蕪蘁蜈蝥螐蟱誈誣誤譕诬误趶躌迕逜邬郚鄔釫鋈鋘鋙錻鎢钨铻阢陚隖雺雾霚霧霿靰騖骛鯃鰞鴮鵐鵡鶩鷡鹀鹉鹜鼯鼿齀齬𠒄𠘻𠛆𠞆𠥢𠯃𠵦𠼘𡈎𡈞𡬫𡯇𡵉𡷤𢁢𢃀𢄓𢋹𢑟𢓲𢗳𢘃𢙁𢜮𢝴𢨂𢩈𢫸𣟒𣨓𣬽𣯎𣲘𣺀𤆡𤕘𤣬𤭑𤵐𤸼𥁡𥎈𥎮𥏒𥒀𥕻𥟽𥭠𥲐𥾕𦆞𦌬𦎦𦥁𦨉𦨳𦬂𦶀𦷽𦼇𧆹𧈭𧎄𧎻𧐙𧑕𧨆𧨈𧰈𧳎𧴇𧺴𧽋𧿉𨂣𨑥𨖴𨡡𨧗𨨡𨲬𨶇𨼊𨿏𩄯𩒾𩓦𩗽𩝕𩝷𩠟𩳌𩵱𩶭𩺰𩻚𪄝𪑱"},
    {U"xi", U"㑶㓾㔒㕃㕧㗩㗭㘊㙾㚀㚛㛓㛫㛭㜎㜯㞒㠄㣟㤅㤸㦦㦻㨙㩉㩗㩦㪧㬛㭡㮩㯕㰥㰻㰿㱆㱇㱤㲸㴔㴧㶉㶼㸍㺣㽯㾙㾷㿇㿽䀌䁯䂀䈢䈪䊠䏩䏮䐅䐖䐼䒁䒊䓇䖒䖷䙎䙵䚙䚫䚷䚿䛊䛥䜁䟇䢄䣛䧍䧿䨛䨳䫣䬣䭒䮎䯜䲪䳶䵱䶋习係俙傒僖兮凞匸卌卤卥卻厀吚吸呬呰咥咦咭唏唽喜喺嘻噏嚊嚱囍塈墍壐夕奊奚娭媐媳嬆嬉屃屎屖屣屭嵇嵠嶍嶲巂巇希席徆徙徯忚忥怬怸恄恓息悉悕惁惜愾慀憘憙戏戱戲扱扸摡撕擊既昔晞晰晳暿曦杫析枲桸棲椞椺榽槢樨橀橲檄欪欯欷歖歙氣氥汐洒洗浠淅渓溪滊漇漝潝潟澙濕灑烯焁焈焟焬煕熂熄熈熙熹熺熻燍燨爔牺犀犔犠犧狶猎獻玺琋璽瓕瘜皙盻睎瞦矖矽硒碏磎磶礂禊禧稀稧穸窸粞糦系細綌緆縘縰繥繫纚细绤羛義羲習翕翖肸肹脅腊膝舃舄舾茜莃莔菥葈葸蒠蒵蓆蓰蔇蕮薂虒虩蜤蜥蜴蝷螅螇蟋蟢蠵衋袭裼褶襲西覀覡覤觋觹觽觿訢詑誒諰謑謚謵譆谿豀豨豯貕赥赩趇趘蹊蹝躧遟邜郄郋郗郤鄎酅醯釐釳釸鈒鈢鉨鉩銑錫錯鎴鏭鑴铣锡闟阋隙隟隰隵雟雭霫霼飁餏餙餼饩饻騱騽驨鬩鯑鰓鰼鱚鳛鵗鸂黖鼳鼷𠅤𠆱𠉢𠔃𠔍𠘕𠜗𠟊𠤴𠦌𠦜𠨚𠩺𠪙𠬬𠴭𠶨𠺒𡁱𡃢𡅕𡏛𡗞𡗳𡘐𡘡𡙋𡜧𡝧𡡒𡦎𡩤𡳚𡶯𡻎𢀊𢊚𢋼𢑧𢒩𢒲𢗴𢙅𢜣𢟪𢠹𢡁𢤋𢧽𢨟𢬾𢭁𢹍𣅾𣎮𣒃𣚔𣟵𣢁𣢂𣢇𣢍𣢎𣢑𣢓𣣉𣤢𣤳𣤴𣨗𣯪𣳦𣳬𤃪𤄎𤄬𤇴𤌷𤓔𤓚𤟧𤠓𤡡𤡬𤡯𤢀𤤱𤥒𤨐𤬕𤬘𤮆𤮙𤲺𤳥𤶈𤶰𤷡𤹊𤺊𥄖𥄛𥇛𥈜𥈻𥋁𥋟𥎃𥪦𥮬𥰝𥰥𥱵𥺚𥻥𥿭𦂅𦃝𦐠𦙝𦜱𦞝𦞽𦠪𦤈𦩭𦪿𦮐𦱓𦷲𦸚𦼗𧀬𧂙𧈅𧈈𧈍𧈼𧉁𧋐𧎵𧐔𧚃𧣩𧤟𧥤𧦁𧦭𧧹𧪢𧬈𧬊𧯈𧯊𧯗𧱲𧲘𧳐𧶖𧹨𧹶𧹽𧺨𧻶𧿅𧿝𨀙𨋦𨐛𨛳𨜐𨝫𨞘𨡂𨭎𨮪𨰿𨳛𨳿𨵎𨷘𨻁𨻥𨿛𩅖𩊿𩍆𩎉𩎥𩒽𩗊𩗱𩛹𩦇𩭡𩲁𩽨𩾼𩿛𪃼𪄛𪄶𪅲𪓷𪕯𪖥𫍰𫘬𬭳𬶮"},
    {U"xia", U"㔠㗇㗿㘡㙈㙤㛍㦆㮫㰨㰰㰺㽠㾎䒠䖎䖖䘥䛅䞩䠍䦖䪗䫗䯟丅下乤侠俠假傄匣叚吓呀呷呼哧唬嗄嗑嚇圷埉夏夓夾岈峡峽廈徦懗押捾搳敮斜昰暇柙梺欱歃毳浹炠烚煆狎狭狹珨瑕疜疨瘕睱瞎硖硤碬磍祫笚筪給縀縖罅翈舝舺芐葭蕸虲虾蝦螛諕謑谺赮轄辖遐郃鍜鎋鎼鏬閕閜陜陿霞颬騢魻鰕鶷黠𠀼𠖹𠢆𠩘𠽫𡈮𡏘𡨄𡺷𢈙𢈤𢑓𢗄𢘉𢚌𢝅𢩹𢻗𣢗𣹱𤗭𤙇𤟝𤪆𤪍𥁆𥯾𥰶𥻴𦦕𦵯𦾏𧆥𧇍𧈄𧔂𧕱𧦎𧪕𧪹𧫒𧯋𨩽𨲑𨳉𨻲𨽯𩄗𩉾𩎲𩏓𩐀𩝛𩮂𪄂𪗾𪘁𪘘"},
    {U"xian", U"㔾㘅㘋㛍㛾㜷㡉㡾㢺㦑㦓㦥㧥㪇㫫㬎㬗㭠㭹㮭㯗㰊㰹㲔㳄㳭㵪㶍㷿㸝㺌㺤㽉㾾㿅㿌䁂䂅䃱䃸䄳䆎䉯䉳䊱䍎䏹䐄䕔䕭䗾䘆䙹䚚䜢䝨䢭䢾䤼䥪䦘䦥䧋䧟䧮䨘䨷䩂䩙䭑䯭䯹䱤䲗䵇䵌䵤䶟䶢仙仚伣伭佡俔僊僩僲僴先冼县咁咞咸哯唌啣嗛嘕垷埳堿壏奾妗妶姍姭姺娊娨娴娹婱嫌嫺嫻嬐孅宪寰尟尠屳岘峴崄嶮幰廯弦彡忺慊慳憪憲憸懢挦捍掀探揱搚搟撊撏攇攕显晛暹杴枮梘槏橌櫶欦毨氙洒洗涀涎湺溓澖濂瀗灑灦烍燹狝猃献獫獮獻玁现玹珗現甉痫癇癎盷省県睍瞯瞷矣硍碱礆礥祆禒禰秈稴筅筧箲籼粯糮絃絤綅綖綫線縣縿繊纎纖纤线缐羡羨羬肩胘脅腺膁臔臤臽舷苋苮莧莶薟藓藖蘚蘞蚬蚿蛝蜆衔衘褼襳見誢誸諴譀譣豏賢贒贤赻跣跹蹮躚軐軒輱酰醎醶釤銑銛銜鋧錎錟鍁鍌鏾鑦铦锨閑閒闞闲限陥险陷険險霰韅韯韱顈顕顯餡饀馅馦鮮鰔鱻鲜鶱鷳鷴鷼鹇鹹麙麲黹鼸𠊏𠏓𠏡𠓌𠚆𠛑𠜎𠠁𠤕𠫄𠯟𠷢𠿢𡐖𡒓𡗏𡞣𡫹𡰲𡸃𡾮𡿤𢁗𢅮𢋮𢎙𢐐𢒆𢕖𢕭𢖋𢖎𢖝𢚀𢛆𢥌𢫿𢮂𢷑𢹚𣆕𣊺𣑹𣔙𣕎𣞘𣟲𣢟𣭡𣮾𣰷𤁦𤈷𤉌𤑃𤓤𤞤𤟅𤡥𤼂𥑻𥓒𥙆𥜦𥜬𥜲𥟕𥦶𥬍𥰳𥲋𥻇𥻧𥽏𦋈𦎵𦑘𦒜𦠹𦢙𦥖𦧐𦩢𦭶𦱁𦸊𦽭𧂞𧈁𧕇𧖙𧗹𧠒𧫹𧱀𧸂𧻒𧼏𧾨𨁅𨇤𨍒𨏥𨐊𨖱𨘙𨘞𨙡𨚾𨵬𨸄𨺘𩆵𩈖𩏩𩝈𩤊𩤥𩤦𩦂𩦹𩧩𩨡𩱆𩱊𩶤𪂶𪄏𪄷𪎉𪔩𪾢𫍯𬀪𬸣"},
    {U"xiang", U"㐮㗽㟄㟟䄈䇨䊑䐟䔗䖮䜶䢽䦳䬕䴂乡亨享亯佭傢像儴勨勷厢向响啌啍嚮塂姠嶑巷庠廂忀想攘晑曏栙楿樣橡欀洋湘潒珦瓖瓨皀相祥稥箱絴緗纕缃缿羏翔膷舡芗萫葙薌蘘蚃蟓蠁衖襄襐詳详象跭迒郷鄉鄊鄕銄銗鐌鑲镶閧闂降響項项飨餉饗饟饷香驤骧鬨鮝鯗鱌鱜鱶鲞鴹麘𠸮𡨇𡹷𢄵𢛖𢞡𢠷𢪷𢭎𣂝𣅰𣨳𣻍𤉪𤍀𤖽𤝷𤩪𤭬𤰅𤷼𥀾𥊾𥗵𥣟𥥩𥫖𥿧𦍲𦍴𦎈𦕺𦺣𦺨𧖿𧬰𨀘𨉽𨖶𨙵𨛜𨧑𨷄𨷿𩑇𩝾𩞃𩞥𩡄𩡌𩡠𩾬𪂼𬙋"},
    {U"xiao", U"㔅㕺㗛㚠㚣㤊㩋㪣㬵㮁㲖㵿㹲㺒䉰䊥䌃䎄䐹䒕䒝䕧䟁䥵䨭䬒䬘䳂䳋䴛佼侾俏俲傚削効叟号呺呼咲咻哓哨哮唬啋啸嗃嘋嘐嘨嘮嘯嘵嚣嚻囂奡姣婋孝宯宵小崤庨彇恔恷憢捎揱搜撓撨效敩斅斆晓暁曉枭枵校梟梢橚櫹歊歒歗殽毊洨消涍淆漻潇潚澩瀟灱灲烋焇熇熽燆爻狡猇獟獢痚痟皛皢睄硝硣穘窙笑筊筱筿箫箾篠簘簫絞綃縿绡翛肖胶脩膮芍茭莦萧萷蕭薂藃虈虓蛸蟂蟏蟰蠨訤詨誟誵謏謞謼譊踃較轇逍郩銷销霄颵騷驍驕骁骹髇髐魈鴞鴵鵁鷍鷕鸮𠈬𠏕𠑪𠴡𠴳𠹎𡟣𡣾𡥍𡦝𡦳𡧕𡯩𡱉𡷸𡼚𢓮𢛘𢪶𢭦𢸳𢹳𢽾𣂬𣏠𣔷𣕇𣟇𣠎𣤡𣱓𣹝𣿣𤑳𤕢𤞚𤟞𤠖𤡔𤣌𤣠𤷤𤺃𤽳𤿨𥆔𥔑𥕾𥽁𥾤𦏷𦐺𦟞𦢩𦦛𦯪𦱜𦺔𧄤𧍂𧡼𧢬𧩮𧮻𧱐𧳍𧵱𨅋𨊅𨠦𨴹𨶅𩋍𩙚𩙮𩧓𩫂𩫳𩱴𩵖𩵮𩾒𩾓𩾾𪁎𪊷𪛀𫍲"},
    {U"xie", U"㐖㒠㓔㔎㕐㖑㖿㗨㙝㙦㙰㝍㞒㞕㡜㢵㣯㣰㥟㦪㨙㨝㩉㩦㩪㭨㰔㰡㱔㳦㳿㴬㴮㴽㸉㽊㾚䀘䁋䁯䉏䉣䊝䋶䓳䔑䕈䕵䙊䙎䙝䙽䚸䝱䡡䢡䥱䥾䦏䦖䩤䩧䪥䬅䲒䵦些亵伳偕偞偰僁儶写冩劦勰协協卨卸叶吤唏喈嗋噧垥塮夑夾奊契娎媟孈寫屑屓屟屧峫嶰廨徢恊愶慀懈拹挟挾接揳搚携摺撷擕擷攜斜旪暬枻桔梋械楔榍榝榭槷檞欸歇歙殺汁泄泻洩渫湝溉滊潰澥瀉瀣灺炧炨烲焎熁燮燲爕猲獦獬瑎眭碿祄禼糏紲絏絜絬綊緤緳繲纈绁缬缷翓耶胁脅脇脋膎苴薢薤藛蝎蝑蝢蟹蠍蠏血衺裌褉褻襭觟解諜諧謝譮讗谐谢豫跬躞躠迦邂邪鍱隰鞋鞢鞵韰頁頡骱鬹魼鮭齂齘齛齥龤𠅱𠑄𠖹𠗉𠢲𠨆𠲊𠸴𠿇𡀺𡃂𡄕𡗼𡛶𡞘𡟩𡣹𡤋𡭥𡰢𡸔𡽖𢂐𢌀𢓬𢖆𢗊𢜨𢞜𢤯𢤰𢥘𢬿𢯉𢴲𢹒𢿡𣆟𣒄𣞐𣣩𣣲𣣶𣫴𣬕𣹩𣻠𣽒𤑪𤗈𤙒𤞡𤡧𤢺𤣑𤫉𤮯𤱷𤺎𥀺𥆥𥇱𥊯𥌨𥍆𥍋𥎎𥗧𥝖𥢹𦁛𦋅𦎈𦔼𦖐𦚡𦚫𦞚𦩌𦪬𦳃𦵱𧀢𧀺𧌊𧌋𧌖𧍁𧏂𧏃𧐃𧑦𧓂𧓺𧖁𧛼𧜔𧜵𧝫𧟃𧥤𧭠𧭸𧳧𧷑𧷧𨁂𨄌𨇨𨈙𨏳𨤴𨧥𨫔𨳚𨵚𨵪𨷥𨼬𨾗𩂪𩃖𩋘𩋧𩍝𩎃𩐁𩐉𩙜𩤠𩫲𩰳𩷂𩺫𩽍𪆋𪍛𪕯𪙥𬹼"},
    {U"xin", U"㐰㔤㖕㚯㛙㛛㜦㣺㭄㭡㭢㽎㾙䅽䒖䚱䛥䛨䜗䜣伈伩信俽噷噺囟妡姰嬜孞尋庍廞心忄忻惞愖憖撢新昕杺枔橝欣款歆炘焮盺礥脪興舋芯莘薪衅襑訢訫軐辛邤釁鈊鋅鐔鑫锌阠顖馨馫馸鬵𠑰𠔋𠷓𡈏𡌜𢋆𢗀𢠝𢩲𢭧𣂗𣂜𣃄𣥇𤙖𤙣𤜢𤣲𤫨𤴾𤷓𤹩𦁍𦉝𦜓𦞤𦢯𦤟𦰸𧗹𧳄𧴢𨊳𨓇𩒦𩔙𩖣𩖰𩟍𩾽𩿃𪀽𫷷𬒘"},
    {U"xing", U"㐩㓑㓝㙚㚔㝭㣜㨘㷣㼛㼬䁄䂔䃏䓷䕟䗌䛭䣆䤯䰢䳙侀倖兴刑哘坓型垶姓娙婞嫈嬹巠幸形性悻惺擤星曐杏洐涬滎煋熒狌猩瑆皨省睲研硎箵篂緈胜腥臖興荇荥莕蛵行裄觪觲謃邢郉醒鈃鉶銒鋞钘铏陉陘餳騂骍鮏鯹𠀦𠬋𡃳𡶭𡷨𢙼𢜫𣢝𣨾𣸝𤏽𤙡𤬐𤶲𥠀𥨕𦂅𦈒𦈨𦈵𦖤𦢯𦩠𧊞𧊽𧌚𧗦𧛟𨌍𨞾𩈡𩩋𫰛"},
    {U"xiong", U"㐫㚾䠗䧺兄兇凶匂匈哅夐宪忷恟敻昫汹洶焸焽熊胷胸能芎訩詗詾讻诇賯赨雄𠓙𡨳𡪰𡬁𢢹𢿌𣅷𤔫𤛪𥃴𥥧𥦥𦈤𦓈𦙄𦬺𦵡𧘮𧞞𧰯𧵣𧽒𧿖𨥍𩌠𩧊𩴂"},
    {U"xiu", U"㗜㱗㱙㳜㵻㹋㽲㾋䏫䐰䗛䡭休俢修咻嗅嚊宿岫峀庥朽樇櫹溴滫潃烋烌煦珛琇璓秀糔綇綉繍繡绣羞脙脩臭臹苬茠莠蓨螑袖褎褏貅銝銹鎀鏅鏥鏽锈飍饈馐髤髹鮴鱃鵂鸺齅𡔨𡜨𡟞𡯐𢊒𢓵𢕦𢩨𣧬𤚯𥌪𥝠𥞼𦈋𦟤𦪋𧌌𧙏𩘭𩛢𩡎𩢮𩭘𩮄𪀪𪁮𪕦𪘆"},
    {U"xu", U"㐨㑔㑯㕛㖅㖪㗵㘧㚜㜅㜿㞊㞰㡏㥠㦌㰭㰲㳚㵰㷦㺷㽳䂆䅡䇓䈝䋶䍱䎉䏏䔓䘏䙒䛙䢕䣱䣴䦗䦽䧁䬄䬎䱛䱬䳳于伃休伵余侐俆偦冔勖勗卹叙吁呴呼咻喣嘔嘘嘼噓圩垿墟壻妶姁姐婿媭嬃嶼幁序徐怴怵恓恤惐慉戌掝揟敍敘旭旮旴昫晇暊朂朐栩楈槒欨欰欻歔殈汿沀洫淢湑溆滀滸漵潊烅烼煦獝珝珬畜疞盢盨盱眗瞁瞲矞砉稰稸窢糈絮続緒緖緰縃繻續绪续聓聟肷胥芋芧蒣蓄蓲蓿蕦藇藚虗虚虛蚼蛡蝑裇規訏許訹詡諝諿謣謳譃许诩谞賉邪鄦酗醑鉏鉥銊鑐雩需須頊须顼馘驉鬚魆魖魣鱮𠆐𠜄𠧰𠯜𠷙𠹘𠾫𡦁𡱣𡹲𢄼𢖳𢨁𢨰𢩕𣅤𣆒𣊞𣚏𣢊𣧵𣨤𣰃𣸃𤆞𤇳𤚉𤟠𤡣𤡶𤬱𤭽𤲸𤴙𤷇𤸀𥄵𥅺𥆛𥇏𥇿𥈈𥊊𥍟𥎕𥎗𥕰𥚩𥮪𥳗𦄼𦅏𦈡𦑍𦓔𦔰𦕓𦘼𦜃𦝳𦠷𦪡𦯅𦰰𦰲𦲰𧁃𧆜𧆡𧊥𧏺𧕼𧙆𧟬𧧓𧪮𧶍𧹭𧹴𧼑𨂠𨅑𨋾𨌎𨍐𨜿𨞣𨣦𨬗𨴎𨵮𨷔𨹘𨻍𨼋𩂉𩌮𩌲𩍳𩑕𩒇𩒧𩓣𩔴𩔼𩖕𩝔𩠋𩣊𩪉𩰠𩽆𩾊𪆛𪖩𪙫𫚈𬣙"},
    {U"xuan", U"㓩㔯㔵㘣㝁㦏㦥㧋㧦㩊㯀㳙㳬㹡㻹㻽㾌䀏䁔䁢䃠䆭䍗䍻䗠䘩䚙䚭䚯䛹䝮䠣䡓䡣䧎䩙䩰䮄䲂䲻䳦䴉䴋亘儇券吅咺喛喧塇夐妶姰媗嫙嬛宣弲怰悬愃愋懁懸揈揎撰擐旋昍昕昡晅暄暅暖暶梋楥楦檈泫洵涓渲滋漩澴炫烜煇煊煖狟玄玆玹琁琄瑄璇璿瓊痃癣癬盤眩眴睻矎碹禤箮絃絢絹縇縣縼繏绚翧翾萱萲蓒蔙蕿藼蘐蜁蜎蝖蠉衒袨諠諼譞讂谖贙軒轩选選還鉉鋗鍹鏇鐶铉镟鞙顈颴饌駨駽鰚𠗻𠣖𠤰𠥞𠵷𡈣𡈴𡬳𡾥𢂄𢆯𢈋𢏧𢙂𢰊𢳄𢿌𣉖𣎓𣟳𤂿𤟿𥌭𥥾𥶷𦈝𦌔𦐽𦑙𦛔𧉎𧐗𧑩𧔤𧜽𧟨𧡚𧡢𧤎𧭦𧸾𧾆𧾎𧾩𨀮𨁁𨊼𨏂𨹆𩃚𩉥𩋢𩋫𩋱𩑹𩕖𩕪𩘒𩙢𩙽𩟁𩤡𪍧𫍽𫓶𫠊"},
    {U"xue", U"㔃㔢㕰㖸㗾㞽㡜㦜㨹㰒㶅㷤㻡㿱䀗䆝䆷䋉䎀䒸䛎䤕䦑䨮䫼䬂䭥䱑乴削吷哮噱嚯坹壆学學岤峃嶨怴斅斈桖樰決泧泬泶滈澩瀥炔燢狘疦疶瞲矆穴膤艝茓蒆薛血袕觷謔谑趐踅蹻轌辥辪雤雪靴鞾韡鱈鳕鷽鸴𢪎𢯳𢼺𣎛𣧌𣧡𣧵𣪨𣺭𤀰𤇳𥀣𥄎𥄒𥄴𥅧𦐍𦥯𦰾𧉢𧋍𧔗𧮞𧸗𨂂𨑣𨒲𨨺𨭁𩋾𩌊𩖱𩖶𩶫𪃅"},
    {U"xun", U"㖊㜄㡄㢲㨚㰬㵌㽦䀏䋸䖲䗼䘩䙉䛜䞊䠝䡅䭀䵫伨侚偱勋勛勲勳卂咰噀噚嚑坃埙塤壎壦奞姰孫寻尋峋巡巺巽廵徇循恂悛愻揗撏攳旬曛杊栒桪梭樳殉殾毥汛洒洵浔浚潠潭潯濬灥焄煇熏燂燅燖燻爋爓狥狻獯珣璕畃眴矄稄窨筍篔紃絢纁臐荀荨葷蔒蕁蕈薫薰蘍蟫蟳訊訓訙詢训讯询賐迅迿逊逡遁遜郇鄩醺鑂鑫顨馴駨驯鱏鱘鲟鶽𠊫𠹀𡑎𡺕𡿼𢏤𣌨𣖼𣹯𤃺𤑕𤛧𤟤𤿟𥒘𥙣𥳍𥾡𦅀𦅑𦘶𦠅𦫯𦳣𧂇𧥿𧰣𧸩𧾝𧾠𧾩𨀴𨺮𨼔𩊻𩖰𩠇𩪱𩷰𩾄𩾧𪀠𪀽𪇑𫄸𬊈𬍤𬘓𬩽"},
    {U"ya", U"㗇㝞㤉㧎㰳㳌㴫㹞㾎㾏㿿䀴䀹䃁䄰䅉䆘䝟䢝䦪䪵䫔䰲䵝丫乛亚亜亞伢俹劜厊压厑厓厭吖吾呀呾哑唖啞圔圠圧垭埡堊堐壓姶娅婭孲岈崕崖庌庘御押拁挜掗揠札枒桠椏椻歇氩氬浥涯漄潝烏牙犽猚猰玡琊瑘疋疨痖瘂睚砑碣磍稏穵窫笌聐芽蕥蚜衙襾訝讶軋輅輵轧迓邪釾錏鐚铔閘雅顔鴉鴨鵪鵶鸦鸭齖齾𠄮𠋗𠜲𠮜𠵣𡇼𡴭𡶦𡷻𡸗𡹄𡿖𢛄𢛟𢮊𣉩𣏎𤘅𤘆𤴓𤵭𤷅𤸱𤹎𥇠𥌚𥏝𥐕𥒧𦉟𦉧𦜖𧈝𧓪𧡋𧧝𧬬𨁶𨓴𨖭𨨙𨸺𨹝𩃐𩑔𩖁𩨠𩭯𩮝𩸋𩿔𪆰𪖐𪗹𪘬𪘲"},
    {U"yan", U"㕣㖶㗴㘖㘙㚧㛪㜝㝚㢂㢛㤿㥼㦔㫃㫟㬫㭺㮒㰽㳂㵪㶄㷔㷳㷼㸶㺂㿕㿼䀋䀽䁙䂩䂴䄋䅖䅧䇾䉷䊙䌪䍾䎦䑍䓂䕾䖗䗎䗡䗺䛳䜩䞁䞛䢥䢭䣍䤷䦲䨄䪜䫡䮗䲓䳛䳡䳺䴏䶫䶮严乵但俨俺偃偐偣傿儼兖兗剡剦匽厂厃厌厣厭厳厴咽唁唌啱喭噞嚥嚴囐埏埯堰塩墕壛壧夵奄妍妟姲姸娫娮媕嫣嬊嬐嬮嬿孍宴屵岩崦嵃嵒嵓嶖嶮巌巖巗巘巚巡广庵延弇彥彦恹愝懕懨戭扊抁挻捝掞掩揅揜揞敥昖晏晻暥曕曣曮棪椻椼楌樮橪檐檿櫩欕殗殷氤汧沇沿洇洝涎淊淡淫淹渰渷湮溎滟演漹灎灔灧灩炎炏烟烻焉焑焔焰焱煙熖燄燕爓牪狠狿猒珚琂琰瓛甗癌盐眼研砚硏硯硽碞礹筵篶簷綖縯罨羡羬胭腌膁臙艳艶艷芫莚菴菸萒葕蔅蔫薟虤蜒蝘衍裺褗覃覎觃觾言訁訮詽諺讌讞讠谚谳豓豔豣贋贗赝趼躽軅這遃郔郾鄢酀酓酽醃醶醼釅鉛鋋錟閆閹閻閼闫阉阎阭隁隒險雁靨顏顑顔顩颜餍饜騐験騴驗驠验鬳魇魘鰋鳫鳱鴈鴳鶠鷃鷰鹽麙麣麲黡黤黫黬黭黰黶鼴鼹齗齞齴龑𠆲𠉬𠍛𠘥𠛭𠝢𠰖𠻤𡙶𡚇𡣽𡹶𡹼𡻡𢅠𢇈𢇘𢈂𢉘𢌨𢔂𢜰𢤍𢯼𢸴𢾑𣃧𣃳𣃾𣄉𣄑𣄝𣚕𣝎𣡕𣡞𣡶𣥡𣩙𣭻𣼞𣼠𤂠𤅊𤅸𤎄𤒦𤖝𤗎𤜵𤟇𤟟𤡖𤡥𤢋𤫣𤬝𤯇𤯐𤲩𤸹𥀬𥂁𥃩𥃿𥍓𥍻𥕼𥜒𥣘𥤟𥤴𥦍𥧑𥯃𥴿𥶿𥷀𦁏𦁙𦌚𦎣𦏥𦏹𦑎𦖈𦖧𦛞𦛣𦝪𦧡𦫤𧇱𧉮𧊔𧍢𧎘𧞣𧠦𧥜𧩅𧬌𧴣𧹬𧺅𧻃𧽉𧽞𧾤𨀅𨁍𨁹𨂪𨃰𨆁𨒄𨟹𨠭𨡄𨡎𨡣𨣻𨤎𨪶𨴣𨶁𨷽𨸮𨺥𨻂𨻳𨽑𩃀𩇠𩈯𩒖𩗷𩜽𩣲𩩄𩩴𩩶𩪴𩳢𩸞𩻖𩾝𩾸𪁡𪂈𪐘𪑈𪑸𪒝𪒠𪗙𪗤𪙊𪩘𫄧𬙂𬸘"},
    {U"yang", U"㒕㔦㟅㦹㨾㬕㺊㿮䁑䄃䍩䑆䒋䖹䬗䬺䭐䱀䵮仰佒佯傟养劷勜卬咉坱垟央姎婸將岟崵崸徉怏恙愓慃懩扬抰揚攁敭旸昂昜映暘杨柍样楊楧様樣歍殃氜氧氱泱洋湯漾潒瀁炀炴烊煬珜瑒疡痒瘍癢眏眻礢禓秧紻羊羏羕羪胦英蛘蝆詇詳諹軮輰鉠鍚鐊钖阦阳陽雵霙霷鞅颺飏養駚鰑鴦鴹鸉鸯𠍵𠢴𠮴𡠘𡡂𡩶𡹕𢏙𢟣𢵇𢽕𣃝𣉚𣐫𣗹𤞢𤢐𤢮𤸡𥂸𥃽𥒞𥠜𥥵𥬴𥳜𦍕𦍲𦍹𦏱𦭵𦯒𦴊𦼴𧓲𧥴𧫛𧲱𧵌𨋕𨋽𨎔𨒫𨖌𨦡𨱝𨹈𨽐𩊑𩋬𩤟𩧫𩲴𩴨𪓛𪕫𪚻"},
    {U"yao", U"㑸㑾㔽㙘㝔㞁㟱㢓㨱㫍㫏㫐㴭㵸㹓㿑㿢䁏䁘䂚䆗䆙䆞䋂䌁䌊䌛䑬䔄䖴䙅䚺䚻䛂䠛䢣䬙䯚䳩䴠䶧么仸佻倄偠傜僥匋吆咬喓嗂嚙垚堯夭妖姚婹媱嬈宎尧尭岆峣崤崾嶢嶤幺幼徭徼怮恌愮抭揄揺搖摇摿撽暚曜杳枖柼楆榚榣樂殀殽洮淫溔滧瀹烄烑熎燿爻狕猶猺獟玅珧瑤瑶由眑矅磘祅穾窅窈窑窔窯窰筄箹約繇纅耀肴腰舀艞苭药葯葽蓔蕘薬藥蘨袎要覞訞詏謠謡讑谣趯踰軺轺遙遥邀邎銚鎐鑰钥闄陶隃靿顤颻飖餆餚騕驁鰩鳐鴁鴢鷂鷕鹞鼼齩𠌠𠍩𠏈𠑐𠕻𠟋𠢩𠣑𠹑𡆩𡔜𡛙𡝛𡝩𡢹𡣠𡨇𡩸𡶂𡺯𢂊𢅹𢆷𢆽𢈆𢊙𢋇𢑈𢝍𢺇𣣳𣨘𤂼𤄶𤒝𤚭𤫺𤬔𤬖𤺃𤾫𥁒𥃧𥃺𥌺𥤣𥤹𥦖𥪯𥬓𥹱𦆸𦇬𦔷𦡱𦤋𦥝𦦌𦾺𦾾𧄎𧇠𧍔𧠽𧢢𧤮𧷋𧽎𨍳𨓳𨘔𨱧𨹋𩋃𩑗𩑴𩜸𩢒𩥣𩨴𩩼𩬗𩯛𩲻𩳔𩿕𪐯𪖐"},
    {U"ye", U"㖡㖶㖿㗼㙒㜇㡋㥷㧉㩎㪑㱉㱌㸣䁆䅖䇩䈎䊦䋵䎨䓉䔑䡾䢡䤳䤶䥟䥡䥺䧨䭇䭎䭟䮜䱒䲜业也亪亱倻偞僷冶叶吔咽啘喝嘢噎嚈埜堨墅墷壄夜射峫嶪嶫懕抴拽捓捙掖揞揲揶擖擛擨擪擫斜晔暍曄曅曗曳曵枒枼枽椰楪業歋殕殗洂洇涂液漜潱澲烨焆煠熀燁爗爷爺璍瓛痷皣瞱瞸礏窫緤耶聶腋荼葉虵蠮蠱謁谒邪邺鄓鄴野釶釾鋣鍱鎁鎑鐷铘靥靨頁页餘餣饁饐馌驜鵺鸈黦𠀸𠄅𠟪𠥇𠱝𡀽𡁁𡑀𡛌𡛽𡽣𢀘𢉥𢢜𢪧𢱴𣃳𣎩𣐂𣚋𣚕𣩫𣩯𣰛𤑷𤝇𤝉𤝱𤳪𥀬𥌅𥠍𥮧𥯘𦀕𦂡𦕆𦠜𦤪𦰳𧌄𧎭𧏽𧐓𧒐𧔦𧗖𨂒𨈺𨉅𨶮𨸌𨼥𨽀𩉂𩐱𩑃𩘏𩜺𩱝𩸾𩼋𩼴𪋫𪍅𪑦𪒲"},
    {U"yi", U"㐌㐹㑊㑜㑥㓷㔴㕈㖂㘁㘈㘊㙠㙪㙯㚤㚦㚶㛄㛕㛳㜋㜒㝖㝣㞔㠖㠯㡫㡼㢊㢞㣇㣻㥋㥴㦉㦤㦾㫊㮛㰘㰝㰻㱅㱞㱲㲼㳑㳖㴁㴒㴔㵝㵧㵩㶠㹫㹭㺿㼢㽈㾨䁺䃜䄁䄩䄬䄿䆿䇩䇵䇼䈕䉗䉝䉨䋚䋵䌻䎈䏌䑄䒾䓃䓈䓹䔟䔬䔱䕍䖁䖊䖌䗑䗟䗷䘝䘸䚷䝘䝝䝯䞅䡾䢃䣡䣧䦴䧅䧇䧧䩟䪰䫑䬁䬥䬮䭂䭇䭞䭲䭿䮊䯆䰙䰯䱌䱛䲑䳀䴊䴬䴰䵝一丿乁乂义乊乙也亄亦亿仡以仪伇伊伿佁佗佚佾侇依俋倚偯儀億儗兿冝刈劓劮勚勩匇匜医印厭叕台叹吚听呓呭呹咦咿唈喦嗌噎噫囈圛圪圯坄坨垼埶埸墿壱壹夁夕失夷奇奕妷姨姬媐嫕嫛嬄嬑嬟孴它宐宜宧寱寲射尾屹峄峓崎崺嶧嶬嶷已巳巸帟帠幆庡廙异弈弋弌弬彛彜彝彞彵役忆忔怈怠怡怿恞悒悘悥意憶懌懿戲戺扅扆扡抑拸挹掎掜揖搋搤撎擇攺敡敼斁施旑旖易昳晹暆曀曎杙杝枍枻柂栘栧栺桋棭椅椬椸榏槷槸樴檍檥檹欥欭欹歖歝殔殪殹毅毉汽沂沶泄泆洢洩洫浂浥浳渏渫湙溢漪潩澤澺瀷炈焉焬焱焲熙熠熤熪熼燚燡燱犄狋狏猗獈玴珆瑿瓵畩異疑疙疫痍痬瘗瘞瘱癔益眙睪瞖矣硛硪礒礙祎禕秇移稦穓竩笖箷簃籎紲絏維綺縊繄繶繹绎缢羛羠羡義羿翊翌翳翼耛耴肄肊胰膉臆舣艗艤艺艾芅苅苡苢荑萓萟蓺薏藙藝蘙虉蚁蛇蛜蛡蛦蛾蜴螔螘螠蟻衣衤衪衵袂袘袣裔裛裿褘褹襗襼觺訑訲訳詍詑詒詣誃誒誼謚謻譩譯議譺讉讛议译诒诣谊豙豛豷貖貤貽賹贀贻跇跠踦軼輗輢轙轶辥辷迆迤迭迱迻逘逸遗遺邑郼酏醫醳醷釋釔釴釶鈘鈠鉇鉈鉯銕銥錡鎰鏔鐿钀钇铱镒镱阣阤陁陭隶隿雉霅霬靉靾頉頤頥顊顗颐食飴饐饴駅驛驿骮鮧鮨鯣鳦鴺鶂鶃鶍鷁鷊鷖鷧鷾鸃鹝鹢鹥黓黝黟黳齮齸𠂆𠄱𠅌𠈶𠍫𠍳𠏩𠐀𠓋𠗺𠚮𠛃𠜁𠡔𠡝𠤕𠤗𠤘𠥦𠨾𠩗𠩫𠪗𠮙𠯋𠰄𠲔𠲖𠲚𠲺𠲻𠶷𠼪𠽜𠿣𡄵𡄻𡉛𡊁𡊶𡍡𡙘𡜬𡥁𡬐𡬓𡰥𡱐𡷪𡻣𡼎𡾾𢀁𢂒𢂗𢂼𢄅𢇙𢇚𢇸𢈶𢊘𢍰𢎀𢎃𢎉𢏗𢓀𢓡𢕷𢖅𢖫𢖴𢖺𢗎𢘽𢙇𢞉𢡃𢣉𢦕𢨮𢨳𢩮𢩼𢱁𢷔𣎅𣐓𣐵𣐿𣕁𣘦𣙛𣚘𣡊𣢭𣢷𣤪𣦌𣧄𣨟𣫙𣶫𣷩𣸘𣿉𤆾𤇴𤈙𤑹𤖪𤘊𤝳𤝻𤣨𤣮𤤺𤥿𤧕𤬩𤴧𤶛𤷅𤸸𤻂𤼌𥃠𥃸𥄻𥄿𥅓𥌟𥍟𥍴𥏜𥑴𥒵𥘒𥘠𥙁𥙇𥜃𥜥𥟘𥡪𥢧𥥌𥥴𥩖𥫃𥫜𥫝𥰧𥱃𥸊𥹋𥾐𥿫𥿹𦌩𦎝𦏸𦓻𦔜𦔥𦔫𦘳𦙨𦚟𦟧𦠉𦡫𦥱𦨇𦭥𦮸𦶂𧃟𧅖𧆦𧈻𧉅𧉮𧊣𧊤𧋏𧐔𧑌𧓗𧔮𧙡𧜤𧡇𧢂𧣟𧣬𧦧𧩹𧫦𧬇𧮒𧱊𧱏𧳁𧷅𧷥𧺎𧺝𧺢𧺿𧾰𨋯𨛯𨜶𨜽𨠑𨠶𨣠𨣬𨦯𨱁𨳷𨹝𨻊𨻏𨽹𨾓𩂒𩂹𩈭𩋌𩍖𩎭𩎷𩓧𩔦𩕲𩖹𩖾𩗑𩘧𩚂𩚇𩛆𩛮𩞾𩟉𩠂𩡣𩣞𩤒𩥯𩧭𩪟𩪣𩮵𩳇𩴜𩴮𩷍𩷘𩸨𩼨𩾘𩾠𩾢𪀓𪀕𪁚𪁛𪈨𪎈𪐔𪐘𪐣𪒕𪕶𪗷𪘃𪘬𪘲𪙴𫄷𫍙𫍟𫖮𬟁𬬩𬺈"},
    {U"yin", U"㐆㐺㒚㕂㖗㖶㙬㝙㞤㡥㣧㥯㥼㦩㧈㧢㪦㮒㱃㴈㶏㶣㸒㹜㹞䃌䄄䅧䇙䌥䏖䒞䒡䓄䓰䕃䕾䖐䖜䚿䜾䡛䤃䤺䨸䪩䰼䲟䴦乑乚伒众侌傿冘凐印吟听吲唫喑噖噾嚚囙因圁圻垔垠垦垽堙堷壹夤姻婣婬寅尹峾崟崯嶾币廕廴引愔慇慭憖憗懚斦朄栶梀檃檭檼櫽欭欽歅殥殷氤沂泿洇洕淫淾湚湛湮溵滛潭潯濥濦烎烟犾狺猌玪珢璌瘖瘾癊癮硍碒磤禋秵窨筃粌絪緸縯胤芩苂茚茵荫荶蒑蔩蔭蘟蚓螾蟫裀言訔訚訡訢誾諲讔赺趛輑鄞酓酳釿鈏鈝銀銦铟银闇闉阥阴陰陻隂隐隠隱霒霠霪靷鞇音韾飮飲饮駰骃鮣鷣齗齦龂𠃊𠖟𠪚𠮜𠽨𡇂𡈲𡋪𡐔𡓓𡓿𡖣𡩘𡸛𡼽𢂨𢉩𢋻𢌲𢓕𢓙𢛦𢝯𢪪𢳃𢷍𣓆𣔸𣘴𣦫𣱐𣱜𣸊𣽮𤂹𤎭𤝎𤢦𤯸𤵯𤷏𤻘𥖵𥤷𥬜𥮍𦈑𦈠𦜲𦝴𦟘𦾻𧊭𧍢𧥸𧦸𧦹𧩬𨈧𨋙𨏈𨐐𨒦𨓮𨛊𨟏𨟴𨡢𨢂𨢎𨦆𩂢𩂥𩃬𩐞𩖄𩚕𩬵𪔰𪔽𪘎𪙤𪙾𪛊𫡑𬘡𬤇𬮱"},
    {U"ying", U"㑞㕲㡕㢍㨕㯋㲟㵬㶈㹚㹵㿘䀴䁐䁝䃷䇦䊔䋼䑉䑍䓨䕦䙬䚆䣐䤝䤰䦫䧹䨍䩕䪯䬬䭊䭗䭘䴍䵴俓偀僌哽啨営嘤噟嚶塋夃央婴媖媵嫈嬰嬴孆孾嵤巆巊应廮影応愥應摬撄攍攖旲映景暎朠柍桜桯梬楹樱櫻櫿泂浧渶溁溋滎滢潁潆濙濚濴瀅瀛瀠瀯瀴灐灜焸煐熒營珱瑛瑩璎瓔甇甖甸瘿癭盁盈眏矨硬碤礯禜穎籝籯緓縈繩纓绬缨罂罃罌耺膡膺英茔荧莖莹莺萤营萦萾蓥藀蘡蛍蝇蝧蝿螢蠅蠳褮覮謍譍譻賏贏赢軈迎逞郢鍈鎣鐛鑍锳霙鞕韹韺頴颍颕颖鱦鴬鶑鶧鶯鷪鷹鸎鸚鹦鹰𠊶𠝟𠠜𠮳𠸄𡀘𡁊𡂚𡃅𡄖𡎘𡢘𡺡𢄋𢣙𢥏𣋋𣟤𣤵𣲜𤁽𤇾𤌌𤜉𤟣𤣎𤭫𤹜𤹥𥌽𥌾𥍼𥏎𥐑𥚿𦔃𦖽𦝚𦡺𦢆𦦿𦩩𧅋𧓀𧕄𧕍𧢛𧭓𧮆𧯀𨍞𨜏𨟙𨠸𨪄𨵛𩂝𩄪𩋹𩖍𩘑𩳍𩸥𩹅𩻷𪊵"},
    {U"yo", U"哟唷喲嚛"},
    {U"yong", U"㐯㙲㛚㜉㝘㞲㟾㦶㦷㴄㴩㶲㷏㻾㽫䗤䗸䞻䧡佣俑傛傭勇勈咏喁嗈噰埇塎墉壅嫞容嵱庸廱彮怺恿悀惥愑愹慂慵拥揘擁柡栐槦永泳涌湧滽澭灉牅用甬痈癕癰砽硧禜筩臃臾苚蕹蛹詠踊踴遇邕郺鄘醟銿鏞镛雍雝顒颙飬饔鯒鰫鱅鲬鳙鷛𠆌𠳀𠷸𠹍𡵜𡶤𢀍𢢓𢧳𣋿𣏀𤛑𥁎𥑿𥥝𥧱𦃽𦤘𦨤𦨬𧖇𧗴𧙇𧝸𧲤𧴄𧴗𧺸𧻹𨓨𨤂𨦡𨴭𨶽𩆄𩍓𩔔𩜳𩟀𩟷𩤛𩫏𩬮𪄉𪅟𪇛𪪝"},
    {U"you", U"㒡㓜㕗㕱㗀㘥㚭㛜㤑㫍㮋㰶㱊㳊㳛㳺㴗㶭㹨㺠㽕㾞䀁䅎䆜䍃䑻䒴䖻䚃䚻䛻䞥䢊䢟䥳䬀䱂䳑丣亴优佑侑偤優冘卣又友右叹呦哊唀嚘囿坳奥妋姷孧宥尢尤峟峳幼幽庮忧怞怣怮悠憂懮戭扰揂揄攸斿有朓柚栯梄梎楢槱櫌櫾汓汼沋油泅泑浟游湵滺瀀牖牗牰犹狖猶猷獶由甴疣痏祐禉秞糿繇纋羐羑羗耰聈聱肬脜脩苃莜莠莤莸蒏蕕蚘蚰蚴蜏蝣蝤褎訧誘诱貁輏輶迶逌逰遊邮郵鄾酉酭釉鈾銪銹铀铕駀魷鮋鯈鱿鲉麀黝鼬𠖋𠘳𠢢𠧠𠧪𠧴𠨦𠮫𡇀𡈙𡈰𡈵𡊧𡋧𡜳𡯉𡯙𡺒𡺖𢆶𢈓𢋣𢓿𢖟𢟅𢪥𢿚𣁨𣅄𣅺𣏞𣓐𣓛𣢄𣢜𣣜𣣸𣤎𣤡𣧗𣧥𣸠𣿤𤄘𤍕𤘜𤣙𤤧𤤬𤪎𤱎𤴨𤸈𥙾𥜚𥝘𥣯𥯞𥴕𥽟𦎓𦏇𦑸𦚥𦥣𦩲𦳧𦳩𦳷𦵵𦷿𦸙𧀥𧅲𧆕𧆘𧍘𧙏𧠶𧡹𧰰𧳫𨑫𨗰𨘁𨘵𨙂𨛕𨡴𨴜𨸙𩈏𩑣𩗚𩘈𩘓𩜷𩤹𩥘𩲎𩴑𩴙𩹊𩼈𩽇𩿬𬨎"},
    {U"yu", U"㑨㒁㒜㔱㙑㚜㚥㝢㝼㠘㠨㡰㢏㣃㤢㤤㥔㥚㥥㦛㦽㧒㪀㬂㬰㰲㲾㳛㳼㵄㶛㷒㺄㺞㺮㻀㼌㼶㽣䁌䁩䂊䂛䃋䄏䄨䆰䈅䉛䋖䋭䍂䍞䏸䐳䓊䔡䖇䗨䘘䘱䘻䛕䜡䜽䞕䞝䢓䢖䢩䣁䣿䤋䥏䨒䨞䩒䩱䩽䫻䬑䬔䮇䮙䰥䰻䱷䲣䴁䵥䵫丂与乻予于亏亐伃伛余俁俞俣俼偊傴僪儥兪匬吁吳吾唷唹喁喅喐喩喻噊噢噳圄圉圩圫域堉堣堬奥妤妪娛娪娯娱媀媮嫗嬩宇宛寓寙尉屿峪峿崛崳嵎嵛嶎嶼庽庾彧御忬悆悇惌惐愈愉愚慾懊懙或戫扜扵拗挧捓捥揄敔斔斞於旕旟昙昱杅栩栯桙梧棛棜棫楀楡楰榆櫲欎欝欤欥欲歈歟歶毓毹氀汙汩浴淢淤淯渔渝湡滪漁潏澚澞澦澳灪灹焴煜煨熨燏燠爩牏狱狳獄獝玉王玗玙琙琟瑀瑜璵畬畭痏瘀瘉瘐癒盂盓睮矞砡硢硲礇礖礜祤禦禹禺秗稢稶穥穻窬窳竽箊篽籅籞籲粥紆緎緰繘纡罭羭羽聿肀育腧腴臾舁舆與舒艅艈芋芌苑茟茰菀菸萭萮萸蒮蓣蓹蔚蕍蕷薁藇蘌蘛虞虶蜍蜟蜮蝓蝺螸蟈衘衙衧袬裕褕覦觎誉語諛諭謣譽语谀谕谷豫貍貐貗踰軉輍輿轝込迂迃逳逾遇遹邘郁郚鄅酑醧釪鈺銉鋊鋙錥鍝鐍鐭钰閼閾阈陓隃隅隩雓雨雩霱預頨顒预飫餘饇饫馀馭騟驈驭骬髃鬰鬱鬻魊魚魣鮽鯲鰅鱊鱮鱼鳿鴥鴧鴪鵒鷠鷸鸆鸒鹆鹬麌齬齵龉龥𠀛𠊏𠋟𠎳𠏚𠕦𠧇𠫣𠱐𠱙𠴫𠸹𠽵𡁎𡂊𡇺𡈨𡋬𡑾𡒃𡒊𡔴𡨣𡨿𡫽𡬊𡬞𡷎𡻢𡿥𡿯𢊧𢋅𢌻𢎻𢒰𢔢𢔥𢔬𢔴𢖡𢛨𢡎𢮁𢯮𢹏𢺴𢾄𣄊𣋉𣍛𣕃𣝑𣟰𣡉𣢒𣢦𣣎𣨝𣩺𤀝𤕘𤗃𤚎𤜹𤞞𤥽𤧙𤳕𤸒𤹪𥄵𥆉𥉑𥎐𥒾𥔢𥘄𥙿𥛩𥝍𥝨𥯮𥷔𥷞𥸤𥸪𥹔𥺞𦀡𦈣𦈸𦋢𦋯𦎘𦏜𦏻𦐛𦒑𦡭𦥉𦦩𦦫𦦲𦩞𦭳𦱀𦱂𦱃𦳅𧃠𧈯𧉣𧊠𧍪𧐄𧑐𧗪𧗱𧙶𧞏𧫊𧯱𧰇𧱬𧶠𧼫𧼯𧾚𧿷𨄯𨉗𨊱𨖛𨗝𨜖𨝈𨞓𨨶𨩬𨪎𨮔𨰸𨵉𨵦𨶢𨾌𩂧𩃯𩈕𩊇𩋉𩋤𩎹𩏟𩏴𩘤𩘳𩘻𩚄𩛪𩛭𩝗𩟑𩟳𩡃𩢶𩤺𩥭𩦡𩦢𩨈𩨗𩨙𩩑𩩘𩰪𩱌𩱱𩲾𩵎𩺰𪁀𪂉𪂕𪂵𪃍𪃎𪇝𪉐𪊻𪋉𪋬𪋮𪑆𪑌𪑝𪓊"},
    {U"yuan", U"㘣㟶㠾㤪㥐㥳㭇㱧㹉㾓䅈䈠䏍䖠䛄䛇䡝䥉䦾䨊䩩䬇䬧䬼䱲䲮䲻䳒䳣傆允元円冤剈原厡厵员咽員喛噮囦园圆圎園圓圜垣垸塬夗妧妴媛媴嫄嫚嬽宛寃弲怨悁惌愿捐掾援杬棩楥榞榬橼櫞沅涓涴淵渁渆渊渕湲源溒灁焆爰猨猿獂瑗畹盶眢禐穿笎箢緣縁缘羱肙芫苑茒葾蒝蒬薗薳蚖蜎蜵蝝蝯螈衏袁裫裷褑褤謜貟贠輐轅辕远逺遠邍邧酛鈨鋺鎱阮院隕願駌騵魭鳶鴛鵷鶢鶰鸢鸳鹓黿鼋鼘鼝𠒜𠛭𠝳𠩠𡈒𡈓𡢊𡣬𡯱𡷡𢂱𢆀𢍈𢏮𢐄𢕋𢗯𢱽𢷻𣹠𤬌𥭞𥰟𥿎𦍼𦿂𧉗𧔞𧙮𧳭𧻚𨀮𨓯𨕗𨖳𨸘𨻣𩉯𩌑𩍻𩐘𩔃𩕾𩘍𩛟𩝸𩟁𩰵𩾢𪀝𪂦𪄁𪔅𪔗𪔙𪕀𫘪"},
    {U"yue", U"㜧㜰㬦㰛㹊䆕䆢䋐䋤䖃䟑䟠䠯䡇䢁䢲䤦䥃䶳刖哾噦囝块妁妜嬳岄岳嶽彟彠恱悅悦戉扚抈捳擽曰曱月枂樂樾櫟汋瀹焆爍爚玥矆矱礿禴箹篗籆籥籰粤粵約约臒蘥蚎蚏蜕蠖説越趯跀跃躍躒軏鈅鉞鋭鑠鑰钺閱閲阅髺鸑鸙黦龠𠏃𠔠𠨲𠩉𠪶𠯲𠾲𠿋𡆦𡆽𡛟𡡕𢁞𢦰𢯵𢾔𣌗𣎱𣐋𣤰𣦏𣨡𣻮𤑓𤓝𥆟𥩡𥸘𦋩𦘹𦚢𦣜𦤕𧀲𧅚𧇓𧕋𧤯𧤽𧨄𧯡𧹊𨁑𨈋𨊸𨒋𨙄𨳕𨵃𨷲𨸀𨸎𨿁𩁯𩎙𩓥𩚈𩜌𩱪𩱲𩿠𪁑𪒥𪘳𫐄𬸚"},
    {U"yun", U"㚃㚺㛣㜏㞌㟦㩈䆬䇖䉙䚋䞫䡝䢵䤞䨶䩵䪳䲰䵴云伝傊允勻匀員喗囩均夽奫妘媪孕宛尉尹怨恽惲愠愪慍抎抣昀晕暈枟榅橒殒殞氲氳沄涒涢温溳澐煇煴煾熅熉熨狁玧畇瘟盾眃磒秐筍筠筼篔紜緷緼縕縜繧纭缊耘耺腪芸苑荺菀蒀蒕蒷蕓蕰蕴薀藴蘊蜵蝹褞賱贇赟輼运運郓郧鄆鄖酝醖醞鈗鋆阭陨隕雲霣韗韞韫韵韻頵餫馧馻齫齳𠈤𠚓𠣐𠱳𡅙𡖒𡢘𡲪𡽅𣂊𣍯𣖆𤈶𤶧𤸫𥐩𥠺𥬀𦅿𦈉𧉃𧡡𧥺𧥼𧬞𧶊𧼐𨍆𨛡𨷐𨸮𩁴𩂿𩋢𩏅𩏆𩴉𪉂𪍝𪏔𪏚𪘩𫖳"},
    {U"za", U"㞉㦫䆘䕹䞙䣠䨿䪞偺匝咂咋咱啈啐啑喒嘁噈囃囋囐帀扎拶杂沞沯灒砸磼籴紥紮臜臢襍迊鉔雑雜雥韴魳𠂝𠛱𠯗𠱥𠶿𠷿𠽷𡁕𢶍𢹼𢽜𣗁𣤶𣤷𣤺𣴖𣸐𤄔𤄖𤠀𥷩𦠛𦾬𧄽𧌃𧬩𧾁𨠿𨤃𩇺𩞶𩯳𩵢𪚇"},
    {U"zai", U"㱰䏁䔂䣬䮨䵧仔傤儎再哉在宰崽才扗栽洅渽溨災灾烖甾睵縡菑賳載载酨𠎶𡉄𡿧𢎋𢦏𢦒𣅃𣔮𣪮𤆄𤌊𤝖𤞳𦞁𦳦𧯥𨀬𨙴𨚵𩛥𩛳"},
    {U"zan", U"㔆㜺㟛㣅㨻㭮㳫䍼䐶䟅䬤䭕儧儹兂兓咱喒噆囋寁拶揝撍攅攒攢昝暂暫桚涔淺湔濺濽灒瓉瓒瓚禶穳篸簪簮糌臢襸讃讚賛贊赞趱趲蹔鄼酇錾鏨鏩鐕鐟饡𠼗𡡖𢄤𣤶𣸄𥎑𥜙𥳋𥸢𧄽𨖋𨘄𨙏𩛻𩯒𩯳"},
    {U"zang", U"㘸㮜匨塟奘弉戕牂牫羘脏臓臟臧葬蔵藏賍賘贓贜赃銺駔驡驵髒𡁧𡅆𢈜𢍈𢍿𣻟𤃼𤛻𤞛𦟃𧕨𨌄𪓅"},
    {U"zao", U"㡟㯥㯾㷮䖣䗢䜊䥣䲃傮凿唕唣喿噪慥早枣栆梍棗槽澡灶煰燥璅璪皁皂窖竃竈簉糟繅繰艁草薻藻蚤謲譟趮蹧躁造遭醩鑿𠙬𠴵𡌣𡐋𡨗𢄀𢑖𢲵𢵥𣍖𣩒𣴢𤍜𤞋𤟀𤩨𥀛𥖨𦯑𦵩𧈹𧎮𧻰𨎮𨐉𨒽𨚰𨠷𨤇𪙡"},
    {U"ze", U"㖽㚖㟙㣱㤞㥽㳁㳻㺓䇥䔼䕉䕪䟄䯔䰹䶡䶦仄伬側则則咋唶啧嘖夨嫧崱帻幘庂廁択择捑措擇昃昗柞樍歵汄沢泎泽溭澤灂皟睪瞔矠礋稄稷笮箦簀耫舴葃蔶蘀蠌襗諎謫謮責賾责赜迮飵鰂鸅齚齰𠟻𠨻𠷌𡵗𡸈𡸦𡹨𢍰𢧠𢮚𢯩𣆽𣛸𣤈𣬿𣼦𣽎𣿐𤖓𤢟𤾀𥍱𥎍𥟔𥼃𦔈𦟜𧶷𨕠𨖊𩂖𩄾𩌪𩔳𩾸𪌟"},
    {U"zei", U"戝蠈賊贼鯽鰂鱡鲗𢨗𦽒𧒿𨆎"},
    {U"zen", U"㻸䫈僭囎怎撍譖譛谮"},
    {U"zeng", U"㽪䎖䙢䰝䵴増增憎曾橧熷璔甑矰磳綜縡繒缯罾譄贈赠鄫鋥锃鬷鱛𡡑𡾽𤎯𦀓𦼏𧢐𨲯𪒟𪙭"},
    {U"zha", U"㒀㔍㗬㞚㡸㦋㪥㱜㳐㴙㷢㾴䀹䃎䄍䆛䈟䋏䋾䐒䕢䖳䙄䛽䟻䥷䮜䮢䱹䵙䶥乍偞偧册剳劄厏吒咋咤哆哳喋喥喳囃奓宱怍扎扠抯拃挓插揸搩搾摣擖札柞柤查柵査栅楂榨樝渣渫溠潳灹炸煠牐甴痄皶皻眨砟笮箑箚紮耫膪苲苴蔖藸蚱蚻蜡觰詐諎謯譇譗诈踷蹅軋醡鍘铡閘闸霅鞢鮓鮺鰈鲊鲝齄齇齟齰𠍹𠓣𠝚𠝝𠢙𠢡𠭯𠯩𠰏𠽣𡎫𡗸𡟢𢄄𢕮𢧖𢧻𣛽𣟦𣧖𣹧𣽛𤁳𤈩𤡨𤰦𤵦𤹡𥀈𥀉𥝾𥡧𥬰𥰭𥹁𦂉𦑯𦝥𦟰𦳏𧄠𧉫𧧻𧨊𧨿𧩫𧬅𧶇𧼰𧼶𧽅𧿌𨂵𨅓𨋘𨡗𨢦𩃡𩃹𩥠𩩥𩬟𩮎𩳶𩶱𩻢𩼫𩽽𩿤𪗭𪗵"},
    {U"zhai", U"㒀㡯㩟㾹䍉䐱䔝亝债側債厇厏啇嚌夈宅寨度抧捚摘擇擿斋斎柴榸檡牴疵瘵砦祭窄簀粂翟膪責鉙駘骴齊齋𠏰𠑞𠞶𠫸𠵠𠷒𡅓𡍥𢋿𢯌𢴨𣩭𤞮𤢒𤻦𥍪𥞅𥰾𦑱𦤧𧲻𧻍𨅪𨝋𩏪𩏽𩝿𩬫𩱳𪀥𪑽𪗒𪗓𪘇𪘨𪚎"},
    {U"zhan", U"㔊㜊㞡㟞㟻㠭㣶㮵㺘㻵䁪䁴䆄䋎䎒䏼䗃䘺䟋䡀䦓䧯䩅䩆䩇䪌䬤䱠䱳䱼䳻䶨亶佔偡儃占單噡嫸孱展崭嵁嶃嶄嶘嶦怗惉战戦戰拃搌撣斩斬旃旜枬栈栴桟棧椾榐橏欃毡氈氊沾湔湛澶琖皽盏盞瞻碊站粘綻绽菚薝蘸虥虦蛅袒襢覱詀詹謙譧譫讝谵趈跕蹍躔輚輾轏邅醆醮閚霑顫颭飐飦餰饘驏驙魙鱣鳣鳽鸇鹯點黵𠌲𠟉𠟧𡁳𡅹𡓦𡕁𡭞𡽻𢅺𢈽𢤚𢧗𢵈𣀁𣛷𣢤𣮿𣳤𤖆𤘇𤜇𥇢𥙡𥮒𥴐𥶕𥿜𦈻𦗢𦧚𦪣𧀡𧂁𧋱𧎰𧒝𧔡𧖉𧙭𧝑𧬆𧮪𧮺𧲮𧸪𧽆𧾍𨇩𨊈𨣁𨣚𨫀𨭖𨺿𨼈𨼮𩆯𩉗𩔣𩕊𩛻𩥇𩨍𩰃𩼼𪃋𪆲𪉜𪏉𪗦𪘪𪡏𫗴"},
    {U"zhang", U"㙊㙣㽴䛫䩨丈仉仗傽墇嫜嶂帐帳幛幥张弡張彰慞扙掌暲杖樟涨涱漲漳獐璋痮瘬瘴瞕礃章粀粻胀脹蔁蟑賬账遧鄣鏱長长障鞝餦騿鱆麞𠅹𠫝𡈠𡑄𡚹𢕎𢕔𢩰𢪾𢷢𣌞𣾦𤍤𤓯𤕄𥇔𥳶𦺡𧐊𧽣𨄰𩌬𩭫𪅂𫗠"},
    {U"zhao", U"㑿㕚㡽㨄㷖㷹㹿䃍䈃䈇䍜䍮䑲䜈䝖䞴䰫佋佻兆召啁啅嘲垗妱巶找招旐昭晁曌朝枛桃棹櫂沼淖濯炤照燳爪爫狣瑵皽盄着瞾窼笊箌罀罩羄肁肇肈菬著蚤詔诏赵趙釗釽鉊鍣钊駋鮡鳭鼂𠕖𠕭𠟅𠠄𠻥𡖎𡱜𢁬𢗈𢡰𣊧𣋍𣠜𤍒𤑗𤙔𤿘𥏨𥢔𥵤𥷮𦗔𦬔𦹫𦺓𧳝𧳻𨱻𨹸𩘀𩙩𬬿𬶐"},
    {U"zhe", U"㙷㝂㞏㡇㢎㪿㭙㭯㯙㯰㰅㸙㸞䁋䁤䂞䇽䊞䎲䏳䐑䐲䓆䖳䗪䜆䝃䝕䠦䩾䮰䵭乇乽仛厇哲啠啫喆嗻嘀嚞囁埑堵嫬庶悊慴慹扸折摺攝斥晢晣杔柘棏樀樜歽浙淛潪着矺砓磔禇籷粍者耷聑聶著蔗虴蛰蜇蟄蟅袩褚褶襵詟謫謶謺讁讋谪赭軼輒輙轍辄辙这這適遮銸鍺锗陬馲鮿鷓鷙鹧𠌮𠚱𠝚𠝝𠞃𠯓𠽻𠾀𡂭𡄡𡇠𡘭𡜯𡝊𢟯𢢍𢫰𢬴𣇧𣙵𣠞𣶋𣻩𤜤𤟍𤮱𥏯𥐽𥑡𥕣𥛧𥤋𥧮𥭙𦅄𦔮𦗑𦗗𦞥𦠟𦠠𦠣𦣀𦬃𦯍𧀹𧎴𧑧𧤠𧲢𨅊𨐃𨰵𨵊𩊵𩐶𩢐𩣩𩤜𪁊𪐏𪚥"},
    {U"zhei", U"這"},
    {U"zhen", U"㐱㓄㖘㘰㣀㪛㮳㯢㱽㲀㴨㼉䀕䂦䂧䃌䈯䊶䏖䑐䝩䟴䠴䡩䨯䪴䪾䫃䫬䲴䳲侦侲偵唇圳坫塦填姫嫃寊屒帪弫慎戡抮挋振揕搸敶斟昣朕枕枮栕栚桢桭椹楨榐榛槇樼殝沴浈湞溱滇潧澵獉珍珎瑧瑱甄甽畛疹眕眞真眹砧碪祯禎禛稹竧箴籈紖紾絼縝縥纼缜聄胗臻萙葴蒖蓁薽蜄袗裖診誫謓诊貞賑贞赈趁軫轃轸辴遉酖酙針鈂鉁鋴錱鍖鍼鎭鎮针镇阵陣陳震靕駗鬒鮝鱵鴆鸩黮黰黱鼎𠘱𠛶𠠹𠬓𠵧𠸸𡇑𡇖𡈿𡻈𢏈𣃵𣏖𣒅𣓀𣘘𣦵𣬻𣱽𣿎𤚨𤷌𥅘𥌃𥖘𥤤𥪘𥿜𦁟𦳳𦸮𧠝𧤛𧮬𨌑𨏤𨱅𨳌𨸬𩄛𩇜𩊡𩊨𩑘𩒀𩒈𩒪𩬖𩾺𪁧𪇳𪉕𪐲𪑳"},
    {U"zheng", U"㡠㡧㬹㱏㽀䂻䆸䇰䈣䋊䋫䍵䛫䟓䡕䥌䥭䦛䦶䱢丁丞争佂倀偵凧埥埩塣奠姃媜峥崝崢嶒帧幀征徎徰徴徵怔愸憕承抍拯挣掙掟揁撜政敞整晸朾正氶浧炡烝爭狰猙町症癥眐睁睜瞠禎筝箏篜糽綪聇脀蒸証諍證证诤貞趟踭郑鄭鉦錚钲铮鬇鮏鯖鴊𠏫𠑅𠔻𠲜𡈼𡪺𢁿𢌦𢏰𢓞𢮐𢹑𢾧𤪡𤸲𤿆𥊼𥒛𦓺𦙫𦚦𦜎𦡅𦱊𧗆𧗲𧘿𧪣𧯫𧶄𨀧𨋬𨌢𨚣𨛰𨜓𨟃𨢹𨧭𨺟𩏠𩗲𩗵𩘼𩘽𩚫𩺄𪎻"},
    {U"zhi", U"㕄㗌㗧㘉㙷㛿㜱㜼㝂㡳㡶㣥㥀㧗㨁㨖㩼㫑㮹㯄㲍㲛㲳㴛㴯㸟㽻㿃䃽䄺䅩䆈䇛䇧䇽䉅䉜䌤䎺䏄䏯䐈䐭䑇䓋䓌䓜䓡䕌䖨䘣䘭䚦䚳䛊䛗䝰䝷䞃䞠䟈䟡䡹䣽䤠䥍䦯䧝䧴䩢䬹䭁䱃䱥䲀䳅䵂䵹之乿伎侄俧倁値值偫傂儨凪制剬劕劧卮厎厔只吱呮咥咫嗭嚔址坁坧垁埃埴執墆墌夂妷姪娡媞嬂寘實峙崻巵帋帙帜幟庢庤廌彘徏徔徝徵志忮怾恃恉慹憄懥懫戠执扺扻抧抵拓挃指挚捗掷搘搱摕摨摭摯擲擳擿支斦旘旨昵晊晢智杝杫枝枳柣栀栉栺桎梔梽植椥楖榰樀樲樴櫍櫛止歭殖氏氐汁汥汦沚治泜洔洷淔淽滍滞滯漐潌潪瀄炙熫犆狾猘璏瓆瓡畤疐疷疻痔痣瘈直眰知砋砥礩示祁祇祉祑祗祬禃禔秇秓秖秩秪积秲秷稙稚稺穉窒筫紙紩絷絺綕緻縶織纸织置翐耆聀职職肢胑胝胵脂膣膱至致臷臸芖芝芷茋茝菭薙藢蘵虒蚔蛭蜘螲蟙衹衼袟袠製襧覟觗觝觯觶訨誌識豑豒豸貭質贄质贽趾跂跖跱踬踯踶蹛蹠蹢躑躓軄軹軽輊轵轾迣遟遲郅酈酯釞鉄銍銴鋕鑕铚锧阤阯陁陟隲隻雉馶馽駤騭騺驇骘鯯鳩鳷鴙鴲鶨鷙鸷黹鼅𠊤𠊷𠋤𠍜𠓶𠘖𠚅𠟅𠦧𠮡𠰅𠼠𡀹𡁉𡂒𡂣𡈊𡈼𡌴𡍶𡏀𡏚𡑘𡖧𡖻𡙑𡠗𡠹𡮞𡰹𡱔𡸜𡽆𢃜𢄢𢄱𢅁𢇨𢊁𢍧𢎈𢐂𢕞𢖇𢖿𢙺𢚨𢛍𢡒𢦕𢧤𢯶𢰙𢴠𢴧𢴨𢷸𢻙𢽃𢽗𢾫𣔐𣖌𣖭𣖿𣗻𣚠𣥰𣨋𣱐𣲵𣳀𣽚𤃲𤆒𤓕𤖞𤛱𤞂𤞌𤦄𤦮𤧜𤴛𤴟𤴢𤵋𤶓𤸓𤽁𤿙𥃫𥇕𥇭𥍭𥎹𥏄𥏅𥏊𥏰𥏷𥒓𥒗𥔊𥘡𥙆𥝑𥝮𥠈𥠽𥣮𥭡𥮖𥴒𥹩𥻬𥾣𥿮𦃘𦏤𦐖𦛧𦜋𦝔𦟔𦤻𦥎𦥏𦥐𦭜𦭮𦯫𦯯𦰘𦳮𦴀𧀿𧃐𧊙𧌔𧏸𧐉𧓸𧙁𧛢𧜚𧝉𧠫𧠴𧣭𧣾𧤡𧨰𧩼𧫡𧱒𧸅𧸕𧸲𧹛𧼴𧽦𧾂𨁷𨂂𨃯𨆧𨈙𨌌𨎉𨎌𨑨𨒉𨕕𨖋𨖹𨙸𨜎𨟊𨟾𨡐𨢮𨤱𨧵𨫔𨬚𨰛𨵂𨻆𨼓𨿛𩅀𩊝𩊴𩋩𩍲𩍵𩙾𩧄𩬺𩯈𩷓𩹈𩻼𪁊𪁓𪁩𪂅𪉆𪏀𪐏𪑜𪒊𪗨𪗻𪙹𫛛𬃊"},
    {U"zhong", U"㐺㣫㲁㲴㳊㹣㼿䇗䈺䝦䱰中乑仲伀众偅冢刣喠堹塚塜夂妐妕媑尰幒彸徸忠忪柊橦歱汷泈潼炂煄狆瘇盅眾祌种種穜童筗籦終緟终肿腫舂舯茽董蔠蚛蚣蝩螤螽蟲衆衳衶衷褈諥踵蹱重鈆鈡銿鍾鐘钟锺鴤鼨𠊥𠛀𠱧𡖌𡥿𡰒𡻑𢁷𢃭𢝆𢨱𣱧𣷡𣹞𤚏𤝅𤯚𤺄𥗦𥷈𥻝𦉂𦌋𦔉𦬕𧆼𧑆𧬤𧳮𨉢𨳗𩅞𩅧𩾋𩿀𬑔"},
    {U"zhou", U"㑇㑳㗙㛩㣙㤘㥮㨄㫶㼙㾭㿒䈙䋓䎇䎻䏔䐢䑼䓟䖞䛆䧓䩜䶇伷侏侜倜僽冑周呪咒咮啁啄喌喙噣嚋妯婤宙州帚徟扭掫昼晝晭柚椆注洀洲淍炿烐珘甃疛皱皺盩睭矪碡祝箒籀籒籕粙粥紂紬縐繇纣绉翢肘育胄胕舟舳荮菷葤薵詋詶調諏謅譸诌诪賙赒軸輈輖轴辀逐週郮酎鈾銂霌駎駲騆騶驟骤鬻鯞鵃鸼𠊣𠋢𠚴𠣘𠤍𠱙𡀑𡊡𢃸𢏝𢐫𢓟𢫧𢷗𢼲𢽧𣆔𣥯𣻱𤏲𤵥𥀙𥌆𥑸𥖠𥣙𥲝𥺝𥺞𥼫𥾓𥿦𦁖𦂈𦅸𦈺𦡴𦩈𦭴𧇟𧛸𧣷𧧔𧭍𧳜𧻖𨉜𨏺𨥇𨦞𨨯𩊄𩋰𩌄𩍌𩍧𩗪𩢸𩧨𩧳𩶣𪆀𪇞"},
    {U"zhu", U"㑏㑳㔉㝉㤖㦵㧣㫂㵭㶆㹥㺛㾻㿾䃴䆝䇠䇡䇬䌵䍆䎷䐗䐢䕽䘄䘚䘢䝒䝬䟉䠱䡤䣷䥮䪒䬡䭖䮱䰞䳠丶主之予伫佇住侏兪劚助劯咮嗻嘱噣囑坾墸壴孎宁宔尌屬嵀庶拄搊敱斀斗斸曯朝朮术朱杼枓柠柱柷株楮槠樦橥櫡櫧櫫欘殶泏泞注洙渚潴澍濐瀦灟炢炷烛煑煮燭爥猪珠疰瘃眝瞩矚砫硃磩祝祩秼窋竚竹竺笁笜筑筯箸築篫篴紵紸絑纻罜羜翥舳芧苎苧茁茱茿莇著蓫薥藷藸蚰蛀蛛蝫蠋蠩蠾袾褚註詝誅諸诛诸豬貯贮跓跙跦躅軴軸迬逐逗逫邾鉏鉒銖鋳鑄钃铢铸阻除陼霔飳馵駐駯騶驻鮢鯺鱁鴸鸀麆麈鼄𠧀𠩈𠮌𠰍𠴦𠴫𠷅𡎺𡤗𡧨𡱱𡴅𡸌𡺐𡻌𡻠𢁼𢔪𢚻𢥃𢩄𢲿𣀻𣔯𣤁𣥼𣵸𣽆𤆼𤋰𤎧𤕞𤝹𤥮𤲑𤳯𤾄𥋛𥛂𥞏𥩣𥯸𥵟𥹍𥾅𦅷𦉐𦙴𦧙𦬸𧈚𧉞𧏿𧑤𧵤𨆄𨈫𨙔𨞕𨥇𨭅𨲈𩊣𩋵𩒊𩞈𩣷𩨻𩲠𩲬𩳥𩴀𩶂𩶄𪊹𪋏𪋑𪋰𪏿𪚹𬣞"},
    {U"zhua", U"抓摣撾檛爪簻膼髽𠆣𡎬𣑃𥬲𥮣"},
    {U"zhuai", U"尵拽睉跩顡𢶀"},
    {U"zhuan", U"㛯㼷䉵䏙䏝䡱䧘专傳僎僝剸叀啭囀堟塼嫥孨専專巽恮摶撰沌湍漙灷瑑瑼甎砖磗磚竱篆篹篿簨籑縳耑腞膞蒃蟤襈諯譔賺赚転轉转鄟顓颛饌馔鱄𠊩𠨎𡇰𡢀𡤛𡩚𡭇𡰞𢂘𢐎𢞬𣂵𣕏𣚢𤂤𤩄𤪪𤮳𥛥𥫛𦁆𦄯𦓝𦝏𦠆𦧸𧂍𧶲𧸖𨷱𩔊𩧜𩳏𩻝𬤥"},
    {U"zhuang", U"䚒僮壮壯壵奘妆妝娤幢庄庒憧戇撞桩梉樁湷漴焋状狀獞粧糚艟荘莊装裝贛𠌴𢙳𢤤𢦅𣞝𣴣𣶍𣻛𤕯𤘲𤶜𦀜𦚏𩅃𩮱𩯲𪁈𪄻𪅖𪉉"},
    {U"zhui", U"㗓㚝㩾㮅㾽䄌䢡䧳䨨䳡䶆倕坠垂墜娷惴揣桘椎槌沝甀畷硾磓礈笍箠綴縋缀缒腏膇致萑諈贅赘轛追醀醊錐錗錣鎚鑆锥隊隧隹餟騅骓鵻𠄌𡑻𡯑𢊅𣝸𣦬𣨫𤺅𥟒𦥻𧿲𨒉𨧨𨪗𨺵𨾻𩛵𩜀𩪀𩬳𪋇"},
    {U"zhun", U"㡒䐃准凖啍圫埻宒屯忳敦旽淳準甽盹稕窀純綧肫胗衠訰諄谆踆迍飩𥇜𥚠𬘯"},
    {U"zhuo", U"㑁㒂㓸㣿㧳㧻㪬㭬㹿㺟䂐䃗䅵䆯䐁䓬䕴䖦䞵䟾䦃䪼䫎䮓䮕䵠䵵䶂丵倬剢劅勺卓叕啄啅啜噣圴墌妰娺彴拙捉捔掇撯擆擢斀斫斮斱斲斵晫杓桌梲棁棳棹椓槕櫡汋浊浞涿淖準濁濯灂灼炪烵焯燋犳狵琢琸着矠硺禚穛穱窡窧箸篧籗籱繳罬聉肫茁著蓔蕞藋蝃蠗蠿諁諑謶诼趠趵踔蹠躅酌鉵鋜鐯鐲镯鵫鷟𠡑𠭴𠿡𡷿𢁁𢢗𢧈𢳇𢺡𢽚𣃈𣃑𣄻𤃮𤉐𤏸𤓦𥇍𥋮𥐊𥗁𥞺𥢔𥮥𥯩𥷘𥷮𥼚𦜰𦰹𦳡𧂒𧃔𧘑𧞐𧢼𧨳𧰵𧱒𧱰𨁿𨑽𨖮𨡸𨢬𨧧𨮿𨺝𩆸𩋁𩑂𩩔𩲃𩷹𬸦"},
    {U"zi", U"㜽㞨㠿㧗㧘㬐㰣㰷㱴㺭㽧㾅㿳䁒䅆䅔䆅䎩䐉䔂䖪䘣䣎䦻䰵乲事仔倳兹剚吇吱呰呲咨啙嗞姉姊姕姿子孖字孜孳孶崰嵫恣杍柴栥梓椔榟橴次沝泚洓淄渍湽滋滓漬澬牸玆璾甾疵眥眦矷禌秄秭秶稵穧笫籽粢紎純紫緇缁耔胏胔胾自芓茈茊茡茲荢菑葘蓻薋虸觜訾訿諮谘貲資赀资赼趑趦跐載輜輺辎鄑釨鈭鋅錙鍿鎡锱镃頾頿髭鯔鰦鲻鶅鼒齊齍齜龇𠀢𠂔𠡸𡉗𡗈𡙛𡥎𡪒𡸟𡸪𢆶𢱆𢼱𣄮𣓊𣖨𣚀𣚁𣣊𣣌𣥨𣯃𣳩𣸆𤲸𥀖𥕁𥚉𥞎𥫞𥬳𥲕𥻍𥼩𥼻𥿩𦍺𦎸𦖺𦞤𦣹𦺱𧀗𧂐𧆰𧕓𧙁𧛏𧣤𧥕𧧕𧨴𧹌𧿞𨀥𨍢𨚖𨝳𨧫𨩲𨹀𩄚𩐍𩜊𪅵𪑿𪕊𪗉𪗋𪗐"},
    {U"zong", U"㙡㚇㢔㣭㨑㯶㷓㹅䁓䈦䍟䑸䖲䗥䙕䝋䡯䢨䰌倊倧偬傯堫宗嵏嵕嵸從总惣惾愡捴揔搃摠昮朡棕椶樅潈潨熜熧燪猔猣疭瘲碂磫稯粽糉糭綜緃総緵縂縦縱總纵综翪腙艐葼蓗蓯蝬豵踨踪蹤錝鍐鏓鑁騌騣骔鬃鬉鬷鯮鯼𠏭𠕌𠡻𠵻𡕰𡞧𣀒𣯨𤡆𥍺𥓻𥚾𥠡𦖸𦡙𧺣𨌰𨍈𨍉𨎢𨺡𩋯𩤗𩦲𩭤𩮀𩰽𪖁"},
    {U"zou", U"㔌㔿㵵㻓䅳䠫偢奏媰掫揍搊族棷棸楱箃緅芻菆諏诹走赱趣邹郰鄒鄹陬騶驺鯐鯫鲰黀齱齺𠂑𣙻𣠏𥋜𥶈𧌗𨂡𨃘𨑿𨜗𨝮𨽁𩼦𪃆"},
    {U"zu", U"㞺㧻㩆㰵㲞㵀䃚䅸䔃䖕䙘䚝䯿䱣伹俎倅傶卆卒哫唨啐嘁姐岨崒崪怚族柤槭沮淬爼珇砠祖租稡箤組綷组苴菹葅蒩詛謯诅趲足踤踿蹴鉃鉏鉐錊鎐鎺鏃鑿镞阻靻顇駔𠻏𡻬𢅪𢉺𢫵𢳇𢳈𣇙𣢰𣤶𣨛𤓵𤬧𤱌𤽱𥛜𥞯𥣆𥼀𥼪𦑋𦵬𧇈𧇿𧎲𧐈𧑙𧗎𧞰𧺒𨂀𨃭𨄕𨧰𨨳𨩰𩐡𩥿𩩠𩲲𩺯𪋍𪘧𪙳"},
    {U"zuan", U"㸇䂎䌣䡽䤸䰖劗揝撮攢攥欑篹籫繤纂纉纘缵賺躜躦鑚鑽钻𡉺𣀶𣪁𦆈𦙉𨇃𨉖𨰭𩎑𩛻𩯳"},
    {U"zui", U"㝡㠑㭰㰎䓱䘒䘹䮔䯿厜嗺嘴噿嫢嶉嶊嶵摧撮晬最朘枠栬槜槯樶檇檌欈濢璻睟祽稡穝絊纗罪羧蕝蕞蟕觜辠酔酨酻醉鋷錊雋𠲋𠾋𠿘𡙭𡡔𡽁𡽛𢈡𢊛𣖱𣩑𥍋𥳣𥳵𦆈𦈬𦏳𦙈𦸺𧎹𧻝𨢅𨻵𨿇𩚻𩣷𩲨𪋌𪓌"},
    {U"zun", U"䔿僎僔噂墫壿奠尊嶟拵捘捽撙栫樽瀳繜罇袸譐跧踆蹲遵銌鐏鱒鳟鶎鷷𠟃𠱜𤮐𥊭𥞘𥢎𦢐𦨆𦪚𨱔𩯄"},
    {U"zuo", U"㑅㘀㘴㘸㛗㝾㤰㩇㭮㵶㸲䋏䎰䔘䝫䞢䞰䟄䟭䟶䦈乍佐作侳做咗唑嘬坐岝岞左座怍挫捽撮昨柞柮椊琢砟祚秨稓笮筰糳繓胙苲莋葃葄蓙袏諎迮酢醋鈼鑿阼飵𠂇𠱯𠹠𡪳𡯨𡹥𢀡𢂃𢤎𣠹𣩈𣹧𤿀𥅁𥙀𥥏𥹁𥽿𦁎𦈛𦥬𦦹𧃘𧨀𧮙𧲭𨀨𨐳𨝨𨞒𩛠𪎇𪎲"},
    {U",", U"，"},
    {U".", U"。"},
    {U"?", U"？"},
    {U"!", U"！"},
    {U"-", U"—"},
    {U"\"", U"“”"},
    {U"(", U"（"},
    {U")", U"）"}
};

#endif // PINYIN_DATA_H
