// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A0__F325

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_192_0_T_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x60bb3c89u, 0xb117fafbu, 0, 138 }, // 60bb3c89b117fafb = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950__P__16_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x0a12d774u, 0x69cfd185u, 0, 63 }, // 0a12d77469cfd185 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x7115de1du, 0xc599f8f0u, 40, 46 }, // 7115de1dc599f8f0 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x126eb1dcu, 0x54b986c7u, 40, 63 }, // 126eb1dc54b986c7 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xdaae8410u, 0x37b60bf9u, 97, 46 }, // daae841037b60bf9 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950__P__32_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x0edc4ac0u, 0xbb5c03b4u, 120, 80 }, // 0edc4ac0bb5c03b4 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0xd234ab5du, 0xc1c872feu, 120, 155 }, // d234ab5dc1c872fe = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950__P__64_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xe6263084u, 0xe688e9d0u, 126, 46 }, // e6263084e688e9d0 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xd3a7f8c3u, 0xa9434d4bu, 132, 138 }, // d3a7f8c3a9434d4b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_0_T_F_1___gfx950__P__64_64__CO__wave1_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{2,3,8,8,8,8,8,8,8,8},
 {1,1,0,1,1,0,8,8,4,4},
 {0,3,3,3,3,3,4,4,4,4},
 {0,1,1,1,2,3,8,8,4,4},
 {3,3,3,3,3,3,3,7,3,3},
 {5,5,5,3,3,3,3,7,6,6},
 {3,3,3,3,7,7,7,7,3,4},
 {3,3,3,3,3,7,7,7,4,4},
 {3,3,3,3,3,3,3,4,4,8},
 {3,3,3,3,3,3,3,3,3,3}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A0__F325 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 3;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 3;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 192
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 192 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

