/* Copyright (C) 2020-2021 Björn Esser <besser82@fedoraproject.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* Simple test to ensure compatibility of our implementations of the
   md5crypt, sha256crypt, and sha512crypt hashing method with the ones
   found in other commonly used libcrypt libraries.  Their implementations
   allow for any ASCII character (with one exception to the colon ':'
   character, that has special meaning in the Unix shadow file) to be
   present in a salt string.  */

#include "crypt-port.h"

#if INCLUDE_md5crypt || INCLUDE_sha256crypt || INCLUDE_sha512crypt || INCLUDE_sm3crypt

#include <stdio.h>

struct testcase
{
  const char *setting;
  const char *expected;
};

static const struct testcase testcases[] =
{
#if INCLUDE_md5crypt
  {"$1$", "$1$$Nxrr.XTiS6AY.7S2ye3OA1"},
  {"$1$\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01", "*0"},
  {"$1$\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02", "*0"},
  {"$1$\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03", "*0"},
  {"$1$\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04", "*0"},
  {"$1$\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05", "*0"},
  {"$1$\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06", "*0"},
  {"$1$\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07", "*0"},
  {"$1$\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08", "*0"},
  {"$1$\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09", "*0"},
  {"$1$\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a", "*0"},
  {"$1$\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b", "*0"},
  {"$1$\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c", "*0"},
  {"$1$\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d", "*0"},
  {"$1$\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e", "*0"},
  {"$1$\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f", "*0"},
  {"$1$\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10", "*0"},
  {"$1$\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11", "*0"},
  {"$1$\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12", "*0"},
  {"$1$\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13", "*0"},
  {"$1$\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14", "*0"},
  {"$1$\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15", "*0"},
  {"$1$\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16", "*0"},
  {"$1$\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17", "*0"},
  {"$1$\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18", "*0"},
  {"$1$\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19", "*0"},
  {"$1$\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a", "*0"},
  {"$1$\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b", "*0"},
  {"$1$\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c", "*0"},
  {"$1$\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d", "*0"},
  {"$1$\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e", "*0"},
  {"$1$\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f", "*0"},
  {"$1$                ", "*0"},
  {"$1$!!!!!!!!!!!!!!!!", "*0"},
  {"$1$\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"", "$1$\"\"\"\"\"\"\"\"$epL1smbL5e8WnI4OmGUIQ."},
  {"$1$################", "$1$########$BTXbxyZ1vI4TivVy61i3y."},
  {"$1$$$$$$$$$$$$$$$$$", "$1$$Nxrr.XTiS6AY.7S2ye3OA1"},
  {"$1$%%%%%%%%%%%%%%%%", "$1$%%%%%%%%$b.qcij9q4onx9Z11Io6Rm0"},
  {"$1$&&&&&&&&&&&&&&&&", "$1$&&&&&&&&$VED/ZzenKsKk0VWdWPvZd."},
  {"$1$''''''''''''''''", "$1$''''''''$ug/mEvIhArMo2ZmDR.ukL0"},
  {"$1$((((((((((((((((", "$1$(((((((($D1sJfyMGDYn68nzKKauI20"},
  {"$1$))))))))))))))))", "$1$))))))))$koy9i6GsfoQJUdGDeKje.."},
  {"$1$****************", "*0"},
  {"$1$++++++++++++++++", "$1$++++++++$N3V3THe53iandIYWj9u3Z0"},
  {"$1$,,,,,,,,,,,,,,,,", "$1$,,,,,,,,$0LQlmUbUFJ.0UMPEHhJOJ1"},
  {"$1$----------------", "$1$--------$Ik.e0Cgi84iak0gQPBRU81"},
  {"$1$................", "$1$........$YNN4lBU4qVf9ienSk8brr/"},
  {"$1$////////////////", "$1$////////$1PzmjkqpnpM2Xx9sUHZrJ1"},
  {"$1$0000000000000000", "$1$00000000$WIv0qOX13MxRPPkNATuzz/"},
  {"$1$1111111111111111", "$1$11111111$kfWauknBJcj7MOR0nPgWG1"},
  {"$1$2222222222222222", "$1$22222222$pN8.GpS.W6mUUsBd3w0aP0"},
  {"$1$3333333333333333", "$1$33333333$A36NAKszUDbvHb2RGqIL5."},
  {"$1$4444444444444444", "$1$44444444$meQvyWbhi5NZtY7q60pKc/"},
  {"$1$5555555555555555", "$1$55555555$lWRVMPymhoz3hU/1FZ5f70"},
  {"$1$6666666666666666", "$1$66666666$Exx.2frtfKms4Iw22klhE0"},
  {"$1$7777777777777777", "$1$77777777$42lxQ94MiNIq60eIwMX3G0"},
  {"$1$8888888888888888", "$1$88888888$7uqcuVSxDGygAoyYUppiH/"},
  {"$1$9999999999999999", "$1$99999999$s7b4BUN.GQWtpw9H1.DIa."},
  {"$1$::::::::::::::::", "*0"},
  {"$1$;;;;;;;;;;;;;;;;", "*0"},
  {"$1$<<<<<<<<<<<<<<<<", "$1$<<<<<<<<$TMxdcN3jjYkKA1tduDLFq."},
  {"$1$================", "$1$========$JmHCPrUJf5ZwatxiIEQzd1"},
  {"$1$>>>>>>>>>>>>>>>>", "$1$>>>>>>>>$hc9L8QUv8Sr48QqgT9OU/1"},
  {"$1$????????????????", "$1$????????$kl3sFY2vkCbiT1jGkTk4D."},
  {"$1$@@@@@@@@@@@@@@@@", "$1$@@@@@@@@$qG.6p.sulKDSneQYdmumq."},
  {"$1$AAAAAAAAAAAAAAAA", "$1$AAAAAAAA$rARmimYWpVtDMOkOCHZsV0"},
  {"$1$BBBBBBBBBBBBBBBB", "$1$BBBBBBBB$ycTOhmpddCcf7lVvKX0tJ/"},
  {"$1$CCCCCCCCCCCCCCCC", "$1$CCCCCCCC$meJSeRpVQnG8QHFUxconF."},
  {"$1$DDDDDDDDDDDDDDDD", "$1$DDDDDDDD$aeVX.La08fM6hqD4fwu3m/"},
  {"$1$EEEEEEEEEEEEEEEE", "$1$EEEEEEEE$l7d7o3eXNRs1SjnbIKEgw/"},
  {"$1$FFFFFFFFFFFFFFFF", "$1$FFFFFFFF$CxqkiERpodlM9sU3YBeGf1"},
  {"$1$GGGGGGGGGGGGGGGG", "$1$GGGGGGGG$z6TSf5KIElqpRoooKKqTv/"},
  {"$1$HHHHHHHHHHHHHHHH", "$1$HHHHHHHH$hoDSzfcrTMYCWkoJhy6sE0"},
  {"$1$IIIIIIIIIIIIIIII", "$1$IIIIIIII$TkU6VWUwWZUeFYOXwtAq51"},
  {"$1$JJJJJJJJJJJJJJJJ", "$1$JJJJJJJJ$WGaWHRzJ6KqPyTTNHz5Sx/"},
  {"$1$KKKKKKKKKKKKKKKK", "$1$KKKKKKKK$jrb7xaFxBqUOSTL2Mqpuz0"},
  {"$1$LLLLLLLLLLLLLLLL", "$1$LLLLLLLL$gjCuwxKCso6nkNlA/H1t4/"},
  {"$1$MMMMMMMMMMMMMMMM", "$1$MMMMMMMM$nDq6vl28MDntXIk7XjhYa0"},
  {"$1$NNNNNNNNNNNNNNNN", "$1$NNNNNNNN$xOFVP1RpOGrEbbCqj5wQR/"},
  {"$1$OOOOOOOOOOOOOOOO", "$1$OOOOOOOO$6nRSEBBy1ugB2IpKUmjBZ/"},
  {"$1$PPPPPPPPPPPPPPPP", "$1$PPPPPPPP$V.oSP3Xx4yLeNMwbHtVri0"},
  {"$1$QQQQQQQQQQQQQQQQ", "$1$QQQQQQQQ$j5CplJvskamHbbJr/KDdc/"},
  {"$1$RRRRRRRRRRRRRRRR", "$1$RRRRRRRR$ggNTLaAVAnOGIqWBWxezW."},
  {"$1$SSSSSSSSSSSSSSSS", "$1$SSSSSSSS$/CE/Y6QawKcGuhWKYyRdu0"},
  {"$1$TTTTTTTTTTTTTTTT", "$1$TTTTTTTT$WADhbmdTKuZwV.qGr2jUN0"},
  {"$1$UUUUUUUUUUUUUUUU", "$1$UUUUUUUU$qkEJoJb4OnrFpH6bnH4g5/"},
  {"$1$VVVVVVVVVVVVVVVV", "$1$VVVVVVVV$2wuSe47ONM48I0MEMRITG."},
  {"$1$WWWWWWWWWWWWWWWW", "$1$WWWWWWWW$vH/XOSxWseOr.Fw5EdFxr/"},
  {"$1$XXXXXXXXXXXXXXXX", "$1$XXXXXXXX$70M55QoHBsLko.FFc6Cp11"},
  {"$1$YYYYYYYYYYYYYYYY", "$1$YYYYYYYY$ntauSgB0OwKqb8.ik6HMM/"},
  {"$1$ZZZZZZZZZZZZZZZZ", "$1$ZZZZZZZZ$j1htqKO9MRZP455gQ4OBR1"},
  {"$1$[[[[[[[[[[[[[[[[", "$1$[[[[[[[[$YOdsKzhIoPKXHuBW8ffeo0"},
  {"$1$\\\\\\\\\\\\\\\\", "*0"},
  {"$1$]]]]]]]]]]]]]]]]", "$1$]]]]]]]]$6zvmbgSu3saeTevjvPBA70"},
  {"$1$^^^^^^^^^^^^^^^^", "$1$^^^^^^^^$Wv2rkpIRDuiQ.Blobxcx8."},
  {"$1$________________", "$1$________$LUlsmTnGQigLnnlsnkX8o0"},
  {"$1$````````````````", "$1$````````$oKci.FIXaRhOaiOcA6Mzi/"},
  {"$1$aaaaaaaaaaaaaaaa", "$1$aaaaaaaa$.yfukZepOC7STW9Bwbw0f1"},
  {"$1$bbbbbbbbbbbbbbbb", "$1$bbbbbbbb$B8AEHJyJP9pdKiVqdezmU."},
  {"$1$cccccccccccccccc", "$1$cccccccc$CjWos3pM1lGi2p84Ta4kG0"},
  {"$1$dddddddddddddddd", "$1$dddddddd$i9ye4TY0noHzrRDXPNzpu/"},
  {"$1$eeeeeeeeeeeeeeee", "$1$eeeeeeee$NTNWA/D.glIRU/hTgj5/x."},
  {"$1$ffffffffffffffff", "$1$ffffffff$VERjgO485ZgxByW.ZW5Pr."},
  {"$1$gggggggggggggggg", "$1$gggggggg$.d7Evd2ZuF5.5HKiC9ztA1"},
  {"$1$hhhhhhhhhhhhhhhh", "$1$hhhhhhhh$8yUmO.8la8.NKuY93.vN0/"},
  {"$1$iiiiiiiiiiiiiiii", "$1$iiiiiiii$Nm7qy3PE/l67qH5ChuGO8."},
  {"$1$jjjjjjjjjjjjjjjj", "$1$jjjjjjjj$eozS0hxT1TYnIGyS97ytQ1"},
  {"$1$kkkkkkkkkkkkkkkk", "$1$kkkkkkkk$vgL1OKMr7doJdjjG1Nn2C0"},
  {"$1$llllllllllllllll", "$1$llllllll$hcK96/tM.gWe1IAI6MS7E1"},
  {"$1$mmmmmmmmmmmmmmmm", "$1$mmmmmmmm$y1uAEuAGCYBBtWVqCZSek1"},
  {"$1$nnnnnnnnnnnnnnnn", "$1$nnnnnnnn$.6Ky5wTLYM5HLS5FbwSrZ1"},
  {"$1$oooooooooooooooo", "$1$oooooooo$syXICQSw8utrPrAOz41yZ1"},
  {"$1$pppppppppppppppp", "$1$pppppppp$Qd0rS5ZsHCXWjZeJkGCDZ/"},
  {"$1$qqqqqqqqqqqqqqqq", "$1$qqqqqqqq$NKTfQNaTgQdMzqwSH/yH01"},
  {"$1$rrrrrrrrrrrrrrrr", "$1$rrrrrrrr$eCx3wAEkuOeWKq.TxmGZ31"},
  {"$1$ssssssssssssssss", "$1$ssssssss$dmgoNRxFJ.Uvvi48RkcdV."},
  {"$1$tttttttttttttttt", "$1$tttttttt$d1OszLtpuCf0I6KjqFXd30"},
  {"$1$uuuuuuuuuuuuuuuu", "$1$uuuuuuuu$y4iwsDsCwenDsUI0hGg72."},
  {"$1$vvvvvvvvvvvvvvvv", "$1$vvvvvvvv$IKlHLyWn9rUXcumgVCVwC."},
  {"$1$wwwwwwwwwwwwwwww", "$1$wwwwwwww$RavCRC59VdCiezcgyiIwC1"},
  {"$1$xxxxxxxxxxxxxxxx", "$1$xxxxxxxx$GZDIpernfuj7a/Iy2CTL70"},
  {"$1$yyyyyyyyyyyyyyyy", "$1$yyyyyyyy$bNX5MvbK.MKW6x8MNYhgW0"},
  {"$1$zzzzzzzzzzzzzzzz", "$1$zzzzzzzz$Bz5/HkvZ/FJMCOCNWfXEg0"},
  {"$1${{{{{{{{{{{{{{{{", "$1${{{{{{{{$pzvQi.ekvLBgGAGyu/EmW/"},
  {"$1$||||||||||||||||", "$1$||||||||$MQEVd9ZBTfnBpn3fHH9vV0"},
  {"$1$}}}}}}}}}}}}}}}}", "$1$}}}}}}}}$SP2cIcNL58YC/LwTmJEed0"},
  {"$1$~~~~~~~~~~~~~~~~", "$1$~~~~~~~~$3smC2d7JWcLL2lGnzsVww1"},
  {"$1$\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f", "*0"},
  {"$1$\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80", "*0"},
  {"$1$\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81", "*0"},
  {"$1$\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82", "*0"},
  {"$1$\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83", "*0"},
  {"$1$\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84", "*0"},
  {"$1$\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85", "*0"},
  {"$1$\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86", "*0"},
  {"$1$\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87", "*0"},
  {"$1$\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88", "*0"},
  {"$1$\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89", "*0"},
  {"$1$\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a", "*0"},
  {"$1$\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b", "*0"},
  {"$1$\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c", "*0"},
  {"$1$\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d", "*0"},
  {"$1$\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e", "*0"},
  {"$1$\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f", "*0"},
  {"$1$\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90", "*0"},
  {"$1$\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91", "*0"},
  {"$1$\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92", "*0"},
  {"$1$\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93", "*0"},
  {"$1$\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94", "*0"},
  {"$1$\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95", "*0"},
  {"$1$\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96", "*0"},
  {"$1$\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97", "*0"},
  {"$1$\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98", "*0"},
  {"$1$\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99", "*0"},
  {"$1$\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a", "*0"},
  {"$1$\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b", "*0"},
  {"$1$\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c", "*0"},
  {"$1$\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d", "*0"},
  {"$1$\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e", "*0"},
  {"$1$\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f", "*0"},
  {"$1$\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0", "*0"},
  {"$1$\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1", "*0"},
  {"$1$\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2", "*0"},
  {"$1$\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3", "*0"},
  {"$1$\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4", "*0"},
  {"$1$\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5", "*0"},
  {"$1$\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6", "*0"},
  {"$1$\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7", "*0"},
  {"$1$\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8", "*0"},
  {"$1$\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9", "*0"},
  {"$1$\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa", "*0"},
  {"$1$\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab", "*0"},
  {"$1$\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac", "*0"},
  {"$1$\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad", "*0"},
  {"$1$\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae", "*0"},
  {"$1$\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf", "*0"},
  {"$1$\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0", "*0"},
  {"$1$\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1", "*0"},
  {"$1$\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2", "*0"},
  {"$1$\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3", "*0"},
  {"$1$\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4", "*0"},
  {"$1$\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5", "*0"},
  {"$1$\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6", "*0"},
  {"$1$\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7", "*0"},
  {"$1$\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8", "*0"},
  {"$1$\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9", "*0"},
  {"$1$\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba", "*0"},
  {"$1$\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb", "*0"},
  {"$1$\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc", "*0"},
  {"$1$\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd", "*0"},
  {"$1$\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe", "*0"},
  {"$1$\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf", "*0"},
  {"$1$\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0", "*0"},
  {"$1$\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1", "*0"},
  {"$1$\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2", "*0"},
  {"$1$\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3", "*0"},
  {"$1$\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4", "*0"},
  {"$1$\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5", "*0"},
  {"$1$\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6", "*0"},
  {"$1$\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7", "*0"},
  {"$1$\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8", "*0"},
  {"$1$\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9", "*0"},
  {"$1$\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca", "*0"},
  {"$1$\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb", "*0"},
  {"$1$\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc", "*0"},
  {"$1$\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd", "*0"},
  {"$1$\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce", "*0"},
  {"$1$\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf", "*0"},
  {"$1$\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0", "*0"},
  {"$1$\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1", "*0"},
  {"$1$\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2", "*0"},
  {"$1$\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3", "*0"},
  {"$1$\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4", "*0"},
  {"$1$\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5", "*0"},
  {"$1$\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6", "*0"},
  {"$1$\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7", "*0"},
  {"$1$\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8", "*0"},
  {"$1$\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9", "*0"},
  {"$1$\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda", "*0"},
  {"$1$\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb", "*0"},
  {"$1$\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc", "*0"},
  {"$1$\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd", "*0"},
  {"$1$\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde", "*0"},
  {"$1$\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf", "*0"},
  {"$1$\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0", "*0"},
  {"$1$\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1", "*0"},
  {"$1$\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2", "*0"},
  {"$1$\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3", "*0"},
  {"$1$\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4", "*0"},
  {"$1$\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5", "*0"},
  {"$1$\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6", "*0"},
  {"$1$\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7", "*0"},
  {"$1$\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8", "*0"},
  {"$1$\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9", "*0"},
  {"$1$\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea", "*0"},
  {"$1$\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb", "*0"},
  {"$1$\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec", "*0"},
  {"$1$\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed", "*0"},
  {"$1$\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee", "*0"},
  {"$1$\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef", "*0"},
  {"$1$\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0", "*0"},
  {"$1$\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1", "*0"},
  {"$1$\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2", "*0"},
  {"$1$\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3", "*0"},
  {"$1$\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4", "*0"},
  {"$1$\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5", "*0"},
  {"$1$\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6", "*0"},
  {"$1$\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7", "*0"},
  {"$1$\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8", "*0"},
  {"$1$\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9", "*0"},
  {"$1$\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa", "*0"},
  {"$1$\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb", "*0"},
  {"$1$\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc", "*0"},
  {"$1$\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd", "*0"},
  {"$1$\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe", "*0"},
  {"$1$\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff", "*0"},
#endif
#if INCLUDE_sm3crypt
  {"$sm3$", "$sm3$$HhPEDu1bjf.xru5vFLDfb/JywfIdph0sazvY4rLf7vD"},
  {"$sm3$\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01", "*0"},
  {"$sm3$\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02", "*0"},
  {"$sm3$\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03", "*0"},
  {"$sm3$\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04", "*0"},
  {"$sm3$\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05", "*0"},
  {"$sm3$\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06", "*0"},
  {"$sm3$\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07", "*0"},
  {"$sm3$\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08", "*0"},
  {"$sm3$\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09", "*0"},
  {"$sm3$\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a", "*0"},
  {"$sm3$\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b", "*0"},
  {"$sm3$\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c", "*0"},
  {"$sm3$\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d", "*0"},
  {"$sm3$\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e", "*0"},
  {"$sm3$\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f", "*0"},
  {"$sm3$\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10", "*0"},
  {"$sm3$\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11", "*0"},
  {"$sm3$\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12", "*0"},
  {"$sm3$\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13", "*0"},
  {"$sm3$\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14", "*0"},
  {"$sm3$\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15", "*0"},
  {"$sm3$\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16", "*0"},
  {"$sm3$\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17", "*0"},
  {"$sm3$\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18", "*0"},
  {"$sm3$\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19", "*0"},
  {"$sm3$\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a", "*0"},
  {"$sm3$\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b", "*0"},
  {"$sm3$\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c", "*0"},
  {"$sm3$\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d", "*0"},
  {"$sm3$\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e", "*0"},
  {"$sm3$\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f", "*0"},
  {"$sm3$                ", "*0"},
  {"$sm3$!!!!!!!!!!!!!!!!", "*0"},
  {"$sm3$\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"", "$sm3$\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"$rotR.HwtkJKBXpNsFTuZB4R3viMAQUfrYmcGjocBPM1"},
  {"$sm3$################", "$sm3$################$9n039tE799I46RPFM.o9E41MvlKcUvz9tLrfQcsEdZC"},
  {"$sm3$%%%%%%%%%%%%%%%%", "$sm3$%%%%%%%%%%%%%%%%$661EqK8/5q69rZHLFpmYyYBQWcmX/LiUe8RyaGLRKlD"},
  {"$sm3$&&&&&&&&&&&&&&&&", "$sm3$&&&&&&&&&&&&&&&&$VmSc06G/ZI.t6W0LWlZHjLMn3kWi3r7i5CDdMirqIp/"},
  {"$sm3$''''''''''''''''", "$sm3$''''''''''''''''$s4KixzCP9YBybPlaFpTJxEHSCbYkc/vudwSmtY9/qF1"},
  {"$sm3$((((((((((((((((", "$sm3$(((((((((((((((($zsgVpKMSuppBhZDuDvfH8cEVVPkvJQ1DBkZa9hszMN2"},
  {"$sm3$))))))))))))))))", "$sm3$))))))))))))))))$H12uGU70P8sKLt6hiVN7kZN9KhJcxaqWauLQslli/qC"},
  {"$sm3$++++++++++++++++", "$sm3$++++++++++++++++$N.4x7wI.768Jp/LeMryviDUd2dXWwCHFYmc25qgws5."},
  {"$sm3$,,,,,,,,,,,,,,,,", "$sm3$,,,,,,,,,,,,,,,,$EuBOskVycfHLyCbhd1/LXSCvzcxsEslq2Q2kRhbTGf3"},
  {"$sm3$----------------", "$sm3$----------------$UykKwOl45sxrgWbzkSX2e1KJeh86Cdut6k.MLVB2QC."},
  {"$sm3$................", "$sm3$................$41qwH0Q2iCPIJRIRHU.TiSm/TUhqD7pYY1pfwRmLHQ6"},
  {"$sm3$////////////////", "$sm3$////////////////$pQlGWtRKKVR3XHbFZhcd8mvTZNriAav0alSCZQf3ar5"},
  {"$sm3$0000000000000000", "$sm3$0000000000000000$5SihUZa0diNkVMWju/Dgmw96rE7Ya7nduLr.2NdSTo9"},
  {"$sm3$1111111111111111", "$sm3$1111111111111111$RYiw/RShgd9geFnT9et0oiTINSbzgZiInliGLklJbx5"},
  {"$sm3$2222222222222222", "$sm3$2222222222222222$toe8tenehDVdnbp3J9ej36ZEUcbigkWbdKa2vhgB580"},
  {"$sm3$3333333333333333", "$sm3$3333333333333333$BsgOwxWAye3esGMecZA.CWBDaxkQF6VtHHUyHvfZ1h4"},
  {"$sm3$4444444444444444", "$sm3$4444444444444444$rBDZf61BLefacVapw4Oidmcg3v2SxY/dPFrTnZK0TtB"},
  {"$sm3$5555555555555555", "$sm3$5555555555555555$nLvDt7JxzhIsVQNzsTBUK4C5cpe/ObEtkpJtsxVdyZ9"},
  {"$sm3$6666666666666666", "$sm3$6666666666666666$1VliklUe.bMtXGRfvXK1pqPQVddn5Q3D9UJv/ixQp14"},
  {"$sm3$7777777777777777", "$sm3$7777777777777777$B8NCD50PzK6hPicvC9hkPrvYbhaSkmz.vZlGc0TAat9"},
  {"$sm3$8888888888888888", "$sm3$8888888888888888$As83spSU14r1YzwQ.iHfybJ6zvPju9VpP1u2vqmeqfC"},
  {"$sm3$9999999999999999", "$sm3$9999999999999999$ybFZUgqh80x4FMTRyh99Xc.AlYyEGrNkIqZ6EPsnQe0"},
  {"$sm3$::::::::::::::::", "*0"},
  {"$sm3$;;;;;;;;;;;;;;;;", "*0"},
  {"$sm3$<<<<<<<<<<<<<<<<", "$sm3$<<<<<<<<<<<<<<<<$ed.OwIxIwaByO8R7jf4HJE8IeefxD.lRYMpoX9eNOuC"},
  {"$sm3$================", "$sm3$================$DEKH.cIvUJjFPex/AxVDIQU925IHAuuCeanH4MbPVd9"},
  {"$sm3$>>>>>>>>>>>>>>>>", "$sm3$>>>>>>>>>>>>>>>>$OqR4fJiVv.mnsryKt3cL8XpQgKRVmuZIg2fF8QSIcG4"},
  {"$sm3$????????????????", "$sm3$????????????????$wPIbzWinDJcdmugdaOJ6Qhr2sFqyye9DTjnxll/9zw8"},
  {"$sm3$@@@@@@@@@@@@@@@@", "$sm3$@@@@@@@@@@@@@@@@$McJNNmlRv/OngIUCEdj2kvNAmfN6EOdWu6eaU0AC.39"},
  {"$sm3$AAAAAAAAAAAAAAAA", "$sm3$AAAAAAAAAAAAAAAA$QVnx1uttJDvm4tHlr4dysQTENEdms40gPSDsXfoBhND"},
  {"$sm3$BBBBBBBBBBBBBBBB", "$sm3$BBBBBBBBBBBBBBBB$615myiFdDt9/s5YAhqY35sDbgHgvqABzHTPZY8qltv7"},
  {"$sm3$CCCCCCCCCCCCCCCC", "$sm3$CCCCCCCCCCCCCCCC$NljEz23Shmj2b9gR4CK65fkgaIYbOfhW.TbaKKXvdg9"},
  {"$sm3$DDDDDDDDDDDDDDDD", "$sm3$DDDDDDDDDDDDDDDD$6hhg8Vuwixb8LnVzwvS4MDZpkOBBFnVAZmgUye2bmg5"},
  {"$sm3$EEEEEEEEEEEEEEEE", "$sm3$EEEEEEEEEEEEEEEE$L7GuK5kdMMUnzpG4ghruHWvyPaMNZxkqijePlHzPGW."},
  {"$sm3$FFFFFFFFFFFFFFFF", "$sm3$FFFFFFFFFFFFFFFF$CKjNBsUkFP1O8kyuetByO8qTIAOFpFQYHNktDR56tx3"},
  {"$sm3$GGGGGGGGGGGGGGGG", "$sm3$GGGGGGGGGGGGGGGG$ZspbB5E685aZG7V3HzSCb98kf/Xqq1enk4v8bUkAkS3"},
  {"$sm3$HHHHHHHHHHHHHHHH", "$sm3$HHHHHHHHHHHHHHHH$740Kc6mj/hbVIi1jFbohimaHjJ5UW8BlhnO9JJxzYA0"},
  {"$sm3$IIIIIIIIIIIIIIII", "$sm3$IIIIIIIIIIIIIIII$EDqGkw7oXp/WP1vt467mSDRmNevEs26RkmWHn5rGh8B"},
  {"$sm3$JJJJJJJJJJJJJJJJ", "$sm3$JJJJJJJJJJJJJJJJ$eYMJPEJHwSn4dVaSsGFSh28609C.I2f9O57GaqBXVt."},
  {"$sm3$KKKKKKKKKKKKKKKK", "$sm3$KKKKKKKKKKKKKKKK$y70uJ/vpCD/4iGu6CLW5RbPoaEhFToyBIBK37OxNNyD"},
  {"$sm3$LLLLLLLLLLLLLLLL", "$sm3$LLLLLLLLLLLLLLLL$eoLMHi6kcbQmUT619rOHg2RkfKqvZGWu3HEGbUa8olC"},
  {"$sm3$MMMMMMMMMMMMMMMM", "$sm3$MMMMMMMMMMMMMMMM$UCgx1Tm7mN3r3bOSJhB9zHpdPFax4j33d7npFLKxcYB"},
  {"$sm3$NNNNNNNNNNNNNNNN", "$sm3$NNNNNNNNNNNNNNNN$y4PlpQbyQdaamI.slv5JmS.YQv1txLkIwmk6k7CVttD"},
  {"$sm3$OOOOOOOOOOOOOOOO", "$sm3$OOOOOOOOOOOOOOOO$oAgyc2rpzZQIH2MW9UiE2GAd.w5rQJDiNkp6F4WKQR5"},
  {"$sm3$PPPPPPPPPPPPPPPP", "$sm3$PPPPPPPPPPPPPPPP$Oa/r5NmFHcjhpcTL.eqrlU5ZaF3gFZW4DCMtbDwjXa3"},
  {"$sm3$QQQQQQQQQQQQQQQQ", "$sm3$QQQQQQQQQQQQQQQQ$FvQmpACpogoRqU6lImLJIUhhNeyqDx.pzs.PM4SYK/8"},
  {"$sm3$RRRRRRRRRRRRRRRR", "$sm3$RRRRRRRRRRRRRRRR$y82r9hNLOEIu1YGxOpnY9W5aN/RPkdoEqotDXOpvrQ4"},
  {"$sm3$SSSSSSSSSSSSSSSS", "$sm3$SSSSSSSSSSSSSSSS$1xrhD5tFFRyHnRoDC2x20ZuH9yQX0oTWJTMlCc7moz."},
  {"$sm3$TTTTTTTTTTTTTTTT", "$sm3$TTTTTTTTTTTTTTTT$.ZfNLRvLuvEucTiQotv0GxIQ7m2MIH75EvbOkunPCE9"},
  {"$sm3$UUUUUUUUUUUUUUUU", "$sm3$UUUUUUUUUUUUUUUU$6s0rERfzx.5Y4y86nxV4qSacYjhV7gj5/njuvUptCC8"},
  {"$sm3$VVVVVVVVVVVVVVVV", "$sm3$VVVVVVVVVVVVVVVV$P66ODLYxQzJkwrlxIAz9SiqLHyHMDRE3WbcxZVZge18"},
  {"$sm3$WWWWWWWWWWWWWWWW", "$sm3$WWWWWWWWWWWWWWWW$KCOfBl4lt9JmbdbTq50lerZWwjG9yDWQTmiHLyGf3F8"},
  {"$sm3$XXXXXXXXXXXXXXXX", "$sm3$XXXXXXXXXXXXXXXX$MvWWfbbjjwTstV.I45mKeXPodrXKJfKP3JowNf8oPS."},
  {"$sm3$YYYYYYYYYYYYYYYY", "$sm3$YYYYYYYYYYYYYYYY$K5RgEOXt0IwK5YYjJSFxx2dDgtcGdLSRLQ5wgSNuvM6"},
  {"$sm3$ZZZZZZZZZZZZZZZZ", "$sm3$ZZZZZZZZZZZZZZZZ$bXbOyyE/lNafvCIXs1vxjTS9qnxxFBW/t1emzhAhTB5"},
  {"$sm3$[[[[[[[[[[[[[[[[", "$sm3$[[[[[[[[[[[[[[[[$VBpnOTvA6cbEw2dga7d4rKUNPA7ZDmH5/lYrhc9.DPA"},
  {"$sm3$\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\", "*0"},
  {"$sm3$]]]]]]]]]]]]]]]]", "$sm3$]]]]]]]]]]]]]]]]$W0Y5yUpXYmrbJwMFu..6vzpKpGvW.QHtNs0SUXxELl6"},
  {"$sm3$^^^^^^^^^^^^^^^^", "$sm3$^^^^^^^^^^^^^^^^$Z.fBSo1IsC11naMAy/SbZNpFsdmIPdSAk0ea178FHM9"},
  {"$sm3$________________", "$sm3$________________$z.MrCO.StIK/piDWn9geSKrDuuFClBPffZBZ2vRI010"},
  {"$sm3$````````````````", "$sm3$````````````````$Lfd2TyYUjFnK5xc7rYC2WjWLky.ivWo7BYIivZiBSvC"},
  {"$sm3$aaaaaaaaaaaaaaaa", "$sm3$aaaaaaaaaaaaaaaa$xCVY.YbpISXGfs1znx5f1er4ZfMmvy3tOT6dKMssub."},
  {"$sm3$bbbbbbbbbbbbbbbb", "$sm3$bbbbbbbbbbbbbbbb$POZlgEI.maXDED6iaKRzS/b.9wEtfPqy178V5mF/3u1"},
  {"$sm3$cccccccccccccccc", "$sm3$cccccccccccccccc$vwqpi5/j0..HatglZVP.sLG9cDbUG3BdCQjJqjeVgd."},
  {"$sm3$dddddddddddddddd", "$sm3$dddddddddddddddd$jR8EmJC9z2hc5J4xUXY5.DxfMVD6thTnnm4lUTIR3h5"},
  {"$sm3$eeeeeeeeeeeeeeee", "$sm3$eeeeeeeeeeeeeeee$4mWcyWAeLhMOZURPu0/P3x6R3H8sK4czLDNiIaojXV6"},
  {"$sm3$ffffffffffffffff", "$sm3$ffffffffffffffff$sXNxxiWxealPK6A25IRkQGYwtLdp3pppktpFyMmvHp4"},
  {"$sm3$gggggggggggggggg", "$sm3$gggggggggggggggg$NbXuIsg0DSg40l1xsfW7thXy0fLhlvWsoDpDqLVoYP4"},
  {"$sm3$hhhhhhhhhhhhhhhh", "$sm3$hhhhhhhhhhhhhhhh$OK69HzW12PVP26o0R62TziUY8fjMx4T9Lsp/ba7mwV6"},
  {"$sm3$iiiiiiiiiiiiiiii", "$sm3$iiiiiiiiiiiiiiii$LOHzx6lA1QBG5BebnTmsGCNTrS29KMu4iJ7BgMOMgh2"},
  {"$sm3$jjjjjjjjjjjjjjjj", "$sm3$jjjjjjjjjjjjjjjj$CybcR8TV4Ogim.t45WA7KiGB14sco8RLwhkpogJHwP2"},
  {"$sm3$kkkkkkkkkkkkkkkk", "$sm3$kkkkkkkkkkkkkkkk$z6ECqf30WpcXY9bkTZyil79npM.x8GXWVtfjnqo4I42"},
  {"$sm3$llllllllllllllll", "$sm3$llllllllllllllll$qEgEdKFyNgr3na1/XoaicAA2fr1eRqKAALS/K2V5qM1"},
  {"$sm3$mmmmmmmmmmmmmmmm", "$sm3$mmmmmmmmmmmmmmmm$w.am8GBYSsJFE0MXxZvyVYCq2qUTyXXPKaM2hjdF1K/"},
  {"$sm3$nnnnnnnnnnnnnnnn", "$sm3$nnnnnnnnnnnnnnnn$VbDIQ7phZwxaRxM2VY0xg/7mNuQiFNKilVi8vBXqlI4"},
  {"$sm3$oooooooooooooooo", "$sm3$oooooooooooooooo$M6lub.VPCcLNeqJQzaGG5wJX4X/wxzNmaaPH59renLB"},
  {"$sm3$pppppppppppppppp", "$sm3$pppppppppppppppp$3LXjalmIyjnuhmp.ToydE1.Y29SME06iWz.GwbtRueC"},
  {"$sm3$qqqqqqqqqqqqqqqq", "$sm3$qqqqqqqqqqqqqqqq$vq3esHm8bIEsYGAVCno.mlx5JYzDvVltS96kj7CRVv0"},
  {"$sm3$rrrrrrrrrrrrrrrr", "$sm3$rrrrrrrrrrrrrrrr$oQ.VFcTK5dToElTi0DpPaXj/zXf5GWwxdh4OXZhlAn."},
  {"$sm3$ssssssssssssssss", "$sm3$ssssssssssssssss$/ROaq.PHX8TYaRIlTtjR45FgjHD4WE20DxtNs9cUWd0"},
  {"$sm3$tttttttttttttttt", "$sm3$tttttttttttttttt$gjec8y8rnvqlm5AGevnusnCU3ER9ah8tGhWUtqM/0i9"},
  {"$sm3$uuuuuuuuuuuuuuuu", "$sm3$uuuuuuuuuuuuuuuu$LqeUCy/yOcaADjaUPIhbGDbh/Jy5SvPI5h3.khbhJz5"},
  {"$sm3$vvvvvvvvvvvvvvvv", "$sm3$vvvvvvvvvvvvvvvv$FKjdE0qSfRn47TjZQM36eeaD9u/sdtLIpTn5iobBnE1"},
  {"$sm3$wwwwwwwwwwwwwwww", "$sm3$wwwwwwwwwwwwwwww$dIo08ED5QpbJOx/xNZjcBEVf/ApUIqKCfE0CoMZqe72"},
  {"$sm3$xxxxxxxxxxxxxxxx", "$sm3$xxxxxxxxxxxxxxxx$uqEgp0ez71drTLD.JuyWoGJEvuhRkr8C/Zsg6cxtmH0"},
  {"$sm3$yyyyyyyyyyyyyyyy", "$sm3$yyyyyyyyyyyyyyyy$B8C1f1/j.BXi4JpN8CncjmtGmFiszAerGWwCYW4uVy6"},
  {"$sm3$zzzzzzzzzzzzzzzz", "$sm3$zzzzzzzzzzzzzzzz$jBwR/9Ns9jjcTnPVkd17j5tyd9DELu3Hk5whtx3Mil5"},
  {"$sm3${{{{{{{{{{{{{{{{", "$sm3${{{{{{{{{{{{{{{{$i.QtmWhYhF1TF5Jm5BRlahJTMfW7MSpXKrwDwJIJWf/"},
  {"$sm3$||||||||||||||||", "$sm3$||||||||||||||||$oaNC4l5cg5OMZ49H86OfLeZ5cEJTCvSzjlARrH5Ry6/"},
  {"$sm3$}}}}}}}}}}}}}}}}", "$sm3$}}}}}}}}}}}}}}}}$xg.U9oYOumcnXlOBWd.SivWFM3EZVrXlnXRIQkZ1fa7"},
  {"$sm3$~~~~~~~~~~~~~~~~", "$sm3$~~~~~~~~~~~~~~~~$fG6QnaQxzvRSESuTO02UL9ffMKsWuzGd5eyy6Lasbe8"},
  {"$sm3$\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f", "*0"},
  {"$sm3$\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80", "*0"},
  {"$sm3$\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81", "*0"},
  {"$sm3$\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82", "*0"},
  {"$sm3$\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83", "*0"},
  {"$sm3$\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84", "*0"},
  {"$sm3$\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85", "*0"},
  {"$sm3$\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86", "*0"},
  {"$sm3$\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87", "*0"},
  {"$sm3$\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88", "*0"},
  {"$sm3$\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89", "*0"},
  {"$sm3$\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a", "*0"},
  {"$sm3$\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b", "*0"},
  {"$sm3$\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c", "*0"},
  {"$sm3$\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d", "*0"},
  {"$sm3$\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e", "*0"},
  {"$sm3$\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f", "*0"},
  {"$sm3$\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90", "*0"},
  {"$sm3$\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91", "*0"},
  {"$sm3$\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92", "*0"},
  {"$sm3$\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93", "*0"},
  {"$sm3$\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94", "*0"},
  {"$sm3$\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95", "*0"},
  {"$sm3$\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96", "*0"},
  {"$sm3$\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97", "*0"},
  {"$sm3$\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98", "*0"},
  {"$sm3$\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99", "*0"},
  {"$sm3$\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a", "*0"},
  {"$sm3$\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b", "*0"},
  {"$sm3$\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c", "*0"},
  {"$sm3$\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d", "*0"},
  {"$sm3$\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e", "*0"},
  {"$sm3$\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f", "*0"},
  {"$sm3$\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0", "*0"},
  {"$sm3$\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1", "*0"},
  {"$sm3$\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2", "*0"},
  {"$sm3$\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3", "*0"},
  {"$sm3$\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4", "*0"},
  {"$sm3$\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5", "*0"},
  {"$sm3$\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6", "*0"},
  {"$sm3$\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7", "*0"},
  {"$sm3$\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8", "*0"},
  {"$sm3$\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9", "*0"},
  {"$sm3$\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa", "*0"},
  {"$sm3$\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab", "*0"},
  {"$sm3$\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac", "*0"},
  {"$sm3$\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad", "*0"},
  {"$sm3$\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae", "*0"},
  {"$sm3$\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf", "*0"},
  {"$sm3$\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0", "*0"},
  {"$sm3$\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1", "*0"},
  {"$sm3$\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2", "*0"},
  {"$sm3$\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3", "*0"},
  {"$sm3$\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4", "*0"},
  {"$sm3$\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5", "*0"},
  {"$sm3$\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6", "*0"},
  {"$sm3$\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7", "*0"},
  {"$sm3$\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8", "*0"},
  {"$sm3$\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9", "*0"},
  {"$sm3$\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba", "*0"},
  {"$sm3$\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb", "*0"},
  {"$sm3$\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc", "*0"},
  {"$sm3$\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd", "*0"},
  {"$sm3$\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe", "*0"},
  {"$sm3$\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf", "*0"},
  {"$sm3$\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0", "*0"},
  {"$sm3$\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1", "*0"},
  {"$sm3$\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2", "*0"},
  {"$sm3$\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3", "*0"},
  {"$sm3$\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4", "*0"},
  {"$sm3$\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5", "*0"},
  {"$sm3$\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6", "*0"},
  {"$sm3$\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7", "*0"},
  {"$sm3$\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8", "*0"},
  {"$sm3$\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9", "*0"},
  {"$sm3$\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca", "*0"},
  {"$sm3$\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb", "*0"},
  {"$sm3$\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc", "*0"},
  {"$sm3$\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd", "*0"},
  {"$sm3$\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce", "*0"},
  {"$sm3$\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf", "*0"},
  {"$sm3$\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0", "*0"},
  {"$sm3$\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1", "*0"},
  {"$sm3$\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2", "*0"},
  {"$sm3$\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3", "*0"},
  {"$sm3$\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4", "*0"},
  {"$sm3$\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5", "*0"},
  {"$sm3$\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6", "*0"},
  {"$sm3$\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7", "*0"},
  {"$sm3$\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8", "*0"},
  {"$sm3$\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9", "*0"},
  {"$sm3$\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda", "*0"},
  {"$sm3$\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb", "*0"},
  {"$sm3$\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc", "*0"},
  {"$sm3$\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd", "*0"},
  {"$sm3$\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde", "*0"},
  {"$sm3$\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf", "*0"},
  {"$sm3$\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0", "*0"},
  {"$sm3$\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1", "*0"},
  {"$sm3$\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2", "*0"},
  {"$sm3$\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3", "*0"},
  {"$sm3$\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4", "*0"},
  {"$sm3$\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5", "*0"},
  {"$sm3$\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6", "*0"},
  {"$sm3$\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7", "*0"},
  {"$sm3$\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8", "*0"},
  {"$sm3$\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9", "*0"},
  {"$sm3$\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea", "*0"},
  {"$sm3$\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb", "*0"},
  {"$sm3$\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec", "*0"},
  {"$sm3$\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed", "*0"},
  {"$sm3$\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee", "*0"},
  {"$sm3$\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef", "*0"},
  {"$sm3$\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0", "*0"},
  {"$sm3$\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1", "*0"},
  {"$sm3$\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2", "*0"},
  {"$sm3$\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3", "*0"},
  {"$sm3$\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4", "*0"},
  {"$sm3$\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5", "*0"},
  {"$sm3$\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6", "*0"},
  {"$sm3$\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7", "*0"},
  {"$sm3$\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8", "*0"},
  {"$sm3$\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9", "*0"},
  {"$sm3$\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa", "*0"},
  {"$sm3$\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb", "*0"},
  {"$sm3$\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc", "*0"},
  {"$sm3$\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd", "*0"},
  {"$sm3$\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe", "*0"},
  {"$sm3$\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff", "*0"},
#endif
#if INCLUDE_sha256crypt
  {"$5$", "$5$$.9cmwUsB0JN26/tNlJo6GVzbqQBfZNC8a9LW5Oa2eb4"},
  {"$5$\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01", "*0"},
  {"$5$\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02", "*0"},
  {"$5$\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03", "*0"},
  {"$5$\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04", "*0"},
  {"$5$\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05", "*0"},
  {"$5$\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06", "*0"},
  {"$5$\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07", "*0"},
  {"$5$\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08", "*0"},
  {"$5$\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09", "*0"},
  {"$5$\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a", "*0"},
  {"$5$\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b", "*0"},
  {"$5$\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c", "*0"},
  {"$5$\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d", "*0"},
  {"$5$\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e", "*0"},
  {"$5$\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f", "*0"},
  {"$5$\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10", "*0"},
  {"$5$\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11", "*0"},
  {"$5$\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12", "*0"},
  {"$5$\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13", "*0"},
  {"$5$\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14", "*0"},
  {"$5$\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15", "*0"},
  {"$5$\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16", "*0"},
  {"$5$\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17", "*0"},
  {"$5$\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18", "*0"},
  {"$5$\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19", "*0"},
  {"$5$\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a", "*0"},
  {"$5$\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b", "*0"},
  {"$5$\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c", "*0"},
  {"$5$\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d", "*0"},
  {"$5$\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e", "*0"},
  {"$5$\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f", "*0"},
  {"$5$                ", "*0"},
  {"$5$!!!!!!!!!!!!!!!!", "*0"},
  {"$5$\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"", "$5$\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"$IcbPfVlA.OKFHHR2I4noozy9uUvPh4YQQ5sHzvnG7A6"},
  {"$5$################", "$5$################$tZlGRcmONL7295tQG1dOXPeIiDTMFpQ5H7Bde1ImpB7"},
  {"$5$$$$$$$$$$$$$$$$$", "$5$$.9cmwUsB0JN26/tNlJo6GVzbqQBfZNC8a9LW5Oa2eb4"},
  {"$5$%%%%%%%%%%%%%%%%", "$5$%%%%%%%%%%%%%%%%$YHB.sYi3as/2N2AwT0I7N9J.EGSQiamsBQjp8j6lSoB"},
  {"$5$&&&&&&&&&&&&&&&&", "$5$&&&&&&&&&&&&&&&&$4oI9Wfty8ldlgbZHaRMQy6MBtO./RPe5BzNdYFvsRm7"},
  {"$5$''''''''''''''''", "$5$''''''''''''''''$4fEpjZD1/JVDY7kCN5WMChbLbzurZm8HYJTIABoCYUB"},
  {"$5$((((((((((((((((", "$5$(((((((((((((((($Z2HXAU0EkiVyIOzHbutfuKpYET4rVnN9WGrY0B1UVt2"},
  {"$5$))))))))))))))))", "$5$))))))))))))))))$s.vQuKjcyxh0GA4uT3NpzutBhYefNl3XzhZ8S6HFxrC"},
  {"$5$****************", "*0"},
  {"$5$++++++++++++++++", "$5$++++++++++++++++$1KHSxdU2GMHr.JMUPLXgTt2c4ecpTPEIcF3gyli/qp8"},
  {"$5$,,,,,,,,,,,,,,,,", "$5$,,,,,,,,,,,,,,,,$7KkI4Coa5cA4yZkhPJiIPFi88djn2RdZIdw4uyqQEUC"},
  {"$5$----------------", "$5$----------------$mSaWjPpj2YF659UA/tuqnlyubF4dk7jvQegmE0rWA82"},
  {"$5$................", "$5$................$W/v8EEWv1B6Dve1bi0HbAyllF5g4TkxQH2.xYrdxN.7"},
  {"$5$////////////////", "$5$////////////////$33BfmIgmhgRtT2vlfY3SRqs7lTi3brXBU0FIy8FaAn9"},
  {"$5$0000000000000000", "$5$0000000000000000$eR21UOvSFdYsepsAlt3eZMsjRy8T9ScbPpuKIShlivB"},
  {"$5$1111111111111111", "$5$1111111111111111$bySBjJ4r0QdUw/fgomcB7uzl4LjvJ65XyOyMdtHZxQ3"},
  {"$5$2222222222222222", "$5$2222222222222222$yuZndC7TYE2j6FuS0bEJsX92Iu5iWdaHMX6l9eAEYW8"},
  {"$5$3333333333333333", "$5$3333333333333333$xA0Pfyll.4Vzhn9l3V/aBht2p.AVY3QiZUEuGIC19W0"},
  {"$5$4444444444444444", "$5$4444444444444444$vIya5BnS8SPjCiEA7gcEAa.TcuMcVu3MtuLcx2GIcA3"},
  {"$5$5555555555555555", "$5$5555555555555555$jDmTar0YvVlBA5vVNpoJnWs3068L7jO9ttRmAU/FzR8"},
  {"$5$6666666666666666", "$5$6666666666666666$vSBmNVyAn14oQBjBkoF6Q48MTV4Bu5qW/fyre6L94kD"},
  {"$5$7777777777777777", "$5$7777777777777777$d5UbZ4jniwMil/pZXZM1EaM/JgLzZPosbP85sU3A9c6"},
  {"$5$8888888888888888", "$5$8888888888888888$i0dOqdLRFCW6tofALeQlF2gjZVCWQAej6NdQDyuLNJ1"},
  {"$5$9999999999999999", "$5$9999999999999999$CEmRQOHPWAwLCZI3XnswPssRh9Tpkx4dVwIiIwnGsvD"},
  {"$5$::::::::::::::::", "*0"},
  {"$5$;;;;;;;;;;;;;;;;", "*0"},
  {"$5$<<<<<<<<<<<<<<<<", "$5$<<<<<<<<<<<<<<<<$E0pdB/7wA41LmFDfODIv7TBUlX5eXIMpxAl3yG8fnUA"},
  {"$5$================", "$5$================$E6DSf64H4ccRpeFzxY2ohyOgUmqISQaHQe/589h8ap7"},
  {"$5$>>>>>>>>>>>>>>>>", "$5$>>>>>>>>>>>>>>>>$bn5mLiXIVhqRLuFhx2DeSOGOP1kJjvVSa752SGiQN38"},
  {"$5$????????????????", "$5$????????????????$eoE8xbResh/24NojQoK.rtKkn273RYxIDUQxhnW6dq1"},
  {"$5$@@@@@@@@@@@@@@@@", "$5$@@@@@@@@@@@@@@@@$3i7VYDfPzWYla8vo11pZjLBDetTFv91Qthzp8F/D5I8"},
  {"$5$AAAAAAAAAAAAAAAA", "$5$AAAAAAAAAAAAAAAA$VUpd0T9ADzWfkEFV3opuuUXuOKQqK9z2PwhFhG1tbg."},
  {"$5$BBBBBBBBBBBBBBBB", "$5$BBBBBBBBBBBBBBBB$fa7J/F.WdiyMbAIcj5Fy3lYRNRtx2Z2Saluo3QNeq/1"},
  {"$5$CCCCCCCCCCCCCCCC", "$5$CCCCCCCCCCCCCCCC$N8Af2vgy2oUYPWsZJK4amWU6GEvgOzqn39AfWnGdnU5"},
  {"$5$DDDDDDDDDDDDDDDD", "$5$DDDDDDDDDDDDDDDD$cqbztx0HqFgxi2AkDyeW9VQDFTPJqgWAwf.otoqcNq8"},
  {"$5$EEEEEEEEEEEEEEEE", "$5$EEEEEEEEEEEEEEEE$zJZjssAORvqD1rBWcfFVXM1DW89W.R9yvKsdNODm3H7"},
  {"$5$FFFFFFFFFFFFFFFF", "$5$FFFFFFFFFFFFFFFF$9swLQspTseegU2GZSD5oAkdCj7qyiQ2Fp8Ol5Ofrs6A"},
  {"$5$GGGGGGGGGGGGGGGG", "$5$GGGGGGGGGGGGGGGG$JwKDv.wdK2EDvhCo/AL.Nkhjbp7Cq0V.IYtdIVmS/1C"},
  {"$5$HHHHHHHHHHHHHHHH", "$5$HHHHHHHHHHHHHHHH$1dByOvQtmP83itXuRwBRejTs911SoKAgGgQiPrkvqmD"},
  {"$5$IIIIIIIIIIIIIIII", "$5$IIIIIIIIIIIIIIII$Z/ldZHX2MiJHPnVcJCOY/AEjrWAG2to0RoT9qfIZsRD"},
  {"$5$JJJJJJJJJJJJJJJJ", "$5$JJJJJJJJJJJJJJJJ$fhwQW6rdMJary.ixfiX/Dt9hmCpd0996dALH78CJ8sC"},
  {"$5$KKKKKKKKKKKKKKKK", "$5$KKKKKKKKKKKKKKKK$HHwts31Cns.ctW5icGIsxZl4AW0Gytgt..vdo91dHm2"},
  {"$5$LLLLLLLLLLLLLLLL", "$5$LLLLLLLLLLLLLLLL$d28JjgKAP3qlPfZYJs3hdSzaB5J7K.V06pLkuojR7f0"},
  {"$5$MMMMMMMMMMMMMMMM", "$5$MMMMMMMMMMMMMMMM$P4Wo0Hj8GL1q9oT.zJMhChwwsbGDLle6WfY4Lo8uPi3"},
  {"$5$NNNNNNNNNNNNNNNN", "$5$NNNNNNNNNNNNNNNN$YlRLxmjBFuF5g0XhKJBHaXnUdyEWK7OKW/HGcLMwcG3"},
  {"$5$OOOOOOOOOOOOOOOO", "$5$OOOOOOOOOOOOOOOO$DRn.JeS9KRIyaJ1Y7YBd3r6uoBjqzdL4yXYUI3XfvC0"},
  {"$5$PPPPPPPPPPPPPPPP", "$5$PPPPPPPPPPPPPPPP$pMCZDx.QilOTudd58i6D2rcQGZxBb9xsrnONsSN6UdA"},
  {"$5$QQQQQQQQQQQQQQQQ", "$5$QQQQQQQQQQQQQQQQ$iPnfE5yCEEHvOaGPmqOGiOMrx2G7BAyj45GbI98xX.7"},
  {"$5$RRRRRRRRRRRRRRRR", "$5$RRRRRRRRRRRRRRRR$0RG0Haiy4MIGpX1Cb6m0VxK5SgqWU0APzN/yxKiH5r5"},
  {"$5$SSSSSSSSSSSSSSSS", "$5$SSSSSSSSSSSSSSSS$JUGG5tCs4CVVQK98ab4wBR5at/yesizfu9eod.b5LhA"},
  {"$5$TTTTTTTTTTTTTTTT", "$5$TTTTTTTTTTTTTTTT$A0mm2hkjlmS/4N74qNPtQM7XaVCkHhkbTpvjXBLwQL1"},
  {"$5$UUUUUUUUUUUUUUUU", "$5$UUUUUUUUUUUUUUUU$ax8OTuPrxaBU76nbA7MdLvRD5Ghybn.kZdAU7gjcrU4"},
  {"$5$VVVVVVVVVVVVVVVV", "$5$VVVVVVVVVVVVVVVV$OQehmGBGe/3ZIQZ9aWOKr2CttBP8SDx0K3U2bqJ85/1"},
  {"$5$WWWWWWWWWWWWWWWW", "$5$WWWWWWWWWWWWWWWW$EjlH0vYOYZhvVhWruuWqpLC0t2NJzdq5zlnGoIX2bOB"},
  {"$5$XXXXXXXXXXXXXXXX", "$5$XXXXXXXXXXXXXXXX$pVapyZeNqGWfLPAvds.35hdaULvCehrErqrZmlDrGI7"},
  {"$5$YYYYYYYYYYYYYYYY", "$5$YYYYYYYYYYYYYYYY$bqL8kl.IRvZ6EH2j1jxMX.N5Jycve5I1QF3eYV1sTk1"},
  {"$5$ZZZZZZZZZZZZZZZZ", "$5$ZZZZZZZZZZZZZZZZ$BpQW6V.Sned2/ey686Jf/9QyWsal.H7xZ4JZdKvtSf."},
  {"$5$[[[[[[[[[[[[[[[[", "$5$[[[[[[[[[[[[[[[[$Kxx8t7j2tckrKdkQcne.g8WeieU9oanqykH7unPFC8D"},
  {"$5$\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\", "*0"},
  {"$5$]]]]]]]]]]]]]]]]", "$5$]]]]]]]]]]]]]]]]$v/dsYHRr8uSei9wHdHYKgR1kE6/ssXtQe.Cj5w7pnJ7"},
  {"$5$^^^^^^^^^^^^^^^^", "$5$^^^^^^^^^^^^^^^^$p2NLBCuJP2PuLLKYazCIVBHi0deXJSG6MIhK37HqC/D"},
  {"$5$________________", "$5$________________$AHciRH92K4w7nTLbBn1tZq2wvR9M6LjS2UYbYXCTis1"},
  {"$5$````````````````", "$5$````````````````$xYEXvr4fcBaOc3vrrlggJulFVQc9F5Y90nV.vDS9v0C"},
  {"$5$aaaaaaaaaaaaaaaa", "$5$aaaaaaaaaaaaaaaa$xOxMjsC8uGeIeuWiWjuqSpfwUDrq4KGPdOTKyd0KfJ8"},
  {"$5$bbbbbbbbbbbbbbbb", "$5$bbbbbbbbbbbbbbbb$Jb7o7yezHNA9X2ZGhqCSWdPWEJ3D7k/VsuKhMGN1IL5"},
  {"$5$cccccccccccccccc", "$5$cccccccccccccccc$9/rqDusRqEopAIW.t5dYMmgNvmnk5WI2G2L8WrmvOJ1"},
  {"$5$dddddddddddddddd", "$5$dddddddddddddddd$98UTIoNQDn5V2pNRK9RuKVDExLOFmnhHXB9OnJX9Vt."},
  {"$5$eeeeeeeeeeeeeeee", "$5$eeeeeeeeeeeeeeee$yH9ZoFyknDt6Dc05jg7MBeeE0XMPxOsTMV5vSlUZYX8"},
  {"$5$ffffffffffffffff", "$5$ffffffffffffffff$ZQI3fYusr0D2ZkXM.x3LR9Uww3v3RmMvQGLS/43EFnA"},
  {"$5$gggggggggggggggg", "$5$gggggggggggggggg$/UJVmABMZxxloTI7Z.TNEtPiejlmWcPH9Q4xcbP93K8"},
  {"$5$hhhhhhhhhhhhhhhh", "$5$hhhhhhhhhhhhhhhh$yHqyY4G5Gq966erkXmBaw1rO6ir.Fw80Ex1dQUHHCE8"},
  {"$5$iiiiiiiiiiiiiiii", "$5$iiiiiiiiiiiiiiii$nLWcRRmLTRGJ.dKBRCtRKukfCfCceGq7PMtsR4rNrQ5"},
  {"$5$jjjjjjjjjjjjjjjj", "$5$jjjjjjjjjjjjjjjj$ZLZfaKGuAqH1p5KhJ8..oThpB1wXZsu8SYO6sg5Dc.3"},
  {"$5$kkkkkkkkkkkkkkkk", "$5$kkkkkkkkkkkkkkkk$Z120Z4aa9ajtl8YN.OXSbS87Z4XCkP5LEX/tdDxDBY1"},
  {"$5$llllllllllllllll", "$5$llllllllllllllll$ExDgrdZcQuwFAqfWwIXwU8FO9IoQYGh.paiXLnksHR9"},
  {"$5$mmmmmmmmmmmmmmmm", "$5$mmmmmmmmmmmmmmmm$JaWuf4lmgatfz4GWMRhkNmodXwuwx2HLGbqajxxI2H7"},
  {"$5$nnnnnnnnnnnnnnnn", "$5$nnnnnnnnnnnnnnnn$HNrcFfckFkhxPuf65oiVAeVZ4ReR7cT5n3cx5bBQZgA"},
  {"$5$oooooooooooooooo", "$5$oooooooooooooooo$g6DVh2i3Mv1lVu4tbRjg.voWFRVDFvdVwjHlNBS2p/6"},
  {"$5$pppppppppppppppp", "$5$pppppppppppppppp$n45WFITWV.YlZafkp4giNBbFJ7E0K3hk.4ATQQWmpfB"},
  {"$5$qqqqqqqqqqqqqqqq", "$5$qqqqqqqqqqqqqqqq$aZlTD3j/FbEH7JtiYQzNWqHMpsrzOMGdi7NYwa3R5q/"},
  {"$5$rrrrrrrrrrrrrrrr", "$5$rrrrrrrrrrrrrrrr$NZUVfiRHu8kdnjiwUiMdBBZ.DoJCAO4STgltaUv0p.B"},
  {"$5$ssssssssssssssss", "$5$ssssssssssssssss$onSPRcYGEGSFWK.PtbsWLuKL9MLoSTXcWMp989oUdu9"},
  {"$5$tttttttttttttttt", "$5$tttttttttttttttt$evmPxXoulgYCuuPtM0RsqUDlbQcY5VyALX4YB/jzuZC"},
  {"$5$uuuuuuuuuuuuuuuu", "$5$uuuuuuuuuuuuuuuu$zA7OLW4DbHif04U.BxAOsmKQIGOszsDFpgKs4/lDbnC"},
  {"$5$vvvvvvvvvvvvvvvv", "$5$vvvvvvvvvvvvvvvv$ByQtI4ZSnpYYeycSmlI7X0KOWin/frKXnkGvPbkrH/8"},
  {"$5$wwwwwwwwwwwwwwww", "$5$wwwwwwwwwwwwwwww$rYx8MlvDNcbwsDWkPSyglFUbjuTbedp5Ud5px3OXjQ/"},
  {"$5$xxxxxxxxxxxxxxxx", "$5$xxxxxxxxxxxxxxxx$LOl2z2Jw4dWgqW4ukK.IcVS9REnxkVPpMm/vFKcrnO9"},
  {"$5$yyyyyyyyyyyyyyyy", "$5$yyyyyyyyyyyyyyyy$ubCS2POH1zfaPYiEnKZTnkwLqjisxunxmplxwNkG/m/"},
  {"$5$zzzzzzzzzzzzzzzz", "$5$zzzzzzzzzzzzzzzz$lPqbXBcwOQ81XXT7xKjJIGVr4ChEWRmrZdHgv0am1v7"},
  {"$5${{{{{{{{{{{{{{{{", "$5${{{{{{{{{{{{{{{{$ZJN/mpIRViDP0XULRTfFiemdciu8dAnbkcOQfLF0YD0"},
  {"$5$||||||||||||||||", "$5$||||||||||||||||$0wYgvq83T4NnAIcE.rtIiEZqr75cYFLFIHaiYF6eNG2"},
  {"$5$}}}}}}}}}}}}}}}}", "$5$}}}}}}}}}}}}}}}}$CFAky4H9RLcJjTh0wClEOdEVQiL/mHu9i6ogNXFCA0B"},
  {"$5$~~~~~~~~~~~~~~~~", "$5$~~~~~~~~~~~~~~~~$AT92oDPwlSlTL9uTxyAG1mEs.9iJYpKvuNUKzoJIyIA"},
  {"$5$\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f", "*0"},
  {"$5$\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80", "*0"},
  {"$5$\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81", "*0"},
  {"$5$\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82", "*0"},
  {"$5$\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83", "*0"},
  {"$5$\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84", "*0"},
  {"$5$\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85", "*0"},
  {"$5$\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86", "*0"},
  {"$5$\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87", "*0"},
  {"$5$\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88", "*0"},
  {"$5$\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89", "*0"},
  {"$5$\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a", "*0"},
  {"$5$\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b", "*0"},
  {"$5$\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c", "*0"},
  {"$5$\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d", "*0"},
  {"$5$\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e", "*0"},
  {"$5$\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f", "*0"},
  {"$5$\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90", "*0"},
  {"$5$\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91", "*0"},
  {"$5$\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92", "*0"},
  {"$5$\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93", "*0"},
  {"$5$\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94", "*0"},
  {"$5$\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95", "*0"},
  {"$5$\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96", "*0"},
  {"$5$\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97", "*0"},
  {"$5$\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98", "*0"},
  {"$5$\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99", "*0"},
  {"$5$\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a", "*0"},
  {"$5$\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b", "*0"},
  {"$5$\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c", "*0"},
  {"$5$\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d", "*0"},
  {"$5$\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e", "*0"},
  {"$5$\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f", "*0"},
  {"$5$\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0", "*0"},
  {"$5$\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1", "*0"},
  {"$5$\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2", "*0"},
  {"$5$\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3", "*0"},
  {"$5$\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4", "*0"},
  {"$5$\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5", "*0"},
  {"$5$\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6", "*0"},
  {"$5$\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7", "*0"},
  {"$5$\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8", "*0"},
  {"$5$\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9", "*0"},
  {"$5$\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa", "*0"},
  {"$5$\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab", "*0"},
  {"$5$\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac", "*0"},
  {"$5$\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad", "*0"},
  {"$5$\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae", "*0"},
  {"$5$\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf", "*0"},
  {"$5$\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0", "*0"},
  {"$5$\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1", "*0"},
  {"$5$\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2", "*0"},
  {"$5$\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3", "*0"},
  {"$5$\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4", "*0"},
  {"$5$\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5", "*0"},
  {"$5$\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6", "*0"},
  {"$5$\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7", "*0"},
  {"$5$\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8", "*0"},
  {"$5$\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9", "*0"},
  {"$5$\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba", "*0"},
  {"$5$\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb", "*0"},
  {"$5$\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc", "*0"},
  {"$5$\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd", "*0"},
  {"$5$\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe", "*0"},
  {"$5$\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf", "*0"},
  {"$5$\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0", "*0"},
  {"$5$\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1", "*0"},
  {"$5$\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2", "*0"},
  {"$5$\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3", "*0"},
  {"$5$\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4", "*0"},
  {"$5$\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5", "*0"},
  {"$5$\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6", "*0"},
  {"$5$\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7", "*0"},
  {"$5$\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8", "*0"},
  {"$5$\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9", "*0"},
  {"$5$\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca", "*0"},
  {"$5$\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb", "*0"},
  {"$5$\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc", "*0"},
  {"$5$\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd", "*0"},
  {"$5$\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce", "*0"},
  {"$5$\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf", "*0"},
  {"$5$\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0", "*0"},
  {"$5$\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1", "*0"},
  {"$5$\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2", "*0"},
  {"$5$\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3", "*0"},
  {"$5$\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4", "*0"},
  {"$5$\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5", "*0"},
  {"$5$\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6", "*0"},
  {"$5$\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7", "*0"},
  {"$5$\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8", "*0"},
  {"$5$\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9", "*0"},
  {"$5$\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda", "*0"},
  {"$5$\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb", "*0"},
  {"$5$\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc", "*0"},
  {"$5$\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd", "*0"},
  {"$5$\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde", "*0"},
  {"$5$\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf", "*0"},
  {"$5$\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0", "*0"},
  {"$5$\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1", "*0"},
  {"$5$\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2", "*0"},
  {"$5$\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3", "*0"},
  {"$5$\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4", "*0"},
  {"$5$\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5", "*0"},
  {"$5$\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6", "*0"},
  {"$5$\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7", "*0"},
  {"$5$\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8", "*0"},
  {"$5$\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9", "*0"},
  {"$5$\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea", "*0"},
  {"$5$\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb", "*0"},
  {"$5$\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec", "*0"},
  {"$5$\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed", "*0"},
  {"$5$\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee", "*0"},
  {"$5$\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef", "*0"},
  {"$5$\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0", "*0"},
  {"$5$\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1", "*0"},
  {"$5$\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2", "*0"},
  {"$5$\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3", "*0"},
  {"$5$\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4", "*0"},
  {"$5$\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5", "*0"},
  {"$5$\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6", "*0"},
  {"$5$\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7", "*0"},
  {"$5$\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8", "*0"},
  {"$5$\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9", "*0"},
  {"$5$\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa", "*0"},
  {"$5$\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb", "*0"},
  {"$5$\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc", "*0"},
  {"$5$\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd", "*0"},
  {"$5$\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe", "*0"},
  {"$5$\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff", "*0"},
#endif
#if INCLUDE_sha512crypt
  {"$6$", "$6$$t2FbUVbthAvIU7XM/8/NizHqHc/gcVkV6TK7SSXPCv.uHfMVKKmKs1ezXF8vtKC6ACvY0g/lmErEnI4Jy8WY9/"},
  {"$6$\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01\x01", "*0"},
  {"$6$\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02\x02", "*0"},
  {"$6$\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03\x03", "*0"},
  {"$6$\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04", "*0"},
  {"$6$\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05\x05", "*0"},
  {"$6$\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06\x06", "*0"},
  {"$6$\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07\x07", "*0"},
  {"$6$\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08\x08", "*0"},
  {"$6$\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09\x09", "*0"},
  {"$6$\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a\x0a", "*0"},
  {"$6$\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b", "*0"},
  {"$6$\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c", "*0"},
  {"$6$\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d\x0d", "*0"},
  {"$6$\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e\x0e", "*0"},
  {"$6$\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f", "*0"},
  {"$6$\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10\x10", "*0"},
  {"$6$\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11\x11", "*0"},
  {"$6$\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12\x12", "*0"},
  {"$6$\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13\x13", "*0"},
  {"$6$\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14\x14", "*0"},
  {"$6$\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15\x15", "*0"},
  {"$6$\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16\x16", "*0"},
  {"$6$\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17\x17", "*0"},
  {"$6$\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18\x18", "*0"},
  {"$6$\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19\x19", "*0"},
  {"$6$\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a\x1a", "*0"},
  {"$6$\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b\x1b", "*0"},
  {"$6$\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c\x1c", "*0"},
  {"$6$\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d\x1d", "*0"},
  {"$6$\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e\x1e", "*0"},
  {"$6$\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f\x1f", "*0"},
  {"$6$                ", "*0"},
  {"$6$!!!!!!!!!!!!!!!!", "*0"},
  {"$6$\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"", "$6$\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"\"$y4ikCocrdkHNvDZKBGFtSBVNOKsVqCkyAbM40IRT0wRp8zkgj54tc07lPBRodv4Rpu6WT14CHJknrW/9fTwGZ/"},
  {"$6$################", "$6$################$0lXBH7m76aEK5kQdOwPfWOyYPemgcfRbu9Rw/9Bc7SRfFuMv3PZYue6Bn.t1SO.pku7tne.AY0icRIfr2nHuG."},
  {"$6$$$$$$$$$$$$$$$$$", "$6$$t2FbUVbthAvIU7XM/8/NizHqHc/gcVkV6TK7SSXPCv.uHfMVKKmKs1ezXF8vtKC6ACvY0g/lmErEnI4Jy8WY9/"},
  {"$6$%%%%%%%%%%%%%%%%", "$6$%%%%%%%%%%%%%%%%$k9bX/iMS0GLlfZjzuP/ONSSo7dlEzqwGY2aY1jm2A6ZJBIadOxXlBJjNFjHOQ.fa.PIIyRssqGTUTFpTOtztU1"},
  {"$6$&&&&&&&&&&&&&&&&", "$6$&&&&&&&&&&&&&&&&$skVA0eRt3uM2yrsQHaYxEacC63wF9/99RUp7Pnk8aQh9CCXSpd0l3CM9g8.cwCcrRSMfMod0oO2EnAoXnoNJd1"},
  {"$6$''''''''''''''''", "$6$''''''''''''''''$qOfgIMYACfO5W.FEno/Z8JEU9ixO70Sisy.ZyC27.hNCauDc1UssoMAn3S5ppnHzKNwL7M0CAFzMjKt2XDQK4."},
  {"$6$((((((((((((((((", "$6$(((((((((((((((($kJwg/5cH1DxiDMVw9/UB0VpeuqWv8vxLUZSQUhWxs0ND4.olXVb9wg9USMBSv0mXmzDMuTG0N/O1nFHFiEyr.."},
  {"$6$))))))))))))))))", "$6$))))))))))))))))$S5sVQD7UjSWaFs4gGijIm4bs/DH.DVe2G8LuckweSH5s2YkGD.EOoVSSKJ7UYb3J2IvyqonYSHMJyiitzvRxx/"},
  {"$6$****************", "*0"},
  {"$6$++++++++++++++++", "$6$++++++++++++++++$pMPSwZco12aGZGv42JNb7DhWN02wliGAmvO/W5c5quH/Gx/gMDuIyhmVB3FEFMesO8/Kg0Z93FBwWs9S7mdlX1"},
  {"$6$,,,,,,,,,,,,,,,,", "$6$,,,,,,,,,,,,,,,,$r2Cyr1grJ01AvsYn8Zljv4YLt6.mO0vjDwAD4Guc1GTXurKMF1VvSKTIG7eiB4xNPaHmVqVouXcxYYNFo3cZV1"},
  {"$6$----------------", "$6$----------------$a1D5NaZ7Jd1t9/5tKCN/cgmdoFqw0F7PdyTwTLjAh0m07Bg.F/t4CdxKznMRVyvIzBcPuYy8VHnJD7IFxR8Bj."},
  {"$6$................", "$6$................$xkyddLV1z75ZjVK8QlC9547T8tsNpiZWjHJvkSWVOM.Ao/4qwd.c/TtoSJ0MmwyN7blC5rzoG59MdcceI7V/m0"},
  {"$6$////////////////", "$6$////////////////$89PmHUAIFKYJ7FKaiU7G5tAtQ/pN2.Y6ix0dvbcQzSELt3ansXuGJS7D3JqYukb9JS/lSVeYoIBpPUKc5UF6D1"},
  {"$6$0000000000000000", "$6$0000000000000000$MZwTpzth7AFVFvmYJhgKVvtXQ3q665hbFB0XNT6x1zZgkik0uMp3TdlNEbkCggqbGi0CRCpL80X50I9MIYV.R0"},
  {"$6$1111111111111111", "$6$1111111111111111$B/qsncOwVkh1Y/QsBD7QNrfSfwoIPZKEWVdtRfydZk0Z1l2M1sfHqRCxyHUBTVC8JJI3JwqePZor8H75qYzHQ/"},
  {"$6$2222222222222222", "$6$2222222222222222$OYIMW2GpZF.u5iY2JkJn1q0Be3L1RU4vKvBzJoGaVapqE.notZ0X6YUVFBMIY1IgSu1gsFaCS6VupS4yhPlu5."},
  {"$6$3333333333333333", "$6$3333333333333333$qAl/VDu4THkJPwesyyEQMHj9ZJlNCrTckx.wlQOuFpO3AP1v0yaqlKxeLKr8/A7/tovJuHjrLvOP.D39jaLWQ0"},
  {"$6$4444444444444444", "$6$4444444444444444$rCmiLxgLMGvUYPULk.vYKWGbWAiWDzEfRqCUenVLECSOjseJ9jP6pxEiDOuw/owCuH6/miUJGwoQOat5ENm/b/"},
  {"$6$5555555555555555", "$6$5555555555555555$MTaDmYPznCdZTP9fShnjNWJsdsPaMmJfHx9NXzGpiVZz0aqpF2vYw.tXXz8wYHLDpZthqpcqppKb83d42UKn3/"},
  {"$6$6666666666666666", "$6$6666666666666666$mzCKrq0VrJQT0iLeFw8Ff4PXyS9no98H0Rr5zAllcdGC7KFHlVTcTLvaEfSivxHqsAcP16WTn9MRIbJgPINfg1"},
  {"$6$7777777777777777", "$6$7777777777777777$UHZrbj4F.gdace5WIrAmGaqE1XBBG5WZX72C7i3YAR9jVv4dzjvbYOJUmYiAYafZ1WaIBidAsZas8.8xCOM4N/"},
  {"$6$8888888888888888", "$6$8888888888888888$oJQcSrcBjvsVObW7M9aW9KVDf4nUzfALbBKuU0znYkqfdr3ms1Q4F4uF0p1tmc5.AFXqSK5XNu4bZ3RMMywNX/"},
  {"$6$9999999999999999", "$6$9999999999999999$lShnYSRrSGXk/pMhUK2qqlXYB7vQi6COb.YT2/ZvyBmY/RCDCt9oBXi7kZpX8FxUIRC2D9lOFtxWpbLIbLJFM1"},
  {"$6$::::::::::::::::", "*0"},
  {"$6$;;;;;;;;;;;;;;;;", "*0"},
  {"$6$<<<<<<<<<<<<<<<<", "$6$<<<<<<<<<<<<<<<<$IzHC9GvicnkYsJPc76ZnI/QS7/me9q5/kVLmIlWZzIUo3zpliZ0svdxvb73BM7WhAcfIfv7hvTCIt8/MsJ1iY."},
  {"$6$================", "$6$================$ks8n9O0rxQI408VzbA5KlSkpFwJjQyKg4owCQeHBvLs/TfiNZQwp75aZA6Ne6xIs1LIlnMrNhfu34hcwpALWX1"},
  {"$6$>>>>>>>>>>>>>>>>", "$6$>>>>>>>>>>>>>>>>$ae3ShFy5aIq/ozlB3UBwZlm7P8AKn0DQqZ9oMyA0q4aIjLOYpIKNglhKLU.KY4kTX/zPHLe9pLis.B89DwLgR."},
  {"$6$????????????????", "$6$????????????????$TtylS7/us9mPCLMzAF4kb.DN1hTnlB/OXKcsdFcoMDt9JC9m7C7L7T5nlti8OF9U0E16NrqkcaCfChpoSeO1r/"},
  {"$6$@@@@@@@@@@@@@@@@", "$6$@@@@@@@@@@@@@@@@$QEjlvm6.tPwaAws0U8bHfU/ElffdJOFZqAZ4HrT9a539EPJt.3TP6mL3eE.yQtRLIM4IKT3bAKCnBzeIH9XTp/"},
  {"$6$AAAAAAAAAAAAAAAA", "$6$AAAAAAAAAAAAAAAA$pmN8p/Q2O6DIubIh7Y7TqwUDoFMRFzRsBJMpqhAoa00YEy7I2TvxVg1VyVFOP0aOhLv5mTB4SfKnHPB0GH/Lc/"},
  {"$6$BBBBBBBBBBBBBBBB", "$6$BBBBBBBBBBBBBBBB$38jJoryxwIItUBBJLGPCifls9RzTVmapELwm91rwnY2qGLUGgV99mk6yXScu0P4mr7atgAd6mgHIP1O3t.wEx/"},
  {"$6$CCCCCCCCCCCCCCCC", "$6$CCCCCCCCCCCCCCCC$07sW36CtP8XueaTwHGtTMBOXrBew4VFm0jwj4i1FmyWFoRXA0vTbwG.3BZ840nvjo7reW4o96o2iW0oX6fMbY/"},
  {"$6$DDDDDDDDDDDDDDDD", "$6$DDDDDDDDDDDDDDDD$J5Kpz3nqJ3fSgwzYUsLoHsQyVe6ZKRXAkg4qJUKNM37C4CbNa2Kz/TgDVMfm6xB0P1QXHlueFjbW9mpLQu9oY."},
  {"$6$EEEEEEEEEEEEEEEE", "$6$EEEEEEEEEEEEEEEE$KLANbns1faG0VtS3NM7QYwUCV5YN/AiHd2KcHFKHUje0WaA7tiOkQgp58ppdw7dqYOwAt7BY0mCRdn90mA6mU."},
  {"$6$FFFFFFFFFFFFFFFF", "$6$FFFFFFFFFFFFFFFF$..8YbJ8DCIWBcNH5POWVhKEvcuz9cdtdOG/mbDPMQHhaU3DMfFv2NMIKnJChkmgbxhuoLKMOOCnv4lWV6PX44/"},
  {"$6$GGGGGGGGGGGGGGGG", "$6$GGGGGGGGGGGGGGGG$u2WsXEETmQghsHTQ0DaofeEUb3UBi572elJYhRqmXMpjPm/7j2hEf8Vijtj/ml8qPjcHCdLQuYYaAWBmcnCqs1"},
  {"$6$HHHHHHHHHHHHHHHH", "$6$HHHHHHHHHHHHHHHH$kPznwkW7ucXCzHtg9aaJ5asDnw48qKswyjLOSwmsXkF1Ugmv9508ORO/SXaHzYm377WQWE64bCPElYboD1dY90"},
  {"$6$IIIIIIIIIIIIIIII", "$6$IIIIIIIIIIIIIIII$vSJgk2qjP66NNpM/MB58K6VNYWfVN8G5JDMT9YRDj/CzlrD3PdJmfy.ILZxEZw/2Pph06t/H4pr4B92wdJabR1"},
  {"$6$JJJJJJJJJJJJJJJJ", "$6$JJJJJJJJJJJJJJJJ$NtS1J3/SVL1DjL6QQ7nDOCVUF/awovpSqLYQx4a/HXeZmFDxIE91Stg2oRr.TUsWPR3VXCIs9A12p3F1OoIM10"},
  {"$6$KKKKKKKKKKKKKKKK", "$6$KKKKKKKKKKKKKKKK$4df7QnhT78fHr3Yr9ez05FRQWzszzZPnkuG7gPnJmAG5ieH03zfywDJvOVEpdNxuKci/i5TVd6mJLJCdhABnj."},
  {"$6$LLLLLLLLLLLLLLLL", "$6$LLLLLLLLLLLLLLLL$kP9hFAAJGrjBWIK584qYGzRucH3V4ZAnw4uHb0z6ri94gut984Ulrqk9dlsWztXX2wQK3TwdgWmC0ncD/gXiL."},
  {"$6$MMMMMMMMMMMMMMMM", "$6$MMMMMMMMMMMMMMMM$GTXNzFI6VGkJhZkDt.IdPZBOb9TE1ETHJuNsF.BzvVSoFWlaIRHW5IHmbyqbg6eMlhXNTJ9fJ2WcG3uYG97WU."},
  {"$6$NNNNNNNNNNNNNNNN", "$6$NNNNNNNNNNNNNNNN$uPLWLmXuw7eA54ynnQXJ4DCrFLIvxucQfB3ZnD7joNB4USKcd9qD61Haa7dcIh8CqMCOQFJswZg/8s9txOi/e1"},
  {"$6$OOOOOOOOOOOOOOOO", "$6$OOOOOOOOOOOOOOOO$2WDhTCe.fILClG88sDZZEvowkFEwkeyjO56RQeOKGUrkwuO1VAXwOtiWNats56he9f7AJHKvGXkzq1GLwvT0x."},
  {"$6$PPPPPPPPPPPPPPPP", "$6$PPPPPPPPPPPPPPPP$CsGQkUS5yfDCvouG2ELPpyvb16PrVrjSbTFw/SAdno4xZRc4jMiDasW1xDMa8lOerVEyhRDeAfvZ0nc2ZCDpU/"},
  {"$6$QQQQQQQQQQQQQQQQ", "$6$QQQQQQQQQQQQQQQQ$9dq9MRwFZIneAVNNNo2DyI8/0uKrTKXv8iwfu43XbCFtW1YFk/6iW2.vpzeIoohz4hZHFiNsE8OCeLmDAmIpG/"},
  {"$6$RRRRRRRRRRRRRRRR", "$6$RRRRRRRRRRRRRRRR$GtSCVJx6/ZvUWrYpD9URyae061UswUDuqUeRaS2Z2ttyVSRzMQLjA6o7c3Dwkc2eGNPTSzpQUpvDdmqlZ4m2x1"},
  {"$6$SSSSSSSSSSSSSSSS", "$6$SSSSSSSSSSSSSSSS$H.Dn4giCeiyRBW5S83Nq9rireF/./RYP72WPzg3UCQHjhMO0zmLV8Li0xbeHJXwuCFxkCErhxGA7m82XV1s5T."},
  {"$6$TTTTTTTTTTTTTTTT", "$6$TTTTTTTTTTTTTTTT$A65fsPuGvM8KjuAxku5EIHGuS/hBJYQ0PUbofZffIEK02AB/uGOikl30WqOJpfFdKWZr9q0NTtO/4kCmOD58g."},
  {"$6$UUUUUUUUUUUUUUUU", "$6$UUUUUUUUUUUUUUUU$8sQbCkXeG8/3kuGmpbtvo15c.EYmQvqdE1j2kQ8vP3WLfS05yVVsEaytID2zJLBQpg9qNi2uy7KdrDTG0sciF1"},
  {"$6$VVVVVVVVVVVVVVVV", "$6$VVVVVVVVVVVVVVVV$Rm1be9Sg3miBmZGpMpCnbNY.Iyz8aug7ator7kgckT6e390jMS833.os8Y1S1mrxbYbwbzemxx8D5BfH7U5.b1"},
  {"$6$WWWWWWWWWWWWWWWW", "$6$WWWWWWWWWWWWWWWW$Kqmzc1SThYGjwmbcjZhVgzeSopzW9rIU/VbWW3pSb6TzvnvhDi9uUFZlPN.79tSF.DR77PXnP7Q4R0l1gMDPd0"},
  {"$6$XXXXXXXXXXXXXXXX", "$6$XXXXXXXXXXXXXXXX$KHkpQUvbQ95J.8y9iyeeAbduXiM8VSquW4auE0067SUtAIrYMVmvSrxTDoFFl1W03BLW6oT8m5FJbIe.HhrDi."},
  {"$6$YYYYYYYYYYYYYYYY", "$6$YYYYYYYYYYYYYYYY$x2WHwrNrf/qp4SEdZesD4JZpnf.DMmB5Zy3jC/GaYJHJb..ZwdA5Z3z7KqP.2uYJqfRoa/cpcWqB8ZZ9MvQrn0"},
  {"$6$ZZZZZZZZZZZZZZZZ", "$6$ZZZZZZZZZZZZZZZZ$xWF6lbDczxbjDeDQfHNabpEdqmOp1hu7IFHU3AW2uxK1Q55Qhrx0jK7yAM/CCU79dmT2MBMCJP/G2o6UBoRjR0"},
  {"$6$[[[[[[[[[[[[[[[[", "$6$[[[[[[[[[[[[[[[[$dBp6Z.qt5DOG3HcfXKIWCslDlNgvcM0AYfFKYdYlCUgqAh2/8Lc19yiAdVuWVj4avJtOVREeA2UHRZuiiUcFY1"},
  {"$6$\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\", "*0"},
  {"$6$]]]]]]]]]]]]]]]]", "$6$]]]]]]]]]]]]]]]]$Yvco5E.ZnJk70S1mBe7.ul9Qv93ZP3gtTk8qcYnKiWE9u8RGn2R5i3oBx/Dz0asccDNuYITN.RFBsKQeMYTQd1"},
  {"$6$^^^^^^^^^^^^^^^^", "$6$^^^^^^^^^^^^^^^^$nBQ4e7tRXokFDZNuFZUW0wO/kMWIFh6wZlBT4m/jNEu02G9SWFfb9q65H5tB81H9sGR0rzSXvqWvauO8GoL8S1"},
  {"$6$________________", "$6$________________$RCncF7GLNSTeiXTvScoU855UNcTWc0WCEeEql.uwz6aaIiPmPCBBmp.c8HJglimBDKvv7cJJpfhjiJnKOhKNi1"},
  {"$6$````````````````", "$6$````````````````$O0EzXiLYXQcd7rZauStyori1GHiAiGV.mkDQAYRPJMYvJiYnhYHtfoxMb2kkHk4DfCi9/vARhWYLMSQxQUY8X/"},
  {"$6$aaaaaaaaaaaaaaaa", "$6$aaaaaaaaaaaaaaaa$HvV5Pg0tE0/7mpvSbO0FUm.4xHi7r3JYligZBiGOmm926kVVg8VBJ4QgSrWVwXtqYpvBoJhtIe/HMB5wHQlTQ."},
  {"$6$bbbbbbbbbbbbbbbb", "$6$bbbbbbbbbbbbbbbb$GWA7ESqunT3P9cz8HyZWeTWcIh3DJT62eTczPylgaEEanRIOw5VvAgIj5dzsx9OQv/xL7jdxdXS7pH5nCHR0n1"},
  {"$6$cccccccccccccccc", "$6$cccccccccccccccc$1bZ8Z0n.1.fkTrGehCgnOw/QjuB9qist8yDhuqiKPaHTxa9iVk.1etQ0g3zNuiO8..T1wAfPuD5fy8kFAyoxC."},
  {"$6$dddddddddddddddd", "$6$dddddddddddddddd$4CHU4zgyc9iJdxLyXzCY8XzMzbvz23rD5vwRc4LOtVDjDgxavqkxgYDS71IFumL/qCK13.d9l/MgdUPnE29EW."},
  {"$6$eeeeeeeeeeeeeeee", "$6$eeeeeeeeeeeeeeee$sYcbpxDWs3TQhFudTzHlEzyoB07oZi1kcGnLJnCHU6OpxGinRAUjUVzVyTDWs9ZKTbCfeW9YXuSz87GGF6pAp/"},
  {"$6$ffffffffffffffff", "$6$ffffffffffffffff$MHt4IeZsFoNZXaibWBtHb/IQVUFoQ8GjayC3uQM0FWpGmtpRdnYcxa3mZJW8QGjOHhmYrG9zHca1/gxZBhmiA/"},
  {"$6$gggggggggggggggg", "$6$gggggggggggggggg$RvFpqAiZYpgEYKrqgB8TRcUm.HHf2no5IM26azGTIleRbZeCsZBhdiFhNYmTpjjmWZajmrePTCm/utW2dsQh3."},
  {"$6$hhhhhhhhhhhhhhhh", "$6$hhhhhhhhhhhhhhhh$LawQ9KCZkYaXmXGT1UcEsmOq/kx9WQx0MejadhNKj0seppOAoI1LNE5oBEdPOMOdFL6mahHIDsg3UyfuHuXF00"},
  {"$6$iiiiiiiiiiiiiiii", "$6$iiiiiiiiiiiiiiii$4ek2WaF1vMFgGI2IKd7aMrZN15iA3fNvLf94D/PF5S56Inv5TepuFG//ufoCFfRx6JMJ4ei5kbzgLNX7H4yq61"},
  {"$6$jjjjjjjjjjjjjjjj", "$6$jjjjjjjjjjjjjjjj$nyXrZM4DruCulpp.78f3BFjWA7W7fEuyqUrnFqeLdtEZeoIBnqULrw./8V0QT/QFiVf2XvGts9w042798vbsf1"},
  {"$6$kkkkkkkkkkkkkkkk", "$6$kkkkkkkkkkkkkkkk$B9elLzYl/KnzeKIQ4o.TtuQlPClwm64NT4O4Rf3..lo/mnDRQFGFEnVGPE5h16vUBrhnNZzMX7qyAE6BAfsbD1"},
  {"$6$llllllllllllllll", "$6$llllllllllllllll$4IoCGvQh3HA9EwqOHf4EhtpMLmvtiiu.NT0QGkiQPRlde8CeofwSNUjihD3e9kEwFKpdZKgyhYmIOZL5YuPkL."},
  {"$6$mmmmmmmmmmmmmmmm", "$6$mmmmmmmmmmmmmmmm$g2lab03qXNQrYZQglcjAgWmEbDw48yalW0YbmYF/ik77kxTQ5b.XUtHNZu5eWs9QBjv2u3epXAmDs.Nk1lKO0."},
  {"$6$nnnnnnnnnnnnnnnn", "$6$nnnnnnnnnnnnnnnn$hLDWmZssa25zOQAnoxETmCKtrsqxXw0sT01L1LGpn91WLqAyBJjNcN.Fol.uBcYvkUc.TPM4VjgYlsToykzwR0"},
  {"$6$oooooooooooooooo", "$6$oooooooooooooooo$HzxnSVJUvuc50Uy6PoZVQmVtgE/CZJ36mN8czr9lRUh3q.YCBNP0VPfc3DlmwGRavViNjELpYkqdjky6jZpsf/"},
  {"$6$pppppppppppppppp", "$6$pppppppppppppppp$/XQduHAyM3gF2BJIucWGPeL3uMguiuAoCoFk/r8yPhj/bdXF8oSzpto7dI/ToVrSWQSpq8Gbcq9TMpYMk2bcV1"},
  {"$6$qqqqqqqqqqqqqqqq", "$6$qqqqqqqqqqqqqqqq$Zf32Lr4ZWTzklzMD2URBKXarGQjqfPulqUCSJ/FNgCx/AFoLixjSxxGr/F54agkatM.yxN2b6r.4VaOgJzLp/."},
  {"$6$rrrrrrrrrrrrrrrr", "$6$rrrrrrrrrrrrrrrr$55E7q5YLvBVJ640e.dBckt2Yr010n3CSCNhErgsDtRUcxCQe5WPr/VvXHHadZzRBaA5Arx4QtCIqQAhfiouaL."},
  {"$6$ssssssssssssssss", "$6$ssssssssssssssss$mMOS7/bpCnU6q66dgucBA76GD5U76iQ1eH5Auqrp/o59TLK7vJToUY8syxe18ZCyEKXCB88LbldEgx316cD1I0"},
  {"$6$tttttttttttttttt", "$6$tttttttttttttttt$N1PIDJt9hUMyr1NlBAVBgP5S3xe6woFUik83Fo.LcQOgPQ0/fwH85zzpSrxPrGNmeO4XDw/mYx.QBskwA5qO21"},
  {"$6$uuuuuuuuuuuuuuuu", "$6$uuuuuuuuuuuuuuuu$UIy5pdkRqtXwwJHs07VSpdzJVB.JYuET/iDa6PECFUVyEYGEXyuuUse6umRKZOS54fXg78BTJxsPhMcjjkEzY."},
  {"$6$vvvvvvvvvvvvvvvv", "$6$vvvvvvvvvvvvvvvv$AYvRqfz59J233HXW71U4eEZv.HRAP9m5lALriFsY7hJg377G7Z2nu/7dQBGJZe76tM.yg89Ig4t1SGj/iTc/T/"},
  {"$6$wwwwwwwwwwwwwwww", "$6$wwwwwwwwwwwwwwww$VmS/m9M8S.v3sqmnHxtPzD4RjA6F/1qsU/GJQMelFVl1ItdKtSsgRsYyF3bUO9lOfX4XJK.avj/m2Ybnbd/.Z/"},
  {"$6$xxxxxxxxxxxxxxxx", "$6$xxxxxxxxxxxxxxxx$rBzF2pkrsh09n25F26pG3waeIV758datdZonnyi5CsFksOl4Y0fcNf34LGpVPj0s.kEZ04O.JPRdJdeiRo94I1"},
  {"$6$yyyyyyyyyyyyyyyy", "$6$yyyyyyyyyyyyyyyy$PDltzKb/ZJrjCZXYrUZ0YRzITKKkJdNOD7QUFrKsdUNjv6TWhKjb97HlN8Ia4BP4EDsHxkDaP2CGtd0k0TEGP."},
  {"$6$zzzzzzzzzzzzzzzz", "$6$zzzzzzzzzzzzzzzz$mRVbjzrCc9gglrQuHL2CLkt0jBd6IxGIHzGmdrufY1DS4L/KmixXJ1a0jagCWr6JbO0W2HbYzduVRBXvmbDz4/"},
  {"$6${{{{{{{{{{{{{{{{", "$6${{{{{{{{{{{{{{{{$wEw3hFJgTIwMkapwV81fJNoVx9iKzgR.ivUROgb8o/KMcMUP8JZvBFyfo1t6xFG7enHMwOKHVvYnhki/nKeHD."},
  {"$6$||||||||||||||||", "$6$||||||||||||||||$4JldrmpooVZmxae9lZKlE6PDLwMGIjvtUWp45S6Phl72H5YPfD7Uk/nND.Ksnesf6VPq/9W10Il2nmZ3WLit41"},
  {"$6$}}}}}}}}}}}}}}}}", "$6$}}}}}}}}}}}}}}}}$WE9yMayOYHdYipzj6cUqFAkoFLLQvK3eGdN.NUAiWVTfriv7aIG5D0MedHs89iiWl766S9UptxUbZubRetaXB/"},
  {"$6$~~~~~~~~~~~~~~~~", "$6$~~~~~~~~~~~~~~~~$cZRB//NhFoC73bB.bJn/djYKqtFn9r7VBtrdEeGbLS8082wZlOpUilIJAfOkNFk.LPQEAKsZ4hcrKzZdSp9vN0"},
  {"$6$\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f\x7f", "*0"},
  {"$6$\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80\x80", "*0"},
  {"$6$\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81\x81", "*0"},
  {"$6$\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82\x82", "*0"},
  {"$6$\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83\x83", "*0"},
  {"$6$\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84\x84", "*0"},
  {"$6$\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85\x85", "*0"},
  {"$6$\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86\x86", "*0"},
  {"$6$\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87\x87", "*0"},
  {"$6$\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88\x88", "*0"},
  {"$6$\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89\x89", "*0"},
  {"$6$\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a\x8a", "*0"},
  {"$6$\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b\x8b", "*0"},
  {"$6$\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c\x8c", "*0"},
  {"$6$\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d\x8d", "*0"},
  {"$6$\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e\x8e", "*0"},
  {"$6$\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f\x8f", "*0"},
  {"$6$\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90", "*0"},
  {"$6$\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91\x91", "*0"},
  {"$6$\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92\x92", "*0"},
  {"$6$\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93\x93", "*0"},
  {"$6$\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94\x94", "*0"},
  {"$6$\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95\x95", "*0"},
  {"$6$\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96\x96", "*0"},
  {"$6$\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97\x97", "*0"},
  {"$6$\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98\x98", "*0"},
  {"$6$\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99\x99", "*0"},
  {"$6$\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a\x9a", "*0"},
  {"$6$\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b\x9b", "*0"},
  {"$6$\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c\x9c", "*0"},
  {"$6$\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d\x9d", "*0"},
  {"$6$\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e\x9e", "*0"},
  {"$6$\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f\x9f", "*0"},
  {"$6$\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0\xa0", "*0"},
  {"$6$\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1\xa1", "*0"},
  {"$6$\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2", "*0"},
  {"$6$\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3", "*0"},
  {"$6$\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4\xa4", "*0"},
  {"$6$\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5", "*0"},
  {"$6$\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6\xa6", "*0"},
  {"$6$\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7\xa7", "*0"},
  {"$6$\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8\xa8", "*0"},
  {"$6$\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9\xa9", "*0"},
  {"$6$\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa", "*0"},
  {"$6$\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab\xab", "*0"},
  {"$6$\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac", "*0"},
  {"$6$\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad\xad", "*0"},
  {"$6$\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae\xae", "*0"},
  {"$6$\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf\xaf", "*0"},
  {"$6$\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0\xb0", "*0"},
  {"$6$\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1\xb1", "*0"},
  {"$6$\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2\xb2", "*0"},
  {"$6$\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3\xb3", "*0"},
  {"$6$\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4\xb4", "*0"},
  {"$6$\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5\xb5", "*0"},
  {"$6$\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6\xb6", "*0"},
  {"$6$\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7\xb7", "*0"},
  {"$6$\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8\xb8", "*0"},
  {"$6$\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9\xb9", "*0"},
  {"$6$\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba\xba", "*0"},
  {"$6$\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb\xbb", "*0"},
  {"$6$\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc\xbc", "*0"},
  {"$6$\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd", "*0"},
  {"$6$\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe\xbe", "*0"},
  {"$6$\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf\xbf", "*0"},
  {"$6$\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0\xc0", "*0"},
  {"$6$\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1\xc1", "*0"},
  {"$6$\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2\xc2", "*0"},
  {"$6$\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3\xc3", "*0"},
  {"$6$\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4\xc4", "*0"},
  {"$6$\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5\xc5", "*0"},
  {"$6$\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6\xc6", "*0"},
  {"$6$\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7\xc7", "*0"},
  {"$6$\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8\xc8", "*0"},
  {"$6$\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9\xc9", "*0"},
  {"$6$\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca\xca", "*0"},
  {"$6$\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb\xcb", "*0"},
  {"$6$\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc", "*0"},
  {"$6$\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd", "*0"},
  {"$6$\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce\xce", "*0"},
  {"$6$\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf\xcf", "*0"},
  {"$6$\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0\xd0", "*0"},
  {"$6$\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1\xd1", "*0"},
  {"$6$\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2\xd2", "*0"},
  {"$6$\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3\xd3", "*0"},
  {"$6$\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4\xd4", "*0"},
  {"$6$\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5\xd5", "*0"},
  {"$6$\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6\xd6", "*0"},
  {"$6$\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7\xd7", "*0"},
  {"$6$\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8\xd8", "*0"},
  {"$6$\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9\xd9", "*0"},
  {"$6$\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda\xda", "*0"},
  {"$6$\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb\xdb", "*0"},
  {"$6$\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc\xdc", "*0"},
  {"$6$\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd", "*0"},
  {"$6$\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde\xde", "*0"},
  {"$6$\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf\xdf", "*0"},
  {"$6$\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0\xe0", "*0"},
  {"$6$\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1\xe1", "*0"},
  {"$6$\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2\xe2", "*0"},
  {"$6$\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3\xe3", "*0"},
  {"$6$\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4\xe4", "*0"},
  {"$6$\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5\xe5", "*0"},
  {"$6$\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6\xe6", "*0"},
  {"$6$\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7\xe7", "*0"},
  {"$6$\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8\xe8", "*0"},
  {"$6$\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9\xe9", "*0"},
  {"$6$\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea\xea", "*0"},
  {"$6$\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb\xeb", "*0"},
  {"$6$\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec\xec", "*0"},
  {"$6$\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed\xed", "*0"},
  {"$6$\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee\xee", "*0"},
  {"$6$\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef\xef", "*0"},
  {"$6$\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0\xf0", "*0"},
  {"$6$\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1\xf1", "*0"},
  {"$6$\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2\xf2", "*0"},
  {"$6$\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3\xf3", "*0"},
  {"$6$\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4\xf4", "*0"},
  {"$6$\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5\xf5", "*0"},
  {"$6$\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6\xf6", "*0"},
  {"$6$\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7\xf7", "*0"},
  {"$6$\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8\xf8", "*0"},
  {"$6$\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9\xf9", "*0"},
  {"$6$\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa\xfa", "*0"},
  {"$6$\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb\xfb", "*0"},
  {"$6$\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc\xfc", "*0"},
  {"$6$\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd\xfd", "*0"},
  {"$6$\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe", "*0"},
  {"$6$\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff", "*0"},
#endif
};


static void
fprint_not_printable_as_hex (FILE *stream, const char *string,
                             bool with_cstring_escape)
{
  for (size_t i = 0; string[i] != '\0'; ++i)
    {
      if (with_cstring_escape)
        {
          /* The backslash '\' is screen-printable,
             but needs blackslash '\' escape for c strings. */
          if (string[i] == '\\')
            {
              fputs ("\\\\", stream);
              continue;
            }

          /* The quotation mark '"' is screen-printable,
             but needs blackslash '\' escape for c strings. */
          if (string[i] == '"')
            {
              fputs ("\\\"", stream);
              continue;
            }
        }

      /* The character is uncoditionally screen-printable.  */
      if (string[i] >= ' ' && string[i] <= '~')
        {
          fputc (string[i], stream);
          continue;
        }

      /* The character is NOT screen-printable,
         and needs conversion to hex notation.  */
      fprintf (stream, "\\x%02x", (unsigned char) string[i]);
    }
}

int
main (void)
{
  bool ok = true;
  const char *phrase = "foobarbaz";
  struct crypt_data cd;

  for (size_t i = 0; i < ARRAY_SIZE (testcases); ++i)
    {
      crypt_r (phrase, testcases[i].setting, &cd);

#if 0 /* Print the test array to stdout.  */

      fputs ("{\"", stdout);
      fprint_not_printable_as_hex (stdout, testcases[i].setting, true);
      fputs ("\", \"", stdout);
      fprint_not_printable_as_hex (stdout, cd.output, true);
      fputs ("\"},\n", stdout);
      ok = false; /* Ensure this always returns an error on exit.  */

#else /* Operate in real test mode.  */

      if (strcmp (cd.output, testcases[i].expected) != 0)
        {
          fputs ("FAIL:     ", stderr);
          fprint_not_printable_as_hex (stderr, testcases[i].setting, false);
          fputs ("\nexpected: ", stderr);
          fprint_not_printable_as_hex (stderr, testcases[i].expected, false);
          fputs ("\ngot:      ", stderr);
          fprint_not_printable_as_hex (stderr, cd.output, false);
          fputs ("\n\n", stderr);
          ok = false;
        }

#endif
    }

  fprintf (stdout, "%s\n", ok ? "PASS" : "FAIL");
  return ok ? 0 : 1;
}

#else

int
main (void)
{
  return 77; /* UNSUPPORTED */
}

#endif
