# Copyright 2021 by Michiel de Hoon.  All rights reserved.
#
# This code is part of the Biopython distribution and governed by its
# license.  Please see the LICENSE file that should have been included
# as part of this package.
"""Tests for Bio.Align.tabular module."""
import os
import unittest
from tempfile import NamedTemporaryFile

from Bio import Align
from Bio.Align import substitution_matrices
from Bio import SeqIO
from Bio.Seq import Seq

try:
    import numpy as np
except ImportError:
    from Bio import MissingPythonDependencyError

    raise MissingPythonDependencyError(
        "Install numpy if you want to use Bio.Align.emboss."
    ) from None

substitution_matrix = substitution_matrices.load("BLOSUM62")


class TestFastaProtein(unittest.TestCase):
    query = Seq(
        "MPMILGYWNVRGLTHPIRMLLEYTDSSYDEKRYTMGDAPDFDRSQWLNEKFKLGLDFPNLPYLIDGSHKITQSNAILRYLARKHHLDGETEEERIRADIVENQVMDTRMQLIMLCYNPDFEKQKPEFLKTIPEKMKLYSEFLGKRPWFAGDKVTYVDFLAYDILDQYRMFEPKCLDAFPNLRDFLARFEGLKKISAYMKSSRYIATPIFSKMAHWSNK"
    )

    filename = os.path.join("Fasta", "protein_lib.fa")
    records = SeqIO.parse(filename, "fasta")
    targets = {record.id: record.seq.upper() for record in records}

    def test_m8CB(self):
        # Alignment file obtained by running
        # fasta36 -q -m 8CB seq/mgstm1.aa seq/prot_test.lseg
        # in the fasta36 source distribution
        path = "Fasta/protein_m8CB.txt"
        alignments = Align.parse(path, "tabular")
        self.check_m8CB(alignments)
        alignments = iter(alignments)
        self.check_m8CB(alignments)
        with Align.parse(path, "tabular") as alignments:
            self.check_m8CB(alignments)
        with self.assertRaises(AttributeError):
            alignments._stream
        with Align.parse(path, "tabular") as alignments:
            pass
        with self.assertRaises(AttributeError):
            alignments._stream
        with open(path) as stream:
            data = stream.read()
        stream = NamedTemporaryFile("w+t")
        stream.write(data)
        stream.seek(0)
        alignments = Align.parse(stream, "tabular")
        self.check_m8CB(alignments)

    def check_m8CB(self, alignments):
        self.assertEqual(
            alignments.metadata["Command line"],
            "fasta36 -q -m 8CB seq/mgstm1.aa seq/prot_test.lseg",
        )
        self.assertEqual(alignments.metadata["Program"], "FASTA")
        self.assertEqual(alignments.metadata["Version"], "36.3.8h May, 2020")
        self.assertEqual(alignments.metadata["Database"], "seq/prot_test.lseg")
        # sp|P10649|GSTM1_MOUSE   sp|P09488|GSTM1_HUMAN
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 77.98)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 218)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 218)
        self.assertEqual(alignment.shape, (2, 218 + 0))
        self.assertEqual(alignment.sequences[0].id, "sp|P09488|GSTM1_HUMAN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 48)
        self.assertAlmostEqual(alignment.annotations["evalue"], 6.1e-78, places=79)
        self.assertAlmostEqual(alignment.annotations["bit score"], 275.6)
        self.assertEqual(
            str(alignment),
            """\
sp|P09488         0 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
sp|P10649         0 ????????????????????????????????????????????????????????????

sp|P09488        60 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
sp|P10649        60 ????????????????????????????????????????????????????????????

sp|P09488       120 ????????????????????????????????????????????????????????????
                120 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
sp|P10649       120 ????????????????????????????????????????????????????????????

sp|P09488       180 ?????????????????????????????????????? 218
                180 |||||||||||||||||||||||||||||||||||||| 218
sp|P10649       180 ?????????????????????????????????????? 218
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[0, 218], [0, 218]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 218)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "MPMILGYWDIRGLAHAIRLLLEYTDSSYEEKKYTMGDAPDYDRSQWLNEKFKLGLDFPNLPYLIDGAHKITQSNAILCYIARKHNLCGETEEEKIRVDILENQTMDNHMQLGMICYNPEFEKLKPKYLEELPEKLKLYSEFLGKRPWFAGNKITFVDFLVYDVLDLHRIFEPKCLDAFPNLKDFISRFEGLEKISAYMKSSRFLPRPVFSKMAVWGNK",
        )
        self.assertEqual(
            alignment[1],
            "MPMILGYWNVRGLTHPIRMLLEYTDSSYDEKRYTMGDAPDFDRSQWLNEKFKLGLDFPNLPYLIDGSHKITQSNAILRYLARKHHLDGETEEERIRADIVENQVMDTRMQLIMLCYNPDFEKQKPEFLKTIPEKMKLYSEFLGKRPWFAGDKVTYVDFLAYDILDQYRMFEPKCLDAFPNLRDFLARFEGLKKISAYMKSSRYIATPIFSKMAHWSNK",
        )
        # sp|P10649|GSTM1_MOUSE   sp|P00502|GSTA1_RAT
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['M', 'P', 'M', 'I', 'L', 'G', 'Y', 'W', 'D', 'I', 'R', 'G', 'L',
           'A', 'H', 'A', 'I', 'R', 'L', 'L', 'L', 'E', 'Y', 'T', 'D', 'S',
           'S', 'Y', 'E', 'E', 'K', 'K', 'Y', 'T', 'M', 'G', 'D', 'A', 'P',
           'D', 'Y', 'D', 'R', 'S', 'Q', 'W', 'L', 'N', 'E', 'K', 'F', 'K',
           'L', 'G', 'L', 'D', 'F', 'P', 'N', 'L', 'P', 'Y', 'L', 'I', 'D',
           'G', 'A', 'H', 'K', 'I', 'T', 'Q', 'S', 'N', 'A', 'I', 'L', 'C',
           'Y', 'I', 'A', 'R', 'K', 'H', 'N', 'L', 'C', 'G', 'E', 'T', 'E',
           'E', 'E', 'K', 'I', 'R', 'V', 'D', 'I', 'L', 'E', 'N', 'Q', 'T',
           'M', 'D', 'N', 'H', 'M', 'Q', 'L', 'G', 'M', 'I', 'C', 'Y', 'N',
           'P', 'E', 'F', 'E', 'K', 'L', 'K', 'P', 'K', 'Y', 'L', 'E', 'E',
           'L', 'P', 'E', 'K', 'L', 'K', 'L', 'Y', 'S', 'E', 'F', 'L', 'G',
           'K', 'R', 'P', 'W', 'F', 'A', 'G', 'N', 'K', 'I', 'T', 'F', 'V',
           'D', 'F', 'L', 'V', 'Y', 'D', 'V', 'L', 'D', 'L', 'H', 'R', 'I',
           'F', 'E', 'P', 'K', 'C', 'L', 'D', 'A', 'F', 'P', 'N', 'L', 'K',
           'D', 'F', 'I', 'S', 'R', 'F', 'E', 'G', 'L', 'E', 'K', 'I', 'S',
           'A', 'Y', 'M', 'K', 'S', 'S', 'R', 'F', 'L', 'P', 'R', 'P', 'V',
           'F', 'S', 'K', 'M', 'A', 'V', 'W', 'G', 'N', 'K'],
          ['M', 'P', 'M', 'I', 'L', 'G', 'Y', 'W', 'N', 'V', 'R', 'G', 'L',
           'T', 'H', 'P', 'I', 'R', 'M', 'L', 'L', 'E', 'Y', 'T', 'D', 'S',
           'S', 'Y', 'D', 'E', 'K', 'R', 'Y', 'T', 'M', 'G', 'D', 'A', 'P',
           'D', 'F', 'D', 'R', 'S', 'Q', 'W', 'L', 'N', 'E', 'K', 'F', 'K',
           'L', 'G', 'L', 'D', 'F', 'P', 'N', 'L', 'P', 'Y', 'L', 'I', 'D',
           'G', 'S', 'H', 'K', 'I', 'T', 'Q', 'S', 'N', 'A', 'I', 'L', 'R',
           'Y', 'L', 'A', 'R', 'K', 'H', 'H', 'L', 'D', 'G', 'E', 'T', 'E',
           'E', 'E', 'R', 'I', 'R', 'A', 'D', 'I', 'V', 'E', 'N', 'Q', 'V',
           'M', 'D', 'T', 'R', 'M', 'Q', 'L', 'I', 'M', 'L', 'C', 'Y', 'N',
           'P', 'D', 'F', 'E', 'K', 'Q', 'K', 'P', 'E', 'F', 'L', 'K', 'T',
           'I', 'P', 'E', 'K', 'M', 'K', 'L', 'Y', 'S', 'E', 'F', 'L', 'G',
           'K', 'R', 'P', 'W', 'F', 'A', 'G', 'D', 'K', 'V', 'T', 'Y', 'V',
           'D', 'F', 'L', 'A', 'Y', 'D', 'I', 'L', 'D', 'Q', 'Y', 'R', 'M',
           'F', 'E', 'P', 'K', 'C', 'L', 'D', 'A', 'F', 'P', 'N', 'L', 'R',
           'D', 'F', 'L', 'A', 'R', 'F', 'E', 'G', 'L', 'K', 'K', 'I', 'S',
           'A', 'Y', 'M', 'K', 'S', 'S', 'R', 'Y', 'I', 'A', 'T', 'P', 'I',
           'F', 'S', 'K', 'M', 'A', 'H', 'W', 'S', 'N', 'K']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 967.0; 218 aligned letters; 170 identities; 48 mismatches; 201 positives; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 967.0,
    aligned = 218:
        identities = 170,
        positives = 201,
        mismatches = 48.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 218)
        self.assertEqual(counts.identities, 170)
        self.assertEqual(counts.mismatches, 48)
        self.assertEqual(counts.positives, 201)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 29.76)
        self.assertEqual(alignment.annotations["gap opens"], 18)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 205)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 205)
        self.assertEqual(alignment.shape, (2, 205 + 18))
        self.assertEqual(alignment.sequences[0].id, "sp|P00502|GSTA1_RAT")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 144)
        self.assertAlmostEqual(alignment.annotations["evalue"], 3.1e-13, places=14)
        self.assertAlmostEqual(alignment.annotations["bit score"], 60.7)
        self.assertEqual(
            str(alignment),
            """\
sp|P00502         5 ????????????????????????????---------???????????????????????
                  0 ||||||||||||||||||||||||||||---------|||||||||||||--||||||||
sp|P10649         3 ??????????????????????????????????????????????????--????????

sp|P00502        56 ??????-?????????????????????????????????????????????????????
                 60 ||-|||-|||||||||||||||||||||||||||||||||||||||-|||||||||||||
sp|P10649        61 ??-???????????????????????????????????????????-?????????????

sp|P00502       115 ????????????????????????????????????????????????????????????
                120 ||||||||||--|||||||||||||||--|||||||||||||||||||||||||||||||
sp|P10649       119 ??????????--???????????????--???????????????????????????????

sp|P00502       175 ??????????????????????????????????????????? 218
                180 ||||||||||||||||||||||||||||||||||||||||||| 223
sp|P10649       175 ??????????????????????????????????????????? 218
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[ 5,  33,  33,  46,  48,  58,  59,  62,
                           62, 101, 102, 125, 127, 142, 144, 218],
                          [ 3,  31,  40,  53,  53,  63,  63,  66,
                           67, 106, 106, 129, 129, 144, 144, 218]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 218)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "VLHYFNARGRMECIRWLLAAAGVEFDEK---------FIQSPEDLEKLKKDGNLMFDQVPMVEIDG-MKLAQTRAILNYIATKYDLYGKDMKERALIDMYTEGILDLTEMIMQLVICPPDQKEAKTALAKDRTKNRYLPAFEKVLKSHGQDYLVGNRLTRVDIHLLELLLYVEEFDASLLTSFPLLKAFKSRISSLPNVKKFLQPGSQRKLPVDAKQIEEARK",
        )
        self.assertEqual(
            alignment[1],
            "ILGYWNVRGLTHPIRMLLEYTDSSYDEKRYTMGDAPDFDRSQWLNEKFKL--GLDFPNLPYL-IDGSHKITQSNAILRYLARKHHLDGETEEERIRADIVENQVMD-TRMQLIMLCYNPDFEKQKPEFLK--TIPEKMKLYSEFLGK--RPWFAGDKVTYVDFLAYDILDQYRMFEPKCLDAFPNLRDFLARFEGLKKISAYMKSSRYIATPIFSKMAHWSNK",
        )
        # sp|P10649|GSTM1_MOUSE   sp|P69905|HBA_HUMAN
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['V', 'L', 'H', 'Y', 'F', 'N', 'A', 'R', 'G', 'R', 'M', 'E', 'C',
           'I', 'R', 'W', 'L', 'L', 'A', 'A', 'A', 'G', 'V', 'E', 'F', 'D',
           'E', 'K', '-', '-', '-', '-', '-', '-', '-', '-', '-', 'F', 'I',
           'Q', 'S', 'P', 'E', 'D', 'L', 'E', 'K', 'L', 'K', 'K', 'D', 'G',
           'N', 'L', 'M', 'F', 'D', 'Q', 'V', 'P', 'M', 'V', 'E', 'I', 'D',
           'G', '-', 'M', 'K', 'L', 'A', 'Q', 'T', 'R', 'A', 'I', 'L', 'N',
           'Y', 'I', 'A', 'T', 'K', 'Y', 'D', 'L', 'Y', 'G', 'K', 'D', 'M',
           'K', 'E', 'R', 'A', 'L', 'I', 'D', 'M', 'Y', 'T', 'E', 'G', 'I',
           'L', 'D', 'L', 'T', 'E', 'M', 'I', 'M', 'Q', 'L', 'V', 'I', 'C',
           'P', 'P', 'D', 'Q', 'K', 'E', 'A', 'K', 'T', 'A', 'L', 'A', 'K',
           'D', 'R', 'T', 'K', 'N', 'R', 'Y', 'L', 'P', 'A', 'F', 'E', 'K',
           'V', 'L', 'K', 'S', 'H', 'G', 'Q', 'D', 'Y', 'L', 'V', 'G', 'N',
           'R', 'L', 'T', 'R', 'V', 'D', 'I', 'H', 'L', 'L', 'E', 'L', 'L',
           'L', 'Y', 'V', 'E', 'E', 'F', 'D', 'A', 'S', 'L', 'L', 'T', 'S',
           'F', 'P', 'L', 'L', 'K', 'A', 'F', 'K', 'S', 'R', 'I', 'S', 'S',
           'L', 'P', 'N', 'V', 'K', 'K', 'F', 'L', 'Q', 'P', 'G', 'S', 'Q',
           'R', 'K', 'L', 'P', 'V', 'D', 'A', 'K', 'Q', 'I', 'E', 'E', 'A',
           'R', 'K'],
          ['I', 'L', 'G', 'Y', 'W', 'N', 'V', 'R', 'G', 'L', 'T', 'H', 'P',
           'I', 'R', 'M', 'L', 'L', 'E', 'Y', 'T', 'D', 'S', 'S', 'Y', 'D',
           'E', 'K', 'R', 'Y', 'T', 'M', 'G', 'D', 'A', 'P', 'D', 'F', 'D',
           'R', 'S', 'Q', 'W', 'L', 'N', 'E', 'K', 'F', 'K', 'L', '-', '-',
           'G', 'L', 'D', 'F', 'P', 'N', 'L', 'P', 'Y', 'L', '-', 'I', 'D',
           'G', 'S', 'H', 'K', 'I', 'T', 'Q', 'S', 'N', 'A', 'I', 'L', 'R',
           'Y', 'L', 'A', 'R', 'K', 'H', 'H', 'L', 'D', 'G', 'E', 'T', 'E',
           'E', 'E', 'R', 'I', 'R', 'A', 'D', 'I', 'V', 'E', 'N', 'Q', 'V',
           'M', 'D', '-', 'T', 'R', 'M', 'Q', 'L', 'I', 'M', 'L', 'C', 'Y',
           'N', 'P', 'D', 'F', 'E', 'K', 'Q', 'K', 'P', 'E', 'F', 'L', 'K',
           '-', '-', 'T', 'I', 'P', 'E', 'K', 'M', 'K', 'L', 'Y', 'S', 'E',
           'F', 'L', 'G', 'K', '-', '-', 'R', 'P', 'W', 'F', 'A', 'G', 'D',
           'K', 'V', 'T', 'Y', 'V', 'D', 'F', 'L', 'A', 'Y', 'D', 'I', 'L',
           'D', 'Q', 'Y', 'R', 'M', 'F', 'E', 'P', 'K', 'C', 'L', 'D', 'A',
           'F', 'P', 'N', 'L', 'R', 'D', 'F', 'L', 'A', 'R', 'F', 'E', 'G',
           'L', 'K', 'K', 'I', 'S', 'A', 'Y', 'M', 'K', 'S', 'S', 'R', 'Y',
           'I', 'A', 'T', 'P', 'I', 'F', 'S', 'K', 'M', 'A', 'H', 'W', 'S',
           'N', 'K']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 257.0; 205 aligned letters; 61 identities; 144 mismatches; 102 positives; 18 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 257.0,
    aligned = 205:
        identities = 61,
        positives = 102,
        mismatches = 144.
    gaps = 18:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 18:
            internal_insertions = 10:
                open_internal_insertions = 2,
                extend_internal_insertions = 8;
            internal_deletions = 8:
                open_internal_deletions = 5,
                extend_internal_deletions = 3;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 10)
        self.assertEqual(counts.internal_deletions, 8)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 18)
        self.assertEqual(counts.insertions, 10)
        self.assertEqual(counts.deletions, 8)
        self.assertEqual(counts.gaps, 18)
        self.assertEqual(counts.aligned, 205)
        self.assertEqual(counts.identities, 61)
        self.assertEqual(counts.mismatches, 144)
        self.assertEqual(counts.positives, 102)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 27.03)
        self.assertEqual(alignment.annotations["gap opens"], 2)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 37)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 37)
        self.assertEqual(alignment.shape, (2, 37 + 2))
        self.assertEqual(alignment.sequences[0].id, "sp|P69905|HBA_HUMAN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 27)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.19)
        self.assertAlmostEqual(alignment.annotations["bit score"], 20.9)
        self.assertEqual(
            str(alignment),
            """\
sp|P69905        35 ?????????????-?????????????????????????  73
                  0 |||||||||||||-||||||||||-||||||||||||||  39
sp|P10649       176 ????????????????????????-?????????????? 214
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[ 35,  48,  48,  58,  59,  73],
                          [176, 189, 190, 200, 200, 214]]),
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 73)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "SFPTTKTYFPHFD-LSHGSAQVKGHGKKVADALTNAVAH")
        self.assertEqual(alignment[1], "AFPNLRDFLARFEGLKKISAYMKS-SRYIATPIFSKMAH")
        # sp|P10649|GSTM1_MOUSE   sp|P00517|KAPCA_BOVIN
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['S', 'F', 'P', 'T', 'T', 'K', 'T', 'Y', 'F', 'P', 'H', 'F', 'D',
           '-', 'L', 'S', 'H', 'G', 'S', 'A', 'Q', 'V', 'K', 'G', 'H', 'G',
           'K', 'K', 'V', 'A', 'D', 'A', 'L', 'T', 'N', 'A', 'V', 'A', 'H'],
          ['A', 'F', 'P', 'N', 'L', 'R', 'D', 'F', 'L', 'A', 'R', 'F', 'E',
           'G', 'L', 'K', 'K', 'I', 'S', 'A', 'Y', 'M', 'K', 'S', '-', 'S',
           'R', 'Y', 'I', 'A', 'T', 'P', 'I', 'F', 'S', 'K', 'M', 'A', 'H']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 54.0; 37 aligned letters; 10 identities; 27 mismatches; 20 positives; 2 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 54.0,
    aligned = 37:
        identities = 10,
        positives = 20,
        mismatches = 27.
    gaps = 2:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 2:
            internal_insertions = 1:
                open_internal_insertions = 1,
                extend_internal_insertions = 0;
            internal_deletions = 1:
                open_internal_deletions = 1,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 1)
        self.assertEqual(counts.internal_deletions, 1)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 2)
        self.assertEqual(counts.insertions, 1)
        self.assertEqual(counts.deletions, 1)
        self.assertEqual(counts.gaps, 2)
        self.assertEqual(counts.aligned, 37)
        self.assertEqual(counts.identities, 10)
        self.assertEqual(counts.mismatches, 27)
        self.assertEqual(counts.positives, 20)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 24.29)
        self.assertEqual(alignment.annotations["gap opens"], 2)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 70)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 70)
        self.assertEqual(alignment.shape, (2, 70 + 2))
        self.assertEqual(alignment.sequences[0].id, "sp|P00517|KAPCA_BOVIN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 53)
        self.assertAlmostEqual(alignment.annotations["evalue"], 1.4)
        self.assertAlmostEqual(alignment.annotations["bit score"], 19.2)
        self.assertEqual(
            str(alignment),
            """\
sp|P00517       228 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||--||||||||||||
sp|P10649       136 ??????????????????????????????????????????????--????????????

sp|P00517       288 ???????????? 300
                 60 ||||||||||||  72
sp|P10649       194 ???????????? 206
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[228, 274, 276, 300],
                          [136, 182, 182, 206]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 300)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "IYEMAAGYPPFFADQPIQIYEKIVSGKVRFPSHFSSDLKDLLRNLLQVDLTKRFGNLKNGVNDIKNHKWFAT",
        )
        self.assertEqual(
            alignment[1],
            "LYSEFLGKRPWFAGDKVTYVDFLAYDILDQYRMFEPKCLDAFPNLR--DFLARFEGLKKISAYMKSSRYIAT",
        )
        # sp|P10649|GSTM1_MOUSE   sp|P14960|RBS_GUITH
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['I', 'Y', 'E', 'M', 'A', 'A', 'G', 'Y', 'P', 'P', 'F', 'F', 'A',
           'D', 'Q', 'P', 'I', 'Q', 'I', 'Y', 'E', 'K', 'I', 'V', 'S', 'G',
           'K', 'V', 'R', 'F', 'P', 'S', 'H', 'F', 'S', 'S', 'D', 'L', 'K',
           'D', 'L', 'L', 'R', 'N', 'L', 'L', 'Q', 'V', 'D', 'L', 'T', 'K',
           'R', 'F', 'G', 'N', 'L', 'K', 'N', 'G', 'V', 'N', 'D', 'I', 'K',
           'N', 'H', 'K', 'W', 'F', 'A', 'T'],
          ['L', 'Y', 'S', 'E', 'F', 'L', 'G', 'K', 'R', 'P', 'W', 'F', 'A',
           'G', 'D', 'K', 'V', 'T', 'Y', 'V', 'D', 'F', 'L', 'A', 'Y', 'D',
           'I', 'L', 'D', 'Q', 'Y', 'R', 'M', 'F', 'E', 'P', 'K', 'C', 'L',
           'D', 'A', 'F', 'P', 'N', 'L', 'R', '-', '-', 'D', 'F', 'L', 'A',
           'R', 'F', 'E', 'G', 'L', 'K', 'K', 'I', 'S', 'A', 'Y', 'M', 'K',
           'S', 'S', 'R', 'Y', 'I', 'A', 'T']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 49.0; 70 aligned letters; 17 identities; 53 mismatches; 27 positives; 2 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 49.0,
    aligned = 70:
        identities = 17,
        positives = 27,
        mismatches = 53.
    gaps = 2:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 2:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 2:
                open_internal_deletions = 1,
                extend_internal_deletions = 1;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 2)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 2)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 2)
        self.assertEqual(counts.gaps, 2)
        self.assertEqual(counts.aligned, 70)
        self.assertEqual(counts.identities, 17)
        self.assertEqual(counts.mismatches, 53)
        self.assertEqual(counts.positives, 27)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 57.14)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 7)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 7)
        self.assertEqual(alignment.shape, (2, 7 + 0))
        self.assertEqual(alignment.sequences[0].id, "sp|P14960|RBS_GUITH")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 3)
        self.assertAlmostEqual(alignment.annotations["evalue"], 1.6)
        self.assertAlmostEqual(alignment.annotations["bit score"], 17.7)
        self.assertEqual(
            str(alignment),
            """\
sp|P14960        46 ??????? 53
                  0 |||||||  7
sp|P10649         6 ??????? 13
""",
        )
        self.assertTrue(
            np.array_equal(alignment.coordinates, np.array([[46, 53], [6, 13]]))
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 53)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "YWDLWGL")
        self.assertEqual(alignment[1], "YWNVRGL")
        # sp|P10649|GSTM1_MOUSE   sp|P01593|KV101_HUMAN
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['Y', 'W', 'D', 'L', 'W', 'G', 'L'],
          ['Y', 'W', 'N', 'V', 'R', 'G', 'L']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 27.0; 7 aligned letters; 4 identities; 3 mismatches; 6 positives; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 27.0,
    aligned = 7:
        identities = 4,
        positives = 6,
        mismatches = 3.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 7)
        self.assertEqual(counts.identities, 4)
        self.assertEqual(counts.mismatches, 3)
        self.assertEqual(counts.positives, 6)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 45.00)
        self.assertEqual(alignment.annotations["gap opens"], 10)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 40)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 40)
        self.assertEqual(alignment.shape, (2, 40 + 10))
        self.assertEqual(alignment.sequences[0].id, "sp|P01593|KV101_HUMAN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 22)
        self.assertAlmostEqual(alignment.annotations["evalue"], 2.5)
        self.assertAlmostEqual(alignment.annotations["bit score"], 16.6)
        self.assertEqual(
            str(alignment),
            """\
sp|P01593        15 ????????????????????????????????????-?????????????  64
                  0 ||||||||||||||---||||||||---||||--||-|||||||-|||||  50
sp|P10649       149 ??????????????---????????---????--??????????-????? 190
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[ 15,  29,  32,  40,  43,  47,
                            49,  51,  51,  58,  59,  64],
                          [149, 163, 163, 171, 171, 175,
                           175, 177, 178, 185, 185, 190]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 64)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0], "GDRVTITCQASQDINHYLNWYQQGPKKAPKILIYDA-SNLETGVPSRFSG"
        )
        self.assertEqual(
            alignment[1], "GDKVTYVDFLAYDI---LDQYRMFE---PKCL--DAFPNLRDFL-ARFEG"
        )
        # sp|P10649|GSTM1_MOUSE   sp|P99998|CYC_PANTR
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['G', 'D', 'R', 'V', 'T', 'I', 'T', 'C', 'Q', 'A', 'S', 'Q', 'D',
           'I', 'N', 'H', 'Y', 'L', 'N', 'W', 'Y', 'Q', 'Q', 'G', 'P', 'K',
           'K', 'A', 'P', 'K', 'I', 'L', 'I', 'Y', 'D', 'A', '-', 'S', 'N',
           'L', 'E', 'T', 'G', 'V', 'P', 'S', 'R', 'F', 'S', 'G'],
          ['G', 'D', 'K', 'V', 'T', 'Y', 'V', 'D', 'F', 'L', 'A', 'Y', 'D',
           'I', '-', '-', '-', 'L', 'D', 'Q', 'Y', 'R', 'M', 'F', 'E', '-',
           '-', '-', 'P', 'K', 'C', 'L', '-', '-', 'D', 'A', 'F', 'P', 'N',
           'L', 'R', 'D', 'F', 'L', '-', 'A', 'R', 'F', 'E', 'G']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 81.0; 40 aligned letters; 18 identities; 22 mismatches; 24 positives; 10 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 81.0,
    aligned = 40:
        identities = 18,
        positives = 24,
        mismatches = 22.
    gaps = 10:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 10:
            internal_insertions = 1:
                open_internal_insertions = 1,
                extend_internal_insertions = 0;
            internal_deletions = 9:
                open_internal_deletions = 4,
                extend_internal_deletions = 5;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 1)
        self.assertEqual(counts.internal_deletions, 9)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 10)
        self.assertEqual(counts.insertions, 1)
        self.assertEqual(counts.deletions, 9)
        self.assertEqual(counts.gaps, 10)
        self.assertEqual(counts.aligned, 40)
        self.assertEqual(counts.identities, 18)
        self.assertEqual(counts.mismatches, 22)
        self.assertEqual(counts.positives, 24)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 31.03)
        self.assertEqual(alignment.annotations["gap opens"], 10)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 58)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 58)
        self.assertEqual(alignment.shape, (2, 58 + 10))
        self.assertEqual(alignment.sequences[0].id, "sp|P99998|CYC_PANTR")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 40)
        self.assertAlmostEqual(alignment.annotations["evalue"], 2.7)
        self.assertAlmostEqual(alignment.annotations["bit score"], 16.4)
        self.assertEqual(
            str(alignment),
            """\
sp|P99998        27 ???????????????????????????????-??????????-?????--?????????-
                  0 ||||||||||||||||||||---||||||||-||||||||||-|||||--|||||||||-
sp|P10649       128 ????????????????????---?????????????????????????????????????

sp|P99998        82 --??????  88
                 60 --||||||  68
sp|P10649       185 ???????? 193
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[ 27,  47,  50,  58,  58,  68,
                            68,  73,  73,  82,  82,  88],
                          [128, 148, 148, 156, 157, 167,
                           168, 173, 175, 184, 187, 193]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 88)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "KTGPNLHGLFGRKTGQAPGYSYTAANKNKGI-IWGEDTLMEY-LENPK--KYIPGTKMI---FVGIKK",
        )
        self.assertEqual(
            alignment[1],
            "KTIPEKMKLYSEFLGKRPWF---AGDKVTYVDFLAYDILDQYRMFEPKCLDAFPNLRDFLARFEGLKK",
        )
        # sp|P10649|GSTM1_MOUSE   sp|P02585|TNNC2_HUMAN
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['K', 'T', 'G', 'P', 'N', 'L', 'H', 'G', 'L', 'F', 'G', 'R', 'K',
           'T', 'G', 'Q', 'A', 'P', 'G', 'Y', 'S', 'Y', 'T', 'A', 'A', 'N',
           'K', 'N', 'K', 'G', 'I', '-', 'I', 'W', 'G', 'E', 'D', 'T', 'L',
           'M', 'E', 'Y', '-', 'L', 'E', 'N', 'P', 'K', '-', '-', 'K', 'Y',
           'I', 'P', 'G', 'T', 'K', 'M', 'I', '-', '-', '-', 'F', 'V', 'G',
           'I', 'K', 'K'],
          ['K', 'T', 'I', 'P', 'E', 'K', 'M', 'K', 'L', 'Y', 'S', 'E', 'F',
           'L', 'G', 'K', 'R', 'P', 'W', 'F', '-', '-', '-', 'A', 'G', 'D',
           'K', 'V', 'T', 'Y', 'V', 'D', 'F', 'L', 'A', 'Y', 'D', 'I', 'L',
           'D', 'Q', 'Y', 'R', 'M', 'F', 'E', 'P', 'K', 'C', 'L', 'D', 'A',
           'F', 'P', 'N', 'L', 'R', 'D', 'F', 'L', 'A', 'R', 'F', 'E', 'G',
           'L', 'K', 'K']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 76.0; 58 aligned letters; 18 identities; 40 mismatches; 27 positives; 10 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 76.0,
    aligned = 58:
        identities = 18,
        positives = 27,
        mismatches = 40.
    gaps = 10:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 10:
            internal_insertions = 7:
                open_internal_insertions = 4,
                extend_internal_insertions = 3;
            internal_deletions = 3:
                open_internal_deletions = 1,
                extend_internal_deletions = 2;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 7)
        self.assertEqual(counts.internal_deletions, 3)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 10)
        self.assertEqual(counts.insertions, 7)
        self.assertEqual(counts.deletions, 3)
        self.assertEqual(counts.gaps, 10)
        self.assertEqual(counts.aligned, 58)
        self.assertEqual(counts.identities, 18)
        self.assertEqual(counts.mismatches, 40)
        self.assertEqual(counts.positives, 27)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 31.11)
        self.assertEqual(alignment.annotations["gap opens"], 9)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 45)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 45)
        self.assertEqual(alignment.shape, (2, 45 + 9))
        self.assertEqual(alignment.sequences[0].id, "sp|P02585|TNNC2_HUMAN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 31)
        self.assertAlmostEqual(alignment.annotations["evalue"], 3.9)
        self.assertAlmostEqual(alignment.annotations["bit score"], 16.4)
        self.assertEqual(
            str(alignment),
            """\
sp|P02585        12 ??????????????----???????????????????????????????????? 62
                  0 ||||||||||||||----||||||||||--|||||||||||---|||||||||| 54
sp|P10649        43 ????????????????????????????--???????????---?????????? 92
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array(
                    [[12, 26, 26, 36, 38, 49, 52, 62],
                     [43, 57, 61, 71, 71, 82, 82, 92]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 62)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0], "SEEMIAEFKAAFDM----FDADGGGDISVKELGTVMRMLGQTPTKEELDAIIEE"
        )
        self.assertEqual(
            alignment[1], "SQWLNEKFKLGLDFPNLPYLIDGSHKIT--QSNAILRYLAR---KHHLDGETEE"
        )
        # sp|P10649|GSTM1_MOUSE   sp|P60615|NXL1A_BUNMU
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['S', 'E', 'E', 'M', 'I', 'A', 'E', 'F', 'K', 'A', 'A', 'F', 'D',
           'M', '-', '-', '-', '-', 'F', 'D', 'A', 'D', 'G', 'G', 'G', 'D',
           'I', 'S', 'V', 'K', 'E', 'L', 'G', 'T', 'V', 'M', 'R', 'M', 'L',
           'G', 'Q', 'T', 'P', 'T', 'K', 'E', 'E', 'L', 'D', 'A', 'I', 'I',
           'E', 'E'],
          ['S', 'Q', 'W', 'L', 'N', 'E', 'K', 'F', 'K', 'L', 'G', 'L', 'D',
           'F', 'P', 'N', 'L', 'P', 'Y', 'L', 'I', 'D', 'G', 'S', 'H', 'K',
           'I', 'T', '-', '-', 'Q', 'S', 'N', 'A', 'I', 'L', 'R', 'Y', 'L',
           'A', 'R', '-', '-', '-', 'K', 'H', 'H', 'L', 'D', 'G', 'E', 'T',
           'E', 'E']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 65.0; 45 aligned letters; 14 identities; 31 mismatches; 23 positives; 9 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 65.0,
    aligned = 45:
        identities = 14,
        positives = 23,
        mismatches = 31.
    gaps = 9:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 9:
            internal_insertions = 4:
                open_internal_insertions = 1,
                extend_internal_insertions = 3;
            internal_deletions = 5:
                open_internal_deletions = 2,
                extend_internal_deletions = 3;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 4)
        self.assertEqual(counts.internal_deletions, 5)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 9)
        self.assertEqual(counts.insertions, 4)
        self.assertEqual(counts.deletions, 5)
        self.assertEqual(counts.gaps, 9)
        self.assertEqual(counts.aligned, 45)
        self.assertEqual(counts.identities, 14)
        self.assertEqual(counts.mismatches, 31)
        self.assertEqual(counts.positives, 23)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 66.67)
        self.assertEqual(alignment.annotations["gap opens"], 2)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 9)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 9)
        self.assertEqual(alignment.shape, (2, 9 + 2))
        self.assertEqual(alignment.sequences[0].id, "sp|P60615|NXL1A_BUNMU")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 3)
        self.assertAlmostEqual(alignment.annotations["evalue"], 4.2)
        self.assertAlmostEqual(alignment.annotations["bit score"], 15.6)
        self.assertEqual(
            str(alignment),
            """\
sp|P60615        85 ?-???-?????  94
                  0 |-|||-|||||  11
sp|P10649       114 ??????????? 125
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[ 85,  86,  86,  89,  89,  94],
                          [114, 115, 116, 119, 120, 125]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 94)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "C-NPH-PKQRP")
        self.assertEqual(alignment[1], "CYNPDFEKQKP")
        # sp|P10649|GSTM1_MOUSE   sp|P00193|FER_PEPAS
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', '-', 'N', 'P', 'H', '-', 'P', 'K', 'Q', 'R', 'P'],
          ['C', 'Y', 'N', 'P', 'D', 'F', 'E', 'K', 'Q', 'K', 'P']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 39.0; 9 aligned letters; 6 identities; 3 mismatches; 7 positives; 2 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 39.0,
    aligned = 9:
        identities = 6,
        positives = 7,
        mismatches = 3.
    gaps = 2:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 2:
            internal_insertions = 2:
                open_internal_insertions = 2,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 2)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 2)
        self.assertEqual(counts.insertions, 2)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 2)
        self.assertEqual(counts.aligned, 9)
        self.assertEqual(counts.identities, 6)
        self.assertEqual(counts.mismatches, 3)
        self.assertEqual(counts.positives, 7)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 50.00)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 4)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 4)
        self.assertEqual(alignment.shape, (2, 4 + 0))
        self.assertEqual(alignment.sequences[0].id, "sp|P00193|FER_PEPAS")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 2)
        self.assertAlmostEqual(alignment.annotations["evalue"], 4.2)
        self.assertAlmostEqual(alignment.annotations["bit score"], 14.7)
        self.assertEqual(
            str(alignment),
            """\
sp|P00193        14 ????  18
                  0 ||||   4
sp|P10649       170 ???? 174
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[14, 18], [170, 174]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 18)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "KPEC")
        self.assertEqual(alignment[1], "EPKC")
        # sp|P10649|GSTM1_MOUSE   sp|P03435|HEMA_I75A3
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['K', 'P', 'E', 'C'],
          ['E', 'P', 'K', 'C']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 18.0; 4 aligned letters; 2 identities; 2 mismatches; 4 positives; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 18.0,
    aligned = 4:
        identities = 2,
        positives = 4,
        mismatches = 2.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 4)
        self.assertEqual(counts.identities, 2)
        self.assertEqual(counts.mismatches, 2)
        self.assertEqual(counts.positives, 4)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 28.95)
        self.assertEqual(alignment.annotations["gap opens"], 1)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 38)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 38)
        self.assertEqual(alignment.shape, (2, 38 + 1))
        self.assertEqual(alignment.sequences[0].id, "sp|P03435|HEMA_I75A3")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 27)
        self.assertAlmostEqual(alignment.annotations["evalue"], 4.7)
        self.assertAlmostEqual(alignment.annotations["bit score"], 17.9)
        self.assertEqual(
            str(alignment),
            """\
sp|P03435       398 ??????????????????????????????????????? 437
                  0 ||||||||||-||||||||||||||||||||||||||||  39
sp|P10649        73 ??????????-???????????????????????????? 111
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[398, 408, 409, 437], [73, 83, 83, 111]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 437)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "NRVIEKTNEKFHQIEKEFSEVEGRIQDLEKYVEDTKIDL")
        self.assertEqual(alignment[1], "NAILRYLARK-HHLDGETEEERIRADIVENQVMDTRMQL")
        # sp|P10649|GSTM1_MOUSE   sp|P01834|IGKC_HUMAN
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['N', 'R', 'V', 'I', 'E', 'K', 'T', 'N', 'E', 'K', 'F', 'H', 'Q',
           'I', 'E', 'K', 'E', 'F', 'S', 'E', 'V', 'E', 'G', 'R', 'I', 'Q',
           'D', 'L', 'E', 'K', 'Y', 'V', 'E', 'D', 'T', 'K', 'I', 'D', 'L'],
          ['N', 'A', 'I', 'L', 'R', 'Y', 'L', 'A', 'R', 'K', '-', 'H', 'H',
           'L', 'D', 'G', 'E', 'T', 'E', 'E', 'E', 'R', 'I', 'R', 'A', 'D',
           'I', 'V', 'E', 'N', 'Q', 'V', 'M', 'D', 'T', 'R', 'M', 'Q', 'L']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 48.0; 38 aligned letters; 11 identities; 27 mismatches; 18 positives; 1 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 48.0,
    aligned = 38:
        identities = 11,
        positives = 18,
        mismatches = 27.
    gaps = 1:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 1:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 1:
                open_internal_deletions = 1,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 1)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 1)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 1)
        self.assertEqual(counts.gaps, 1)
        self.assertEqual(counts.aligned, 38)
        self.assertEqual(counts.identities, 11)
        self.assertEqual(counts.mismatches, 27)
        self.assertEqual(counts.positives, 18)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 19.40)
        self.assertEqual(alignment.annotations["gap opens"], 4)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 67)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 67)
        self.assertEqual(alignment.shape, (2, 67 + 4))
        self.assertEqual(alignment.sequences[0].id, "sp|P01834|IGKC_HUMAN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 54)
        self.assertAlmostEqual(alignment.annotations["evalue"], 6.3)
        self.assertAlmostEqual(alignment.annotations["bit score"], 14.9)
        self.assertEqual(
            str(alignment),
            """\
sp|P01834        11 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||--||||||||||||||||||||||||||||||||||||||||--||
sp|P10649        57 ??????????????--????????????????????????????????????????--??

sp|P01834        71 ???????????  82
                 60 |||||||||||  71
sp|P10649       113 ??????????? 124
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[11, 25, 27,  67,  69,  82],
                          [57, 71, 71, 111, 111, 124]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 82)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "PSDEQLKSGTASVVCLLNNFYPREAKVQWKVDNALQSGNSQESVTEQDSKDSTYSLSSTLTLSKADYEKHK",
        )
        self.assertEqual(
            alignment[1],
            "PNLPYLIDGSHKIT--QSNAILRYLARKHHLDGETEEERIRADIVENQVMDTRMQL--IMLCYNPDFEKQK",
        )
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['P', 'S', 'D', 'E', 'Q', 'L', 'K', 'S', 'G', 'T', 'A', 'S', 'V',
           'V', 'C', 'L', 'L', 'N', 'N', 'F', 'Y', 'P', 'R', 'E', 'A', 'K',
           'V', 'Q', 'W', 'K', 'V', 'D', 'N', 'A', 'L', 'Q', 'S', 'G', 'N',
           'S', 'Q', 'E', 'S', 'V', 'T', 'E', 'Q', 'D', 'S', 'K', 'D', 'S',
           'T', 'Y', 'S', 'L', 'S', 'S', 'T', 'L', 'T', 'L', 'S', 'K', 'A',
           'D', 'Y', 'E', 'K', 'H', 'K'],
          ['P', 'N', 'L', 'P', 'Y', 'L', 'I', 'D', 'G', 'S', 'H', 'K', 'I',
           'T', '-', '-', 'Q', 'S', 'N', 'A', 'I', 'L', 'R', 'Y', 'L', 'A',
           'R', 'K', 'H', 'H', 'L', 'D', 'G', 'E', 'T', 'E', 'E', 'E', 'R',
           'I', 'R', 'A', 'D', 'I', 'V', 'E', 'N', 'Q', 'V', 'M', 'D', 'T',
           'R', 'M', 'Q', 'L', '-', '-', 'I', 'M', 'L', 'C', 'Y', 'N', 'P',
           'D', 'F', 'E', 'K', 'Q', 'K']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 43.0; 67 aligned letters; 13 identities; 54 mismatches; 25 positives; 4 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 43.0,
    aligned = 67:
        identities = 13,
        positives = 25,
        mismatches = 54.
    gaps = 4:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 4:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 4:
                open_internal_deletions = 2,
                extend_internal_deletions = 2;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 4)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 4)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 4)
        self.assertEqual(counts.gaps, 4)
        self.assertEqual(counts.aligned, 67)
        self.assertEqual(counts.identities, 13)
        self.assertEqual(counts.mismatches, 54)
        self.assertEqual(counts.positives, 25)
        with self.assertRaises(StopIteration):
            next(alignments)

    def test_m8CC(self):
        # Alignment file obtained by running
        # fasta36 -q -m 8CB seq/mgstm1.aa seq/prot_test.lseg
        # in the fasta36 source distribution
        path = "Fasta/protein_m8CC.txt"
        alignments = Align.parse(path, "tabular")
        self.assertEqual(
            alignments.metadata["Command line"],
            "fasta36 -q -m 8CC seq/mgstm1.aa seq/prot_test.lseg",
        )
        self.assertEqual(alignments.metadata["Program"], "FASTA")
        self.assertEqual(alignments.metadata["Version"], "36.3.8h May, 2020")
        self.assertEqual(alignments.metadata["Database"], "seq/prot_test.lseg")
        # sp|P10649|GSTM1_MOUSE   sp|P09488|GSTM1_HUMAN
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 77.98)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 218)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 218)
        self.assertEqual(alignment.shape, (2, 218 + 0))
        self.assertEqual(alignment.sequences[0].id, "sp|P09488|GSTM1_HUMAN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 48)
        self.assertAlmostEqual(alignment.annotations["evalue"], 7.6e-83, places=84)
        self.assertAlmostEqual(alignment.annotations["bit score"], 291.9)
        self.assertEqual(
            str(alignment),
            """\
sp|P09488         0 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
sp|P10649         0 ????????????????????????????????????????????????????????????

sp|P09488        60 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
sp|P10649        60 ????????????????????????????????????????????????????????????

sp|P09488       120 ????????????????????????????????????????????????????????????
                120 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
sp|P10649       120 ????????????????????????????????????????????????????????????

sp|P09488       180 ?????????????????????????????????????? 218
                180 |||||||||||||||||||||||||||||||||||||| 218
sp|P10649       180 ?????????????????????????????????????? 218
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[0, 218], [0, 218]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 218)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "MPMILGYWDIRGLAHAIRLLLEYTDSSYEEKKYTMGDAPDYDRSQWLNEKFKLGLDFPNLPYLIDGAHKITQSNAILCYIARKHNLCGETEEEKIRVDILENQTMDNHMQLGMICYNPEFEKLKPKYLEELPEKLKLYSEFLGKRPWFAGNKITFVDFLVYDVLDLHRIFEPKCLDAFPNLKDFISRFEGLEKISAYMKSSRFLPRPVFSKMAVWGNK",
        )
        self.assertEqual(
            alignment[1],
            "MPMILGYWNVRGLTHPIRMLLEYTDSSYDEKRYTMGDAPDFDRSQWLNEKFKLGLDFPNLPYLIDGSHKITQSNAILRYLARKHHLDGETEEERIRADIVENQVMDTRMQLIMLCYNPDFEKQKPEFLKTIPEKMKLYSEFLGKRPWFAGDKVTYVDFLAYDILDQYRMFEPKCLDAFPNLRDFLARFEGLKKISAYMKSSRYIATPIFSKMAHWSNK",
        )
        # sp|P10649|GSTM1_MOUSE   sp|P00502|GSTA1_RAT
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['M', 'P', 'M', 'I', 'L', 'G', 'Y', 'W', 'D', 'I', 'R', 'G', 'L',
           'A', 'H', 'A', 'I', 'R', 'L', 'L', 'L', 'E', 'Y', 'T', 'D', 'S',
           'S', 'Y', 'E', 'E', 'K', 'K', 'Y', 'T', 'M', 'G', 'D', 'A', 'P',
           'D', 'Y', 'D', 'R', 'S', 'Q', 'W', 'L', 'N', 'E', 'K', 'F', 'K',
           'L', 'G', 'L', 'D', 'F', 'P', 'N', 'L', 'P', 'Y', 'L', 'I', 'D',
           'G', 'A', 'H', 'K', 'I', 'T', 'Q', 'S', 'N', 'A', 'I', 'L', 'C',
           'Y', 'I', 'A', 'R', 'K', 'H', 'N', 'L', 'C', 'G', 'E', 'T', 'E',
           'E', 'E', 'K', 'I', 'R', 'V', 'D', 'I', 'L', 'E', 'N', 'Q', 'T',
           'M', 'D', 'N', 'H', 'M', 'Q', 'L', 'G', 'M', 'I', 'C', 'Y', 'N',
           'P', 'E', 'F', 'E', 'K', 'L', 'K', 'P', 'K', 'Y', 'L', 'E', 'E',
           'L', 'P', 'E', 'K', 'L', 'K', 'L', 'Y', 'S', 'E', 'F', 'L', 'G',
           'K', 'R', 'P', 'W', 'F', 'A', 'G', 'N', 'K', 'I', 'T', 'F', 'V',
           'D', 'F', 'L', 'V', 'Y', 'D', 'V', 'L', 'D', 'L', 'H', 'R', 'I',
           'F', 'E', 'P', 'K', 'C', 'L', 'D', 'A', 'F', 'P', 'N', 'L', 'K',
           'D', 'F', 'I', 'S', 'R', 'F', 'E', 'G', 'L', 'E', 'K', 'I', 'S',
           'A', 'Y', 'M', 'K', 'S', 'S', 'R', 'F', 'L', 'P', 'R', 'P', 'V',
           'F', 'S', 'K', 'M', 'A', 'V', 'W', 'G', 'N', 'K'],
          ['M', 'P', 'M', 'I', 'L', 'G', 'Y', 'W', 'N', 'V', 'R', 'G', 'L',
           'T', 'H', 'P', 'I', 'R', 'M', 'L', 'L', 'E', 'Y', 'T', 'D', 'S',
           'S', 'Y', 'D', 'E', 'K', 'R', 'Y', 'T', 'M', 'G', 'D', 'A', 'P',
           'D', 'F', 'D', 'R', 'S', 'Q', 'W', 'L', 'N', 'E', 'K', 'F', 'K',
           'L', 'G', 'L', 'D', 'F', 'P', 'N', 'L', 'P', 'Y', 'L', 'I', 'D',
           'G', 'S', 'H', 'K', 'I', 'T', 'Q', 'S', 'N', 'A', 'I', 'L', 'R',
           'Y', 'L', 'A', 'R', 'K', 'H', 'H', 'L', 'D', 'G', 'E', 'T', 'E',
           'E', 'E', 'R', 'I', 'R', 'A', 'D', 'I', 'V', 'E', 'N', 'Q', 'V',
           'M', 'D', 'T', 'R', 'M', 'Q', 'L', 'I', 'M', 'L', 'C', 'Y', 'N',
           'P', 'D', 'F', 'E', 'K', 'Q', 'K', 'P', 'E', 'F', 'L', 'K', 'T',
           'I', 'P', 'E', 'K', 'M', 'K', 'L', 'Y', 'S', 'E', 'F', 'L', 'G',
           'K', 'R', 'P', 'W', 'F', 'A', 'G', 'D', 'K', 'V', 'T', 'Y', 'V',
           'D', 'F', 'L', 'A', 'Y', 'D', 'I', 'L', 'D', 'Q', 'Y', 'R', 'M',
           'F', 'E', 'P', 'K', 'C', 'L', 'D', 'A', 'F', 'P', 'N', 'L', 'R',
           'D', 'F', 'L', 'A', 'R', 'F', 'E', 'G', 'L', 'K', 'K', 'I', 'S',
           'A', 'Y', 'M', 'K', 'S', 'S', 'R', 'Y', 'I', 'A', 'T', 'P', 'I',
           'F', 'S', 'K', 'M', 'A', 'H', 'W', 'S', 'N', 'K']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 967.0; 218 aligned letters; 170 identities; 48 mismatches; 201 positives; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 967.0,
    aligned = 218:
        identities = 170,
        positives = 201,
        mismatches = 48.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 218)
        self.assertEqual(counts.identities, 170)
        self.assertEqual(counts.mismatches, 48)
        self.assertEqual(counts.positives, 201)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 29.76)
        self.assertEqual(alignment.annotations["gap opens"], 18)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 205)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 205)
        self.assertEqual(alignment.shape, (2, 205 + 18))
        self.assertEqual(alignment.sequences[0].id, "sp|P00502|GSTA1_RAT")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 144)
        self.assertAlmostEqual(alignment.annotations["evalue"], 4.4e-14, places=15)
        self.assertAlmostEqual(alignment.annotations["bit score"], 63.5)
        self.assertEqual(
            str(alignment),
            """\
sp|P00502         5 ????????????????????????????---------???????????????????????
                  0 ||||||||||||||||||||||||||||---------|||||||||||||--||||||||
sp|P10649         3 ??????????????????????????????????????????????????--????????

sp|P00502        56 ??????-?????????????????????????????????????????????????????
                 60 ||-|||-|||||||||||||||||||||||||||||||||||||||-|||||||||||||
sp|P10649        61 ??-???????????????????????????????????????????-?????????????

sp|P00502       115 ????????????????????????????????????????????????????????????
                120 ||||||||||--|||||||||||||||--|||||||||||||||||||||||||||||||
sp|P10649       119 ??????????--???????????????--???????????????????????????????

sp|P00502       175 ??????????????????????????????????????????? 218
                180 ||||||||||||||||||||||||||||||||||||||||||| 223
sp|P10649       175 ??????????????????????????????????????????? 218
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[ 5,  33,  33,  46,  48,  58,  59,  62,
                           62, 101, 102, 125, 127, 142, 144, 218],
                          [ 3,  31,  40,  53,  53,  63,  63,  66,
                           67, 106, 106, 129, 129, 144, 144, 218]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 218)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "VLHYFNARGRMECIRWLLAAAGVEFDEK---------FIQSPEDLEKLKKDGNLMFDQVPMVEIDG-MKLAQTRAILNYIATKYDLYGKDMKERALIDMYTEGILDLTEMIMQLVICPPDQKEAKTALAKDRTKNRYLPAFEKVLKSHGQDYLVGNRLTRVDIHLLELLLYVEEFDASLLTSFPLLKAFKSRISSLPNVKKFLQPGSQRKLPVDAKQIEEARK",
        )
        self.assertEqual(
            alignment[1],
            "ILGYWNVRGLTHPIRMLLEYTDSSYDEKRYTMGDAPDFDRSQWLNEKFKL--GLDFPNLPYL-IDGSHKITQSNAILRYLARKHHLDGETEEERIRADIVENQVMD-TRMQLIMLCYNPDFEKQKPEFLK--TIPEKMKLYSEFLGK--RPWFAGDKVTYVDFLAYDILDQYRMFEPKCLDAFPNLRDFLARFEGLKKISAYMKSSRYIATPIFSKMAHWSNK",
        )
        # sp|P10649|GSTM1_MOUSE   sp|P69905|HBA_HUMAN
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['V', 'L', 'H', 'Y', 'F', 'N', 'A', 'R', 'G', 'R', 'M', 'E', 'C',
           'I', 'R', 'W', 'L', 'L', 'A', 'A', 'A', 'G', 'V', 'E', 'F', 'D',
           'E', 'K', '-', '-', '-', '-', '-', '-', '-', '-', '-', 'F', 'I',
           'Q', 'S', 'P', 'E', 'D', 'L', 'E', 'K', 'L', 'K', 'K', 'D', 'G',
           'N', 'L', 'M', 'F', 'D', 'Q', 'V', 'P', 'M', 'V', 'E', 'I', 'D',
           'G', '-', 'M', 'K', 'L', 'A', 'Q', 'T', 'R', 'A', 'I', 'L', 'N',
           'Y', 'I', 'A', 'T', 'K', 'Y', 'D', 'L', 'Y', 'G', 'K', 'D', 'M',
           'K', 'E', 'R', 'A', 'L', 'I', 'D', 'M', 'Y', 'T', 'E', 'G', 'I',
           'L', 'D', 'L', 'T', 'E', 'M', 'I', 'M', 'Q', 'L', 'V', 'I', 'C',
           'P', 'P', 'D', 'Q', 'K', 'E', 'A', 'K', 'T', 'A', 'L', 'A', 'K',
           'D', 'R', 'T', 'K', 'N', 'R', 'Y', 'L', 'P', 'A', 'F', 'E', 'K',
           'V', 'L', 'K', 'S', 'H', 'G', 'Q', 'D', 'Y', 'L', 'V', 'G', 'N',
           'R', 'L', 'T', 'R', 'V', 'D', 'I', 'H', 'L', 'L', 'E', 'L', 'L',
           'L', 'Y', 'V', 'E', 'E', 'F', 'D', 'A', 'S', 'L', 'L', 'T', 'S',
           'F', 'P', 'L', 'L', 'K', 'A', 'F', 'K', 'S', 'R', 'I', 'S', 'S',
           'L', 'P', 'N', 'V', 'K', 'K', 'F', 'L', 'Q', 'P', 'G', 'S', 'Q',
           'R', 'K', 'L', 'P', 'V', 'D', 'A', 'K', 'Q', 'I', 'E', 'E', 'A',
           'R', 'K'],
          ['I', 'L', 'G', 'Y', 'W', 'N', 'V', 'R', 'G', 'L', 'T', 'H', 'P',
           'I', 'R', 'M', 'L', 'L', 'E', 'Y', 'T', 'D', 'S', 'S', 'Y', 'D',
           'E', 'K', 'R', 'Y', 'T', 'M', 'G', 'D', 'A', 'P', 'D', 'F', 'D',
           'R', 'S', 'Q', 'W', 'L', 'N', 'E', 'K', 'F', 'K', 'L', '-', '-',
           'G', 'L', 'D', 'F', 'P', 'N', 'L', 'P', 'Y', 'L', '-', 'I', 'D',
           'G', 'S', 'H', 'K', 'I', 'T', 'Q', 'S', 'N', 'A', 'I', 'L', 'R',
           'Y', 'L', 'A', 'R', 'K', 'H', 'H', 'L', 'D', 'G', 'E', 'T', 'E',
           'E', 'E', 'R', 'I', 'R', 'A', 'D', 'I', 'V', 'E', 'N', 'Q', 'V',
           'M', 'D', '-', 'T', 'R', 'M', 'Q', 'L', 'I', 'M', 'L', 'C', 'Y',
           'N', 'P', 'D', 'F', 'E', 'K', 'Q', 'K', 'P', 'E', 'F', 'L', 'K',
           '-', '-', 'T', 'I', 'P', 'E', 'K', 'M', 'K', 'L', 'Y', 'S', 'E',
           'F', 'L', 'G', 'K', '-', '-', 'R', 'P', 'W', 'F', 'A', 'G', 'D',
           'K', 'V', 'T', 'Y', 'V', 'D', 'F', 'L', 'A', 'Y', 'D', 'I', 'L',
           'D', 'Q', 'Y', 'R', 'M', 'F', 'E', 'P', 'K', 'C', 'L', 'D', 'A',
           'F', 'P', 'N', 'L', 'R', 'D', 'F', 'L', 'A', 'R', 'F', 'E', 'G',
           'L', 'K', 'K', 'I', 'S', 'A', 'Y', 'M', 'K', 'S', 'S', 'R', 'Y',
           'I', 'A', 'T', 'P', 'I', 'F', 'S', 'K', 'M', 'A', 'H', 'W', 'S',
           'N', 'K']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 257.0; 205 aligned letters; 61 identities; 144 mismatches; 102 positives; 18 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 257.0,
    aligned = 205:
        identities = 61,
        positives = 102,
        mismatches = 144.
    gaps = 18:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 18:
            internal_insertions = 10:
                open_internal_insertions = 2,
                extend_internal_insertions = 8;
            internal_deletions = 8:
                open_internal_deletions = 5,
                extend_internal_deletions = 3;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 10)
        self.assertEqual(counts.internal_deletions, 8)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 18)
        self.assertEqual(counts.insertions, 10)
        self.assertEqual(counts.deletions, 8)
        self.assertEqual(counts.gaps, 18)
        self.assertEqual(counts.aligned, 205)
        self.assertEqual(counts.identities, 61)
        self.assertEqual(counts.mismatches, 144)
        self.assertEqual(counts.positives, 102)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 27.03)
        self.assertEqual(alignment.annotations["gap opens"], 2)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 37)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 37)
        self.assertEqual(alignment.shape, (2, 37 + 2))
        self.assertEqual(alignment.sequences[0].id, "sp|P69905|HBA_HUMAN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 27)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.15)
        self.assertAlmostEqual(alignment.annotations["bit score"], 21.2)
        self.assertEqual(
            str(alignment),
            """\
sp|P69905        35 ?????????????-?????????????????????????  73
                  0 |||||||||||||-||||||||||-||||||||||||||  39
sp|P10649       176 ????????????????????????-?????????????? 214
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[ 35,  48,  48,  58,  59,  73],
                          [176, 189, 190, 200, 200, 214]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 73)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "SFPTTKTYFPHFD-LSHGSAQVKGHGKKVADALTNAVAH")
        self.assertEqual(alignment[1], "AFPNLRDFLARFEGLKKISAYMKS-SRYIATPIFSKMAH")
        # sp|P10649|GSTM1_MOUSE   sp|P00517|KAPCA_BOVIN
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['S', 'F', 'P', 'T', 'T', 'K', 'T', 'Y', 'F', 'P', 'H', 'F', 'D',
           '-', 'L', 'S', 'H', 'G', 'S', 'A', 'Q', 'V', 'K', 'G', 'H', 'G',
           'K', 'K', 'V', 'A', 'D', 'A', 'L', 'T', 'N', 'A', 'V', 'A', 'H'],
          ['A', 'F', 'P', 'N', 'L', 'R', 'D', 'F', 'L', 'A', 'R', 'F', 'E',
           'G', 'L', 'K', 'K', 'I', 'S', 'A', 'Y', 'M', 'K', 'S', '-', 'S',
           'R', 'Y', 'I', 'A', 'T', 'P', 'I', 'F', 'S', 'K', 'M', 'A', 'H']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 54.0; 37 aligned letters; 10 identities; 27 mismatches; 20 positives; 2 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 54.0,
    aligned = 37:
        identities = 10,
        positives = 20,
        mismatches = 27.
    gaps = 2:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 2:
            internal_insertions = 1:
                open_internal_insertions = 1,
                extend_internal_insertions = 0;
            internal_deletions = 1:
                open_internal_deletions = 1,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 1)
        self.assertEqual(counts.internal_deletions, 1)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 2)
        self.assertEqual(counts.insertions, 1)
        self.assertEqual(counts.deletions, 1)
        self.assertEqual(counts.gaps, 2)
        self.assertEqual(counts.aligned, 37)
        self.assertEqual(counts.identities, 10)
        self.assertEqual(counts.mismatches, 27)
        self.assertEqual(counts.positives, 20)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 24.29)
        self.assertEqual(alignment.annotations["gap opens"], 2)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 70)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 70)
        self.assertEqual(alignment.shape, (2, 70 + 2))
        self.assertEqual(alignment.sequences[0].id, "sp|P00517|KAPCA_BOVIN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 53)
        self.assertAlmostEqual(alignment.annotations["evalue"], 1.2)
        self.assertAlmostEqual(alignment.annotations["bit score"], 19.4)
        self.assertEqual(
            str(alignment),
            """\
sp|P00517       228 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||--||||||||||||
sp|P10649       136 ??????????????????????????????????????????????--????????????

sp|P00517       288 ???????????? 300
                 60 ||||||||||||  72
sp|P10649       194 ???????????? 206
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[228, 274, 276, 300],
                          [136, 182, 182, 206]]),
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 300)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "IYEMAAGYPPFFADQPIQIYEKIVSGKVRFPSHFSSDLKDLLRNLLQVDLTKRFGNLKNGVNDIKNHKWFAT",
        )
        self.assertEqual(
            alignment[1],
            "LYSEFLGKRPWFAGDKVTYVDFLAYDILDQYRMFEPKCLDAFPNLR--DFLARFEGLKKISAYMKSSRYIAT",
        )
        # sp|P10649|GSTM1_MOUSE   sp|P14960|RBS_GUITH
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['I', 'Y', 'E', 'M', 'A', 'A', 'G', 'Y', 'P', 'P', 'F', 'F', 'A',
           'D', 'Q', 'P', 'I', 'Q', 'I', 'Y', 'E', 'K', 'I', 'V', 'S', 'G',
           'K', 'V', 'R', 'F', 'P', 'S', 'H', 'F', 'S', 'S', 'D', 'L', 'K',
           'D', 'L', 'L', 'R', 'N', 'L', 'L', 'Q', 'V', 'D', 'L', 'T', 'K',
           'R', 'F', 'G', 'N', 'L', 'K', 'N', 'G', 'V', 'N', 'D', 'I', 'K',
           'N', 'H', 'K', 'W', 'F', 'A', 'T'],
          ['L', 'Y', 'S', 'E', 'F', 'L', 'G', 'K', 'R', 'P', 'W', 'F', 'A',
           'G', 'D', 'K', 'V', 'T', 'Y', 'V', 'D', 'F', 'L', 'A', 'Y', 'D',
           'I', 'L', 'D', 'Q', 'Y', 'R', 'M', 'F', 'E', 'P', 'K', 'C', 'L',
           'D', 'A', 'F', 'P', 'N', 'L', 'R', '-', '-', 'D', 'F', 'L', 'A',
           'R', 'F', 'E', 'G', 'L', 'K', 'K', 'I', 'S', 'A', 'Y', 'M', 'K',
           'S', 'S', 'R', 'Y', 'I', 'A', 'T']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 49.0; 70 aligned letters; 17 identities; 53 mismatches; 27 positives; 2 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 49.0,
    aligned = 70:
        identities = 17,
        positives = 27,
        mismatches = 53.
    gaps = 2:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 2:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 2:
                open_internal_deletions = 1,
                extend_internal_deletions = 1;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 2)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 2)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 2)
        self.assertEqual(counts.gaps, 2)
        self.assertEqual(counts.aligned, 70)
        self.assertEqual(counts.identities, 17)
        self.assertEqual(counts.mismatches, 53)
        self.assertEqual(counts.positives, 27)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 57.14)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 7)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 7)
        self.assertEqual(alignment.shape, (2, 7 + 0))
        self.assertEqual(alignment.sequences[0].id, "sp|P14960|RBS_GUITH")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 3)
        self.assertAlmostEqual(alignment.annotations["evalue"], 1.5)
        self.assertAlmostEqual(alignment.annotations["bit score"], 17.8)
        self.assertEqual(
            str(alignment),
            """\
sp|P14960        46 ??????? 53
                  0 |||||||  7
sp|P10649         6 ??????? 13
""",
        )
        self.assertTrue(
            np.array_equal(alignment.coordinates, np.array([[46, 53], [6, 13]]))
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 53)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "YWDLWGL")
        self.assertEqual(alignment[1], "YWNVRGL")
        # sp|P10649|GSTM1_MOUSE   sp|P01593|KV101_HUMAN
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['Y', 'W', 'D', 'L', 'W', 'G', 'L'],
          ['Y', 'W', 'N', 'V', 'R', 'G', 'L']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 27.0; 7 aligned letters; 4 identities; 3 mismatches; 6 positives; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 27.0,
    aligned = 7:
        identities = 4,
        positives = 6,
        mismatches = 3.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 7)
        self.assertEqual(counts.identities, 4)
        self.assertEqual(counts.mismatches, 3)
        self.assertEqual(counts.positives, 6)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 45.00)
        self.assertEqual(alignment.annotations["gap opens"], 10)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 40)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 40)
        self.assertEqual(alignment.shape, (2, 40 + 10))
        self.assertEqual(alignment.sequences[0].id, "sp|P01593|KV101_HUMAN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 22)
        self.assertAlmostEqual(alignment.annotations["evalue"], 2.4)
        self.assertAlmostEqual(alignment.annotations["bit score"], 16.7)
        self.assertEqual(
            str(alignment),
            """\
sp|P01593        15 ????????????????????????????????????-?????????????  64
                  0 ||||||||||||||---||||||||---||||--||-|||||||-|||||  50
sp|P10649       149 ??????????????---????????---????--??????????-????? 190
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[15,  29,  32,  40,  43,  47,
                           49,  51,  51,  58,  59,  64],
                         [149, 163, 163, 171, 171, 175,
                          175, 177, 178, 185, 185, 190]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 64)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0], "GDRVTITCQASQDINHYLNWYQQGPKKAPKILIYDA-SNLETGVPSRFSG"
        )
        self.assertEqual(
            alignment[1], "GDKVTYVDFLAYDI---LDQYRMFE---PKCL--DAFPNLRDFL-ARFEG"
        )
        # sp|P10649|GSTM1_MOUSE   sp|P99998|CYC_PANTR
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['G', 'D', 'R', 'V', 'T', 'I', 'T', 'C', 'Q', 'A', 'S', 'Q', 'D',
           'I', 'N', 'H', 'Y', 'L', 'N', 'W', 'Y', 'Q', 'Q', 'G', 'P', 'K',
           'K', 'A', 'P', 'K', 'I', 'L', 'I', 'Y', 'D', 'A', '-', 'S', 'N',
           'L', 'E', 'T', 'G', 'V', 'P', 'S', 'R', 'F', 'S', 'G'],
          ['G', 'D', 'K', 'V', 'T', 'Y', 'V', 'D', 'F', 'L', 'A', 'Y', 'D',
           'I', '-', '-', '-', 'L', 'D', 'Q', 'Y', 'R', 'M', 'F', 'E', '-',
           '-', '-', 'P', 'K', 'C', 'L', '-', '-', 'D', 'A', 'F', 'P', 'N',
           'L', 'R', 'D', 'F', 'L', '-', 'A', 'R', 'F', 'E', 'G']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 81.0; 40 aligned letters; 18 identities; 22 mismatches; 24 positives; 10 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 81.0,
    aligned = 40:
        identities = 18,
        positives = 24,
        mismatches = 22.
    gaps = 10:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 10:
            internal_insertions = 1:
                open_internal_insertions = 1,
                extend_internal_insertions = 0;
            internal_deletions = 9:
                open_internal_deletions = 4,
                extend_internal_deletions = 5;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 1)
        self.assertEqual(counts.internal_deletions, 9)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 10)
        self.assertEqual(counts.insertions, 1)
        self.assertEqual(counts.deletions, 9)
        self.assertEqual(counts.gaps, 10)
        self.assertEqual(counts.aligned, 40)
        self.assertEqual(counts.identities, 18)
        self.assertEqual(counts.mismatches, 22)
        self.assertEqual(counts.positives, 24)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 31.03)
        self.assertEqual(alignment.annotations["gap opens"], 10)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 58)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 58)
        self.assertEqual(alignment.shape, (2, 58 + 10))
        self.assertEqual(alignment.sequences[0].id, "sp|P99998|CYC_PANTR")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 40)
        self.assertAlmostEqual(alignment.annotations["evalue"], 2.7)
        self.assertAlmostEqual(alignment.annotations["bit score"], 16.5)
        self.assertEqual(
            str(alignment),
            """\
sp|P99998        27 ???????????????????????????????-??????????-?????--?????????-
                  0 ||||||||||||||||||||---||||||||-||||||||||-|||||--|||||||||-
sp|P10649       128 ????????????????????---?????????????????????????????????????

sp|P99998        82 --??????  88
                 60 --||||||  68
sp|P10649       185 ???????? 193
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[ 27,  47,  50,  58,  58,  68,
                            68,  73,  73,  82,  82,  88],
                          [128, 148, 148, 156, 157, 167,
                           168, 173, 175, 184, 187, 193]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 88)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "KTGPNLHGLFGRKTGQAPGYSYTAANKNKGI-IWGEDTLMEY-LENPK--KYIPGTKMI---FVGIKK",
        )
        self.assertEqual(
            alignment[1],
            "KTIPEKMKLYSEFLGKRPWF---AGDKVTYVDFLAYDILDQYRMFEPKCLDAFPNLRDFLARFEGLKK",
        )
        # sp|P10649|GSTM1_MOUSE   sp|P02585|TNNC2_HUMAN
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['K', 'T', 'G', 'P', 'N', 'L', 'H', 'G', 'L', 'F', 'G', 'R', 'K',
           'T', 'G', 'Q', 'A', 'P', 'G', 'Y', 'S', 'Y', 'T', 'A', 'A', 'N',
           'K', 'N', 'K', 'G', 'I', '-', 'I', 'W', 'G', 'E', 'D', 'T', 'L',
           'M', 'E', 'Y', '-', 'L', 'E', 'N', 'P', 'K', '-', '-', 'K', 'Y',
           'I', 'P', 'G', 'T', 'K', 'M', 'I', '-', '-', '-', 'F', 'V', 'G',
           'I', 'K', 'K'],
          ['K', 'T', 'I', 'P', 'E', 'K', 'M', 'K', 'L', 'Y', 'S', 'E', 'F',
           'L', 'G', 'K', 'R', 'P', 'W', 'F', '-', '-', '-', 'A', 'G', 'D',
           'K', 'V', 'T', 'Y', 'V', 'D', 'F', 'L', 'A', 'Y', 'D', 'I', 'L',
           'D', 'Q', 'Y', 'R', 'M', 'F', 'E', 'P', 'K', 'C', 'L', 'D', 'A',
           'F', 'P', 'N', 'L', 'R', 'D', 'F', 'L', 'A', 'R', 'F', 'E', 'G',
           'L', 'K', 'K']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 76.0; 58 aligned letters; 18 identities; 40 mismatches; 27 positives; 10 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 76.0,
    aligned = 58:
        identities = 18,
        positives = 27,
        mismatches = 40.
    gaps = 10:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 10:
            internal_insertions = 7:
                open_internal_insertions = 4,
                extend_internal_insertions = 3;
            internal_deletions = 3:
                open_internal_deletions = 1,
                extend_internal_deletions = 2;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 7)
        self.assertEqual(counts.internal_deletions, 3)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 10)
        self.assertEqual(counts.insertions, 7)
        self.assertEqual(counts.deletions, 3)
        self.assertEqual(counts.gaps, 10)
        self.assertEqual(counts.aligned, 58)
        self.assertEqual(counts.identities, 18)
        self.assertEqual(counts.mismatches, 40)
        self.assertEqual(counts.positives, 27)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 31.11)
        self.assertEqual(alignment.annotations["gap opens"], 9)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 45)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 45)
        self.assertEqual(alignment.shape, (2, 45 + 9))
        self.assertEqual(alignment.sequences[0].id, "sp|P02585|TNNC2_HUMAN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 31)
        self.assertAlmostEqual(alignment.annotations["evalue"], 3.8)
        self.assertAlmostEqual(alignment.annotations["bit score"], 16.5)
        self.assertEqual(
            str(alignment),
            """\
sp|P02585        12 ??????????????----???????????????????????????????????? 62
                  0 ||||||||||||||----||||||||||--|||||||||||---|||||||||| 54
sp|P10649        43 ????????????????????????????--???????????---?????????? 92
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[12, 26, 26, 36, 38, 49, 52, 62],
                          [43, 57, 61, 71, 71, 82, 82, 92]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 62)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0], "SEEMIAEFKAAFDM----FDADGGGDISVKELGTVMRMLGQTPTKEELDAIIEE"
        )
        self.assertEqual(
            alignment[1], "SQWLNEKFKLGLDFPNLPYLIDGSHKIT--QSNAILRYLAR---KHHLDGETEE"
        )
        # sp|P10649|GSTM1_MOUSE   sp|P60615|NXL1A_BUNMU
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['S', 'E', 'E', 'M', 'I', 'A', 'E', 'F', 'K', 'A', 'A', 'F', 'D',
           'M', '-', '-', '-', '-', 'F', 'D', 'A', 'D', 'G', 'G', 'G', 'D',
           'I', 'S', 'V', 'K', 'E', 'L', 'G', 'T', 'V', 'M', 'R', 'M', 'L',
           'G', 'Q', 'T', 'P', 'T', 'K', 'E', 'E', 'L', 'D', 'A', 'I', 'I',
           'E', 'E'],
          ['S', 'Q', 'W', 'L', 'N', 'E', 'K', 'F', 'K', 'L', 'G', 'L', 'D',
           'F', 'P', 'N', 'L', 'P', 'Y', 'L', 'I', 'D', 'G', 'S', 'H', 'K',
           'I', 'T', '-', '-', 'Q', 'S', 'N', 'A', 'I', 'L', 'R', 'Y', 'L',
           'A', 'R', '-', '-', '-', 'K', 'H', 'H', 'L', 'D', 'G', 'E', 'T',
           'E', 'E']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 65.0; 45 aligned letters; 14 identities; 31 mismatches; 23 positives; 9 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 65.0,
    aligned = 45:
        identities = 14,
        positives = 23,
        mismatches = 31.
    gaps = 9:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 9:
            internal_insertions = 4:
                open_internal_insertions = 1,
                extend_internal_insertions = 3;
            internal_deletions = 5:
                open_internal_deletions = 2,
                extend_internal_deletions = 3;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 4)
        self.assertEqual(counts.internal_deletions, 5)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 9)
        self.assertEqual(counts.insertions, 4)
        self.assertEqual(counts.deletions, 5)
        self.assertEqual(counts.gaps, 9)
        self.assertEqual(counts.aligned, 45)
        self.assertEqual(counts.identities, 14)
        self.assertEqual(counts.mismatches, 31)
        self.assertEqual(counts.positives, 23)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 66.67)
        self.assertEqual(alignment.annotations["gap opens"], 2)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 9)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 9)
        self.assertEqual(alignment.shape, (2, 9 + 2))
        self.assertEqual(alignment.sequences[0].id, "sp|P60615|NXL1A_BUNMU")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 3)
        self.assertAlmostEqual(alignment.annotations["evalue"], 4.2)
        self.assertAlmostEqual(alignment.annotations["bit score"], 15.6)
        self.assertEqual(
            str(alignment),
            """\
sp|P60615        85 ?-???-?????  94
                  0 |-|||-|||||  11
sp|P10649       114 ??????????? 125
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[ 85,  86,  86,  89,  89,  94],
                          [114, 115, 116, 119, 120, 125]]),
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 94)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "C-NPH-PKQRP")
        self.assertEqual(alignment[1], "CYNPDFEKQKP")
        # sp|P10649|GSTM1_MOUSE   sp|P03435|HEMA_I75A3
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', '-', 'N', 'P', 'H', '-', 'P', 'K', 'Q', 'R', 'P'],
          ['C', 'Y', 'N', 'P', 'D', 'F', 'E', 'K', 'Q', 'K', 'P']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 39.0; 9 aligned letters; 6 identities; 3 mismatches; 7 positives; 2 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 39.0,
    aligned = 9:
        identities = 6,
        positives = 7,
        mismatches = 3.
    gaps = 2:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 2:
            internal_insertions = 2:
                open_internal_insertions = 2,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 2)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 2)
        self.assertEqual(counts.insertions, 2)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 2)
        self.assertEqual(counts.aligned, 9)
        self.assertEqual(counts.identities, 6)
        self.assertEqual(counts.mismatches, 3)
        self.assertEqual(counts.positives, 7)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 28.95)
        self.assertEqual(alignment.annotations["gap opens"], 1)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 38)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 38)
        self.assertEqual(alignment.shape, (2, 38 + 1))
        self.assertEqual(alignment.sequences[0].id, "sp|P03435|HEMA_I75A3")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 27)
        self.assertAlmostEqual(alignment.annotations["evalue"], 4.4)
        self.assertAlmostEqual(alignment.annotations["bit score"], 18.1)
        self.assertEqual(
            str(alignment),
            """\
sp|P03435       398 ??????????????????????????????????????? 437
                  0 ||||||||||-||||||||||||||||||||||||||||  39
sp|P10649        73 ??????????-???????????????????????????? 111
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[398, 408, 409, 437],
                          [ 73,  83,  83, 111]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 437)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "NRVIEKTNEKFHQIEKEFSEVEGRIQDLEKYVEDTKIDL")
        self.assertEqual(alignment[1], "NAILRYLARK-HHLDGETEEERIRADIVENQVMDTRMQL")
        # sp|P10649|GSTM1_MOUSE   sp|P00193|FER_PEPAS
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['N', 'R', 'V', 'I', 'E', 'K', 'T', 'N', 'E', 'K', 'F', 'H', 'Q',
           'I', 'E', 'K', 'E', 'F', 'S', 'E', 'V', 'E', 'G', 'R', 'I', 'Q',
           'D', 'L', 'E', 'K', 'Y', 'V', 'E', 'D', 'T', 'K', 'I', 'D', 'L'],
          ['N', 'A', 'I', 'L', 'R', 'Y', 'L', 'A', 'R', 'K', '-', 'H', 'H',
           'L', 'D', 'G', 'E', 'T', 'E', 'E', 'E', 'R', 'I', 'R', 'A', 'D',
           'I', 'V', 'E', 'N', 'Q', 'V', 'M', 'D', 'T', 'R', 'M', 'Q', 'L']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 48.0; 38 aligned letters; 11 identities; 27 mismatches; 18 positives; 1 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 48.0,
    aligned = 38:
        identities = 11,
        positives = 18,
        mismatches = 27.
    gaps = 1:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 1:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 1:
                open_internal_deletions = 1,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 1)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 1)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 1)
        self.assertEqual(counts.gaps, 1)
        self.assertEqual(counts.aligned, 38)
        self.assertEqual(counts.identities, 11)
        self.assertEqual(counts.mismatches, 27)
        self.assertEqual(counts.positives, 18)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 50.00)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 4)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 4)
        self.assertEqual(alignment.shape, (2, 4 + 0))
        self.assertEqual(alignment.sequences[0].id, "sp|P00193|FER_PEPAS")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 2)
        self.assertAlmostEqual(alignment.annotations["evalue"], 4.4)
        self.assertAlmostEqual(alignment.annotations["bit score"], 14.7)
        self.assertEqual(
            str(alignment),
            """\
sp|P00193        14 ????  18
                  0 ||||   4
sp|P10649       170 ???? 174
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[14, 18], [170, 174]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 18)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "KPEC")
        self.assertEqual(alignment[1], "EPKC")
        # sp|P10649|GSTM1_MOUSE   sp|P01834|IGKC_HUMAN
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['K', 'P', 'E', 'C'],
          ['E', 'P', 'K', 'C']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 18.0; 4 aligned letters; 2 identities; 2 mismatches; 4 positives; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 18.0,
    aligned = 4:
        identities = 2,
        positives = 4,
        mismatches = 2.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 4)
        self.assertEqual(counts.identities, 2)
        self.assertEqual(counts.mismatches, 2)
        self.assertEqual(counts.positives, 4)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 19.40)
        self.assertEqual(alignment.annotations["gap opens"], 4)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 67)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 67)
        self.assertEqual(alignment.shape, (2, 67 + 4))
        self.assertEqual(alignment.sequences[0].id, "sp|P01834|IGKC_HUMAN")
        self.assertEqual(alignment.sequences[1].id, "sp|P10649|GSTM1_MOUSE")
        self.assertEqual(alignment.annotations["mismatches"], 54)
        self.assertAlmostEqual(alignment.annotations["evalue"], 6.4)
        self.assertAlmostEqual(alignment.annotations["bit score"], 14.9)
        self.assertEqual(
            str(alignment),
            """\
sp|P01834        11 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||--||||||||||||||||||||||||||||||||||||||||--||
sp|P10649        57 ??????????????--????????????????????????????????????????--??

sp|P01834        71 ???????????  82
                 60 |||||||||||  71
sp|P10649       113 ??????????? 124
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[11, 25, 27,  67,  69,  82],
                          [57, 71, 71, 111, 111, 124]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 82)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "PSDEQLKSGTASVVCLLNNFYPREAKVQWKVDNALQSGNSQESVTEQDSKDSTYSLSSTLTLSKADYEKHK",
        )
        self.assertEqual(
            alignment[1],
            "PNLPYLIDGSHKIT--QSNAILRYLARKHHLDGETEEERIRADIVENQVMDTRMQL--IMLCYNPDFEKQK",
        )
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['P', 'S', 'D', 'E', 'Q', 'L', 'K', 'S', 'G', 'T', 'A', 'S', 'V',
           'V', 'C', 'L', 'L', 'N', 'N', 'F', 'Y', 'P', 'R', 'E', 'A', 'K',
           'V', 'Q', 'W', 'K', 'V', 'D', 'N', 'A', 'L', 'Q', 'S', 'G', 'N',
           'S', 'Q', 'E', 'S', 'V', 'T', 'E', 'Q', 'D', 'S', 'K', 'D', 'S',
           'T', 'Y', 'S', 'L', 'S', 'S', 'T', 'L', 'T', 'L', 'S', 'K', 'A',
           'D', 'Y', 'E', 'K', 'H', 'K'],
          ['P', 'N', 'L', 'P', 'Y', 'L', 'I', 'D', 'G', 'S', 'H', 'K', 'I',
           'T', '-', '-', 'Q', 'S', 'N', 'A', 'I', 'L', 'R', 'Y', 'L', 'A',
           'R', 'K', 'H', 'H', 'L', 'D', 'G', 'E', 'T', 'E', 'E', 'E', 'R',
           'I', 'R', 'A', 'D', 'I', 'V', 'E', 'N', 'Q', 'V', 'M', 'D', 'T',
           'R', 'M', 'Q', 'L', '-', '-', 'I', 'M', 'L', 'C', 'Y', 'N', 'P',
           'D', 'F', 'E', 'K', 'Q', 'K']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts(substitution_matrix)
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (substitution score = 43.0; 67 aligned letters; 13 identities; 54 mismatches; 25 positives; 4 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    substitution_score = 43.0,
    aligned = 67:
        identities = 13,
        positives = 25,
        mismatches = 54.
    gaps = 4:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 4:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 4:
                open_internal_deletions = 2,
                extend_internal_deletions = 2;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 4)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 4)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 4)
        self.assertEqual(counts.gaps, 4)
        self.assertEqual(counts.aligned, 67)
        self.assertEqual(counts.identities, 13)
        self.assertEqual(counts.mismatches, 54)
        self.assertEqual(counts.positives, 25)
        with self.assertRaises(StopIteration):
            next(alignments)


class TestFastaNucleotide(unittest.TestCase):
    query = Seq(
        "ATGCCTATGATACTGGGATACTGGAACGTCCGCGGACTGACACACCCGATCCGCATGCTCCTGGAATACACAGACTCAAGCTATGATGAGAAGAGATACACCATGGGTGACGCTCCCGACTTTGACAGAAGCCAGTGGCTGAATGAGAAGTTCAAGCTGGGCCTGGACTTTCCCAATCTGCCTTACTTGATCGATGGATCACACAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGATGGAGAGACAGAGGAGGAGAGGATCCGTGCAGACATTGTGGAGAACCAGGTCATGGACACCCGCATGCAGCTCATCATGCTCTGTTACAACCCTGACTTTGAGAAGCAGAAGCCAGAGTTCTTGAAGACCATCCCTGAGAAAATGAAGCTCTACTCTGAGTTCCTGGGCAAGAGGCCATGGTTTGCAGGGGACAAGGTCACCTATGTGGATTTCCTTGCTTATGACATTCTTGACCAGTACCGTATGTTTGAGCCCAAGTGCCTGGACGCCTTCCCAAACCTGAGGGACTTCCTGGCCCGCTTCGAGGGCCTCAAGAAGATCTCTGCCTACATGAAGAGTAGCCGCTACATCGCAACACCTATATTTTCAAAGATGGCCCACTGGAGTAACAAGTAG"
    )

    filename = os.path.join("Fasta", "nucleotide_lib.fa")
    records = SeqIO.parse(filename, "fasta")
    targets = {record.id: record.seq.upper() for record in records}

    def test_m8CB(self):
        # Alignment file obtained by running
        # fasta36 -m 8CB seq/mgstm1.nt seq/gst.nlib
        # in the fasta36 source distribution
        path = "Fasta/nucleotide_m8CB.txt"
        alignments = Align.parse(path, "tabular")
        self.assertEqual(
            alignments.metadata["Command line"],
            "fasta36 -m 8CB seq/mgstm1.nt seq/gst.nlib",
        )
        self.assertEqual(alignments.metadata["Program"], "FASTA")
        self.assertEqual(alignments.metadata["Version"], "36.3.8h May, 2020")
        self.assertEqual(alignments.metadata["Database"], "seq/gst.nlib")
        # pGT875   pGT875
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 657)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 657)
        self.assertEqual(alignment.shape, (2, 657 + 0))
        self.assertEqual(alignment.sequences[0].id, "pGT875")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 0)
        self.assertAlmostEqual(alignment.annotations["evalue"], 3.6e-194, places=195)
        self.assertAlmostEqual(alignment.annotations["bit score"], 666.0)
        self.assertEqual(
            str(alignment),
            """\
pGT875           37 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875            0 ????????????????????????????????????????????????????????????

pGT875           97 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875           60 ????????????????????????????????????????????????????????????

pGT875          157 ????????????????????????????????????????????????????????????
                120 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          120 ????????????????????????????????????????????????????????????

pGT875          217 ????????????????????????????????????????????????????????????
                180 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          180 ????????????????????????????????????????????????????????????

pGT875          277 ????????????????????????????????????????????????????????????
                240 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          240 ????????????????????????????????????????????????????????????

pGT875          337 ????????????????????????????????????????????????????????????
                300 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          300 ????????????????????????????????????????????????????????????

pGT875          397 ????????????????????????????????????????????????????????????
                360 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          360 ????????????????????????????????????????????????????????????

pGT875          457 ????????????????????????????????????????????????????????????
                420 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          420 ????????????????????????????????????????????????????????????

pGT875          517 ????????????????????????????????????????????????????????????
                480 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          480 ????????????????????????????????????????????????????????????

pGT875          577 ????????????????????????????????????????????????????????????
                540 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          540 ????????????????????????????????????????????????????????????

pGT875          637 ????????????????????????????????????????????????????????? 694
                600 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||| 657
pGT875          600 ????????????????????????????????????????????????????????? 657
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[37, 694], [0, 657]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 694)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "ATGCCTATGATACTGGGATACTGGAACGTCCGCGGACTGACACACCCGATCCGCATGCTCCTGGAATACACAGACTCAAGCTATGATGAGAAGAGATACACCATGGGTGACGCTCCCGACTTTGACAGAAGCCAGTGGCTGAATGAGAAGTTCAAGCTGGGCCTGGACTTTCCCAATCTGCCTTACTTGATCGATGGATCACACAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGATGGAGAGACAGAGGAGGAGAGGATCCGTGCAGACATTGTGGAGAACCAGGTCATGGACACCCGCATGCAGCTCATCATGCTCTGTTACAACCCTGACTTTGAGAAGCAGAAGCCAGAGTTCTTGAAGACCATCCCTGAGAAAATGAAGCTCTACTCTGAGTTCCTGGGCAAGAGGCCATGGTTTGCAGGGGACAAGGTCACCTATGTGGATTTCCTTGCTTATGACATTCTTGACCAGTACCGTATGTTTGAGCCCAAGTGCCTGGACGCCTTCCCAAACCTGAGGGACTTCCTGGCCCGCTTCGAGGGCCTCAAGAAGATCTCTGCCTACATGAAGAGTAGCCGCTACATCGCAACACCTATATTTTCAAAGATGGCCCACTGGAGTAACAAGTAG",
        )
        self.assertEqual(
            alignment[1],
            "ATGCCTATGATACTGGGATACTGGAACGTCCGCGGACTGACACACCCGATCCGCATGCTCCTGGAATACACAGACTCAAGCTATGATGAGAAGAGATACACCATGGGTGACGCTCCCGACTTTGACAGAAGCCAGTGGCTGAATGAGAAGTTCAAGCTGGGCCTGGACTTTCCCAATCTGCCTTACTTGATCGATGGATCACACAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGATGGAGAGACAGAGGAGGAGAGGATCCGTGCAGACATTGTGGAGAACCAGGTCATGGACACCCGCATGCAGCTCATCATGCTCTGTTACAACCCTGACTTTGAGAAGCAGAAGCCAGAGTTCTTGAAGACCATCCCTGAGAAAATGAAGCTCTACTCTGAGTTCCTGGGCAAGAGGCCATGGTTTGCAGGGGACAAGGTCACCTATGTGGATTTCCTTGCTTATGACATTCTTGACCAGTACCGTATGTTTGAGCCCAAGTGCCTGGACGCCTTCCCAAACCTGAGGGACTTCCTGGCCCGCTTCGAGGGCCTCAAGAAGATCTCTGCCTACATGAAGAGTAGCCGCTACATCGCAACACCTATATTTTCAAAGATGGCCCACTGGAGTAACAAGTAG",
        )
        # pGT875   RABGLTR
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (657 aligned letters; 657 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 657:
        identities = 657,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 657)
        self.assertEqual(counts.identities, 657)
        self.assertEqual(counts.mismatches, 0)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 79.10)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 646)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 646)
        self.assertEqual(alignment.shape, (2, 646 + 0))
        self.assertEqual(alignment.sequences[0].id, "RABGLTR")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 135)
        self.assertAlmostEqual(alignment.annotations["evalue"], 1.9e-118, places=119)
        self.assertAlmostEqual(alignment.annotations["bit score"], 414.4)
        self.assertEqual(
            str(alignment),
            """\
RABGLTR          33 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875            0 ????????????????????????????????????????????????????????????

RABGLTR          93 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875           60 ????????????????????????????????????????????????????????????

RABGLTR         153 ????????????????????????????????????????????????????????????
                120 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          120 ????????????????????????????????????????????????????????????

RABGLTR         213 ????????????????????????????????????????????????????????????
                180 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          180 ????????????????????????????????????????????????????????????

RABGLTR         273 ????????????????????????????????????????????????????????????
                240 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          240 ????????????????????????????????????????????????????????????

RABGLTR         333 ????????????????????????????????????????????????????????????
                300 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          300 ????????????????????????????????????????????????????????????

RABGLTR         393 ????????????????????????????????????????????????????????????
                360 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          360 ????????????????????????????????????????????????????????????

RABGLTR         453 ????????????????????????????????????????????????????????????
                420 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          420 ????????????????????????????????????????????????????????????

RABGLTR         513 ????????????????????????????????????????????????????????????
                480 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          480 ????????????????????????????????????????????????????????????

RABGLTR         573 ????????????????????????????????????????????????????????????
                540 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          540 ????????????????????????????????????????????????????????????

RABGLTR         633 ?????????????????????????????????????????????? 679
                600 |||||||||||||||||||||||||||||||||||||||||||||| 646
pGT875          600 ?????????????????????????????????????????????? 646
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[33, 679], [0, 646]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 679)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "ATGCCCATGACGCTGGGTTACTGGGACGTCCGTGGGCTGGCTCTGCCAATCCGCATGCTCCTGGAATACACGGACACCAGCTATGAGGAAAAGAAATACACCATGGGGGATGCTCCCAACTATGACCAAAGCAAGTGGCTGAGTGAGAAGTTCACCCTGGGCCTGGACTTTCCCAATCTGCCCTACCTAATTGATGGGACTCACAAGCTCACGCAGAGCAACGCCATCCTGCGCTACCTGGCCCGCAAGCACGGCCTGTGTGGGGAGACGGAAGAGGAGAGGATTCGCGTGGACATTCTGGAGAATCAGCTGATGGACAACCGCTTCCAACTTGTAAACGTCTGCTACAGTCCCGACTTTGAGAAGCTCAAGCCCGAGTACCTGAAGGGGCTCCCTGAGAAGCTGCAGCTGTACTCGCAGTTCCTGGGAAGCCTCCCCTGGTTCGCAGGGGACAAGATCACCTTCGCCGATTTCCTTGTCTACGACGTTCTTGACCAGAACCGGATATTTGTGCCTGGGTGCCTGGACGCGTTCCCAAACCTGAAGGACTTTCATGTCCGCTTTGAGGGCCTGCCGAAGATCTCTGCCTACATGAAGTCCAGCCGCTTTATCCGAGTCCCTGTGTTTTTAAAGAAGGCCACGTGGA",
        )
        self.assertEqual(
            alignment[1],
            "ATGCCTATGATACTGGGATACTGGAACGTCCGCGGACTGACACACCCGATCCGCATGCTCCTGGAATACACAGACTCAAGCTATGATGAGAAGAGATACACCATGGGTGACGCTCCCGACTTTGACAGAAGCCAGTGGCTGAATGAGAAGTTCAAGCTGGGCCTGGACTTTCCCAATCTGCCTTACTTGATCGATGGATCACACAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGATGGAGAGACAGAGGAGGAGAGGATCCGTGCAGACATTGTGGAGAACCAGGTCATGGACACCCGCATGCAGCTCATCATGCTCTGTTACAACCCTGACTTTGAGAAGCAGAAGCCAGAGTTCTTGAAGACCATCCCTGAGAAAATGAAGCTCTACTCTGAGTTCCTGGGCAAGAGGCCATGGTTTGCAGGGGACAAGGTCACCTATGTGGATTTCCTTGCTTATGACATTCTTGACCAGTACCGTATGTTTGAGCCCAAGTGCCTGGACGCCTTCCCAAACCTGAGGGACTTCCTGGCCCGCTTCGAGGGCCTCAAGAAGATCTCTGCCTACATGAAGAGTAGCCGCTACATCGCAACACCTATATTTTCAAAGATGGCCCACTGGA",
        )
        # pGT875   BTGST
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (646 aligned letters; 511 identities; 135 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 646:
        identities = 511,
        mismatches = 135.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 646)
        self.assertEqual(counts.identities, 511)
        self.assertEqual(counts.mismatches, 135)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 59.56)
        self.assertEqual(alignment.annotations["gap opens"], 21)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 413)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 413)
        self.assertEqual(alignment.shape, (2, 413 + 21))
        self.assertEqual(alignment.sequences[0].id, "BTGST")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 167)
        self.assertAlmostEqual(alignment.annotations["evalue"], 1.2e-07, places=8)
        self.assertAlmostEqual(alignment.annotations["bit score"], 46.4)
        self.assertEqual(
            str(alignment),
            """\
BTGST           227 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          175 ????????????????????????????????????????????????????????????

BTGST           287 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          235 ????????????????????????????????????????????????????????????

BTGST           347 ?????????????????????????????-?????????????????????????---??
                120 |||||||||||||||||||||||||||||-|||||||-|||||||||||||||||---||
pGT875          295 ?????????????????????????????????????-??????????????????????

BTGST           403 ????????????????????????????????????????????????????????????
                180 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||--
pGT875          354 ??????????????????????????????????????????????????????????--

BTGST           463 ????????????????????????????????????????????????????????????
                240 ---||||||-|||||||||||||--|||||||||||||----||||||||||||||||||
pGT875          412 ---??????-?????????????--?????????????----??????????????????

BTGST           523 ????????????????????--??????????????????????????????????????
                300 ||||||||||||--||||||--||||||||||||||||||||||||||||||||||||||
pGT875          462 ????????????--??????????????????????????????????????????????

BTGST           581 ????????????????????????????????????????????????????????????
                360 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          520 ????????????????????????????????????????????????????????????

BTGST           641 ?????????????? 655
                420 |||||||||||||| 434
pGT875          580 ?????????????? 594
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[227, 376, 376, 383, 384, 401, 401, 461, 466, 472,
                           473, 486, 488, 501, 505, 535, 537, 543, 543, 655],
                          [175, 324, 325, 332, 332, 349, 352, 412, 412, 418,
                           418, 431, 431, 444, 444, 474, 474, 480, 482, 594]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 655)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "AGCTCCCCAAGTTCCAGGACGGAGACCTCACGCTGTACCAGTCCAATGCCATCCTGCGGCACCTGGGCCGCACCCTCGGGCTGTATGGGAAGGACCAGCAGGAGGCGGCCCTGGTGGACATGGTGAATGACGGTGTAGAGGACCTTCGC-TGCAAATACGTCTCCCTCATTTACA---CCAACTACGAGGCGGGCAAGGAGGACTATGTGAAGGCGCTGCCCCAGCACCTGAAGCCTTTCGAGACCCTGCTGTCCCAGAACAAGGGTGGCCAGGCCTTCATCGTGGGCGACCAGATCTCCTTTGCGGACTACAACCTGCT--GGACCTGCTTCGGATTCACCAGGTCCTGGCCCCCAGCTGTCTGGACTCCTTCCCCCTGCTCTCAGCCTACGTGGCCCGTCTCAACTCCCGGCCCAAGCTCAAGGCCTTCCTG",
        )
        self.assertEqual(
            alignment[1],
            "ATCTGCCTTACTTGATCGATGGATCACACAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGATGGAGAGACAGAGGAGGAGAGGATCCGTGCAGACATTGTGGAGAACCAGGTCATGGACACCCGCATGCAGCT-CATCATGCTCTGTTACAACCCTGACTTTGAGAAGCAGAAGCCAGAGTTCTTGAAGACCATCCCTGAGAAAATGAAGCTCT-----ACTCTG-AGTTCCTGGGCAA--GAGGCCATGGTTT----GCAGGGGACAAGGTCACCTATGTGGATTTC--CTTGCTTATGACATTCTTGACCAGTACCGTATGTTTGAGCCCAAGTGCCTGGACGCCTTCCCAAACCTGAGGGACTTCCTGGCCCGCTTCGAGGGCCTCAAGAAGATCTCTGCCTACATG",
        )
        # pGT875   RABGSTB
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['A', 'G', 'C', 'T', 'C', 'C', 'C', 'C', 'A', 'A', 'G', 'T', 'T',
           'C', 'C', 'A', 'G', 'G', 'A', 'C', 'G', 'G', 'A', 'G', 'A', 'C',
           'C', 'T', 'C', 'A', 'C', 'G', 'C', 'T', 'G', 'T', 'A', 'C', 'C',
           'A', 'G', 'T', 'C', 'C', 'A', 'A', 'T', 'G', 'C', 'C', 'A', 'T',
           'C', 'C', 'T', 'G', 'C', 'G', 'G', 'C', 'A', 'C', 'C', 'T', 'G',
           'G', 'G', 'C', 'C', 'G', 'C', 'A', 'C', 'C', 'C', 'T', 'C', 'G',
           'G', 'G', 'C', 'T', 'G', 'T', 'A', 'T', 'G', 'G', 'G', 'A', 'A',
           'G', 'G', 'A', 'C', 'C', 'A', 'G', 'C', 'A', 'G', 'G', 'A', 'G',
           'G', 'C', 'G', 'G', 'C', 'C', 'C', 'T', 'G', 'G', 'T', 'G', 'G',
           'A', 'C', 'A', 'T', 'G', 'G', 'T', 'G', 'A', 'A', 'T', 'G', 'A',
           'C', 'G', 'G', 'T', 'G', 'T', 'A', 'G', 'A', 'G', 'G', 'A', 'C',
           'C', 'T', 'T', 'C', 'G', 'C', '-', 'T', 'G', 'C', 'A', 'A', 'A',
           'T', 'A', 'C', 'G', 'T', 'C', 'T', 'C', 'C', 'C', 'T', 'C', 'A',
           'T', 'T', 'T', 'A', 'C', 'A', '-', '-', '-', 'C', 'C', 'A', 'A',
           'C', 'T', 'A', 'C', 'G', 'A', 'G', 'G', 'C', 'G', 'G', 'G', 'C',
           'A', 'A', 'G', 'G', 'A', 'G', 'G', 'A', 'C', 'T', 'A', 'T', 'G',
           'T', 'G', 'A', 'A', 'G', 'G', 'C', 'G', 'C', 'T', 'G', 'C', 'C',
           'C', 'C', 'A', 'G', 'C', 'A', 'C', 'C', 'T', 'G', 'A', 'A', 'G',
           'C', 'C', 'T', 'T', 'T', 'C', 'G', 'A', 'G', 'A', 'C', 'C', 'C',
           'T', 'G', 'C', 'T', 'G', 'T', 'C', 'C', 'C', 'A', 'G', 'A', 'A',
           'C', 'A', 'A', 'G', 'G', 'G', 'T', 'G', 'G', 'C', 'C', 'A', 'G',
           'G', 'C', 'C', 'T', 'T', 'C', 'A', 'T', 'C', 'G', 'T', 'G', 'G',
           'G', 'C', 'G', 'A', 'C', 'C', 'A', 'G', 'A', 'T', 'C', 'T', 'C',
           'C', 'T', 'T', 'T', 'G', 'C', 'G', 'G', 'A', 'C', 'T', 'A', 'C',
           'A', 'A', 'C', 'C', 'T', 'G', 'C', 'T', '-', '-', 'G', 'G', 'A',
           'C', 'C', 'T', 'G', 'C', 'T', 'T', 'C', 'G', 'G', 'A', 'T', 'T',
           'C', 'A', 'C', 'C', 'A', 'G', 'G', 'T', 'C', 'C', 'T', 'G', 'G',
           'C', 'C', 'C', 'C', 'C', 'A', 'G', 'C', 'T', 'G', 'T', 'C', 'T',
           'G', 'G', 'A', 'C', 'T', 'C', 'C', 'T', 'T', 'C', 'C', 'C', 'C',
           'C', 'T', 'G', 'C', 'T', 'C', 'T', 'C', 'A', 'G', 'C', 'C', 'T',
           'A', 'C', 'G', 'T', 'G', 'G', 'C', 'C', 'C', 'G', 'T', 'C', 'T',
           'C', 'A', 'A', 'C', 'T', 'C', 'C', 'C', 'G', 'G', 'C', 'C', 'C',
           'A', 'A', 'G', 'C', 'T', 'C', 'A', 'A', 'G', 'G', 'C', 'C', 'T',
           'T', 'C', 'C', 'T', 'G'],
          ['A', 'T', 'C', 'T', 'G', 'C', 'C', 'T', 'T', 'A', 'C', 'T', 'T',
           'G', 'A', 'T', 'C', 'G', 'A', 'T', 'G', 'G', 'A', 'T', 'C', 'A',
           'C', 'A', 'C', 'A', 'A', 'G', 'A', 'T', 'C', 'A', 'C', 'C', 'C',
           'A', 'G', 'A', 'G', 'C', 'A', 'A', 'T', 'G', 'C', 'C', 'A', 'T',
           'C', 'C', 'T', 'G', 'C', 'G', 'C', 'T', 'A', 'C', 'C', 'T', 'T',
           'G', 'C', 'C', 'C', 'G', 'A', 'A', 'A', 'G', 'C', 'A', 'C', 'C',
           'A', 'C', 'C', 'T', 'G', 'G', 'A', 'T', 'G', 'G', 'A', 'G', 'A',
           'G', 'A', 'C', 'A', 'G', 'A', 'G', 'G', 'A', 'G', 'G', 'A', 'G',
           'A', 'G', 'G', 'A', 'T', 'C', 'C', 'G', 'T', 'G', 'C', 'A', 'G',
           'A', 'C', 'A', 'T', 'T', 'G', 'T', 'G', 'G', 'A', 'G', 'A', 'A',
           'C', 'C', 'A', 'G', 'G', 'T', 'C', 'A', 'T', 'G', 'G', 'A', 'C',
           'A', 'C', 'C', 'C', 'G', 'C', 'A', 'T', 'G', 'C', 'A', 'G', 'C',
           'T', '-', 'C', 'A', 'T', 'C', 'A', 'T', 'G', 'C', 'T', 'C', 'T',
           'G', 'T', 'T', 'A', 'C', 'A', 'A', 'C', 'C', 'C', 'T', 'G', 'A',
           'C', 'T', 'T', 'T', 'G', 'A', 'G', 'A', 'A', 'G', 'C', 'A', 'G',
           'A', 'A', 'G', 'C', 'C', 'A', 'G', 'A', 'G', 'T', 'T', 'C', 'T',
           'T', 'G', 'A', 'A', 'G', 'A', 'C', 'C', 'A', 'T', 'C', 'C', 'C',
           'T', 'G', 'A', 'G', 'A', 'A', 'A', 'A', 'T', 'G', 'A', 'A', 'G',
           'C', 'T', 'C', 'T', '-', '-', '-', '-', '-', 'A', 'C', 'T', 'C',
           'T', 'G', '-', 'A', 'G', 'T', 'T', 'C', 'C', 'T', 'G', 'G', 'G',
           'C', 'A', 'A', '-', '-', 'G', 'A', 'G', 'G', 'C', 'C', 'A', 'T',
           'G', 'G', 'T', 'T', 'T', '-', '-', '-', '-', 'G', 'C', 'A', 'G',
           'G', 'G', 'G', 'A', 'C', 'A', 'A', 'G', 'G', 'T', 'C', 'A', 'C',
           'C', 'T', 'A', 'T', 'G', 'T', 'G', 'G', 'A', 'T', 'T', 'T', 'C',
           '-', '-', 'C', 'T', 'T', 'G', 'C', 'T', 'T', 'A', 'T', 'G', 'A',
           'C', 'A', 'T', 'T', 'C', 'T', 'T', 'G', 'A', 'C', 'C', 'A', 'G',
           'T', 'A', 'C', 'C', 'G', 'T', 'A', 'T', 'G', 'T', 'T', 'T', 'G',
           'A', 'G', 'C', 'C', 'C', 'A', 'A', 'G', 'T', 'G', 'C', 'C', 'T',
           'G', 'G', 'A', 'C', 'G', 'C', 'C', 'T', 'T', 'C', 'C', 'C', 'A',
           'A', 'A', 'C', 'C', 'T', 'G', 'A', 'G', 'G', 'G', 'A', 'C', 'T',
           'T', 'C', 'C', 'T', 'G', 'G', 'C', 'C', 'C', 'G', 'C', 'T', 'T',
           'C', 'G', 'A', 'G', 'G', 'G', 'C', 'C', 'T', 'C', 'A', 'A', 'G',
           'A', 'A', 'G', 'A', 'T', 'C', 'T', 'C', 'T', 'G', 'C', 'C', 'T',
           'A', 'C', 'A', 'T', 'G']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (413 aligned letters; 246 identities; 167 mismatches; 21 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 413:
        identities = 246,
        mismatches = 167.
    gaps = 21:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 21:
            internal_insertions = 6:
                open_internal_insertions = 3,
                extend_internal_insertions = 3;
            internal_deletions = 15:
                open_internal_deletions = 6,
                extend_internal_deletions = 9;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 6)
        self.assertEqual(counts.internal_deletions, 15)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 21)
        self.assertEqual(counts.insertions, 6)
        self.assertEqual(counts.deletions, 15)
        self.assertEqual(counts.gaps, 21)
        self.assertEqual(counts.aligned, 413)
        self.assertEqual(counts.identities, 246)
        self.assertEqual(counts.mismatches, 167)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 66.93)
        self.assertEqual(alignment.annotations["gap opens"], 8)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 127)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 127)
        self.assertEqual(alignment.shape, (2, 127 + 8))
        self.assertEqual(alignment.sequences[0].id, "RABGSTB")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 42)
        self.assertAlmostEqual(alignment.annotations["evalue"], 2.1e-07, places=8)
        self.assertAlmostEqual(alignment.annotations["bit score"], 45.6)
        self.assertEqual(
            str(alignment),
            """\
RABGSTB         156 ??????????????????????????????????--????????????????????????
                  0 |||||||||||||||--|||||||||||||||||--|||||||||||-||||||||||||
pGT875          158 ???????????????--??????????????????????????????-????????????

RABGSTB         214 ??????????????????????????????????????????????????????????-?
                 60 ||||||||||||||||||||||||||||||||||||||||||||||-|||||||||||-|
pGT875          215 ??????????????????????????????????????????????-?????????????

RABGSTB         273 ??????-???????? 287
                120 ||||||-|||||||| 135
pGT875          274 ??????????????? 289
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[156, 171, 173, 190, 190, 201, 202,
                           260, 261, 272, 272, 279, 279, 287],
                          [158, 173, 173, 190, 192, 203, 203,
                           261, 261, 272, 273, 280, 281, 289]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 287)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "GGGTATTGATGTTCCAGCAAGTGCCCATGGTTGA--GATTGATGGGATGAAGCTGGTGCAGACCAGAGCCATTTTCAACTACATTGCAGACAAGCACAACCTGTATGGGAAAGACATA-AAGGAGA-GAGCCCTG",
        )
        self.assertEqual(
            alignment[1],
            "GGGCCTGGACTTTCC--CAATCTGCCTTACTTGATCGATGGATCACA-CAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGAT-GGAGAGACAGAGGAGGAGAGGATCCGTG",
        )
        # pGT875   OCDHPR
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['G', 'G', 'G', 'T', 'A', 'T', 'T', 'G', 'A', 'T', 'G', 'T', 'T',
           'C', 'C', 'A', 'G', 'C', 'A', 'A', 'G', 'T', 'G', 'C', 'C', 'C',
           'A', 'T', 'G', 'G', 'T', 'T', 'G', 'A', '-', '-', 'G', 'A', 'T',
           'T', 'G', 'A', 'T', 'G', 'G', 'G', 'A', 'T', 'G', 'A', 'A', 'G',
           'C', 'T', 'G', 'G', 'T', 'G', 'C', 'A', 'G', 'A', 'C', 'C', 'A',
           'G', 'A', 'G', 'C', 'C', 'A', 'T', 'T', 'T', 'T', 'C', 'A', 'A',
           'C', 'T', 'A', 'C', 'A', 'T', 'T', 'G', 'C', 'A', 'G', 'A', 'C',
           'A', 'A', 'G', 'C', 'A', 'C', 'A', 'A', 'C', 'C', 'T', 'G', 'T',
           'A', 'T', 'G', 'G', 'G', 'A', 'A', 'A', 'G', 'A', 'C', 'A', 'T',
           'A', '-', 'A', 'A', 'G', 'G', 'A', 'G', 'A', '-', 'G', 'A', 'G',
           'C', 'C', 'C', 'T', 'G'],
          ['G', 'G', 'G', 'C', 'C', 'T', 'G', 'G', 'A', 'C', 'T', 'T', 'T',
           'C', 'C', '-', '-', 'C', 'A', 'A', 'T', 'C', 'T', 'G', 'C', 'C',
           'T', 'T', 'A', 'C', 'T', 'T', 'G', 'A', 'T', 'C', 'G', 'A', 'T',
           'G', 'G', 'A', 'T', 'C', 'A', 'C', 'A', '-', 'C', 'A', 'A', 'G',
           'A', 'T', 'C', 'A', 'C', 'C', 'C', 'A', 'G', 'A', 'G', 'C', 'A',
           'A', 'T', 'G', 'C', 'C', 'A', 'T', 'C', 'C', 'T', 'G', 'C', 'G',
           'C', 'T', 'A', 'C', 'C', 'T', 'T', 'G', 'C', 'C', 'C', 'G', 'A',
           'A', 'A', 'G', 'C', 'A', 'C', 'C', 'A', 'C', 'C', 'T', 'G', 'G',
           'A', 'T', '-', 'G', 'G', 'A', 'G', 'A', 'G', 'A', 'C', 'A', 'G',
           'A', 'G', 'G', 'A', 'G', 'G', 'A', 'G', 'A', 'G', 'G', 'A', 'T',
           'C', 'C', 'G', 'T', 'G']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (127 aligned letters; 85 identities; 42 mismatches; 8 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 127:
        identities = 85,
        mismatches = 42.
    gaps = 8:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 8:
            internal_insertions = 4:
                open_internal_insertions = 3,
                extend_internal_insertions = 1;
            internal_deletions = 4:
                open_internal_deletions = 3,
                extend_internal_deletions = 1;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 4)
        self.assertEqual(counts.internal_deletions, 4)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 8)
        self.assertEqual(counts.insertions, 4)
        self.assertEqual(counts.deletions, 4)
        self.assertEqual(counts.gaps, 8)
        self.assertEqual(counts.aligned, 127)
        self.assertEqual(counts.identities, 85)
        self.assertEqual(counts.mismatches, 42)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 91.30)
        self.assertEqual(alignment.annotations["gap opens"], 1)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 23)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 23)
        self.assertEqual(alignment.shape, (2, 23 + 1))
        self.assertEqual(alignment.sequences[0].id, "OCDHPR")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 2)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.0092)
        self.assertAlmostEqual(alignment.annotations["bit score"], 30.1)
        self.assertEqual(
            str(alignment),
            """\
OCDHPR         2302 ?????????????????-?????? 2325
                  0 |||||||||||||||||-||||||   24
pGT875          265 ????????????????????????  289
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[2302, 2319, 2319, 2325],
                          [ 265,  282,  283,  289]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 2325)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "AGACAGAGGAGGAGAAG-TCTGTG")
        self.assertEqual(alignment[1], "AGACAGAGGAGGAGAGGATCCGTG")
        # pGT875   RABALP1A
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['A', 'G', 'A', 'C', 'A', 'G', 'A', 'G', 'G', 'A', 'G', 'G', 'A',
           'G', 'A', 'A', 'G', '-', 'T', 'C', 'T', 'G', 'T', 'G'],
          ['A', 'G', 'A', 'C', 'A', 'G', 'A', 'G', 'G', 'A', 'G', 'G', 'A',
           'G', 'A', 'G', 'G', 'A', 'T', 'C', 'C', 'G', 'T', 'G']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (23 aligned letters; 21 identities; 2 mismatches; 1 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 23:
        identities = 21,
        mismatches = 2.
    gaps = 1:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 1:
            internal_insertions = 1:
                open_internal_insertions = 1,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 1)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 1)
        self.assertEqual(counts.insertions, 1)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 1)
        self.assertEqual(counts.aligned, 23)
        self.assertEqual(counts.identities, 21)
        self.assertEqual(counts.mismatches, 2)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 76.19)
        self.assertEqual(alignment.annotations["gap opens"], 4)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 42)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 42)
        self.assertEqual(alignment.shape, (2, 42 + 4))
        self.assertEqual(alignment.sequences[0].id, "RABALP1A")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 10)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.036)
        self.assertAlmostEqual(alignment.annotations["bit score"], 28.1)
        self.assertEqual(
            str(alignment),
            """\
RABALP1A       4973 ??????????????--???-?????????????????????????? 5016
                  0 ||||||||||||||--|||-||||||||||||-|||||||||||||   46
pGT875          240 ????????????????????????????????-?????????????  285
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[4973, 4987, 4987, 4990, 4990, 5002, 5003, 5016],
                          [ 240,  254,  256,  259,  260,  272,  272,  285]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 5016)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "GCTGGAGAGAGCCA--TGG-TGGAGGCTGCGATGGAGGAGAGGATC")
        self.assertEqual(alignment[1], "GCCCGAAAGCACCACCTGGATGGAGAGACAGA-GGAGGAGAGGATC")
        # pGT875   OCDHPR
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['G', 'C', 'T', 'G', 'G', 'A', 'G', 'A', 'G', 'A', 'G', 'C', 'C',
           'A', '-', '-', 'T', 'G', 'G', '-', 'T', 'G', 'G', 'A', 'G', 'G',
           'C', 'T', 'G', 'C', 'G', 'A', 'T', 'G', 'G', 'A', 'G', 'G', 'A',
           'G', 'A', 'G', 'G', 'A', 'T', 'C'],
          ['G', 'C', 'C', 'C', 'G', 'A', 'A', 'A', 'G', 'C', 'A', 'C', 'C',
           'A', 'C', 'C', 'T', 'G', 'G', 'A', 'T', 'G', 'G', 'A', 'G', 'A',
           'G', 'A', 'C', 'A', 'G', 'A', '-', 'G', 'G', 'A', 'G', 'G', 'A',
           'G', 'A', 'G', 'G', 'A', 'T', 'C']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (42 aligned letters; 32 identities; 10 mismatches; 4 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 42:
        identities = 32,
        mismatches = 10.
    gaps = 4:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 4:
            internal_insertions = 3:
                open_internal_insertions = 2,
                extend_internal_insertions = 1;
            internal_deletions = 1:
                open_internal_deletions = 1,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 3)
        self.assertEqual(counts.internal_deletions, 1)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 4)
        self.assertEqual(counts.insertions, 3)
        self.assertEqual(counts.deletions, 1)
        self.assertEqual(counts.gaps, 4)
        self.assertEqual(counts.aligned, 42)
        self.assertEqual(counts.identities, 32)
        self.assertEqual(counts.mismatches, 10)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 12)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 12)
        self.assertEqual(alignment.shape, (2, 12 + 0))
        self.assertEqual(alignment.sequences[0].id, "OCDHPR")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 0)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.071)
        self.assertAlmostEqual(alignment.annotations["bit score"], 27.2)
        self.assertEqual(
            str(alignment),
            """\
OCDHPR         1499 ???????????? 1511
                  0 ||||||||||||   12
pGT875          316 ????????????  304
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[1499, 1511], [316, 304]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 1511)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "CCATGACCTGGT")
        self.assertEqual(alignment[1], "CCATGACCTGGT")
        # pGT875   RABALP1A
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', 'C', 'A', 'T', 'G', 'A', 'C', 'C', 'T', 'G', 'G', 'T'],
          ['C', 'C', 'A', 'T', 'G', 'A', 'C', 'C', 'T', 'G', 'G', 'T']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (12 aligned letters; 12 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 12:
        identities = 12,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 12)
        self.assertEqual(counts.identities, 12)
        self.assertEqual(counts.mismatches, 0)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 12)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 12)
        self.assertEqual(alignment.shape, (2, 12 + 0))
        self.assertEqual(alignment.sequences[0].id, "RABALP1A")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 0)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.071)
        self.assertAlmostEqual(alignment.annotations["bit score"], 27.2)
        self.assertEqual(
            str(alignment),
            """\
RABALP1A       1499 ???????????? 1511
                  0 ||||||||||||   12
pGT875          316 ????????????  304
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[1499, 1511], [316, 304]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 1511)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "CCATGACCTGGT")
        self.assertEqual(alignment[1], "CCATGACCTGGT")
        # pGT875   RABGSTB
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', 'C', 'A', 'T', 'G', 'A', 'C', 'C', 'T', 'G', 'G', 'T'],
          ['C', 'C', 'A', 'T', 'G', 'A', 'C', 'C', 'T', 'G', 'G', 'T']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (12 aligned letters; 12 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 12:
        identities = 12,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 12)
        self.assertEqual(counts.identities, 12)
        self.assertEqual(counts.mismatches, 0)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 87.50)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 16)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 16)
        self.assertEqual(alignment.shape, (2, 16 + 0))
        self.assertEqual(alignment.sequences[0].id, "RABGSTB")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 2)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.14)
        self.assertAlmostEqual(alignment.annotations["bit score"], 26.2)
        self.assertEqual(
            str(alignment),
            """\
RABGSTB         490 ???????????????? 506
                  0 ||||||||||||||||  16
pGT875          160 ???????????????? 144
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[490, 506], [160, 144]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 506)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "CCTGGTTGAACTTCTC")
        self.assertEqual(alignment[1], "CCAGCTTGAACTTCTC")
        # pGT875   RABGLTR
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', 'C', 'T', 'G', 'G', 'T', 'T', 'G', 'A', 'A', 'C', 'T', 'T',
           'C', 'T', 'C'],
          ['C', 'C', 'A', 'G', 'C', 'T', 'T', 'G', 'A', 'A', 'C', 'T', 'T',
           'C', 'T', 'C']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (16 aligned letters; 14 identities; 2 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 16:
        identities = 14,
        mismatches = 2.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 16)
        self.assertEqual(counts.identities, 14)
        self.assertEqual(counts.mismatches, 2)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 53.57)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 84)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 84)
        self.assertEqual(alignment.shape, (2, 84 + 0))
        self.assertEqual(alignment.sequences[0].id, "RABGLTR")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 39)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.28)
        self.assertAlmostEqual(alignment.annotations["bit score"], 25.2)
        self.assertEqual(
            str(alignment),
            """\
RABGLTR        1116 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          242 ????????????????????????????????????????????????????????????

RABGLTR        1176 ???????????????????????? 1200
                 60 ||||||||||||||||||||||||   84
pGT875          182 ????????????????????????  158
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[1116, 1200], [242, 158]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 1200)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "GCATGGCTGGGTGGGGCAGGATTAGTGTGGGGGGAGTTGGGTGCTCAGGCAGGGCTATGAGGGATCTTGTTCATTTCCGGGCCC",
        )
        self.assertEqual(
            alignment[1],
            "GCAAGGTAGCGCAGGATGGCATTGCTCTGGGTGATCTTGTGTGATCCATCGATCAAGTAAGGCAGATTGGGAAAGTCCAGGCCC",
        )
        # pGT875   pGT875
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['G', 'C', 'A', 'T', 'G', 'G', 'C', 'T', 'G', 'G', 'G', 'T', 'G',
           'G', 'G', 'G', 'C', 'A', 'G', 'G', 'A', 'T', 'T', 'A', 'G', 'T',
           'G', 'T', 'G', 'G', 'G', 'G', 'G', 'G', 'A', 'G', 'T', 'T', 'G',
           'G', 'G', 'T', 'G', 'C', 'T', 'C', 'A', 'G', 'G', 'C', 'A', 'G',
           'G', 'G', 'C', 'T', 'A', 'T', 'G', 'A', 'G', 'G', 'G', 'A', 'T',
           'C', 'T', 'T', 'G', 'T', 'T', 'C', 'A', 'T', 'T', 'T', 'C', 'C',
           'G', 'G', 'G', 'C', 'C', 'C'],
          ['G', 'C', 'A', 'A', 'G', 'G', 'T', 'A', 'G', 'C', 'G', 'C', 'A',
           'G', 'G', 'A', 'T', 'G', 'G', 'C', 'A', 'T', 'T', 'G', 'C', 'T',
           'C', 'T', 'G', 'G', 'G', 'T', 'G', 'A', 'T', 'C', 'T', 'T', 'G',
           'T', 'G', 'T', 'G', 'A', 'T', 'C', 'C', 'A', 'T', 'C', 'G', 'A',
           'T', 'C', 'A', 'A', 'G', 'T', 'A', 'A', 'G', 'G', 'C', 'A', 'G',
           'A', 'T', 'T', 'G', 'G', 'G', 'A', 'A', 'A', 'G', 'T', 'C', 'C',
           'A', 'G', 'G', 'C', 'C', 'C']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (84 aligned letters; 45 identities; 39 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 84:
        identities = 45,
        mismatches = 39.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 84)
        self.assertEqual(counts.identities, 45)
        self.assertEqual(counts.mismatches, 39)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 10)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 10)
        self.assertEqual(alignment.shape, (2, 10 + 0))
        self.assertEqual(alignment.sequences[0].id, "pGT875")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 0)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.28)
        self.assertAlmostEqual(alignment.annotations["bit score"], 25.2)
        self.assertEqual(
            str(alignment),
            """\
pGT875          792 ?????????? 802
                  0 ||||||||||  10
pGT875          310 ?????????? 300
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[792, 802], [310, 300]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 802)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "CCTGGTTCTC")
        self.assertEqual(alignment[1], "CCTGGTTCTC")
        # pGT875   BTGST
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', 'C', 'T', 'G', 'G', 'T', 'T', 'C', 'T', 'C'],
          ['C', 'C', 'T', 'G', 'G', 'T', 'T', 'C', 'T', 'C']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (10 aligned letters; 10 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 10:
        identities = 10,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 10)
        self.assertEqual(counts.identities, 10)
        self.assertEqual(counts.mismatches, 0)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 59.15)
        self.assertEqual(alignment.annotations["gap opens"], 3)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 71)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 71)
        self.assertEqual(alignment.shape, (2, 71 + 3))
        self.assertEqual(alignment.sequences[0].id, "BTGST")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 29)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.56)
        self.assertAlmostEqual(alignment.annotations["bit score"], 24.2)
        self.assertEqual(
            str(alignment),
            """\
BTGST           280 ????????????????????????-????????--?????????????????????????
                  0 ||||||||||||||||||||||||-||||||||--|||||||||||||||||||||||||
pGT875          378 ????????????????????????????????????????????????????????????

BTGST           337 ?????????????? 351
                 60 ||||||||||||||  74
pGT875          318 ?????????????? 304
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[280, 304, 304, 312, 312, 351],
                          [378, 354, 353, 345, 343, 304]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 351)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "CTGCGGCACCTGGGCCGCACCCTC-GGGCTGTA--TGGGAAGGACCAGCAGGAGGCGGCCCTGGTGGACATGGT",
        )
        self.assertEqual(
            alignment[1],
            "CTCTGGCTTCTGCTTCTCAAAGTCAGGGTTGTAACAGAGCATGATGAGCTGCATGCGGGTGTCCATGACCTGGT",
        )
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', 'T', 'G', 'C', 'G', 'G', 'C', 'A', 'C', 'C', 'T', 'G', 'G',
           'G', 'C', 'C', 'G', 'C', 'A', 'C', 'C', 'C', 'T', 'C', '-', 'G',
           'G', 'G', 'C', 'T', 'G', 'T', 'A', '-', '-', 'T', 'G', 'G', 'G',
           'A', 'A', 'G', 'G', 'A', 'C', 'C', 'A', 'G', 'C', 'A', 'G', 'G',
           'A', 'G', 'G', 'C', 'G', 'G', 'C', 'C', 'C', 'T', 'G', 'G', 'T',
           'G', 'G', 'A', 'C', 'A', 'T', 'G', 'G', 'T'],
          ['C', 'T', 'C', 'T', 'G', 'G', 'C', 'T', 'T', 'C', 'T', 'G', 'C',
           'T', 'T', 'C', 'T', 'C', 'A', 'A', 'A', 'G', 'T', 'C', 'A', 'G',
           'G', 'G', 'T', 'T', 'G', 'T', 'A', 'A', 'C', 'A', 'G', 'A', 'G',
           'C', 'A', 'T', 'G', 'A', 'T', 'G', 'A', 'G', 'C', 'T', 'G', 'C',
           'A', 'T', 'G', 'C', 'G', 'G', 'G', 'T', 'G', 'T', 'C', 'C', 'A',
           'T', 'G', 'A', 'C', 'C', 'T', 'G', 'G', 'T']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (71 aligned letters; 42 identities; 29 mismatches; 3 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 71:
        identities = 42,
        mismatches = 29.
    gaps = 3:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 3:
            internal_insertions = 3:
                open_internal_insertions = 2,
                extend_internal_insertions = 1;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 3)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 3)
        self.assertEqual(counts.insertions, 3)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 3)
        self.assertEqual(counts.aligned, 71)
        self.assertEqual(counts.identities, 42)
        self.assertEqual(counts.mismatches, 29)
        with self.assertRaises(StopIteration):
            next(alignments)

    def test_m8CC(self):
        # Alignment file obtained by running
        # fasta36 -m 8CC seq/mgstm1.nt seq/gst.nlib
        # in the fasta36 source distribution
        path = "Fasta/nucleotide_m8CC.txt"
        alignments = Align.parse(path, "tabular")
        self.assertEqual(
            alignments.metadata["Command line"],
            "fasta36 -m 8CC seq/mgstm1.nt seq/gst.nlib",
        )
        self.assertEqual(alignments.metadata["Program"], "FASTA")
        self.assertEqual(alignments.metadata["Version"], "36.3.8h May, 2020")
        self.assertEqual(alignments.metadata["Database"], "seq/gst.nlib")
        # pGT875   pGT875
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 657)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 657)
        self.assertEqual(alignment.shape, (2, 657 + 0))
        self.assertEqual(alignment.sequences[0].id, "pGT875")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 0)
        self.assertAlmostEqual(alignment.annotations["evalue"], 4.6e-191, places=192)
        self.assertAlmostEqual(alignment.annotations["bit score"], 655.6)
        self.assertEqual(
            str(alignment),
            """\
pGT875           37 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875            0 ????????????????????????????????????????????????????????????

pGT875           97 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875           60 ????????????????????????????????????????????????????????????

pGT875          157 ????????????????????????????????????????????????????????????
                120 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          120 ????????????????????????????????????????????????????????????

pGT875          217 ????????????????????????????????????????????????????????????
                180 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          180 ????????????????????????????????????????????????????????????

pGT875          277 ????????????????????????????????????????????????????????????
                240 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          240 ????????????????????????????????????????????????????????????

pGT875          337 ????????????????????????????????????????????????????????????
                300 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          300 ????????????????????????????????????????????????????????????

pGT875          397 ????????????????????????????????????????????????????????????
                360 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          360 ????????????????????????????????????????????????????????????

pGT875          457 ????????????????????????????????????????????????????????????
                420 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          420 ????????????????????????????????????????????????????????????

pGT875          517 ????????????????????????????????????????????????????????????
                480 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          480 ????????????????????????????????????????????????????????????

pGT875          577 ????????????????????????????????????????????????????????????
                540 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          540 ????????????????????????????????????????????????????????????

pGT875          637 ????????????????????????????????????????????????????????? 694
                600 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||| 657
pGT875          600 ????????????????????????????????????????????????????????? 657
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[37, 694], [0, 657]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 694)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "ATGCCTATGATACTGGGATACTGGAACGTCCGCGGACTGACACACCCGATCCGCATGCTCCTGGAATACACAGACTCAAGCTATGATGAGAAGAGATACACCATGGGTGACGCTCCCGACTTTGACAGAAGCCAGTGGCTGAATGAGAAGTTCAAGCTGGGCCTGGACTTTCCCAATCTGCCTTACTTGATCGATGGATCACACAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGATGGAGAGACAGAGGAGGAGAGGATCCGTGCAGACATTGTGGAGAACCAGGTCATGGACACCCGCATGCAGCTCATCATGCTCTGTTACAACCCTGACTTTGAGAAGCAGAAGCCAGAGTTCTTGAAGACCATCCCTGAGAAAATGAAGCTCTACTCTGAGTTCCTGGGCAAGAGGCCATGGTTTGCAGGGGACAAGGTCACCTATGTGGATTTCCTTGCTTATGACATTCTTGACCAGTACCGTATGTTTGAGCCCAAGTGCCTGGACGCCTTCCCAAACCTGAGGGACTTCCTGGCCCGCTTCGAGGGCCTCAAGAAGATCTCTGCCTACATGAAGAGTAGCCGCTACATCGCAACACCTATATTTTCAAAGATGGCCCACTGGAGTAACAAGTAG",
        )
        self.assertEqual(
            alignment[1],
            "ATGCCTATGATACTGGGATACTGGAACGTCCGCGGACTGACACACCCGATCCGCATGCTCCTGGAATACACAGACTCAAGCTATGATGAGAAGAGATACACCATGGGTGACGCTCCCGACTTTGACAGAAGCCAGTGGCTGAATGAGAAGTTCAAGCTGGGCCTGGACTTTCCCAATCTGCCTTACTTGATCGATGGATCACACAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGATGGAGAGACAGAGGAGGAGAGGATCCGTGCAGACATTGTGGAGAACCAGGTCATGGACACCCGCATGCAGCTCATCATGCTCTGTTACAACCCTGACTTTGAGAAGCAGAAGCCAGAGTTCTTGAAGACCATCCCTGAGAAAATGAAGCTCTACTCTGAGTTCCTGGGCAAGAGGCCATGGTTTGCAGGGGACAAGGTCACCTATGTGGATTTCCTTGCTTATGACATTCTTGACCAGTACCGTATGTTTGAGCCCAAGTGCCTGGACGCCTTCCCAAACCTGAGGGACTTCCTGGCCCGCTTCGAGGGCCTCAAGAAGATCTCTGCCTACATGAAGAGTAGCCGCTACATCGCAACACCTATATTTTCAAAGATGGCCCACTGGAGTAACAAGTAG",
        )
        # pGT875   RABGLTR
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (657 aligned letters; 657 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 657:
        identities = 657,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 657)
        self.assertEqual(counts.identities, 657)
        self.assertEqual(counts.mismatches, 0)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 79.10)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 646)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 646)
        self.assertEqual(alignment.shape, (2, 646 + 0))
        self.assertEqual(alignment.sequences[0].id, "RABGLTR")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 135)
        self.assertAlmostEqual(alignment.annotations["evalue"], 1.6e-116, places=117)
        self.assertAlmostEqual(alignment.annotations["bit score"], 408.0)
        self.assertEqual(
            str(alignment),
            """\
RABGLTR          33 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875            0 ????????????????????????????????????????????????????????????

RABGLTR          93 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875           60 ????????????????????????????????????????????????????????????

RABGLTR         153 ????????????????????????????????????????????????????????????
                120 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          120 ????????????????????????????????????????????????????????????

RABGLTR         213 ????????????????????????????????????????????????????????????
                180 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          180 ????????????????????????????????????????????????????????????

RABGLTR         273 ????????????????????????????????????????????????????????????
                240 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          240 ????????????????????????????????????????????????????????????

RABGLTR         333 ????????????????????????????????????????????????????????????
                300 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          300 ????????????????????????????????????????????????????????????

RABGLTR         393 ????????????????????????????????????????????????????????????
                360 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          360 ????????????????????????????????????????????????????????????

RABGLTR         453 ????????????????????????????????????????????????????????????
                420 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          420 ????????????????????????????????????????????????????????????

RABGLTR         513 ????????????????????????????????????????????????????????????
                480 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          480 ????????????????????????????????????????????????????????????

RABGLTR         573 ????????????????????????????????????????????????????????????
                540 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          540 ????????????????????????????????????????????????????????????

RABGLTR         633 ?????????????????????????????????????????????? 679
                600 |||||||||||||||||||||||||||||||||||||||||||||| 646
pGT875          600 ?????????????????????????????????????????????? 646
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[33, 679], [0, 646]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 679)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "ATGCCCATGACGCTGGGTTACTGGGACGTCCGTGGGCTGGCTCTGCCAATCCGCATGCTCCTGGAATACACGGACACCAGCTATGAGGAAAAGAAATACACCATGGGGGATGCTCCCAACTATGACCAAAGCAAGTGGCTGAGTGAGAAGTTCACCCTGGGCCTGGACTTTCCCAATCTGCCCTACCTAATTGATGGGACTCACAAGCTCACGCAGAGCAACGCCATCCTGCGCTACCTGGCCCGCAAGCACGGCCTGTGTGGGGAGACGGAAGAGGAGAGGATTCGCGTGGACATTCTGGAGAATCAGCTGATGGACAACCGCTTCCAACTTGTAAACGTCTGCTACAGTCCCGACTTTGAGAAGCTCAAGCCCGAGTACCTGAAGGGGCTCCCTGAGAAGCTGCAGCTGTACTCGCAGTTCCTGGGAAGCCTCCCCTGGTTCGCAGGGGACAAGATCACCTTCGCCGATTTCCTTGTCTACGACGTTCTTGACCAGAACCGGATATTTGTGCCTGGGTGCCTGGACGCGTTCCCAAACCTGAAGGACTTTCATGTCCGCTTTGAGGGCCTGCCGAAGATCTCTGCCTACATGAAGTCCAGCCGCTTTATCCGAGTCCCTGTGTTTTTAAAGAAGGCCACGTGGA",
        )
        self.assertEqual(
            alignment[1],
            "ATGCCTATGATACTGGGATACTGGAACGTCCGCGGACTGACACACCCGATCCGCATGCTCCTGGAATACACAGACTCAAGCTATGATGAGAAGAGATACACCATGGGTGACGCTCCCGACTTTGACAGAAGCCAGTGGCTGAATGAGAAGTTCAAGCTGGGCCTGGACTTTCCCAATCTGCCTTACTTGATCGATGGATCACACAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGATGGAGAGACAGAGGAGGAGAGGATCCGTGCAGACATTGTGGAGAACCAGGTCATGGACACCCGCATGCAGCTCATCATGCTCTGTTACAACCCTGACTTTGAGAAGCAGAAGCCAGAGTTCTTGAAGACCATCCCTGAGAAAATGAAGCTCTACTCTGAGTTCCTGGGCAAGAGGCCATGGTTTGCAGGGGACAAGGTCACCTATGTGGATTTCCTTGCTTATGACATTCTTGACCAGTACCGTATGTTTGAGCCCAAGTGCCTGGACGCCTTCCCAAACCTGAGGGACTTCCTGGCCCGCTTCGAGGGCCTCAAGAAGATCTCTGCCTACATGAAGAGTAGCCGCTACATCGCAACACCTATATTTTCAAAGATGGCCCACTGGA",
        )
        # pGT875   BTGST
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (646 aligned letters; 511 identities; 135 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 646:
        identities = 511,
        mismatches = 135.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 646)
        self.assertEqual(counts.identities, 511)
        self.assertEqual(counts.mismatches, 135)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 59.56)
        self.assertEqual(alignment.annotations["gap opens"], 21)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 413)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 413)
        self.assertEqual(alignment.shape, (2, 413 + 21))
        self.assertEqual(alignment.sequences[0].id, "BTGST")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 167)
        self.assertAlmostEqual(alignment.annotations["evalue"], 1.9e-07, places=8)
        self.assertAlmostEqual(alignment.annotations["bit score"], 45.7)
        self.assertEqual(
            str(alignment),
            """\
BTGST           227 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          175 ????????????????????????????????????????????????????????????

BTGST           287 ????????????????????????????????????????????????????????????
                 60 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          235 ????????????????????????????????????????????????????????????

BTGST           347 ?????????????????????????????-?????????????????????????---??
                120 |||||||||||||||||||||||||||||-|||||||-|||||||||||||||||---||
pGT875          295 ?????????????????????????????????????-??????????????????????

BTGST           403 ????????????????????????????????????????????????????????????
                180 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||--
pGT875          354 ??????????????????????????????????????????????????????????--

BTGST           463 ????????????????????????????????????????????????????????????
                240 ---||||||-|||||||||||||--|||||||||||||----||||||||||||||||||
pGT875          412 ---??????-?????????????--?????????????----??????????????????

BTGST           523 ????????????????????--??????????????????????????????????????
                300 ||||||||||||--||||||--||||||||||||||||||||||||||||||||||||||
pGT875          462 ????????????--??????????????????????????????????????????????

BTGST           581 ????????????????????????????????????????????????????????????
                360 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          520 ????????????????????????????????????????????????????????????

BTGST           641 ?????????????? 655
                420 |||||||||||||| 434
pGT875          580 ?????????????? 594
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[227, 376, 376, 383, 384, 401, 401, 461, 466, 472,
                           473, 486, 488, 501, 505, 535, 537, 543, 543, 655],
                          [175, 324, 325, 332, 332, 349, 352, 412, 412, 418,
                           418, 431, 431, 444, 444, 474, 474, 480, 482, 594]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 655)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "AGCTCCCCAAGTTCCAGGACGGAGACCTCACGCTGTACCAGTCCAATGCCATCCTGCGGCACCTGGGCCGCACCCTCGGGCTGTATGGGAAGGACCAGCAGGAGGCGGCCCTGGTGGACATGGTGAATGACGGTGTAGAGGACCTTCGC-TGCAAATACGTCTCCCTCATTTACA---CCAACTACGAGGCGGGCAAGGAGGACTATGTGAAGGCGCTGCCCCAGCACCTGAAGCCTTTCGAGACCCTGCTGTCCCAGAACAAGGGTGGCCAGGCCTTCATCGTGGGCGACCAGATCTCCTTTGCGGACTACAACCTGCT--GGACCTGCTTCGGATTCACCAGGTCCTGGCCCCCAGCTGTCTGGACTCCTTCCCCCTGCTCTCAGCCTACGTGGCCCGTCTCAACTCCCGGCCCAAGCTCAAGGCCTTCCTG",
        )
        self.assertEqual(
            alignment[1],
            "ATCTGCCTTACTTGATCGATGGATCACACAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGATGGAGAGACAGAGGAGGAGAGGATCCGTGCAGACATTGTGGAGAACCAGGTCATGGACACCCGCATGCAGCT-CATCATGCTCTGTTACAACCCTGACTTTGAGAAGCAGAAGCCAGAGTTCTTGAAGACCATCCCTGAGAAAATGAAGCTCT-----ACTCTG-AGTTCCTGGGCAA--GAGGCCATGGTTT----GCAGGGGACAAGGTCACCTATGTGGATTTC--CTTGCTTATGACATTCTTGACCAGTACCGTATGTTTGAGCCCAAGTGCCTGGACGCCTTCCCAAACCTGAGGGACTTCCTGGCCCGCTTCGAGGGCCTCAAGAAGATCTCTGCCTACATG",
        )
        # pGT875   RABGSTB
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['A', 'G', 'C', 'T', 'C', 'C', 'C', 'C', 'A', 'A', 'G', 'T', 'T',
           'C', 'C', 'A', 'G', 'G', 'A', 'C', 'G', 'G', 'A', 'G', 'A', 'C',
           'C', 'T', 'C', 'A', 'C', 'G', 'C', 'T', 'G', 'T', 'A', 'C', 'C',
           'A', 'G', 'T', 'C', 'C', 'A', 'A', 'T', 'G', 'C', 'C', 'A', 'T',
           'C', 'C', 'T', 'G', 'C', 'G', 'G', 'C', 'A', 'C', 'C', 'T', 'G',
           'G', 'G', 'C', 'C', 'G', 'C', 'A', 'C', 'C', 'C', 'T', 'C', 'G',
           'G', 'G', 'C', 'T', 'G', 'T', 'A', 'T', 'G', 'G', 'G', 'A', 'A',
           'G', 'G', 'A', 'C', 'C', 'A', 'G', 'C', 'A', 'G', 'G', 'A', 'G',
           'G', 'C', 'G', 'G', 'C', 'C', 'C', 'T', 'G', 'G', 'T', 'G', 'G',
           'A', 'C', 'A', 'T', 'G', 'G', 'T', 'G', 'A', 'A', 'T', 'G', 'A',
           'C', 'G', 'G', 'T', 'G', 'T', 'A', 'G', 'A', 'G', 'G', 'A', 'C',
           'C', 'T', 'T', 'C', 'G', 'C', '-', 'T', 'G', 'C', 'A', 'A', 'A',
           'T', 'A', 'C', 'G', 'T', 'C', 'T', 'C', 'C', 'C', 'T', 'C', 'A',
           'T', 'T', 'T', 'A', 'C', 'A', '-', '-', '-', 'C', 'C', 'A', 'A',
           'C', 'T', 'A', 'C', 'G', 'A', 'G', 'G', 'C', 'G', 'G', 'G', 'C',
           'A', 'A', 'G', 'G', 'A', 'G', 'G', 'A', 'C', 'T', 'A', 'T', 'G',
           'T', 'G', 'A', 'A', 'G', 'G', 'C', 'G', 'C', 'T', 'G', 'C', 'C',
           'C', 'C', 'A', 'G', 'C', 'A', 'C', 'C', 'T', 'G', 'A', 'A', 'G',
           'C', 'C', 'T', 'T', 'T', 'C', 'G', 'A', 'G', 'A', 'C', 'C', 'C',
           'T', 'G', 'C', 'T', 'G', 'T', 'C', 'C', 'C', 'A', 'G', 'A', 'A',
           'C', 'A', 'A', 'G', 'G', 'G', 'T', 'G', 'G', 'C', 'C', 'A', 'G',
           'G', 'C', 'C', 'T', 'T', 'C', 'A', 'T', 'C', 'G', 'T', 'G', 'G',
           'G', 'C', 'G', 'A', 'C', 'C', 'A', 'G', 'A', 'T', 'C', 'T', 'C',
           'C', 'T', 'T', 'T', 'G', 'C', 'G', 'G', 'A', 'C', 'T', 'A', 'C',
           'A', 'A', 'C', 'C', 'T', 'G', 'C', 'T', '-', '-', 'G', 'G', 'A',
           'C', 'C', 'T', 'G', 'C', 'T', 'T', 'C', 'G', 'G', 'A', 'T', 'T',
           'C', 'A', 'C', 'C', 'A', 'G', 'G', 'T', 'C', 'C', 'T', 'G', 'G',
           'C', 'C', 'C', 'C', 'C', 'A', 'G', 'C', 'T', 'G', 'T', 'C', 'T',
           'G', 'G', 'A', 'C', 'T', 'C', 'C', 'T', 'T', 'C', 'C', 'C', 'C',
           'C', 'T', 'G', 'C', 'T', 'C', 'T', 'C', 'A', 'G', 'C', 'C', 'T',
           'A', 'C', 'G', 'T', 'G', 'G', 'C', 'C', 'C', 'G', 'T', 'C', 'T',
           'C', 'A', 'A', 'C', 'T', 'C', 'C', 'C', 'G', 'G', 'C', 'C', 'C',
           'A', 'A', 'G', 'C', 'T', 'C', 'A', 'A', 'G', 'G', 'C', 'C', 'T',
           'T', 'C', 'C', 'T', 'G'],
          ['A', 'T', 'C', 'T', 'G', 'C', 'C', 'T', 'T', 'A', 'C', 'T', 'T',
           'G', 'A', 'T', 'C', 'G', 'A', 'T', 'G', 'G', 'A', 'T', 'C', 'A',
           'C', 'A', 'C', 'A', 'A', 'G', 'A', 'T', 'C', 'A', 'C', 'C', 'C',
           'A', 'G', 'A', 'G', 'C', 'A', 'A', 'T', 'G', 'C', 'C', 'A', 'T',
           'C', 'C', 'T', 'G', 'C', 'G', 'C', 'T', 'A', 'C', 'C', 'T', 'T',
           'G', 'C', 'C', 'C', 'G', 'A', 'A', 'A', 'G', 'C', 'A', 'C', 'C',
           'A', 'C', 'C', 'T', 'G', 'G', 'A', 'T', 'G', 'G', 'A', 'G', 'A',
           'G', 'A', 'C', 'A', 'G', 'A', 'G', 'G', 'A', 'G', 'G', 'A', 'G',
           'A', 'G', 'G', 'A', 'T', 'C', 'C', 'G', 'T', 'G', 'C', 'A', 'G',
           'A', 'C', 'A', 'T', 'T', 'G', 'T', 'G', 'G', 'A', 'G', 'A', 'A',
           'C', 'C', 'A', 'G', 'G', 'T', 'C', 'A', 'T', 'G', 'G', 'A', 'C',
           'A', 'C', 'C', 'C', 'G', 'C', 'A', 'T', 'G', 'C', 'A', 'G', 'C',
           'T', '-', 'C', 'A', 'T', 'C', 'A', 'T', 'G', 'C', 'T', 'C', 'T',
           'G', 'T', 'T', 'A', 'C', 'A', 'A', 'C', 'C', 'C', 'T', 'G', 'A',
           'C', 'T', 'T', 'T', 'G', 'A', 'G', 'A', 'A', 'G', 'C', 'A', 'G',
           'A', 'A', 'G', 'C', 'C', 'A', 'G', 'A', 'G', 'T', 'T', 'C', 'T',
           'T', 'G', 'A', 'A', 'G', 'A', 'C', 'C', 'A', 'T', 'C', 'C', 'C',
           'T', 'G', 'A', 'G', 'A', 'A', 'A', 'A', 'T', 'G', 'A', 'A', 'G',
           'C', 'T', 'C', 'T', '-', '-', '-', '-', '-', 'A', 'C', 'T', 'C',
           'T', 'G', '-', 'A', 'G', 'T', 'T', 'C', 'C', 'T', 'G', 'G', 'G',
           'C', 'A', 'A', '-', '-', 'G', 'A', 'G', 'G', 'C', 'C', 'A', 'T',
           'G', 'G', 'T', 'T', 'T', '-', '-', '-', '-', 'G', 'C', 'A', 'G',
           'G', 'G', 'G', 'A', 'C', 'A', 'A', 'G', 'G', 'T', 'C', 'A', 'C',
           'C', 'T', 'A', 'T', 'G', 'T', 'G', 'G', 'A', 'T', 'T', 'T', 'C',
           '-', '-', 'C', 'T', 'T', 'G', 'C', 'T', 'T', 'A', 'T', 'G', 'A',
           'C', 'A', 'T', 'T', 'C', 'T', 'T', 'G', 'A', 'C', 'C', 'A', 'G',
           'T', 'A', 'C', 'C', 'G', 'T', 'A', 'T', 'G', 'T', 'T', 'T', 'G',
           'A', 'G', 'C', 'C', 'C', 'A', 'A', 'G', 'T', 'G', 'C', 'C', 'T',
           'G', 'G', 'A', 'C', 'G', 'C', 'C', 'T', 'T', 'C', 'C', 'C', 'A',
           'A', 'A', 'C', 'C', 'T', 'G', 'A', 'G', 'G', 'G', 'A', 'C', 'T',
           'T', 'C', 'C', 'T', 'G', 'G', 'C', 'C', 'C', 'G', 'C', 'T', 'T',
           'C', 'G', 'A', 'G', 'G', 'G', 'C', 'C', 'T', 'C', 'A', 'A', 'G',
           'A', 'A', 'G', 'A', 'T', 'C', 'T', 'C', 'T', 'G', 'C', 'C', 'T',
           'A', 'C', 'A', 'T', 'G']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (413 aligned letters; 246 identities; 167 mismatches; 21 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 413:
        identities = 246,
        mismatches = 167.
    gaps = 21:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 21:
            internal_insertions = 6:
                open_internal_insertions = 3,
                extend_internal_insertions = 3;
            internal_deletions = 15:
                open_internal_deletions = 6,
                extend_internal_deletions = 9;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 6)
        self.assertEqual(counts.internal_deletions, 15)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 21)
        self.assertEqual(counts.insertions, 6)
        self.assertEqual(counts.deletions, 15)
        self.assertEqual(counts.gaps, 21)
        self.assertEqual(counts.aligned, 413)
        self.assertEqual(counts.identities, 246)
        self.assertEqual(counts.mismatches, 167)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 66.93)
        self.assertEqual(alignment.annotations["gap opens"], 8)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 127)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 127)
        self.assertEqual(alignment.shape, (2, 127 + 8))
        self.assertEqual(alignment.sequences[0].id, "RABGSTB")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 42)
        self.assertAlmostEqual(alignment.annotations["evalue"], 3.2e-07, places=8)
        self.assertAlmostEqual(alignment.annotations["bit score"], 45.0)
        self.assertEqual(
            str(alignment),
            """\
RABGSTB         156 ??????????????????????????????????--????????????????????????
                  0 |||||||||||||||--|||||||||||||||||--|||||||||||-||||||||||||
pGT875          158 ???????????????--??????????????????????????????-????????????

RABGSTB         214 ??????????????????????????????????????????????????????????-?
                 60 ||||||||||||||||||||||||||||||||||||||||||||||-|||||||||||-|
pGT875          215 ??????????????????????????????????????????????-?????????????

RABGSTB         273 ??????-???????? 287
                120 ||||||-|||||||| 135
pGT875          274 ??????????????? 289
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[156, 171, 173, 190, 190, 201, 202,
                           260, 261, 272, 272, 279, 279, 287],
                          [158, 173, 173, 190, 192, 203, 203,
                           261, 261, 272, 273, 280, 281, 289]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 287)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "GGGTATTGATGTTCCAGCAAGTGCCCATGGTTGA--GATTGATGGGATGAAGCTGGTGCAGACCAGAGCCATTTTCAACTACATTGCAGACAAGCACAACCTGTATGGGAAAGACATA-AAGGAGA-GAGCCCTG",
        )
        self.assertEqual(
            alignment[1],
            "GGGCCTGGACTTTCC--CAATCTGCCTTACTTGATCGATGGATCACA-CAAGATCACCCAGAGCAATGCCATCCTGCGCTACCTTGCCCGAAAGCACCACCTGGAT-GGAGAGACAGAGGAGGAGAGGATCCGTG",
        )
        # pGT875   OCDHPR
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['G', 'G', 'G', 'T', 'A', 'T', 'T', 'G', 'A', 'T', 'G', 'T', 'T',
           'C', 'C', 'A', 'G', 'C', 'A', 'A', 'G', 'T', 'G', 'C', 'C', 'C',
           'A', 'T', 'G', 'G', 'T', 'T', 'G', 'A', '-', '-', 'G', 'A', 'T',
           'T', 'G', 'A', 'T', 'G', 'G', 'G', 'A', 'T', 'G', 'A', 'A', 'G',
           'C', 'T', 'G', 'G', 'T', 'G', 'C', 'A', 'G', 'A', 'C', 'C', 'A',
           'G', 'A', 'G', 'C', 'C', 'A', 'T', 'T', 'T', 'T', 'C', 'A', 'A',
           'C', 'T', 'A', 'C', 'A', 'T', 'T', 'G', 'C', 'A', 'G', 'A', 'C',
           'A', 'A', 'G', 'C', 'A', 'C', 'A', 'A', 'C', 'C', 'T', 'G', 'T',
           'A', 'T', 'G', 'G', 'G', 'A', 'A', 'A', 'G', 'A', 'C', 'A', 'T',
           'A', '-', 'A', 'A', 'G', 'G', 'A', 'G', 'A', '-', 'G', 'A', 'G',
           'C', 'C', 'C', 'T', 'G'],
          ['G', 'G', 'G', 'C', 'C', 'T', 'G', 'G', 'A', 'C', 'T', 'T', 'T',
           'C', 'C', '-', '-', 'C', 'A', 'A', 'T', 'C', 'T', 'G', 'C', 'C',
           'T', 'T', 'A', 'C', 'T', 'T', 'G', 'A', 'T', 'C', 'G', 'A', 'T',
           'G', 'G', 'A', 'T', 'C', 'A', 'C', 'A', '-', 'C', 'A', 'A', 'G',
           'A', 'T', 'C', 'A', 'C', 'C', 'C', 'A', 'G', 'A', 'G', 'C', 'A',
           'A', 'T', 'G', 'C', 'C', 'A', 'T', 'C', 'C', 'T', 'G', 'C', 'G',
           'C', 'T', 'A', 'C', 'C', 'T', 'T', 'G', 'C', 'C', 'C', 'G', 'A',
           'A', 'A', 'G', 'C', 'A', 'C', 'C', 'A', 'C', 'C', 'T', 'G', 'G',
           'A', 'T', '-', 'G', 'G', 'A', 'G', 'A', 'G', 'A', 'C', 'A', 'G',
           'A', 'G', 'G', 'A', 'G', 'G', 'A', 'G', 'A', 'G', 'G', 'A', 'T',
           'C', 'C', 'G', 'T', 'G']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (127 aligned letters; 85 identities; 42 mismatches; 8 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 127:
        identities = 85,
        mismatches = 42.
    gaps = 8:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 8:
            internal_insertions = 4:
                open_internal_insertions = 3,
                extend_internal_insertions = 1;
            internal_deletions = 4:
                open_internal_deletions = 3,
                extend_internal_deletions = 1;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 4)
        self.assertEqual(counts.internal_deletions, 4)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 8)
        self.assertEqual(counts.insertions, 4)
        self.assertEqual(counts.deletions, 4)
        self.assertEqual(counts.gaps, 8)
        self.assertEqual(counts.aligned, 127)
        self.assertEqual(counts.identities, 85)
        self.assertEqual(counts.mismatches, 42)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 91.30)
        self.assertEqual(alignment.annotations["gap opens"], 1)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 23)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 23)
        self.assertEqual(alignment.shape, (2, 23 + 1))
        self.assertEqual(alignment.sequences[0].id, "OCDHPR")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 2)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.012)
        self.assertAlmostEqual(alignment.annotations["bit score"], 29.7)
        self.assertEqual(
            str(alignment),
            """\
OCDHPR         2302 ?????????????????-?????? 2325
                  0 |||||||||||||||||-||||||   24
pGT875          265 ????????????????????????  289
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[2302, 2319, 2319, 2325],
                           [265,  282,  283,  289]]),
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 2325)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "AGACAGAGGAGGAGAAG-TCTGTG")
        self.assertEqual(alignment[1], "AGACAGAGGAGGAGAGGATCCGTG")
        # pGT875   RABALP1A
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['A', 'G', 'A', 'C', 'A', 'G', 'A', 'G', 'G', 'A', 'G', 'G', 'A',
           'G', 'A', 'A', 'G', '-', 'T', 'C', 'T', 'G', 'T', 'G'],
          ['A', 'G', 'A', 'C', 'A', 'G', 'A', 'G', 'G', 'A', 'G', 'G', 'A',
           'G', 'A', 'G', 'G', 'A', 'T', 'C', 'C', 'G', 'T', 'G']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (23 aligned letters; 21 identities; 2 mismatches; 1 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 23:
        identities = 21,
        mismatches = 2.
    gaps = 1:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 1:
            internal_insertions = 1:
                open_internal_insertions = 1,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 1)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 1)
        self.assertEqual(counts.insertions, 1)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 1)
        self.assertEqual(counts.aligned, 23)
        self.assertEqual(counts.identities, 21)
        self.assertEqual(counts.mismatches, 2)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 76.19)
        self.assertEqual(alignment.annotations["gap opens"], 4)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 42)
        self.assertEqual(sum(end - start for start, end in aligned[1]), 42)
        self.assertEqual(alignment.shape, (2, 42 + 4))
        self.assertEqual(alignment.sequences[0].id, "RABALP1A")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 10)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.046)
        self.assertAlmostEqual(alignment.annotations["bit score"], 27.8)
        self.assertEqual(
            str(alignment),
            """\
RABALP1A       4973 ??????????????--???-?????????????????????????? 5016
                  0 ||||||||||||||--|||-||||||||||||-|||||||||||||   46
pGT875          240 ????????????????????????????????-?????????????  285
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[4973, 4987, 4987, 4990, 4990, 5002, 5003, 5016],
                          [ 240,  254,  256,  259,  260,  272,  272,  285]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 5016)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "GCTGGAGAGAGCCA--TGG-TGGAGGCTGCGATGGAGGAGAGGATC")
        self.assertEqual(alignment[1], "GCCCGAAAGCACCACCTGGATGGAGAGACAGA-GGAGGAGAGGATC")
        # pGT875   OCDHPR
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['G', 'C', 'T', 'G', 'G', 'A', 'G', 'A', 'G', 'A', 'G', 'C', 'C',
           'A', '-', '-', 'T', 'G', 'G', '-', 'T', 'G', 'G', 'A', 'G', 'G',
           'C', 'T', 'G', 'C', 'G', 'A', 'T', 'G', 'G', 'A', 'G', 'G', 'A',
           'G', 'A', 'G', 'G', 'A', 'T', 'C'],
          ['G', 'C', 'C', 'C', 'G', 'A', 'A', 'A', 'G', 'C', 'A', 'C', 'C',
           'A', 'C', 'C', 'T', 'G', 'G', 'A', 'T', 'G', 'G', 'A', 'G', 'A',
           'G', 'A', 'C', 'A', 'G', 'A', '-', 'G', 'G', 'A', 'G', 'G', 'A',
           'G', 'A', 'G', 'G', 'A', 'T', 'C']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (42 aligned letters; 32 identities; 10 mismatches; 4 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 42:
        identities = 32,
        mismatches = 10.
    gaps = 4:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 4:
            internal_insertions = 3:
                open_internal_insertions = 2,
                extend_internal_insertions = 1;
            internal_deletions = 1:
                open_internal_deletions = 1,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 3)
        self.assertEqual(counts.internal_deletions, 1)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 4)
        self.assertEqual(counts.insertions, 3)
        self.assertEqual(counts.deletions, 1)
        self.assertEqual(counts.gaps, 4)
        self.assertEqual(counts.aligned, 42)
        self.assertEqual(counts.identities, 32)
        self.assertEqual(counts.mismatches, 10)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 12)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 12)
        self.assertEqual(alignment.shape, (2, 12 + 0))
        self.assertEqual(alignment.sequences[0].id, "OCDHPR")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 0)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.09)
        self.assertAlmostEqual(alignment.annotations["bit score"], 26.8)
        self.assertEqual(
            str(alignment),
            """\
OCDHPR         1499 ???????????? 1511
                  0 ||||||||||||   12
pGT875          316 ????????????  304
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[1499, 1511], [316, 304]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 1511)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "CCATGACCTGGT")
        self.assertEqual(alignment[1], "CCATGACCTGGT")
        # pGT875   RABALP1A
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', 'C', 'A', 'T', 'G', 'A', 'C', 'C', 'T', 'G', 'G', 'T'],
          ['C', 'C', 'A', 'T', 'G', 'A', 'C', 'C', 'T', 'G', 'G', 'T']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (12 aligned letters; 12 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 12:
        identities = 12,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 12)
        self.assertEqual(counts.identities, 12)
        self.assertEqual(counts.mismatches, 0)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 12)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 12)
        self.assertEqual(alignment.shape, (2, 12 + 0))
        self.assertEqual(alignment.sequences[0].id, "RABALP1A")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 0)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.09)
        self.assertAlmostEqual(alignment.annotations["bit score"], 26.8)
        self.assertEqual(
            str(alignment),
            """\
RABALP1A       1499 ???????????? 1511
                  0 ||||||||||||   12
pGT875          316 ????????????  304
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[1499, 1511], [316, 304]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 1511)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "CCATGACCTGGT")
        self.assertEqual(alignment[1], "CCATGACCTGGT")
        # pGT875   RABGSTB
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', 'C', 'A', 'T', 'G', 'A', 'C', 'C', 'T', 'G', 'G', 'T'],
          ['C', 'C', 'A', 'T', 'G', 'A', 'C', 'C', 'T', 'G', 'G', 'T']],
         dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (12 aligned letters; 12 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 12:
        identities = 12,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 12)
        self.assertEqual(counts.identities, 12)
        self.assertEqual(counts.mismatches, 0)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 87.50)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 16)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 16)
        self.assertEqual(alignment.shape, (2, 16 + 0))
        self.assertEqual(alignment.sequences[0].id, "RABGSTB")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 2)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.18)
        self.assertAlmostEqual(alignment.annotations["bit score"], 25.8)
        self.assertEqual(
            str(alignment),
            """\
RABGSTB         490 ???????????????? 506
                  0 ||||||||||||||||  16
pGT875          160 ???????????????? 144
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[490, 506], [160, 144]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 506)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "CCTGGTTGAACTTCTC")
        self.assertEqual(alignment[1], "CCAGCTTGAACTTCTC")
        # pGT875   RABGLTR
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', 'C', 'T', 'G', 'G', 'T', 'T', 'G', 'A', 'A', 'C', 'T', 'T',
           'C', 'T', 'C'],
          ['C', 'C', 'A', 'G', 'C', 'T', 'T', 'G', 'A', 'A', 'C', 'T', 'T',
           'C', 'T', 'C']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (16 aligned letters; 14 identities; 2 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 16:
        identities = 14,
        mismatches = 2.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 16)
        self.assertEqual(counts.identities, 14)
        self.assertEqual(counts.mismatches, 2)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 53.57)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 84)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 84)
        self.assertEqual(alignment.shape, (2, 84 + 0))
        self.assertEqual(alignment.sequences[0].id, "RABGLTR")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 39)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.35)
        self.assertAlmostEqual(alignment.annotations["bit score"], 24.9)
        self.assertEqual(
            str(alignment),
            """\
RABGLTR        1116 ????????????????????????????????????????????????????????????
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
pGT875          242 ????????????????????????????????????????????????????????????

RABGLTR        1176 ???????????????????????? 1200
                 60 ||||||||||||||||||||||||   84
pGT875          182 ????????????????????????  158
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[1116, 1200], [242, 158]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 1200)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "GCATGGCTGGGTGGGGCAGGATTAGTGTGGGGGGAGTTGGGTGCTCAGGCAGGGCTATGAGGGATCTTGTTCATTTCCGGGCCC",
        )
        self.assertEqual(
            alignment[1],
            "GCAAGGTAGCGCAGGATGGCATTGCTCTGGGTGATCTTGTGTGATCCATCGATCAAGTAAGGCAGATTGGGAAAGTCCAGGCCC",
        )
        # pGT875   pGT875
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['G', 'C', 'A', 'T', 'G', 'G', 'C', 'T', 'G', 'G', 'G', 'T', 'G',
           'G', 'G', 'G', 'C', 'A', 'G', 'G', 'A', 'T', 'T', 'A', 'G', 'T',
           'G', 'T', 'G', 'G', 'G', 'G', 'G', 'G', 'A', 'G', 'T', 'T', 'G',
           'G', 'G', 'T', 'G', 'C', 'T', 'C', 'A', 'G', 'G', 'C', 'A', 'G',
           'G', 'G', 'C', 'T', 'A', 'T', 'G', 'A', 'G', 'G', 'G', 'A', 'T',
           'C', 'T', 'T', 'G', 'T', 'T', 'C', 'A', 'T', 'T', 'T', 'C', 'C',
           'G', 'G', 'G', 'C', 'C', 'C'],
          ['G', 'C', 'A', 'A', 'G', 'G', 'T', 'A', 'G', 'C', 'G', 'C', 'A',
           'G', 'G', 'A', 'T', 'G', 'G', 'C', 'A', 'T', 'T', 'G', 'C', 'T',
           'C', 'T', 'G', 'G', 'G', 'T', 'G', 'A', 'T', 'C', 'T', 'T', 'G',
           'T', 'G', 'T', 'G', 'A', 'T', 'C', 'C', 'A', 'T', 'C', 'G', 'A',
           'T', 'C', 'A', 'A', 'G', 'T', 'A', 'A', 'G', 'G', 'C', 'A', 'G',
           'A', 'T', 'T', 'G', 'G', 'G', 'A', 'A', 'A', 'G', 'T', 'C', 'C',
           'A', 'G', 'G', 'C', 'C', 'C']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (84 aligned letters; 45 identities; 39 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 84:
        identities = 45,
        mismatches = 39.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 84)
        self.assertEqual(counts.identities, 45)
        self.assertEqual(counts.mismatches, 39)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
        self.assertEqual(alignment.annotations["gap opens"], 0)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 10)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 10)
        self.assertEqual(alignment.shape, (2, 10 + 0))
        self.assertEqual(alignment.sequences[0].id, "pGT875")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 0)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.35)
        self.assertAlmostEqual(alignment.annotations["bit score"], 24.9)
        self.assertEqual(
            str(alignment),
            """\
pGT875          792 ?????????? 802
                  0 ||||||||||  10
pGT875          310 ?????????? 300
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                np.array([[792, 802], [310, 300]]),
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 802)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(alignment[0], "CCTGGTTCTC")
        self.assertEqual(alignment[1], "CCTGGTTCTC")
        # pGT875   BTGST
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', 'C', 'T', 'G', 'G', 'T', 'T', 'C', 'T', 'C'],
          ['C', 'C', 'T', 'G', 'G', 'T', 'T', 'C', 'T', 'C']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (10 aligned letters; 10 identities; 0 mismatches; 0 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 10:
        identities = 10,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 0)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 0)
        self.assertEqual(counts.insertions, 0)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 0)
        self.assertEqual(counts.aligned, 10)
        self.assertEqual(counts.identities, 10)
        self.assertEqual(counts.mismatches, 0)
        alignment = next(alignments)
        self.assertEqual(len(alignment), 2)
        self.assertAlmostEqual(alignment.annotations["% identity"], 59.15)
        self.assertEqual(alignment.annotations["gap opens"], 3)
        aligned = alignment.aligned
        self.assertEqual(sum(end - start for start, end in aligned[0]), 71)
        self.assertEqual(sum(start - end for start, end in aligned[1]), 71)
        self.assertEqual(alignment.shape, (2, 71 + 3))
        self.assertEqual(alignment.sequences[0].id, "BTGST")
        self.assertEqual(alignment.sequences[1].id, "pGT875")
        self.assertEqual(alignment.annotations["mismatches"], 29)
        self.assertAlmostEqual(alignment.annotations["evalue"], 0.68)
        self.assertAlmostEqual(alignment.annotations["bit score"], 23.9)
        self.assertEqual(
            str(alignment),
            """\
BTGST           280 ????????????????????????-????????--?????????????????????????
                  0 ||||||||||||||||||||||||-||||||||--|||||||||||||||||||||||||
pGT875          378 ????????????????????????????????????????????????????????????

BTGST           337 ?????????????? 351
                 60 ||||||||||||||  74
pGT875          318 ?????????????? 304
""",
        )
        self.assertTrue(
            np.array_equal(
                alignment.coordinates,
                # fmt: off
                np.array([[280, 304, 304, 312, 312, 351],
                          [378, 354, 353, 345, 343, 304]])
                # fmt: on
            )
        )
        query = self.query
        target = self.targets[alignment.sequences[0].id]
        self.assertEqual(len(alignment.sequences[0].seq), 351)
        self.assertEqual(len(alignment.sequences[1].seq), len(query))
        alignment.sequences[0].seq = target
        alignment.sequences[1].seq = query
        self.assertEqual(
            alignment[0],
            "CTGCGGCACCTGGGCCGCACCCTC-GGGCTGTA--TGGGAAGGACCAGCAGGAGGCGGCCCTGGTGGACATGGT",
        )
        self.assertEqual(
            alignment[1],
            "CTCTGGCTTCTGCTTCTCAAAGTCAGGGTTGTAACAGAGCATGATGAGCTGCATGCGGGTGTCCATGACCTGGT",
        )
        self.assertTrue(
            np.array_equal(
                np.array(alignment, "U"),
                # fmt: off
np.array([['C', 'T', 'G', 'C', 'G', 'G', 'C', 'A', 'C', 'C', 'T', 'G', 'G',
           'G', 'C', 'C', 'G', 'C', 'A', 'C', 'C', 'C', 'T', 'C', '-', 'G',
           'G', 'G', 'C', 'T', 'G', 'T', 'A', '-', '-', 'T', 'G', 'G', 'G',
           'A', 'A', 'G', 'G', 'A', 'C', 'C', 'A', 'G', 'C', 'A', 'G', 'G',
           'A', 'G', 'G', 'C', 'G', 'G', 'C', 'C', 'C', 'T', 'G', 'G', 'T',
           'G', 'G', 'A', 'C', 'A', 'T', 'G', 'G', 'T'],
          ['C', 'T', 'C', 'T', 'G', 'G', 'C', 'T', 'T', 'C', 'T', 'G', 'C',
           'T', 'T', 'C', 'T', 'C', 'A', 'A', 'A', 'G', 'T', 'C', 'A', 'G',
           'G', 'G', 'T', 'T', 'G', 'T', 'A', 'A', 'C', 'A', 'G', 'A', 'G',
           'C', 'A', 'T', 'G', 'A', 'T', 'G', 'A', 'G', 'C', 'T', 'G', 'C',
           'A', 'T', 'G', 'C', 'G', 'G', 'G', 'T', 'G', 'T', 'C', 'C', 'A',
           'T', 'G', 'A', 'C', 'C', 'T', 'G', 'G', 'T']], dtype='U')
                # fmt: on
            )
        )
        counts = alignment.counts()
        self.assertEqual(
            repr(counts),
            "<AlignmentCounts object (71 aligned letters; 42 identities; 29 mismatches; 3 gaps) at 0x%x>"
            % id(counts),
        )
        self.assertEqual(
            str(counts),
            """\
AlignmentCounts object with
    aligned = 71:
        identities = 42,
        mismatches = 29.
    gaps = 3:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 3:
            internal_insertions = 3:
                open_internal_insertions = 2,
                extend_internal_insertions = 1;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
        )
        self.assertEqual(counts.left_insertions, 0)
        self.assertEqual(counts.left_deletions, 0)
        self.assertEqual(counts.right_insertions, 0)
        self.assertEqual(counts.right_deletions, 0)
        self.assertEqual(counts.internal_insertions, 3)
        self.assertEqual(counts.internal_deletions, 0)
        self.assertEqual(counts.left_gaps, 0)
        self.assertEqual(counts.right_gaps, 0)
        self.assertEqual(counts.internal_gaps, 3)
        self.assertEqual(counts.insertions, 3)
        self.assertEqual(counts.deletions, 0)
        self.assertEqual(counts.gaps, 3)
        self.assertEqual(counts.aligned, 71)
        self.assertEqual(counts.identities, 42)
        self.assertEqual(counts.mismatches, 29)
        with self.assertRaises(StopIteration):
            next(alignments)


class TestFastaBasic(unittest.TestCase):
    def test_empty(self):
        import io

        stream = io.StringIO()
        with self.assertRaisesRegex(ValueError, "Empty file."):
            Align.parse(stream, "tabular")


class TestBlast(unittest.TestCase):
    def test_2226_tblastn_001(self):
        path = "Blast/tab_2226_tblastn_001.txt"
        with open(path) as stream:
            with self.assertRaisesRegex(ValueError, "Missing header."):
                Align.parse(stream, "tabular")

    def test_2226_tblastn_002(self):
        path = "Blast/tab_2226_tblastn_002.txt"
        with open(path) as stream:
            with self.assertRaisesRegex(ValueError, "Empty file."):
                Align.parse(stream, "tabular")

    def test_2226_tblastn_003(self):
        path = "Blast/tab_2226_tblastn_003.txt"
        with open(path) as stream:
            with self.assertRaisesRegex(ValueError, "Missing header."):
                Align.parse(stream, "tabular")

    def test_2226_tblastn_004(self):
        path = "Blast/tab_2226_tblastn_004.txt"
        with open(path) as stream:
            with self.assertRaisesRegex(ValueError, "Missing header."):
                Align.parse(stream, "tabular")

    def test_2226_tblastn_005(self):
        path = "Blast/tab_2226_tblastn_005.txt"
        with open(path) as stream:
            alignments = Align.parse(stream, "tabular")
            self.assertEqual(alignments.metadata["Program"], "TBLASTN")
            self.assertEqual(alignments.metadata["Version"], "2.2.26+")
            self.assertEqual(alignments.metadata["Database"], "db/minirefseq_mrna")
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|145479850|ref|XM_001425911.1|")
            self.assertEqual(alignment.query.annotations["start"], 30)
            self.assertEqual(alignment.query.annotations["end"], 73)
            self.assertEqual(alignment.target.annotations["start"], 1743)
            self.assertEqual(alignment.target.annotations["end"], 1872)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 34.7)
            self.assertAlmostEqual(alignment.annotations["% identity"], 34.88)
            self.assertEqual(alignment.annotations["alignment length"], 43)
            self.assertEqual(alignment.annotations["mismatches"], 28)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|72012412|ref|XM_777959.1|")
            self.assertEqual(alignment.query.annotations["start"], 43)
            self.assertEqual(alignment.query.annotations["end"], 94)
            self.assertEqual(alignment.target.annotations["start"], 1056)
            self.assertEqual(alignment.target.annotations["end"], 1233)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 31.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 33.90)
            self.assertEqual(alignment.annotations["alignment length"], 59)
            self.assertEqual(alignment.annotations["mismatches"], 31)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|115975252|ref|XM_001180111.1|")
            self.assertEqual(alignment.query.annotations["start"], 43)
            self.assertEqual(alignment.query.annotations["end"], 94)
            self.assertEqual(alignment.target.annotations["start"], 1056)
            self.assertEqual(alignment.target.annotations["end"], 1233)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 31.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 33.90)
            self.assertEqual(alignment.annotations["alignment length"], 59)
            self.assertEqual(alignment.annotations["mismatches"], 31)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|350596019|ref|XM_003360601.2|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 94)
            self.assertEqual(alignment.target.annotations["end"], 388)
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 199)
            self.assertAlmostEqual(alignment.annotations["% identity"], 95.92)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 4)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|350596019|ref|XM_003360601.2|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 29)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 541)
            self.assertEqual(alignment.target.annotations["end"], 754)
            self.assertAlmostEqual(alignment.annotations["evalue"], 4e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 32.7)
            self.assertAlmostEqual(alignment.annotations["% identity"], 29.58)
            self.assertEqual(alignment.annotations["alignment length"], 71)
            self.assertEqual(alignment.annotations["mismatches"], 46)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|301779869|ref|XM_002925302.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 77)
            self.assertEqual(alignment.target.annotations["end"], 371)
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|301779869|ref|XM_002925302.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 2)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 803)
            self.assertEqual(alignment.target.annotations["end"], 1103)
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.00)
            self.assertEqual(alignment.annotations["alignment length"], 100)
            self.assertEqual(alignment.annotations["mismatches"], 64)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|296223671|ref|XM_002757683.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 160)
            self.assertEqual(alignment.target.annotations["end"], 454)
            self.assertAlmostEqual(alignment.annotations["evalue"], 4e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|296223671|ref|XM_002757683.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 2)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 865)
            self.assertEqual(alignment.target.annotations["end"], 1165)
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.00)
            self.assertEqual(alignment.annotations["alignment length"], 100)
            self.assertEqual(alignment.annotations["mismatches"], 64)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|338714227|ref|XM_001492113.3|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 172)
            self.assertEqual(alignment.target.annotations["end"], 466)
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-66, places=66)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|338714227|ref|XM_001492113.3|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 2)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 898)
            self.assertEqual(alignment.target.annotations["end"], 1198)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 46.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 31.00)
            self.assertEqual(alignment.annotations["alignment length"], 100)
            self.assertEqual(alignment.annotations["mismatches"], 63)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|365982352|ref|XM_003667962.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 11)
            self.assertEqual(alignment.query.annotations["end"], 54)
            self.assertEqual(alignment.target.annotations["start"], 3180)
            self.assertEqual(alignment.target.annotations["end"], 3336)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1.7)
            self.assertAlmostEqual(alignment.annotations["bit score"], 19.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.77)
            self.assertEqual(alignment.annotations["alignment length"], 52)
            self.assertEqual(alignment.annotations["mismatches"], 27)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            with self.assertRaises(StopIteration):
                next(alignments)

    def test_2226_tblastn_007(self):
        path = "Blast/tab_2226_tblastn_007.txt"
        with open(path) as stream:
            alignments = Align.parse(stream, "tabular")
            self.assertEqual(alignments.metadata["Program"], "TBLASTN")
            self.assertEqual(alignments.metadata["Version"], "2.2.26+")
            self.assertEqual(alignments.metadata["Database"], "db/minirefseq_mrna")
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|145479850|ref|XM_001425911.1|")
            self.assertEqual(
                alignment.query.description,
                "membrane bound lipoprotein [Bacillus subtilis subsp. subtilis str. 168]",
            )
            self.assertEqual(alignment.query.annotations["start"], 30)
            self.assertEqual(alignment.query.annotations["end"], 73)
            self.assertEqual(alignment.target.annotations["start"], 1743)
            self.assertEqual(alignment.target.annotations["end"], 1872)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 34.7)
            self.assertAlmostEqual(alignment.annotations["% identity"], 34.88)
            self.assertEqual(alignment.annotations["alignment length"], 43)
            self.assertEqual(alignment.annotations["mismatches"], 28)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|72012412|ref|XM_777959.1|")
            self.assertEqual(
                alignment.query.description,
                "membrane bound lipoprotein [Bacillus subtilis subsp. subtilis str. 168]",
            )
            self.assertEqual(alignment.query.annotations["start"], 43)
            self.assertEqual(alignment.query.annotations["end"], 94)
            self.assertEqual(alignment.target.annotations["start"], 1056)
            self.assertEqual(alignment.target.annotations["end"], 1233)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 31.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 33.90)
            self.assertEqual(alignment.annotations["alignment length"], 59)
            self.assertEqual(alignment.annotations["mismatches"], 31)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|115975252|ref|XM_001180111.1|")
            self.assertEqual(
                alignment.query.description,
                "membrane bound lipoprotein [Bacillus subtilis subsp. subtilis str. 168]",
            )
            self.assertEqual(alignment.query.annotations["start"], 43)
            self.assertEqual(alignment.query.annotations["end"], 94)
            self.assertEqual(alignment.target.annotations["start"], 1056)
            self.assertEqual(alignment.target.annotations["end"], 1233)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 31.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 33.90)
            self.assertEqual(alignment.annotations["alignment length"], 59)
            self.assertEqual(alignment.annotations["mismatches"], 31)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            with self.assertRaises(StopIteration):
                next(alignments)

    def test_2226_tblastn_008(self):
        path = "Blast/tab_2226_tblastn_008.txt"
        with open(path) as stream:
            alignments = Align.parse(stream, "tabular")
            self.assertEqual(alignments.metadata["Program"], "TBLASTN")
            self.assertEqual(alignments.metadata["Version"], "2.2.26+")
            self.assertEqual(alignments.metadata["Database"], "db/minirefseq_mrna")

            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|350596019|ref|XM_003360601.2|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 94)
            self.assertEqual(alignment.target.annotations["end"], 388)
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 199)
            self.assertAlmostEqual(alignment.annotations["% identity"], 95.92)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 4)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|350596019|ref|XM_003360601.2|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 29)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 541)
            self.assertEqual(alignment.target.annotations["end"], 754)
            self.assertAlmostEqual(alignment.annotations["evalue"], 4e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 32.7)
            self.assertAlmostEqual(alignment.annotations["% identity"], 29.58)
            self.assertEqual(alignment.annotations["alignment length"], 71)
            self.assertEqual(alignment.annotations["mismatches"], 46)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|301779869|ref|XM_002925302.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 77)
            self.assertEqual(alignment.target.annotations["end"], 371)
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|301779869|ref|XM_002925302.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 2)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 803)
            self.assertEqual(alignment.target.annotations["end"], 1103)
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.00)
            self.assertEqual(alignment.annotations["alignment length"], 100)
            self.assertEqual(alignment.annotations["mismatches"], 64)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|296223671|ref|XM_002757683.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 160)
            self.assertEqual(alignment.target.annotations["end"], 454)
            self.assertAlmostEqual(alignment.annotations["evalue"], 4e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|296223671|ref|XM_002757683.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 2)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 865)
            self.assertEqual(alignment.target.annotations["end"], 1165)
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.00)
            self.assertEqual(alignment.annotations["alignment length"], 100)
            self.assertEqual(alignment.annotations["mismatches"], 64)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|338714227|ref|XM_001492113.3|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 172)
            self.assertEqual(alignment.target.annotations["end"], 466)
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-66, places=66)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|338714227|ref|XM_001492113.3|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 2)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 898)
            self.assertEqual(alignment.target.annotations["end"], 1198)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 46.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 31.00)
            self.assertEqual(alignment.annotations["alignment length"], 100)
            self.assertEqual(alignment.annotations["mismatches"], 63)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|365982352|ref|XM_003667962.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 11)
            self.assertEqual(alignment.query.annotations["end"], 54)
            self.assertEqual(alignment.target.annotations["start"], 3180)
            self.assertEqual(alignment.target.annotations["end"], 3336)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1.7)
            self.assertAlmostEqual(alignment.annotations["bit score"], 19.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.77)
            self.assertEqual(alignment.annotations["alignment length"], 52)
            self.assertEqual(alignment.annotations["mismatches"], 27)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            with self.assertRaises(StopIteration):
                next(alignments)

    def test_2226_tblastn_009(self):
        path = "Blast/tab_2226_tblastn_009.txt"
        with open(path) as stream:
            with self.assertRaisesRegex(ValueError, "Missing header."):
                Align.parse(stream, "tabular")

    def test_2226_tblastn_010(self):
        path = "Blast/tab_2226_tblastn_010.txt"
        with open(path) as stream:
            alignments = Align.parse(stream, "tabular")
            self.assertEqual(alignments.metadata["Program"], "TBLASTN")
            self.assertEqual(alignments.metadata["Version"], "2.2.26+")
            self.assertEqual(alignments.metadata["Database"], "db/minirefseq_mrna")

            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|145479850|ref|XM_001425911.1|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 34.7)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|72012412|ref|XM_777959.1|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 31.6)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|115975252|ref|XM_001180111.1|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 31.6)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|350596019|ref|XM_003360601.2|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 199)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|350596019|ref|XM_003360601.2|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 4e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 32.7)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|301779869|ref|XM_002925302.1|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|301779869|ref|XM_002925302.1|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|296223671|ref|XM_002757683.1|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 4e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|296223671|ref|XM_002757683.1|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|338714227|ref|XM_001492113.3|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-66, places=66)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|338714227|ref|XM_001492113.3|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 46.6)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|365982352|ref|XM_003667962.1|")
            self.assertAlmostEqual(alignment.annotations["evalue"], 1.7)
            self.assertAlmostEqual(alignment.annotations["bit score"], 19.6)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            with self.assertRaises(StopIteration):
                next(alignments)

    def test_2226_tblastn_011(self):
        path = "Blast/tab_2226_tblastn_011.txt"
        with open(path) as stream:
            alignments = Align.parse(stream, "tabular")
            self.assertEqual(alignments.metadata["Program"], "TBLASTN")
            self.assertEqual(alignments.metadata["Version"], "2.2.26+")
            self.assertEqual(alignments.metadata["Database"], "db/minirefseq_mrna")
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(
                alignment.query.description,
                "membrane bound lipoprotein [Bacillus subtilis subsp. subtilis str. 168]",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(
                alignment.query.annotations["acc."], "gi|16080617|ref|NP_391444.1|"
            )
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|16080617|ref|NP_391444.1|"
            )
            self.assertEqual(len(alignment.query.seq), 102)
            self.assertEqual(alignment.target.id, "gi|145479850|ref|XM_001425911.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|145479850|ref|XM_001425911.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|145479850|ref|XM_001425911.1|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|145479850|ref|XM_001425911.1|",
            )
            self.assertEqual(alignment.target.annotations["length"], 4632)
            self.assertEqual(
                str(alignment),
                """\
gi|145479         0 PKTATGTKKGTIIGLLSIHTILFILTSHALSLEVKEQT*KDID 43
                  0 |.....||.||..||...|||.........||...|....|.| 43
gi|160806        30 PDSNIETKEGTYVGLADTHTIEVTVDNEPVSLDITEESTSDLD 73
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    np.array([[0, 43], [30, 73]]),
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 1743)
            self.assertEqual(alignment.target.annotations["end"], 1872)
            self.assertEqual(
                alignment.target.seq, "PKTATGTKKGTIIGLLSIHTILFILTSHALSLEVKEQT*KDID"
            )
            self.assertEqual(
                alignment.query.seq[30:73],
                "PDSNIETKEGTYVGLADTHTIEVTVDNEPVSLDITEESTSDLD",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 34.7)
            self.assertAlmostEqual(alignment.score, 78)
            self.assertEqual(alignment.shape, (2, 43))
            self.assertAlmostEqual(alignment.annotations["% identity"], 34.88)
            self.assertEqual(alignment.annotations["identical"], 15)
            self.assertEqual(alignment.annotations["mismatches"], 28)
            self.assertEqual(alignment.annotations["positives"], 26)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 60.47)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/1")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 77.0; 43 aligned letters; 15 identities; 28 mismatches; 26 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 77.0,
    aligned = 43:
        identities = 15,
        positives = 26,
        mismatches = 28.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 43)
            self.assertEqual(counts.identities, 15)
            self.assertEqual(counts.mismatches, 28)
            self.assertEqual(counts.positives, 26)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(
                alignment.query.description,
                "membrane bound lipoprotein [Bacillus subtilis subsp. subtilis str. 168]",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(
                alignment.query.annotations["acc."], "gi|16080617|ref|NP_391444.1|"
            )
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|16080617|ref|NP_391444.1|"
            )
            self.assertEqual(len(alignment.query.seq), 102)
            self.assertEqual(alignment.target.id, "gi|72012412|ref|XM_777959.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|72012412|ref|XM_777959.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|72012412|ref|XM_777959.1|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"], "gi|72012412|ref|XM_777959.1|"
            )
            self.assertEqual(alignment.target.annotations["length"], 1593)
            self.assertEqual(
                str(alignment),
                """\
gi|720124         0 GLVPDHTLILPVGHYQSMLDLTEEVQTELDQFKSALRKYYLSKGKTCVIYERNFRTQHL
                  0 ||...||....|......||.|||....||.|.|.--------.|....||.|...|.|
gi|160806        43 GLADTHTIEVTVDNEPVSLDITEESTSDLDKFNSG--------DKVTITYEKNDEGQLL

gi|720124        59
                 59
gi|160806        94
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[ 0, 35, 43, 59],
                              [43, 78, 78, 94]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 1056)
            self.assertEqual(alignment.target.annotations["end"], 1233)
            self.assertEqual(
                alignment.target.seq,
                "GLVPDHTLILPVGHYQSMLDLTEEVQTELDQFKSALRKYYLSKGKTCVIYERNFRTQHL",
            )
            self.assertEqual(
                alignment.query.seq[43:94],
                "GLADTHTIEVTVDNEPVSLDITEESTSDLDKFNSGDKVTITYEKNDEGQLL",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 31.6)
            self.assertAlmostEqual(alignment.score, 70)
            self.assertEqual(alignment.shape, (2, 59))
            self.assertAlmostEqual(alignment.annotations["% identity"], 33.90)
            self.assertEqual(alignment.annotations["identical"], 20)
            self.assertEqual(alignment.annotations["mismatches"], 31)
            self.assertEqual(alignment.annotations["positives"], 29)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            self.assertEqual(alignment.annotations["gaps"], 8)
            self.assertAlmostEqual(alignment.annotations["% positives"], 49.15)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/1")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 92.0; 51 aligned letters; 20 identities; 31 mismatches; 29 positives; 8 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 92.0,
    aligned = 51:
        identities = 20,
        positives = 29,
        mismatches = 31.
    gaps = 8:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 8:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 8:
                open_internal_deletions = 1,
                extend_internal_deletions = 7;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 8)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 8)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 8)
            self.assertEqual(counts.gaps, 8)
            self.assertEqual(counts.aligned, 51)
            self.assertEqual(counts.identities, 20)
            self.assertEqual(counts.mismatches, 31)
            self.assertEqual(counts.positives, 29)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(
                alignment.query.description,
                "membrane bound lipoprotein [Bacillus subtilis subsp. subtilis str. 168]",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(
                alignment.query.annotations["acc."], "gi|16080617|ref|NP_391444.1|"
            )
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|16080617|ref|NP_391444.1|"
            )
            self.assertEqual(len(alignment.query.seq), 102)
            self.assertEqual(alignment.target.id, "gi|115975252|ref|XM_001180111.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|115975252|ref|XM_001180111.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|115975252|ref|XM_001180111.1|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|115975252|ref|XM_001180111.1|",
            )
            self.assertEqual(alignment.target.annotations["length"], 1593)
            self.assertEqual(
                str(alignment),
                """\
gi|115975         0 GLVPDHTLILPVGHYQSMLDLTEEVQTELDQFKSALRKYYLSKGKTCVIYERNFRTQHL
                  0 ||...||....|......||.|||....||.|.|.--------.|....||.|...|.|
gi|160806        43 GLADTHTIEVTVDNEPVSLDITEESTSDLDKFNSG--------DKVTITYEKNDEGQLL

gi|115975        59
                 59
gi|160806        94
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[ 0, 35, 43, 59],
                              [43, 78, 78, 94]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 1056)
            self.assertEqual(alignment.target.annotations["end"], 1233)
            self.assertEqual(
                alignment.target.seq,
                "GLVPDHTLILPVGHYQSMLDLTEEVQTELDQFKSALRKYYLSKGKTCVIYERNFRTQHL",
            )
            self.assertEqual(
                alignment.query.seq[43:94],
                "GLADTHTIEVTVDNEPVSLDITEESTSDLDKFNSGDKVTITYEKNDEGQLL",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 31.6)
            self.assertAlmostEqual(alignment.score, 70)
            self.assertEqual(alignment.shape, (2, 59))
            self.assertAlmostEqual(alignment.annotations["% identity"], 33.90)
            self.assertEqual(alignment.annotations["identical"], 20)
            self.assertEqual(alignment.annotations["mismatches"], 31)
            self.assertEqual(alignment.annotations["positives"], 29)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            self.assertEqual(alignment.annotations["gaps"], 8)
            self.assertAlmostEqual(alignment.annotations["% positives"], 49.15)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/1")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 92.0; 51 aligned letters; 20 identities; 31 mismatches; 29 positives; 8 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 92.0,
    aligned = 51:
        identities = 20,
        positives = 29,
        mismatches = 31.
    gaps = 8:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 8:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 8:
                open_internal_deletions = 1,
                extend_internal_deletions = 7;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 8)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 8)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 8)
            self.assertEqual(counts.gaps, 8)
            self.assertEqual(counts.aligned, 51)
            self.assertEqual(counts.identities, 20)
            self.assertEqual(counts.mismatches, 31)
            self.assertEqual(counts.positives, 29)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|11464971:4-101")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|11464971:4-101"
            )
            self.assertEqual(len(alignment.query.seq), 98)
            self.assertEqual(alignment.target.id, "gi|350596019|ref|XM_003360601.2|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|350596019|ref|XM_003360601.2|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|350596019|ref|XM_003360601.2|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|350596019|ref|XM_003360601.2|",
            )
            self.assertEqual(alignment.target.annotations["length"], 772)
            self.assertEqual(
                str(alignment),
                """\
gi|350596         0 KRIREGYLVKKGSMFNTWKPMWVILLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFG
                  0 |||||||||||||.|||||||||.||||||||||||||||||||||||||||||||||||
gi|114649         0 KRIREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFG

gi|350596        60 KRMFVFKITTTKQQDHFFQAAFLEERDGWVRDIKKAIK 98
                 60 |||||.|||||||||||||||||||||.|||||||||| 98
gi|114649        60 KRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK 98
""",
            )
            self.assertTrue(
                np.array_equal(alignment.coordinates, np.array([[0, 98], [0, 98]]))
            )
            self.assertEqual(alignment.target.annotations["start"], 94)
            self.assertEqual(alignment.target.annotations["end"], 388)
            self.assertEqual(
                alignment.target.seq,
                "KRIREGYLVKKGSMFNTWKPMWVILLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVFKITTTKQQDHFFQAAFLEERDGWVRDIKKAIK",
            )
            self.assertEqual(
                alignment.query.seq[0:98],
                "KRIREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 199)
            self.assertAlmostEqual(alignment.score, 506)
            self.assertEqual(alignment.shape, (2, 98))
            self.assertAlmostEqual(alignment.annotations["% identity"], 95.92)
            self.assertEqual(alignment.annotations["identical"], 94)
            self.assertEqual(alignment.annotations["mismatches"], 4)
            self.assertEqual(alignment.annotations["positives"], 96)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 97.96)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/2")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "2")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 508.0; 98 aligned letters; 94 identities; 4 mismatches; 96 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 508.0,
    aligned = 98:
        identities = 94,
        positives = 96,
        mismatches = 4.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 98)
            self.assertEqual(counts.identities, 94)
            self.assertEqual(counts.mismatches, 4)
            self.assertEqual(counts.positives, 96)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|11464971:4-101")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|11464971:4-101"
            )
            self.assertEqual(len(alignment.query.seq), 98)
            self.assertEqual(alignment.target.id, "gi|350596019|ref|XM_003360601.2|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|350596019|ref|XM_003360601.2|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|350596019|ref|XM_003360601.2|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|350596019|ref|XM_003360601.2|",
            )
            self.assertEqual(alignment.target.annotations["length"], 772)
            self.assertEqual(
                str(alignment),
                """\
gi|350596         0 LHYYDPAGGEDPLGAIHLRGCVVTSVESNTDGKNGFLWERAXXITADEVHYFLQAANPKE
                  0 ...|.......|.|.|.|.|...||-.....||..|...---..|......|.|||...|
gi|114649        29 IEFYKKKSDNSPKGMIPLKGSTLTS-PCQDFGKRMFVLK---ITTTKQQDHFFQAAFLEE

gi|350596        60 RTEWIKAIQVA 71
                 60 |..|...|..| 71
gi|114649        85 RDAWVRDIKKA 96
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[ 0, 25, 26, 39, 42, 71],
                              [29, 54, 54, 67, 67, 96]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 541)
            self.assertEqual(alignment.target.annotations["end"], 754)
            self.assertEqual(
                alignment.target.seq,
                "LHYYDPAGGEDPLGAIHLRGCVVTSVESNTDGKNGFLWERAXXITADEVHYFLQAANPKERTEWIKAIQVA",
            )
            self.assertEqual(
                alignment.query.seq[29:96],
                "IEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKA",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 4e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 32.7)
            self.assertAlmostEqual(alignment.score, 73)
            self.assertEqual(alignment.shape, (2, 71))
            self.assertAlmostEqual(alignment.annotations["% identity"], 29.58)
            self.assertEqual(alignment.annotations["identical"], 21)
            self.assertEqual(alignment.annotations["mismatches"], 46)
            self.assertEqual(alignment.annotations["positives"], 33)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            self.assertEqual(alignment.annotations["gaps"], 4)
            self.assertAlmostEqual(alignment.annotations["% positives"], 46.48)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/2")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "2")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 95.0; 67 aligned letters; 21 identities; 46 mismatches; 33 positives; 4 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 95.0,
    aligned = 67:
        identities = 21,
        positives = 33,
        mismatches = 46.
    gaps = 4:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 4:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 4:
                open_internal_deletions = 2,
                extend_internal_deletions = 2;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 4)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 4)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 4)
            self.assertEqual(counts.gaps, 4)
            self.assertEqual(counts.aligned, 67)
            self.assertEqual(counts.identities, 21)
            self.assertEqual(counts.mismatches, 46)
            self.assertEqual(counts.positives, 33)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|11464971:4-101")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|11464971:4-101"
            )
            self.assertEqual(len(alignment.query.seq), 98)
            self.assertEqual(alignment.target.id, "gi|301779869|ref|XM_002925302.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|301779869|ref|XM_002925302.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|301779869|ref|XM_002925302.1|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|301779869|ref|XM_002925302.1|",
            )
            self.assertEqual(alignment.target.annotations["length"], 1144)
            self.assertEqual(
                str(alignment),
                """\
gi|301779         0 KRIREGYLVKRGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFG
                  0 ||||||||||.|||||||||||||||||||||||||||||||||||||||||||||||||
gi|114649         0 KRIREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFG

gi|301779        60 KRMFVFKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK 98
                 60 |||||.|||||||||||||||||||||||||||||||| 98
gi|114649        60 KRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK 98
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    np.array([[0, 98], [0, 98]]),
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 77)
            self.assertEqual(alignment.target.annotations["end"], 371)
            self.assertEqual(
                alignment.target.seq,
                "KRIREGYLVKRGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVFKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK",
            )
            self.assertEqual(
                alignment.query.seq[0:98],
                "KRIREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.score, 515)
            self.assertEqual(alignment.shape, (2, 98))
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["identical"], 96)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["positives"], 97)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 98.98)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/3")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "3")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 513.0; 98 aligned letters; 96 identities; 2 mismatches; 97 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 513.0,
    aligned = 98:
        identities = 96,
        positives = 97,
        mismatches = 2.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 98)
            self.assertEqual(counts.identities, 96)
            self.assertEqual(counts.mismatches, 2)
            self.assertEqual(counts.positives, 97)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|11464971:4-101")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|11464971:4-101"
            )
            self.assertEqual(len(alignment.query.seq), 98)
            self.assertEqual(alignment.target.id, "gi|301779869|ref|XM_002925302.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|301779869|ref|XM_002925302.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|301779869|ref|XM_002925302.1|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|301779869|ref|XM_002925302.1|",
            )
            self.assertEqual(alignment.target.annotations["length"], 1144)
            self.assertEqual(
                str(alignment),
                """\
gi|301779         0 IKQGCLLKQGHRRKNWKVRKFILREDPAYLHYYDPAGGEDPLGAIHLRGCVVTSVESNPD
                  0 |..|.|.|.|.....||.....|.||.--...|.......|.|.|.|.|...||----|.
gi|114649         2 IREGYLVKKGSVFNTWKPMWVVLLEDG--IEFYKKKSDNSPKGMIPLKGSTLTS----PC

gi|301779        60 VRKSEEENLFEIITADEVHYFLQAATPKERTEWIKAIQVA 100
                 60 ...........|.|......|.|||...||..|...|..| 100
gi|114649        56 QDFGKRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKA  96
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[0, 27, 29, 54, 58, 100],
                              [2, 29, 29, 54, 54,  96]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 803)
            self.assertEqual(alignment.target.annotations["end"], 1103)
            self.assertEqual(
                alignment.target.seq,
                "IKQGCLLKQGHRRKNWKVRKFILREDPAYLHYYDPAGGEDPLGAIHLRGCVVTSVESNPDVRKSEEENLFEIITADEVHYFLQAATPKERTEWIKAIQVA",
            )
            self.assertEqual(
                alignment.query.seq[2:96],
                "IREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKA",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            self.assertAlmostEqual(alignment.score, 105)
            self.assertEqual(alignment.shape, (2, 100))
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.00)
            self.assertEqual(alignment.annotations["identical"], 30)
            self.assertEqual(alignment.annotations["mismatches"], 64)
            self.assertEqual(alignment.annotations["positives"], 48)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            self.assertEqual(alignment.annotations["gaps"], 6)
            self.assertAlmostEqual(alignment.annotations["% positives"], 48.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/3")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "3")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 133.0; 94 aligned letters; 30 identities; 64 mismatches; 48 positives; 6 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 133.0,
    aligned = 94:
        identities = 30,
        positives = 48,
        mismatches = 64.
    gaps = 6:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 6:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 6:
                open_internal_deletions = 2,
                extend_internal_deletions = 4;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 6)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 6)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 6)
            self.assertEqual(counts.gaps, 6)
            self.assertEqual(counts.aligned, 94)
            self.assertEqual(counts.identities, 30)
            self.assertEqual(counts.mismatches, 64)
            self.assertEqual(counts.positives, 48)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|11464971:4-101")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|11464971:4-101"
            )
            self.assertEqual(len(alignment.query.seq), 98)
            self.assertEqual(alignment.target.id, "gi|296223671|ref|XM_002757683.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|296223671|ref|XM_002757683.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|296223671|ref|XM_002757683.1|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|296223671|ref|XM_002757683.1|",
            )
            self.assertEqual(alignment.target.annotations["length"], 1183)
            self.assertEqual(
                str(alignment),
                """\
gi|296223         0 KRIREGYLVKKGSMFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFG
                  0 |||||||||||||.||||||||||||||||||||||||||||||||||||||||||||||
gi|114649         0 KRIREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFG

gi|296223        60 KRMFVFKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK 98
                 60 |||||.|||||||||||||||||||||||||||||||| 98
gi|114649        60 KRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK 98
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    np.array([[0, 98], [0, 98]]),
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 160)
            self.assertEqual(alignment.target.annotations["end"], 454)
            self.assertEqual(
                alignment.target.seq,
                "KRIREGYLVKKGSMFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVFKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK",
            )
            self.assertEqual(
                alignment.query.seq[0:98],
                "KRIREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 4e-67, places=67)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.score, 515)
            self.assertEqual(alignment.shape, (2, 98))
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["identical"], 96)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["positives"], 97)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 98.98)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/2")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "2")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 513.0; 98 aligned letters; 96 identities; 2 mismatches; 97 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 513.0,
    aligned = 98:
        identities = 96,
        positives = 97,
        mismatches = 2.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 98)
            self.assertEqual(counts.identities, 96)
            self.assertEqual(counts.mismatches, 2)
            self.assertEqual(counts.positives, 97)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|11464971:4-101")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|11464971:4-101"
            )
            self.assertEqual(len(alignment.query.seq), 98)
            self.assertEqual(alignment.target.id, "gi|296223671|ref|XM_002757683.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|296223671|ref|XM_002757683.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|296223671|ref|XM_002757683.1|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|296223671|ref|XM_002757683.1|",
            )
            self.assertEqual(alignment.target.annotations["length"], 1183)
            self.assertEqual(
                str(alignment),
                """\
gi|296223         0 IKQGCLLKQGHRRKNWKVRKFILREDPAYLHYYDPAGGEDPLGAIHLRGCVVTSVESNSD
                  0 |..|.|.|.|.....||.....|.||.--...|.......|.|.|.|.|...||......
gi|114649         2 IREGYLVKKGSVFNTWKPMWVVLLEDG--IEFYKKKSDNSPKGMIPLKGSTLTSPCQDFG

gi|296223        60 GRKSEEENLFEIITADEVHYFLQAATPKERTEWIKAIQVA 100
                 60 .|..----...|.|......|.|||...||..|...|..| 100
gi|114649        60 KRMF----VLKITTTKQQDHFFQAAFLEERDAWVRDIKKA  96
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[0, 27, 29, 64, 68, 100],
                              [2, 29, 29, 64, 64,  96]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 865)
            self.assertEqual(alignment.target.annotations["end"], 1165)
            self.assertEqual(
                alignment.target.seq,
                "IKQGCLLKQGHRRKNWKVRKFILREDPAYLHYYDPAGGEDPLGAIHLRGCVVTSVESNSDGRKSEEENLFEIITADEVHYFLQAATPKERTEWIKAIQVA",
            )
            self.assertEqual(
                alignment.query.seq[2:96],
                "IREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKA",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            self.assertAlmostEqual(alignment.score, 105)
            self.assertEqual(alignment.shape, (2, 100))
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.00)
            self.assertEqual(alignment.annotations["identical"], 30)
            self.assertEqual(alignment.annotations["mismatches"], 64)
            self.assertEqual(alignment.annotations["positives"], 48)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            self.assertEqual(alignment.annotations["gaps"], 6)
            self.assertAlmostEqual(alignment.annotations["% positives"], 48.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/2")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "2")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 132.0; 94 aligned letters; 30 identities; 64 mismatches; 48 positives; 6 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 132.0,
    aligned = 94:
        identities = 30,
        positives = 48,
        mismatches = 64.
    gaps = 6:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 6:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 6:
                open_internal_deletions = 2,
                extend_internal_deletions = 4;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 6)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 6)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 6)
            self.assertEqual(counts.gaps, 6)
            self.assertEqual(counts.aligned, 94)
            self.assertEqual(counts.identities, 30)
            self.assertEqual(counts.mismatches, 64)
            self.assertEqual(counts.positives, 48)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|11464971:4-101")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|11464971:4-101"
            )
            self.assertEqual(len(alignment.query.seq), 98)
            self.assertEqual(alignment.target.id, "gi|338714227|ref|XM_001492113.3|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|338714227|ref|XM_001492113.3|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|338714227|ref|XM_001492113.3|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|338714227|ref|XM_001492113.3|",
            )
            self.assertEqual(alignment.target.annotations["length"], 1390)
            self.assertEqual(
                str(alignment),
                """\
gi|338714         0 KRIREGYLVKRGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFG
                  0 ||||||||||.|||||||||||||||||||||||||||||||||||||||||||||||||
gi|114649         0 KRIREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFG

gi|338714        60 KRMFVFKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK 98
                 60 |||||.|||||||||||||||||||||||||||||||| 98
gi|114649        60 KRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK 98
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    np.array([[0, 98], [0, 98]]),
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 172)
            self.assertEqual(alignment.target.annotations["end"], 466)
            self.assertEqual(
                alignment.target.seq,
                "KRIREGYLVKRGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVFKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK",
            )
            self.assertEqual(
                alignment.query.seq[0:98],
                "KRIREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKAIK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-66, places=66)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.score, 515)
            self.assertEqual(alignment.shape, (2, 98))
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["identical"], 96)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["positives"], 97)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 98.98)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/2")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "2")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 513.0; 98 aligned letters; 96 identities; 2 mismatches; 97 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 513.0,
    aligned = 98:
        identities = 96,
        positives = 97,
        mismatches = 2.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 98)
            self.assertEqual(counts.identities, 96)
            self.assertEqual(counts.mismatches, 2)
            self.assertEqual(counts.positives, 97)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|11464971:4-101")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|11464971:4-101"
            )
            self.assertEqual(len(alignment.query.seq), 98)
            self.assertEqual(alignment.target.id, "gi|338714227|ref|XM_001492113.3|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|338714227|ref|XM_001492113.3|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|338714227|ref|XM_001492113.3|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|338714227|ref|XM_001492113.3|",
            )
            self.assertEqual(alignment.target.annotations["length"], 1390)
            self.assertEqual(
                str(alignment),
                """\
gi|338714         0 IKQGCLLKQGHRRKNWKVRKFVLREDPAYVHYYDPAGGEEPLGAIHLRGCVVTSVEGNPD
                  0 |..|.|.|.|.....||....||.||.--...|.......|.|.|.|.|...||----|.
gi|114649         2 IREGYLVKKGSVFNTWKPMWVVLLEDG--IEFYKKKSDNSPKGMIPLKGSTLTS----PC

gi|338714        60 GKKSEEENLFEIITADEVHYFLQAATPKERTEWIKAIQVA 100
                 60 ...........|.|......|.|||...||..|...|..| 100
gi|114649        56 QDFGKRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKA  96
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[0, 27, 29, 54, 58, 100],
                              [2, 29, 29, 54, 54,  96]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 898)
            self.assertEqual(alignment.target.annotations["end"], 1198)
            self.assertEqual(
                alignment.target.seq,
                "IKQGCLLKQGHRRKNWKVRKFVLREDPAYVHYYDPAGGEEPLGAIHLRGCVVTSVEGNPDGKKSEEENLFEIITADEVHYFLQAATPKERTEWIKAIQVA",
            )
            self.assertEqual(
                alignment.query.seq[2:96],
                "IREGYLVKKGSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTSPCQDFGKRMFVLKITTTKQQDHFFQAAFLEERDAWVRDIKKA",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 46.6)
            self.assertAlmostEqual(alignment.score, 109)
            self.assertEqual(alignment.shape, (2, 100))
            self.assertAlmostEqual(alignment.annotations["% identity"], 31.00)
            self.assertEqual(alignment.annotations["identical"], 31)
            self.assertEqual(alignment.annotations["mismatches"], 63)
            self.assertEqual(alignment.annotations["positives"], 48)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            self.assertEqual(alignment.annotations["gaps"], 6)
            self.assertAlmostEqual(alignment.annotations["% positives"], 48.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/2")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "2")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 136.0; 94 aligned letters; 31 identities; 63 mismatches; 48 positives; 6 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 136.0,
    aligned = 94:
        identities = 31,
        positives = 48,
        mismatches = 63.
    gaps = 6:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 6:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 6:
                open_internal_deletions = 2,
                extend_internal_deletions = 4;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 6)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 6)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 6)
            self.assertEqual(counts.gaps, 6)
            self.assertEqual(counts.aligned, 94)
            self.assertEqual(counts.identities, 31)
            self.assertEqual(counts.mismatches, 63)
            self.assertEqual(counts.positives, 48)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|11464971:4-101")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|11464971:4-101"
            )
            self.assertEqual(len(alignment.query.seq), 98)
            self.assertEqual(alignment.target.id, "gi|365982352|ref|XM_003667962.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|365982352|ref|XM_003667962.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "0")
            self.assertEqual(alignment.target.annotations["gis"], "0")
            self.assertEqual(
                alignment.target.annotations["acc."], "gi|365982352|ref|XM_003667962.1|"
            )
            self.assertEqual(
                alignment.target.annotations["acc.ver"],
                "gi|365982352|ref|XM_003667962.1|",
            )
            self.assertEqual(alignment.target.annotations["length"], 4932)
            self.assertEqual(
                str(alignment),
                """\
gi|365982         0 GSCFPTWDLIFIEVLNPFLKEKLWEADNEEISKFVDLTLKGLVDLYPSHFTS 52
                  0 ||.|.||.......|---------|...|...|..|...||...|..|..|| 52
gi|114649        11 GSVFNTWKPMWVVLL---------EDGIEFYKKKSDNSPKGMIPLKGSTLTS 54
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[ 0, 15, 24, 52],
                              [11, 26, 26, 54]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.target.annotations["start"], 3180)
            self.assertEqual(alignment.target.annotations["end"], 3336)
            self.assertEqual(
                alignment.target.seq,
                "GSCFPTWDLIFIEVLNPFLKEKLWEADNEEISKFVDLTLKGLVDLYPSHFTS",
            )
            self.assertEqual(
                alignment.query.seq[11:54],
                "GSVFNTWKPMWVVLLEDGIEFYKKKSDNSPKGMIPLKGSTLTS",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1.7)
            self.assertAlmostEqual(alignment.annotations["bit score"], 19.6)
            self.assertAlmostEqual(alignment.score, 39)
            self.assertEqual(alignment.shape, (2, 52))
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.77)
            self.assertEqual(alignment.annotations["identical"], 16)
            self.assertEqual(alignment.annotations["mismatches"], 27)
            self.assertEqual(alignment.annotations["positives"], 23)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            self.assertEqual(alignment.annotations["gaps"], 9)
            self.assertAlmostEqual(alignment.annotations["% positives"], 44.23)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "0/1")
            self.assertEqual(alignment.query.annotations["frame"], "0")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 60.0; 43 aligned letters; 16 identities; 27 mismatches; 23 positives; 9 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 60.0,
    aligned = 43:
        identities = 16,
        positives = 23,
        mismatches = 27.
    gaps = 9:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 9:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 9:
                open_internal_deletions = 1,
                extend_internal_deletions = 8;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 9)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 9)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 9)
            self.assertEqual(counts.gaps, 9)
            self.assertEqual(counts.aligned, 43)
            self.assertEqual(counts.identities, 16)
            self.assertEqual(counts.mismatches, 27)
            self.assertEqual(counts.positives, 23)
            with self.assertRaises(StopIteration):
                next(alignments)

    def test_2226_tblastn_012(self):
        path = "Blast/tab_2226_tblastn_012.txt"
        with open(path) as stream:
            alignments = Align.parse(stream, "tabular")
            self.assertEqual(alignments.metadata["Program"], "TBLASTN")
            self.assertEqual(alignments.metadata["Version"], "2.2.26+")
            self.assertEqual(alignments.metadata["RID"], "X76FDCG9016")
            self.assertEqual(alignments.metadata["Database"], "refseq_rna")

            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|145479850|ref|XM_001425911.1|")
            self.assertEqual(alignment.query.annotations["start"], 30)
            self.assertEqual(alignment.query.annotations["end"], 73)
            self.assertEqual(alignment.target.annotations["start"], 1743)
            self.assertEqual(alignment.target.annotations["end"], 1872)
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.76)
            self.assertAlmostEqual(alignment.annotations["bit score"], 34.7)
            self.assertAlmostEqual(alignment.annotations["% identity"], 34.88)
            self.assertEqual(alignment.annotations["alignment length"], 43)
            self.assertEqual(alignment.annotations["mismatches"], 28)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|115975252|ref|XM_001180111.1|")
            self.assertEqual(alignment.query.annotations["start"], 43)
            self.assertEqual(alignment.query.annotations["end"], 94)
            self.assertEqual(alignment.target.annotations["start"], 1056)
            self.assertEqual(alignment.target.annotations["end"], 1233)
            self.assertAlmostEqual(alignment.annotations["evalue"], 7.2)
            self.assertAlmostEqual(alignment.annotations["bit score"], 31.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 33.90)
            self.assertEqual(alignment.annotations["alignment length"], 59)
            self.assertEqual(alignment.annotations["mismatches"], 31)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|16080617|ref|NP_391444.1|")
            self.assertEqual(alignment.target.id, "gi|390342090|ref|XM_777959.2|")
            self.assertEqual(alignment.query.annotations["start"], 43)
            self.assertEqual(alignment.query.annotations["end"], 94)
            self.assertEqual(alignment.target.annotations["start"], 1266)
            self.assertEqual(alignment.target.annotations["end"], 1443)
            self.assertAlmostEqual(alignment.annotations["evalue"], 7.6)
            self.assertAlmostEqual(alignment.annotations["bit score"], 31.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 33.90)
            self.assertEqual(alignment.annotations["alignment length"], 59)
            self.assertEqual(alignment.annotations["mismatches"], 31)
            self.assertEqual(alignment.annotations["gap opens"], 1)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|354480463|ref|XM_003502378.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 75)
            self.assertEqual(alignment.target.annotations["end"], 369)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-63, places=63)
            self.assertAlmostEqual(alignment.annotations["bit score"], 205)
            self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 0)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|354480463|ref|XM_003502378.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 2)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 801)
            self.assertEqual(alignment.target.annotations["end"], 1101)
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 43.9)
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.00)
            self.assertEqual(alignment.annotations["alignment length"], 100)
            self.assertEqual(alignment.annotations["mismatches"], 64)
            self.assertEqual(alignment.annotations["gap opens"], 3)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|350596019|ref|XM_003360601.2|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 94)
            self.assertEqual(alignment.target.annotations["end"], 388)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-62, places=62)
            self.assertAlmostEqual(alignment.annotations["bit score"], 199)
            self.assertAlmostEqual(alignment.annotations["% identity"], 95.92)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 4)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|350596019|ref|XM_003360601.2|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 29)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 541)
            self.assertEqual(alignment.target.annotations["end"], 754)
            self.assertAlmostEqual(alignment.annotations["evalue"], 2.8)
            self.assertAlmostEqual(alignment.annotations["bit score"], 32.7)
            self.assertAlmostEqual(alignment.annotations["% identity"], 29.58)
            self.assertEqual(alignment.annotations["alignment length"], 71)
            self.assertEqual(alignment.annotations["mismatches"], 46)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|301779869|ref|XM_002925302.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 77)
            self.assertEqual(alignment.target.annotations["end"], 371)
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-62, places=62)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|301779869|ref|XM_002925302.1|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 2)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 803)
            self.assertEqual(alignment.target.annotations["end"], 1103)
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            self.assertAlmostEqual(alignment.annotations["% identity"], 30.00)
            self.assertEqual(alignment.annotations["alignment length"], 100)
            self.assertEqual(alignment.annotations["mismatches"], 64)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|338714227|ref|XM_001492113.3|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 172)
            self.assertEqual(alignment.target.annotations["end"], 466)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-61, places=61)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|338714227|ref|XM_001492113.3|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 2)
            self.assertEqual(alignment.query.annotations["end"], 96)
            self.assertEqual(alignment.target.annotations["start"], 898)
            self.assertEqual(alignment.target.annotations["end"], 1198)
            self.assertAlmostEqual(alignment.annotations["evalue"], 9e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 46.6)
            self.assertAlmostEqual(alignment.annotations["% identity"], 31.00)
            self.assertEqual(alignment.annotations["alignment length"], 100)
            self.assertEqual(alignment.annotations["mismatches"], 63)
            self.assertEqual(alignment.annotations["gap opens"], 2)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|11464971:4-101")
            self.assertEqual(alignment.target.id, "gi|390474391|ref|XM_002757683.2|")
            self.assertEqual(alignment.query.description, "pleckstrin [Mus musculus]")
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 98)
            self.assertEqual(alignment.target.annotations["start"], 160)
            self.assertEqual(alignment.target.annotations["end"], 454)
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-61, places=61)
            self.assertAlmostEqual(alignment.annotations["bit score"], 202)
            self.assertAlmostEqual(alignment.annotations["% identity"], 97.96)
            self.assertEqual(alignment.annotations["alignment length"], 98)
            self.assertEqual(alignment.annotations["mismatches"], 2)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            with self.assertRaises(StopIteration):
                next(alignments)

    def test_2228_tblastn_001(self):
        path = "Blast/tab_2228_tblastn_001.txt"
        with open(path) as stream:
            alignments = Align.parse(stream, "tabular")
            self.assertEqual(alignments.metadata["Program"], "TBLASTN")
            self.assertEqual(alignments.metadata["Version"], "2.2.28+")
            self.assertEqual(alignments.metadata["RID"], "M6BMVNA2015")
            self.assertEqual(alignments.metadata["Database"], "nr")

            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|148227874|ref|NP_001088636.1|")
            self.assertEqual(alignment.query.description, "sirtuin 2 [Xenopus laevis]")
            self.assertEqual(
                alignment.target.annotations["ids"],
                "gi|148227873|ref|NM_001095167.1|;gi|55250552|gb|BC086280.1|",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|148227874|ref|NP_001088636.1|")
            self.assertEqual(alignment.query.description, "sirtuin 2 [Xenopus laevis]")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|512812550|ref|XM_002935781.2|"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|148227874|ref|NP_001088636.1|")
            self.assertEqual(alignment.query.description, "sirtuin 2 [Xenopus laevis]")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|512812554|ref|XM_004910718.1|"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|148227874|ref|NP_001088636.1|")
            self.assertEqual(alignment.query.description, "sirtuin 2 [Xenopus laevis]")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|327289356|ref|XM_003229343.1|"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|148227874|ref|NP_001088636.1|")
            self.assertEqual(alignment.query.description, "sirtuin 2 [Xenopus laevis]")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|602661018|ref|XM_007436108.1|"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|148227874|ref|NP_001088636.1|")
            self.assertEqual(alignment.query.description, "sirtuin 2 [Xenopus laevis]")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|564242014|ref|XM_006277753.1|"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.0)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|148227874|ref|NP_001088636.1|")
            self.assertEqual(alignment.query.description, "sirtuin 2 [Xenopus laevis]")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|591387858|ref|XM_007068281.1|"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-180, places=180)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|148227874|ref|NP_001088636.1|")
            self.assertEqual(alignment.query.description, "sirtuin 2 [Xenopus laevis]")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|558185265|ref|XM_006128143.1|"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 8e-180, places=180)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|148227874|ref|NP_001088636.1|")
            self.assertEqual(alignment.query.description, "sirtuin 2 [Xenopus laevis]")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|573878747|ref|XM_006627536.1|"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-173, places=173)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            alignment = next(alignments)
            self.assertEqual(alignment.query.id, "gi|148227874|ref|NP_001088636.1|")
            self.assertEqual(alignment.query.description, "sirtuin 2 [Xenopus laevis]")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|410910671|ref|XM_003968765.1|"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 8e-173, places=173)
            counts = alignment.counts()
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (0 aligned letters; 0 identities; 0 mismatches; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    aligned = 0:
        identities = 0,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 0)
            self.assertEqual(counts.identities, 0)
            self.assertEqual(counts.mismatches, 0)
            with self.assertRaises(StopIteration):
                next(alignments)

    def test_2228_tblastx_001(self):
        path = "Blast/tab_2228_tblastx_001.txt"
        with open(path) as stream:
            alignments = Align.parse(stream, "tabular")
            self.assertEqual(alignments.metadata["Program"], "TBLASTX")
            self.assertEqual(alignments.metadata["Version"], "2.2.28+")
            self.assertEqual(alignments.metadata["RID"], "P06P5RN0015")
            self.assertEqual(alignments.metadata["Database"], "refseq_rna")
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|296147         0 WP*TLEGLTPCKGNLKQNCVLYLPNRKEEIQPFAMLVINPLRY*KEYIVLRS*KDIRISH
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
gi|296147         0 WP*TLEGLTPCKGNLKQNCVLYLPNRKEEIQPFAMLVINPLRY*KEYIVLRS*KDIRISH

gi|296147        60 SLSCWLANQGMLK*RPWQCNAYRDCQPFHLFLEAGCLKFWMPSLRLLISRWRFN*K 116
                 60 |||||||||||||||||||||||||||||||||||||||||||||||||||||||| 116
gi|296147        60 SLSCWLANQGMLK*RPWQCNAYRDCQPFHLFLEAGCLKFWMPSLRLLISRWRFN*K 116
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 116],
                              [  0, 116]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|296147483|ref|NM_001183135.1|")
            self.assertEqual(
                alignment.target.annotations["ids"],
                "gi|296147483|ref|NM_001183135.1|;gi|116616412|gb|EF059095.1|",
            )
            self.assertEqual(alignment.target.annotations["gi"], "296147483")
            self.assertEqual(alignment.target.annotations["gis"], "296147483;116616412")
            self.assertEqual(alignment.target.annotations["acc."], "NM_001183135")
            self.assertEqual(
                alignment.target.annotations["accs."], "NM_001183135;EF059095"
            )
            self.assertEqual(alignment.target.annotations["acc.ver"], "NM_001183135.1")
            self.assertEqual(alignment.target.annotations["length"], 4911)
            self.assertEqual(alignment.query.annotations["start"], 1)
            self.assertEqual(alignment.query.annotations["end"], 349)
            self.assertEqual(alignment.target.annotations["start"], 1)
            self.assertEqual(alignment.target.annotations["end"], 349)
            self.assertEqual(
                alignment.target.seq,
                "WP*TLEGLTPCKGNLKQNCVLYLPNRKEEIQPFAMLVINPLRY*KEYIVLRS*KDIRISHSLSCWLANQGMLK*RPWQCNAYRDCQPFHLFLEAGCLKFWMPSLRLLISRWRFN*K",
            )
            self.assertEqual(
                alignment.query.seq,
                "WP*TLEGLTPCKGNLKQNCVLYLPNRKEEIQPFAMLVINPLRY*KEYIVLRS*KDIRISHSLSCWLANQGMLK*RPWQCNAYRDCQPFHLFLEAGCLKFWMPSLRLLISRWRFN*K",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-76, places=76)
            self.assertAlmostEqual(alignment.annotations["bit score"], 289)
            self.assertAlmostEqual(alignment.score, 626)
            self.assertEqual(alignment.shape, (2, 116))
            self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
            self.assertEqual(alignment.annotations["identical"], 116)
            self.assertEqual(alignment.annotations["mismatches"], 0)
            self.assertEqual(alignment.annotations["positives"], 116)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 100.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "2/2")
            self.assertEqual(alignment.query.annotations["frame"], "2")
            self.assertEqual(alignment.target.annotations["frame"], "2")
            self.assertEqual(alignment.target.annotations["tax ids"], "32630;559292")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA<>Synthetic construct Saccharomyces cerevisiae clone FLH203015.01X MON2, complete sequence",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 99)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 626.0; 116 aligned letters; 116 identities; 0 mismatches; 116 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 626.0,
    aligned = 116:
        identities = 116,
        positives = 116,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 116)
            self.assertEqual(counts.identities, 116)
            self.assertEqual(counts.mismatches, 0)
            self.assertEqual(counts.positives, 116)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|296147         0 LLIESPSRDE*PQ*RHPKFQTAGFEE*MERLTVPVGIALPGSSF*HSLIGKPTRKGVRNP
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
gi|296147         0 LLIESPSRDE*PQ*RHPKFQTAGFEE*MERLTVPVGIALPGSSF*HSLIGKPTRKGVRNP

gi|296147        60 DVFLAPQNYVLFSISQWIYH*HGEWLNFFFSIRKIKNAILLQVAFAWSQTLQCSWP 116
                 60 |||||||||||||||||||||||||||||||||||||||||||||||||||||||| 116
gi|296147        60 DVFLAPQNYVLFSISQWIYH*HGEWLNFFFSIRKIKNAILLQVAFAWSQTLQCSWP 116
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[0, 116],
                              [0, 116]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|296147483|ref|NM_001183135.1|")
            self.assertEqual(
                alignment.target.annotations["ids"],
                "gi|296147483|ref|NM_001183135.1|;gi|116616412|gb|EF059095.1|",
            )
            self.assertEqual(alignment.target.annotations["gi"], "296147483")
            self.assertEqual(alignment.target.annotations["gis"], "296147483;116616412")
            self.assertEqual(alignment.target.annotations["acc."], "NM_001183135")
            self.assertEqual(
                alignment.target.annotations["accs."], "NM_001183135;EF059095"
            )
            self.assertEqual(alignment.target.annotations["acc.ver"], "NM_001183135.1")
            self.assertEqual(alignment.target.annotations["length"], 4911)
            self.assertEqual(alignment.query.annotations["start"], 349)
            self.assertEqual(alignment.query.annotations["end"], 1)
            self.assertEqual(alignment.target.annotations["start"], 349)
            self.assertEqual(alignment.target.annotations["end"], 1)
            self.assertEqual(
                alignment.target.seq,
                "LLIESPSRDE*PQ*RHPKFQTAGFEE*MERLTVPVGIALPGSSF*HSLIGKPTRKGVRNPDVFLAPQNYVLFSISQWIYH*HGEWLNFFFSIRKIKNAILLQVAFAWSQTLQCSWP",
            )
            self.assertEqual(
                alignment.query.seq,
                "LLIESPSRDE*PQ*RHPKFQTAGFEE*MERLTVPVGIALPGSSF*HSLIGKPTRKGVRNPDVFLAPQNYVLFSISQWIYH*HGEWLNFFFSIRKIKNAILLQVAFAWSQTLQCSWP",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-73, places=73)
            self.assertAlmostEqual(alignment.annotations["bit score"], 278)
            self.assertAlmostEqual(alignment.score, 602)
            self.assertEqual(alignment.shape, (2, 116))
            self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
            self.assertEqual(alignment.annotations["identical"], 116)
            self.assertEqual(alignment.annotations["mismatches"], 0)
            self.assertEqual(alignment.annotations["positives"], 116)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 100.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-2/-3")
            self.assertEqual(alignment.query.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["tax ids"], "32630;559292")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA<>Synthetic construct Saccharomyces cerevisiae clone FLH203015.01X MON2, complete sequence",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 99)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 602.0; 116 aligned letters; 116 identities; 0 mismatches; 116 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 602.0,
    aligned = 116:
        identities = 116,
        positives = 116,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 116)
            self.assertEqual(counts.identities, 116)
            self.assertEqual(counts.mismatches, 0)
            self.assertEqual(counts.positives, 116)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|296147         0 F*LNLHREMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIR
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
gi|296147         0 F*LNLHREMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIR

gi|296147        60 MSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKLPLHGVKPSSVHGH 116
                 60 |||||||||||||||||||||||||||||||||||||||||||||||||||||||| 116
gi|296147        60 MSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKLPLHGVKPSSVHGH 116
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 116],
                              [  0, 116]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|296147483|ref|NM_001183135.1|")
            self.assertEqual(
                alignment.target.annotations["ids"],
                "gi|296147483|ref|NM_001183135.1|;gi|116616412|gb|EF059095.1|",
            )
            self.assertEqual(alignment.target.annotations["gi"], "296147483")
            self.assertEqual(alignment.target.annotations["gis"], "296147483;116616412")
            self.assertEqual(alignment.target.annotations["acc."], "NM_001183135")
            self.assertEqual(
                alignment.target.annotations["accs."], "NM_001183135;EF059095"
            )
            self.assertEqual(alignment.target.annotations["acc.ver"], "NM_001183135.1")
            self.assertEqual(alignment.target.annotations["length"], 4911)
            self.assertEqual(alignment.query.annotations["start"], 348)
            self.assertEqual(alignment.query.annotations["end"], 0)
            self.assertEqual(alignment.target.annotations["start"], 348)
            self.assertEqual(alignment.target.annotations["end"], 0)
            self.assertEqual(
                alignment.target.seq,
                "F*LNLHREMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKLPLHGVKPSSVHGH",
            )
            self.assertEqual(
                alignment.query.seq,
                "F*LNLHREMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKLPLHGVKPSSVHGH",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-72, places=72)
            self.assertAlmostEqual(alignment.annotations["bit score"], 274)
            self.assertAlmostEqual(alignment.score, 593)
            self.assertEqual(alignment.shape, (2, 116))
            self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
            self.assertEqual(alignment.annotations["identical"], 116)
            self.assertEqual(alignment.annotations["mismatches"], 0)
            self.assertEqual(alignment.annotations["positives"], 116)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 100.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-3/-1")
            self.assertEqual(alignment.query.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["tax ids"], "32630;559292")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA<>Synthetic construct Saccharomyces cerevisiae clone FLH203015.01X MON2, complete sequence",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 99)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 593.0; 116 aligned letters; 116 identities; 0 mismatches; 116 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 593.0,
    aligned = 116:
        identities = 116,
        positives = 116,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 116)
            self.assertEqual(counts.identities, 116)
            self.assertEqual(counts.mismatches, 0)
            self.assertEqual(counts.positives, 116)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|296147         0 GHEHWRV*LHAKAT*SRIAFFIFRIEKKKFNHSPC***IH*DIEKST*F*GARKTSGFRT
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
gi|296147         0 GHEHWRV*LHAKAT*SRIAFFIFRIEKKKFNHSPC***IH*DIEKST*F*GARKTSGFRT

gi|296147        60 PFRVGLPIKEC*NDDPGNAMPTGTVNRSIYSSKPAV*NFGCLH*GYSSRDGDSIKS 116
                 60 |||||||||||||||||||||||||||||||||||||||||||||||||||||||| 116
gi|296147        60 PFRVGLPIKEC*NDDPGNAMPTGTVNRSIYSSKPAV*NFGCLH*GYSSRDGDSIKS 116
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 116],
                              [  0, 116]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|296147483|ref|NM_001183135.1|")
            self.assertEqual(
                alignment.target.annotations["ids"],
                "gi|296147483|ref|NM_001183135.1|;gi|116616412|gb|EF059095.1|",
            )
            self.assertEqual(alignment.target.annotations["gi"], "296147483")
            self.assertEqual(alignment.target.annotations["gis"], "296147483;116616412")
            self.assertEqual(alignment.target.annotations["acc."], "NM_001183135")
            self.assertEqual(
                alignment.target.annotations["accs."], "NM_001183135;EF059095"
            )
            self.assertEqual(alignment.target.annotations["acc.ver"], "NM_001183135.1")
            self.assertEqual(alignment.target.annotations["length"], 4911)
            self.assertEqual(alignment.query.annotations["start"], 2)
            self.assertEqual(alignment.query.annotations["end"], 350)
            self.assertEqual(alignment.target.annotations["start"], 2)
            self.assertEqual(alignment.target.annotations["end"], 350)
            self.assertEqual(
                alignment.target.seq,
                "GHEHWRV*LHAKAT*SRIAFFIFRIEKKKFNHSPC***IH*DIEKST*F*GARKTSGFRTPFRVGLPIKEC*NDDPGNAMPTGTVNRSIYSSKPAV*NFGCLH*GYSSRDGDSIKS",
            )
            self.assertEqual(
                alignment.query.seq,
                "GHEHWRV*LHAKAT*SRIAFFIFRIEKKKFNHSPC***IH*DIEKST*F*GARKTSGFRTPFRVGLPIKEC*NDDPGNAMPTGTVNRSIYSSKPAV*NFGCLH*GYSSRDGDSIKS",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-70, places=70)
            self.assertAlmostEqual(alignment.annotations["bit score"], 270)
            self.assertAlmostEqual(alignment.score, 583)
            self.assertEqual(alignment.shape, (2, 116))
            self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
            self.assertEqual(alignment.annotations["identical"], 116)
            self.assertEqual(alignment.annotations["mismatches"], 0)
            self.assertEqual(alignment.annotations["positives"], 116)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 100.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "3/3")
            self.assertEqual(alignment.query.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["tax ids"], "32630;559292")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA<>Synthetic construct Saccharomyces cerevisiae clone FLH203015.01X MON2, complete sequence",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 99)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 583.0; 116 aligned letters; 116 identities; 0 mismatches; 116 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 583.0,
    aligned = 116:
        identities = 116,
        positives = 116,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 116)
            self.assertEqual(counts.identities, 116)
            self.assertEqual(counts.mismatches, 0)
            self.assertEqual(counts.positives, 116)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTKGSAKS
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTKGSAKS

gi|296147        60 GCLSSSSKLCTLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCLCMESNPPVFMA 116
                 60 |||||||||||||||||||||||||||||||||||||||||||||||||||||||| 116
gi|296147        60 GCLSSSSKLCTLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCLCMESNPPVFMA 116
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 116],
                              [  0, 116]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|296147483|ref|NM_001183135.1|")
            self.assertEqual(
                alignment.target.annotations["ids"],
                "gi|296147483|ref|NM_001183135.1|;gi|116616412|gb|EF059095.1|",
            )
            self.assertEqual(alignment.target.annotations["gi"], "296147483")
            self.assertEqual(alignment.target.annotations["gis"], "296147483;116616412")
            self.assertEqual(alignment.target.annotations["acc."], "NM_001183135")
            self.assertEqual(
                alignment.target.annotations["accs."], "NM_001183135;EF059095"
            )
            self.assertEqual(alignment.target.annotations["acc.ver"], "NM_001183135.1")
            self.assertEqual(alignment.target.annotations["length"], 4911)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 2)
            self.assertEqual(alignment.target.annotations["start"], 350)
            self.assertEqual(alignment.target.annotations["end"], 2)
            self.assertEqual(
                alignment.target.seq,
                "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTKGSAKSGCLSSSSKLCTLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCLCMESNPPVFMA",
            )
            self.assertEqual(
                alignment.query.seq,
                "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTKGSAKSGCLSSSSKLCTLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCLCMESNPPVFMA",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-58, places=58)
            self.assertAlmostEqual(alignment.annotations["bit score"], 229)
            self.assertAlmostEqual(alignment.score, 495)
            self.assertEqual(alignment.shape, (2, 116))
            self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
            self.assertEqual(alignment.annotations["identical"], 116)
            self.assertEqual(alignment.annotations["mismatches"], 0)
            self.assertEqual(alignment.annotations["positives"], 116)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 100.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "32630;559292")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA<>Synthetic construct Saccharomyces cerevisiae clone FLH203015.01X MON2, complete sequence",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 99)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 586.0; 116 aligned letters; 116 identities; 0 mismatches; 116 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 586.0,
    aligned = 116:
        identities = 116,
        positives = 116,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 116)
            self.assertEqual(counts.identities, 116)
            self.assertEqual(counts.mismatches, 0)
            self.assertEqual(counts.positives, 116)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|296147         0 TIRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSI
                  0 ||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
gi|296147         0 TIRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSI

gi|296147        60 PRSRLSEILDAFIEATHLAMEIQLK 85
                 60 ||||||||||||||||||||||||| 85
gi|296147        60 PRSRLSEILDAFIEATHLAMEIQLK 85
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  85],
                              [  0,  85]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|296147483|ref|NM_001183135.1|")
            self.assertEqual(
                alignment.target.annotations["ids"],
                "gi|296147483|ref|NM_001183135.1|;gi|116616412|gb|EF059095.1|",
            )
            self.assertEqual(alignment.target.annotations["gi"], "296147483")
            self.assertEqual(alignment.target.annotations["gis"], "296147483;116616412")
            self.assertEqual(alignment.target.annotations["acc."], "NM_001183135")
            self.assertEqual(
                alignment.target.annotations["accs."], "NM_001183135;EF059095"
            )
            self.assertEqual(alignment.target.annotations["acc.ver"], "NM_001183135.1")
            self.assertEqual(alignment.target.annotations["length"], 4911)
            self.assertEqual(alignment.query.annotations["start"], 93)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 93)
            self.assertEqual(alignment.target.annotations["end"], 348)
            self.assertEqual(
                alignment.target.seq,
                "TIRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "TIRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-53, places=53)
            self.assertAlmostEqual(alignment.annotations["bit score"], 197)
            self.assertAlmostEqual(alignment.score, 425)
            self.assertEqual(alignment.shape, (2, 85))
            self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
            self.assertEqual(alignment.annotations["identical"], 85)
            self.assertEqual(alignment.annotations["mismatches"], 0)
            self.assertEqual(alignment.annotations["positives"], 85)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 100.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "32630;559292")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA<>Synthetic construct Saccharomyces cerevisiae clone FLH203015.01X MON2, complete sequence",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 73)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 425.0; 85 aligned letters; 85 identities; 0 mismatches; 85 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 425.0,
    aligned = 85:
        identities = 85,
        positives = 85,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 85)
            self.assertEqual(counts.identities, 85)
            self.assertEqual(counts.mismatches, 0)
            self.assertEqual(counts.positives, 85)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|296147         0 MAMNTGGFDSMQRQ 14
                  0 |||||||||||||| 14
gi|296147         0 MAMNTGGFDSMQRQ 14
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[ 0, 14],
                              [ 0, 14]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|296147483|ref|NM_001183135.1|")
            self.assertEqual(
                alignment.target.annotations["ids"],
                "gi|296147483|ref|NM_001183135.1|;gi|116616412|gb|EF059095.1|",
            )
            self.assertEqual(alignment.target.annotations["gi"], "296147483")
            self.assertEqual(alignment.target.annotations["gis"], "296147483;116616412")
            self.assertEqual(alignment.target.annotations["acc."], "NM_001183135")
            self.assertEqual(
                alignment.target.annotations["accs."], "NM_001183135;EF059095"
            )
            self.assertEqual(alignment.target.annotations["acc.ver"], "NM_001183135.1")
            self.assertEqual(alignment.target.annotations["length"], 4911)
            self.assertEqual(alignment.query.annotations["start"], 0)
            self.assertEqual(alignment.query.annotations["end"], 42)
            self.assertEqual(alignment.target.annotations["start"], 0)
            self.assertEqual(alignment.target.annotations["end"], 42)
            self.assertEqual(alignment.target.seq, "MAMNTGGFDSMQRQ")
            self.assertEqual(alignment.query.seq, "MAMNTGGFDSMQRQ")
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-53, places=53)
            self.assertAlmostEqual(alignment.annotations["bit score"], 36.3)
            self.assertAlmostEqual(alignment.score, 73)
            self.assertEqual(alignment.shape, (2, 14))
            self.assertAlmostEqual(alignment.annotations["% identity"], 100.00)
            self.assertEqual(alignment.annotations["identical"], 14)
            self.assertEqual(alignment.annotations["mismatches"], 0)
            self.assertEqual(alignment.annotations["positives"], 14)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 100.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "32630;559292")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A;N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Saccharomyces cerevisiae S288c Mon2p (MON2), mRNA<>Synthetic construct Saccharomyces cerevisiae clone FLH203015.01X MON2, complete sequence",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 12)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 73.0; 14 aligned letters; 14 identities; 0 mismatches; 14 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 73.0,
    aligned = 14:
        identities = 14,
        positives = 14,
        mismatches = 0.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 14)
            self.assertEqual(counts.identities, 14)
            self.assertEqual(counts.mismatches, 0)
            self.assertEqual(counts.positives, 14)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|365982         0 TIKHASDKSIDILKTIQNIEELVRHPDFVTPLVLACSSRNAKLTSIAMQCLQGLASVPSI
                  0 ||.|||||||.|||.....|||.|||||..|.||||.|||||.|..||||||||..||||
gi|296147         0 TIRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSI

gi|365982        60 PESRIPEVLDGFIEATQLAMEIQLK 85
                 60 |.||..|.||.|||||.|||||||| 85
gi|296147        60 PRSRLSEILDAFIEATHLAMEIQLK 85
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  85],
                              [  0,  85]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|365982352|ref|XM_003667962.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|365982352|ref|XM_003667962.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "365982352")
            self.assertEqual(alignment.target.annotations["gis"], "365982352")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003667962.1")
            self.assertEqual(alignment.target.annotations["length"], 4932)
            self.assertEqual(alignment.query.annotations["start"], 93)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 87)
            self.assertEqual(alignment.target.annotations["end"], 342)
            self.assertEqual(
                alignment.target.seq,
                "TIKHASDKSIDILKTIQNIEELVRHPDFVTPLVLACSSRNAKLTSIAMQCLQGLASVPSIPESRIPEVLDGFIEATQLAMEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "TIRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-37, places=37)
            self.assertAlmostEqual(alignment.annotations["bit score"], 152)
            self.assertAlmostEqual(alignment.score, 327)
            self.assertEqual(alignment.shape, (2, 85))
            self.assertAlmostEqual(alignment.annotations["% identity"], 72.94)
            self.assertEqual(alignment.annotations["identical"], 62)
            self.assertEqual(alignment.annotations["mismatches"], 23)
            self.assertEqual(alignment.annotations["positives"], 73)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 85.88)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071378")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 94)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 73)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 327.0; 85 aligned letters; 62 identities; 23 mismatches; 73 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 327.0,
    aligned = 85:
        identities = 62,
        positives = 73,
        mismatches = 23.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 85)
            self.assertEqual(counts.identities, 62)
            self.assertEqual(counts.mismatches, 23)
            self.assertEqual(counts.positives, 73)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|365982         0 FRI*KKKFNH*TC* 14
                  0 |||.||||||..|| 14
gi|296147         0 FRIEKKKFNHSPC* 14
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[ 0, 14],
                              [ 0, 14]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|365982352|ref|XM_003667962.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|365982352|ref|XM_003667962.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "365982352")
            self.assertEqual(alignment.target.annotations["gis"], "365982352")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003667962.1")
            self.assertEqual(alignment.target.annotations["length"], 4932)
            self.assertEqual(alignment.query.annotations["start"], 68)
            self.assertEqual(alignment.query.annotations["end"], 110)
            self.assertEqual(alignment.target.annotations["start"], 62)
            self.assertEqual(alignment.target.annotations["end"], 104)
            self.assertEqual(alignment.target.seq, "FRI*KKKFNH*TC*")
            self.assertEqual(alignment.query.seq, "FRIEKKKFNHSPC*")
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-37, places=37)
            self.assertAlmostEqual(alignment.annotations["bit score"], 26.3)
            self.assertAlmostEqual(alignment.score, 51)
            self.assertEqual(alignment.shape, (2, 14))
            self.assertAlmostEqual(alignment.annotations["% identity"], 78.57)
            self.assertEqual(alignment.annotations["identical"], 11)
            self.assertEqual(alignment.annotations["mismatches"], 3)
            self.assertEqual(alignment.annotations["positives"], 11)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 78.57)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "3/3")
            self.assertEqual(alignment.query.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071378")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 94)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 12)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 51.0; 14 aligned letters; 11 identities; 3 mismatches; 11 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 51.0,
    aligned = 14:
        identities = 11,
        positives = 11,
        mismatches = 3.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 14)
            self.assertEqual(counts.identities, 11)
            self.assertEqual(counts.mismatches, 3)
            self.assertEqual(counts.positives, 11)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|365982         0 TFNCISIAS*VASMNPSRTSGILDSGIEGTDANPCRHCMAIEVNLAFLDEQARTK 55
                  0 |||.||||.|||||..|..|.....||.||...|||||.|..|.|||||.||.|| 55
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTK 55
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  55],
                              [  0,  55]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|365982352|ref|XM_003667962.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|365982352|ref|XM_003667962.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "365982352")
            self.assertEqual(alignment.target.annotations["gis"], "365982352")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003667962.1")
            self.assertEqual(alignment.target.annotations["length"], 4932)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 185)
            self.assertEqual(alignment.target.annotations["start"], 344)
            self.assertEqual(alignment.target.annotations["end"], 179)
            self.assertEqual(
                alignment.target.seq,
                "TFNCISIAS*VASMNPSRTSGILDSGIEGTDANPCRHCMAIEVNLAFLDEQARTK",
            )
            self.assertEqual(
                alignment.query.seq,
                "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 6e-20, places=20)
            self.assertAlmostEqual(alignment.annotations["bit score"], 68.0)
            self.assertAlmostEqual(alignment.score, 142)
            self.assertEqual(alignment.shape, (2, 55))
            self.assertAlmostEqual(alignment.annotations["% identity"], 61.82)
            self.assertEqual(alignment.annotations["identical"], 34)
            self.assertEqual(alignment.annotations["mismatches"], 21)
            self.assertEqual(alignment.annotations["positives"], 38)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 69.09)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071378")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 94)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 47)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 142.0; 55 aligned letters; 34 identities; 21 mismatches; 38 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 142.0,
    aligned = 55:
        identities = 34,
        positives = 38,
        mismatches = 21.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 55)
            self.assertEqual(counts.identities, 34)
            self.assertEqual(counts.mismatches, 21)
            self.assertEqual(counts.positives, 38)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|365982         0 VFRISIDLSLACLMVEFLLLDSEERECNSWSNCFRKEVN 39
                  0 .|.||.|||||..||||||.|||..|.||.|.|...|.| 39
gi|296147         0 LFNISMDLSLAWRMVEFLLFDSEDKERNSASSCLCMESN 39
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  39],
                              [  0,  39]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|365982352|ref|XM_003667962.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|365982352|ref|XM_003667962.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "365982352")
            self.assertEqual(alignment.target.annotations["gis"], "365982352")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003667962.1")
            self.assertEqual(alignment.target.annotations["length"], 4932)
            self.assertEqual(alignment.query.annotations["start"], 137)
            self.assertEqual(alignment.query.annotations["end"], 20)
            self.assertEqual(alignment.target.annotations["start"], 131)
            self.assertEqual(alignment.target.annotations["end"], 14)
            self.assertEqual(
                alignment.target.seq, "VFRISIDLSLACLMVEFLLLDSEERECNSWSNCFRKEVN"
            )
            self.assertEqual(
                alignment.query.seq, "LFNISMDLSLAWRMVEFLLFDSEDKERNSASSCLCMESN"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 6e-20, places=20)
            self.assertAlmostEqual(alignment.annotations["bit score"], 52.8)
            self.assertAlmostEqual(alignment.score, 109)
            self.assertEqual(alignment.shape, (2, 39))
            self.assertAlmostEqual(alignment.annotations["% identity"], 61.54)
            self.assertEqual(alignment.annotations["identical"], 24)
            self.assertEqual(alignment.annotations["mismatches"], 15)
            self.assertEqual(alignment.annotations["positives"], 29)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 74.36)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071378")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 94)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 33)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 109.0; 39 aligned letters; 24 identities; 15 mismatches; 29 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 109.0,
    aligned = 39:
        identities = 24,
        positives = 29,
        mismatches = 15.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 39)
            self.assertEqual(counts.identities, 24)
            self.assertEqual(counts.mismatches, 15)
            self.assertEqual(counts.positives, 29)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|365982         0 NWTRNYTPFLQNLKEEIQPLNMLATNQSIF*RLFKI*KNL*GTLILLHLWSWPVHREMPN
                  0 |...|....|.|.||||||..||..|....|.............|...|..|.....|..
gi|296147         0 NLKQNCVLYLPNRKEEIQPFAMLVINPLRY*KEYIVLRS*KDIRISHSLSCWLANQGMLK

gi|365982        60 *LQLPCNAYKDWRPYLQYQSQESQKF*TDSLKPLS*RWRYN*K 103
                 60 |....||||.|..|..........||...||..|..|||.||| 103
gi|296147        60 *RPWQCNAYRDCQPFHLFLEAGCLKFWMPSLRLLISRWRFN*K 103
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 103],
                              [  0, 103]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|365982352|ref|XM_003667962.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|365982352|ref|XM_003667962.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "365982352")
            self.assertEqual(alignment.target.annotations["gis"], "365982352")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003667962.1")
            self.assertEqual(alignment.target.annotations["length"], 4932)
            self.assertEqual(alignment.query.annotations["start"], 40)
            self.assertEqual(alignment.query.annotations["end"], 349)
            self.assertEqual(alignment.target.annotations["start"], 34)
            self.assertEqual(alignment.target.annotations["end"], 343)
            self.assertEqual(
                alignment.target.seq,
                "NWTRNYTPFLQNLKEEIQPLNMLATNQSIF*RLFKI*KNL*GTLILLHLWSWPVHREMPN*LQLPCNAYKDWRPYLQYQSQESQKF*TDSLKPLS*RWRYN*K",
            )
            self.assertEqual(
                alignment.query.seq,
                "NLKQNCVLYLPNRKEEIQPFAMLVINPLRY*KEYIVLRS*KDIRISHSLSCWLANQGMLK*RPWQCNAYRDCQPFHLFLEAGCLKFWMPSLRLLISRWRFN*K",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 9e-08, places=8)
            self.assertAlmostEqual(alignment.annotations["bit score"], 61.1)
            self.assertAlmostEqual(alignment.score, 127)
            self.assertEqual(alignment.shape, (2, 103))
            self.assertAlmostEqual(alignment.annotations["% identity"], 34.95)
            self.assertEqual(alignment.annotations["identical"], 36)
            self.assertEqual(alignment.annotations["mismatches"], 67)
            self.assertEqual(alignment.annotations["positives"], 52)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 50.49)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "2/2")
            self.assertEqual(alignment.query.annotations["frame"], "2")
            self.assertEqual(alignment.target.annotations["frame"], "2")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071378")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 94)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 88)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 127.0; 103 aligned letters; 36 identities; 67 mismatches; 52 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 127.0,
    aligned = 103:
        identities = 36,
        positives = 52,
        mismatches = 67.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 103)
            self.assertEqual(counts.identities, 36)
            self.assertEqual(counts.mismatches, 67)
            self.assertEqual(counts.positives, 52)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|365982         0 GRQSL*ALHGN*S*FGISR*TGQDQRCNKIRVPYKFFYILNSL*NID*FVASMFNG*ISS
                  0 |.|||||||.....|.|.....|..|...||..........|......|..||.||||||
gi|296147         0 G*QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISS

gi|365982        60 FRF*R 65
                 60 |||.| 65
gi|296147        60 FRFGR 65
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  65],
                              [  0,  65]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|365982352|ref|XM_003667962.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|365982352|ref|XM_003667962.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "365982352")
            self.assertEqual(alignment.target.annotations["gis"], "365982352")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003667962")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003667962.1")
            self.assertEqual(alignment.target.annotations["length"], 4932)
            self.assertEqual(alignment.query.annotations["start"], 261)
            self.assertEqual(alignment.query.annotations["end"], 66)
            self.assertEqual(alignment.target.annotations["start"], 255)
            self.assertEqual(alignment.target.annotations["end"], 60)
            self.assertEqual(
                alignment.target.seq,
                "GRQSL*ALHGN*S*FGISR*TGQDQRCNKIRVPYKFFYILNSL*NID*FVASMFNG*ISSFRF*R",
            )
            self.assertEqual(
                alignment.query.seq,
                "G*QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.029)
            self.assertAlmostEqual(alignment.annotations["bit score"], 42.8)
            self.assertAlmostEqual(alignment.score, 87)
            self.assertEqual(alignment.shape, (2, 65))
            self.assertAlmostEqual(alignment.annotations["% identity"], 43.08)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 37)
            self.assertEqual(alignment.annotations["positives"], 36)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 55.38)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-3/-1")
            self.assertEqual(alignment.query.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071378")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Naumovozyma dairenensis CBS 421 hypothetical protein (NDAI0A06120), mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 94)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 56)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 87.0; 65 aligned letters; 28 identities; 37 mismatches; 36 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 87.0,
    aligned = 65:
        identities = 28,
        positives = 36,
        mismatches = 37.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 65)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 37)
            self.assertEqual(counts.positives, 36)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|366988         0 SIKHASDKSIEILKTIQNIEDLASHPDFVTPLVESCLSRNAKLTSIAMQCLQGLASAPSI
                  0 .|.|||||||||||.....|.|..||||..|.|..|.|||||.|..||||||||...|||
gi|296147         0 TIRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSI

gi|366988        60 PESRLSGVLDGFIEATHLAIEIQLK 85
                 60 |.||||..||.||||||||.||||| 85
gi|296147        60 PRSRLSEILDAFIEATHLAMEIQLK 85
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  85],
                              [  0,  85]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|366988334|ref|XM_003673886.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|366988334|ref|XM_003673886.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "366988334")
            self.assertEqual(alignment.target.annotations["gis"], "366988334")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003673886")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003673886")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003673886.1")
            self.assertEqual(alignment.target.annotations["length"], 4938)
            self.assertEqual(alignment.query.annotations["start"], 93)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 93)
            self.assertEqual(alignment.target.annotations["end"], 348)
            self.assertEqual(
                alignment.target.seq,
                "SIKHASDKSIEILKTIQNIEDLASHPDFVTPLVESCLSRNAKLTSIAMQCLQGLASAPSIPESRLSGVLDGFIEATHLAIEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "TIRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-32, places=32)
            self.assertAlmostEqual(alignment.annotations["bit score"], 143)
            self.assertAlmostEqual(alignment.score, 306)
            self.assertEqual(alignment.shape, (2, 85))
            self.assertAlmostEqual(alignment.annotations["% identity"], 68.24)
            self.assertEqual(alignment.annotations["identical"], 58)
            self.assertEqual(alignment.annotations["mismatches"], 27)
            self.assertEqual(alignment.annotations["positives"], 71)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 83.53)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "1064592")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Naumovozyma castellii CBS 4309 hypothetical protein (NCAS0A09950) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Naumovozyma castellii CBS 4309 hypothetical protein (NCAS0A09950) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 73)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 306.0; 85 aligned letters; 58 identities; 27 mismatches; 71 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 306.0,
    aligned = 85:
        identities = 58,
        positives = 71,
        mismatches = 27.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 85)
            self.assertEqual(counts.identities, 58)
            self.assertEqual(counts.mismatches, 27)
            self.assertEqual(counts.positives, 71)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|366988         0 TFSCISIARCVASINPSRTPDSLLSGMDGADAKPCKHCIAIEVNLAFLDRHDST 54
                  0 ||..|||||.|||...|...||.|.|.||....||.||||..|.|||||....| 54
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANT 54
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  54],
                              [  0,  54]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|366988334|ref|XM_003673886.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|366988334|ref|XM_003673886.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "366988334")
            self.assertEqual(alignment.target.annotations["gis"], "366988334")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003673886")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003673886")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003673886.1")
            self.assertEqual(alignment.target.annotations["length"], 4938)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 188)
            self.assertEqual(alignment.target.annotations["start"], 350)
            self.assertEqual(alignment.target.annotations["end"], 188)
            self.assertEqual(
                alignment.target.seq,
                "TFSCISIARCVASINPSRTPDSLLSGMDGADAKPCKHCIAIEVNLAFLDRHDST",
            )
            self.assertEqual(
                alignment.query.seq,
                "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANT",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 7e-16, places=16)
            self.assertAlmostEqual(alignment.annotations["bit score"], 62.5)
            self.assertAlmostEqual(alignment.score, 130)
            self.assertEqual(alignment.shape, (2, 54))
            self.assertAlmostEqual(alignment.annotations["% identity"], 55.56)
            self.assertEqual(alignment.annotations["identical"], 30)
            self.assertEqual(alignment.annotations["mismatches"], 24)
            self.assertEqual(alignment.annotations["positives"], 36)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 66.67)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "1064592")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Naumovozyma castellii CBS 4309 hypothetical protein (NCAS0A09950) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Naumovozyma castellii CBS 4309 hypothetical protein (NCAS0A09950) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 46)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 130.0; 54 aligned letters; 30 identities; 24 mismatches; 36 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 130.0,
    aligned = 54:
        identities = 30,
        positives = 36,
        mismatches = 24.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 54)
            self.assertEqual(counts.identities, 30)
            self.assertEqual(counts.mismatches, 24)
            self.assertEqual(counts.positives, 36)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|366988         0 VFKISIDLSLACLIEVFLLLDSEDNECNSRSNC 33
                  0 .|.||.|||||.....|||.||||.|.||.|.| 33
gi|296147         0 LFNISMDLSLAWRMVEFLLFDSEDKERNSASSC 33
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  33],
                              [  0,  33]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|366988334|ref|XM_003673886.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|366988334|ref|XM_003673886.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "366988334")
            self.assertEqual(alignment.target.annotations["gis"], "366988334")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003673886")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003673886")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003673886.1")
            self.assertEqual(alignment.target.annotations["length"], 4938)
            self.assertEqual(alignment.query.annotations["start"], 137)
            self.assertEqual(alignment.query.annotations["end"], 38)
            self.assertEqual(alignment.target.annotations["start"], 137)
            self.assertEqual(alignment.target.annotations["end"], 38)
            self.assertEqual(alignment.target.seq, "VFKISIDLSLACLIEVFLLLDSEDNECNSRSNC")
            self.assertEqual(alignment.query.seq, "LFNISMDLSLAWRMVEFLLFDSEDKERNSASSC")
            self.assertAlmostEqual(alignment.annotations["evalue"], 7e-16, places=16)
            self.assertAlmostEqual(alignment.annotations["bit score"], 44.6)
            self.assertAlmostEqual(alignment.score, 91)
            self.assertEqual(alignment.shape, (2, 33))
            self.assertAlmostEqual(alignment.annotations["% identity"], 60.61)
            self.assertEqual(alignment.annotations["identical"], 20)
            self.assertEqual(alignment.annotations["mismatches"], 13)
            self.assertEqual(alignment.annotations["positives"], 24)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 72.73)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "1064592")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Naumovozyma castellii CBS 4309 hypothetical protein (NCAS0A09950) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Naumovozyma castellii CBS 4309 hypothetical protein (NCAS0A09950) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 28)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 91.0; 33 aligned letters; 20 identities; 13 mismatches; 24 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 91.0,
    aligned = 33:
        identities = 20,
        positives = 24,
        mismatches = 13.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 33)
            self.assertEqual(counts.identities, 20)
            self.assertEqual(counts.mismatches, 13)
            self.assertEqual(counts.positives, 24)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|366988         0 *LYFNCKMCRFYKSIENSR*SAFRYGWCRC*AL*TLHSNRGQLSVP**T*FYQWSDKIRM
                  0 ||.....|......|.|.|..|.|..|.....||.||......|.||..........|||
gi|296147         0 *LNLHREMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIRM

gi|366988        60 ACQIFNVLDSF*DLDRLVTCMLDRSVPSFRFGRQ*MQF*VQLFLKGLKTCSLNSH 115
                 60 .........||..|....|.|......||||||...||...|.|.|.|..|...| 115
gi|296147        60 SF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKLPLHGVKPSSVHGH 115
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 115],
                              [  0, 115]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|366988334|ref|XM_003673886.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|366988334|ref|XM_003673886.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "366988334")
            self.assertEqual(alignment.target.annotations["gis"], "366988334")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003673886")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003673886")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003673886.1")
            self.assertEqual(alignment.target.annotations["length"], 4938)
            self.assertEqual(alignment.query.annotations["start"], 345)
            self.assertEqual(alignment.query.annotations["end"], 0)
            self.assertEqual(alignment.target.annotations["start"], 345)
            self.assertEqual(alignment.target.annotations["end"], 0)
            self.assertEqual(
                alignment.target.seq,
                "*LYFNCKMCRFYKSIENSR*SAFRYGWCRC*AL*TLHSNRGQLSVP**T*FYQWSDKIRMACQIFNVLDSF*DLDRLVTCMLDRSVPSFRFGRQ*MQF*VQLFLKGLKTCSLNSH",
            )
            self.assertEqual(
                alignment.query.seq,
                "*LNLHREMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKLPLHGVKPSSVHGH",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 54.2)
            self.assertAlmostEqual(alignment.score, 112)
            self.assertEqual(alignment.shape, (2, 115))
            self.assertAlmostEqual(alignment.annotations["% identity"], 33.04)
            self.assertEqual(alignment.annotations["identical"], 38)
            self.assertEqual(alignment.annotations["mismatches"], 77)
            self.assertEqual(alignment.annotations["positives"], 58)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 50.43)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-3/-1")
            self.assertEqual(alignment.query.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["tax ids"], "1064592")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Naumovozyma castellii CBS 4309 hypothetical protein (NCAS0A09950) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Naumovozyma castellii CBS 4309 hypothetical protein (NCAS0A09950) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 99)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 112.0; 115 aligned letters; 38 identities; 77 mismatches; 58 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 112.0,
    aligned = 115:
        identities = 38,
        positives = 58,
        mismatches = 77.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 115)
            self.assertEqual(counts.identities, 38)
            self.assertEqual(counts.mismatches, 77)
            self.assertEqual(counts.positives, 58)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|255710         0 VRHASDKSIEILKTVHEFEDLPRHPDFVTPFVLSCASKNAKLTSVSVQCLQKMSSVNCIP
                  0 .||||||||||||.||.||.|.|||||..||||.|.|.|||.|....||||..|.|..||
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|255710        60 EDRIEDVLDAFIDSTHLAAEIQLK 84
                 60 ..|....|||||..||||.||||| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|255710474|ref|XM_002551475.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|255710474|ref|XM_002551475.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "255710474")
            self.assertEqual(alignment.target.annotations["gis"], "255710474")
            self.assertEqual(alignment.target.annotations["acc."], "XM_002551475")
            self.assertEqual(alignment.target.annotations["accs."], "XM_002551475")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_002551475.1")
            self.assertEqual(alignment.target.annotations["length"], 4845)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 93)
            self.assertEqual(alignment.target.annotations["end"], 345)
            self.assertEqual(
                alignment.target.seq,
                "VRHASDKSIEILKTVHEFEDLPRHPDFVTPFVLSCASKNAKLTSVSVQCLQKMSSVNCIPEDRIEDVLDAFIDSTHLAAEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-32, places=32)
            self.assertAlmostEqual(alignment.annotations["bit score"], 141)
            self.assertAlmostEqual(alignment.score, 303)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 65.48)
            self.assertEqual(alignment.annotations["identical"], 55)
            self.assertEqual(alignment.annotations["mismatches"], 29)
            self.assertEqual(alignment.annotations["positives"], 71)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 84.52)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "559295")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Lachancea thermotolerans CBS 6340 KLTH0A01342p (KLTH0A01342g) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Lachancea thermotolerans CBS 6340 KLTH0A01342p (KLTH0A01342g) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 73)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 303.0; 84 aligned letters; 55 identities; 29 mismatches; 71 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 303.0,
    aligned = 84:
        identities = 55,
        positives = 71,
        mismatches = 29.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 55)
            self.assertEqual(counts.mismatches, 29)
            self.assertEqual(counts.positives, 71)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|255710         0 TLSCISAARWVESMNASSTSSILSSGMQLTDDIFCRHCTETLVSLAFLEAHERTK 55
                  0 |...||.||.|.||.||..|.....|...|.|..||||....|.||||.....|| 55
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTK 55
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  55],
                              [  0,  55]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|255710474|ref|XM_002551475.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|255710474|ref|XM_002551475.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "255710474")
            self.assertEqual(alignment.target.annotations["gis"], "255710474")
            self.assertEqual(alignment.target.annotations["acc."], "XM_002551475")
            self.assertEqual(alignment.target.annotations["accs."], "XM_002551475")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_002551475.1")
            self.assertEqual(alignment.target.annotations["length"], 4845)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 185)
            self.assertEqual(alignment.target.annotations["start"], 347)
            self.assertEqual(alignment.target.annotations["end"], 182)
            self.assertEqual(
                alignment.target.seq,
                "TLSCISAARWVESMNASSTSSILSSGMQLTDDIFCRHCTETLVSLAFLEAHERTK",
            )
            self.assertEqual(
                alignment.query.seq,
                "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.006)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            self.assertAlmostEqual(alignment.score, 92)
            self.assertEqual(alignment.shape, (2, 55))
            self.assertAlmostEqual(alignment.annotations["% identity"], 45.45)
            self.assertEqual(alignment.annotations["identical"], 25)
            self.assertEqual(alignment.annotations["mismatches"], 30)
            self.assertEqual(alignment.annotations["positives"], 29)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 52.73)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "559295")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Lachancea thermotolerans CBS 6340 KLTH0A01342p (KLTH0A01342g) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Lachancea thermotolerans CBS 6340 KLTH0A01342p (KLTH0A01342g) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 73)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 47)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 92.0; 55 aligned letters; 25 identities; 30 mismatches; 29 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 92.0,
    aligned = 55:
        identities = 25,
        positives = 29,
        mismatches = 30.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 55)
            self.assertEqual(counts.identities, 25)
            self.assertEqual(counts.mismatches, 30)
            self.assertEqual(counts.positives, 29)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|254579         0 IRNASDKSIEILKVVHSYEELSRHPDFIVPLVMSCASKNAKLTTISMQCFQKLATVPCIP
                  0 ||.||||||||||.|||.|||.|||||..|.|..|.|.|||.||..|||.|.|.|||.||
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|254579        60 VDKLSDVLDAFIEANQLAMDIKLK 84
                 60 ...||..|||||||..|||.|.|| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|254579534|ref|XM_002495708.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|254579534|ref|XM_002495708.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "254579534")
            self.assertEqual(alignment.target.annotations["gis"], "254579534")
            self.assertEqual(alignment.target.annotations["acc."], "XM_002495708")
            self.assertEqual(alignment.target.annotations["accs."], "XM_002495708")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_002495708.1")
            self.assertEqual(alignment.target.annotations["length"], 4866)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 96)
            self.assertEqual(alignment.target.annotations["end"], 348)
            self.assertEqual(
                alignment.target.seq,
                "IRNASDKSIEILKVVHSYEELSRHPDFIVPLVMSCASKNAKLTTISMQCFQKLATVPCIPVDKLSDVLDAFIEANQLAMDIKLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 6e-32, places=32)
            self.assertAlmostEqual(alignment.annotations["bit score"], 141)
            self.assertAlmostEqual(alignment.score, 302)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 67.86)
            self.assertEqual(alignment.annotations["identical"], 57)
            self.assertEqual(alignment.annotations["mismatches"], 27)
            self.assertEqual(alignment.annotations["positives"], 72)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 85.71)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "559307")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Zygosaccharomyces rouxii hypothetical protein (ZYRO0C02266g) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Zygosaccharomyces rouxii hypothetical protein (ZYRO0C02266g) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 96)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 302.0; 84 aligned letters; 57 identities; 27 mismatches; 72 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 302.0,
    aligned = 84:
        identities = 57,
        positives = 72,
        mismatches = 27.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 57)
            self.assertEqual(counts.mismatches, 27)
            self.assertEqual(counts.positives, 72)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|254579         0 TFSFISIASWLASINASSTSESLSTGIHGTVANFWKHCIDIVVNLAFLD 49
                  0 ||..||||...||..||..|.|...||.|||.....|||..||.||||| 49
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLD 49
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  49],
                              [  0,  49]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|254579534|ref|XM_002495708.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|254579534|ref|XM_002495708.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "254579534")
            self.assertEqual(alignment.target.annotations["gis"], "254579534")
            self.assertEqual(alignment.target.annotations["acc."], "XM_002495708")
            self.assertEqual(alignment.target.annotations["accs."], "XM_002495708")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_002495708.1")
            self.assertEqual(alignment.target.annotations["length"], 4866)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 203)
            self.assertEqual(alignment.target.annotations["start"], 350)
            self.assertEqual(alignment.target.annotations["end"], 203)
            self.assertEqual(
                alignment.target.seq,
                "TFSFISIASWLASINASSTSESLSTGIHGTVANFWKHCIDIVVNLAFLD",
            )
            self.assertEqual(
                alignment.query.seq, "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLD"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 51.0)
            self.assertAlmostEqual(alignment.score, 105)
            self.assertEqual(alignment.shape, (2, 49))
            self.assertAlmostEqual(alignment.annotations["% identity"], 55.10)
            self.assertEqual(alignment.annotations["identical"], 27)
            self.assertEqual(alignment.annotations["mismatches"], 22)
            self.assertEqual(alignment.annotations["positives"], 33)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 67.35)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "559307")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Zygosaccharomyces rouxii hypothetical protein (ZYRO0C02266g) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Zygosaccharomyces rouxii hypothetical protein (ZYRO0C02266g) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 96)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 42)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 105.0; 49 aligned letters; 27 identities; 22 mismatches; 33 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 105.0,
    aligned = 49:
        identities = 27,
        positives = 33,
        mismatches = 22.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 49)
            self.assertEqual(counts.identities, 27)
            self.assertEqual(counts.mismatches, 22)
            self.assertEqual(counts.positives, 33)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|254579         0 TTFKISIDLSLAFLIAVFLFFDSDESECNSELSSFQKLPNSP 42
                  0 |.|.||.|||||.....||.|||...|.||..|......|.| 42
gi|296147         0 TLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCLCMESNPP 42
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  42],
                              [  0,  42]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|254579534|ref|XM_002495708.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|254579534|ref|XM_002495708.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "254579534")
            self.assertEqual(alignment.target.annotations["gis"], "254579534")
            self.assertEqual(alignment.target.annotations["acc."], "XM_002495708")
            self.assertEqual(alignment.target.annotations["accs."], "XM_002495708")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_002495708.1")
            self.assertEqual(alignment.target.annotations["length"], 4866)
            self.assertEqual(alignment.query.annotations["start"], 140)
            self.assertEqual(alignment.query.annotations["end"], 14)
            self.assertEqual(alignment.target.annotations["start"], 140)
            self.assertEqual(alignment.target.annotations["end"], 14)
            self.assertEqual(
                alignment.target.seq, "TTFKISIDLSLAFLIAVFLFFDSDESECNSELSSFQKLPNSP"
            )
            self.assertEqual(
                alignment.query.seq, "TLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCLCMESNPP"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 41.8)
            self.assertAlmostEqual(alignment.score, 85)
            self.assertEqual(alignment.shape, (2, 42))
            self.assertAlmostEqual(alignment.annotations["% identity"], 47.62)
            self.assertEqual(alignment.annotations["identical"], 20)
            self.assertEqual(alignment.annotations["mismatches"], 22)
            self.assertEqual(alignment.annotations["positives"], 25)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 59.52)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "559307")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Zygosaccharomyces rouxii hypothetical protein (ZYRO0C02266g) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Zygosaccharomyces rouxii hypothetical protein (ZYRO0C02266g) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 96)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 36)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 85.0; 42 aligned letters; 20 identities; 22 mismatches; 25 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 85.0,
    aligned = 42:
        identities = 20,
        positives = 25,
        mismatches = 22.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 42)
            self.assertEqual(counts.identities, 20)
            self.assertEqual(counts.mismatches, 22)
            self.assertEqual(counts.positives, 25)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|254579         0 RVAFTFIRVEEKEYCY*EC*R*IN*NFESSSQL*GIIKTSRFYSTASDVMCIQECQIDYY
                  0 |.||...|.|.|......||.||.|..|.|...||..|||.|.........|.||..|..
gi|296147         0 RIAFFIFRIEKKKFNHSPC***IH*DIEKST*F*GARKTSGFRTPFRVGLPIKEC*NDDP

gi|254579        60 INAMFPKIGHSAMYTGR*TLRRT*CVYRGQPAGNGYKTK 99
                 60 .|||.........|..........|...|.....|...| 99
gi|296147        60 GNAMPTGTVNRSIYSSKPAV*NFGCLH*GYSSRDGDSIK 99
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  99],
                              [  0,  99]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|254579534|ref|XM_002495708.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|254579534|ref|XM_002495708.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "254579534")
            self.assertEqual(alignment.target.annotations["gis"], "254579534")
            self.assertEqual(alignment.target.annotations["acc."], "XM_002495708")
            self.assertEqual(alignment.target.annotations["accs."], "XM_002495708")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_002495708.1")
            self.assertEqual(alignment.target.annotations["length"], 4866)
            self.assertEqual(alignment.query.annotations["start"], 50)
            self.assertEqual(alignment.query.annotations["end"], 347)
            self.assertEqual(alignment.target.annotations["start"], 50)
            self.assertEqual(alignment.target.annotations["end"], 347)
            self.assertEqual(
                alignment.target.seq,
                "RVAFTFIRVEEKEYCY*EC*R*IN*NFESSSQL*GIIKTSRFYSTASDVMCIQECQIDYYINAMFPKIGHSAMYTGR*TLRRT*CVYRGQPAGNGYKTK",
            )
            self.assertEqual(
                alignment.query.seq,
                "RIAFFIFRIEKKKFNHSPC***IH*DIEKST*F*GARKTSGFRTPFRVGLPIKEC*NDDPGNAMPTGTVNRSIYSSKPAV*NFGCLH*GYSSRDGDSIK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.006)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            self.assertAlmostEqual(alignment.score, 92)
            self.assertEqual(alignment.shape, (2, 99))
            self.assertAlmostEqual(alignment.annotations["% identity"], 31.31)
            self.assertEqual(alignment.annotations["identical"], 31)
            self.assertEqual(alignment.annotations["mismatches"], 68)
            self.assertEqual(alignment.annotations["positives"], 53)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 53.54)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "3/3")
            self.assertEqual(alignment.query.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["tax ids"], "559307")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Zygosaccharomyces rouxii hypothetical protein (ZYRO0C02266g) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Zygosaccharomyces rouxii hypothetical protein (ZYRO0C02266g) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 96)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 85)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 92.0; 99 aligned letters; 31 identities; 68 mismatches; 53 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 92.0,
    aligned = 99:
        identities = 31,
        positives = 53,
        mismatches = 68.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 99)
            self.assertEqual(counts.identities, 31)
            self.assertEqual(counts.mismatches, 68)
            self.assertEqual(counts.positives, 53)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|156843         0 VKHASDKSIEILKTVTNINDLTRHPDFVVPFILACSSGNAKLTSISMQCIQVISTVQCIP
                  0 ..|||||||||||.|.....|.|||||..||.|||.|.|||.|...|||.|..|||..||
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|156843        60 STRISEILDAFINATHLAVEIQLK 84
                 60 ..|.||||||||.|||||.||||| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|156843402|ref|XM_001644719.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|156843402|ref|XM_001644719.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "156843402")
            self.assertEqual(alignment.target.annotations["gis"], "156843402")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001644719.1")
            self.assertEqual(alignment.target.annotations["length"], 4914)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 96)
            self.assertEqual(alignment.target.annotations["end"], 348)
            self.assertEqual(
                alignment.target.seq,
                "VKHASDKSIEILKTVTNINDLTRHPDFVVPFILACSSGNAKLTSISMQCIQVISTVQCIPSTRISEILDAFINATHLAVEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-31, places=31)
            self.assertAlmostEqual(alignment.annotations["bit score"], 139)
            self.assertAlmostEqual(alignment.score, 299)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 66.67)
            self.assertEqual(alignment.annotations["identical"], 56)
            self.assertEqual(alignment.annotations["mismatches"], 28)
            self.assertEqual(alignment.annotations["positives"], 71)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 84.52)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "436907")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 299.0; 84 aligned letters; 56 identities; 28 mismatches; 71 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 299.0,
    aligned = 84:
        identities = 56,
        positives = 71,
        mismatches = 28.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 56)
            self.assertEqual(counts.mismatches, 28)
            self.assertEqual(counts.positives, 71)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|156843         0 WQLNLLILQPLKSN*LQTYNLYLLNRKEKVQMLNTLVINLLKF*KL*LI*MI*QDIQIS*
                  0 |...|..|.|.|.|..|...|||.||||..|....||||.|..||.......|.||.||.
gi|296147         0 WP*TLEGLTPCKGNLKQNCVLYLPNRKEEIQPFAMLVINPLRY*KEYIVLRS*KDIRISH

gi|156843        60 YHSFWHVLQEMQS*LQFLCNAFKLYQLYNVYQVLEFLKF*MLLLMLLI*RWKYN*K 116
                 60 ..|.|...|.|..|....|||....|..........|||.|..|.|||.||..||| 116
gi|296147        60 SLSCWLANQGMLK*RPWQCNAYRDCQPFHLFLEAGCLKFWMPSLRLLISRWRFN*K 116
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 116],
                              [  0, 116]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|156843402|ref|XM_001644719.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|156843402|ref|XM_001644719.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "156843402")
            self.assertEqual(alignment.target.annotations["gis"], "156843402")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001644719.1")
            self.assertEqual(alignment.target.annotations["length"], 4914)
            self.assertEqual(alignment.query.annotations["start"], 1)
            self.assertEqual(alignment.query.annotations["end"], 349)
            self.assertEqual(alignment.target.annotations["start"], 1)
            self.assertEqual(alignment.target.annotations["end"], 349)
            self.assertEqual(
                alignment.target.seq,
                "WQLNLLILQPLKSN*LQTYNLYLLNRKEKVQMLNTLVINLLKF*KL*LI*MI*QDIQIS*YHSFWHVLQEMQS*LQFLCNAFKLYQLYNVYQVLEFLKF*MLLLMLLI*RWKYN*K",
            )
            self.assertEqual(
                alignment.query.seq,
                "WP*TLEGLTPCKGNLKQNCVLYLPNRKEEIQPFAMLVINPLRY*KEYIVLRS*KDIRISHSLSCWLANQGMLK*RPWQCNAYRDCQPFHLFLEAGCLKFWMPSLRLLISRWRFN*K",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 7e-14, places=14)
            self.assertAlmostEqual(alignment.annotations["bit score"], 81.3)
            self.assertAlmostEqual(alignment.score, 171)
            self.assertEqual(alignment.shape, (2, 116))
            self.assertAlmostEqual(alignment.annotations["% identity"], 42.24)
            self.assertEqual(alignment.annotations["identical"], 49)
            self.assertEqual(alignment.annotations["mismatches"], 67)
            self.assertEqual(alignment.annotations["positives"], 66)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 56.90)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "2/2")
            self.assertEqual(alignment.query.annotations["frame"], "2")
            self.assertEqual(alignment.target.annotations["frame"], "2")
            self.assertEqual(alignment.target.annotations["tax ids"], "436907")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 99)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 171.0; 116 aligned letters; 49 identities; 67 mismatches; 66 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 171.0,
    aligned = 116:
        identities = 49,
        positives = 66,
        mismatches = 67.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 116)
            self.assertEqual(counts.identities, 49)
            self.assertEqual(counts.mismatches, 67)
            self.assertEqual(counts.positives, 66)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|156843         0 F*LYFHR*MSSINKSI*NFRNSSTWYTLYS*YNLNALHRN*S*LCIS*RTCQNEWYYEIW
                  0 |||..||.|||.|..|.|||..........|..|.|||.......|.|...|.|...||.
gi|296147         0 F*LNLHREMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIR

gi|156843        60 MSC*IIYISYSFQNFNRFITSVFNICTFSFRFRR*RL*VCSQLLFKGCKIKRFNCH 116
                 60 ||.|.....||||..|.||||..|....||||.|||...|..|...|.|......| 116
gi|296147        60 MSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKLPLHGVKPSSVHGH 116
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 116],
                              [  0, 116]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|156843402|ref|XM_001644719.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|156843402|ref|XM_001644719.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "156843402")
            self.assertEqual(alignment.target.annotations["gis"], "156843402")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001644719.1")
            self.assertEqual(alignment.target.annotations["length"], 4914)
            self.assertEqual(alignment.query.annotations["start"], 348)
            self.assertEqual(alignment.query.annotations["end"], 0)
            self.assertEqual(alignment.target.annotations["start"], 348)
            self.assertEqual(alignment.target.annotations["end"], 0)
            self.assertEqual(
                alignment.target.seq,
                "F*LYFHR*MSSINKSI*NFRNSSTWYTLYS*YNLNALHRN*S*LCIS*RTCQNEWYYEIWMSC*IIYISYSFQNFNRFITSVFNICTFSFRFRR*RL*VCSQLLFKGCKIKRFNCH",
            )
            self.assertEqual(
                alignment.query.seq,
                "F*LNLHREMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKLPLHGVKPSSVHGH",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 4e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 72.1)
            self.assertAlmostEqual(alignment.score, 151)
            self.assertEqual(alignment.shape, (2, 116))
            self.assertAlmostEqual(alignment.annotations["% identity"], 42.24)
            self.assertEqual(alignment.annotations["identical"], 49)
            self.assertEqual(alignment.annotations["mismatches"], 67)
            self.assertEqual(alignment.annotations["positives"], 60)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 51.72)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-3/-1")
            self.assertEqual(alignment.query.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["tax ids"], "436907")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 99)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 151.0; 116 aligned letters; 49 identities; 67 mismatches; 60 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 151.0,
    aligned = 116:
        identities = 49,
        positives = 60,
        mismatches = 67.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 116)
            self.assertEqual(counts.identities, 49)
            self.assertEqual(counts.mismatches, 67)
            self.assertEqual(counts.positives, 60)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|156843         0 TFNCISTAK*VALIKASKISEILVLGIHCTVDIT*MHCIEIEVSFAF 47
                  0 |||.||.|.|||..||||||.....||..|||....|||...|..|| 47
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAF 47
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  47],
                              [  0,  47]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|156843402|ref|XM_001644719.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|156843402|ref|XM_001644719.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "156843402")
            self.assertEqual(alignment.target.annotations["gis"], "156843402")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001644719.1")
            self.assertEqual(alignment.target.annotations["length"], 4914)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 209)
            self.assertEqual(alignment.target.annotations["start"], 350)
            self.assertEqual(alignment.target.annotations["end"], 209)
            self.assertEqual(
                alignment.target.seq, "TFNCISTAK*VALIKASKISEILVLGIHCTVDIT*MHCIEIEVSFAF"
            )
            self.assertEqual(
                alignment.query.seq, "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAF"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-10, places=10)
            self.assertAlmostEqual(alignment.annotations["bit score"], 47.3)
            self.assertAlmostEqual(alignment.score, 97)
            self.assertEqual(alignment.shape, (2, 47))
            self.assertAlmostEqual(alignment.annotations["% identity"], 55.32)
            self.assertEqual(alignment.annotations["identical"], 26)
            self.assertEqual(alignment.annotations["mismatches"], 21)
            self.assertEqual(alignment.annotations["positives"], 30)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 63.83)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "436907")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 40)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 97.0; 47 aligned letters; 26 identities; 21 mismatches; 30 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 97.0,
    aligned = 47:
        identities = 26,
        positives = 30,
        mismatches = 21.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 47)
            self.assertEqual(counts.identities, 26)
            self.assertEqual(counts.mismatches, 21)
            self.assertEqual(counts.positives, 30)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|156843         0 HFLKNMPK*MVLRNLDVLLNHLY*LQFSKFQ*IYH*RV*HLYFFFSIQKIKIVSL*SIAF
                  0 |.|...|.....||.||.|.......||..|.||||....|.|||||.|||...|...||
gi|296147         0 HSLIGKPTRKGVRNPDVFLAPQNYVLFSISQWIYH*HGEWLNFFFSIRKIKNAILLQVAF

gi|156843        60 
                 60 
gi|296147        60 
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  60],
                              [  0,  60]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|156843402|ref|XM_001644719.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|156843402|ref|XM_001644719.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "156843402")
            self.assertEqual(alignment.target.annotations["gis"], "156843402")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001644719.1")
            self.assertEqual(alignment.target.annotations["length"], 4914)
            self.assertEqual(alignment.query.annotations["start"], 214)
            self.assertEqual(alignment.query.annotations["end"], 34)
            self.assertEqual(alignment.target.annotations["start"], 214)
            self.assertEqual(alignment.target.annotations["end"], 34)
            self.assertEqual(
                alignment.target.seq,
                "HFLKNMPK*MVLRNLDVLLNHLY*LQFSKFQ*IYH*RV*HLYFFFSIQKIKIVSL*SIAF",
            )
            self.assertEqual(
                alignment.query.seq,
                "HSLIGKPTRKGVRNPDVFLAPQNYVLFSISQWIYH*HGEWLNFFFSIRKIKNAILLQVAF",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-10, places=10)
            self.assertAlmostEqual(alignment.annotations["bit score"], 40.5)
            self.assertAlmostEqual(alignment.score, 82)
            self.assertEqual(alignment.shape, (2, 60))
            self.assertAlmostEqual(alignment.annotations["% identity"], 45.00)
            self.assertEqual(alignment.annotations["identical"], 27)
            self.assertEqual(alignment.annotations["mismatches"], 33)
            self.assertEqual(alignment.annotations["positives"], 31)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 51.67)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-2/-3")
            self.assertEqual(alignment.query.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["tax ids"], "436907")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 51)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 82.0; 60 aligned letters; 27 identities; 33 mismatches; 31 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 82.0,
    aligned = 60:
        identities = 27,
        positives = 31,
        mismatches = 33.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 60)
            self.assertEqual(counts.identities, 27)
            self.assertEqual(counts.mismatches, 33)
            self.assertEqual(counts.positives, 31)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|156843         0 KAIDYRLTIFIF*IEKKKYRC*TR***IY*NFENCN*YK*FNKTSRFRSTIHFGMFFRKC
                  0 ||...|...|||.|||||......||||.|..|...|.....|||.||.....|.....|
gi|296147         0 KAT*SRIAFFIFRIEKKKFNHSPC***IH*DIEKST*F*GARKTSGFRTPFRVGLPIKEC

gi|156843        60 KANFNFYAMHSSYINCTMYTKY*NF*NFRCFY*CYSFSGGNTIKS 105
                 60 .......||.....|...|......|||.|..|.||...|..||| 105
gi|296147        60 *NDDPGNAMPTGTVNRSIYSSKPAV*NFGCLH*GYSSRDGDSIKS 105
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 105],
                              [  0, 105]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|156843402|ref|XM_001644719.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|156843402|ref|XM_001644719.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "156843402")
            self.assertEqual(alignment.target.annotations["gis"], "156843402")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001644719.1")
            self.assertEqual(alignment.target.annotations["length"], 4914)
            self.assertEqual(alignment.query.annotations["start"], 35)
            self.assertEqual(alignment.query.annotations["end"], 350)
            self.assertEqual(alignment.target.annotations["start"], 35)
            self.assertEqual(alignment.target.annotations["end"], 350)
            self.assertEqual(
                alignment.target.seq,
                "KAIDYRLTIFIF*IEKKKYRC*TR***IY*NFENCN*YK*FNKTSRFRSTIHFGMFFRKCKANFNFYAMHSSYINCTMYTKY*NF*NFRCFY*CYSFSGGNTIKS",
            )
            self.assertEqual(
                alignment.query.seq,
                "KAT*SRIAFFIFRIEKKKFNHSPC***IH*DIEKST*F*GARKTSGFRTPFRVGLPIKEC*NDDPGNAMPTGTVNRSIYSSKPAV*NFGCLH*GYSSRDGDSIKS",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-08, places=8)
            self.assertAlmostEqual(alignment.annotations["bit score"], 62.5)
            self.assertAlmostEqual(alignment.score, 130)
            self.assertEqual(alignment.shape, (2, 105))
            self.assertAlmostEqual(alignment.annotations["% identity"], 38.10)
            self.assertEqual(alignment.annotations["identical"], 40)
            self.assertEqual(alignment.annotations["mismatches"], 65)
            self.assertEqual(alignment.annotations["positives"], 58)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 55.24)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "3/3")
            self.assertEqual(alignment.query.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["tax ids"], "436907")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 90)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 130.0; 105 aligned letters; 40 identities; 65 mismatches; 58 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 130.0,
    aligned = 105:
        identities = 40,
        positives = 58,
        mismatches = 65.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 105)
            self.assertEqual(counts.identities, 40)
            self.assertEqual(counts.mismatches, 65)
            self.assertEqual(counts.positives, 58)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|156843         0 TVFKISIDLSLACLTSVLFLFDSEDKDCKSVVNC 34
                  0 |.|.||.|||||.......|||||||...|...| 34
gi|296147         0 TLFNISMDLSLAWRMVEFLLFDSEDKERNSASSC 34
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  34],
                              [  0,  34]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|156843402|ref|XM_001644719.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|156843402|ref|XM_001644719.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "156843402")
            self.assertEqual(alignment.target.annotations["gis"], "156843402")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001644719")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001644719.1")
            self.assertEqual(alignment.target.annotations["length"], 4914)
            self.assertEqual(alignment.query.annotations["start"], 140)
            self.assertEqual(alignment.query.annotations["end"], 38)
            self.assertEqual(alignment.target.annotations["start"], 140)
            self.assertEqual(alignment.target.annotations["end"], 38)
            self.assertEqual(alignment.target.seq, "TVFKISIDLSLACLTSVLFLFDSEDKDCKSVVNC")
            self.assertEqual(alignment.query.seq, "TLFNISMDLSLAWRMVEFLLFDSEDKERNSASSC")
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.19)
            self.assertAlmostEqual(alignment.annotations["bit score"], 40.0)
            self.assertAlmostEqual(alignment.score, 81)
            self.assertEqual(alignment.shape, (2, 34))
            self.assertAlmostEqual(alignment.annotations["% identity"], 52.94)
            self.assertEqual(alignment.annotations["identical"], 18)
            self.assertEqual(alignment.annotations["mismatches"], 16)
            self.assertEqual(alignment.annotations["positives"], 22)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 64.71)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "436907")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Vanderwaltozyma polyspora DSM 70294 hypothetical protein (Kpol_1020p19) partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 100)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 29)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 81.0; 34 aligned letters; 18 identities; 16 mismatches; 22 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 81.0,
    aligned = 34:
        identities = 18,
        positives = 22,
        mismatches = 16.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 34)
            self.assertEqual(counts.identities, 18)
            self.assertEqual(counts.mismatches, 16)
            self.assertEqual(counts.positives, 22)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|410075         0 VKHASEKSLKILKTVHDSGDFLRHPDFVVPFVLACSSRSAKLTTIGLQGLQNLSSTNCIP
                  0 ..|||.||..|||.||......|||||..||||||.||.||.||...|.||.||....||
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|410075        60 KDRLIEVLDGFIDATHLAMEIQLK 84
                 60 ..||.|.||.||.||||||||||| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|410075642|ref|XM_003955355.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|410075642|ref|XM_003955355.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "410075642")
            self.assertEqual(alignment.target.annotations["gis"], "410075642")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003955355")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003955355")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003955355.1")
            self.assertEqual(alignment.target.annotations["length"], 4881)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 96)
            self.assertEqual(alignment.target.annotations["end"], 348)
            self.assertEqual(
                alignment.target.seq,
                "VKHASEKSLKILKTVHDSGDFLRHPDFVVPFVLACSSRSAKLTTIGLQGLQNLSSTNCIPKDRLIEVLDGFIDATHLAMEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-28, places=28)
            self.assertAlmostEqual(alignment.annotations["bit score"], 129)
            self.assertAlmostEqual(alignment.score, 276)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 61.90)
            self.assertEqual(alignment.annotations["identical"], 52)
            self.assertEqual(alignment.annotations["mismatches"], 32)
            self.assertEqual(alignment.annotations["positives"], 67)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 79.76)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071382")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Kazachstania africana CBS 2517 hypothetical protein (KAFR0A08350), partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Kazachstania africana CBS 2517 hypothetical protein (KAFR0A08350), partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 90)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 276.0; 84 aligned letters; 52 identities; 32 mismatches; 67 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 276.0,
    aligned = 84:
        identities = 52,
        positives = 67,
        mismatches = 32.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 52)
            self.assertEqual(counts.mismatches, 32)
            self.assertEqual(counts.positives, 67)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|410075         0 TFS*ISIAKCVASINPSNTSINRSLGIQLVDDKFCKP*RPIVVNLALRDEQAKTNGTTKS
                  0 ||.|||||..|||...|..|..|..||....|..|......||.||..|.||.|.|..||
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTKGSAKS

gi|410075        60 GCLRKSPLSCTVFKILRDFSLACLTSEFLLLASDDKE*SPVSSCL 105
                 60 |||..|...||.|.|..|.|||....||||..|.|||....|||| 105
gi|296147        60 GCLSSSSKLCTLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCL 105
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 105],
                              [  0, 105]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|410075642|ref|XM_003955355.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|410075642|ref|XM_003955355.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "410075642")
            self.assertEqual(alignment.target.annotations["gis"], "410075642")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003955355")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003955355")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003955355.1")
            self.assertEqual(alignment.target.annotations["length"], 4881)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 35)
            self.assertEqual(alignment.target.annotations["start"], 350)
            self.assertEqual(alignment.target.annotations["end"], 35)
            self.assertEqual(
                alignment.target.seq,
                "TFS*ISIAKCVASINPSNTSINRSLGIQLVDDKFCKP*RPIVVNLALRDEQAKTNGTTKSGCLRKSPLSCTVFKILRDFSLACLTSEFLLLASDDKE*SPVSSCL",
            )
            self.assertEqual(
                alignment.query.seq,
                "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTKGSAKSGCLSSSSKLCTLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCL",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-09, places=9)
            self.assertAlmostEqual(alignment.annotations["bit score"], 67.5)
            self.assertAlmostEqual(alignment.score, 141)
            self.assertEqual(alignment.shape, (2, 105))
            self.assertAlmostEqual(alignment.annotations["% identity"], 49.52)
            self.assertEqual(alignment.annotations["identical"], 52)
            self.assertEqual(alignment.annotations["mismatches"], 53)
            self.assertEqual(alignment.annotations["positives"], 61)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 58.10)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071382")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Kazachstania africana CBS 2517 hypothetical protein (KAFR0A08350), partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Kazachstania africana CBS 2517 hypothetical protein (KAFR0A08350), partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 90)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 90)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 198.0; 105 aligned letters; 52 identities; 53 mismatches; 61 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 198.0,
    aligned = 105:
        identities = 52,
        positives = 61,
        mismatches = 53.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 105)
            self.assertEqual(counts.identities, 52)
            self.assertEqual(counts.mismatches, 53)
            self.assertEqual(counts.positives, 61)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|470736         0 VKHASDRSLQILRIVHSFEELERHPDFILPFVLSCKSGNAKFTSLSMQSLQRLAIHQSIP
                  0 ..||||.|..||..|||||||||||||.|||||.|.|.|||.|.|.||.||.|....|||
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|470736        60 REQIEHVLEALIDSTQLAVEIQLK 84
                 60 |......|.|.|..|.||.||||| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|47073627|ref|NM_210010.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|47073627|ref|NM_210010.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "47073627")
            self.assertEqual(alignment.target.annotations["gis"], "47073627")
            self.assertEqual(alignment.target.annotations["acc."], "NM_210010")
            self.assertEqual(alignment.target.annotations["accs."], "NM_210010")
            self.assertEqual(alignment.target.annotations["acc.ver"], "NM_210010.1")
            self.assertEqual(alignment.target.annotations["length"], 4776)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 93)
            self.assertEqual(alignment.target.annotations["end"], 345)
            self.assertEqual(
                alignment.target.seq,
                "VKHASDRSLQILRIVHSFEELERHPDFILPFVLSCKSGNAKFTSLSMQSLQRLAIHQSIPREQIEHVLEALIDSTQLAVEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-28, places=28)
            self.assertAlmostEqual(alignment.annotations["bit score"], 129)
            self.assertAlmostEqual(alignment.score, 276)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 61.90)
            self.assertEqual(alignment.annotations["identical"], 52)
            self.assertEqual(alignment.annotations["mismatches"], 32)
            self.assertEqual(alignment.annotations["positives"], 70)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 83.33)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "284811")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Ashbya gossypii ATCC 10895 AEL204Cp (AGOS_AEL204C) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Ashbya gossypii ATCC 10895 AEL204Cp (AGOS_AEL204C) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 94)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 276.0; 84 aligned letters; 52 identities; 32 mismatches; 70 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 276.0,
    aligned = 84:
        identities = 52,
        positives = 70,
        mismatches = 32.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 52)
            self.assertEqual(counts.mismatches, 32)
            self.assertEqual(counts.positives, 70)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|470736         0 TFS*ISTASCVESISASSTCSICSRGIDWWIARRCRLCIDSEVNFAFPDLHERTNGRIKS
                  0 ||.|||.|..|.|..||.......||||......||.||...|..||.|....|.|..||
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTKGSAKS

gi|470736        60 GCRSSSSKLWTIRNICNERSLACFTSELRRLDSEARACRSCVSCLTVLVN 110
                 60 ||.||||||.|..||....|||....|....|||.....|..|||....| 110
gi|296147        60 GCLSSSSKLCTLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCLCMESN 110
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 110],
                              [  0, 110]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|47073627|ref|NM_210010.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|47073627|ref|NM_210010.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "47073627")
            self.assertEqual(alignment.target.annotations["gis"], "47073627")
            self.assertEqual(alignment.target.annotations["acc."], "NM_210010")
            self.assertEqual(alignment.target.annotations["accs."], "NM_210010")
            self.assertEqual(alignment.target.annotations["acc.ver"], "NM_210010.1")
            self.assertEqual(alignment.target.annotations["length"], 4776)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 20)
            self.assertEqual(alignment.target.annotations["start"], 347)
            self.assertEqual(alignment.target.annotations["end"], 17)
            self.assertEqual(
                alignment.target.seq,
                "TFS*ISTASCVESISASSTCSICSRGIDWWIARRCRLCIDSEVNFAFPDLHERTNGRIKSGCRSSSSKLWTIRNICNERSLACFTSELRRLDSEARACRSCVSCLTVLVN",
            )
            self.assertEqual(
                alignment.query.seq,
                "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTKGSAKSGCLSSSSKLCTLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCLCMESN",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 4e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 52.4)
            self.assertAlmostEqual(alignment.score, 108)
            self.assertEqual(alignment.shape, (2, 110))
            self.assertAlmostEqual(alignment.annotations["% identity"], 44.55)
            self.assertEqual(alignment.annotations["identical"], 49)
            self.assertEqual(alignment.annotations["mismatches"], 61)
            self.assertEqual(alignment.annotations["positives"], 56)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 50.91)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "284811")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Ashbya gossypii ATCC 10895 AEL204Cp (AGOS_AEL204C) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Ashbya gossypii ATCC 10895 AEL204Cp (AGOS_AEL204C) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 94)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 94)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 172.0; 110 aligned letters; 49 identities; 61 mismatches; 56 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 172.0,
    aligned = 110:
        identities = 49,
        positives = 56,
        mismatches = 61.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 110)
            self.assertEqual(counts.identities, 49)
            self.assertEqual(counts.mismatches, 61)
            self.assertEqual(counts.positives, 56)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|470736         0 F*LNLNGQLCGINQCFEYMFYLLPWNRLVDSQTLQTLH*QRSKLRIS*LT*EDERKNKVW
                  0 |||||.......|............||....|.|..||.|.....|.||....||.....
gi|296147         0 F*LNLHREMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIR

gi|470736        60 VSF*FFKTMDYPQYL*RAIACMFHFRITSLRFRS*GVQILCQLP 104
                 60 .|||..|||...|||...|..|....|.|.||..|..|....|| 104
gi|296147        60 MSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKLP 104
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0, 104],
                              [  0, 104]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|47073627|ref|NM_210010.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|47073627|ref|NM_210010.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "47073627")
            self.assertEqual(alignment.target.annotations["gis"], "47073627")
            self.assertEqual(alignment.target.annotations["acc."], "NM_210010")
            self.assertEqual(alignment.target.annotations["accs."], "NM_210010")
            self.assertEqual(alignment.target.annotations["acc.ver"], "NM_210010.1")
            self.assertEqual(alignment.target.annotations["length"], 4776)
            self.assertEqual(alignment.query.annotations["start"], 348)
            self.assertEqual(alignment.query.annotations["end"], 36)
            self.assertEqual(alignment.target.annotations["start"], 345)
            self.assertEqual(alignment.target.annotations["end"], 33)
            self.assertEqual(
                alignment.target.seq,
                "F*LNLNGQLCGINQCFEYMFYLLPWNRLVDSQTLQTLH*QRSKLRIS*LT*EDERKNKVWVSF*FFKTMDYPQYL*RAIACMFHFRITSLRFRS*GVQILCQLP",
            )
            self.assertEqual(
                alignment.query.seq,
                "F*LNLHREMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKLP",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-04)
            self.assertAlmostEqual(alignment.annotations["bit score"], 50.1)
            self.assertAlmostEqual(alignment.score, 103)
            self.assertEqual(alignment.shape, (2, 104))
            self.assertAlmostEqual(alignment.annotations["% identity"], 35.58)
            self.assertEqual(alignment.annotations["identical"], 37)
            self.assertEqual(alignment.annotations["mismatches"], 67)
            self.assertEqual(alignment.annotations["positives"], 52)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 50.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-3/-1")
            self.assertEqual(alignment.query.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["tax ids"], "284811")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Ashbya gossypii ATCC 10895 AEL204Cp (AGOS_AEL204C) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Ashbya gossypii ATCC 10895 AEL204Cp (AGOS_AEL204C) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 94)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 89)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 103.0; 104 aligned letters; 37 identities; 67 mismatches; 52 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 103.0,
    aligned = 104:
        identities = 37,
        positives = 52,
        mismatches = 67.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 104)
            self.assertEqual(counts.identities, 37)
            self.assertEqual(counts.mismatches, 67)
            self.assertEqual(counts.positives, 52)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|502941         0 VKEASHKSIEILKTIKTINDLENHPDFVVPFALACKTKNAKMTTIAMQCLQNMASTRCIP
                  0 ...||.|||||||.......||.||||..||.|||...||||||.||||||.......||
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|502941        60 EKRMDMILDAFIEATQLAMDIQLK 84
                 60 ..|...|||||||||.|||.|||| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|50294190|ref|XM_449507.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|50294190|ref|XM_449507.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "50294190")
            self.assertEqual(alignment.target.annotations["gis"], "50294190")
            self.assertEqual(alignment.target.annotations["acc."], "XM_449507")
            self.assertEqual(alignment.target.annotations["accs."], "XM_449507")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_449507.1")
            self.assertEqual(alignment.target.annotations["length"], 4980)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 183)
            self.assertEqual(alignment.target.annotations["end"], 435)
            self.assertEqual(
                alignment.target.seq,
                "VKEASHKSIEILKTIKTINDLENHPDFVVPFALACKTKNAKMTTIAMQCLQNMASTRCIPEKRMDMILDAFIEATQLAMDIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-28, places=28)
            self.assertAlmostEqual(alignment.annotations["bit score"], 128)
            self.assertAlmostEqual(alignment.score, 274)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 60.71)
            self.assertEqual(alignment.annotations["identical"], 51)
            self.assertEqual(alignment.annotations["mismatches"], 33)
            self.assertEqual(alignment.annotations["positives"], 66)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 78.57)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "284593")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Candida glabrata CBS 138 hypothetical protein partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Candida glabrata CBS 138 hypothetical protein partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 90)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 274.0; 84 aligned letters; 51 identities; 33 mismatches; 66 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 274.0,
    aligned = 84:
        identities = 51,
        positives = 66,
        mismatches = 33.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 51)
            self.assertEqual(counts.mismatches, 33)
            self.assertEqual(counts.positives, 66)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|502941         0 TFNCISMASCVASINASNIISILFSGIHLVEAMFCRHCIAIVVILAFL 48
                  0 |||.||.|..|||..||.|......||.......||||||.||||||| 48
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFL 48
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  48],
                              [  0,  48]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|50294190|ref|XM_449507.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|50294190|ref|XM_449507.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "50294190")
            self.assertEqual(alignment.target.annotations["gis"], "50294190")
            self.assertEqual(alignment.target.annotations["acc."], "XM_449507")
            self.assertEqual(alignment.target.annotations["accs."], "XM_449507")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_449507.1")
            self.assertEqual(alignment.target.annotations["length"], 4980)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 206)
            self.assertEqual(alignment.target.annotations["start"], 437)
            self.assertEqual(alignment.target.annotations["end"], 293)
            self.assertEqual(
                alignment.target.seq, "TFNCISMASCVASINASNIISILFSGIHLVEAMFCRHCIAIVVILAFL"
            )
            self.assertEqual(
                alignment.query.seq, "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFL"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-07)
            self.assertAlmostEqual(alignment.annotations["bit score"], 49.2)
            self.assertAlmostEqual(alignment.score, 101)
            self.assertEqual(alignment.shape, (2, 48))
            self.assertAlmostEqual(alignment.annotations["% identity"], 56.25)
            self.assertEqual(alignment.annotations["identical"], 27)
            self.assertEqual(alignment.annotations["mismatches"], 21)
            self.assertEqual(alignment.annotations["positives"], 29)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 60.42)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "284593")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Candida glabrata CBS 138 hypothetical protein partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Candida glabrata CBS 138 hypothetical protein partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 90)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 41)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 101.0; 48 aligned letters; 27 identities; 21 mismatches; 29 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 101.0,
    aligned = 48:
        identities = 27,
        positives = 29,
        mismatches = 21.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 48)
            self.assertEqual(counts.identities, 27)
            self.assertEqual(counts.mismatches, 21)
            self.assertEqual(counts.positives, 29)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|502941         0 VFNISIDLWLASLTSELRLRASATSA*RSASNCL 34
                  0 .||||.||.||....|..|..|......|||.|| 34
gi|296147         0 LFNISMDLSLAWRMVEFLLFDSEDKERNSASSCL 34
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  34],
                              [  0,  34]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|50294190|ref|XM_449507.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|50294190|ref|XM_449507.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "50294190")
            self.assertEqual(alignment.target.annotations["gis"], "50294190")
            self.assertEqual(alignment.target.annotations["acc."], "XM_449507")
            self.assertEqual(alignment.target.annotations["accs."], "XM_449507")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_449507.1")
            self.assertEqual(alignment.target.annotations["length"], 4980)
            self.assertEqual(alignment.query.annotations["start"], 137)
            self.assertEqual(alignment.query.annotations["end"], 35)
            self.assertEqual(alignment.target.annotations["start"], 224)
            self.assertEqual(alignment.target.annotations["end"], 122)
            self.assertEqual(alignment.target.seq, "VFNISIDLWLASLTSELRLRASATSA*RSASNCL")
            self.assertEqual(alignment.query.seq, "LFNISMDLSLAWRMVEFLLFDSEDKERNSASSCL")
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-07)
            self.assertAlmostEqual(alignment.annotations["bit score"], 27.6)
            self.assertAlmostEqual(alignment.score, 54)
            self.assertEqual(alignment.shape, (2, 34))
            self.assertAlmostEqual(alignment.annotations["% identity"], 47.06)
            self.assertEqual(alignment.annotations["identical"], 16)
            self.assertEqual(alignment.annotations["mismatches"], 18)
            self.assertEqual(alignment.annotations["positives"], 19)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 55.88)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "284593")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Candida glabrata CBS 138 hypothetical protein partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Candida glabrata CBS 138 hypothetical protein partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 90)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 29)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 54.0; 34 aligned letters; 16 identities; 18 mismatches; 19 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 54.0,
    aligned = 34:
        identities = 16,
        positives = 19,
        mismatches = 18.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 34)
            self.assertEqual(counts.identities, 16)
            self.assertEqual(counts.mismatches, 18)
            self.assertEqual(counts.positives, 19)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|444314         0 VKHTSDKALEILKTCHSNLDLKRHPDFIIPLIKACSSKSAKLTTIAMQCLQRMSSVDCIP
                  0 ..|.|||..||||..||...|.|||||..|...||.|..||.||.||||||..|.|..||
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|444314        60 DSRISQVLDSFIEATLLASDIQLK 84
                 60 .||.|..||.|||||.||..|||| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|444314872|ref|XM_004178046.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|444314872|ref|XM_004178046.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "444314872")
            self.assertEqual(alignment.target.annotations["gis"], "444314872")
            self.assertEqual(alignment.target.annotations["acc."], "XM_004178046")
            self.assertEqual(alignment.target.annotations["accs."], "XM_004178046")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_004178046.1")
            self.assertEqual(alignment.target.annotations["length"], 5190)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 99)
            self.assertEqual(alignment.target.annotations["end"], 351)
            self.assertEqual(
                alignment.target.seq,
                "VKHTSDKALEILKTCHSNLDLKRHPDFIIPLIKACSSKSAKLTTIAMQCLQRMSSVDCIPDSRISQVLDSFIEATLLASDIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-27, places=27)
            self.assertAlmostEqual(alignment.annotations["bit score"], 127)
            self.assertAlmostEqual(alignment.score, 271)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 59.52)
            self.assertEqual(alignment.annotations["identical"], 50)
            self.assertEqual(alignment.annotations["mismatches"], 34)
            self.assertEqual(alignment.annotations["positives"], 69)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 82.14)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071380")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Tetrapisispora blattae CBS 6284 hypothetical protein (TBLA0A07860) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Tetrapisispora blattae CBS 6284 hypothetical protein (TBLA0A07860) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 82)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 271.0; 84 aligned letters; 50 identities; 34 mismatches; 69 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 271.0,
    aligned = 84:
        identities = 50,
        positives = 69,
        mismatches = 34.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 50)
            self.assertEqual(counts.mismatches, 34)
            self.assertEqual(counts.positives, 69)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|444314         0 TFS*ISEANRVASMKESNT*DILESGIQSTDDILCKHCIAIVVNLALFELHAFIK 55
                  0 ||.|||.|..|||||.|...|....||..|.|..|.||||.||.||.....|..| 55
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTK 55
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  55],
                              [  0,  55]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|444314872|ref|XM_004178046.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|444314872|ref|XM_004178046.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "444314872")
            self.assertEqual(alignment.target.annotations["gis"], "444314872")
            self.assertEqual(alignment.target.annotations["acc."], "XM_004178046")
            self.assertEqual(alignment.target.annotations["accs."], "XM_004178046")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_004178046.1")
            self.assertEqual(alignment.target.annotations["length"], 5190)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 185)
            self.assertEqual(alignment.target.annotations["start"], 353)
            self.assertEqual(alignment.target.annotations["end"], 188)
            self.assertEqual(
                alignment.target.seq,
                "TFS*ISEANRVASMKESNT*DILESGIQSTDDILCKHCIAIVVNLALFELHAFIK",
            )
            self.assertEqual(
                alignment.query.seq,
                "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 48.7)
            self.assertAlmostEqual(alignment.score, 100)
            self.assertEqual(alignment.shape, (2, 55))
            self.assertAlmostEqual(alignment.annotations["% identity"], 50.91)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 27)
            self.assertEqual(alignment.annotations["positives"], 31)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 56.36)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071380")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Tetrapisispora blattae CBS 6284 hypothetical protein (TBLA0A07860) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Tetrapisispora blattae CBS 6284 hypothetical protein (TBLA0A07860) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 82)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 47)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 100.0; 55 aligned letters; 28 identities; 27 mismatches; 31 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 100.0,
    aligned = 55:
        identities = 28,
        positives = 31,
        mismatches = 27.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 55)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 27)
            self.assertEqual(counts.positives, 31)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|444314         0 VLRISKALSLVCLTSEFFLLDSETK 25
                  0 ...||..|||.....||.|.|||.| 25
gi|296147         0 LFNISMDLSLAWRMVEFLLFDSEDK 25
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  25],
                              [  0,  25]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|444314872|ref|XM_004178046.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|444314872|ref|XM_004178046.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "444314872")
            self.assertEqual(alignment.target.annotations["gis"], "444314872")
            self.assertEqual(alignment.target.annotations["acc."], "XM_004178046")
            self.assertEqual(alignment.target.annotations["accs."], "XM_004178046")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_004178046.1")
            self.assertEqual(alignment.target.annotations["length"], 5190)
            self.assertEqual(alignment.query.annotations["start"], 137)
            self.assertEqual(alignment.query.annotations["end"], 62)
            self.assertEqual(alignment.target.annotations["start"], 140)
            self.assertEqual(alignment.target.annotations["end"], 65)
            self.assertEqual(alignment.target.seq, "VLRISKALSLVCLTSEFFLLDSETK")
            self.assertEqual(alignment.query.seq, "LFNISMDLSLAWRMVEFLLFDSEDK")
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-05)
            self.assertAlmostEqual(alignment.annotations["bit score"], 23.5)
            self.assertAlmostEqual(alignment.score, 45)
            self.assertEqual(alignment.shape, (2, 25))
            self.assertAlmostEqual(alignment.annotations["% identity"], 48.00)
            self.assertEqual(alignment.annotations["identical"], 12)
            self.assertEqual(alignment.annotations["mismatches"], 13)
            self.assertEqual(alignment.annotations["positives"], 13)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 52.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071380")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Tetrapisispora blattae CBS 6284 hypothetical protein (TBLA0A07860) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Tetrapisispora blattae CBS 6284 hypothetical protein (TBLA0A07860) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 82)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 21)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 45.0; 25 aligned letters; 12 identities; 13 mismatches; 13 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 45.0,
    aligned = 25:
        identities = 12,
        positives = 13,
        mismatches = 13.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 25)
            self.assertEqual(counts.identities, 12)
            self.assertEqual(counts.mismatches, 13)
            self.assertEqual(counts.positives, 13)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|444314         0 SL*ALHCDSSQFGTLRTTCFYQRYYEIRVTF*IQIAMACLKDLQSFITCMFNF*IFSSGF
                  0 |||||||....|..........|..|||..||....|.....|..|||.|.|.||.|..|
gi|296147         0 SL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFRF

gi|444314        60 
                 60 
gi|296147        60 
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  60],
                              [  0,  60]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|444314872|ref|XM_004178046.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|444314872|ref|XM_004178046.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "444314872")
            self.assertEqual(alignment.target.annotations["gis"], "444314872")
            self.assertEqual(alignment.target.annotations["acc."], "XM_004178046")
            self.assertEqual(alignment.target.annotations["accs."], "XM_004178046")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_004178046.1")
            self.assertEqual(alignment.target.annotations["length"], 5190)
            self.assertEqual(alignment.query.annotations["start"], 252)
            self.assertEqual(alignment.query.annotations["end"], 72)
            self.assertEqual(alignment.target.annotations["start"], 255)
            self.assertEqual(alignment.target.annotations["end"], 75)
            self.assertEqual(
                alignment.target.seq,
                "SL*ALHCDSSQFGTLRTTCFYQRYYEIRVTF*IQIAMACLKDLQSFITCMFNF*IFSSGF",
            )
            self.assertEqual(
                alignment.query.seq,
                "SL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFRF",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.075)
            self.assertAlmostEqual(alignment.annotations["bit score"], 41.4)
            self.assertAlmostEqual(alignment.score, 84)
            self.assertEqual(alignment.shape, (2, 60))
            self.assertAlmostEqual(alignment.annotations["% identity"], 41.67)
            self.assertEqual(alignment.annotations["identical"], 25)
            self.assertEqual(alignment.annotations["mismatches"], 35)
            self.assertEqual(alignment.annotations["positives"], 31)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 51.67)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-3/-1")
            self.assertEqual(alignment.query.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071380")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Tetrapisispora blattae CBS 6284 hypothetical protein (TBLA0A07860) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Tetrapisispora blattae CBS 6284 hypothetical protein (TBLA0A07860) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 82)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 51)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 84.0; 60 aligned letters; 25 identities; 35 mismatches; 31 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 84.0,
    aligned = 60:
        identities = 25,
        positives = 31,
        mismatches = 35.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 60)
            self.assertEqual(counts.identities, 25)
            self.assertEqual(counts.mismatches, 35)
            self.assertEqual(counts.positives, 31)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|363751         0 VKHASDKSLQILRIVHSFQELERHPDFVHPFVLSCKSGNAKFTTLSMQCLQRLAIHRSIS
                  0 ..||||||..||..||||.||||||||..||||.|.|.|||.|||.|||||.|....||.
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|363751        60 KEQIEPVLEALIDSTQLAVEIQLK 84
                 60 .......|.|.|..|.||.||||| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|363751207|ref|XM_003645773.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|363751207|ref|XM_003645773.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "363751207")
            self.assertEqual(alignment.target.annotations["gis"], "363751207")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003645773.1")
            self.assertEqual(alignment.target.annotations["length"], 4794)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 93)
            self.assertEqual(alignment.target.annotations["end"], 345)
            self.assertEqual(
                alignment.target.seq,
                "VKHASDKSLQILRIVHSFQELERHPDFVHPFVLSCKSGNAKFTTLSMQCLQRLAIHRSISKEQIEPVLEALIDSTQLAVEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-27, places=27)
            self.assertAlmostEqual(alignment.annotations["bit score"], 127)
            self.assertAlmostEqual(alignment.score, 271)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 60.71)
            self.assertEqual(alignment.annotations["identical"], 51)
            self.assertEqual(alignment.annotations["mismatches"], 33)
            self.assertEqual(alignment.annotations["positives"], 69)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 82.14)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "931890")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 89)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 271.0; 84 aligned letters; 51 identities; 33 mismatches; 69 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 271.0,
    aligned = 84:
        identities = 51,
        positives = 69,
        mismatches = 33.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 51)
            self.assertEqual(counts.mismatches, 33)
            self.assertEqual(counts.positives, 69)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|363751         0 TFNWISTAN*VESINASKTGSICSLEIDLWIANRCKHCIERVVNLAFPDLHDNTK 55
                  0 |||.||.|.||.|..|||........||......|.|||.|||.|||.|...||| 55
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTK 55
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  55],
                              [  0,  55]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|363751207|ref|XM_003645773.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|363751207|ref|XM_003645773.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "363751207")
            self.assertEqual(alignment.target.annotations["gis"], "363751207")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003645773.1")
            self.assertEqual(alignment.target.annotations["length"], 4794)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 185)
            self.assertEqual(alignment.target.annotations["start"], 347)
            self.assertEqual(alignment.target.annotations["end"], 182)
            self.assertEqual(
                alignment.target.seq,
                "TFNWISTAN*VESINASKTGSICSLEIDLWIANRCKHCIERVVNLAFPDLHDNTK",
            )
            self.assertEqual(
                alignment.query.seq,
                "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQANTK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 8e-08, places=8)
            self.assertAlmostEqual(alignment.annotations["bit score"], 52.4)
            self.assertAlmostEqual(alignment.score, 108)
            self.assertEqual(alignment.shape, (2, 55))
            self.assertAlmostEqual(alignment.annotations["% identity"], 50.91)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 27)
            self.assertEqual(alignment.annotations["positives"], 32)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 58.18)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "931890")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 89)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 47)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 108.0; 55 aligned letters; 28 identities; 27 mismatches; 32 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 108.0,
    aligned = 55:
        identities = 28,
        positives = 32,
        mismatches = 27.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 55)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 27)
            self.assertEqual(counts.positives, 32)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|363751         0 TILKI*SDLSDACLTSEFLLLDSED 25
                  0 |...|..|||.|....||||.|||| 25
gi|296147         0 TLFNISMDLSLAWRMVEFLLFDSED 25
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  25],
                              [  0,  25]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|363751207|ref|XM_003645773.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|363751207|ref|XM_003645773.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "363751207")
            self.assertEqual(alignment.target.annotations["gis"], "363751207")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003645773.1")
            self.assertEqual(alignment.target.annotations["length"], 4794)
            self.assertEqual(alignment.query.annotations["start"], 140)
            self.assertEqual(alignment.query.annotations["end"], 65)
            self.assertEqual(alignment.target.annotations["start"], 137)
            self.assertEqual(alignment.target.annotations["end"], 62)
            self.assertEqual(alignment.target.seq, "TILKI*SDLSDACLTSEFLLLDSED")
            self.assertEqual(alignment.query.seq, "TLFNISMDLSLAWRMVEFLLFDSED")
            self.assertAlmostEqual(alignment.annotations["evalue"], 8e-08, places=8)
            self.assertAlmostEqual(alignment.annotations["bit score"], 27.2)
            self.assertAlmostEqual(alignment.score, 53)
            self.assertEqual(alignment.shape, (2, 25))
            self.assertAlmostEqual(alignment.annotations["% identity"], 56.00)
            self.assertEqual(alignment.annotations["identical"], 14)
            self.assertEqual(alignment.annotations["mismatches"], 11)
            self.assertEqual(alignment.annotations["positives"], 15)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 60.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "931890")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 89)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 21)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 53.0; 25 aligned letters; 14 identities; 11 mismatches; 15 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 53.0,
    aligned = 25:
        identities = 14,
        positives = 15,
        mismatches = 11.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 25)
            self.assertEqual(counts.identities, 14)
            self.assertEqual(counts.mismatches, 11)
            self.assertEqual(counts.positives, 15)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|363751         0 QSLQTLHRKSSKFSIS*FT**YKRVYKIGMSFQFLKTMNYS*DLKRFIRCMFNLRISSFG
                  0 |||..||.....|||.|......|...|.|||..||||.....|..||..|.|..||||.
gi|296147         0 QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFR

gi|363751        60 FRRQRIQILIQL 72
                 60 |.|.|.|....| 72
gi|296147        60 FGR*RTQFCFKL 72
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  72],
                              [  0,  72]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|363751207|ref|XM_003645773.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|363751207|ref|XM_003645773.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "363751207")
            self.assertEqual(alignment.target.annotations["gis"], "363751207")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003645773.1")
            self.assertEqual(alignment.target.annotations["length"], 4794)
            self.assertEqual(alignment.query.annotations["start"], 255)
            self.assertEqual(alignment.query.annotations["end"], 39)
            self.assertEqual(alignment.target.annotations["start"], 252)
            self.assertEqual(alignment.target.annotations["end"], 36)
            self.assertEqual(
                alignment.target.seq,
                "QSLQTLHRKSSKFSIS*FT**YKRVYKIGMSFQFLKTMNYS*DLKRFIRCMFNLRISSFGFRRQRIQILIQL",
            )
            self.assertEqual(
                alignment.query.seq,
                "QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQFCFKL",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.006)
            self.assertAlmostEqual(alignment.annotations["bit score"], 45.1)
            self.assertAlmostEqual(alignment.score, 92)
            self.assertEqual(alignment.shape, (2, 72))
            self.assertAlmostEqual(alignment.annotations["% identity"], 44.44)
            self.assertEqual(alignment.annotations["identical"], 32)
            self.assertEqual(alignment.annotations["mismatches"], 40)
            self.assertEqual(alignment.annotations["positives"], 37)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 51.39)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-3/-1")
            self.assertEqual(alignment.query.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["tax ids"], "931890")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 89)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 62)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 92.0; 72 aligned letters; 32 identities; 40 mismatches; 37 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 92.0,
    aligned = 72:
        identities = 32,
        positives = 37,
        mismatches = 40.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 72)
            self.assertEqual(counts.identities, 32)
            self.assertEqual(counts.mismatches, 40)
            self.assertEqual(counts.positives, 37)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|363751         0 *HFLIYMIIQKGVQNRDVVPIPENYELFLRFEAIYQMHV*PQNFFFWIQK 50
                  0 ||.||.....|||.|.||...|.||.||.....||..|....||||.|.| 50
gi|296147         0 *HSLIGKPTRKGVRNPDVFLAPQNYVLFSISQWIYH*HGEWLNFFFSIRK 50
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  50],
                              [  0,  50]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|363751207|ref|XM_003645773.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|363751207|ref|XM_003645773.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "363751207")
            self.assertEqual(alignment.target.annotations["gis"], "363751207")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003645773.1")
            self.assertEqual(alignment.target.annotations["length"], 4794)
            self.assertEqual(alignment.query.annotations["start"], 217)
            self.assertEqual(alignment.query.annotations["end"], 67)
            self.assertEqual(alignment.target.annotations["start"], 214)
            self.assertEqual(alignment.target.annotations["end"], 64)
            self.assertEqual(
                alignment.target.seq,
                "*HFLIYMIIQKGVQNRDVVPIPENYELFLRFEAIYQMHV*PQNFFFWIQK",
            )
            self.assertEqual(
                alignment.query.seq,
                "*HSLIGKPTRKGVRNPDVFLAPQNYVLFSISQWIYH*HGEWLNFFFSIRK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.021)
            self.assertAlmostEqual(alignment.annotations["bit score"], 43.2)
            self.assertAlmostEqual(alignment.score, 88)
            self.assertEqual(alignment.shape, (2, 50))
            self.assertAlmostEqual(alignment.annotations["% identity"], 48.00)
            self.assertEqual(alignment.annotations["identical"], 24)
            self.assertEqual(alignment.annotations["mismatches"], 26)
            self.assertEqual(alignment.annotations["positives"], 29)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 58.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-2/-3")
            self.assertEqual(alignment.query.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["tax ids"], "931890")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 89)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 43)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 88.0; 50 aligned letters; 24 identities; 26 mismatches; 29 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 88.0,
    aligned = 50:
        identities = 24,
        positives = 29,
        mismatches = 26.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 50)
            self.assertEqual(counts.identities, 24)
            self.assertEqual(counts.mismatches, 26)
            self.assertEqual(counts.positives, 29)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|363751         0 VF*IQKKKF*G*TCI**IASNLKNSS*FSGIGTTSRFCTPFCIIM*IRKC*IYYSFYAMF
                  0 .|.|.||||....|.|||......|.||.|...||.|.|||.....|..||......||.
gi|296147         0 IFRIEKKKFNHSPC***IH*DIEKST*F*GARKTSGFRTPFRVGLPIKEC*NDDPGNAMP

gi|363751        60 AAIGNPQIY 69
                 60 ....|..|| 69
gi|296147        60 TGTVNRSIY 69
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  69],
                              [  0,  69]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|363751207|ref|XM_003645773.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|363751207|ref|XM_003645773.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "363751207")
            self.assertEqual(alignment.target.annotations["gis"], "363751207")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003645773")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003645773.1")
            self.assertEqual(alignment.target.annotations["length"], 4794)
            self.assertEqual(alignment.query.annotations["start"], 65)
            self.assertEqual(alignment.query.annotations["end"], 272)
            self.assertEqual(alignment.target.annotations["start"], 62)
            self.assertEqual(alignment.target.annotations["end"], 269)
            self.assertEqual(
                alignment.target.seq,
                "VF*IQKKKF*G*TCI**IASNLKNSS*FSGIGTTSRFCTPFCIIM*IRKC*IYYSFYAMFAAIGNPQIY",
            )
            self.assertEqual(
                alignment.query.seq,
                "IFRIEKKKFNHSPC***IH*DIEKST*F*GARKTSGFRTPFRVGLPIKEC*NDDPGNAMPTGTVNRSIY",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.27)
            self.assertAlmostEqual(alignment.annotations["bit score"], 39.6)
            self.assertAlmostEqual(alignment.score, 80)
            self.assertEqual(alignment.shape, (2, 69))
            self.assertAlmostEqual(alignment.annotations["% identity"], 40.58)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 41)
            self.assertEqual(alignment.annotations["positives"], 38)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 55.07)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "3/3")
            self.assertEqual(alignment.query.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["tax ids"], "931890")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Eremothecium cymbalariae DBVPG#7215 hypothetical protein (Ecym_3526) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 89)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 59)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 80.0; 69 aligned letters; 28 identities; 41 mismatches; 38 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 80.0,
    aligned = 69:
        identities = 28,
        positives = 38,
        mismatches = 41.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 69)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 41)
            self.assertEqual(counts.positives, 38)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|366997         0 IHQASDKSIEILKTVKSIDELRRHPDFILPFIIATSSGNAKLTSIALQCIQRFLTVEYIP
                  0 |..||||||||||.|.|..||.|||||.|||..|..|.|||.|..|.||.|...||..||
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|366997        60 KSQLGALLDSFISATHLADEIKLK 84
                 60 .|.|...||.||.|||||.||.|| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|366997674|ref|XM_003683526.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|366997674|ref|XM_003683526.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "366997674")
            self.assertEqual(alignment.target.annotations["gis"], "366997674")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003683526")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003683526")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003683526.1")
            self.assertEqual(alignment.target.annotations["length"], 4890)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 99)
            self.assertEqual(alignment.target.annotations["end"], 351)
            self.assertEqual(
                alignment.target.seq,
                "IHQASDKSIEILKTVKSIDELRRHPDFILPFIIATSSGNAKLTSIALQCIQRFLTVEYIPKSQLGALLDSFISATHLADEIKLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-26, places=26)
            self.assertAlmostEqual(alignment.annotations["bit score"], 123)
            self.assertAlmostEqual(alignment.score, 263)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 61.90)
            self.assertEqual(alignment.annotations["identical"], 52)
            self.assertEqual(alignment.annotations["mismatches"], 32)
            self.assertEqual(alignment.annotations["positives"], 65)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 77.38)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071381")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Tetrapisispora phaffii CBS 4417 hypothetical protein (TPHA0A00550) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Tetrapisispora phaffii CBS 4417 hypothetical protein (TPHA0A00550) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 96)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 263.0; 84 aligned letters; 52 identities; 32 mismatches; 65 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 263.0,
    aligned = 84:
        identities = 52,
        positives = 65,
        mismatches = 32.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 52)
            self.assertEqual(counts.mismatches, 32)
            self.assertEqual(counts.positives, 65)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|366997         0 NQDVFSIHQLT*LFSKFQWTYH*LDESHCLFFLIRLIKDPSLIIVIF*KPRRLQ 54
                  0 |.|||...|...|||..||.|||..|....||.||.||...|..|.|.....|| 54
gi|296147         0 NPDVFLAPQNYVLFSISQWIYH*HGEWLNFFFSIRKIKNAILLQVAFAWSQTLQ 54
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  54],
                              [  0,  54]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|366997674|ref|XM_003683526.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|366997674|ref|XM_003683526.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "366997674")
            self.assertEqual(alignment.target.annotations["gis"], "366997674")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003683526")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003683526")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003683526.1")
            self.assertEqual(alignment.target.annotations["length"], 4890)
            self.assertEqual(alignment.query.annotations["start"], 175)
            self.assertEqual(alignment.query.annotations["end"], 13)
            self.assertEqual(alignment.target.annotations["start"], 178)
            self.assertEqual(alignment.target.annotations["end"], 16)
            self.assertEqual(
                alignment.target.seq,
                "NQDVFSIHQLT*LFSKFQWTYH*LDESHCLFFLIRLIKDPSLIIVIF*KPRRLQ",
            )
            self.assertEqual(
                alignment.query.seq,
                "NPDVFLAPQNYVLFSISQWIYH*HGEWLNFFFSIRKIKNAILLQVAFAWSQTLQ",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-07)
            self.assertAlmostEqual(alignment.annotations["bit score"], 40.0)
            self.assertAlmostEqual(alignment.score, 81)
            self.assertEqual(alignment.shape, (2, 54))
            self.assertAlmostEqual(alignment.annotations["% identity"], 46.30)
            self.assertEqual(alignment.annotations["identical"], 25)
            self.assertEqual(alignment.annotations["mismatches"], 29)
            self.assertEqual(alignment.annotations["positives"], 28)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 51.85)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-2/-3")
            self.assertEqual(alignment.query.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071381")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Tetrapisispora phaffii CBS 4417 hypothetical protein (TPHA0A00550) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Tetrapisispora phaffii CBS 4417 hypothetical protein (TPHA0A00550) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 96)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 46)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 81.0; 54 aligned letters; 25 identities; 29 mismatches; 28 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 81.0,
    aligned = 54:
        identities = 25,
        positives = 28,
        mismatches = 29.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 54)
            self.assertEqual(counts.identities, 25)
            self.assertEqual(counts.mismatches, 29)
            self.assertEqual(counts.positives, 28)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|366997         0 TFNFISSAKWVAEIKESSKAPN*LLGIYSTVKNLCIHCKAILVNFA 46
                  0 |||.||.|..||..|.|......|.||..||...|.||.|..|..| 46
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILA 46
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  46],
                              [  0,  46]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|366997674|ref|XM_003683526.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|366997674|ref|XM_003683526.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "366997674")
            self.assertEqual(alignment.target.annotations["gis"], "366997674")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003683526")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003683526")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003683526.1")
            self.assertEqual(alignment.target.annotations["length"], 4890)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 212)
            self.assertEqual(alignment.target.annotations["start"], 353)
            self.assertEqual(alignment.target.annotations["end"], 215)
            self.assertEqual(
                alignment.target.seq, "TFNFISSAKWVAEIKESSKAPN*LLGIYSTVKNLCIHCKAILVNFA"
            )
            self.assertEqual(
                alignment.query.seq, "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILA"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-07)
            self.assertAlmostEqual(alignment.annotations["bit score"], 36.8)
            self.assertAlmostEqual(alignment.score, 74)
            self.assertEqual(alignment.shape, (2, 46))
            self.assertAlmostEqual(alignment.annotations["% identity"], 45.65)
            self.assertEqual(alignment.annotations["identical"], 21)
            self.assertEqual(alignment.annotations["mismatches"], 25)
            self.assertEqual(alignment.annotations["positives"], 27)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 58.70)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071381")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Tetrapisispora phaffii CBS 4417 hypothetical protein (TPHA0A00550) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Tetrapisispora phaffii CBS 4417 hypothetical protein (TPHA0A00550) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 96)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 39)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 74.0; 46 aligned letters; 21 identities; 25 mismatches; 27 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 74.0,
    aligned = 46:
        identities = 21,
        positives = 27,
        mismatches = 25.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 46)
            self.assertEqual(counts.identities, 21)
            self.assertEqual(counts.mismatches, 25)
            self.assertEqual(counts.positives, 27)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|366997         0 QMGS*NKRV*QSAQLTLRNIFNS*KSLYTLQSNTC*FCITRRGSNYKWQYKIRMSSQFIN
                  0 .|.|.|.......|...||..|.|.||..|......|.|....|.......||||.....
gi|296147         0 EMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLK

gi|366997        60 *LNCFQNFNGLITSLMNLIVSSF*FV**KTQ 91
                 60 ....||..||.|||..|...|||.|..|.|| 91
gi|296147        60 TMYSFQYLNGFITSMANG*ISSFRFGR*RTQ 91
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  91],
                              [  0,  91]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|366997674|ref|XM_003683526.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|366997674|ref|XM_003683526.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "366997674")
            self.assertEqual(alignment.target.annotations["gis"], "366997674")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003683526")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003683526")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003683526.1")
            self.assertEqual(alignment.target.annotations["length"], 4890)
            self.assertEqual(alignment.query.annotations["start"], 327)
            self.assertEqual(alignment.query.annotations["end"], 54)
            self.assertEqual(alignment.target.annotations["start"], 330)
            self.assertEqual(alignment.target.annotations["end"], 57)
            self.assertEqual(
                alignment.target.seq,
                "QMGS*NKRV*QSAQLTLRNIFNS*KSLYTLQSNTC*FCITRRGSNYKWQYKIRMSSQFIN*LNCFQNFNGLITSLMNLIVSSF*FV**KTQ",
            )
            self.assertEqual(
                alignment.query.seq,
                "EMSSLNEGIQNFRQPASRNRWNG*QSL*ALHCQGRHFSIP*LASQHERECEIRMSF*LLKTMYSFQYLNGFITSMANG*ISSFRFGR*RTQ",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.001)
            self.assertAlmostEqual(alignment.annotations["bit score"], 47.3)
            self.assertAlmostEqual(alignment.score, 97)
            self.assertEqual(alignment.shape, (2, 91))
            self.assertAlmostEqual(alignment.annotations["% identity"], 36.26)
            self.assertEqual(alignment.annotations["identical"], 33)
            self.assertEqual(alignment.annotations["mismatches"], 58)
            self.assertEqual(alignment.annotations["positives"], 47)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 51.65)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-3/-1")
            self.assertEqual(alignment.query.annotations["frame"], "-3")
            self.assertEqual(alignment.target.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071381")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Tetrapisispora phaffii CBS 4417 hypothetical protein (TPHA0A00550) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Tetrapisispora phaffii CBS 4417 hypothetical protein (TPHA0A00550) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 96)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 78)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 97.0; 91 aligned letters; 33 identities; 58 mismatches; 47 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 97.0,
    aligned = 91:
        identities = 33,
        positives = 47,
        mismatches = 58.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 91)
            self.assertEqual(counts.identities, 33)
            self.assertEqual(counts.mismatches, 58)
            self.assertEqual(counts.positives, 47)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|366997         0 QTWVFYQTNQKEETMRFIKLVISPLKF*KQLSQLMN*EDILI 42
                  0 |..|.|..|.|||...|..|||.||..||....|..|.||.| 42
gi|296147         0 QNCVLYLPNRKEEIQPFAMLVINPLRY*KEYIVLRS*KDIRI 42
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[ 0, 42],
                              [ 0, 42]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|366997674|ref|XM_003683526.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|366997674|ref|XM_003683526.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "366997674")
            self.assertEqual(alignment.target.annotations["gis"], "366997674")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003683526")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003683526")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003683526.1")
            self.assertEqual(alignment.target.annotations["length"], 4890)
            self.assertEqual(alignment.query.annotations["start"], 49)
            self.assertEqual(alignment.query.annotations["end"], 175)
            self.assertEqual(alignment.target.annotations["start"], 52)
            self.assertEqual(alignment.target.annotations["end"], 178)
            self.assertEqual(
                alignment.target.seq, "QTWVFYQTNQKEETMRFIKLVISPLKF*KQLSQLMN*EDILI"
            )
            self.assertEqual(
                alignment.query.seq, "QNCVLYLPNRKEEIQPFAMLVINPLRY*KEYIVLRS*KDIRI"
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 0.075)
            self.assertAlmostEqual(alignment.annotations["bit score"], 41.4)
            self.assertAlmostEqual(alignment.score, 84)
            self.assertEqual(alignment.shape, (2, 42))
            self.assertAlmostEqual(alignment.annotations["% identity"], 47.62)
            self.assertEqual(alignment.annotations["identical"], 20)
            self.assertEqual(alignment.annotations["mismatches"], 22)
            self.assertEqual(alignment.annotations["positives"], 27)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 64.29)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "2/2")
            self.assertEqual(alignment.query.annotations["frame"], "2")
            self.assertEqual(alignment.target.annotations["frame"], "2")
            self.assertEqual(alignment.target.annotations["tax ids"], "1071381")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Tetrapisispora phaffii CBS 4417 hypothetical protein (TPHA0A00550) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Tetrapisispora phaffii CBS 4417 hypothetical protein (TPHA0A00550) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 96)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 36)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 84.0; 42 aligned letters; 20 identities; 22 mismatches; 27 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 84.0,
    aligned = 42:
        identities = 20,
        positives = 27,
        mismatches = 22.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 42)
            self.assertEqual(counts.identities, 20)
            self.assertEqual(counts.mismatches, 22)
            self.assertEqual(counts.positives, 27)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|367014         0 IKQASDKSIEILRTVRNYEELSNRSEFLAPFLMSCSSKNAKLTSISMQCLQRLSSTPSLS
                  0 |..|||||||||..|...|||.....|..||...|.|.|||.|...|||||.||..||..
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|367014        60 KDKLSDVLEAFIVATQLALDMKLK 84
                 60 ...||..|.|||.||.||....|| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|367014048|ref|XM_003681476.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|367014048|ref|XM_003681476.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "367014048")
            self.assertEqual(alignment.target.annotations["gis"], "367014048")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003681476")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003681476")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003681476.1")
            self.assertEqual(alignment.target.annotations["length"], 4902)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 96)
            self.assertEqual(alignment.target.annotations["end"], 348)
            self.assertEqual(
                alignment.target.seq,
                "IKQASDKSIEILRTVRNYEELSNRSEFLAPFLMSCSSKNAKLTSISMQCLQRLSSTPSLSKDKLSDVLEAFIVATQLALDMKLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-24, places=24)
            self.assertAlmostEqual(alignment.annotations["bit score"], 115)
            self.assertAlmostEqual(alignment.score, 246)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 52.38)
            self.assertEqual(alignment.annotations["identical"], 44)
            self.assertEqual(alignment.annotations["mismatches"], 40)
            self.assertEqual(alignment.annotations["positives"], 68)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 80.95)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "4950")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Torulaspora delbrueckii hypothetical protein (TDEL0E00700) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Torulaspora delbrueckii hypothetical protein (TDEL0E00700) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 90)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 246.0; 84 aligned letters; 44 identities; 40 mismatches; 68 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 246.0,
    aligned = 84:
        identities = 44,
        positives = 68,
        mismatches = 40.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 44)
            self.assertEqual(counts.mismatches, 40)
            self.assertEqual(counts.positives, 68)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|367014         0 TFNFMSRAN*VATMNASRTSDNLSFDRDGVDDNLCKHCIDIDVNLAFFEEQ 51
                  0 |||..|.|.|||.|.||..||......||..|..|.|||...|.|||...| 51
gi|296147         0 TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQ 51
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  51],
                              [  0,  51]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|367014048|ref|XM_003681476.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|367014048|ref|XM_003681476.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "367014048")
            self.assertEqual(alignment.target.annotations["gis"], "367014048")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003681476")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003681476")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003681476.1")
            self.assertEqual(alignment.target.annotations["length"], 4902)
            self.assertEqual(alignment.query.annotations["start"], 350)
            self.assertEqual(alignment.query.annotations["end"], 197)
            self.assertEqual(alignment.target.annotations["start"], 350)
            self.assertEqual(alignment.target.annotations["end"], 197)
            self.assertEqual(
                alignment.target.seq,
                "TFNFMSRAN*VATMNASRTSDNLSFDRDGVDDNLCKHCIDIDVNLAFFEEQ",
            )
            self.assertEqual(
                alignment.query.seq,
                "TFN*ISIAR*VASMKASKISDSRLRGIDGTVDSPCRHCIARVVILAFLDWQ",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 48.3)
            self.assertAlmostEqual(alignment.score, 99)
            self.assertEqual(alignment.shape, (2, 51))
            self.assertAlmostEqual(alignment.annotations["% identity"], 49.02)
            self.assertEqual(alignment.annotations["identical"], 25)
            self.assertEqual(alignment.annotations["mismatches"], 26)
            self.assertEqual(alignment.annotations["positives"], 32)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 62.75)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "4950")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Torulaspora delbrueckii hypothetical protein (TDEL0E00700) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Torulaspora delbrueckii hypothetical protein (TDEL0E00700) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 90)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 44)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 99.0; 51 aligned letters; 25 identities; 26 mismatches; 32 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 99.0,
    aligned = 51:
        identities = 25,
        positives = 32,
        mismatches = 26.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 51)
            self.assertEqual(counts.identities, 25)
            self.assertEqual(counts.mismatches, 26)
            self.assertEqual(counts.positives, 32)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|367014         0 TVLRISIDLSLACLMAELLFFASEDKSCNSEFNCL 35
                  0 |...||.|||||..|.|.|.|.||||..||...|| 35
gi|296147         0 TLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCL 35
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  35],
                              [  0,  35]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|367014048|ref|XM_003681476.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|367014048|ref|XM_003681476.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "367014048")
            self.assertEqual(alignment.target.annotations["gis"], "367014048")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003681476")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003681476")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003681476.1")
            self.assertEqual(alignment.target.annotations["length"], 4902)
            self.assertEqual(alignment.query.annotations["start"], 140)
            self.assertEqual(alignment.query.annotations["end"], 35)
            self.assertEqual(alignment.target.annotations["start"], 140)
            self.assertEqual(alignment.target.annotations["end"], 35)
            self.assertEqual(
                alignment.target.seq, "TVLRISIDLSLACLMAELLFFASEDKSCNSEFNCL"
            )
            self.assertEqual(alignment.query.seq, "TLFNISMDLSLAWRMVEFLLFDSEDKERNSASSCL")
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 43.7)
            self.assertAlmostEqual(alignment.score, 89)
            self.assertEqual(alignment.shape, (2, 35))
            self.assertAlmostEqual(alignment.annotations["% identity"], 57.14)
            self.assertEqual(alignment.annotations["identical"], 20)
            self.assertEqual(alignment.annotations["mismatches"], 15)
            self.assertEqual(alignment.annotations["positives"], 23)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 65.71)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "-1/-2")
            self.assertEqual(alignment.query.annotations["frame"], "-1")
            self.assertEqual(alignment.target.annotations["frame"], "-2")
            self.assertEqual(alignment.target.annotations["tax ids"], "4950")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Torulaspora delbrueckii hypothetical protein (TDEL0E00700) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Torulaspora delbrueckii hypothetical protein (TDEL0E00700) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 90)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 30)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 89.0; 35 aligned letters; 20 identities; 15 mismatches; 23 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 89.0,
    aligned = 35:
        identities = 20,
        positives = 23,
        mismatches = 15.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 35)
            self.assertEqual(counts.identities, 20)
            self.assertEqual(counts.mismatches, 15)
            self.assertEqual(counts.positives, 23)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|503027         0 IKQACEKSIEILQRSHSEEELVRHPDFVDPFITACLSGNAKLTSISMQSMQRISGIRCIC
                  0 |..|..||||||.|.||.|||.|||||..||..||.|.|||.|...||..|..|....|.
gi|296147         0 IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIP

gi|503027        60 TSKMESLLNALLKSTELAIDIQLK 84
                 60 .|.....|.|....|.||..|||| 84
gi|296147        60 RSRLSEILDAFIEATHLAMEIQLK 84
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  84],
                              [  0,  84]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|50302786|ref|XM_451330.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|50302786|ref|XM_451330.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "50302786")
            self.assertEqual(alignment.target.annotations["gis"], "50302786")
            self.assertEqual(alignment.target.annotations["acc."], "XM_451330")
            self.assertEqual(alignment.target.annotations["accs."], "XM_451330")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_451330.1")
            self.assertEqual(alignment.target.annotations["length"], 4809)
            self.assertEqual(alignment.query.annotations["start"], 96)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 93)
            self.assertEqual(alignment.target.annotations["end"], 345)
            self.assertEqual(
                alignment.target.seq,
                "IKQACEKSIEILQRSHSEEELVRHPDFVDPFITACLSGNAKLTSISMQSMQRISGIRCICTSKMESLLNALLKSTELAIDIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "IRHASDKSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-20, places=20)
            self.assertAlmostEqual(alignment.annotations["bit score"], 103)
            self.assertAlmostEqual(alignment.score, 219)
            self.assertEqual(alignment.shape, (2, 84))
            self.assertAlmostEqual(alignment.annotations["% identity"], 51.19)
            self.assertEqual(alignment.annotations["identical"], 43)
            self.assertEqual(alignment.annotations["mismatches"], 41)
            self.assertEqual(alignment.annotations["positives"], 63)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 75.00)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "284590")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Kluyveromyces lactis NRRL Y-1140 hypothetical protein partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Kluyveromyces lactis NRRL Y-1140 hypothetical protein partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 72)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 72)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 219.0; 84 aligned letters; 43 identities; 41 mismatches; 63 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 219.0,
    aligned = 84:
        identities = 43,
        positives = 63,
        mismatches = 41.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 84)
            self.assertEqual(counts.identities, 43)
            self.assertEqual(counts.mismatches, 41)
            self.assertEqual(counts.positives, 63)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|367029         0 KNLRASTEVQASEELAQRPNFVNPFIIACGTKNVKFTGIAIVCLQRLIVSRALPRARLSQ
                  0 |.......|...|||...|.|..||..||...|.|.|..|..|||.|......||.|||.
gi|296147         0 KSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSE

gi|367029        60 VLEALQQATSAGLDVQLK 78
                 60 .|.|...||......||| 78
gi|296147        60 ILDAFIEATHLAMEIQLK 78
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  78],
                              [  0,  78]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|367029360|ref|XM_003663916.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|367029360|ref|XM_003663916.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "367029360")
            self.assertEqual(alignment.target.annotations["gis"], "367029360")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003663916")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003663916")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003663916.1")
            self.assertEqual(alignment.target.annotations["length"], 5145)
            self.assertEqual(alignment.query.annotations["start"], 114)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 105)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "KNLRASTEVQASEELAQRPNFVNPFIIACGTKNVKFTGIAIVCLQRLIVSRALPRARLSQVLEALQQATSAGLDVQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "KSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-13, places=13)
            self.assertAlmostEqual(alignment.annotations["bit score"], 79.9)
            self.assertAlmostEqual(alignment.score, 168)
            self.assertEqual(alignment.shape, (2, 78))
            self.assertAlmostEqual(alignment.annotations["% identity"], 39.74)
            self.assertEqual(alignment.annotations["identical"], 31)
            self.assertEqual(alignment.annotations["mismatches"], 47)
            self.assertEqual(alignment.annotations["positives"], 52)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 66.67)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "573729")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Myceliophthora thermophila ATCC 42464 hypothetical protein (MYCTH_2139484) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Myceliophthora thermophila ATCC 42464 hypothetical protein (MYCTH_2139484) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 67)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 67)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 168.0; 78 aligned letters; 31 identities; 47 mismatches; 52 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 168.0,
    aligned = 78:
        identities = 31,
        positives = 52,
        mismatches = 47.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 78)
            self.assertEqual(counts.identities, 31)
            self.assertEqual(counts.mismatches, 47)
            self.assertEqual(counts.positives, 52)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|367039         0 KSLRASSEPQFADELAQRPNFVNPFIIACGTKNSKFTGIAIVCLQRLIIARAFPRSKLSQ
                  0 ||...........||...|.|..||..||...|.|.|..|..|||.|......|||.||.
gi|296147         0 KSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSE

gi|367039        60 ILDALQQATSAGLDVQLK 78
                 60 ||||...||......||| 78
gi|296147        60 ILDAFIEATHLAMEIQLK 78
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  78],
                              [  0,  78]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|367039842|ref|XM_003650254.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|367039842|ref|XM_003650254.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "367039842")
            self.assertEqual(alignment.target.annotations["gis"], "367039842")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003650254")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003650254")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003650254.1")
            self.assertEqual(alignment.target.annotations["length"], 5163)
            self.assertEqual(alignment.query.annotations["start"], 114)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 105)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "KSLRASSEPQFADELAQRPNFVNPFIIACGTKNSKFTGIAIVCLQRLIIARAFPRSKLSQILDALQQATSAGLDVQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "KSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-13, places=13)
            self.assertAlmostEqual(alignment.annotations["bit score"], 78.5)
            self.assertAlmostEqual(alignment.score, 165)
            self.assertEqual(alignment.shape, (2, 78))
            self.assertAlmostEqual(alignment.annotations["% identity"], 41.03)
            self.assertEqual(alignment.annotations["identical"], 32)
            self.assertEqual(alignment.annotations["mismatches"], 46)
            self.assertEqual(alignment.annotations["positives"], 50)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 64.10)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "578455")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Thielavia terrestris NRRL 8126 hypothetical protein (THITE_2039383) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Thielavia terrestris NRRL 8126 hypothetical protein (THITE_2039383) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 67)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 67)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 165.0; 78 aligned letters; 32 identities; 46 mismatches; 50 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 165.0,
    aligned = 78:
        identities = 32,
        positives = 50,
        mismatches = 46.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 78)
            self.assertEqual(counts.identities, 32)
            self.assertEqual(counts.mismatches, 46)
            self.assertEqual(counts.positives, 50)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|302895         0 EELSQKVNFVNPFIIACGTKNAKFTAIAIVCLQRLIVAQALPRSKLNQVLEALMQATSAG
                  0 |||.....|..||..||...|||.|..|..|||.|......|||.|...|.|...||...
gi|296147         0 EELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLA

gi|302895        60 LDVQLK 66
                 60 ...||| 66
gi|296147        60 MEIQLK 66
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  66],
                              [  0,  66]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|302895812|ref|XM_003046741.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|302895812|ref|XM_003046741.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "302895812")
            self.assertEqual(alignment.target.annotations["gis"], "302895812")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003046741")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003046741")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003046741.1")
            self.assertEqual(alignment.target.annotations["length"], 4878)
            self.assertEqual(alignment.query.annotations["start"], 150)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 141)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "EELSQKVNFVNPFIIACGTKNAKFTAIAIVCLQRLIVAQALPRSKLNQVLEALMQATSAGLDVQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "EELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 1e-12, places=12)
            self.assertAlmostEqual(alignment.annotations["bit score"], 77.1)
            self.assertAlmostEqual(alignment.score, 162)
            self.assertEqual(alignment.shape, (2, 66))
            self.assertAlmostEqual(alignment.annotations["% identity"], 42.42)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 38)
            self.assertEqual(alignment.annotations["positives"], 48)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 72.73)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "660122")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Nectria haematococca mpVI 77-13-4 hypothetical protein, mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Nectria haematococca mpVI 77-13-4 hypothetical protein, mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 57)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 57)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 162.0; 66 aligned letters; 28 identities; 38 mismatches; 48 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 162.0,
    aligned = 66:
        identities = 28,
        positives = 48,
        mismatches = 38.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 66)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 38)
            self.assertEqual(counts.positives, 48)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|302403         0 ELTQRPNFVNPFVIACGTKNTKFTAIAIVCLQRLILVRALPRGKLSHVLEALREATSAGL
                  0 ||...|.|..|||.||...|.|.|..|..|||.|..|...||..||..|.|..|||....
gi|296147         0 ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAM

gi|302403        60 DVQLK 65
                 60 ..||| 65
gi|296147        60 EIQLK 65
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  65],
                              [  0,  65]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|302403860|ref|XM_002999723.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|302403860|ref|XM_002999723.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "302403860")
            self.assertEqual(alignment.target.annotations["gis"], "302403860")
            self.assertEqual(alignment.target.annotations["acc."], "XM_002999723")
            self.assertEqual(alignment.target.annotations["accs."], "XM_002999723")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_002999723.1")
            self.assertEqual(alignment.target.annotations["length"], 5097)
            self.assertEqual(alignment.query.annotations["start"], 153)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 150)
            self.assertEqual(alignment.target.annotations["end"], 345)
            self.assertEqual(
                alignment.target.seq,
                "ELTQRPNFVNPFVIACGTKNTKFTAIAIVCLQRLILVRALPRGKLSHVLEALREATSAGLDVQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-12, places=12)
            self.assertAlmostEqual(alignment.annotations["bit score"], 76.7)
            self.assertAlmostEqual(alignment.score, 161)
            self.assertEqual(alignment.shape, (2, 65))
            self.assertAlmostEqual(alignment.annotations["% identity"], 46.15)
            self.assertEqual(alignment.annotations["identical"], 30)
            self.assertEqual(alignment.annotations["mismatches"], 35)
            self.assertEqual(alignment.annotations["positives"], 45)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 69.23)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "526221")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Verticillium albo-atrum VaMs.102 conserved hypothetical protein, mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Verticillium albo-atrum VaMs.102 conserved hypothetical protein, mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 56)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 56)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 161.0; 65 aligned letters; 30 identities; 35 mismatches; 45 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 161.0,
    aligned = 65:
        identities = 30,
        positives = 45,
        mismatches = 35.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 65)
            self.assertEqual(counts.identities, 30)
            self.assertEqual(counts.mismatches, 35)
            self.assertEqual(counts.positives, 45)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|154323         0 DLTQRPNFVTPFLIACGTRNVKFTGIAVVCLQRLVVSRALPKSRLKEVLEALREATSAGL
                  0 .|...|.|..||..||..||.|.|..|..|||.|......|.|||.|.|.|..|||....
gi|296147         0 ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAM

gi|154323        60 DVQLK 65
                 60 ..||| 65
gi|296147        60 EIQLK 65
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  65],
                              [  0,  65]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|154323271|ref|XM_001560900.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|154323271|ref|XM_001560900.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "154323271")
            self.assertEqual(alignment.target.annotations["gis"], "154323271")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001560900")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001560900")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001560900.1")
            self.assertEqual(alignment.target.annotations["length"], 3186)
            self.assertEqual(alignment.query.annotations["start"], 153)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 141)
            self.assertEqual(alignment.target.annotations["end"], 336)
            self.assertEqual(
                alignment.target.seq,
                "DLTQRPNFVTPFLIACGTRNVKFTGIAVVCLQRLVVSRALPKSRLKEVLEALREATSAGLDVQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 5e-12, places=12)
            self.assertAlmostEqual(alignment.annotations["bit score"], 75.3)
            self.assertAlmostEqual(alignment.score, 158)
            self.assertEqual(alignment.shape, (2, 65))
            self.assertAlmostEqual(alignment.annotations["% identity"], 44.62)
            self.assertEqual(alignment.annotations["identical"], 29)
            self.assertEqual(alignment.annotations["mismatches"], 36)
            self.assertEqual(alignment.annotations["positives"], 45)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 69.23)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "332648")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Botryotinia fuckeliana B05.10 hypothetical protein (BC1G_00035) partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Botryotinia fuckeliana B05.10 hypothetical protein (BC1G_00035) partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 56)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 56)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 158.0; 65 aligned letters; 29 identities; 36 mismatches; 45 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 158.0,
    aligned = 65:
        identities = 29,
        positives = 45,
        mismatches = 36.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 65)
            self.assertEqual(counts.identities, 29)
            self.assertEqual(counts.mismatches, 36)
            self.assertEqual(counts.positives, 45)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|389631         0 KSLNISSEAQLGPELTQKTNFANPFIIACGTKNAKFTGIAIVCLQRLIVSKALPRPRLNQ
                  0 ||..|........||.....||.||..||...|||.|..|..|||.|......||.||..
gi|296147         0 KSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSE

gi|389631        60 VLEALQGATSAGLDVQLK 78
                 60 .|.|...||......||| 78
gi|296147        60 ILDAFIEATHLAMEIQLK 78
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  78],
                              [  0,  78]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|389631003|ref|XM_003713107.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|389631003|ref|XM_003713107.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "389631003")
            self.assertEqual(alignment.target.annotations["gis"], "389631003")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003713107")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003713107")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003713107.1")
            self.assertEqual(alignment.target.annotations["length"], 5566)
            self.assertEqual(alignment.query.annotations["start"], 114)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 293)
            self.assertEqual(alignment.target.annotations["end"], 527)
            self.assertEqual(
                alignment.target.seq,
                "KSLNISSEAQLGPELTQKTNFANPFIIACGTKNAKFTGIAIVCLQRLIVSKALPRPRLNQVLEALQGATSAGLDVQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "KSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 73.5)
            self.assertAlmostEqual(alignment.score, 154)
            self.assertEqual(alignment.shape, (2, 78))
            self.assertAlmostEqual(alignment.annotations["% identity"], 39.74)
            self.assertEqual(alignment.annotations["identical"], 31)
            self.assertEqual(alignment.annotations["mismatches"], 47)
            self.assertEqual(alignment.annotations["positives"], 49)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 62.82)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/3")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "3")
            self.assertEqual(alignment.target.annotations["tax ids"], "242507")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Magnaporthe oryzae 70-15 hypothetical protein (MGG_07913) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Magnaporthe oryzae 70-15 hypothetical protein (MGG_07913) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 67)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 67)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 154.0; 78 aligned letters; 31 identities; 47 mismatches; 49 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 154.0,
    aligned = 78:
        identities = 31,
        positives = 49,
        mismatches = 47.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 78)
            self.assertEqual(counts.identities, 31)
            self.assertEqual(counts.mismatches, 47)
            self.assertEqual(counts.positives, 49)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|336276         0 KSLKVTSEAQISAELSQRSNFVNPFIIACGTKNVKFTGIAIVCLQRLIASRALPRFKLSQ
                  0 ||...........||.....|..||..||...|.|.|..|..|||.|......||..||.
gi|296147         0 KSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSE

gi|336276        60 VLEALQQATSAGLDVQLK 78
                 60 .|.|...||......||| 78
gi|296147        60 ILDAFIEATHLAMEIQLK 78
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  78],
                              [  0,  78]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|336276565|ref|XM_003352988.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|336276565|ref|XM_003352988.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "336276565")
            self.assertEqual(alignment.target.annotations["gis"], "336276565")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003352988")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003352988")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003352988.1")
            self.assertEqual(alignment.target.annotations["length"], 5166)
            self.assertEqual(alignment.query.annotations["start"], 114)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 105)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "KSLKVTSEAQISAELSQRSNFVNPFIIACGTKNVKFTGIAIVCLQRLIASRALPRFKLSQVLEALQQATSAGLDVQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "KSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 72.5)
            self.assertAlmostEqual(alignment.score, 152)
            self.assertEqual(alignment.shape, (2, 78))
            self.assertAlmostEqual(alignment.annotations["% identity"], 35.90)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 50)
            self.assertEqual(alignment.annotations["positives"], 49)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 62.82)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "771870")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Sordaria macrospora k-hell hypothetical protein (SMAC_03354), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Sordaria macrospora k-hell hypothetical protein (SMAC_03354), mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 67)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 67)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 152.0; 78 aligned letters; 28 identities; 50 mismatches; 49 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 152.0,
    aligned = 78:
        identities = 28,
        positives = 49,
        mismatches = 50.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 78)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 50)
            self.assertEqual(counts.positives, 49)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|164425         0 KSLKVTSEAQISAELSQRSNFVNPFIIACGTKNVKFTGIAIVCLQRLIASRALPRFKLSQ
                  0 ||...........||.....|..||..||...|.|.|..|..|||.|......||..||.
gi|296147         0 KSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSE

gi|164425        60 VLEALQQATSAGLDVQLK 78
                 60 .|.|...||......||| 78
gi|296147        60 ILDAFIEATHLAMEIQLK 78
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  78],
                              [  0,  78]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|164425502|ref|XM_955020.2|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|164425502|ref|XM_955020.2|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "164425502")
            self.assertEqual(alignment.target.annotations["gis"], "164425502")
            self.assertEqual(alignment.target.annotations["acc."], "XM_955020")
            self.assertEqual(alignment.target.annotations["accs."], "XM_955020")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_955020.2")
            self.assertEqual(alignment.target.annotations["length"], 5253)
            self.assertEqual(alignment.query.annotations["start"], 114)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 105)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "KSLKVTSEAQISAELSQRSNFVNPFIIACGTKNVKFTGIAIVCLQRLIASRALPRFKLSQVLEALQQATSAGLDVQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "KSIEILKRVHSFEELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 3e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 72.5)
            self.assertAlmostEqual(alignment.score, 152)
            self.assertEqual(alignment.shape, (2, 78))
            self.assertAlmostEqual(alignment.annotations["% identity"], 35.90)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 50)
            self.assertEqual(alignment.annotations["positives"], 49)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 62.82)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "367110")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Neurospora crassa OR74A hypothetical protein partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Neurospora crassa OR74A hypothetical protein partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 67)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 67)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 152.0; 78 aligned letters; 28 identities; 50 mismatches; 49 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 152.0,
    aligned = 78:
        identities = 28,
        positives = 49,
        mismatches = 50.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 78)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 50)
            self.assertEqual(counts.positives, 49)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|296807         0 DLARKPQFARPFVLACQTRHTRLAAIGVANLQRLVTIGALPQERLKDVLQGLHETANLSL
                  0 .|.|.|.||.|||||||.|...........||.|.|....|..||...|....|...|..
gi|296147         0 ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAM

gi|296807        60 EIQLK 65
                 60 ||||| 65
gi|296147        60 EIQLK 65
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  65],
                              [  0,  65]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|296807956|ref|XM_002844271.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|296807956|ref|XM_002844271.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "296807956")
            self.assertEqual(alignment.target.annotations["gis"], "296807956")
            self.assertEqual(alignment.target.annotations["acc."], "XM_002844271")
            self.assertEqual(alignment.target.annotations["accs."], "XM_002844271")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_002844271.1")
            self.assertEqual(alignment.target.annotations["length"], 5127)
            self.assertEqual(alignment.query.annotations["start"], 153)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 144)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "DLARKPQFARPFVLACQTRHTRLAAIGVANLQRLVTIGALPQERLKDVLQGLHETANLSLEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 6e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 71.6)
            self.assertAlmostEqual(alignment.score, 150)
            self.assertEqual(alignment.shape, (2, 65))
            self.assertAlmostEqual(alignment.annotations["% identity"], 43.08)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 37)
            self.assertEqual(alignment.annotations["positives"], 44)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 67.69)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "554155")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Arthroderma otae CBS 113480 conserved hypothetical protein, mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Arthroderma otae CBS 113480 conserved hypothetical protein, mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 56)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 56)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 150.0; 65 aligned letters; 28 identities; 37 mismatches; 44 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 150.0,
    aligned = 65:
        identities = 28,
        positives = 44,
        mismatches = 37.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 65)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 37)
            self.assertEqual(counts.positives, 44)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|327295         0 DLARKPQFARPFVLACQTRHTRLAAIGVTNLQRLVTIGALPHERLKDVLQGLHETANLSL
                  0 .|.|.|.||.|||||||.|...........||.|.|....|..||...|....|...|..
gi|296147         0 ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAM

gi|327295        60 EIQLK 65
                 60 ||||| 65
gi|296147        60 EIQLK 65
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  65],
                              [  0,  65]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|327295565|ref|XM_003232430.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|327295565|ref|XM_003232430.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "327295565")
            self.assertEqual(alignment.target.annotations["gis"], "327295565")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003232430")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003232430")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003232430.1")
            self.assertEqual(alignment.target.annotations["length"], 5157)
            self.assertEqual(alignment.query.annotations["start"], 153)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 144)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "DLARKPQFARPFVLACQTRHTRLAAIGVTNLQRLVTIGALPHERLKDVLQGLHETANLSLEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 8e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 71.2)
            self.assertAlmostEqual(alignment.score, 149)
            self.assertEqual(alignment.shape, (2, 65))
            self.assertAlmostEqual(alignment.annotations["% identity"], 43.08)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 37)
            self.assertEqual(alignment.annotations["positives"], 43)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 66.15)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "559305")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Trichophyton rubrum CBS 118892 hypothetical protein (TERG_07323) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Trichophyton rubrum CBS 118892 hypothetical protein (TERG_07323) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 56)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 56)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 149.0; 65 aligned letters; 28 identities; 37 mismatches; 43 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 149.0,
    aligned = 65:
        identities = 28,
        positives = 43,
        mismatches = 37.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 65)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 37)
            self.assertEqual(counts.positives, 43)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|317026         0 DLVRKPKFANPFILACHSRHAKLAGIGVVCLQRLVASRSLPSERLKDVLAGLKETTNMSL
                  0 .|.|.|.||.||.|||.||.||.......|||.|....|.|..||...|....|.|....
gi|296147         0 ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAM

gi|317026        60 DIQLK 65
                 60 .|||| 65
gi|296147        60 EIQLK 65
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  65],
                              [  0,  65]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|317026200|ref|XM_001389120.2|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|317026200|ref|XM_001389120.2|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "317026200")
            self.assertEqual(alignment.target.annotations["gis"], "317026200")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001389120")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001389120")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001389120.2")
            self.assertEqual(alignment.target.annotations["length"], 5229)
            self.assertEqual(alignment.query.annotations["start"], 153)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 144)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "DLVRKPKFANPFILACHSRHAKLAGIGVVCLQRLVASRSLPSERLKDVLAGLKETTNMSLDIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 8e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 71.2)
            self.assertAlmostEqual(alignment.score, 149)
            self.assertEqual(alignment.shape, (2, 65))
            self.assertAlmostEqual(alignment.annotations["% identity"], 44.62)
            self.assertEqual(alignment.annotations["identical"], 29)
            self.assertEqual(alignment.annotations["mismatches"], 36)
            self.assertEqual(alignment.annotations["positives"], 43)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 66.15)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "425011")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Aspergillus niger CBS 513.88 endosomal peripheral membrane protein (Mon2), mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Aspergillus niger CBS 513.88 endosomal peripheral membrane protein (Mon2), mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 56)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 56)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 149.0; 65 aligned letters; 29 identities; 36 mismatches; 43 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 149.0,
    aligned = 65:
        identities = 29,
        positives = 43,
        mismatches = 36.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 65)
            self.assertEqual(counts.identities, 29)
            self.assertEqual(counts.mismatches, 36)
            self.assertEqual(counts.positives, 43)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|156058         0 DLTQRANFVTPFLIACGTKNVKFTGIAVVCLQRLVVSRALPRSRLREVLEALREATSAGL
                  0 .|.....|..||..||...|.|.|..|..|||.|......|||||.|.|.|..|||....
gi|296147         0 ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAM

gi|156058        60 DVQLK 65
                 60 ..||| 65
gi|296147        60 EIQLK 65
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  65],
                              [  0,  65]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|156058004|ref|XM_001594876.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|156058004|ref|XM_001594876.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "156058004")
            self.assertEqual(alignment.target.annotations["gis"], "156058004")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001594876")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001594876")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001594876.1")
            self.assertEqual(alignment.target.annotations["length"], 4755)
            self.assertEqual(alignment.query.annotations["start"], 153)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 144)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "DLTQRANFVTPFLIACGTKNVKFTGIAVVCLQRLVVSRALPRSRLREVLEALREATSAGLDVQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 8e-11, places=11)
            self.assertAlmostEqual(alignment.annotations["bit score"], 71.2)
            self.assertAlmostEqual(alignment.score, 149)
            self.assertEqual(alignment.shape, (2, 65))
            self.assertAlmostEqual(alignment.annotations["% identity"], 43.08)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 37)
            self.assertEqual(alignment.annotations["positives"], 44)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 67.69)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "665079")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Sclerotinia sclerotiorum 1980 hypothetical protein (SS1G_04734) partial mRNA",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Sclerotinia sclerotiorum 1980 hypothetical protein (SS1G_04734) partial mRNA",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 56)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 56)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 149.0; 65 aligned letters; 28 identities; 37 mismatches; 44 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 149.0,
    aligned = 65:
        identities = 28,
        positives = 44,
        mismatches = 37.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 65)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 37)
            self.assertEqual(counts.positives, 44)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|315042         0 DLARKPQFSRPFVLACQTRHARLAAIGVANLQRLVTIGALPHERLKDVVQGLHETANLSL
                  0 .|.|.|.|..|||||||.|.|.........||.|.|....|..||........|...|..
gi|296147         0 ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAM

gi|315042        60 EIQLK 65
                 60 ||||| 65
gi|296147        60 EIQLK 65
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  65],
                              [  0,  65]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|315042717|ref|XM_003170687.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|315042717|ref|XM_003170687.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "315042717")
            self.assertEqual(alignment.target.annotations["gis"], "315042717")
            self.assertEqual(alignment.target.annotations["acc."], "XM_003170687")
            self.assertEqual(alignment.target.annotations["accs."], "XM_003170687")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_003170687.1")
            self.assertEqual(alignment.target.annotations["length"], 5178)
            self.assertEqual(alignment.query.annotations["start"], 153)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 144)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "DLARKPQFSRPFVLACQTRHARLAAIGVANLQRLVTIGALPHERLKDVVQGLHETANLSLEIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-10, places=10)
            self.assertAlmostEqual(alignment.annotations["bit score"], 70.3)
            self.assertAlmostEqual(alignment.score, 147)
            self.assertEqual(alignment.shape, (2, 65))
            self.assertAlmostEqual(alignment.annotations["% identity"], 41.54)
            self.assertEqual(alignment.annotations["identical"], 27)
            self.assertEqual(alignment.annotations["mismatches"], 38)
            self.assertEqual(alignment.annotations["positives"], 44)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 67.69)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "535722")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Arthroderma gypseum CBS 118893 hypothetical protein (MGYG_06723) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Arthroderma gypseum CBS 118893 hypothetical protein (MGYG_06723) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 56)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 56)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 147.0; 65 aligned letters; 27 identities; 38 mismatches; 44 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 147.0,
    aligned = 65:
        identities = 27,
        positives = 44,
        mismatches = 38.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 65)
            self.assertEqual(counts.identities, 27)
            self.assertEqual(counts.mismatches, 38)
            self.assertEqual(counts.positives, 44)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|255947         0 DLVRKPNFVEPFIIACHTRHAKLAGIGVICLQRLIASRSLPSSRLKDVLGGLKETTSLSL
                  0 .|.|.|.|..||..||..|.||.......|||.|....|.|.|||...|....|.|.|..
gi|296147         0 ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAM

gi|255947        60 DIQLK 65
                 60 .|||| 65
gi|296147        60 EIQLK 65
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  65],
                              [  0,  65]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|255947577|ref|XM_002564510.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|255947577|ref|XM_002564510.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "255947577")
            self.assertEqual(alignment.target.annotations["gis"], "255947577")
            self.assertEqual(alignment.target.annotations["acc."], "XM_002564510")
            self.assertEqual(alignment.target.annotations["accs."], "XM_002564510")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_002564510.1")
            self.assertEqual(alignment.target.annotations["length"], 5004)
            self.assertEqual(alignment.query.annotations["start"], 153)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 144)
            self.assertEqual(alignment.target.annotations["end"], 339)
            self.assertEqual(
                alignment.target.seq,
                "DLVRKPNFVEPFIIACHTRHAKLAGIGVICLQRLIASRSLPSSRLKDVLGGLKETTSLSLDIQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-10, places=10)
            self.assertAlmostEqual(alignment.annotations["bit score"], 69.8)
            self.assertAlmostEqual(alignment.score, 146)
            self.assertEqual(alignment.shape, (2, 65))
            self.assertAlmostEqual(alignment.annotations["% identity"], 43.08)
            self.assertEqual(alignment.annotations["identical"], 28)
            self.assertEqual(alignment.annotations["mismatches"], 37)
            self.assertEqual(alignment.annotations["positives"], 43)
            self.assertEqual(alignment.annotations["gap opens"], 0)
            self.assertEqual(alignment.annotations["gaps"], 0)
            self.assertAlmostEqual(alignment.annotations["% positives"], 66.15)
            self.assertEqual(alignment.annotations["query/sbjct frames"], "1/1")
            self.assertEqual(alignment.query.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["frame"], "1")
            self.assertEqual(alignment.target.annotations["tax ids"], "500485")
            self.assertEqual(alignment.target.annotations["sci names"], "N/A")
            self.assertEqual(alignment.target.annotations["com names"], "N/A")
            self.assertEqual(alignment.target.annotations["blast names"], "N/A")
            self.assertEqual(alignment.target.annotations["super kingdoms"], "N/A")
            self.assertEqual(
                alignment.target.annotations["title"],
                "Penicillium chrysogenum Wisconsin 54-1255 hypothetical protein (Pc22g05210) mRNA, complete cds",
            )
            self.assertEqual(
                alignment.target.annotations["titles"],
                "Penicillium chrysogenum Wisconsin 54-1255 hypothetical protein (Pc22g05210) mRNA, complete cds",
            )
            self.assertEqual(alignment.target.annotations["strand"], "N/A")
            self.assertAlmostEqual(alignment.target.annotations["% coverage"], 56)
            self.assertAlmostEqual(alignment.annotations["% hsp coverage"], 56)
            counts = alignment.counts(substitution_matrix)
            self.assertEqual(
                repr(counts),
                "<AlignmentCounts object (substitution score = 146.0; 65 aligned letters; 28 identities; 37 mismatches; 43 positives; 0 gaps) at 0x%x>"
                % id(counts),
            )
            self.assertEqual(
                str(counts),
                """\
AlignmentCounts object with
    substitution_score = 146.0,
    aligned = 65:
        identities = 28,
        positives = 43,
        mismatches = 37.
    gaps = 0:
        left_gaps = 0:
            left_insertions = 0:
                open_left_insertions = 0,
                extend_left_insertions = 0;
            left_deletions = 0:
                open_left_deletions = 0,
                extend_left_deletions = 0;
        internal_gaps = 0:
            internal_insertions = 0:
                open_internal_insertions = 0,
                extend_internal_insertions = 0;
            internal_deletions = 0:
                open_internal_deletions = 0,
                extend_internal_deletions = 0;
        right_gaps = 0:
            right_insertions = 0:
                open_right_insertions = 0,
                extend_right_insertions = 0;
            right_deletions = 0:
                open_right_deletions = 0,
                extend_right_deletions = 0.
""",
            )
            self.assertEqual(counts.left_insertions, 0)
            self.assertEqual(counts.left_deletions, 0)
            self.assertEqual(counts.right_insertions, 0)
            self.assertEqual(counts.right_deletions, 0)
            self.assertEqual(counts.internal_insertions, 0)
            self.assertEqual(counts.internal_deletions, 0)
            self.assertEqual(counts.left_gaps, 0)
            self.assertEqual(counts.right_gaps, 0)
            self.assertEqual(counts.internal_gaps, 0)
            self.assertEqual(counts.insertions, 0)
            self.assertEqual(counts.deletions, 0)
            self.assertEqual(counts.gaps, 0)
            self.assertEqual(counts.aligned, 65)
            self.assertEqual(counts.identities, 28)
            self.assertEqual(counts.mismatches, 37)
            self.assertEqual(counts.positives, 43)
            alignment = next(alignments)
            self.assertEqual(
                str(alignment),
                """\
gi|121708         0 DLVRKPKFVDPFILACHSRHAKLSGIGVVCLQRLVASRSLPSSRLKDVLGGLRETTSLNL
                  0 .|.|.|.|..||.|||.||.||.......|||.|....|.|.|||...|....|.|.|..
gi|296147         0 ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAM

gi|121708        60 DVQLK 65
                 60 ..||| 65
gi|296147        60 EIQLK 65
""",
            )
            self.assertTrue(
                np.array_equal(
                    alignment.coordinates,
                    # fmt: off
                    np.array([[  0,  65],
                              [  0,  65]])
                    # fmt: on
                )
            )
            self.assertEqual(alignment.query.id, "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.description,
                "Saccharomyces cerevisiae S288c Mon2p (MON2) mRNA, complete cds",
            )
            self.assertEqual(alignment.query.annotations["gi"], "0")
            self.assertEqual(alignment.query.annotations["acc."], "gi|296147483:1-350")
            self.assertEqual(
                alignment.query.annotations["acc.ver"], "gi|296147483:1-350"
            )
            self.assertEqual(alignment.target.id, "gi|121708212|ref|XM_001272061.1|")
            self.assertEqual(
                alignment.target.annotations["ids"], "gi|121708212|ref|XM_001272061.1|"
            )
            self.assertEqual(alignment.target.annotations["gi"], "121708212")
            self.assertEqual(alignment.target.annotations["gis"], "121708212")
            self.assertEqual(alignment.target.annotations["acc."], "XM_001272061")
            self.assertEqual(alignment.target.annotations["accs."], "XM_001272061")
            self.assertEqual(alignment.target.annotations["acc.ver"], "XM_001272061.1")
            self.assertEqual(alignment.target.annotations["length"], 5226)
            self.assertEqual(alignment.query.annotations["start"], 153)
            self.assertEqual(alignment.query.annotations["end"], 348)
            self.assertEqual(alignment.target.annotations["start"], 150)
            self.assertEqual(alignment.target.annotations["end"], 345)
            self.assertEqual(
                alignment.target.seq,
                "DLVRKPKFVDPFILACHSRHAKLSGIGVVCLQRLVASRSLPSSRLKDVLGGLRETTSLNLDVQLK",
            )
            self.assertEqual(
                alignment.query.seq,
                "ELERHPDFALPFVLACQSRNAKMTTLAMQCLQGLSTVPSIPRSRLSEILDAFIEATHLAMEIQLK",
            )
            self.assertAlmostEqual(alignment.annotations["evalue"], 2e-10, places=10)
            self.assertAlmostEqual(alignment.annotations["bit score"], 69.